export const MODULE_ID = "ripper-premium-dice-ea";

Hooks.on("init", () => {
    registerDice();
});

function registerDice() {
    Hooks.once("diceSoNiceReady", async (dice3d) => {
        for (const def of TEXTURE_DEFINITIONS) {
            await dice3d.addTexture(def.id, { name: def.name, source: def.source, composite: "source-over" });
        }

        COLORSETS.forEach((colorset) => dice3d.addColorset(colorset, "default"));

        const standardSystem = dice3d.getLoadedDiceSystems().get("standard");

        standardSystem.registerBeforeShaderCompileCallback((shader, material, diceType, appearance) => {
            const texId = material.userData?.materialData?.texture?.name;
            if (!texId) return;
            const isSpecialDice = SPECIAL_DICE.some((d) => texId == d.id);
            if (!isSpecialDice) return;
            shader.fragmentShader = shader.fragmentShader.replace("#include <map_fragment>", SHADERS[SPECIAL_DICE.find((d) => texId == d.id).shader].fragment);
            const colorBG = new PIXI.Color(material.userData?.materialData?.background || "#ffffff");
            const rgbBG = [colorBG.red, colorBG.green, colorBG.blue];
            shader.uniforms.time = game.dice3d.uniforms.time;
            shader.uniforms.uColor = { value: rgbBG };
            shader.fragmentShader = `uniform float time;\nuniform vec3 uColor;\n` + shader.fragmentShader;

            if (SPECIAL_DICE.find((d) => texId == d.id).shader.includes("WORLD")) {
                shader.fragmentShader = `varying vec3 sWorldPos;\n` + shader.fragmentShader;
                shader.uniforms.sWorldPos = { value: [0, 0, 0] };
                shader.vertexShader = `varying vec3 sWorldPos;\n` + shader.vertexShader;
                shader.vertexShader = shader.vertexShader.replace(
                    "#include <begin_vertex>",
                    `
                    #include <begin_vertex>
                    sWorldPos = (modelMatrix * vec4( position , 1.0 )).xyz;
                    `,
                );
            }
        });
    });
}

const SHADERS = {
    PATTERN_SQUARE: {
        fragment: `
        #include <map_fragment>
        vec2 uv = vMapUv;
        diffuseColor.rgb *= uColor;
        float time_speed = time * 5.0;
        float square_scale = 60.0;
        vec2 square_fract = square_scale * uv;
        vec2 tile = fract(square_fract);
        float dist = min(min(tile.x, 1.0-tile.x), min(tile.y, 1.0-tile.y));
        float square_dist = length(floor(square_fract));
        float e_side = sin(time_speed-square_dist*square_scale)*0.5;
        e_side = mod(e_side * e_side, e_side / e_side);
        float value = mix(dist, 1.0-dist, step(1.0, e_side));
        e_side = pow(abs(1.0-e_side), 2.2) * 0.5;
        value = smoothstep( e_side-0.05, e_side, 0.95*value);
        value = (value + square_dist*0.1)*0.6;
        diffuseColor.rgb *= value * 0.6;
        `,
    },
    PATTERN_CIRCLE: {
        fragment: `
        #include <map_fragment>
        vec2 uv = vMapUv;
        diffuseColor.rgb *= uColor;
        float time_speed = time * 3.0;

        float divisions = 32.0;
        
        vec2 closestCenterToUV = vec2(floor(uv.x * divisions) / divisions, floor(uv.y * divisions) / divisions);
        closestCenterToUV += 1.0 / divisions / 2.0;
        
        float circleDistance = (sin(time_speed + (closestCenterToUV.x + closestCenterToUV.y) * 10.0) * 0.5 + 0.5) / divisions;

        float distanceToClosestCenter = distance(closestCenterToUV, uv);

        float value = (1.0 - smoothstep(0.0, circleDistance, distanceToClosestCenter)) * 0.6 + 0.4;


        diffuseColor.rgb *= value;
        `,
    },
    PATTERN_CIRCLE_DARK: {
        fragment: `
        #include <map_fragment>
        vec2 uv = vMapUv;
        float time_speed = time * 3.0;

        float divisions = 32.0;
        
        vec2 closestCenterToUV = vec2(floor(uv.x * divisions) / divisions, floor(uv.y * divisions) / divisions);
        closestCenterToUV += 1.0 / divisions / 2.0;
        
        float circleDistance = (sin(time_speed + (closestCenterToUV.x + closestCenterToUV.y) * 10.0) * 0.5 + 0.5) / divisions;

        float distanceToClosestCenter = distance(closestCenterToUV, uv);

        float value = (1.0 - smoothstep(0.0, circleDistance, distanceToClosestCenter)) * 0.6 + 0.4;
        float totalDiffuseColor = (diffuseColor.r + diffuseColor.g + diffuseColor.b) / 3.0;
        if(totalDiffuseColor < 0.1){

            diffuseColor.rgb += uColor * clamp(pow(value, 5.0), 0.0, 1.0);
        }
        `,
    },
    PATTERN_CIRCLE_WORLD: {
        fragment: `
        #include <map_fragment>
        vec2 uv = sWorldPos.xy * 0.003;
        diffuseColor.rgb *= uColor;
        float time_speed = time * 3.0;

        float divisions = 32.0;
        
        vec2 closestCenterToUV = vec2(floor(uv.x * divisions) / divisions, floor(uv.y * divisions) / divisions);
        closestCenterToUV += 1.0 / divisions / 2.0;
        
        float circleDistance = (sin(time_speed + (closestCenterToUV.x + closestCenterToUV.y) * 10.0) * 0.5 + 0.5) / divisions;

        float distanceToClosestCenter = distance(closestCenterToUV, uv);

        float value = (1.0 - smoothstep(0.0, circleDistance, distanceToClosestCenter)) * 0.6 + 0.4;


        diffuseColor.rgb *= value;
        `,
    },
    PATTERN_MELT: {
        fragment: `
        #include <map_fragment>
        vec2 uv = vMapUv*10.0;
        diffuseColor.rgb *= uColor;
        float time_speed = time * 0.3;

        for(float i = 1.0; i < 10.0; i++){
            uv.x += 0.6 / i * cos(i * 2.5* uv.y + time_speed);
            uv.y += 0.6 / i * cos(i * 1.5 * uv.x + time_speed);
        }
        
        vec3 finalValue = vec3(vec3(0.1)/abs(sin(time_speed-uv.y-uv.x)));
        float avgColor = (finalValue.r + finalValue.g + finalValue.b) / 3.0;

        diffuseColor.rgb *= clamp(avgColor + 0.2, 0.0, 1.0);
        `,
    },
    PATTERN_GOLDEN: {
        fragment: `
        #include <map_fragment>
        vec2 uv = vMapUv*10.0;
        float time_speed = time * 0.3;

        for(float i = 1.0; i < 10.0; i++){
            uv.x += 0.6 / i * cos(i * 2.5* uv.y + time_speed);
            uv.y += 0.6 / i * cos(i * 1.5 * uv.x + time_speed);
        }
        
        vec3 finalValue = vec3(vec3(0.1)/abs(sin(time_speed-uv.y-uv.x)));
        float avgColor = (finalValue.r + finalValue.g + finalValue.b) / 3.0;
        float totalDiffuseColor = (diffuseColor.r + diffuseColor.g + diffuseColor.b) / 3.0;
        if(totalDiffuseColor < 0.1){

            diffuseColor.rgb += uColor * clamp(pow(avgColor, 3.0), 0.0, 1.0);
        }
        `,
    },
    PATTERN_ALIENTECH: {
        fragment: `
        #include <map_fragment>
        vec2 uv = vMapUv;
        diffuseColor.rgb *= uColor;
        float time_speed = time * 1.0;

        uv *= mat2(1, 1, -1, 1) / sqrt(2.0);
        uv += 0.5;
        uv *= 15.0;
    
        vec2 repeatingPattern = fract(uv);
        repeatingPattern = min(repeatingPattern, 1.0 - repeatingPattern);
        float value = fract(4.0 * min(repeatingPattern.x, repeatingPattern.y));
        float squareDist = length(floor(uv) - 4.5);
        float edge = 2.0 * fract((time_speed - squareDist * 0.5) / 4.3) - 1.0;
        value = edge < 0.0 ? value : 1.0 - value;
        value = smoothstep(-0.04, 0.0, 0.96 * value - edge * edge) + squareDist * 0.1;
        vec4 outputColor = mix(vec4(1), vec4(0.55, 0.70, 1.0, 1.0), value);
        float avgColor = (outputColor.r + outputColor.g + outputColor.b) / 3.0;

        diffuseColor.rgb *= avgColor;
        `,
    },
    PATTERN_ALIENTECH_DARK: {
        fragment: `
        #include <map_fragment>
        vec2 uv = vMapUv;
        float time_speed = time * 1.0;

        uv *= mat2(1, 1, -1, 1) / sqrt(2.0);
        uv += 0.5;
        uv *= 15.0;
    
        vec2 repeatingPattern = fract(uv);
        repeatingPattern = min(repeatingPattern, 1.0 - repeatingPattern);
        float value = fract(4.0 * min(repeatingPattern.x, repeatingPattern.y));
        float squareDist = length(floor(uv) - 4.5);
        float edge = 2.0 * fract((time_speed - squareDist * 0.5) / 4.3) - 1.0;
        value = edge < 0.0 ? value : 1.0 - value;
        value = smoothstep(-0.04, 0.0, 0.96 * value - edge * edge) + squareDist * 0.1;
        vec4 outputColor = mix(vec4(1), vec4(0.55, 0.70, 1.0, 1.0), value);
        float avgColor = (outputColor.r + outputColor.g + outputColor.b) / 3.0;

        float totalDiffuseColor = (diffuseColor.r + diffuseColor.g + diffuseColor.b) / 3.0;
        if(totalDiffuseColor < 0.1){

            diffuseColor.rgb += uColor * clamp(pow(avgColor, 3.0), 0.0, 1.0);
        }
        `,
    },
    PATTERN_MELT_WORLD: {
        fragment: `
        #include <map_fragment>
        vec2 uv = sWorldPos.xy * 0.05;
        diffuseColor.rgb *= uColor;
        float time_speed = time * 0.5;

        for(float i = 1.0; i < 10.0; i++){
            uv.x += 0.6 / i * cos(i * 2.5* uv.y + time_speed);
            uv.y += 0.6 / i * cos(i * 1.5 * uv.x + time_speed);
        }
        
        vec3 finalValue = vec3(vec3(0.1)/abs(sin(time_speed-uv.y-uv.x)));
        float avgColor = (finalValue.r + finalValue.g + finalValue.b) / 3.0;

        diffuseColor.rgb *= clamp(avgColor + 0.2, 0.0, 1.0);
        `,
    },
};

const SPECIAL_DICE = [
    {
        id: "ripper-premium-dice-pixelize",
        shader: "PATTERN_SQUARE",
    },
    {
        id: "ripper-premium-dice-circle",
        shader: "PATTERN_CIRCLE",
    },
    {
        id: "ripper-premium-dice-darkcircle",
        shader: "PATTERN_CIRCLE_DARK",
    },
    {
        id: "ripper-premium-dice-worldcircle",
        shader: "PATTERN_CIRCLE_WORLD",
    },
    {
        id: "ripper-premium-dice-melt",
        shader: "PATTERN_MELT",
    },
    {
        id: "ripper-premium-dice-worldmelt",
        shader: "PATTERN_MELT_WORLD",
    },
    {
        id: "ripper-premium-dice-alientech",
        shader: "PATTERN_ALIENTECH",
    },
    {
        id: "ripper-premium-dice-darkalientech",
        shader: "PATTERN_ALIENTECH_DARK",
    },
    {
        id: "ripper-premium-dice-golden",
        shader: "PATTERN_GOLDEN",
    },
];

const COLORSETS = [
    {
        name: "absolute-elegance",
        description: "Absolute Elegance",
        category: "The Ripper Premium Dice",
        foreground: "#c5c0aa",
        background: "#ff780a",
        texture: "ripper-premium-dice-golden",
        edge: "#ffbb00",
        outline: "#b0b0b0",
        material: "chrome",
        font: "Courier",
        visibility: "visible",
    },
    {
        name: "dinner-with-the-vampire",
        description: "Dinner with the Vampire",
        category: "The Ripper Premium Dice",
        foreground: "#ffdd00",
        background: "#ff0a0a",
        texture: "ripper-premium-dice-golden",
        edge: "#911d1d",
        outline: "#4d1f1f",
        material: "chrome",
        font: "Courier",
        visibility: "visible",
    },
    {
        name: "deep-sea",
        description: "Deep Sea",
        category: "The Ripper Premium Dice",
        foreground: "#ffffff",
        background: "#40689c",
        texture: "ripper-premium-dice-golden",
        edge: "#40689c",
        outline: "#ffffff",
        material: "chrome",
        font: "Courier",
        visibility: "visible",
    },
    {
        name: "hakerman",
        description: "Hakerman",
        category: "The Ripper Premium Dice",
        texture: "ripper-premium-dice-darkalientech",
        foreground: "#04ff00",
        background: "#227724",
        outline: "#000000",
        edge: "#04ff00",
        material: "chrome",
        font: "Courier New",
        visibility: "visible",
    },
    {
        name: "astronomer",
        description: "Astronomer",
        category: "The Ripper Premium Dice",
        texture: "ripper-premium-dice-darkcircle",
        foreground: "#ffffff",
        background: "#1841a0",
        outline: "#344760",
        edge: "#ccae28",
        material: "chrome",
        font: "Courier New",
        visibility: "visible",
    },
    {
        name: "bad-omen",
        description: "Bad Omen",
        category: "The Ripper Premium Dice",
        texture: "ripper-premium-dice-darkcircle",
        foreground: "#ffffff",
        background: "#ad0000",
        outline: "#603452",
        edge: "#ccae28",
        material: "chrome",
        font: "Courier New",
        visibility: "visible",
    },
    {
        name: "light-the-fire",
        description: "Light the Fire",
        category: "The Ripper Premium Dice",
        texture: "ripper-premium-dice-darkcircle",
        foreground: "#ffffff",
        background: "#ff4000",
        outline: "#604134",
        edge: "#7a4e00",
        material: "chrome",
        font: "Courier New",
        visibility: "visible",
    },
    {
        name: "pixel-dreams",
        description: "Pixel Dreams",
        category: "The Ripper Premium Dice",
        texture: "ripper-premium-dice-pixelize",
        foreground: "#000000",
        background: "#2896cc",
        outline: "#000000",
        edge: "#000000",
        material: "chrome",
        font: "Courier New",
        visibility: "visible",
    },
    {
        name: "enchanted-forest",
        description: "Enchanted Forest",
        category: "The Ripper Premium Dice",
        texture: "ripper-premium-dice-circle",
        foreground: "#000000",
        background: "#2896cc",
        outline: "#000000",
        edge: "#000000",
        material: "chrome",
        font: "Courier New",
        visibility: "visible",
    },
    {
        name: "limitless",
        description: "Limitless",
        category: "The Ripper Premium Dice",
        texture: "ripper-premium-dice-circle",
        foreground: "#000000",
        background: "#a694ff",
        outline: "#000000",
        edge: "#ffffff",
        material: "chrome",
        font: "Courier New",
        visibility: "visible",
    },
    {
        name: "endless-sea",
        description: "Endless Sea",
        category: "The Ripper Premium Dice",
        texture: "ripper-premium-dice-golden",
        foreground: "#ffffff",
        background: "#388eff",
        outline: "#000000",
        edge: "#000000",
        material: "chrome",
        font: "Courier New",
        visibility: "visible",
    },
    {
        name: "forbidden-technology",
        description: "Forbidden Technology",
        category: "The Ripper Premium Dice",
        texture: "ripper-premium-dice-alientech",
        foreground: "#000000",
        background: "#a6e3f7",
        outline: "#000000",
        edge: "#000000",
        material: "chrome",
        font: "Courier New",
        visibility: "visible",
    },
];

const TEXTURE_DEFINITIONS = [
    {
        id: "ripper-premium-dice-pixelize",
        name: "Pixelize (The Ripper Premium Dice)",
        source: `modules/${MODULE_ID}/assets/blank.jpg`,
    },
    {
        id: "ripper-premium-dice-circle",
        name: "Dot.Matrix (The Ripper Premium Dice)",
        source: `modules/${MODULE_ID}/assets/blank.jpg`,
    },
    {
        id: "ripper-premium-dice-darkcircle",
        name: "Dot.Matrix Dark (The Ripper Premium Dice)",
        source: `modules/${MODULE_ID}/assets/black.jpg`,
    },
    {
        id: "ripper-premium-dice-worldcircle",
        name: "Dot.Matrix Fusion (The Ripper Premium Dice)",
        source: `modules/${MODULE_ID}/assets/blank.jpg`,
    },
    {
        id: "ripper-premium-dice-melt",
        name: "Liquid Metal (The Ripper Premium Dice)",
        source: `modules/${MODULE_ID}/assets/blank.jpg`,
    },
    {
        id: "ripper-premium-dice-golden",
        name: "Liquid Metal Dark (The Ripper Premium Dice)",
        source: `modules/${MODULE_ID}/assets/black.jpg`,
    },
    {
        id: "ripper-premium-dice-worldmelt",
        name: "Liquid Metal Fusion (The Ripper Premium Dice)",
        source: `modules/${MODULE_ID}/assets/blank.jpg`,
    },
    {
        id: "ripper-premium-dice-alientech",
        name: "Nano Nexus (The Ripper Premium Dice)",
        source: `modules/${MODULE_ID}/assets/blank.jpg`,
    },
    {
        id: "ripper-premium-dice-darkalientech",
        name: "Nano Nexus Dark (The Ripper Premium Dice)",
        source: `modules/${MODULE_ID}/assets/black.jpg`,
    },
];
