import { getSetting } from "./settings.js";

export async function flip(token, id) {
    const tokenDocument = token.document ?? token;
    const nextFace = determineTargetFace(tokenDocument, id);
    if (!nextFace) return;

    const updateData = getUpdateData(nextFace);

    const animationSpeed = Math.max(0.01, getSetting("flipSpeed"));

    const transition = nextFace.transition || "flip";

    if(transition === "flip") return await updateWithFlipAnimation(tokenDocument, updateData, animationSpeed);

    return await tokenDocument.update(updateData, { animation: { duration: animationSpeed * 10000, transition } });
}

async function updateWithFlipAnimation(tokenDocument, updateData, animationSpeed) {
    animationSpeed *= 2;
    await tokenDocument.update(
        {
            texture: {
                scaleX: 0.0001,
            },
        },
        { animation: { duration: animationSpeed * 1000 } },
    );
    const p = foundry.canvas.animation.CanvasAnimation.getAnimation("Token." + tokenDocument.id + ".animate")?.promise;
    if (p) await p;
    else await new Promise((r) => setTimeout(r, animationSpeed * 1000));
    return await tokenDocument.update(updateData, { animation: { duration: animationSpeed * 1000 } });
}

function determineTargetFace(tokenDocument, id) {
    const faces = tokenDocument.getFlag("tokenflip", "tokenfaces") ?? [];
    if (!faces.length) {
        ui.notifications.warn("Token flip: No faces found");
        return false;
    }
    const manualFace = faces.find((f) => f.id === id);
    if (manualFace) return manualFace;
    try {
        const currentId = tokenDocument.flags?.tokenflip?.currentId;
        const currentIndex = faces.indexOf(faces.find((f) => f.id === currentId));
        const targetId = faces[(currentIndex + 1) % faces.length].id ?? faces[0].id;
        const targetFace = faces.find((f) => f.id === targetId) ?? faces[0];
        return targetFace;
    } catch (e) {
        return faces[0];
    }
}

function getUpdateData(nextFace) {
    const updateData = {
        actorId: nextFace.actorId,
        texture: {
            src: nextFace.img,
            scaleX: nextFace.scaleX,
            scaleY: nextFace.scaleY,
        },
        ring: {
            subject: {},
        },
        flags: {
            tokenflip: {
                currentId: nextFace.id,
            },
            "levels-3d-preview": {},
        },
    };
    if (nextFace.width) updateData.width = nextFace.width;
    if (nextFace.height) updateData.height = nextFace.height;
    if (nextFace.subject) updateData.ring.subject.texture = nextFace.subject;
    if (nextFace.model3d) updateData.flags["levels-3d-preview"].model3d = nextFace.model3d;
    return updateData;
}