/**
 * Modèle de données pour les arts magiques dans le système Myriad.
 * @extends {MyriadSystemItemBase}
 */
import MyriadSystemItemBase from "./base-item.mjs";

export default class MyriadSystemArt extends MyriadSystemItemBase {
  /** @override */
  static defineSchema() {
    const fields = foundry.data.fields;
    const requiredInteger = { required: true, nullable: false, integer: true };
    const baseSchema = super.defineSchema();

    return foundry.utils.mergeObject(baseSchema, {
      // Caractéristique liée à l'art magique
      linkedCharacteristic: new fields.StringField({
        required: true, 
        choices: ["puissance", "coordination", "ingeniosite", "perspicacite"],
        initial: "ingeniosite"
      }),
      
      // Avantages conférés par l'art
      benefits: new fields.ArrayField(
        new fields.StringField({ required: true })
      ),
      
      // Contraintes imposées par l'art
      constraints: new fields.ArrayField(
        new fields.StringField({ required: true })
      ),
      
      // Informations d'acquisition (facultatif)
      acquisition: new fields.SchemaField({
        xpCost: new fields.NumberField({ ...requiredInteger, initial: 25 }),
        prerequisites: new fields.ArrayField(
          new fields.StringField({ required: true })
        )
      })
    });
  }

  /**
   * Calcule le bonus de points de magie accordé par cet art.
   * Le bonus est calculé comme 2 × la dizaine de la caractéristique liée.
   * @param {MyriadSystemCharacter} actor - L'acteur possédant l'art
   * @returns {number} Le bonus de points de magie
   */
  calculateMagicPointsBonus(actor) {
    if (!actor?.system?.characteristics || !this.linkedCharacteristic) return 0;
    
    const charValue = actor.system.characteristics[this.linkedCharacteristic]?.value || 0;
    const tens = Math.floor(charValue / 10);
    return tens * 2;
  }
}
