/**
 * Extend the base Actor document by defining a custom roll data structure which is ideal for the Simple system.
 * @extends {Actor}
 */
export class MyriadSystemActor extends Actor {
  /** @override */
  prepareData() {
    // Prepare data for the actor. Calling the super version of this executes
    // the following, in order: data reset (to clear active effects),
    // prepareBaseData(), prepareEmbeddedDocuments() (including active effects),
    // prepareDerivedData().
    super.prepareData();
  }

  /** @override */
  getBarAttribute(name, options) {
    // Support Action Points token bar. Always return the root object path so Foundry can manage edits (value/max).
    if ([
      'actionPoints',
      'ap',
      'system.actionPoints',
      'system.actionPoints.value'
    ].includes(name)) {
      const ap = this.system.actionPoints;
      if (!ap || (ap.max ?? 0) <= 0) return null;
      return {
        attribute: 'system.actionPoints', // root object expected to contain {value,max}
        value: ap.value ?? 0,
        max: ap.max ?? 0
      };
    }
    return super.getBarAttribute(name, options);
  }

  /** @override */
  prepareBaseData() {
    // Data modifications in this step occur before processing embedded
    // documents or derived data.
  }

  /**
   * @override
   * Augment the actor source data with additional dynamic data that isn't 
   * handled by the actor's DataModel. Data calculated in this step should be
   * available both inside and outside of character sheets (such as if an actor
   * is queried and has a roll executed directly from it).
   */
  prepareDerivedData() {
    const actorData = this;
    const flags = actorData.flags.myriadsystem || {};
  }

  /**
   * 
   * @override
   * Augment the actor's default getRollData() method by appending the data object
   * generated by the its DataModel's getRollData(), or null. This polymorphic 
   * approach is useful when you have actors & items that share a parent Document, 
   * but have slightly different data preparation needs.
   */
  getRollData() {
    return { ...super.getRollData(), ...this.system.getRollData?.() ?? null };
  }

  /**
   * Convert the actor document to a plain object.
   * 
   * The built in `toObject()` method will ignore derived data when using Data Models.
   * This method creates a safe copy without circular references.
   * 
   * @returns {object} Plain object without circular references.
   */
  toPlainObject() {
    // Use the built-in toObject method to get a safe copy
    const result = this.toObject();
    
    // Add any additional derived data if needed
    // The toObject method already handles circular references safely
    return result;
  }

}
