/**
 * Modèle de données pour les domaines magiques dans le système Myriad.
 * @extends {MyriadSystemItemBase}
 */
import MyriadSystemItemBase from "./base-item.mjs";

export default class MyriadSystemDomain extends MyriadSystemItemBase {
  /** @override */
  static defineSchema() {
    const fields = foundry.data.fields;
    const requiredInteger = { required: true, nullable: false, integer: true };
    const baseSchema = super.defineSchema();

    return foundry.utils.mergeObject(baseSchema, {
      // Type de domaine (élément, corps, immatériel, etc.)
      type: new fields.StringField({
        required: true,
        initial: "other",
        choices: [
          "element", // Élémentaire (feu, eau, air, terre, etc.)
          "body",    // Corporel (corps, sang, os, etc.)
          "mind",    // Mental (esprit, émotion, mémoire, etc.)
          "nature",  // Nature (plantes, animaux, etc.)
          "energy",  // Énergie (électricité, chaleur, lumière, etc.)
          "matter",  // Matière (métal, pierre, etc.)
          "space",   // Espace (téléportation, dimensionnel, etc.)
          "time",    // Temps (ralentir, accélérer, etc.)
          "death",   // Mort (nécromancie, pourriture, etc.)
          "life",    // Vie (guérison, croissance, etc.)
          "other"    // Autre
        ]
      }),
      
      // Effets particuliers liés au domaine
      effects: new fields.ArrayField(
        new fields.StringField({ required: true })
      ),
      
      // Liste des sorts associés à ce domaine
      spells: new fields.ArrayField(
        new fields.StringField({ required: true })
      ),
      
      // Informations d'acquisition (facultatif)
      acquisition: new fields.SchemaField({
        xpCost: new fields.NumberField({ ...requiredInteger, initial: 20 }),
        prerequisites: new fields.ArrayField(
          new fields.StringField({ required: true })
        )
      })
    });
  }

  /**
   * Préparation des données dérivées pour les domaines magiques.
   * @override
   */
  prepareDerivedData() {
    super.prepareDerivedData();
    
    // Localisation du type de domaine
    this._locType = this.type ? game.i18n.localize(`MYRIAD_SYSTEM.DomainTypes.${this.type}`) : "";
  }

  /**
   * Vérifie si ce domaine est compatible avec un art magique donné.
   * Cette méthode peut être étendue avec des règles de compatibilité spécifiques.
   * @param {MyriadSystemArt} art - L'art magique à vérifier
   * @returns {boolean} true si compatible, false sinon
   */
  isCompatibleWithArt(art) {
    // Par défaut, tous les arts sont compatibles avec tous les domaines
    // Cette méthode peut être étendue avec des règles spécifiques
    return true;
  }

  /**
   * Calcule le coût en points de magie pour utiliser ce domaine avec un art spécifique.
   * @param {number} spellGrade - Le grade du sort (1-5)
   * @param {MyriadSystemArt} art - L'art magique utilisé
   * @returns {number} Le coût en points de magie
   */
  calculateMagicCost(spellGrade, art) {
    // Base : le coût est égal au grade du sort
    let cost = spellGrade || 1;
    
    // Des règles supplémentaires peuvent être ajoutées ici
    
    return Math.max(1, cost);
  }
}
