import MyriadSystemItemBase from "./base-item.mjs";

export default class MyriadSystemSpell extends MyriadSystemItemBase {

  static defineSchema() {
    const fields = foundry.data.fields;
    const requiredInteger = { required: true, nullable: false, integer: true };
    const baseSchema = super.defineSchema();

    // Schéma des sorts conformément à MODELES.md
    const spellSchema = {
      // Coût du sort en points de magie (égal au grade du sort)
      magicPointCost: new fields.NumberField({ initial: 1, min: 1 }),
      
      // Art magique associé au sort
      associatedArt: new fields.StringField({ required: false, nullable: true }),
      
      // Domaine magique associé au sort
      associatedDomain: new fields.StringField({ required: false, nullable: true }),
      
      // Temps d'incantation
      castingTime: new fields.StringField({ initial: "1 action" }),
      
      // Durée de l'effet
      duration: new fields.StringField({ initial: "instantané" }),
      
      // Portée du sort
      range: new fields.StringField({ initial: "personnel" }),
      
      // Zone d'effet
      area: new fields.StringField({ required: false, nullable: true }),
      
      // Effet supplémentaire en cas de réussite critique
      criticalSuccess: new fields.HTMLField(),
      
      // Conséquence en cas d'échec critique
      criticalFailure: new fields.HTMLField(),
      
      // Indique s'il s'agit d'un sort personnalisé créé par le joueur
      isCustom: new fields.BooleanField({ initial: false })
    };

    return foundry.utils.mergeObject(baseSchema, spellSchema);
  }
  
  prepareDerivedData() {
    super.prepareDerivedData();
    
    // Par défaut, le coût en points de magie est égal au grade du sort
    if (!this.magicPointCost) {
      this.magicPointCost = this.grade || 1;
    }
  }
  
  /**
   * Lance le sort avec un jet de dés contextualisé.
   * @param {Object} options - Options pour le jet
   * @returns {Promise} - Promesse résolue avec le résultat du jet
   */
  async cast(options = {}) {
    if (!this.parent) return null;
    
    // Déterminer la caractéristique à utiliser
    // Par défaut, on utilise l'ingéniosité
    let characteristic = "ingeniosite";
    
    // Si le sort a un art associé, on récupère la caractéristique liée à cet art
    if (this.associatedArt) {
      const art = this.parent.items.find(i => 
        i.type === "art" && 
        i.name === this.associatedArt
      );
      if (art?.system?.linkedCharacteristic) {
        characteristic = art.system.linkedCharacteristic;
      }
    }
    
    // Construire les options du jet
    const rollOptions = foundry.utils.mergeObject({
      title: game.i18n.format("MYRIAD_SYSTEM.SpellCast", {name: this.name}),
      characteristic: characteristic,
      isMagical: true,
      spellData: {
        id: this.id,
        grade: this.grade,
        cost: this.magicPointCost
      }
    }, options);
    
    // Effectuer le jet
    return this.parent.rollCharacteristic(characteristic, rollOptions);
  }
}