import { MODULE_ID } from "../main.js";
import { getSetting, setSetting } from "../settings.js";
import { Sortable } from "../lib/sortable.js";
import {SoundLibrary} from "./soundLibrary.js";

let _instance = null;

const searchCache = {};

export class SidebarVideoPlayer extends Application {
    constructor() {
        super();
        ui.sidebarVideoPlayer = this;
        this.YTPlayer = null;
        this.setSyncSettingDebounced = foundry.utils.debounce(this.setSyncSetting, 1500);
        Hooks.on("globalPlaylistVolumeChanged", this.setVolume.bind(this));
        if (game.user.isGM) {
            setInterval(this.setSyncSettingDebounced.bind(this), 10000);
        }
    }

    static get APP_ID() {
        return this.name
            .split(/(?=[A-Z])/)
            .join("-")
            .toLowerCase();
    }

    get APP_ID() {
        return this.constructor.APP_ID;
    }

    static get defaultOptions() {
        return foundry.utils.mergeObject(super.defaultOptions, {
            id: this.APP_ID,
            template: `modules/${MODULE_ID}/templates/${this.APP_ID}.hbs`,
            popOut: false,
            resizable: false,
            minimizable: false,
            width: 400,
            height: 600,
            title: game.i18n.localize(`${MODULE_ID}.${this.APP_ID}.title`),
        });
    }

    async getData() {
        const data = {};
        return { canEdit: game.user.isGM };
    }

    activateListeners(html) {
        super.activateListeners(html);
        html = html[0] ?? html;
        this.videoPlayerContainer = html.querySelector(`.video-player-container`);
        this.playControls = html.querySelector(`.play-controls`);
        this.trackList = html.querySelector(`#track-list`);
        html.querySelectorAll("button").forEach((button) => {
            button.addEventListener("click", this.onButtonClick.bind(this));
        });
        html.querySelectorAll(".playlist-controls i").forEach((icon) => {
            icon.addEventListener("click", this.onPlaylistControlClick.bind(this));
        });
        html.querySelector("select").addEventListener("change", this.onChangePlaylist.bind(this));
        const currentVideo = getSetting("currentVideo");
        html.querySelector("#url").value = currentVideo;
        //if enter is pressed in the playlist name input,rename the current playlist
        html.querySelector("#playlist-name").addEventListener("keypress", (event) => {
            if (event.key === "Enter") {
                const playlists = getSetting("playlists");
                const current = playlists.find((playlist) => playlist.active);
                current.name = event.currentTarget.value;
                setSetting("playlists", playlists);
            }
        });
        html.querySelector("#search-youtube").addEventListener("search", this.onYoutubeSearch.bind(this));
        const moveHandle = html.querySelector(".widget-drag-handle");
        this.moveHandle = moveHandle;
        moveHandle.addEventListener("mousedown", this._onMoveStart.bind(this));
        moveHandle.addEventListener("mouseenter", () => html.classList.toggle("hovering-handle", true));
        moveHandle.addEventListener("mouseleave", () => html.classList.toggle("hovering-handle", false));
        if (currentVideo) this.loadVideo(currentVideo);
        this.renderPlaylists();
        const position = getSetting("windowPosition");
        this.append();
        if (position.x !== 0 && position.y !== 0) {
            this.element[0].style.opacity = 0;
            setTimeout(() => {
                this._initial = { bottom: position.y, left: position.x };
                this._onMoveDetach();
                this.element[0].style.opacity = 1;
            }, 100);
        }
        this._initContextMenu();
    }

    append() {
        document.querySelector("#controls").after(this.element[0]);
        this.element[0].classList.add("tyw-docked");
        this.isDocked = true;
    }

    onButtonClick(event) {
        event.preventDefault();
        const url = this.playControls.querySelector("#url").value;
        const button = event.currentTarget;
        const dataAction = button.dataset.action;
        if (dataAction === "play") {
            setSetting("currentVideo", url);
        }
        if (dataAction === "stop") {
            setSetting("currentVideo", "");
            const playlists = getSetting("playlists");
            playlists.forEach((playlist) => {
                playlist.tracks.forEach((track) => (track.active = false));
            });
            setSetting("playlists", playlists);
        }
        if (dataAction === "playlist") {
            this.element[0].querySelector(".playlist-section").classList.toggle("show-playlist");
            setTimeout(() => {
                this.renderPlaylists();
            }, 300);
        }
        if (dataAction === "add-to-playlist") {
            this.addTrackToPlaylist(url);
        }
        if (dataAction === "search") {
            const searchContainer = this.element[0].querySelector(".search-container");
            searchContainer.classList.toggle("expanded");
            if (searchContainer.classList.contains("expanded")) {
                this.element[0].querySelector("#search-youtube").focus();
            }
        }
        if (dataAction === "search-youtube") {
            this.onYoutubeSearch(event);
        }
        if(dataAction === "sound-library"){
            SoundLibrary.open();
        }
    }

    async onYoutubeSearch(event) {
        const query = this.element[0].querySelector("#search-youtube").value;
        if (!query) return;
        const results = searchCache[query] ?? (await this.searchVideos(query));
        searchCache[query] = results;
        const resultsContainer = this.element[0].querySelector(".search-results");
        resultsContainer.innerHTML = "";
        results.forEach((result) => {
            const li = document.createElement("li");
            li.innerHTML = `
            <div class="video-image" style="background-image:url('${result.snippet.thumbnails.default.url}')">
            <span class="duration">${SidebarVideoPlayer.parseDuration(result.duration)}</span>
            </div>
            <article>
            <span class="title">${result.snippet.title}</span>
            <span class="description">${result.snippet.description}</span>
            </article>
            `;
            li.dataset.url = result.id.videoId;
            li.addEventListener("click", (event) => {
                const videoUrl = `https://www.youtube.com/watch?v=${event.currentTarget.dataset.url}`;
                setSetting("currentVideo", videoUrl);
                this.playControls.querySelector("#url").value = videoUrl;
            });
            resultsContainer.appendChild(li);
        });
    }

    async onPlaylistControlClick(event) {
        event.preventDefault();
        const action = event.currentTarget.dataset.action;
        if (action === "add") {
            const playlistName = this.element[0].querySelector("#playlist-name").value;
            //check if the playlist name is unique
            const playlists = getSetting("playlists");
            if (playlists.find((p) => p.name === playlistName)) {
                return;
            }
            playlists.forEach((playlist) => (playlist.active = false));
            playlists.push({ name: playlistName, tracks: [], active: true, id: foundry.utils.randomID() });
            setSetting("playlists", playlists);
        }
        if (action === "remove") {
            const confirmed = await this.confirmDelete(`${MODULE_ID}.deletePlaylist`, `${MODULE_ID}.deletePlaylistContent`);
            if (!confirmed) return;
            const playlists = getSetting("playlists");
            const selected = this.element[0].querySelector("select").value;
            const current = playlists.find((playlist) => playlist.id === selected);
            playlists.splice(playlists.indexOf(current), 1);
            //remove option from the select and set the first playlist as active
            if (playlists.length > 0) {
                this.element[0].querySelector("select").value = playlists[0].id;
                playlists[0].active = true;
            }
            await setSetting("playlists", playlists);

            this.renderPlaylists();
        }
        if (action === "toggle") {
            const playlists = getSetting("playlists");
            const current = playlists.find((playlist) => playlist.active);
            current.autoplay = !(current.autoplay ?? true);
            setSetting("playlists", playlists);
        }
    }

    _onMoveStart(event) {
        this.moveListener = this._onMove.bind(this);
        this.moveEndListener = this._onMoveEnd.bind(this);
        document.addEventListener("mousemove", this.moveListener);
        document.addEventListener("mouseup", this.moveEndListener);
        // Record initial position
        const offsetX = this.isDocked ? 10 : 0;
        const offsetY = this.isDocked ? -15 : -15;
        this._initial = {
            x: event.clientX,
            y: event.clientY,
            left: this.element[0].offsetLeft + offsetY,
            bottom: window.innerHeight - this.element[0].offsetTop - this.element[0].offsetHeight - offsetX - 10,
        };
        this._onMoveDetach();
    }

    _onMoveDetach() {
        if (!this.isDocked) return;
        this.element[0].style.position = "absolute";
        this.element[0].style.bottom = `${this._initial.bottom}px`;
        this.element[0].style.left = `${this._initial.left}px`;
        this.element[0].style.zIndex = 100;
        this.element[0].classList.remove("tyw-docked");
        document.body.appendChild(this.element[0]);
        this.isDocked = false;
    }

    _onMoveAttach() {
        if (this.isDocked) return;
        this.element[0].style.position = null;
        this.element[0].style.bottom = null;
        this.element[0].style.left = null;
        this.element[0].style.zIndex = null;
        this.append();
    }

    _onMove(event) {
        event.preventDefault();
        event.stopPropagation();
        const dx = event.clientX - this._initial.x;
        const dy = event.clientY - this._initial.y;
        this.current = { x: this._initial.left + dx, y: this._initial.bottom - dy };
        const minLeft = 10;
        const maxLeft = window.innerWidth - this.element[0].offsetWidth;
        const minBottom = 10;
        const maxBottom = window.innerHeight - this.element[0].offsetHeight;
        this.current.x = SidebarVideoPlayer.clamp(this.current.x, minLeft, maxLeft);
        this.current.y = SidebarVideoPlayer.clamp(this.current.y, minBottom, maxBottom);
        this.element[0].style.left = `${this.current.x}px`;
        this.element[0].style.bottom = `${this.current.y}px`;
    }

    _onMoveEnd(event) {
        document.removeEventListener("mousemove", this.moveListener);
        document.removeEventListener("mouseup", this.moveEndListener);
        if (event.clientX < 300 && event.clientY > window.innerHeight - 300) {
            this._onMoveAttach();
            setSetting("windowPosition", { x: 0, y: 0 });
        } else {
            setSetting("windowPosition", this.current);
        }
    }

    async loadVideo(url) {
        url = url || getSetting("currentVideo");
        if (url.includes("list")) {
            ui.notifications.error("You cannot play a playlist directly, please add the playlist to a playlist first.");
            url = "";
            return;
        }
        if (this.YTPlayer) {
            this.YTPlayer.stopVideo();
            this.YTPlayer.destroy();
        }
        if (!url) {
            //stop the video
            this.YTPlayer = null;
            this.videoId = null;
            this.videoPlayerContainer.innerHTML = "";
            this.element[0].classList.remove("has-video");
            if (game.user === game.users.activeGM) setSetting("currentTime", { videoId: "", time: 0 });
            return;
        }
        //load a youtube video to auto play with sound on inside the videoPlayerContainer

        let videoId = await this.getVideoId(url);
        //use youtube iframe api to load the video
        const player = new YT.Player(this.videoPlayerContainer, {
            height: "100%",
            width: "100%",
            videoId: videoId,
            host: "https://www.youtube-nocookie.com",
            playerVars: {
                showinfo: 0,
                modestbranding: 1,
            },
            events: {
                onReady: (event) => {
                    const currentTime = getSetting("currentTime");
                    if (currentTime.videoId === videoId) {
                        event.target.seekTo(currentTime.time, true);
                    }
                    this.setVolume();
                    if(currentTime.paused) event.target.pauseVideo();
                    if (currentTime.videoId === videoId && currentTime.paused) return;
                    event.target.playVideo();
                },
                onStateChange: (event) => {
                    if (event.data === YT.PlayerState.ENDED) {
                        this.advancePlaylist();
                    }
                    if (event.data === YT.PlayerState.PLAYING) {
                        this.setSyncSettingDebounced();
                    }
                    if (event.data === YT.PlayerState.PAUSED) {
                        this.setSyncSetting();
                    }
                },
            },
        });
        player.addEventListener("onVolumeChange", (e) => {
            if (this._volumeSetProgrammatically) {
                this._volumeSetProgrammatically = false;
                return;
            }
            const vol = foundry.audio.AudioHelper.inputToVolume(e.target.getVolume() / 100);
            game.settings.set("core", "globalPlaylistVolume", vol);
            document.querySelector(`input[name="globalPlaylistVolume"]`).value = foundry.audio.AudioHelper.volumeToInput(vol);
        });
        this.element[0].classList.add("has-video");
        this.YTPlayer = player;
        this.videoId = videoId;
    }

    async getVideoId(url) {
        // Determine if the URL is a playlist
        if (url.includes("list=")) {
            const playlistId = url.match(/list=([a-zA-Z0-9_-]+)/)[1];
            return await this.fetchPlaylist(playlistId);
        }

        // Regular expression to match YouTube video ID
        var regExp = /^.*(youtu\.be\/|v\/|u\/\w\/|embed\/|watch\?v=|\&v=)([^#\&\?]*).*/;
        var match = url.match(regExp);
        if (match && match[2].length === 11) {
            var videoId = match[2];
            var embedUrl = videoId;
            return embedUrl;
        } else {
            return url;
        }
    }

    syncTime() {
        const currentTime = getSetting("currentTime");
        if (!currentTime.videoId) return;
        if (!this.YTPlayer || !this.YTPlayer.getCurrentTime) return;
        if (this.YTPlayer.getVideoData().video_id !== currentTime.videoId) return;
        //set paused state
        if (currentTime.paused) {
            this.YTPlayer.pauseVideo();
        } else {
            this.YTPlayer.playVideo();
        }
        const deltaTime = Math.abs(this.YTPlayer.getCurrentTime() - currentTime.time);
        if (Math.abs(deltaTime) < 3) return;
        this.YTPlayer.seekTo(currentTime.time, true);
    }

    setSyncSetting() {
        if (!this.YTPlayer || game.user !== game.users.activeGM || !this.YTPlayer.getCurrentTime) return;
        const currentVideoTime = this.YTPlayer.getCurrentTime();
        const videoId = this.YTPlayer.getVideoData().video_id;
        setSetting("currentTime", { videoId, time: currentVideoTime, paused: this.YTPlayer.getPlayerState() === 2 });
    }

    setVolume() {
        this._volumeSetProgrammatically = true;
        const volume = foundry.audio.AudioHelper.volumeToInput(game.settings.get("core", "globalPlaylistVolume"));
        if (this.YTPlayer) {
            this.YTPlayer.setVolume(parseFloat(volume) * 100);
        }
    }

    advancePlaylist() {
        if (game.user !== game.users.activeGM) return;
        const playlists = getSetting("playlists");
        const current = playlists.find((playlist) => playlist.active);
        if (!current) return;
        const autoplay = current.autoplay ?? true;
        const currentIndex = current.tracks.findIndex((track) => track.active);
        if (currentIndex === -1) return;
        const isRepeat = current.tracks[currentIndex].repeat;
        if (isRepeat) {
            this.YTPlayer.seekTo(0, true);
            this.YTPlayer.playVideo();
            this.setSyncSetting();
            return;
        }
        if (!autoplay) {
            //stop the video
            setSetting("currentVideo", "");
            return;
        }
        const nextIndex = (currentIndex + 1) % current.tracks.length;
        current.tracks.forEach((track) => (track.active = false));
        current.tracks[nextIndex].active = true;
        setSetting("playlists", playlists);
        setSetting("currentVideo", current.tracks[nextIndex].url);
    }

    renderPlaylists() {
        const playlists = getSetting("playlists");
        const current = playlists.find((playlist) => playlist.active);
        if (!current) return;
        const toggleEl = this.element[0].querySelector(".playlist-controls i[data-action='toggle']");
        const autoplay = current.autoplay ?? true;
        toggleEl.classList.toggle("fa-toggle-on", autoplay);
        toggleEl.classList.toggle("fa-toggle-off", !autoplay);
        this.trackList.innerHTML = "";
        current.tracks.forEach((track) => {
            const li = document.createElement("li");
            if (track.active) li.classList.add("active");
            li.innerHTML = `<span data-tooltip="${track.title}">${track.title}</span><i class="fas fa-repeat ${track.repeat ? "repeat" : ""}" data-action="loop" data-tooltip="fvtt-youtube-player.TOOLTIP.loop"></i><i class="fas fa-trash" data-action="delete-track" data-tooltip="fvtt-youtube-player.TOOLTIP.delete-track"></i>`;
            li.querySelector("i[data-action='delete-track']").addEventListener("click", async (e) => {
                const isShift = e.shiftKey;
                const confirmed = isShift || (await this.confirmDelete(`${MODULE_ID}.deleteTrack`, `${MODULE_ID}.deleteTrackContent`));
                if (!confirmed) return;
                const setting = getSetting("playlists");
                const playlist = setting.find((playlist) => playlist.active);
                playlist.tracks = playlist.tracks.filter((t) => t.url !== track.url);
                setSetting("playlists", setting);
            });
            li.querySelector("i[data-action='loop']").addEventListener("click", () => {
                const playlists = getSetting("playlists");
                const playlist = playlists.find((playlist) => playlist.active);
                const t = playlist.tracks.find((t) => t.url === track.url);
                t.repeat = !t.repeat;
                setSetting("playlists", playlists);
            });
            li.querySelector("span").addEventListener("click", () => {
                setSetting("currentVideo", track.url);
                const setting = getSetting("playlists");
                //set all tracks to inactive
                setting.forEach((playlist) => {
                    playlist.tracks.forEach((t) => (t.active = false));
                });
                const playlist = setting.find((playlist) => playlist.active);
                const currentTrack = playlist.tracks.find((t) => t.url === track.url);
                currentTrack.active = true;
                setSetting("playlists", setting);
            });
            //rename on context menu
            li.addEventListener("contextmenu", (event) => {
                event.preventDefault();
                Dialog.prompt({
                    title: "Rename track",
                    content: "<input type='text' id='track-name' value='" + track.title + "'>",
                    label: game.i18n.localize("Save"),
                    callback: (html) => {
                        const newName = html[0].querySelector("#track-name").value;
                        const ps = getSetting("playlists");
                        const p = ps.find((playlist) => playlist.active);
                        const t = p.tracks.find((t) => t.url === track.url);
                        t.title = newName;
                        setSetting("playlists", ps);
                    },
                });
            });
            this.trackList.appendChild(li);
        });
        Sortable.create(this.trackList, {
            onEnd: (event) => {
                const setting = getSetting("playlists");
                const playlist = setting.find((playlist) => playlist.active);
                const track = playlist.tracks.splice(event.oldIndex, 1)[0];
                playlist.tracks.splice(event.newIndex, 0, track);
                setSetting("playlists", setting);
            },
        });
        const select = this.element[0].querySelector("select");
        select.innerHTML = "";
        //sort the playlists by name
        playlists.sort((a, b) => a.name.localeCompare(b.name));
        playlists.forEach((playlist) => {
            const option = document.createElement("option");
            option.value = playlist.id;
            option.innerHTML = playlist.name;
            if (playlist.active) {
                this.element[0].querySelector("#playlist-name").value = playlist.name;
                option.selected = true;
            }
            select.appendChild(option);
        });
    }

    onChangePlaylist(event) {
        const playlists = getSetting("playlists");
        const select = event.currentTarget;
        const playlistId = select.value;
        playlists.forEach((playlist) => {
            playlist.active = playlist.id === playlistId;
        });
        setSetting("playlists", playlists);
    }

    async processUrl(url) {
        const pUrl = await this.getVideoId(url);
        const videoIds = Array.isArray(pUrl) ? pUrl : [pUrl];
        const results = [];
        for (const videoId of videoIds) {
            if (videoId.title) {
                results.push({ id: videoId.id, title: videoId.title });
                continue;
            }
            const title = await this.getVideoTitle(videoId);
            results.push({ id: videoId, title: title });
        }
        return results;
    }

    async addTrackToPlaylist(url) {
        const playlists = getSetting("playlists");
        let current = playlists.find((playlist) => playlist.active);
        const videoId = await this.getVideoId(url);
        if (Array.isArray(videoId)) {
            const playlistName = await this.getPlaylistNameDialog();
            if (playlistName) {
                playlists.forEach((playlist) => (playlist.active = false));
                current = { name: playlistName, tracks: [], active: true, id: foundry.utils.randomID() };
                playlists.push(current);
            }
        }
        if (!current) {
            playlists.forEach((playlist) => (playlist.active = false));
            current = { name: "Playlist", tracks: [], active: true, id: foundry.utils.randomID() };
            playlists.push(current);
        }
        if (Array.isArray(videoId)) {
            for (const videoData of videoId) {
                current.tracks.push({ url: videoData.id, title: videoData.title, active: false });
            }
        } else {
            const title = await this.getVideoTitle(videoId);
            current.tracks.push({ url, title: title, active: false });
        }
        setSetting("playlists", playlists);
    }

    async confirmDelete(title, content) {
        const confirmed = await Dialog.confirm({
            title: game.i18n.localize(title),
            content: game.i18n.localize(content),
            yes: () => {
                return true;
            },
            no: () => {
                return false;
            },
        });
        return confirmed;
    }

    async getVideoTitle(videoId) {
        ui.notifications.info(`${MODULE_ID}.fetchingTitle`, { localize: true });
        return new Promise((resolve, reject) => {
            const el = document.createElement("div");
            el.style.position = "absolute";
            el.style.width = "200px";
            el.style.height = "200px";
            el.style.zIndex = -1;
            document.body.appendChild(el);
            const player = new YT.Player(el, {
                height: "100%",
                width: "100%",
                videoId: videoId,
                host: "https://www.youtube-nocookie.com",
                events: {
                    onReady: (event) => {
                        resolve(event.target.getVideoData().title);
                        player.destroy();
                    },
                },
            });
        });
    }

    async fetchPlaylist(playlistId) {
        const apiKey = getSetting("googleApiKey");
        if (!apiKey) {
            const res = await fetch("https://theripper93.com/api/youtube?id=" + playlistId);
            const data = await res.json();
            return data.data;
        }
        let allItems = [];
        let nextPageToken = "";
        do {
            let url = `https://www.googleapis.com/youtube/v3/playlistItems?part=snippet&playlistId=${playlistId}&key=${apiKey}&maxResults=50&pageToken=${nextPageToken}`;
            const response = await fetch(url);
            const data = await response.json();
            allItems = allItems.concat(data.items);
            nextPageToken = data.nextPageToken;
        } while (nextPageToken);

        return allItems.map((item) => {
            return { id: item.snippet.resourceId.videoId, title: item.snippet.title };
        });
    }

    async getPlaylistNameDialog() {
        const res = await Dialog.prompt({
            title: game.i18n.localize(`${MODULE_ID}.playlistName`),
            content: `<p>${game.i18n.localize(`${MODULE_ID}.playlistNameContent`)}</p><input type='text' id='track-name' placeholder='${game.i18n.localize(`${MODULE_ID}.playlistName`)}'>`,
            label: game.i18n.localize("Yes"),
            callback: (html) => {
                const newName = html[0].querySelector("#track-name").value;
                return newName;
            },
            rejectClose: false,
        });
        return res;
    }

    async searchVideos(query) {
        if (!query) return [];
        //encode the query
        query = encodeURIComponent(query);
        const apiKey = getSetting("googleApiKey");
        if (!apiKey) {
            const msg = game.i18n.localize(`${MODULE_ID}.noApiKey`) + " <a href='https://wiki.theripper93.com/premium/fvtt-youtube-player#using-your-own-youtube-api-key' target='_blank'>Getting an API Key</a>";
            ui.notifications.warn(msg, { permanent: true });
            return [];
        }
        const url = `https://www.googleapis.com/youtube/v3/search?part=snippet&maxResults=10&q=${query}&type=video&key=${apiKey}`;
        try {
            const res = await fetch(url);
            const data = await res.json();
            //fetch video durations next
            const videoIds = data.items.map((item) => item.id.videoId).join(",");
            const durationRes = await fetch(`https://www.googleapis.com/youtube/v3/videos?part=contentDetails&id=${videoIds}&key=${apiKey}`);
            const durationData = await durationRes.json();
            data.items.forEach((item, index) => {
                item.duration = durationData.items[index].contentDetails.duration;
            });
            return data.items;
        } catch (error) {
            console.error(error);
            return [];
        }
    }

    _initContextMenu() {
        const contextMenu = new ContextMenu(this.element[0], ".search-results li", [
            {
                name: `${MODULE_ID}.TOOLTIP.add-to-playlist`,
                icon: '<i class="fas fa-plus"></i>',
                callback: (li) => {
                    li = li[0] ?? li;
                    const url = `https://www.youtube.com/watch?v=${li.dataset.url}`;
                    this.addTrackToPlaylist(url);
                },
            },
            {
                name: `${MODULE_ID}.TOOLTIP.copy-url`,
                icon: '<i class="fas fa-copy"></i>',
                callback: (li) => {
                    li = li[0] ?? li;
                    const url = `https://www.youtube.com/watch?v=${li.dataset.url}`;
                    navigator.clipboard.writeText(url);
                },
            },
        ]);
    }

    static inject() {
        if (_instance) return _instance.append();
        _instance = new this();
        _instance.render(true);
    }

    static getInstance() {
        return _instance;
    }

    static _onPreRender() {
        if (!_instance) return;
        _instance._onPreRender();
    }

    static parseDuration(duration) {
        const match = duration.match(/PT(\d+H)?(\d+M)?(\d+S)?/);

        if (!match) return "";

        const hours = parseInt(match[1]) || 0;
        let minutes = parseInt(match[2]) || 0;
        let seconds = parseInt(match[3]) || 0;

        //append a leading zero if the number is less than 10
        minutes = minutes < 10 ? `0${minutes}` : `${minutes}`;
        seconds = seconds < 10 ? `0${seconds}` : `${seconds}`;

        return (hours ? hours + ":" : "") + (minutes ? minutes + ":" : "") + (seconds ? seconds + "" : "");
    }

    static clamp(num, min, max) {
        return num <= min ? min : num >= max ? max : num;
    }
}
