import { MODULE_ID } from "../main.js";
import { getSetting, setSetting } from "../settings.js";

export class ImportExport extends FormApplication {
    constructor() {
        super();
        this._importedData = null;
    }

    static get APP_ID() {
        return this.name
            .split(/(?=[A-Z])/)
            .join("-")
            .toLowerCase();
    }

    get APP_ID() {
        return this.constructor.APP_ID;
    }

    static get SETTING_KEY() {
        return this.name.charAt(0).toLowerCase() + this.name.slice(1);
    }

    get SETTING_KEY() {
        return this.constructor.SETTING_KEY;
    }

    static get defaultOptions() {
        return foundry.utils.mergeObject(super.defaultOptions, {
            id: this.APP_ID,
            template: `modules/${MODULE_ID}/templates/${this.APP_ID}.hbs`,
            popOut: true,
            minimizable: true,
            title: game.i18n.localize(`${MODULE_ID}.${this.APP_ID}.title`),
            closeOnSubmit: true,
            width: 600,
        });
    }

    async getData() {
        const playlists = getSetting("playlists");
        playlists.sort((a, b) => a.name.localeCompare(b.name));
        return { playlists, importedPlaylists: this._importedData };
    }

    activateListeners(html) {
        super.activateListeners(html);
        html = html[0] ?? html;
        html.querySelectorAll("button").forEach((button) => {
            button.addEventListener("click", this._onButtonClick.bind(this));
        });
        html.querySelector("input[type=file]").addEventListener("change", this._onFileChange.bind(this));
        this.setPosition({ height: "auto", width: 600 });
    }

    async _onButtonClick(event) {
        event.preventDefault();
        const action = event.target.dataset.action;
        if (action === "import") {
            const playlists = getSetting("playlists");
            const importItems = this.element[0].querySelectorAll("#imported-playlists li input:checked");
            const imported = Array.from(importItems).map((item) => item.value);
            const importedPlaylists = this._importedData.filter((playlist) => imported.includes(playlist.id));
            //regenerate ids
            importedPlaylists.forEach((playlist) => {
                playlist.id = foundry.utils.randomID();
            });
            playlists.push(...importedPlaylists);
            setSetting("playlists", playlists).then(() => {
                this.render(true);
                ui.notifications.info(game.i18n.localize(`${MODULE_ID}.import-export.imported`));
            });
        } else if (action === "export") {
            const exportItems = this.element[0].querySelectorAll("#exported-playlists li input:checked");
            const ids = Array.from(exportItems).map((item) => item.value);
            const playlists = getSetting("playlists");
            const exported = playlists.filter((playlist) => ids.includes(playlist.id));
            const data = JSON.stringify(exported);
            saveDataToFile(data, "application/json", "playlists.json");
        }
    }

    _onFileChange(event) {
        const file = event.target.files[0];
        if (!file) return;
        const reader = new FileReader();
        reader.onload = (event) => {
            const data = JSON.parse(event.target.result);
            this._importedData = data;
            this.render(true);
        };
        reader.readAsText(file);
    }

    async _updateObject(event, formData) {}

    static register() {
        game.settings.registerMenu(MODULE_ID, this.APP_ID, {
            name: game.i18n.localize(`${MODULE_ID}.settings.${this.SETTING_KEY}.name`),
            label: game.i18n.localize(`${MODULE_ID}.settings.${this.SETTING_KEY}.label`),
            hint: game.i18n.localize(`${MODULE_ID}.settings.${this.SETTING_KEY}.hint`),
            icon: "fas fa-cogs",
            type: this,
            restricted: true,
            scope: "world",
        });
    }
}
