import { DEFAULT_CELL_CONFIG, getCellData, HexplorerApp } from "../app/HexplorerApp.js";
import { Socket } from "../lib/socket.js";
import { l, mergeClone, mergeObject } from "../lib/utils.js";
import { MODULE_ID } from "../main.js";
import { Fog } from "./Fog.js";
import { GridUtils } from "./GridUtils.js";
import { HexplorerHighlightLayer } from "./HexplorerHighlightLayer.js";

export class Hexplorer {
    constructor(scene) {
        this.scene = scene;
        canvas.hexplorer = this;
        this.fog = new Fog(scene, this);
        this.highlightLayer = game.user.isGM ? new HexplorerHighlightLayer(scene) : null;
        this.setHooks();
        this.activateCanvasListeners();
        this._onCanvasMove = foundry.utils.debounce(this._onCanvasMove.bind(this), 200);
    }

    #app;

    get app() {
        return this.#app;
    }

    set app(app) {
        this.#app = app;
        this.highlightLayer?.toggle(!!app);
    }

    activateCanvasListeners() {
        this.boundOnCanvasMove = this._onCanvasMove.bind(this);
        this.boundOnCanvasClick = this._onCanvasClick.bind(this);
        document.querySelector("#board").addEventListener("mousemove", this.boundOnCanvasMove);
        document.querySelector("#board").addEventListener("click", this.boundOnCanvasClick);
    }

    deactivateCanvasListeners() {
        document.querySelector("#board").removeEventListener("mousemove", this.boundOnCanvasMove);
        document.querySelector("#board").removeEventListener("click", this.boundOnCanvasClick);
    }

    _onCanvasClick(event) {
        const isShift = event.shiftKey;
        if (!isShift) return;
        const cell = canvas.grid.getOffset(canvas.mousePosition);
        const cellData = getCellData(cell);
        if (!cellData.journalEntry) return;
        const journalEntry = fromUuidSync(cellData.journalEntry);
        if (!journalEntry.testUserPermission(game.user, CONST.DOCUMENT_OWNERSHIP_LEVELS.OBSERVER)) return;
        journalEntry.parent.sheet.render(true, { pageId: journalEntry.id });
    }

    async _onCanvasMove(event) {
        const cell = canvas.grid.getOffset(canvas.mousePosition);
        if (!this._lasCell) {
            this._lasCell = cell;
            return;
        }
        if (cell.i === this._lasCell.i && cell.j === this._lasCell.j) {
            if (this.tooltipAnchor && this.tooltipAnchor.dataset.i === cell.i && this.tooltipAnchor.dataset.j === cell.j) return;
            if (this.tooltipAnchor) {
                this.tooltipAnchor.remove();
                this.tooltipAnchor = null;
            }
            const cellData = getCellData(cell);
            const tooltipUUID = cellData.tooltip ?? "";
            const tooltipDocument = fromUuidSync(tooltipUUID);
            const allowTooltip = tooltipDocument ? tooltipDocument.testUserPermission(game.user, CONST.DOCUMENT_OWNERSHIP_LEVELS.OBSERVER) : false;
            if (!allowTooltip || !tooltipUUID || !tooltipDocument || !tooltipDocument.text.content) {
                this._lasCell = cell;
                return;
            }
            const topLeft = canvas.grid.getTopLeftPoint(cell);
            const width = canvas.grid.sizeX;
            const height = canvas.grid.sizeY;
            const tempEl = document.createElement("div");
            tempEl.dataset.i = cell.i;
            tempEl.dataset.j = cell.j;
            tempEl.style.position = "absolute";
            tempEl.style.top = `${topLeft.y}px`;
            tempEl.style.left = `${topLeft.x}px`;
            tempEl.style.width = `${width}px`;
            tempEl.style.height = `${height}px`;
            tempEl.style.backgroundColor = "#00000000";
            document.querySelector("#hud").appendChild(tempEl);
            const enriched = await TextEditor.enrichHTML(tooltipDocument.text.content, { secrets: tooltipDocument.isOwner });
            game.tooltip.activate(tempEl, { text: enriched, cssClass: "hexplorer-tooltip", direction: TooltipManager.TOOLTIP_DIRECTIONS.UP });
            this.tooltipAnchor = tempEl;
        } else {
            this.tooltipAnchor?.remove();
            if (game.tooltip.tooltip.classList.contains("hexplorer-tooltip")) {
                game.tooltip.deactivate();
            }
            this.tooltipAnchor = null;
        }
        this._lasCell = cell;
    }

    setHooks() {
        Hooks.on("preUpdateToken", Hexplorer._onPreUpdateToken);
        Hooks.on("updateToken", Hexplorer._onUpdateToken);
        Hooks.on("updateScene", Hexplorer._onUpdateScene);
    }

    unsetHooks() {
        Hooks.off("preUpdateToken", Hexplorer._onPreUpdateToken);
        Hooks.off("updateToken", Hexplorer._onUpdateToken);
        Hooks.off("updateScene", Hexplorer._onUpdateScene);
    }

    async requestHexplorerUpdate(tokenDocument, destination) {
        const rulerSegments = canvas.controls.ruler.segments.filter((s) => s.distance > 0).map((s) => s.ray);
        const segments = rulerSegments.length ? rulerSegments : [new Ray(GridUtils.offsetToCenter(tokenDocument, tokenDocument), GridUtils.offsetToCenter(destination, tokenDocument))];
        const offsetRays = segments;

        const pointsOnPath = offsetRays.reduce((acc, ray) => {
            const points = canvas.grid.getDirectPath([ray.A, ray.B]);
            if (offsetRays.indexOf(ray) === offsetRays.length - 1) points.shift();
            return acc.concat(points);
        }, []);

        if (!pointsOnPath.length) return;
        let totalTime = 0;
        const cells = pointsOnPath.map((p) => getCellData(p));
        let messageContent = "<div class='hexplorer-request-message'><h2>Hexplorer Request</h2><hr>";
        for (const cell of cells) {
            const index = cells.indexOf(cell);
            const point = pointsOnPath[index];
            messageContent += `<h3>${cell.region} (${point.i},${point.j})</h3>`;
            messageContent += `<table><thead><tr><th>Action</th><th>Value</th></tr></thead><tbody>`;
            messageContent += `<tr><td>${l("hexplorer.request.move")}</td><td>${tokenDocument.name}</td></tr>`;
            if (!cell.revealed) {
                messageContent += `<tr><td>${l("hexplorer.request.explore")}</td><td>${point.i},${point.j}</td></tr>`;
            }
            if (cell.journalEntry) {
                const journalEntry = await fromUuid(cell.journalEntry);
                messageContent += `<tr><td>${l("hexplorer.request.journal-entry")}</td><td>@UUID[${journalEntry.uuid}]</td></tr>`;
            }
            if (cell.reveal) {
                messageContent += `<tr><td>${l("hexplorer.request.reveal")}</td><td>${cell.reveal}</td></tr>`;
            }
            const advanceTime = Hexplorer.getAdvanceTime(cell, tokenDocument);
            if (advanceTime) {
                totalTime += advanceTime;
                messageContent += `<tr><td>${l("hexplorer.request.advance-time")}</td><td>${advanceTime.toFixed(1)}h</td></tr>`;
            }
            if (cell.rollTables) {
                for (const uuid of cell.rollTables) {
                    const table = await fromUuid(uuid);
                    messageContent += `<tr><td>${l("hexplorer.request.roll-tables")}</td><td>@UUID[${table.uuid}]</td></tr>`;
                }
            }
            //end table
            messageContent += `</tbody></table>`;
        }
        messageContent += `</div>`;
        if (totalTime) {
            messageContent += `<h1 style="border: none;">
            <strong>${l("hexplorer.request.total-time")}</strong>: ${totalTime.toFixed(1)}h
            </h1>`;
        }
        messageContent += `<button class="hexplorer-request-button" data-destination-x="${destination.x}" data-destination-y="${destination.y}" data-uuid="${tokenDocument.uuid}"><i class="fas fa-badge-check"></i> Accept</button>`;

        await ChatMessage.create({
            user: game.user,
            speaker: {
                actor: tokenDocument.actor,
                token: tokenDocument,
                alias: tokenDocument.name,
            },
            content: l("hexplorer.request.player-message"),
        });

        ChatMessage.create({
            user: game.user,
            speaker: {
                actor: tokenDocument.actor,
                token: tokenDocument,
                alias: tokenDocument.name,
            },
            flags: {
                [MODULE_ID]: {
                    points: pointsOnPath.map((p) => ({ i: p.i, j: p.j })),
                },
            },
            whisper: game.users.filter((u) => u.isGM).map((u) => u.id),
            blind: true,
            content: messageContent,
        });
    }

    destroy() {
        delete canvas.hexplorer;
        this.unsetHooks();
        this.deactivateCanvasListeners();
        this.fog.destroy();
        this.highlightLayer?.destroy();
    }

    async revealCell(cell, scene, tokenDocument) {
        scene ??= canvas.scene;
        if (scene !== canvas.scene) return ui.notifications.warn(`${MODULE_ID}.request.not-in-scene`);

        const cellData = getCellData(cell);
        const exploreActivities = cellData.exploreActivities;

        // Reveal Hidden
        if (exploreActivities.removeHidden.value.length) {
            const removeHidden = exploreActivities.removeHidden.value.map((uuid) => fromUuidSync(uuid));
            for (const document of removeHidden) {
                await document.update({ hidden: false });
            }
        }

        // Macros
        if (exploreActivities.macros.value.length) {
            for (const uuid of exploreActivities.macros.value) {
                const macro = await fromUuid(uuid);
                await macro.execute({ cell: { ...cell, ...cellData }, token: tokenDocument });
            }
        }

        // Reveal Tooltip
        if (exploreActivities.revealTooltip) {
            const tooltip = await fromUuid(cellData.tooltip);
            if (tooltip) {
                await tooltip.update({ "ownership.default": CONST.DOCUMENT_OWNERSHIP_LEVELS.OBSERVER });
            }
        }

        // Reveal Journal Entry
        if (exploreActivities.revealJournalEntry) {
            const journalEntry = await fromUuid(cellData.journalEntry);
            if (journalEntry) {
                await journalEntry.update({ "ownership.default": CONST.DOCUMENT_OWNERSHIP_LEVELS.OBSERVER });
            }
        }

        // Reveal Map Notes
        if (exploreActivities.revealMapNotes) {
            const topLeft = canvas.grid.getTopLeftPoint(cell);
            const quad = new PIXI.Rectangle(topLeft.x, topLeft.y, topLeft.x + canvas.grid.sizeX, topLeft.y + canvas.grid.sizeY);
            const notes = Array.from(canvas.notes.quadtree.getObjects(quad)).filter((n) => n.page);
            for (const note of notes) {
                await note.page.update({ "ownership.default": CONST.DOCUMENT_OWNERSHIP_LEVELS.OBSERVER });
            }
        }
    }

    async moveIntoCell(cell, scene, tokenDocument) {
        scene ??= this.scene ?? canvas.scene;
        if (scene !== canvas.scene) return ui.notifications.warn(`${MODULE_ID}.request.not-in-scene`);
        const cellData = getCellData(cell);

        if (tokenDocument) {
            const centerCoords = canvas.grid.getCenterPoint(cell);
            const currentTokenRegion = tokenDocument.getFlag(MODULE_ID, "region") ?? "";

            // Display region change
            if (cellData.region && currentTokenRegion !== cellData.region) {
                await tokenDocument.setFlag(MODULE_ID, "region", cellData.region);
                Socket.createScrollingText({ origin: centerCoords, content: cellData.region });
                const discovered = scene.getFlag(MODULE_ID, "discovered") ?? [];
                if (!discovered.includes(cellData.region)) {
                    discovered.push(cellData.region);
                    await scene.setFlag(MODULE_ID, "discovered", discovered);
                    Socket.showBannerText({ text: cellData.region });
                }
            }
            const advanceTime = Hexplorer.getAdvanceTime(cellData, tokenDocument);
            // Advance time
            if (advanceTime) {
                await game.time.advance(advanceTime * 3600);
            }
        }
    }

    async exploreCell(cell, scene, tokenDocument, force = false) {
        scene ??= this.scene ?? canvas.scene;
        if (scene !== canvas.scene) return ui.notifications.warn(`${MODULE_ID}.request.not-in-scene`);
        const key = GridUtils.getKey(cell);
        const exploration = scene.flags[MODULE_ID]?.exploration ?? {};
        exploration[key] ??= {};
        const firstTime = !exploration[key].discovered || force;
        mergeObject(exploration[key], { discovered: true });
        await scene.setFlag(MODULE_ID, "exploration", exploration);
        const cellData = getCellData(cell);
        const discoverActivities = cellData.discoverActivities;

        // Remove Hidden
        if (discoverActivities.removeHidden.value.length) {
            const removeHidden = discoverActivities.removeHidden.value.map((uuid) => fromUuidSync(uuid));
            for (const document of removeHidden) {
                await document.update({ hidden: false });
            }
        }

        // Roll Tables
        const shouldRollTables = !discoverActivities.rollTables.firstTimeOnly || (discoverActivities.rollTables.firstTimeOnly && firstTime);
        if (discoverActivities.rollTables.value.length && shouldRollTables) {
            for (const uuid of discoverActivities.rollTables.value) {
                const table = await fromUuid(uuid);
                table.draw();
            }
        }

        // Macros
        const shouldMacros = !discoverActivities.macros.firstTimeOnly || (discoverActivities.macros.firstTimeOnly && firstTime);
        if (discoverActivities.macros.value.length && shouldMacros) {
            for (const uuid of discoverActivities.macros.value) {
                const macro = await fromUuid(uuid);
                await macro.execute({ cell: { ...cell, ...cellData }, token: tokenDocument });
            }
        }

        // Reveal Journal Entry
        if (discoverActivities.revealJournalEntry) {
            const journalEntry = await fromUuid(cellData.journalEntry);
            if (journalEntry) {
                await journalEntry.update({ "ownership.default": CONST.DOCUMENT_OWNERSHIP_LEVELS.OBSERVER });
            }
        }

        // Reveal Tooltip
        if (discoverActivities.revealTooltip) {
            const tooltip = await fromUuid(cellData.tooltip);
            if (tooltip) {
                await tooltip.update({ "ownership.default": CONST.DOCUMENT_OWNERSHIP_LEVELS.OBSERVER });
            }
        }

        // Reveal Map Notes
        if (discoverActivities.revealMapNotes) {
            const topLeft = canvas.grid.getTopLeftPoint(cell);
            const quad = new PIXI.Rectangle(topLeft.x, topLeft.y, topLeft.x + canvas.grid.sizeX, topLeft.y + canvas.grid.sizeY);
            const notes = Array.from(canvas.notes.quadtree.getObjects(quad)).filter((n) => n.page);
            for (const note of notes) {
                await note.page.update({ "ownership.default": CONST.DOCUMENT_OWNERSHIP_LEVELS.OBSERVER });
            }
        }
    }

    static getAdvanceTime(cell, tokenDocument) {
        const scene = tokenDocument.parent;
        const unitsPerCell = scene.grid.distance;
        const unitsPerHour = (tokenDocument.getFlag(MODULE_ID, "speed") ?? 0) * (cell.speedMultiplier ?? 1);
        if (!unitsPerHour) return 0;
        return unitsPerCell / unitsPerHour;
    }

    static _onPreUpdateToken(document, update, options) {
        //if(game.user.isGM) return;
        if (options.hexplorerConfirmed) return;
        const isHexplorer = document.getFlag(MODULE_ID, "enabled");
        if (!isHexplorer) return;
        const xChanged = "x" in update && update.x !== document.x;
        const yChanged = "y" in update && update.y !== document.y;
        if (!xChanged && !yChanged) return;
        const destination = { x: update.x ?? document.x, y: update.y ?? document.y };
        canvas.hexplorer.requestHexplorerUpdate(document, destination);
        return false;
    }

    static _onUpdateToken(document, update) {
        if (!game.users.activeGM.isSelf || !document.getFlag(MODULE_ID, "enabled")) return;
        const xChanged = "x" in update && update.x !== document.x;
        const yChanged = "y" in update && update.y !== document.y;
        if (!xChanged && !yChanged) return;
        const origin = GridUtils.offsetToCenter({ x: document.x, y: document.y }, document);
        const destination = GridUtils.offsetToCenter({ x: update.x ?? document.x, y: update.y ?? document.y }, document);
        canvas.hexplorer.fog.processPath(origin, destination, document);
    }

    static _onUpdateScene(scene, update, options) {
        if (scene !== canvas.hexplorer.scene) return;
        if (update.flags?.[MODULE_ID]?.exploration || update.flags?.[MODULE_ID]?.config) {
            let redraw = false;
            if (update.flags?.[MODULE_ID]?.exploration) {
                const exploration = update.flags[MODULE_ID].exploration;
                for (const [key, value] of Object.entries(exploration)) {
                    if (value.revealed) {
                        const isFirstGMinScene = game.user === Array.from(game.users).find((u) => u.isGM && u.active && u.viewedScene === scene.id);
                        if (isFirstGMinScene) {
                            const cell = GridUtils.getOffsetFromKey(key);
                            canvas.hexplorer.revealCell(cell, scene);
                        }
                    }
                    if (value.color) {
                        redraw = true;
                    }
                    if (value.revealed !== undefined) redraw = true;
                }
            }

            // Redraw Fog
            if (redraw) {
                canvas.hexplorer.fog.draw();
                canvas.hexplorer.highlightLayer?.draw(false);
            }
        }
    }
}
