import { MODULE_ID } from "../main.js";
import { confirm, deepClone, HandlebarsApplication, l, mergeClone, mergeObject } from "../lib/utils.js";
import { GridUtils } from "../canvas/GridUtils.js";
import { FormBuilder } from "../lib/formBuilder.js";
import { HexplorerDrawPreviewLayer } from "../canvas/HexplorerDrawPreviewLayer.js";
import { getSetting, setSetting } from "../settings.js";

export const DEFAULT_CELL_CONFIG = {
    color: "#000000",
    region: "",
    reveal: "",
    revealRadius: -1,
    journalEntry: "",
    speedMultiplier: 1,
    rollTables: [],
    removeHidden: [],
    macros: [],
    tooltip: "",
    executeOnlyOnDiscover: false,
    explored: false,
    revealed: false,
    exploreActivities: {
        rollTables: {
            value: [],
        },
        macros: {
            value: [],
        },
        removeHidden: {
            value: [],
        },
        revealTooltip: false,
        revealJournalEntry: false,
        revealMapNotes: false,
    },
    discoverActivities: {
        rollTables: {
            value: [],
            firstTimeOnly: false,
        },
        macros: {
            value: [],
            firstTimeOnly: true,
        },
        removeHidden: {
            value: [],
            firstTimeOnly: true,
        },
        revealTooltip: false,
        revealJournalEntry: false,
        revealMapNotes: false,
    },
};

export function getCellData(cell) {
    const key = GridUtils.getKey(cell);
    const current = (canvas.scene.getFlag(MODULE_ID, "exploration") ?? {})[key] ?? {};
    return mergeClone(DEFAULT_CELL_CONFIG, current);
}

let revealInput;

export class HexplorerApp extends HandlebarsApplication {
    constructor(scene) {
        super();
        this.scene = scene;
        if (canvas.hexplorer.app) {
            canvas.hexplorer.app.deactivateCanvasListeners();
            Hooks.off("updateScene", canvas.hexplorer.app.updateSceneHook);
            canvas.hexplorer.app.drawPreviewLayer.destroy();
        }
        canvas.hexplorer.app = this;
        this.activateCanvasListeners();
        this.updateSceneHook = Hooks.on("updateScene", this._onUpdateScene.bind(this));
        //this.applyBrush = foundry.utils.throttle(this.applyBrush, 1);
        this.drawPreviewLayer = new HexplorerDrawPreviewLayer(scene);
    }

    #activeBrush;

    get activeBrush() {
        return this.#activeBrush;
    }

    set activeBrush(brush) {
        this.#activeBrush = brush;
        this.drawPreviewLayer.clear();
    }

    getCombinedBrushes() {
        const brushes = this.scene.getFlag(MODULE_ID, "brushes") ?? {};
        const basicBrushes = {
            explore: {
                name: "Reveal",
                noDelete: true,
                data: {
                    revealed: true,
                },
            },
            hide: {
                name: "Hide",
                noDelete: true,
                data: {
                    revealed: false,
                },
            },
            clear: {
                name: "Clear",
                noDelete: true,
                data: deepClone(DEFAULT_CELL_CONFIG),
            },
        };
        return mergeClone(basicBrushes, brushes);
    }

    activateCanvasListeners() {
        //Setup bound functions
        this.boundOnCanvasClick = this._onCanvasClick.bind(this);
        this.boundOnCanvasMouseMove = this._onCanvasMouseMove.bind(this);
        this.boundOnCanvasMouseDown = this._onCanvasMouseDown.bind(this);
        this.boundOnCanvasMouseUp = this._onCanvasMouseUp.bind(this);
        //Add listeners
        const canvas = document.querySelector("#board");
        canvas.addEventListener("click", this.boundOnCanvasClick);
        canvas.addEventListener("mousemove", this.boundOnCanvasMouseMove);
        canvas.addEventListener("mousedown", this.boundOnCanvasMouseDown);
        canvas.addEventListener("mouseup", this.boundOnCanvasMouseUp);
    }

    deactivateCanvasListeners() {
        const canvas = document.querySelector("#board");
        canvas.removeEventListener("click", this.boundOnCanvasClick);
        canvas.removeEventListener("mousemove", this.boundOnCanvasMouseMove);
        canvas.removeEventListener("mousedown", this.boundOnCanvasMouseDown);
        canvas.removeEventListener("mouseup", this.boundOnCanvasMouseUp);
    }

    _onUpdateScene(scene, update) {
        if (scene !== this.scene) return;
        if (update.flags?.[MODULE_ID]) {
            this.render();
        }
    }

    static get DEFAULT_OPTIONS() {
        return mergeClone(super.DEFAULT_OPTIONS, {
            tag: "form",
            classes: [this.APP_ID],
            id: this.APP_ID,
            window: {
                title: `${MODULE_ID}.${this.APP_ID}.title`,
                icon: "",
                resizable: true,
                contentClasses: ["standard-form"],
            },
            actions: {
                "reset-fow": this.onResetFow,
            },
            position: {
                width: 360,
                height: "auto",
            },
            form: {
                handler: this.#onSubmit,
                submitOnChange: true,
                submitOnClose: true,
            },
        });
    }

    static get PARTS() {
        return {
            content: {
                template: `modules/${MODULE_ID}/templates/${this.APP_ID}.hbs`,
                classes: ["standard-form"],
                scrollable: [".brushes-list"],
            },
            footer: {
                template: "templates/generic/form-footer.hbs",
            },
        };
    }

    async _prepareContext(options) {
        const brushes = this.getCombinedBrushes();
        if (this.activeBrush && brushes[this.activeBrush.id]) brushes[this.activeBrush.id].active = true;
        const config = this.scene.getFlag(MODULE_ID, "config") ?? {};
        return {
            brushes,
            config,
            brushRadius: this.brushRadius,
            buttons: [
                {
                    type: "button",
                    icon: "fas fa-cloud",
                    action: "reset-fow",
                    label: `${MODULE_ID}.${this.APP_ID}.reset-fow.label`,
                },
            ],
        };
    }

    _onCanvasClick(event) {
        const isShift = event.shiftKey;
        if (revealInput && isShift) {
            let newValue = revealInput.value + "," + GridUtils.getKey(canvas.grid.getOffset(canvas.mousePosition));
            newValue = newValue.trim();
            if (newValue.startsWith(",")) newValue = newValue.substring(1);
            revealInput.value = newValue;
            ui.notifications.info(l(`${MODULE_ID}.cell-config.reveal.notification`));
            return;
        }
        if (this.activeBrush) return;
        if (!canvas.activeLayer.hover) configureCell(canvas.grid.getOffset(canvas.mousePosition), this.scene);
    }

    _onCanvasMouseMove(event) {
        if (this.activeBrush) {
            const cell = canvas.grid.getOffset(canvas.mousePosition);
            this.drawPreviewLayer.updateHoverPreview(cell, this.activeBrush.data.color, this.brushRadius);
        }
        if (this.isLeftDown) {
            event.preventDefault();
            event.stopPropagation();
            event.stopImmediatePropagation();
            canvas.activeLayer._onDragLeftCancel(event);
            this.applyBrush();
        }
    }

    get brushRadius() {
        return parseInt(getSetting("brushRadius"));
    }

    applyBrush() {
        if (!this.activeBrush) return;
        const cell = canvas.grid.getOffset(canvas.mousePosition);
        const cells = GridUtils.getCells(cell, this.brushRadius);
        this.drawPreviewLayer.addPreview(cells, this.activeBrush.data.color);
    }

    async finalizeBrush() {
        this.applyBrush();
        const cells = this.drawPreviewLayer.cells;
        this.drawPreviewLayer.clear();
        if (!cells.length) return;
        const exploration = this.scene.flags[MODULE_ID]?.exploration ?? {};
        for (const cell of cells) {
            const key = GridUtils.getKey(cell);
            exploration[key] ??= {};
            mergeObject(exploration[key], this.activeBrush.data);
        }
        await this.scene.setFlag(MODULE_ID, "exploration", exploration);
    }

    _onCanvasMouseDown(event) {
        this.isLeftDown = event.button === 0;
        this.isRightDown = event.button === 2;
    }

    _onCanvasMouseUp(event) {
        const wasLeftDown = this.isLeftDown;
        this.isLeftDown = false;
        this.isRightDown = false;
        const isRightMouse = event.button === 2;
        if (isRightMouse) {
            return this.drawPreviewLayer.clear();
        }
        if (wasLeftDown) this.finalizeBrush();
    }

    _onRender(context, options) {
        super._onRender(context, options);
        const html = this.element;
        html.querySelectorAll(".brush").forEach((brush) => {
            brush.addEventListener("click", this._onBrushClick.bind(this));
        });
        html.querySelectorAll(".brush-delete").forEach((brush) => {
            brush.addEventListener("click", this._onBrushDelete.bind(this));
        });
        html.querySelectorAll(".brush-apply").forEach((brush) => {
            brush.addEventListener("click", this._onBrushApply.bind(this));
        });
        html.querySelector("#brush-radius").addEventListener("change", (e) => {
            const value = e.target.value;
            setSetting("brushRadius", value);
        });
    }

    _onBrushClick(event) {
        const brushId = event.currentTarget.dataset.brushId;
        const active = event.currentTarget.classList.contains("active");
        if (active) {
            event.currentTarget.classList.remove("active");
            this.activeBrush = null;
        } else {
            this.element.querySelectorAll(".brush").forEach((brush) => {
                brush.classList.remove("active");
            });
            event.currentTarget.classList.add("active");
            this.activeBrush = mergeClone(this.getCombinedBrushes()[brushId], { id: brushId });
        }
    }

    async _onBrushDelete(event) {
        event.preventDefault();
        event.stopPropagation();
        const brushId = event.currentTarget.closest(".brush").dataset.brushId;
        const brushes = this.scene.getFlag(MODULE_ID, "brushes") ?? {};
        if (!brushes[brushId]) return;
        const deleteKey = "-=" + brushId;
        brushes[deleteKey] = null;
        await this.scene.setFlag(MODULE_ID, "brushes", brushes);
    }

    async _onBrushApply(event) {
        event.preventDefault();
        event.stopPropagation();
        const brushId = event.currentTarget.closest(".brush").dataset.brushId;
        const brushes = this.scene.getFlag(MODULE_ID, "brushes") ?? {};
        if (!brushes[brushId]) return;
        const brush = brushes[brushId];
        const dialog = await new FormBuilder()
            .title(`${MODULE_ID}.hexplorer-app.brushes.apply.title`)
            .checkbox({ name: "sameName", label: `${MODULE_ID}.hexplorer-app.brushes.apply.sameName`, hint: `${MODULE_ID}.hexplorer-app.brushes.apply.sameNameHint`, value: true })
            .checkbox({ name: "sameColor", label: `${MODULE_ID}.hexplorer-app.brushes.apply.sameColor`, hint: `${MODULE_ID}.hexplorer-app.brushes.apply.sameColorHint`, value: false })
            .render();
        if (!dialog) return;
        const {sameName, sameColor} = dialog;
        const cells = this.scene.getFlag(MODULE_ID, "exploration") ?? {};
        let count = 0;
        for (const [key, cell] of Object.entries(cells)) {
            if ((sameName && brush.data.region === cell.region) || (sameColor && brush.data.color === cell.color)) {
                mergeObject(cells[key], brush.data);
                count++;
            }
        }
        ui.notifications.info(l(`${MODULE_ID}.hexplorer-app.brushes.apply.notification`).replace("{count}", count));
        await this.scene.setFlag(MODULE_ID, "exploration", cells);
    }

    _onClose(options) {
        super._onClose(options);
        this.deactivateCanvasListeners();
        Hooks.off("updateScene", this.updateSceneHook);
        this.drawPreviewLayer.destroy();
        canvas.hexplorer.app = null;
    }

    static #onSubmit(event, form, formData) {
        const data = foundry.utils.expandObject(formData.object);
        this.scene.setFlag(MODULE_ID, "config", data.config);
    }

    static async onResetFow() {
        if (await confirm(`${MODULE_ID}.reset-fow.title`, `${MODULE_ID}.reset-fow.content`)) {
            canvas.hexplorer.fog.reset();
        }
    }
}

async function configureCell(cell, scene) {
    const key = GridUtils.getKey(cell);
    const current = (scene.getFlag(MODULE_ID, "exploration") ?? {})[key] ?? {};
    const data = mergeClone(DEFAULT_CELL_CONFIG, current);

    const form = new FormBuilder();
    form.object(data)
        .title(l(`${MODULE_ID}.cell-config.title`) + " (" + key + ")")
        .onRender(() => {
            const html = form.app.element;
            const reveal = html.querySelector(`[name="reveal"]`);
            revealInput = reveal;
        })
        .tab({ id: "basic", icon: "fas fa-cogs", label: `${MODULE_ID}.cell-config.tabs.basic.label` })
        .number({ name: "revealRadius", label: `${MODULE_ID}.cell-config.revealRadius.label`, hint: `${MODULE_ID}.cell-config.revealRadius.hint` })
        .color({ name: "color", label: `${MODULE_ID}.cell-config.color.label` })
        .text({ name: "region", label: `${MODULE_ID}.cell-config.region.label` })
        .number({ name: "speedMultiplier", label: `${MODULE_ID}.cell-config.speed-multiplier.label` })
        .uuid({ name: "tooltip", label: `${MODULE_ID}.cell-config.tooltip.label`, type: "JournalEntryPage", multiple: false })
        .uuid({ name: "journalEntry", label: `${MODULE_ID}.cell-config.journal-entry.label`, hint: `${MODULE_ID}.cell-config.journal-entry.hint`, type: "JournalEntryPage", multiple: false })
        .tab({ id: "explore-activities", icon: "fas fa-eye", label: `${MODULE_ID}.cell-config.tabs.explore-activities.label` })
        .html(`<p class="hint">${l(`${MODULE_ID}.cell-config.tabs.explore-activities.hint`)}</p>`)
        .checkbox({ name: "revealed", label: `${MODULE_ID}.cell-config.revealed.label` })
        .text({ name: "reveal", label: `${MODULE_ID}.cell-config.reveal.label`, hint: `${MODULE_ID}.cell-config.reveal.hint` })
        .uuid({ name: "exploreActivities.removeHidden.value", label: `${MODULE_ID}.cell-config.remove-hidden.label`, hint: `${MODULE_ID}.cell-config.remove-hidden.hint`, multiple: true })
        .uuid({ name: "exploreActivities.macros.value", label: `${MODULE_ID}.cell-config.macros.label`, hint: `${MODULE_ID}.cell-config.macros.hint`, type: "Macro", multiple: true })
        .checkbox({ name: "exploreActivities.revealTooltip", label: `${MODULE_ID}.cell-config.reveal-tooltip.label`, hint: `${MODULE_ID}.cell-config.reveal-tooltip.hint` })
        .checkbox({ name: "exploreActivities.revealJournalEntry", label: `${MODULE_ID}.cell-config.reveal-journal-entry.label`, hint: `${MODULE_ID}.cell-config.reveal-journal-entry.hint` })
        .checkbox({ name: "exploreActivities.revealMapNotes", label: `${MODULE_ID}.cell-config.reveal-map-notes.label`, hint: `${MODULE_ID}.cell-config.reveal-map-notes.hint` })
        .tab({ id: "discover-activities", icon: "fas fa-person-running-fast", label: `${MODULE_ID}.cell-config.tabs.discover-activities.label` })
        .html(`<p class="hint">${l(`${MODULE_ID}.cell-config.tabs.discover-activities.hint`)}</p>`)
        .checkbox({ name: "discovered", label: `${MODULE_ID}.cell-config.discovered.label`, hint: `${MODULE_ID}.cell-config.discovered.hint` })
        .uuid({ name: "discoverActivities.removeHidden.value", label: `${MODULE_ID}.cell-config.remove-hidden.label`, hint: `${MODULE_ID}.cell-config.remove-hidden.hint`, multiple: true })
        .checkbox({ name: "discoverActivities.revealJournalEntry", label: `${MODULE_ID}.cell-config.reveal-journal-entry.label`, hint: `${MODULE_ID}.cell-config.reveal-journal-entry.hint` })
        .checkbox({ name: "discoverActivities.revealTooltip", label: `${MODULE_ID}.cell-config.reveal-tooltip.label`, hint: `${MODULE_ID}.cell-config.reveal-tooltip.hint` })
        .checkbox({ name: "discoverActivities.revealMapNotes", label: `${MODULE_ID}.cell-config.reveal-map-notes.label`, hint: `${MODULE_ID}.cell-config.reveal-map-notes.hint` })
        .html(`<fieldset> <legend>${l(`${MODULE_ID}.cell-config.roll-tables.label`)}</legend>`)
        .uuid({ name: "discoverActivities.rollTables.value", label: `${MODULE_ID}.cell-config.roll-tables.label`, hint: `${MODULE_ID}.cell-config.roll-tables.hint`, type: "RollTable", multiple: true })
        .checkbox({ name: "discoverActivities.rollTables.firstTimeOnly", label: `${MODULE_ID}.cell-config.first-time-only.label` })
        .html(`</fieldset> <fieldset> <legend>${l(`${MODULE_ID}.cell-config.macros.label`)}</legend>`)
        .uuid({ name: "discoverActivities.macros.value", label: `${MODULE_ID}.cell-config.macros.label`, hint: `${MODULE_ID}.cell-config.macros.hint`, type: "Macro", multiple: true })
        .checkbox({ name: "discoverActivities.macros.firstTimeOnly", label: `${MODULE_ID}.cell-config.first-time-only.label` })
        .html(`</fieldset>`);

    form.button({
        label: `${MODULE_ID}.cell-config.save-as-brush.label`,
        action: "saveAsBrush",
        icon: "fas fa-brush",
        callback: async () => {
            const data = form.app.getFormData();
            delete data.explored;
            delete data.revealed;
            const brushConfig = await new FormBuilder().text({ name: "brushName", label: `${MODULE_ID}.cell-config.brush-name.label`, value: data.region }).render();
            if (!brushConfig) return;
            const brushes = scene.getFlag(MODULE_ID, "brushes") ?? {};
            const brushId = foundry.utils.randomID();
            brushes[brushId] = {
                name: brushConfig.brushName,
                data: data,
            };
            await scene.setFlag(MODULE_ID, "brushes", brushes);
        },
    });

    //Discover button
    form.button({
        label: `${MODULE_ID}.cell-config.discover-button.label`,
        action: "discover",
        icon: "fas fa-map",
        callback: () => {
            const token = canvas.tokens.controlled.find((t) => t.document.getFlag(MODULE_ID, "enabled")) ?? canvas.tokens.placeables.find((t) => t.document.getFlag(MODULE_ID, "enabled"));
            if (!token) return ui.notifications.warn(`${MODULE_ID}.cell-config.discover-button.no-token`);
            canvas.hexplorer.exploreCell(cell, canvas.scene, token.document);
        },
    });

    const result = await form.render();
    revealInput = null;
    if (result) {
        const exploration = (await scene.getFlag(MODULE_ID, "exploration")) ?? {};
        exploration[key] = result;
        await scene.setFlag(MODULE_ID, "exploration", exploration);
    }
}
