import {HandlebarsApplication} from "../lib/utils.js";

const MODULE_ID = "journal-page-popout";

let classesCache = null;
export function setClassesCache(classes) {
    classesCache = classes;
}

export class JournalPopout extends HandlebarsApplication {
    constructor(page, classes) {
        super();
        classes ??= classesCache;
        this.page = page;
        this.classes = classes?.value ?? "";
        this.classes = this.classes.replace("app window-app", "");
    }

    static DEFAULT_OPTIONS = {
        classes: [this.APP_ID],
        tag: "div",
        window: {
            frame: true,
            positioned: true,
            title: `${MODULE_ID}.${this.APP_ID}.title`,
            icon: "fas fa-page",
            controls: [],
            minimizable: true,
            resizable: true,
            contentTag: "section",
            contentClasses: [],
        },
        actions: {},
        form: {
            handler: undefined,
            submitOnChange: false,
            closeOnSubmit: false,
        },
        position: {
            width: 900,
            height: 800,
        },
        actions: {},
    };

    static PARTS = {
        content: {
            template: `modules/${MODULE_ID}/templates/${this.APP_ID}.hbs`,
        },
    };

    static get APP_ID() {
        return this.name
            .split(/(?=[A-Z])/)
            .join("-")
            .toLowerCase();
    }

    get APP_ID() {
        return this.constructor.APP_ID;
    }

    get title() {
        return this.page.name;
    }

    async _prepareContext(options) {
        let content;
        const page = this.page;
        if (page.type === "text") {
            content = await TextEditor.enrichHTML(page.text.content, { secrets: game.user.isGM, relativeTo: page, async: true });
        } else if (page.type === "image") {
            content = `<div class="simple-quest-image-journal"><img src="${page.src}" alt="${page.name}"><p>${page.image.caption}</p></div>`;
        } else if (page.type === "pdf") {
            const params = page.sheet._getViewerParams();
            const frame = `<iframe src="scripts/pdfjs/web/viewer.html?${params}" style="width: 100%; height: 95%;"></iframe>`;
            content = frame;
        }
        else {
            content = `<p>This Page Type is not Supported</p>`;
        }

        const clonedToc = JSON.parse(JSON.stringify(page.toc));
        for (const entry of Object.values(clonedToc)) {
            const enriched = await TextEditor.enrichHTML(entry.text, {secrets: game.user.isGM, relativeTo: page, async: true});
            if(enriched === entry.text) continue;
            const div = document.createElement("div");
            div.innerHTML = enriched;
            entry.text = div.innerText;
            entry.slug = div.innerText.slugify({strict: true});
        }

        return { content, classes: this.classes, toc: clonedToc };
    }

    _onRender(context, options) {
        super._onRender(context, options);
        const html = this.element;
        html.querySelectorAll(".toc-item").forEach((item) => {
            item.addEventListener("click", (event) => {
                event.preventDefault();
                const slug = event.currentTarget.dataset.slug;
                const target = Array.from(html.querySelectorAll('h1, h2, h3, h4, h5, h6')).find((h) => h.innerText.slugify({strict: true}) === slug);
                target.scrollIntoView({
                    block: "start",
                });
            });
        });
    }

    _onClose(options) {
        super._onClose(options);
    }
}