Hooks.once("dragRuler.ready", (SpeedProvider) => {
  class LancerSpeedProvider extends SpeedProvider {
    get colors() {
      return [
        {
          id: "standard",
          default: 0x1e88e5,
          name: "lancer-speed-provider.standard",
        },
        {
          id: "boost",
          default: 0xffc107,
          name: "lancer-speed-provider.boost",
        },
        {
          id: "over-boost",
          default: 0xd81b60,
          name: "lancer-speed-provider.over-boost",
        },
      ];
    }

    get defaultUnreachableColor() {
      return 0x000000;
    }

    /** @param token {Token} The token to check movement */
    getRanges(token) {
      const actor = token.actor;
      const effects = Array.from(actor.statuses);
      /**@type{number}*/
      const speed = tokenSpeed(token);
      const boost_bonus =
        nerveweave_boost_bonus(actor) +
        (actor.getFlag("lancer-speed-provider", "boost-bonus") ?? 0);
      const stunned = isStunned(token);
      // Cant move if stunned or immobilized
      if (stunned) return [{ range: -1, color: "standard" }];

      const prone = effects.some((e) => e.endsWith("prone"));
      /** @type {boolean} */
      const startedProne = token.combatant
        ?.getFlag("lancer-speed-provider", "turn-status")
        ?.some((e) => e.endsWith("prone"));
      const slowed = prone || effects.some((e) => e.endsWith("slow"));

      let range = speed;
      const ranges = [];
      if (prone || !startedProne) {
        ranges.push({ range, color: "standard" });
        range += speed + boost_bonus;
      }
      if (!slowed) {
        ranges.push({ range, color: "boost" });
        range += speed + boost_bonus;
      }
      if (!slowed && canOvercharge(actor)) {
        ranges.push({ range, color: "over-boost" });
      }

      return ranges;
    }
  }

  dragRuler.registerModule("lancer-speed-provider", LancerSpeedProvider);
});

Hooks.on("aeris-tokens.preGetMovementValue", (actor, ranges, mode) => {
  /**@type{number}*/
  let speed = actor.system.speed;
  if (actor.statuses.has("prone")) speed = Math.floor(speed / 2);
  const boost_bonus = nerveweave_boost_bonus(actor);
  const stunned = ["stunned", "immobilized", "shutdown", "downandout"].some(
    (e) => actor.statuses.has(e),
  );
  // Cant move if stunned or immobilized
  if (stunned) {
    ranges.push({ value: 0, preset: "available" });
    return;
  }

  const prone = actor.statuses.has("prone");
  /** @type {boolean} */
  const startedProne = actor
    .getActiveTokens()
    .pop()
    ?.combatant?.getFlag("lancer-speed-provider", "turn-status")
    ?.some((e) => e.endsWith("prone"));
  const slowed = prone || actor.statuses.has("slow");

  let value = speed;
  if (prone || !startedProne) {
    ranges.push({
      value,
      rgb: Color.from(
        game.settings.get("lancer-speed-provider", "color-standard"),
      ).valueOf(),
      a: 0.25,
    });
    value += speed + boost_bonus;
  }
  if (!slowed) {
    ranges.push({
      value,
      rgb: Color.from(
        game.settings.get("lancer-speed-provider", "color-boost"),
      ).valueOf(),
      a: 0.25,
    });
    value += speed + boost_bonus;
  }
  if (!slowed && canOvercharge(actor)) {
    ranges.push({
      value,
      rgb: Color.from(
        game.settings.get("lancer-speed-provider", "color-over-boost"),
      ).valueOf(),
      a: 0.25,
    });
  }
});

function initAerisTokenIntegration() {
  const LRITokenDocument = (TokenDocument) =>
    class extends TokenDocument {
      _onRelatedUpdate(update, options) {
        super._onRelatedUpdate(update, options);
        this.object?.dragActionHandler?.refreshMovementData();
      }
    };

  CONFIG.Token.documentClass = LRITokenDocument(CONFIG.Token.documentClass);

  game.settings.settings.get("aeris-tokens.gridUnreachableTilesColor").default =
    "#00000026";
}

Hooks.once("init", () => {
  const mods = ["elevationruler", "drag-ruler", "aeris-tokens"].reduce(
    (a, m) => a + +(game.modules.get(m)?.active ?? 0),
    0,
  );

  if (mods > 1) {
    Hooks.once("ready", () => {
      Dialog.prompt({
        title: "WARNING",
        content:
          "WARNING: Multiple ruler modules detected, please disable all but one. (Elevation Ruler is the recommended ruler module in v12)",
      })
        .catch(() => {})
        .then(() => new ModuleManagement().render(true));
    });
  }

  if (
    !game.modules.get("elevationruler")?.active &&
    !game.modules.get("aeris-tokens")?.active
  )
    return;

  if (game.modules.get("aeris-tokens")?.active) initAerisTokenIntegration();

  const isV12 = game.release.generation >= 12;
  if (!isV12) {
    Hooks.on("renderSettingsConfig", renderSettingsConfig);
    Hooks.on("canvasInit", (gameCanvas) => {
      const r = (rule) => {
        switch (rule) {
          case "121":
            return "5105";
          case "222":
            return "MANHATTAN";
          case "euc":
            return "EUCL";
          default:
            return "555";
        }
      };
      if (gameCanvas.grid.isHex) canvas.grid.diagonalRule = r("111");
      else
        canvas.grid.diagonalRule = r(
          game.settings.get("lancer", "squareGridDiagonals"),
        );
    });
  }

  game.settings.register("lancer-speed-provider", "color-standard", {
    name: "lancer-speed-provider.settings.color-standard.label",
    scope: "client",
    type: isV12
      ? new foundry.data.fields.ColorField({
          required: true,
          blank: false,
          initial: "#1e88e5",
        })
      : String,
    default: "#1e88e5",
    config: true,
    onChange: (val) => {
      if (game.modules.get("elevationruler")?.active)
        CONFIG.elevationruler.SPEED.CATEGORIES.find(
          (c) => c.name === "lancer-speed-provider.standard",
        ).color = Color.from(val);
    },
  });
  game.settings.register("lancer-speed-provider", "color-boost", {
    name: "lancer-speed-provider.settings.color-boost.label",
    scope: "client",
    type: isV12
      ? new foundry.data.fields.ColorField({
          required: true,
          blank: false,
          initial: "#ffc107",
        })
      : String,
    default: "#ffc107",
    config: true,
    onChange: (val) => {
      if (game.modules.get("elevationruler")?.active)
        CONFIG.elevationruler.SPEED.CATEGORIES.find(
          (c) => c.name === "lancer-speed-provider.boost",
        ).color = Color.from(val);
    },
  });
  game.settings.register("lancer-speed-provider", "color-over-boost", {
    name: "lancer-speed-provider.settings.color-over-boost.label",
    scope: "client",
    type: isV12
      ? new foundry.data.fields.ColorField({
          required: true,
          blank: false,
          initial: "#d81b60",
        })
      : String,
    default: "#d81b60",
    config: true,
    onChange: (val) => {
      if (game.modules.get("elevationruler")?.active)
        CONFIG.elevationruler.SPEED.CATEGORIES.find(
          (c) => c.name === "lancer-speed-provider.over-boost",
        ).color = Color.from(val);
    },
  });

  if (!game.modules.get("elevationruler")?.active) return;
  CONFIG.elevationruler.SPEED.ATTRIBUTES.WALK = "actor.system.speed";
  CONFIG.elevationruler.SPEED.CATEGORIES = [
    {
      name: "lancer-speed-provider.standard",
      color: Color.from(
        game.settings.get("lancer-speed-provider", "color-standard"),
      ),
      multiplier: 1,
    },
    {
      name: "lancer-speed-provider.boost",
      color: Color.from(
        game.settings.get("lancer-speed-provider", "color-boost"),
      ),
      multiplier: 2,
    },
    {
      name: "lancer-speed-provider.over-boost",
      color: Color.from(
        game.settings.get("lancer-speed-provider", "color-over-boost"),
      ),
      multiplier: 3,
    },
    {
      name: "Unreachable",
      color: Color.from(0),
      multiplier: Number.POSITIVE_INFINITY,
    },
  ];
  CONFIG.elevationruler.SPEED.tokenSpeed = tokenSpeed;
  CONFIG.elevationruler.SPEED.maximumCategoryDistance = maximumCategoryDistance;
});

Hooks.once("setup", () => {
  const cpe = game.lancer.fromLidSync("core_power_active");
  if (!cpe) {
    Item.implementation.create({
      name: game.i18n.localize("lancer-speed-provider.statuses.core_power"),
      type: "status",
      img: "systems/lancer/assets/icons/white/corepower.svg",
      "system.lid": "core_power_active",
      "flags.lancer-speed-provider.core-status": true,
    });
  }
  if (game.modules.get("terrainmapper")?.active) {
    CONFIG.terrainmapper.defaultTerrainJSONs = [
      "modules/lancer-speed-provider/terrain/difficult.json",
    ];
  }
});

Hooks.on("preDeleteItem", (item) => {
  if (
    !item.parent &&
    item.type === "status" &&
    item.getFlag("lancer-speed-provider", "core-status")
  ) {
    ui.notifications.info(
      "Cannot delete Core Power Status with Lancer Ruler Integration active",
    );
    return false;
  }
});

Hooks.on("preCreateActiveEffect", (effect) => {
  const changes = [...effect.changes];
  const MODES = CONST.ACTIVE_EFFECT_MODES;
  if (
    effect.statuses.size === 1 &&
    effect.statuses.has("dangerzone") &&
    getFrameLid(effect.parent) === "mf_tokugawa_alt_enkidu"
  ) {
    changes.push({
      key: "system.speed",
      value: "3",
      mode: MODES.ADD,
      priority: "100",
    });
    effect.updateSource({ changes });
  } else if (
    effect.statuses.size === 1 &&
    effect.statuses.has("core_power_active") &&
    getFrameLid(effect.parent) === "mf_lycan"
  ) {
    changes.push({
      key: "system.speed",
      value: "3",
      mode: MODES.ADD,
      priority: "100",
    });
    effect.updateSource({ changes });
  }
});

Hooks.once("lancer.registerFlows", (steps, flows) => {
  steps.set("addCorePowerSE", async ({ actor }) => {
    if (actor.statuses.has("core_power_active")) return true;
    if (game.release.generation < 12) {
      const createData = foundry.utils.deepClone(
        CONFIG.statusEffects.find((e) => e.id === "core_power_active"),
      );
      foundry.utils.mergeObject(createData, {
        statuses: [createData.id],
        "flags.lancer-speed-provider.status": true,
      });
      // biome-ignore lint/performance/noDelete: This is how core does it
      delete createData.id;
      const ae = getDocumentClass("ActiveEffect");
      ae.migrateDataSafe(createData);
      ae.cleanData(createData);
      await ae.create(createData, { parent: actor });
    } else {
      actor.toggleStatusEffect("core_power_active", { active: true });
    }
    return true;
  });
  flows
    .get("CoreActiveFlow")
    ?.insertStepAfter("consumeCorePower", "addCorePowerSE");
});

Hooks.on("updateCombat", (combat, change) => {
  if (!("turn" in change) || !combat.current.tokenId) return;
  const token = game.canvas.tokens.get(combat.current.tokenId);
  if (!token?.isOwner) return;
  const combatant = combat.combatants.get(combat.current.combatantId);
  if (!combatant?.isOwner) return;
  const conditionIds = Array.from(token.actor.statuses);
  combatant.setFlag("lancer-speed-provider", "turn-status", conditionIds);
});

Hooks.on("preDeleteCombatant", (combatant) => {
  if (game.release.generation < 12) {
    for (const e of combatant.actor?.effects ?? []) {
      if (e.getFlag("lancer-speed-provider", "status")) e.delete();
    }
  } else {
    combatant.actor?.toggleStatusEffect("core_power_active", { active: false });
  }
});

Hooks.on("preDeleteCombat", (combat) => {
  for (const c of combat.combatants) {
    if (game.release.generation < 12) {
      for (const e of c.actor?.effects ?? []) {
        if (e.getFlag("lancer-speed-provider", "status")) e.delete();
      }
    } else {
      c.actor?.toggleStatusEffect("core_power_active", { active: false });
    }
  }
});

function tokenSpeed(token) {
  const actor = token.actor;
  let speed = actor.system.speed;
  if (actor.statuses.has("prone")) speed = Math.floor(speed / 2);
  return speed;
}

function maximumCategoryDistance(token, speedCategory, tokenSpeed) {
  const tSpeed = tokenSpeed ?? CONFIG.elevationruler.SPEED.tokenSpeed(token);
  const CATEGORIES = CONFIG.elevationruler.SPEED.CATEGORIES;
  if (speedCategory.name === "Unreachable")
    return tSpeed * Number.POSITIVE_INFINITY;

  const idx = CATEGORIES.indexOf(speedCategory);

  const startedProne = token.combatant
    ?.getFlag("lancer-speed-provider", "turn-status")
    ?.includes("prone");
  const prone = token.actor.statuses.has("prone");
  const firstMove = prone || !startedProne;

  if (
    isStunned(token) ||
    (idx === 2 && !canOvercharge(token.actor)) ||
    (idx > 0 && (prone || token.actor.statuses.has("slow"))) ||
    (idx === 0 && !firstMove)
  )
    return 0;
  const accum =
    idx > 0 ? maximumCategoryDistance(token, CATEGORIES[idx - 1], tSpeed) : 0;

  return (
    accum +
    tSpeed +
    (idx > 0
      ? nerveweave_boost_bonus(token.actor) +
        (token.actor.getFlag("lancer-speed-provider", "boost-bonus") ?? 0)
      : 0)
  );
}

/**
 * @returns {boolean}
 */
function canOvercharge(actor) {
  if (actor.is_npc()) {
    return actor.itemTypes.npc_feature.some((i) =>
      i.system.lid.startsWith("npcf_limitless"),
    );
  }
  return actor.is_mech();
}

const stunnedSet = ["stunned", "immobilized", "shutdown", "downandout"];
function isStunned(token) {
  return stunnedSet.some((e) => token.actor.statuses.has(e));
}

function getFrameLid(actor) {
  return actor.system?.loadout?.frame?.value?.system?.lid ?? null;
}

function nerveweave_boost_bonus(actor) {
  if (!actor.is_mech()) return 0;
  const pilot = actor.system.pilot?.value;
  if (pilot?.items.some((i) => i.system.lid === "cb_integrated_nerveweave"))
    return 2;
  return 0;
}

function renderSettingsConfig(_app, el) {
  const standard = game.settings.get("lancer-speed-provider", "color-standard");
  const boost = game.settings.get("lancer-speed-provider", "color-boost");
  const ovboost = game.settings.get(
    "lancer-speed-provider",
    "color-over-boost",
  );

  el.find('[name="lancer-speed-provider.color-standard"]')
    .parent()
    .append(
      `<input type="color" value="${standard}" data-edit="lancer-speed-provider.color-standard">`,
    );
  el.find('[name="lancer-speed-provider.color-boost"]')
    .parent()
    .append(
      `<input type="color" value="${boost}" data-edit="lancer-speed-provider.color-boost">`,
    );
  el.find('[name="lancer-speed-provider.color-over-boost"]')
    .parent()
    .append(
      `<input type="color" value="${ovboost}" data-edit="lancer-speed-provider.color-over-boost">`,
    );
}
