import {Socket} from "./lib/socket.js";

export class LightSwitch extends PIXI.Container {
    constructor(light) {
        super();
        this.light = light;
        this.light.lightSwitch = this;
    }
    async draw() {
        // Control Icon
        this.icon = this.icon || this.addChild(new PIXI.Sprite());
        this.icon.width = this.icon.height = 40;
        this.icon.alpha = 0.6;
        this.icon.texture = await this._getTexture();
        this.icon.tint = !this.light.document.hidden ? 0xffffff : 0xff0000;

        // Add control interactivity
        this.eventMode = "static";
        this.interactiveChildren = false;
        this.hitArea = new PIXI.Rectangle(-2, -2, 44, 44);
        this.cursor = "pointer";

        // Set position
        this.reposition();
        this.alpha = 1.0;

        // Activate listeners
        this.removeAllListeners();
        this.on("mouseover", this._onMouseOver).on("mouseout", this._onMouseOut).on("mousedown", this._onMouseDown);

        this.visible = this.isVisible;

        // Return the control icon
        return this;
    }
    async _getTexture(invert = false) {
        if (this.light.document.flags["light-switch"]?.img) {
            await loadTexture(this.light.document.flags["light-switch"].img);
            return getTexture(this.light.document.flags["light-switch"].img);
        }
        if (invert) return getTexture(!this.light.document.hidden ? CONFIG.controlIcons.lightOff : CONFIG.controlIcons.light);
        return getTexture(this.light.document.hidden ? CONFIG.controlIcons.lightOff : CONFIG.controlIcons.light);
    }
    reposition() {
        this.position.set(this.light.center.x - this.width / 2, this.light.center.y - this.height / 2);
    }
    get isVisible() {
        const l = this.light;
        if (!this.isLighswitch) return false;
        if (game.modules.get("levels")?.active) {
            if (!CONFIG.Levels.currentToken && game.user.isGM) return true;
            if (!CONFIG.Levels.currentToken) return false;
            if (!CONFIG.Levels.helpers.inRange(l.document, CONFIG.Levels.currentToken.losHeight)) return false;
        }
        const point = new PIXI.Point(l.center.x, l.center.y);
        return canvas.visibility.testVisibility(point, { tolerance: 2, object: this });
    }

    get isLighswitch() {
        const l = this.light;
        if (!l.document.flags["light-switch"]?.isLighswitch || !canvas?.tokens?.active) return false;
        return true;
    }

    _onMouseOver(event) {
        event.stopPropagation();
        const blockPaused = game.paused && !game.user.isGM;
        if (blockPaused) return false;
        this.icon.alpha = 1.0;
    }
    _onMouseOut(event) {
        event.stopPropagation();
        if (game.paused && !game.user.isGM) return false;
        this.icon.alpha = 0.6;
    }
    async _onMouseDown(event) {
        if (event.data.originalEvent.button !== 0) return;
        event.stopPropagation();
        this.icon.tint = this.light.document.hidden ? 0xffffff : 0xff0000;
        this.icon.texture = await this._getTexture(true);
        return Socket.toggleLight({uuid: this.light.document.uuid});
    }

    static drawLightswitches(wrapped, ...args) {
        wrapped(...args);
        this.drawLightswitches();
    }

    static restrictVisibility(wrapped, ...args) {
        for (let l of canvas.controls.lightSwitches.children) {
            l.visible = l.isLighswitch && (!this.tokenVision || l.isVisible);
        }
        wrapped(...args);
    }

    static toggleLight({uuid}) {
        if (game.users.activeGM.isSelf) {
            const light = fromUuidSync(uuid);
            light.update({ hidden: !light.hidden });
        }
    }
}
