class MacroWheel extends Application{
    constructor(){
        super();
    }

    get folderSize(){
        if(this._folderSize) return this._folderSize;
        this._folderSize = parseInt(getComputedStyle(document.documentElement).getPropertyValue('--macroFolderSize').replace("px", ""));
        return this._folderSize;
    }

    set folderSize(size){
        this._folderSize = size;
        document.documentElement.style.setProperty('--macroFolderSize', `${size}px`);
    }

    get itemSize(){
        if(this._itemSize) return this._itemSize;
        this._itemSize = parseInt(getComputedStyle(document.documentElement).getPropertyValue('--macroItemSize').replace("px", ""));
        return this._itemSize;
    }

    set itemSize(size){
        this._itemSize = size;
        document.documentElement.style.setProperty('--macroItemSize', `${size}px`);
    }

    static get defaultOptions() {
        return {
          ...super.defaultOptions,
          id: "macro-wheel",
          template: "modules/macro-wheel/templates/macro-wheel.hbs",
          resizable: false,
          width: 400,
          height: 400,
          dragDrop: [{ dragSelector: null, dropSelector: null }],
          popOut: false,
        };
    }

    activateListeners(html){
        super.activateListeners(html);
        this.positionItems(html[0] ?? html);
        setTimeout(() => {
            html.on("mouseover", ".macro-folder", (event) => {
                event.currentTarget.classList.add("hovered");
                setTimeout(() => {
                    if(event.currentTarget.classList.contains("hovered")){
                        html[0].querySelectorAll(".macro-folder").forEach(folder => {
                            if(folder.dataset.tooltipText) folder.dataset.tooltip = folder.dataset.tooltipText;
                            folder.classList.remove("selected")
                        });
                        event.currentTarget.classList.add("selected");
                        const macroLists = html[0].querySelectorAll(".macro-list");
                        macroLists.forEach(macroList => macroList.classList.add("macro-hidden"));
                        const macroList = event.currentTarget.querySelector(".macro-list");
                        macroList.classList.remove("macro-hidden");
                    }
                }, game.settings.get("macro-wheel", "hoverDelay"));
            });
            html.on("mouseleave", ".macro-folder", (event) => {
                event.currentTarget.classList.remove("hovered");
                delete event.currentTarget.dataset.tooltip;
            });
            html.on("mousedown", ".macro-folder", (event) => {
                if(event.button === 2){
                    const folder = game.folders.get(event.currentTarget.dataset.folderId);
                    if(folder){
                        folder.sheet.render(true);
                        this.close();
                    }
                }
            });
            html.on("click", ".macro-item", async (event) => {
                const uuid = event.currentTarget.dataset.uuid;
                const macro = await fromUuid(uuid);
                macro.execute();
            });
        }, 100);


    }

    positionItems(html){
        this.folderSize = game.settings.get("macro-wheel", "folderSize");
        this.itemSize = game.settings.get("macro-wheel", "macroSize");
        const folders = html.querySelectorAll(".macro-folder");
        const folderCount = folders.length;
        const c = this.folderSize * (folderCount + 1);
        const folderRingRadius = c/(2*Math.PI);
        for(let i=0; i<folderCount; i++){
            const folder = folders[i];
            const angle = ((2 * Math.PI) / folderCount) * (i);
            const x = (folderRingRadius * Math.cos(angle));
            const y = (folderRingRadius * Math.sin(angle));
            folder.style.transform = `translate(${x}px, ${y}px)`;
            const macroList = folder.querySelector(".macro-list");
            macroList.style.transform = `translate(${-x}px, ${-y}px)`;

            const macros = folder.querySelectorAll(".macro-item");
            const macroCount = macros.length;
            const innerMaxRadius = Math.PI * 2;
            const itemRingRadius = folderRingRadius + this.folderSize/2 + this.itemSize*0.6;
            const maxCircumference = (innerMaxRadius * itemRingRadius);
            const maxItems = Math.floor(maxCircumference / (this.itemSize * 1.1));
            const itemAngle = ((innerMaxRadius) / maxItems);
            const currentAngle = itemAngle * macroCount;

            for(let j=0; j<macroCount; j++){
                const ringNumber = Math.floor(j/(maxItems));
                const macro = macros[j];
                const mangle = (angle - ((currentAngle - itemAngle)/2)) + (currentAngle / macroCount) * (j);
                const finalRadius = itemRingRadius + (ringNumber * this.itemSize * 1.1);
                const x = (finalRadius * Math.cos(mangle));
                const y = (finalRadius * Math.sin(mangle));
                macro.style.transform = `translate(${x}px, ${y}px)`;
            }

        }
    }

    setPosition(pos){
        this.element[0].style.left = `${this.clientX}px`;
        this.element[0].style.top = `${this.clientY}px`;
    };

    get macroBarIcons(){
        return [
            "icons/commodities/tech/cog-bronze.webp",
            "icons/commodities/tech/cog-gold.webp",
            "icons/commodities/tech/cog-large-steel-white.webp",
            "icons/commodities/tech/cog-steel-grey.webp",
            "icons/commodities/tech/cog-steel.webp",
        ];
    }

    async getData(){
        const isFolder = game.folders.getName("MacroWheel");
        const useRootSett = game.settings.get("macro-wheel", "useRoot");
        const useRoot = useRootSett || !isFolder;
        function hasAncestorFolderName(ancestors){
            if(useRoot) return true;
            if(!ancestors?.length) return false;
            return ancestors.some(ancestor => ancestor.name === "MacroWheel");
        }
        if(!isFolder && !useRootSett){
            ui.notifications.error("macro-wheel.errors.no-folder", {localize: true, permanent: true});
        }
        const macros = Array.from(game.macros).filter(m => m.folder && m.canExecute && (hasAncestorFolderName(m.folder?.ancestors) || hasAncestorFolderName(m.folder?.ancestors?.ancestors)));
        const foldersList = new Set(macros.map(m => m.folder));
        const folders = [];
        let folderArray = Array.from(foldersList);
        if (isFolder && isFolder.sorting !== "a") folderArray = folderArray.sort((a, b) => a.sort - b.sort);
        else folderArray = folderArray.sort((a, b) => a.name.localeCompare(b.name));
        for(let f of folderArray){
            const img = f.getFlag("macro-wheel", "folderImg") ?? "icons/svg/d20-grey.svg";
            const condition = f.getFlag("macro-wheel", "condition") ?? "";
            const presetCondition = f.getFlag("macro-wheel", "presetCondition") ?? "";
            const conditionToCheck = presetCondition || condition;
            const fulfilled = await this.evaluateCondition(conditionToCheck);
            let wMacros = macros.filter(m => m.folder === f);
            if(f.sorting === "a") wMacros = wMacros.sort((a, b) => a.name.localeCompare(b.name));
            if(!fulfilled) continue;
            folders.push({
                name: f.name,
                id: f.id,
                img: img,
                macros: wMacros
            });
        }
        const addHotbar = game.settings.get("macro-wheel", "addHotbar")
        if(addHotbar){
            for(let i=1; i<6; i++){
                const hotbarMacros = game.user.getHotbarMacros(i).filter(m => m.macro).map(m => m.macro);
                if(hotbarMacros.length){
                    folders.push({
                        name: `Hotbar ${i}`,
                        id: `hotbar${i}`,
                        img: this.macroBarIcons[i-1] ?? this.macroBarIcons[0],
                        macros: hotbarMacros
                    });
                }
            }
        }
        if(useRoot){
            const macrosWithoutFolder = Array.from(game.macros).filter(m => !m.folder && m.canExecute);
            if(macrosWithoutFolder.length){
                folders.push({
                    name: "Macros",
                    id: "root",
                    img: "icons/commodities/tech/cog-brass.webp",
                    macros: macrosWithoutFolder
                });
            }
        }

        for (let f of folders) { 
            for (let m of f.macros) {
                const condition = m.flags?.["macro-wheel"]?.activeCondition ?? "";
                m.mwIsToggle = !!condition;
                if (!m.mwIsToggle) continue;
                const fulfilled = await this.evaluateCondition(condition);
                m.mwToggle = fulfilled;
            }
        }

        const enableTooltips = game.settings.get("macro-wheel", "enableTooltips");
        return {folders, enableTooltips};
    }

    async evaluateCondition(condition){
        if(!condition) return true;
        const AsyncFunction = (function () {}).constructor;
        const fn = new AsyncFunction("return " + condition);
        let result = true;
        try {
            result = await fn();
        } catch (e) {
            console.error(e);
        }
        return result;
    }

    _onMacroClick(event){
        event.preventDefault();
        const macro = game.macros.get(event.currentTarget.dataset.macroId);
        if(macro){
            macro.execute();
        }
    }

    async close(){
        const offset = this.element[0].getBoundingClientRect();
        this.element[0].style.transformOrigin = `${this.clientX - offset.left}px ${this.clientY - offset.top}px`;
        this.element[0].classList.add("closing");
        this.element[0].ontransitionend = () => {
            super.close();
        }
    }
}