import { MODULE_ID } from "./main.js";
import { getBinSettingId } from "./settings.js";

export class RecycleBin extends Application {
    constructor() {
        super();
        this._bin = [];
        this._search = "";
        this.loadSavedBin();
        this.setHooks();
    }

    static get defaultOptions() {
        return foundry.utils.mergeObject(super.defaultOptions, {
            id: "recycle-bin",
            template: `modules/${MODULE_ID}/templates/recycle-bin.hbs`,
            popOut: true,
            resizable: true,
            minimizable: true,
            width: 400,
            height: 600,
            title: game.i18n.localize(`${MODULE_ID}.app.title`),
        });
    }

    get title() {
        return game.i18n.localize(`${MODULE_ID}.app.title`) + ` | ${this._bin.length}` + (this._bin.length > game.settings.get(MODULE_ID, "maxPermanentItems") ? " ❗" : "");
    }

    getItemName(item) {
        item = this.getItem(item) ?? item;
        if (!item) return "";
        return item.name ?? item.title ?? item.speaker?.alias ?? this.getItemImageName(item) ?? item._id ?? item.id;
    }

    getItemImageName(item) {
        const img = item.img ?? item.thumb ?? item.texture?.src ?? item.icon;
        if (!img) return null;
        return img.split("/").pop();
    }

    getItemImage(item) {
        const img = item.img ?? item.thumb ?? item.texture?.src ?? item.icon;
        if (img) return img;
        const layerName = CONFIG[item._documentName]?.layerClass?.layerOptions?.name;
        if (!layerName) return "fas fa-file";
        const icon = SceneControls.prototype._getControlButtons().find((c) => c.layer === layerName)?.icon;
        if (icon) return icon.replace("fas", "fad").replace("far", "fad").replace("fa-solid", "fa-duotone").replace("fa-regular", "fa-duotone");
        return "fas fa-file";
    }

    async getData() {
        const maxPermanentItems = game.settings.get(MODULE_ID, "maxPermanentItems");
        const items = [];
        const timeString = game.i18n.localize(`${MODULE_ID}.app.time`);
        for (const item of this._bin) {
            const howLongAgo = Date.now() - (item._deleteTime ?? Date.now());
            const date = new Date(howLongAgo);
            const DDHHMM = timeString.replace("{time}", `${date.getUTCDate() - 1}d ${date.getUTCHours()}h ${date.getUTCMinutes()}m`);
            const img = this.getItemImage(item);
            items.push({
                parent: item._parent ? `${this.getItemName(await fromUuid(item._parent)) ?? ""} | ` + item._parent : game.i18n.localize(`${MODULE_ID}.app.noParent`),
                name: this.getItemName(item),
                img: img,
                icon: !img.includes(".") ? img : null,
                documentName: item._documentName ?? "Unknown",
                index: this._bin.indexOf(item),
                time: DDHHMM,
                willBeDeleted: this._bin.indexOf(item) < this._bin.length - maxPermanentItems,
            });
        }
        items.reverse();
        return {
            items: items,
        };
    }

    activateListeners(html) {
        super.activateListeners(html);
        html = html[0];
        html.querySelectorAll(".restore-item").forEach((element) => {
            element.addEventListener("click", async (event) => {
                const item = parseInt(event.target.closest("li").dataset.index);
                await this.restoreItem(item);
            });
        });

        html.querySelectorAll(".preview-item").forEach((element) => {
            element.addEventListener("click", async (event) => {
                const item = parseInt(event.target.closest("li").dataset.index);
                await this.previewItem(item);
            });
        });

        html.querySelectorAll(".permanently-delete").forEach((element) => {
            element.addEventListener("click", async (event) => {
                const item = parseInt(event.target.closest("li").dataset.index);
                Dialog.confirm({
                    title: game.i18n.localize(`${MODULE_ID}.dialog.permanentlyDelete.title`) + ` | ${this.getItemName(item)}`,
                    content: `<p>${game.i18n.localize(`${MODULE_ID}.dialog.permanentlyDelete.content`)}</p>`,
                    yes: () => {
                        this.deleteForever(item);
                    },
                    no: () => {},
                    defaultYes: false,
                });
            });
        });

        html.querySelector("#search").value = this._search;
        this._onSearch();

        html.querySelector("#search").addEventListener("keyup", (event) => {
            const search = event.target.value.toLowerCase();
            this._search = search;
            this._onSearch();
        });
        html.querySelector("#search").addEventListener("search", (event) => {
            const search = event.target.value.toLowerCase();
            this._search = search;
            this._onSearch();
        });
    }

    _onSearch() {
        const search = this._search;
        this.element[0].querySelectorAll("li").forEach((element) => {
            if (!search) {
                element.classList.remove("hidden");
            } else if (element.dataset.search.toLowerCase().includes(search)) {
                element.classList.remove("hidden");
            } else {
                element.classList.add("hidden");
            }
        });
    }

    getItem(item) {
        if (Number.isInteger(item)) {
            item = this._bin[item];
        }
        if (!this._bin.includes(item)) {
            return null;
        }
        return item;
    }

    loadSavedBin() {
        this._bin = game.settings.get(MODULE_ID, getBinSettingId()) || [];
    }

    addItem(item) {
        const alreadyInBin = this._bin.find((i) => i._uuid === item._uuid);
        if (alreadyInBin) return;
        const maxTemporaryItems = game.settings.get(MODULE_ID, "maxTemporaryItems");
        this._bin.push(item);
        if (this._bin.length > maxTemporaryItems) {
            this._bin.shift();
        }
        this.savePermanentItems();
    }

    async restoreItem(item) {
        item = this.getItem(item);
        const documentName = item._documentName;
        const parent = item._parent;
        const time = item._deleteTime;
        const uuid = item._uuid;
        delete item._documentName;
        delete item._parent;
        delete item._deleteTime;
        delete item._uuid;
        if (parent) {
            let parentDocument = await fromUuid(parent);
            if (!parentDocument) {
                const binParent = this._bin.find((i) => i._uuid === parent);
                let proceed = false;
                if (binParent) {
                    proceed = await new Promise((resolve) => {
                        Dialog.confirm({
                            title: game.i18n.localize(`${MODULE_ID}.dialog.restoreParent.title`) + ` | ${this.getItemName(binParent)}`,
                            content: `<p>${game.i18n.localize(`${MODULE_ID}.dialog.restoreParent.content`)}</p>`,
                            yes: async () => {
                                await this.restoreItem(binParent);
                                resolve(true);
                            },
                            no: () => {
                                resolve(false);
                            },
                            defaultYes: true,
                            close: () => {
                                resolve(false);
                            },
                        });
                    });
                }
                if (!proceed) {
                    item._parent = parent;
                    item._documentName = documentName;
                    item._deleteTime = time;
                    item._uuid = uuid;
                    return ui.notifications.error(game.i18n.localize(`${MODULE_ID}.error.restore`).replace("{item}", `${documentName} - ${this.getItemName(item)}`) + ` ${parent}`);
                } else {
                    parentDocument = await fromUuid(parent);
                }
            }
            await parentDocument.createEmbeddedDocuments(documentName, [item], { keepId: true });
        } else {
            await CONFIG[documentName].documentClass.create(item, { keepId: true });
        }
        ui.notifications.info(game.i18n.localize(`${MODULE_ID}.success.restore`).replace("{item}", `${documentName} - ${this.getItemName(item)}`));
        this.deleteForever(item);
    }

    async previewItem(item) {
        item = this.getItem(item);
        try {
            item = JSON.parse(JSON.stringify(item));
            const documentName = item._documentName;
            const parent = item._parent;
            delete item._documentName;
            delete item._parent;
            delete item._deleteTime;
            delete item._uuid;
            if (parent) {
                const parentDocument = await fromUuid(parent);
                const parentClone = parentDocument.toObject();
                const parentCloneDoc = new CONFIG[parentDocument.documentName].documentClass(parentClone);
                const document = new DocumentsClasses[documentName](item, { parent: parentCloneDoc });
                document.sheet.render(true);
            } else {
                const document = new CONFIG[documentName].documentClass(item);
                document.sheet.render(true);
            }
        } catch (e) {
            ui.notifications.error(game.i18n.localize(`${MODULE_ID}.error.preview`)).replace("{item}", `${documentName} - ${this.getItemName(item)}`);
            console.error(e);
        }
    }

    deleteForever(item) {
        item = this.getItem(item);
        const index = this._bin.indexOf(item);
        if (index > -1) {
            this._bin.splice(index, 1);
        }
        this.savePermanentItems();
    }

    savePermanentItems() {
        const maxPermanentItems = game.settings.get(MODULE_ID, "maxPermanentItems");
        const permanentItems = this._bin.slice(-maxPermanentItems);
        game.settings.set(MODULE_ID, getBinSettingId(), permanentItems);
        if (this.rendered) {
            this.render(true);
        }
    }

    static setSidebarHook() {
        Hooks.on("renderSidebarTab", (app, html) => {
            if (!app instanceof Settings) return;
            html = html[0];
            const gd = html.querySelector("#game-details");
            if (!gd) return;
            const button = document.createElement("button");
            button.classList.add("recycle-bin");
            button.innerHTML = game.i18n.localize(`${MODULE_ID}.app.title`);
            button.addEventListener("click", () => {
                game.recycleBin.render(true);
            });
            gd.append(button);
        });
    }

    setHooks() {
        const deleteHook = game.user.isGM ? "delete" : "preDelete";
        const EMBEDDED_DOCUMENTS = this.getEmbeddedDocuments();
        [...CONST.WORLD_DOCUMENT_TYPES].concat(EMBEDDED_DOCUMENTS).forEach((documentType) => {
            Hooks.on(`${deleteHook}${documentType}`, (document, b, c) => {
                const data = document.toObject();
                data._parent = document?.parent?.uuid;
                data._documentName = document.documentName;
                data._deleteTime = Date.now();
                data._uuid = document.uuid;
                this.addItem(data);
            });
        });
    }

    getEmbeddedDocuments() {
        const embeddedDocs = new Set();
        CONST.WORLD_DOCUMENT_TYPES.forEach((documentType) => {
            Object.values(CONFIG[documentType].documentClass.hierarchy).forEach((d) => {
                embeddedDocs.add(d?.element?.documentName);
                DocumentsClasses[d?.element?.documentName] = d?.element?.implementation;
            });
        });
        const embeddedDocsArray = Array.from(embeddedDocs);
        return embeddedDocsArray.filter((d) => d);
    }
}

const DocumentsClasses = {};
