import { MODULE_ID } from "../main.js";
import { HandlebarsApplication, l, mergeClone } from "../lib/utils.js";

export class TokenNotes extends HandlebarsApplication {
    constructor(object) {
        super();
        this.object = object ?? null;
    }

    #editMode = false;

    #useActor = false;

    get actor() {
        return game.actors.get(this.object.document.actorId);
    }

    get useActor() {
        return this.#useActor;
    }

    get editMode() {
        return this.#editMode;
    }

    static get DEFAULT_OPTIONS() {
        return mergeClone(super.DEFAULT_OPTIONS, {
            classes: [this.APP_ID],
            id: this.APP_ID,
            window: {
                title: `${MODULE_ID}.${this.APP_ID}.title`,
                icon: "",
                resizable: true,
                savePosition: true,
            },
            position: {},
            actions: {
                toggleEdit: this.#toggleEdit,
                toggleTokenActor: this.#toggleTokenActor,
            },
        });
    }

    static get PARTS() {
        return {
            content: {
                template: `modules/${MODULE_ID}/templates/${this.APP_ID}.hbs`,
                classes: [],
                scrollable: [],
            },
        };
    }

    get title() {
        const tokenActorText = this.useActor ? l("DOCUMENT.Actor") : l("DOCUMENT.Token");
        return this.object.document.name + ` (${tokenActorText})`;
    }

    async _prepareContext(options) {
        const data = {};
        return { data };
    }

    bind(object) {
        const previousObject = this.object;
        this.object = object;
        if(!object) return this.close();
        if (previousObject !== object) this.render(true, { window: { title: this.title } });
    }

    _onRender(context, options) {
        super._onRender(context, options);
        const html = this.element;
        const notes = this.useActor ? this.actor.getFlag("token-notes", "notes") ?? "" : this.object.document.getFlag("token-notes", "notes") ?? "";
        const textarea = html.querySelector("textarea");
        const readarea = html.querySelector(".token-notes-text-read");
        this.textarea = textarea;
        this.readarea = readarea;
        this.readOnly = game.settings.get("token-notes", "allowplayers") === 1 && !game.user.isGM;
        textarea.value = notes;
        this._generateReadArea();
        this.textarea.classList.add("hidden");
    }

    static #toggleEdit(event) {
        event.preventDefault();
        this._toggleNoteState();
        this._save();
        //switch icon
        const icon = this.editMode ? "fa-comment" : "fa-edit";
        const oldIcon = this.editMode ? "fa-edit" : "fa-comment";
        this.element.querySelector('.header-control[data-action="toggleEdit"]').classList.replace(oldIcon, icon);
    }

    static async #toggleTokenActor(event) {
        event.preventDefault();
        const oldIcon = this.useActor ? "fa-user" : "fa-user-circle";
        await this._save();
        await this._toggleTokenActor();
        const icon = this.useActor ? "fa-user" : "fa-user-circle";
        this.element.querySelector('.header-control[data-action="toggleTokenActor"]').classList.replace(oldIcon, icon);
    }

    _save() {
        if (this.useActor) return this.actor.setFlag("token-notes", "notes", this.textarea.value);
        return this.object.document.setFlag("token-notes", "notes", this.textarea.value);
    }

    async _toggleTokenActor() {
        this.#useActor = !this.#useActor;
        await this.object.document.setFlag("token-notes", "useActor", this.#useActor);
        this.render(true, { window: { title: this.title } });
    }

    _toggleNoteState() {
        if (!this.textarea.classList.contains("hidden")) {
            this._readMode();
        } else {
            this._editMode();
        }
    }

    _editMode() {
        this.#editMode = true;
        this._updateTextArea();
        this.textarea.classList.remove("hidden");
        this.readarea.classList.add("hidden");
    }

    _readMode() {
        this.#editMode = false;
        this.textarea.classList.add("hidden");
        this.readarea.classList.remove("hidden");
        this._generateReadArea();
    }

    _generateReadArea() {
        const text = this.textarea.value;
        const lines = text.split("\n");
        let parts = [];
        for (let line of lines) {
            parts = parts.concat(line.split(" ").filter((p) => p.length > 0));
            parts.push("\n");
        }
        parts.pop();
        let content = "";
        for (let part of parts) {
            const isNumber = part.match(/^\d+$/) != null;
            const number = isNumber ? parseInt(part) : undefined;
            const length = part.length;
            if (part == "\n") {
                content += "<hr>";
                continue;
            }
            if (isNumber) {
                content += `<input type="number" value="${number}" min="0" style="width:${length + 1}rem;">`;
            } else {
                content += `<span class="token-notes-text-read-word">${part}</span>`;
            }
        }
        this.readarea.innerHTML = content;
        this.readarea.querySelectorAll("input").forEach((e) => e.addEventListener("change", this._updateTextArea.bind(this)));
    }

    _updateTextArea() {
        let text = "";
        for (let c of this.readarea.children) {
            if (c.nodeName == "HR") {
                text += "\n";
            } else {
                text += c.value ? c.value : c.innerText;
                text += " ";
            }
        }
        text.trim();
        this.textarea.value = text;
        this._save();
    }

    async _onDrop(event) {
        event.preventDefault();
    }

    _onClose(options) {
        this._save();
        super._onClose(options);
    }

    async _renderFrame(options) {
        const frame = await super._renderFrame(options);
        if (!this.readOnly) {
            const toggleEditLabel = l("tokennotes.tooltip.toggle-edit");
            const icon = !this.editMode ? "fa-solid fa-edit" : "fa-solid fa-comment";
            const toggleEdit = `<button type="button" class="header-control ${icon}" data-action="toggleEdit"
                                  data-tooltip="${toggleEditLabel}" aria-label="${toggleEditLabel}"></button>`;
            this.window.close.insertAdjacentHTML("beforebegin", toggleEdit);
        }
        //toggle token\actor button
        const toggleTokenActorLabel = l("tokennotes.tooltip.toggle-token-actor");
        const taIcon = !this.useActor ? "fa-solid fa-user-circle" : "fa-solid fa-user";
        const ta = `<button type="button" class="header-control ${taIcon}" data-action="toggleTokenActor"
                          data-tooltip="${toggleTokenActorLabel}" aria-label="${toggleTokenActorLabel}"></button>`;
        this.window.close.insertAdjacentHTML("beforebegin", ta);
        return frame;
    }

    static checkPermission(token) {
        const permissionLevel = game.settings.get("token-notes", "allowplayers");
        if (game.user.isGM) return true;
        if (!token.isOwner) return false;
        if (permissionLevel == 0) return false;
        if (permissionLevel >= 1) return true;
    }
}
