import { MODULE_ID } from "../main.js";
import { HandlebarsApplication } from "../lib/utils.js";
import { mergeClone } from "../lib/utils.js";

export const AUTO_FLIP = {
    NONE: 0,
    IN_COMBAT: 1,
    OUT_OF_COMBAT: 2,
};

export class TokenFlipConfig extends HandlebarsApplication {
    constructor(token, app) {
        super(token, app);
        this.object = token;
        this.actor = app.actor;
        this.app = app;
        this.isPrototype = this.app.isPrototype;
    }

    static get DEFAULT_OPTIONS() {
        return mergeClone(super.DEFAULT_OPTIONS, {
            classes: [this.APP_ID],
            tag: "form",
            id: this.APP_ID,
            window: {
                title: `${MODULE_ID}.${this.APP_ID}.title`,
                icon: "",
                resizable: true,
                contentTag: "section",
            },
            position: {
                width: 450,
                height: "auto",
            },
        });
    }

    static get PARTS() {
        return {
            content: {
                template: `modules/${MODULE_ID}/templates/${this.APP_ID}.hbs`,
                classes: [],
                scrollable: [".scrollable"],
            },
        };
    }

    get AUTO_FLIP_OPTIONS() {
        return {
            0: `${MODULE_ID}.token-flip-config.autoFlip.none`,
            1: `${MODULE_ID}.token-flip-config.autoFlip.inCombat`,
            2: `${MODULE_ID}.token-flip-config.autoFlip.outOfCombat`,
        };
    }

    get TRANSITION_OPTIONS() {
        const options = ["flip", ...Object.values(TextureTransitionFilter.TYPES).sort((a, b) => a.localeCompare(b))];
        return options.reduce((acc, val) => {
            acc[val] = val;
            acc[val] = acc[val].charAt(0).toUpperCase() + acc[val].slice(1);
            return acc;
        }, {});
    }

    async _prepareContext(options) {
        let faces = this.currentData;
        const actors = Array.from(game.actors).filter(a=>a.isOwner).map((a) => ({ name: a.name, id: a.id }));
        if (!faces.length) {
            faces = [
                this.newFace,
            ];
            await this.object.update({ flags: { tokenflip: { tokenfaces: faces } } });
        } else {
            faces.forEach((f) => {
                if (f.model3d === undefined) f.model3d = this.object.flags["levels-3d-preview"]?.model3d ?? "";
                if (!f.width) f.width = this.object.width;
                if (!f.height) f.height = this.object.height;
            });
        }

        faces = foundry.utils.deepClone(faces);

        return {
            actors,
            tokenFaces: faces,
            isPrototype: this.isPrototype,
            autoFlipOptions: this.AUTO_FLIP_OPTIONS,
            transitionOptions: this.TRANSITION_OPTIONS,
        };
    }

    _onRender(context, options) {
        super._onRender(context, options);
        const html = this.element;

        this.element.querySelector("button#add").addEventListener("click", this._onAdd.bind(this));
        this.element.querySelectorAll("button.delete").forEach((button) => {
            button.addEventListener("click", this._onDelete.bind(this));
        });
        this.element.querySelectorAll("button.expand-collapse").forEach((button) => {
            button.addEventListener("click", (e) => button.closest("fieldset").classList.toggle("collapsed"));
        });
        this.element.querySelectorAll("a.document-id-link").forEach((link) => {
            link.addEventListener("click", (e) => {
                e.preventDefault();
                const id = link.dataset.id;
                game.clipboard.copyPlainText(id);
                ui.notifications.info(game.i18n.localize("tokenflip.token-flip-config.copiedId"));
            });
        });
    }

    get currentData() {
        if (this.isPrototype) return this.actor.prototypeToken.flags?.tokenflip?.tokenfaces ?? [];
        return this.object.flags?.tokenflip?.tokenfaces ?? [];
    }

    async makeRandomActorUpdate() {
        if (!this.actor || !this.isPrototype) return;
        await this.actor.update({
            flags: {
                tokenflip: { unfuck: !this.actor.flags?.tokenflip?.unfuck },
            },
        });
        return;
    }

    async _onDelete(e) {
        e.currentTarget.closest("li").remove();
        await this.saveData();
        await this.makeRandomActorUpdate();
        this.render(true);
    }

    get newFace() {
        return {
            actorId: this.object.actor.id,
            img: this.object.texture.src,
            subject: this.object.ring.subject.texture,
            id: foundry.utils.randomID(20),
            scaleX: this.object.texture.scaleX,
            scaleY: this.object.texture.scaleY,
            width: this.object.width,
            height: this.object.height,
            model3d: this.object.flags["levels-3d-preview"]?.model3d ?? "",
            autoFlip: 0,
            transition: "flip",
            autoFlipMacro: "",
        };
    }

    async _onAdd(e) {
        e.preventDefault();
        const data = this.currentData;
        data.push(this.newFace);
        await this.object.update({ flags: { tokenflip: { tokenfaces: data } } });
        await this.makeRandomActorUpdate();
        this.render(true);
    }

    async close(...args) {
        this.saveData();
        return super.close(...args);
    }

    async _updateObject() {
        return await this.saveData();
    }

    async saveData() {
        const ul = this.element.querySelector("ul");
        //loop ul
        let data = [];
        for (const li of ul.children) {
            const actorId = li.querySelector(`select`).value;
            const img = li.querySelector(`[name="img"]`).value;
            const subject = li.querySelector(`[name="subject"]`).value;
            const id = li.id;
            const scaleX = parseFloat(li.querySelector(`[name="scaleX"]`).value);
            const scaleY = parseFloat(li.querySelector(`[name="scaleY"]`).value);
            const width = parseFloat(li.querySelector(`[name="width"]`).value);
            const height = parseFloat(li.querySelector(`[name="height"]`).value);
            const model3d = li.querySelector(`[name="model3d"]`).value;
            const autoFlip = parseInt(li.querySelector(`[name="autoFlip"]`).value);
            const transition = li.querySelector(`[name="transition"]`).value;
            const autoFlipMacro = li.querySelector(`[name="autoFlipMacro"]`).value;
            data.push({ actorId, img, id, scaleX, scaleY, width, height, model3d, subject, autoFlip, transition, autoFlipMacro });
        }
        await this.object.update({ flags: { tokenflip: { tokenfaces: data } } });
        return await this.makeRandomActorUpdate();
    }
}
