import { i as inputRules, e as ellipsis, w as wrappingInputRule, t as textblockTypeInputRule, I as InputRule, k as keymap, u as undo, r as redo, a as undoInputRule, j as joinUp, b as joinDown, l as lift, s as selectParentNode, c as toggleMark, d as wrapInList, f as liftListItem, g as sinkListItem, h as setBlockType, m as chainCommands, n as exitCode, P as Plugin, o as wrapIn, p as deleteTable, q as addColumnAfter, v as addColumnBefore, x as deleteColumn, y as addRowAfter, z as addRowBefore, A as deleteRow, B as mergeCells, C as splitCell, T as TextSelection, D as liftTarget, E as autoJoin, R as ResolvedPos, F as tableNodes, S as Schema, G as splitListItem, H as DOMParser$2, J as DOMSerializer, K as Slice, L as tableEditing, M as columnResizing, N as history$1, O as gapCursor, Q as dropCursor, U as baseKeymap, V as AllSelection, W as EditorState, X as EditorView, Y as PluginKey, Z as Step, _ as index, $ as index$1, a0 as index$2, a1 as index$3, a2 as index$4, a3 as index$5, a4 as index$6, a5 as basicSetup, a6 as markdown, a7 as json, a8 as linter, a9 as javascript, aa as lintGutter, ab as esLint, ac as Linter, ad as html, ae as syntaxHighlighting, af as HighlightStyle, ag as tags, ah as markdownLanguage, ai as jsonLanguage, aj as javascriptLanguage, ak as htmlLanguage, al as jsonParseLinter, am as indentUnit, an as keymap$1, ao as indentWithTab, ap as EditorView$1, aq as Compartment, ar as EditorSelection } from './vendor.mjs';

/**
 * @import {ColorSource} from "../_types.mjs";
 */

/**
 * A representation of a color in hexadecimal format.
 * This class provides methods for transformations and manipulations of colors.
 */
class Color extends Number {

  /**
   * Is this a valid color?
   * @type {boolean}
   */
  get valid() {
    const v = this.valueOf();
    return Number.isInteger(v) && v >= 0 && v <= 0xFFFFFF;
  }

  /* ------------------------------------------ */

  /**
   * A CSS-compatible color string.
   * If this color is not valid, the empty string is returned.
   * An alias for Color#toString.
   * @type {string}
   */
  get css() {
    return this.toString(16);
  }

  /* ------------------------------------------ */

  /**
   * The color represented as an RGB array.
   * @type {[number, number, number]}
   */
  get rgb() {
    return [((this >> 16) & 0xFF) / 255, ((this >> 8) & 0xFF) / 255, (this & 0xFF) / 255];
  }

  /* ------------------------------------------ */

  /**
   * The numeric value of the red channel between [0, 1].
   * @type {number}
   */
  get r() {
    return ((this >> 16) & 0xFF) / 255;
  }

  /* ------------------------------------------ */

  /**
   * The numeric value of the green channel between [0, 1].
   * @type {number}
   */
  get g() {
    return ((this >> 8) & 0xFF) / 255;
  }

  /* ------------------------------------------ */

  /**
   * The numeric value of the blue channel between [0, 1].
   * @type {number}
   */
  get b() {
    return (this & 0xFF) / 255;
  }

  /* ------------------------------------------ */

  /**
   * The maximum value of all channels.
   * @type {number}
   */
  get maximum() {
    return Math.max(...this);
  }

  /* ------------------------------------------ */

  /**
   * The minimum value of all channels.
   * @type {number}
   */
  get minimum() {
    return Math.min(...this);
  }

  /* ------------------------------------------ */

  /**
   * Get the value of this color in little endian format.
   * @type {number}
   */
  get littleEndian() {
    return ((this >> 16) & 0xFF) + (this & 0x00FF00) + ((this & 0xFF) << 16);
  }

  /* ------------------------------------------ */

  /**
   * The color represented as an HSV array.
   * Conversion formula adapted from http://en.wikipedia.org/wiki/HSV_color_space.
   * Assumes r, g, and b are contained in the set [0, 1] and returns h, s, and v in the set [0, 1].
   * @type {[number, number, number]}
   */
  get hsv() {
    const [r, g, b] = this.rgb;
    const max = Math.max(r, g, b);
    const min = Math.min(r, g, b);
    const d = max - min;

    let h;
    const s = max === 0 ? 0 : d / max;
    const v = max;

    // Achromatic colors
    if (max === min) return [0, s, v];

    // Normal colors
    switch (max) {
      case r: h = ((g - b) / d) + (g < b ? 6 : 0); break;
      case g: h = ((b - r) / d) + 2; break;
      case b: h = ((r - g) / d) + 4; break;
    }
    h /= 6;
    return [h, s, v];
  }

  /* ------------------------------------------ */

  /**
   * The color represented as an HSL array.
   * Assumes r, g, and b are contained in the set [0, 1] and returns h, s, and l in the set [0, 1].
   * @type {[number, number, number]}
   */
  get hsl() {
    const [r, g, b] = this.rgb;

    // Compute luminosity, saturation and hue
    const l = Math.max(r, g, b);
    const s = l - Math.min(r, g, b);
    let h = 0;
    if ( s > 0 ) {
      if ( l === r ) {
        h = (g - b) / s;
      } else if ( l === g ) {
        h = 2 + ((b - r) / s);
      } else {
        h = 4 + ((r - g) / s);
      }
    }
    const finalHue = (60 * h < 0 ? (60 * h) + 360 : 60 * h) / 360;
    const finalSaturation = s ? (l <= 0.5 ? s / ((2 * l) - s) : s / (2 - ((2 * l) - s))) : 0;
    const finalLuminance = ((2 * l) - s) / 2;
    return [finalHue, finalSaturation, finalLuminance];
  }

  /* ------------------------------------------ */

  /**
   * The color represented as a linear RGB array.
   * Assumes r, g, and b are contained in the set [0, 1] and returns linear r, g, and b in the set [0, 1].
   * @see {@link https://en.wikipedia.org/wiki/SRGB#Transformation}
   * @type {Color}
   */
  get linear() {
    const toLinear = c => (c > 0.04045) ? Math.pow((c + 0.055) / 1.055, 2.4) : (c / 12.92);
    return this.constructor.fromRGB([toLinear(this.r), toLinear(this.g), toLinear(this.b)]);
  }

  /* ------------------------------------------ */
  /*  Color Manipulation Methods                */
  /* ------------------------------------------ */

  /** @override */
  toString(radix) {
    if ( !this.valid ) return "";
    return `#${super.toString(16).padStart(6, "0")}`;
  }

  /* ------------------------------------------ */

  /**
   * Serialize the Color.
   * @returns {string}    The color as a CSS string
   */
  toJSON() {
    return this.css;
  }

  /* ------------------------------------------ */

  /**
   * Returns the color as a CSS string.
   * @returns {string}    The color as a CSS string
   */
  toHTML() {
    return this.css;
  }

  /* ------------------------------------------ */

  /**
   * Test whether this color equals some other color
   * @param {Color|number} other  Some other color or hex number
   * @returns {boolean}           Are the colors equal?
   */
  equals(other) {
    return this.valueOf() === other.valueOf();
  }

  /* ------------------------------------------ */

  /**
   * Get a CSS-compatible RGBA color string.
   * @param {number} alpha      The desired alpha in the range [0, 1]
   * @returns {string}          A CSS-compatible RGBA string
   */
  toRGBA(alpha) {
    const rgba = [(this >> 16) & 0xFF, (this >> 8) & 0xFF, this & 0xFF, alpha];
    return `rgba(${rgba.join(", ")})`;
  }

  /* ------------------------------------------ */

  /**
   * Mix this Color with some other Color using a provided interpolation weight.
   * @param {Color} other       Some other Color to mix with
   * @param {number} weight     The mixing weight placed on this color where weight is placed on the other color
   * @returns {Color}           The resulting mixed Color
   */
  mix(other, weight) {
    return new Color(Color.mix(this, other, weight));
  }

  /* ------------------------------------------ */

  /**
   * Multiply this Color by another Color or a static scalar.
   * @param {Color|number} other  Some other Color or a static scalar.
   * @returns {Color}             The resulting Color.
   */
  multiply(other) {
    if ( other instanceof Color ) return new Color(Color.multiply(this, other));
    return new Color(Color.multiplyScalar(this, other));
  }

  /* ------------------------------------------ */

  /**
   * Add this Color by another Color or a static scalar.
   * @param {Color|number} other  Some other Color or a static scalar.
   * @returns {Color}             The resulting Color.
   */
  add(other) {
    if ( other instanceof Color ) return new Color(Color.add(this, other));
    return new Color(Color.addScalar(this, other));
  }

  /* ------------------------------------------ */

  /**
   * Subtract this Color by another Color or a static scalar.
   * @param {Color|number} other  Some other Color or a static scalar.
   * @returns {Color}             The resulting Color.
   */
  subtract(other) {
    if ( other instanceof Color ) return new Color(Color.subtract(this, other));
    return new Color(Color.subtractScalar(this, other));
  }

  /* ------------------------------------------ */

  /**
   * Max this color by another Color or a static scalar.
   * @param {Color|number} other  Some other Color or a static scalar.
   * @returns {Color}             The resulting Color.
   */
  maximize(other) {
    if ( other instanceof Color ) return new Color(Color.maximize(this, other));
    return new Color(Color.maximizeScalar(this, other));
  }

  /* ------------------------------------------ */

  /**
   * Min this color by another Color or a static scalar.
   * @param {Color|number} other  Some other Color or a static scalar.
   * @returns {Color}             The resulting Color.
   */
  minimize(other) {
    if ( other instanceof Color ) return new Color(Color.minimize(this, other));
    return new Color(Color.minimizeScalar(this, other));
  }

  /* ------------------------------------------ */
  /*  Iterator                                  */
  /* ------------------------------------------ */

  /**
   * Iterating over a Color is equivalent to iterating over its [r,g,b] color channels.
   * @returns {Generator<number>}
   */
  *[Symbol.iterator]() {
    yield this.r;
    yield this.g;
    yield this.b;
  }

  /* ------------------------------------------------------------------------------------------- */
  /*                      Real-time performance Methods and Properties                           */
  /*  Important Note:                                                                            */
  /*  These methods are not a replacement, but a tool when real-time performance is needed.      */
  /*  They do not have the flexibility of the "classic" methods and come with some limitations.  */
  /*  Unless you have to deal with real-time performance, you should use the "classic" methods.  */
  /* ------------------------------------------------------------------------------------------- */

  /**
   * Set an rgb array with the rgb values contained in this Color class.
   * @param {number[]} vec3  Receive the result. Must be an array with at least a length of 3.
   */
  applyRGB(vec3) {
    vec3[0] = ((this >> 16) & 0xFF) / 255;
    vec3[1] = ((this >> 8) & 0xFF) / 255;
    vec3[2] = (this & 0xFF) / 255;
  }

  /* ------------------------------------------ */

  /**
   * Apply a linear interpolation between two colors, according to the weight.
   * @param {number}        color1       The first color to mix.
   * @param {number}        color2       The second color to mix.
   * @param {number}        weight       Weight of the linear interpolation.
   * @returns {number}                   The resulting mixed color
   */
  static mix(color1, color2, weight) {
    return ((((((color1 >> 16) & 0xFF) * (1 - weight)) + (((color2 >> 16) & 0xFF) * weight)) << 16) & 0xFF0000)
      | ((((((color1 >> 8) & 0xFF) * (1 - weight)) + (((color2 >> 8) & 0xFF) * weight)) << 8) & 0x00FF00)
      | ((((color1 & 0xFF) * (1 - weight)) + ((color2 & 0xFF) * weight)) & 0x0000FF);
  }

  /* ------------------------------------------ */

  /**
   * Multiply two colors.
   * @param {number}        color1       The first color to multiply.
   * @param {number}        color2       The second color to multiply.
   * @returns {number}                   The result.
   */
  static multiply(color1, color2) {
    return ((((color1 >> 16) & 0xFF) / 255 * ((color2 >> 16) & 0xFF) / 255) * 255 << 16)
      | ((((color1 >> 8) & 0xFF) / 255 * ((color2 >> 8) & 0xFF) / 255) * 255 << 8)
      | (((color1 & 0xFF) / 255 * ((color2 & 0xFF) / 255)) * 255);
  }

  /* ------------------------------------------ */

  /**
   * Multiply a color by a scalar
   * @param {number} color        The color to multiply.
   * @param {number} scalar       A static scalar to multiply with.
   * @returns {number}            The resulting color as a number.
   */
  static multiplyScalar(color, scalar) {
    return (Math.clamp(((color >> 16) & 0xFF) / 255 * scalar, 0, 1) * 255 << 16)
      | (Math.clamp(((color >> 8) & 0xFF) / 255 * scalar, 0, 1) * 255 << 8)
      | (Math.clamp((color & 0xFF) / 255 * scalar, 0, 1) * 255);
  }

  /* ------------------------------------------ */

  /**
   * Maximize two colors.
   * @param {number}        color1       The first color.
   * @param {number}        color2       The second color.
   * @returns {number}                   The result.
   */
  static maximize(color1, color2) {
    return (Math.clamp(Math.max((color1 >> 16) & 0xFF, (color2 >> 16) & 0xFF), 0, 0xFF) << 16)
      | (Math.clamp(Math.max((color1 >> 8) & 0xFF, (color2 >> 8) & 0xFF), 0, 0xFF) << 8)
      | Math.clamp(Math.max(color1 & 0xFF, color2 & 0xFF), 0, 0xFF);
  }

  /* ------------------------------------------ */

  /**
   * Maximize a color by a static scalar.
   * @param {number} color         The color to maximize.
   * @param {number} scalar        Scalar to maximize with (normalized).
   * @returns {number}             The resulting color as a number.
   */
  static maximizeScalar(color, scalar) {
    return (Math.clamp(Math.max((color >> 16) & 0xFF, scalar * 255), 0, 0xFF) << 16)
      | (Math.clamp(Math.max((color >> 8) & 0xFF, scalar * 255), 0, 0xFF) << 8)
      | Math.clamp(Math.max(color & 0xFF, scalar * 255), 0, 0xFF);
  }

  /* ------------------------------------------ */

  /**
   * Add two colors.
   * @param {number}        color1       The first color.
   * @param {number}        color2       The second color.
   * @returns {number}                   The resulting color as a number.
   */
  static add(color1, color2) {
    return (Math.clamp((((color1 >> 16) & 0xFF) + ((color2 >> 16) & 0xFF)), 0, 0xFF) << 16)
      | (Math.clamp((((color1 >> 8) & 0xFF) + ((color2 >> 8) & 0xFF)), 0, 0xFF) << 8)
      | Math.clamp(((color1 & 0xFF) + (color2 & 0xFF)), 0, 0xFF);
  }

  /* ------------------------------------------ */

  /**
   * Add a static scalar to a color.
   * @param {number} color         The color.
   * @param {number} scalar        Scalar to add with (normalized).
   * @returns {number}             The resulting color as a number.
   */
  static addScalar(color, scalar) {
    return (Math.clamp((((color >> 16) & 0xFF) + (scalar * 255)), 0, 0xFF) << 16)
      | (Math.clamp((((color >> 8) & 0xFF) + (scalar * 255)), 0, 0xFF) << 8)
      | Math.clamp(((color & 0xFF) + (scalar * 255)), 0, 0xFF);
  }

  /* ------------------------------------------ */

  /**
   * Subtract two colors.
   * @param {number}        color1       The first color.
   * @param {number}        color2       The second color.
   */
  static subtract(color1, color2) {
    return (Math.clamp((((color1 >> 16) & 0xFF) - ((color2 >> 16) & 0xFF)), 0, 0xFF) << 16)
      | (Math.clamp((((color1 >> 8) & 0xFF) - ((color2 >> 8) & 0xFF)), 0, 0xFF) << 8)
      | Math.clamp(((color1 & 0xFF) - (color2 & 0xFF)), 0, 0xFF);
  }

  /* ------------------------------------------ */

  /**
   * Subtract a color by a static scalar.
   * @param {number} color         The color.
   * @param {number} scalar        Scalar to subtract with (normalized).
   * @returns {number}             The resulting color as a number.
   */
  static subtractScalar(color, scalar) {
    return (Math.clamp((((color >> 16) & 0xFF) - (scalar * 255)), 0, 0xFF) << 16)
      | (Math.clamp((((color >> 8) & 0xFF) - (scalar * 255)), 0, 0xFF) << 8)
      | Math.clamp(((color & 0xFF) - (scalar * 255)), 0, 0xFF);
  }

  /* ------------------------------------------ */

  /**
   * Minimize two colors.
   * @param {number}        color1       The first color.
   * @param {number}        color2       The second color.
   */
  static minimize(color1, color2) {
    return (Math.clamp(Math.min((color1 >> 16) & 0xFF, (color2 >> 16) & 0xFF), 0, 0xFF) << 16)
      | (Math.clamp(Math.min((color1 >> 8) & 0xFF, (color2 >> 8) & 0xFF), 0, 0xFF) << 8)
      | Math.clamp(Math.min(color1 & 0xFF, color2 & 0xFF), 0, 0xFF);
  }

  /* ------------------------------------------ */

  /**
   * Minimize a color by a static scalar.
   * @param {number} color         The color.
   * @param {number} scalar        Scalar to minimize with (normalized).
   */
  static minimizeScalar(color, scalar) {
    return (Math.clamp(Math.min((color >> 16) & 0xFF, scalar * 255), 0, 0xFF) << 16)
      | (Math.clamp(Math.min((color >> 8) & 0xFF, scalar * 255), 0, 0xFF) << 8)
      | Math.clamp(Math.min(color & 0xFF, scalar * 255), 0, 0xFF);
  }

  /* ------------------------------------------ */

  /**
   * Convert a color to RGB and assign values to a passed array.
   * @param {number} color   The color to convert to RGB values.
   * @param {number[]} vec3  Receive the result. Must be an array with at least a length of 3.
   */
  static applyRGB(color, vec3) {
    vec3[0] = ((color >> 16) & 0xFF) / 255;
    vec3[1] = ((color >> 8) & 0xFF) / 255;
    vec3[2] = (color & 0xFF) / 255;
  }

  /* ------------------------------------------ */
  /*  Factory Methods                           */
  /* ------------------------------------------ */

  /**
   * Create a Color instance from an RGB array.
   * @param {ColorSource} color     A color input
   * @returns {Color}               The hex color instance or NaN
   */
  static from(color) {
    if ( (color === null) || (color === undefined) ) return new this(NaN);
    if ( typeof color === "string" ) return this.fromString(color);
    if ( typeof color === "number" ) return new this(color);
    if ( (color instanceof Array) && (color.length === 3) ) return this.fromRGB(color);
    if ( color instanceof Color ) return color;
    return new this(color);
  }

  /* ------------------------------------------ */

  /**
   * Create a Color instance from a color string which either includes or does not include a leading #.
   * @param {string} color                      A color string
   * @returns {Color}                           The hex color instance
   */
  static fromString(color) {
    return new this(parseInt(color.startsWith("#") ? color.substring(1) : color, 16));
  }

  /* ------------------------------------------ */

  /**
   * Create a Color instance from an RGB array.
   * @param {[number, number, number]} rgb      An RGB tuple
   * @returns {Color}                           The hex color instance
   */
  static fromRGB(rgb) {
    return new this(((rgb[0] * 255) << 16) + ((rgb[1] * 255) << 8) + (rgb[2] * 255 | 0));
  }

  /* ------------------------------------------ */

  /**
   * Create a Color instance from an RGB normalized values.
   * @param {number} r                          The red value
   * @param {number} g                          The green value
   * @param {number} b                          The blue value
   * @returns {Color}                           The hex color instance
   */
  static fromRGBvalues(r, g, b) {
    return new this(((r * 255) << 16) + ((g * 255) << 8) + (b * 255 | 0));
  }

  /* ------------------------------------------ */

  /**
   * Create a Color instance from an HSV array.
   * Conversion formula adapted from http://en.wikipedia.org/wiki/HSV_color_space.
   * Assumes h, s, and v are contained in the set [0, 1].
   * @param {[number, number, number]} hsv      An HSV tuple
   * @returns {Color}                           The hex color instance
   */
  static fromHSV(hsv) {
    const [h, s, v] = hsv;
    const i = Math.floor(h * 6);
    const f = (h * 6) - i;
    const p = v * (1 - s);
    const q = v * (1 - (f * s));
    const t = v * (1 - ((1 - f) * s));
    let rgb;
    switch (i % 6) {
      case 0: rgb = [v, t, p]; break;
      case 1: rgb = [q, v, p]; break;
      case 2: rgb = [p, v, t]; break;
      case 3: rgb = [p, q, v]; break;
      case 4: rgb = [t, p, v]; break;
      case 5: rgb = [v, p, q]; break;
    }
    return this.fromRGB(rgb);
  }

  /* ------------------------------------------ */

  /**
   * Create a Color instance from an HSL array.
   * Assumes h, s, and l are contained in the set [0, 1].
   * @param {[number, number, number]} hsl      An HSL tuple
   * @returns {Color}                           The hex color instance
   */
  static fromHSL(hsl) {
    const [h, s, l] = hsl;

    // Calculate intermediate values for the RGB components
    const chroma = (1 - Math.abs((2 * l) - 1)) * s;
    const hue = h * 6;
    const x = chroma * (1 - Math.abs((hue % 2) - 1));
    const m = l - (chroma / 2);

    let r;
    let g;
    let b;
    switch (Math.floor(hue)) {
      case 0: [r, g, b] = [chroma, x, 0]; break;
      case 1: [r, g, b] = [x, chroma, 0]; break;
      case 2: [r, g, b] = [0, chroma, x]; break;
      case 3: [r, g, b] = [0, x, chroma]; break;
      case 4: [r, g, b] = [x, 0, chroma]; break;
      case 5:
      case 6: [r, g, b] = [chroma, 0, x]; break;
      default: [r, g, b] = [0, 0, 0]; break;
    }

    // Adjust for luminance
    r += m;
    g += m;
    b += m;
    return this.fromRGB([r, g, b]);
  }

  /* ------------------------------------------ */

  /**
   * Create a Color instance (sRGB) from a linear rgb array.
   * Assumes r, g, and b are contained in the set [0, 1].
   * @see {@link https://en.wikipedia.org/wiki/SRGB#Transformation}
   * @param {[number, number, number]} linear   The linear rgb array
   * @returns {Color}                           The hex color instance
   */
  static fromLinearRGB(linear) {
    const [r, g, b] = linear;
    const tosrgb = c => (c <= 0.0031308) ? (12.92 * c) : ((1.055 * Math.pow(c, 1 / 2.4)) - 0.055);
    return this.fromRGB([tosrgb(r), tosrgb(g), tosrgb(b)]);
  }
}

/** @module helpers */


/**
 * @import {DeepReadonly} from "../_types.mjs";
 * @import {ResolvedUUID} from "./_types.mjs";
 */

/**
 * Benchmark the performance of a function, calling it a requested number of iterations.
 * @param {Function} func       The function to benchmark
 * @param {number} iterations   The number of iterations to test
 * @param {...any} args         Additional arguments passed to the benchmarked function
 */
async function benchmark(func, iterations, ...args) {
  const start = performance.now();
  for ( let i=0; i<iterations; i++ ) {
    await func(...args, i);
  }
  const end = performance.now();
  const t = Math.round((end - start) * 100) / 100;
  const name = func.name ?? "Evaluated Function";
  console.log(`${name} | ${iterations} iterations | ${t}ms | ${t / iterations}ms per`);
}

/* -------------------------------------------- */

/**
 * A debugging function to test latency or timeouts by forcibly locking the thread for an amount of time.
 * @param {number} ms         A number of milliseconds to lock
 * @param {boolean} debug     Log debugging information?
 * @returns {Promise<void>}
 */
async function threadLock(ms, debug=false) {
  const t0 = performance.now();
  let d = 0;
  while ( d < ms ) {
    d = performance.now() - t0;
    if ( debug && (d % 1000 === 0) ) {
      console.debug(`Thread lock for ${d / 1000} of ${ms / 1000} seconds`);
    }
  }
}

/* -------------------------------------------- */

/**
 * Wrap a callback in a debounced timeout.
 * Delay execution of the callback function until the function has not been called for delay milliseconds
 * @param {Function} callback       A function to execute once the debounced threshold has been passed
 * @param {number} delay            An amount of time in milliseconds to delay
 * @returns {Function}              A wrapped function which can be called to debounce execution
 */
function debounce(callback, delay) {
  let timeoutId;
  return function(...args) {
    clearTimeout(timeoutId);
    timeoutId = setTimeout(() => {
      callback.apply(this, args);
    }, delay);
  };
}

/* -------------------------------------------- */

/**
 * Wrap a callback in a throttled timeout.
 * Delay execution of the callback function when the last time the function was called was delay milliseconds ago
 * @param {Function} callback       A function to execute once the throttled threshold has been passed
 * @param {number} delay            A maximum amount of time in milliseconds between to execution
 * @returns {Function}              A wrapped function which can be called to throttle execution
 */
function throttle(callback, delay) {
  let pending;
  let lastTime = -delay;
  return function(...args) {
    if ( pending ) {
      pending.thisArg = this;
      pending.args = args;
      return;
    }
    pending = {thisArg: this, args};
    setTimeout(() => {
      const {thisArg, args} = pending;
      pending = null;
      callback.apply(thisArg, args);
      lastTime = performance.now();
    }, Math.max(delay - (performance.now() - lastTime), 0));
  };
}

/* -------------------------------------------- */

/**
 * A utility function to request a debounced page reload.
 * @type {() => void}
 */
const debouncedReload = debounce( () => window.location.reload(), 250);

/* -------------------------------------------- */

/**
 * Recursively freezes (`Object.freeze`) the object (or value).
 * This method DOES NOT support cyclical data structures.
 * This method DOES NOT support advanced object types like Set, Map, or other specialized classes.
 * @template {object} const T
 * @param {T} obj                             The object (or value)
 * @param {object} [options]                  Options to configure the behaviour of deepFreeze
 * @param {boolean} [options.strict=false]    Throw an Error if deepFreeze is unable to seal something instead of
 *                                            returning the original
 * @returns {Readonly<T>}                     The same object (or value) that was passed in
 */
function deepFreeze(obj, {strict=false}={}) {
  return _deepFreeze(obj, strict, 0);
}

/**
 * An inner function does the work of {@link foundry.utils.deepFreeze}.
 * @param {any} obj           Some sort of data
 * @param {boolean} strict    Throw an Error if deepClone is unable to clone something instead of returning the original
 * @param {number} _d         The depth tracker
 */
function _deepFreeze(obj, strict, _d) {
  if ( _d > 100 ) {
    throw new Error("Maximum depth exceeded. Be sure your object does not contain cyclical data structures.");
  }
  _d++;

  if ( obj instanceof Array ) obj.forEach(o => _deepFreeze(o, strict, _d));
  else if ( typeof obj === "object" ) {
    if ( obj === null ) return null;

    // Unsupported advanced objects
    if ( obj.constructor && (obj.constructor !== Object) ) {
      if ( strict ) throw new Error("deepFreeze cannot freeze advanced objects");
      return obj;
    }

    for ( const key in obj ) _deepFreeze(obj[key], strict, _d);
  }
  return Object.freeze(obj);
}

/* -------------------------------------------- */

/**
 * Recursively seals (`Object.seal`) the object (or value).
 * This method DOES NOT support cyclical data structures.
 * This method DOES NOT support advanced object types like Set, Map, or other specialized classes.
 * @template {object} T
 * @param {T} obj                             The object (or value)
 * @param {object} [options]                  Options to configure the behaviour of deepSeal
 * @param {boolean} [options.strict=false]    Throw an Error if deepSeal is unable to seal something
 * @returns {T}                               The same object (or value) that was passed in
 */
function deepSeal(obj, {strict=false}={}) {
  return _deepSeal(obj, strict, 0);
}

/**
 * An inner function does the work of {@link foundry.utils.deepSeal}.
 * @param {any} obj           Some sort of data
 * @param {boolean} strict    Throw an Error if deepClone is unable to clone something instead of returning the original
 * @param {number} _d         The depth tracker
 */
function _deepSeal(obj, strict, _d) {
  if ( _d > 100 ) {
    throw new Error("Maximum depth exceeded. Be sure your object does not contain cyclical data structures.");
  }
  _d++;

  if ( obj instanceof Array ) obj.forEach(o => _deepSeal(o, strict, _d));
  else if ( typeof obj === "object" ) {
    if ( obj === null ) return null;

    // Unsupported advanced objects
    if ( obj.constructor && (obj.constructor !== Object) ) {
      if ( strict ) throw new Error("deepSeal cannot seal advanced objects");
      return obj;
    }

    for ( const key in obj ) _deepSeal(obj[key], strict, _d);
  }
  return Object.seal(obj);
}

/* -------------------------------------------- */

/**
 * Quickly clone a simple piece of data, returning a copy which can be mutated safely.
 * This method DOES support recursive data structures containing inner objects or arrays.
 * This method DOES NOT support cyclical data structures.
 * This method DOES NOT support advanced object types like Set, Map, or other specialized classes.
 * @template {object} T
 * @param {T} original                      Some sort of data
 * @param {object} [options]                Options to configure the behaviour of deepClone
 * @param {boolean} [options.strict=false]  Throw an Error if deepClone is unable to clone something instead of
 *                                          returning the original
 * @returns {T}                             The clone of that data
 */
function deepClone(original, {strict=false}={}) {
  return _deepClone(original, strict, 0);
}

/**
 * An inner function does the work of the deepClone operation and is optimized to avoid object creation.
 * @param {any} original      Some sort of data
 * @param {boolean} strict    Throw an Error if deepClone is unable to clone something instead of returning the original
 * @param {number} _d         The depth tracker
 */
function _deepClone(original, strict, _d) {
  if ( _d > 100 ) {
    throw new Error("Maximum depth exceeded. Be sure your object does not contain cyclical data structures.");
  }
  _d++;

  // Simple types
  if ( (typeof original !== "object") || (original === null) ) return original;

  // Arrays
  if ( original instanceof Array ) return original.map(o => _deepClone(o, strict, _d));

  // Dates
  if ( original instanceof Date ) return new Date(original);

  // Unsupported advanced objects
  if ( original.constructor && (original.constructor !== Object) ) {
    if ( strict ) throw new Error("deepClone cannot clone advanced objects");
    return original;
  }

  // Other objects
  const clone = {};
  for ( const k of Object.keys(original) ) {
    clone[k] = _deepClone(original[k], strict, _d);
  }
  return clone;
}

/* -------------------------------------------- */

/**
 * Deeply difference an object against some other, returning the update keys and values.
 * @param {object} original       An object comparing data against which to compare
 * @param {object} other          An object containing potentially different data
 * @param {object} [options={}]   Additional options which configure the diff operation
 * @param {boolean} [options.inner=false]  Only recognize differences in other for keys which also exist in original
 * @param {boolean} [options.deletionKeys=false] Apply special logic to deletion keys. They will only be kept if the
 *                                               original object has a corresponding key that could be deleted.
 * @param {number} [options._d]           An internal depth tracker
 * @returns {object}              An object of the data in other which differs from that in original
 */
function diffObject(original, other, {inner=false, deletionKeys=false, _d=0}={}) {
  return _diffObject(original, other, inner, deletionKeys, _d);
}

/**
 * An inner function does the work of the diffObject operation and is optimized to avoid object creation.
 * @param {object} original
 * @param {object} other
 * @param {boolean} inner
 * @param {boolean} deletionKeys
 * @param {number} _d
 * @returns {object}
 */
function _diffObject(original, other, inner, deletionKeys, _d=0) {
  if ( _d > 100 ) throw new Error("Maximum diffObject depth exceeded. Be careful of cyclical data structures.");
  const diff = {};
  for ( const key in other ) {
    if ( deletionKeys && isDeletionKey(key) ) {
      const [isDifferent, difference] = _diffSpecial(original, key, other[key], inner);
      if ( isDifferent ) diff[key] = difference;
    } else {
      const [isDifferent, difference] = _diffValue(original, key, other[key], inner, deletionKeys, _d);
      if ( isDifferent ) diff[key] = difference;
    }
  }
  return diff;
}

/**
 * Special handling for deletion "-=" and forced replacement "==" keys.
 * @param {object} original
 * @param {string} key
 * @param {any} value
 * @param {boolean} inner
 * @returns {[boolean, any]}
 */
function _diffSpecial(original, key, value, inner) {
  const targetKey = key.substring(2);
  const hasKey = targetKey in original;
  if ( inner && !hasKey ) return [false, undefined];

  // Deletion
  if ( key[0] === "-" ) {
    if ( value !== null ) throw new Error("Removing a key using the -= deletion syntax requires the value of that"
      + " deletion key to be null, for example {-=key: null}");
    return [hasKey, null];
  }

  // Forced Replacement
  else if ( key[0] === "=" ) return [true, applySpecialKeys(value)];
  return [false, undefined];
}

/**
 * Identify differences in individual keys.
 * @param {object} original
 * @param {string} key
 * @param {any} v1
 * @param {boolean} inner
 * @param {boolean} deletionKeys
 * @param {number} _d
 * @returns {[boolean, any]}
 */
function _diffValue(original, key, v1, inner, deletionKeys, _d) {
  const hasKey = key in original;
  if ( inner && !hasKey ) return [false, undefined];
  const v0 = original[key];

  // Set to null or to undefined
  if ( (v1 === undefined) || (v1 === null) ) return [v0 !== v1, v1];

  // Change object type
  const t0 = getType(v0);
  const t1 = getType(v1);
  if ( t0 !== t1 ) return [true, applySpecialKeys(v1)];

  // Use an explicitly provided equality testing method, if available
  if ( v0?.equals instanceof Function ) {
    if ( v0.equals(v1) ) return [false, undefined];
    return [true, applySpecialKeys(v1)];
  }

  // Recursively diff objects
  if ( (t0 === "Object") && (t1 === "Object") ) {
    if ( isEmpty(v1) ) return [false, undefined];
    const d = _diffObject(v0, v1, inner, deletionKeys, _d+1);
    return [!isEmpty(d), d];
  }

  // Differences in primitives
  return [v0.valueOf() !== v1.valueOf(), v1];
}

/* -------------------------------------------- */

/**
 * Recurse through an object, applying all special keys.
 * Deletion keys ("-=") are removed.
 * Forced replacement keys ("==") are assigned.
 * @param {*} obj
 * @returns {*}
 */
function applySpecialKeys(obj) {
  const type = getType(obj);
  if ( type === "Array" ) return obj.map(applySpecialKeys);
  if ( type !== "Object" ) return obj;
  const clone = {};
  for ( const key in obj ) {
    const v = obj[key];
    if ( isDeletionKey(key) ) {
      if ( key[0] === "-" ) {
        if ( v !== null ) throw new Error("Removing a key using the -= deletion syntax requires the value of that"
          + " deletion key to be null, for example {-=key: null}");
        delete clone[key.substring(2)];
        continue;
      }
      if ( key[0] === "=" ) {
        clone[key.substring(2)] = applySpecialKeys(v);
        continue;
      }
    }
    clone[key] = applySpecialKeys(v);
  }
  return clone;
}

/* -------------------------------------------- */

/**
 * Test if two objects contain the same enumerable keys and values.
 * @param {object} a  The first object.
 * @param {object} b  The second object.
 * @returns {boolean}
 */
function objectsEqual(a, b) {
  if ( (a == null) || (b == null) ) return a === b;
  if ( (getType(a) !== "Object") || (getType(b) !== "Object") ) return a === b;
  if ( Object.keys(a).length !== Object.keys(b).length ) return false;
  return Object.entries(a).every(([k, v0]) => {
    const v1 = b[k];
    const t0 = getType(v0);
    const t1 = getType(v1);
    if ( t0 !== t1 ) return false;
    if ( v0?.equals instanceof Function ) return v0.equals(v1);
    if ( t0 === "Object" ) return objectsEqual(v0, v1);
    return v0 === v1;
  });
}

/* -------------------------------------------- */

/**
 * A cheap data duplication trick which is relatively robust.
 * For a subset of cases the deepClone function will offer better performance.
 * @param {Object} original   Some sort of data
 */
function duplicate(original) {
  return JSON.parse(JSON.stringify(original));
}

/* -------------------------------------------- */

/**
 * Is a string key of an object used for certain deletion or forced replacement operations.
 * @param {string} key
 * @returns {boolean}
 */
function isDeletionKey(key) {
  if ( !(typeof key === "string") ) return false;
  return (key[1] === "=") && ((key[0] === "=") || (key[0] === "-"));
}

/* -------------------------------------------- */

/**
 * Test whether some class is a subclass of a parent.
 * Returns true if the classes are identical.
 * @param {Function} cls        The class to test
 * @param {Function} parent     Some other class which may be a parent
 * @returns {boolean}           Is the class a subclass of the parent?
 */
function isSubclass(cls, parent) {
  if ( typeof cls !== "function" ) return false;
  if ( cls === parent ) return true;
  return parent.isPrototypeOf(cls);
}

/* -------------------------------------------- */

/**
 * Search up the prototype chain and return the class that defines the given property.
 * @param {Object|Constructor} obj    A class instance or class definition which contains a property.
 *                                    If a class instance is passed the property is treated as an instance attribute.
 *                                    If a class constructor is passed the property is treated as a static attribute.
 * @param {string} property           The property name
 * @returns {Constructor<Object>}             The class that defines the property
 */
function getDefiningClass(obj, property) {
  const isStatic = obj.hasOwnProperty("prototype");
  let target = isStatic ? obj : Object.getPrototypeOf(obj);
  while ( target ) {
    if ( target.hasOwnProperty(property) ) {
      target = isStatic ? target : target.constructor;
      break;
    }
    target = Object.getPrototypeOf(target);
  }
  return target;
}

/* -------------------------------------------- */

/**
 * Encode an url-like string by replacing any characters which need encoding.
 * To reverse this encoding, the native decodeURIComponent can be used on the whole encoded string, without adjustment.
 * @param {string} path     A fully-qualified URL or url component (like a relative path)
 * @returns {string}         An encoded URL string
 */
function encodeURL(path) {

  // Determine whether the path is a well-formed URL
  const url = URL.parseSafe(path);

  // If URL, remove the initial protocol
  if ( url ) path = path.replace(url.protocol, "");

  // Split and encode each URL part
  path = path.split("/").map(p => encodeURIComponent(p).replace(/'/g, "%27")).join("/");

  // Return the encoded URL
  return url ? url.protocol + path : path;
}

/* -------------------------------------------- */

/**
 * Expand a flattened object to be a standard nested Object by converting all dot-notation keys to inner objects.
 * Only simple objects will be expanded. Other Object types like class instances will be retained as-is.
 * @param {object} obj      The object to expand
 * @returns {object}        An expanded object
 */
function expandObject(obj) {
  const _expand = (value, depth) => {
    if ( depth > 32 ) throw new Error("Maximum object expansion depth exceeded");
    if ( !value ) return value;
    if ( Array.isArray(value) ) return value.map(v => _expand(v, depth+1)); // Map arrays
    if ( getType(value) !== "Object" ) return value;                        // Return advanced objects directly
    const expanded = {};                                                    // Expand simple objects
    for ( const [k, v] of Object.entries(value) ) {
      setProperty(expanded, k, _expand(v, depth+1));
    }
    return expanded;
  };
  return _expand(obj, 0);
}

/* -------------------------------------------- */

/**
 * Filter the contents of some source object using the structure of a template object.
 * Only keys which exist in the template are preserved in the source object.
 *
 * @param {object} source           An object which contains the data you wish to filter
 * @param {object} template         An object which contains the structure you wish to preserve
 * @param {object} [options={}]     Additional options which customize the filtration
 * @param {boolean} [options.deletionKeys=false]    Whether to keep deletion keys
 * @param {boolean} [options.templateValues=false]  Instead of keeping values from the source, instead draw values
 *                                                  from the template
 * @returns {object}                The filtered object
 *
 * @example Filter an object
 * ```js
 * const source = {foo: {number: 1, name: "Tim", topping: "olives"}, bar: "baz"};
 * const template = {foo: {number: 0, name: "Mit", style: "bold"}, other: 72};
 * filterObject(source, template); // {foo: {number: 1, name: "Tim"}};
 * filterObject(source, template, {templateValues: true}); // {foo: {number: 0, name: "Mit"}};
 * ```
 */
function filterObject(source, template, {deletionKeys=false, templateValues=false}={}) {

  // Validate input
  const ts = getType(source);
  const tt = getType(template);
  if ( (ts !== "Object") || (tt !== "Object")) throw new Error("One of source or template are not Objects!");

  // Define recursive filtering function
  const _filter = function(s, t, filtered) {
    for ( const [k, v] of Object.entries(s) ) {
      const has = t.hasOwnProperty(k);
      const x = t[k];

      // Case 1 - inner object
      if ( has && (getType(v) === "Object") && (getType(x) === "Object") ) filtered[k] = _filter(v, x, {});

      // Case 2 - inner key
      else if ( has ) filtered[k] = templateValues ? x : v;

      // Case 3 - special key
      else if ( deletionKeys && isDeletionKey(k) ) filtered[k] = v;
    }
    return filtered;
  };

  // Begin filtering at the outermost layer
  return _filter(source, template, {});
}

/* -------------------------------------------- */

/**
 * Flatten a possibly multidimensional object to a one-dimensional one by converting all nested keys to dot notation
 * @param {object} obj        The object to flatten
 * @param {number} [_d=0]     Track the recursion depth to prevent overflow
 * @returns {object}          A flattened object
 */
function flattenObject(obj, _d=0) {
  const flat = {};
  if ( _d > 100 ) {
    throw new Error("Maximum depth exceeded");
  }
  for ( const [k, v] of Object.entries(obj) ) {
    const t = getType(v);
    if ( t === "Object" ) {
      if ( isEmpty(v) ) flat[k] = v;
      const inner = flattenObject(v, _d+1);
      for ( const [ik, iv] of Object.entries(inner) ) {
        flat[`${k}.${ik}`] = iv;
      }
    }
    else flat[k] = v;
  }
  return flat;
}

/* -------------------------------------------- */

/**
 * Obtain references to the parent classes of a certain class.
 * @param {Function} cls            An class definition
 * @returns {Array<typeof Object>}  An array of parent classes which the provided class extends
 */
function getParentClasses(cls) {
  if ( typeof cls !== "function" ) {
    throw new Error("The provided class is not a type of Function");
  }
  const parents = [];
  let parent = Object.getPrototypeOf(cls);
  while ( parent ) {
    parents.push(parent);
    parent = Object.getPrototypeOf(parent);
  }
  return parents.slice(0, -2);
}

/* -------------------------------------------- */

/**
 * Get the URL route for a certain path which includes a path prefix, if one is set
 * @param {string} path             The Foundry URL path
 * @param {string|null} [prefix]    A path prefix to apply
 * @returns {string}                The absolute URL path
 */
function getRoute(path, {prefix}={}) {
  prefix = prefix === undefined ? globalThis.ROUTE_PREFIX : prefix || null;
  path = path.replace(/(^\/+)|(\/+$)/g, ""); // Strip leading and trailing slashes
  let paths = [""];
  if ( prefix ) paths.push(prefix);
  paths = paths.concat([path.replace(/(^\/)|(\/$)/g, "")]);
  return paths.join("/");
}

/* -------------------------------------------- */

/**
 * The identifiable class types.
 * @type {Array<[class: Function, name: string]>}
 */
const typePrototypes = [
  [Array, "Array"],
  [Set, "Set"],
  [Map, "Map"],
  [Promise, "Promise"],
  [Error, "Error"],
  [Color, "number"]
];

/**
 * Learn the underlying data type of some variable. Supported identifiable types include:
 * undefined, null, number, string, boolean, function, Array, Set, Map, Promise, Error,
 * HTMLElement (client side only), Object (plain objects).
 * If the type isn't identifiable, Unknown is returned.
 * @param {*} variable  A provided variable
 * @returns {string}    The named type of the token
 */
function getType(variable) {

  // Primitive types, handled with simple typeof check
  const typeOf = typeof variable;
  if ( typeOf !== "object" ) return typeOf;

  // Special cases of object
  if ( variable === null ) return "null";
  if ( !variable.constructor ) return "Object"; // Object with the null prototype.
  if ( variable.constructor === Object ) return "Object"; // Simple objects

  // Match prototype instances
  for ( const [cls, type] of typePrototypes ) {
    if ( variable instanceof cls ) return type;
  }
  if ( ("HTMLElement" in globalThis) && (variable instanceof globalThis.HTMLElement) ) return "HTMLElement";

  // Unknown Object type
  return "Unknown";
}

/* -------------------------------------------- */

/**
 * A helper function which tests whether an object has a property or nested property given a string key.
 * The method also supports arrays if the provided key is an integer index of the array.
 * The string key supports the notation a.b.c which would return true if object[a][b][c] exists
 * @param {object} object   The object to traverse
 * @param {string} key      An object property with notation a.b.c
 * @returns {boolean}       An indicator for whether the property exists
 */
function hasProperty(object, key) {
  if ( !key || !object ) return false;
  if ( key in object ) return true;
  let target = object;
  for ( const p of key.split(".") ) {
    if ( !target || (typeof target !== "object") ) return false;
    if ( p in target ) target = target[p];
    else return false;
  }
  return true;
}

/* -------------------------------------------- */

/**
 * A helper function which searches through an object to retrieve a value by a string key.
 * The method also supports arrays if the provided key is an integer index of the array.
 * The string key supports the notation a.b.c which would return object[a][b][c]
 * @param {object} object   The object to traverse
 * @param {string} key      An object property with notation a.b.c
 * @returns {*}             The value of the found property
 */
function getProperty(object, key) {
  if ( !key || !object ) return undefined;
  if ( key in object ) return object[key];
  let target = object;
  for ( const p of key.split(".") ) {
    if ( !target ) return undefined;
    const type = typeof target;
    if ( (type !== "object") && (type !== "function") ) return undefined;
    if ( p in target ) target = target[p];
    else return undefined;
  }
  return target;
}

/* -------------------------------------------- */

const SKIPPED_PROPERTIES = new Set(["__proto__", "constructor", "prototype"]);

/**
 * A helper function which searches through an object to assign a value using a string key
 * This string key supports the notation a.b.c which would target object[a][b][c]
 * @param {object} object   The object to update
 * @param {string} key      The string key
 * @param {*} value         The value to be assigned
 * @returns {boolean}       Whether the value was changed from its previous value
 */
function setProperty(object, key, value) {
  if ( !key || SKIPPED_PROPERTIES.has(key) ) return false;

  // Convert the key to an object reference if it contains dot notation
  let target = object;
  if ( key.indexOf(".") !== -1 ) {
    const parts = key.split(".");
    if ( parts.some(p => SKIPPED_PROPERTIES.has(p)) ) return false;
    key = parts.pop();
    target = parts.reduce((target, p) => {
      if ( !(p in target) ) target[p] = {};
      return target[p];
    }, object);
  }

  // Update the target
  if ( !(key in target) || (target[key] !== value) ) {
    target[key] = value;
    return true;
  }
  return false;
}

/* -------------------------------------------- */

/**
 * A helper function which searches through an object to delete a value by a string key.
 * The string key supports the notation a.b.c which would delete object[a][b][c]
 * @param {object} object   The object to traverse
 * @param {string} key      An object property with notation a.b.c
 * @returns {boolean}       Was the property deleted?
 */
function deleteProperty(object, key) {
  if ( !key || !object ) return false;
  let parent;
  let target = object;
  const parts = key.split(".");
  for ( const p of parts ) {
    if ( !target ) return false;
    const type = typeof target;
    if ( (type !== "object") && (type !== "function") ) return false;
    if ( !(p in target) ) return false;
    parent = target;
    target = parent[p];
  }
  delete parent[parts.at(-1)];
  return true;
}

/* -------------------------------------------- */

/**
 * Invert an object by assigning its values as keys and its keys as values.
 * @param {object} obj    The original object to invert
 * @returns {object}      The inverted object with keys and values swapped
 */
function invertObject(obj) {
  const inverted = {};
  for ( const [k, v] of Object.entries(obj) ) {
    if ( v in inverted ) throw new Error("The values of the provided object must be unique in order to invert it.");
    inverted[v] = k;
  }
  return inverted;
}

/* -------------------------------------------- */

/**
 * Return whether a target version (v1) is more advanced than some other reference version (v0).
 * Supports either numeric or string version comparison with version parts separated by periods.
 * @param {number|string} v1    The target version
 * @param {number|string} v0    The reference version
 * @returns {boolean}           Is v1 a more advanced version than v0?
 */
function isNewerVersion(v1, v0) {

  // Handle numeric versions
  if ( (typeof v1 === "number") && (typeof v0 === "number") ) return v1 > v0;

  // Handle string parts
  const v1Parts = String(v1).split(".");
  const v0Parts = String(v0).split(".");

  // Iterate over version parts
  for ( const [i, p1] of v1Parts.entries() ) {
    const p0 = v0Parts[i];

    // If the prior version doesn't have a part, v1 wins
    if ( p0 === undefined ) return true;

    // If both parts are numbers, use numeric comparison to avoid cases like "12" < "5"
    if ( Number.isNumeric(p0) && Number.isNumeric(p1) ) {
      if ( Number(p1) !== Number(p0) ) return Number(p1) > Number(p0);
    }

    // Otherwise, compare as strings
    if ( p1 !== p0 ) return p1 > p0;
  }

  // If there are additional parts to v0, it is not newer
  if ( v0Parts.length > v1Parts.length ) return false;

  // If we have not returned false by now, it's either newer or the same
  return !v1Parts.equals(v0Parts);
}

/* -------------------------------------------- */

/**
 * Test whether a value is empty-like; either undefined or a content-less object.
 * @param {*} value       The value to test
 * @returns {boolean}     Is the value empty-like?
 */
function isEmpty(value) {
  const t = getType(value);
  switch ( t ) {
    case "undefined":
      return true;
    case "null":
      return true;
    case "Array":
      return !value.length;
    case "Object":
      return !Object.keys(value).length;
    case "Set":
    case "Map":
      return !value.size;
    default:
      return false;
  }
}

/* -------------------------------------------- */

/**
 * Update a source object by replacing its keys and values with those from a target object.
 *
 * @param {object} original                           The initial object which should be updated with values from the
 *                                                    target
 * @param {object} [other={}]                         A new object whose values should replace those in the source
 * @param {object} [options={}]                       Additional options which configure the merge
 * @param {boolean} [options.insertKeys=true]         Control whether to insert new top-level objects into the resulting
 *                                                    structure which do not previously exist in the original object.
 * @param {boolean} [options.insertValues=true]       Control whether to insert new nested values into child objects in
 *                                                    the resulting structure which did not previously exist in the
 *                                                    original object.
 * @param {boolean} [options.overwrite=true]          Control whether to replace existing values in the source, or only
 *                                                    merge values which do not already exist in the original object.
 * @param {boolean} [options.recursive=true]          Control whether to merge inner-objects recursively (if true), or
 *                                                    whether to simply replace inner objects with a provided new value.
 * @param {boolean} [options.inplace=true]            Control whether to apply updates to the original object in-place
 *                                                    (if true), otherwise the original object is duplicated and the
 *                                                    copy is merged.
 * @param {boolean} [options.enforceTypes=false]      Control whether strict type checking requires that the value of a
 *                                                    key in the other object must match the data type in the original
 *                                                    data to be merged.
 * @param {boolean} [options.performDeletions=false]  Control whether to perform deletions on the original object if
 *                                                    deletion keys are present in the other object.
 * @param {number} [_d=0]                             A privately used parameter to track recursion depth.
 * @returns {object}                                  The original source object including updated, inserted, or
 *                                                    overwritten records.
 *
 * @example Control how new keys and values are added
 * ```js
 * mergeObject({k1: "v1"}, {k2: "v2"}, {insertKeys: false}); // {k1: "v1"}
 * mergeObject({k1: "v1"}, {k2: "v2"}, {insertKeys: true});  // {k1: "v1", k2: "v2"}
 * mergeObject({k1: {i1: "v1"}}, {k1: {i2: "v2"}}, {insertValues: false}); // {k1: {i1: "v1"}}
 * mergeObject({k1: {i1: "v1"}}, {k1: {i2: "v2"}}, {insertValues: true}); // {k1: {i1: "v1", i2: "v2"}}
 * ```
 *
 * @example Control how existing data is overwritten
 * ```js
 * mergeObject({k1: "v1"}, {k1: "v2"}, {overwrite: true}); // {k1: "v2"}
 * mergeObject({k1: "v1"}, {k1: "v2"}, {overwrite: false}); // {k1: "v1"}
 * ```
 *
 * @example Control whether merges are performed recursively
 * ```js
 * mergeObject({k1: {i1: "v1"}}, {k1: {i2: "v2"}}, {recursive: false}); // {k1: {i2: "v2"}}
 * mergeObject({k1: {i1: "v1"}}, {k1: {i2: "v2"}}, {recursive: true}); // {k1: {i1: "v1", i2: "v2"}}
 * ```
 *
 * @example Deleting an existing object key
 * ```js
 * mergeObject({k1: "v1", k2: "v2"}, {"-=k1": null}, {performDeletions: true});   // {k2: "v2"}
 * ```
 *
 * @example Explicitly replacing an inner object key
 * ```js
 * mergeObject({k1: {i1: "v1"}}, {"==k1": {i2: "v2"}}, {performDeletions: true}); // {k1: {i2: "v2"}}
 * ```
 */
function mergeObject(original, other={}, {insertKeys=true, insertValues=true, overwrite=true, recursive=true,
  inplace=true, enforceTypes=false, performDeletions=false}={}, _d=0) {
  other = other || {};
  if (!(original instanceof Object) || !(other instanceof Object)) {
    throw new Error("One of original or other are not Objects!");
  }
  const options = {insertKeys, insertValues, overwrite, recursive, enforceTypes, performDeletions};

  // Special handling at depth 0
  if ( _d === 0 ) {
    if ( Object.keys(other).some(k => /\./.test(k)) ) other = expandObject(other);
    if ( Object.keys(original).some(k => /\./.test(k)) ) {
      const expanded = expandObject(original);
      if ( inplace ) {
        Object.keys(original).forEach(k => delete original[k]);
        Object.assign(original, expanded);
      }
      else original = expanded;
    }
    else if ( !inplace ) original = deepClone(original);
  }

  // Iterate over the other object
  for ( const k of Object.keys(other) ) {
    const v = other[k];
    if ( original.hasOwnProperty(k) ) _mergeUpdate(original, k, v, _d+1, options);
    else _mergeInsert(original, k, v, _d+1, options);
  }
  return original;
}

/**
 * A helper function for merging objects when the target key does not exist in the original.
 * @ignore
 */
function _mergeInsert(original, k, v, _d, {insertKeys, insertValues, performDeletions}={}) {

  // Force replace a specific key
  if ( performDeletions && k.startsWith("==") ) {
    original[k.slice(2)] = applySpecialKeys(v);
    return;
  }

  // Delete a specific key
  if ( performDeletions && k.startsWith("-=") ) {
    if ( v !== null ) throw new Error("Removing a key using the -= deletion syntax requires the value of that"
      + " deletion key to be null, for example {-=key: null}");
    delete original[k.slice(2)];
    return;
  }

  // Insert a new object, either recursively or directly
  const canInsert = ((_d <= 1) && insertKeys) || ((_d > 1) && insertValues);
  if ( !canInsert ) return;
  if ( getType(v) === "Object" ) {
    original[k] = mergeObject({}, v, {insertKeys: true, inplace: true, performDeletions});
    return;
  }
  original[k] = v;
}

/**
 * A helper function for merging objects when the target key exists in the original.
 * @ignore
 */
function _mergeUpdate(original, k, v, _d, {insertKeys, insertValues, enforceTypes, overwrite, recursive,
  performDeletions}={}) {
  const x = original[k];
  const tv = getType(v);
  const tx = getType(x);
  const ov = (tv === "Object") || (tv === "Unknown");
  const ox = (tx === "Object") || (tx === "Unknown");

  // Recursively merge an inner object
  if ( ov && ox && recursive ) {
    return mergeObject(x, v, {
      insertKeys, insertValues, overwrite, enforceTypes, performDeletions,
      inplace: true
    }, _d);
  }

  // Overwrite an existing value
  if ( overwrite ) {
    if ( (tx !== "undefined") && (tv !== tx) && enforceTypes ) {
      throw new Error("Mismatched data types encountered during object merge.");
    }
    original[k] = applySpecialKeys(v);
  }
}

/* -------------------------------------------- */

/**
 * Parse an S3 key to learn the bucket and the key prefix used for the request.
 * @param {string} key  A fully qualified key name or prefix path.
 * @returns {{bucket: string|null, keyPrefix: string}}
 */
function parseS3URL(key) {
  const url = URL.parseSafe(key);
  if ( url ) return {
    bucket: url.host.split(".").shift(),
    keyPrefix: url.pathname.slice(1)
  };
  return {
    bucket: null,
    keyPrefix: ""
  };
}

/* -------------------------------------------- */

/**
 * Generate a random alphanumeric string ID of a given requested length using `crypto.getRandomValues()`.
 * @param {number} length    The length of the random string to generate, which must be at most 16384.
 * @returns {string}         A string containing random letters (A-Z, a-z) and numbers (0-9).
 */
function randomID(length=16) {
  const chars = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789";
  const cutoff = 0x100000000 - (0x100000000 % chars.length);
  const random = new Uint32Array(length);
  do {
    crypto.getRandomValues(random);
  } while ( random.some(x => x >= cutoff) );
  let id = "";
  for ( let i = 0; i < length; i++ ) id += chars[random[i] % chars.length];
  return id;
}

/* -------------------------------------------- */

/**
 * Format a file size to an appropriate order of magnitude.
 * @param {number} size  The size in bytes.
 * @param {object} [options]
 * @param {number} [options.decimalPlaces=2]  The number of decimal places to round to.
 * @param {2|10} [options.base=10]            The base to use. In base 10 a kilobyte is 1000 bytes. In base 2 it is
 *                                            1024 bytes.
 * @returns {string}
 */
function formatFileSize(size, {decimalPlaces=2, base=10}={}) {
  const units = ["B", "kB", "MB", "GB", "TB"];
  const divisor = base === 2 ? 1024 : 1000;
  let iterations = 0;
  while ( (iterations < units.length) && (size > divisor) ) {
    size /= divisor;
    iterations++;
  }
  return `${size.toFixed(decimalPlaces)} ${units[iterations]}`;
}

/* -------------------------------------------- */

/**
 * Parse a UUID into its constituent parts, identifying the type and ID of the referenced document.
 * The ResolvedUUID result also identifies a "primary" document which is a root-level document either in the game
 * World or in a Compendium pack which is a parent of the referenced document.
 * @param {string} uuid                  The UUID to parse.
 * @param {object} [options]             Options to configure parsing behavior.
 * @param {foundry.abstract.Document} [options.relative]  A document to resolve relative UUIDs against.
 * @returns {ResolvedUUID|null} Returns, if possible, the Collection, Document Type, and Document ID to resolve the
 *                              parent document, as well as the remaining Embedded Document parts, if any.
 */
function parseUuid(uuid, {relative}={}) {
  if ( !uuid ) return null;

  // Relative UUID
  if ( uuid.startsWith(".") && relative ) return _resolveRelativeUuid(uuid, relative);

  // Split UUID parts
  const parts = uuid.split(".");
  let id;
  let type;
  let primaryId;
  let primaryType;
  let collection;

  // Compendium Documents
  if ( parts[0] === "Compendium" ) {
    // Re-interpret legacy compendium UUIDs which did not explicitly include their parent document type.
    let packType = _resolvePrimaryType(parts);
    if ( packType !== parts[3] ) parts.splice(3, 0, packType);

    // Check for redirects.
    if ( game.compendiumUUIDRedirects ) {
      const node = game.compendiumUUIDRedirects.nodeAtPrefix(parts, { hasLeaves: true });
      const leaves = node?.[foundry.utils.StringTree.leaves];
      if ( leaves.length ) {
        const redirect = leaves[0];
        if ( redirect?.length ) {
          parts.splice(0, redirect.length, ...redirect);
          packType ??= _resolvePrimaryType(parts);
          parts[3] = packType;
        }
      }
    }
    const [, scope, packName] = parts.splice(0, 3);
    collection = game.packs.get(`${scope}.${packName}`);
    [primaryType, primaryId] = parts.splice(0, 2);
    if ( primaryType && (primaryType === packType) ) {
      uuid = ["Compendium", scope, packName, primaryType, primaryId, ...parts].join(".");
    }
  }

  // World Documents
  else {
    [primaryType, primaryId] = parts.splice(0, 2);
    collection = globalThis.db?.[primaryType] ?? CONFIG[primaryType]?.collection?.instance;
  }

  // Embedded Documents
  if ( parts.length ) {
    if ( parts.length % 2 ) return null;
    id = parts.at(-1);
    type = parts.at(-2);
  }

  // Primary Documents
  else {
    id = primaryId;
    type = primaryType ?? undefined;
    primaryId = primaryType = undefined;
  }

  // Return resolved UUID
  return {uuid, type, id, collection, embedded: parts, primaryType, primaryId,
    documentType: primaryType ?? type, documentId: primaryId ?? id};
}

/* -------------------------------------------- */

/**
 * Resolve a UUID relative to another document.
 * The general-purpose algorithm for resolving relative UUIDs is as follows:
 * 1. If the number of parts is odd, remove the first part and resolve it against the current document and update the
 *    current document.
 * 2. If the number of parts is even, resolve embedded documents against the current document.
 * @param {string} uuid        The UUID to resolve.
 * @param {foundry.abstract.Document} relative  The document to resolve against.
 * @returns {ResolvedUUID|null}     A resolved UUID object, if possible to create, or otherwise `null`.
 */
function _resolveRelativeUuid(uuid, relative) {
  if ( !(relative instanceof foundry.abstract.Document) ) {
    throw new Error("A relative Document instance must be provided to _resolveRelativeUuid");
  }
  uuid = uuid.substring(1);
  const parts = uuid.split(".");
  if ( !parts.length ) return null;
  let id;
  let type;
  let root;
  let primaryType;
  let primaryId;

  // Identify the root document and its collection
  const getRoot = doc => {
    if ( doc.parent ) parts.unshift(doc.documentName, doc.id);
    return doc.parent ? getRoot(doc.parent) : doc;
  };

  // Even-numbered parts include an explicit child document type
  if ( (parts.length % 2) === 0 ) {
    root = getRoot(relative);
    id = parts.at(-1);
    type = parts.at(-2);
    primaryType = root.documentName;
    primaryId = root.id;
    uuid = [primaryType, primaryId, ...parts].join(".");
  }

  // Relative Embedded Document
  else if ( relative.parent ) {
    id = parts.at(-1);
    type = relative.documentName;
    parts.unshift(type);
    root = getRoot(relative.parent);
    primaryType = root.documentName;
    primaryId = root.id;
    uuid = [primaryType, primaryId, ...parts].join(".");
  }

  // Relative Document
  else {
    root = relative;
    id = parts.pop();
    type = relative.documentName;
    uuid = [type, id].join(".");
  }

  // Recreate fully-qualified UUID and return the resolved result
  if ( root.pack ) uuid = `Compendium.${root.pack}.${uuid}`;
  return {uuid, type, id, collection: root.collection, primaryType, primaryId, embedded: parts,
    documentType: primaryType ?? type, documentId: primaryId ?? id};
}

/* -------------------------------------------- */

/**
 * Attempt to resolve a possibly missing primary Document type for a legacy Compendium UUID.
 * @param {string[]} parts
 * @returns {string|null} The Document type, if found, or null
 */
function _resolvePrimaryType(parts) {
  if ( CONST.COMPENDIUM_DOCUMENT_TYPES.includes(parts[3]) || (parts[3] === "Folder") ) return parts[3];
  return game.packs.get(`${parts[1]}.${parts[2]}`)?.documentName ?? null;
}

/* -------------------------------------------- */

/**
 * Build a Universally Unique Identifier (uuid) from possibly limited data. An attempt will be made to resolve omitted
 * components, but an identifier and at least one of documentName, parent, and pack are required.
 * @param {object} context  Data for building the uuid
 * @param {string} context.id              The identifier of the document
 * @param {string} [context.documentName]  The document name (or type)
 * @param {Document|null} [context.parent] The document's parent, if any
 * @param {string|null} [context.pack]     The document's compendium pack, if applicable
 * @returns {string|null} A well-formed Document uuid unless one is unable to be created
 */
function buildUuid({id, documentName, parent, pack}) {
  if ( !id || (!documentName && !parent && !pack) ) return null;
  if ( !pack && !parent && !CONST.WORLD_DOCUMENT_TYPES.includes(documentName) ) {
    console.warn("Only a documentName were provided, but it is not a valid world Document type.");
  }
  if ( pack ) documentName ||= game.packs.get(pack)?.documentName;
  if ( parent && !documentName ) {
    // Note: the possibility exists, however unlikely, that multiple embedded collections will have the same ID
    for ( const collection of Object.values(parent.collections) ) {
      if ( collection.has(id) ) {
        documentName = collection.documentName;
        break;
      }
    }
  }
  return [
    parent?.uuid,
    pack && !parent ? ["Compendium", pack] : null,
    documentName,
    id
  ].flat().filterJoin(".");
}

/* -------------------------------------------- */

/**
 * The table used for escaping bad characters.
 * @type {Record<string, string>}
 */
const ESCAPE_HTML_TABLE = {
  "&": "&amp;",
  "<": "&lt;",
  ">": "&gt;",
  '"': "&quot;",
  "'": "&#x27;"
};

/* -------------------------------------------- */

/**
 * The regex of bad characters that need to be escaped.
 * @type {RegExp}
 */
const ESCAPE_HTML_REGEX = new RegExp(`[${Object.keys(ESCAPE_HTML_TABLE).join("")}]`, "g");

/* -------------------------------------------- */

/**
 * The table used for unescaping HTML character entities of bad characters.
 * @type {Record<string, string>}
 */
const UNESCAPE_HTML_TABLE = invertObject(ESCAPE_HTML_TABLE);

/* -------------------------------------------- */

/**
 * The regex of HTML entities of bad characters that need to be unescaped.
 * @type {RegExp}
 */
const UNESCAPE_HTML_REGEX = new RegExp(`${Object.keys(UNESCAPE_HTML_TABLE).join("|")}`, "g");

/* -------------------------------------------- */

/**
 * Escape the given unescaped string.
 *
 * Escaped strings are safe to use inside inner HTML of most tags and in most quoted HTML attributes.
 * They are not NOT safe to use in `<script>` tags, unquoted attributes, `href`, `onmouseover`, and similar.
 * They must be unescaped first if they are used inside a context that would escape them.
 *
 * Handles only `&`, `<`, `>`, `"`, and `'`.
 * @see {@link foundry.utils.unescapeHTML}
 * @param {string|any} value    An unescaped string
 * @returns {string}            The escaped string
 */
function escapeHTML(value) {
  return String(value).replace(ESCAPE_HTML_REGEX, c => ESCAPE_HTML_TABLE[c]);
}

/* -------------------------------------------- */

/**
 * Unescape the given escaped string.
 *
 * Handles only `&amp;`, `&lt;`, `&gt;`, `&quot;`, and `&#x27;`.
 * @see {@link foundry.utils.escapeHTML}
 * @param {string} value    An escaped string
 * @returns {string}        The escaped string
 */
function unescapeHTML(value) {
  return value.replace(UNESCAPE_HTML_REGEX, c => UNESCAPE_HTML_TABLE[c]);
}

/**
 * Flatten nested arrays by concatenating their contents
 * @returns {any[]}    An array containing the concatenated inner values
 */
function deepFlatten() {
  return this.reduce((acc, val) => Array.isArray(val) ? acc.concat(val.deepFlatten()) : acc.concat(val), []);
}

/**
 * Test element-wise equality of the values of this array against the values of another array
 * @param {any[]} other   Some other array against which to test equality
 * @returns {boolean}     Are the two arrays element-wise equal?
 */
function equals$1(other) {
  if ( !(other instanceof Array) || (other.length !== this.length) ) return false;
  return this.every((v0, i) => {
    const v1 = other[i];
    const t0 = getType(v0);
    const t1 = getType(v1);
    if ( t0 !== t1 ) return false;
    if ( v0?.equals instanceof Function ) return v0.equals(v1);
    if ( t0 === "Object" ) return objectsEqual(v0, v1);
    return v0 === v1;
  });
}

/**
 * Partition an original array into two children array based on a logical test
 * Elements which test as false go into the first result while elements testing as true appear in the second
 * @template T
 * @param {(element: T) => boolean} rule
 * @returns {[T[], T[]]}    An Array of length two whose elements are the partitioned pieces of the original
 */
function partition(rule) {
  return this.reduce((acc, val) => {
    const test = rule(val);
    acc[Number(test)].push(val);
    return acc;
  }, [[], []]);
}

/**
 * Join an Array using a string separator, first filtering out any parts which return a false-y value
 * @param {string} sep    The separator string
 * @returns {string}      The joined string, filtered of any false values
 */
function filterJoin(sep) {
  return this.filter(p => !!p).join(sep);
}

/**
 * Find an element within the Array and remove it from the array
 * @template T
 * @param {(element: T) => boolean} find   A function to use as input to findIndex
 * @param {T} [replace]     A replacement for the spliced element
 * @returns {T|null}        The replacement element, the removed element, or null if no element was found.
 * @see Array#splice
 */
function findSplice(find, replace) {
  const idx = this.findIndex(find);
  if ( idx === -1 ) return null;
  if ( replace !== undefined ) {
    this.splice(idx, 1, replace);
    return replace;
  } else {
    const item = this[idx];
    this.splice(idx, 1);
    return item;
  }
}

/**
 * Create and initialize an array of length n with integers from 0 to n-1
 * @param {number} n        The desired array length
 * @param {number} [min=0]  A desired minimum number from which the created array starts
 * @returns {number[]}      An array of integers from min to min+n
 */
function fromRange(n, min=0) {
  return Array.from({length: n}, (v, i) => i + min);
}

// Define primitives on the Array prototype
Object.defineProperties(Array.prototype, {
  deepFlatten: {value: deepFlatten},
  equals: {value: equals$1},
  filterJoin: {value: filterJoin},
  findSplice: {value: findSplice},
  partition: {value: partition}
});
Object.defineProperties(Array, {
  fromRange: {value: fromRange}
});

/**
 * Test whether a Date instance is valid.
 * A valid date returns a number for its timestamp, and NaN otherwise.
 * NaN is never equal to itself.
 * @returns {boolean}
 */
function isValid() {
  // eslint-disable-next-line no-self-compare
  return this.getTime() === this.getTime();
}

/**
 * Return a standard YYYY-MM-DD string for the Date instance.
 * @returns {string}    The date in YYYY-MM-DD format
 */
function toDateInputString() {
  const yyyy = this.getFullYear();
  const mm = (this.getMonth() + 1).paddedString(2);
  const dd = this.getDate().paddedString(2);
  return `${yyyy}-${mm}-${dd}`;
}

/**
 * Return a standard H:M:S.Z string for the Date instance.
 * @returns {string}    The time in H:M:S format
 */
function toTimeInputString() {
  return this.toTimeString().split(" ")[0];
}

// Define primitives on the Date prototype
Object.defineProperties(Date.prototype, {
  isValid: {value: isValid},
  toDateInputString: {value: toDateInputString},
  toTimeInputString: {value: toTimeInputString}
});

/**
 * √3
 * @type {number}
 */
const SQRT3 = 1.7320508075688772;

/**
 * √⅓
 * @type {number}
 */
const SQRT1_3 = 0.5773502691896257;

/**
 * Bound a number between some minimum and maximum value, inclusively.
 * @param {number} num    The current value
 * @param {number} min    The minimum allowed value
 * @param {number} max    The maximum allowed value
 * @returns {number}      The clamped number
 */
function clamp(num, min, max) {
  return Math.min(Math.max(num, min), max);
}

/**
 * @deprecated since v12
 * @ignore
 */
function clamped(num, min, max) {
  const msg = "Math.clamped is deprecated in favor of Math.clamp.";
  foundry.utils.logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
  return clamp(num, min, max);
}

/**
 * Linear interpolation function
 * @param {number} a   An initial value when weight is 0.
 * @param {number} b   A terminal value when weight is 1.
 * @param {number} w   A weight between 0 and 1.
 * @returns {number}   The interpolated value between a and b with weight w.
 */
function mix(a, b, w) {
  return (a * (1 - w)) + (b * w);
}

/**
 * Transform an angle in degrees to be bounded within the domain [0, 360)
 * @param {number} degrees  An angle in degrees
 * @param {number} _base    DEPRECATED
 * @returns {number}        The same angle on the range [0, 360)
 */
function normalizeDegrees(degrees, _base) {
  const d = degrees % 360;
  if ( _base !== undefined ) {
    const msg = "Math.normalizeDegrees(degrees, base) is deprecated.";
    foundry.utils.logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    if ( _base === 360 ) return d <= 0 ? d + 360 : d;
  }
  return d < 0 ? d + 360 : d;
}

/**
 * Transform an angle in radians to be bounded within the domain [-PI, PI]
 * @param {number} radians  An angle in degrees
 * @returns {number}        The same angle on the range [-PI, PI]
 */
function normalizeRadians(radians) {
  const pi = Math.PI;
  const pi2 = pi * 2;
  return radians - (pi2 * Math.floor((radians + pi) / pi2));
}

/**
 * @deprecated since v12
 * @ignore
 */
function roundDecimals(number, places) {
  const msg = "Math.roundDecimals is deprecated.";
  foundry.utils.logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
  places = Math.max(Math.trunc(places), 0);
  const scl = Math.pow(10, places);
  return Math.round(number * scl) / scl;
}

/**
 * Transform an angle in radians to a number in degrees
 * @param {number} angle    An angle in radians
 * @returns {number}        An angle in degrees
 */
function toDegrees(angle) {
  return angle * (180 / Math.PI);
}

/**
 * Transform an angle in degrees to an angle in radians
 * @param {number} angle    An angle in degrees
 * @returns {number}        An angle in radians
 */
function toRadians(angle) {
  return angle * (Math.PI / 180);
}

/**
 * Returns the value of the oscillation between `a` and `b` at time `t`.
 * @param {number} a                              The minimium value of the oscillation
 * @param {number} b                              The maximum value of the oscillation
 * @param {number} t                              The time
 * @param {number} [p=1]                          The period (must be nonzero)
 * @param {(x: number) => number} [f=Math.cos]    The periodic function (its period must be 2π)
 * @returns {number}                              `((b - a) * (f(2π * t / p) + 1) / 2) + a`
 */
function oscillation(a, b, t, p=1, f=Math.cos) {
  return ((b - a) * (f((2 * Math.PI * t) / p) + 1) / 2) + a;
}

// Define properties on the Math environment
Object.defineProperties(Math, {
  SQRT3: {value: SQRT3},
  SQRT1_3: {value: SQRT1_3},
  clamp: {
    value: clamp,
    configurable: true,
    writable: true
  },
  clamped: {
    value: clamped,
    configurable: true,
    writable: true
  },
  mix: {
    value: mix,
    configurable: true,
    writable: true
  },
  normalizeDegrees: {
    value: normalizeDegrees,
    configurable: true,
    writable: true
  },
  normalizeRadians: {
    value: normalizeRadians,
    configurable: true,
    writable: true
  },
  roundDecimals: {
    value: roundDecimals,
    configurable: true,
    writable: true
  },
  toDegrees: {
    value: toDegrees,
    configurable: true,
    writable: true
  },
  toRadians: {
    value: toRadians,
    configurable: true,
    writable: true
  },
  oscillation: {
    value: oscillation,
    configurable: true,
    writable: true
  }
});

/**
 * Test for near-equivalence of two numbers within some permitted epsilon
 * @param {number} n      Some other number
 * @param {number} e      Some permitted epsilon, by default 1e-8
 * @returns {boolean}     Are the numbers almost equal?
 */
function almostEqual(n, e=1e-8) {
  return Math.abs(this - n) < e;
}

/**
 * Transform a number to an ordinal string representation. i.e.
 * 1 => 1st
 * 2 => 2nd
 * 3 => 3rd
 * @returns {string}
 */
function ordinalString() {
  const s = ["th", "st", "nd", "rd"];
  const v = this % 100;
  return this + (s[(v-20)%10]||s[v]||s[0]);
}

/**
 * Return a string front-padded by zeroes to reach a certain number of numeral characters
 * @param {number} digits     The number of characters desired
 * @returns {string}          The zero-padded number
 */
function paddedString(digits) {
  return this.toString().padStart(digits, "0");
}

/**
 * Return a locally formatted string prefaced by the explicit sign of the number (+) or (-). Use of this method is
 * intended for display purposes only.
 * @this {number}
 * @returns {string}          The signed number as a locally formatted string
 */
function signedString() {
  const n = this.toLocaleString(game.i18n.lang);
  if ( this === 0 ) return n;
  if ( this < 0 ) return n.replace("-", "−"); // Minus sign character
  else return `+${n}`;
}

/**
 * Round a number to the closest number which substracted from the base is a multiple of the provided interval.
 * This is a convenience function intended to humanize issues of floating point precision.
 * The interval is treated as a standard string representation to determine the amount of decimal truncation applied.
 * @param {number} interval                            The step interval
 * @param {"round"|"floor"|"ceil"} [method="round"]    The rounding method
 * @param {number} [base=0]                            The step base
 * @returns {number}                                   The rounded number
 *
 * @example Round a number to the nearest step interval
 * ```js
 * let n = 17.18;
 * n.toNearest(5); // 15
 * n.toNearest(10); // 20
 * n.toNearest(10, "floor"); // 10
 * n.toNearest(10, "ceil"); // 20
 * n.toNearest(0.25); // 17.25
 * n.toNearest(2, "round", 1); // 17
 * ```
 */
function toNearest(interval=1, method="round", base=0) {
  if ( interval < 0 ) throw new Error("Number#toNearest interval must not be negative");
  const eps = method === "floor" ? 1e-8 : method === "ceil" ? -1e-8 : 0;
  const float = base + (Math[method](((this - base) / interval) + eps) * interval);
  const trunc1 = Number.isInteger(base) ? 0 : String(base).length - 2;
  const trunc2 = Number.isInteger(interval) ? 0 : String(interval).length - 2;
  return Number(float.toFixed(Math.max(trunc1, trunc2)));
}

/**
 * A faster numeric between check which avoids type coercion to the Number object.
 * Since this avoids coercion, if non-numbers are passed in unpredictable results will occur. Use with caution.
 * @param {number} a            The lower-bound
 * @param {number} b            The upper-bound
 * @param {boolean} inclusive   Include the bounding values as a true result?
 * @returns {boolean}           Is the number between the two bounds?
 */
function between(a, b, inclusive=true) {
  const min = Math.min(a, b);
  const max = Math.max(a, b);
  return inclusive ? (this >= min) && (this <= max) : (this > min) && (this < max);
}

/**
 * @see {@link Number#between}
 * @ignore
 */
Number.between = function(num, a, b, inclusive=true) {
  const min = Math.min(a, b);
  const max = Math.max(a, b);
  return inclusive ? (num >= min) && (num <= max) : (num > min) && (num < max);
};

/**
 * Test whether a value is numeric.
 * This is the highest performing algorithm currently available, per https://jsperf.com/isnan-vs-typeof/5
 * @param {*} n        A value to test
 * @returns {boolean}  Is it a number?
 */
function isNumeric(n) {
  if ( n instanceof Array ) return false;
  else if ( [null, ""].includes(n) ) return false;
  // eslint-disable-next-line no-implicit-coercion, no-self-compare
  return +n === +n;
}

/**
 * Attempt to create a number from a user-provided string.
 * @param {string|number} n    The value to convert; typically a string, but may already be a number.
 * @returns {number}           The number that the string represents, or NaN if no number could be determined.
 */
function fromString(n) {
  if ( typeof n === "number" ) return n;
  if ( (typeof n !== "string") || !n.length ) return NaN;
  n = n.replace(/\s+/g, "");
  return Number(n);
}

// Define properties on the Number environment
Object.defineProperties(Number.prototype, {
  almostEqual: {value: almostEqual},
  between: {value: between},
  ordinalString: {value: ordinalString},
  paddedString: {value: paddedString},
  signedString: {value: signedString},
  toNearest: {value: toNearest}
});
Object.defineProperties(Number, {
  isNumeric: {value: isNumeric},
  fromString: {value: fromString}
});

/**
 * Test whether this set is equal to some other set.
 * Sets are equal if they share the same members, independent of order
 * @param {Set<any>} other       Some other set to compare against
 * @returns {boolean}       Are the sets equal?
 */
function equals(other) {
  if ( !(other instanceof Set ) ) return false;
  if ( other.size !== this.size ) return false;
  for ( const element of this ) {
    if ( !other.has(element) ) return false;
  }
  return true;
}

/**
 * Return the first value from the set.
 * @template T
 * @returns {T|undefined} The first element in the set, or undefined
 */
function first() {
  return this.values().next().value;
}


/**
 * Test whether this set has an intersection with another set.
 * @param {Set} other       Another set to compare against
 * @returns {boolean}       Do the sets intersect?
 */
function intersects(other) {
  return !this.isDisjointFrom(other);
}

/**
 * Test whether this set is a subset of some other set.
 * A set is a subset if all its members are also present in the other set.
 * @param {Set} other       Some other set that may be a subset of this one
 * @returns {boolean}       Is the other set a subset of this one?
 * @deprecated since v13
 */
function isSubset(other) {
  const message = "Set#isSubset is deprecated in favor of the native Set#isSubsetOf.";
  foundry.utils.logCompatibilityWarning(message, {since: 13, until: 15, once: true});
  return this.isSubsetOf(other);
}

/**
 * Convert a set to a JSON object by mapping its contents to an array
 * @template T
 * @returns {T[]}           The set elements as an array.
 */
function toObject() {
  return Array.from(this);
}

/**
 * Test whether every element in this Set satisfies a certain test criterion.
 * @template T
 * @param {(element: T, index: number, set: Set<T>) => boolean} test The test criterion to apply. Positional arguments
 *                                                                   are the value, the index of iteration, and the set
 *                                                                   being tested.
 * @returns {boolean}  Does every element in the set satisfy the test criterion?
 * @see Array#every
 */
function every(test) {
  let i = 0;
  for ( const v of this ) {
    if ( !test(v, i, this) ) return false;
    i++;
  }
  return true;
}

/**
 * Filter this set to create a subset of elements which satisfy a certain test criterion.
 * @template T
 * @param {(element: T, index: number, set: Set) => boolean} test The test criterion to apply. Positional arguments are
 *                                                                the value, the index of iteration, and the set being
 *                                                                filtered.
 * @returns {Set<T>} A new Set containing only elements which satisfy the test criterion.
 * @see Array#filter
 */
function filter(test) {
  const filtered = new Set();
  let i = 0;
  for ( const v of this ) {
    if ( test(v, i, this) ) filtered.add(v);
    i++;
  }
  return filtered;
}

/**
 * Find the first element in this set which satisfies a certain test criterion.
 * @template T
 * @param {(element: T, index: number, set: Set<T>) => boolean} test The test criterion to apply. Positional arguments
 *                                                                   are the value, the index of iteration, and the set
 *                                                                   being searched.
 * @returns {T|undefined} The first element in the set which satisfies the test criterion, or undefined.
 * @see Array#find
 */
function find(test) {
  let i = 0;
  for ( const v of this ) {
    if ( test(v, i, this) ) return v;
    i++;
  }
  return undefined;
}

/**
 * Create a new Set where every element is modified by a provided transformation function.
 * @template T
 * @template U
 * @param {(element: T, index: number, set: Set<T>) => U} transform The transformation function to apply. Positional
 *                                                                  arguments are the value, the index of iteration, and
 *                                                                  the set being transformed.
 * @returns {Set<U>} A new Set of equal size containing transformed elements.
 * @see Array#map
 */
function map(transform) {
  const mapped = new Set();
  let i = 0;
  for ( const v of this ) {
    mapped.add(transform(v, i, this));
    i++;
  }
  if ( mapped.size !== this.size ) {
    throw new Error("The Set#map operation illegally modified the size of the set");
  }
  return mapped;
}

/**
 * Create a new Set with elements that are filtered and transformed by a provided reducer function.
 * @template T
 * @param {(accum: any, element: T, index: number, set: Set<T>) => any} reducer A reducer function applied to each
 *                                                                              value. Positional arguments are the
 *                                                                              accumulator, the value, the index of
 *                                                                              iteration, and the set being reduced.
 * @param {any} [initial]         The initial value of the returned accumulator.
 * @returns {any}                 The final value of the accumulator.
 * @see Array#reduce
 */
function reduce(reducer, initial) {
  let i = 0;
  for ( const v of this ) {
    initial = reducer(initial, v, i, this);
    i++;
  }
  return initial;
}

/**
 * Test whether any element in this Set satisfies a certain test criterion.
 * @template T
 * @param {(element: T, index: number, set: Set<T>) => boolean} test The test criterion to apply. Positional arguments
 *                                                                   are the value, the index of iteration, and the set
 *                                                                   being tested.
 * @returns {boolean} Does any element in the set satisfy the test criterion?
 * @see Array#some
 */
function some(test) {
  let i = 0;
  for ( const v of this ) {
    if ( test(v, i, this) ) return true;
    i++;
  }
  return false;
}

// Assign primitives to Set prototype
Object.defineProperties(Set.prototype, {
  equals: {value: equals},
  every: {value: every},
  filter: {value: filter},
  find: {value: find},
  first: {value: first},
  intersects: {value: intersects},
  isSubset: {value: isSubset},
  map: {value: map},
  reduce: {value: reduce},
  some: {value: some},
  toObject: {value: toObject}
});

/**
 * Capitalize a string, transforming it's first character to a capital letter.
 * @returns {string}
 */
function capitalize() {
  if ( !this.length ) return this;
  return this.charAt(0).toUpperCase() + this.slice(1);
}

/**
 * Compare this string (x) with the other string (y) by comparing each character's Unicode code point value.
 * Returns a negative Number if x < y, a positive Number if x > y, or a zero otherwise.
 * This is the same comparision function that used by Array#sort if the compare function argument is omitted.
 * The result is host/locale-independent.
 * @param {string} other    The other string to compare this string to.
 * @returns {number}
 */
function compare(other) {
  return this < other ? -1 : this > other ? 1 : 0;
}

/**
 * Convert a string to Title Case where the first letter of each word is capitalized.
 * @returns {string}
 */
function titleCase() {
  if (!this.length) return this;
  return this.toLowerCase().split(' ').reduce((parts, word) => {
    if ( !word ) return parts;
    const title = word.replace(word[0], word[0].toUpperCase());
    parts.push(title);
    return parts;
  }, []).join(' ');
}

/**
 * Strip any script tags which were included within a provided string.
 * @returns {string}
 */
function stripScripts() {
  let el = document.createElement("div");
  el.innerHTML = this;
  for ( let s of el.getElementsByTagName("script") ) {
    s.parentNode.removeChild(s);
  }
  return el.innerHTML;
}

/**
 * Map characters to lower case ASCII
 * @type {Record<string, string>}
 */
const CHAR_MAP = JSON.parse('{"$":"dollar","%":"percent","&":"and","<":"less",">":"greater","|":"or","¢":"cent","£":"pound","¤":"currency","¥":"yen","©":"(c)","ª":"a","®":"(r)","º":"o","À":"A","Á":"A","Â":"A","Ã":"A","Ä":"A","Å":"A","Æ":"AE","Ç":"C","È":"E","É":"E","Ê":"E","Ë":"E","Ì":"I","Í":"I","Î":"I","Ï":"I","Ð":"D","Ñ":"N","Ò":"O","Ó":"O","Ô":"O","Õ":"O","Ö":"O","Ø":"O","Ù":"U","Ú":"U","Û":"U","Ü":"U","Ý":"Y","Þ":"TH","ß":"ss","à":"a","á":"a","â":"a","ã":"a","ä":"a","å":"a","æ":"ae","ç":"c","è":"e","é":"e","ê":"e","ë":"e","ì":"i","í":"i","î":"i","ï":"i","ð":"d","ñ":"n","ò":"o","ó":"o","ô":"o","õ":"o","ö":"o","ø":"o","ù":"u","ú":"u","û":"u","ü":"u","ý":"y","þ":"th","ÿ":"y","Ā":"A","ā":"a","Ă":"A","ă":"a","Ą":"A","ą":"a","Ć":"C","ć":"c","Č":"C","č":"c","Ď":"D","ď":"d","Đ":"DJ","đ":"dj","Ē":"E","ē":"e","Ė":"E","ė":"e","Ę":"e","ę":"e","Ě":"E","ě":"e","Ğ":"G","ğ":"g","Ģ":"G","ģ":"g","Ĩ":"I","ĩ":"i","Ī":"i","ī":"i","Į":"I","į":"i","İ":"I","ı":"i","Ķ":"k","ķ":"k","Ļ":"L","ļ":"l","Ľ":"L","ľ":"l","Ł":"L","ł":"l","Ń":"N","ń":"n","Ņ":"N","ņ":"n","Ň":"N","ň":"n","Ő":"O","ő":"o","Œ":"OE","œ":"oe","Ŕ":"R","ŕ":"r","Ř":"R","ř":"r","Ś":"S","ś":"s","Ş":"S","ş":"s","Š":"S","š":"s","Ţ":"T","ţ":"t","Ť":"T","ť":"t","Ũ":"U","ũ":"u","Ū":"u","ū":"u","Ů":"U","ů":"u","Ű":"U","ű":"u","Ų":"U","ų":"u","Ŵ":"W","ŵ":"w","Ŷ":"Y","ŷ":"y","Ÿ":"Y","Ź":"Z","ź":"z","Ż":"Z","ż":"z","Ž":"Z","ž":"z","ƒ":"f","Ơ":"O","ơ":"o","Ư":"U","ư":"u","ǈ":"LJ","ǉ":"lj","ǋ":"NJ","ǌ":"nj","Ș":"S","ș":"s","Ț":"T","ț":"t","˚":"o","Ά":"A","Έ":"E","Ή":"H","Ί":"I","Ό":"O","Ύ":"Y","Ώ":"W","ΐ":"i","Α":"A","Β":"B","Γ":"G","Δ":"D","Ε":"E","Ζ":"Z","Η":"H","Θ":"8","Ι":"I","Κ":"K","Λ":"L","Μ":"M","Ν":"N","Ξ":"3","Ο":"O","Π":"P","Ρ":"R","Σ":"S","Τ":"T","Υ":"Y","Φ":"F","Χ":"X","Ψ":"PS","Ω":"W","Ϊ":"I","Ϋ":"Y","ά":"a","έ":"e","ή":"h","ί":"i","ΰ":"y","α":"a","β":"b","γ":"g","δ":"d","ε":"e","ζ":"z","η":"h","θ":"8","ι":"i","κ":"k","λ":"l","μ":"m","ν":"n","ξ":"3","ο":"o","π":"p","ρ":"r","ς":"s","σ":"s","τ":"t","υ":"y","φ":"f","χ":"x","ψ":"ps","ω":"w","ϊ":"i","ϋ":"y","ό":"o","ύ":"y","ώ":"w","Ё":"Yo","Ђ":"DJ","Є":"Ye","І":"I","Ї":"Yi","Ј":"J","Љ":"LJ","Њ":"NJ","Ћ":"C","Џ":"DZ","А":"A","Б":"B","В":"V","Г":"G","Д":"D","Е":"E","Ж":"Zh","З":"Z","И":"I","Й":"J","К":"K","Л":"L","М":"M","Н":"N","О":"O","П":"P","Р":"R","С":"S","Т":"T","У":"U","Ф":"F","Х":"H","Ц":"C","Ч":"Ch","Ш":"Sh","Щ":"Sh","Ъ":"U","Ы":"Y","Ь":"","Э":"E","Ю":"Yu","Я":"Ya","а":"a","б":"b","в":"v","г":"g","д":"d","е":"e","ж":"zh","з":"z","и":"i","й":"j","к":"k","л":"l","м":"m","н":"n","о":"o","п":"p","р":"r","с":"s","т":"t","у":"u","ф":"f","х":"h","ц":"c","ч":"ch","ш":"sh","щ":"sh","ъ":"u","ы":"y","ь":"","э":"e","ю":"yu","я":"ya","ё":"yo","ђ":"dj","є":"ye","і":"i","ї":"yi","ј":"j","љ":"lj","њ":"nj","ћ":"c","ѝ":"u","џ":"dz","Ґ":"G","ґ":"g","Ғ":"GH","ғ":"gh","Қ":"KH","қ":"kh","Ң":"NG","ң":"ng","Ү":"UE","ү":"ue","Ұ":"U","ұ":"u","Һ":"H","һ":"h","Ә":"AE","ә":"ae","Ө":"OE","ө":"oe","฿":"baht","ა":"a","ბ":"b","გ":"g","დ":"d","ე":"e","ვ":"v","ზ":"z","თ":"t","ი":"i","კ":"k","ლ":"l","მ":"m","ნ":"n","ო":"o","პ":"p","ჟ":"zh","რ":"r","ს":"s","ტ":"t","უ":"u","ფ":"f","ქ":"k","ღ":"gh","ყ":"q","შ":"sh","ჩ":"ch","ც":"ts","ძ":"dz","წ":"ts","ჭ":"ch","ხ":"kh","ჯ":"j","ჰ":"h","Ẁ":"W","ẁ":"w","Ẃ":"W","ẃ":"w","Ẅ":"W","ẅ":"w","ẞ":"SS","Ạ":"A","ạ":"a","Ả":"A","ả":"a","Ấ":"A","ấ":"a","Ầ":"A","ầ":"a","Ẩ":"A","ẩ":"a","Ẫ":"A","ẫ":"a","Ậ":"A","ậ":"a","Ắ":"A","ắ":"a","Ằ":"A","ằ":"a","Ẳ":"A","ẳ":"a","Ẵ":"A","ẵ":"a","Ặ":"A","ặ":"a","Ẹ":"E","ẹ":"e","Ẻ":"E","ẻ":"e","Ẽ":"E","ẽ":"e","Ế":"E","ế":"e","Ề":"E","ề":"e","Ể":"E","ể":"e","Ễ":"E","ễ":"e","Ệ":"E","ệ":"e","Ỉ":"I","ỉ":"i","Ị":"I","ị":"i","Ọ":"O","ọ":"o","Ỏ":"O","ỏ":"o","Ố":"O","ố":"o","Ồ":"O","ồ":"o","Ổ":"O","ổ":"o","Ỗ":"O","ỗ":"o","Ộ":"O","ộ":"o","Ớ":"O","ớ":"o","Ờ":"O","ờ":"o","Ở":"O","ở":"o","Ỡ":"O","ỡ":"o","Ợ":"O","ợ":"o","Ụ":"U","ụ":"u","Ủ":"U","ủ":"u","Ứ":"U","ứ":"u","Ừ":"U","ừ":"u","Ử":"U","ử":"u","Ữ":"U","ữ":"u","Ự":"U","ự":"u","Ỳ":"Y","ỳ":"y","Ỵ":"Y","ỵ":"y","Ỷ":"Y","ỷ":"y","Ỹ":"Y","ỹ":"y","‘":"\'","’":"\'","“":"\\\"","”":"\\\"","†":"+","•":"*","…":"...","₠":"ecu","₢":"cruzeiro","₣":"french franc","₤":"lira","₥":"mill","₦":"naira","₧":"peseta","₨":"rupee","₩":"won","₪":"new shequel","₫":"dong","€":"euro","₭":"kip","₮":"tugrik","₯":"drachma","₰":"penny","₱":"peso","₲":"guarani","₳":"austral","₴":"hryvnia","₵":"cedi","₸":"kazakhstani tenge","₹":"indian rupee","₽":"russian ruble","₿":"bitcoin","℠":"sm","™":"tm","∂":"d","∆":"delta","∑":"sum","∞":"infinity","♥":"love","元":"yuan","円":"yen","﷼":"rial"}');

/**
 * Transform any string into an url-viable slug string
 * @param {object} [options]      Optional arguments which customize how the slugify operation is performed
 * @param {string} [options.replacement="-"]  The replacement character to separate terms, default is '-'
 * @param {boolean} [options.strict=false]    Replace all non-alphanumeric characters, or allow them? Default false
 * @param {boolean} [options.lowercase=true]  Lowercase the string.
 * @returns {string}              The slugified input string
 */
function slugify({replacement='-', strict=false, lowercase=true}={}) {
  let slug = this.split("").reduce((result, char) => result + (CHAR_MAP[char] || char), "").trim();
  if ( lowercase ) slug = slug.toLowerCase();

  // Convert any spaces to the replacement character and de-dupe
  slug = slug.replace(new RegExp('[\\s' + replacement + ']+', 'g'), replacement);

  // If we're being strict, replace anything that is not alphanumeric
  if ( strict ) slug = slug.replace(new RegExp('[^a-zA-Z0-9' + replacement + ']', 'g'), '');
  return slug;
}

// Define properties on the String environment
Object.defineProperties(String.prototype, {
  capitalize: {value: capitalize},
  compare: {value: compare},
  titleCase: {value: titleCase},
  stripScripts: {value: stripScripts},
  slugify: {value: slugify}
});

/**
 * Escape a given input string, prefacing special characters with backslashes for use in a regular expression
 * @param {string} string     The un-escaped input string
 * @returns {string}          The escaped string, suitable for use in regular expression
 */
function escape$1(string) {
  return string.replace(/[-\/\\^$*+?.()|[\]{}]/g, '\\$&');
}

// Define properties on the RegExp environment
Object.defineProperties(RegExp, {
  escape: {value: escape$1}
});

/**
 * Attempt to parse a URL without throwing an error.
 * @param {string} url  The string to parse.
 * @returns {URL|null}  The parsed URL if successful, otherwise null.
 */
function parseSafe(url) {
  try {
    return new URL(url);
  } catch (err) {}
  return null;
}

// Define properties on the URL environment
Object.defineProperties(URL, {
  parseSafe: {value: parseSafe}
});

/**
 * Constant definitions used throughout the Foundry Virtual Tabletop framework.
 * @module CONST
 */


/**
 * The shortened software name
 */
const vtt = "Foundry VTT";

/**
 * The full software name
 */
const VTT = "Foundry Virtual Tabletop";

/**
 * The software website URL
 */
const WEBSITE_URL = "https://foundryvtt.com";

/**
 * The serverless API URL
 */
const WEBSITE_API_URL = "https://api.foundryvtt.com";

/**
 * An ASCII greeting displayed to the client
 * @type {string}
 */
const ASCII = `_______________________________________________________________
 _____ ___  _   _ _   _ ____  ______   __ __     _______ _____
|  ___/ _ \\| | | | \\ | |  _ \\|  _ \\ \\ / / \\ \\   / |_   _|_   _|
| |_ | | | | | | |  \\| | | | | |_) \\ V /   \\ \\ / /  | |   | |
|  _|| |_| | |_| | |\\  | |_| |  _ < | |     \\ V /   | |   | |
|_|   \\___/ \\___/|_| \\_|____/|_| \\_\\|_|      \\_/    |_|   |_|
===============================================================`;

/**
 * Define the allowed ActiveEffect application modes.
 * Other arbitrary mode numbers can be used by systems and modules to identify special behaviors and are ignored
 */
const ACTIVE_EFFECT_MODES = Object.freeze({
  /**
   * Used to denote that the handling of the effect is programmatically provided by a system or module.
   */
  CUSTOM: 0,

  /**
   * Multiplies a numeric base value by the numeric effect value
   * @example
   * 2 (base value) * 3 (effect value) = 6 (derived value)
   */
  MULTIPLY: 1,

  /**
   * Adds a numeric base value to a numeric effect value, or concatenates strings
   * @example
   * 2 (base value) + 3 (effect value) = 5 (derived value)
   * @example
   * "Hello" (base value) + " World" (effect value) = "Hello World"
   */
  ADD: 2,

  /**
   * Keeps the lower value of the base value and the effect value
   * @example
   * 2 (base value), 0 (effect value) = 0 (derived value)
   * @example
   * 2 (base value), 3 (effect value) = 2 (derived value)
   */
  DOWNGRADE: 3,

  /**
   * Keeps the greater value of the base value and the effect value
   * @example
   * 2 (base value), 4 (effect value) = 4 (derived value)
   * @example
   * 2 (base value), 1 (effect value) = 2 (derived value)
   */
  UPGRADE: 4,

  /**
   * Directly replaces the base value with the effect value
   * @example
   * 2 (base value), 4 (effect value) = 4 (derived value)
   */
  OVERRIDE: 5
});

/**
 * Define the string name used for the base document type when specific sub-types are not defined by the system
 */
const BASE_DOCUMENT_TYPE = "base";

/**
 * Define the methods by which a Card can be drawn from a Cards stack
 */
const CARD_DRAW_MODES = Object.freeze({
  /**
   * Draw the first card from the stack
   * Synonymous with `TOP`
   */
  FIRST: 0,

  /**
   * Draw the top card from the stack
   * Synonymous with `FIRST`
   */
  TOP: 0,

  /**
   * Draw the last card from the stack
   * Synonymous with `BOTTOM`
   */
  LAST: 1,

  /**
   * Draw the bottom card from the stack
   * Synonymous with `LAST`
   */
  BOTTOM: 1,

  /**
   * Draw a random card from the stack
   */
  RANDOM: 2
});

/**
 * An enumeration of canvas performance modes.
 */
const CANVAS_PERFORMANCE_MODES = Object.freeze({
  LOW: 0,
  MED: 1,
  HIGH: 2,
  MAX: 3
});

/**
 * @typedef {typeof CANVAS_PERFORMANCE_MODES[keyof typeof CANVAS_PERFORMANCE_MODES]} CanvasPerformanceMode
 */

/**
 * Valid Chat Message styles which affect how the message is presented in the chat log.
 */
const CHAT_MESSAGE_STYLES = /** @type {const} */ ({
  /**
   * An uncategorized chat message
   */
  OTHER: 0,

  /**
   * The message is spoken out of character (OOC).
   * OOC messages will be outlined by the player's color to make them more easily recognizable.
   */
  OOC: 1,

  /**
   * The message is spoken by an associated character.
   */
  IC: 2,

  /**
   * The message is an emote performed by the selected character.
   * Entering "/emote waves his hand." while controlling a character named Simon will send the message, "Simon waves his
   * hand."
   */
  EMOTE: 3
});
Object.defineProperties(CHAT_MESSAGE_STYLES, {
  /** @deprecated since v12 */
  ROLL: {
    get() {
      foundry.utils.logCompatibilityWarning("CONST.CHAT_MESSAGE_STYLES.ROLL is deprecated in favor of defining "
          + "rolls directly in ChatMessage#rolls", {since: 12, until: 14, once: true});
      return 0;
    }
  },
  /** @deprecated since v12 */
  WHISPER: {
    get() {
      foundry.utils.logCompatibilityWarning("CONST.CHAT_MESSAGE_STYLES.WHISPER is deprecated in favor of defining "
        + "whisper recipients directly in ChatMessage#whisper", {since: 12, until: 14, once: true});
      return 0;
    }
  }
});
Object.freeze(CHAT_MESSAGE_STYLES);

/**
 * @typedef {typeof CHAT_MESSAGE_STYLES[keyof typeof CHAT_MESSAGE_STYLES]} ChatMessageStyle
 */

/**
 * Define the set of languages which have built-in support in the core software.
 */
const CORE_SUPPORTED_LANGUAGES = Object.freeze(["en"]);

/**
 * Configure the severity of compatibility warnings.
 */
const COMPATIBILITY_MODES = Object.freeze({
  /**
   * Nothing will be logged
   */
  SILENT: 0,

  /**
   * A message will be logged at the "warn" level
   */
  WARNING: 1,

  /**
   * A message will be logged at the "error" level
   */
  ERROR: 2,

  /**
   * An Error will be thrown
   */
  FAILURE: 3
});


/**
 * Configure custom cursor images to use when interacting with the application.
 */
const CURSOR_STYLES = Object.freeze({
  default: "default",
  "default-down": "default",
  pointer: "pointer",
  "pointer-down": "pointer",
  grab: "grab",
  "grab-down": "grabbing",
  text: "text",
  "text-down": "text"
});

/**
 * The lighting illumination levels which are supported.
 */
const LIGHTING_LEVELS = Object.freeze({
  DARKNESS: -2,
  HALFDARK: -1,
  UNLIT: 0,
  DIM: 1,
  BRIGHT: 2,
  BRIGHTEST: 3
});

/**
 * @typedef {typeof LIGHTING_LEVELS[keyof typeof LIGHTING_LEVELS]} LightingLevel
 */

/**
 * The CSS themes which are currently supported for the V11 Setup menu.
 */
const CSS_THEMES = Object.freeze({
  dark: "THEME.foundry",
  fantasy: "THEME.fantasy",
  scifi: "THEME.scifi"
});

/**
 * The default artwork used for Token images if none is provided
 */
const DEFAULT_TOKEN = "icons/svg/mystery-man.svg";

/**
 * The primary Document types.
 */
const PRIMARY_DOCUMENT_TYPES = Object.freeze([
  "Actor",
  "Adventure",
  "Cards",
  "ChatMessage",
  "Combat",
  "FogExploration",
  "Folder",
  "Item",
  "JournalEntry",
  "Macro",
  "Playlist",
  "RollTable",
  "Scene",
  "Setting",
  "User"
]);

/**
 * The embedded Document types.
 */
const EMBEDDED_DOCUMENT_TYPES = Object.freeze([
  "ActiveEffect",
  "ActorDelta",
  "AmbientLight",
  "AmbientSound",
  "Card",
  "Combatant",
  "CombatantGroup",
  "Drawing",
  "Item",
  "JournalEntryCategory",
  "JournalEntryPage",
  "MeasuredTemplate",
  "Note",
  "PlaylistSound",
  "Region",
  "RegionBehavior",
  "TableResult",
  "Tile",
  "Token",
  "Wall"
]);

/**
 * A listing of all valid Document types, both primary and embedded.
 * @type {readonly ["ActiveEffect", "Actor", "ActorDelta", "Adventure", "AmbientLight", "AmbientSound", "Card", "Cards",
 *   "ChatMessage", "Combat", "Combatant", "CombatantGroup", "Drawing", "FogExploration", "Folder", "Item",
 *   "JournalEntry", "JournalEntryCategory", "JournalEntryPage", "Macro", "MeasuredTemplate", "Note", "Playlist",
 *   "PlaylistSound", "Region", "RegionBehavior", "RollTable", "Scene", "Setting", "TableResult", "Tile", "Token",
 *   "User", "Wall"]}
 */
const ALL_DOCUMENT_TYPES = Object.freeze(Array.from(new Set([
  ...PRIMARY_DOCUMENT_TYPES,
  ...EMBEDDED_DOCUMENT_TYPES
])).sort());

/**
 * The allowed primary Document types which may exist within a World.
 */
const WORLD_DOCUMENT_TYPES = Object.freeze([
  "Actor",
  "Cards",
  "ChatMessage",
  "Combat",
  "FogExploration",
  "Folder",
  "Item",
  "JournalEntry",
  "Macro",
  "Playlist",
  "RollTable",
  "Scene",
  "Setting",
  "User"
]);

/**
 * The allowed primary Document types which may exist within a Compendium pack.
 */
const COMPENDIUM_DOCUMENT_TYPES = Object.freeze([
  "Actor",
  "Adventure",
  "Cards",
  "Item",
  "JournalEntry",
  "Macro",
  "Playlist",
  "RollTable",
  "Scene"
]);

/**
 * Define the allowed ownership levels for a Document.
 * Each level is assigned a value in ascending order.
 * Higher levels grant more permissions.
 * @see {@link https://foundryvtt.com/article/users/}
 */
const DOCUMENT_OWNERSHIP_LEVELS = Object.freeze({
  /**
   * The User inherits permissions from the parent Folder.
   */
  INHERIT: -1,

  /**
   * Restricts the associated Document so that it may not be seen by this User.
   */
  NONE: 0,

  /**
   * Allows the User to interact with the Document in basic ways, allowing them to see it in sidebars and see only
   * limited aspects of its contents. The limits of this interaction are defined by the game system being used.
   */
  LIMITED: 1,

  /**
   * Allows the User to view this Document as if they were owner, but prevents them from making any changes to it.
   */
  OBSERVER: 2,

  /**
   * Allows the User to view and make changes to the Document as its owner. Owned documents cannot be deleted by anyone
   * other than a gamemaster level User.
   */
  OWNER: 3
});

/**
 * @typedef {typeof DOCUMENT_OWNERSHIP_LEVELS[keyof typeof DOCUMENT_OWNERSHIP_LEVELS]} DocumentOwnershipNumber
 * @typedef {keyof typeof DOCUMENT_OWNERSHIP_LEVELS|DocumentOwnershipNumber} DocumentOwnershipLevel
 */

/**
 * Meta ownership levels that are used in the UI but never stored.
 */
const DOCUMENT_META_OWNERSHIP_LEVELS = Object.freeze({
  DEFAULT: -20,
  NOCHANGE: -10
});

/**
 * Define the allowed Document types which may be dynamically linked in chat
 */
const DOCUMENT_LINK_TYPES = Object.freeze(["Actor", "Cards", "Item", "Scene", "JournalEntry", "Macro",
  "RollTable", "PlaylistSound"]);

/**
 * The supported dice roll visibility modes
 * @see {@link https://foundryvtt.com/article/dice/}
 */
const DICE_ROLL_MODES = Object.freeze({
  /**
   * This roll is visible to all players.
   */
  PUBLIC: "publicroll",

  /**
   * Rolls of this type are only visible to the player that rolled and any Game Master users.
   */
  PRIVATE: "gmroll",

  /**
   * A private dice roll only visible to Gamemaster users. The rolling player will not see the result of their own roll.
   */
  BLIND: "blindroll",

  /**
   * A private dice roll which is only visible to the user who rolled it.
   */
  SELF: "selfroll"
});

/**
 * The allowed fill types which a Drawing object may display
 * @see {@link https://foundryvtt.com/article/drawings/}
 */
const DRAWING_FILL_TYPES = Object.freeze({
  /**
   * The drawing is not filled
   */
  NONE: 0,

  /**
   * The drawing is filled with a solid color
   */
  SOLID: 1,

  /**
   * The drawing is filled with a tiled image pattern
   */
  PATTERN: 2
});

/**
 * Define the allowed Document types which Folders may contain
 */
const FOLDER_DOCUMENT_TYPES = Object.freeze(["Actor", "Adventure", "Item", "Scene", "JournalEntry", "Playlist",
  "RollTable", "Cards", "Macro", "Compendium"]);

/**
 * The maximum allowed level of depth for Folder nesting
 */
const FOLDER_MAX_DEPTH = 4;

/**
 * A list of allowed game URL names
 */
const GAME_VIEWS = Object.freeze(["game", "stream"]);

/**
 * The directions of movement.
 */
const MOVEMENT_DIRECTIONS = Object.freeze({
  UP: 0x1,
  DOWN: 0x2,
  LEFT: 0x4,
  RIGHT: 0x8,
  UP_LEFT: /** @type {5} */ (0x1 | 0x4),
  UP_RIGHT: /** @type {9} */ (0x1 | 0x8),
  DOWN_LEFT: /** @type {6} */ (0x2 | 0x4),
  DOWN_RIGHT: /** @type {10} */ (0x2 | 0x8),
  DESCEND: 0x10,
  ASCEND: 0x20
});

/**
 * The minimum allowed grid size which is supported by the software
 */
const GRID_MIN_SIZE = 20;

/**
 * The allowed Grid types which are supported by the software
 * @see {@link https://foundryvtt.com/article/scenes/}
 */
const GRID_TYPES = Object.freeze({
  /**
   * No fixed grid is used on this Scene allowing free-form point-to-point measurement without grid lines.
   */
  GRIDLESS: 0,

  /**
   * A square grid is used with width and height of each grid space equal to the chosen grid size.
   */
  SQUARE: 1,

  /**
   * A row-wise hexagon grid (pointy-topped) where odd-numbered rows are offset.
   */
  HEXODDR: 2,

  /**
   * A row-wise hexagon grid (pointy-topped) where even-numbered rows are offset.
   */
  HEXEVENR: 3,

  /**
   * A column-wise hexagon grid (flat-topped) where odd-numbered columns are offset.
   */
  HEXODDQ: 4,

  /**
   * A column-wise hexagon grid (flat-topped) where even-numbered columns are offset.
   */
  HEXEVENQ: 5
});

/**
 * @typedef {typeof GRID_TYPES[keyof typeof GRID_TYPES]} GridType
 */

/**
 * The different rules to define and measure diagonal distance/cost in a square grid.
 * The description of each option refers to the distance/cost of moving diagonally relative to the distance/cost of a
 * horizontal or vertical move.
 */
const GRID_DIAGONALS = Object.freeze({
  /**
   * The diagonal distance is 1. Diagonal movement costs the same as horizontal/vertical movement.
   */
  EQUIDISTANT: 0,

  /**
   * The diagonal distance is √2. Diagonal movement costs √2 times as much as horizontal/vertical movement.
   */
  EXACT: 1,

  /**
   * The diagonal distance is 1.5. Diagonal movement costs 1.5 times as much as horizontal/vertical movement.
   */
  APPROXIMATE: 2,

  /**
   * The diagonal distance is 2. Diagonal movement costs 2 times as much as horizontal/vertical movement.
   */
  RECTILINEAR: 3,

  /**
   * The diagonal distance alternates between 1 and 2 starting at 1.
   * The first diagonal movement costs the same as horizontal/vertical movement
   * The second diagonal movement costs 2 times as much as horizontal/vertical movement.
   * And so on...
   */
  ALTERNATING_1: 4,

  /**
   * The diagonal distance alternates between 2 and 1 starting at 2.
   * The first diagonal movement costs 2 times as much as horizontal/vertical movement.
   * The second diagonal movement costs the same as horizontal/vertical movement.
   * And so on...
   */
  ALTERNATING_2: 5,

  /**
   * The diagonal distance is ∞. Diagonal movement is not allowed/possible.
   */
  ILLEGAL: 6
});

/**
 * @typedef {typeof GRID_DIAGONALS[keyof typeof GRID_DIAGONALS]} GridDiagonalRule
 */

/**
 * The grid snapping modes.
 */
const GRID_SNAPPING_MODES = Object.freeze({
  /**
   * Nearest center point.
   */
  CENTER: 0x1,

  /**
   * Nearest edge midpoint.
   */
  EDGE_MIDPOINT: 0x2,

  /**
   * Nearest top-left vertex.
   */
  TOP_LEFT_VERTEX: 0x10,

  /**
   * Nearest top-right vertex.
   */
  TOP_RIGHT_VERTEX: 0x20,

  /**
   * Nearest bottom-left vertex.
   */
  BOTTOM_LEFT_VERTEX: 0x40,

  /**
   * Nearest bottom-right vertex.
   */
  BOTTOM_RIGHT_VERTEX: 0x80,

  /**
   * Nearest vertex.
   * Alias for `TOP_LEFT_VERTEX | TOP_RIGHT_VERTEX | BOTTOM_LEFT_VERTEX | BOTTOM_RIGHT_VERTEX`.
   */
  VERTEX: 0xF0,

  /**
   * Nearest top-left corner.
   */
  TOP_LEFT_CORNER: 0x100,

  /**
   * Nearest top-right corner.
   */
  TOP_RIGHT_CORNER: 0x200,

  /**
   * Nearest bottom-left corner.
   */
  BOTTOM_LEFT_CORNER: 0x400,

  /**
   * Nearest bottom-right corner.
   */
  BOTTOM_RIGHT_CORNER: 0x800,

  /**
   * Nearest corner.
   * Alias for `TOP_LEFT_CORNER | TOP_RIGHT_CORNER | BOTTOM_LEFT_CORNER | BOTTOM_RIGHT_CORNER`.
   */
  CORNER: 0xF00,

  /**
   * Nearest top side midpoint.
   */
  TOP_SIDE_MIDPOINT: 0x1000,

  /**
   * Nearest bottom side midpoint.
   */
  BOTTOM_SIDE_MIDPOINT: 0x2000,

  /**
   * Nearest left side midpoint.
   */
  LEFT_SIDE_MIDPOINT: 0x4000,

  /**
   * Nearest right side midpoint.
   */
  RIGHT_SIDE_MIDPOINT: 0x8000,

  /**
   * Nearest side midpoint.
   * Alias for `TOP_SIDE_MIDPOINT | BOTTOM_SIDE_MIDPOINT | LEFT_SIDE_MIDPOINT | RIGHT_SIDE_MIDPOINT`.
   */
  SIDE_MIDPOINT: 0xF000
});

/**
 * A list of supported setup URL names
 */
const SETUP_VIEWS = Object.freeze(["auth", "license", "setup", "players", "join", "update"]);

/**
 * An Array of valid MacroAction scope values
 */
const MACRO_SCOPES = Object.freeze(["global", "actors", "actor"]);

/**
 * An enumeration of valid Macro types
 * @see {@link https://foundryvtt.com/article/macros/}
 */
const MACRO_TYPES = Object.freeze({
  /**
   * Complex and powerful macros which leverage the FVTT API through plain JavaScript to perform functions as simple or
   * as advanced as you can imagine.
   */
  SCRIPT: "script",

  /**
   * Simple and easy to use, chat macros post pre-defined chat messages to the chat log when executed. All users can
   * execute chat macros by default.
   */
  CHAT: "chat"
});

/**
 * The allowed channels for audio playback.
 */
const AUDIO_CHANNELS = Object.freeze({
  music: "AUDIO.CHANNELS.MUSIC.label",
  environment: "AUDIO.CHANNELS.ENVIRONMENT.label",
  interface: "AUDIO.CHANNELS.INTERFACE.label"
});

/**
 * The allowed playback modes for an audio Playlist
 * @see {@link https://foundryvtt.com/article/playlists/}
 */
const PLAYLIST_MODES = Object.freeze({
  /**
   * The playlist does not play on its own, only individual Sound tracks played as a soundboard.
   */
  DISABLED: -1,

  /**
   * The playlist plays sounds one at a time in sequence.
   */
  SEQUENTIAL: 0,

  /**
   * The playlist plays sounds one at a time in randomized order.
   */
  SHUFFLE: 1,

  /**
   * The playlist plays all contained sounds at the same time.
   */
  SIMULTANEOUS: 2
});

/**
 * The available sort modes for an audio Playlist.
 * @see {@link https://foundryvtt.com/article/playlists/}
 */
const PLAYLIST_SORT_MODES = Object.freeze({
  /**
   * Sort sounds alphabetically.
   */
  ALPHABETICAL: "a",

  /**
   * Sort sounds by manual drag-and-drop.
   */
  MANUAL: "m"
});

/**
 * The available modes for searching within a DirectoryCollection
 */
const DIRECTORY_SEARCH_MODES = Object.freeze({
  FULL: "full",
  NAME: "name"
});

/**
 * The allowed package types
 */
const PACKAGE_TYPES$1 = Object.freeze(["world", "system", "module"]);

/**
 * Encode the reasons why a package may be available or unavailable for use
 */
const PACKAGE_AVAILABILITY_CODES = Object.freeze({
  /**
   * Package availability could not be determined
   */
  UNKNOWN: 0,

  /**
   * The Package is verified to be compatible with the current core software build
   */
  VERIFIED: 1,

  /**
   * Package is available for use, but not verified for the current core software build
   */
  UNVERIFIED_BUILD: 2,

  /**
   * One or more installed system is incompatible with the Package.
   */
  UNVERIFIED_SYSTEM: 3,

  /**
   * Package is available for use, but not verified for the current core software generation
   */
  UNVERIFIED_GENERATION: 4,

  /**
   * The System that the Package relies on is not available
   */
  MISSING_SYSTEM: 5,

  /**
   * A dependency of the Package is not available
   */
  MISSING_DEPENDENCY: 6,

  /**
   * The Package is compatible with an older version of Foundry than the currently installed version
   */
  REQUIRES_CORE_DOWNGRADE: 7,

  /**
   * The Package is compatible with a newer version of Foundry than the currently installed version, and that version is
   * Stable
   */
  REQUIRES_CORE_UPGRADE_STABLE: 8,

  /**
   * The Package is compatible with a newer version of Foundry than the currently installed version, and that version is
   * not yet Stable
   */
  REQUIRES_CORE_UPGRADE_UNSTABLE: 9,

  /**
   * A required dependency is not compatible with the current version of Foundry
   */
  REQUIRES_DEPENDENCY_UPDATE: 10
});

/**
 * A safe password string which can be displayed
 * @type {"••••••••••••••••"}
 */
const PASSWORD_SAFE_STRING = "•".repeat(16);

/**
 * The allowed software update channels
 */
const SOFTWARE_UPDATE_CHANNELS = Object.freeze({
  /**
   * The Stable release channel
   */
  stable: "SETUP.UpdateStable",

  /**
   * The User Testing release channel
   */
  testing: "SETUP.UpdateTesting",

  /**
   * The Development release channel
   */
  development: "SETUP.UpdateDevelopment",

  /**
   * The Prototype release channel
   */
  prototype: "SETUP.UpdatePrototype"
});

/**
 * The default sorting density for manually ordering child objects within a parent
 */
const SORT_INTEGER_DENSITY = 100000;

/**
 * The allowed types of a TableResult document
 * @see {@link https://foundryvtt.com/article/roll-tables/}
 */
const TABLE_RESULT_TYPES = /** @type {const} */ ({
  /**
   *  Plain text or HTML scripted entries which will be output to Chat.
   */
  TEXT: "text",

  /**
   * An in-World Document reference which will be linked to in the chat message.
   */
  DOCUMENT: "document"
});
Object.defineProperties(TABLE_RESULT_TYPES, {
  /** @deprecated since v13 */
  COMPENDIUM: {
    get() {
      const message = "CONST.TABLE_RESULT_TYPES.COMPENDIUM is is deprecated in favor of CONST.TABLE_RESULT_TYPES.DOCUMENT"
      + ' due to the "compendium" being merged with the "document" type.';
      foundry.utils.logCompatibilityWarning(message, {since: 13, until: 15, once: true});
      return TABLE_RESULT_TYPES.DOCUMENT;
    }
  }
});
Object.freeze(TABLE_RESULT_TYPES);

/**
 * The allowed formats of a Journal Entry Page.
 * @see {@link https://foundryvtt.com/article/journal/}
 */
const JOURNAL_ENTRY_PAGE_FORMATS = Object.freeze({
  /**
   * The page is formatted as HTML.
   */
  HTML: 1,

  /**
   * The page is formatted as Markdown.
   */
  MARKDOWN: 2
});

/**
 * Define the valid anchor locations for a Tooltip displayed on a Placeable Object
 * @see {@link foundry.helpers.interaction.TooltipManager}
 */
const TEXT_ANCHOR_POINTS = Object.freeze({
  /**
   * Anchor the tooltip to the center of the element.
   */
  CENTER: 0,

  /**
   * Anchor the tooltip to the bottom of the element.
   */
  BOTTOM: 1,

  /**
   * Anchor the tooltip to the top of the element.
   */
  TOP: 2,

  /**
   * Anchor the tooltip to the left of the element.
   */
  LEFT: 3,

  /**
   * Anchor the tooltip to the right of the element.
   */
  RIGHT: 4
});

/**
 * @typedef {typeof TEXT_ANCHOR_POINTS[keyof typeof TEXT_ANCHOR_POINTS]} TextAnchorPoint
 */

/**
 * Define the valid occlusion modes which a tile can use
 * @see {@link https://foundryvtt.com/article/tiles/}
 */
const OCCLUSION_MODES = Object.freeze({
  /**
   * Turns off occlusion, making the tile never fade while tokens are under it.
   */
  NONE: 0,

  /**
   * Causes the whole tile to fade when an actor token moves under it.
   */
  FADE: 1,

  // ROOF: 2,  This mode is no longer supported so we don't use 2 for any other mode

  /**
   * Causes the tile to reveal the background in the vicinity of an actor token under it. The radius is determined by
   * the token's size.
   */
  RADIAL: 3,

  /**
   * Causes the tile to be partially revealed based on the vision of the actor, which does not need to be under the tile
   * to see what's beneath it.
   * This is useful for rooves on buildings where players could see through a window or door, viewing only a portion of
   * what is obscured by the roof itself.
   */
  VISION: 4
});

/**
 * Alias for old tile occlusion modes definition
 */
const TILE_OCCLUSION_MODES = OCCLUSION_MODES;

/**
 * The occlusion modes that define the set of tokens that trigger occlusion.
 */
const TOKEN_OCCLUSION_MODES = Object.freeze({

  /**
   * Owned tokens that aren't hidden.
   */
  OWNED: 0x1,

  /**
   * Controlled tokens.
   */
  CONTROLLED: 0x2,

  /**
   * Hovered tokens that are visible.
   */
  HOVERED: 0x4,

  /**
   * Highlighted tokens that are visible.
   */
  HIGHLIGHTED: 0x8,

  /**
   * All visible tokens.
   */
  VISIBLE: 0x10
});

/**
 * Describe the various thresholds of token control upon which to show certain pieces of information
 * @see {@link https://foundryvtt.com/article/tokens/}
 */
const TOKEN_DISPLAY_MODES = Object.freeze({
  /**
   * No information is displayed.
   */
  NONE: 0,

  /**
   * Displayed when the token is controlled.
   */
  CONTROL: 10,

  /**
   * Displayed when hovered by a GM or a user who owns the actor.
   */
  OWNER_HOVER: 20,

  /**
   * Displayed when hovered by any user.
   */
  HOVER: 30,

  /**
   * Always displayed for a GM or for a user who owns the actor.
   */
  OWNER: 40,

  /**
   * Always displayed for everyone.
   */
  ALWAYS: 50
});

/**
 * @typedef {typeof TOKEN_DISPLAY_MODES[keyof typeof TOKEN_DISPLAY_MODES]} TokenDisplayMode
 */

/**
 * The allowed Token disposition types
 * @see {@link https://foundryvtt.com/article/tokens/}
 */
const TOKEN_DISPOSITIONS = Object.freeze({
  /**
   * Displayed with a purple borders for owners and with no borders for others (and no pointer change).
   */
  SECRET: -2,

  /**
   * Displayed as an enemy with a red border.
   */
  HOSTILE: -1,

  /**
   * Displayed as neutral with a yellow border.
   */
  NEUTRAL: 0,

  /**
   * Displayed as an ally with a cyan border.
   */
  FRIENDLY: 1
});

/**
 * The allowed token turn markers modes.
 */
const TOKEN_TURN_MARKER_MODES = Object.freeze({
  /**
   * The turn marker is disabled for this token.
   */
  DISABLED: 0,

  /**
   * The turn marker for this token is using the combat tracker settings (which could be disabled).
   */
  DEFAULT: 1,

  /**
   * The turn marker is using the token settings (unless the combat tracker turn marker setting is disabled)
   */
  CUSTOM: 2
});

/**
 * The possible shapes of Tokens.
 */
const TOKEN_SHAPES = Object.freeze({
  /**
   * Ellipse (Variant 1)
   */
  ELLIPSE_1: 0,

  /**
   * Ellipse (Variant 2)
   */
  ELLIPSE_2: 1,

  /**
   * Trapezoid (Variant 1)
   */
  TRAPEZOID_1: 2,

  /**
   * Trapezoid (Variant 2)
   */
  TRAPEZOID_2: 3,

  /**
   * Rectangle (Variant 1)
   */
  RECTANGLE_1: 4,

  /**
   * Rectangle (Variant 2)
   */
  RECTANGLE_2: 5
});

/**
 * @typedef {typeof TOKEN_SHAPES[keyof typeof TOKEN_SHAPES]} TokenShapeType
 */

/**
 * Define the allowed User permission levels.
 * Each level is assigned a value in ascending order. Higher levels grant more permissions.
 * @see {@link https://foundryvtt.com/article/users/}
 */
const USER_ROLES = Object.freeze({
  /**
   * The User is blocked from taking actions in Foundry Virtual Tabletop.
   * You can use this role to temporarily or permanently ban a user from joining the game.
   */
  NONE: 0,

  /**
   * The User is able to join the game with permissions available to a standard player.
   * They cannot take some more advanced actions which require Trusted permissions, but they have the basic
   * functionalities needed to operate in the virtual tabletop.
   */
  PLAYER: 1,

  /**
   * Similar to the Player role, except a Trusted User has the ability to perform some more advanced actions like create
   * drawings, measured templates, or even to (optionally) upload media files to the server.
   */
  TRUSTED: 2,

  /**
   * A special User who has many of the same in-game controls as a Game Master User, but does not have the ability to
   * perform administrative actions like changing User roles or modifying World-level settings.
   */
  ASSISTANT: 3,

  /**
   * A special User who has administrative control over this specific World.
   * Game Masters behave quite differently than Players in that they have the ability to see all Documents and Objects
   * within the world as well as the capability to configure World settings.
   */
  GAMEMASTER: 4
});

/**
 * Invert the User Role mapping to recover role names from a role integer
 * @type {{0: "NONE"; 1: "PLAYER"; 2: "TRUSTED"; 3: "ASSISTANT"; 4: "GAMEMASTER"}}
 * @see {@link CONST.USER_ROLES}
 */
const USER_ROLE_NAMES = Object.entries(USER_ROLES).reduce((obj, r) => {
  obj[r[1]] = r[0];
  return obj;
}, {});

/**
 * An enumeration of the allowed types for a MeasuredTemplate embedded document
 * @see {@link https://foundryvtt.com/article/measurement/}
 */
const MEASURED_TEMPLATE_TYPES = Object.freeze({
  /**
   * Circular templates create a radius around the starting point.
   */
  CIRCLE: "circle",

  /**
   * Cones create an effect in the shape of a triangle or pizza slice from the starting point.
   */
  CONE: "cone",

  /**
   * A rectangle uses the origin point as a corner, treating the origin as being inside of the rectangle's area.
   */
  RECTANGLE: "rect",

  /**
   * A ray creates a single line that is one square in width and as long as you want it to be.
   */
  RAY: "ray"
});

/**
 * Define the recognized User capabilities which individual Users or role levels may be permitted to perform
 */
const USER_PERMISSIONS = deepFreeze({
  ACTOR_CREATE: {
    label: "PERMISSION.ActorCreate",
    hint: "PERMISSION.ActorCreateHint",
    disableGM: false,
    defaultRole: USER_ROLES.ASSISTANT
  },
  BROADCAST_AUDIO: {
    label: "PERMISSION.BroadcastAudio",
    hint: "PERMISSION.BroadcastAudioHint",
    disableGM: true,
    defaultRole: USER_ROLES.TRUSTED
  },
  BROADCAST_VIDEO: {
    label: "PERMISSION.BroadcastVideo",
    hint: "PERMISSION.BroadcastVideoHint",
    disableGM: true,
    defaultRole: USER_ROLES.TRUSTED
  },
  CARDS_CREATE: {
    label: "PERMISSION.CardsCreate",
    hint: "PERMISSION.CardsCreateHint",
    disableGM: false,
    defaultRole: USER_ROLES.ASSISTANT
  },
  DRAWING_CREATE: {
    label: "PERMISSION.DrawingCreate",
    hint: "PERMISSION.DrawingCreateHint",
    disableGM: false,
    defaultRole: USER_ROLES.TRUSTED
  },
  ITEM_CREATE: {
    label: "PERMISSION.ItemCreate",
    hint: "PERMISSION.ItemCreateHint",
    disableGM: false,
    defaultRole: USER_ROLES.ASSISTANT
  },
  FILES_BROWSE: {
    label: "PERMISSION.FilesBrowse",
    hint: "PERMISSION.FilesBrowseHint",
    disableGM: false,
    defaultRole: USER_ROLES.TRUSTED
  },
  FILES_UPLOAD: {
    label: "PERMISSION.FilesUpload",
    hint: "PERMISSION.FilesUploadHint",
    disableGM: false,
    defaultRole: USER_ROLES.ASSISTANT
  },
  JOURNAL_CREATE: {
    label: "PERMISSION.JournalCreate",
    hint: "PERMISSION.JournalCreateHint",
    disableGM: false,
    defaultRole: USER_ROLES.TRUSTED
  },
  MACRO_SCRIPT: {
    label: "PERMISSION.MacroScript",
    hint: "PERMISSION.MacroScriptHint",
    disableGM: false,
    defaultRole: USER_ROLES.PLAYER
  },
  MANUAL_ROLLS: {
    label: "PERMISSION.ManualRolls",
    hint: "PERMISSION.ManualRollsHint",
    disableGM: true,
    defaultRole: USER_ROLES.TRUSTED
  },
  MESSAGE_WHISPER: {
    label: "PERMISSION.MessageWhisper",
    hint: "PERMISSION.MessageWhisperHint",
    disableGM: false,
    defaultRole: USER_ROLES.PLAYER
  },
  NOTE_CREATE: {
    label: "PERMISSION.NoteCreate",
    hint: "PERMISSION.NoteCreateHint",
    disableGM: false,
    defaultRole: USER_ROLES.TRUSTED
  },
  PING_CANVAS: {
    label: "PERMISSION.PingCanvas",
    hint: "PERMISSION.PingCanvasHint",
    disableGM: true,
    defaultRole: USER_ROLES.PLAYER
  },
  PLAYLIST_CREATE: {
    label: "PERMISSION.PlaylistCreate",
    hint: "PERMISSION.PlaylistCreateHint",
    disableGM: false,
    defaultRole: USER_ROLES.ASSISTANT
  },
  SETTINGS_MODIFY: {
    label: "PERMISSION.SettingsModify",
    hint: "PERMISSION.SettingsModifyHint",
    disableGM: false,
    defaultRole: USER_ROLES.ASSISTANT
  },
  SHOW_CURSOR: {
    label: "PERMISSION.ShowCursor",
    hint: "PERMISSION.ShowCursorHint",
    disableGM: true,
    defaultRole: USER_ROLES.PLAYER
  },
  SHOW_RULER: {
    label: "PERMISSION.ShowRuler",
    hint: "PERMISSION.ShowRulerHint",
    disableGM: true,
    defaultRole: USER_ROLES.PLAYER
  },
  TEMPLATE_CREATE: {
    label: "PERMISSION.TemplateCreate",
    hint: "PERMISSION.TemplateCreateHint",
    disableGM: false,
    defaultRole: USER_ROLES.PLAYER
  },
  TOKEN_CREATE: {
    label: "PERMISSION.TokenCreate",
    hint: "PERMISSION.TokenCreateHint",
    disableGM: false,
    defaultRole: USER_ROLES.ASSISTANT
  },
  TOKEN_DELETE: {
    label: "PERMISSION.TokenDelete",
    hint: "PERMISSION.TokenDeleteHint",
    disableGM: false,
    defaultRole: USER_ROLES.ASSISTANT
  },
  TOKEN_CONFIGURE: {
    label: "PERMISSION.TokenConfigure",
    hint: "PERMISSION.TokenConfigureHint",
    disableGM: false,
    defaultRole: USER_ROLES.TRUSTED
  },
  WALL_DOORS: {
    label: "PERMISSION.WallDoors",
    hint: "PERMISSION.WallDoorsHint",
    disableGM: false,
    defaultRole: USER_ROLES.PLAYER
  },
  QUERY_USER: {
    label: "PERMISSION.QueryUser",
    hint: "PERMISSION.QueryUserHint",
    disableGM: false,
    defaultRole: USER_ROLES.PLAYER
  }
});

/**
 * The allowed directions of effect that a Wall can have
 * @see {@link https://foundryvtt.com/article/walls/}
 */
const WALL_DIRECTIONS = Object.freeze({
  /**
   * The wall collides from both directions.
   */
  BOTH: 0,

  /**
   * The wall collides only when a ray strikes its left side.
   */
  LEFT: 1,

  /**
   * The wall collides only when a ray strikes its right side.
   */
  RIGHT: 2
});

/**
 * @typedef {typeof WALL_DIRECTIONS[keyof typeof WALL_DIRECTIONS]} WallDirection
 */

/**
 * The allowed door types which a Wall may contain
 * @see {@link https://foundryvtt.com/article/walls/}
 */
const WALL_DOOR_TYPES = Object.freeze({
  /**
   * The wall does not contain a door.
   */
  NONE: 0,

  /**
   *  The wall contains a regular door.
   */
  DOOR: 1,

  /**
   * The wall contains a secret door.
   */
  SECRET: 2
});

/**
 * The allowed door states which may describe a Wall that contains a door
 * @see {@link https://foundryvtt.com/article/walls/}
 */
const WALL_DOOR_STATES = Object.freeze({
  /**
   * The door is closed.
   */
  CLOSED: 0,

  /**
   * The door is open.
   */
  OPEN: 1,

  /**
   * The door is closed and locked.
   */
  LOCKED: 2
});

/**
 * The possible ways to interact with a door
 */
const WALL_DOOR_INTERACTIONS = Object.freeze(["open", "close", "lock", "unlock", "test"]);

/**
 * The wall properties which restrict the way interaction occurs with a specific wall
 */
const WALL_RESTRICTION_TYPES = Object.freeze(["light", "sight", "sound", "move"]);

/**
 * @typedef {typeof WALL_RESTRICTION_TYPES[number]} WallRestrictionType
 */

/**
 * The types of sensory collision which a Wall may impose
 * @see {@link https://foundryvtt.com/article/walls/}
 */
const WALL_SENSE_TYPES = Object.freeze({
  /**
   * Senses do not collide with this wall.
   */
  NONE: 0,

  /**
   * Senses collide with this wall.
   */
  LIMITED: 10,

  /**
   * Senses collide with the second intersection, bypassing the first.
   */
  NORMAL: 20,

  /**
   * Senses bypass the wall within a certain proximity threshold.
   */
  PROXIMITY: 30,

  /**
   * Senses bypass the wall outside a certain proximity threshold.
   */
  DISTANCE: 40
});

/**
 * @typedef {typeof WALL_SENSE_TYPES[keyof typeof WALL_SENSE_TYPES]} WallSenseType
 */

/**
 * The types of movement collision which a Wall may impose
 * @see {@link https://foundryvtt.com/article/walls/}
 */
const WALL_MOVEMENT_TYPES = Object.freeze({
  /**
   * Movement does not collide with this wall.
   */
  NONE: WALL_SENSE_TYPES.NONE,

  /**
   * Movement collides with this wall.
   */
  NORMAL: WALL_SENSE_TYPES.NORMAL
});

/**
 * The possible precedence values a Keybinding might run in
 * @see {@link https://foundryvtt.com/article/keybinds/}
 */
const KEYBINDING_PRECEDENCE = Object.freeze({
  /**
   * Runs in the first group along with other PRIORITY keybindings.
   */
  PRIORITY: 0,

  /**
   * Runs after the PRIORITY group along with other NORMAL keybindings.
   */
  NORMAL: 1,

  /**
   * Runs in the last group along with other DEFERRED keybindings.
   */
  DEFERRED: 2
});

/**
 * Directories in the public storage path.
 */
const FILE_PICKER_PUBLIC_DIRS = Object.freeze([
  "cards", "css", "fonts", "icons", "lang", "scripts", "sounds", "ui"
]);

/**
 * The allowed set of HTML template extensions
 */
const HTML_FILE_EXTENSIONS = Object.freeze({
  handlebars: "text/x-handlebars-template",
  hbs: "text/x-handlebars-template",
  html: "text/html"
});

/**
 * The supported file extensions for image-type files, and their corresponding mime types.
 */
const IMAGE_FILE_EXTENSIONS = Object.freeze({
  apng: "image/apng",
  avif: "image/avif",
  bmp: "image/bmp",
  gif: "image/gif",
  jpeg: "image/jpeg",
  jpg: "image/jpeg",
  png: "image/png",
  svg: "image/svg+xml",
  tiff: "image/tiff",
  webp: "image/webp"
});

/**
 * The supported file extensions for video-type files, and their corresponding mime types.
 */
const VIDEO_FILE_EXTENSIONS = Object.freeze({
  m4v: "video/mp4",
  mp4: "video/mp4",
  ogv: "video/ogg",
  webm: "video/webm"
});

/**
 * The supported file extensions for audio-type files, and their corresponding mime types.
 */
const AUDIO_FILE_EXTENSIONS = Object.freeze({
  aac: "audio/aac",
  flac: "audio/flac",
  m4a: "audio/mp4",
  mid: "audio/midi",
  mp3: "audio/mpeg",
  ogg: "audio/ogg",
  opus: "audio/opus",
  wav: "audio/wav",
  webm: "audio/webm"
});

/**
 * The supported file extensions for text files, and their corresponding mime types.
 */
const TEXT_FILE_EXTENSIONS = Object.freeze({
  csv: "text/csv",
  json: "application/json",
  md: "text/markdown",
  pdf: "application/pdf",
  tsv: "text/tab-separated-values",
  txt: "text/plain",
  xml: "application/xml",
  yml: "application/yaml",
  yaml: "application/yaml"
});

/**
 * Supported file extensions for font files, and their corresponding mime types.
 */
const FONT_FILE_EXTENSIONS = Object.freeze({
  otf: "font/otf",
  ttf: "font/ttf",
  woff: "font/woff",
  woff2: "font/woff2"
});

/**
 * Supported file extensions for 3D files, and their corresponding mime types.
 */
const GRAPHICS_FILE_EXTENSIONS = Object.freeze({
  fbx: "application/octet-stream",
  glb: "model/gltf-binary",
  gltf: "model/gltf+json",
  mtl: "model/mtl",
  obj: "model/obj",
  stl: "model/stl",
  usdz: "model/vnd.usdz+zip"
});

/**
 * A consolidated mapping of all extensions permitted for upload.
 */
const UPLOADABLE_FILE_EXTENSIONS = Object.freeze({
  ...IMAGE_FILE_EXTENSIONS,
  ...AUDIO_FILE_EXTENSIONS,
  ...VIDEO_FILE_EXTENSIONS,
  ...TEXT_FILE_EXTENSIONS,
  ...FONT_FILE_EXTENSIONS,
  ...GRAPHICS_FILE_EXTENSIONS
});

/**
 * An enumeration of file type categories which can be selected.
 */
const FILE_CATEGORIES = {
  HTML: HTML_FILE_EXTENSIONS,
  IMAGE: IMAGE_FILE_EXTENSIONS,
  VIDEO: VIDEO_FILE_EXTENSIONS,
  AUDIO: AUDIO_FILE_EXTENSIONS,
  TEXT: TEXT_FILE_EXTENSIONS,
  FONT: FONT_FILE_EXTENSIONS,
  GRAPHICS: GRAPHICS_FILE_EXTENSIONS,

  /**
   * @deprecated since v13
   * @ignore
   */
  get MEDIA() {
    const message = "CONST.FILE_CATEGORIES.MEDIA is deprecated. Use CONST.MEDIA_MIME_TYPES instead.";
    foundry.utils.logCompatibilityWarning(message, {since: 13, until: 15, once: true});
    return MEDIA_MIME_TYPES;
  }
};
Object.defineProperties(FILE_CATEGORIES, {MEDIA: {enumerable: false}});
Object.freeze(FILE_CATEGORIES);

/**
 * The list of file categories that are "media".
 */
const MEDIA_FILE_CATEGORIES = Object.freeze(["IMAGE", "VIDEO", "AUDIO", "TEXT", "FONT", "GRAPHICS"]);

/**
 * A list of MIME types which are treated as uploaded "media", which are allowed to overwrite existing files.
 * Any non-media MIME type is not allowed to replace an existing file.
 */
const MEDIA_MIME_TYPES = Array.from(new Set(MEDIA_FILE_CATEGORIES.flatMap(
  c => Object.values(FILE_CATEGORIES[c])))).sort();

/**
 * A font weight to name mapping.
 */
const FONT_WEIGHTS = Object.freeze({
  Thin: 100,
  ExtraLight: 200,
  Light: 300,
  Regular: 400,
  Medium: 500,
  SemiBold: 600,
  Bold: 700,
  ExtraBold: 800,
  Black: 900
});

/**
 * Stores shared commonly used timeouts, measured in MS
 */
const TIMEOUTS = Object.freeze({
  /**
   * The default timeout for interacting with the foundryvtt.com API.
   */
  FOUNDRY_WEBSITE: 10000,

  /**
   * The specific timeout for loading the list of packages from the foundryvtt.com API.
   */
  PACKAGE_REPOSITORY: 10000,

  /**
   * The specific timeout for the IP address lookup service.
   */
  IP_DISCOVERY: 5000,

  /**
   * A remote package manifest JSON or download ZIP.
   */
  REMOTE_PACKAGE: 5000
});

/**
 * A subset of Compendium types which require a specific system to be designated
 */
const SYSTEM_SPECIFIC_COMPENDIUM_TYPES = Object.freeze(["Actor", "Item"]);

/**
 * The configured showdown bi-directional HTML <-> Markdown converter options.
 */
const SHOWDOWN_OPTIONS = Object.freeze({
  disableForced4SpacesIndentedSublists: true,
  noHeaderId: true,
  parseImgDimensions: true,
  strikethrough: true,
  tables: true,
  tablesHeaderId: true
});

/**
 * The list of allowed HTML tags.
 */
const ALLOWED_HTML_TAGS = Object.freeze([
  "header", "main", "section", "article", "aside", "nav", "footer", "div", "address", // Structural Elements
  "h1", "h2", "h3", "h4", "h5", "h6", "hr", "br", // Headers and Dividers
  "p", "blockquote", "summary", "details", "span", "code", "pre", "a", "label", "abbr", "cite",
  "mark", "q", "ruby", "rp", "rt", "small", "time", "var", "kbd", "samp", // Text Types
  "dfn", "sub", "sup", "strong", "em", "b", "i", "u", "s", "del", "ins", // Text Styles
  "ol", "ul", "li", "dl", "dd", "dt", "menu", // Lists
  "table", "thead", "tbody", "tfoot", "tr", "th", "td", "col", "colgroup", // Tables
  "form", "input", "select", "option", "button", "datalist", "fieldset", "legend", "meter",
  "optgroup", "progress", "textarea", "output", // Forms
  "figure", "figcaption", "caption", "img", "video", "map", "area", "track", "picture",
  "source", "audio", // Media
  "iframe", // Embedded content
  "color-picker", "code-mirror", "document-embed", "document-tags", "enriched-content", "file-picker", "hue-slider",
  "multi-select", "multi-checkbox", "range-picker", "secret-block", "string-tags", "prose-mirror" // Custom elements
]);

/**
 * The list of allowed attributes in HTML elements.
 */
const ALLOWED_HTML_ATTRIBUTES = deepFreeze({
  "*": [
    "class", "data-*", "id", "title", "style", "draggable", "aria-*", "tabindex", "dir", "hidden", "inert", "role",
    "is", "lang", "popover", "autocapitalize", "autocorrect", "autofocus", "contenteditable", "spellcheck", "translate"
  ],
  a: ["href", "name", "target", "rel"],
  area: ["alt", "coords", "href", "rel", "shape", "target"],
  audio: ["controls", "loop", "muted", "src", "autoplay"],
  blockquote: ["cite"],
  button: ["disabled", "name", "type", "value"],
  col: ["span"],
  colgroup: ["span"],
  "code-mirror": ["disabled", "name", "value", "placeholder", "readonly", "required", "language", "indent", "nowrap"],
  "color-picker": ["disabled", "name", "value", "placeholder", "readonly", "required"],
  details: ["open"],
  "document-embed": ["uuid"],
  "document-tags": ["disabled", "name", "value", "placeholder", "readonly", "required", "type", "single", "max"],
  "enriched-content": ["enricher"],
  fieldset: ["disabled"],
  "file-picker": ["disabled", "name", "value", "placeholder", "readonly", "required", "type", "noupload"],
  form: ["name"],
  "hue-slider": ["disabled", "name", "value", "readonly", "required"],
  iframe: ["src", "srcdoc", "name", "height", "width", "loading", "sandbox"],
  img: ["height", "src", "width", "usemap", "sizes", "srcset", "alt"],
  input: [
    "checked", "disabled", "name", "value", "placeholder", "type", "alt", "height", "list",
    "max", "min", "readonly", "size", "src", "step", "width", "required"
  ],
  label: ["for"],
  li: ["value"],
  map: ["name"],
  meter: ["value", "min", "max", "low", "high", "optimum"],
  "multi-checkbox": ["disabled", "name", "required"],
  "multi-select": ["disabled", "name", "required"],
  ol: ["reversed", "start", "type"],
  optgroup: ["disabled", "label"],
  option: ["disabled", "selected", "label", "value"],
  output: ["for", "form", "name"],
  progress: ["max", "value"],
  "prose-mirror": ["disabled", "name", "value", "placeholder", "readonly", "required", "toggled", "open"],
  "range-picker": ["disabled", "name", "value", "placeholder", "readonly", "min", "max", "step"],
  select: ["name", "disabled", "multiple", "size", "required"],
  source: ["media", "sizes", "src", "srcset", "type"],
  "string-tags": ["disabled", "name", "value", "placeholder", "readonly", "required"],
  table: ["border"],
  td: ["colspan", "headers", "rowspan"],
  textarea: ["rows", "cols", "disabled", "name", "readonly", "wrap", "required"],
  time: ["datetime"],
  th: ["abbr", "colspan", "headers", "rowspan", "scope", "sorted"],
  track: ["default", "kind", "label", "src", "srclang"],
  video: ["controls", "height", "width", "loop", "muted", "poster", "src", "autoplay"]
});

/**
 * The list of allowed URL schemes.
 */
const ALLOWED_URL_SCHEMES = Object.freeze(["http", "https", "data", "mailto", "obsidian",
  "syrinscape-online"]);

/**
 * The list of attributes validated as URLs.
 */
const ALLOWED_URL_SCHEMES_APPLIED_TO_ATTRIBUTES = Object.freeze(["href", "src", "cite"]);

/**
 * The list of trusted iframe domains.
 */
const TRUSTED_IFRAME_DOMAINS = Object.freeze(["google.com", "youtube.com"]);

/**
 * Available themes for the world join page.
 */
const WORLD_JOIN_THEMES = Object.freeze({
  default: "WORLD.JOIN_THEMES.default",
  minimal: "WORLD.JOIN_THEMES.minimal"
});

/**
 * Setup page package progress protocol.
 */
const SETUP_PACKAGE_PROGRESS = deepFreeze({
  ACTIONS: {
    CREATE_BACKUP: "createBackup",
    RESTORE_BACKUP: "restoreBackup",
    DELETE_BACKUP: "deleteBackup",
    CREATE_SNAPSHOT: "createSnapshot",
    RESTORE_SNAPSHOT: "restoreSnapshot",
    DELETE_SNAPSHOT: "deleteSnapshot",
    INSTALL_PKG: "installPackage",
    LAUNCH_WORLD: "launchWorld",
    UPDATE_CORE: "updateCore",
    UPDATE_DOWNLOAD: "updateDownload"
  },
  STEPS: {
    ARCHIVE: "archive",
    CHECK_DISK_SPACE: "checkDiskSpace",
    CLEAN_WORLD: "cleanWorld",
    EXTRACT_DEMO: "extractDemo",
    CONNECT_WORLD: "connectWorld",
    MIGRATE_WORLD: "migrateWorld",
    CONNECT_PKG: "connectPackage",
    MIGRATE_PKG: "migratePackage",
    MIGRATE_CORE: "migrateCore",
    MIGRATE_SYSTEM: "migrateSystem",
    DOWNLOAD: "download",
    EXTRACT: "extract",
    INSTALL: "install",
    CLEANUP: "cleanup",
    COMPLETE: "complete",
    DELETE: "delete",
    ERROR: "error",
    VEND: "vend",
    SNAPSHOT_MODULES: "snapshotModules",
    SNAPSHOT_SYSTEMS: "snapshotSystems",
    SNAPSHOT_WORLDS: "snapshotWorlds"
  }
});

/**
 * The combat announcements.
 */
const COMBAT_ANNOUNCEMENTS = Object.freeze(["startEncounter", "nextUp", "yourTurn"]);

/**
 * The fit modes of {@link foundry.data.TextureData}.
 */
const TEXTURE_DATA_FIT_MODES = Object.freeze(["fill", "contain", "cover", "width", "height"]);

/**
 * @typedef {typeof TEXTURE_DATA_FIT_MODES[number]} TextureDataFitMode
 */

/**
 * The maximum depth to recurse to when embedding enriched text.
 */
const TEXT_ENRICH_EMBED_MAX_DEPTH = 5;

/**
 * The Region events that are supported by core.
 */
const REGION_EVENTS = /** @type {const} */ ({
  /**
   * Triggered when the shapes or bottom/top elevation of the Region are changed.
   *
   * @see {@link foundry.documents.types.RegionRegionBoundaryEvent}
   */
  REGION_BOUNDARY: "regionBoundary",

  /**
   * Triggered when the Region Behavior becomes active, i.e. is enabled or created without being disabled.
   *
   * The event is triggered only for this Region Behavior.
   *
   * @see {@link foundry.documents.types.RegionBehaviorActivatedEvent}
   */
  BEHAVIOR_ACTIVATED: "behaviorActivated",

  /**
   * Triggered when the Region Behavior becomes inactive, i.e. is disabled or deleted without being disabled.
   *
   * The event is triggered only for this Region Behavior.
   *
   * @see {@link foundry.documents.types.RegionBehaviorDeactivatedEvent}
   */
  BEHAVIOR_DEACTIVATED: "behaviorDeactivated",

  /**
   * Triggered when the Region Behavior becomes viewed, i.e. active and the Scene of its Region is viewed.
   *
   * The event is triggered only for this Region Behavior.
   *
   * @see {@link foundry.documents.types.RegionBehaviorViewedEvent}
   */
  BEHAVIOR_VIEWED: "behaviorViewed",

  /**
   * Triggered when the Region Behavior becomes unviewed, i.e. inactive or the Scene of its Region is unviewed.
   *
   * The event is triggered only for this Region Behavior.
   *
   * @see {@link foundry.documents.types.RegionBehaviorUnviewedEvent}
   */
  BEHAVIOR_UNVIEWED: "behaviorUnviewed",

  /**
   * Triggered when a Token enters a Region.
   *
   * A Token enters a Region whenever ...
   *   - it is created within the Region,
   *   - the boundary of the Region has changed such that the Token is now inside the Region,
   *   - the Token moves into the Region (the Token's x, y, elevation, width, height, or shape
   *     has changed such that it is now inside the Region), or
   *   - a Region Behavior becomes active (i.e., is enabled or created while enabled), in which case
   *     the event it triggered only for this Region Behavior.
   *
   * @see {@link foundry.documents.types.RegionTokenEnterEvent}
   */
  TOKEN_ENTER: "tokenEnter",

  /**
   * Triggered when a Token exits a Region.
   *
   * A Token exits a Region whenever ...
   *   - it is deleted while inside the Region,
   *   - the boundary of the Region has changed such that the Token is no longer inside the Region,
   *   - the Token moves out of the Region (the Token's x, y, elevation, width, height, or shape
   *     has changed such that it is no longer inside the Region), or
   *   - a Region Behavior becomes inactive (i.e., is disabled or deleted while enabled), in which case
   *     the event it triggered only for this Region Behavior.
   *
   * @see {@link foundry.documents.types.RegionTokenExitEvent}
   */
  TOKEN_EXIT: "tokenExit",

  /**
   * Triggered when a Token moves into a Region.
   *
   * A Token moves whenever its x, y, elevation, width, height, or shape is changed.
   *
   * @see {@link foundry.documents.types.RegionTokenMoveInEvent}
   */
  TOKEN_MOVE_IN: "tokenMoveIn",

  /**
   * Triggered when a Token moves out of a Region.
   *
   * A Token moves whenever its x, y, elevation, width, height, or shape is changed.
   *
   * @see {@link foundry.documents.types.RegionTokenMoveOutEvent}
   */
  TOKEN_MOVE_OUT: "tokenMoveOut",

  /**
   * Triggered when a Token moves within a Region.
   *
   * A token moves whenever its x, y, elevation, width, height, or shape is changed.
   *
   * @see {@link foundry.documents.types.RegionTokenMoveWithinEvent}
   */
  TOKEN_MOVE_WITHIN: "tokenMoveWithin",

  /**
   * Triggered when a Token animates into a Region.
   *
   * This event is only triggered only if the Scene the Token is in is viewed.
   *
   * @see {@link foundry.documents.types.RegionTokenAnimateInEvent}
   */
  TOKEN_ANIMATE_IN: "tokenAnimateIn",

  /**
   * Triggered when a Token animates out of a Region.
   *
   * This event is triggered only if the Scene the Token is in is viewed.
   *
   * @see {@link foundry.documents.types.RegionTokenAnimateOutEvent}
   */
  TOKEN_ANIMATE_OUT: "tokenAnimateOut",

  /**
   * Triggered when a Token starts its Combat turn in a Region.
   *
   * @see {@link foundry.documents.types.RegionTokenTurnStartEvent}
   */
  TOKEN_TURN_START: "tokenTurnStart",

  /**
   * Triggered when a Token ends its Combat turn in a Region.
   *
   * @see {@link foundry.documents.types.RegionTokenTurnEndEvent}
   */
  TOKEN_TURN_END: "tokenTurnEnd",

  /**
   * Triggered when a Token starts the Combat round in a Region.
   *
   * @see {@link foundry.documents.types.RegionTokenRoundStartEvent}
   */
  TOKEN_ROUND_START: "tokenRoundStart",

  /**
   * Triggered when a Token ends the Combat round in a Region.
   *
   * @see {@link foundry.documents.types.RegionTokenRoundEndEvent}
   */
  TOKEN_ROUND_END: "tokenRoundEnd"

});

Object.defineProperties(REGION_EVENTS, {
  /** @deprecated since v13 */
  BEHAVIOR_STATUS: {
    get() {
      const message = "CONST.REGION_EVENTS.BEHAVIOR_STATUS is deprecated in favor of BEHAVIOR_ACTIVATED"
    + "BEHAVIOR_DEACTIVATED, BEHAVIOR_VIEWED, and BEHAVIOR_UNVIEWED.";
      foundry.utils.logCompatibilityWarning(message, {since: 13, until: 15, once: true});
      return "behaviorStatus";
    }
  },
  /** @deprecated since v13 */
  TOKEN_PRE_MOVE: {
    get() {
      foundry.utils.logCompatibilityWarning("CONST.REGION_EVENTS.TOKEN_PRE_MOVE is deprecated without replacement. "
          + "The TOKEN_PRE_MOVE event is not longer triggered.", {since: 13, until: 15, once: true});
      return "tokenPreMove";
    }
  },
  /** @deprecated since v13 */
  TOKEN_MOVE: {
    get() {
      foundry.utils.logCompatibilityWarning("CONST.REGION_EVENTS.TOKEN_MOVE is deprecated without replacement. "
        + "The TOKEN_MOVE event is not longer triggered.", {since: 13, until: 15, once: true});
      return "tokenMove";
    }
  }
});
Object.freeze(REGION_EVENTS);

/**
 * @typedef {typeof REGION_EVENTS[keyof typeof REGION_EVENTS]} RegionEventType
 */

/**
 * The possible visibility state of Region.
 */
const REGION_VISIBILITY = Object.freeze({

  /**
   * Only visible on the RegionLayer.
   */
  LAYER: 0,

  /**
   * Only visible to Gamemasters.
   */
  GAMEMASTER: 1,

  /**
   * Visible to anyone.
   */
  ALWAYS: 2
});

/**
 * The types of a Region movement segment.
 */
const REGION_MOVEMENT_SEGMENTS = Object.freeze({

  /**
   * The segment crosses the boundary of the Region and exits it.
   */
  EXIT: -1,

  /**
   * The segment does not cross the boundary of the Region and is contained within it.
   */
  MOVE: 0,

  /**
   * The segment crosses the boundary of the Region and enters it.
   */
  ENTER: 1
});

/**
 * @typedef {typeof REGION_MOVEMENT_SEGMENTS[keyof typeof REGION_MOVEMENT_SEGMENTS]} RegionMovementSegmentType
 */

/**
 * Available setting scopes.
 */
const SETTING_SCOPES = Object.freeze({
  /**
   * Settings scoped to the client device. Stored in localStorage.
   */
  CLIENT: "client",

  /**
   * Settings scoped to the game World. Applies to all Users in the World. Stored in the Settings database.
   */
  WORLD: "world",

  /**
   * Settings scoped to an individual User in the World. Stored in the Settings database.
   */
  USER: "user"
});

/* -------------------------------------------- */

/**
 * The scaling factor that is used for Clipper polygons/paths consistently everywhere core performs Clipper operations.
 */
const CLIPPER_SCALING_FACTOR = 100;

/* -------------------------------------------- */
/*  Deprecations and Compatibility              */
/* -------------------------------------------- */

/**
 * @deprecated since v12
 * @ignore
 */
const CHAT_MESSAGE_TYPES = new Proxy(CHAT_MESSAGE_STYLES, {
  get(target, prop, receiver) {
    const msg = "CONST.CHAT_MESSAGE_TYPES is deprecated in favor of CONST.CHAT_MESSAGE_STYLES because the "
      + "ChatMessage#type field has been renamed to ChatMessage#style";
    foundry.utils.logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return Reflect.get(...arguments);
  }
});


/**
 * @deprecated since v12
 * @ignore
 */
const _DOCUMENT_TYPES = Object.freeze(WORLD_DOCUMENT_TYPES.filter(t => {
  const excluded = ["FogExploration", "Setting"];
  return !excluded.includes(t);
}));

/**
 * @deprecated since v12
 * @ignore
 */
const DOCUMENT_TYPES = new Proxy(_DOCUMENT_TYPES, {
  get(target, prop, receiver) {
    const msg = "CONST.DOCUMENT_TYPES is deprecated in favor of either CONST.WORLD_DOCUMENT_TYPES or "
      + "CONST.COMPENDIUM_DOCUMENT_TYPES.";
    foundry.utils.logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return Reflect.get(...arguments);
  }
});

/**
 * @deprecated since v13
 * @ignore
 */
const TOKEN_HEXAGONAL_SHAPES = new Proxy(TOKEN_SHAPES, {
  get(target, prop, receiver) {
    const msg = "CONST.TOKEN_HEXAGONAL_SHAPES is deprecated in favor of CONST.TOKEN_SHAPES.";
    foundry.utils.logCompatibilityWarning(msg, {since: 13, until: 15, once: true});
    return Reflect.get(...arguments);
  }
});

var constants = /*#__PURE__*/Object.freeze({
  __proto__: null,
  ACTIVE_EFFECT_MODES: ACTIVE_EFFECT_MODES,
  ALLOWED_HTML_ATTRIBUTES: ALLOWED_HTML_ATTRIBUTES,
  ALLOWED_HTML_TAGS: ALLOWED_HTML_TAGS,
  ALLOWED_URL_SCHEMES: ALLOWED_URL_SCHEMES,
  ALLOWED_URL_SCHEMES_APPLIED_TO_ATTRIBUTES: ALLOWED_URL_SCHEMES_APPLIED_TO_ATTRIBUTES,
  ALL_DOCUMENT_TYPES: ALL_DOCUMENT_TYPES,
  ASCII: ASCII,
  AUDIO_CHANNELS: AUDIO_CHANNELS,
  AUDIO_FILE_EXTENSIONS: AUDIO_FILE_EXTENSIONS,
  BASE_DOCUMENT_TYPE: BASE_DOCUMENT_TYPE,
  CANVAS_PERFORMANCE_MODES: CANVAS_PERFORMANCE_MODES,
  CARD_DRAW_MODES: CARD_DRAW_MODES,
  CHAT_MESSAGE_STYLES: CHAT_MESSAGE_STYLES,
  CHAT_MESSAGE_TYPES: CHAT_MESSAGE_TYPES,
  CLIPPER_SCALING_FACTOR: CLIPPER_SCALING_FACTOR,
  COMBAT_ANNOUNCEMENTS: COMBAT_ANNOUNCEMENTS,
  COMPATIBILITY_MODES: COMPATIBILITY_MODES,
  COMPENDIUM_DOCUMENT_TYPES: COMPENDIUM_DOCUMENT_TYPES,
  CORE_SUPPORTED_LANGUAGES: CORE_SUPPORTED_LANGUAGES,
  CSS_THEMES: CSS_THEMES,
  CURSOR_STYLES: CURSOR_STYLES,
  DEFAULT_TOKEN: DEFAULT_TOKEN,
  DICE_ROLL_MODES: DICE_ROLL_MODES,
  DIRECTORY_SEARCH_MODES: DIRECTORY_SEARCH_MODES,
  DOCUMENT_LINK_TYPES: DOCUMENT_LINK_TYPES,
  DOCUMENT_META_OWNERSHIP_LEVELS: DOCUMENT_META_OWNERSHIP_LEVELS,
  DOCUMENT_OWNERSHIP_LEVELS: DOCUMENT_OWNERSHIP_LEVELS,
  DOCUMENT_TYPES: DOCUMENT_TYPES,
  DRAWING_FILL_TYPES: DRAWING_FILL_TYPES,
  EMBEDDED_DOCUMENT_TYPES: EMBEDDED_DOCUMENT_TYPES,
  FILE_CATEGORIES: FILE_CATEGORIES,
  FILE_PICKER_PUBLIC_DIRS: FILE_PICKER_PUBLIC_DIRS,
  FOLDER_DOCUMENT_TYPES: FOLDER_DOCUMENT_TYPES,
  FOLDER_MAX_DEPTH: FOLDER_MAX_DEPTH,
  FONT_FILE_EXTENSIONS: FONT_FILE_EXTENSIONS,
  FONT_WEIGHTS: FONT_WEIGHTS,
  GAME_VIEWS: GAME_VIEWS,
  GRAPHICS_FILE_EXTENSIONS: GRAPHICS_FILE_EXTENSIONS,
  GRID_DIAGONALS: GRID_DIAGONALS,
  GRID_MIN_SIZE: GRID_MIN_SIZE,
  GRID_SNAPPING_MODES: GRID_SNAPPING_MODES,
  GRID_TYPES: GRID_TYPES,
  HTML_FILE_EXTENSIONS: HTML_FILE_EXTENSIONS,
  IMAGE_FILE_EXTENSIONS: IMAGE_FILE_EXTENSIONS,
  JOURNAL_ENTRY_PAGE_FORMATS: JOURNAL_ENTRY_PAGE_FORMATS,
  KEYBINDING_PRECEDENCE: KEYBINDING_PRECEDENCE,
  LIGHTING_LEVELS: LIGHTING_LEVELS,
  MACRO_SCOPES: MACRO_SCOPES,
  MACRO_TYPES: MACRO_TYPES,
  MEASURED_TEMPLATE_TYPES: MEASURED_TEMPLATE_TYPES,
  MEDIA_FILE_CATEGORIES: MEDIA_FILE_CATEGORIES,
  MEDIA_MIME_TYPES: MEDIA_MIME_TYPES,
  MOVEMENT_DIRECTIONS: MOVEMENT_DIRECTIONS,
  OCCLUSION_MODES: OCCLUSION_MODES,
  PACKAGE_AVAILABILITY_CODES: PACKAGE_AVAILABILITY_CODES,
  PACKAGE_TYPES: PACKAGE_TYPES$1,
  PASSWORD_SAFE_STRING: PASSWORD_SAFE_STRING,
  PLAYLIST_MODES: PLAYLIST_MODES,
  PLAYLIST_SORT_MODES: PLAYLIST_SORT_MODES,
  PRIMARY_DOCUMENT_TYPES: PRIMARY_DOCUMENT_TYPES,
  REGION_EVENTS: REGION_EVENTS,
  REGION_MOVEMENT_SEGMENTS: REGION_MOVEMENT_SEGMENTS,
  REGION_VISIBILITY: REGION_VISIBILITY,
  SETTING_SCOPES: SETTING_SCOPES,
  SETUP_PACKAGE_PROGRESS: SETUP_PACKAGE_PROGRESS,
  SETUP_VIEWS: SETUP_VIEWS,
  SHOWDOWN_OPTIONS: SHOWDOWN_OPTIONS,
  SOFTWARE_UPDATE_CHANNELS: SOFTWARE_UPDATE_CHANNELS,
  SORT_INTEGER_DENSITY: SORT_INTEGER_DENSITY,
  SYSTEM_SPECIFIC_COMPENDIUM_TYPES: SYSTEM_SPECIFIC_COMPENDIUM_TYPES,
  TABLE_RESULT_TYPES: TABLE_RESULT_TYPES,
  TEXTURE_DATA_FIT_MODES: TEXTURE_DATA_FIT_MODES,
  TEXT_ANCHOR_POINTS: TEXT_ANCHOR_POINTS,
  TEXT_ENRICH_EMBED_MAX_DEPTH: TEXT_ENRICH_EMBED_MAX_DEPTH,
  TEXT_FILE_EXTENSIONS: TEXT_FILE_EXTENSIONS,
  TILE_OCCLUSION_MODES: TILE_OCCLUSION_MODES,
  TIMEOUTS: TIMEOUTS,
  TOKEN_DISPLAY_MODES: TOKEN_DISPLAY_MODES,
  TOKEN_DISPOSITIONS: TOKEN_DISPOSITIONS,
  TOKEN_HEXAGONAL_SHAPES: TOKEN_HEXAGONAL_SHAPES,
  TOKEN_OCCLUSION_MODES: TOKEN_OCCLUSION_MODES,
  TOKEN_SHAPES: TOKEN_SHAPES,
  TOKEN_TURN_MARKER_MODES: TOKEN_TURN_MARKER_MODES,
  TRUSTED_IFRAME_DOMAINS: TRUSTED_IFRAME_DOMAINS,
  UPLOADABLE_FILE_EXTENSIONS: UPLOADABLE_FILE_EXTENSIONS,
  USER_PERMISSIONS: USER_PERMISSIONS,
  USER_ROLES: USER_ROLES,
  USER_ROLE_NAMES: USER_ROLE_NAMES,
  VIDEO_FILE_EXTENSIONS: VIDEO_FILE_EXTENSIONS,
  VTT: VTT,
  WALL_DIRECTIONS: WALL_DIRECTIONS,
  WALL_DOOR_INTERACTIONS: WALL_DOOR_INTERACTIONS,
  WALL_DOOR_STATES: WALL_DOOR_STATES,
  WALL_DOOR_TYPES: WALL_DOOR_TYPES,
  WALL_MOVEMENT_TYPES: WALL_MOVEMENT_TYPES,
  WALL_RESTRICTION_TYPES: WALL_RESTRICTION_TYPES,
  WALL_SENSE_TYPES: WALL_SENSE_TYPES,
  WEBSITE_API_URL: WEBSITE_API_URL,
  WEBSITE_URL: WEBSITE_URL,
  WORLD_DOCUMENT_TYPES: WORLD_DOCUMENT_TYPES,
  WORLD_JOIN_THEMES: WORLD_JOIN_THEMES,
  vtt: vtt
});

/** @module validators */

/**
 * Test whether a string is a valid 16 character UID
 * @param {string} id
 * @return {boolean}
 */
function isValidId(id) {
  return /^[a-zA-Z0-9]{16}$/.test(id);
}

/**
 * Test whether a file path has an extension in a list of provided extensions
 * @param {string} path
 * @param {string[]} extensions
 * @return {boolean}
 */
function hasFileExtension(path, extensions) {
  const xts = extensions.map(ext => `\\.${ext}`).join("|");
  const rgx = new RegExp(`(${xts})(\\?.*)?$`, "i");
  return !!path && rgx.test(path);
}

/**
 * Test whether a string data blob contains base64 data, optionally of a specific type or types
 * @param {string} data       The candidate string data
 * @param {string[]} [types]  An array of allowed mime types to test
 * @return {boolean}
 */
function isBase64Data(data, types) {
  if ( types === undefined ) return /^data:([a-z]+)\/([a-z0-9]+);base64,/.test(data);
  return types.some(type => data.startsWith(`data:${type};base64,`))
}

/**
 * Test whether an input represents a valid 6-character color string
 * @param {string} color      The input string to test
 * @return {boolean}          Is the string a valid color?
 */
function isColorString(color) {
  return /^#[0-9A-Fa-f]{6}$/.test(color);
}

/**
 * Assert that the given value parses as a valid JSON string
 * @param {string} val        The value to test
 * @return {boolean}          Is the String valid JSON?
 */
function isJSON(val) {
  try {
    JSON.parse(val);
    return true;
  } catch(err) {
    return false;
  }
}

var validators = /*#__PURE__*/Object.freeze({
  __proto__: null,
  hasFileExtension: hasFileExtension,
  isBase64Data: isBase64Data,
  isColorString: isColorString,
  isJSON: isJSON,
  isValidId: isValidId
});

/**
 * @import {ElementValidationFailure} from "./_types.mjs";
 */

/**
 * A class responsible for recording information about a validation failure.
 */
class DataModelValidationFailure {
  /**
   * @param {object} [options]
   * @param {any} [options.invalidValue]       The value that failed validation for this field.
   * @param {any} [options.fallback]           The value it was replaced by, if any.
   * @param {boolean} [options.dropped=false]  Whether the value was dropped from some parent collection.
   * @param {string} [options.message]         The validation error message.
   * @param {boolean} [options.unresolved=false]     Whether this failure was unresolved
   */
  constructor({invalidValue, fallback, dropped=false, message, unresolved=false}={}) {
    this.invalidValue = invalidValue;
    this.fallback = fallback;
    this.dropped = dropped;
    this.message = message;
    this.unresolved = unresolved;
  }

  /**
   * The value that failed validation for this field.
   * @type {any}
   */
  invalidValue;

  /**
   * The value it was replaced by, if any.
   * @type {any}
   */
  fallback;

  /**
   * Whether the value was dropped from some parent collection.
   * @type {boolean}
   */
  dropped;

  /**
   * The validation error message.
   * @type {string}
   */
  message;

  /**
   * If this field contains other fields that are validated as part of its validation, their results are recorded here.
   * @type {Record<string, DataModelValidationFailure>}
   */
  fields = {};

  /**
   * If this field contains a list of elements that are validated as part of its validation, their results are recorded
   * here.
   * @type {ElementValidationFailure[]}
   */
  elements = [];

  /**
   * Record whether a validation failure is unresolved.
   * This reports as true if validation for this field or any hierarchically contained field is unresolved.
   * A failure is unresolved if the value was invalid and there was no valid fallback value available.
   * @type {boolean}
   */
  unresolved;

  /* -------------------------------------------- */

  /**
   * Return this validation failure as an Error object.
   * @returns {DataModelValidationError}
   */
  asError() {
    return new DataModelValidationError(this);
  }

  /* -------------------------------------------- */

  /**
   * Whether this failure contains other sub-failures.
   * @returns {boolean}
   */
  isEmpty() {
    return isEmpty(this.fields) && isEmpty(this.elements);
  }

  /* -------------------------------------------- */

  /**
   * Return the base properties of this failure, omitting any nested failures.
   * @returns {{invalidValue: any, fallback: any, dropped: boolean, message: string}}
   */
  toObject() {
    const {invalidValue, fallback, dropped, message} = this;
    return {invalidValue, fallback, dropped, message};
  }

  /* -------------------------------------------- */

  /**
   * Represent the DataModelValidationFailure as a string.
   * @returns {string}
   */
  toString() {
    return DataModelValidationFailure.#formatString(this);
  }

  /* -------------------------------------------- */

  /**
   * Format a DataModelValidationFailure instance as a string message.
   * @param {DataModelValidationFailure} failure    The failure instance
   * @param {number} _d                             An internal depth tracker
   * @returns {string}                              The formatted failure string
   */
  static #formatString(failure, _d=0) {
    let message = failure.message ?? "";
    _d++;
    if ( !isEmpty(failure.fields) ) {
      message += "\n";
      const messages = [];
      for ( const [key, subFailure] of Object.entries(failure.fields) ) {
        const name = isDeletionKey(key) ? key.slice(2) : key;
        const subMessage = DataModelValidationFailure.#formatString(subFailure, _d);
        messages.push(`${" ".repeat(2 * _d)}${name}: ${subMessage}`);
      }
      message += messages.join("\n");
    }
    if ( !isEmpty(failure.elements) ) {
      message += "\n";
      const messages = [];
      for ( const element of failure.elements ) {
        const subMessage = DataModelValidationFailure.#formatString(element.failure, _d);
        messages.push(`${" ".repeat(2 * _d)}${element.id}: ${subMessage}`);
      }
      message += messages.join("\n");
    }
    return message;
  }
}

/* -------------------------------------------- */

/**
 * A specialised Error to indicate a model validation failure.
 * @extends {Error}
 */
class DataModelValidationError extends Error {
  /**
   * @param {DataModelValidationFailure|string} failure  The failure that triggered this error or an error message
   * @param {...any} [params]                            Additional Error constructor parameters
   */
  constructor(failure, ...params) {
    super(failure.toString(), ...params);
    if ( failure instanceof DataModelValidationFailure ) this.#failure = failure;
  }

  /**
   * The root validation failure that triggered this error.
   * @type {DataModelValidationFailure}
   */
  #failure;

  /* -------------------------------------------- */

  /**
   * Retrieve the root failure that caused this error, or a specific sub-failure via a path.
   * @param {string} [path]  The property path to the failure.
   * @returns {DataModelValidationFailure}
   *
   * @example Retrieving a failure.
   * ```js
   * const changes = {
   *   "foo.bar": "validValue",
   *   "foo.baz": "invalidValue"
   * };
   * try {
   *   doc.validate(expandObject(changes));
   * } catch ( err ) {
   *   const failure = err.getFailure("foo.baz");
   *   console.log(failure.invalidValue); // "invalidValue"
   * }
   * ```
   */
  getFailure(path) {
    if ( !this.#failure ) return;
    if ( !path ) return this.#failure;
    let failure = this.#failure;
    for ( const p of path.split(".") ) {
      if ( !failure ) return;
      if ( !isEmpty(failure.fields) ) failure = failure.fields[p];
      else if ( !isEmpty(failure.elements) ) failure = failure.elements.find(e => e.id?.toString() === p);
    }
    return failure;
  }

  /* -------------------------------------------- */

  /**
   * Retrieve a flattened object of all the properties that failed validation as part of this error.
   * @returns {Record<string, DataModelValidationFailure>}
   *
   * @example Removing invalid changes from an update delta.
   * ```js
   * const changes = {
   *   "foo.bar": "validValue",
   *   "foo.baz": "invalidValue"
   * };
   * try {
   *   doc.validate(expandObject(changes));
   * } catch ( err ) {
   *   const failures = err.getAllFailures();
   *   if ( failures ) {
   *     for ( const prop in failures ) delete changes[prop];
   *     doc.validate(expandObject(changes));
   *   }
   * }
   * ```
   */
  getAllFailures() {
    if ( !this.#failure || this.#failure.isEmpty() ) return;
    return DataModelValidationError.#aggregateFailures(this.#failure);
  }

  /* -------------------------------------------- */

  /**
   * Log the validation error as a table.
   */
  logAsTable() {
    const failures = this.getAllFailures();
    if ( isEmpty(failures) ) return;
    console.table(Object.entries(failures).reduce((table, [p, failure]) => {
      table[p] = failure.toObject();
      return table;
    }, {}));
  }

  /* -------------------------------------------- */

  /**
   * Generate a nested tree view of the error as an HTML string.
   * @returns {string}
   */
  asHTML() {
    const renderFailureNode = failure => {
      if ( failure.isEmpty() ) return `<li>${foundry.utils.escapeHTML(failure.message || "")}</li>`;
      const nodes = [];
      for ( const [field, subFailure] of Object.entries(failure.fields) ) {
        nodes.push(`<li><details><summary>${field}</summary><ul>${renderFailureNode(subFailure)}</ul></details></li>`);
      }
      for ( const element of failure.elements ) {
        const name = element.name || element.id;
        const html = `
          <li><details><summary>${name}</summary><ul>${renderFailureNode(element.failure)}</ul></details></li>
        `;
        nodes.push(html);
      }
      return nodes.join("");
    };
    return `<ul class="summary-tree">${renderFailureNode(this.#failure)}</ul>`;
  }

  /* -------------------------------------------- */

  /**
   * Collect nested failures into an aggregate object.
   * @param {DataModelValidationFailure} failure                               The failure.
   * @returns {DataModelValidationFailure|Record<string, DataModelValidationFailure>}  Returns the failure at the leaf of the
   *                                                                           tree, otherwise an object of
   *                                                                           sub-failures.
   */
  static #aggregateFailures(failure) {
    if ( failure.isEmpty() ) return failure;
    const failures = {};
    const recordSubFailures = (field, subFailures) => {
      if ( subFailures instanceof DataModelValidationFailure ) failures[field] = subFailures;
      else {
        for ( const [k, v] of Object.entries(subFailures) ) {
          failures[`${field}.${k}`] = v;
        }
      }
    };
    for ( const [field, subFailure] of Object.entries(failure.fields) ) {
      recordSubFailures(field, DataModelValidationError.#aggregateFailures(subFailure));
    }
    for ( const element of failure.elements ) {
      recordSubFailures(element.id, DataModelValidationError.#aggregateFailures(element.failure));
    }
    return failures;
  }
}

var validationFailure = /*#__PURE__*/Object.freeze({
  __proto__: null,
  DataModelValidationError: DataModelValidationError,
  DataModelValidationFailure: DataModelValidationFailure
});

/**
 * A reusable storage concept which blends the functionality of an Array with the efficient key-based lookup of a Map.
 * This concept is reused throughout Foundry VTT where a collection of uniquely identified elements is required.
 * @template {string} K
 * @template V
 * @extends {Map<K, V>}
 */
class Collection extends Map {

  /**
   * Then iterating over a Collection, we should iterate over its values instead of over its entries
   * @returns {MapIterator<V>}
   */
  [Symbol.iterator]() {
    return this.values();
  }

  /* -------------------------------------------- */

  /**
   * Return an Array of all the entry values in the Collection
   * @type {V[]}
   */
  get contents() {
    return Array.from(this.values());
  }

  /* -------------------------------------------- */
  /**
   * Find an entry in the Map using a functional condition.
   * @see {Array#find}
   * @param {(value: V, index: number, collection: Collection<K, V>) => unknown} condition The functional condition to
   *                                                                                       test.
   * @returns {V|undefined} The value, if found, otherwise undefined
   *
   * @example Create a new Collection and reference its contents
   * ```js
   * let c = new Collection([["a", "A"], ["b", "B"], ["c", "C"]]);
   * c.get("a") === c.find(entry => entry === "A"); // true
   * ```
   */
  find(condition) {
    let i = 0;
    for ( const v of this.values() ) {
      if ( condition(v, i, this) ) return v;
      i++;
    }
    return undefined;
  }

  /* -------------------------------------------- */

  /**
   * Filter the Collection, returning an Array of entries which match a functional condition.
   * @see {Array#filter}
   * @param {(value: V, index: number, collection: Collection<K, V>) => unknown} condition The functional condition to
   *                                                                                       test.
   * @returns {V[]}           An Array of matched values
   *
   * @example Filter the Collection for specific entries
   * ```js
   * let c = new Collection([["a", "AA"], ["b", "AB"], ["c", "CC"]]);
   * let hasA = c.filters(entry => entry.slice(0) === "A");
   * ```
   */
  filter(condition) {
    const entries = [];
    let i = 0;
    for ( const v of this.values() ) {
      if ( condition(v, i, this) ) entries.push(v);
      i++;
    }
    return entries;
  }

  /* -------------------------------------------- */

  /**
   * Apply a function to each element of the collection
   * @see Array#forEach
   * @param {(value: V) => void} fn A function to apply to each element
   *
   * @example Apply a function to each value in the collection
   * ```js
   * let c = new Collection([["a", {active: false}], ["b", {active: false}], ["c", {active: false}]]);
   * c.forEach(e => e.active = true);
   * ```
   */
  forEach(fn) {
    for ( const e of this.values() ) {
      fn(e);
    }
  }

  /* -------------------------------------------- */

  /**
   * Get an element from the Collection by its key.
   * @param {string} key      The key of the entry to retrieve
   * @param {object} [options]  Additional options that affect how entries are retrieved
   * @param {boolean} [options.strict=false] Throw an Error if the requested key does not exist. Default false.
   * @returns {V|undefined} The retrieved entry value, if the key exists, otherwise undefined
   *
   * @example Get an element from the Collection by key
   * ```js
   * let c = new Collection([["a", "Alfred"], ["b", "Bob"], ["c", "Cynthia"]]);
   * c.get("a"); // "Alfred"
   * c.get("d"); // undefined
   * c.get("d", {strict: true}); // throws Error
   * ```
   */
  get(key, {strict=false}={}) {
    const entry = super.get(key);
    if ( strict && (entry === undefined) ) {
      throw new Error(`The key ${key} does not exist in the ${this.constructor.name} Collection`);
    }
    return entry;
  }

  /* -------------------------------------------- */

  /**
   * Get an entry from the Collection by name.
   * Use of this method assumes that the objects stored in the collection have a "name" attribute.
   * @param {string} name       The name of the entry to retrieve
   * @param {object} [options]  Additional options that affect how entries are retrieved
   * @param {boolean} [options.strict=false] Throw an Error if the requested name does not exist. Default false.
   * @returns {V|undefined} The retrieved entry value, if one was found, otherwise undefined
   *
   * @example Get an element from the Collection by name (if applicable)
   * ```js
   * let c = new Collection([["a", "Alfred"], ["b", "Bob"], ["c", "Cynthia"]]);
   * c.getName("Alfred"); // "Alfred"
   * c.getName("D"); // undefined
   * c.getName("D", {strict: true}); // throws Error
   * ```
   */
  getName(name, {strict=false}={}) {
    const entry = this.find(e => e.name === name);
    if ( strict && (entry === undefined) ) {
      throw new Error(`An entry with name ${name} does not exist in the collection`);
    }
    return entry ?? undefined;
  }

  /* -------------------------------------------- */

  /**
   * Transform each element of the Collection into a new form, returning an Array of transformed values
   * @param {function(V,number,Collection): *} transformer  A transformation function applied to each entry value.
   * Positional arguments are the value, the index of iteration, and the collection being mapped.
   * @returns {Array<*>}  An Array of transformed values
   */
  map(transformer) {
    const transformed = [];
    let i = 0;
    for ( const v of this.values() ) {
      transformed.push(transformer(v, i, this));
      i++;
    }
    return transformed;
  }

  /* -------------------------------------------- */

  /**
   * Reduce the Collection by applying an evaluator function and accumulating entries
   * @see {Array#reduce}
   * @param {function(*,V,number,Collection): *} reducer  A reducer function applied to each entry value. Positional
   * arguments are the accumulator, the value, the index of iteration, and the collection being reduced.
   * @param {*} initial             An initial value which accumulates with each iteration
   * @returns {*}                    The accumulated result
   *
   * @example Reduce a collection to an array of transformed values
   * ```js
   * let c = new Collection([["a", "A"], ["b", "B"], ["c", "C"]]);
   * let letters = c.reduce((s, l) => {
   *   return s + l;
   * }, ""); // "ABC"
   * ```
   */
  reduce(reducer, initial) {
    let accumulator = initial;
    let i = 0;
    for ( const v of this.values() ) {
      accumulator = reducer(accumulator, v, i, this);
      i++;
    }
    return accumulator;
  }

  /* -------------------------------------------- */

  /**
   * Test whether a condition is met by some entry in the Collection.
   * @see {Array#some}
   * @param {function(V,number,Collection): boolean} condition  The functional condition to test. Positional
   * arguments are the value, the index of iteration, and the collection being tested.
   * @returns {boolean}  Was the test condition passed by at least one entry?
   */
  some(condition) {
    let i = 0;
    for ( const v of this.values() ) {
      const pass = condition(v, i, this);
      i++;
      if ( pass ) return true;
    }
    return false;
  }

  /* -------------------------------------------- */

  /**
   * Convert the Collection to a primitive array of its contents.
   * @returns {object[]}  An array of contained values
   */
  toJSON() {
    return this.map(e => e.toJSON ? e.toJSON() : e);
  }
}

/**
 * @import DataModel from "./data.mjs";
 * @import Document from "./document.mjs";
 * @import {DatabaseAction, DatabaseOperation} from "./_types.mjs";
 * @import BaseUser from "../documents/user.mjs";
 * @import {DocumentConstructionContext} from "./_types.mjs";
 */

/**
 * An extension of the Collection.
 * Used for the specific task of containing embedded Document instances within a parent Document.
 * @template {Document} TDocument
 * @extends Collection<string, TDocument>
 */
class EmbeddedCollection extends Collection {
  /**
   * @param {string} name           The name of this collection in the parent Document.
   * @param {Document} parent       The parent Document instance to which this collection belongs.
   * @param {object[]} sourceArray  The source data array for the collection in the parent Document data.
   */
  constructor(name, parent, sourceArray) {
    if ( typeof name !== "string" ) throw new Error("The signature of EmbeddedCollection has changed in v11.");
    super();
    Object.defineProperties(this, {
      _source: {value: sourceArray, writable: false},
      documentClass: {value: parent.constructor.hierarchy[name].model, writable: false},
      name: {value: name, writable: false},
      model: {value: parent, writable: false}
    });
  }

  /**
   * The Document implementation used to construct instances within this collection.
   * @type {typeof Document}
   */
  documentClass;

  /**
   * The Document name of Documents stored in this collection.
   * @returns {string|void}
   */
  get documentName() {
    return this.documentClass?.documentName;
  }

  /**
   * The name of this collection in the parent Document.
   * @type {string}
   */
  name;

  /**
   * The parent Document to which this EmbeddedCollection instance belongs.
   * @type {Document}
   */
  model; // TODO: Should we rename this property parentDocument?

  /**
   * Has this embedded collection been initialized as a one-time workflow?
   * @type {boolean}
   * @protected
   */
  _initialized = false;

  /**
   * The source data array from which the embedded collection is created
   * @type {object[]}
   * @public
   */
  _source;

  /**
   * Record the set of document ids where the Document was not initialized because of invalid source data
   * @type {Set<string>}
   */
  invalidDocumentIds = new Set();

  /**
   * A cache of this collection's contents grouped by subtype
   * @type {Record<string, TDocument[]>|null}
   */
  #documentsByType = null;

  /* -------------------------------------------- */

  /**
   * This collection's contents grouped by subtype, lazily (re-)computed as needed.
   * If the document type does not support subtypes, all will be in the "base" group.
   * @type {Record<string, TDocument[]>}
   */
  get documentsByType() {
    if ( this.#documentsByType ) return this.#documentsByType;
    const typeName = this.documentClass.metadata.name;
    const types = Object.fromEntries(game.documentTypes[typeName].map(t => [t, []]));
    for ( const document of this.values() ) {
      types[document._source.type ?? "base"]?.push(document);
    }
    return this.#documentsByType = types;
  }

  /* -------------------------------------------- */
  /*  Collection Initialization                   */
  /* -------------------------------------------- */

  /**
   * Initialize the EmbeddedCollection by synchronizing its Document instances with existing _source data.
   * Importantly, this method does not make any modifications to the _source array.
   * It is responsible for creating, updating, or removing Documents from the Collection.
   * @param {DocumentConstructionContext} [options]  Initialization options.
   */
  initialize(options={}) {
    this._initialized = false;
    this.#documentsByType = null;

    // Re-initialize all records in source
    const initializedIds = new Set();
    for ( const obj of this._source ) {
      const doc = this._initializeDocument(obj, options);
      if ( doc ) initializedIds.add(doc.id);
    }

    // Remove documents that no longer exist in source
    if ( this.size !== initializedIds.size ) {
      for ( const k of this.keys() ) {
        if ( !initializedIds.has(k) ) this.delete(k, {modifySource: false});
      }
    }
    this._initialized = true;
  }

  /* -------------------------------------------- */

  /**
   * Initialize an embedded document and store it in the collection.
   * The document may already exist, in which case we are reinitializing it with new _source data.
   * The document may not yet exist, in which case we create a new Document instance using the provided source.
   *
   * @param {object} data                    The Document data.
   * @param {DocumentConstructionContext} [options]  Initialization options.
   * @returns {TDocument|null}               The initialized document or null if no document was initialized
   * @protected
   */
  _initializeDocument(data, options) {
    let doc = this.get(data._id);

    // Re-initialize an existing document
    if ( doc ) {
      doc._initialize(options);
      return doc;
    }

    // Create a new document
    if ( !data._id ) data._id = randomID(16); // TODO should this throw an error?
    try {
      doc = this.createDocument(data, options);
      super.set(doc.id, doc);
    } catch(err) {
      this._handleInvalidDocument(data._id, err, options);
      return null;
    }
    return doc;
  }

  /* -------------------------------------------- */

  /**
   * Instantiate a Document for inclusion in the Collection.
   * @param {object} data       The Document data.
   * @param {DocumentConstructionContext} [context]  Document creation context.
   * @returns {TDocument}
   */
  createDocument(data, context={}) {
    return new this.documentClass(data, {
      ...context,
      parent: this.model,
      parentCollection: this.name,
      pack: this.model.pack
    });
  }

  /* -------------------------------------------- */

  /**
   * Log warnings or errors when a Document is found to be invalid.
   * @param {string} id                      The invalid Document's ID.
   * @param {Error} err                      The validation error.
   * @param {object} [options]               Options to configure invalid Document handling.
   * @param {boolean} [options.strict=true]  Whether to throw an error or only log a warning.
   * @protected
   */
  _handleInvalidDocument(id, err, {strict=true}={}) {
    const documentName = this.documentClass.documentName;
    const parent = this.model;
    this.invalidDocumentIds.add(id);

    // Wrap the error with more information
    const uuid = foundry.utils.buildUuid({id, documentName, parent});
    const msg = `Failed to initialize ${documentName} [${uuid}]:\n${err.message}`;
    const error = new Error(msg, {cause: err});

    if ( strict ) globalThis.logger.error(error);
    else globalThis.logger.warn(error);
    if ( strict ) {
      globalThis.Hooks?.onError(`${this.constructor.name}#_initializeDocument`, error, {id, documentName});
    }
  }

  /* -------------------------------------------- */
  /*  Collection Methods                          */
  /* -------------------------------------------- */

  /**
   * Get a document from the EmbeddedCollection by its ID.
   * @param {string} id                         The ID of the Embedded Document to retrieve.
   * @param {object} [options]                  Additional options to configure retrieval.
   * @param {boolean} [options.strict=false]    Throw an Error if the requested Embedded Document does not exist.
   * @param {boolean} [options.invalid=false]   Allow retrieving an invalid Embedded Document.
   * @returns {TDocument}                       The retrieved document instance, or undefined
   * @throws {Error}                            If strict is true and the Embedded Document cannot be found.
   */
  get(id, {invalid=false, strict=false}={}) {
    let result = super.get(id);
    if ( !result && invalid ) result = this.getInvalid(id, { strict: false });
    if ( !result && strict ) throw new Error(`${this.constructor.documentName} id [${id}] does not exist in the `
      + `${this.constructor.name} collection.`);
    return result;
  }

  /* ---------------------------------------- */

  /**
   * Add a document to the collection.
   * @param {string} key                           The embedded Document ID.
   * @param {TDocument} value                      The embedded Document instance.
   * @param {object} [options]                     Additional options to the set operation.
   * @param {boolean} [options.modifySource=true]  Whether to modify the collection's source as part of the operation.
   * */
  set(key, value, {modifySource=true, ...options}={}) {
    if ( modifySource ) this._set(key, value, options);
    if ( super.get(key) !== value ) this.#documentsByType = null;
    return super.set(key, value);
  }

  /* -------------------------------------------- */

  /**
   * Modify the underlying source array to include the Document.
   * @param {string} key      The Document ID key.
   * @param {Document} value  The Document.
   * @protected
   */
  _set(key, value) {
    if ( this.has(key) || this.invalidDocumentIds.has(key) ) this._source.findSplice(d => d._id === key, value._source);
    else this._source.push(value._source);
  }

  /* ---------------------------------------- */

  /**
   * Remove a document from the collection.
   * @param {string} key                           The embedded Document ID.
   * @param {object} [options]                     Additional options to the delete operation.
   * @param {boolean} [options.modifySource=true]  Whether to modify the collection's source as part of the operation.
   * */
  delete(key, {modifySource=true, ...options}={}) {
    if ( modifySource ) this._delete(key, options);
    const result = super.delete(key);
    if ( result ) this.#documentsByType = null;
    return result;
  }

  /* -------------------------------------------- */

  /**
   * Remove the value from the underlying source array.
   * @param {string} key        The Document ID key.
   * @param {object} [options]  Additional options to configure deletion behavior.
   * @protected
   */
  _delete(key, options={}) {
    if ( this.has(key) || this.invalidDocumentIds.has(key) ) this._source.findSplice(d => d._id === key);
  }

  /* ---------------------------------------- */

  /**
   * Obtain a temporary Document instance for a document id which currently has invalid source data.
   * @param {string} id                      A document ID with invalid source data.
   * @param {object} [options]               Additional options to configure retrieval.
   * @param {boolean} [options.strict=true]  Throw an Error if the requested ID is not in the set of invalid IDs for
   *                                         this collection.
   * @returns {TDocument|void}               An in-memory instance for the invalid Document
   * @throws If strict is true and the requested ID is not in the set of invalid IDs for this collection.
   */
  getInvalid(id, {strict=true}={}) {
    if ( !this.invalidDocumentIds.has(id) ) {
      if ( strict ) throw new Error(`${this.constructor.documentName} id [${id}] is not in the set of invalid ids`);
      return;
    }
    const data = this._source.find(d => d._id === id);
    return this.documentClass.fromSource(foundry.utils.deepClone(data), {parent: this.model});
  }

  /* ---------------------------------------- */

  /**
   * Convert the EmbeddedCollection to an array of simple objects.
   * @param {boolean} [source=true]     Draw data for contained Documents from the underlying data source?
   * @returns {object[]}                The extracted array of primitive objects
   */
  toObject(source=true) {
    const arr = [];
    for ( const doc of this.values() ) {
      arr.push(doc.toObject(source));
    }
    return arr;
  }

  /* -------------------------------------------- */

  /**
   * Follow-up actions to take when a database operation modifies Documents in this EmbeddedCollection.
   * @param {DatabaseAction} action         The database action performed
   * @param {TDocument[]} documents         The array of modified Documents
   * @param {any[]} result                  The result of the database operation
   *
   * @param {DatabaseOperation} operation   Database operation details
   * @param {BaseUser} user                 The User who performed the operation
   * @internal
   */
  _onModifyContents(action, documents, result, operation, user) {
    // Propagate upwards to the parent collection
    const parentResult = action === "delete" ? [this.toObject()] : result;
    this.model?.collection?._onModifyContents?.("update", [this.model], [{[this.name]: parentResult}], operation, user);
  }
}

/**
 * This class provides a {@link foundry.utils.Collection} wrapper around a singleton embedded Document
 * so that it can be interacted with via a common interface.
 */
class SingletonEmbeddedCollection extends EmbeddedCollection {
  /** @inheritdoc */
  set(key, value) {
    if ( this.size && !this.has(key) ) {
      const embeddedName = this.documentClass.documentName;
      const parentName = this.model.documentName;
      throw new Error(`Cannot create singleton embedded ${embeddedName} [${key}] in parent ${parentName} `
        + `[${this.model.id}] as it already has one assigned.`);
    }
    return super.set(key, value);
  }

  /* -------------------------------------------- */

  /** @override */
  _set(key, value) {
    this.model._source[this.name] = value?._source ?? null;
  }

  /* -------------------------------------------- */

  /** @override */
  _delete(key) {
    this.model._source[this.name] = null;
  }
}

/**
 * @import { DataModelUpdateOptions } from "./_types.mjs";
 */

/**
 * An embedded collection delta contains delta source objects that can be compared against other objects inside a base
 * embedded collection, and generate new embedded Documents by combining them.
 * @template {Document} TDocument
 * @extends EmbeddedCollection<string, TDocument>
 */
class EmbeddedCollectionDelta extends EmbeddedCollection {
  /**
   * Maintain a list of IDs that are managed by this collection delta to distinguish from those IDs that are inherited
   * from the base collection.
   * @type {Set<string>}
   */
  #managedIds = new Set();

  /* -------------------------------------------- */

  /**
   * Maintain a list of IDs that are tombstone Documents.
   * @type {Set<string>}
   */
  #tombstones = new Set();

  /* -------------------------------------------- */

  /**
   * A convenience getter to return the corresponding base collection.
   * @type {EmbeddedCollection}
   */
  get baseCollection() {
    return this.model.getBaseCollection?.(this.name);
  }

  /* -------------------------------------------- */

  /**
   * A convenience getter to return the corresponding synthetic collection.
   * @type {EmbeddedCollection}
   */
  get syntheticCollection() {
    return this.model.syntheticActor?.getEmbeddedCollection(this.name);
  }

  /* -------------------------------------------- */

  /**
   * Determine whether a given ID is managed directly by this collection delta or inherited from the base collection.
   * @param {string} key  The Document ID.
   * @returns {boolean}
   */
  manages(key) {
    return this.#managedIds.has(key);
  }

  /* -------------------------------------------- */

  /**
   * Determine whether a given ID exists as a tombstone Document in the collection delta.
   * @param {string} key  The Document ID.
   * @returns {boolean}
   */
  isTombstone(key) {
    return this.#tombstones.has(key);
  }

  /* -------------------------------------------- */
  /*  Collection Initialization                   */
  /* -------------------------------------------- */

  /** @override */
  initialize({full=false, ...options} = {}) {
    const sc = this.syntheticCollection;
    this._initialized = false;
    this.#tombstones.clear();
    this.#managedIds.clear();
    if ( full ) this.clear();

    // Register tombstones and add custom managed records
    const initializedIds = new Set();
    for ( const obj of this._source ) {
      if ( obj._tombstone ) {
        this.#tombstones.add(obj._id);
        this.#managedIds.add(obj._id);
        continue;
      }
      let doc = sc?.get(obj._id);
      if ( doc ) super.set(doc.id, doc, {modifySource: false});
      else doc = this._initializeDocument(obj, options);
      if ( doc ) {
        initializedIds.add(doc.id);
        this.#managedIds.add(doc.id);
      }
    }

    // Add non-tombstone documents from the base collection
    if ( this.baseCollection?.size ) {
      for ( const baseDoc of this.baseCollection ) {
        if ( this.#managedIds.has(baseDoc.id) ) continue; // Skip managed Documents.
        let doc = sc?.get(baseDoc.id);
        if ( doc ) super.set(doc.id, doc, {modifySource: false});
        else doc = this._initializeDocument(baseDoc, options);
        if ( doc ) initializedIds.add(doc.id);
      }
    }

    // Remove any documents that no longer exist in source
    if ( this.size !== initializedIds.size ) {
      for ( const k of this.keys() ) {
        if ( !initializedIds.has(k) ) super.delete(k, {modifySource: false});
      }
    }

    this._initialized = true;
  }

  /* -------------------------------------------- */

  /** @override */
  createDocument(data, context={}) {
    return new this.documentClass(data, {
      ...context,
      parent: this.model.syntheticActor ?? this.model,
      parentCollection: this.name,
      pack: this.model.pack
    });
  }

  /* -------------------------------------------- */

  /**
   * Restore a Document so that it is no longer managed by the collection delta and instead inherits from the base
   * Document.
   * @param {string} id            The Document ID.
   * @returns {Promise<TDocument>} The restored Document.
   */
  async restoreDocument(id) {
    const docs = await this.restoreDocuments([id]);
    return docs.shift();
  }

  /* -------------------------------------------- */

  /**
   * Restore the given Documents so that they are no longer managed by the collection delta and instead inherit directly
   * from their counterparts in the base Actor.
   * @param {string[]} ids           The IDs of the Documents to restore.
   * @returns {Promise<TDocument[]>} An array of updated Document instances.
   */
  async restoreDocuments(ids) {
    if ( !this.model.syntheticActor ) return [];
    const baseActor = this.model.parent.baseActor;
    const embeddedName = this.documentClass.documentName;
    const {deltas, tombstones} = ids.reduce((obj, id) => {
      if ( !this.manages(id) ) return obj;
      const doc = baseActor.getEmbeddedCollection(this.name).get(id);
      if ( this.isTombstone(id) ) obj.tombstones.push(doc.toObject());
      else obj.deltas.push(doc.toObject());
      return obj;
    }, {deltas: [], tombstones: []});

    // For the benefit of downstream CRUD workflows, we emulate events from the perspective of the synthetic Actor.
    // Restoring an Item to the version on the base Actor is equivalent to updating that Item on the synthetic Actor
    // with the version of the Item on the base Actor.
    // Restoring an Item that has been deleted on the synthetic Actor is equivalent to creating a new Item on the
    // synthetic Actor with the contents of the version on the base Actor.
    // On the ActorDelta, those Items are removed from this collection delta so that they are once again 'linked' to the
    // base Actor's Item, as though they had never been modified from the original in the first place.

    let updated = [];
    if ( deltas.length ) {
      updated = await this.model.syntheticActor.updateEmbeddedDocuments(embeddedName, deltas, {
        diff: false, recursive: false, restoreDelta: true
      });
    }

    let created = [];
    if ( tombstones.length ) {
      created = await this.model.syntheticActor.createEmbeddedDocuments(embeddedName, tombstones, {
        keepId: true, restoreDelta: true
      });
    }

    return updated.concat(created);
  }

  /* -------------------------------------------- */

  /**
   * Prepare changes to this delta collection.
   * @param {object[]} changes                Candidate source changes.
   * @param {DataModelUpdateOptions} options  Options which determine how the new data is merged.
   * @internal
   */
  _prepareDeltaUpdate(changes, options) {
    for ( const change of changes ) {
      // If this entry is already managed, the update can be applied as normal.
      if ( !change._id || this.manages(change._id) ) continue;

      // Otherwise we must first adopt entry from the base collection.
      const existing = this.baseCollection.get(change._id)?._source;
      if ( existing ) foundry.utils.mergeObject(change, existing, {
        performDeletions: true,
        overwrite: false
      });
    }
  }

  /* -------------------------------------------- */
  /*  Collection Methods                          */
  /* -------------------------------------------- */

  /** @inheritdoc */
  set(key, value, options={}) {
    super.set(key, value, options);
    this.syntheticCollection?.set(key, value, options);
  }

  /* -------------------------------------------- */

  /** @override */
  _set(key, value, {restoreDelta=false}={}) {
    if ( restoreDelta ) return this._delete(key, {restoreDelta});
    if ( this.manages(key) ) this._source.findSplice(d => d._id === key, value._source);
    else this._source.push(value._source);
    this.#managedIds.add(key);
  }

  /* -------------------------------------------- */

  /** @inheritdoc */
  delete(key, options={}) {
    super.delete(key, options);
    this.syntheticCollection?.delete(key, options);
  }

  /* -------------------------------------------- */

  /** @override */
  _delete(key, {restoreDelta=false}={}) {
    if ( !this.baseCollection ) return;

    // Remove the document from this collection, if it exists.
    if ( this.manages(key) ) {
      this._source.findSplice(entry => entry._id === key);
      this.#managedIds.delete(key);
      this.#tombstones.delete(key);
    }

    // If the document exists in the base collection, push a tombstone in its place.
    if ( !restoreDelta && this.baseCollection.has(key) ) {
      this._source.push({_id: key, _tombstone: true});
      this.#managedIds.add(key);
      this.#tombstones.add(key);
    }
  }
}

/**
 * @import {LineCircleIntersection, LineIntersection} from "./_types.mjs";
 */

/**
 * Determine the relative orientation of three points in two-dimensional space.
 * The result is also an approximation of twice the signed area of the triangle defined by the three points.
 * This method is fast - but not robust against issues of floating point precision. Best used with integer coordinates.
 * Adapted from https://github.com/mourner/robust-predicates.
 * @param {Point} a     An endpoint of segment AB, relative to which point C is tested
 * @param {Point} b     An endpoint of segment AB, relative to which point C is tested
 * @param {Point} c     A point that is tested relative to segment AB
 * @returns {number}    The relative orientation of points A, B, and C
 *                      A positive value if the points are in counter-clockwise order (C lies to the left of AB)
 *                      A negative value if the points are in clockwise order (C lies to the right of AB)
 *                      Zero if the points A, B, and C are collinear.
 */
function orient2dFast(a, b, c) {
  return (a.y - c.y) * (b.x - c.x) - (a.x - c.x) * (b.y - c.y);
}

/* -------------------------------------------- */

/**
 * Quickly test whether the line segment AB intersects with the line segment CD.
 * This method does not determine the point of intersection, for that use lineLineIntersection.
 * @param {Point} a                   The first endpoint of segment AB
 * @param {Point} b                   The second endpoint of segment AB
 * @param {Point} c                   The first endpoint of segment CD
 * @param {Point} d                   The second endpoint of segment CD
 * @returns {boolean}                 Do the line segments intersect?
 */
function lineSegmentIntersects(a, b, c, d) {

  // First test the orientation of A and B with respect to CD to reject collinear cases
  const xa = foundry.utils.orient2dFast(a, b, c);
  const xb = foundry.utils.orient2dFast(a, b, d);
  if ( !xa && !xb ) return false;
  const xab = (xa * xb) <= 0;

  // Also require an intersection of CD with respect to AB
  const xcd = (foundry.utils.orient2dFast(c, d, a) * foundry.utils.orient2dFast(c, d, b)) <= 0;
  return xab && xcd;
}

/* -------------------------------------------- */

/**
 * An internal helper method for computing the intersection between two infinite-length lines.
 * Adapted from http://paulbourke.net/geometry/pointlineplane/.
 * @param {Point} a                   The first endpoint of segment AB
 * @param {Point} b                   The second endpoint of segment AB
 * @param {Point} c                   The first endpoint of segment CD
 * @param {Point} d                   The second endpoint of segment CD
 * @param {object} [options]          Options which affect the intersection test
 * @param {boolean} [options.t1=false]    Return the optional vector distance from C to D on CD
 * @returns {LineIntersection|null}   An intersection point, or null if no intersection occurred
 */
function lineLineIntersection(a, b, c, d, {t1=false}={}) {

  // If either line is length 0, they cannot intersect
  if (((a.x === b.x) && (a.y === b.y)) || ((c.x === d.x) && (c.y === d.y))) return null;

  // Check denominator - avoid parallel lines where d = 0
  const dnm = ((d.y - c.y) * (b.x - a.x) - (d.x - c.x) * (b.y - a.y));
  if (dnm === 0) return null;

  // Vector distances
  const t0 = ((d.x - c.x) * (a.y - c.y) - (d.y - c.y) * (a.x - c.x)) / dnm;
  t1 = t1 ? ((b.x - a.x) * (a.y - c.y) - (b.y - a.y) * (a.x - c.x)) / dnm : undefined;

  // Return the point of intersection
  return {
    x: a.x + t0 * (b.x - a.x),
    y: a.y + t0 * (b.y - a.y),
    t0: t0,
    t1: t1
  }
}

/* -------------------------------------------- */

/**
 * An internal helper method for computing the intersection between two finite line segments.
 * Adapted from http://paulbourke.net/geometry/pointlineplane/
 * @param {Point} a                   The first endpoint of segment AB
 * @param {Point} b                   The second endpoint of segment AB
 * @param {Point} c                   The first endpoint of segment CD
 * @param {Point} d                   The second endpoint of segment CD
 * @param {number} [epsilon]          A small epsilon which defines a tolerance for near-equality
 * @returns {LineIntersection|null}   An intersection point, or null if no intersection occurred
 */
function lineSegmentIntersection(a, b, c, d, epsilon=1e-8) {

  // If either line is length 0, they cannot intersect
  if (((a.x === b.x) && (a.y === b.y)) || ((c.x === d.x) && (c.y === d.y))) return null;

  // Check denominator - avoid parallel lines where d = 0
  const dnm = ((d.y - c.y) * (b.x - a.x) - (d.x - c.x) * (b.y - a.y));
  if (dnm === 0) return null;

  // Vector distance from a
  const t0 = ((d.x - c.x) * (a.y - c.y) - (d.y - c.y) * (a.x - c.x)) / dnm;
  if ( !Number.between(t0, 0-epsilon, 1+epsilon) ) return null;

  // Vector distance from c
  const t1 = ((b.x - a.x) * (a.y - c.y) - (b.y - a.y) * (a.x - c.x)) / dnm;
  if ( !Number.between(t1, 0-epsilon, 1+epsilon) ) return null;

  // Return the point of intersection and the vector distance from both line origins
  return {
    x: a.x + t0 * (b.x - a.x),
    y: a.y + t0 * (b.y - a.y),
    t0: Math.clamp(t0, 0, 1),
    t1: Math.clamp(t1, 0, 1)
  }
}

/* -------------------------------------------- */

/**
 * Determine the intersection between a line segment and a circle.
 * @param {Point} a                   The first vertex of the segment
 * @param {Point} b                   The second vertex of the segment
 * @param {Point} center              The center of the circle
 * @param {number} radius             The radius of the circle
 * @param {number} epsilon            A small tolerance for floating point precision
 * @returns {LineCircleIntersection}  The intersection of the segment AB with the circle
 */
function lineCircleIntersection(a, b, center, radius, epsilon=1e-8) {
  const r2 = Math.pow(radius, 2);
  let intersections = [];

  // Test whether endpoint A is contained
  const ar2 = Math.pow(a.x - center.x, 2) + Math.pow(a.y - center.y, 2);
  const aInside = ar2 < r2 - epsilon;

  // Test whether endpoint B is contained
  const br2 = Math.pow(b.x - center.x, 2) + Math.pow(b.y - center.y, 2);
  const bInside = br2 < r2 - epsilon;

  // Find quadratic intersection points
  const contained = aInside && bInside;
  if ( !contained ) intersections = quadraticIntersection(a, b, center, radius, epsilon);

  // Return the intersection data
  return {
    aInside,
    bInside,
    contained,
    outside: !contained && !intersections.length,
    tangent: !aInside && !bInside && intersections.length === 1,
    intersections
  };
}

/* -------------------------------------------- */

/**
 * Identify the point closest to C on segment AB
 * @param {Point} c     The reference point C
 * @param {Point} a     Point A on segment AB
 * @param {Point} b     Point B on segment AB
 * @returns {Point}     The closest point to C on segment AB
 */
function closestPointToSegment(c, a, b) {
  const dx = b.x - a.x;
  const dy = b.y - a.y;
  if (( dx === 0 ) && ( dy === 0 )) {
    throw new Error("Zero-length segment AB not supported");
  }
  const u = (((c.x - a.x) * dx) + ((c.y - a.y) * dy)) / (dx * dx + dy * dy);
  if ( u < 0 ) return a;
  if ( u > 1 ) return b;
  else return {
    x: a.x + (u * dx),
    y: a.y + (u * dy)
  }
}

/* -------------------------------------------- */

/**
 * Determine the points of intersection between a line segment (p0,p1) and a circle.
 * There will be zero, one, or two intersections
 * See https://math.stackexchange.com/a/311956.
 * @param {Point} p0            The initial point of the line segment
 * @param {Point} p1            The terminal point of the line segment
 * @param {Point} center        The center of the circle
 * @param {number} radius       The radius of the circle
 * @param {number} [epsilon=0]  A small tolerance for floating point precision
 */
function quadraticIntersection(p0, p1, center, radius, epsilon=0) {
  const dx = p1.x - p0.x;
  const dy = p1.y - p0.y;

  // Quadratic terms where at^2 + bt + c = 0
  const a = Math.pow(dx, 2) + Math.pow(dy, 2);
  const b = (2 * dx * (p0.x - center.x)) + (2 * dy * (p0.y - center.y));
  const c = Math.pow(p0.x - center.x, 2) + Math.pow(p0.y - center.y, 2) - Math.pow(radius, 2);

  // Discriminant
  let disc2 = Math.pow(b, 2) - (4 * a * c);
  if ( disc2.almostEqual(0) ) disc2 = 0; // segment endpoint touches the circle; 1 intersection
  else if ( disc2 < 0 ) return []; // no intersections

  // Roots
  const disc = Math.sqrt(disc2);
  const t1 = (-b - disc) / (2 * a);

  // If t1 hits (between 0 and 1) it indicates an "entry"
  const intersections = [];
  if ( t1.between(0-epsilon, 1+epsilon) ) {
    intersections.push({
      x: p0.x + (dx * t1),
      y: p0.y + (dy * t1)
    });
  }
  if ( !disc2 ) return intersections; // 1 intersection

  // If t2 hits (between 0 and 1) it indicates an "exit"
  const t2 = (-b + disc) / (2 * a);
  if ( t2.between(0-epsilon, 1+epsilon) ) {
    intersections.push({
      x: p0.x + (dx * t2),
      y: p0.y + (dy * t2)
    });
  }
  return intersections;
}

/* -------------------------------------------- */

/**
 * Calculate the centroid non-self-intersecting closed polygon.
 * See https://en.wikipedia.org/wiki/Centroid#Of_a_polygon.
 * @param {Point[]|number[]} points    The points of the polygon
 * @returns {Point}                    The centroid of the polygon
 */
function polygonCentroid(points) {
  const n = points.length;
  if ( n === 0 ) return {x: 0, y: 0};
  let x = 0;
  let y = 0;
  let a = 0;
  if ( typeof points[0] === "number" ) {
    let x0 = points[n - 2];
    let y0 = points[n - 1];
    for ( let i = 0; i < n; i += 2 ) {
      const x1 = points[i];
      const y1 = points[i + 1];
      const z = (x0 * y1) - (x1 * y0);
      x += (x0 + x1) * z;
      y += (y0 + y1) * z;
      x0 = x1;
      y0 = y1;
      a += z;
    }
  } else {
    let {x: x0, y: y0} = points[n - 1];
    for ( let i = 0; i < n; i++ ) {
      const {x: x1, y: y1} = points[i];
      const z = (x0 * y1) - (x1 * y0);
      x += (x0 + x1) * z;
      y += (y0 + y1) * z;
      x0 = x1;
      y0 = y1;
      a += z;
    }
  }
  a *= 3;
  x /= a;
  y /= a;
  return {x, y};
}

/* -------------------------------------------- */

/**
 * Test whether the circle given by the center and radius intersects the path (open or closed).
 * @param {Point[]|number[]} points    The points of the path
 * @param {boolean} close              If true, the edge from the last to the first point is tested
 * @param {Point} center               The center of the circle
 * @param {number} radius              The radius of the circle
 * @returns {boolean}                  Does the circle intersect the path?
 */
function pathCircleIntersects(points, close, center, radius) {
  const n = points.length;
  if ( n === 0 ) return false;
  const {x: cx, y: cy} = center;
  const rr = radius * radius;
  let i;
  let x0;
  let y0;
  if ( typeof points[0] === "number" ) {
    if ( close ) {
      i = 0;
      x0 = points[n - 2];
      y0 = points[n - 1];
    } else {
      i = 2;
      x0 = points[0];
      y0 = points[1];
    }
    for ( ; i < n; i += 2 ) {
      const x1 = points[i];
      const y1 = points[i + 1];
      let dx = cx - x0;
      let dy = cy - y0;
      const nx = x1 - x0;
      const ny = y1 - y0;
      const t = Math.clamp(((dx * nx) + (dy * ny)) / ((nx * nx) + (ny * ny)), 0, 1);
      dx = (t * nx) - dx;
      dy = (t * ny) - dy;
      if ( (dx * dx) + (dy * dy) <= rr ) return true;
      x0 = x1;
      y0 = y1;
    }
  } else {
    if ( close ) {
      i = 0;
      ({x: x0, y: y0} = points[n - 1]);
    } else {
      i = 1;
      ({x: x0, y: y0} = points[0]);
    }
    for ( ; i < n; i++ ) {
      const {x: x1, y: y1} = points[i];
      let dx = cx - x0;
      let dy = cy - y0;
      const nx = x1 - x0;
      const ny = y1 - y0;
      const t = Math.clamp(((dx * nx) + (dy * ny)) / ((nx * nx) + (ny * ny)), 0, 1);
      dx = (t * nx) - dx;
      dy = (t * ny) - dy;
      if ( (dx * dx) + (dy * dy) <= rr ) return true;
      x0 = x1;
      y0 = y1;
    }
  }
  return false;
}

/* -------------------------------------------- */

/**
 * Test whether two circles (with position and radius) intersect.
 * @param {number} x0    x center coordinate of circle A.
 * @param {number} y0    y center coordinate of circle A.
 * @param {number} r0    radius of circle A.
 * @param {number} x1    x center coordinate of circle B.
 * @param {number} y1    y center coordinate of circle B.
 * @param {number} r1    radius of circle B.
 * @returns {boolean}    True if the two circles intersect, false otherwise.
 */
function circleCircleIntersects(x0, y0, r0, x1, y1, r1) {
  return Math.hypot(x0 - x1, y0 - y1) <= (r0 + r1);
}

/**
 * A wrapper method around `fetch` that attaches an AbortController signal to the `fetch` call for clean timeouts
 * @see {@link https://developer.mozilla.org/en-US/docs/Web/API/AbortSignal#aborting_a_fetch_with_timeout_or_explicit_abort}
 * @param {string} url            The URL to make the Request to
 * @param {RequestInit} data      The data of the Request
 * @param {object} [options]                 Additional options
 * @param {number|null} [options.timeoutMs]  How long to wait for a Response before cleanly aborting.
 *                                           If null, no timeout is applied. Default: `30000`.
 * @param {Function} [options.onTimeout]     A method to invoke if and when the timeout is reached
 * @returns {Promise<Response>}
 * @throws {HttpError}
 */
async function fetchWithTimeout(url, data={}, {timeoutMs=30000, onTimeout=() => {}}={}) {
  const controller = new AbortController();
  data.signal = controller.signal;
  let timedOut = false;
  const enforceTimeout = timeoutMs !== null;

  // Enforce a timeout
  let timeout;
  if ( enforceTimeout ) {
    timeout = setTimeout(() => {
      timedOut = true;
      controller.abort();
      onTimeout();
    }, timeoutMs);
  }

  // Attempt the request
  let response;
  try {
    response = await fetch(url, data);
  } catch(err) {
    if ( timedOut ) {
      const timeoutS = Math.round(timeoutMs / 1000);
      const msg = game.i18n
        ? game.i18n.format("SETUP.ErrorTimeout", { url, timeout: timeoutS })
        : `The request to ${url} timed out after ${timeoutS}s.`;
      throw new HttpError("Timed Out", 408, msg);
    }
    throw err;
  } finally {
    if ( enforceTimeout ) clearTimeout(timeout);
  }

  // Return the response
  if ( !response.ok && (response.type !== "opaqueredirect") ) {
    const responseBody = response.body ? await response.text() : "";
    throw new HttpError(response.statusText, response.status, responseBody);
  }
  return response;
}

/* ----------------------------------------- */

/**
 * A small wrapper that automatically asks for JSON with a Timeout
 * @param {string} url          The URL to make the Request to
 * @param {Object} data         The data of the Request
 * @param {object} [options]                 Additional options
 * @param {number|null} [options.timeoutMs]  How long to wait for a Response before cleanly aborting.
 *                                           If null, no timeout is applied. Default: `30000`.
 * @param {Function} [options.onTimeout]     A method to invoke if and when the timeout is reached
 * @returns {Promise<*>}
 */
async function fetchJsonWithTimeout(url, data = {}, {timeoutMs=30000, onTimeout = () => {}} = {}) {
  const response = await fetchWithTimeout(url, data, {timeoutMs, onTimeout: onTimeout});
  return response.json();
}

/* -------------------------------------------- */

/**
 * Test whether a file source exists by performing a HEAD request against it
 * @param {string} src          The source URL or path to test
 * @returns {Promise<boolean>}   Does the file exist at the provided url?
 */
async function srcExists$1(src) {
  return foundry.utils.fetchWithTimeout(src, { method: "HEAD" }).then(resp => {
    return resp.status < 400;
  }).catch(() => false);
}

/* ----------------------------------------- */

/**
 * Represents an HTTP Error when a non-OK response is returned by Fetch
 * @extends {Error}
 */
class HttpError extends Error {
  constructor(statusText, code, displayMessage="") {
    super(statusText);
    this.code = code;
    this.displayMessage = displayMessage;
  }

  /* -------------------------------------------- */

  /** @override */
  toString() {
    return this.displayMessage;
  }
}

/**
 * The messages that have been logged already and should not be logged again.
 * @type {Set<string>}
 */
const loggedCompatibilityWarnings = new Set();

/**
 * Log a compatibility warning which is filtered based on the client's defined compatibility settings.
 * @param {string} message            The original warning or error message
 * @param {object} [options={}]       Additional options which customize logging
 * @param {number} [options.mode]          A logging level in COMPATIBILITY_MODES which overrides the configured default
 * @param {number|string} [options.since]  A version identifier since which a change was made
 * @param {number|string} [options.until]  A version identifier until which a change remains supported
 * @param {string} [options.details]       Additional details to append to the logged message
 * @param {boolean} [options.stack=true]   Include the message stack trace
 * @param {boolean} [options.once=false]   Log this the message only once?
 * @throws                            An Error if the mode is ERROR
 */
function logCompatibilityWarning(message, {mode, since, until, details, stack=true, once=false}={}) {

  // Determine the logging mode
  const modes = COMPATIBILITY_MODES;
  const compatibility = globalThis.CONFIG?.compatibility || {
    mode: modes.WARNING,
    includePatterns: [],
    excludePatterns: []
  };
  mode ??= compatibility.mode;
  if ( mode === modes.SILENT ) return;

  // Compose the message
  since = since ? `Deprecated since Version ${since}` : null;
  until = until ? `Backwards-compatible support will be removed in Version ${until}`: null;
  message = [message, since, until, details].filterJoin("\n");

  // Filter the message by its stack trace
  const error = new Error(message);
  if ( compatibility.includePatterns.length ) {
    if ( !compatibility.includePatterns.some(rgx => rgx.test(error.message) || rgx.test(error.stack)) ) return;
  }
  if ( compatibility.excludePatterns.length ) {
    if ( compatibility.excludePatterns.some(rgx => rgx.test(error.message) || rgx.test(error.stack)) ) return;
  }

  // Log the message
  const log = !(once && loggedCompatibilityWarnings.has(error.stack));
  switch ( mode ) {
    case modes.WARNING:
      if ( log ) globalThis.logger.warn(stack ? error : error.message);
      break;
    case modes.ERROR:
      if ( log ) globalThis.logger.error(stack ? error : error.message);
      break;
    case modes.FAILURE:
      throw error;
  }
  if ( log && once ) loggedCompatibilityWarnings.add(error.stack);
}

/**
 * @import ApplicationV2 from "../../client/applications/api/application.mjs"
 * @import {Constructor} from "../_types.mjs"
 * @import {EmittedEventListener} from "./_types.mjs"
 */

/**
 * Augment a base class with EventEmitter behavior.
 * @template {Function} TBaseClass
 * @param {TBaseClass} [BaseClass] Some base class to be augmented with event emitter functionality: defaults to an
 *                                 anonymous empty class.
 */
function EventEmitterMixin(BaseClass=class {}) {
  /**
   * A mixin class which implements the behavior of EventTarget.
   * This is useful in cases where a class wants EventTarget-like behavior but needs to extend some other class.
   * @see {@link https://developer.mozilla.org/en-US/docs/Web/API/EventTarget}
   */
  class EventEmitter extends BaseClass {

    /**
     * An array of event types which are valid for this class.
     * @type {string[]}
     */
    static emittedEvents = [];

    /**
     * A mapping of registered events.
     * @type {Record<string, Map<EmittedEventListener, {fn: EmittedEventListener, once: boolean}>>}
     */
    #events = {};

    /* -------------------------------------------- */

    /**
     * Add a new event listener for a certain type of event.
     * @see {@link https://developer.mozilla.org/en-US/docs/Web/API/EventTarget/addEventListener}
     * @param {string} type                     The type of event being registered for
     * @param {EmittedEventListener} listener   The listener function called when the event occurs
     * @param {object} [options={}]             Options which configure the event listener
     * @param {boolean} [options.once=false]      Should the event only be responded to once and then removed
     */
    addEventListener(type, listener, {once = false} = {}) {
      if ( !this.constructor.emittedEvents.includes(type) ) {
        throw new Error(`"${type}" is not a supported event of the ${this.constructor.name} class`);
      }
      this.#events[type] ||= new Map();
      this.#events[type].set(listener, {fn: listener, once});
    }

    /* -------------------------------------------- */

    /**
     * Remove an event listener for a certain type of event.
     * @see {@link https://developer.mozilla.org/en-US/docs/Web/API/EventTarget/removeEventListener}
     * @param {string} type                     The type of event being removed
     * @param {EmittedEventListener} listener   The listener function being removed
     */
    removeEventListener(type, listener) {
      this.#events[type]?.delete(listener);
    }

    /* -------------------------------------------- */

    /**
     * Dispatch an event on this target.
     * @see {@link https://developer.mozilla.org/en-US/docs/Web/API/EventTarget/dispatchEvent}
     * @param {Event} event                     The Event to dispatch
     * @returns {boolean}                       Was default behavior for the event prevented?
     */
    dispatchEvent(event) {
      if ( !(event instanceof Event) ) {
        throw new Error("EventEmitter#dispatchEvent must be provided an Event instance");
      }
      if ( !this.constructor.emittedEvents.includes(event?.type) ) {
        throw new Error(`"${event.type}" is not a supported event of the ${this.constructor.name} class`);
      }
      const listeners = this.#events[event.type];
      if ( !listeners ) return true;

      // Extend and configure the Event
      Object.defineProperties(event, {
        target: {value: this},
        stopPropagation: {value: function() {
          event.propagationStopped = true;
          Event.prototype.stopPropagation.call(this);
        }},
        stopImmediatePropagation: {value: function() {
          event.propagationStopped = true;
          Event.prototype.stopImmediatePropagation.call(this);
        }}
      });

      // Call registered listeners
      for ( const listener of listeners.values() ) {
        listener.fn(event);
        if ( listener.once ) this.removeEventListener(event.type, listener.fn);
        if ( event.propagationStopped ) break;
      }
      return event.defaultPrevented;
    }
  }
  return EventEmitter;
}

/**
 * @import {IterableWeakMapHeldValue, IterableWeakMapValue} from "./_types.mjs";
 */

/**
 * Stores a map of objects with weak references to the keys, allowing them to be garbage collected. Both keys and values
 * can be iterated over, unlike a WeakMap.
 */
class IterableWeakMap extends WeakMap {

  /**
   * A set of weak refs to the map's keys, allowing enumeration.
   * @type {Set<WeakRef<any>>}
   */
  #refs = new Set();

  /**
   * A FinalizationRegistry instance to clean up the ref set when objects are garbage collected.
   * @type {FinalizationRegistry<IterableWeakMapHeldValue>}
   */
  #finalizer = new FinalizationRegistry(IterableWeakMap.#cleanup);

  /**
   * @param {Iterable<[any, any]>} [entries]  The initial entries.
   */
  constructor(entries=[]) {
    super();
    for ( const [key, value] of entries ) this.set(key, value);
  }

  /* -------------------------------------------- */

  /**
   * Clean up the corresponding ref in the set when its value is garbage collected.
   * @param {IterableWeakMapHeldValue} heldValue  The value held by the finalizer.
   */
  static #cleanup({ set, ref }) {
    set.delete(ref);
  }

  /* -------------------------------------------- */

  /**
   * Remove a key from the map.
   * @param {any} key  The key to remove.
   * @returns {boolean}
   */
  delete(key) {
    const entry = super.get(key);
    if ( !entry ) return false;
    super.delete(key);
    this.#refs.delete(entry.ref);
    this.#finalizer.unregister(key);
    return true;
  }

  /* -------------------------------------------- */

  /**
   * Retrieve a value from the map.
   * @param {any} key  The value's key.
   * @returns {any}
   */
  get(key) {
    /** @type {IterableWeakMapValue|undefined} */
    const entry = super.get(key);
    return entry && entry.value;
  }

  /* -------------------------------------------- */

  /**
   * Place a value in the map.
   * @param {any} key    The key.
   * @param {any} value  The value.
   * @returns {IterableWeakMap}
   */
  set(key, value) {
    const entry = super.get(key);
    if ( entry ) this.#refs.delete(entry.ref);
    const ref = new WeakRef(key);
    super.set(key, /** @type {IterableWeakMapValue} */ { value, ref });
    this.#refs.add(ref);
    this.#finalizer.register(key, { ref, set: this.#refs }, key);
    return this;
  }

  /* -------------------------------------------- */

  /**
   * Clear all values from the map.
   */
  clear() {
    for ( const ref of this.#refs ) {
      const key = ref.deref();
      if ( key ) this.delete(key);
      else this.#refs.delete(ref);
    }
  }

  /* -------------------------------------------- */

  /**
   * Enumerate the entries.
   * @returns {Generator<[any, any], void, any>}
   */
  *[Symbol.iterator]() {
    for ( const ref of this.#refs ) {
      const key = ref.deref();
      if ( !key ) continue;
      const { value } = super.get(key);
      yield [key, value];
    }
  }

  /* -------------------------------------------- */

  /**
   * Enumerate the entries.
   * @returns {Generator<[any, any], void, any>}
   */
  entries() {
    return this[Symbol.iterator]();
  }

  /* -------------------------------------------- */

  /**
   * Enumerate the keys.
   * @returns {Generator<any, void, any>}
   */
  *keys() {
    for ( const [key] of this ) yield key;
  }

  /* -------------------------------------------- */

  /**
   * Enumerate the values.
   * @returns {Generator<any, void, any>}
   */
  *values() {
    for ( const [, value] of this ) yield value;
  }
}

/**
 * Stores a set of objects with weak references to them, allowing them to be garbage collected. Can be iterated over,
 * unlike a WeakSet.
 */
class IterableWeakSet extends WeakSet {
  /**
   * The backing iterable weak map.
   * @type {IterableWeakMap<any, any>}
   */
  #map = new IterableWeakMap();

  /**
   * @param {Iterable<any>} [entries]  The initial entries.
   */
  constructor(entries=[]) {
    super();
    for ( const entry of entries ) this.add(entry);
  }

  /* -------------------------------------------- */

  /**
   * Enumerate the values.
   * @returns {Generator<any, void, any>}
   */
  [Symbol.iterator]() {
    return this.values();
  }

  /* -------------------------------------------- */

  /**
   * Add a value to the set.
   * @param {any} value  The value to add.
   * @returns {IterableWeakSet}
   */
  add(value) {
    this.#map.set(value, value);
    return this;
  }

  /* -------------------------------------------- */

  /**
   * Delete a value from the set.
   * @param {any} value  The value to delete.
   * @returns {boolean}
   */
  delete(value) {
    return this.#map.delete(value);
  }

  /* -------------------------------------------- */

  /**
   * Whether this set contains the given value.
   * @param {any} value  The value to test.
   * @returns {boolean}
   */
  has(value) {
    return this.#map.has(value);
  }

  /* -------------------------------------------- */

  /**
   * Enumerate the collection.
   * @returns {Generator<any, void, any>}
   */
  values() {
    return this.#map.values();
  }

  /* -------------------------------------------- */

  /**
   * Clear all values from the set.
   */
  clear() {
    this.#map.clear();
  }
}

/**
 * A simple Semaphore implementation which provides a limited queue for ensuring proper concurrency.
 * @param {number} [max=1]    The maximum number of tasks which are allowed concurrently.
 *
 * @example Using a Semaphore
 * ```js
 * // Some async function that takes time to execute
 * function fn(x) {
 *   return new Promise(resolve => {
 *     setTimeout(() => {
 *       console.log(x);
 *       resolve(x);
 *     }, 1000);
 *   });
 * }
 *
 * // Create a Semaphore and add many concurrent tasks
 * const semaphore = new Semaphore(1);
 * for ( let i of Array.fromRange(100) ) {
 *   semaphore.add(fn, i);
 * }
 * ```
 */
class Semaphore {
  constructor(max=1) {

    /**
     * The maximum number of tasks which can be simultaneously attempted.
     * @type {number}
     */
    this.max = max;
  }

  /**
   * A queue of pending function signatures
   * @type {Array<[fn: Function, args: any[], resolve: (result: any) => void, reject: (error: Error) => void]>}
   */
  #queue = [];

  /**
   * The number of tasks which are currently underway
   * @type {number}
   */
  #active = 0;

  /**
   * The number of pending tasks remaining in the queue
   * @type {number}
   */
  get remaining() {
    return this.#queue.length;
  }

  /**
   * The number of actively executing tasks
   * @type {number}
   */
  get active() {
    return this.#active;
  }

  /**
   * Add a new tasks to the managed queue
   * @param {Function} fn     A callable function
   * @param {...*} [args]     Function arguments
   * @returns {Promise}       A promise that resolves once the added function is executed
   */
  add(fn, ...args) {
    return new Promise((resolve, reject) => {
      this.#queue.push([fn, args, resolve, reject]);
      this.#try();
    });
  }

  /**
   * Abandon any tasks which have not yet concluded
   */
  clear() {
    this.#queue = [];
  }

  /**
   * Attempt to perform a task from the queue.
   * If all workers are busy, do nothing.
   * If successful, try again.
   */
  async #try() {
    if ( (this.active === this.max) || !this.remaining ) return false;

    // Obtain the next task from the queue
    const next = this.#queue.shift();
    if ( !next ) return;
    this.#active += 1;

    // Try and execute it, resolving its promise
    const [fn, args, resolve, reject] = next;
    try {
      const r = await fn(...args);
      resolve(r);
    }
    catch(err) {
      reject(err);
    }

    // Try the next function in the queue
    this.#active -= 1;
    return this.#try();
  }
}

/**
 * Create a new BitMask instance.
 * @param {Record<string, boolean>} [states=null] An object containing valid states and their corresponding initial boolean values (default is null).
 */
class BitMask extends Number {
  constructor(states=null) {
    super();
    this.#generateValidStates(states);
    this.#generateEnum();
    this.#value = this.#computeValue(states);
  }

  /**
   * The real value behind the bitmask instance.
   * @type {number}
   */
  #value;

  /**
   * The structure of valid states and their associated values.
   * @type {Map<string, number>}
   */
  #validStates;

  /**
   * The enum associated with this structure.
   * @type {Record<string, string>}
   * @readonly
   */
  states;

  /* -------------------------------------------- */
  /*  Internals                                   */
  /* -------------------------------------------- */

  /**
   * Generates the valid states and their associated values.
   * @param {Record<string, boolean>} [states=null] The structure defining the valid states and their associated values.
   */
  #generateValidStates(states) {
    this.#validStates = new Map();
    let bitIndex = 0;
    for ( const state of Object.keys(states || {}) ) {
      if ( bitIndex >= 32 ) throw new Error("A bitmask can't handle more than 32 states");
      this.#validStates.set(state, 1 << bitIndex++);
    }
  }

  /* -------------------------------------------- */

  /**
   * Generates an enum based on the provided valid states.
   */
  #generateEnum() {
    this.states = {};
    for ( const state of this.#validStates.keys() ) this.states[state] = state;
    Object.freeze(this.states);
  }

  /* -------------------------------------------- */

  /**
   * Calculate the default value of the bitmask based on the initial states
   * @param {Record<string, boolean>} [initialStates={}] The structure defining the valid states and their associated values.
   * @returns {number}
   */
  #computeValue(initialStates={}) {
    let defaultValue = 0;
    for ( const state in initialStates ) {
      if ( !initialStates.hasOwnProperty(state) ) continue;
      this.#checkState(state);
      if ( initialStates[state] ) defaultValue |= this.#validStates.get(state);
    }
    return defaultValue;
  }

  /* -------------------------------------------- */

  /**
   * Checks a state and throws an error if it doesn't exist.
   * @param {string} state   Name of the state to check.
   */
  #checkState(state) {
    if ( !this.#validStates.has(state) ) {
      throw new Error(`${state} is an invalid state for this BitMask instance: ${this.toJSON()}`);
    }
  }

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * True if this bitmask is empty (no active states).
   * @type {boolean}
   */
  get isEmpty() {
    return this.#value === 0;
  }

  /* -------------------------------------------- */
  /*  Methods for Handling states                 */
  /* -------------------------------------------- */

  /**
   * Check if a specific state is active.
   * @param {string} state The state to check.
   * @returns {boolean} True if the state is active, false otherwise.
   */
  hasState(state) {
    return (this.#value & this.#validStates.get(state)) !== 0;
  }

  /* -------------------------------------------- */

  /**
   * Add a state to the bitmask.
   * @param {string} state The state to add.
   * @throws {Error} Throws an error if the provided state is not valid.
   */
  addState(state) {
    this.#checkState(state);
    this.#value |= this.#validStates.get(state);
  }

  /* -------------------------------------------- */

  /**
   * Remove a state from the bitmask.
   * @param {string} state The state to remove.
   * @throws {Error} Throws an error if the provided state is not valid.
   */
  removeState(state) {
    this.#checkState(state);
    this.#value &= ~this.#validStates.get(state);
  }

  /* -------------------------------------------- */

  /**
   * Toggle the state of a specific state in the bitmask.
   * @param {string} state The state to toggle.
   * @param {boolean} [enabled] Toggle on (true) or off (false)? If undefined, the state is switched automatically.
   * @throws {Error} Throws an error if the provided state is not valid.
   */
  toggleState(state, enabled) {
    this.#checkState(state);
    if ( enabled === undefined ) return (this.#value ^= this.#validStates.get(state));
    if ( enabled ) this.addState(state);
    else this.removeState(state);
  }

  /* -------------------------------------------- */

  /**
   * Clear the bitmask, setting all states to inactive.
   */
  clear() {
    this.#value = 0;
  }

  /* -------------------------------------------- */
  /*  bitmask representations                     */
  /* -------------------------------------------- */

  /**
   * Get the current value of the bitmask.
   * @returns {number} The current value of the bitmask.
   */
  valueOf() {
    return this.#value;
  }

  /* -------------------------------------------- */

  /**
   * Get a string representation of the bitmask in binary format.
   * @returns {string} The string representation of the bitmask.
   */
  toString() {
    return String(this.#value.toString(2)).padStart(this.#validStates.size, '0');
  }

  /* -------------------------------------------- */

  /**
   * Checks if two bitmasks structures are compatible (the same valid states).
   * @param {BitMask} otherBitMask The bitmask structure to compare with.
   * @returns {boolean} True if the two bitmasks have the same structure, false otherwise.
   */
  isCompatible(otherBitMask) {
    const states1 = Array.from(this.#validStates.keys()).sort().join(',');
    const states2 = Array.from(otherBitMask.#validStates.keys()).sort().join(',');
    return states1 === states2;
  }

  /* -------------------------------------------- */

  /**
   * Serializes the bitmask to a JSON string.
   * @returns {string} The JSON string representing the bitmask.
   */
  toJSON() {
    return JSON.stringify(this.toObject());
  }

  /* -------------------------------------------- */

  /**
   * Creates a new BitMask instance from a JSON string.
   * @param {string} jsonString The JSON string representing the bitmask.
   * @returns {BitMask} A new BitMask instance created from the JSON string.
   */
  static fromJSON(jsonString) {
    const data = JSON.parse(jsonString);
    return new BitMask(data);
  }

  /* -------------------------------------------- */

  /**
   * Convert value of this BitMask to object representation according to structure.
   * @returns {Object} The data represented by the bitmask.
   */
  toObject() {
    const result = {};
    for ( const [validState, value] of this.#validStates ) result[validState] = ((this.#value & value) !== 0);
    return result;
  }

  /* -------------------------------------------- */

  /**
   * Creates a clone of this BitMask instance.
   * @returns {BitMask} A new BitMask instance with the same value and valid states as this instance.
   */
  clone() {
    return new BitMask(this.toObject());
  }

  /* -------------------------------------------- */
  /*  Static Helpers                              */
  /* -------------------------------------------- */

  /**
   * Generates shader constants based on the provided states.
   * @param {string[]} states An array containing valid states.
   * @returns {string} Shader bit mask constants generated from the states.
   */
  static generateShaderBitMaskConstants(states) {
    let shaderConstants = '';
    let bitIndex = 0;
    for ( const state of states ) {
      shaderConstants += `const uint ${state.toUpperCase()} = 0x${(1 << bitIndex).toString(16).toUpperCase()}U;\n`;
      bitIndex++;
    }
    return shaderConstants;
  }
}

/**
 * @import {StringTreeEntryFilter, StringTreeNode} from "./_types.mjs";
 */

/**
 * A data structure representing a tree of string nodes with arbitrary object leaves.
 */
class StringTree {
  /**
   * The key symbol that stores the leaves of any given node.
   * @type {symbol}
   */
  static get leaves() {
    return StringTree.#leaves;
  }

  static #leaves = Symbol();

  /* -------------------------------------------- */

  /**
   * The tree's root.
   * @type {StringTreeNode}
   */
  #root = this.#createNode();

  /* -------------------------------------------- */

  /**
   * Create a new node.
   * @returns {StringTreeNode}
   */
  #createNode() {
    return { [StringTree.leaves]: [] };
  }

  /* -------------------------------------------- */

  /**
   * Insert an entry into the tree.
   * @param {string[]} strings  The string parents for the entry.
   * @param {any} entry         The entry to store.
   * @returns {StringTreeNode}  The node the entry was added to.
   */
  addLeaf(strings, entry) {
    let node = this.#root;
    for ( const string of strings ) {
      node[string] ??= this.#createNode();
      node = node[string];
    }

    // Once we've traversed the tree, we add our entry.
    node[StringTree.leaves].push(entry);
    return node;
  }

  /* -------------------------------------------- */

  /**
   * Traverse the tree along the given string path and return any entries reachable from the node.
   * @param {string[]} strings                               The string path to the desired node.
   * @param {object} [options]
   * @param {number} [options.limit]                         The maximum number of items to retrieve.
   * @param {StringTreeEntryFilter} [options.filterEntries]  A filter function to apply to each candidate entry.
   * @returns {any[]}
   */
  lookup(strings, { limit, filterEntries }={}) {
    const entries = [];
    const node = this.nodeAtPrefix(strings);
    if ( !node ) return []; // No matching entries.
    const queue = [node];
    while ( queue.length ) {
      if ( limit && (entries.length >= limit) ) break;
      this._breadthFirstSearch(queue.shift(), entries, queue, { limit, filterEntries });
    }
    return entries;
  }

  /* -------------------------------------------- */

  /**
   * Returns the node at the given path through the tree.
   * @param {string[]} strings                    The string path to the desired node.
   * @param {object} [options]
   * @param {boolean} [options.hasLeaves=false]   Only return the most recently visited node that has leaves, otherwise
   *                                              return the exact node at the prefix, if it exists.
   * @returns {StringTreeNode|void}
   */
  nodeAtPrefix(strings, { hasLeaves=false }={}) {
    let node = this.#root;
    let withLeaves = node;
    for ( const string of strings ) {
      if ( !(string in node) ) return hasLeaves ? withLeaves : undefined;
      node = node[string];
      if ( node[StringTree.leaves].length ) withLeaves = node;
    }
    return hasLeaves ? withLeaves : node;
  }

  /* -------------------------------------------- */

  /**
   * Perform a breadth-first search starting from the given node and retrieving any entries reachable from that node,
   * until we reach the limit.
   * @param {StringTreeNode} node                            The starting node.
   * @param {any[]} entries                                  The accumulated entries.
   * @param {StringTreeNode[]} queue                         The working queue of nodes to search.
   * @param {object} [options]
   * @param {number} [options.limit]                         The maximum number of entries to retrieve before stopping.
   * @param {StringTreeEntryFilter} [options.filterEntries]  A filter function to apply to each candidate entry.
   * @protected
   */
  _breadthFirstSearch(node, entries, queue, { limit, filterEntries }={}) {
    // Retrieve the entries at this node.
    let leaves = node[StringTree.leaves];
    if ( filterEntries instanceof Function ) leaves = leaves.filter(filterEntries);
    entries.push(...leaves);
    if ( limit && (entries.length >= limit) ) return;
    // Push this node's children onto the end of the queue.
    for ( const key of Object.keys(node) ) {
      if ( typeof key === "string" ) queue.push(node[key]);
    }
  }
}

/**
 * @import {StringTreeEntryFilter, StringTreeNode, WordTreeEntry} from "./_types.mjs";
 */

/**
 * A data structure for quickly retrieving objects by a string prefix.
 * Note that this works well for languages with alphabets (latin, cyrillic, korean, etc.), but may need more nuanced
 * handling for languages that compose characters and letters.
 * @extends {StringTree}
 */
class WordTree extends StringTree {
  /**
   * Insert an entry into the tree.
   * @param {string} string        The string key for the entry.
   * @param {WordTreeEntry} entry  The entry to store.
   * @returns {StringTreeNode}     The node the entry was added to.
   */
  addLeaf(string, entry) {
    string = string.toLocaleLowerCase(game.i18n.lang);
    return super.addLeaf(Array.from(string), entry);
  }

  /* -------------------------------------------- */

  /**
   * Return entries that match the given string prefix.
   * @param {string} prefix              The prefix.
   * @param {object} [options]           Additional options to configure behaviour.
   * @param {number} [options.limit=10]  The maximum number of items to retrieve. It is important to set this value as
   *                                     very short prefixes will naturally match large numbers of entries.
   * @param {StringTreeEntryFilter} [options.filterEntries]  A filter function to apply to each candidate entry.
   * @returns {WordTreeEntry[]}          A number of entries that have the given prefix.
   */
  lookup(prefix, { limit=10, filterEntries }={}) {
    return super.lookup(prefix, { limit, filterEntries });
  }

  /* -------------------------------------------- */

  /**
   * Returns the node at the given prefix.
   * @param {string} prefix  The prefix.
   * @returns {StringTreeNode}
   */
  nodeAtPrefix(prefix) {
    prefix = prefix.toLocaleLowerCase(game.i18n.lang);
    return super.nodeAtPrefix(Array.from(prefix));
  }
}

/**
 * The constructor of an async function.
 * @type {typeof AsyncFunction}
 */
const AsyncFunction = (async function() {}).constructor;

/**
 * This module contains data field classes which are used to define a data schema.
 * A data field is responsible for cleaning, validation, and initialization of the value assigned to it.
 * Each data field extends the {@link foundry.data.fields.DataField} class to implement logic specific to its
 * contained data type.
 * @module fields
 */


/**
 * @import {EffectChangeData} from "../documents/_types.mjs";
 * @import {
 *   ArrayFieldOptions,
 *   ChoiceInputConfig,
 *   CodeMirrorInputConfig,
 *   DataFieldContext,
 *   DataFieldOptions,
 *   DataFieldValidationOptions,
 *   DocumentStats,
 *   DocumentUUIDFieldOptions,
 *   FilePathFieldOptions,
 *   FormGroupConfig,
 *   FormInputConfig,
 *   JavaScriptFieldOptions,
 *   NumberFieldOptions,
 *   StringFieldInputConfig,
 *   StringFieldOptions
 * } from "./_types.mjs";
 * @import {Document, DataModel} from "../abstract/_module.mjs";
 * @import {DataSchema, DataModelUpdateOptions} from "../abstract/_types.mjs";
 * @import {FormSelectOption} from "../../client/applications/forms/fields.mjs"
 */

/* ---------------------------------------- */
/*  Abstract Data Field                     */
/* ---------------------------------------- */

/**
 * An abstract class that defines the base pattern for a data field within a data schema.
 * @property {string} name                The name of this data field within the schema that contains it.
 * @mixes DataFieldOptions
 */
class DataField {
  /**
   * @param {DataFieldOptions} [options]    Options which configure the behavior of the field
   * @param {DataFieldContext} [context]    Additional context which describes the field
   */
  constructor(options={}, {name, parent}={}) {
    this.name = name;
    this.parent = parent;
    this.options = options;
    for ( const k in this.constructor._defaults ) {
      this[k] = k in this.options ? this.options[k] : this.constructor._defaults[k];
    }
  }

  /**
   * The field name of this DataField instance.
   * This is assigned by SchemaField#initialize.
   * @internal
   */
  name;

  /**
   * A reference to the parent schema to which this DataField belongs.
   * This is assigned by SchemaField#initialize.
   * @internal
   */
  parent;

  /**
   * The initially provided options which configure the data field
   * @type {DataFieldOptions}
   */
  options;

  /**
   * Whether this field defines part of a Document/Embedded Document hierarchy.
   * @type {boolean}
   */
  static hierarchical = false;

  /**
   * Does this field type contain other fields in a recursive structure?
   * Examples of recursive fields are SchemaField, ArrayField, or TypeDataField
   * Examples of non-recursive fields are StringField, NumberField, or ObjectField
   * @type {boolean}
   */
  static recursive = false;

  /**
   * Default parameters for this field type
   * @returns {DataFieldOptions}
   * @protected
   */
  static get _defaults() {
    return {
      required: false,
      nullable: false,
      initial: undefined,
      readonly: false,
      gmOnly: false,
      label: "",
      hint: "",
      validationError: "is not a valid value"
    };
  }

  /**
   * A dot-separated string representation of the field path within the parent schema.
   * @type {string}
   */
  get fieldPath() {
    return [this.parent?.fieldPath, this.name].filterJoin(".");
  }

  /**
   * Apply a function to this DataField which propagates through recursively to any contained data schema.
   * @param {string|Function} fn          The function to apply
   * @param {*} value                     The current value of this field
   * @param {object} [options={}]         Additional options passed to the applied function
   * @returns {object}                    The results object
   */
  apply(fn, value, options={}) {
    if ( typeof fn === "string" ) fn = this[fn];
    return fn.call(this, value, options);
  }

  /* -------------------------------------------- */

  /**
   * Add types of the source to the data if they are missing.
   * @param {*} source                           The source data
   * @param {*} changes                          The partial data
   * @param {object} [options]                   Additional options
   * @param {object} [options.source]            The root data model source
   * @param {object} [options.changes]           The root data model changes
   * @internal
   */
  _addTypes(source, changes, options) {}

  /* -------------------------------------------- */

  /**
   * Recursively traverse a schema and retrieve a field specification by a given path
   * @param {string[]} path             The field path as an array of strings
   * @returns {DataField|undefined}     The corresponding DataField definition for that field, or undefined
   * @internal
   */
  _getField(path) {
    return path.length ? undefined : this;
  }

  /* -------------------------------------------- */
  /*  Field Cleaning                              */
  /* -------------------------------------------- */

  /**
   * Coerce source data to ensure that it conforms to the correct data type for the field.
   * Data coercion operations should be simple and synchronous as these are applied whenever a DataModel is constructed.
   * For one-off cleaning of user-provided input the sanitize method should be used.
   * @param {*} value           An initial requested value
   * @param {object} [options]  Additional options for how the field is cleaned
   * @param {boolean} [options.partial]   Whether to perform partial cleaning?
   * @param {object} [options.source]     The root data model being cleaned
   * @returns {*}               The cast value
   */
  clean(value, options={}) {

    // Get an initial value for the field
    if ( value === undefined ) return this.getInitialValue(options.source);

    // Keep allowed special values
    try {
      const isValid = this._validateSpecial(value);
      if ( isValid === true ) return value;
    } catch(err) {
      return this.getInitialValue(options.source);
    }

    // Cast a provided value to the correct type
    value = this._cast(value);

    // Cleaning logic specific to the DataField.
    return this._cleanType(value, options);
  }

  /* -------------------------------------------- */

  /**
   * Apply any cleaning logic specific to this DataField type.
   * @param {*} value           The appropriately coerced value.
   * @param {object} [options]  Additional options for how the field is cleaned.
   * @returns {*}               The cleaned value.
   * @protected
   */
  _cleanType(value, options) {
    return value;
  }

  /* -------------------------------------------- */

  /**
   * Cast a non-default value to ensure it is the correct type for the field
   * @param {*} value       The provided non-default value
   * @returns {*}           The standardized value
   * @protected
   */
  _cast(value) {
    return value;
  }

  /* -------------------------------------------- */

  /**
   * Attempt to retrieve a valid initial value for the DataField.
   * @param {object} data   The source data object for which an initial value is required
   * @returns {*}           A proposed initial value
   */
  getInitialValue(data) {
    if ( this.initial instanceof Function ) return this.initial(data);  // Explicit function
    else if ( this.initial !== undefined ) return this.initial;         // Explicit value
    if ( !this.required ) return undefined;                             // Prefer undefined if non-required
    if ( this.nullable ) return null;                                   // Prefer explicit null
    return undefined;                                                   // Otherwise undefined
  }

  /* -------------------------------------------- */

  /**
   * Export the current value of the field into a serializable object.
   * @param {*} value                   The initialized value of the field
   * @returns {*}                       An exported representation of the field
   */
  toObject(value) {
    return value;
  }

  /* -------------------------------------------- */
  /*  Field Validation                            */
  /* -------------------------------------------- */

  /**
   * Validate a candidate input for this field, ensuring it meets the field requirements.
   * A validation failure can be provided as a raised Error (with a string message), by returning false, or by returning
   * a DataModelValidationFailure instance.
   * A validator which returns true denotes that the result is certainly valid and further validations are unnecessary.
   * @param {*} value                                  The initial value
   * @param {DataFieldValidationOptions} [options={}]  Options which affect validation behavior
   * @returns {DataModelValidationFailure|void}        Returns a DataModelValidationFailure if a validation failure
   *                                                   occurred.
   */
  validate(value, options={}) {
    const validators = [this._validateSpecial, this._validateType];
    if ( this.options.validate ) validators.push(this.options.validate);
    try {
      for ( const validator of validators ) {
        const isValid = validator.call(this, value, options);
        if ( isValid === true ) return undefined;
        if ( isValid === false ) {
          return new DataModelValidationFailure({
            invalidValue: value,
            message: this.validationError,
            unresolved: true
          });
        }
        if ( isValid instanceof DataModelValidationFailure ) return isValid;
      }
    } catch(err) {
      return new DataModelValidationFailure({invalidValue: value, message: err.message, unresolved: true});
    }
  }

  /* -------------------------------------------- */

  /**
   * Special validation rules which supersede regular field validation.
   * This validator screens for certain values which are otherwise incompatible with this field like null or undefined.
   * @param {*} value               The candidate value
   * @returns {boolean|void}        A boolean to indicate with certainty whether the value is valid.
   *                                Otherwise, return void.
   * @throws {Error}                May throw a specific error if the value is not valid
   * @protected
   */
  _validateSpecial(value) {

    // Allow null values for explicitly nullable fields
    if ( value === null ) {
      if ( this.nullable ) return true;
      else throw new Error("may not be null");
    }

    // Allow undefined if the field is not required
    if ( value === undefined ) {
      if ( this.required ) throw new Error("may not be undefined");
      else return true;
    }
  }

  /* -------------------------------------------- */

  /**
   * A default type-specific validator that can be overridden by child classes
   * @param {*} value                                    The candidate value
   * @param {DataFieldValidationOptions} [options={}]    Options which affect validation behavior
   * @returns {boolean|DataModelValidationFailure|void}  A boolean to indicate with certainty whether the value is
   *                                                     valid, or specific DataModelValidationFailure information,
   *                                                     otherwise void.
   * @throws                                             May throw a specific error if the value is not valid
   * @protected
   */
  _validateType(value, options={}) {}

  /* -------------------------------------------- */

  /**
   * Certain fields may declare joint data validation criteria.
   * This method will only be called if the field is designated as recursive.
   * @param {object} data       Candidate data for joint model validation
   * @param {object} options    Options which modify joint model validation
   * @throws  An error if joint model validation fails
   * @internal
   */
  _validateModel(data, options={}) {}

  /* -------------------------------------------- */
  /*  Initialization and Updates                  */
  /* -------------------------------------------- */

  /**
   * Initialize the original source data into a mutable copy for the DataModel instance.
   * @param {*} value                   The source value of the field
   * @param {Object} model              The DataModel instance that this field belongs to
   * @param {object} [options]          Initialization options
   * @returns {*}                       An initialized copy of the source data
   */
  initialize(value, model, options={}) {
    return value;
  }

  /* -------------------------------------------- */

  /**
   * Update the source data for a DataModel which includes this DataField.
   * This method is responsible for modifying the provided source data as well as updating the tracked diff included
   * in provided metadata.
   * @param {object} source               Source data of the DataModel which should be updated. This object is always
   *                                      a partial node of source data, relative to which this field belongs.
   * @param {string} key                  The name of this field within the context of the source data.
   * @param {any} value                   The candidate value that should be applied as an update.
   * @param {object} difference           The accumulated diff that is recursively populated as the model traverses
   *                                      through its schema fields.
   * @param {DataModelUpdateOptions} options Options which modify how this update workflow is performed.
   * @throws {Error}                      An error if the requested update cannot be performed.
   * @internal
   */
  _updateDiff(source, key, value, difference, options) {
    const current = source[key];
    if ( value === current ) return;
    difference[key] = value;
    source[key] = value;
  }

  /* -------------------------------------------- */

  /**
   * Commit a prepared update to DataModel#_source.
   * @param {object} source               The parent source object within which the `key` field exists
   * @param {string} key                  The named field in source to commit
   * @param {object} value                The new value of the field which should be committed to source
   * @param {object} diff                 The reported change to the field
   * @param {DataModelUpdateOptions} options Options which modify how this update workflow is performed.
   * @internal
   */
  _updateCommit(source, key, value, diff, options) {
    source[key] = value;
  }

  /* -------------------------------------------- */
  /*  Form Field Integration                      */
  /* -------------------------------------------- */

  /**
   * Does this form field class have defined form support?
   * @type {boolean}
   */
  static get hasFormSupport() {
    return this.prototype._toInput !== DataField.prototype._toInput;
  }

  /* -------------------------------------------- */

  /**
   * Render this DataField as an HTML element.
   * @param {FormInputConfig} config        Form element configuration parameters
   * @throws {Error}                        An Error if this DataField subclass does not support input rendering
   * @returns {HTMLElement|HTMLCollection}  A rendered HTMLElement for the field
   */
  toInput(config={}) {
    const inputConfig = {name: this.fieldPath, ...config};
    if ( inputConfig.input instanceof Function ) return config.input(this, inputConfig);
    return this._toInput(inputConfig);
  }

  /* -------------------------------------------- */

  // eslint-disable-next-line jsdoc/require-returns-check
  /**
   * Render this DataField as an HTML element.
   * Subclasses should implement this method rather than the public toInput method which wraps it.
   * @param {FormInputConfig} config        Form element configuration parameters
   * @throws {Error}                        An Error if this DataField subclass does not support input rendering
   * @returns {HTMLElement|HTMLCollection}  A rendered HTMLElement for the field
   * @protected
   */
  _toInput(config) {
    throw new Error(`The ${this.constructor.name} class does not implement the _toInput method`);
  }

  /* -------------------------------------------- */

  /**
   * Render this DataField as a standardized form-group element.
   * @param {FormGroupConfig} groupConfig   Configuration options passed to the wrapping form-group
   * @param {FormInputConfig} inputConfig   Input element configuration options passed to DataField#toInput
   * @returns {HTMLDivElement}              The rendered form group element
   */
  toFormGroup(groupConfig={}, inputConfig={}) {
    if ( groupConfig.widget instanceof Function ) return groupConfig.widget(this, groupConfig, inputConfig);
    groupConfig.label ??= this.label ?? this.fieldPath;
    groupConfig.hint ??= this.hint;
    groupConfig.input ??= this.toInput(inputConfig);
    return foundry.applications.fields.createFormGroup(groupConfig);
  }

  /* -------------------------------------------- */
  /*  Active Effect Integration                   */
  /* -------------------------------------------- */

  /**
   * Apply an ActiveEffectChange to this field.
   * @param {*} value                  The field's current value.
   * @param {DataModel} model          The model instance.
   * @param {EffectChangeData} change  The change to apply.
   * @returns {*}                      The updated value.
   */
  applyChange(value, model, change) {
    const delta = this._castChangeDelta(change.value);
    switch ( change.mode ) {
      case CONST.ACTIVE_EFFECT_MODES.ADD: return this._applyChangeAdd(value, delta, model, change);
      case CONST.ACTIVE_EFFECT_MODES.MULTIPLY: return this._applyChangeMultiply(value, delta, model, change);
      case CONST.ACTIVE_EFFECT_MODES.OVERRIDE: return this._applyChangeOverride(value, delta, model, change);
      case CONST.ACTIVE_EFFECT_MODES.UPGRADE: return this._applyChangeUpgrade(value, delta, model, change);
      case CONST.ACTIVE_EFFECT_MODES.DOWNGRADE: return this._applyChangeDowngrade(value, delta, model, change);
    }
    return this._applyChangeCustom(value, delta, model, change);
  }

  /* -------------------------------------------- */

  /**
   * Cast a change delta into an appropriate type to be applied to this field.
   * @param {*} delta  The change delta.
   * @returns {*}
   * @internal
   */
  _castChangeDelta(delta) {
    return this._cast(delta);
  }

  /* -------------------------------------------- */

  /**
   * Apply an ADD change to this field.
   * @param {*} value                  The field's current value.
   * @param {*} delta                  The change delta.
   * @param {DataModel} model          The model instance.
   * @param {EffectChangeData} change  The original change data.
   * @returns {*}                      The updated value.
   * @protected
   */
  _applyChangeAdd(value, delta, model, change) {
    return value + delta;
  }

  /* -------------------------------------------- */

  /**
   * Apply a MULTIPLY change to this field.
   * @param {*} value                  The field's current value.
   * @param {*} delta                  The change delta.
   * @param {DataModel} model          The model instance.
   * @param {EffectChangeData} change  The original change data.
   * @returns {*}                      The updated value.
   * @protected
   */
  _applyChangeMultiply(value, delta, model, change) {}

  /* -------------------------------------------- */

  /**
   * Apply an OVERRIDE change to this field.
   * @param {*} value                  The field's current value.
   * @param {*} delta                  The change delta.
   * @param {DataModel} model          The model instance.
   * @param {EffectChangeData} change  The original change data.
   * @returns {*}                      The updated value.
   * @protected
   */
  _applyChangeOverride(value, delta, model, change) {
    return delta;
  }

  /* -------------------------------------------- */

  /**
   * Apply an UPGRADE change to this field.
   * @param {*} value                  The field's current value.
   * @param {*} delta                  The change delta.
   * @param {DataModel} model          The model instance.
   * @param {EffectChangeData} change  The original change data.
   * @returns {*}                      The updated value.
   * @protected
   */
  _applyChangeUpgrade(value, delta, model, change) {}

  /* -------------------------------------------- */

  /**
   * Apply a DOWNGRADE change to this field.
   * @param {*} value                  The field's current value.
   * @param {*} delta                  The change delta.
   * @param {DataModel} model          The model instance.
   * @param {EffectChangeData} change  The original change data.
   * @returns {*}                      The updated value.
   * @protected
   */
  _applyChangeDowngrade(value, delta, model, change) {}

  /* -------------------------------------------- */

  /**
   * Apply a CUSTOM change to this field.
   * @param {*} value                  The field's current value.
   * @param {*} delta                  The change delta.
   * @param {DataModel} model          The model instance.
   * @param {EffectChangeData} change  The original change data.
   * @returns {*}                      The updated value.
   * @protected
   */
  _applyChangeCustom(value, delta, model, change) {
    const preHook = foundry.utils.getProperty(model, change.key);
    Hooks.call("applyActiveEffect", model, change, value, delta, {});
    const postHook = foundry.utils.getProperty(model, change.key);
    if ( postHook !== preHook ) return postHook;
  }
}

/* -------------------------------------------- */
/*  Data Schema Field                           */
/* -------------------------------------------- */

/**
 * A special class of {@link foundry.data.fields.DataField} which defines a data schema.
 */
class SchemaField extends DataField {
  /**
   * @param {DataSchema} fields                 The contained field definitions
   * @param {DataFieldOptions} [options]        Options which configure the behavior of the field
   * @param {DataFieldContext} [context]        Additional context which describes the field
   */
  constructor(fields, options, context={}) {
    super(options, context);
    this.fields = this._initialize(fields);
  }

  /* -------------------------------------------- */

  /** @inheritdoc */
  static get _defaults() {
    return Object.assign(super._defaults, {required: true, nullable: false});
  }

  /** @override */
  static recursive = true;

  /* -------------------------------------------- */

  /**
   * The contained field definitions.
   * @type {DataSchema}
   */
  fields;

  /* -------------------------------------------- */

  /**
   * Initialize and validate the structure of the provided field definitions.
   * @param {DataSchema} fields     The provided field definitions
   * @returns {DataSchema}          The validated schema
   * @protected
   */
  _initialize(fields) {
    if ( getType(fields) !== "Object" ) {
      throw new Error("A DataSchema must be an object with string keys and DataField values.");
    }
    fields = {...fields};
    for ( const [name, field] of Object.entries(fields) ) {
      if ( name === "_source" ) throw new Error('"_source" is not a valid name for a field of a SchemaField.');
      if ( !(field instanceof DataField) ) {
        throw new Error(`The "${name}" field is not an instance of the DataField class.`);
      }
      if ( field.parent !== undefined ) {
        throw new Error(`The "${field.fieldPath}" field already belongs to some other parent and may not be reused.`);
      }
      field.name = name;
      field.parent = this;
    }
    return fields;
  }

  /* -------------------------------------------- */
  /*  Schema Iteration                            */
  /* -------------------------------------------- */

  /**
   * Iterate over a SchemaField by iterating over its fields.
   * @type {Iterable<DataField>}
   */
  *[Symbol.iterator]() {
    for ( const field of Object.values(this.fields) ) {
      yield field;
    }
  }

  /**
   * An array of field names which are present in the schema.
   * @returns {string[]}
   */
  keys() {
    return Object.keys(this.fields);
  }

  /**
   * An array of DataField instances which are present in the schema.
   * @returns {DataField[]}
   */
  values() {
    return Object.values(this.fields);
  }

  /**
   * An array of [name, DataField] tuples which define the schema.
   * @returns {Array<[string, DataField]>}
   */
  entries() {
    return Object.entries(this.fields);
  }

  /**
   * Test whether a certain field name belongs to this schema definition.
   * @param {string} fieldName    The field name
   * @returns {boolean}           Does the named field exist in this schema?
   */
  has(fieldName) {
    return Object.hasOwn(this.fields, fieldName);
  }

  /**
   * Get a DataField instance from the schema by name.
   * @param {string} fieldName    The field name
   * @returns {DataField|void}    The DataField instance or undefined
   */
  get(fieldName) {
    if ( !this.has(fieldName) ) return;
    return this.fields[fieldName];
  }

  /**
   * Traverse the schema, obtaining the DataField definition for a particular field.
   * @param {string[]|string} fieldName       A field path like ["abilities", "strength"] or "abilities.strength"
   * @returns {DataField|undefined}           The corresponding DataField definition for that field, or undefined
   */
  getField(fieldName) {
    let path;
    if ( typeof fieldName === "string" ) path = fieldName.split(".");
    else if ( Array.isArray(fieldName) ) path = fieldName.slice();
    else throw new Error("A field path must be an array of strings or a dot-delimited string");
    return this._getField(path);
  }

  /** @override */
  _getField(path) {
    if ( !path.length ) return this;
    const field = this.get(path.shift());
    return field?._getField(path);
  }

  /* -------------------------------------------- */
  /*  Data Field Methods                          */
  /* -------------------------------------------- */

  /** @override */
  getInitialValue(data) {
    const initial = super.getInitialValue(data);
    if ( this.required && (initial === undefined) ) return this._cleanType({});
    return initial;
  }

  /* -------------------------------------------- */

  /** @override */
  _cast(value) {
    return getType(value) === "Object" ? value : {};
  }

  /* -------------------------------------------- */

  /** @inheritdoc */
  _cleanType(data, options={}) {
    options.source = options.source || data;

    // Clean each field which belongs to the schema
    for ( const [name, field] of this.entries() ) {
      const k = `==${name}`;
      if ( k in data ) {
        data[k] = field.clean(applySpecialKeys(data[k]), {...options, partial: false});
      } else if ( !options.partial || (name in data) ) {
        data[name] = field.clean(data[name], options);
      }
    }

    // Delete any keys which do not belong to the schema
    for ( const k in data ) {
      if ( this.has(k) ) continue;
      if ( isDeletionKey(k) ) {
        const key = k.slice(2);
        if ( this.has(key) ) continue;
      }
      delete data[k];
    }
    return data;
  }

  /* -------------------------------------------- */

  /** @override */
  initialize(value, model, options={}) {
    if ( !value ) return value;
    const data = {};
    for ( const [name, field] of this.entries() ) {
      const v = field.initialize(value[name], model, options);

      // Readonly fields
      if ( field.readonly ) {
        Object.defineProperty(data, name, {value: v, writable: false});
      }

      // Getter fields
      else if ( (typeof v === "function") && !v.prototype ) {
        Object.defineProperty(data, name, {get: v, set() {}, configurable: true});
      }

      // Writable fields
      else data[name] = v;
    }
    return data;
  }

  /* -------------------------------------------- */

  /**
   * The SchemaField#update method plays a special role of recursively dispatching DataField#update operations to the
   * constituent fields within the schema.
   * @override
   */
  _updateDiff(source, key, value, difference, options) {

    // * -> undefined, or * -> null
    if ( (value === undefined) || (value === null) || ((options.recursive === false) && (key !== "_source")) ) {
      value = applySpecialKeys(value);
      if ( options.recursive === false ) value = this.clean(value);
      super._updateDiff(source, key, value, difference, options);
      return;
    }

    // Pass type to fields
    const hasTypeData = this.fields.system instanceof TypeDataField;
    if ( hasTypeData && (("==type" in value) || ("-=type" in value)) ) {
      throw new Error("The type of a Document cannot be updated with ==type or -=type");
    }

    // {} -> {}, undefined -> {}, or null -> {}
    source[key] ||= {};
    source = source[key];
    const schemaDiff = difference[key] = {};
    for ( const [k, v] of Object.entries(value) ) {
      let name = k;
      const specialKey = isDeletionKey(k);
      if ( specialKey ) name = k.slice(2);

      // Require the changed field to exist
      const field = this.get(name);
      if ( !field ) continue;

      // Special operations for deletion or forced replacement
      if ( specialKey ) {
        if ( k[0] === "-" ) {
          if ( v !== null ) throw new Error("Removing a key using the -= deletion syntax requires the value of that"
            + " deletion key to be null, for example {-=key: null}");
          if ( name in source ) {
            schemaDiff[k] = v;
            delete source[name];
          }
        }
        else if ( k[0] === "=" ) schemaDiff[k] = source[name] = applySpecialKeys(v);
        continue;
      }

      // Perform field-specific update
      field._updateDiff(source, k, v, schemaDiff, options);
    }

    if ( hasTypeData && ("type" in schemaDiff) && !(("==system" in value) || (("system" in value) && (options.recursive === false))) ) {
      throw new Error("The type of a Document can be changed only if the system field is force-replaced (==) or updated with {recursive: false}");
    }

    // No updates applied
    if ( isEmpty(schemaDiff) ) delete difference[key];
  }

  /* -------------------------------------------- */

  /** @override */
  _updateCommit(source, key, value, diff, options) {
    const s = source[key];

    // Special Cases: * -> undefined, * -> null, undefined -> *, null -> *
    if ( !s || !value ) {
      source[key] = value;
      return;
    }

    // Clear system field if the type changed
    const hasTypeData = this.fields.system instanceof TypeDataField;
    if ( hasTypeData && ("type" in diff) ) s.system = undefined;

    // Update fields in source which changed in the diff
    for ( let [k, d] of Object.entries(diff) ) {
      k = isDeletionKey(k) ? k.slice(2) : k;
      const field = this.get(k);
      if ( !field ) continue;
      field._updateCommit(s, k, value[k], d, options);
    }
  }

  /* -------------------------------------------- */

  /** @override */
  _validateType(data, options={}) {
    if ( !(data instanceof Object) ) throw new Error("must be an object");
    options.source = options.source || data;
    const schemaFailure = new DataModelValidationFailure();
    for ( const [name, field] of this.entries() ) {
      for ( const prefix of ["", "-=", "=="] ) {
        const key = prefix + name;
        if ( (prefix || options.partial) && !(key in data) ) continue;

        // Validate the field's current value
        let value = data[key];
        if ( prefix === "-=" ) {
          if ( value !== null ) throw new Error("Removing a key using the -= deletion syntax requires the value of that"
            + " deletion key to be null, for example {-=key: null}");
          value = undefined;
        }
        const failure = field.validate(value, options);

        // Failure may be permitted if fallback replacement is allowed
        if ( failure ) {
          schemaFailure.fields[key] = failure;

          // If the field internally applied fallback logic
          if ( !failure.unresolved ) continue;

          // If fallback is allowed at the schema level
          if ( options.fallback && !prefix ) {
            const initial = field.getInitialValue(options.source);
            const fallbackFailure = field.validate(initial, {fallback: false, source: options.source});
            if ( fallbackFailure ) failure.unresolved = schemaFailure.unresolved = true;
            else {
              data[name] = failure.fallback = initial;
              failure.unresolved = false;
            }
          }

          // Otherwise the field-level failure is unresolved
          else failure.unresolved = schemaFailure.unresolved = true;
        }
      }
    }
    if ( !isEmpty(schemaFailure.fields) ) return schemaFailure;
  }

  /* ---------------------------------------- */

  /** @override */
  _validateModel(changes, options={}) {
    options.source = options.source || changes;
    if ( !changes ) return;
    for ( const [name, field] of this.entries() ) {
      const change = changes[name];  // May be nullish
      if ( change && field.constructor.recursive ) field._validateModel(change, options);
    }
  }

  /* -------------------------------------------- */

  /** @override */
  toObject(value) {
    if ( (value === undefined) || (value === null) ) return value;
    const data = {};
    for ( const [name, field] of this.entries() ) {
      data[name] = field.toObject(value[name]);
    }
    return data;
  }

  /* -------------------------------------------- */

  /** @override */
  apply(fn, data={}, options={}) {

    // Apply to this SchemaField
    const thisFn = typeof fn === "string" ? this[fn] : fn;
    thisFn?.call(this, data, options);
    if ( !data || (typeof data !== "object") ) return data; // Do not recurse for non-object types or null

    // Recursively apply to inner fields
    const results = {};
    for ( const [key, field] of this.entries() ) {
      if ( options.partial && !(key in data) ) continue;
      const r = field.apply(fn, data[key], options);
      if ( !options.filter || !isEmpty(r) ) results[key] = r;
    }
    return results;
  }

  /* -------------------------------------------- */

  /** @override */
  _addTypes(source, changes, options={}) {
    if ( getType(source) !== "Object" ) return;
    if ( getType(changes) !== "Object" ) return;
    options.source ??= source;
    options.changes ??= changes;
    const hasTypeData = this.fields.system instanceof TypeDataField;
    if ( hasTypeData ) {
      if ( "type" in changes ) changes.type ??= this.fields.type.getInitialValue(source);
      else changes.type = source.type;
    }
    for ( const key in changes ) {
      const field = this.get(key);
      field?._addTypes(source[key], changes[key], options);
    }
  }

  /* -------------------------------------------- */

  /**
   * Migrate this field's candidate source data.
   * @param {object} sourceData   Candidate source data of the root model
   * @param {any} fieldData       The value of this field within the source data
   */
  migrateSource(sourceData, fieldData) {
    if ( getType(fieldData) !== "Object" ) return;
    for ( const key in fieldData ) {
      const isDeletion = isDeletionKey(key);
      if ( isDeletion && (key[0] === "-") ) continue;
      const field = this.get(isDeletion ? key.slice(2) : key);
      if ( !field || !(field.migrateSource instanceof Function) ) continue;
      field.migrateSource(sourceData, fieldData[key]);
    }
  }
}

/* -------------------------------------------- */
/*  Basic Field Types                           */
/* -------------------------------------------- */

/**
 * A subclass of {@link foundry.data.fields.DataField} which deals with boolean-typed data.
 */
class BooleanField extends DataField {

  /** @inheritdoc */
  static get _defaults() {
    return Object.assign(super._defaults, {required: true, nullable: false, initial: false});
  }

  /** @override */
  _cast(value) {
    if ( typeof value === "string" ) return value === "true";
    if ( typeof value === "object" ) return false;
    return Boolean(value);
  }

  /** @override */
  _validateType(value) {
    if (typeof value !== "boolean") throw new Error("must be a boolean");
  }

  /** @override */
  _toInput(config) {
    config.value ??= this.initial;
    return foundry.applications.fields.createCheckboxInput(config);
  }

  /* -------------------------------------------- */
  /*  Active Effect Integration                   */
  /* -------------------------------------------- */

  /** @override */
  _applyChangeAdd(value, delta, model, change) {
    return value || delta;
  }

  /** @override */
  _applyChangeMultiply(value, delta, model, change) {
    return value && delta;
  }

  /** @override */
  _applyChangeUpgrade(value, delta, model, change) {
    return delta > value ? delta : value;
  }

  /** @override */
  _applyChangeDowngrade(value, delta, model, change) {
    return delta < value ? delta : value;
  }
}

/* ---------------------------------------- */

/**
 * A subclass of {@link foundry.data.fields.DataField} which deals with number-typed data.
 *
 * @property {number} min                 A minimum allowed value
 * @property {number} max                 A maximum allowed value
 * @property {number} step                A permitted step size
 * @property {boolean} integer=false      Must the number be an integer?
 * @property {boolean} positive=false     Must the number be positive?
 * @property {number[]|object|Function} [choices] An array of values or an object of values/labels which represent
 *                                        allowed choices for the field. A function may be provided which dynamically
 *                                        returns the array of choices.
 */
class NumberField extends DataField {
  /**
   * @param {NumberFieldOptions} options  Options which configure the behavior of the field
   * @param {DataFieldContext} [context]  Additional context which describes the field
   */
  constructor(options={}, context={}) {
    super(options, context);
    // If choices are provided, the field should not be null by default
    if ( this.choices ) {
      this.nullable = options.nullable ?? false;
    }
    if ( Number.isFinite(this.min) && Number.isFinite(this.max) && (this.min > this.max) ) {
      throw new Error("NumberField minimum constraint cannot exceed its maximum constraint");
    }
  }

  /** @inheritdoc */
  static get _defaults() {
    return Object.assign(super._defaults, {
      nullable: true,
      min: undefined,
      max: undefined,
      step: undefined,
      integer: false,
      positive: false,
      choices: undefined
    });
  }

  /* -------------------------------------------- */

  /** @override */
  _cast(value) {
    return this.nullable && (value === "") ? null : Number(value);
  }

  /** @inheritdoc */
  _cleanType(value, options) {
    value = super._cleanType(value, options);
    if ( typeof value !== "number" ) return value;
    if ( this.integer ) value = Math.round(value);
    if ( Number.isFinite(this.step) ) {
      let base = 0;
      if ( Number.isFinite(this.min) ) value = Math.max(value, base = this.min);
      value = value.toNearest(this.step, "round", base);
      if ( Number.isFinite(this.max) ) value = Math.min(value, this.max.toNearest(this.step, "floor", base));
    } else {
      if ( Number.isFinite(this.min) ) value = Math.max(value, this.min);
      if ( Number.isFinite(this.max) ) value = Math.min(value, this.max);
    }
    return value;
  }

  /** @override */
  _validateType(value) {
    if ( typeof value !== "number" ) throw new Error("must be a number");
    if ( this.positive && (value <= 0) ) throw new Error("must be a positive number");
    if ( Number.isFinite(this.min) && (value < this.min) ) throw new Error(`must be at least ${this.min}`);
    if ( Number.isFinite(this.max) && (value > this.max) ) throw new Error(`must be at most ${this.max}`);
    if ( Number.isFinite(this.step) && (value.toNearest(this.step, "round", Number.isFinite(this.min) ? this.min : 0) !== value) ) {
      if ( Number.isFinite(this.min) && (this.min !== 0) ) throw new Error(`must be an increment of ${this.step} after subtracting ${this.min}`);
      else throw new Error(`must be an increment of ${this.step}`);
    }
    if ( this.choices && !this.#isValidChoice(value) ) throw new Error(`${value} is not a valid choice`);
    if ( this.integer ) {
      if ( !Number.isInteger(value) ) throw new Error("must be an integer");
    }
    else if ( !Number.isFinite(value) ) throw new Error("must be a finite number");
  }

  /**
   * Test whether a provided value is a valid choice from the allowed choice set
   * @param {number} value      The provided value
   * @returns {boolean}         Is the choice valid?
   */
  #isValidChoice(value) {
    let choices = this.choices;
    if ( choices instanceof Function ) choices = choices();
    if ( choices instanceof Array ) return choices.includes(value);
    return String(value) in choices;
  }

  /* -------------------------------------------- */
  /*  Form Field Integration                      */
  /* -------------------------------------------- */

  /** @override */
  _toInput(config) {
    config.min ??= this.min;
    config.max ??= this.max;
    config.step ??= this.step;
    if ( config.value === undefined ) config.value = this.getInitialValue({});
    if ( this.integer ) {
      if ( Number.isNumeric(config.value) ) config.value = Math.round(config.value);
      config.step ??= 1;
    }
    if ( this.positive && Number.isFinite(config.step) ) config.min ??= config.step;

    // Number Select
    config.choices ??= this.choices;
    StringField._prepareChoiceConfig(config);
    if ( config.options ) {
      config.dataset ||= {};
      config.dataset.dtype = "Number";
      return foundry.applications.fields.createSelectInput(config);
    }

    // Range Slider
    if ( ["min", "max", "step"].every(k => config[k] !== undefined) && (config.type !== "number") ) {
      return foundry.applications.elements.HTMLRangePickerElement.create(config);
    }

    // Number Input
    return foundry.applications.fields.createNumberInput(config);
  }

  /* -------------------------------------------- */
  /*  Active Effect Integration                   */
  /* -------------------------------------------- */

  /** @override */
  _applyChangeMultiply(value, delta, model, change) {
    return value * delta;
  }

  /** @override */
  _applyChangeUpgrade(value, delta, model, change) {
    return delta > value ? delta : value;
  }

  /** @override */
  _applyChangeDowngrade(value, delta, model, change) {
    return delta < value ? delta : value;
  }
}

/* ---------------------------------------- */

/**
 * A subclass of {@link foundry.data.fields.DataField} which deals with string-typed data.
 */
class StringField extends DataField {
  /**
   * @param {StringFieldOptions} [options]  Options which configure the behavior of the field
   * @param {DataFieldContext} [context]    Additional context which describes the field
   */
  constructor(options={}, context={}) {
    super(options, context);

    // If choices are provided, the field should not be null or blank by default
    if ( this.choices ) {
      this.nullable = options.nullable ?? false;
      this.blank = options.blank ?? false;
    }
  }

  /** @inheritdoc */
  static get _defaults() {
    return Object.assign(super._defaults, {blank: true, trim: true, choices: undefined, textSearch: false});
  }

  /**
   * Is the string allowed to be blank (empty)?
   * @type {boolean}
   */
  blank = this.blank;

  /**
   * Should any provided string be trimmed as part of cleaning?
   * @type {boolean}
   */
  trim = this.trim;

  /**
   * An array of values or an object of values/labels which represent
   * allowed choices for the field. A function may be provided which dynamically
   * returns the array of choices.
   * @type {string[]|object|Function}
   */
  choices = this.choices;

  /**
   * Is this string field a target for text search?
   * @type {boolean}
   */
  textSearch = this.textSearch;

  /* -------------------------------------------- */

  /** @inheritdoc */
  clean(value, options) {
    if ( (typeof value === "string") && this.trim ) value = value.trim(); // Trim input strings
    return super.clean(value, options);
  }

  /* -------------------------------------------- */

  /** @override */
  getInitialValue(data) {
    const initial = super.getInitialValue(data);
    if ( this.blank && this.required && !initial ) return "";  // Prefer blank to null for required fields
    return initial;
  }

  /* -------------------------------------------- */

  /** @override */
  _cast(value) {
    return String(value);
  }

  /* -------------------------------------------- */

  /** @inheritdoc */
  _validateSpecial(value) {
    if ( value === "" ) {
      if ( this.blank ) return true;
      else throw new Error("may not be a blank string");
    }
    return super._validateSpecial(value);
  }

  /* -------------------------------------------- */

  /** @override */
  _validateType(value) {
    if ( typeof value !== "string" ) throw new Error("must be a string");
    else if ( this.choices ) {
      if ( this._isValidChoice(value) ) return true;
      else throw new Error(`${value} is not a valid choice`);
    }
  }

  /* -------------------------------------------- */

  /**
   * Test whether a provided value is a valid choice from the allowed choice set
   * @param {string} value      The provided value
   * @returns {boolean}         Is the choice valid?
   * @protected
   */
  _isValidChoice(value) {
    let choices = this.choices;
    if ( choices instanceof Function ) choices = choices();
    if ( choices instanceof Array ) return choices.includes(value);
    return String(value) in choices;
  }

  /* -------------------------------------------- */
  /*  Form Field Integration                      */
  /* -------------------------------------------- */

  /**
   * Prepare form input configuration to accept a limited choice set of options.
   * @param {FormInputConfig & Partial<ChoiceInputConfig>} [config]
   * @internal
   */
  static _prepareChoiceConfig(config) {
    if ( config.options || !("choices" in config) ) return;
    let choices;
    try {
      choices = typeof config.choices === "function" ? config.choices() : config.choices;
    } catch(error) {
      logger.error(error);
    }

    // Prepare options array - only accept arrays or records
    if ( (typeof choices === "object") && (choices !== null) ) {
      config.options = [];
      for ( const [value, entry] of Object.entries(choices) ) {
        const choice = {value, ...StringField.#getChoiceFromEntry(entry, config)};
        config.options.push(choice);
      }
    }

    // Remove consumed options
    delete config.choices;
    delete config.valueAttr;
    delete config.labelAttr;
  }

  /* -------------------------------------------- */

  /**
   * Convert a choice entry into a standardized FormSelectOption
   * @param {string|object} entry
   * @param {{labelAttr?: string; valueAttr?: string; localize?: boolean}} options
   * @returns {FormSelectOption}
   */
  static #getChoiceFromEntry(entry, {labelAttr="label", valueAttr, localize}) {
    const choice = {};
    if ( foundry.utils.getType(entry) === "Object" ) {
      if ( valueAttr && (valueAttr in entry) ) choice.value = entry[valueAttr];
      if ( labelAttr && (labelAttr in entry) ) choice.label = entry[labelAttr];
      for ( const k of ["group", "disabled", "rule"] ) {
        if ( k in entry ) choice[k] = entry[k];
      }
    }
    else choice.label = String(entry);
    if ( localize && choice.label ) choice.label = game.i18n.localize(choice.label);
    return choice;
  }

  /* -------------------------------------------- */

  /**
   * @param {FormInputConfig & StringFieldInputConfig} config
   * @override
   */
  _toInput(config) {
    if ( config.value === undefined ) config.value = this.getInitialValue({});
    config.choices ??= this.choices;

    // Choice Select
    StringField._prepareChoiceConfig(config);
    if ( config.options ) {
      if ( this.blank || this.nullable || !this.required ) config.blank ??= "";
      return foundry.applications.fields.createSelectInput(config);
    }

    // One of several options for element type
    switch ( config.elementType ?? "input" ) {
      case "input":
        return foundry.applications.fields.createTextInput(config);
      case "textarea":
        return foundry.applications.fields.createTextareaInput(config);
      case "file-picker":
        return foundry.applications.elements.HTMLFilePickerElement.create(config);
      case "prose-mirror":
        return foundry.applications.elements.HTMLProseMirrorElement.create(config);
      case "code-mirror":
        return foundry.applications.elements.HTMLCodeMirrorElement.create(config);
      default:
        throw new Error(`Unrecognized element type for StringField input: ${config.elementType}`);
    }
  }
}

/* ---------------------------------------- */

/**
 * A subclass of DataField which deals with object-typed data.
 */
class ObjectField extends DataField {

  /** @inheritdoc */
  static get _defaults() {
    return Object.assign(super._defaults, {required: true, nullable: false});
  }

  /* -------------------------------------------- */

  /** @override */
  getInitialValue(data) {
    const initial = super.getInitialValue(data);
    if ( this.required && (initial === undefined) ) return {};
    return initial;
  }

  /* -------------------------------------------- */

  /** @override */
  _cast(value) {
    if ( value.toObject instanceof Function ) value = value.toObject();
    return getType(value) === "Object" ? value : {};
  }

  /** @override */
  initialize(value, model, options={}) {
    if ( !value ) return value;
    return deepClone(value);
  }

  /* ---------------------------------------- */

  /** @override */
  _updateDiff(source, key, value, difference, options) {

    // {} -> {}, null -> {}, undefined -> {}
    if ( (getType(value) === "Object") && (options.recursive !== false) ) {
      if ( getType(source[key]) !== "Object" ) source[key] = {};
      const diff = diffObject(source[key], value, {deletionKeys: true});
      if ( isEmpty(diff) ) return;
      difference[key] = diff;
      mergeObject(source[key], value, {insertKeys: true, insertValues: true, performDeletions: true});
    }

    // {} -> null or {} -> undefined
    else super._updateDiff(source, key, applySpecialKeys(value), difference, options);
  }

  /* ---------------------------------------- */

  /** @inheritDoc */
  _updateCommit(source, key, value, diff, options) {
    const s = source[key];

    // Special Cases: * -> undefined, * -> null, undefined -> *, null -> *
    if ( !s || !value || Object.isSealed(s) ) {
      source[key] = value;
      return;
    }

    for ( const k of Object.keys(s) ) {
      if ( !(k in value) ) delete s[k];
    }
    Object.assign(s, value);
  }

  /* ---------------------------------------- */

  /** @override */
  toObject(value) {
    return deepClone(value);
  }

  /** @override */
  _validateType(value, options={}) {
    if ( getType(value) !== "Object" ) throw new Error("must be an object");
  }
}

/* -------------------------------------------- */

/**
 * A subclass of ObjectField that represents a mapping of keys to the provided DataField type.
 */
class TypedObjectField extends ObjectField {
  /**
   * @param {DataField} element             The value type of each entry in this object.
   * @param {DataFieldOptions} [options]    Options which configure the behavior of the field.
   * @param {DataFieldContext} [context]    Additional context which describes the field
   */
  constructor(element, options, context) {
    super(options, context);
    if ( !(element instanceof DataField) ) throw new Error("The element must be a DataField");
    if ( element.parent !== undefined ) throw new Error("The element DataField already has a parent");
    element.name ||= "element";
    element.parent = this;
    this.element = element;
  }

  /* -------------------------------------------- */

  /**
   * The value type of each entry in this object.
   * @type {DataField}
   */
  element;

  /* -------------------------------------------- */

  /** @override */
  static recursive = true;

  /* -------------------------------------------- */

  /** @inheritDoc */
  static get _defaults() {
    return mergeObject(super._defaults, {validateKey: undefined});
  }

  /* -------------------------------------------- */

  /** @override */
  _cleanType(data, options) {
    options.source = options.source || data;
    for ( const key in data ) {
      const isDeletion = isDeletionKey(key);
      const k = isDeletion ? key.slice(2) : key;
      let valid;
      try {
        valid = this.validateKey?.(k);
      } catch {
        valid = false;
      }
      if ( valid === false ) {
        delete data[key];
        continue;
      }
      if ( isDeletion && (key[0] === "-") ) continue;
      data[key] = this.element.clean(data[key], options);
    }
    return data;
  }

  /* -------------------------------------------- */

  /** @override */
  _validateType(data, options={}) {
    if ( foundry.utils.getType(data) !== "Object" ) throw new Error("must be an object");
    options.source = options.source || data;
    const mappingFailure = new DataModelValidationFailure();
    for ( const key in data ) {
      if ( key.startsWith("-=") ) continue;

      // Validate the field's current value
      const value = data[key];
      const failure = this.element.validate(value, options);

      // Failure may be permitted if fallback replacement is allowed
      if ( failure ) {
        mappingFailure.fields[key] = failure;

        // If the field internally applied fallback logic
        if ( !failure.unresolved ) continue;

        // If fallback is allowed at the object level
        if ( options.fallback && !key.startsWith("==") ) {
          const initial = this.element.getInitialValue(options.source);
          if ( this.element.validate(initial, {source: options.source}) === undefined ) {  // Ensure initial is valid
            data[key] = initial;
            failure.fallback = initial;
            failure.unresolved = false;
          }
          else failure.unresolved = mappingFailure.unresolved = true;
        }

        // Otherwise the field-level failure is unresolved
        else failure.unresolved = mappingFailure.unresolved = true;
      }
    }
    if ( !foundry.utils.isEmpty(mappingFailure.fields) ) return mappingFailure;
  }

  /* -------------------------------------------- */

  /** @override */
  _validateModel(changes, options={}) {
    options.source = options.source || changes;
    if ( !changes ) return;
    for ( const key in changes ) {
      const change = changes[key];  // May be nullish
      if ( change && this.element.constructor.recursive ) this.element._validateModel(change, options);
    }
  }

  /* -------------------------------------------- */

  /** @override */
  initialize(value, model, options={}) {
    const object = {};
    for ( const key in value ) object[key] = this.element.initialize(value[key], model, options);
    return object;
  }

  /* -------------------------------------------- */

  /** @override */
  _updateDiff(source, key, value, difference, options) {

    // * -> undefined, or * -> null
    if ( (value === undefined) || (value === null) || (options.recursive === false) ) {
      super._updateDiff(source, key, value, difference, options);
      return;
    }

    // {} -> {}, undefined -> {}, or null -> {}
    source[key] ||= {};
    value ||= {};
    source = source[key];
    const schemaDiff = difference[key] = {};
    for ( const [k, v] of Object.entries(value) ) {
      let name = k;
      const specialKey = isDeletionKey(k);
      if ( specialKey ) name = k.slice(2);

      // Special operations for deletion or forced replacement
      if ( specialKey ) {
        if ( k[0] === "-" ) {
          if ( v !== null ) throw new Error("Removing a key using the -= deletion syntax requires the value of that"
            + " deletion key to be null, for example {-=key: null}");
          if ( name in source ) {
            schemaDiff[k] = v;
            delete source[name];
          }
        }
        else if ( k[0] === "=" ) schemaDiff[k] = source[name] = applySpecialKeys(v);
        continue;
      }

      // Perform type-specific update
      this.element._updateDiff(source, k, v, schemaDiff, options);
    }

    // No updates applied
    if ( isEmpty(schemaDiff) ) delete difference[key];
  }

  /* -------------------------------------------- */

  /** @override */
  _updateCommit(source, key, value, diff, options) {
    const s = source[key];

    // Special Cases: * -> undefined, * -> null, undefined -> *, null -> *
    if ( !s || !value || Object.isSealed(s) ) {
      source[key] = value;
      return;
    }

    // Remove keys which no longer exist in the new value
    for ( const k of Object.keys(s) ) {
      if ( !(k in value) ) delete s[k];
    }

    // Update fields in source which changed in the diff
    for ( let [k, d] of Object.entries(diff) ) {
      if ( isDeletionKey(k) ) {
        if ( k[0] === "-" ) continue;
        k = k.slice(2);
      }
      this.element._updateCommit(s, k, value[k], d, options);
    }
  }

  /* -------------------------------------------- */

  /** @override */
  toObject(value) {
    if ( (value === undefined) || (value === null) ) return value;
    const object = {};
    for ( const key in value ) object[key] = this.element.toObject(value[key]);
    return object;
  }

  /* -------------------------------------------- */

  /** @override */
  apply(fn, data={}, options={}) {

    // Apply to this TypedObjectField
    const thisFn = typeof fn === "string" ? this[fn] : fn;
    thisFn?.call(this, data, options);

    // Recursively apply to inner fields
    const results = {};
    for ( const key in data ) {
      const r = this.element.apply(fn, data[key], options);
      if ( !options.filter || !isEmpty(r) ) results[key] = r;
    }
    return results;
  }

  /* -------------------------------------------- */

  /** @override */
  _addTypes(source, changes, options={}) {
    if ( (getType(source) !== "Object") || (getType(changes) !== "Object") ) return;
    for ( const key in changes ) this.element._addTypes(source[key], changes[key], options);
  }

  /* -------------------------------------------- */

  /** @override */
  _getField(path) {
    if ( path.length === 0 ) return this;
    if ( path.shift() !== this.element.name ) return undefined;
    return this.element._getField(path);
  }

  /* -------------------------------------------- */

  /**
   * Migrate this field's candidate source data.
   * @param {object} sourceData   Candidate source data of the root model
   * @param {any} fieldData       The value of this field within the source data
   */
  migrateSource(sourceData, fieldData) {
    if ( !(this.element.migrateSource instanceof Function) ) return;
    if ( getType(fieldData) !== "Object" ) return;
    for ( const key in fieldData ) {
      if ( key.startsWith("-=") ) continue;
      this.element.migrateSource(sourceData, fieldData[key]);
    }
  }
}

/* -------------------------------------------- */

/**
 * A subclass of {@link foundry.data.fields.DataField} which deals with array-typed data.
 * @template [ElementType=DataField]
 * @property {number} min     The minimum number of elements.
 * @property {number} max     The maximum number of elements.
 */
class ArrayField extends DataField {
  /**
   * @param {ElementType} element          The type of element contained in the Array
   * @param {ArrayFieldOptions} [options]  Options which configure the behavior of the field
   * @param {DataFieldContext} [context]   Additional context which describes the field
   */
  constructor(element, options={}, context={}) {
    super(options, context);
    this.element = this.constructor._validateElementType(element);
    if ( this.element instanceof DataField ) {
      this.element.name ||= "element";
      this.element.parent = this;
    }
    if ( this.min > this.max ) throw new Error("ArrayField minimum length cannot exceed maximum length");
  }

  /* ---------------------------------------- */

  /**
   * The data type of each element in this array
   * @type {ElementType}
   */
  element;

  /* ---------------------------------------- */

  /** @inheritdoc */
  static get _defaults() {
    return Object.assign(super._defaults, {
      required: true,
      nullable: false,
      empty: true,
      exact: undefined,
      min: 0,
      max: Infinity
    });
  }

  /* ---------------------------------------- */

  /** @override */
  static recursive = true;

  /* ---------------------------------------- */

  /**
   * Validate the contained element type of the ArrayField
   * @param {*} element        The type of Array element
   * @returns {ElementType}    The validated element type
   * @throws                   An error if the element is not a valid type
   * @protected
   */
  static _validateElementType(element) {
    if ( !(element instanceof DataField) ) {
      throw new Error(`${this.name} must have a DataField as its contained element`);
    }
    if ( element.parent !== undefined ) throw new Error("The element DataField already has a parent");
    return element;
  }

  /* -------------------------------------------- */

  /** @override */
  getInitialValue(data) {
    const initial = super.getInitialValue(data);
    if ( this.required && (initial === undefined) ) return [];
    return initial;
  }

  /* ---------------------------------------- */

  /** @override */
  _validateModel(changes, options) {
    if ( !this.element.constructor.recursive ) return;
    for ( const element of changes ) {
      this.element._validateModel(element, options);
    }
  }

  /* ---------------------------------------- */

  /** @override */
  _cast(value) {
    const t = getType(value);
    if ( t === "Object" ) {
      const arr = [];
      for ( const [k, v] of Object.entries(value) ) {
        const i = Number(k);
        if ( Number.isInteger(i) && (i >= 0) ) arr[i] = v;
      }
      return arr;
    }
    else if ( t === "Set" ) return Array.from(value);
    return value instanceof Array ? value : [value];
  }

  /** @override */
  _cleanType(value, options) {
    // Force partial as false for array cleaning. Arrays are updated by replacing the entire array, so partial data
    // must be initialized.
    return value.map(v => this.element.clean(v, { ...options, partial: false }));
  }

  /** @override */
  _validateType(value, options={}) {
    if ( !(value instanceof Array) ) throw new Error("must be an Array");
    if ( value.length < this.min ) throw new Error(`cannot have fewer than ${this.min} elements`);
    if ( value.length > this.max ) throw new Error(`cannot have more than ${this.max} elements`);
    return this._validateElements(value, options);
  }

  /**
   * Validate every element of the ArrayField
   * @param {Array} value                         The array to validate
   * @param {DataFieldValidationOptions} options  Validation options
   * @returns {DataModelValidationFailure|void}   A validation failure if any of the elements failed validation,
   *                                              otherwise void.
   * @protected
   */
  _validateElements(value, options) {
    const arrayFailure = new DataModelValidationFailure();
    for ( let i=0; i<value.length; i++ ) {
      // Force partial as false for array validation. Arrays are updated by replacing the entire array, so there cannot
      // be partial data in the elements.
      const failure = this._validateElement(value[i], { ...options, partial: false });
      if ( failure ) {
        arrayFailure.elements.push({id: i, failure});
        arrayFailure.unresolved ||= failure.unresolved;
      }
    }
    if ( arrayFailure.elements.length ) return arrayFailure;
  }

  /**
   * Validate a single element of the ArrayField.
   * @param {*} value                       The value of the array element
   * @param {DataFieldValidationOptions} options  Validation options
   * @returns {DataModelValidationFailure}  A validation failure if the element failed validation
   * @protected
   */
  _validateElement(value, options) {
    return this.element.validate(value, options);
  }

  /** @override */
  initialize(value, model, options={}) {
    if ( !value ) return value;
    return value.map(v => this.element.initialize(v, model, options));
  }

  /* ---------------------------------------- */

  /** @override */
  _updateDiff(source, key, value, difference, options) {
    const current = source[key];
    value = applySpecialKeys(value);
    if ( (value === current) || value?.equals(current) ) return;
    source[key] = value;
    difference[key] = deepClone(value);
  }

  /* ---------------------------------------- */

  /**
   * Commit array field changes by replacing array contents while preserving the array reference itself.
   * @override
   */
  _updateCommit(source, key, value, diff, options) {
    const s = source[key];

    // Special Cases: * -> undefined, * -> null, undefined -> *, null -> *
    if ( !s || !value ) {
      source[key] = value;
      return;
    }

    s.length = 0;
    s.push(...value);
  }

  /* ---------------------------------------- */

  /** @override */
  toObject(value) {
    if ( !value ) return value;
    return value.map(v => this.element.toObject(v));
  }

  /** @override */
  apply(fn, value=[], options={}) {

    // Apply to this ArrayField
    const thisFn = typeof fn === "string" ? this[fn] : fn;
    thisFn?.call(this, value, options);
    if ( !Array.isArray(value) ) return value; // Do not recurse for non-array types

    // Recursively apply to array elements
    const results = [];
    if ( !value.length && options.initializeArrays ) value = [undefined];
    for ( const v of value ) {
      const r = this.element.apply(fn, v, options);
      if ( !options.filter || !isEmpty(r) ) results.push(r);
    }
    return results;
  }

  /** @override */
  _getField(path) {
    if ( path.length === 0 ) return this;
    if ( path.shift() !== this.element.name ) return undefined;
    return this.element._getField(path);
  }

  /**
   * Migrate this field's candidate source data.
   * @param {object} sourceData   Candidate source data of the root model
   * @param {any} fieldData       The value of this field within the source data
   */
  migrateSource(sourceData, fieldData) {
    if ( !(this.element.migrateSource instanceof Function) ) return;
    if ( getType(fieldData) !== "Array" ) return;
    for ( const entry of fieldData ) this.element.migrateSource(sourceData, entry);
  }

  /* -------------------------------------------- */
  /*  Active Effect Integration                   */
  /* -------------------------------------------- */

  /** @override */
  _castChangeDelta(raw) {
    let delta;
    try {
      delta = JSON.parse(raw);
      delta = Array.isArray(delta) ? delta : [delta];
    } catch(_err) {
      delta = [raw];
    }
    return delta.map(value => this.element._castChangeDelta(value));
  }

  /** @override */
  _applyChangeAdd(value, delta, model, change) {
    value.push(...delta);
    return value;
  }
}

/* -------------------------------------------- */
/*  Specialized Field Types                     */
/* -------------------------------------------- */

/**
 * A subclass of {@link foundry.data.fields.ArrayField} which supports a set of contained elements.
 * Elements in this set are treated as fungible and may be represented in any order or discarded if invalid.
 */
class SetField extends ArrayField {

  /** @override */
  _validateElements(value, options) {
    const setFailure = new DataModelValidationFailure();
    for ( let i=value.length-1; i>=0; i-- ) {  // Iterate backwards so we can splice as we go
      const failure = this._validateElement(value[i], options);
      if ( failure ) {
        setFailure.elements.unshift({id: i, failure});

        // The failure may have been internally resolved by fallback logic
        if ( !failure.unresolved && failure.fallback ) continue;

        // If fallback is allowed, remove invalid elements from the set
        if ( options.fallback ) {
          value.splice(i, 1);
          failure.dropped = true;
        }

        // Otherwise the set failure is unresolved
        else setFailure.unresolved = true;
      }
    }

    // Return a record of any failed set elements
    if ( setFailure.elements.length ) {
      if ( options.fallback && !setFailure.unresolved ) setFailure.fallback = value;
      return setFailure;
    }
  }

  /** @override */
  initialize(value, model, options={}) {
    if ( !value ) return value;
    return new Set(super.initialize(value, model, options));
  }

  /** @override */
  toObject(value) {
    if ( !value ) return value;
    return Array.from(value).map(v => this.element.toObject(v));
  }

  /* -------------------------------------------- */
  /*  Form Field Integration                      */
  /* -------------------------------------------- */

  /** @override */
  _toInput(config) {
    const element = this.element;

    // Document UUIDs
    if ( element instanceof DocumentUUIDField ) {
      Object.assign(config, {type: element.type, single: false});
      return foundry.applications.elements.HTMLDocumentTagsElement.create(config);
    }

    // Multi-Select Input
    if ( element.choices && !config.options ) {
      config.choices ??= element.choices;
      StringField._prepareChoiceConfig(config);
    }
    if ( config.options ) {
      if ( element instanceof NumberField ) mergeObject(config, {dataset: {dtype: "Number"}});
      return foundry.applications.fields.createMultiSelectInput(config);
    }

    // Arbitrary String Tags
    if ( element instanceof StringField ) return foundry.applications.elements.HTMLStringTagsElement.create(config);
    throw new Error(`SetField#toInput is not supported for a ${element.constructor.name} element type`);
  }

  /* -------------------------------------------- */
  /*  Active Effect Integration                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  _castChangeDelta(raw) {
    return new Set(super._castChangeDelta(raw));
  }

  /** @override */
  _applyChangeAdd(value, delta, model, change) {
    for ( const element of delta ) value.add(element);
    return value;
  }
}

/* ---------------------------------------- */

/**
 * A subclass of {@link foundry.data.fields.SchemaField} which embeds some other DataModel definition as an inner
 * object.
 */
class EmbeddedDataField extends SchemaField {
  /**
   * @param {typeof DataModel} model          The class of DataModel which should be embedded in this field
   * @param {DataFieldOptions} [options]      Options which configure the behavior of the field
   * @param {DataFieldContext} [context]      Additional context which describes the field
   */
  constructor(model, options={}, context={}) {
    if ( !isSubclass(model, foundry.abstract.DataModel) ) {
      throw new Error("An EmbeddedDataField must specify a DataModel class as its type");
    }

    // Create an independent copy of the model schema
    const fields = model.defineSchema();
    super(fields, options, context);

    /**
     * The base DataModel definition which is contained in this field.
     * @type {typeof DataModel}
     */
    this.model = model;
  }

  /** @inheritdoc */
  clean(value, options) {
    return super.clean(value, {...options, source: value});
  }

  /** @override */
  _cast(value) {
    if ( value.toObject instanceof Function ) value = value.toObject();
    return getType(value) === "Object" ? value : {};
  }

  /** @inheritdoc */
  validate(value, options) {
    return super.validate(value, {...options, source: value});
  }

  /** @override */
  initialize(value, model, options={}) {
    if ( !value ) return value;
    // FIXME it should be unnecessary to construct a new instance of the model every time we initialize.
    const m = new this.model(value, {parent: model, ...options});
    Object.defineProperty(m, "schema", {value: this});
    return m;
  }

  /** @override */
  toObject(value) {
    if ( !value ) return value;
    return value.toObject(false);
  }

  /** @override */
  migrateSource(sourceData, fieldData) {
    if ( getType(fieldData) !== "Object" ) return;
    this.model.migrateDataSafe(fieldData);
  }

  /** @override */
  _validateModel(changes, options) {
    this.model.validateJoint(changes);
  }
}

/* ---------------------------------------- */

/**
 * A subclass of {@link foundry.data.fields.ArrayField} which supports an embedded Document collection.
 * Invalid elements will be dropped from the collection during validation rather than failing for the field entirely.
 * @extends {ArrayField<typeof Document>}
 */
class EmbeddedCollectionField extends ArrayField {
  /**
   * @param {typeof Document} element     The type of Document which belongs to this embedded collection
   * @param {DataFieldOptions} [options]  Options which configure the behavior of the field
   * @param {DataFieldContext} [context]  Additional context which describes the field
   */
  constructor(element, options={}, context={}) {
    super(element, options, context);
    this.readonly = true; // Embedded collections are always immutable
  }

  /** @override */
  static _validateElementType(element) {
    if ( isSubclass(element, foundry.abstract.Document) ) return element;
    throw new Error("An EmbeddedCollectionField must specify a Document subclass as its type");
  }

  /**
   * The Collection implementation to use when initializing the collection.
   * @type {typeof EmbeddedCollection}
   */
  static get implementation() {
    return EmbeddedCollection;
  }

  /** @override */
  static hierarchical = true;

  /**
   * A reference to the DataModel subclass of the embedded document element
   * @type {typeof Document}
   */
  get model() {
    return this.element.implementation;
  }

  /**
   * The DataSchema of the contained Document model.
   * @type {SchemaField}
   */
  get schema() {
    return this.model.schema;
  }

  /** @inheritDoc */
  _cast(value) {
    if ( getType(value) !== "Map" ) return super._cast(value);
    const arr = [];
    for ( const [id, v] of value.entries() ) {
      if ( !("_id" in v) ) v._id = id;
      arr.push(v);
    }
    return super._cast(arr);
  }

  /* -------------------------------------------- */

  /** @override */
  _cleanType(value, options={}) {
    if ( options.recursive === false ) options = {...options, partial: false};
    return value.map(v => this._cleanElement(v, options));
  }

  /* -------------------------------------------- */

  /**
   * Clean data for an individual element in the collection.
   * @param {object} value      Unclean data for the candidate embedded record
   * @param {object} options    Options which control how data is cleaned
   * @returns {object}          Cleaned data for the candidate embedded record
   * @protected
   */
  _cleanElement(value, options={}) {
    if ( !options.partial ) value._id ||= randomID(16); // Should this be left to the server side?
    return this.schema.clean(value, {...options, source: value});
  }

  /* -------------------------------------------- */

  /** @override */
  _validateElements(value, options) {
    const collectionFailure = new DataModelValidationFailure();
    for ( const v of value ) {
      const failure = this.schema.validate(v, {...options, source: v});
      if ( failure && !options.dropInvalidEmbedded ) {
        collectionFailure.elements.push({id: v._id, name: v.name, failure});
        collectionFailure.unresolved ||= failure.unresolved;
      }
    }
    if ( collectionFailure.elements.length ) return collectionFailure;
  }

  /* -------------------------------------------- */

  /** @override */
  initialize(value, model, options={}) {
    const collection = model.collections[this.name];
    collection.initialize(options);
    return collection;
  }

  /* -------------------------------------------- */

  /**
   * Dry-run an update of an EmbeddedCollection, modifying the contents of the safe copy of the source data.
   * @override
   */
  _updateDiff(source, key, value, difference, options) {
    if ( !Array.isArray(value) ) return;

    // Non-recursive updates replace the entire array
    if ( options.recursive === false ) {
      value = applySpecialKeys(value);
      source[key] = value;
      difference[key] = deepClone(value);
      return;
    }

    // Otherwise create or diff individual array members
    const sourceIdMap = {};
    for ( const obj of source[key] ?? [] ) sourceIdMap[obj._id] = obj;
    const diffArray = difference[key] = [];
    for ( const v of value ) {

      // Get the diff for each existing record
      const existing = sourceIdMap[v._id];
      if ( existing ) {
        const elementDiff = {};
        const typeChanged = "type" in v;
        this.schema._addTypes(existing, v);
        this.schema._updateDiff({_source: existing}, "_source", v, elementDiff, options);
        const d = elementDiff._source || {};
        if ( !isEmpty(d) ) {
          d._id = v._id;
          diffArray.push(d);
        }
        if ( !typeChanged ) delete v.type;
      }

      // Create new records using cleaned data
      else {
        const created = this._cleanElement(applySpecialKeys(v), {partial: false});
        source[key].push(created);
        diffArray.push(created);
      }
    }
    if ( !diffArray.length ) delete difference[key];
  }

  /* -------------------------------------------- */

  /** @override */
  _updateCommit(source, key, value, diff, options) {
    const src = source[key];

    // Special Cases: * -> undefined, * -> null, undefined -> *, null -> *
    if ( !src || !value ) {
      source[key] = value;
      return;
    }

    // Map the existing source objects
    const existing = {};
    for ( const obj of src ) existing[obj._id] = obj;
    const changed = {};
    for ( const obj of diff ) changed[obj._id] = obj;

    // Reconstruct the source array, retaining object references
    src.length = 0;
    for ( const obj of value ) {
      const prior = existing[obj._id];
      if ( prior ) {
        const d = changed[obj._id];
        if ( d ) this.schema._updateCommit({_source: prior}, "_source", obj, d, options);
        src.push(prior);
      }
      else src.push(obj);
    }
  }

  /* -------------------------------------------- */

  /** @override */
  toObject(value) {
    return value.toObject(false);
  }

  /** @override */
  apply(fn, value=[], options={}) {

    // Include this field in the options since it's not introspectable from the SchemaField
    options = {...options, collection: this};

    // Apply to this EmbeddedCollectionField
    const thisFn = typeof fn === "string" ? this[fn] : fn;
    thisFn?.call(this, value, options);

    // Recursively apply to inner fields
    const results = [];
    if ( !value.length && options.initializeArrays ) value = [undefined];
    for ( const v of value ) {
      const r = this.schema.apply(fn, v, options);
      if ( !options.filter || !isEmpty(r) ) results.push(r);
    }
    return results;
  }

  /**
   * Migrate this field's candidate source data.
   * @param {object} sourceData   Candidate source data of the root model
   * @param {any} fieldData       The value of this field within the source data
   */
  migrateSource(sourceData, fieldData) {
    if ( !Array.isArray(fieldData) ) return;
    for ( const entry of fieldData ) {
      if ( getType(entry) !== "Object" ) continue;
      this.model.migrateDataSafe(entry);
    }
  }

  /* -------------------------------------------- */
  /*  Embedded Document Operations                */
  /* -------------------------------------------- */

  /**
   * Return the embedded document(s) as a Collection.
   * @param {Document} parent  The parent document.
   * @returns {DocumentCollection}
   */
  getCollection(parent) {
    return parent[this.name];
  }
}

/* -------------------------------------------- */

/**
 * A subclass of {@link foundry.data.fields.EmbeddedCollectionField} which manages a collection of delta objects
 * relative to another collection.
 */
class EmbeddedCollectionDeltaField extends EmbeddedCollectionField {
  /** @override */
  static get implementation() {
    return EmbeddedCollectionDelta;
  }

  /* -------------------------------------------- */

  /** @override */
  _cleanElement(value, options={}) {
    const schema = value._tombstone ? foundry.data.TombstoneData.schema : this.schema;
    if ( !value._tombstone && !options.partial ) value._id ||= randomID(16); // Should this be left to the server side?
    return schema.clean(value, {...options, source: value});
  }

  /* -------------------------------------------- */

  /** @override */
  _validateElements(value, options) {
    const collectionFailure = new DataModelValidationFailure();
    for ( const v of value ) {
      const validationOptions = {...options, source: v};
      const schema = v._tombstone ? foundry.data.TombstoneData.schema : this.schema;
      const failure = schema.validate(v, validationOptions);
      if ( failure && !options.dropInvalidEmbedded ) {
        collectionFailure.elements.push({id: v._id, name: v.name, failure});
        collectionFailure.unresolved ||= failure.unresolved;
      }
    }
    if ( collectionFailure.elements.length ) return collectionFailure;
  }
}

/* -------------------------------------------- */

/**
 * A subclass of {@link foundry.data.fields.EmbeddedDataField} which supports a single embedded Document.
 */
class EmbeddedDocumentField extends EmbeddedDataField {
  /**
   * @param {typeof Document} model       The type of Document which is embedded.
   * @param {DataFieldOptions} [options]  Options which configure the behavior of the field.
   * @param {DataFieldContext} [context]  Additional context which describes the field
   */
  constructor(model, options={}, context={}) {
    if ( !isSubclass(model, foundry.abstract.Document) ) {
      throw new Error("An EmbeddedDocumentField must specify a Document subclass as its type.");
    }
    super(model.implementation, options, context);
  }

  /** @inheritdoc */
  static get _defaults() {
    return Object.assign(super._defaults, {nullable: true});
  }

  /** @override */
  static hierarchical = true;

  /** @override */
  initialize(value, model, options={}) {
    if ( !value ) return value;
    if ( model[this.name] ) {
      model[this.name]._initialize(options);
      return model[this.name];
    }
    const m = new this.model(value, {...options, parent: model, parentCollection: this.name});
    Object.defineProperty(m, "schema", {value: this});
    return m;
  }

  /* -------------------------------------------- */
  /*  Embedded Document Operations                */
  /* -------------------------------------------- */

  /**
   * Return the embedded document(s) as a Collection.
   * @param {Document} parent  The parent document.
   * @returns {Collection<string, Document>}
   */
  getCollection(parent) {
    const collection = new SingletonEmbeddedCollection(this.name, parent, []);
    const doc = parent[this.name];
    if ( !doc ) return collection;
    collection.set(doc.id, doc);
    return collection;
  }
}

/* -------------------------------------------- */
/*  Special Field Types                         */
/* -------------------------------------------- */

/**
 * A subclass of {@link foundry.data.fields.StringField} which provides the primary _id for a Document.
 * The field may be initially null, but it must be non-null when it is saved to the database.
 */
class DocumentIdField extends StringField {

  /** @inheritdoc */
  static get _defaults() {
    return Object.assign(super._defaults, {
      required: true,
      blank: false,
      nullable: true,
      readonly: true,
      validationError: "is not a valid Document ID string"
    });
  }

  /** @override */
  _cast(value) {
    if ( value instanceof foundry.abstract.Document ) return value._id;
    else return String(value);
  }

  /** @override */
  _validateType(value, options) {
    if ( !isValidId(value) ) throw new Error("must be a valid 16-character alphanumeric ID");
  }
}

/* ---------------------------------------- */

/**
 * A subclass of {@link foundry.data.fields.StringField} which supports referencing some other Document by its UUID.
 * This field may not be blank, but may be null to indicate that no UUID is referenced.
 */
class DocumentUUIDField extends StringField {
  /**
   * @param {DocumentUUIDFieldOptions} [options] Options which configure the behavior of the field
   * @param {DataFieldContext} [context]    Additional context which describes the field
   */
  constructor(options, context) {
    super(options, context);
  }

  /** @inheritdoc */
  static get _defaults() {
    return Object.assign(super._defaults, {
      required: true,
      blank: false,
      nullable: true,
      initial: null,
      type: undefined,
      embedded: undefined
    });
  }

  /** @override */
  _validateType(value) {
    const p = parseUuid(value);
    if ( this.type ) {
      if ( p.type !== this.type ) throw new Error(`Invalid document type "${p.type}" which must be a "${this.type}"`);
    }
    else if ( p.type && !ALL_DOCUMENT_TYPES.includes(p.type) ) throw new Error(`Invalid document type "${p.type}"`);
    if ( (this.embedded === true) && !p.embedded.length ) throw new Error("must be an embedded document");
    if ( (this.embedded === false) && p.embedded.length ) throw new Error("may not be an embedded document");
    if ( !isValidId(p.documentId) ) throw new Error(`Invalid document ID "${p.documentId}"`);
  }

  /* -------------------------------------------- */
  /*  Form Field Integration                      */
  /* -------------------------------------------- */

  /** @override */
  _toInput(config) {
    Object.assign(config, {type: this.type, single: true});
    return foundry.applications.elements.HTMLDocumentTagsElement.create(config);
  }
}

/* ---------------------------------------- */

/**
 * A special class of {@link foundry.data.fields.StringField} field which references another DataModel by its id.
 * This field may also be null to indicate that no foreign model is linked.
 */
class ForeignDocumentField extends DocumentIdField {
  /**
   * @param {typeof Document} model  The foreign DataModel class definition which this field links to
   * @param {StringFieldOptions} [options]    Options which configure the behavior of the field
   * @param {DataFieldContext} [context]      Additional context which describes the field
   */
  constructor(model, options={}, context={}) {
    super(options, context);
    if ( !isSubclass(model, foundry.abstract.DataModel) ) {
      throw new Error("A ForeignDocumentField must specify a DataModel subclass as its type");
    }
    /**
     * A reference to the model class which is stored in this field
     * @type {typeof Document}
     */
    this.model = model;
  }

  /** @inheritdoc */
  static get _defaults() {
    return Object.assign(super._defaults, {nullable: true, readonly: false, idOnly: false});
  }

  /** @override */
  _cast(value) {
    if ( typeof value === "string" ) return value;
    if ( (value instanceof this.model) ) return value._id;
    throw new Error(`The value provided to a ForeignDocumentField must be a ${this.model.name} instance.`);
  }

  /** @inheritdoc */
  initialize(value, model, options={}) {
    if ( this.idOnly ) return value;
    if ( model?.pack && !foundry.utils.isSubclass(this.model, foundry.documents.BaseFolder) ) return null;
    if ( !game.collections ) return value; // Server-side
    return () => this.model?.get(value, {pack: model?.pack, ...options}) ?? null;
  }

  /** @inheritdoc */
  toObject(value) {
    return value?._id ?? value;
  }

  /* -------------------------------------------- */
  /*  Form Field Integration                      */
  /* -------------------------------------------- */

  /** @override */
  _toInput(config) {
    config.choices ??= this.choices;

    // Prepare passed choices
    StringField._prepareChoiceConfig(config);

    // Prepare visible Document instances as options
    const collection = game.collections.get(this.model.documentName);
    if ( collection && !config.options ) {
      const current = collection.get(config.value);
      let hasCurrent = false;
      const options = collection.reduce((arr, doc) => {
        if ( !doc.visible ) return arr;
        if ( doc === current ) hasCurrent = true;
        arr.push({value: doc.id, label: doc.name});
        return arr;
      }, []);
      if ( current && !hasCurrent ) options.unshift({value: config.value, label: current.name});
      config.options = options;
    }

    // Allow blank
    if ( !this.required || this.nullable ) config.blank ??= "";

    // Create select input
    return foundry.applications.fields.createSelectInput(config);
  }
}

/* -------------------------------------------- */

/**
 * A special {@link foundry.data.fields.StringField} which records a standardized CSS color string.
 */
class ColorField extends StringField {

  /** @inheritdoc */
  static get _defaults() {
    return Object.assign(super._defaults, {nullable: true, initial: null, blank: false});
  }

  /** @override */
  initialize(value, model, options={}) {
    if ( (value === null) || (value === undefined) ) return value;
    return Color.from(value);
  }

  /** @override */
  _cast(value) {
    return Color.from(value).css;
  }

  /** @inheritdoc */
  _validateType(value, options) {
    if ( !isColorString(value) ) throw new Error("must be a valid color string");
    return super._validateType(value, options);
  }

  /* -------------------------------------------- */
  /*  Form Field Integration                      */
  /* -------------------------------------------- */

  /** @override */
  _toInput(config) {
    if ( (config.placeholder === undefined) && !this.nullable && !(this.initial instanceof Function) ) {
      config.placeholder = this.initial;
    }
    return foundry.applications.elements.HTMLColorPickerElement.create(config);
  }
}

/* -------------------------------------------- */

/**
 * A special {@link foundry.data.fields.StringField} which records a file path or inline base64 data.
 *
 * When using the `FilePathField` in a data model that is persisted to the database, for example a Document sub-type, it is essential to declare this field in the package manifest so that it receives proper server-side validation of its contents.
 * See {@link foundry.packages.types.ServerSanitizationFields} for information about this structure.
 *
 * @property {string[]} categories      A set of categories in CONST.FILE_CATEGORIES which this field supports
 * @property {boolean} base64=false     Is embedded base64 data supported in lieu of a file path?
 * @property {boolean} texture=false    Does the file path field allow specifying a virtual file path which must begin
 *                                      with the "#" character?
 * @property {boolean} wildcard=false   Does this file path field allow wildcard characters?
 */
class FilePathField extends StringField {
  /**
   * @param {FilePathFieldOptions} [options]  Options which configure the behavior of the field
   * @param {DataFieldContext} [context]      Additional context which describes the field
   */
  constructor(options={}, context={}) {
    super(options, context);
    if ( this.categories.includes("MEDIA") ) {
      foundry.utils.logCompatibilityWarning('The "MEDIA" file category is deprecated. '
        + "Use CONST.MEDIA_FILE_CATEGORIES instead.", {since: 13, until: 15, once: true});
      this.categories = Array.from(new Set(this.categories.filter(c => c !== "MEDIA").concat(CONST.MEDIA_FILE_CATEGORIES)));
      if ( "categories" in this.options ) this.options.categories = this.categories;
    }
    if ( !this.categories.length || this.categories.some(c => !(c in FILE_CATEGORIES)) ) {
      throw new Error("The categories of a FilePathField must be keys in CONST.FILE_CATEGORIES");
    }
  }

  /** @inheritdoc */
  static get _defaults() {
    return Object.assign(super._defaults, {
      categories: [],
      base64: false,
      wildcard: false,
      virtual: false,
      nullable: true,
      blank: false,
      initial: null
    });
  }

  /* -------------------------------------------- */

  /** @inheritdoc */
  _validateType(value) {

    // Wildcard or virtual paths
    if ( this.virtual && (value[0] === "#") && value.length > 1 ) return true;
    if ( this.wildcard && value.includes("*") ) return true;

    // Allowed extension or base64
    const isValid = this.categories.some(c => {
      const category = FILE_CATEGORIES[c];
      if ( hasFileExtension(value, Object.keys(category)) ) return true;
      return isBase64Data(value, Object.values(category));
    });

    // Throw an error for invalid paths
    if ( !isValid ) {
      let err = "does not have a valid file extension";
      if ( this.base64 ) err += " or provide valid base64 data";
      throw new Error(err);
    }
  }

  /* -------------------------------------------- */
  /*  Form Field Integration                      */
  /* -------------------------------------------- */

  /** @override */
  _toInput(config) {
    // FIXME: This logic is fragile and would require a mapping between CONST.FILE_CATEGORIES and FilePicker.TYPES
    config.type = this.categories.length === 1 ? this.categories[0].toLowerCase() : "any";
    return foundry.applications.elements.HTMLFilePickerElement.create(config);
  }
}

/* -------------------------------------------- */

/**
 * A special {@link foundry.data.fields.NumberField} which represents an angle of rotation in degrees between 0 and 360.
 * @property {boolean} normalize Whether the angle should be normalized to [0,360) before being clamped to [0,360]. The
 *                               default is true.
 */
class AngleField extends NumberField {
  constructor(options={}, context={}) {
    super(options, context);
    if ( "base" in this.options ) this.base = this.options.base;
  }

  /** @inheritdoc */
  static get _defaults() {
    return Object.assign(super._defaults, {
      required: true,
      nullable: false,
      initial: 0,
      normalize: true,
      min: 0,
      max: 360,
      validationError: "is not a number between 0 and 360"
    });
  }

  /** @inheritdoc */
  _cast(value) {
    value = super._cast(value);
    if ( !this.normalize ) return value;
    value = Math.normalizeDegrees(value);
    /** @deprecated since v12 */
    if ( (this.#base === 360) && (value === 0) ) value = 360;
    return value;
  }

  /* -------------------------------------------- */
  /*  Deprecations and Compatibility              */
  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  get base() {
    const msg = "The AngleField#base is deprecated in favor of AngleField#normalize.";
    foundry.utils.logCompatibilityWarning(msg, {since: 12, until: 14});
    return this.#base;
  }

  /**
   * @deprecated since v12
   * @ignore
   */
  set base(v) {
    const msg = "The AngleField#base is deprecated in favor of AngleField#normalize.";
    foundry.utils.logCompatibilityWarning(msg, {since: 12, until: 14});
    this.#base = v;
  }

  /**
   * @deprecated since v12
   * @ignore
   */
  #base = 0;
}

/* -------------------------------------------- */

/**
 * A special {@link foundry.data.fields.NumberField} represents a number between 0 and 1.
 */
class AlphaField extends NumberField {
  static get _defaults() {
    return Object.assign(super._defaults, {
      required: true,
      nullable: false,
      initial: 1,
      min: 0,
      max: 1,
      validationError: "is not a number between 0 and 1"
    });
  }
}

/* -------------------------------------------- */

/**
 * A special {@link foundry.data.fields.NumberField} represents a number between 0 (inclusive) and 1 (exclusive).
 * Its values are normalized (modulo 1) to the range [0, 1) instead of being clamped.
 */
class HueField extends NumberField {
  static get _defaults() {
    return Object.assign(super._defaults, {
      required: true,
      nullable: false,
      initial: 0,
      min: 0,
      max: 1,
      validationError: "is not a number between 0 (inclusive) and 1 (exclusive)"
    });
  }

  /* -------------------------------------------- */

  /** @inheritdoc */
  _cast(value) {
    value = super._cast(value) % 1;
    if ( value < 0 ) value += 1;
    return value;
  }

  /* -------------------------------------------- */

  /** @override */
  _toInput(config) {
    return foundry.applications.elements.HTMLHueSelectorSlider.create(config);
  }
}

/* -------------------------------------------- */

/**
 * A special {@link foundry.data.fields.ForeignDocumentField} which defines the original author of a document.
 * This can only be changed later by GM users.
 */
class DocumentAuthorField extends ForeignDocumentField {

  /** @inheritdoc */
  static get _defaults() {
    return Object.assign(super._defaults, {
      nullable: false,
      gmOnly: true,
      label: "Author",
      initial: () => game.user?.id
    });
  }
}

/* -------------------------------------------- */

/**
 * A special {@link foundry.data.fields.ObjectField} which captures a mapping of User IDs to Document permission levels.
 */
class DocumentOwnershipField extends ObjectField {

  /** @inheritdoc */
  static get _defaults() {
    return Object.assign(super._defaults, {
      initial: {default: DOCUMENT_OWNERSHIP_LEVELS.NONE},
      validationError: "is not a mapping of user IDs and document permission levels",
      gmOnly: true
    });
  }

  /** @override */
  _validateType(value) {
    for ( const [k, v] of Object.entries(value) ) {
      if ( k.startsWith("-=") ) return isValidId(k.slice(2)) && (v === null);   // Allow removals
      if ( (k !== "default") && !isValidId(k) ) return false;
      if ( !Object.values(DOCUMENT_OWNERSHIP_LEVELS).includes(v) ) return false;
    }
  }
}

/* -------------------------------------------- */

/**
 * A special {@link foundry.data.fields.StringField} which contains serialized JSON data.
 */
class JSONField extends StringField {
  constructor(options, context) {
    super(options, context);
    this.blank = false;
    this.trim = false;
    this.choices = undefined;
  }

  /** @inheritdoc */
  static get _defaults() {
    return Object.assign(super._defaults, {
      blank: false,
      trim: false,
      initial: undefined,
      validationError: "is not a valid JSON string"
    });
  }

  /** @inheritdoc */
  clean(value, options) {
    if ( value === "" ) return '""';  // Special case for JSON fields
    return super.clean(value, options);
  }

  /** @override */
  _cast(value) {
    if ( (typeof value !== "string") || !isJSON(value) ) return JSON.stringify(value);
    return value;
  }

  /** @override */
  _validateType(value, options) {
    if ( (typeof value !== "string") || !isJSON(value) ) throw new Error("must be a serialized JSON string");
  }

  /** @override */
  initialize(value, model, options={}) {
    if ( (value === undefined) || (value === null) ) return value;
    return JSON.parse(value);
  }

  /** @override */
  toObject(value) {
    if ( (value === undefined) || (this.nullable && (value === null)) ) return value;
    return JSON.stringify(value);
  }

  /* -------------------------------------------- */
  /*  Form Field Integration                      */
  /* -------------------------------------------- */

  /**
   * @param {FormInputConfig & CodeMirrorInputConfig} config
   * @override
   */
  _toInput(config) {
    config.language = "json";
    config.indent ??= 2;
    config.value = foundry.data.validators.isJSON(config.value)
      ? JSON.stringify(JSON.parse(config.value), null, config.indent)
      : JSON.stringify(config.value, null, config.indent);
    return foundry.applications.elements.HTMLCodeMirrorElement.create(config);
  }
}

/* -------------------------------------------- */

/**
 * A special subclass of {@link foundry.data.fields.DataField} which can contain any value of any type.
 * Any input is accepted and is treated as valid.
 * It is not recommended to use this class except for very specific circumstances.
 */
class AnyField extends DataField {

  /** @override */
  _validateType(value) {
    return true;
  }
}


/* -------------------------------------------- */

/**
 * A subclass of {@link foundry.data.fields.StringField} which contains a sanitized HTML string.
 * This class does not override any StringField behaviors, but is used by the server-side to identify fields which
 * require sanitization of user input.
 *
 * When using the `HTMLField` in a data model that is persisted to the database, for example a Document sub-type, it is essential to declare this field in the package manifest so that it receives proper server-side validation of its contents.
 * See {@link foundry.packages.types.ServerSanitizationFields} for information about this structure.
 */
class HTMLField extends StringField {

  /** @inheritDoc */
  static get _defaults() {
    return Object.assign(super._defaults, {required: true, blank: true});
  }

  /** @inheritDoc */
  toFormGroup(groupConfig={}, inputConfig={}) {
    groupConfig.stacked ??= inputConfig.elementType !== "input";
    return super.toFormGroup(groupConfig, inputConfig);
  }

  /** @inheritDoc */
  _toInput(config) {
    config.elementType ??= "prose-mirror";
    return super._toInput(config);
  }
}

/* ---------------------------------------- */

/**
 * A subclass of {@link foundry.data.fields.NumberField} which is used for storing integer sort keys.
 */
class IntegerSortField extends NumberField {
  /** @inheritdoc */
  static get _defaults() {
    return Object.assign(super._defaults, {
      required: true,
      nullable: false,
      integer: true,
      initial: 0
    });
  }
}
/* ---------------------------------------- */

/**
 * A subclass of {@link foundry.data.fields.TypedObjectField} that is used specifically for the Document "flags" field.
 */
class DocumentFlagsField extends TypedObjectField {
  /**
   * @param {DataFieldOptions} [options]    Options which configure the behavior of the field
   * @param {DataFieldContext} [context]    Additional context which describes the field
   */
  constructor(options, context) {
    super(new ObjectField(), options, context);
  }

  /** @inheritdoc */
  static get _defaults() {
    return Object.assign(super._defaults, {
      validateKey: k => {
        try {
          foundry.packages.BasePackage.validateId(k);
        } catch {
          return false;
        }
        return true;
      }
    });
  }
}

/* ---------------------------------------- */

/**
 * A subclass of {@link foundry.data.fields.SchemaField} which stores document metadata in the _stats field.
 * @mixes DocumentStats
 */
class DocumentStatsField extends SchemaField {
  /**
   * @param {DataFieldOptions} [options]        Options which configure the behavior of the field
   * @param {DataFieldContext} [context]        Additional context which describes the field
   */
  constructor(options={}, context={}) {
    super({
      coreVersion: new StringField({required: true, blank: false, nullable: true, initial: () => game.release.version}),
      systemId: new StringField({required: true, blank: false, nullable: true, initial: () => game.system?.id ?? null}),
      systemVersion: new StringField({
        required: true,
        blank: false,
        nullable: true,
        initial: () => game.system?.version ?? null
      }),
      createdTime: new NumberField(),
      modifiedTime: new NumberField(),
      lastModifiedBy: new ForeignDocumentField(foundry.documents.BaseUser, {idOnly: true}),
      compendiumSource: new DocumentUUIDField(),
      duplicateSource: new DocumentUUIDField(),
      exportSource: new SchemaField({
        worldId: new StringField({required: true, blank: false, nullable: true}),
        uuid: new DocumentUUIDField({initial: undefined}),
        coreVersion: new StringField({required: true, blank: false, nullable: true}),
        systemId: new StringField({required: true, blank: false, nullable: true}),
        systemVersion: new StringField({required: true, blank: false, nullable: true})
      }, {nullable: true})
    }, options, context);
  }

  /**
   * All Document stats.
   * @type {string[]}
   */
  static fields = [
    "coreVersion", "systemId", "systemVersion", "createdTime", "modifiedTime", "lastModifiedBy", "compendiumSource",
    "duplicateSource", "exportSource"
  ];

  /**
   * These fields are managed by the server and are ignored if they appear in creation or update data.
   * @type {string[]}
   */
  static managedFields = ["coreVersion", "systemId", "systemVersion", "createdTime", "modifiedTime", "lastModifiedBy"];

  /* -------------------------------------------- */

  /**
   * Migrate deprecated core flags to `_stats` properties.
   * @param {typeof Document} document
   * @param {object} source
   * @internal
   */
  static _migrateData(document, source) {

    /**
     * Migrate flags.core.sourceId.
     * @deprecated since v12
     */
    document._addDataFieldMigration(source, "flags.core.sourceId", "_stats.compendiumSource");

    /**
     * Migrate flags.exportSource.
     * @deprecated since v13
     */
    if ( source.flags ) {
      document._addDataFieldMigration(source, "flags.exportSource", "_stats.exportSource", d => {
        const exportSource = foundry.utils.getProperty(d, "flags.exportSource");
        if ( !exportSource ) return null;
        return {
          worldId: exportSource.world ?? null,
          uuid: null,
          coreVersion: exportSource.coreVersion ?? null,
          systemId: exportSource.system ?? null,
          systemVersion: exportSource.systemVersion ?? null
        };
      });
    }
  }

  /* -------------------------------------------- */

  /**
   * Shim the deprecated core flag `exportSource` on Document source data.
   * @param {typeof Document} document
   * @param {object} source
   * @param {object} [options]
   * @internal
   */
  static _shimData(document, source, options) {
    if ( source.flags ) {
      /**
       * Shim flags.exportSource.
       * @deprecated since v13
       */
      Object.defineProperty(source.flags, "exportSource", {
        get: () => {
          document._logDataFieldMigration("flags.exportSource", "_stats.exportSource", {since: 13, until: 15});
          const exportSource = source._stats.exportSource;
          if ( !exportSource ) return undefined;
          return {
            world: exportSource.worldId,
            coreVersion: exportSource.coreVersion,
            system: exportSource.systemId,
            systemVersion: exportSource.systemVersion
          };
        },
        configurable: true,
        enumerable: false
      });
    }
  }

  /* -------------------------------------------- */

  /**
   * Shim the deprecated core flag `exportSource` on Documents.
   * @param {typeof Document} document
   * @internal
   */
  static _shimDocument(document) {
    /**
     * Shim flags.exportSource.
     * @deprecated since v13
     */
    Object.defineProperty(document.flags, "exportSource", {
      get: () => {
        document.constructor._logDataFieldMigration("flags.exportSource", "_stats.exportSource", {since: 13, until: 15});
        const exportSource = document._stats.exportSource;
        if ( !exportSource ) return undefined;
        return {
          world: exportSource.worldId,
          coreVersion: exportSource.coreVersion,
          system: exportSource.systemId,
          systemVersion: exportSource.systemVersion
        };
      },
      configurable: true,
      enumerable: false
    });
  }
}

/* ---------------------------------------- */

/**
 * A subclass of {@link foundry.data.fields.StringField} that is used specifically for the Document "type" field.
 */
class DocumentTypeField extends StringField {
  /**
   * @param {typeof Document} documentClass  The base document class which belongs in this field
   * @param {StringFieldOptions} [options]  Options which configure the behavior of the field
   * @param {DataFieldContext} [context]    Additional context which describes the field
   */
  constructor(documentClass, options={}, context={}) {
    options.choices = () => documentClass.TYPES;
    options.validationError = `is not a valid type for the ${documentClass.documentName} Document class`;
    super(options, context);
  }

  /** @inheritdoc */
  static get _defaults() {
    return Object.assign(super._defaults, {required: true, nullable: false, blank: false});
  }

  /** @override */
  _validateType(value, options) {
    if ( (typeof value !== "string") || !value ) throw new Error("must be a non-blank string");
    if ( this._isValidChoice(value) ) return true;
    // Allow unrecognized types if we are allowed to fallback (non-strict validation)
    if (options.fallback ) return true;
    throw new Error(`"${value}" ${this.options.validationError}`);
  }
}

/* ---------------------------------------- */

/**
 * A subclass of {@link foundry.data.fields.ObjectField} which supports a type-specific data object.
 */
class TypeDataField extends ObjectField {
  /**
   * @param {typeof Document} document      The base document class which belongs in this field
   * @param {DataFieldOptions} [options]    Options which configure the behavior of the field
   * @param {DataFieldContext} [context]    Additional context which describes the field
   */
  constructor(document, options={}, context={}) {
    super(options, context);
    /**
     * The canonical document name of the document type which belongs in this field
     * @type {typeof Document}
     */
    this.document = document;
  }

  /** @inheritdoc */
  static get _defaults() {
    return Object.assign(super._defaults, {required: true});
  }

  /** @override */
  static recursive = true;

  /**
   * Return the package that provides the sub-type for the given model.
   * @param {DataModel} model       The model instance created for this sub-type.
   * @returns {System|Module|null}
   */
  static getModelProvider(model) {
    const document = model.parent;
    if ( !document ) return null;
    const documentClass = document.constructor;
    const documentName = documentClass.documentName;
    const type = document.type;

    // Unrecognized type
    if ( !documentClass.TYPES.includes(type) ) return null;

    // Core-defined sub-type
    const coreTypes = documentClass.metadata.coreTypes;
    if ( coreTypes.includes(type) ) return null;

    // System-defined sub-type
    const systemTypes = game.system.documentTypes[documentName];
    if ( systemTypes && (type in systemTypes) ) return game.system;

    // Module-defined sub-type
    const moduleId = type.substring(0, type.indexOf("."));
    return game.modules.get(moduleId) ?? null;
  }

  /**
   * A convenience accessor for the name of the document type associated with this TypeDataField
   * @type {string}
   */
  get documentName() {
    return this.document.documentName;
  }

  /**
   * Get the DataModel definition that should be used for this type of document.
   * @param {string} type              The Document instance type
   * @returns {typeof DataModel|null}  The DataModel class or null
   */
  getModelForType(type) {
    if ( !type ) return null;
    return globalThis.CONFIG?.[this.documentName]?.dataModels?.[type] ?? null;
  }

  /** @override */
  getInitialValue(data) {
    const initial = super.getInitialValue(data); // ObjectField could return this.initial, undefined, null, or {}
    if ( getType(initial) === "Object" ) return this._cleanType(initial, {partial: false, source: data});
    return initial;
  }

  /** @override */
  _cleanType(value, options) {
    if ( !(typeof value === "object") ) value = {};

    // Use a defined DataModel
    const type = options.source?.type;
    const cls = this.getModelForType(type);
    if ( cls ) return cls.cleanData(value, {...options, source: value});
    if ( options.partial ) return value;

    // Use the defined template.json
    const template = game?.model[this.documentName]?.[type] || {};
    const insertKeys = (type === BASE_DOCUMENT_TYPE) || !game?.system?.strictDataCleaning;
    return mergeObject(template, value, {insertKeys, inplace: false});
  }

  /** @override */
  initialize(value, model, options={}) {
    const cls = this.getModelForType(model._source.type);
    if ( cls ) {
      const instance = new cls(value, {parent: model, ...options});
      if ( !("modelProvider" in instance) ) Object.defineProperty(instance, "modelProvider", {
        value: this.constructor.getModelProvider(instance),
        writable: false
      });
      return instance;
    }
    return deepClone(value);
  }

  /* ---------------------------------------- */

  /** @inheritDoc */
  _updateDiff(source, key, value, difference, options) {
    const cls = this.getModelForType(source.type);
    if ( cls ) cls.schema._updateDiff(source, key, value, difference, options);
    else super._updateDiff(source, key, value, difference, options);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _updateCommit(source, key, value, diff, options) {
    const cls = this.getModelForType(source.type);
    if ( cls ) cls.schema._updateCommit(source, key, value, diff, options);
    else super._updateCommit(source, key, value, diff, options);
  }

  /* ---------------------------------------- */

  /** @inheritdoc */
  _validateType(data, options={}) {
    const result = super._validateType(data, options);
    if ( result !== undefined ) return result;
    const cls = this.getModelForType(options.source?.type);
    const schema = cls?.schema;
    return schema?.validate(data, {...options, source: data});
  }

  /* ---------------------------------------- */

  /** @override */
  _validateModel(changes, options={}) {
    const cls = this.getModelForType(options.source?.type);
    return cls?.validateJoint(changes);
  }

  /* ---------------------------------------- */

  /** @override */
  toObject(value) {
    return value.toObject instanceof Function ? value.toObject(false) : deepClone(value);
  }

  /* -------------------------------------------- */

  /** @override */
  _addTypes(source, changes, options={}) {
    const cls = this.getModelForType(options.changes?.type ?? options.source?.type);
    cls?.schema._addTypes(source, changes, options);
  }

  /* -------------------------------------------- */

  /**
   * Migrate this field's candidate source data.
   * @param {object} sourceData   Candidate source data of the root model
   * @param {any} fieldData       The value of this field within the source data
   */
  migrateSource(sourceData, fieldData) {
    if ( getType(fieldData) !== "Object" ) return;
    const cls = this.getModelForType(sourceData.type);
    if ( cls ) cls.migrateDataSafe(fieldData);
  }
}

/* ---------------------------------------- */

/**
 * A subclass of {@link foundry.data.fields.DataField} that defines a union of schema-constrained objects discriminable
 * via a `type` property.
 */
class TypedSchemaField extends DataField {
  /**
   * @param {Record<string, DataSchema|SchemaField|typeof DataModel>} types The different types this field can represent
   * @param {DataFieldOptions} [options]                                    Options for configuring the field
   * @param {DataFieldContext} [context]                                    Additional context describing the field
   */
  constructor(types, options, context) {
    super(options, context);
    this.types = this.#configureTypes(types);
  }

  /* ---------------------------------------- */

  /** @inheritdoc */
  static get _defaults() {
    return Object.assign(super._defaults, {required: true});
  }

  /* ---------------------------------------- */

  /** @override */
  static recursive = true;

  /* ---------------------------------------- */

  /**
   * The types of this field.
   * @type {{[type: string]: SchemaField}}
   */
  types;

  /* -------------------------------------------- */

  /**
   * Initialize and validate the structure of the provided type definitions.
   * @param {{[type: string]: DataSchema|SchemaField|typeof DataModel}} types The provided field definitions
   * @returns {{[type: string]: SchemaField}}                                 The validated fields
   */
  #configureTypes(types) {
    if ( (typeof types !== "object") ) {
      throw new Error("A DataFields must be an object with string keys and DataField values.");
    }
    types = {...types};
    for ( let [type, field] of Object.entries(types) ) {
      if ( isSubclass(field, foundry.abstract.DataModel) ) field = new EmbeddedDataField(field, {name: type});
      if ( getType(field) === "Object" ) {
        const schema = {...field};
        if ( !("type" in schema) ) {
          schema.type = new StringField({required: true, blank: false, initial: type,
            validate: value => value === type, validationError: `must be equal to "${type}"`});
        }
        field = new SchemaField(schema, {name: type});
      }
      if ( !(field instanceof SchemaField) ) {
        throw new Error(`The "${type}" field is not an instance of the SchemaField class or a subclass of DataModel.`);
      }
      field.name ??= type;
      if ( field.name !== type ) throw new Error(`The name of the "${this.fieldPath}.${type}" field must be "${type}".`);
      if ( field.parent !== undefined ) {
        throw new Error(`The "${field.fieldPath}" field already belongs to some other parent and may not be reused.`);
      }
      types[type] = field;
      field.parent = this;
      if ( !field.required ) throw new Error(`The "${field.fieldPath}" field must be required.`);
      if ( field.nullable ) throw new Error(`The "${field.fieldPath}" field must not be nullable.`);
      const typeField = field.fields.type;
      if ( !(typeField instanceof StringField) ) throw new Error(`The "${field.fieldPath}" field must have a "type" StringField.`);
      if ( !typeField.required ) throw new Error(`The "${typeField.fieldPath}" field must be required.`);
      if ( typeField.nullable ) throw new Error(`The "${typeField.fieldPath}" field must not be nullable.`);
      if ( typeField.blank ) throw new Error(`The "${typeField.fieldPath}" field must not be blank.`);
      if ( typeField.validate(type, {fallback: false}) !== undefined ) throw new Error(`"${type}" must be a valid type of "${typeField.fieldPath}".`);
    }
    return types;
  }

  /* -------------------------------------------- */

  /**
   * Get the schema for the given type.
   * @param {*} type
   * @returns {SchemaField|void}
   */
  #getTypeSchema(type) {
    if ( typeof type !== "string" ) return;
    if ( !Object.hasOwn(this.types, type) ) return;
    return this.types[type];
  }

  /* ---------------------------------------- */

  /** @override */
  _getField(path) {
    if ( !path.length ) return this;
    return this.#getTypeSchema(path.shift())?._getField(path);
  }

  /* -------------------------------------------- */
  /*  Data Field Methods                          */
  /* -------------------------------------------- */

  /** @override */
  _cleanType(value, options) {
    const schema = this.#getTypeSchema(value?.type);
    if ( !schema ) return value;
    return schema.clean(value, options);
  }

  /* ---------------------------------------- */

  /** @override */
  _cast(value) {
    if ( value.toObject instanceof Function ) value = value.toObject();
    return getType(value) === "Object" ? value : {};
  }

  /* ---------------------------------------- */

  /** @override */
  _validateSpecial(value) {
    const result = super._validateSpecial(value);
    if ( result !== undefined ) return result;
    const schema = this.#getTypeSchema(value?.type);
    if ( !schema ) throw new Error("does not have a valid type");
  }

  /* ---------------------------------------- */

  /** @override */
  _validateType(value, options) {
    return this.types[value.type].validate(value, options);
  }

  /* ---------------------------------------- */

  /** @override */
  initialize(value, model, options) {
    const schema = this.#getTypeSchema(value?.type);
    if ( !schema ) return value;
    return schema.initialize(value, model, options);
  }

  /* ---------------------------------------- */

  /** @inheritDoc */
  _updateDiff(source, key, value, difference, options) {
    const sourceType = source[key]?.type;
    const valueType = value?.type;
    if ( value && (("==type" in value) || ("-=type" in value)) ) throw new Error("The type of a TypedSchemaField cannot be updated with ==type or -=type");
    if ( sourceType && valueType && (sourceType !== valueType) && (options.recursive !== false) ) {
      throw new Error("The type of a TypedSchemaField can be changed only by forced replacement (==) of the entire field value or with {recursive: false}");
    }
    const schema = this.#getTypeSchema(valueType);
    if ( schema ) schema._updateDiff(source, key, value, difference, options);
    else super._updateDiff(source, key, applySpecialKeys(value), difference, options);
  }

  /* ---------------------------------------- */

  /** @inheritDoc */
  _updateCommit(source, key, value, diff, options) {
    const schema = this.#getTypeSchema(value?.type);
    if ( schema ) {
      if ( "type" in diff ) source[key] = undefined;
      schema._updateCommit(source, key, value, diff, options);
    }
    else super._updateCommit(source, key, value, diff, options);
  }

  /* ---------------------------------------- */

  /** @override */
  toObject(value) {
    if ( !value ) return value;
    return this.#getTypeSchema(value.type)?.toObject(value) ?? value;
  }

  /* -------------------------------------------- */

  /** @override */
  apply(fn, data={}, options={}) {

    // Apply to this TypedSchemaField
    const thisFn = typeof fn === "string" ? this[fn] : fn;
    thisFn?.call(this, data, options);

    // Apply to the schema of the type
    const schema = this.#getTypeSchema(data.type);
    return schema?.apply(fn, data, options) ?? {};
  }

  /* -------------------------------------------- */

  /** @override */
  _addTypes(source, changes, options={}) {
    if ( getType(source) !== "Object" ) return;
    if ( getType(changes) !== "Object" ) return;
    const type = changes.type ??= source.type;
    this.#getTypeSchema(type)?._addTypes(source, changes, options);
  }

  /* -------------------------------------------- */

  /**
   * Migrate this field's candidate source data.
   * @param {object} sourceData   Candidate source data of the root model
   * @param {any} fieldData       The value of this field within the source data
   */
  migrateSource(sourceData, fieldData) {
    if ( getType(fieldData) !== "Object" ) return;
    const schema = this.#getTypeSchema(fieldData.type);
    const canMigrate = schema?.migrateSource instanceof Function;
    if ( canMigrate ) schema.migrateSource(sourceData, fieldData);
  }
}

/* ---------------------------------------- */

/**
 * A subclass of {@link foundry.data.fields.StringField} which contains JavaScript code.
 */
class JavaScriptField extends StringField {
  /**
   * @param {JavaScriptFieldOptions} [options] Options which configure the behavior of the field
   * @param {DataFieldContext} [context]    Additional context which describes the field
   */
  constructor(options, context) {
    super(options, context);
    this.choices = undefined;
  }

  /** @inheritdoc */
  static get _defaults() {
    return Object.assign(super._defaults, {required: true, blank: true, nullable: false, async: false});
  }

  /** @inheritdoc */
  _validateType(value, options) {
    const result = super._validateType(value, options);
    if ( result !== undefined ) return result;
    try {
      new (this.async ? AsyncFunction : Function)(value);
    } catch(err) {
      const scope = this.async ? "an asynchronous" : "a synchronous";
      err.message = `must be valid JavaScript for ${scope} scope:\n${err.message}`;
      throw new Error(err);
    }
  }

  /* -------------------------------------------- */
  /*  Form Field Integration                      */
  /* -------------------------------------------- */

  /** @override */
  toFormGroup(groupConfig={}, inputConfig={}) {
    groupConfig.stacked ??= true;
    return super.toFormGroup(groupConfig, inputConfig);
  }

  /**
   * @param {FormInputConfig & CodeMirrorInputConfig} config
   * @override
   */
  _toInput(config) {
    config.language = "javascript";
    config.indent ??= 2;
    return foundry.applications.elements.HTMLCodeMirrorElement.create(config);
  }
}

/**
 * @import {DataField} from "../data/fields.mjs";
 * @import {DataModelConstructionContext, DataModelFromSourceOptions, DataModelUpdateOptions,
 *   DataModelValidationOptions, DataSchema} from "./_types.mjs";
 */

/**
 * An abstract class which is a fundamental building block of numerous structures and concepts in Foundry Virtual
 * Tabletop. Data models perform several essential roles:
 *
 * * A static schema definition that all instances of that model adhere to.
 * * A workflow for data migration, cleaning, validation, and initialization such that provided input data is structured
 *   according to the rules of the model's declared schema.
 * * A workflow for transacting differential updates to the instance data and serializing its data into format suitable
 *   for storage or transport.
 *
 * DataModel subclasses can be used for a wide variety of purposes ranging from simple game settings to high complexity
 * objects like `Scene` documents. Data models are often nested; see the {@link DataModel.parent} property for more.
 *
 * @abstract
 * @template {object} [ModelData=object]
 * @template {DataModelConstructionContext} [ModelContext=DataModelConstructionContext]
 */
class DataModel {
  /**
   * @param {Partial<ModelData>} [data={}] Initial data used to construct the data object. The provided object will be
   *                                       owned by the constructed model instance and may be mutated.
   * @param {ModelContext} [options={}]    Context and data validation options which affects initial model construction.
   */
  constructor(data={}, {parent=null, strict=true, ...options}={}) {

    // Parent model
    Object.defineProperty(this, "parent", {
      value: (() => {
        if ( parent === null ) return null;
        if ( parent instanceof DataModel ) return parent;
        throw new Error("The provided parent must be a DataModel instance");
      })(),
      writable: false,
      enumerable: false
    });

    // Source data
    Object.defineProperty(this, "_source", {
      value: this._initializeSource(data, {strict, ...options}),
      writable: false,
      enumerable: false
    });
    Object.seal(this._source);

    // Additional subclass configurations
    this._configure(options);

    // Data validation and initialization
    const fallback = options.fallback ?? !strict;
    const dropInvalidEmbedded = options.dropInvalidEmbedded ?? !strict;
    this.validate({strict, fallback, dropInvalidEmbedded, fields: true, joint: true});
    this._initialize({strict, ...options});
  }

  /**
   * Configure the data model instance before validation and initialization workflows are performed.
   * @param {object} [options] Additional options modifying the configuration
   * @protected
   */
  _configure(options={}) {}

  /* -------------------------------------------- */

  /**
   * The source data object for this DataModel instance.
   * Once constructed, the source object is sealed such that no keys may be added nor removed.
   * @type {ModelData}
   * @public
   */
  _source;

  /**
   * The defined and cached Data Schema for all instances of this DataModel.
   * @type {SchemaField}
   * @internal
   */
  static _schema;

  /**
   * An immutable reverse-reference to a parent DataModel to which this model belongs.
   * @type {DataModel|null}
   */
  parent;

  /* ---------------------------------------- */
  /*  Data Schema                             */
  /* ---------------------------------------- */

  /**
   * Define the data schema for models of this type.
   * The schema is populated the first time it is accessed and cached for future reuse.
   *
   * The schema, through its fields, provide the essential cleaning, validation, and initialization methods to turn the
   * {@link _source} values into direct properties of the data model. The schema is a static property of the model and
   * is reused by all instances to perform validation.
   *
   * The schemas defined by the core software in classes like {@link foundry.documents.BaseActor} are validated by the
   * server, where user code does not run. However, almost all documents have a `flags` field to store data, and many
   * have a `system` field that can be configured to be a {@link foundry.abstract.TypeDataModel} instance. Those models
   * are *not* constructed on the server and rely purely on client-side code, which means certain extra-sensitive fields
   * must be also be registered through your package manifest. {@link foundry.packages.types.ServerSanitizationFields}
   *
   * @returns {DataSchema}
   * @abstract
   *
   * @example
   * ```js
   * class SomeModel extends foundry.abstract.DataModel {
   *   static defineSchema() {
   *     return {
   *       foo: new foundry.data.fields.StringField()
   *     }
   *   }
   * }
   *
   * class AnotherModel extends SomeModel {
   *   static defineSchema() {
   *     // Inheritance and object oriented principles apply to schema definition
   *     const schema = super.defineSchema()
   *
   *     schema.bar = new foundry.data.fields.NumberField()
   *
   *     return schema;
   *   }
   * }
   * ```
   */
  static defineSchema() {
    throw new Error(`The ${this.name} subclass of DataModel must define its Document schema`);
  }

  /* ---------------------------------------- */

  /**
   * The Data Schema for all instances of this DataModel.
   * @type {SchemaField}
   */
  static get schema() {
    if ( this.hasOwnProperty("_schema") ) return this._schema;
    const schema = new SchemaField(Object.freeze(this.defineSchema()));
    Object.defineProperty(this, "_schema", {value: schema, writable: false});
    return schema;
  }

  /* ---------------------------------------- */

  /**
   * Define the data schema for this document instance.
   * @type {SchemaField}
   */
  get schema() {
    return this.constructor.schema;
  }

  /* ---------------------------------------- */

  /**
   * Is the current state of this DataModel invalid?
   * The model is invalid if there is any unresolved failure.
   * @type {boolean}
   */
  get invalid() {
    return Object.values(this.#validationFailures).some(f => f?.unresolved);
  }

  /**
   * An array of validation failure instances which may have occurred when this instance was last validated.
   * @type {{fields: DataModelValidationFailure|null, joint: DataModelValidationFailure|null}}
   */
  get validationFailures() {
    return this.#validationFailures;
  }

  #validationFailures = Object.seal({fields: null, joint: null });

  /**
   * A set of localization prefix paths which are used by this DataModel. This provides an alternative to defining the
   * `label` and `hint` property of each field by having foundry map the labels to a structure inside the path
   * provided by the prefix.
   *
   * @type {string[]}
   *
   * @example
   * JavaScript class definition and localization call.
   * ```js
   * class MyDataModel extends foundry.abstract.DataModel {
   *   static defineSchema() {
   *     return {
   *       foo: new foundry.data.fields.StringField(),
   *       bar: new foundry.data.fields.NumberField()
   *     };
   *   }
   *   static LOCALIZATION_PREFIXES = ["MYMODULE.MYDATAMODEL"];
   * }
   *
   * Hooks.on("i18nInit", () => {
   *   // Foundry will attempt to automatically localize models registered for a document subtype, so this step is only
   *   // needed for other data model usage, e.g. for a Setting.
   *   Localization.localizeDataModel(MyDataModel);
   * });
   * ```
   *
   * JSON localization file
   * ```json
   * {
   *   "MYMODULE": {
   *     "MYDATAMODEL": {
   *       "FIELDS" : {
   *         "foo": {
   *           "label": "Foo",
   *           "hint": "Instructions for foo"
   *         },
   *         "bar": {
   *           "label": "Bar",
   *           "hint": "Instructions for bar"
   *         }
   *       }
   *     }
   *   }
   * }
   * ```
   */
  static LOCALIZATION_PREFIXES = [];

  /* ---------------------------------------- */
  /*  Data Cleaning Methods                   */
  /* ---------------------------------------- */

  /**
   * Initialize the source data for a new DataModel instance.
   * One-time migrations and initial cleaning operations are applied to the source data.
   * @param {object|DataModel} data   The candidate source data from which the model will be constructed
   * @param {object} [options]        Options provided to the model constructor
   * @returns {object}                Migrated and cleaned source data which will be stored to the model instance,
   *                                  which is the same object as the `data` argument
   * @protected
   */
  _initializeSource(data, options={}) {
    if ( data instanceof DataModel ) data = data.toObject();

    // Migrate old data to the new format
    data = this.constructor.migrateDataSafe(data);
    const dt = getType(data);
    if ( dt !== "Object" ) {
      throw new Error(`${this.constructor.name} was incorrectly constructed with a ${dt} instead of an object.`);
    }

    // Clean data and apply shims for backwards compatibility
    data = this.constructor.cleanData(data);
    return this.constructor.shimData(data);
  }

  /* ---------------------------------------- */

  /**
   * Clean a data source object to conform to a specific provided schema.
   * @param {object} [source]         The source data object
   * @param {object} [options={}]     Additional options which are passed to field cleaning methods
   * @returns {object}                The cleaned source data, which is the same object as the `source` argument
   */
  static cleanData(source={}, options={}) {
    return this.schema.clean(source, options);
  }

  /* ---------------------------------------- */
  /*  Data Initialization                     */
  /* ---------------------------------------- */

  /**
   * A generator that orders the DataFields in the DataSchema into an expected initialization order.
   * @returns {Generator<[string,DataField]>}
   * @yields {DataField}
   * @protected
   */
  static *_initializationOrder() {
    for ( const entry of this.schema.entries() ) yield entry;
  }

  /* ---------------------------------------- */

  /**
   * Initialize the instance by copying data from the source object to instance attributes.
   * This mirrors the workflow of SchemaField#initialize but with some added functionality.
   * @param {object} [options]        Options provided to the model constructor
   * @protected
   */
  _initialize(options={}) {
    for ( const [name, field] of this.constructor._initializationOrder() ) {
      const sourceValue = this._source[name];

      // Field initialization
      const value = field.initialize(sourceValue, this, options);

      // Special handling for Document IDs.
      if ( (name === "_id") && (!Object.getOwnPropertyDescriptor(this, "_id") || (this._id === null)) ) {
        Object.defineProperty(this, name, {value, writable: false, configurable: true});
      }

      // Readonly fields
      else if ( field.readonly ) {
        if ( this[name] !== undefined ) continue;
        Object.defineProperty(this, name, {value, writable: false});
      }

      // Getter fields
      else if ( value instanceof Function ) {
        Object.defineProperty(this, name, {get: value, set() {}, configurable: true});
      }

      // Writable fields
      else this[name] = value;
    }
  }

  /* ---------------------------------------- */

  /**
   * Reset the state of this data instance back to mirror the contained source data, erasing any changes.
   */
  reset() {
    this._initialize();
  }

  /* ---------------------------------------- */

  /**
   * Clone a model, creating a new data model by combining current data with provided overrides.
   * @param {object} [data={}]             Additional data which overrides current document data at the time of creation
   * @param {DataModelConstructionContext} [context={}]          Context options passed to the data model constructor
   * @returns {DataModel|Promise<DataModel>} The cloned instance
   */
  clone(data={}, context={}) {
    data = mergeObject(this.toObject(), data, {insertKeys: false, performDeletions: true, inplace: true});
    return new this.constructor(data, {parent: this.parent, ...context});
  }

  /* ---------------------------------------- */
  /*  Data Validation Methods                 */
  /* ---------------------------------------- */

  /**
   * Validate the data contained in the document to check for type and content.
   * If changes are provided, missing types are added to it before cleaning and validation.
   * This mutates the provided changes. This function throws an error if data within the document is not valid.
   * @param {DataModelValidationOptions} options    Options which modify how the model is validated
   * @returns {boolean}                             Whether the data source or proposed change is reported as valid.
   *                                                A boolean is always returned if validation is non-strict.
   * @throws {Error}                                An error thrown if validation is strict and a failure occurs.
   */
  validate({changes, clean=false, fallback=false, dropInvalidEmbedded=false, strict=true,
    fields=true, joint}={}) {
    let source = changes ?? this._source;

    // Determine whether we are performing partial or joint validation
    joint = joint ?? !changes;
    const partial = !joint;

    // Add types where missing in a set of partial changes
    if ( partial ) {
      if ( !clean ) source = deepClone(source);
      this.schema._addTypes(this._source, source);
    }

    // Optionally clean the data before validating
    if ( clean ) this.constructor.cleanData(source, {partial});

    // Validate individual fields in the data or in a specific change-set, throwing errors if validation fails
    if ( fields ) {
      this.#validationFailures.fields = null;
      const failure = this.schema.validate(source, {partial, fallback, dropInvalidEmbedded});
      if ( failure ) {
        const id = this._source._id ? `[${this._source._id}] ` : "";
        failure.message = `${this.constructor.name} ${id}validation errors:`;
        this.#validationFailures.fields = failure;
        if ( strict && failure.unresolved ) throw failure.asError();
        else logger.warn(failure.asError());
      }
    }

    // Perform joint document-level validations which consider all fields together
    if ( joint ) {
      this.#validationFailures.joint = null;
      try {
        this.schema._validateModel(source);     // Validate inner models
        this.constructor.validateJoint(source); // Validate this model
      } catch(err) {
        const id = this._source._id ? `[${this._source._id}] ` : "";
        const message = [this.constructor.name, id, `Joint Validation Error:\n${err.message}`].filterJoin(" ");
        const failure = new DataModelValidationFailure({message, unresolved: true});
        this.#validationFailures.joint = failure;
        if ( strict ) throw failure.asError();
        else logger.warn(failure.asError());
      }
    }
    return !this.invalid;
  }

  /* ---------------------------------------- */

  /**
   * Evaluate joint validation rules which apply validation conditions across multiple fields of the model.
   * Field-specific validation rules should be defined as part of the DataSchema for the model.
   * This method allows for testing aggregate rules which impose requirements on the overall model.
   * @param {object} data     Candidate data for the model
   * @throws {Error}          An error if a validation failure is detected
   */
  static validateJoint(data) {}

  /* ---------------------------------------- */
  /*  Data Management                         */
  /* ---------------------------------------- */

  /**
   * Update the DataModel locally by applying an object of changes to its source data.
   * The provided changes are expanded, cleaned, validated, and stored to the source data object for this model.
   * The provided changes argument is mutated in this process.
   * The source data is then re-initialized to apply those changes to the prepared data.
   * The method returns an object of differential changes which modified the original data.
   *
   * @param {object} changes                  New values which should be applied to the data model
   * @param {DataModelUpdateOptions} options  Options which determine how the new data is merged
   * @returns {object}                        An object containing differential keys and values that were changed
   * @throws {DataModelValidationError}       An error if the requested data model changes were invalid
   */
  updateSource(changes={}, options={}) {
    const rootKey = "_source";
    const rootDiff = {[rootKey]: {}};

    // Expand the object, if dot-notation keys are provided
    if ( Object.keys(changes).some(k => /\./.test(k)) ) {
      const expandedChanges = expandObject(changes);
      for ( const key in changes ) delete changes[key];
      Object.assign(changes, expandedChanges);
    }

    // Clean proposed changes
    this.schema._addTypes(this._source, changes);
    this.constructor.cleanData(changes, {partial: true});

    // Perform updates on the safe copy of source data
    const copy = this.#prepareSafeSource(changes);
    this.schema._updateDiff({_source: copy}, rootKey, changes, rootDiff, options);
    const diff = rootDiff[rootKey] || {};
    if ( isEmpty(diff) ) return diff;

    // Validate final field-level changes only on the subset of fields which changed
    const typeChanged = "type" in diff;
    this.validate({changes: diff, fields: true, joint: false, strict: true});
    if ( !typeChanged ) delete diff.type;

    // Validate the final model on the safe copy of changes
    this.validate({changes: copy, fields: false, joint: true, strict: true});

    // If this is not a dry run, enact the final changes
    if ( !options.dryRun ) {
      this.schema._updateCommit(this, rootKey, copy, diff, options);
      this._initialize();
    }

    // Return the diff of enacted changes
    return diff;
  }

  /* ---------------------------------------- */

  /**
   * Prepare a mutation-safe version of the source data.
   * The resulting object contains a copy of the source data for any field present in the proposed set of model changes.
   * For fields not present in the proposed changes, the resulting object directly references the true source.
   * This approach is used because of superior performance for complex data structures.
   * @param {object} changes
   * @returns {object}
   */
  #prepareSafeSource(changes) {
    const copy = {};
    for ( const k of Object.keys(changes) ) {
      const key = isDeletionKey(k) ? k.slice(2) : k;
      if ( key in this._source ) copy[key] = deepClone(this._source[key]); // Copy for changed fields
    }
    for ( const k of Object.keys(this._source) ) {
      if ( !(k in copy) ) copy[k] = this._source[k]; // Direct reference for unchanged fields
    }
    return copy;
  }

  /* ---------------------------------------- */
  /*  Serialization and Storage               */
  /* ---------------------------------------- */

  /**
   * Copy and transform the DataModel into a plain object.
   * Draw the values of the extracted object from the data source (by default) otherwise from its transformed values.
   * @param {boolean} [source=true]     Draw values from the underlying data source rather than transformed values
   * @returns {object}                  The extracted primitive object
   */
  toObject(source=true) {
    if ( source ) return deepClone(this._source);

    // We have use the schema of the class instead of the schema of the instance to prevent an infinite recursion:
    // the EmbeddedDataField replaces the schema of its model instance with itself
    // and EmbeddedDataField#toObject calls DataModel#toObject.
    return this.constructor.schema.toObject(this);
  }

  /* ---------------------------------------- */

  /**
   * Extract the source data for the DataModel into a simple object format that can be serialized.
   * @returns {object}          The document source data expressed as a plain object
   */
  toJSON() {
    return this.toObject(true);
  }

  /* -------------------------------------------- */

  /**
   * Create a new instance of this DataModel from a source record.
   * The source is presumed to be trustworthy and is not strictly validated.
   * @param {object} source    Initial document data which comes from a trusted source.
   * @param {Omit<DataModelConstructionContext, "strict"> & DataModelFromSourceOptions} [context]
   *                           Model construction context
   * @returns {DataModel}
   */
  static fromSource(source, {strict=false, ...context}={}) {
    return new this(source, {strict, ...context});
  }

  /* ---------------------------------------- */

  /**
   * Create a DataModel instance using a provided serialized JSON string.
   * @param {string} json       Serialized document data in string format
   * @returns {DataModel}       A constructed data model instance
   */
  static fromJSON(json) {
    return this.fromSource(JSON.parse(json));
  }

  /* -------------------------------------------- */
  /*  Deprecations and Compatibility              */
  /* -------------------------------------------- */

  /**
   * Migrate candidate source data for this DataModel which may require initial cleaning or transformations.
   * @param {object} source           The candidate source data from which the model will be constructed
   * @returns {object}                Migrated source data, which is the same object as the `source` argument
   */
  static migrateData(source) {
    this.schema.migrateSource(source, source);
    return source;
  }

  /* ---------------------------------------- */

  /**
   * Wrap data migration in a try/catch which attempts it safely
   * @param {object} source           The candidate source data from which the model will be constructed
   * @returns {object}                Migrated source data, which is the same object as the `source` argument
   */
  static migrateDataSafe(source) {
    try {
      this.migrateData(source);
    } catch(err) {
      err.message = `Failed data migration for ${this.name}: ${err.message}`;
      logger.warn(err);
    }
    return source;
  }

  /* ---------------------------------------- */

  /**
   * Take data which conforms to the current data schema and add backwards-compatible accessors to it in order to
   * support older code which uses this data.
   * @param {object} data         Data which matches the current schema
   * @param {object} [options={}] Additional shimming options
   * @param {boolean} [options.embedded=true] Apply shims to embedded models?
   * @returns {object}            Data with added backwards-compatible properties, which is the same object as
   *                              the `data` argument
   */
  static shimData(data, {embedded=true}={}) {
    if ( Object.isSealed(data) ) return data;
    const schema = this.schema;
    if ( embedded ) {
      for ( const [name, value] of Object.entries(data) ) {
        const field = schema.get(name);
        if ( (field instanceof EmbeddedDataField) && !Object.isSealed(value) ) {
          data[name] = field.model.shimData(value || {});
        }
        else if ( field instanceof EmbeddedCollectionField ) {
          for ( const d of (value || []) ) {
            if ( !Object.isSealed(d) ) field.model.shimData(d);
          }
        }
      }
    }
    return data;
  }
}

/**
 * @import DatabaseBackend from "./backend.mjs";
 * @import BaseUser from "../documents/user.mjs";
 * @import {
 *   DatabaseCreateOperation,
 *   DatabaseGetOperation,
 *   DatabaseUpdateOperation,
 *   DatabaseDeleteOperation,
 *   DocumentCloneOptions,
 *   DocumentClassMetadata
 * } from "./_types.mjs";
 * @import {DocumentConstructionContext} from "./_types.mjs";
 * @import {DocumentOwnershipLevel, DocumentOwnershipNumber} from "../constants.mjs";
 * @import {DocumentFlags, DocumentStats} from "../data/_types.mjs";
 */

/**
 * An extension of the base DataModel which defines a Document.
 * Documents are special in that they are persisted to the database and referenced by _id.
 * @abstract
 *
 * @template {object} [DocumentData=object] Initial data from which to construct the Document
 * @template {DocumentConstructionContext} [DocumentContext=DocumentConstructionContext] Construction context options
 *
 * @property {string|null} _id                    The document identifier, unique within its Collection, or null if the
 *                                                Document has not yet been assigned an identifier
 * @property {string} [name]                      Documents typically have a human-readable name
 * @property {DataModel} [system]                 Certain document types may have a system data model which contains
 *                                                subtype-specific data defined by the game system or a module
 * @property {DocumentStats} [_stats]             Primary document types have a _stats object which provides metadata
 *                                                about their status
 * @property {DocumentFlags} flags                Documents each have an object of arbitrary flags which are used by
 *                                                systems or modules to store additional Document-specific data
 * @extends {DataModel<DocumentData, DocumentContext>}
 */
class Document extends DataModel {

  /** @override */
  _configure({pack=null, parentCollection=null}={}) {
    /**
     * An immutable reverse-reference to the name of the collection that this Document exists in on its parent, if any.
     * @type {string|null}
     */
    Object.defineProperty(this, "parentCollection", {
      value: this._getParentCollection(parentCollection),
      writable: false
    });

    /**
     * An immutable reference to a containing Compendium collection to which this Document belongs.
     * @type {string|null}
     */
    Object.defineProperty(this, "pack", {
      value: (() => {
        if ( typeof pack === "string" ) return pack;
        if ( this.parent?.pack ) return this.parent.pack;
        if ( pack === null ) return null;
        throw new Error("The provided compendium pack ID must be a string");
      })(),
      writable: false
    });

    // Construct Embedded Collections
    const collections = {};
    for ( const [fieldName, field] of Object.entries(this.constructor.hierarchy) ) {
      if ( !field.constructor.implementation ) continue;
      const data = this._source[fieldName];
      const c = collections[fieldName] = new field.constructor.implementation(fieldName, this, data);
      Object.defineProperty(this, fieldName, {value: c, writable: false});
    }

    /**
     * A mapping of embedded Document collections which exist in this model.
     * @type {Record<string, EmbeddedCollection>}
     */
    Object.defineProperty(this, "collections", {value: Object.seal(collections), writable: false});
  }

  /* ---------------------------------------- */

  /**
   * Ensure that all Document classes share the same schema of their base declaration.
   * @type {SchemaField}
   * @override
   */
  static get schema() {
    if ( this._schema ) return this._schema;
    const base = this.baseDocument;
    if ( !base.hasOwnProperty("_schema") ) {
      const schema = new SchemaField(Object.freeze(base.defineSchema()));
      Object.defineProperty(base, "_schema", {value: schema, writable: false});
    }
    Object.defineProperty(this, "_schema", {value: base._schema, writable: false});
    return base._schema;
  }

  /* -------------------------------------------- */

  /** @override */
  static *_initializationOrder() {
    const hierarchy = this.hierarchy;

    // Initialize non-hierarchical fields first
    for ( const [name, field] of this.schema.entries() ) {
      if ( name in hierarchy ) continue;
      yield [name, field];
    }

    // Initialize hierarchical fields last
    for ( const [name, field] of Object.entries(hierarchy) ) {
      yield [name, field];
    }
  }

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /**
   * Default metadata which applies to each instance of this Document type.
   * @type {Readonly<DocumentClassMetadata>}
   */
  static metadata = Object.freeze({
    name: "Document",
    label: "DOCUMENT.Document",
    coreTypes: [BASE_DOCUMENT_TYPE],
    collection: "documents",
    embedded: {},
    hasTypeData: false,
    indexed: false,
    compendiumIndexFields: [],
    permissions: {
      view: "LIMITED",      // At least limited permission is required to view the Document
      create: "ASSISTANT",  // Assistants or Gamemasters can create Documents
      update: "OWNER",      // Document owners can update Documents (this includes GM users)
      delete: "ASSISTANT"   // Assistants or Gamemasters can delete Documents
    },
    preserveOnImport: ["_id", "sort", "ownership", "folder"],
    /*
     * The metadata has to include the version of this Document schema, which needs to be increased
     * whenever the schema is changed such that Document data created before this version
     * would come out different if `fromSource(data).toObject()` was applied to it so that
     * we always vend data to client that is in the schema of the current core version.
     * The schema version needs to be bumped if
     *   - a field was added or removed,
     *   - the class/type of any field was changed,
     *   - the casting or cleaning behavior of any field class was changed,
     *   - the data model of an embedded data field was changed,
     *   - certain field properties are changed (e.g. required, nullable, blank, ...), or
     *   - there have been changes to cleanData or migrateData of the Document.
     *
     * Moreover, the schema version needs to be bumped if the sanitization behavior
     * of any field in the schema was changed.
     */
    schemaVersion: undefined
  });

  /* -------------------------------------------- */

  /** @override */
  static LOCALIZATION_PREFIXES = ["DOCUMENT"];

  /* -------------------------------------------- */

  /**
   * The database backend used to execute operations and handle results.
   * @type {DatabaseBackend}
   */
  static get database() {
    return globalThis.CONFIG.DatabaseBackend;
  }

  /* -------------------------------------------- */

  /**
   * Return a reference to the configured subclass of this base Document type.
   * @type {typeof Document}
   */
  static get implementation() {
    return globalThis.CONFIG[this.documentName]?.documentClass || this;
  }

  /* -------------------------------------------- */

  /**
   * The base document definition that this document class extends from.
   * @type {typeof Document}
   */
  static get baseDocument() {
    let cls;
    let parent = this;
    while ( parent ) {
      cls = parent;
      parent = Object.getPrototypeOf(cls);
      if ( parent === Document ) return cls;
    }
    throw new Error(`Base Document class identification failed for "${this.documentName}"`);
  }

  /* -------------------------------------------- */

  /**
   * The named collection to which this Document belongs.
   * @type {string}
   */
  static get collectionName() {
    return this.metadata.collection;
  }

  get collectionName() {
    return this.constructor.collectionName;
  }

  /* -------------------------------------------- */

  /**
   * The canonical name of this Document type, for example "Actor".
   * @type {string}
   */
  static get documentName() {
    return this.metadata.name;
  }

  get documentName() {
    return this.constructor.documentName;
  }

  /* ---------------------------------------- */

  /**
   * The allowed types which may exist for this Document class.
   * @type {string[]}
   */
  static get TYPES() {
    return Object.keys(game.model[this.metadata.name]);
  }

  /* -------------------------------------------- */

  /**
   * Does this Document support additional subtypes?
   * @type {boolean}
   */
  static get hasTypeData() {
    return this.metadata.hasTypeData;
  }

  /* -------------------------------------------- */
  /*  Model Properties                            */
  /* -------------------------------------------- */

  /**
   * The Embedded Document hierarchy for this Document.
   * @returns {Readonly<Record<string, EmbeddedCollectionField|EmbeddedDocumentField>>}
   */
  static get hierarchy() {
    const hierarchy = {};
    for ( const [fieldName, field] of this.schema.entries() ) {
      if ( field.constructor.hierarchical ) hierarchy[fieldName] = field;
    }
    Object.defineProperty(this, "hierarchy", {value: Object.freeze(hierarchy), writable: false});
    return hierarchy;
  }

  /* -------------------------------------------- */

  /**
   * Identify the collection in a parent Document that this Document belongs to, if any.
   * @param {string|null} [parentCollection]  An explicitly provided parent collection name.
   * @returns {string|null}
   * @internal
   */
  _getParentCollection(parentCollection) {
    if ( !this.parent ) return null;
    if ( parentCollection ) return parentCollection;
    return this.parent.constructor.getCollectionName(this.documentName);
  }

  /**
   * The canonical identifier for this Document.
   * @type {string|null}
   */
  get id() {
    return this._id;
  }

  /**
   * A reference to the Compendium Collection containing this Document, if any, and otherwise null.
   * @returns {CompendiumCollection|null}
   * @abstract
   */
  get compendium() {
    throw new Error("A subclass of Document must implement this getter.");
  }

  /**
   * Is this document embedded within a parent document?
   * @returns {boolean}
   */
  get isEmbedded() {
    return !!(this.parent && this.parentCollection);
  }

  /**
   * Is this document in a compendium?
   * @returns {boolean}
   */
  get inCompendium() {
    return !!this.pack;
  }

  /* -------------------------------------------- */

  /**
   * A Universally Unique Identifier (uuid) for this Document instance.
   * @type {string}
   */
  get uuid() {
    return foundry.utils.buildUuid(this);
  }

  /* ---------------------------------------- */
  /*  Model Permissions                       */
  /* ---------------------------------------- */

  /**
   * Test whether a given User has sufficient permissions to create Documents of this type in general. This does not
   * guarantee that the User is able to create all Documents of this type, as certain document-specific requirements
   * may also be present.
   *
   * Generally speaking, this method is used to verify whether a User should be presented with the option to create
   * Documents of this type in the UI.
   *
   * @param {BaseUser} user       The User being tested
   * @returns {boolean}           Does the User have a sufficient role to create?
   */
  static canUserCreate(user) {
    const perm = this.metadata.permissions.create;

    // Require a custom User permission
    if ( perm in USER_PERMISSIONS ) return user.hasPermission(perm);

    // Require a specific User role
    if ( perm in USER_ROLES ) return user.hasRole(perm);

    // Construct a sample Document
    let doc;
    try {
      doc = this.fromSource(this.cleanData(), {strict: true});
    } catch(err) {
      return false;
    }

    // Use a specialized permission test function
    if ( perm instanceof Function ) return doc.canUserModify(user, "create");

    // Require Document ownership
    if ( perm in DOCUMENT_OWNERSHIP_LEVELS ) return doc.testUserPermission(user, perm);
    return false;
  }

  /* ---------------------------------------- */

  /**
   * Get the explicit permission level that a User has over this Document, a value in CONST.DOCUMENT_OWNERSHIP_LEVELS.
   * Compendium content ignores the ownership field in favor of User role-based ownership. Otherwise, Documents use
   * granular per-User ownership definitions and Embedded Documents defer to their parent ownership.
   *
   * This method returns the value recorded in Document ownership, regardless of the User's role, for example a
   * GAMEMASTER user might still return a result of NONE if they are not explicitly denoted as having a level.
   *
   * To test whether a user has a certain capability over the document, testUserPermission should be used.
   *
   * @param {BaseUser} [user=game.user] The User being tested
   * @returns {DocumentOwnershipNumber} A numeric permission level from {@link CONST.DOCUMENT_OWNERSHIP_LEVELS}
   */
  getUserLevel(user) {
    user ||= game.user;
    if ( this.pack ) return this.compendium.getUserLevel(user);               // Compendium User role
    if ( this.schema.has("ownership") ) {
      const level = this.ownership[user.id] ?? this.ownership.default ?? DOCUMENT_OWNERSHIP_LEVELS.NONE;
      if ( level !== DOCUMENT_OWNERSHIP_LEVELS.INHERIT ) return level;        // Defer inherited for Embedded
    }
    if ( this.parent ) return this.parent.getUserLevel(user);                 // Embedded Documents
    return DOCUMENT_OWNERSHIP_LEVELS.NONE;                                    // Otherwise, NONE
  }

  /* ---------------------------------------- */

  /**
   * Test whether a certain User has a requested permission level (or greater) over the Document
   * @param {BaseUser} user                 The User being tested
   * @param {DocumentOwnershipLevel} permission The permission level from DOCUMENT_OWNERSHIP_LEVELS to test
   * @param {object} options                Additional options involved in the permission test
   * @param {boolean} [options.exact=false] Require the exact permission level requested?
   * @returns {boolean}                     Does the user have this permission level over the Document?
   */
  testUserPermission(user, permission, {exact=false}={}) {
    const perms = DOCUMENT_OWNERSHIP_LEVELS;
    let level;
    if ( user.isGM ) level = perms.OWNER;
    else if ( user.isBanned ) level = perms.NONE;
    else level = this.getUserLevel(user);
    const target = (typeof permission === "string") ? (perms[permission] ?? perms.OWNER) : permission;
    return exact ? level === target : level >= target;
  }

  /* ---------------------------------------- */

  /**
   * Test whether a given User has permission to perform some action on this Document
   * @param {BaseUser} user             The User attempting modification
   * @param {string} action             The attempted action
   * @param {object} [data]             Data involved in the attempted action
   * @returns {boolean}                 Does the User have permission?
   */
  canUserModify(user, action, data={}) {
    const permissions = this.constructor.metadata.permissions;
    const perm = permissions[action];

    // Use a specialized permission test function
    if ( perm instanceof Function ) return perm(user, this, data);

    // Require a custom User permission
    if ( perm in USER_PERMISSIONS ) return user.hasPermission(perm);

    // Require a specific User role
    if ( perm in USER_ROLES ) return user.hasRole(perm);

    // Require Document ownership
    if ( perm in DOCUMENT_OWNERSHIP_LEVELS ) return this.testUserPermission(user, perm);
    return false;
  }

  /* ---------------------------------------- */
  /*  Model Methods                           */
  /* ---------------------------------------- */

  /**
   * Clone a document, creating a new document by combining current data with provided overrides.
   * The cloned document is ephemeral and not yet saved to the database.
   * @param {object} [data={}]    Additional data which overrides current document data at the time of creation
   * @param {DocumentConstructionContext & DocumentCloneOptions} [context]
   *                                          Additional context options passed to the create method
   * @returns {Document|Promise<Document>}    The cloned Document instance
   */
  clone(data={}, context={}) {
    const {save=false, keepId=false, addSource=false, ...remaining} = context;
    context = remaining;
    context.parent = this.parent;
    context.pack = this.pack;
    context.strict = false;
    data = mergeObject(this.toObject(), data, {insertKeys: false, performDeletions: true, inplace: true});
    if ( !keepId ) delete data._id;
    if ( addSource ) {
      data._stats.duplicateSource = this.uuid;
      data._stats.exportSource = null;
    }
    return save ? this.constructor.create(data, context) : new this.constructor(data, context);
  }

  /* -------------------------------------------- */

  /**
   * For Documents which include game system data, migrate the system data object to conform to its latest data model.
   * The data model is defined by the template.json specification included by the game system.
   * @returns {object}              The migrated system data object
   */
  migrateSystemData() {
    if ( !this.constructor.hasTypeData ) {
      throw new Error(`The ${this.documentName} Document does not include a TypeDataField.`);
    }
    if ( (this.system instanceof DataModel) && !(this.system.modelProvider instanceof foundry.packages.BaseSystem) ) {
      throw new Error(`The ${this.documentName} Document does not have system-provided package data.`);
    }
    const model = game.model[this.documentName]?.[this.type] ?? {};
    return mergeObject(model, this.system, {
      insertKeys: false,
      insertValues: true,
      enforceTypes: false,
      overwrite: true,
      inplace: false
    });
  }

  /* ---------------------------------------- */

  /** @inheritDoc */
  toObject(source=true) {
    const data = super.toObject(source);
    return this.constructor.shimData(data);
  }

  /* -------------------------------------------- */
  /*  Database Operations                         */
  /* -------------------------------------------- */

  /**
   * Create multiple Documents using provided input data.
   * Data is provided as an array of objects where each individual object becomes one new Document.
   *
   * @param {Array<object|Document>} data  An array of data objects or existing Documents to persist.
   * @param {Partial<Omit<DatabaseCreateOperation, "data">>} [operation={}]  Parameters of the requested creation
   *                                  operation
   * @returns {Promise<Document[]>}        An array of created Document instances
   *
   * @example Create a single Document
   * ```js
   * const data = [{name: "New Actor", type: "character", img: "path/to/profile.jpg"}];
   * const created = await Actor.implementation.createDocuments(data);
   * ```
   *
   * @example Create multiple Documents
   * ```js
   * const data = [{name: "Tim", type: "npc"], [{name: "Tom", type: "npc"}];
   * const created = await Actor.implementation.createDocuments(data);
   * ```
   *
   * @example Create multiple embedded Documents within a parent
   * ```js
   * const actor = game.actors.getName("Tim");
   * const data = [{name: "Sword", type: "weapon"}, {name: "Breastplate", type: "equipment"}];
   * const created = await Item.implementation.createDocuments(data, {parent: actor});
   * ```
   *
   * @example Create a Document within a Compendium pack
   * ```js
   * const data = [{name: "Compendium Actor", type: "character", img: "path/to/profile.jpg"}];
   * const created = await Actor.implementation.createDocuments(data, {pack: "mymodule.mypack"});
   * ```
   */
  static async createDocuments(data=[], operation={}) {
    if ( operation.parent?.pack ) operation.pack = operation.parent.pack;
    operation.data = data;
    const created = await this.database.create(this.implementation, operation);

    /** @deprecated since v12 */
    if ( getDefiningClass(this, "_onCreateDocuments") !== Document ) {
      foundry.utils.logCompatibilityWarning("The Document._onCreateDocuments static method is deprecated in favor of "
        + "Document._onCreateOperation", {since: 12, until: 14});
      await this._onCreateDocuments(created, operation);
    }
    return created;
  }

  /* -------------------------------------------- */

  /**
   * Update multiple Document instances using provided differential data.
   * Data is provided as an array of objects where each individual object updates one existing Document.
   *
   * @param {object[]} updates          An array of differential data objects, each used to update a single Document
   * @param {Partial<Omit<DatabaseUpdateOperation, "updates">>} [operation={}] Parameters of the database update
   *                                    operation
   * @returns {Promise<Document[]>}     An array of updated Document instances
   *
   * @example Update a single Document
   * ```js
   * const updates = [{_id: "12ekjf43kj2312ds", name: "Timothy"}];
   * const updated = await Actor.implementation.updateDocuments(updates);
   * ```
   *
   * @example Update multiple Documents
   * ```js
   * const updates = [{_id: "12ekjf43kj2312ds", name: "Timothy"}, {_id: "kj549dk48k34jk34", name: "Thomas"}]};
   * const updated = await Actor.implementation.updateDocuments(updates);
   * ```
   *
   * @example Update multiple embedded Documents within a parent
   * ```js
   * const actor = game.actors.getName("Timothy");
   * const updates = [{_id: sword.id, name: "Magic Sword"}, {_id: shield.id, name: "Magic Shield"}];
   * const updated = await Item.implementation.updateDocuments(updates, {parent: actor});
   * ```
   *
   * @example Update Documents within a Compendium pack
   * ```js
   * const actor = await pack.getDocument(documentId);
   * const updated = await Actor.implementation.updateDocuments([{_id: actor.id, name: "New Name"}],
   *   {pack: "mymodule.mypack"});
   * ```
   */
  static async updateDocuments(updates=[], operation={}) {
    if ( operation.parent?.pack ) operation.pack = operation.parent.pack;
    operation.updates = updates;
    const updated = await this.database.update(this.implementation, operation);

    /** @deprecated since v12 */
    if ( getDefiningClass(this, "_onUpdateDocuments") !== Document ) {
      foundry.utils.logCompatibilityWarning("The Document._onUpdateDocuments static method is deprecated in favor of "
        + "Document._onUpdateOperation", {since: 12, until: 14});
      await this._onUpdateDocuments(updated, operation);
    }
    return updated;
  }

  /* -------------------------------------------- */

  /**
   * Delete one or multiple existing Documents using an array of provided ids.
   * Data is provided as an array of string ids for the documents to delete.
   *
   * @param {string[]} ids              An array of string ids for the documents to be deleted
   * @param {Partial<Omit<DatabaseDeleteOperation, "ids">>} [operation={}]  Parameters of the database deletion
   *                                    operation
   * @returns {Promise<Document[]>}     An array of deleted Document instances
   *
   * @example Delete a single Document
   * ```js
   * const tim = game.actors.getName("Tim");
   * const deleted = await Actor.implementation.deleteDocuments([tim.id]);
   * ```
   *
   * @example Delete multiple Documents
   * ```js
   * const tim = game.actors.getName("Tim");
   * const tom = game.actors.getName("Tom");
   * const deleted = await Actor.implementation.deleteDocuments([tim.id, tom.id]);
   * ```
   *
   * @example Delete multiple embedded Documents within a parent
   * ```js
   * const tim = game.actors.getName("Tim");
   * const sword = tim.items.getName("Sword");
   * const shield = tim.items.getName("Shield");
   * const deleted = await Item.implementation.deleteDocuments([sword.id, shield.id], parent: actor});
   * ```
   *
   * @example Delete Documents within a Compendium pack
   * ```js
   * const actor = await pack.getDocument(documentId);
   * const deleted = await Actor.implementation.deleteDocuments([actor.id], {pack: "mymodule.mypack"});
   * ```
   */
  static async deleteDocuments(ids=[], operation={}) {
    if ( operation.parent?.pack ) operation.pack = operation.parent.pack;
    operation.ids = ids;
    const deleted = await this.database.delete(this.implementation, operation);

    /** @deprecated since v12 */
    if ( getDefiningClass(this, "_onDeleteDocuments") !== Document ) {
      foundry.utils.logCompatibilityWarning("The Document._onDeleteDocuments static method is deprecated in favor of "
        + "Document._onDeleteOperation", {since: 12, until: 14});
      await this._onDeleteDocuments(deleted, operation);
    }
    return deleted;
  }

  /* -------------------------------------------- */

  /**
   * Create a new Document using provided input data, saving it to the database.
   * @see {@link Document.createDocuments}
   * @param {object|Document|(object|Document)[]} [data={}] Initial data used to create this Document, or a Document
   *                                                        instance to persist.
   * @param {Partial<Omit<DatabaseCreateOperation, "data">>} [operation={}]  Parameters of the creation operation
   * @returns {Promise<Document | Document[] | undefined>}        The created Document instance(s)
   *
   * @example Create a World-level Item
   * ```js
   * const data = [{name: "Special Sword", type: "weapon"}];
   * const created = await Item.implementation.create(data);
   * ```
   *
   * @example Create an Actor-owned Item
   * ```js
   * const data = [{name: "Special Sword", type: "weapon"}];
   * const actor = game.actors.getName("My Hero");
   * const created = await Item.implementation.create(data, {parent: actor});
   * ```
   *
   * @example Create an Item in a Compendium pack
   * ```js
   * const data = [{name: "Special Sword", type: "weapon"}];
   * const created = await Item.implementation.create(data, {pack: "mymodule.mypack"});
   * ```
   */
  static async create(data={}, operation={}) {
    const isArray = Array.isArray(data);
    const createData = isArray ? data : [data];
    const created = await this.createDocuments(createData, operation);
    return isArray ? created : created.shift();
  }

  /* -------------------------------------------- */

  /**
   * Update this Document using incremental data, saving it to the database.
   * @see {@link Document.updateDocuments}
   * @param {object} [data={}]          Differential update data which modifies the existing values of this document
   * @param {Partial<Omit<DatabaseUpdateOperation, "updates">>} [operation={}]  Parameters of the update operation
   * @returns {Promise<Document|undefined>}       The updated Document instance, or undefined not updated
   */
  async update(data={}, operation={}) {
    data._id = this.id;
    operation.parent = this.parent;
    operation.pack = this.pack;
    const updates = await this.constructor.updateDocuments([data], operation);
    return updates.shift();
  }

  /* -------------------------------------------- */

  /**
   * Delete this Document, removing it from the database.
   * @see {@link Document.deleteDocuments}
   * @param {Partial<Omit<DatabaseDeleteOperation, "ids">>} [operation={}]  Parameters of the deletion operation
   * @returns {Promise<Document|undefined>}       The deleted Document instance, or undefined if not deleted
   */
  async delete(operation={}) {
    operation.parent = this.parent;
    operation.pack = this.pack;
    const deleted = await this.constructor.deleteDocuments([this.id], operation);
    return deleted.shift();
  }

  /* -------------------------------------------- */

  /**
   * Get a World-level Document of this type by its id.
   * @param {string} documentId         The Document ID
   * @param {DatabaseGetOperation} [operation={}] Parameters of the get operation
   * @returns {Document|null}  The retrieved Document, or null
   */
  static get(documentId, operation={}) {
    if ( !documentId ) return null;
    if ( operation.pack ) {
      const pack = game.packs.get(operation.pack);
      return pack?.index.get(documentId) || null;
    }
    else {
      const collection = game.collections?.get(this.documentName);
      return collection?.get(documentId) || null;
    }
  }

  /* -------------------------------------------- */
  /*  Embedded Operations                         */
  /* -------------------------------------------- */

  /**
   * A compatibility method that returns the appropriate name of an embedded collection within this Document.
   * @param {string} name    An existing collection name or a document name.
   * @returns {string|null}  The provided collection name if it exists, the first available collection for the
   *                         document name provided, or null if no appropriate embedded collection could be found.
   * @example Passing an existing collection name.
   * ```js
   * Actor.implementation.getCollectionName("items");
   * // returns "items"
   * ```
   *
   * @example Passing a document name.
   * ```js
   * Actor.implementation.getCollectionName("Item");
   * // returns "items"
   * ```
   */
  static getCollectionName(name) {
    if ( name in this.hierarchy ) return name;
    for ( const [collectionName, field] of Object.entries(this.hierarchy) ) {
      if ( field.model.documentName === name ) return collectionName;
    }
    return null;
  }

  /* -------------------------------------------- */

  /**
   * Obtain a reference to the Array of source data within the data object for a certain embedded Document name
   * @param {string} embeddedName   The name of the embedded Document type
   * @returns {DocumentCollection}  The Collection instance of embedded Documents of the requested type
   */
  getEmbeddedCollection(embeddedName) {
    const collectionName = this.constructor.getCollectionName(embeddedName);
    if ( !collectionName ) {
      throw new Error(`${embeddedName} is not a valid embedded Document within the ${this.documentName} Document`);
    }
    const field = this.constructor.hierarchy[collectionName];
    return field.getCollection(this);
  }

  /* -------------------------------------------- */

  /**
   * Get an embedded document by its id from a named collection in the parent document.
   * @param {string} embeddedName              The name of the embedded Document type
   * @param {string} id                        The id of the child document to retrieve
   * @param {object} [options]                 Additional options which modify how embedded documents are retrieved
   * @param {boolean} [options.strict=false]   Throw an Error if the requested id does not exist. See Collection#get
   * @param {boolean} [options.invalid=false]  Allow retrieving an invalid Embedded Document.
   * @returns {Document}                       The retrieved embedded Document instance, or undefined
   * @throws If the embedded collection does not exist, or if strict is true and the Embedded Document could not be
   *         found.
   */
  getEmbeddedDocument(embeddedName, id, {invalid=false, strict=false}={}) {
    const collection = this.getEmbeddedCollection(embeddedName);
    return collection.get(id, {invalid, strict});
  }

  /* -------------------------------------------- */

  /**
   * Create multiple embedded Document instances within this parent Document using provided input data.
   * @see {@link Document.createDocuments}
   * @param {string} embeddedName                     The name of the embedded Document type
   * @param {object[]} data                           An array of data objects used to create multiple documents
   * @param {DatabaseCreateOperation} [operation={}]  Parameters of the database creation workflow
   * @returns {Promise<Document[]>}                   An array of created Document instances
   */
  async createEmbeddedDocuments(embeddedName, data=[], operation={}) {
    this.getEmbeddedCollection(embeddedName); // Validation only
    operation.parent = this;
    operation.pack = this.pack;
    const cls = getDocumentClass(embeddedName);
    return cls.createDocuments(data, operation);
  }

  /* -------------------------------------------- */

  /**
   * Update multiple embedded Document instances within a parent Document using provided differential data.
   * @see {@link Document.updateDocuments}
   * @param {string} embeddedName                     The name of the embedded Document type
   * @param {object[]} updates                        An array of differential data objects, each used to update a
   *                                                  single Document
   * @param {DatabaseUpdateOperation} [operation={}]  Parameters of the database update workflow
   * @returns {Promise<Document[]>}                   An array of updated Document instances
   */
  async updateEmbeddedDocuments(embeddedName, updates=[], operation={}) {
    this.getEmbeddedCollection(embeddedName); // Validation only
    operation.parent = this;
    operation.pack = this.pack;
    const cls = getDocumentClass(embeddedName);
    return cls.updateDocuments(updates, operation);
  }

  /* -------------------------------------------- */

  /**
   * Delete multiple embedded Document instances within a parent Document using provided string ids.
   * @see {@link Document.deleteDocuments}
   * @param {string} embeddedName                     The name of the embedded Document type
   * @param {string[]} ids                            An array of string ids for each Document to be deleted
   * @param {DatabaseDeleteOperation} [operation={}]  Parameters of the database deletion workflow
   * @returns {Promise<Document[]>}                   An array of deleted Document instances
   */
  async deleteEmbeddedDocuments(embeddedName, ids, operation={}) {
    this.getEmbeddedCollection(embeddedName); // Validation only
    operation.parent = this;
    operation.pack = this.pack;
    const cls = getDocumentClass(embeddedName);
    return cls.deleteDocuments(ids, operation);
  }

  /* -------------------------------------------- */

  /**
   * Iterate over all embedded Documents that are hierarchical children of this Document.
   * @param {string} [_parentPath]                      A parent field path already traversed
   * @yields {[string, Document]}
   */
  * traverseEmbeddedDocuments(_parentPath) {
    for ( const [fieldName, field] of Object.entries(this.constructor.hierarchy) ) {
      const fieldPath = _parentPath ? `${_parentPath}.${fieldName}` : fieldName;

      // Singleton embedded document
      if ( field instanceof foundry.data.fields.EmbeddedDocumentField ) {
        const document = this[fieldName];
        if ( document ) {
          yield [fieldPath, document];
          yield* document.traverseEmbeddedDocuments(fieldPath);
        }
      }

      // Embedded document collection
      else if ( field instanceof foundry.data.fields.EmbeddedCollectionField ) {
        const collection = this[fieldName];
        const isDelta = field instanceof foundry.data.fields.EmbeddedCollectionDeltaField;
        for ( const document of collection.values() ) {
          if ( isDelta && !collection.manages(document.id) ) continue;
          yield [fieldPath, document];
          yield* document.traverseEmbeddedDocuments(fieldPath);
        }
      }
    }
  }

  /* -------------------------------------------- */
  /*  Flag Operations                             */
  /* -------------------------------------------- */

  /**
   * Get the value of a "flag" for this document
   * See the setFlag method for more details on flags
   *
   * @param {string} scope        The flag scope which namespaces the key
   * @param {string} key          The flag key
   * @returns {*}                 The flag value
   */
  getFlag(scope, key) {
    const scopes = this.constructor.database.getFlagScopes();
    if ( !scopes.includes(scope) ) throw new Error(`Flag scope "${scope}" is not valid or not currently active`);

    /** @deprecated since v12 */
    if ( (scope === "core") && (key === "sourceId") ) {
      foundry.utils.logCompatibilityWarning("The core.sourceId flag has been deprecated. "
        + "Please use the _stats.compendiumSource property instead.", { since: 12, until: 14 });
      return this._stats?.compendiumSource;
    }

    if ( !this.flags || !(scope in this.flags) ) return undefined;
    return getProperty(this.flags?.[scope], key);
  }

  /* -------------------------------------------- */

  /**
   * Assign a "flag" to this document.
   * Flags represent key-value type data which can be used to store flexible or arbitrary data required by either
   * the core software, game systems, or user-created modules.
   *
   * Each flag should be set using a scope which provides a namespace for the flag to help prevent collisions.
   *
   * Flags set by the core software use the "core" scope.
   * Flags set by game systems or modules should use the canonical name attribute for the module
   * Flags set by an individual world should "world" as the scope.
   *
   * Flag values can assume almost any data type. Setting a flag value to null will delete that flag.
   *
   * @param {string} scope        The flag scope which namespaces the key
   * @param {string} key          The flag key
   * @param {*} value             The flag value
   * @returns {Promise<Document>} A Promise resolving to the updated document
   */
  async setFlag(scope, key, value) {
    const scopes = this.constructor.database.getFlagScopes();
    if ( !scopes.includes(scope) ) throw new Error(`Flag scope "${scope}" is not valid or not currently active`);
    return this.update({
      flags: {
        [scope]: {
          [key]: value
        }
      }
    });
  }

  /* -------------------------------------------- */

  /**
   * Remove a flag assigned to the document
   * @param {string} scope        The flag scope which namespaces the key
   * @param {string} key          The flag key
   * @returns {Promise<Document>} The updated document instance
   */
  async unsetFlag(scope, key) {
    const scopes = this.constructor.database.getFlagScopes();
    if ( !scopes.includes(scope) ) throw new Error(`Flag scope "${scope}" is not valid or not currently active`);
    const head = key.split(".");
    const tail = `-=${head.pop()}`;
    key = ["flags", scope, ...head, tail].join(".");
    return this.update({[key]: null});
  }

  /* -------------------------------------------- */
  /*  Database Creation Operations                */
  /* -------------------------------------------- */

  /**
   * Pre-process a creation operation for a single Document instance. Pre-operation events only occur for the client
   * which requested the operation.
   *
   * Modifications to the pending Document instance must be performed using {@link updateSource}.
   *
   * @param {object} data               The initial data object provided to the document creation request
   * @param {object} options            Additional options which modify the creation request
   * @param {BaseUser} user             The User requesting the document creation
   * @returns {Promise<boolean|void>}   Return false to exclude this Document from the creation operation
   * @protected
   */
  async _preCreate(data, options, user) {}

  /**
   * Post-process a creation operation for a single Document instance. Post-operation events occur for all connected
   * clients.
   *
   * @param {object} data                         The initial data object provided to the document creation request
   * @param {object} options                      Additional options which modify the creation request
   * @param {string} userId                       The id of the User requesting the document update
   * @protected
   */
  _onCreate(data, options, userId) {}

  /**
   * Pre-process a creation operation, potentially altering its instructions or input data. Pre-operation events only
   * occur for the client which requested the operation.
   *
   * This batch-wise workflow occurs after individual {@link _preCreate} workflows and provides a final pre-flight check
   * before a database operation occurs.
   *
   * Modifications to pending documents must mutate the documents array or alter individual document instances using
   * {@link updateSource}.
   *
   * @param {Document[]} documents                Pending document instances to be created
   * @param {DatabaseCreateOperation} operation   Parameters of the database creation operation
   * @param {BaseUser} user                       The User requesting the creation operation
   * @returns {Promise<boolean|void>}             Return false to cancel the creation operation entirely
   * @protected
   */
  static async _preCreateOperation(documents, operation, user) {}

  /**
   * Post-process a creation operation, reacting to database changes which have occurred. Post-operation events occur
   * for all connected clients.
   *
   * This batch-wise workflow occurs after individual {@link _onCreate} workflows.
   *
   * @param {Document[]} documents                The Document instances which were created
   * @param {DatabaseCreateOperation} operation   Parameters of the database creation operation
   * @param {BaseUser} user                       The User who performed the creation operation
   * @returns {Promise<void>}
   * @protected
   */
  static async _onCreateOperation(documents, operation, user) {}

  /* -------------------------------------------- */
  /*  Database Update Operations                  */
  /* -------------------------------------------- */

  /**
   * Pre-process an update operation for a single Document instance. Pre-operation events only occur for the client
   * which requested the operation.
   *
   * @param {object} changes            The candidate changes to the Document
   * @param {object} options            Additional options which modify the update request
   * @param {BaseUser} user             The User requesting the document update
   * @returns {Promise<boolean|void>}   A return value of false indicates the update operation should be cancelled.
   * @protected
   */
  async _preUpdate(changes, options, user) {}

  /**
   * Post-process an update operation for a single Document instance. Post-operation events occur for all connected
   * clients.
   *
   * @param {object} changed            The differential data that was changed relative to the documents prior values
   * @param {object} options            Additional options which modify the update request
   * @param {string} userId             The id of the User requesting the document update
   * @protected
   */
  _onUpdate(changed, options, userId) {}

  /**
   * Pre-process an update operation, potentially altering its instructions or input data. Pre-operation events only
   * occur for the client which requested the operation.
   *
   * This batch-wise workflow occurs after individual {@link _preUpdate} workflows and provides a final pre-flight check
   * before a database operation occurs.
   *
   * Modifications to the requested updates are performed by mutating the data array of the operation.
   *
   * @param {Document[]} documents                Document instances to be updated
   * @param {DatabaseUpdateOperation} operation   Parameters of the database update operation
   * @param {BaseUser} user                       The User requesting the update operation
   * @returns {Promise<boolean|void>}             Return false to cancel the update operation entirely
   * @protected
   */
  static async _preUpdateOperation(documents, operation, user) {}

  /**
   * Post-process an update operation, reacting to database changes which have occurred. Post-operation events occur
   * for all connected clients.
   *
   * This batch-wise workflow occurs after individual {@link _onUpdate} workflows.
   *
   * @param {Document[]} documents                The Document instances which were updated
   * @param {DatabaseUpdateOperation} operation   Parameters of the database update operation
   * @param {BaseUser} user                       The User who performed the update operation
   * @returns {Promise<void>}
   * @protected
   */
  static async _onUpdateOperation(documents, operation, user) {}

  /* -------------------------------------------- */
  /*  Database Delete Operations                  */
  /* -------------------------------------------- */

  /**
   * Pre-process a deletion operation for a single Document instance. Pre-operation events only occur for the client
   * which requested the operation.
   *
   * @param {object} options            Additional options which modify the deletion request
   * @param {BaseUser} user             The User requesting the document deletion
   * @returns {Promise<boolean|void>}   A return value of false indicates the deletion operation should be cancelled.
   * @protected
   */
  async _preDelete(options, user) {}

  /**
   * Post-process a deletion operation for a single Document instance. Post-operation events occur for all connected
   * clients.
   *
   * @param {object} options            Additional options which modify the deletion request
   * @param {string} userId             The id of the User requesting the document update
   * @protected
   */
  _onDelete(options, userId) {}

  /**
   * Pre-process a deletion operation, potentially altering its instructions or input data. Pre-operation events only
   * occur for the client which requested the operation.
   *
   * This batch-wise workflow occurs after individual {@link _preDelete} workflows and provides a final pre-flight check
   * before a database operation occurs.
   *
   * Modifications to the requested deletions are performed by mutating the operation object.
   * {@link updateSource}.
   *
   * @param {Document[]} documents                Document instances to be deleted
   * @param {DatabaseDeleteOperation} operation   Parameters of the database update operation
   * @param {BaseUser} user                       The User requesting the deletion operation
   * @returns {Promise<boolean|void>}             Return false to cancel the deletion operation entirely
   * @protected
   */
  static async _preDeleteOperation(documents, operation, user) {}

  /**
   * Post-process a deletion operation, reacting to database changes which have occurred. Post-operation events occur
   * for all connected clients.
   *
   * This batch-wise workflow occurs after individual {@link _onDelete} workflows.
   *
   * @param {Document[]} documents                The Document instances which were deleted
   * @param {DatabaseDeleteOperation} operation   Parameters of the database deletion operation
   * @param {BaseUser} user                       The User who performed the deletion operation
   * @returns {Promise<void>}
   * @protected
   */
  static async _onDeleteOperation(documents, operation, user) {}

  /* -------------------------------------------- */
  /*  Deprecations and Compatibility              */
  /* -------------------------------------------- */

  /**
   * A reusable helper for adding migration shims.
   * @param {object} data                       The data object being shimmed
   * @param {{[oldKey: string]: string}} shims  The mapping of old keys to new keys
   * @param {object} [options]                  Options passed to {@link foundry.utils.logCompatibilityWarning}
   * @param {string} [options.warning]          The deprecation message
   * @param {any} [options.value]               The value of the shim
   * @internal
   */
  static _addDataFieldShims(data, shims, options) {
    for ( const [oldKey, newKey] of Object.entries(shims) ) {
      this._addDataFieldShim(data, oldKey, newKey, options);
    }
  }

  /* ---------------------------------------- */

  /**
   * A reusable helper for adding a migration shim
   * The value of the data can be transformed during the migration by an optional application function.
   * @param {object} data               The data object being shimmed
   * @param {string} oldKey             The old field name
   * @param {string} newKey             The new field name
   * @param {object} [options]          Options passed to {@link foundry.utils.logCompatibilityWarning}
   * @param {string} [options.warning]  The deprecation message
   * @param {any} [options.value]       The value of the shim
   * @internal
   */
  static _addDataFieldShim(data, oldKey, newKey, options={}) {
    if ( hasProperty(data, oldKey) ) return;
    let oldTarget = data;
    let oldTargetKey = oldKey;
    if ( oldKey.includes(".") ) {
      const parts = oldKey.split(".");
      oldTarget = getProperty(data, parts.slice(0, -1).join("."));
      oldTargetKey = parts.at(-1);
    }
    Object.defineProperty(oldTarget, oldTargetKey, {
      get: () => {
        if ( options.warning ) logCompatibilityWarning(options.warning, options);
        else this._logDataFieldMigration(oldKey, newKey, options);
        return ("value" in options) ? options.value : getProperty(data, newKey);
      },
      set: value => {
        if ( newKey ) setProperty(data, newKey, value);
      },
      configurable: true
    });
  }

  /* ---------------------------------------- */

  /**
   * Define a simple migration from one field name to another.
   * The value of the data can be transformed during the migration by an optional application function.
   * @param {object} data     The data object being migrated
   * @param {string} oldKey   The old field name
   * @param {string} newKey   The new field name
   * @param {(data: object) => any} [apply]  An application function, otherwise the old value is applied
   * @returns {boolean}       Whether a migration was applied.
   * @internal
   */
  static _addDataFieldMigration(data, oldKey, newKey, apply) {
    if ( !hasProperty(data, newKey) && hasProperty(data, oldKey) ) {
      let oldTarget = data;
      let oldTargetKey = oldKey;
      if ( oldKey.includes(".") ) {
        const parts = oldKey.split(".");
        oldTarget = getProperty(data, parts.slice(0, -1).join("."));
        oldTargetKey = parts.at(-1);
      }
      const oldProp = Object.getOwnPropertyDescriptor(oldTarget, oldTargetKey);
      if ( oldProp && !oldProp.writable ) return false;
      setProperty(data, newKey, apply ? apply(data) : getProperty(data, oldKey));
      deleteProperty(data, oldKey);
      return true;
    }
    return false;
  }

  /* ---------------------------------------- */

  /**
   * Log a compatbility warning for the data field migration.
   * @param {string} oldKey       The old field name
   * @param {string} newKey       The new field name
   * @param {object} [options]    Options passed to {@link foundry.utils.logCompatibilityWarning}
   * @internal
   */
  static _logDataFieldMigration(oldKey, newKey, options={}) {
    const msg = `You are accessing ${this.name}#${oldKey} which has been migrated to ${this.name}#${newKey}`;
    logCompatibilityWarning(msg, options);
  }

  /* ---------------------------------------- */

  /**
   * @callback RecursiveFieldClearCallback
   * @param {object} data       The (partial) Document data.
   * @param {string} fieldName  The name of the field to clear.
   */

  /**
   * Clear the fields from the given Document data recursively.
   * @param {object} data                                     The (partial) Document data
   * @param {string[]} fieldNames                             The fields that are cleared
   * @param {object} [options]
   * @param {RecursiveFieldClearCallback} [options.callback]  A callback that is invoked on each field in order to clear
   *                                                          it.
   * @internal
   */
  static _clearFieldsRecursively(data, fieldNames, options={}) {
    if ( fieldNames.length === 0 ) return;
    const { callback } = options;
    for ( const fieldName of fieldNames ) {
      if ( typeof callback === "function" ) callback(data, fieldName);
      else deleteProperty(data, fieldName);
    }
    for ( const [collectionName, field] of Object.entries(this.hierarchy) ) {
      const collection = data[collectionName];
      if ( !collection ) continue;
      if ( field instanceof foundry.data.fields.EmbeddedDocumentField ) {
        field.model._clearFieldsRecursively(collection, fieldNames, options);
        continue;
      }
      for ( const embeddedData of collection ) {
        if ( embeddedData._tombstone ) continue;
        field.model._clearFieldsRecursively(embeddedData, fieldNames, options);
      }
    }
  }

  /* ---------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  static async _onCreateDocuments(documents, operation) {}

  /**
   * @deprecated since v12
   * @ignore
   */
  static async _onUpdateDocuments(documents, operation) {}

  /**
   * @deprecated since v12
   * @ignore
   */
  static async _onDeleteDocuments(documents, operation) {}
}

/**
 * @import {ActiveEffectData} from "./_types.mjs";
 */

/**
 * The ActiveEffect Document.
 * Defines the DataSchema and common behaviors for an ActiveEffect which are shared between both client and server.
 * @extends {Document<ActiveEffectData>}
 * @mixes ActiveEffectData
 * @category Documents
 */
class BaseActiveEffect extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "ActiveEffect",
    collection: "effects",
    hasTypeData: true,
    label: "DOCUMENT.ActiveEffect",
    labelPlural: "DOCUMENT.ActiveEffects",
    schemaVersion: "13.341",
    permissions: {
      create: "OWNER",
      delete: "OWNER"
    }
  }, {inplace: false}));

  /* -------------------------------------------- */

  /** @inheritdoc */
  static defineSchema() {
    return {
      _id: new DocumentIdField(),
      name: new StringField({required: true, blank: false, textSearch: true}),
      img: new FilePathField({categories: ["IMAGE"]}),
      type: new DocumentTypeField(this, {initial: BASE_DOCUMENT_TYPE}),
      system: new TypeDataField(this),
      changes: new ArrayField(new SchemaField({
        key: new StringField({required: true}),
        value: new StringField({required: true}),
        mode: new NumberField({required: true, nullable: false, integer: true,
          initial: ACTIVE_EFFECT_MODES.ADD}),
        priority: new NumberField()
      })),
      disabled: new BooleanField(),
      duration: new SchemaField({
        startTime: new NumberField({initial: null}),
        seconds: new NumberField({integer: true, min: 0}),
        combat: new ForeignDocumentField(foundry.documents.BaseCombat),
        rounds: new NumberField({integer: true, min: 0}),
        turns: new NumberField({integer: true, min: 0}),
        startRound: new NumberField({integer: true, min: 0}),
        startTurn: new NumberField({integer: true, min: 0})
      }),
      description: new HTMLField({textSearch: true}),
      origin: new StringField({nullable: true, blank: false, initial: null}),
      tint: new ColorField({nullable: false, initial: "#ffffff"}),
      transfer: new BooleanField({initial: true}),
      statuses: new SetField(new StringField({required: true, blank: false})),
      sort: new IntegerSortField(),
      flags: new DocumentFlagsField(),
      _stats: new DocumentStatsField()
    };
  }

  /* -------------------------------------------- */

  /** @override */
  static LOCALIZATION_PREFIXES = ["DOCUMENT", "EFFECT"];

  /* -------------------------------------------- */
  /*  Database Event Handlers                     */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _preCreate(data, options, user) {
    const allowed = await super._preCreate(data, options, user);
    if ( allowed === false ) return false;
    if ( this.parent instanceof foundry.documents.BaseActor ) {
      this.updateSource({transfer: false});
    }
  }

  /* -------------------------------------------- */
  /*  Deprecations and Compatibility              */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static migrateData(data) {
    /**
     * icon -> img
     * @deprecated since v12
     */
    this._addDataFieldMigration(data, "icon", "img");
    return super.migrateData(data);
  }

  /* ---------------------------------------- */

  /** @inheritdoc */
  static shimData(data, options) {
    this._addDataFieldShim(data, "icon", "img", {since: 12, until: 14});
    return super.shimData(data, options);
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  get icon() {
    this.constructor._logDataFieldMigration("icon", "img", {since: 12, until: 14, once: true});
    return this.img;
  }

  /**
   * @deprecated since v12
   * @ignore
   */
  set icon(value) {
    this.constructor._logDataFieldMigration("icon", "img", {since: 12, until: 14, once: true});
    this.img = value;
  }
}

/**
 * @import {ActorDeltaData} from "./_types.mjs";
 * @import BaseActor from "./actor.mjs";
 * @import {DataModelUpdateOptions} from "@common/abstract/_types.mjs";
 */

/**
 * The ActorDelta Document.
 * Defines the DataSchema and common behaviors for an ActorDelta which are shared between both client and server.
 * ActorDeltas store a delta that can be applied to a particular Actor in order to produce a new Actor.
 * @extends {Document<ActorDeltaData>}
 * @mixes ActorDeltaData
 * @category Documents
 */
class BaseActorDelta extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "ActorDelta",
    collection: "delta",
    label: "DOCUMENT.ActorDelta",
    labelPlural: "DOCUMENT.ActorDeltas",
    isEmbedded: true,
    embedded: {
      Item: "items",
      ActiveEffect: "effects"
    },
    permissions: {
      create: "OWNER",
      delete: "OWNER"
    },
    schemaVersion: "13.341"
  }, {inplace: false}));

  /** @override */
  static defineSchema() {
    const {BaseItem, BaseActiveEffect} = foundry.documents;
    return {
      _id: new DocumentIdField(),
      name: new StringField({required: false, nullable: true, initial: null}),
      type: new StringField({required: false, nullable: true, initial: null}),
      img: new FilePathField({categories: ["IMAGE"], nullable: true, initial: null, required: false}),
      system: new ObjectField(),
      items: new EmbeddedCollectionDeltaField(BaseItem),
      effects: new EmbeddedCollectionDeltaField(BaseActiveEffect),
      ownership: new DocumentOwnershipField({required: false, nullable: true, initial: null}),
      flags: new DocumentFlagsField()
    };
  }

  /* -------------------------------------------- */
  /*  Methods                                     */
  /* -------------------------------------------- */

  /** @override */
  getUserLevel(user) {
    user ||= game.user;
    const {INHERIT, NONE} = CONST.DOCUMENT_OWNERSHIP_LEVELS;
    if ( this.ownership ) {
      const level = this.ownership[user.id] ?? this.ownership.default ?? NONE;
      if ( level !== INHERIT ) return level;                                  // Defer inherited for Embedded
    }
    if ( this.parent ) return this.parent.getUserLevel(user);                 // Embedded Documents
    return NONE;                                                              // Otherwise, NONE
  }

  /* -------------------------------------------- */

  /**
   * Retrieve the base actor's collection, if it exists.
   * @param {string} collectionName  The collection name.
   * @returns {Collection}
   */
  getBaseCollection(collectionName) {
    const baseActor = this.parent?.baseActor;
    return baseActor?.getEmbeddedCollection(collectionName);
  }

  /* -------------------------------------------- */

  /**
   * Apply an ActorDelta to an Actor and return the resultant synthetic Actor.
   * @param {ActorDelta} delta    The ActorDelta.
   * @param {BaseActor} baseActor The base Actor.
   * @param {object} [context]    Context to supply to synthetic Actor instantiation.
   * @returns {BaseActor|null}
   */
  static applyDelta(delta, baseActor, context={}) {
    if ( !baseActor ) return null;
    if ( delta.parent?.isLinked ) return baseActor;

    // Get base actor data.
    const cls = getDocumentClass("Actor");
    const actorData = baseActor.toObject();
    const deltaData = delta.toObject();
    delete deltaData._id;

    // Merge embedded collections.
    BaseActorDelta.#mergeEmbeddedCollections(cls, actorData, deltaData);

    // Merge the rest of the delta.
    mergeObject(actorData, deltaData);
    return new cls(actorData, {parent: delta.parent, ...context});
  }

  /* -------------------------------------------- */

  /**
   * Merge delta Document embedded collections with the base Document.
   * @param {typeof Document} documentClass  The parent Document class.
   * @param {object} baseData                The base Document data.
   * @param {object} deltaData               The delta Document data.
   */
  static #mergeEmbeddedCollections(documentClass, baseData, deltaData) {
    for ( const collectionName of Object.keys(documentClass.hierarchy) ) {
      const baseCollection = baseData[collectionName];
      const deltaCollection = deltaData[collectionName];
      baseData[collectionName] = BaseActorDelta.#mergeEmbeddedCollection(baseCollection, deltaCollection);
      delete deltaData[collectionName];
    }
  }

  /* -------------------------------------------- */

  /**
   * Apply an embedded collection delta.
   * @param {object[]} base   The base embedded collection.
   * @param {object[]} delta  The delta embedded collection.
   * @returns {object[]}
   */
  static #mergeEmbeddedCollection(base=[], delta=[]) {
    const deltaIds = new Set();
    const records = [];
    for ( const record of delta ) {
      if ( !record._tombstone ) records.push(record);
      deltaIds.add(record._id);
    }
    for ( const record of base ) {
      if ( !deltaIds.has(record._id) ) records.push(record);
    }
    return records;
  }

  /* -------------------------------------------- */

  /** @override */
  static migrateData(source) {
    return foundry.documents.BaseActor.migrateData(source);
  }

  /* -------------------------------------------- */

  /**
   * Prepare changes to a descendent delta collection.
   * @param {object} changes                  Candidate source changes.
   * @param {DataModelUpdateOptions} options  Options which determine how the new data is merged.
   * @internal
   */
  _prepareDeltaUpdate(changes={}, options={}) {
    for ( const collectionName of Object.keys(this.constructor.hierarchy) ) {
      if ( collectionName in changes ) {
        this.getEmbeddedCollection(collectionName)._prepareDeltaUpdate(changes[collectionName], options);
      }
    }
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  updateSource(changes={}, options={}) {
    this._prepareDeltaUpdate(changes, options);
    return super.updateSource(changes, options);
  }

  /* -------------------------------------------- */
  /*  Serialization                               */
  /* -------------------------------------------- */

  /** @override */
  toObject(source=true) {
    const data = {};
    const value = source ? this._source : this;
    for ( const [name, field] of this.schema.entries() ) {
      const v = value[name];
      if ( !field.required && ((v === undefined) || (v === null)) ) continue; // Drop optional fields
      data[name] = source ? deepClone(value[name]) : field.toObject(value[name]);
    }
    return data;
  }
}

/**
 * The collection of data schema and document definitions for primary documents which are shared between the both the
 * client and the server.
 * @module data
 */


/**
 * @import {EmbeddedCollectionDelta} from "../abstract/_module.mjs";
 * @import BaseActor from "../documents/actor.mjs";
 * @import {PrototypeTokenData} from "@common/documents/_types.mjs";
 * @import {TextureDataFitMode} from "../constants.mjs";
 * @import {DataFieldOptions, FilePathFieldOptions, LightAnimationData} from "./_types.mjs";
 */

/**
 * A reusable document structure for the internal data used to render the appearance of a light source.
 * This is re-used by both the AmbientLightData and TokenData classes.
 *
 * @property {boolean} negative           Is this light source a negative source? (i.e. darkness source)
 * @property {number} priority            The priority of this source
 * @property {number} alpha               An opacity for the emitted light, if any
 * @property {number} angle               The angle of emission for this point source
 * @property {number} bright              The allowed radius of bright vision or illumination
 * @property {number} color               A tint color for the emitted light, if any
 * @property {number} coloration          The coloration technique applied in the shader
 * @property {number} contrast            The amount of contrast this light applies to the background texture
 * @property {number} dim                 The allowed radius of dim vision or illumination
 * @property {number} attenuation         Fade the difference between bright, dim, and dark gradually?
 * @property {number} luminosity          The luminosity applied in the shader
 * @property {number} saturation          The amount of color saturation this light applies to the background texture
 * @property {number} shadows             The depth of shadows this light applies to the background texture
 * @property {LightAnimationData} animation  An animation configuration for the source
 * @property {{min: number, max: number}} darkness  A darkness range (min and max) for which the source should be active
 */
class LightData extends DataModel {
  static defineSchema() {
    return {
      negative: new BooleanField(),
      priority: new NumberField({required: true, nullable: false, integer: true, initial: 0, min: 0}),
      alpha: new AlphaField({initial: 0.5}),
      angle: new AngleField({initial: 360, normalize: false}),
      bright: new NumberField({required: true, nullable: false, initial: 0, min: 0, step: 0.01}),
      color: new ColorField({}),
      coloration: new NumberField({required: true, integer: true, initial: 1}),
      dim: new NumberField({required: true, nullable: false, initial: 0, min: 0, step: 0.01}),
      attenuation: new AlphaField({initial: 0.5}),
      luminosity: new NumberField({required: true, nullable: false, initial: 0.5, min: 0, max: 1}),
      saturation: new NumberField({required: true, nullable: false, initial: 0, min: -1, max: 1}),
      contrast: new NumberField({required: true, nullable: false, initial: 0, min: -1, max: 1}),
      shadows: new NumberField({required: true, nullable: false, initial: 0, min: 0, max: 1}),
      animation: new SchemaField({
        type: new StringField({nullable: true, blank: false, initial: null}),
        speed: new NumberField({required: true, nullable: false, integer: true, initial: 5, min: 0, max: 10,
          validationError: "Light animation speed must be an integer between 0 and 10"}),
        intensity: new NumberField({required: true, nullable: false, integer: true, initial: 5, min: 1, max: 10,
          validationError: "Light animation intensity must be an integer between 1 and 10"}),
        reverse: new BooleanField()
      }),
      darkness: new SchemaField({
        min: new AlphaField({initial: 0}),
        max: new AlphaField({initial: 1})
      }, {
        validate: d => (d.min ?? 0) <= (d.max ?? 1),
        validationError: "darkness.max may not be less than darkness.min"
      })
    };
  }

  /** @override */
  static LOCALIZATION_PREFIXES = ["LIGHT"];

  /* -------------------------------------------- */
  /*  Deprecations and Compatibility              */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static migrateData(data) {
    /**
     * Migration of negative luminosity
     * @deprecated since v12
     */
    const luminosity = data.luminosity;
    if ( luminosity < 0) {
      data.luminosity = 1 - luminosity;
      data.negative = true;
    }
    return super.migrateData(data);
  }
}

/* ---------------------------------------- */

/**
 * A data model intended to be used as an inner EmbeddedDataField which defines a geometric shape.
 *
 * @property {string} type                The type of shape, a value in ShapeData.TYPES.
 *                                        For rectangles, the x/y coordinates are the top-left corner.
 *                                        For circles, the x/y coordinates are the center of the circle.
 *                                        For polygons, the x/y coordinates are the first point of the polygon.
 * @property {number|null} width          For rectangles, the pixel width of the shape.
 * @property {number|null} height         For rectangles, the pixel width of the shape.
 * @property {number|null} radius         For circles, the pixel radius of the shape.
 * @property {number[]} [points]          For polygons, the array of polygon coordinates which comprise the shape.
 */
class ShapeData extends DataModel {
  static defineSchema() {
    return {
      type: new StringField({required: true, blank: false, choices: Object.values(this.TYPES), initial: "r"}),
      width: new NumberField({required: true, integer: true, min: 0, label: "Width"}),
      height: new NumberField({required: true, integer: true, min: 0, label: "Height"}),
      radius: new NumberField({required: true, integer: true, positive: true}),
      points: new ArrayField(new NumberField({required: true, nullable: false}))
    };
  }

  /**
   * The primitive shape types which are supported
   * @enum {string}
   */
  static TYPES = {
    RECTANGLE: "r",
    CIRCLE: "c",
    ELLIPSE: "e",
    POLYGON: "p"
  };
}

/* ---------------------------------------- */

/**
 * A data model intended to be used as an inner EmbeddedDataField which defines a geometric shape.
 * @abstract
 *
 * @property {string} type                                          The type of shape, a value in BaseShapeData.TYPES
 * @property {boolean} [hole=false]                                 Is this shape a hole?
 */
class BaseShapeData extends DataModel {

  /**
   * The possible shape types.
   * @type {Readonly<{
   *   rectangle: RectangleShapeData,
   *   circle: CircleShapeData,
   *   ellipse: EllipseShapeData,
   *   polygon: PolygonShapeData
   * }>}
   */
  static get TYPES() {
    return BaseShapeData.#TYPES ??= Object.freeze({
      [RectangleShapeData.TYPE]: RectangleShapeData,
      [CircleShapeData.TYPE]: CircleShapeData,
      [EllipseShapeData.TYPE]: EllipseShapeData,
      [PolygonShapeData.TYPE]: PolygonShapeData
    });
  }

  static #TYPES;

  /* -------------------------------------------- */

  /**
   * The type of this shape.
   * @type {string}
   */
  static TYPE = "";

  /* -------------------------------------------- */

  /** @override */
  static defineSchema() {
    return {
      type: new StringField({required: true, blank: false, initial: this.TYPE,
        validate: value => value === this.TYPE, validationError: `must be equal to "${this.TYPE}"`}),
      hole: new BooleanField()
    };
  }
}

/* -------------------------------------------- */

/**
 * The data model for a rectangular shape.
 *
 * @property {number} x               The top-left x-coordinate in pixels before rotation.
 * @property {number} y               The top-left y-coordinate in pixels before rotation.
 * @property {number} width           The width of the rectangle in pixels.
 * @property {number} height          The height of the rectangle in pixels.
 * @property {number} rotation        The rotation around the center of the rectangle in degrees.
 */
class RectangleShapeData extends BaseShapeData {

  static {
    Object.defineProperty(this, "TYPE", {value: "rectangle"});
  }

  /** @inheritdoc */
  static defineSchema() {
    return Object.assign(super.defineSchema(), {
      x: new NumberField({required: true, nullable: false, initial: undefined}),
      y: new NumberField({required: true, nullable: false, initial: undefined}),
      width: new NumberField({required: true, nullable: false, initial: undefined, positive: true}),
      height: new NumberField({required: true, nullable: false, initial: undefined, positive: true}),
      rotation: new AngleField()
    });
  }
}

/* -------------------------------------------- */

/**
 * The data model for a circle shape.
 *
 * @property {number} x         The x-coordinate of the center point in pixels.
 * @property {number} y         The y-coordinate of the center point in pixels.
 * @property {number} radius    The radius of the circle in pixels.
 */
class CircleShapeData extends BaseShapeData {

  static {
    Object.defineProperty(this, "TYPE", {value: "circle"});
  }

  /** @inheritdoc */
  static defineSchema() {
    return Object.assign(super.defineSchema(), {
      x: new NumberField({required: true, nullable: false, initial: undefined}),
      y: new NumberField({required: true, nullable: false, initial: undefined}),
      radius: new NumberField({required: true, nullable: false, initial: undefined, positive: true})
    });
  }
}

/* -------------------------------------------- */

/**
 * The data model for an ellipse shape.
 *
 * @property {number} x               The x-coordinate of the center point in pixels.
 * @property {number} y               The y-coordinate of the center point in pixels.
 * @property {number} radiusX         The x-radius of the circle in pixels.
 * @property {number} radiusY         The y-radius of the circle in pixels.
 * @property {number} rotation        The rotation around the center of the rectangle in degrees.
 */
class EllipseShapeData extends BaseShapeData {

  static {
    Object.defineProperty(this, "TYPE", {value: "ellipse"});
  }

  /** @inheritdoc */
  static defineSchema() {
    return Object.assign(super.defineSchema(), {
      x: new NumberField({required: true, nullable: false, initial: undefined}),
      y: new NumberField({required: true, nullable: false, initial: undefined}),
      radiusX: new NumberField({required: true, nullable: false, initial: undefined, positive: true}),
      radiusY: new NumberField({required: true, nullable: false, initial: undefined, positive: true}),
      rotation: new AngleField()
    });
  }
}

/* -------------------------------------------- */

/**
 * The data model for a polygon shape.
 *
 * @property {number[]} points      The points of the polygon ([x0, y0, x1, y1, ...]).
 *                                  The polygon must not be self-intersecting.
 */
class PolygonShapeData extends BaseShapeData {

  static {
    Object.defineProperty(this, "TYPE", {value: "polygon"});
  }

  /** @inheritdoc */
  static defineSchema() {
    return Object.assign(super.defineSchema(), {
      points: new ArrayField(new NumberField({required: true, nullable: false, initial: undefined}),
        {validate: value => {
          if ( value.length % 2 !== 0 ) throw new Error("must have an even length");
          if ( value.length < 6 ) throw new Error("must have at least 3 points");
        }})
    });
  }
}

/* ---------------------------------------- */

/**
 * A {@link foundry.data.fields.SchemaField} subclass used to represent texture data.
 * @property {string|null} src            The URL of the texture source.
 * @property {number} anchorX             The X coordinate of the texture anchor.
 * @property {number} anchorY             The Y coordinate of the texture anchor.
 * @property {number} scaleX              The scale of the texture in the X dimension.
 * @property {number} scaleY              The scale of the texture in the Y dimension.
 * @property {number} offsetX             The X offset of the texture with (0,0) in the top left.
 * @property {number} offsetY             The Y offset of the texture with (0,0) in the top left.
 * @property {TextureDataFitMode} fit     The texture fit mode.
 * @property {number} rotation            An angle of rotation by which this texture is rotated around its center.
 * @property {string} tint                The tint applied to the texture.
 * @property {number} alphaThreshold      Only pixels with an alpha value at or above this value are consider solid
 *                                        w.r.t. to occlusion testing and light/weather blocking.
 */
class TextureData extends SchemaField {
  /**
   * @param {DataFieldOptions} options        Options which are forwarded to the SchemaField constructor
   * @param {Pick<FilePathFieldOptions, "categories"|"initial"|"wildcard"|"label">} srcOptions
   *                                          Additional options for the src field
   */
  constructor(options={}, {categories=["IMAGE", "VIDEO"], initial={}, wildcard=false, label=""}={}) {
    /** @deprecated since v12 */
    if ( typeof initial === "string" ) {
      const msg = "Passing the initial value of the src field as a string is deprecated. Pass {src} instead.";
      logCompatibilityWarning(msg, {since: 12, until: 14});
      initial = {src: initial};
    }
    super({
      src: new FilePathField({required: true, categories, initial: initial.src ?? null, label,
        virtual: !wildcard, wildcard}),
      anchorX: new NumberField({required: true, nullable: false, initial: initial.anchorX ?? 0}),
      anchorY: new NumberField({required: true, nullable: false, initial: initial.anchorY ?? 0}),
      offsetX: new NumberField({required: true, nullable: false, integer: true, initial: initial.offsetX ?? 0}),
      offsetY: new NumberField({required: true, nullable: false, integer: true, initial: initial.offsetY ?? 0}),
      fit: new StringField({required: true, initial: initial.fit ?? "fill", choices: CONST.TEXTURE_DATA_FIT_MODES}),
      scaleX: new NumberField({required: true, nullable: false, initial: initial.scaleX ?? 1}),
      scaleY: new NumberField({required: true, nullable: false, initial: initial.scaleY ?? 1}),
      rotation: new AngleField({initial: initial.rotation ?? 0}),
      tint: new ColorField({required: true, nullable: false, initial: initial.tint ?? "#ffffff"}),
      alphaThreshold: new AlphaField({nullable: false, initial: initial.alphaThreshold ?? 0})
    }, options);
  }
}

/* ---------------------------------------- */

/**
 * Extend the base TokenData to define a PrototypeToken which exists within a parent Actor.
 * @extends DataModel<PrototypeTokenData>
 * @property {boolean} randomImg      Does the prototype token use a random wildcard image?
 */
class PrototypeToken extends DataModel {
  constructor(data={}, options={}) {
    super(data, options);
    Object.defineProperty(this, "apps", {value: {}});
  }

  /** @override */
  static defineSchema() {
    const schema = foundry.documents.BaseToken.defineSchema();
    const excluded = ["_id", "actorId", "delta", "x", "y", "elevation", "shape", "sort", "hidden", "locked",
      "_movementHistory", "_regions"];
    for ( const fieldName of excluded ) {
      delete schema[fieldName];
    }
    schema.name.textSearch = schema.name.options.textSearch = false;
    schema.randomImg = new BooleanField();
    schema.appendNumber = new BooleanField();
    schema.prependAdjective = new BooleanField();
    return schema;
  }

  /** @override */
  static LOCALIZATION_PREFIXES = ["DOCUMENT", "TOKEN"];

  /**
   * The Actor which owns this Prototype Token
   * @type {BaseActor}
   */
  get actor() {
    return this.parent;
  }

  /** @inheritdoc */
  toObject(source=true) {
    const data = super.toObject(source);
    data.actorId = this.document?.id;
    return data;
  }

  /** @ignore */
  static get database() {
    return globalThis.CONFIG.DatabaseBackend;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _initializeSource(data, options={}) {
    if ( data instanceof PrototypeToken ) data = data.toObject();
    PrototypeTokenOverrides.applyOverrides(data, this.parent?.type);
    return super._initializeSource(data, options);
  }

  /* -------------------------------------------- */
  /*  Document Compatibility Methods              */
  /* -------------------------------------------- */

  /**
   * @see {@link foundry.abstract.Document#update}
   * @ignore
   */
  update(data, options) {
    return this.actor.update({prototypeToken: data}, options);
  }

  /* -------------------------------------------- */

  /**
   * @see {@link foundry.abstract.Document#getFlag}
   * @ignore
   */
  getFlag(...args) {
    return foundry.abstract.Document.prototype.getFlag.call(this, ...args);
  }

  /* -------------------------------------------- */

  /**
   * @see {@link foundry.abstract.Document#getFlag}
   * @ignore
   */
  setFlag(...args) {
    return foundry.abstract.Document.prototype.setFlag.call(this, ...args);
  }

  /* -------------------------------------------- */

  /**
   * @see {@link foundry.abstract.Document#unsetFlag}
   * @ignore
   */
  async unsetFlag(...args) {
    return foundry.abstract.Document.prototype.unsetFlag.call(this, ...args);
  }

  /* -------------------------------------------- */

  /**
   * @see {@link foundry.abstract.Document#testUserPermission}
   * @ignore
   */
  testUserPermission(user, permission, {exact=false}={}) {
    return this.actor.testUserPermission(user, permission, {exact});
  }

  /* -------------------------------------------- */

  /**
   * @see {@link foundry.documents.BaseActor#isOwner}
   * @ignore
   */
  get isOwner() {
    return this.actor.isOwner;
  }
}

/* -------------------------------------------- */

/**
 * The data model for the the core.prototypeTokenOverrides setting.
 */
class PrototypeTokenOverrides extends DataModel {
  /** @override */
  static defineSchema() {
    const labelFor = n => `TOKEN.FIELDS.${n}.label`;
    return getDocumentClass("Actor").TYPES.reduce((types, type) => {
      const prototypeSchema = PrototypeToken.defineSchema();

      // Select a subset of the prototype schema, make all non-recursive fields nullable, and set an initial of null.
      const subSchema = {
        sight: new SchemaField({
          enabled: new BooleanField({required: false, initial: undefined, label: labelFor("sight.enabled")})
        }),
        ring: new SchemaField({
          enabled: new BooleanField({required: false, initial: undefined, label: labelFor("ring.enabled")})
        }),
        turnMarker: prototypeSchema.turnMarker
      };
      for ( const fieldName of ["displayName", "displayBars", "disposition", "lockRotation"] ) {
        const field = prototypeSchema[fieldName];
        field.label = labelFor(fieldName);
        field.required = false;
        field.initial = undefined;
        subSchema[fieldName] = field;
      }
      for (const field of Object.values(subSchema.turnMarker.fields)) {
        field.label = labelFor(`turnMarker.${field.name}`);
        field.required = false;
        field.initial = undefined;
      }
      types[type] = new SchemaField(subSchema);
      return types;
    }, {});
  }

  /* -------------------------------------------- */

  /**
   * Localize all non-recursive data fields on first load of the application.
   * @param {fields.DataField[]} [fields] Subfields of a recursive field
   * @param {Record<string, string>} [cache] A running cache of localization results
   */
  static localizeFields(fields, cache={}) {
    const Cls = PrototypeTokenOverrides;
    if ( Cls.#localized ) return;
    fields ??= Object.values(Cls.schema.fields).flatMap(f => Object.values(f.fields));
    for ( const field of fields ) {
      if ( field instanceof foundry.data.fields.SchemaField ) {
        Cls.localizeFields(Object.values(field.fields, cache));
        continue;
      }
      field.label = (cache[field.label] ??= game.i18n.localize(field.label));
    }
    if ( !fields ) Cls.#localized = true;
  }

  /* -------------------------------------------- */

  /** @override */
  static LOCALIZATION_PREFIXES = ["TOKEN"];

  /**
   * The named of the world setting that stores the prototype token overrides
   * @type {"prototypeTokenOverrides"}
   */
  static SETTING = "prototypeTokenOverrides";

  /**
   * A cached copy of the currently-configured overrides
   * @returns {PrototypeTokenOverrides}
   */
  static get overrides() {
    return this.#overrides ??= game.settings.get("core", PrototypeTokenOverrides.SETTING);
  }

  /**
   * Set or clear the cached overrides.
   * @param {PrototypeTokenOverrides|null} value
   */
  static set overrides(value) {
    this.#overrides = value;
  }

  /**
   * @type {PrototypeTokenOverrides|null}
   */
  static #overrides = null;

  /** Have the fields been localized? Done later since this model may be initialized before i18n is ready. */
  static #localized = false;

  /* -------------------------------------------- */

  /**
   * Apply configured overrides to prototype token data.
   * @param {object} source The prototype token source data on which to operate
   * @param {string} [actorType] The prototype parent's actor type: used to retrieve type-specific overrides
   */
  static applyOverrides(source, actorType) {
    if ( !game.settings || !game.model ) return; // Server-side or Setup
    const setting = PrototypeTokenOverrides.overrides;
    const global = flattenObject(setting.base);
    const typed = flattenObject(setting[actorType] ?? {});
    for ( const [path, baseValue] of Object.entries(global) ) {
      const override = typed[path] ?? baseValue;
      if ( override !== undefined ) setProperty(source, path, override);
    }
  }

  /* -------------------------------------------- */

  /**
   * Apply configured overrides to all Actor documents within the World.
   */
  static applyAll() {
    PrototypeTokenOverrides.#overrides = null;
    for ( const actor of game.actors ) {
      this.applyOverrides(actor.prototypeToken._source, actor.type);
      actor.prototypeToken.reset();
      actor.render();
    }
  }
}

/* -------------------------------------------- */

/**
 * A minimal data model used to represent a tombstone entry inside an {@link foundry.abstract.EmbeddedCollectionDelta}.
 *
 * @property {string} _id              The _id of the base Document that this tombstone represents.
 * @property {boolean} _tombstone      A property that identifies this entry as a tombstone.
 */
class TombstoneData extends DataModel {
  /** @override */
  static defineSchema() {
    return {
      _id: new DocumentIdField(),
      _tombstone: new BooleanField({initial: true, validate: v => v === true, validationError: "must be true"})
    };
  }
}

/**
 * @import {ActorData} from "./_types.mjs";
 * @import {DocumentPermissionTest} from "@common/abstract/_types.mjs";
 */

/**
 * The Actor Document.
 * Defines the DataSchema and common behaviors for an Actor which are shared between both client and server.
 * @extends {Document<ActorData>}
 * @mixes ActorData
 * @category Documents
 */
class BaseActor extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "Actor",
    collection: "actors",
    indexed: true,
    compendiumIndexFields: ["_id", "name", "img", "type", "sort", "folder"],
    embedded: {ActiveEffect: "effects", Item: "items"},
    hasTypeData: true,
    label: "DOCUMENT.Actor",
    labelPlural: "DOCUMENT.Actors",
    permissions: {
      create: this.#canCreate,
      update: this.#canUpdate
    },
    schemaVersion: "13.341"
  }, {inplace: false}));

  /* ---------------------------------------- */

  /** @inheritdoc */
  static defineSchema() {
    const {BaseItem, BaseActiveEffect, BaseFolder} = foundry.documents;
    return {
      _id: new DocumentIdField(),
      name: new StringField({required: true, blank: false, textSearch: true}),
      img: new FilePathField({categories: ["IMAGE"], initial: data => {
        return this.implementation.getDefaultArtwork(data).img;
      }}),
      type: new DocumentTypeField(this),
      system: new TypeDataField(this),
      prototypeToken: new EmbeddedDataField(PrototypeToken),
      items: new EmbeddedCollectionField(BaseItem),
      effects: new EmbeddedCollectionField(BaseActiveEffect),
      folder: new ForeignDocumentField(BaseFolder),
      sort: new IntegerSortField(),
      ownership: new DocumentOwnershipField(),
      flags: new DocumentFlagsField(),
      _stats: new DocumentStatsField()
    };
  }

  /* ---------------------------------------- */

  /**
   * The default icon used for newly created Actor documents.
   * @type {string}
   */
  static DEFAULT_ICON = DEFAULT_TOKEN;

  /* -------------------------------------------- */

  /**
   * Determine default artwork based on the provided actor data.
   * @param {ActorData} actorData                      The source actor data.
   * @returns {{img: string, texture: {src: string}}}  Candidate actor image and prototype token artwork.
   */
  static getDefaultArtwork(actorData) {
    return {
      img: this.DEFAULT_ICON,
      texture: {
        src: this.DEFAULT_ICON
      }
    };
  }

  /* ---------------------------------------- */

  /** @inheritdoc */
  _initializeSource(source, options) {
    source = super._initializeSource(source, options);
    source.prototypeToken.name = source.prototypeToken.name || source.name;
    source.prototypeToken.texture.src = source.prototypeToken.texture.src || source.img;
    return source;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _initialize(options) {
    super._initialize(options);
    DocumentStatsField._shimDocument(this);
  }

  /* -------------------------------------------- */

  /** @override */
  static canUserCreate(user) {
    return user.hasPermission("ACTOR_CREATE");
  }

  /* ---------------------------------------- */

  /**
   * Is a user able to create this actor?
   * @type {DocumentPermissionTest}
   */
  static #canCreate(user, doc) {
    if ( !user.hasPermission("ACTOR_CREATE") ) return false;      // User cannot create actors at all
    if ( doc._source.prototypeToken.randomImg && !user.hasPermission("FILES_BROWSE") ) return false;
    return true;
  }

  /* -------------------------------------------- */

  /**
   * Is a user able to update an existing actor?
   * @type {DocumentPermissionTest}
   */
  static #canUpdate(user, doc, data) {
    if ( !doc.testUserPermission(user, "OWNER") ) return false; // Ownership is required.

    // Users can only enable token wildcard images if they have FILES_BROWSE permission.
    const tokenChange = data?.prototypeToken || {};
    const enablingRandomImage = tokenChange.randomImg === true;
    if ( enablingRandomImage ) return user.hasPermission("FILES_BROWSE");

    // Users can only change a token wildcard path if they have FILES_BROWSE permission.
    const randomImageEnabled = doc._source.prototypeToken.randomImg && (tokenChange.randomImg !== false);
    const changingRandomImage = ("img" in tokenChange) && randomImageEnabled;
    if ( changingRandomImage ) return user.hasPermission("FILES_BROWSE");
    return true;
  }

  /* ---------------------------------------- */

  /** @inheritDoc */
  async _preCreate(data, options, user) {
    const allowed = await super._preCreate(data, options, user);
    if ( allowed === false ) return false;
    if ( !this.prototypeToken.name ) this.prototypeToken.updateSource({name: this.name});
    if ( !this.prototypeToken.texture.src || (this.prototypeToken.texture.src === DEFAULT_TOKEN)) {
      const { texture } = this.constructor.getDefaultArtwork(this.toObject());
      this.prototypeToken.updateSource("img" in data ? { texture: { src: this.img } } : { texture });
    }
  }

  /* ---------------------------------------- */

  /** @inheritDoc */
  async _preUpdate(changed, options, user) {
    const allowed = await super._preUpdate(changed, options, user);
    if ( allowed === false ) return false;
    if ( changed.img && !getProperty(changed, "prototypeToken.texture.src") ) {
      const { texture } = this.constructor.getDefaultArtwork(foundry.utils.mergeObject(this.toObject(), changed));
      if ( !this.prototypeToken.texture.src || (this.prototypeToken.texture.src === texture?.src) ) {
        setProperty(changed, "prototypeToken.texture.src", changed.img);
      }
    }
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static migrateData(source) {
    DocumentStatsField._migrateData(this, source);
    return super.migrateData(source);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static shimData(source, options) {
    DocumentStatsField._shimData(this, source, options);
    return super.shimData(source, options);
  }
}

/**
 * @import {AdventureData} from "./_types.mjs";
 */

/**
 * The Adventure Document.
 * Defines the DataSchema and common behaviors for an Adventure which are shared between both client and server.
 * @extends {Document<AdventureData>}
 * @mixes AdventureData
 * @category Documents
 */
class BaseAdventure extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "Adventure",
    collection: "adventures",
    compendiumIndexFields: ["_id", "name", "description", "img", "sort", "folder"],
    label: "DOCUMENT.Adventure",
    labelPlural: "DOCUMENT.Adventures",
    schemaVersion: "13.341"
  }, {inplace: false}));

  /** @inheritdoc */
  static defineSchema() {
    const documents = foundry.documents;
    return {
      _id: new DocumentIdField(),
      name: new StringField({required: true, blank: false, textSearch: true}),
      img: new FilePathField({categories: ["IMAGE"]}),
      caption: new HTMLField(),
      description: new HTMLField({textSearch: true}),
      actors: new SetField(new EmbeddedDataField(documents.BaseActor)),
      combats: new SetField(new EmbeddedDataField(documents.BaseCombat)),
      items: new SetField(new EmbeddedDataField(documents.BaseItem)),
      journal: new SetField(new EmbeddedDataField(documents.BaseJournalEntry)),
      scenes: new SetField(new EmbeddedDataField(documents.BaseScene)),
      tables: new SetField(new EmbeddedDataField(documents.BaseRollTable)),
      macros: new SetField(new EmbeddedDataField(documents.BaseMacro)),
      cards: new SetField(new EmbeddedDataField(documents.BaseCards)),
      playlists: new SetField(new EmbeddedDataField(documents.BasePlaylist)),
      folders: new SetField(new EmbeddedDataField(documents.BaseFolder)),
      folder: new ForeignDocumentField(documents.BaseFolder),
      sort: new IntegerSortField(),
      flags: new DocumentFlagsField(),
      _stats: new DocumentStatsField()
    };
  }

  /** @override */
  static LOCALIZATION_PREFIXES = ["DOCUMENT", "ADVENTURE"];

  /* -------------------------------------------- */
  /*  Model Properties                            */
  /* -------------------------------------------- */

  /**
   * An array of the fields which provide imported content from the Adventure.
   * @type {Record<string, typeof Document>}
   */
  static get contentFields() {
    const content = {};
    for ( const field of this.schema ) {
      if ( field instanceof SetField ) content[field.name] = field.element.model.implementation;
    }
    return content;
  }

  /**
   * Provide a thumbnail image path used to represent the Adventure document.
   * @type {string}
   */
  get thumbnail() {
    return this.img;
  }
}

/**
 * @import {AmbientLightData} from "./_types.mjs";
 */

/**
 * The AmbientLight Document.
 * Defines the DataSchema and common behaviors for an AmbientLight which are shared between both client and server.
 * @extends {Document<AmbientLightData>}
 * @mixes AmbientLightData
 * @category Documents
 */
class BaseAmbientLight extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "AmbientLight",
    collection: "lights",
    label: "DOCUMENT.AmbientLight",
    labelPlural: "DOCUMENT.AmbientLights",
    schemaVersion: "13.341"
  }, {inplace: false}));

  /** @inheritdoc */
  static defineSchema() {
    return {
      _id: new DocumentIdField(),
      x: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
      y: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
      elevation: new NumberField({required: true, nullable: false, initial: 0}),
      rotation: new AngleField(),
      walls: new BooleanField({initial: true}),
      vision: new BooleanField(),
      config: new EmbeddedDataField(LightData),
      hidden: new BooleanField(),
      flags: new DocumentFlagsField()
    };
  }

  /** @override */
  static LOCALIZATION_PREFIXES = ["DOCUMENT", "AMBIENT_LIGHT"];
}

/**
 * @import {AmbientSoundData} from "./_types.mjs";
 */

/**
 * The AmbientSound Document.
 * Defines the DataSchema and common behaviors for an AmbientSound which are shared between both client and server.
 * @extends {Document<AmbientSoundData>}
 * @mixes AmbientSoundData
 * @category Documents
 */
class BaseAmbientSound extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "AmbientSound",
    collection: "sounds",
    label: "DOCUMENT.AmbientSound",
    labelPlural: "DOCUMENT.AmbientSounds",
    isEmbedded: true,
    schemaVersion: "13.341"
  }, {inplace: false}));

  /** @inheritdoc */
  static defineSchema() {
    const oneToTen = {required: true, nullable: false, integer: true, initial: 5, min: 1, max: 10, step: 1};
    return {
      _id: new DocumentIdField(),
      x: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
      y: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
      elevation: new NumberField({required: true, nullable: false, initial: 0}),
      radius: new NumberField({required: true, nullable: false, initial: 0, min: 0, step: 0.01}),
      path: new FilePathField({categories: ["AUDIO"]}),
      repeat: new BooleanField(),
      volume: new AlphaField({initial: 0.5, step: 0.01}),
      walls: new BooleanField({initial: true}),
      easing: new BooleanField({initial: true}),
      hidden: new BooleanField(),
      darkness: new SchemaField({
        min: new AlphaField({initial: 0}),
        max: new AlphaField({initial: 1})
      }),
      effects: new SchemaField({
        base: new SchemaField({
          type: new StringField(),
          intensity: new NumberField({...oneToTen})
        }),
        muffled: new SchemaField({
          type: new StringField(),
          intensity: new NumberField({...oneToTen})
        })
      }),
      flags: new DocumentFlagsField()
    };
  }

  /** @override */
  static LOCALIZATION_PREFIXES = ["DOCUMENT", "AMBIENT_SOUND"];
}

/**
 * @import {CardData} from "./_types.mjs";
 * @import {DocumentPermissionTest} from "@common/abstract/_types.mjs";
 */

/**
 * The Card Document.
 * Defines the DataSchema and common behaviors for a Card which are shared between both client and server.
 * @extends {Document<CardData>}
 * @mixes CardData
 * @category Documents
 */
class BaseCard extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "Card",
    collection: "cards",
    hasTypeData: true,
    indexed: true,
    label: "DOCUMENT.Card",
    labelPlural: "DOCUMENT.CardPlural",
    permissions: {
      create: this.#canCreate,
      update: this.#canUpdate,
      delete: "OWNER"
    },
    compendiumIndexFields: ["name", "type", "suit", "sort"],
    schemaVersion: "13.341"
  }, {inplace: false}));

  /** @inheritdoc */
  static defineSchema() {
    return {
      _id: new DocumentIdField(),
      name: new StringField({required: true, blank: false, textSearch: true}),
      description: new HTMLField(),
      type: new DocumentTypeField(this, {initial: BASE_DOCUMENT_TYPE}),
      system: new TypeDataField(this),
      suit: new StringField({required: true}),
      value: new NumberField({required: true}),
      back: new SchemaField({
        name: new StringField(),
        text: new HTMLField(),
        img: new FilePathField({categories: ["IMAGE", "VIDEO"]})
      }),
      faces: new ArrayField(new SchemaField({
        name: new StringField(),
        text: new HTMLField(),
        img: new FilePathField({categories: ["IMAGE", "VIDEO"], initial: () => this.DEFAULT_ICON})
      })),
      face: new NumberField({required: true, initial: null, integer: true, min: 0}),
      drawn: new BooleanField(),
      origin: new ForeignDocumentField(foundry.documents.BaseCards),
      width: new NumberField({integer: true, positive: true}),
      height: new NumberField({integer: true, positive: true}),
      rotation: new AngleField(),
      sort: new IntegerSortField(),
      flags: new DocumentFlagsField(),
      _stats: new DocumentStatsField()
    };
  }

  /**
   * The default icon used for a Card face that does not have a custom image set
   * @type {string}
   */
  static DEFAULT_ICON = "icons/svg/card-joker.svg";

  /** @override */
  static LOCALIZATION_PREFIXES = ["DOCUMENT", "CARD"];

  /* -------------------------------------------- */

  /**
   * Is a User able to create a new Card within this parent?
   * @type {DocumentPermissionTest}
   */
  static #canCreate(user, doc, data) {
    if ( user.isGM ) return true;                             // GM users can always create
    if ( doc.parent.type !== "deck" ) return true;            // Users can pass cards to card hands or piles
    return doc.parent.testUserPermission(user, "OWNER");      // Otherwise require owner permission of the parent document
  }

  /* -------------------------------------------- */

  /**
   * Is a user able to update an existing Card?
   * @type {DocumentPermissionTest}
   */
  static #canUpdate(user, doc, data) {
    if ( user.isGM ) return true;                               // GM users can always update
    const wasDrawn = new Set(["drawn", "_id"]);                 // Users can draw cards from a deck
    if ( new Set(Object.keys(data)).equals(wasDrawn) ) return true;
    return doc.parent.testUserPermission(user, "OWNER");        // Otherwise require owner permission of the parent document
  }
}

/**
 * @import {CardsData} from "./_types.mjs";
 */

/**
 * The Cards Document.
 * Defines the DataSchema and common behaviors for a Cards Document which are shared between both client and server.
 * @extends {Document<CardsData>}
 * @mixes CardsData
 * @category Documents
 */
class BaseCards extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "Cards",
    collection: "cards",
    indexed: true,
    compendiumIndexFields: ["_id", "name", "description", "img", "type", "sort", "folder"],
    embedded: {Card: "cards"},
    hasTypeData: true,
    label: "DOCUMENT.Cards",
    labelPlural: "DOCUMENT.CardsPlural",
    permissions: {
      create: "CARDS_CREATE",
      delete: "OWNER"
    },
    coreTypes: ["deck", "hand", "pile"],
    schemaVersion: "13.341"
  }, {inplace: false}));

  /** @inheritdoc */
  static defineSchema() {
    const {BaseCard, BaseFolder} = foundry.documents;
    return {
      _id: new DocumentIdField(),
      name: new StringField({required: true, blank: false, textSearch: true}),
      type: new DocumentTypeField(this),
      description: new HTMLField({textSearch: true}),
      img: new FilePathField({categories: ["IMAGE", "VIDEO"], initial: () => this.DEFAULT_ICON}),
      system: new TypeDataField(this),
      cards: new EmbeddedCollectionField(BaseCard),
      width: new NumberField({integer: true, positive: true}),
      height: new NumberField({integer: true, positive: true}),
      rotation: new AngleField(),
      displayCount: new BooleanField(),
      folder: new ForeignDocumentField(BaseFolder),
      sort: new IntegerSortField(),
      ownership: new DocumentOwnershipField(),
      flags: new DocumentFlagsField(),
      _stats: new DocumentStatsField()
    };
  }

  /** @override */
  static LOCALIZATION_PREFIXES = ["DOCUMENT", "CARDS"];

  /**
   * The default icon used for a cards stack that does not have a custom image set
   * @type {string}
   */
  static DEFAULT_ICON = "icons/svg/card-hand.svg";

  /* -------------------------------------------- */

  /** @inheritDoc */
  _initialize(options) {
    super._initialize(options);
    DocumentStatsField._shimDocument(this);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static migrateData(source) {
    DocumentStatsField._migrateData(this, source);
    return super.migrateData(source);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static shimData(source, options) {
    DocumentStatsField._shimData(this, source, options);
    return super.shimData(source, options);
  }
}

/**
 * @import {ChatMessageData} from "./_types.mjs";
 * @import {DocumentPermissionTest} from "@common/abstract/_types.mjs";
 */

/**
 * The ChatMessage Document.
 * Defines the DataSchema and common behaviors for a ChatMessage which are shared between both client and server.
 * @extends {Document<ChatMessageData>}
 * @mixes ChatMessageData
 * @category Documents
 */
class BaseChatMessage extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "ChatMessage",
    collection: "messages",
    label: "DOCUMENT.ChatMessage",
    labelPlural: "DOCUMENT.ChatMessages",
    hasTypeData: true,
    isPrimary: true,
    permissions: {
      create: this.#canCreate,
      delete: "OWNER"
    },
    schemaVersion: "13.341"
  }, {inplace: false}));

  /** @inheritdoc */
  static defineSchema() {
    const documents = foundry.documents;
    return {
      _id: new DocumentIdField(),
      type: new DocumentTypeField(this, {initial: BASE_DOCUMENT_TYPE}),
      system: new TypeDataField(this),
      style: new NumberField({required: true, choices: Object.values(CHAT_MESSAGE_STYLES),
        initial: CHAT_MESSAGE_STYLES.OTHER, validationError: "must be a value in CONST.CHAT_MESSAGE_STYLES"}),
      author: new DocumentAuthorField(documents.BaseUser),
      timestamp: new NumberField({required: true, nullable: false, initial: Date.now}),
      flavor: new HTMLField(),
      title: new StringField(),
      content: new HTMLField({textSearch: true}),
      speaker: new SchemaField({
        scene: new ForeignDocumentField(documents.BaseScene, {idOnly: true}),
        actor: new ForeignDocumentField(documents.BaseActor, {idOnly: true}),
        token: new ForeignDocumentField(documents.BaseToken, {idOnly: true}),
        alias: new StringField()
      }),
      whisper: new ArrayField(new ForeignDocumentField(documents.BaseUser, {idOnly: true})),
      blind: new BooleanField(),
      rolls: new ArrayField(new JSONField({validate: BaseChatMessage.#validateRoll})),
      sound: new FilePathField({categories: ["AUDIO"]}),
      emote: new BooleanField(),
      flags: new DocumentFlagsField(),
      _stats: new DocumentStatsField()
    };
  }

  /**
   * Is a user able to create a new chat message?
   * @type {DocumentPermissionTest}
   */
  static #canCreate(user, doc) {
    if ( user.isGM ) return true;
    if ( user.id !== doc._source.author ) return false; // You cannot impersonate a different user
    return user.hasRole("PLAYER");                      // Any player can create messages
  }

  /* -------------------------------------------- */

  /**
   * Validate that Rolls belonging to the ChatMessage document are valid
   * @param {string} rollJSON     The serialized Roll data
   */
  static #validateRoll(rollJSON) {
    const roll = JSON.parse(rollJSON);
    if ( !roll.evaluated ) throw new Error("Roll objects added to ChatMessage documents must be evaluated");
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  getUserLevel(user) {
    user ||= game.user;
    if ( user.id === this._source.author ) return DOCUMENT_OWNERSHIP_LEVELS.OWNER;
    return super.getUserLevel(user);
  }

  /* -------------------------------------------- */
  /*  Deprecations and Compatibility              */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static migrateData(data) {
    /**
     * V12 migration from user to author
     * @deprecated since v12
     */
    this._addDataFieldMigration(data, "user", "author");
    BaseChatMessage.#migrateTypeToStyle(data);
    return super.migrateData(data);
  }

  /* ---------------------------------------- */

  /**
   * Migrate the type field to the style field in order to allow the type field to be used for system sub-types.
   * @param {Partial<ChatMessageData>} data
   */
  static #migrateTypeToStyle(data) {
    if ( (typeof data.type !== "number") || ("style" in data) ) return;
    // WHISPER, ROLL, and any other invalid style are redirected to OTHER
    data.style = Object.values(CHAT_MESSAGE_STYLES).includes(data.type) ? data.type : 0;
    data.type = BASE_DOCUMENT_TYPE;
  }

  /* ---------------------------------------- */

  /** @inheritdoc */
  static shimData(data, options) {
    this._addDataFieldShim(data, "user", "author", {since: 12, until: 14});
    return super.shimData(data, options);
  }

  /* ---------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  get user() {
    this.constructor._logDataFieldMigration("user", "author", {since: 12, until: 14});
    return this.author;
  }
}

/**
 * @import {CombatData, CombatantData} from "./_types.mjs";
 * @import {DocumentPermissionTest} from "@common/abstract/_types.mjs";
 */

/**
 * The Combat Document.
 * Defines the DataSchema and common behaviors for a Combat which are shared between both client and server.
 * @extends {Document<CombatData>}
 * @mixes CombatData
 * @category Documents
 */
class BaseCombat extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "Combat",
    collection: "combats",
    label: "DOCUMENT.Combat",
    labelPlural: "DOCUMENT.Combats",
    embedded: {
      Combatant: "combatants",
      CombatantGroup: "groups"
    },
    hasTypeData: true,
    permissions: {
      update: this.#canUpdate
    },
    schemaVersion: "13.341"
  }, {inplace: false}));

  /* -------------------------------------------- */

  /** @override */
  static LOCALIZATION_PREFIXES = ["DOCUMENT", "COMBAT"];

  /* -------------------------------------------- */

  /** @inheritdoc */
  static defineSchema() {
    const {BaseScene, BaseCombatant, BaseCombatantGroup} = foundry.documents;
    return {
      _id: new DocumentIdField(),
      type: new DocumentTypeField(this, {initial: BASE_DOCUMENT_TYPE}),
      system: new TypeDataField(this),
      scene: new ForeignDocumentField(BaseScene),
      groups: new EmbeddedCollectionField(BaseCombatantGroup),
      combatants: new EmbeddedCollectionField(BaseCombatant),
      active: new BooleanField(),
      round: new NumberField({required: true, nullable: false, integer: true, min: 0, initial: 0}),
      turn: new NumberField({required: true, integer: true, min: 0, initial: null}),
      sort: new IntegerSortField(),
      flags: new DocumentFlagsField(),
      _stats: new DocumentStatsField()
    };
  }

  /* -------------------------------------------- */

  /**
   * Is a user able to update an existing Combat?
   * @type {DocumentPermissionTest}
   */
  static #canUpdate(user, doc, data) {
    if ( user.isGM ) return true;                             // GM users can do anything
    const turnOnly = ["_id", "round", "turn", "combatants"];  // Players may only modify a subset of fields
    if ( Object.keys(data).some(k => !turnOnly.includes(k)) ) return false;
    if ( ("round" in data) && !doc._canChangeRound(user) ) return false;
    if ( ("turn" in data) && !doc._canChangeTurn(user) ) return false;
    if ( ("combatants" in data) && !doc.#canModifyCombatants(user, data.combatants) ) return false;
    return true;
  }

  /* -------------------------------------------- */

  /**
   * Can a certain User change the Combat round?
   * @param {User} user     The user attempting to change the round
   * @returns {boolean}     Is the user allowed to change the round?
   * @protected
   */
  _canChangeRound(user) {
    return true;
  }

  /* -------------------------------------------- */

  /**
   * Can a certain User change the Combat turn?
   * @param {documents.User} user The user attempting to change the turn
   * @returns {boolean} Is the user allowed to change the turn?
   * @protected
   */
  _canChangeTurn(user) {
    return true;
  }

  /* -------------------------------------------- */

  /**
   * Can a certain user make modifications to the array of Combatants?
   * @param {documents.User} user                 The user attempting to modify combatants
   * @param {Partial<CombatantData>[]} combatants Proposed combatant changes
   * @returns {boolean} Is the user allowed to make this change?
   */
  #canModifyCombatants(user, combatants) {
    for ( const {_id, ...change} of combatants ) {
      const c = this.combatants.get(_id);
      if ( !c ) return false;
      if ( !c.canUserModify(user, "update", change) ) return false;
    }
    return true;
  }

  /* -------------------------------------------- */
  /*  Event Handlers                              */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _preUpdate(changed, options, user) {
    const allowed = await super._preUpdate(changed, options, user);
    if ( allowed === false ) return false;
    // Don't allow linking to a Scene that doesn't contain all its Combatants
    if ( !("scene" in changed) ) return;
    const sceneId = this.schema.fields.scene.clean(changed.scene);
    if ( (sceneId !== null) && isValidId(sceneId)
      && this.combatants.some(c => c.sceneId && (c.sceneId !== sceneId)) ) {
      throw new Error("You cannot link the Combat to a Scene that doesn't contain all its Combatants.");
    }
  }
}

/**
 * @import {CombatantData} from "./_types.mjs";
 * @import {DocumentPermissionTest} from "@common/abstract/_types.mjs";
 */

/**
 * The Combatant Document.
 * Defines the DataSchema and common behaviors for a Combatant which are shared between both client and server.
 * @extends {Document<CombatantData>}
 * @mixes CombatantData
 * @category Documents
 */
class BaseCombatant extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "Combatant",
    collection: "combatants",
    label: "DOCUMENT.Combatant",
    labelPlural: "DOCUMENT.Combatants",
    isEmbedded: true,
    hasTypeData: true,
    permissions: {
      create: "OWNER",
      update: this.#canUpdate,
      delete: "OWNER"
    },
    schemaVersion: "13.341"
  }, {inplace: false}));

  /* -------------------------------------------- */

  /** @inheritdoc */
  static defineSchema() {
    const {BaseActor, BaseToken, BaseScene} = foundry.documents;
    return {
      _id: new DocumentIdField(),
      type: new DocumentTypeField(this, {initial: BASE_DOCUMENT_TYPE}),
      system: new TypeDataField(this),
      actorId: new ForeignDocumentField(BaseActor, {label: "COMBAT.CombatantActor", idOnly: true}),
      tokenId: new ForeignDocumentField(BaseToken, {label: "COMBAT.CombatantToken", idOnly: true}),
      sceneId: new ForeignDocumentField(BaseScene, {label: "COMBAT.CombatantScene", idOnly: true}),
      name: new StringField({label: "COMBAT.CombatantName", textSearch: true}),
      img: new FilePathField({categories: ["IMAGE"], label: "COMBAT.CombatantImage"}),
      initiative: new NumberField({required: true, label: "COMBAT.CombatantInitiative"}),
      hidden: new BooleanField({label: "COMBAT.CombatantHidden"}),
      defeated: new BooleanField({label: "COMBAT.CombatantDefeated"}),
      group: new DocumentIdField({readonly: false}),
      flags: new DocumentFlagsField(),
      _stats: new DocumentStatsField()
    };
  }

  /* -------------------------------------------- */

  /**
   * Is a user able to update an existing Combatant?
   * @type {DocumentPermissionTest}
   */
  static #canUpdate(user, doc, data) {
    if ( user.isGM ) return true; // GM users can do anything
    if ( !doc.testUserPermission(user, "OWNER") ) return false;

    // Players may only update a subset of fields
    const updateKeys = Object.keys(data);
    const allowedKeys = ["_id", "initiative", "flags", "defeated", "system"];
    return updateKeys.every(k => allowedKeys.includes(k));
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  getUserLevel(user) {
    if ( this.actor ) return this.actor.getUserLevel(user);
    return super.getUserLevel(user);
  }
}

/**
 * @import {CombatantGroupData} from "./_types.mjs";
 */

/**
 * A Document that represents a grouping of individual Combatants in a Combat.
 * Defines the DataSchema and common behaviors for a CombatantGroup which are shared between both client and server.
 * @extends {Document<CombatantGroupData>}
 * @mixes CombatantGroupData
 * @category Documents
 */
class BaseCombatantGroup extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "CombatantGroup",
    collection: "groups",
    label: "DOCUMENT.CombatantGroup",
    labelPlural: "DOCUMENT.CombatantGroups",
    isEmbedded: true,
    hasTypeData: true,
    schemaVersion: "13.341"
  }, { inplace: false }));

  /* -------------------------------------------- */

  /** @inheritDoc */
  static defineSchema() {
    return {
      _id: new DocumentIdField(),
      type: new DocumentTypeField(this, { initial: BASE_DOCUMENT_TYPE }),
      system: new TypeDataField(this),
      name: new StringField({ textSearch: true }),
      img: new FilePathField({ categories: ["IMAGE"] }),
      initiative: new NumberField({ required: true }),
      ownership: new DocumentOwnershipField(),
      flags: new DocumentFlagsField(),
      _stats: new DocumentStatsField()
    };
  }
}

/**
 * @import {DrawingData} from "./_types.mjs";
 * @import {DocumentPermissionTest} from "@common/abstract/_types.mjs";
 */

/**
 * The Drawing Document.
 * Defines the DataSchema and common behaviors for a Drawing which are shared between both client and server.
 * @extends {Document<DrawingData>}
 * @mixes DrawingData
 * @category Documents
 */
class BaseDrawing extends Document {

  /* ---------------------------------------- */
  /*  Model Configuration                     */
  /* ---------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "Drawing",
    collection: "drawings",
    label: "DOCUMENT.Drawing",
    labelPlural: "DOCUMENT.Drawings",
    isEmbedded: true,
    permissions: {
      create: this.#canCreate,
      delete: "OWNER"
    },
    schemaVersion: "13.341"
  }, {inplace: false}));

  /* -------------------------------------------- */

  /** @override */
  static LOCALIZATION_PREFIXES = ["DOCUMENT", "DRAWING"];

  /* ---------------------------------------- */

  /** @inheritDoc */
  static defineSchema() {
    const fillTypeChoices = Object.entries(DRAWING_FILL_TYPES).reduce((types, [key, value]) => {
      types[value] = `DRAWING.FillType${key.titleCase()}`;
      return types;
    }, {});
    return {
      _id: new DocumentIdField(),
      author: new DocumentAuthorField(foundry.documents.BaseUser),
      shape: new EmbeddedDataField(ShapeData),
      x: new NumberField({required: true, nullable: false, initial: 0}),
      y: new NumberField({required: true, nullable: false, initial: 0}),
      elevation: new NumberField({required: true, nullable: false, initial: 0}),
      sort: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
      rotation: new AngleField(),
      bezierFactor: new AlphaField({initial: 0, label: "DRAWING.SmoothingFactor", max: 0.5,
        hint: "DRAWING.SmoothingFactorHint"}),
      fillType: new NumberField({
        required: true,
        nullable: false,
        initial: DRAWING_FILL_TYPES.NONE,
        choices: fillTypeChoices,
        label: "DRAWING.FillTypes",
        validationError: "must be a value in CONST.DRAWING_FILL_TYPES"
      }),
      fillColor: new ColorField({nullable: false, initial: () => game.user?.color.css || "#ffffff", label: "DRAWING.FillColor"}),
      fillAlpha: new AlphaField({initial: 0.5, label: "DRAWING.FillOpacity"}),
      strokeWidth: new NumberField({nullable: false, integer: true, initial: 8, min: 0, label: "DRAWING.LineWidth"}),
      strokeColor: new ColorField({nullable: false, initial: () => game.user?.color.css || "#ffffff", label: "DRAWING.StrokeColor"}),
      strokeAlpha: new AlphaField({initial: 1, label: "DRAWING.LineOpacity"}),
      texture: new FilePathField({categories: ["IMAGE"], label: "DRAWING.FillTexture"}),
      text: new StringField({label: "DRAWING.TextLabel"}),
      fontFamily: new StringField({blank: false, label: "DRAWING.FontFamily",
        initial: () => globalThis.CONFIG?.defaultFontFamily || "Signika"}),
      fontSize: new NumberField({nullable: false, integer: true, min: 8, max: 256, initial: 48, label: "DRAWING.FontSize",
        validationError: "must be an integer between 8 and 256"}),
      textColor: new ColorField({nullable: false, initial: "#ffffff", label: "DRAWING.TextColor"}),
      textAlpha: new AlphaField({label: "DRAWING.TextOpacity"}),
      hidden: new BooleanField(),
      locked: new BooleanField(),
      interface: new BooleanField(),
      flags: new DocumentFlagsField()
    };
  }

  /* ---------------------------------------- */

  /**
   * Validate whether the drawing has some visible content (as required by validation).
   * @param {Partial<Pick<DrawingData, "shape">> & Pick<DrawingData, "text"|"textAlpha"|"fillType"|"fillAlpha"
   *   |"strokeWidth"|"strokeAlpha">} data
   * @returns {boolean}
   * @internal
   */
  static _validateVisibleContent(data) {
    let isEmpty;
    switch ( data.shape?.type ) {
      case "r":
      case "e": isEmpty = (data.shape.width <= data.strokeWidth) || (data.shape.height <= data.strokeWidth); break;
      case "p": isEmpty = ((data.shape.width === 0) && (data.shape.height === 0)) || (data.shape.points.length < 4); break;
      case "c": isEmpty = (data.shape.radius <= (data.strokeWidth / 2)); break;
    }
    if ( isEmpty ) return false;
    const hasText = (data.text !== "") && (data.textAlpha > 0);
    const hasFill = (data.fillType !== DRAWING_FILL_TYPES.NONE) && (data.fillAlpha > 0);
    const hasLine = (data.strokeWidth > 0) && (data.strokeAlpha > 0);
    return hasText || hasFill || hasLine;
  }

  /* ---------------------------------------- */

  /** @inheritdoc */
  static validateJoint(data) {
    if ( !BaseDrawing._validateVisibleContent(data) ) {
      throw new Error(game.i18n.localize("DRAWING.JointValidationError"));
    }
  }

  /* -------------------------------------------- */

  /** @override */
  static canUserCreate(user) {
    return user.hasPermission("DRAWING_CREATE");
  }

  /* ---------------------------------------- */

  /**
   * Is a user able to create a new Drawing?
   * @type {DocumentPermissionTest}
   */
  static #canCreate(user, doc) {
    if ( !user.isGM && (doc._source.author !== user.id) ) return false;
    return user.hasPermission("DRAWING_CREATE");
  }

  /* ---------------------------------------- */
  /*  Model Methods                           */
  /* ---------------------------------------- */

  /** @inheritDoc */
  getUserLevel(user) {
    user ||= game.user;
    if ( user.id === this._source.author ) return DOCUMENT_OWNERSHIP_LEVELS.OWNER;
    return super.getUserLevel(user);
  }

  /* ---------------------------------------- */
  /*  Deprecations and Compatibility          */
  /* ---------------------------------------- */

  /** @inheritdoc */
  static migrateData(data) {
    /**
     * V12 migration to elevation and sort fields
     * @deprecated since v12
     */
    this._addDataFieldMigration(data, "z", "elevation");
    return super.migrateData(data);
  }

  /* ---------------------------------------- */

  /** @inheritdoc */
  static shimData(data, options) {
    this._addDataFieldShim(data, "z", "elevation", {since: 12, until: 14});
    return super.shimData(data, options);
  }

  /* ---------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  get z() {
    this.constructor._logDataFieldMigration("z", "elevation", {since: 12, until: 14});
    return this.elevation;
  }
}

/**
 * @import {FogExplorationData} from "./_types.mjs";
 * @import {DocumentPermissionTest} from "@common/abstract/_types.mjs";
 */

/**
 * The FogExploration Document.
 * Defines the DataSchema and common behaviors for a FogExploration which are shared between both client and server.
 * @extends {Document<FogExplorationData>}
 * @mixes FogExplorationData
 * @category Documents
 */
class BaseFogExploration extends Document {

  /* ---------------------------------------- */
  /*  Model Configuration                     */
  /* ---------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "FogExploration",
    collection: "fog",
    label: "DOCUMENT.FogExploration",
    labelPlural: "DOCUMENT.FogExplorations",
    isPrimary: true,
    permissions: {
      create: "PLAYER",
      update: this.#canModify,
      delete: this.#canModify
    },
    schemaVersion: "13.341"
  }, {inplace: false}));

  /** @inheritdoc */
  static defineSchema() {
    const {BaseScene, BaseUser} = foundry.documents;
    return {
      _id: new DocumentIdField(),
      scene: new ForeignDocumentField(BaseScene, {initial: () => canvas?.scene?.id}),
      user: new ForeignDocumentField(BaseUser, {initial: () => game?.user?.id}),
      explored: new FilePathField({categories: ["IMAGE"], required: true, base64: true}),
      positions: new ObjectField(),
      timestamp: new NumberField({nullable: false, initial: Date.now}),
      flags: new DocumentFlagsField(),
      _stats: new DocumentStatsField()
    };
  }

  /**
   * Test whether a User can modify a FogExploration document.
   * @type {DocumentPermissionTest}
   */
  static #canModify(user, doc) {
    return (user.id === doc._source.user) || user.hasRole("ASSISTANT");
  }

  /* ---------------------------------------- */
  /*  Database Event Handlers                 */
  /* ---------------------------------------- */

  /** @inheritDoc */
  async _preUpdate(changed, options, user) {
    const allowed = await super._preUpdate(changed, options, user);
    if ( allowed === false ) return false;
    changed.timestamp = Date.now();
  }
}

/**
 * @import {FolderData} from "./_types.mjs";
 */

/**
 * The Folder Document.
 * Defines the DataSchema and common behaviors for a Folder which are shared between both client and server.
 * @extends {Document<FolderData>}
 * @mixes FolderData
 * @category Documents
 */
class BaseFolder extends Document {

  /* ---------------------------------------- */
  /*  Model Configuration                     */
  /* ---------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "Folder",
    collection: "folders",
    label: "DOCUMENT.Folder",
    labelPlural: "DOCUMENT.Folders",
    coreTypes: FOLDER_DOCUMENT_TYPES,
    schemaVersion: "13.341"
  }, {inplace: false}));

  /** @inheritdoc */
  static defineSchema() {
    return {
      _id: new DocumentIdField(),
      name: new StringField({required: true, blank: false, textSearch: true}),
      type: new DocumentTypeField(this),
      description: new HTMLField({textSearch: true}),
      folder: new ForeignDocumentField(BaseFolder),
      sorting: new StringField({required: true, initial: "a", choices: this.SORTING_MODES}),
      sort: new IntegerSortField(),
      color: new ColorField(),
      flags: new DocumentFlagsField(),
      _stats: new DocumentStatsField()
    };
  }

  /** @override */
  static LOCALIZATION_PREFIXES = ["DOCUMENT", "FOLDER"];

  /** @inheritdoc */
  static validateJoint(data) {
    if ( (data.folder !== null) && (data.folder === data._id) ) {
      throw new Error("A Folder may not contain itself");
    }
  }

  /**
   * Allow folder sorting modes
   * @type {string[]}
   */
  static SORTING_MODES = ["a", "m"];

  /* -------------------------------------------- */

  /** @override */
  static get(documentId, options={}) {
    if ( !documentId ) return null;
    if ( !options.pack ) return super.get(documentId, options);
    const pack = game.packs.get(options.pack);
    if ( !pack ) {
      console.error(`The ${this.name} model references a non-existent pack ${options.pack}.`);
      return null;
    }
    return pack.folders.get(documentId);
  }
}

/**
 * @import {ItemData} from "./_types.mjs";
 * @import {DocumentPermissionTest} from "@common/abstract/_types.mjs";
 */

/**
 * The Item Document.
 * Defines the DataSchema and common behaviors for a Item which are shared between both client and server.
 * @extends {Document<ItemData>}
 * @mixes ItemData
 * @category Documents
 */
class BaseItem extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "Item",
    collection: "items",
    hasTypeData: true,
    indexed: true,
    compendiumIndexFields: ["_id", "name", "img", "type", "sort", "folder"],
    embedded: {ActiveEffect: "effects"},
    label: "DOCUMENT.Item",
    labelPlural: "DOCUMENT.Items",
    permissions: {
      create: BaseItem.#canCreate,
      delete: "OWNER"
    },
    schemaVersion: "13.341"
  }, {inplace: false}));

  /* ---------------------------------------- */

  /** @inheritdoc */
  static defineSchema() {
    const {BaseActiveEffect, BaseFolder} = foundry.documents;
    return {
      _id: new DocumentIdField(),
      name: new StringField({required: true, blank: false, textSearch: true}),
      type: new DocumentTypeField(this),
      img: new FilePathField({categories: ["IMAGE"], initial: data => {
        return this.implementation.getDefaultArtwork(data).img;
      }}),
      system: new TypeDataField(this),
      effects: new EmbeddedCollectionField(BaseActiveEffect),
      folder: new ForeignDocumentField(BaseFolder),
      sort: new IntegerSortField(),
      ownership: new DocumentOwnershipField(),
      flags: new DocumentFlagsField(),
      _stats: new DocumentStatsField()
    };
  }

  /* ---------------------------------------- */

  /**
   * The default icon used for newly created Item documents
   * @type {string}
   */
  static DEFAULT_ICON = "icons/svg/item-bag.svg";

  /* -------------------------------------------- */

  /**
   * Determine default artwork based on the provided item data.
   * @param {ItemData} itemData  The source item data.
   * @returns {{img: string}}    Candidate item image.
   */
  static getDefaultArtwork(itemData) {
    return { img: this.DEFAULT_ICON };
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _initialize(options) {
    super._initialize(options);
    DocumentStatsField._shimDocument(this);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  getUserLevel(user) {
    // Embedded Items require a special exception because they ignore their own ownership field.
    if ( this.parent ) return this.parent.getUserLevel(user);
    return super.getUserLevel(user);
  }

  /* -------------------------------------------- */

  /** @override */
  static canUserCreate(user) {
    return user.hasPermission("ITEM_CREATE");
  }

  /* -------------------------------------------- */

  /**
   * Is a User able to create a new Item?
   * Embedded Items depend on Actor ownership.
   * Otherwise, the ITEM_CREATE permission is required.
   * @type {DocumentPermissionTest}
   */
  static #canCreate(user, doc) {
    if ( doc.parent ) return doc.parent.testUserPermission(user, "OWNER");
    return user.hasPermission("ITEM_CREATE");
  }

  /* -------------------------------------------- */


  /** @inheritDoc */
  static migrateData(source) {
    DocumentStatsField._migrateData(this, source);
    return super.migrateData(source);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static shimData(source, options) {
    DocumentStatsField._shimData(this, source, options);
    return super.shimData(source, options);
  }
}

/**
 * @import {JournalEntryData} from "./_types.mjs";
 */

/**
 * The JournalEntry Document.
 * Defines the DataSchema and common behaviors for a JournalEntry which are shared between both client and server.
 * @extends {Document<JournalEntryData>}
 * @mixes JournalEntryData
 * @category Documents
 */
class BaseJournalEntry extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "JournalEntry",
    collection: "journal",
    indexed: true,
    compendiumIndexFields: ["_id", "name", "sort", "folder"],
    embedded: {
      JournalEntryCategory: "categories",
      JournalEntryPage: "pages"
    },
    label: "DOCUMENT.JournalEntry",
    labelPlural: "DOCUMENT.JournalEntries",
    permissions: {
      create: "JOURNAL_CREATE",
      delete: "OWNER"
    },
    schemaVersion: "13.341"
  }, {inplace: false}));

  /** @inheritdoc */
  static defineSchema() {
    const {BaseJournalEntryPage, BaseJournalEntryCategory, BaseFolder} = foundry.documents;
    return {
      _id: new DocumentIdField(),
      name: new StringField({required: true, blank: false, textSearch: true}),
      pages: new EmbeddedCollectionField(BaseJournalEntryPage),
      folder: new ForeignDocumentField(BaseFolder),
      categories: new EmbeddedCollectionField(BaseJournalEntryCategory),
      sort: new IntegerSortField(),
      ownership: new DocumentOwnershipField(),
      flags: new DocumentFlagsField(),
      _stats: new DocumentStatsField()
    };
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _initialize(options) {
    super._initialize(options);
    DocumentStatsField._shimDocument(this);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static migrateData(source) {
    DocumentStatsField._migrateData(this, source);
    return super.migrateData(source);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static shimData(source, options) {
    DocumentStatsField._shimData(this, source, options);
    return super.shimData(source, options);
  }
}

/**
 * @import {JournalEntryCategoryData} from "./_types.mjs";
 */

/**
 * An embedded Document that represents a category in a JournalEntry.
 * Defines the DataSchema and common behaviors for a JournalEntryCategory which are shared between both client and
 * server.
 * @extends {Document<JournalEntryCategoryData>}
 * @mixes JournalEntryCategoryData
 * @category Documents
 */
class BaseJournalEntryCategory extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "JournalEntryCategory",
    collection: "categories",
    label: "DOCUMENT.JournalEntryCategory",
    labelPlural: "DOCUMENT.JournalEntryCategories",
    isEmbedded: true,
    schemaVersion: "13.341"
  }, { inplace: false }));

  /** @override */
  static defineSchema() {
    return {
      _id: new DocumentIdField(),
      name: new StringField({ required: true, blank: true, textSearch: true }),
      sort: new IntegerSortField(),
      flags: new DocumentFlagsField(),
      _stats: new DocumentStatsField()
    };
  }
}

/**
 * @import {JournalEntryPageData} from "./_types.mjs";
 */

/**
 * The JournalEntryPage Document.
 * Defines the DataSchema and common behaviors for a JournalEntryPage which are shared between both client and server.
 * @extends {Document<JournalEntryPageData>}
 * @mixes JournalEntryPageData
 * @category Documents
 */
class BaseJournalEntryPage extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "JournalEntryPage",
    collection: "pages",
    hasTypeData: true,
    indexed: true,
    label: "DOCUMENT.JournalEntryPage",
    labelPlural: "DOCUMENT.JournalEntryPages",
    coreTypes: ["text", "image", "pdf", "video"],
    compendiumIndexFields: ["name", "type", "sort"],
    permissions: {
      create: "OWNER",
      delete: "OWNER"
    },
    schemaVersion: "13.341"
  }, {inplace: false}));

  /** @inheritdoc */
  static defineSchema() {
    return {
      _id: new DocumentIdField(),
      name: new StringField({required: true, blank: false, label: "JOURNALENTRYPAGE.PageTitle", textSearch: true}),
      type: new DocumentTypeField(this, {initial: "text"}),
      system: new TypeDataField(this),
      title: new SchemaField({
        show: new BooleanField({initial: true}),
        level: new NumberField({required: true, initial: 1, min: 1, max: 6, integer: true, nullable: false})
      }),
      image: new SchemaField({
        caption: new StringField({required: false, initial: undefined})
      }),
      text: new SchemaField({
        content: new HTMLField({required: false, initial: undefined, textSearch: true}),
        markdown: new StringField({required: false, initial: undefined}),
        format: new NumberField({label: "JOURNALENTRYPAGE.Format",
          initial: JOURNAL_ENTRY_PAGE_FORMATS.HTML, choices: Object.values(JOURNAL_ENTRY_PAGE_FORMATS)})
      }),
      video: new SchemaField({
        controls: new BooleanField({initial: true}),
        loop: new BooleanField({required: false, initial: undefined}),
        autoplay: new BooleanField({required: false, initial: undefined}),
        volume: new AlphaField({required: true, step: 0.01, initial: .5}),
        timestamp: new NumberField({required: false, min: 0, initial: undefined}),
        width: new NumberField({required: false, positive: true, integer: true, initial: undefined}),
        height: new NumberField({required: false, positive: true, integer: true, initial: undefined})
      }),
      src: new StringField({required: false, blank: false, nullable: true, initial: null,
        label: "JOURNALENTRYPAGE.Source"}),
      category: new DocumentIdField({readonly: false}),
      sort: new IntegerSortField(),
      ownership: new DocumentOwnershipField({initial: {default: DOCUMENT_OWNERSHIP_LEVELS.INHERIT}}),
      flags: new DocumentFlagsField(),
      _stats: new DocumentStatsField()
    };
  }
}

/**
 * @import {MacroData} from "./_types.mjs";
 * @import {DocumentPermissionTest} from "@common/abstract/_types.mjs";
 */

/**
 * The Macro Document.
 * Defines the DataSchema and common behaviors for a Macro which are shared between both client and server.
 * @extends {Document<MacroData>}
 * @mixes MacroData
 * @category Documents
 */
class BaseMacro extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "Macro",
    collection: "macros",
    indexed: true,
    compendiumIndexFields: ["_id", "name", "img", "sort", "folder"],
    label: "DOCUMENT.Macro",
    labelPlural: "DOCUMENT.Macros",
    coreTypes: Object.values(MACRO_TYPES),
    permissions: {
      create: this.#canCreate,
      update: this.#canUpdate,
      delete: "OWNER"
    },
    schemaVersion: "13.341"
  }, {inplace: false}));

  /** @inheritdoc */
  static defineSchema() {
    const {BaseUser, BaseFolder} = foundry.documents;
    return {
      _id: new DocumentIdField(),
      name: new StringField({required: true, blank: false, textSearch: true}),
      type: new DocumentTypeField(this, {initial: MACRO_TYPES.CHAT}),
      author: new DocumentAuthorField(BaseUser),
      img: new FilePathField({categories: ["IMAGE"], initial: () => this.DEFAULT_ICON}),
      scope: new StringField({required: true, choices: MACRO_SCOPES, initial: MACRO_SCOPES[0],
        validationError: "must be a value in CONST.MACRO_SCOPES"}),
      command: new StringField({required: true, blank: true}),
      folder: new ForeignDocumentField(BaseFolder),
      sort: new IntegerSortField(),
      ownership: new DocumentOwnershipField(),
      flags: new DocumentFlagsField(),
      _stats: new DocumentStatsField()
    };
  }

  /** @override */
  static LOCALIZATION_PREFIXES = ["DOCUMENT", "MACRO"];

  /**
   * The default icon used for newly created Macro documents.
   * @type {string}
   */
  static DEFAULT_ICON = "icons/svg/dice-target.svg";

  /* -------------------------------------------- */

  /** @inheritDoc */
  _initialize(options) {
    super._initialize(options);
    DocumentStatsField._shimDocument(this);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static migrateData(source) {
    DocumentStatsField._migrateData(this, source);
    return super.migrateData(source);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static shimData(source, options) {
    DocumentStatsField._shimData(this, source, options);
    return super.shimData(source, options);
  }

  /* -------------------------------------------- */
  /*  Model Methods                               */
  /* -------------------------------------------- */

  /** @override */
  static validateJoint(data) {
    if ( data.type !== MACRO_TYPES.SCRIPT ) return;
    const field = new JavaScriptField({ async: true });
    const failure = field.validate(data.command);
    if ( failure ) throw failure.asError();
  }

  /* -------------------------------------------- */

  /** @override */
  static canUserCreate(user) {
    return user.hasRole("PLAYER");
  }

  /* ---------------------------------------- */

  /**
   * Is a user able to create the Macro document?
   * @type {DocumentPermissionTest}
   */
  static #canCreate(user, doc) {
    if ( !user.isGM && (doc._source.author !== user.id) ) return false;
    if ( (doc._source.type === "script") && !user.hasPermission("MACRO_SCRIPT") ) return false;
    return user.hasRole("PLAYER");
  }

  /* ---------------------------------------- */

  /**
   * Is a user able to update the Macro document?
   * @type {DocumentPermissionTest}
   */
  static #canUpdate(user, doc, data) {
    if ( !user.hasPermission("MACRO_SCRIPT") ) {
      if ( (data.type === "script") || (data["==type"] === "script") ) return false;
      if ( (doc._source.type === "script") && ("command" in data) ) return false;
    }
    return doc.testUserPermission(user, "OWNER");
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  getUserLevel(user) {
    user ||= game.user;
    if ( user.id === this._source.author ) return DOCUMENT_OWNERSHIP_LEVELS.OWNER;
    return super.getUserLevel(user);
  }

  /* -------------------------------------------- */
  /*  Database Event Handlers                     */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _preCreate(data, options, user) {
    const allowed = await super._preCreate(data, options, user);
    if ( allowed === false ) return false;
    this.updateSource({author: user.id});
  }
}

/**
 * @import {MeasuredTemplateData} from "./_types.mjs";
 * @import {DocumentPermissionTest} from "@common/abstract/_types.mjs";
 */

/**
 * The MeasuredTemplate Document.
 * Defines the DataSchema and common behaviors for a MeasuredTemplate which are shared between both client and server.
 * @extends {Document<MeasuredTemplateData>}
 * @mixes MeasuredTemplateData
 * @category Documents
 */
class BaseMeasuredTemplate extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = mergeObject(super.metadata, {
    name: "MeasuredTemplate",
    collection: "templates",
    label: "DOCUMENT.MeasuredTemplate",
    labelPlural: "DOCUMENT.MeasuredTemplates",
    isEmbedded: true,
    permissions: {
      create: this.#canCreate,
      delete: "OWNER"
    },
    schemaVersion: "13.341"
  }, {inplace: false});

  /** @inheritdoc */
  static defineSchema() {
    return {
      _id: new DocumentIdField(),
      author: new DocumentAuthorField(foundry.documents.BaseUser),
      t: new StringField({required: true, choices: Object.values(MEASURED_TEMPLATE_TYPES),
        initial: MEASURED_TEMPLATE_TYPES.CIRCLE,
        validationError: "must be a value in CONST.MEASURED_TEMPLATE_TYPES"
      }),
      x: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
      y: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
      elevation: new NumberField({required: true, nullable: false, initial: 0}),
      sort: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
      distance: new NumberField({required: true, nullable: false, initial: 0, min: 0}),
      direction: new AngleField(),
      angle: new AngleField({normalize: false}),
      width: new NumberField({required: true, nullable: false, initial: 0, min: 0, step: 0.01}),
      borderColor: new ColorField({nullable: false, initial: "#000000"}),
      fillColor: new ColorField({nullable: false, initial: () => game.user?.color.css || "#ffffff"}),
      texture: new FilePathField({categories: ["IMAGE", "VIDEO"]}),
      hidden: new BooleanField(),
      flags: new DocumentFlagsField()
    };
  }

  /** @override */
  static LOCALIZATION_PREFIXES = ["DOCUMENT", "TEMPLATE"];

  /* ---------------------------------------- */

  /**
   * Is a user able to create a new MeasuredTemplate?
   * @type {DocumentPermissionTest}
   */
  static #canCreate(user, doc) {
    if ( !user.isGM && (doc._source.author !== user.id) ) return false;
    return user.hasPermission("TEMPLATE_CREATE");
  }

  /* -------------------------------------------- */
  /*  Model Methods                               */
  /* -------------------------------------------- */

  /** @inheritDoc */
  getUserLevel(user) {
    user ||= game.user;
    if ( user.id === this._source.author ) return DOCUMENT_OWNERSHIP_LEVELS.OWNER;
    return super.getUserLevel(user);
  }

  /* -------------------------------------------- */
  /*  Deprecations and Compatibility              */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static migrateData(data) {
    /**
     * V12 migration from user to author
     * @deprecated since v12
     */
    this._addDataFieldMigration(data, "user", "author");
    return super.migrateData(data);
  }

  /* ---------------------------------------- */

  /** @inheritdoc */
  static shimData(data, options) {
    this._addDataFieldShim(data, "user", "author", {since: 12, until: 14});
    return super.shimData(data, options);
  }

  /* ---------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  get user() {
    this.constructor._logDataFieldMigration("user", "author", {since: 12, until: 14});
    return this.author;
  }
}

/**
 * @import {NoteData} from "./_types.mjs";
 * @import {DocumentPermissionTest} from "@common/abstract/_types.mjs";
 */

/**
 * The Note Document.
 * Defines the DataSchema and common behaviors for a Note which are shared between both client and server.
 * @extends {Document<NoteData>}
 * @mixes NoteData
 * @category Documents
 */
class BaseNote extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "Note",
    collection: "notes",
    label: "DOCUMENT.Note",
    labelPlural: "DOCUMENT.Notes",
    permissions: {
      create: BaseNote.#canCreate,
      delete: "OWNER"
    },
    schemaVersion: "13.341"
  }, {inplace: false}));

  /** @inheritdoc */
  static defineSchema() {
    const {BaseJournalEntry, BaseJournalEntryPage} = foundry.documents;
    return {
      _id: new DocumentIdField(),
      entryId: new ForeignDocumentField(BaseJournalEntry, {idOnly: true}),
      pageId: new ForeignDocumentField(BaseJournalEntryPage, {idOnly: true}),
      x: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
      y: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
      elevation: new NumberField({required: true, nullable: false, initial: 0}),
      sort: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
      texture: new TextureData({}, {categories: ["IMAGE"],
        initial: {src: () => this.DEFAULT_ICON, anchorX: 0.5, anchorY: 0.5, fit: "contain"}}),
      iconSize: new NumberField({required: true, nullable: false, integer: true, min: 32, initial: 40,
        validationError: "must be an integer greater than 32"}),
      text: new StringField({textSearch: true}),
      fontFamily: new StringField({required: true,
        initial: () => globalThis.CONFIG?.defaultFontFamily || "Signika"}),
      fontSize: new NumberField({required: true, nullable: false, integer: true, min: 8, max: 128, initial: 32,
        validationError: "must be an integer between 8 and 128"}),
      textAnchor: new NumberField({required: true, choices: Object.values(TEXT_ANCHOR_POINTS),
        initial: TEXT_ANCHOR_POINTS.BOTTOM, validationError: "must be a value in CONST.TEXT_ANCHOR_POINTS"}),
      textColor: new ColorField({required: true, nullable: false, initial: "#ffffff"}),
      global: new BooleanField(),
      flags: new DocumentFlagsField()
    };
  }

  /** @override */
  static LOCALIZATION_PREFIXES = ["DOCUMENT", "NOTE"];

  /**
   * The default icon used for newly created Note documents.
   * @type {string}
   */
  static DEFAULT_ICON = "icons/svg/book.svg";

  /* -------------------------------------------- */
  /*  Model Methods                               */
  /* -------------------------------------------- */

  /** @override */
  getUserLevel(user) {
    if ( this.page ) return this.page.getUserLevel(user);
    if ( this.entry ) return this.entry.getUserLevel(user);
    if ( user.isGM || user.hasPermission("NOTE_CREATE") ) return DOCUMENT_OWNERSHIP_LEVELS.OWNER;
    return DOCUMENT_OWNERSHIP_LEVELS.NONE;
  }

  /* -------------------------------------------- */

  /** @override */
  static canUserCreate(user) {
    return user.hasPermission("NOTE_CREATE");
  }

  /* -------------------------------------------- */

  /**
   * To create a Note document, the player must have both the NOTE_CREATE permission and at least OBSERVER
   * permission over the referenced JournalEntry.
   * @type {DocumentPermissionTest}
   */
  static #canCreate(user, doc) {
    if ( !user.hasPermission("NOTE_CREATE") ) return false;
    if ( doc._source.entryId ) return doc.entry.testUserPermission(user, "OBSERVER");
    return true;
  }
}

/**
 * @import {PlaylistData} from "./_types.mjs";
 */

/**
 * The Playlist Document.
 * Defines the DataSchema and common behaviors for a Playlist which are shared between both client and server.
 * @extends {Document<PlaylistData>}
 * @mixes PlaylistData
 * @category Documents
 */
class BasePlaylist extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "Playlist",
    collection: "playlists",
    indexed: true,
    compendiumIndexFields: ["_id", "name", "description", "sort", "folder"],
    embedded: {PlaylistSound: "sounds"},
    label: "DOCUMENT.Playlist",
    labelPlural: "DOCUMENT.Playlists",
    permissions: {
      create: "PLAYLIST_CREATE",
      delete: "OWNER"
    },
    schemaVersion: "13.341"
  }, {inplace: false}));

  /** @inheritdoc */
  static defineSchema() {
    const {BasePlaylistSound, BaseFolder} = foundry.documents;
    return {
      _id: new DocumentIdField(),
      name: new StringField({required: true, blank: false, textSearch: true}),
      description: new StringField({textSearch: true}),
      sounds: new EmbeddedCollectionField(BasePlaylistSound),
      channel: new StringField({required: true, choices: AUDIO_CHANNELS, initial: "music", blank: false}),
      mode: new NumberField({required: true, choices: Object.values(PLAYLIST_MODES),
        initial: PLAYLIST_MODES.SEQUENTIAL, validationError: "must be a value in CONST.PLAYLIST_MODES"}),
      playing: new BooleanField(),
      fade: new NumberField({integer: true, positive: true}),
      folder: new ForeignDocumentField(BaseFolder),
      sorting: new StringField({required: true, choices: Object.values(PLAYLIST_SORT_MODES),
        initial: PLAYLIST_SORT_MODES.ALPHABETICAL,
        validationError: "must be a value in CONST.PLAYLIST_SORTING_MODES"}),
      seed: new NumberField({integer: true, min: 0}),
      sort: new IntegerSortField(),
      ownership: new DocumentOwnershipField(),
      flags: new DocumentFlagsField(),
      _stats: new DocumentStatsField()
    };
  }

  /* -------------------------------------------- */

  /** @override */
  static LOCALIZATION_PREFIXES = ["DOCUMENT", "PLAYLIST"];

  /* -------------------------------------------- */

  /** @inheritDoc */
  _initialize(options) {
    super._initialize(options);
    DocumentStatsField._shimDocument(this);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static migrateData(source) {
    DocumentStatsField._migrateData(this, source);
    return super.migrateData(source);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static shimData(source, options) {
    DocumentStatsField._shimData(this, source, options);
    return super.shimData(source, options);
  }
}

/**
 * @import {PlaylistSoundData} from "./_types.mjs";
 */

/**
 * The PlaylistSound Document.
 * Defines the DataSchema and common behaviors for a PlaylistSound which are shared between both client and server.
 * @extends {Document<PlaylistSoundData>}
 * @mixes PlaylistSoundData
 * @category Documents
 */
class BasePlaylistSound extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "PlaylistSound",
    collection: "sounds",
    indexed: true,
    label: "DOCUMENT.PlaylistSound",
    labelPlural: "DOCUMENT.PlaylistSounds",
    compendiumIndexFields: ["name", "sort"],
    schemaVersion: "13.341",
    permissions: {
      ...super.metadata.permissions,
      create: "OWNER",
      update: "OWNER",
      delete: "OWNER"
    }
  }, {inplace: false}));

  /** @inheritdoc */
  static defineSchema() {
    return {
      _id: new DocumentIdField(),
      name: new StringField({required: true, blank: false, textSearch: true}),
      description: new StringField(),
      path: new FilePathField({categories: ["AUDIO"]}),
      channel: new StringField({required: true, choices: AUDIO_CHANNELS, initial: "", blank: true}),
      playing: new BooleanField(),
      pausedTime: new NumberField({min: 0}),
      repeat: new BooleanField(),
      volume: new AlphaField({initial: 0.5, step: 0.01}),
      fade: new NumberField({integer: true, min: 0}),
      sort: new IntegerSortField(),
      flags: new DocumentFlagsField()
    };
  }

  /** @override */
  static LOCALIZATION_PREFIXES = ["DOCUMENT", "PLAYLIST_SOUND"];
}

/**
 * @import {RollTableData} from "./_types.mjs";
 */

/**
 * The RollTable Document.
 * Defines the DataSchema and common behaviors for a RollTable which are shared between both client and server.
 * @extends {Document<RollTableData>}
 * @mixes RollTableData
 * @category Documents
 */
class BaseRollTable extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "RollTable",
    collection: "tables",
    indexed: true,
    compendiumIndexFields: ["_id", "name", "description", "img", "sort", "folder"],
    embedded: {TableResult: "results"},
    label: "DOCUMENT.RollTable",
    labelPlural: "DOCUMENT.RollTables",
    schemaVersion: "13.341"
  }, {inplace: false}));

  /** @override */
  static LOCALIZATION_PREFIXES = ["DOCUMENT", "TABLE"];

  /**
   * The default icon used for newly created Macro documents
   * @type {string}
   */
  static DEFAULT_ICON = "icons/svg/d20-grey.svg";

  /* -------------------------------------------- */

  /** @inheritDoc */
  static defineSchema() {
    const {BaseTableResult, BaseFolder} = foundry.documents;
    return {
      _id: new DocumentIdField(),
      name: new StringField({required: true, blank: false, textSearch: true}),
      img: new FilePathField({categories: ["IMAGE"], initial: () => this.DEFAULT_ICON}),
      description: new HTMLField({textSearch: true}),
      results: new EmbeddedCollectionField(BaseTableResult),
      formula: new StringField(),
      replacement: new BooleanField({initial: true}),
      displayRoll: new BooleanField({initial: true}),
      folder: new ForeignDocumentField(BaseFolder),
      sort: new IntegerSortField(),
      ownership: new DocumentOwnershipField(),
      flags: new DocumentFlagsField(),
      _stats: new DocumentStatsField()
    };
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _initialize(options) {
    super._initialize(options);
    DocumentStatsField._shimDocument(this);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static migrateData(source) {
    DocumentStatsField._migrateData(this, source);
    return super.migrateData(source);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static shimData(source, options) {
    DocumentStatsField._shimData(this, source, options);
    return super.shimData(source, options);
  }
}

/**
 * @import {GridConfiguration, GridOffset2D, GridOffset3D, GridCoordinates2D, GridCoordinates3D, GridSnappingBehavior,
 *   GridMeasurePathWaypointData2D, GridMeasurePathWaypointData3D, GridMeasurePathResult, GridMeasurePathCostFunction2D,
 *   GridMeasurePathCostFunction3D} from "./_types.mjs";
 * @import {Point, ElevatedPoint, Rectangle} from "../_types.mjs";
 * @import {GridType, MOVEMENT_DIRECTIONS} from "../constants.mjs";
 */

/**
 * The base grid class.
 * @template [Coordinates2D=GridCoordinates2D]
 * @template [Coordinates3D=GridCoordinates3D]
 * @abstract
 */
class BaseGrid {
  /**
   * The base grid constructor.
   * @param {GridConfiguration} config    The grid configuration
   */
  constructor(config) {
    let {size, distance=1, units="", style="solidLines", thickness=1, color, alpha=1} = config;
    /** @deprecated since v12 */
    if ( "dimensions" in config ) {
      const msg = "The constructor BaseGrid({dimensions, color, alpha}) is deprecated "
        + "in favor of BaseGrid({size, distance, units, style, thickness, color, alpha}).";
      logCompatibilityWarning(msg, {since: 12, until: 14});
      const dimensions = config.dimensions;
      size = dimensions.size;
      distance = dimensions.distance || 1;
    }

    if ( size === undefined ) throw new Error(`${this.constructor.name} cannot be constructed without a size`);
    if ( !(size > 0) ) throw new Error("The size must be positive");
    if ( !(distance > 0) ) throw new Error("The distance must be positive");

    // Convert the color to a Color
    if ( color ) color = Color.from(color);
    if ( !color?.valid ) color = new Color(0);

    /**
     * The size of a grid space in pixels.
     * @type {number}
     * @readonly
     */
    this.size = size;

    /**
     * The width of a grid space in pixels.
     * @type {number}
     * @readonly
     */
    this.sizeX = size;

    /**
     * The height of a grid space in pixels.
     * @type {number}
     * @readonly
     */
    this.sizeY = size;

    /**
     * The distance of a grid space in units.
     * @type {number}
     * @readonly
     */
    this.distance = distance;

    /**
     * The distance units used in this grid.
     * @type {string}
     * @readonly
     */
    this.units = units;

    /**
     * The style of the grid.
     * @type {string}
     * @readonly
     */
    this.style = style;

    /**
     * The thickness of the grid.
     * @type {number}
     * @readonly
     */
    this.thickness = thickness;

    /**
     * The color of the grid.
     * @type {Color}
     * @readonly
     */
    this.color = color;

    /**
     * The opacity of the grid.
     * @type {number}
     * @readonly
     */
    this.alpha = alpha;
  }

  /* -------------------------------------------- */

  /**
   * The grid type (see {@link CONST.GRID_TYPES}).
   * @type {GridType}
   * @readonly
   */
  type;

  /* -------------------------------------------- */

  /**
   * Is this a gridless grid?
   * @type {boolean}
   */
  get isGridless() {
    return this.type === GRID_TYPES.GRIDLESS;
  }

  /* -------------------------------------------- */

  /**
   * Is this a square grid?
   * @type {boolean}
   */
  get isSquare() {
    return this.type === GRID_TYPES.SQUARE;
  }

  /* -------------------------------------------- */

  /**
   * Is this a hexagonal grid?
   * @type {boolean}
   */
  get isHexagonal() {
    return (this.type >= GRID_TYPES.HEXODDR) && (this.type <= GRID_TYPES.HEXEVENQ);
  }

  /* -------------------------------------------- */

  /**
   * Calculate the total size of the canvas with padding applied, as well as the top-left coordinates of the inner
   * rectangle that houses the scene.
   * @param {number} sceneWidth         The width of the scene.
   * @param {number} sceneHeight        The height of the scene.
   * @param {number} padding            The percentage of padding.
   * @returns {{width: number, height: number, x: number, y: number, rows: number, columns: number}}
   * @abstract
   */
  calculateDimensions(sceneWidth, sceneHeight, padding) {
    throw new Error("A subclass of the BaseGrid must implement the calculateDimensions method");
  }

  /* -------------------------------------------- */

  /**
   * Returns the offset of the grid space corresponding to the given coordinates.
   * @overload
   * @param {Coordinates2D} coords    The coordinates
   * @returns {GridOffset2D}          The offset
   */
  /**
   * @overload
   * @param {Coordinates3D} coords    The coordinates
   * @returns {GridOffset3D}          The offset
   * @abstract
   */
  getOffset(coords) {
    throw new Error("A subclass of the BaseGrid must implement the getOffset method");
  }

  /* -------------------------------------------- */

  /**
   * Returns the smallest possible range containing the offsets of all grid spaces that intersect the given bounds.
   * If the bounds are empty (nonpositive width or height), then the offset range is empty.
   * @example
   * ```js
   * const [i0, j0, i1, j1] = grid.getOffsetRange(bounds);
   * for ( let i = i0; i < i1; i++ ) {
   *   for ( let j = j0; j < j1; j++ ) {
   *     const offset = {i, j};
   *     // ...
   *   }
   * }
   * ```
   * @param {Rectangle} bounds                                      The bounds
   * @returns {[i0: number, j0: number, i1: number, j1: number]}    The offset range
   * @abstract
   */
  getOffsetRange({x, y, width, height}) {
    throw new Error("A subclass of the BaseGrid must implement the getOffsetRange method");
  }

  /* -------------------------------------------- */

  /**
   * Returns the offsets of the grid spaces adjacent to the one corresponding to the given coordinates.
   * Returns always an empty array in gridless grids.
   * @overload
   * @param {Coordinates2D} coords    The coordinates
   * @returns {GridOffset2D[]}        The adjacent offsets
   */
  /**
   * @overload
   * @param {Coordinates3D} coords    The coordinates
   * @returns {GridOffset3D[]}        The adjacent offsets
   * @abstract
   */
  getAdjacentOffsets(coords) {
    throw new Error("A subclass of the BaseGrid must implement the getAdjacentOffsets method");
  }

  /* -------------------------------------------- */

  /**
   * Returns true if the grid spaces corresponding to the given coordinates are adjacent to each other.
   * In square and hexagonal grids with illegal diagonals the diagonally neighboring grid spaces are not adjacent.
   * Returns always false in gridless grids.
   * @overload
   * @param {Coordinates2D} coords1    The first coordinates
   * @param {Coordinates2D} coords2    The second coordinates
   * @returns {boolean}
   */
  /**
   * @overload
   * @param {Coordinates3D} coords1    The first coordinates
   * @param {Coordinates3D} coords2    The second coordinates
   * @returns {boolean}
   * @abstract
   */
  testAdjacency(coords1, coords2) {
    throw new Error("A subclass of the BaseGrid must implement the testAdjacency method");
  }

  /* -------------------------------------------- */

  /**
   * Returns the offset of the grid space corresponding to the given coordinates
   * shifted by one grid space in the given direction. The k-coordinate is not changed.
   * In square and hexagonal grids with illegal diagonals the offset of the given coordinates is returned
   * if the direction is diagonal.
   * In gridless grids the point is by the grid size.
   * @overload
   * @param {Coordinates2D} coords             The coordinates
   * @param {MOVEMENT_DIRECTIONS} direction    The direction (see {@link CONST.MOVEMENT_DIRECTIONS})
   * @returns {GridOffset2D}                   The offset
   */
  /**
   * @overload
   * @param {Coordinates3D} coords             The coordinates
   * @param {MOVEMENT_DIRECTIONS} direction    The direction (see {@link CONST.MOVEMENT_DIRECTIONS})
   * @returns {GridOffset3D}                   The offset
   * @abstract
   */
  getShiftedOffset(coords, direction) {
    throw new Error("A subclass of the BaseGrid must implement the getShiftedOffset method");
  }

  /* -------------------------------------------- */

  /**
   * Returns the point shifted by the difference between the grid space corresponding to the given coordinates
   * and the shifted grid space in the given direction. The z-coordinate is not changed.
   * In square and hexagonal grids with illegal diagonals the point is not shifted if the direction is diagonal.
   * In gridless grids the point coordinates are shifted by the grid size.
   * @overload
   * @param {Point} point                      The point that is to be shifted
   * @param {MOVEMENT_DIRECTIONS} direction    The direction (see {@link CONST.MOVEMENT_DIRECTIONS})
   * @returns {Point}                          The shifted point
   */
  /**
   * @overload
   * @param {ElevatedPoint} point              The point that is to be shifted
   * @param {MOVEMENT_DIRECTIONS} direction    The direction (see {@link CONST.MOVEMENT_DIRECTIONS})
   * @returns {ElevatedPoint}                  The shifted point
   * @abstract
   */
  getShiftedPoint(point, direction) {
    throw new Error("A subclass of the BaseGrid must implement the getShiftedPoint method");
  }

  /* -------------------------------------------- */

  /**
   * Returns the top-left point of the grid space bounds corresponding to the given coordinates.
   * If given a point, the top-left point of the grid space bounds that contains it is returned.
   * The top-left point lies in the plane of the bottom face of the 3D grid space.
   * In gridless grids a point with the same coordinates as the given point is returned.
   * @overload
   * @param {Coordinates2D} coords    The coordinates
   * @returns {Point}                 The top-left point
   */
  /**
   * @overload
   * @param {Coordinates3D} coords    The coordinates
   * @returns {ElevatedPoint}         The top-left point
   * @abstract
   */
  getTopLeftPoint(coords) {
    throw new Error("A subclass of the BaseGrid must implement the getTopLeftPoint method");
  }

  /* -------------------------------------------- */

  /**
   * Returns the center point of the grid space corresponding to the given coordinates.
   * If given a point, the center point of the grid space that contains it is returned.
   * The center point lies in the plane of the bottom face of the 3D grid space.
   * In gridless grids a point with the same coordinates as the given point is returned.
   * @overload
   * @param {Coordinates2D} coords    The coordinates
   * @returns {Point}                 The center point
   */
  /**
   * @overload
   * @param {Coordinates3D} coords    The coordinates
   * @returns {ElevatedPoint}         The center point
   * @abstract
   */
  getCenterPoint(coords) {
    throw new Error("A subclass of the BaseGrid must implement the getCenterPoint method");
  }

  /* -------------------------------------------- */

  /**
   * Returns the points of the grid space shape relative to the center point.
   * The points are returned in the same order as in {@link BaseGrid#getVertices}.
   * In gridless grids an empty array is returned.
   * @returns {Point[]}    The points of the polygon
   * @abstract
   */
  getShape() {
    throw new Error("A subclass of the BaseGrid must implement the getShape method");
  }

  /* -------------------------------------------- */

  /**
   * Returns the vertices of the grid space corresponding to the given coordinates.
   * The vertices are returned ordered in positive orientation with the first vertex
   * being the top-left vertex in square grids, the top vertex in row-oriented
   * hexagonal grids, and the left vertex in column-oriented hexagonal grids.
   * In gridless grids an empty array is returned.
   * @param {Coordinates2D} coords      The coordinates
   * @returns {Point[]}                 The vertices
   * @abstract
   */
  getVertices(coords) {
    throw new Error("A subclass of the BaseGrid must implement the getVertices method");
  }

  /* -------------------------------------------- */

  /**
   * Snaps the given point to the grid.
   * In square and hexagonal grids the z-coordinate of the point is rounded to the nearest multiple of the grid size.
   * In gridless grids a point with the same coordinates as the given point is returned regardless of the
   * snapping behavior.
   * @overload
   * @param {Point} point                      The point that is to be snapped
   * @param {GridSnappingBehavior} behavior    The snapping behavior
   * @returns {Point}                          The snapped point
   */
  /**
   * @overload
   * @param {ElevatedPoint} point              The point that is to be snapped
   * @param {GridSnappingBehavior} behavior    The snapping behavior
   * @returns {ElevatedPoint}                  The snapped point
   * @abstract
   */
  getSnappedPoint(point, behavior) {
    throw new Error("A subclass of the BaseGrid must implement the getSnappedPoint method");
  }

  /* -------------------------------------------- */

  /**
   * Measure a shortest, direct path through the given waypoints.
   * @template {{[K in "i"|"j"|"k"|"q"|"r"|"s"|"x"|"y"|"elevation"|"cost"]: never}} [SegmentData={}]
   * @overload
   * @param {(Coordinates2D & Partial<GridMeasurePathWaypointData2D> & SegmentData)[]} waypoints
   *   The waypoints the path must pass through
   * @param {object} [options]                                             Additional measurement options
   * @param {GridMeasurePathCostFunction2D<SegmentData>} [options.cost]    The function that returns the cost
   *   for a given move between grid spaces (default is the distance travelled along the direct path)
   * @returns {GridMeasurePathResult}    The measurements a shortest, direct path through the given waypoints
   */
  /**
   * @overload
   * @param {(Coordinates3D & Partial<GridMeasurePathWaypointData3D> & SegmentData)[]} waypoints
   *   The waypoints the path must pass through
   * @param {object} [options]                                             Additional measurement options
   * @param {GridMeasurePathCostFunction3D<SegmentData>} [options.cost]    The function that returns the cost
   *   for a given move between grid spaces (default is the distance travelled along the direct path)
   * @returns {GridMeasurePathResult}    The measurements a shortest, direct path through the given waypoints
   */
  measurePath(waypoints, options={}) {
    const result = {
      waypoints: [],
      segments: [],
      distance: 0,
      cost: 0,
      spaces: 0,
      diagonals: 0,
      euclidean: 0
    };
    if ( waypoints.length === 0 ) return result;

    // Create result waypoints and segments
    let from = {backward: null, forward: null, distance: 0, cost: 0, spaces: 0, diagonals: 0, euclidean: 0};
    result.waypoints.push(from);
    for ( let i = 1; i < waypoints.length; i++ ) {
      const to = {backward: null, forward: null, distance: 0, cost: 0, spaces: 0, diagonals: 0, euclidean: 0};
      const segment = {from, to, distance: 0, cost: 0, spaces: 0, diagonals: 0, euclidean: 0};
      from.forward = to.backward = segment;
      result.waypoints.push(to);
      result.segments.push(segment);
      from = to;
    }

    // Measure segments
    this._measurePath(waypoints, options, result);

    // Accumulate segment measurements
    for ( let i = 1; i < waypoints.length; i++ ) {
      const waypoint = result.waypoints[i];
      const segment = waypoint.backward;

      // Accumulate measurements
      result.distance += segment.distance;
      result.cost += segment.cost;
      result.spaces += segment.spaces;
      result.diagonals += segment.diagonals;
      result.euclidean += segment.euclidean;

      // Set waypoint measurements
      waypoint.distance = result.distance;
      waypoint.cost = result.cost;
      waypoint.spaces = result.spaces;
      waypoint.diagonals = result.diagonals;
      waypoint.euclidean = result.euclidean;
    }
    return result;
  }

  /* -------------------------------------------- */

  /**
   * Measures the path and writes the segments measurements into the result.
   * The waypoint measurements are filled in by {@link BaseGrid#measurePath}.
   * Called by {@link BaseGrid#measurePath}.
   * @template {{[K in "i"|"j"|"k"|"q"|"r"|"s"|"x"|"y"|"elevation"|"cost"|"measure"]: never}} SegmentData
   * @overload
   * @param {(Coordinates2D & Partial<GridMeasurePathWaypointData2D> & SegmentData)[]} waypoints
   *   The waypoints the path must pass through
   * @param {object} [options]                                             Additional measurement options
   * @param {GridMeasurePathCostFunction2D<SegmentData>} [options.cost]    The function that returns the cost
   *   for a given move between grid spaces (default is the distance travelled)
   * @param {GridMeasurePathResult} result    The measurement result that the measurements need to be written to
   */
  /**
   * @overload
   * @param {(Coordinates3D & Partial<GridMeasurePathWaypointData3D> & SegmentData)[]} waypoints
   *   The waypoints the path must pass through
   * @param {object} [options]                                             Additional measurement options
   * @param {GridMeasurePathCostFunction3D<SegmentData>} [options.cost]    The function that returns the cost
   *   for a given move between grid spaces (default is the distance travelled)
   * @param {GridMeasurePathResult} result    The measurement result that the measurements need to be written to
   * @protected
   * @abstract
   */
  _measurePath(waypoints, options, result) {
    throw new Error("A subclass of the BaseGrid must implement the _measurePath method");
  }

  /* -------------------------------------------- */

  /**
   * Returns the sequence of grid offsets of a shortest, direct path passing through the given waypoints.
   * @overload
   * @param {Coordinates2D[]} waypoints    The waypoints the path must pass through
   * @returns {GridOffset2D[]}             The sequence of grid offsets of a shortest, direct path
   */
  /**
   * @overload
   * @param {Coordinates3D[]} waypoints    The waypoints the path must pass through
   * @returns {GridOffset3D[]}             The sequence of grid offsets of a shortest, direct path
   * @abstract
   */
  getDirectPath(waypoints) {
    throw new Error("A subclass of the BaseGrid must implement the getDirectPath method");
  }

  /* -------------------------------------------- */

  /**
   * Get the point translated in a direction by a distance.
   * The z-coordinate is not changed.
   * @overload
   * @param {Point} point            The point that is to be translated
   * @param {number} direction       The angle of direction in degrees
   * @param {number} distance        The distance in grid units
   * @returns {Point}                The translated point
   */
  /**
   * @overload
   * @param {ElevatedPoint} point    The point that is to be translated
   * @param {number} direction       The angle of direction in degrees
   * @param {number} distance        The distance in grid units
   * @returns {ElevatedPoint}        The translated point
   * @abstract
   */
  getTranslatedPoint(point, direction, distance) {
    throw new Error("A subclass of the BaseGrid must implement the getTranslatedPoint method");
  }

  /* -------------------------------------------- */

  /**
   * Get the circle polygon given the radius in grid units for this grid.
   * The points of the polygon are returned ordered in positive orientation.
   * In gridless grids an approximation of the true circle with a deviation of less than 0.25 pixels is returned.
   * @param {Point} center      The center point of the circle.
   * @param {number} radius     The radius in grid units.
   * @returns {Point[]}         The points of the circle polygon.
   * @abstract
   */
  getCircle(center, radius) {
    throw new Error("A subclass of the BaseGrid must implement the getCircle method");
  }

  /* -------------------------------------------- */

  /**
   * Get the cone polygon given the radius in grid units and the angle in degrees for this grid.
   * The points of the polygon are returned ordered in positive orientation.
   * In gridless grids an approximation of the true cone with a deviation of less than 0.25 pixels is returned.
   * @param {Point} origin        The origin point of the cone
   * @param {number} radius       The radius in grid units
   * @param {number} direction    The direction in degrees
   * @param {number} angle        The angle in degrees
   * @returns {Point[]}           The points of the cone polygon
   */
  getCone(origin, radius, direction, angle) {
    if ( (radius <= 0) || (angle <= 0) ) return [];
    const circle = this.getCircle(origin, radius);
    if ( angle >= 360 ) return circle;
    const n = circle.length;
    const aMin = Math.normalizeRadians(Math.toRadians(direction - (angle / 2)));
    const aMax = aMin + Math.toRadians(angle);
    const pMin = {x: origin.x + (Math.cos(aMin) * this.size), y: origin.y + (Math.sin(aMin) * this.size)};
    const pMax = {x: origin.x + (Math.cos(aMax) * this.size), y: origin.y + (Math.sin(aMax) * this.size)};
    const angles = circle.map(p => {
      const a = Math.atan2(p.y - origin.y, p.x - origin.x);
      return a >= aMin ? a : a + (2 * Math.PI);
    });
    const points = [{x: origin.x, y: origin.y}];
    for ( let i = 0, c0 = circle[n - 1], a0 = angles[n - 1]; i < n; i++ ) {
      let c1 = circle[i];
      let a1 = angles[i];
      if ( a0 > a1 ) {
        const {x: x1, y: y1} = lineLineIntersection(c0, c1, origin, pMin);
        points.push({x: x1, y: y1});
        while ( a1 < aMax ) {
          points.push(c1);
          i = (i + 1) % n;
          c0 = c1;
          c1 = circle[i];
          a0 = a1;
          a1 = angles[i];
          if ( a0 > a1 ) break;
        }
        const {x: x2, y: y2} = lineLineIntersection(c0, c1, origin, pMax);
        points.push({x: x2, y: y2});
        break;
      }
      c0 = c1;
      a0 = a1;
    }
    return points;
  }

  /* -------------------------------------------- */
  /*  Deprecations and Compatibility              */
  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  getRect(w, h) {
    const msg = "BaseGrid#getRect is deprecated. If you need the size of a Token, use Token#getSize instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return new PIXI.Rectangle(0, 0, w * this.sizeX, h * this.sizeY);
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  static calculatePadding(gridType, width, height, size, padding, options) {
    const msg = "BaseGrid.calculatePadding is deprecated in favor of BaseGrid#calculateDimensions.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    let grid;
    if ( gridType === GRID_TYPES.GRIDLESS ) {
      grid = new foundry.grid.GridlessGrid({size});
    } else if ( gridType === GRID_TYPES.SQUARE ) {
      grid = new foundry.grid.SquareGrid({size});
    } else if ( gridType.between(GRID_TYPES.HEXODDR, GRID_TYPES.HEXEVENQ) ) {
      const columns = (gridType === GRID_TYPES.HEXODDQ) || (gridType === GRID_TYPES.HEXEVENQ);
      const HexagonalGrid = foundry.grid.HexagonalGrid;
      if ( options?.legacy ) return HexagonalGrid._calculatePreV10Dimensions(columns, size, width, height, padding);
      grid = new HexagonalGrid({
        columns,
        even: (gridType === GRID_TYPES.HEXEVENR) || (gridType === GRID_TYPES.HEXEVENQ),
        size
      });
    } else {
      throw new Error("Invalid grid type");
    }
    return grid.calculateDimensions(width, height, padding);
  }

  /* -------------------------------------------- */

  /**
   * @deprecated
   * @ignore
   */
  get w() {
    const msg = "BaseGrid#w is deprecated in favor of BaseGrid#sizeX.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return this.sizeX;
  }

  /**
   * @deprecated since v12
   * @ignore
   */
  set w(value) {
    const msg = "BaseGrid#w is deprecated in favor of BaseGrid#sizeX.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    this.sizeX = value;
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  get h() {
    const msg = "BaseGrid#h is deprecated in favor of BaseGrid#sizeY.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return this.sizeY;
  }

  /**
   * @deprecated since v12
   * @ignore
   */
  set h(value) {
    const msg = "BaseGrid#h is deprecated in favor of BaseGrid#sizeY.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    this.sizeY = value;
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  getTopLeft(x, y) {
    const msg = "BaseGrid#getTopLeft is deprecated. Use BaseGrid#getTopLeftPoint instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    const [row, col] = this.getGridPositionFromPixels(x, y);
    return this.getPixelsFromGridPosition(row, col);
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  getCenter(x, y) {
    const msg = "BaseGrid#getCenter is deprecated. Use BaseGrid#getCenterPoint instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return [x, y];
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  getNeighbors(row, col) {
    const msg = "BaseGrid#getNeighbors is deprecated. Use BaseGrid#getAdjacentOffsets instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return this.getAdjacentOffsets({i: row, j: col}).map(({i, j}) => [i, j]);
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  getGridPositionFromPixels(x, y) {
    const msg = "BaseGrid#getGridPositionFromPixels is deprecated. Use BaseGrid#getOffset instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return [y, x].map(Math.round);
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  getPixelsFromGridPosition(row, col) {
    const msg = "BaseGrid#getPixelsFromGridPosition is deprecated. Use BaseGrid#getTopLeftPoint instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return [col, row].map(Math.round);
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  shiftPosition(x, y, dx, dy, options={}) {
    const msg = "BaseGrid#shiftPosition is deprecated. Use BaseGrid#getShiftedPoint instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return [x + (dx * this.size), y + (dy * this.size)];
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  measureDistances(segments, options={}) {
    const msg = "BaseGrid#measureDistances is deprecated. Use BaseGrid#measurePath instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return segments.map(s => {
      return (s.ray.distance / this.size) * this.distance;
    });
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  getSnappedPosition(x, y, interval=null, options={}) {
    const msg = "BaseGrid#getSnappedPosition is deprecated. Use BaseGrid#getSnappedPoint instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    if ( interval === 0 ) return {x: Math.round(x), y: Math.round(y)};
    interval = interval ?? 1;
    return {
      x: Math.round(x.toNearest(this.sizeX / interval)),
      y: Math.round(y.toNearest(this.sizeY / interval))
    };
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  highlightGridPosition(layer, options) {
    const msg = "BaseGrid#highlightGridPosition is deprecated. Use GridLayer#highlightPosition instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    canvas.interface.grid.highlightPosition(layer.name, options);
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  get grid() {
    const msg = "canvas.grid.grid is deprecated. Use canvas.grid instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return this;
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  isNeighbor(r0, c0, r1, c1) {
    const msg = "canvas.grid.isNeighbor is deprecated. Use canvas.grid.testAdjacency instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return this.testAdjacency({i: r0, j: c0}, {i: r1, j: c1});
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  get isHex() {
    const msg = "canvas.grid.isHex is deprecated. Use of canvas.grid.isHexagonal instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return this.isHexagonal;
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  measureDistance(origin, target, options={}) {
    const msg = "canvas.grid.measureDistance is deprecated. "
      + "Use canvas.grid.measurePath instead, which returns grid distance (gridSpaces: true) and Euclidean distance (gridSpaces: false).";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    const ray = new foundry.canvas.geometry.Ray(origin, target);
    const segments = [{ray}];
    return this.measureDistances(segments, options)[0];
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  get highlight() {
    const msg = "canvas.grid.highlight is deprecated. Use canvas.interface.grid.highlight instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return canvas.interface.grid.highlight;
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  get highlightLayers() {
    const msg = "canvas.grid.highlightLayers is deprecated. Use canvas.interface.grid.highlightLayers instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return canvas.interface.grid.highlightLayers;
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  addHighlightLayer(name) {
    const msg = "canvas.grid.addHighlightLayer is deprecated. Use canvas.interface.grid.addHighlightLayer instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return canvas.interface.grid.addHighlightLayer(name);
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  clearHighlightLayer(name) {
    const msg = "canvas.grid.clearHighlightLayer is deprecated. Use canvas.interface.grid.clearHighlightLayer instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    canvas.interface.grid.clearHighlightLayer(name);
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  destroyHighlightLayer(name) {
    const msg = "canvas.grid.destroyHighlightLayer is deprecated. Use canvas.interface.grid.destroyHighlightLayer instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    canvas.interface.grid.destroyHighlightLayer(name);
  }

  /* -------------------------------------------- */


  /**
   * @deprecated since v12
   * @ignore
   */
  getHighlightLayer(name) {
    const msg = "canvas.grid.getHighlightLayer is deprecated. Use canvas.interface.grid.getHighlightLayer instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return canvas.interface.grid.getHighlightLayer(name);
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  highlightPosition(name, options) {
    const msg = "canvas.grid.highlightPosition is deprecated. Use canvas.interface.grid.highlightPosition instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    canvas.interface.grid.highlightPosition(name, options);
  }
}

/**
 * The gridless grid class.
 */
class GridlessGrid extends BaseGrid {

  /**
   * @override
   * @readonly
   */
  type = GRID_TYPES.GRIDLESS;

  /* -------------------------------------------- */

  /** @override */
  calculateDimensions(sceneWidth, sceneHeight, padding) {
    // Note: Do not replace `* (1 / this.size)` by `/ this.size`!
    // It could change the result and therefore break certain scenes.
    const x = Math.ceil((padding * sceneWidth) * (1 / this.size)) * this.size;
    const y = Math.ceil((padding * sceneHeight) * (1 / this.size)) * this.size;
    const width = sceneWidth + (2 * x);
    const height = sceneHeight + (2 * y);
    return {width, height, x, y, rows: Math.ceil(height), columns: Math.ceil(width)};
  }

  /* -------------------------------------------- */

  /** @override */
  getOffset(coords) {
    let i = coords.i;
    if ( i !== undefined ) {
      const {j, k} = coords;
      return k !== undefined ? {i, j, k} : {i, j};
    }
    const {x, y, elevation} = coords;
    i = Math.floor(y) | 0;
    const j = Math.floor(x) | 0;
    if ( elevation === undefined ) return {i, j};
    const k = Math.floor((elevation / this.distance * this.size) + 1e-8) | 0;
    return {i, j, k};
  }

  /* -------------------------------------------- */

  /** @override */
  getOffsetRange({x, y, width, height}) {
    const i0 = Math.floor(y) | 0;
    const j0 = Math.floor(x) | 0;
    if ( !((width > 0) && (height > 0)) ) return [i0, j0, i0, j0];
    return [i0, j0, Math.ceil(y + height) | 0, Math.ceil(x + width) | 0];
  }

  /* -------------------------------------------- */

  /** @override */
  getAdjacentOffsets(coords) {
    return [];
  }

  /* -------------------------------------------- */

  /** @override */
  testAdjacency(coords1, coords2) {
    return false;
  }

  /* -------------------------------------------- */

  /** @override */
  getShiftedOffset(coords, direction) {
    if ( coords.x === undefined ) {
      const {i, j, k} = coords;
      coords = k !== undefined ? {x: j, y: i, elevation: k / this.size * this.distance} : {x: j, y: i};
    }
    return this.getOffset(this.getShiftedPoint(coords, direction));
  }

  /* -------------------------------------------- */

  /** @override */
  getShiftedPoint(point, direction) {
    let di = 0;
    let dj = 0;
    let dk = 0;
    if ( direction & MOVEMENT_DIRECTIONS.UP ) di--;
    if ( direction & MOVEMENT_DIRECTIONS.DOWN ) di++;
    if ( direction & MOVEMENT_DIRECTIONS.LEFT ) dj--;
    if ( direction & MOVEMENT_DIRECTIONS.RIGHT ) dj++;
    if ( direction & MOVEMENT_DIRECTIONS.DESCEND ) dk--;
    if ( direction & MOVEMENT_DIRECTIONS.ASCEND ) dk++;
    const x = point.x + (dj * this.size);
    const y = point.y + (di * this.size);
    const elevation = point.elevation;
    return elevation !== undefined ? {x, y, elevation: elevation + (dk * this.distance)} : {x, y};
  }

  /* -------------------------------------------- */

  /** @override */
  getTopLeftPoint(coords) {
    const i = coords.i;
    if ( i !== undefined ) {
      const {j, k} = coords;
      return k !== undefined ? {x: j, y: i, elevation: k / this.size * this.distance} : {x: j, y: i};
    }
    const {x, y, elevation} = coords;
    return elevation !== undefined ? {x, y, elevation} : {x, y};
  }

  /* -------------------------------------------- */

  /** @override */
  getCenterPoint(coords) {
    const i = coords.i;
    if ( i !== undefined ) {
      const {j, k} = coords;
      return k !== undefined ? {x: j, y: i, elevation: k / this.size * this.distance} : {x: j, y: i};
    }
    const {x, y, elevation} = coords;
    return elevation !== undefined ? {x, y, elevation} : {x, y};
  }

  /* -------------------------------------------- */

  /** @override */
  getShape() {
    return [];
  }

  /* -------------------------------------------- */

  /** @override */
  getVertices(coords) {
    return [];
  }

  /* -------------------------------------------- */

  /** @override */
  getSnappedPoint({x, y, elevation}, behavior) {
    return elevation !== undefined ? {x, y, elevation} : {x, y};
  }

  /* -------------------------------------------- */

  /** @override */
  _measurePath(waypoints, {cost}, result) {

    // Prepare data for the starting point
    const w0 = waypoints[0];
    let o0 = this.getOffset(w0);
    let p0 = this.getCenterPoint(w0);

    // Iterate over additional path points
    const is3D = o0.k !== undefined;
    for ( let i = 1; i < waypoints.length; i++ ) {
      const w1 = waypoints[i];
      const o1 = this.getOffset(w1);
      const p1 = this.getCenterPoint(w1);
      const cost1 = w1.cost ?? cost;

      // Measure segment
      if ( w1.measure !== false ) {
        const segment = result.waypoints[i].backward;
        segment.distance = Math.hypot(p0.x - p1.x, p0.y - p1.y, is3D ? (p0.elevation - p1.elevation) / this.distance
          * this.size : 0) / this.size * this.distance;
        segment.euclidean = segment.distance;
        const offsetDistance = Math.hypot(o0.i - o1.i, o0.j - o1.j, is3D ? o0.k - o1.k : 0) / this.size * this.distance;
        if ( (cost1 === undefined) || (offsetDistance === 0) ) segment.cost = w1.teleport ? 0 : offsetDistance;
        else if ( typeof cost1 === "function" ) segment.cost = cost1(o0, o1, offsetDistance, w1);
        else segment.cost = Number(cost1);
      }

      o0 = o1;
      p0 = p1;
    }
  }

  /* -------------------------------------------- */

  /** @override */
  getDirectPath(waypoints) {
    if ( waypoints.length === 0 ) return [];
    let o0 = this.getOffset(waypoints[0]);
    const path = [o0];
    for ( let i = 1; i < waypoints.length; i++ ) {
      const o1 = this.getOffset(waypoints[i]);
      if ( (o0.i === o1.i) && (o0.j === o1.j) && (o0.k === o1.k) ) continue;
      path.push(o1);
      o0 = o1;
    }
    return path;
  }

  /* -------------------------------------------- */

  /** @override */
  getTranslatedPoint(point, direction, distance) {
    direction = Math.toRadians(direction);
    const dx = Math.cos(direction);
    const dy = Math.sin(direction);
    const s = distance / this.distance * this.size;
    const x = point.x + (dx * s);
    const y = point.y + (dy * s);
    const elevation = point.elevation;
    return elevation !== undefined ? {x, y, elevation} : {x, y};
  }

  /* -------------------------------------------- */

  /** @override */
  getCircle({x, y}, radius) {
    if ( radius <= 0 ) return [];
    const r = radius / this.distance * this.size;
    const n = Math.max(Math.ceil(Math.PI / Math.acos(Math.max(r - 0.25, 0) / r)), 4);
    const points = new Array(n);
    for ( let i = 0; i < n; i++ ) {
      const a = 2 * Math.PI * (i / n);
      points[i] = {x: x + (Math.cos(a) * r), y: y + (Math.sin(a) * r)};
    }
    return points;
  }

  /* -------------------------------------------- */

  /** @override */
  getCone(origin, radius, direction, angle) {
    if ( (radius <= 0) || (angle <= 0) ) return [];
    if ( angle >= 360 ) return this.getCircle(origin, radius);
    const r = radius / this.distance * this.size;
    const n = Math.max(Math.ceil(Math.PI / Math.acos(Math.max(r - 0.25, 0) / r) * (angle / 360)), 4);
    const a0 = Math.toRadians(direction - (angle / 2));
    const a1 = Math.toRadians(direction + (angle / 2));
    const points = new Array(n + 1);
    const {x, y} = origin;
    points[0] = {x, y};
    for ( let i = 0; i <= n; i++ ) {
      const a = Math.mix(a0, a1, i / n);
      points[i + 1] = {x: x + (Math.cos(a) * r), y: y + (Math.sin(a) * r)};
    }
    return points;
  }
}

/**
 * @import {SquareGridConfiguration, GridOffset2D, GridOffset3D, GridCoordinates2D,
 *   GridCoordinates3D} from "./_types.mjs"
 * @import {Point} from "../_types.mjs"
 * @import {GridDiagonalRule} from "../constants.mjs"
 */

/**
 * The square grid class.
 */
class SquareGrid extends BaseGrid {
  /**
   * The square grid constructor.
   * @param {SquareGridConfiguration} config   The grid configuration
   */
  constructor(config) {
    super(config);

    /**
     * The rule for diagonal measurement (see {@link CONST.GRID_DIAGONALS}).
     * @type {GridDiagonalRule}
     * @readonly
     */
    this.diagonals = config.diagonals ?? GRID_DIAGONALS.EQUIDISTANT;
  }

  /* -------------------------------------------- */

  /**
   * @override
   * @readonly
   */
  type = GRID_TYPES.SQUARE;

  /* -------------------------------------------- */

  /** @override */
  getOffset(coords) {
    let i = coords.i;
    let j;
    let k;
    if ( i !== undefined ) {
      j = coords.j;
      k = coords.k;
    } else {
      j = Math.floor(coords.x / this.size);
      i = Math.floor(coords.y / this.size);
      if ( coords.elevation !== undefined ) k = Math.floor((coords.elevation / this.distance) + 1e-8);
    }
    return k !== undefined ? {i, j, k} : {i, j};
  }

  /* -------------------------------------------- */

  /** @override */
  getOffsetRange({x, y, width, height}) {
    const i0 = Math.floor(y / this.size);
    const j0 = Math.floor(x / this.size);
    if ( !((width > 0) && (height > 0)) ) return [i0, j0, i0, j0];
    return [i0, j0, Math.ceil((y + height) / this.size) | 0, Math.ceil((x + width) / this.size) | 0];
  }

  /* -------------------------------------------- */

  /** @override */
  getAdjacentOffsets(coords) {
    const {i, j, k} = this.getOffset(coords);

    // 2D
    if ( k === undefined ) {

      // Non-diagonals
      if ( this.diagonals === GRID_DIAGONALS.ILLEGAL ) return [
        {i: i - 1, j},
        {i, j: j - 1},
        {i, j: j + 1},
        {i: i + 1, j}
      ];

      // Diagonals
      return [
        {i: i - 1, j: j - 1},
        {i: i - 1, j},
        {i: i - 1, j: j + 1},
        {i, j: j - 1},
        {i, j: j + 1},
        {i: i + 1, j: j - 1},
        {i: i + 1, j},
        {i: i + 1, j: j + 1}
      ];
    }

    // 3D
    else {

      // Non-diagonals
      if ( this.diagonals === GRID_DIAGONALS.ILLEGAL ) return [
        {i: i - 1, j, k},
        {i, j: j - 1, k},
        {i, j, k: k - 1},
        {i, j, k: k + 1},
        {i, j: j + 1, k},
        {i: i + 1, j, k}
      ];

      // Diagonals
      const offsets = [];
      for ( let di = -1; di <= 1; di++ ) {
        for ( let dj = -1; dj <= 1; dj++ ) {
          for ( let dk = -1; dk <= 1; dk++ ) {
            if ( (di === 0) && (dj === 0) && (dk === 0) ) continue;
            offsets.push({i: i + di, j: j + dj, k: k + dk});
          }
        }
      }
      return offsets;
    }
  }

  /* -------------------------------------------- */

  /** @override */
  testAdjacency(coords1, coords2) {
    const {i: i1, j: j1, k: k1} = this.getOffset(coords1);
    const {i: i2, j: j2, k: k2} = this.getOffset(coords2);
    const di = Math.abs(i1 - i2);
    const dj = Math.abs(j1 - j2);
    const dk = k1 !== undefined ? Math.abs(k1 - k2) : 0;
    const diagonals = this.diagonals !== GRID_DIAGONALS.ILLEGAL;
    return diagonals ? Math.max(di, dj, dk) === 1 : (di + dj + dk) === 1;
  }

  /* -------------------------------------------- */

  /** @override */
  getShiftedOffset(coords, direction) {
    let di = 0;
    let dj = 0;
    let dk = 0;
    if ( direction & MOVEMENT_DIRECTIONS.UP ) di--;
    if ( direction & MOVEMENT_DIRECTIONS.DOWN ) di++;
    if ( direction & MOVEMENT_DIRECTIONS.LEFT ) dj--;
    if ( direction & MOVEMENT_DIRECTIONS.RIGHT ) dj++;
    if ( direction & MOVEMENT_DIRECTIONS.DESCEND ) dk--;
    if ( direction & MOVEMENT_DIRECTIONS.ASCEND ) dk++;
    if ( ((Math.abs(di) + Math.abs(dj) + Math.abs(dk)) > 1) && (this.diagonals === GRID_DIAGONALS.ILLEGAL) ) {
      // Diagonal movement is not allowed
      di = 0;
      dj = 0;
      dk = 0;
    }
    const offset = this.getOffset(coords);
    offset.i += di;
    offset.j += dj;
    if ( offset.k !== undefined ) offset.k += dk;
    return offset;
  }

  /* -------------------------------------------- */

  /** @override */
  getShiftedPoint(point, direction) {
    const topLeft = this.getTopLeftPoint(point);
    const shifted = this.getTopLeftPoint(this.getShiftedOffset(topLeft, direction));
    shifted.x = point.x + (shifted.x - topLeft.x);
    shifted.y = point.y + (shifted.y - topLeft.y);
    if ( shifted.elevation !== undefined ) {
      shifted.elevation = point.elevation + (shifted.elevation - topLeft.elevation);
    }
    return shifted;
  }

  /* -------------------------------------------- */

  /** @override */
  getTopLeftPoint(coords) {
    let i = coords.i;
    let j;
    let k;
    if ( i !== undefined ) {
      j = coords.j;
      k = coords.k;
    } else {
      const {x, y, elevation} = coords;
      j = Math.floor(x / this.size);
      i = Math.floor(y / this.size);
      if ( elevation !== undefined ) k = Math.floor((elevation / this.distance) + 1e-8);
    }
    const x = j * this.size;
    const y = i * this.size;
    if ( k === undefined ) return {x, y};
    const elevation = k * this.distance;
    return {x, y, elevation};
  }

  /* -------------------------------------------- */

  /** @override */
  getCenterPoint(coords) {
    let i = coords.i;
    let j;
    let k;
    if ( i !== undefined ) {
      j = coords.j;
      k = coords.k;
    } else {
      const {x, y, elevation} = coords;
      j = Math.floor(x / this.size);
      i = Math.floor(y / this.size);
      if ( elevation !== undefined ) k = Math.floor((elevation / this.distance) + 1e-8);
    }
    const x = (j + 0.5) * this.size;
    const y = (i + 0.5) * this.size;
    if ( k === undefined ) return {x, y};
    const elevation = (k + 0.5) * this.distance;
    return {x, y, elevation};
  }

  /* -------------------------------------------- */

  /** @override */
  getShape() {
    const s = this.size / 2;
    return [{x: -s, y: -s}, {x: s, y: -s}, {x: s, y: s}, {x: -s, y: s}];
  }

  /* -------------------------------------------- */

  /** @override */
  getVertices(coords) {
    const {i, j} = this.getOffset(coords);
    const x0 = j * this.size;
    const x1 = (j + 1) * this.size;
    const y0 = i * this.size;
    const y1 = (i + 1) * this.size;
    return [{x: x0, y: y0}, {x: x1, y: y0}, {x: x1, y: y1}, {x: x0, y: y1}];
  }

  /* -------------------------------------------- */

  /** @override */
  getSnappedPoint(point, {mode, resolution=1}) {
    if ( mode & -65524 ) throw new Error("Invalid snapping mode");
    if ( mode === 0 ) {
      return point.elevation !== undefined ? {x: point.x, y: point.y, elevation: point.elevation}
        : {x: point.x, y: point.y};
    }

    let nearest;
    let distance;
    const keepNearest = candidate => {
      if ( !nearest ) return nearest = candidate;
      const {x, y} = point;
      distance ??= ((nearest.x - x) ** 2) + ((nearest.y - y) ** 2);
      const d = ((candidate.x - x) ** 2) + ((candidate.y - y) ** 2);
      if ( d < distance ) {
        nearest = candidate;
        distance = d;
      }
      return nearest;
    };

    // Any edge = Any side
    if ( !(mode & 0x2) ) {
      // Horizontal (Top/Bottom) side + Vertical (Left/Right) side = Any edge
      if ( (mode & 0x3000) && (mode & 0xC000) ) mode |= 0x2;
      // Horizontal (Top/Bottom) side
      else if ( mode & 0x3000 ) keepNearest(this.#snapToTopOrBottom(point, resolution));
      // Vertical (Left/Right) side
      else if ( mode & 0xC000 ) keepNearest(this.#snapToLeftOrRight(point, resolution));
    }

    // With vertices (= corners)
    if ( mode & 0xFF0 ) {
      switch ( mode & -65521 ) {
        case 0x0: keepNearest(this.#snapToVertex(point, resolution)); break;
        case 0x1: keepNearest(this.#snapToVertexOrCenter(point, resolution)); break;
        case 0x2: keepNearest(this.#snapToEdgeOrVertex(point, resolution)); break;
        case 0x3: keepNearest(this.#snapToEdgeOrVertexOrCenter(point, resolution)); break;
      }
    }
    // Without vertices
    else {
      switch ( mode & -65521 ) {
        case 0x1: keepNearest(this.#snapToCenter(point, resolution)); break;
        case 0x2: keepNearest(this.#snapToEdge(point, resolution)); break;
        case 0x3: keepNearest(this.#snapToEdgeOrCenter(point, resolution)); break;
      }
    }

    return point.elevation === undefined ? nearest
      : {x: nearest.x, y: nearest.y, elevation: Math.round((point.elevation / this.distance) + 1e-8) * this.distance};
  }

  /* -------------------------------------------- */

  /**
   * Snap the point to the nearest center of a square.
   * @param {Point} point          The point
   * @param {number} resolution    The grid resolution
   * @returns {Point}              The snapped point
   */
  #snapToCenter({x, y}, resolution) {
    const s = this.size / resolution;
    const t = this.size / 2;
    return {
      x: (Math.round((x - t) / s) * s) + t,
      y: (Math.round((y - t) / s) * s) + t
    };
  }

  /* -------------------------------------------- */

  /**
   * Snap the point to the nearest vertex of a square.
   * @param {Point} point          The point
   * @param {number} resolution    The grid resolution
   * @returns {Point}              The snapped point
   */
  #snapToVertex({x, y}, resolution) {
    const s = this.size / resolution;
    const t = this.size / 2;
    return {
      x: ((Math.floor((x - t) / s) + 0.5) * s) + t,
      y: ((Math.floor((y - t) / s) + 0.5) * s) + t
    };
  }

  /* -------------------------------------------- */

  /**
   * Snap the point to the nearest vertex or center of a square.
   * @param {Point} point          The point
   * @param {number} resolution    The grid resolution
   * @returns {Point}              The snapped point
   */
  #snapToVertexOrCenter({x, y}, resolution) {
    const s = this.size / resolution;
    const t = this.size / 2;
    const c0 = (x - t) / s;
    const r0 = (y - t) / s;
    const c1 = Math.round(c0 + r0);
    const r1 = Math.round(r0 - c0);
    return {
      x: ((c1 - r1) * s / 2) + t,
      y: ((c1 + r1) * s / 2) + t
    };
  }

  /* -------------------------------------------- */

  /**
   * Snap the point to the nearest edge of a square.
   * @param {Point} point          The point
   * @param {number} resolution    The grid resolution
   * @returns {Point}              The snapped point
   */
  #snapToEdge({x, y}, resolution) {
    const s = this.size / resolution;
    const t = this.size / 2;
    const c0 = (x - t) / s;
    const r0 = (y - t) / s;
    const c1 = Math.floor(c0 + r0);
    const r1 = Math.floor(r0 - c0);
    return {
      x: ((c1 - r1) * s / 2) + t,
      y: ((c1 + r1 + 1) * s / 2) + t
    };
  }

  /* -------------------------------------------- */

  /**
   * Snap the point to the nearest edge or center of a square.
   * @param {Point} point          The point
   * @param {number} resolution    The grid resolution
   * @returns {Point}              The snapped point
   */
  #snapToEdgeOrCenter({x, y}, resolution) {
    const s = this.size / resolution;
    const t = this.size / 2;
    const c0 = (x - t) / s;
    const r0 = (y - t) / s;
    const x0 = (Math.round(c0) * s) + t;
    const y0 = (Math.round(r0) * s) + t;
    if ( Math.max(Math.abs(x - x0), Math.abs(y - y0)) <= s / 4 ) {
      return {x: x0, y: y0};
    }
    const c1 = Math.floor(c0 + r0);
    const r1 = Math.floor(r0 - c0);
    return {
      x: ((c1 - r1) * s / 2) + t,
      y: ((c1 + r1 + 1) * s / 2) + t
    };
  }

  /* -------------------------------------------- */

  /**
   * Snap the point to the nearest edge or vertex of a square.
   * @param {Point} point          The point
   * @param {number} resolution    The grid resolution
   * @returns {Point}              The snapped point
   */
  #snapToEdgeOrVertex({x, y}, resolution) {
    const s = this.size / resolution;
    const t = this.size / 2;
    const c0 = (x - t) / s;
    const r0 = (y - t) / s;
    const x0 = ((Math.floor(c0) + 0.5) * s) + t;
    const y0 = ((Math.floor(r0) + 0.5) * s) + t;
    if ( Math.max(Math.abs(x - x0), Math.abs(y - y0)) <= s / 4 ) {
      return {x: x0, y: y0};
    }
    const c1 = Math.floor(c0 + r0);
    const r1 = Math.floor(r0 - c0);
    return {
      x: ((c1 - r1) * s / 2) + t,
      y: ((c1 + r1 + 1) * s / 2) + t
    };
  }

  /* -------------------------------------------- */

  /**
   * Snap the point to the nearest edge, vertex, or center of a square.
   * @param {Point} point          The point
   * @param {number} resolution    The grid resolution
   * @returns {Point}              The snapped point
   */
  #snapToEdgeOrVertexOrCenter({x, y}, resolution) {
    const s = this.size / (resolution * 2);
    return {
      x: Math.round(x / s) * s,
      y: Math.round(y / s) * s
    };
  }

  /* -------------------------------------------- */

  /**
   * Snap the point to the nearest top/bottom side of a square.
   * @param {Point} point          The point
   * @param {number} resolution    The grid resolution
   * @returns {Point}              The snapped point
   */
  #snapToTopOrBottom({x, y}, resolution) {
    const s = this.size / resolution;
    const t = this.size / 2;
    return {
      x: (Math.round((x - t) / s) * s) + t,
      y: ((Math.floor((y - t) / s) + 0.5) * s) + t
    };
  }

  /* -------------------------------------------- */

  /**
   * Snap the point to the nearest left/right side of a square.
   * @param {Point} point          The point
   * @param {number} resolution    The grid resolution
   * @returns {Point}              The snapped point
   */
  #snapToLeftOrRight({x, y}, resolution) {
    const s = this.size / resolution;
    const t = this.size / 2;
    return {
      x: ((Math.floor((x - t) / s) + 0.5) * s) + t,
      y: (Math.round((y - t) / s) * s) + t
    };
  }

  /* -------------------------------------------- */

  /** @override */
  _measurePath(waypoints, {cost}, result) {

    // Convert to point coordiantes
    const toPoint = coords => {
      if ( coords.x !== undefined ) return coords;
      return this.getCenterPoint(coords);
    };

    // Prepare data for the starting point
    const w0 = waypoints[0];
    let o0 = this.getOffset(w0);
    let p0 = toPoint(w0);

    // Iterate over additional path points
    const is3D = o0.k !== undefined;
    const diagonals = this.diagonals;
    let l0 = diagonals === GRID_DIAGONALS.ALTERNATING_2 ? 1.0 : 0.0;
    let dx0 = l0;
    let dy0 = l0;
    let dz0 = l0;
    let nd = l0 * 1.5;
    for ( let i = 1; i < waypoints.length; i++ ) {
      const w1 = waypoints[i];
      const o1 = this.getOffset(w1);
      const p1 = toPoint(w1);
      const cost1 = w1.cost ?? cost;

      // Determine the number of moves total, number of diagonal moves, and cost of the moves
      if ( w1.measure !== false ) {
        let di = Math.abs(o0.i - o1.i);
        let dj = Math.abs(o0.j - o1.j);
        if ( di < dj ) [di, dj] = [dj, di];
        let dk = 0;
        if ( is3D ) {
          dk = Math.abs(o0.k - o1.k);
          if ( dj < dk ) [dj, dk] = [dk, dj];
          if ( di < dj ) [di, dj] = [dj, di];
        }
        let n = di; // The number of moves total
        let d = dj; // The number of diagonal moves
        let c; // The cost of the moves
        const nd0 = nd;
        switch ( diagonals ) {
          case GRID_DIAGONALS.EQUIDISTANT: c = di; break;
          case GRID_DIAGONALS.EXACT: c = di + (((Math.SQRT2 - 1) * (dj - dk)) + ((Math.SQRT3 - 1) * dk)); break;
          case GRID_DIAGONALS.APPROXIMATE: c = di + ((0.5 * (dj - dk)) + (0.75 * dk)); break;
          case GRID_DIAGONALS.RECTILINEAR: c = di + (dj + dk); break;
          case GRID_DIAGONALS.ALTERNATING_1:
          case GRID_DIAGONALS.ALTERNATING_2:
            nd += (dj + (0.5 * dk));
            c = di + (Math.floor(nd / 2) - Math.floor(nd0 / 2));
            break;
          case GRID_DIAGONALS.ILLEGAL:
            n = di + (dj + dk);
            d = 0;
            c = n;
            break;
        }

        // Determine the distance of the segment
        let dx = Math.abs(p0.x - p1.x) / this.size;
        let dy = Math.abs(p0.y - p1.y) / this.size;
        if ( dx < dy ) [dx, dy] = [dy, dx];
        let dz = 0;
        if ( is3D ) {
          dz = Math.abs(p0.elevation - p1.elevation) / this.distance;
          if ( dy < dz ) [dy, dz] = [dz, dy];
          if ( dx < dy ) [dx, dy] = [dy, dx];
        }
        let l; // The distance of the segment
        switch ( diagonals ) {
          case GRID_DIAGONALS.EQUIDISTANT: l = dx; break;
          case GRID_DIAGONALS.EXACT: l = dx + (((Math.SQRT2 - 1) * (dy - dz)) + ((Math.SQRT3 - 1) * dz)); break;
          case GRID_DIAGONALS.APPROXIMATE: l = dx + ((0.5 * (dy - dz)) + (0.75 * dz)); break;
          case GRID_DIAGONALS.RECTILINEAR: l = dx + (dy + dz); break;
          case GRID_DIAGONALS.ALTERNATING_1:
          case GRID_DIAGONALS.ALTERNATING_2: {
            dx0 += dx;
            dy0 += dy;
            dz0 += dz;
            const fx = Math.floor(dx0);
            const fy = Math.floor(dy0);
            const fz = Math.floor(dz0);
            const a = fx + (0.5 * fy) + (0.25 * fz);
            const a0 = Math.floor(a);
            const a1 = Math.floor(a + 1);
            const a2 = Math.floor(a + 1.5);
            const a3 = Math.floor(a + 1.75);
            const mx = dx0 - fx;
            const my = dy0 - fy;
            const mz = dz0 - fz;
            const l1 = (a0 * (1 - mx)) + (a1 * (mx - my)) + (a2 * (my - mz)) + (a3 * mz);
            l = l1 - l0;
            l0 = l1;
            break;
          }
          case GRID_DIAGONALS.ILLEGAL: l = dx + (dy + dz); break;
        }
        if ( l.almostEqual(c) ) l = c;

        const segment = result.waypoints[i].backward;
        segment.distance = l * this.distance;
        if ( (cost1 === undefined) || (c === 0) ) segment.cost = w1.teleport ? 0 : c * this.distance;
        else if ( typeof cost1 === "function" ) segment.cost = w1.teleport ? cost1(o0, o1, c * this.distance, w1)
          : this.#calculateCost(o0, o1, cost1, nd0, w1);
        else segment.cost = Number(cost1);
        segment.spaces = n;
        segment.diagonals = d;
        segment.euclidean = Math.hypot(p0.x - p1.x, p0.y - p1.y, is3D ? (p0.elevation - p1.elevation) / this.distance
          * this.size : 0) / this.size * this.distance;
      }

      o0 = o1;
      p0 = p1;
    }
  }

  /* -------------------------------------------- */

  /**
   * Calculate the cost of the direct path segment.
   * @template SegmentData
   * @overload
   * @param {GridOffset2D} from      The coordinates the segment starts from
   * @param {GridOffset2D} to        The coordinates the segment goes to
   * @param {GridMeasurePathCostFunction2D<SegmentData>} cost    The cost function
   * @param {number} diagonals       The number of diagonal moves that have been performed already
   * @param {SegmentData} segment    The segment data
   * @returns {number}               The cost of the path segment
   */
  /**
   * @overload
   * @param {GridOffset3D} from      The coordinates the segment starts from
   * @param {GridOffset3D} to        The coordinates the segment goes to
   * @param {GridMeasurePathCostFunction3D<SegmentData>} cost    The cost function
   * @param {number} diagonals       The number of diagonal moves that have been performed already
   * @param {SegmentData} segment    The segment data
   * @returns {number}               The cost of the path segment
   */
  #calculateCost(from, to, cost, diagonals, segment) {
    const path = this.getDirectPath([from, to]);
    if ( path.length <= 1 ) return 0;

    // Prepare data for the starting point
    let o0 = path[0];
    let c = 0;

    // Iterate over additional path points
    for ( let i = 1; i < path.length; i++ ) {
      const o1 = path[i];

      // Determine the normalized distance
      let k;
      const m = (o0.i === o1.i) + (o0.j === o1.j) + (o0.k === o1.k);
      if ( m === 2 ) k = 1;
      else if ( m === 1 ) {
        switch ( this.diagonals ) {
          case GRID_DIAGONALS.EQUIDISTANT: k = 1; break;
          case GRID_DIAGONALS.EXACT: k = Math.SQRT2; break;
          case GRID_DIAGONALS.APPROXIMATE: k = 1.5; break;
          case GRID_DIAGONALS.RECTILINEAR: k = 2; break;
          case GRID_DIAGONALS.ALTERNATING_1:
          case GRID_DIAGONALS.ALTERNATING_2:
            k = 1 + (Math.floor((diagonals + 1) / 2) - Math.floor(diagonals / 2));
            break;
        }
        diagonals += 1;
      } else {
        switch ( this.diagonals ) {
          case GRID_DIAGONALS.EQUIDISTANT: k = 1; break;
          case GRID_DIAGONALS.EXACT: k = Math.SQRT3; break;
          case GRID_DIAGONALS.APPROXIMATE: k = 1.75; break;
          case GRID_DIAGONALS.RECTILINEAR: k = 3; break;
          case GRID_DIAGONALS.ALTERNATING_1:
          case GRID_DIAGONALS.ALTERNATING_2:
            k = 1 + (Math.floor((diagonals + 1.5) / 2) - Math.floor(diagonals / 2));
            break;
        }
        diagonals += 1.5;
      }

      // Calculate and accumulate the cost
      c += cost(o0, o1, k * this.distance, segment);

      o0 = o1;
    }

    return c;
  }

  /* -------------------------------------------- */

  /** @override */
  getDirectPath(waypoints) {
    if ( waypoints.length === 0 ) return [];
    const w0 = waypoints[0];
    if ( (w0.k !== undefined) || (w0.elevation !== undefined) ) return this.#getDirectPath3D(waypoints);
    else return this.#getDirectPath2D(waypoints);
  }

  /* -------------------------------------------- */

  /**
   * Returns the sequence of grid offsets of a shortest, direct path passing through the given waypoints.
   * @see {@link https://en.wikipedia.org/wiki/Bresenham's_line_algorithm}
   * @see {@link https://playtechs.blogspot.com/2007/03/raytracing-on-grid.html}
   * @param {GridCoordinates2D[]} waypoints    The waypoints the path must pass through
   * @returns {GridOffset2D[]}                 The sequence of grid offsets of a shortest, direct path
   */
  #getDirectPath2D(waypoints) {

    // Prepare data for the starting point
    const o0 = this.getOffset(waypoints[0]);
    let {i: i0, j: j0} = o0;
    const path = [o0];

    // Iterate over additional path points
    const diagonals = this.diagonals !== GRID_DIAGONALS.ILLEGAL;
    for ( let i = 1; i < waypoints.length; i++ ) {
      const o1 = this.getOffset(waypoints[i]);
      const {i: i1, j: j1} = o1;
      if ( (i0 === i1) && (j0 === j1) ) continue;

      // Walk from (i0, j0) to (i1, j1)
      const di = Math.abs(i0 - i1);
      const dj = 0 - Math.abs(j0 - j1);
      const si = i0 < i1 ? 1 : -1;
      const sj = j0 < j1 ? 1 : -1;
      let e = di + dj;
      if ( diagonals ) {
        for ( ;; ) {
          const e2 = e * 2;
          if ( e2 >= dj ) {
            e += dj;
            i0 += si;
          }
          if ( e2 <= di ) {
            e += di;
            j0 += sj;
          }
          if ( (i0 === i1) && (j0 === j1) ) break;
          path.push({i: i0, j: j0});
        }
      } else {
        const di2 = 2 * di;
        const dj2 = 2 * dj;
        for ( ;; ) {
          if ( e > 0 ) {
            e += dj2;
            i0 += si;
          } else {
            e += di2;
            j0 += sj;
          }
          if ( (i0 === i1) && (j0 === j1) ) break;
          path.push({i: i0, j: j0});
        }
      }
      path.push(o1);

      i0 = i1;
      j0 = j1;
    }

    return path;
  }

  /* -------------------------------------------- */

  /**
   * Returns the sequence of grid offsets of a shortest, direct path passing through the given waypoints.
   * @see {@link https://www.geeksforgeeks.org/bresenhams-algorithm-for-3-d-line-drawing}
   * @see {@link http://www.cse.yorku.ca/~amana/research/grid.pdf}
   * @param {GridCoordinates3D[]} waypoints    The waypoints the path must pass through
   * @returns {GridOffset3D[]}                 The sequence of grid offsets of a shortest, direct path
   */
  #getDirectPath3D(waypoints) {

    // Prepare data for the starting point
    const o0 = this.getOffset(waypoints[0]);
    let {i: i0, j: j0, k: k0} = o0;
    const path = [o0];

    // Iterate over additional path points
    const diagonals = this.diagonals !== GRID_DIAGONALS.ILLEGAL;
    for ( let i = 1; i < waypoints.length; i++ ) {
      const o1 = this.getOffset(waypoints[i]);
      const {i: i1, j: j1, k: k1} = o1;
      if ( (i0 === i1) && (j0 === j1) && (k0 === k1) ) continue;

      // Walk from (i0, j0, k0) to (i1, j1, k1)
      const di = Math.abs(i0 - i1);
      const dj = Math.abs(j0 - j1);
      const dk = Math.abs(k0 - k1);
      const si = i0 < i1 ? 1 : -1;
      const sj = j0 < j1 ? 1 : -1;
      const sk = k0 < k1 ? 1 : -1;
      if ( diagonals ) {
        const di2 = 2 * di;
        const dj2 = 2 * dj;
        const dk2 = 2 * dk;
        if ( (di >= dj) && (di >= dk) ) {
          let ej = 0 - di;
          let ek = ej;
          for ( ;; ) {
            ej += dj2;
            ek += dk2;
            i0 += si;
            if ( ej >= 0 ) {
              ej -= di2;
              j0 += sj;
            }
            if ( ek >= 0 ) {
              ek -= di2;
              k0 += sk;
            }
            if ( i0 === i1 ) break;
            path.push({i: i0, j: j0, k: k0});
          }
        } else if ( (dj >= di) && (dj >= dk) ) {
          let ei = 0 - dj;
          let ek = ei;
          for ( ;; ) {
            ei += di2;
            ek += dk2;
            j0 += sj;
            if ( ei >= 0 ) {
              ei -= dj2;
              i0 += si;
            }
            if ( ek >= 0 ) {
              ek -= dj2;
              k0 += sk;
            }
            if ( j0 === j1 ) break;
            path.push({i: i0, j: j0, k: k0});
          }
        } else {
          let ei = 0 - dk;
          let ej = ei;
          for ( ;; ) {
            ei += di2;
            ej += dj2;
            k0 += sk;
            if ( ei >= 0 ) {
              ei -= dk2;
              i0 += si;
            }
            if ( ej >= 0 ) {
              ej -= dk2;
              j0 += sj;
            }
            if ( k0 === k1 ) break;
            path.push({i: i0, j: j0, k: k0});
          }
        }
      } else {
        const di1 = di || 1;
        const dj1 = dj || 1;
        const dk1 = dk || 1;
        const tdi = dj1 * dk1;
        const tdj = di1 * dk1;
        const tdk = di1 * dj1;
        const tm = (di1 * dj1 * dk1) + 1;
        let ti = di > 0 ? tdi : tm;
        let tj = dj > 0 ? tdj : tm;
        let tk = dk > 0 ? tdk : tm;
        for ( ;; ) {
          if ( ti < tj ) {
            if ( ti <= tk ) {
              ti += tdi;
              i0 += si;
            } else {
              tk += tdk;
              k0 += sk;
            }
          } else {
            if ( tj <= tk ) {
              tj += tdj;
              j0 += sj;
            } else {
              tk += tdk;
              k0 += sk;
            }
          }
          if ( (i0 === i1) && (j0 === j1) && (k0 === k1) ) break;
          path.push({i: i0, j: j0, k: k0});
        }
      }
      path.push(o1);

      i0 = i1;
      j0 = j1;
      k0 = k1;
    }

    return path;
  }

  /* -------------------------------------------- */

  /** @override */
  getTranslatedPoint(point, direction, distance) {
    direction = Math.toRadians(direction);
    const dx = Math.cos(direction);
    const dy = Math.sin(direction);
    const adx = Math.abs(dx);
    const ady = Math.abs(dy);
    let s = distance / this.distance;
    switch ( this.diagonals ) {
      case GRID_DIAGONALS.EQUIDISTANT: s /= Math.max(adx, ady); break;
      case GRID_DIAGONALS.EXACT: s /= (Math.max(adx, ady) + ((Math.SQRT2 - 1) * Math.min(adx, ady))); break;
      case GRID_DIAGONALS.APPROXIMATE: s /= (Math.max(adx, ady) + (0.5 * Math.min(adx, ady))); break;
      case GRID_DIAGONALS.ALTERNATING_1: {
        let a = Math.max(adx, ady);
        const b = Math.min(adx, ady);
        const t = (2 * a) + b;
        let k = Math.floor(s * b / t);
        if ( (s * b) - (k * t) > a ) {
          a += b;
          k = -1 - k;
        }
        s = (s - k) / a;
        break;
      }
      case GRID_DIAGONALS.ALTERNATING_2: {
        let a = Math.max(adx, ady);
        const b = Math.min(adx, ady);
        const t = (2 * a) + b;
        let k = Math.floor(s * b / t);
        if ( (s * b) - (k * t) > a + b ) {
          k += 1;
        } else {
          a += b;
          k = -k;
        }
        s = (s - k) / a;
        break;
      }
      case GRID_DIAGONALS.RECTILINEAR:
      case GRID_DIAGONALS.ILLEGAL: s /= (adx + ady); break;
    }
    s *= this.size;
    const x = point.x + (dx * s);
    const y = point.y + (dy * s);
    const elevation = point.elevation;
    return elevation !== undefined ? {x, y, elevation} : {x, y};
  }

  /* -------------------------------------------- */

  /** @override */
  getCircle(center, radius) {
    if ( radius <= 0 ) return [];
    switch ( this.diagonals ) {
      case GRID_DIAGONALS.EQUIDISTANT: return this.#getCircleEquidistant(center, radius);
      case GRID_DIAGONALS.EXACT: return this.#getCircleExact(center, radius);
      case GRID_DIAGONALS.APPROXIMATE: return this.#getCircleApproximate(center, radius);
      case GRID_DIAGONALS.ALTERNATING_1: return this.#getCircleAlternating(center, radius, false);
      case GRID_DIAGONALS.ALTERNATING_2: return this.#getCircleAlternating(center, radius, true);
      case GRID_DIAGONALS.RECTILINEAR:
      case GRID_DIAGONALS.ILLEGAL: return this.#getCircleRectilinear(center, radius);
    }
  }

  /* -------------------------------------------- */

  /**
   * Get the circle polygon given the radius in grid units (EQUIDISTANT).
   * @param {Point} center      The center point of the circle.
   * @param {number} radius     The radius in grid units (positive).
   * @returns {Point[]}         The points of the circle polygon.
   */
  #getCircleEquidistant({x, y}, radius) {
    const r = radius / this.distance * this.size;
    const x0 = x + r;
    const x1 = x - r;
    const y0 = y + r;
    const y1 = y - r;
    return [{x: x0, y: y0}, {x: x1, y: y0}, {x: x1, y: y1}, {x: x0, y: y1}];
  }

  /* -------------------------------------------- */

  /**
   * Get the circle polygon given the radius in grid units (EXACT).
   * @param {Point} center      The center point of the circle.
   * @param {number} radius     The radius in grid units (positive).
   * @returns {Point[]}         The points of the circle polygon.
   */
  #getCircleExact({x, y}, radius) {
    const r = radius / this.distance * this.size;
    const s = r / Math.SQRT2;
    return [
      {x: x + r, y},
      {x: x + s, y: y + s},
      {x: x, y: y + r },
      {x: x - s, y: y + s},
      {x: x - r, y},
      {x: x - s, y: y - s},
      {x: x, y: y - r},
      {x: x + s, y: y - s}
    ];
  }

  /* -------------------------------------------- */

  /**
   * Get the circle polygon given the radius in grid units (APPROXIMATE).
   * @param {Point} center      The center point of the circle.
   * @param {number} radius     The radius in grid units (positive).
   * @returns {Point[]}         The points of the circle polygon.
   */
  #getCircleApproximate({x, y}, radius) {
    const r = radius / this.distance * this.size;
    const s = r / 1.5;
    return [
      {x: x + r, y},
      {x: x + s, y: y + s},
      {x: x, y: y + r },
      {x: x - s, y: y + s},
      {x: x - r, y},
      {x: x - s, y: y - s},
      {x: x, y: y - r},
      {x: x + s, y: y - s}
    ];
  }

  /* -------------------------------------------- */

  /**
   * Get the circle polygon given the radius in grid units (ALTERNATING_1/2).
   * @param {Point} center           The center point of the circle.
   * @param {number} radius          The radius in grid units (positive).
   * @param {boolean} firstDouble    2/1/2 instead of 1/2/1?
   * @returns {Point[]}              The points of the circle polygon.
   */
  #getCircleAlternating(center, radius, firstDouble) {
    const r = radius / this.distance;
    const points = [];
    let dx = 0;
    let dy = 0;

    // Generate points of the first quarter
    if ( firstDouble ) {
      points.push({x: r - dx, y: dy});
      dx++;
      dy++;
    }
    for ( ;; ) {
      if ( r - dx <= dy ) {
        [dx, dy] = [dy - 1, dx - 1];
        break;
      }
      points.push({x: r - dx, y: dy});
      dy++;
      if ( r - dx <= dy ) {
        points.push({x: r - dx, y: r - dx});
        if ( dx !== 0 ) {
          points.push({x: dy - 1, y: r - dx});
          [dx, dy] = [dy - 2, dx - 1];
        }
        break;
      }
      points.push({x: r - dx, y: dy});
      dx++;
      dy++;
    }
    for ( ;; ) {
      if ( dx === 0 ) break;
      points.push({x: dx, y: r - dy});
      dx--;
      if ( dx === 0 ) break;
      points.push({x: dx, y: r - dy});
      dx--;
      dy--;
    }

    // Generate the points of the other three quarters by mirroring the first
    const n = points.length;
    for ( let i = 0; i < n; i++ ) {
      const p = points[i];
      points.push({x: -p.y, y: p.x});
    }
    for ( let i = 0; i < n; i++ ) {
      const p = points[i];
      points.push({x: -p.x, y: -p.y});
    }
    for ( let i = 0; i < n; i++ ) {
      const p = points[i];
      points.push({x: p.y, y: -p.x});
    }

    // Scale and center the polygon points
    for ( let i = 0; i < 4 * n; i++ ) {
      const p = points[i];
      p.x = (p.x * this.size) + center.x;
      p.y = (p.y * this.size) + center.y;
    }
    return points;
  }

  /* -------------------------------------------- */

  /**
   * Get the circle polygon given the radius in grid units (RECTILINEAR/ILLEGAL).
   * @param {Point} center      The center point of the circle.
   * @param {number} radius     The radius in grid units (positive).
   * @returns {Point[]}         The points of the circle polygon.
   */
  #getCircleRectilinear({x, y}, radius) {
    const r = radius / this.distance * this.size;
    return [{x: x + r, y}, {x, y: y + r}, {x: x - r, y}, {x, y: y - r}];
  }

  /* -------------------------------------------- */

  /** @override */
  calculateDimensions(sceneWidth, sceneHeight, padding) {
    // Note: Do not replace `* (1 / this.size)` by `/ this.size`!
    // It could change the result and therefore break certain scenes.
    const x = Math.ceil((padding * sceneWidth) * (1 / this.size)) * this.size;
    const y = Math.ceil((padding * sceneHeight) * (1 / this.size)) * this.size;
    const width = sceneWidth + (2 * x);
    const height = sceneHeight + (2 * y);
    const rows = Math.ceil((height / this.size) - 1e-6);
    const columns = Math.ceil((width / this.size) - 1e-6);
    return {width, height, x, y, rows, columns};
  }

  /* -------------------------------------------- */
  /*  Deprecations and Compatibility              */
  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  getCenter(x, y) {
    const msg = "SquareGrid#getCenter is deprecated. Use SquareGrid#getCenterPoint instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return this.getTopLeft(x, y).map(c => c + (this.size / 2));
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  getSnappedPosition(x, y, interval=1, options={}) {
    const msg = "SquareGrid#getSnappedPosition is deprecated. "
      + "Use BaseGrid#getSnappedPoint instead for non-Euclidean measurements.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    if ( interval === 0 ) return {x: Math.round(x), y: Math.round(y)};
    const [x0, y0] = this.#getNearestVertex(x, y);
    let dx = 0;
    let dy = 0;
    if ( interval !== 1 ) {
      const delta = this.size / interval;
      dx = Math.round((x - x0) / delta) * delta;
      dy = Math.round((y - y0) / delta) * delta;
    }
    return {
      x: Math.round(x0 + dx),
      y: Math.round(y0 + dy)
    };
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  #getNearestVertex(x, y) {
    return [x.toNearest(this.size), y.toNearest(this.size)];
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  getGridPositionFromPixels(x, y) {
    const msg = "BaseGrid#getGridPositionFromPixels is deprecated. Use BaseGrid#getOffset instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return [Math.floor(y / this.size), Math.floor(x / this.size)];
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  getPixelsFromGridPosition(row, col) {
    const msg = "BaseGrid#getPixelsFromGridPosition is deprecated. Use BaseGrid#getTopLeftPoint instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return [col * this.size, row * this.size];
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  shiftPosition(x, y, dx, dy, options={}) {
    const msg = "BaseGrid#shiftPosition is deprecated. Use BaseGrid#getShiftedPoint instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    const [row, col] = this.getGridPositionFromPixels(x, y);
    return this.getPixelsFromGridPosition(row+dy, col+dx);
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  measureDistances(segments, options={}) {
    const msg = "SquareGrid#measureDistances is deprecated. "
      + "Use SquareGrid#measurePath instead, which returns grid distance (gridSpaces: true) and Euclidean distance (gridSpaces: false).";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    if ( !options.gridSpaces ) return super.measureDistances(segments, options);
    return segments.map(s => {
      const r = s.ray;
      const nx = Math.abs(Math.ceil(r.dx / this.size));
      const ny = Math.abs(Math.ceil(r.dy / this.size));

      // Determine the number of straight and diagonal moves
      const nd = Math.min(nx, ny);
      const ns = Math.abs(ny - nx);

      // Linear distance for all moves
      return (nd + ns) * this.distance;
    });
  }
}

/**
 * @import {HexagonalGridConfiguration, GridOffset2D, GridOffset3D, HexagonalGridCube2D, HexagonalGridCube3D,
 *   HexagonalGridCoordinates2D, HexagonalGridCoordinates3D} from "./_types.mjs"
 * @import {Point, ElevatedPoint} from "../_types.mjs"
 * @import {GridDiagonalRule} from "../constants.mjs"
 */

/**
 * The hexagonal grid class.
 * @extends {BaseGrid<HexagonalGridCoordinates2D, HexagonalGridCoordinates3D>}
 */
class HexagonalGrid extends BaseGrid {
  /**
   * The hexagonal grid constructor.
   * @param {HexagonalGridConfiguration} config   The grid configuration
   */
  constructor(config) {
    super(config);
    const {columns, even} = config;

    // Set the type and size of the grid
    let type;
    if ( columns ) {
      if ( even ) type = GRID_TYPES.HEXEVENQ;
      else type = GRID_TYPES.HEXODDQ;
      this.sizeX *= (2 * Math.SQRT1_3);
    } else {
      if ( even ) type = GRID_TYPES.HEXEVENR;
      else type = GRID_TYPES.HEXODDR;
      this.sizeY *= (2 * Math.SQRT1_3);
    }

    /**
     * @override
     * @readonly
     */
    this.type = type;

    /**
     * Is this grid column-based (flat-topped) or row-based (pointy-topped)?
     * @type {boolean}
     * @readonly
     */
    this.columns = !!columns;

    /**
     * Is this grid even or odd?
     * @type {boolean}
     * @readonly
     */
    this.even = !!even;

    /**
     * The rule for diagonal measurement (see {@link CONST.GRID_DIAGONALS}).
     * @type {GridDiagonalRule}
     * @readonly
     */
    this.diagonals = config.diagonals ?? GRID_DIAGONALS.EQUIDISTANT;
  }

  /* -------------------------------------------- */

  /** @override */
  getOffset(coords) {
    if ( coords.i !== undefined ) {
      const {i, j, k} = coords;
      return k !== undefined ? {i, j, k}: {i, j};
    }
    const cube = coords.q !== undefined ? coords : this.pointToCube(coords);
    return this.cubeToOffset(HexagonalGrid.cubeRound(cube));
  }

  /* -------------------------------------------- */

  /** @override */
  getOffsetRange({x, y, width, height}) {
    const x0 = x;
    const y0 = y;
    const {i: i00, j: j00} = this.getOffset({x: x0, y: y0});
    if ( !((width > 0) && (height > 0)) ) return [i00, j00, i00, j00];
    const x1 = x + width;
    const y1 = y + height;
    const {i: i01, j: j01} = this.getOffset({x: x1, y: y0});
    const {i: i10, j: j10} = this.getOffset({x: x0, y: y1});
    const {i: i11, j: j11} = this.getOffset({x: x1, y: y1});
    let i0 = Math.min(i00, i01, i10, i11);
    let j0 = Math.min(j00, j01, j10, j11);
    let i1 = Math.max(i00, i01, i10, i11) + 1;
    let j1 = Math.max(j00, j01, j10, j11) + 1;
    // While the corners of the rectangle are included in this range, the edges of the rectangle might
    // intersect rows or columns outside of the range. So we need to expand the range if necessary.
    if ( this.columns ) {
      if ( (i00 === i01) && (j00 < j01) && (!(j00 % 2) !== this.even) && (y0 < i00 * this.sizeY) ) i0--;
      if ( (i10 === i11) && (j10 < j11) && (!(j00 % 2) === this.even) && (y1 > (i10 + 0.5) * this.sizeY) ) i1++;
      if ( (j00 === j10) && (i00 < i10) && (x0 < ((j00 * 0.75) + 0.25) * this.sizeX) ) j0--;
      if ( (j01 === j11) && (i01 < i11) && (x1 > ((j01 * 0.75) + 0.75) * this.sizeX) ) j1++;
    } else {
      if ( (j00 === j10) && (i00 < i10) && (!(i00 % 2) !== this.even) && (x0 < j00 * this.sizeX) ) j0--;
      if ( (j01 === j11) && (i01 < i11) && (!(i00 % 2) === this.even) && (x1 > (j01 + 0.5) * this.sizeX) ) j1++;
      if ( (i00 === i01) && (j00 < j01) && (y0 < ((i00 * 0.75) + 0.25) * this.sizeY) ) i0--;
      if ( (i10 === i11) && (j10 < j11) && (y1 > ((i10 * 0.75) + 0.75) * this.sizeY) ) i1++;
    }
    return [i0, j0, i1, j1];
  }

  /* -------------------------------------------- */

  /** @override */
  getAdjacentOffsets(coords) {
    return this.getAdjacentCubes(coords).map(cube => this.getOffset(cube));
  }

  /* -------------------------------------------- */

  /** @override */
  testAdjacency(coords1, coords2) {
    const c1 = this.getCube(coords1);
    const c2 = this.getCube(coords2);
    const d0 = HexagonalGrid.cubeDistance(c1, c2);
    if ( c1.k === undefined ) return d0 === 1;
    if ( d0 > 1 ) return false;
    const d1 = Math.abs(c1.k - c2.k);
    if ( d1 > 1 ) return false;
    if ( this.diagonals === GRID_DIAGONALS.ILLEGAL ) return d0 + d1 === 1;
    return d0 + d1 !== 0;
  }

  /* -------------------------------------------- */

  /** @override */
  getShiftedOffset(coords, direction) {
    const offset = this.getOffset(coords);
    if ( this.columns ) {
      if ( !(direction & MOVEMENT_DIRECTIONS.LEFT) !== !(direction & MOVEMENT_DIRECTIONS.RIGHT) ) {
        const even = (offset.j % 2 === 0) === this.even;
        if ( (even && (direction & MOVEMENT_DIRECTIONS.UP)) || (!even && (direction & MOVEMENT_DIRECTIONS.DOWN)) ) {
          direction &= ~(MOVEMENT_DIRECTIONS.UP | MOVEMENT_DIRECTIONS.DOWN);
        }
      }
    } else {
      if ( !(direction & MOVEMENT_DIRECTIONS.UP) !== !(direction & MOVEMENT_DIRECTIONS.DOWN) ) {
        const even = (offset.i % 2 === 0) === this.even;
        if ( (even && (direction & MOVEMENT_DIRECTIONS.LEFT)) || (!even && (direction & MOVEMENT_DIRECTIONS.RIGHT)) ) {
          direction &= ~(MOVEMENT_DIRECTIONS.LEFT | MOVEMENT_DIRECTIONS.RIGHT);
        }
      }
    }
    let di = 0;
    let dj = 0;
    let dk = 0;
    if ( direction & MOVEMENT_DIRECTIONS.UP ) di--;
    if ( direction & MOVEMENT_DIRECTIONS.DOWN ) di++;
    if ( direction & MOVEMENT_DIRECTIONS.LEFT ) dj--;
    if ( direction & MOVEMENT_DIRECTIONS.RIGHT ) dj++;
    if ( direction & MOVEMENT_DIRECTIONS.DESCEND ) dk--;
    if ( direction & MOVEMENT_DIRECTIONS.ASCEND ) dk++;
    if ( (((Math.abs(di) | Math.abs(dj)) + Math.abs(dk)) > 1) && (this.diagonals === GRID_DIAGONALS.ILLEGAL) ) {
      // Diagonal movement is not allowed
      di = 0;
      dj = 0;
      dk = 0;
    }
    offset.i += di;
    offset.j += dj;
    if ( offset.k !== undefined ) offset.k += dk;
    return offset;
  }

  /* -------------------------------------------- */

  /** @override */
  getShiftedPoint(point, direction) {
    const center = this.getCenterPoint(point);
    const shifted = this.getCenterPoint(this.getShiftedOffset(center, direction));
    shifted.x = point.x + (shifted.x - center.x);
    shifted.y = point.y + (shifted.y - center.y);
    if ( point.elevation !== undefined ) {
      shifted.elevation = point.elevation + (shifted.elevation - center.elevation);
    }
    return shifted;
  }

  /* -------------------------------------------- */

  /**
   * Returns the cube coordinates of the grid space corresponding to the given coordinates.
   * @overload
   * @param {HexagonalGridCoordinates2D} coords    The coordinates
   * @returns {HexagonalGridCube2D}                The cube coordinates
   */
  /**
   * @overload
   * @param {HexagonalGridCoordinates3D} coords    The coordinates
   * @returns {HexagonalGridCube3D}                The cube coordinates
   */
  getCube(coords) {
    if ( coords.i !== undefined ) return this.offsetToCube(coords);
    const cube = coords.q !== undefined ? coords : this.pointToCube(coords);
    return HexagonalGrid.cubeRound(cube);
  }

  /* -------------------------------------------- */

  /**
   * Returns the cube coordinates of grid spaces adjacent to the one corresponding to the given coordinates.
   * @overload
   * @param {HexagonalGridCoordinates2D} coords   The coordinates
   * @returns {HexagonalGridCube2D[]}             The adjacent cube coordinates
   */
  /**
   * @overload
   * @param {HexagonalGridCoordinates3D} coords   The coordinates
   * @returns {HexagonalGridCube3D[]}             The adjacent cube coordinates
   */
  getAdjacentCubes(coords) {
    const {q, r, s, k} = this.getCube(coords);
    const cubes = [
      {q: q - 1, r, s: s + 1},
      {q: q - 1, r: r + 1, s},
      {q, r: r - 1, s: s + 1},
      {q, r: r + 1, s: s - 1},
      {q: q + 1, r: r - 1, s},
      {q: q + 1, r, s: s - 1}
    ];

    // 2D case
    if ( k === undefined ) return cubes;

    // 3D case
    for ( const cube of cubes ) cube.k = k;

    // Add diagonals unless illegal
    if ( this.diagonals !== GRID_DIAGONALS.ILLEGAL ) {
      for ( let i = 0; i < 6; i++ ) {
        const {q, r, s, k} = cubes[i];
        cubes.push({q, r, s, k: k - 1}, {q, r, s, k: k + 1});
      }
    }

    // Add cubes directly above and below
    cubes.push({q, r, s, k: k - 1}, {q, r, s, k: k + 1});
    return cubes;
  }

  /* -------------------------------------------- */

  /**
   * Returns the cube coordinates of the grid space corresponding to the given coordinates
   * shifted by one grid space in the given direction.
   * @overload
   * @param {HexagonalGridCoordinates2D} coords    The coordinates
   * @param {number} direction                     The direction (see {@link CONST.MOVEMENT_DIRECTIONS})
   * @returns {HexagonalGridCube2D}                The cube coordinates
   */
  /**
   * @overload
   * @param {HexagonalGridCoordinates3D} coords    The coordinates
   * @param {number} direction                     The direction (see {@link CONST.MOVEMENT_DIRECTIONS})
   * @returns {HexagonalGridCube3D}                The cube coordinates
   */
  getShiftedCube(coords, direction) {
    return this.getCube(this.getShiftedOffset(coords, direction));
  }

  /* -------------------------------------------- */

  /** @override */
  getTopLeftPoint(coords) {
    if ( coords.i !== undefined ) {
      const {i, j, k} = coords;
      let x;
      let y;
      const size = this.size;
      if ( this.columns ) {
        x = (2 * Math.SQRT1_3) * ((0.75 * j) * size);
        const even = (j + 1) % 2 === 0;
        y = (i - (this.even === even ? 0.5 : 0)) * size;
      } else {
        y = (2 * Math.SQRT1_3) * ((0.75 * i) * size);
        const even = (i + 1) % 2 === 0;
        x = (j - (this.even === even ? 0.5 : 0)) * size;
      }
      return k !== undefined ? {x, y, elevation: k * this.distance} : {x, y};
    }
    const {q, r, k} = HexagonalGrid.cubeRound(coords.q !== undefined ? coords : this.pointToCube(coords));
    let x;
    let y;
    const size = this.size;
    if ( this.columns ) {
      x = (Math.SQRT3 / 2) * (q * size);
      y = ((0.5 * (q - (this.even ? 0 : 1))) + r) * size;
    } else {
      y = (Math.SQRT3 / 2) * (r * size);
      x = ((0.5 * (r - (this.even ? 0 : 1))) + q) * size;
    }
    return k !== undefined ? {x, y, elevation: k * this.distance} : {x, y};
  }

  /* -------------------------------------------- */

  /** @override */
  getCenterPoint(coords) {
    if ( coords.i !== undefined ) {
      const {i, j, k} = coords;
      let x;
      let y;
      const size = this.size;
      if ( this.columns ) {
        x = (2 * Math.SQRT1_3) * (((0.75 * j) + 0.5) * size);
        const even = (j + 1) % 2 === 0;
        y = (i + (this.even === even ? 0 : 0.5)) * size;
      } else {
        y = (2 * Math.SQRT1_3) * (((0.75 * i) + 0.5) * size);
        const even = (i + 1) % 2 === 0;
        x = (j + (this.even === even ? 0 : 0.5)) * size;
      }
      return k !== undefined ? {x, y, elevation: (k + 0.5) * this.distance} : {x, y};
    }
    const cube = HexagonalGrid.cubeRound(coords.q !== undefined ? coords : this.pointToCube(coords));
    if ( cube.k !== undefined ) cube.k += 0.5;
    return this.cubeToPoint(cube);
  }

  /* -------------------------------------------- */

  /** @override */
  getShape() {
    const scaleX = this.sizeX / 4;
    const scaleY = this.sizeY / 4;
    if ( this.columns ) {
      const x0 = -2 * scaleX;
      const x1 = -scaleX;
      const x2 = scaleX;
      const x3 = 2 * scaleX;
      const y0 = -2 * scaleY;
      const y1 = 2 * scaleY;
      return [{x: x0, y: 0}, {x: x1, y: y0}, {x: x2, y: y0}, {x: x3, y: 0}, {x: x2, y: y1}, {x: x1, y: y1}];
    } else {
      const y0 = -2 * scaleY;
      const y1 = -scaleY;
      const y2 = scaleY;
      const y3 = 2 * scaleY;
      const x0 = -2 * scaleX;
      const x1 = 2 * scaleX;
      return [{x: 0, y: y0}, {x: x1, y: y1}, {x: x1, y: y2}, {x: 0, y: y3}, {x: x0, y: y2}, {x: x0, y: y1}];
    }
  }

  /* -------------------------------------------- */

  /** @override */
  getVertices(coords) {
    const {i, j} = this.getOffset(coords);
    const scaleX = this.sizeX / 4;
    const scaleY = this.sizeY / 4;
    if ( this.columns ) {
      const x = 3 * j;
      const x0 = x * scaleX;
      const x1 = (x + 1) * scaleX;
      const x2 = (x + 3) * scaleX;
      const x3 = (x + 4) * scaleX;
      const even = (j + 1) % 2 === 0;
      const y = (4 * i) - (this.even === even ? 2 : 0);
      const y0 = y * scaleY;
      const y1 = (y + 2) * scaleY;
      const y2 = (y + 4) * scaleY;
      return [{x: x0, y: y1}, {x: x1, y: y0}, {x: x2, y: y0}, {x: x3, y: y1}, {x: x2, y: y2}, {x: x1, y: y2}];
    } else {
      const y = 3 * i;
      const y0 = y * scaleY;
      const y1 = (y + 1) * scaleY;
      const y2 = (y + 3) * scaleY;
      const y3 = (y + 4) * scaleY;
      const even = (i + 1) % 2 === 0;
      const x = (4 * j) - (this.even === even ? 2 : 0);
      const x0 = x * scaleX;
      const x1 = (x + 2) * scaleX;
      const x2 = (x + 4) * scaleX;
      return [{x: x1, y: y0}, {x: x2, y: y1}, {x: x2, y: y2}, {x: x1, y: y3}, {x: x0, y: y2}, {x: x0, y: y1}];
    }
  }

  /* -------------------------------------------- */

  /** @override */
  getSnappedPoint(point, {mode, resolution=1}) {
    if ( mode & -65524 ) throw new Error("Invalid snapping mode");
    if ( mode === 0 ) {
      return point.elevation !== undefined ? {x: point.x, y: point.y, elevation: point.elevation}
        : {x: point.x, y: point.y};
    }

    let nearest;
    let distance;
    const keepNearest = candidate => {
      if ( !nearest ) return nearest = candidate;
      const {x, y} = point;
      distance ??= ((nearest.x - x) ** 2) + ((nearest.y - y) ** 2);
      const d = ((candidate.x - x) ** 2) + ((candidate.y - y) ** 2);
      if ( d < distance ) {
        nearest = candidate;
        distance = d;
      }
      return nearest;
    };

    // Symmetries and identities
    if ( this.columns ) {
      // Top-Left = Bottom-Left
      if ( mode & 0x50 ) mode |= 0x50; // Vertex
      if ( mode & 0x500 ) mode |= 0x500; // Corner
      // Top-Right = Bottom-Right
      if ( mode & 0xA0 ) mode |= 0xA0; // Vertex
      if ( mode & 0xA00 ) mode |= 0xA00; // Corner
      // Left Side = Right Vertex
      if ( mode & 0x4000 ) mode |= 0xA0;
      // Right Side = Left Vertex
      if ( mode & 0x8000 ) mode |= 0x50;
    } else {
      // Top-Left = Top-Right
      if ( mode & 0x30 ) mode |= 0x30; // Vertex
      if ( mode & 0x300 ) mode |= 0x300; // Corner
      // Bottom-Left = Bottom-Right
      if ( mode & 0xC0 ) mode |= 0xC0; // Vertex
      if ( mode & 0xC00 ) mode |= 0xC00; // Corner
      // Top Side = Bottom Vertex
      if ( mode & 0x1000 ) mode |= 0xC0;
      // Bottom Side = Top Vertex
      if ( mode & 0x2000 ) mode |= 0x30;
    }

    // Only top/bottom or left/right edges
    if ( !(mode & 0x2) ) {
      if ( this.columns ) {
        // Top/Left side (= edge)
        if ( mode & 0x3000 ) keepNearest(this.#snapToTopOrBottom(point, resolution));
      } else {
        // Left/Right side (= edge)
        if ( mode & 0xC000 ) keepNearest(this.#snapToLeftOrRight(point, resolution));
      }
    }

    // Any vertex (plus edge/center)
    if ( (mode & 0xF0) === 0xF0 ) {
      switch ( mode & 0x3 ) {
        case 0x0: keepNearest(this.#snapToVertex(point, resolution)); break;
        case 0x1: keepNearest(this.#snapToVertexOrCenter(point, resolution)); break;
        case 0x2: keepNearest(this.#snapToEdgeOrVertex(point, resolution)); break;
        case 0x3: keepNearest(this.#snapToEdgeOrVertexOrCenter(point, resolution)); break;
      }
    }
    // A specific vertex
    else if ( mode & 0xF0 ) {
      // Center
      if ( (mode & 0x3) === 0x1 ) {
        keepNearest(this.#snapToSpecificVertexOrCenter(point, !(mode & 0x10), resolution));
      } else {
        // Edge and/or center
        switch ( mode & 0x3 ) {
          case 0x2: keepNearest(this.#snapToEdge(point, resolution)); break;
          case 0x3: keepNearest(this.#snapToEdgeOrCenter(point, resolution)); break;
        }

        // A combination of specific vertices and corners that results in a rectangular grid
        if ( ((mode & 0xF0) ^ ((mode & 0xF00) >> 4)) === 0xF0 ) {
          return keepNearest(this.#snapToRectangularGrid(point, !(mode & 0x100), resolution));
        }

        keepNearest(this.#snapToSpecificVertex(point, !(mode & 0x10), resolution));
      }
    }
    // Edges and/or centers
    else {
      switch ( mode & 0x3 ) {
        case 0x1: keepNearest(this.#snapToCenter(point, resolution)); break;
        case 0x2: keepNearest(this.#snapToEdge(point, resolution)); break;
        case 0x3: keepNearest(this.#snapToEdgeOrCenter(point, resolution)); break;
      }
    }

    // Any corner
    if ( (mode & 0xF00) === 0xF00 ) {
      keepNearest(this.#snapToCorner(point, resolution));
    }
    // A specific corner
    else if ( mode & 0xF00 ) {
      keepNearest(this.#snapToSpecificCorner(point, !(mode & 0x100), resolution));
    }

    return point.elevation === undefined ? nearest
      : {x: nearest.x, y: nearest.y, elevation: Math.round((point.elevation / this.distance) + 1e-8) * this.distance};
  }

  /* -------------------------------------------- */

  /**
   * Snap the point to the nearest center of a hexagon.
   * @param {Point} point          The point
   * @param {number} resolution    The grid resolution
   * @param {number} [dx=0]        The x-translation of the grid
   * @param {number} [dy=0]        The y-translation of the grid
   * @param {boolean} [columns]    Flat-top instead of pointy-top?
   * @param {boolean} [even]       Start at a full grid space?
   * @param {number} [size]        The size of a grid space
   * @returns {Point}              The snapped point
   */
  #snapToCenter({x, y}, resolution, dx=0, dy=0, columns=this.columns, even=this.even, size=this.size) {

    // Subdivide the hex grid
    const grid = HexagonalGrid.#TEMP_GRID;
    grid.columns = columns;
    grid.size = size / resolution;
    if ( columns ) {
      grid.type = GRID_TYPES.HEXODDQ;
      grid.sizeX = grid.size * (2 * Math.SQRT1_3);
      grid.sizeY = grid.size;
    } else {
      grid.type = GRID_TYPES.HEXODDR;
      grid.sizeX = grid.size;
      grid.sizeY = grid.size * (2 * Math.SQRT1_3);
    }

    // Align the subdivided grid with this hex grid
    if ( columns ) {
      dx += ((size - grid.size) * Math.SQRT1_3);
      if ( even ) dy += (size / 2);
    } else {
      if ( even ) dx += (size / 2);
      dy += ((size - grid.size) * Math.SQRT1_3);
    }

    // Get the snapped center point for the subdivision
    const point = HexagonalGrid.#TEMP_POINT;
    point.x = x - dx;
    point.y = y - dy;
    const snapped = grid.getCenterPoint(point);
    snapped.x += dx;
    snapped.y += dy;
    return snapped;
  }

  /* -------------------------------------------- */

  /**
   * Snap the point to the nearest vertex of a hexagon.
   * @param {Point} point          The point
   * @param {number} resolution    The grid resolution
   * @param {number} [dx=0]        The x-offset of the grid
   * @param {number} [dy=0]        The y-offset of the grid
   * @returns {Point}              The snapped point
   */
  #snapToVertex(point, resolution, dx, dy) {
    const center = this.#snapToCenter(point, resolution, dx, dy);
    const {x: x0, y: y0} = center;
    let angle = Math.atan2(point.y - y0, point.x - x0);
    if ( this.columns ) angle = Math.round(angle / (Math.PI / 3)) * (Math.PI / 3);
    else angle = (Math.floor(angle / (Math.PI / 3)) + 0.5) * (Math.PI / 3);
    const radius = Math.max(this.sizeX, this.sizeY) / (2 * resolution);
    const vertex = center; // Reuse the object
    vertex.x = x0 + (Math.cos(angle) * radius);
    vertex.y = y0 + (Math.sin(angle) * radius);
    return vertex;
  }

  /* -------------------------------------------- */

  /**
   * Snap the point to the nearest vertex or center of a hexagon.
   * @param {Point} point          The point
   * @param {number} resolution    The grid resolution
   * @returns {Point}              The snapped point
   */
  #snapToVertexOrCenter(point, resolution) {
    let size;
    let dx = 0;
    let dy = 0;
    if ( this.columns ) {
      size = this.sizeX / 2;
      dy = size * (Math.SQRT1_3 / 2);
    } else {
      size = this.sizeY / 2;
      dx = size * (Math.SQRT1_3 / 2);
    }
    return this.#snapToCenter(point, resolution, dx, dy, !this.columns, !this.even, size);
  }

  /* -------------------------------------------- */

  /**
   * Snap the point to the nearest edge of a hexagon.
   * @param {Point} point          The point
   * @param {number} resolution    The grid resolution
   * @returns {Point}              The snapped point
   */
  #snapToEdge(point, resolution) {
    const center = this.#snapToCenter(point, resolution);
    const {x: x0, y: y0} = center;
    let angle = Math.atan2(point.y - y0, point.x - x0);
    if ( this.columns ) angle = (Math.floor(angle / (Math.PI / 3)) + 0.5) * (Math.PI / 3);
    else angle = Math.round(angle / (Math.PI / 3)) * (Math.PI / 3);
    const radius = Math.min(this.sizeX, this.sizeY) / (2 * resolution);
    const vertex = center; // Reuse the object
    vertex.x = x0 + (Math.cos(angle) * radius);
    vertex.y = y0 + (Math.sin(angle) * radius);
    return vertex;
  }

  /* -------------------------------------------- */

  /**
   * Snap the point to the nearest edge or center of a hexagon.
   * @param {Point} point          The point
   * @param {number} resolution    The grid resolution
   * @returns {Point}              The snapped point
   */
  #snapToEdgeOrCenter(point, resolution) {
    let size;
    let dx = 0;
    let dy = 0;
    if ( this.columns ) {
      size = this.sizeY / 2;
      dx = size * Math.SQRT1_3;
    } else {
      size = this.sizeX / 2;
      dy = size * Math.SQRT1_3;
    }
    return this.#snapToCenter(point, resolution, dx, dy, this.columns, false, size);
  }

  /* -------------------------------------------- */

  /**
   * Snap the point to the nearest edge or vertex of a hexagon.
   * @param {Point} point          The point
   * @param {number} resolution    The grid resolution
   * @returns {Point}              The snapped point
   */
  #snapToEdgeOrVertex(point, resolution) {
    const {x, y} = point;
    point = this.#snapToCenter(point, resolution);
    const {x: x0, y: y0} = point;
    const dx = x - x0;
    const dy = y - y0;
    let angle = Math.atan2(dy, dx);
    if ( this.columns ) angle = (Math.floor(angle / (Math.PI / 3)) + 0.5) * (Math.PI / 3);
    else angle = Math.round(angle / (Math.PI / 3)) * (Math.PI / 3);
    const s = 2 * resolution;
    let radius1 = this.sizeX / s;
    let radius2 = this.sizeY / s;
    if ( radius1 > radius2 ) [radius1, radius2] = [radius2, radius1];
    const cos = Math.cos(angle);
    const sin = Math.sin(angle);
    const d = (cos * dy) - (sin * dx);
    if ( Math.abs(d) <= radius2 / 4 ) {
      point.x = x0 + (cos * radius1);
      point.y = y0 + (sin * radius1);
    } else {
      angle += ((Math.PI / 6) * Math.sign(d));
      point.x = x0 + (Math.cos(angle) * radius2);
      point.y = y0 + (Math.sin(angle) * radius2);
    }
    return point;
  }

  /* -------------------------------------------- */

  /**
   * Snap the point to the nearest edge, vertex, center of a hexagon.
   * @param {Point} point          The point
   * @param {number} resolution    The grid resolution
   * @returns {Point}              The snapped point
   */
  #snapToEdgeOrVertexOrCenter(point, resolution) {
    const {x, y} = point;
    point = this.#snapToCenter(point, resolution);
    const {x: x0, y: y0} = point;
    const dx = x - x0;
    const dy = y - y0;
    let angle = Math.atan2(dy, dx);
    if ( this.columns ) angle = (Math.floor(angle / (Math.PI / 3)) + 0.5) * (Math.PI / 3);
    else angle = Math.round(angle / (Math.PI / 3)) * (Math.PI / 3);
    const s = 2 * resolution;
    let radius1 = this.sizeX / s;
    let radius2 = this.sizeY / s;
    if ( radius1 > radius2 ) [radius1, radius2] = [radius2, radius1];
    const cos = Math.cos(angle);
    const sin = Math.sin(angle);
    const d1 = (cos * dx) + (sin * dy);
    if ( d1 <= radius1 / 2 ) return point;
    const d2 = (cos * dy) - (sin * dx);
    if ( Math.abs(d2) <= radius2 / 4 ) {
      point.x = x0 + (cos * radius1);
      point.y = y0 + (sin * radius1);
    } else {
      angle += ((Math.PI / 6) * Math.sign(d2));
      point.x = x0 + (Math.cos(angle) * radius2);
      point.y = y0 + (Math.sin(angle) * radius2);
    }
    return point;
  }

  /* -------------------------------------------- */

  /**
   * Snap the point to the nearest corner of a hexagon.
   * @param {Point} point          The point
   * @param {number} resolution    The grid resolution
   * @returns {Point}              The snapped point
   */
  #snapToCorner(point, resolution) {
    let dx = 0;
    let dy = 0;
    const s = 2 * resolution;
    if ( this.columns ) dy = this.sizeY / s;
    else dx = this.sizeX / s;
    return this.#snapToVertex(point, resolution, dx, dy);
  }

  /* -------------------------------------------- */

  /**
   * Snap the point to the nearest top/bottom-left/right vertex of a hexagon.
   * @param {Point} point          The point
   * @param {boolean} other        Bottom-right instead of top-left vertex?
   * @param {number} resolution    The grid resolution
   * @returns {Point}              The snapped point
   */
  #snapToSpecificVertex(point, other, resolution) {
    let dx = 0;
    let dy = 0;
    const s = (other ? -2 : 2) * resolution;
    if ( this.columns ) dx = this.sizeX / s;
    else dy = this.sizeY / s;
    return this.#snapToCenter(point, resolution, dx, dy);
  }

  /* -------------------------------------------- */

  /**
   * Snap the point to the nearest top/bottom-left/right vertex or center of a hexagon.
   * @param {Point} point          The point
   * @param {boolean} other        Bottom-right instead of top-left vertex?
   * @param {number} resolution    The grid resolution
   * @returns {Point}              The snapped point
   */
  #snapToSpecificVertexOrCenter(point, other, resolution) {
    let dx = 0;
    let dy = 0;
    const s = (other ? 2 : -2) * resolution;
    if ( this.columns ) dx = this.sizeX / s;
    else dy = this.sizeY / s;
    return this.#snapToVertex(point, resolution, dx, dy);
  }

  /* -------------------------------------------- */

  /**
   * Snap the point to the nearest top/bottom-left/right corner of a hexagon.
   * @param {Point} point          The point
   * @param {boolean} other        Bottom-right instead of top-left corner?
   * @param {number} resolution    The grid resolution
   * @returns {Point}              The snapped point
   */
  #snapToSpecificCorner(point, other, resolution) {
    let dx = 0;
    let dy = 0;
    const s = (other ? -4 : 4) * resolution;
    if ( this.columns ) dx = this.sizeX / s;
    else dy = this.sizeY / s;
    return this.#snapToCenter(point, resolution, dx, dy);
  }

  /* -------------------------------------------- */

  /**
   * Snap the point to the nearest grid intersection of the rectanglar grid.
   * @param {Point} point          The point
   * @param {boolean} other        Align rectangles with top-left vertices instead of top-left corners?
   * @param {number} resolution    The grid resolution
   * @returns {Point}              The snapped point
   */
  #snapToRectangularGrid(point, other, resolution) {
    const tx = this.sizeX / 2;
    const ty = this.sizeY / 2;
    let sx = tx;
    let sy = ty;
    let dx = 0;
    let dy = 0;
    const d = other ? 1 / 3 : 2 / 3;
    if ( this.columns ) {
      sx *= 1.5;
      dx = d;
    } else {
      sy *= 1.5;
      dy = d;
    }
    sx /= resolution;
    sy /= resolution;
    return {
      x: ((Math.round(((point.x - tx) / sx) + dx) - dx) * sx) + tx,
      y: ((Math.round(((point.y - ty) / sy) + dy) - dy) * sy) + ty
    };
  }

  /**
   * Snap the point to the nearest top/bottom side of the bounds of a hexagon.
   * @param {Point} point          The point
   * @param {number} resolution    The grid resolution
   * @returns {Point}              The snapped point
   */
  #snapToTopOrBottom(point, resolution) {
    return this.#snapToCenter(point, resolution, 0, this.sizeY / (2 * resolution));
  }

  /* -------------------------------------------- */

  /**
   * Snap the point to the nearest left/right side of the bounds of a hexagon.
   * @param {Point} point          The point
   * @param {number} resolution    The grid resolution
   * @returns {Point}              The snapped point
   */
  #snapToLeftOrRight(point, resolution) {
    return this.#snapToCenter(point, resolution, this.sizeX / (2 * resolution), 0);
  }

  /* -------------------------------------------- */

  /** @inheritdoc */
  calculateDimensions(sceneWidth, sceneHeight, padding) {
    const {columns, size} = this;
    const sizeX = columns ? (2 * size) / Math.SQRT3 : size;
    const sizeY = columns ? size : (2 * size) / Math.SQRT3;
    const strideX = columns ? 0.75 * sizeX : sizeX;
    const strideY = columns ? sizeY : 0.75 * sizeY;

    // Skip padding computation for Scenes which do not include padding
    if ( !padding ) {
      const cols = Math.ceil(((sceneWidth + (columns ? -sizeX / 4 : sizeX / 2)) / strideX) - 1e-6);
      const rows = Math.ceil(((sceneHeight + (columns ? sizeY / 2 : -sizeY / 4)) / strideY) - 1e-6);
      return {width: sceneWidth, height: sceneHeight, x: 0, y: 0, rows, columns: cols};
    }

    // The grid size is equal to the short diagonal of the hexagon, so padding in that axis will divide evenly by the
    // grid size. In the cross-axis, however, the hexagons do not stack but instead interleave. Multiplying the long
    // diagonal by 75% gives us the amount of space each hexagon takes up in that axis without overlapping.
    // Note: Do not replace `* (1 / strideX)` by `/ strideX` and `* (1 / strideY)` by `/ strideY`!
    // It could change the result and therefore break certain scenes.
    let x = Math.ceil((padding * sceneWidth) * (1 / strideX)) * strideX;
    let y = Math.ceil((padding * sceneHeight) * (1 / strideY)) * strideY;
    // Note: The width and height calculation needs rounded x/y. If we were to remove the rounding here,
    // the result of the rounding of the width and height below would change in certain scenes.
    let width = sceneWidth + (2 * Math.round(Math.ceil((padding * sceneWidth) * (1 / strideX)) / (1 / strideX)));
    let height = sceneHeight + (2 * Math.round(Math.ceil((padding * sceneHeight) * (1 / strideY)) / (1 / strideY)));

    // Ensure that the top-left hexagon of the scene rectangle is always a full hexagon for even grids and always a
    // half hexagon for odd grids, by shifting the padding in the main axis by half a hex if the number of hexagons in
    // the cross-axis is odd.
    const crossEven = Math.round(columns ? x / strideX : y / strideY) % 2 === 0;
    if ( !crossEven ) {
      if ( columns ) {
        y += (sizeY / 2);
        height += sizeY;
      } else {
        x += (sizeX / 2);
        width += sizeX;
      }
    }

    // The height (if column orientation) or width (if row orientation) must be a multiple of the grid size, and
    // the last column (if column orientation) or row (if row orientation) must be fully within the bounds.
    // Note: Do not replace `* (1 / strideX)` by `/ strideX` and `* (1 / strideY)` by `/ strideY`!
    // It could change the result and therefore break certain scenes.
    let cols = Math.round(width * (1 / strideX));
    let rows = Math.round(height * (1 / strideY));
    width = cols * strideX;
    height = rows * strideY;
    if ( columns ) {
      rows++;
      width += (sizeX / 4);
    } else {
      cols++;
      height += (sizeY / 4);
    }
    return {width, height, x, y, rows, columns: cols};
  }

  /* -------------------------------------------- */

  /**
   * Calculate the total size of the canvas with padding applied, as well as the top-left coordinates of the inner
   * rectangle that houses the scene. (Legacy)
   * @param {number} columns            Column or row orientation?
   * @param {number} legacySize         The legacy size of the grid.
   * @param {number} sceneWidth         The width of the scene.
   * @param {number} sceneHeight        The height of the scene.
   * @param {number} padding            The percentage of padding.
   * @returns {{width: number, height: number, x: number, y: number, rows: number, columns: number}}
   * @internal
   * @ignore
   */
  static _calculatePreV10Dimensions(columns, legacySize, sceneWidth, sceneHeight, padding) {
    // Note: Do not replace `* (1 / legacySize)` by `/ legacySize`!
    // It could change the result and therefore break certain scenes.
    const x = Math.ceil((padding * sceneWidth) * (1 / legacySize)) * legacySize;
    const y = Math.ceil((padding * sceneHeight) * (1 / legacySize)) * legacySize;
    const width = sceneWidth + (2 * x);
    const height = sceneHeight + (2 * y);
    const size = legacySize * (Math.SQRT3 / 2);
    const sizeX = columns ? legacySize : size;
    const sizeY = columns ? size : legacySize;
    const strideX = columns ? 0.75 * sizeX : sizeX;
    const strideY = columns ? sizeY : 0.75 * sizeY;
    const cols = Math.floor(((width + (columns ? sizeX / 4 : sizeX)) / strideX) + 1e-6);
    const rows = Math.floor(((height + (columns ? sizeY : sizeY / 4)) / strideY) + 1e-6);
    return {width, height, x, y, rows, columns: cols};
  }

  /* -------------------------------------------- */

  /** @override */
  _measurePath(waypoints, {cost}, result) {

    // Convert to (fractional) cube coordinates
    const toCube = coords => {
      if ( coords.x !== undefined ) return this.pointToCube(coords);
      if ( coords.i !== undefined ) return this.offsetToCube(coords);
      return coords;
    };

    // Prepare data for the starting point
    const w0 = waypoints[0];
    let o0 = this.getOffset(w0);
    let c0 = this.offsetToCube(o0);
    let d0 = toCube(w0);
    let p0 = w0.x !== undefined ? w0 : this.cubeToPoint(d0);

    // Iterate over additional path points
    const is3D = o0.k !== undefined;
    const diagonals = this.diagonals;
    let nd = diagonals === GRID_DIAGONALS.ALTERNATING_2 ? 1 : 0;
    let ld = nd;
    for ( let i = 1; i < waypoints.length; i++ ) {
      const w1 = waypoints[i];
      const o1 = this.getOffset(w1);
      const c1 = this.offsetToCube(o1);
      const d1 = toCube(w1);
      const p1 = w1.x !== undefined ? w1 : this.cubeToPoint(d1);
      const cost1 = w1.cost ?? cost;

      // Determine the number of moves total, number of diagonal moves, and cost of the moves
      if ( w1.measure !== false ) {
        let n = HexagonalGrid.cubeDistance(c0, c1);
        let d = 0;
        if ( is3D ) {
          d = Math.abs(c0.k - c1.k);
          if ( n < d ) [n, d] = [d, n];
        }
        let c;
        const nd0 = nd;
        switch ( diagonals ) {
          case GRID_DIAGONALS.EQUIDISTANT: c = n; break;
          case GRID_DIAGONALS.EXACT: c = n + ((Math.SQRT2 - 1) * d); break;
          case GRID_DIAGONALS.APPROXIMATE: c = n + (0.5 * d); break;
          case GRID_DIAGONALS.RECTILINEAR: c = n + d; break;
          case GRID_DIAGONALS.ALTERNATING_1:
          case GRID_DIAGONALS.ALTERNATING_2:
            nd += d;
            c = n + (Math.floor(nd / 2) - Math.floor(nd0 / 2));
            break;
          case GRID_DIAGONALS.ILLEGAL:
            n = n + d;
            d = 0;
            c = n;
            break;
        }

        // Determine the distance of the segment
        let a = HexagonalGrid.cubeDistance(d0, d1);
        let b = 0;
        if ( is3D ) {
          b = Math.abs(d0.k - d1.k);
          if ( a < b ) [a, b] = [b, a];
        }
        let l;
        switch ( diagonals ) {
          case GRID_DIAGONALS.EQUIDISTANT: l = a; break;
          case GRID_DIAGONALS.EXACT: l = a + ((Math.SQRT2 - 1) * b); break;
          case GRID_DIAGONALS.APPROXIMATE: l = a + (0.5 * b); break;
          case GRID_DIAGONALS.ILLEGAL: l = a + b; break;
          case GRID_DIAGONALS.ALTERNATING_1:
          case GRID_DIAGONALS.ALTERNATING_2: {
            const ld0 = ld;
            ld += b;
            l = a + ((Math.abs(((ld - 1) / 2) - Math.floor(ld / 2)) + ((ld - 1) / 2))
              - (Math.abs(((ld0 - 1) / 2) - Math.floor(ld0 / 2)) + ((ld0 - 1) / 2)));
            break;
          }
          case GRID_DIAGONALS.RECTILINEAR: l = a + b; break;
        }
        if ( l.almostEqual(c) ) l = c;

        const segment = result.segments[i - 1];
        segment.distance = l * this.distance;
        if ( (cost1 === undefined) || (c === 0) ) segment.cost = w1.teleport ? 0 : c * this.distance;
        else if ( typeof cost1 === "function" ) segment.cost = w1.teleport ? cost1(o0, o1, c * this.distance, w1)
          : this.#calculateCost(o0, o1, cost1, nd0, w1);
        else segment.cost = Number(cost1);
        segment.spaces = n;
        segment.diagonals = d;
        segment.euclidean = Math.hypot(p0.x - p1.x, p0.y - p1.y, is3D ? (p0.elevation - p1.elevation) / this.distance
          * this.size : 0) / this.size * this.distance;
      }

      o0 = o1;
      c0 = c1;
      d0 = d1;
      p0 = p1;
    }
  }

  /* -------------------------------------------- */

  /**
   * Calculate the cost of the direct path segment.
   * @template SegmentData
   * @overload
   * @param {GridOffset2D} from      The coordinates the segment starts from
   * @param {GridOffset2D} to        The coordinates the segment goes to
   * @param {GridMeasurePathCostFunction2D<SegmentData>} cost    The cost function
   * @param {number} diagonals       The number of diagonal moves that have been performed already
   * @param {SegmentData} segment    The segment data
   * @returns {number}               The cost of the path segment
   */
  /**
   * @overload
   * @param {GridOffset3D} from      The coordinates the segment starts from
   * @param {GridOffset3D} to        The coordinates the segment goes to
   * @param {GridMeasurePathCostFunction3D<SegmentData>} cost    The cost function
   * @param {number} diagonals       The number of diagonal moves that have been performed already
   * @param {SegmentData} segment    The segment data
   * @returns {number}               The cost of the path segment
   */
  #calculateCost(from, to, cost, diagonals, segment) {
    const path = this.getDirectPath([from, to]);
    if ( path.length <= 1 ) return 0;

    // Prepare data for the starting point
    let o0 = path[0];
    let c = 0;

    // Iterate over additional path points
    for ( let i = 1; i < path.length; i++ ) {
      const o1 = path[i];

      // Determine the normalized distance
      let d;
      if ( (o0.k === o1.k) || ((o0.i === o1.i) && (o0.j === o1.j)) ) d = 1;
      else {
        switch ( this.diagonals ) {
          case GRID_DIAGONALS.EQUIDISTANT: d = 1; break;
          case GRID_DIAGONALS.EXACT: d = Math.SQRT2; break;
          case GRID_DIAGONALS.APPROXIMATE: d = 1.5; break;
          case GRID_DIAGONALS.RECTILINEAR: d = 2; break;
          case GRID_DIAGONALS.ALTERNATING_1:
          case GRID_DIAGONALS.ALTERNATING_2:
            d = 1 + (Math.floor((diagonals + 1) / 2) - Math.floor(diagonals / 2));
            break;
        }
        diagonals++;
      }

      // Calculate and accumulate the cost
      c += cost(o0, o1, d * this.distance, segment);

      o0 = o1;
    }

    return c;
  }

  /* -------------------------------------------- */

  /**
   * @see {@link https://www.redblobgames.com/grids/hexagons/#line-drawing}
   * @override
   */
  getDirectPath(waypoints) {
    if ( waypoints.length === 0 ) return [];

    // Prepare data for the starting point
    let c0 = this.getCube(waypoints[0]);
    let {q: q0, r: r0, k: k0} = c0;
    const is3D = k0 !== undefined;
    const path = [this.getOffset(c0)];

    // Iterate over additional path points
    const diagonals = this.diagonals !== GRID_DIAGONALS.ILLEGAL;
    for ( let i = 1; i < waypoints.length; i++ ) {
      const c1 = this.getCube(waypoints[i]);
      const {q: q1, r: r1, s: s0, k: k1} = c1;
      if ( (q0 === q1) && (r0 === r1) && (k0 === k1) ) continue;

      // Walk from (q0, r0, s0, k0) to (q1, r1, s1, k1)
      const dq = q0 - q1;
      const dr = r0 - r1;
      // If the path segment is collinear with some hexagon edge, we need to nudge
      // the cube coordinates in the right direction so that we get a consistent, clean path.
      const EPS = 1e-6;
      let eq = 0;
      let er = 0;
      if ( this.columns ) {
        // Collinear with SE-NW edges
        if ( dq === dr ) {
          // Prefer movement such that we have rotational symmetry with the E-W case at (0, 0, 0)
          er = !((q0 + r0) & 1) === this.even ? EPS : -1e-6;
          eq = -er;
        }
        // Collinear with SW-NE edges
        else if ( -2 * dq === dr ) {
          // Prefer movement such that we have rotational symmetry with the E-W case at (0, 0, 0)
          eq = !(r0 & 1) === this.even ? EPS : -1e-6;
        }
        // Collinear with E-W edges
        else if ( dq === -2 * dr ) {
          // Move such we don't leave the row that we're in
          er = !(q0 & 1) === this.even ? -1e-6 : EPS;
        }
      } else {
        // Collinear with SE-NW edges
        if ( dq === dr ) {
          // Prefer movement such that we have rotational symmetry with the S-N case at (0, 0, 0)
          eq = !((q0 + r0) & 1) === this.even ? EPS : -1e-6;
          er = -eq;
        }
        // Collinear with SW-NE edges
        else if ( dq === -2 * dr ) {
          // Prefer movement such that we have rotational symmetry with the S-N case at (0, 0, 0)
          er = !(q0 & 1) === this.even ? EPS : -1e-6;
        }
        // Collinear with S-N edges
        else if ( -2 * dq === dr ) {
          // Move such we don't leave the column that we're in
          eq = !(r0 & 1) === this.even ? -1e-6 : EPS;
        }
      }
      const n = HexagonalGrid.cubeDistance(c0, c1);
      if ( is3D ) {
        if ( n !== 0 ) {
          let q = q0;
          let r = r0;
          let s = s0;
          let k = k0;
          let j = 0;
          const sk = k0 < k1 ? 1 : -1;
          if ( diagonals ) {
            const dk = 0 - Math.abs(k0 - k1);
            let e = n + dk;
            for ( ;; ) {
              const e2 = e * 2;
              if ( e2 >= dk ) {
                e += dk;
                j++;

                // Break tries on E-W (if columns) / S-N (if rows) edges
                const t = (j + EPS) / n;
                q = Math.mix(q0, q1, t) + eq;
                r = Math.mix(r0, r1, t) + er;
                s = 0 - q - r;
              }
              if ( e2 <= n ) {
                e += n;
                k += sk;
              }
              if ( (j === n) && (k === k1) ) break;
              path.push(this.getOffset({q, r, s, k}));
            }
          } else {
            const dk1 = Math.abs(k0 - k1) || 1;
            let tc = dk1;
            let tk = n;
            for ( ;; ) {
              if ( tc <= tk ) {
                tc += dk1;
                j++;

                // Break tries on E-W (if columns) / S-N (if rows) edges
                const t = (j + EPS) / n;
                q = Math.mix(q0, q1, t) + eq;
                r = Math.mix(r0, r1, t) + er;
                s = 0 - q - r;
              } else {
                tk += n;
                k += sk;
              }
              if ( (j === n) && (k === k1) ) break;
              path.push(this.getOffset({q, r, s, k}));
            }
          }
          path.push(this.getOffset(c1));
        } else {
          const {i, j} = path.at(-1);
          let k = k0;
          const sk = k0 < k1 ? 1 : -1;
          while ( k !== k1 ) {
            k += sk;
            path.push({i, j, k});
          }
        }
      } else {
        for ( let j = 1; j < n; j++ ) {
          // Break tries on E-W (if columns) / S-N (if rows) edges
          const t = (j + EPS) / n;
          const q = Math.mix(q0, q1, t) + eq;
          const r = Math.mix(r0, r1, t) + er;
          const s = 0 - q - r;
          path.push(this.getOffset({q, r, s}));
        }
        path.push(this.getOffset(c1));
      }

      c0 = c1;
      q0 = q1;
      r0 = r1;
      k0 = k1;
    }

    return path;
  }

  /* -------------------------------------------- */

  /** @override */
  getTranslatedPoint(point, direction, distance) {
    direction = Math.toRadians(direction);
    const dx = Math.cos(direction);
    const dy = Math.sin(direction);
    let q;
    let r;
    if ( this.columns ) {
      q = (2 * Math.SQRT1_3) * dx;
      r = (-0.5 * q) + dy;
    } else {
      r = (2 * Math.SQRT1_3) * dy;
      q = (-0.5 * r) + dx;
    }
    const s = distance / this.distance * this.size / ((Math.abs(r) + Math.abs(q) + Math.abs(q + r)) / 2);
    const x = point.x + (dx * s);
    const y = point.y + (dy * s);
    const elevation = point.elevation;
    return elevation !== undefined ? {x, y, elevation} : {x, y};
  }

  /* -------------------------------------------- */

  /** @override */
  getCircle({x, y}, radius) {
    if ( radius <= 0 ) return [];
    const r = radius / this.distance * this.size;
    if ( this.columns ) {
      const x0 = r * (Math.SQRT3 / 2);
      const x1 = -x0;
      const y0 = r;
      const y1 = y0 / 2;
      const y2 = -y1;
      const y3 = -y0;
      return [{x: x, y: y + y0}, {x: x + x1, y: y + y1}, {x: x + x1, y: y + y2},
        {x: x, y: y + y3}, {x: x + x0, y: y + y2}, {x: x + x0, y: y + y1}];
    } else {
      const y0 = r * (Math.SQRT3 / 2);
      const y1 = -y0;
      const x0 = r;
      const x1 = x0 / 2;
      const x2 = -x1;
      const x3 = -x0;
      return [{x: x + x0, y: y}, {x: x + x1, y: y + y0}, {x: x + x2, y: y + y0},
        {x: x + x3, y: y}, {x: x + x2, y: y + y1}, {x: x + x1, y: y + y1}];
    }
  }

  /* -------------------------------------------- */
  /*  Conversion Functions                        */
  /* -------------------------------------------- */

  /**
   * Round the fractional cube coordinates (q, r, s) / (q, r, s, k).
   * The k-coordinate is floored.
   * @see {@link https://www.redblobgames.com/grids/hexagons/}
   * @overload
   * @param {HexagonalGridCube2D} cube    The fractional cube coordinates
   * @returns {HexagonalGridCube2D}       The rounded integer cube coordinates
   */
  /**
   * @overload
   * @param {HexagonalGridCube3D} cube    The fractional cube coordinates
   * @returns {HexagonalGridCube3D}       The rounded integer cube coordinates
   */
  static cubeRound({q, r, s, k}) {
    let iq = Math.round(q);
    let ir = Math.round(r);
    let is = Math.round(s);
    const dq = Math.abs(iq - q);
    const dr = Math.abs(ir - r);
    const ds = Math.abs(is - s);

    if ( (dq > dr) && (dq > ds) ) {
      iq = -ir - is;
    } else if ( dr > ds ) {
      ir = -iq - is;
    } else {
      is = -iq - ir;
    }

    q = iq | 0;
    r = ir | 0;
    s = is | 0;
    return k !== undefined ? {q, r, s, k: Math.floor(k + 1e-8) | 0} : {q, r, s};
  }

  /* -------------------------------------------- */

  /**
   * Convert point coordinates (x, y) / (x, y, elevation) into cube coordinates (q, r, s) / (q, r, s, k).
   * Inverse of {@link HexagonalGrid#cubeToPoint}.
   * @see {@link https://www.redblobgames.com/grids/hexagons/}
   * @overload
   * @param {Point} point              The point
   * @returns {HexagonalGridCube2D}    The (fractional) cube coordinates
   */
  /**
   * @overload
   * @param {ElevatedPoint} point      The point
   * @returns {HexagonalGridCube3D}    The (fractional) cube coordinates
   */
  pointToCube({x, y, elevation}) {
    let q;
    let r;

    const size = this.size;
    x /= size;
    y /= size;

    if ( this.columns ) {
      q = ((2 * Math.SQRT1_3) * x) - (2 / 3);
      r = (-0.5 * (q + (this.even ? 1 : 0))) + y;
    } else {
      r = ((2 * Math.SQRT1_3) * y) - (2 / 3);
      q = (-0.5 * (r + (this.even ? 1 : 0))) + x;
    }

    const s = 0 - q - r;
    return elevation !== undefined ? {q, r, s, k: elevation / this.distance} : {q, r, s};
  }

  /* -------------------------------------------- */

  /**
   * Convert cube coordinates (q, r, s) / (q, r, s, k) into point coordinates (x, y) / (x, y, elevation).
   * Inverse of {@link HexagonalGrid#pointToCube}.
   * @see {@link https://www.redblobgames.com/grids/hexagons/}
   * @overload
   * @param {HexagonalGridCube2D} cube    The cube coordinates
   * @returns {Point}                     The point coordinates
   */
  /**
   * @overload
   * @param {HexagonalGridCube3D} cube    The cube coordinates
   * @returns {ElevatedPoint}             The point coordinates
   */
  cubeToPoint({q, r, k}) {
    let x;
    let y;

    const size = this.size;
    if ( this.columns ) {
      x = (0.5 * Math.SQRT1_3) * (((3 * q) + 2) * size);
      y = ((0.5 * (q + (this.even ? 1 : 0))) + r) * size;
    } else {
      y = (0.5 * Math.SQRT1_3) * (((3 * r) + 2) * size);
      x = ((0.5 * (r + (this.even ? 1 : 0))) + q) * size;
    }

    return k !== undefined ? {x, y, elevation: k * this.distance} : {x, y};
  }

  /* -------------------------------------------- */

  /**
   * Convert offset coordinates (i, j) / (i, j, k) into integer cube coordinates (q, r, s) / (q, r, s, k).
   * Inverse of {@link HexagonalGrid#cubeToOffset}.
   * @see {@link https://www.redblobgames.com/grids/hexagons/}
   * @overload
   * @param {GridOffset2D} offset      The offset coordinates
   * @returns {HexagonalGridCube2D}    The integer cube coordinates
   */
  /**
   * @overload
   * @param {GridOffset3D} offset      The offset coordinates
   * @returns {HexagonalGridCube3D}    The integer cube coordinates
   */
  offsetToCube({i, j, k}) {
    let q;
    let r;
    if ( this.columns ) {
      q = j;
      r = i - ((j + ((this.even ? 1 : -1) * (j & 1))) >> 1);
    } else {
      q = j - ((i + ((this.even ? 1 : -1) * (i & 1))) >> 1);
      r = i;
    }
    const s = 0 - q - r;
    return k !== undefined ? {q, r, s, k} : {q, r, s};
  }

  /* -------------------------------------------- */

  /**
   * Convert integer cube coordinates (q, r, s) / (q, r, s, k) into offset coordinates (i, j) / (i, j, k).
   * Inverse of {@link HexagonalGrid#offsetToCube}.
   * @see {@link https://www.redblobgames.com/grids/hexagons/}
   * @overload
   * @param {HexagonalGridCube2D} cube    The cube coordinates
   * @returns {GridOffset2D}              The offset coordinates
   */
  /**
   * @overload
   * @param {HexagonalGridCube3D} cube    The cube coordinates
   * @returns {GridOffset3D}              The offset coordinates
   */
  cubeToOffset({q, r, k}) {
    let i;
    let j;
    if ( this.columns ) {
      j = q;
      i = r + ((q + ((this.even ? 1 : -1) * (q & 1))) >> 1);
    } else {
      i = r;
      j = q + ((r + ((this.even ? 1 : -1) * (r & 1))) >> 1);
    }
    return k !== undefined ? {i, j, k} : {i, j};
  }

  /* -------------------------------------------- */

  /**
   * Measure the distance in hexagons between two cube coordinates.
   * @see {@link https://www.redblobgames.com/grids/hexagons/}
   * @param {HexagonalGridCube2D} a    The first cube coordinates
   * @param {HexagonalGridCube2D} b    The second cube coordinates
   * @returns {number}                 The distance between the two cube coordinates in hexagons
   */
  static cubeDistance(a, b) {
    const dq = a.q - b.q;
    const dr = a.r - b.r;
    return (Math.abs(dq) + Math.abs(dr) + Math.abs(dq + dr)) / 2;
  }

  /* -------------------------------------------- */

  /**
   * Used by {@link HexagonalGrid#snapToCenter}.
   * @type {Point}
   */
  static #TEMP_POINT = {x: 0, y: 0};

  /* -------------------------------------------- */

  /**
   * Used by {@link HexagonalGrid#snapToCenter}.
   * Always an odd grid!
   * @type {HexagonalGrid}
   */
  static #TEMP_GRID = new HexagonalGrid({size: 1});

  /* -------------------------------------------- */
  /*  Deprecations and Compatibility              */
  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  static get POINTY_HEX_BORDERS() {
    const msg = "HexagonalGrid.POINTY_HEX_BORDERS is deprecated without replacement.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return this.#POINTY_HEX_BORDERS;
  }

  /**
   * @deprecated since v12
   * @ignore
   */
  static #POINTY_HEX_BORDERS = {
    0.5: [[0, 0.25], [0.5, 0], [1, 0.25], [1, 0.75], [0.5, 1], [0, 0.75]],
    1: [[0, 0.25], [0.5, 0], [1, 0.25], [1, 0.75], [0.5, 1], [0, 0.75]],
    2: [
      [.5, 0], [.75, 1/7], [.75, 3/7], [1, 4/7], [1, 6/7], [.75, 1], [.5, 6/7], [.25, 1], [0, 6/7], [0, 4/7],
      [.25, 3/7], [.25, 1/7]
    ],
    3: [
      [.5, .1], [2/3, 0], [5/6, .1], [5/6, .3], [1, .4], [1, .6], [5/6, .7], [5/6, .9], [2/3, 1], [.5, .9], [1/3, 1],
      [1/6, .9], [1/6, .7], [0, .6], [0, .4], [1/6, .3], [1/6, .1], [1/3, 0]
    ],
    4: [
      [.5, 0], [5/8, 1/13], [.75, 0], [7/8, 1/13], [7/8, 3/13], [1, 4/13], [1, 6/13], [7/8, 7/13], [7/8, 9/13],
      [.75, 10/13], [.75, 12/13], [5/8, 1], [.5, 12/13], [3/8, 1], [.25, 12/13], [.25, 10/13], [1/8, 9/13],
      [1/8, 7/13], [0, 6/13], [0, 4/13], [1/8, 3/13], [1/8, 1/13], [.25, 0], [3/8, 1/13]
    ]
  };

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  static get FLAT_HEX_BORDERS() {
    const msg = "HexagonalGrid.FLAT_HEX_BORDERS is deprecated without replacement.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return this.#FLAT_HEX_BORDERS;
  }

  /**
   * @deprecated since v12
   * @ignore
   */
  static #FLAT_HEX_BORDERS = {
    0.5: [[0, 0.5], [0.25, 0], [0.75, 0], [1, 0.5], [0.75, 1], [0.25, 1]],
    1: [[0, 0.5], [0.25, 0], [0.75, 0], [1, 0.5], [0.75, 1], [0.25, 1]],
    2: [
      [3/7, .25], [4/7, 0], [6/7, 0], [1, .25], [6/7, .5], [1, .75], [6/7, 1], [4/7, 1], [3/7, .75], [1/7, .75],
      [0, .5], [1/7, .25]
    ],
    3: [
      [.4, 0], [.6, 0], [.7, 1/6], [.9, 1/6], [1, 1/3], [.9, .5], [1, 2/3], [.9, 5/6], [.7, 5/6], [.6, 1], [.4, 1],
      [.3, 5/6], [.1, 5/6], [0, 2/3], [.1, .5], [0, 1/3], [.1, 1/6], [.3, 1/6]
    ],
    4: [
      [6/13, 0], [7/13, 1/8], [9/13, 1/8], [10/13, .25], [12/13, .25], [1, 3/8], [12/13, .5], [1, 5/8], [12/13, .75],
      [10/13, .75], [9/13, 7/8], [7/13, 7/8], [6/13, 1], [4/13, 1], [3/13, 7/8], [1/13, 7/8], [0, .75], [1/13, 5/8],
      [0, .5], [1/13, 3/8], [0, .25], [1/13, 1/8], [3/13, 1/8], [4/13, 0]
    ]
  };

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  static get pointyHexPoints() {
    const msg = "HexagonalGrid.pointyHexPoints is deprecated without replacement.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return this.#POINTY_HEX_BORDERS[1];
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  static get flatHexPoints() {
    const msg = "HexagonalGrid.flatHexPoints is deprecated without replacement.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return this.#FLAT_HEX_BORDERS[1];
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  get hexPoints() {
    const msg = "HexagonalGrid#hexPoints is deprecated without replacement.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return this.columns ? this.constructor.flatHexPoints : this.constructor.pointyHexPoints;
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  getPolygon(x, y, w, h, points) {
    const msg = "HexagonalGrid#getPolygon is deprecated. You can get the shape of the hex with HexagonalGrid#getShape "
      + "and the polygon of any hex with HexagonalGrid#getVertices.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    w = w ?? this.sizeX;
    h = h ?? this.sizeY;
    points ??= this.hexPoints;
    const poly = [];
    for ( let i=0; i < points.length; i++ ) {
      poly.push(x + (w * points[i][0]), y + (h * points[i][1]));
    }
    return poly;
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  getBorderPolygon(w, h, p) {
    const msg = "HexagonalGrid#getBorderPolygon is deprecated. "
      + "If you need the shape of a Token, use Token#shape/getShape instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    const points = this.columns ? this.constructor.FLAT_HEX_BORDERS[w] : this.constructor.POINTY_HEX_BORDERS[w];
    if ( (w !== h) || !points ) return null;
    const p2 = p / 2;
    const p4 = p / 4;
    const r = this.getRect(w, h);
    return this.getPolygon(-p4, -p4, r.width + p2, r.height + p2, points);
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  getRect(w, h) {
    const msg = "HexagonalGrid#getRect is deprecated. If you need the size of a Token, use Token#getSize instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    if ( !this.columns || (w < 1) ) w *= this.sizeX;
    else w = (this.sizeX * .75 * (w - 1)) + this.sizeX;
    if ( this.columns || (h < 1) ) h *= this.sizeY;
    else h = (this.sizeY * .75 * (h - 1)) + this.sizeY;
    return new PIXI.Rectangle(0, 0, w, h);
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  _adjustSnapForTokenSize(x, y, token) {
    const msg = "HexagonalGrid#_adjustSnapForTokenSize is deprecated.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    if ( (token.document.width <= 1) && (token.document.height <= 1) ) {
      const [row, col] = this.getGridPositionFromPixels(x, y);
      const [x0, y0] = this.getPixelsFromGridPosition(row, col);
      return [x0 + (this.sizeX / 2) - (token.w / 2), y0 + (this.sizeY / 2) - (token.h / 2)];
    }

    if ( this.columns && (token.document.height > 1) ) y -= this.sizeY / 2;
    if ( !this.columns && (token.document.width > 1) ) x -= this.sizeX / 2;
    return [x, y];
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  static computeDimensions({columns, size, legacy}) {
    const msg = "HexagonalGrid.computeDimensions is deprecated without replacement.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});

    // Legacy dimensions (deprecated)
    if ( legacy ) {
      if ( columns ) return { width: size, height: (Math.SQRT3 / 2) * size };
      return { width: (Math.SQRT3 / 2) * size, height: size };
    }

    // Columnar orientation
    if ( columns ) return { width: (2 * size) / Math.SQRT3, height: size };

    // Row orientation
    return { width: size, height: (2 * size) / Math.SQRT3 };
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  get columnar() {
    const msg = "HexagonalGrid#columnar is deprecated in favor of HexagonalGrid#columns.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return this.columns;
  }

  /**
   * @deprecated since v12
   * @ignore
   */
  set columnar(value) {
    const msg = "HexagonalGrid#columnar is deprecated in favor of HexagonalGrid#columns.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    this.columns = value;
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  getCenter(x, y) {
    const msg = "HexagonalGrid#getCenter is deprecated. Use HexagonalGrid#getCenterPoint instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    const [x0, y0] = this.getTopLeft(x, y);
    return [x0 + (this.sizeX / 2), y0 + (this.sizeY / 2)];
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  getSnappedPosition(x, y, interval=1, {token}={}) {
    const msg = "HexagonalGrid#getSnappedPosition is deprecated. Use HexagonalGrid#getSnappedPoint instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    if ( interval === 0 ) return {x: Math.round(x), y: Math.round(y)};

    // At precision 5, return the center or nearest vertex
    if ( interval === 5) {
      const w4 = this.w / 4;
      const h4 = this.h / 4;

      // Distance relative to center
      const [xc, yc] = this.getCenter(x, y);
      const dx = x - xc;
      const dy = y - yc;
      let ox = dx.between(-w4, w4) ? 0 : Math.sign(dx);
      let oy = dy.between(-h4, h4) ? 0 : Math.sign(dy);

      // Closest to the center
      if ( (ox === 0) && (oy === 0) ) return {x: xc, y: yc};

      // Closest vertex based on offset
      if ( this.columns && (ox === 0) ) ox = Math.sign(dx) ?? -1;
      if ( !this.columns && (oy === 0) ) oy = Math.sign(dy) ?? -1;
      const {x: x0, y: y0 } = this.#getClosestVertex(xc, yc, ox, oy);
      return {x: Math.round(x0), y: Math.round(y0)};
    }

    // Start with the closest top-left grid position
    if ( token ) {
      if ( this.columns && (token.document.height > 1) ) y += this.sizeY / 2;
      if ( !this.columns && (token.document.width > 1) ) x += this.sizeX / 2;
    }
    const options = {
      columns: this.columns,
      even: this.even,
      size: this.size,
      width: this.sizeX,
      height: this.sizeY
    };
    const offset = HexagonalGrid.pixelsToOffset({x, y}, options, "round");
    const point = HexagonalGrid.offsetToPixels(offset, options);

    // Adjust pixel coordinate for token size
    let x0 = point.x;
    let y0 = point.y;
    if ( token ) [x0, y0] = this._adjustSnapForTokenSize(x0, y0, token);

    // Snap directly at interval 1
    if ( interval === 1 ) return {x: x0, y: y0};

    // Round the remainder
    const dx = (x - x0).toNearest(this.w / interval);
    const dy = (y - y0).toNearest(this.h / interval);
    return {x: Math.round(x0 + dx), y: Math.round(y0 + dy)};
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  #getClosestVertex(xc, yc, ox, oy) {
    const b = ox + (oy << 2); // Bit shift to make a unique reference
    const vertices = this.columns
      ? {"-1": 0, "-5": 1, "-3": 2, 1: 3, 5: 4, 3: 5}   // Flat hex vertices
      : {"-5": 0, "-4": 1, "-3": 2, 5: 3, 4: 4, 3: 5};  // Pointy hex vertices
    const idx = vertices[b];
    const pt = this.hexPoints[idx];
    return {
      x: (xc - (this.sizeX / 2)) + (pt[0] * this.sizeX),
      y: (yc - (this.sizeY / 2)) + (pt[1] * this.sizeY)
    };
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  #measureDistance(p0, p1) {
    const [i0, j0] = this.getGridPositionFromPixels(p0.x, p0.y);
    const [i1, j1] = this.getGridPositionFromPixels(p1.x, p1.y);
    const c0 = this.getCube({i: i0, j: j0});
    const c1 = this.getCube({i: i1, j: j1});
    return HexagonalGrid.cubeDistance(c0, c1);
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  getGridPositionFromPixels(x, y) {
    const msg = "HexagonalGrid#getGridPositionFromPixels is deprecated. This function is based on the \"brick wall\" grid. "
    + " For getting the offset coordinates of the hex containing the given point use HexagonalGrid#getOffset.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    const {row, col} = HexagonalGrid.pixelsToOffset({x, y}, {
      columns: this.columns,
      even: this.even,
      size: this.size,
      width: this.sizeX,
      height: this.sizeY
    });
    return [row, col];
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  getPixelsFromGridPosition(row, col) {
    const msg = "HexagonalGrid#getPixelsFromGridPosition is deprecated. This function is based on the \"brick wall\" grid. "
    + " For getting the top-left coordinates of the hex at the given offset coordinates use HexagonalGrid#getTopLeftPoint.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    const {x, y} = HexagonalGrid.offsetToPixels({row, col}, {
      columns: this.columns,
      even: this.even,
      size: this.size,
      width: this.sizeX,
      height: this.sizeY
    });
    return [Math.ceil(x), Math.ceil(y)];
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  shiftPosition(x, y, dx, dy, {token}={}) {
    const msg = "BaseGrid#shiftPosition is deprecated. Use BaseGrid#getShiftedPoint instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    const [row, col] = this.getGridPositionFromPixels(x, y);

    // Adjust diagonal moves for offset
    const isDiagonal = (dx !== 0) && (dy !== 0);
    if ( isDiagonal ) {

      // Column orientation
      if ( this.columns ) {
        const isEven = ((col+1) % 2 === 0) === this.even;
        if ( isEven && (dy > 0)) dy--;
        else if ( !isEven && (dy < 0)) dy++;
      }

      // Row orientation
      else {
        const isEven = ((row + 1) % 2 === 0) === this.even;
        if ( isEven && (dx > 0) ) dx--;
        else if ( !isEven && (dx < 0 ) ) dx++;
      }
    }
    const [shiftX, shiftY] = this.getPixelsFromGridPosition(row+dy, col+dx);
    if ( token ) return this._adjustSnapForTokenSize(shiftX, shiftY, token);
    return [shiftX, shiftY];
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  measureDistance(origin, target, options) {
    const msg = "HexagonalGrid#measureDistance now returns the same result as GridLayer#measureDistance instead of the cube distance "
      + " (breaking). Use HexagonalGrid#measurePath instead to get the number of steps (cube distance) between the origin and target.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return super.measureDistance(origin, target, options);
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  measureDistances(segments, options={}) {
    const msg = "HexagonalGrid#measureDistances is deprecated. "
      + "Use HexagonalGrid#measurePath instead, which returns grid distance (gridSpaces: true) and Euclidean distance (gridSpaces: false).";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    if ( !options.gridSpaces ) return super.measureDistances(segments, options);
    return segments.map(s => {
      const r = s.ray;
      return this.#measureDistance(r.A, r.B) * this.distance;
    });
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  _adjustPositionForTokenSize(row, col, token) {
    const msg = "HexagonalGrid#_adjustPositionForTokenSize is deprecated.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    if ( this.columns && (token.document.height > 1) ) row++;
    if ( !this.columns && (token.document.width > 1) ) col++;
    return [row, col];
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  static getConfig(type, size) {
    const msg = "HexagonalGrid.getConfig is deprecated without replacement.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    const config = {
      columns: [GRID_TYPES.HEXODDQ, GRID_TYPES.HEXEVENQ].includes(type),
      even: [GRID_TYPES.HEXEVENR, GRID_TYPES.HEXEVENQ].includes(type),
      size: size
    };
    const {width, height} = HexagonalGrid.computeDimensions(config);
    config.width = width;
    config.height = height;
    return config;
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  static offsetToCube({row, col}={}, {columns=true, even=false}={}) {
    const msg = "HexagonalGrid.offsetToCube is deprecated. Use HexagonalGrid#offsetToCube instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    return new HexagonalGrid({size: 100, columns, even}).offsetToCube({i: row, j: col});
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  static cubeToOffset(cube={}, {columns=true, even=false}={}) {
    const msg = "HexagonalGrid.cubeToOffset is deprecated. Use HexagonalGrid#cubeToOffset instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    const {i: row, j: col} = new HexagonalGrid({size: 100, columns, even}).cubeToOffset(cube);
    return {row, col};
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  static pixelToCube(point, config) {
    const msg = "HexagonalGrid.pixelToCube is deprecated. Use HexagonalGrid#pointToCube instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    const {x, y} = point ?? {};
    const {size} = config;
    const cx = x / (size / 2);
    const cy = y / (size / 2);

    // Fractional hex coordinates, might not satisfy (fx + fy + fz = 0) due to rounding
    const fr = (2/3) * cx;
    const fq = ((-1/3) * cx) + ((1 / Math.sqrt(3)) * cy);
    const fs = ((-1/3) * cx) - ((1 / Math.sqrt(3)) * cy);

    // Convert to integer triangle coordinates
    const a = Math.ceil(fr - fq);
    const b = Math.ceil(fq - fs);
    const c = Math.ceil(fs - fr);

    // Convert back to cube coordinates
    return {
      q: Math.round((a - c) / 3),
      r: Math.round((c - b) / 3),
      s: Math.round((b - a) / 3)
    };
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  static offsetToPixels({row, col}, {columns, even, width, height}) {
    const msg = "HexagonalGrid.offsetToPixels is deprecated. Use HexagonalGrid#getTopLeftPoint instead.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    let x;
    let y;

    // Flat-topped hexes
    if ( columns ) {
      x = Math.ceil(col * (width * 0.75));
      const isEven = (col + 1) % 2 === 0;
      y = Math.ceil((row - (even === isEven ? 0.5 : 0)) * height);
    }

    // Pointy-topped hexes
    else {
      y = Math.ceil(row * (height * 0.75));
      const isEven = (row + 1) % 2 === 0;
      x = Math.ceil((col - (even === isEven ? 0.5 : 0)) * width);
    }

    // Return the pixel coordinate
    return {x, y};
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  static pixelsToOffset({x, y}, config, method="floor") {
    const msg = "HexagonalGrid.pixelsToOffset is deprecated without replacement. This function is based on the \"brick wall\" grid. "
      + " For getting the offset coordinates of the hex containing the given point use HexagonalGrid#getOffset.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    const {columns, even, width, height} = config;
    const fn = Math[method];
    let row;
    let col;

    // Columnar orientation
    if ( columns ) {
      col = fn(x / (width * 0.75));
      const isEven = (col + 1) % 2 === 0;
      row = fn((y / height) + (even === isEven ? 0.5 : 0));
    }

    // Row orientation
    else {
      row = fn(y / (height * 0.75));
      const isEven = (row + 1) % 2 === 0;
      col = fn((x / width) + (even === isEven ? 0.5 : 0));
    }
    return {row, col};
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  getAStarPath(start, goal, options) {
    const msg = "HexagonalGrid#getAStarPath is deprecated without replacement.";
    logCompatibilityWarning(msg, {since: 12, until: 14, once: true});
    const costs = new Map();

    // Create a prioritized frontier sorted by increasing cost
    const frontier = [];
    const explore = (hex, from, cost) => {
      const idx = frontier.findIndex(l => l.cost > cost);
      if ( idx === -1 ) frontier.push({hex, cost, from});
      else frontier.splice(idx, 0, {hex, cost, from});
      costs.set(hex, cost);
    };
    explore(start, null, 0);

    // Expand the frontier, exploring towards the goal
    let current;
    let solution;
    while ( frontier.length ) {
      current = frontier.shift();
      if ( current.cost === Infinity ) break;
      if ( current.hex.equals(goal) ) {
        solution = current;
        break;
      }
      for ( const next of current.hex.getNeighbors() ) {
        const deltaCost = next.getTravelCost instanceof Function ? next.getTravelCost(current.hex, options) : 1;
        const newCost = current.cost + deltaCost;     // Total cost of reaching this hex
        if ( costs.get(next) <= newCost ) continue;   // We already made it here in the lowest-cost way
        explore(next, current, newCost);
      }
    }

    // Ensure a path was achieved
    if ( !solution ) {
      throw new Error("No valid path between these positions exists");
    }

    // Return the optimal path and cost
    const path = [];
    let c = solution;
    while ( c.from ) {
      path.unshift(c.hex);
      c = c.from;
    }
    return {from: start, to: goal, cost: solution.cost, path};
  }
}

/**
 * @import {SceneData} from "./_types.mjs";
 * @import BaseGrid from "../grid/base.mjs";
 */

/**
 * The Scene Document.
 * Defines the DataSchema and common behaviors for a Scene which are shared between both client and server.
 * @extends {Document<SceneData>}
 * @mixes SceneData
 * @category Documents
 */
class BaseScene extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "Scene",
    collection: "scenes",
    indexed: true,
    compendiumIndexFields: ["_id", "name", "thumb", "sort", "folder"],
    embedded: {
      AmbientLight: "lights",
      AmbientSound: "sounds",
      Drawing: "drawings",
      MeasuredTemplate: "templates",
      Note: "notes",
      Region: "regions",
      Tile: "tiles",
      Token: "tokens",
      Wall: "walls"
    },
    label: "DOCUMENT.Scene",
    labelPlural: "DOCUMENT.Scenes",
    preserveOnImport: [...super.metadata.preserveOnImport, "active"],
    schemaVersion: "13.341"
  }, {inplace: false}));

  /** @inheritDoc */
  static defineSchema() {
    const documents = foundry.documents;
    // Define reusable ambience schema for environment
    const environmentData = init => new SchemaField({
      hue: new HueField({required: true, initial: init.hue}),
      intensity: new AlphaField({required: true, nullable: false, initial: init.intensity}),
      luminosity: new NumberField({required: true, nullable: false, initial: init.luminosity, min: -1, max: 1}),
      saturation: new NumberField({required: true, nullable: false, initial: init.saturation, min: -1, max: 1}),
      shadows: new NumberField({required: true, nullable: false, initial: init.shadows, min: 0, max: 1})
    });
    // Reuse parts of the LightData schema for the global light
    const lightDataSchema = foundry.data.LightData.defineSchema();

    return {
      _id: new DocumentIdField(),
      name: new StringField({required: true, blank: false, textSearch: true}),

      // Navigation
      active: new BooleanField(),
      navigation: new BooleanField({initial: true}),
      navOrder: new NumberField({required: true, nullable: false, integer: true, initial: 0}),
      navName: new StringField({textSearch: true}),

      // Canvas Dimensions
      background: new TextureData(),
      foreground: new FilePathField({categories: ["IMAGE", "VIDEO"], virtual: true}),
      foregroundElevation: new NumberField({required: true, positive: true, integer: true}),
      thumb: new FilePathField({categories: ["IMAGE"]}),
      width: new NumberField({integer: true, positive: true, initial: 4000}),
      height: new NumberField({integer: true, positive: true, initial: 3000}),
      padding: new NumberField({required: true, nullable: false, min: 0, max: 0.5, step: 0.05, initial: 0.25}),
      initial: new SchemaField({
        x: new NumberField({integer: true, required: true}),
        y: new NumberField({integer: true, required: true}),
        scale: new NumberField({required: true, positive: true})
      }),
      backgroundColor: new ColorField({nullable: false, initial: "#999999"}),

      // Grid Configuration
      grid: new SchemaField({
        type: new NumberField({required: true, choices: Object.values(GRID_TYPES),
          initial: () => game.system.grid.type, validationError: "must be a value in CONST.GRID_TYPES"}),
        size: new NumberField({required: true, nullable: false, integer: true, min: GRID_MIN_SIZE,
          initial: 100, validationError: `must be an integer number of pixels, ${GRID_MIN_SIZE} or greater`}),
        style: new StringField({required: true, blank: false, initial: "solidLines"}),
        thickness: new NumberField({required: true, nullable: false, positive: true, integer: true, initial: 1}),
        color: new ColorField({required: true, nullable: false, initial: "#000000"}),
        alpha: new AlphaField({initial: 0.2}),
        distance: new NumberField({required: true, nullable: false, positive: true,
          initial: () => game.system.grid.distance}),
        units: new StringField({required: true, initial: () => game.system.grid.units})
      }),

      // Vision Configuration
      tokenVision: new BooleanField({initial: true}),
      fog: new SchemaField({
        exploration: new BooleanField({initial: true}),
        reset: new NumberField({required: false, initial: undefined}),
        overlay: new FilePathField({categories: ["IMAGE", "VIDEO"], virtual: true}),
        colors: new SchemaField({
          explored: new ColorField(),
          unexplored: new ColorField()
        })
      }),

      // Environment Configuration
      environment: new SchemaField({
        darknessLevel: new AlphaField({initial: 0}),
        darknessLock: new BooleanField({initial: false}),
        globalLight: new SchemaField({
          enabled: new BooleanField({required: true, initial: false}),
          alpha: lightDataSchema.alpha,
          bright: new BooleanField({required: true, initial: false}),
          color: lightDataSchema.color,
          coloration: lightDataSchema.coloration,
          luminosity: new NumberField({required: true, nullable: false, initial: 0, min: 0, max: 1}),
          saturation: lightDataSchema.saturation,
          contrast: lightDataSchema.contrast,
          shadows: lightDataSchema.shadows,
          darkness: lightDataSchema.darkness
        }),
        cycle: new BooleanField({initial: true}),
        base: environmentData({hue: 0, intensity: 0, luminosity: 0, saturation: 0, shadows: 0}),
        dark: environmentData({hue: 257/360, intensity: 0, luminosity: -0.25, saturation: 0, shadows: 0})
      }),

      // Embedded Collections
      drawings: new EmbeddedCollectionField(documents.BaseDrawing),
      tokens: new EmbeddedCollectionField(documents.BaseToken),
      lights: new EmbeddedCollectionField(documents.BaseAmbientLight),
      notes: new EmbeddedCollectionField(documents.BaseNote),
      sounds: new EmbeddedCollectionField(documents.BaseAmbientSound),
      regions: new EmbeddedCollectionField(documents.BaseRegion),
      templates: new EmbeddedCollectionField(documents.BaseMeasuredTemplate),
      tiles: new EmbeddedCollectionField(documents.BaseTile),
      walls: new EmbeddedCollectionField(documents.BaseWall),

      // Linked Documents
      playlist: new ForeignDocumentField(documents.BasePlaylist),
      playlistSound: new ForeignDocumentField(documents.BasePlaylistSound, {idOnly: true}),
      journal: new ForeignDocumentField(documents.BaseJournalEntry),
      journalEntryPage: new ForeignDocumentField(documents.BaseJournalEntryPage, {idOnly: true}),
      weather: new StringField({required: true}),

      // Permissions
      folder: new ForeignDocumentField(documents.BaseFolder),
      sort: new IntegerSortField(),
      ownership: new DocumentOwnershipField(),
      flags: new DocumentFlagsField(),
      _stats: new DocumentStatsField()
    };
  }

  /* -------------------------------------------- */

  /** @override */
  static LOCALIZATION_PREFIXES = ["DOCUMENT", "SCENE"];

  /* -------------------------------------------- */

  /**
   * The default grid defined by the system.
   * @type {BaseGrid}
   */
  static get defaultGrid() {
    if ( BaseScene.#defaultGrid ) return BaseScene.#defaultGrid;

    const T = GRID_TYPES;
    const {type, ...config} = game.system.grid;
    config.size = 100;

    // Gridless grid
    if ( type === T.GRIDLESS ) BaseScene.#defaultGrid = new GridlessGrid(config);

    // Square grid
    if ( type === T.SQUARE ) BaseScene.#defaultGrid = new SquareGrid(config);

    // Hexagonal grid
    if ( type.between(T.HEXODDR, T.HEXEVENQ) ) {
      config.columns = (type === T.HEXODDQ) || (type === T.HEXEVENQ);
      config.even = (type === T.HEXEVENR) || (type === T.HEXEVENQ);
      BaseScene.#defaultGrid = new HexagonalGrid(config);
    }

    return BaseScene.#defaultGrid;
  }

  static #defaultGrid;

  /* -------------------------------------------- */
  /*  Data Management                             */
  /* -------------------------------------------- */

  /** @inheritDoc */
  _initialize(options) {
    super._initialize(options);
    DocumentStatsField._shimDocument(this);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  updateSource(changes={}, options={}) {
    if ( "tokens" in changes ) {
      for ( const tokenChange of changes.tokens ) {
        this.tokens.get(tokenChange._id)?._prepareDeltaUpdate(tokenChange, options);
      }
    }
    return super.updateSource(changes, options);
  }

  /* -------------------------------------------- */
  /*  Deprecations and Compatibility              */
  /* -------------------------------------------- */

  /**
   * Static Initializer Block for deprecated properties.
   */
  static {
    const migrations = {
      fogExploration: "fog.exploration",
      fogReset: "fog.reset",
      fogOverlay: "fog.overlay",
      fogExploredColor: "fog.colors.explored",
      fogUnexploredColor: "fog.colors.unexplored",
      globalLight: "environment.globalLight.enabled",
      globalLightThreshold: "environment.globalLight.darkness.max",
      darkness: "environment.darknessLevel"
    };
    Object.defineProperties(this.prototype, Object.fromEntries(
      Object.entries(migrations).map(([o, n]) => [o, {
        get() {
          this.constructor._logDataFieldMigration(o, n, {since: 12, until: 14});
          return foundry.utils.getProperty(this, n);
        },
        set(v) {
          this.constructor._logDataFieldMigration(o, n, {since: 12, until: 14});
          return foundry.utils.setProperty(this, n, v);
        },
        configurable: true
      }])));
  }

  /* ---------------------------------------- */

  /** @inheritdoc */
  static migrateData(source) {
    /**
     * Migration to fog schema fields. Can be safely removed in V14+
     * @deprecated since v12
     */
    for ( const [oldKey, newKey] of Object.entries({
      fogExploration: "fog.exploration",
      fogReset: "fog.reset",
      fogOverlay: "fog.overlay",
      fogExploredColor: "fog.colors.explored",
      fogUnexploredColor: "fog.colors.unexplored"
    }) ) this._addDataFieldMigration(source, oldKey, newKey);

    /**
     * Migration to global light embedded fields. Can be safely removed in V14+
     * @deprecated since v12
     */
    this._addDataFieldMigration(source, "globalLight", "environment.globalLight.enabled");
    this._addDataFieldMigration(source, "globalLightThreshold", "environment.globalLight.darkness.max",
      d => d.globalLightThreshold ?? 1);

    /**
     * Migration to environment darkness level. Can be safely removed in V14+
     * @deprecated since v12
     */
    this._addDataFieldMigration(source, "darkness", "environment.darknessLevel");

    DocumentStatsField._migrateData(this, source);

    return super.migrateData(source);
  }

  /* ---------------------------------------- */

  /** @inheritdoc */
  static shimData(source, options) {

    /** @deprecated since v12 */
    this._addDataFieldShims(source, {
      fogExploration: "fog.exploration",
      fogReset: "fog.reset",
      fogOverlay: "fog.overlay",
      fogExploredColor: "fog.colors.explored",
      fogUnexploredColor: "fog.colors.unexplored",
      globalLight: "environment.globalLight.enabled",
      globalLightThreshold: "environment.globalLight.darkness.max",
      darkness: "environment.darknessLevel"
    }, {since: 12, until: 14});

    DocumentStatsField._shimData(this, source, options);

    return super.shimData(source, options);
  }
}

/**
 * @import {RegionData} from "./_types.mjs";
 */

/**
 * The Region Document.
 * Defines the DataSchema and common behaviors for a Region which are shared between both client and server.
 * @extends {Document<RegionData>}
 * @mixes RegionData
 * @category Documents
 */
class BaseRegion extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "Region",
    collection: "regions",
    label: "DOCUMENT.Region",
    labelPlural: "DOCUMENT.Regions",
    isEmbedded: true,
    embedded: {
      RegionBehavior: "behaviors"
    },
    schemaVersion: "13.341"
  }, {inplace: false}));

  /** @inheritdoc */
  static defineSchema() {
    const {BaseRegionBehavior} = foundry.documents;
    return {
      _id: new DocumentIdField(),
      name: new StringField({required: true, blank: false, textSearch: true}),
      color: new ColorField({required: true, nullable: false,
        initial: () => Color.fromHSV([Math.random(), 0.8, 0.8]).css}),
      shapes: new ArrayField(new TypedSchemaField(BaseShapeData.TYPES)),
      elevation: new SchemaField({
        bottom: new NumberField({required: true}), // Treat null as -Infinity
        top: new NumberField({required: true}) // Treat null as +Infinity
      }, {
        validate: d => (d.bottom ?? -Infinity) <= (d.top ?? Infinity),
        validationError: "elevation.top may not be less than elevation.bottom"
      }),
      behaviors: new EmbeddedCollectionField(BaseRegionBehavior),
      visibility: new NumberField({required: true,
        initial: CONST.REGION_VISIBILITY.LAYER,
        choices: Object.values(CONST.REGION_VISIBILITY)}),
      locked: new BooleanField(),
      flags: new DocumentFlagsField()
    };
  }

  /** @override */
  static LOCALIZATION_PREFIXES = ["DOCUMENT", "REGION"];
}

/**
 * @import {RegionBehaviorData} from "./_types.mjs";
 * @import {DocumentPermissionTest} from "@common/abstract/_types.mjs";
 */

/**
 * The RegionBehavior Document.
 * Defines the DataSchema and common behaviors for a RegionBehavior which are shared between both client and server.
 * @extends {Document<RegionBehaviorData>}
 * @mixes RegionBehaviorData
 * @category Documents
 */
class BaseRegionBehavior extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "RegionBehavior",
    collection: "behaviors",
    label: "DOCUMENT.RegionBehavior",
    labelPlural: "DOCUMENT.RegionBehaviors",
    coreTypes: ["adjustDarknessLevel", "displayScrollingText", "executeMacro", "executeScript", "modifyMovementCost", "pauseGame", "suppressWeather", "teleportToken", "toggleBehavior"],
    hasTypeData: true,
    isEmbedded: true,
    permissions: {
      create: this.#canCreate,
      update: this.#canUpdate
    },
    schemaVersion: "13.341"
  }, {inplace: false}));

  /** @inheritdoc */
  static defineSchema() {
    return {
      _id: new DocumentIdField(),
      name: new StringField({required: true, blank: true, textSearch: true}),
      type: new DocumentTypeField(this),
      system: new TypeDataField(this),
      disabled: new BooleanField(),
      flags: new DocumentFlagsField(),
      _stats: new DocumentStatsField()
    };
  }

  /** @override */
  static LOCALIZATION_PREFIXES = ["DOCUMENT", "BEHAVIOR"];

  /* -------------------------------------------- */

  /** @override */
  static canUserCreate(user) {
    return user.isGM;
  }

  /* ---------------------------------------- */

  /**
   * Is a user able to create the RegionBehavior document?
   * @type {DocumentPermissionTest}
   */
  static #canCreate(user, doc) {
    if ( (doc._source.type === "executeScript") && !user.hasPermission("MACRO_SCRIPT") ) return false;
    return user.isGM;
  }

  /* ---------------------------------------- */

  /**
   * Is a user able to update the RegionBehavior document?
   * @type {DocumentPermissionTest}
   */
  static #canUpdate(user, doc, data) {
    if ( (((doc._source.type === "executeScript") && ("system" in data) && ("source" in data.system))
      || (data.type === "executeScript")) && !user.hasPermission("MACRO_SCRIPT") ) return false;
    return user.isGM;
  }
}

/**
 * @import {UserData} from "./_types.mjs";
 * @import {DocumentPermissionTest} from "@common/abstract/_types.mjs";
 */

/**
 * The User Document.
 * Defines the DataSchema and common behaviors for a User which are shared between both client and server.
 * @extends {Document<UserData>}
 * @mixes UserData
 * @category Documents
 */
class BaseUser extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "User",
    collection: "users",
    label: "DOCUMENT.User",
    labelPlural: "DOCUMENT.Users",
    permissions: {
      create: this.#canCreate,
      update: this.#canUpdate,
      delete: this.#canDelete
    },
    schemaVersion: "13.341"
  }, {inplace: false}));

  /** @override */
  static LOCALIZATION_PREFIXES = ["DOCUMENT", "USER"];

  /* -------------------------------------------- */

  /** @inheritdoc */
  static defineSchema() {
    return {
      _id: new DocumentIdField(),
      name: new StringField({required: true, blank: false, textSearch: true}),
      role: new NumberField({required: true, choices: Object.values(USER_ROLES),
        initial: USER_ROLES.PLAYER, readonly: true}),
      password: new StringField({required: true, blank: true}),
      passwordSalt: new StringField(),
      avatar: new FilePathField({categories: ["IMAGE"]}),
      character: new ForeignDocumentField(BaseActor),
      color: new ColorField({required: true, nullable: false,
        initial: () => Color.fromHSV([Math.random(), 0.8, 0.8]).css
      }),
      pronouns: new StringField({required: true}),
      hotbar: new ObjectField({required: true, validate: BaseUser.#validateHotbar,
        validationError: "must be a mapping of slots to macro identifiers"}),
      permissions: new ObjectField({required: true, validate: BaseUser.#validatePermissions,
        validationError: "must be a mapping of permission names to booleans"}),
      flags: new DocumentFlagsField(),
      _stats: new DocumentStatsField()
    };
  }

  /* -------------------------------------------- */

  /**
   * Validate the structure of the User hotbar object
   * @param {object} bar      The attempted hotbar data
   * @returns {boolean}
   */
  static #validateHotbar(bar) {
    if ( typeof bar !== "object" ) return false;
    for ( const [k, v] of Object.entries(bar) ) {
      const slot = parseInt(k);
      if ( !slot || slot < 1 || slot > 50 ) return false;
      if ( !isValidId(v) ) return false;
    }
    return true;
  }

  /* -------------------------------------------- */

  /**
   * Validate the structure of the User permissions object
   * @param {object} perms      The attempted permissions data
   * @returns {boolean}
   */
  static #validatePermissions(perms) {
    for ( const [k, v] of Object.entries(perms) ) {
      if ( typeof k !== "string" ) return false;
      if ( k.startsWith("-=") ) {
        if ( v !== null ) return false;
      } else {
        if ( typeof v !== "boolean" ) return false;
      }
    }
    return true;
  }

  /* -------------------------------------------- */
  /*  Model Properties                            */
  /* -------------------------------------------- */

  /**
   * A convenience test for whether this User has the NONE role.
   * @type {boolean}
   */
  get isBanned() {
    return this.role === USER_ROLES.NONE;

  }

  /* -------------------------------------------- */

  /**
   * Test whether the User has a GAMEMASTER or ASSISTANT role in this World?
   * @type {boolean}
   */
  get isGM() {
    return this.hasRole(USER_ROLES.ASSISTANT);
  }

  /* -------------------------------------------- */

  /**
   * Test whether the User is able to perform a certain permission action.
   * The provided permission string may pertain to an explicit permission setting or a named user role.
   *
   * @param {string} action The action to test
   * @returns {boolean} Does the user have the ability to perform this action?
   */
  can(action) {
    if ( action in USER_PERMISSIONS ) return this.hasPermission(action);
    return this.hasRole(action);
  }

  /* ---------------------------------------- */

  /** @inheritdoc */
  getUserLevel(user) {
    return DOCUMENT_OWNERSHIP_LEVELS[user.id === this.id ? "OWNER" : "NONE"];
  }

  /* ---------------------------------------- */

  /**
   * Test whether the User has at least a specific permission
   * @param {string} permission The permission name from USER_PERMISSIONS to test
   * @returns {boolean} Does the user have at least this permission
   */
  hasPermission(permission) {
    if ( this.isBanned ) return false;

    // CASE 1: The user has the permission set explicitly
    const explicit = this.permissions[permission];
    if (explicit !== undefined) return explicit;

    // CASE 2: Permission defined by the user's role
    const rolePerms = game.permissions[permission];
    return rolePerms ? rolePerms.includes(this.role) : false;
  }

  /* ----------------------------------------- */

  /**
   * Test whether the User has at least the permission level of a certain role
   * @param {string|number} role    The role name from USER_ROLES to test
   * @param {boolean} [exact]       Require the role match to be exact
   * @returns {boolean}             Does the user have at this role level (or greater)?
   */
  hasRole(role, {exact = false} = {}) {
    const level = typeof role === "string" ? USER_ROLES[role] : role;
    if (level === undefined) return false;
    return exact ? this.role === level : this.role >= level;
  }

  /* ---------------------------------------- */
  /*  Model Permissions                       */
  /* ---------------------------------------- */

  /**
   * Is a user able to create an existing User?
   * @type {DocumentPermissionTest}
   */
  static #canCreate(user, doc, data) {
    if ( !user.isGM ) return false; // Only Assistants and above can create users.
    // Do not allow Assistants to create a new user with special permissions which might be greater than their own.
    if ( !isEmpty(doc.permissions) ) return user.hasRole(USER_ROLES.GAMEMASTER);
    return user.hasRole(doc.role);
  }

  /* -------------------------------------------- */

  /**
   * Is a user able to update an existing User?
   * @type {DocumentPermissionTest}
   */
  static #canUpdate(user, doc, changes) {
    const roles = USER_ROLES;
    if ( user.role === roles.GAMEMASTER ) return true; // Full GMs can do everything
    if ( user.role === roles.NONE ) return false; // Banned users can do nothing

    // Non-GMs cannot update certain fields.
    const restricted = ["permissions", "passwordSalt"];
    if ( user.role < roles.ASSISTANT ) restricted.push("name", "role");
    if ( doc.role === roles.GAMEMASTER ) restricted.push("password");
    if ( restricted.some(k => k in changes) ) return false;

    // Role changes may not escalate
    if ( ("role" in changes) && !user.hasRole(changes.role) ) return false;

    // Assistant GMs may modify other users. Players may only modify themselves
    return user.isGM || (user.id === doc.id);
  }

  /* -------------------------------------------- */

  /**
   * Is a user able to delete an existing User?
   * @type {DocumentPermissionTest}
   */
  static #canDelete(user, doc) {
    const role = Math.max(USER_ROLES.ASSISTANT, doc.role);
    return user.hasRole(role);
  }
}

/**
 * @import {SettingData} from "./_types.mjs";
 * @import {DocumentPermissionTest} from "@common/abstract/_types.mjs";
 */

/**
 * The Setting Document.
 * Defines the DataSchema and common behaviors for a Setting which are shared between both client and server.
 * @extends {Document<SettingData>}
 * @mixes SettingData
 * @category Documents
 */
class BaseSetting extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "Setting",
    collection: "settings",
    label: "DOCUMENT.Setting",
    labelPlural: "DOCUMENT.Settings",
    permissions: {
      create: this.#canModify,
      update: this.#canModify,
      delete: this.#canModify
    },
    schemaVersion: "13.341"
  }, {inplace: false}));

  /** @inheritdoc */
  static defineSchema() {
    return {
      _id: new DocumentIdField(),
      key: new StringField({required: true, nullable: false, blank: false,
        validate: k => k.split(".").length >= 2,
        validationError: "must have the format {scope}.{field}"}),
      value: new JSONField({required: true, nullable: true, initial: null}),
      user: new ForeignDocumentField(BaseUser, {idOnly: true}),
      _stats: new DocumentStatsField()
    };
  }

  /* -------------------------------------------- */

  /**
   * The settings that only full GMs can modify.
   * @type {string[]}
   */
  static #GAMEMASTER_ONLY_KEYS = ["core.permissions"];

  /* -------------------------------------------- */

  /**
   * The settings that assistant GMs can modify regardless of their permission.
   * @type {string[]}
   */
  static #ALLOWED_ASSISTANT_KEYS = ["core.time", "core.combatTrackerConfig", "core.sheetClasses", "core.scrollingStatusText",
    "core.tokenDragPreview", "core.adventureImports", "core.gridDiagonals", "core.gridTemplates", "core.coneTemplateType"];

  /* -------------------------------------------- */

  /** @override */
  static canUserCreate(user) {
    return user.hasPermission("SETTINGS_MODIFY");
  }

  /* -------------------------------------------- */

  /**
   * Define special rules which allow certain settings to be updated.
   * @type {DocumentPermissionTest}
   */
  static #canModify(user, doc, data) {
    if ( BaseSetting.#GAMEMASTER_ONLY_KEYS.includes(doc._source.key)
      && (!("key" in data) || BaseSetting.#GAMEMASTER_ONLY_KEYS.includes(data.key)) ) return user.hasRole("GAMEMASTER");
    const sourceUser = doc._source.user;
    const targetUser = data?.user;
    const sourceMatch = !sourceUser || (sourceUser === user.id);
    const targetMatch = !targetUser || (targetUser === user.id);
    if ( sourceUser || targetUser ) return user.isGM || (sourceMatch && targetMatch);
    if ( user.hasPermission("SETTINGS_MODIFY") ) return true;
    if ( !user.isGM ) return false;
    return BaseSetting.#ALLOWED_ASSISTANT_KEYS.includes(doc._source.key)
      && (!("key" in data) || BaseSetting.#ALLOWED_ASSISTANT_KEYS.includes(data.key));
  }
}

/**
 * @import {TableResultData} from "./_types.mjs";
 * @import {DocumentPermissionTest} from "@common/abstract/_types.mjs";
 */

/**
 * The TableResult Document.
 * Defines the DataSchema and common behaviors for a TableResult which are shared between both client and server.
 * @extends {Document<TableResultData>}
 * @mixes TableResultData
 * @category Documents
 */
class BaseTableResult extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "TableResult",
    collection: "results",
    label: "DOCUMENT.TableResult",
    labelPlural: "DOCUMENT.TableResults",
    coreTypes: Object.values(TABLE_RESULT_TYPES),
    permissions: {
      create: "OWNER",
      update: this.#canUpdate,
      delete: "OWNER"
    },
    compendiumIndexFields: ["type"],
    schemaVersion: "13.341"
  }, {inplace: false}));

  /** @override */
  static LOCALIZATION_PREFIXES = ["TABLE_RESULT"];

  /* -------------------------------------------- */

  /** @inheritDoc */
  static defineSchema() {
    return {
      _id: new DocumentIdField(),
      type: new DocumentTypeField(this, {initial: TABLE_RESULT_TYPES.TEXT}),
      name: new StringField({required: true, nullable: false, blank: true, initial: "", textSearch: true}),
      img: new FilePathField({categories: ["IMAGE"]}),
      description: new HTMLField({textSearch: true}),
      documentUuid: new DocumentUUIDField({required: false, nullable: true, initial: undefined}),
      weight: new NumberField({required: true, integer: true, positive: true, nullable: false, initial: 1}),
      range: new ArrayField(new NumberField({integer: true}), {
        min: 2,
        max: 2,
        validate: r => r[1] >= r[0],
        validationError: "must be a length-2 array of ascending integers"
      }),
      drawn: new BooleanField(),
      flags: new DocumentFlagsField(),
      _stats: new DocumentStatsField()
    };
  }

  /* -------------------------------------------- */

  /**
   * Is a user able to update an existing TableResult?
   * @type {DocumentPermissionTest}
   */
  static #canUpdate(user, doc, data) {
    if ( user.isGM ) return true;                               // GM users can do anything
    if ( !doc.testUserPermission(user, "OWNER") ) return false;
    const wasDrawn = new Set(["drawn", "_id"]);                 // Users can update the drawn status of a result
    if ( new Set(Object.keys(data)).equals(wasDrawn) ) return true;
    return doc.parent.testUserPermission(user, "OWNER");        // Otherwise, go by parent document permission
  }

  /* ---------------------------------------- */
  /*  Deprecations and Compatibility          */
  /* ---------------------------------------- */

  /**
   * @deprecated since V13
   * @ignore
   */
  get text() {
    const cls = this.constructor.name;
    const message = `${cls}#text is deprecated. Use ${cls}#name or ${cls}#description instead.`;
    foundry.utils.logCompatibilityWarning(message, {since: 13, until: 15, once: true});
    return this.type === "text" ? this.description : this.name;
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since V13
   * @ignore
   */
  get documentId() {
    const cls = this.constructor.name;
    const message = `${cls}#documentId is deprecated. Consult ${cls}#uuid instead.`;
    foundry.utils.logCompatibilityWarning(message, {since: 13, until: 15, once: true});
    return parseUuid(this.documentUuid)?.id ?? null;
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since V13
   * @ignore
   */
  get documentCollection() {
    const cls = this.constructor.name;
    const message = `${cls}#documentCollection is deprecated. Consult ${cls}#uuid instead.`;
    foundry.utils.logCompatibilityWarning(message, {since: 13, until: 15, once: true});
    const parsedUuid = parseUuid(this.documentUuid);
    const collection = parsedUuid?.collection;
    if ( collection instanceof foundry.documents.collections.CompendiumCollection ) return collection.metadata.id;
    return parsedUuid?.type ?? "";
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static migrateData(data) {
    const TYPES = CONST.TABLE_RESULT_TYPES;

    /**
     * V12 migration of type from number to string.
     * @deprecated since v12
     */
    if ( typeof data.type === "number" ) {
      data.type = data.type === 0 ? TYPES.TEXT : TYPES.DOCUMENT;
    }

    // Since V13, the "compendium" type has been dropped.
    if ( data.type === "pack" ) data.type = TYPES.DOCUMENT;
    BaseTableResult.#migrateDocumentUuid(data);

    return super.migrateData(data);
  }

  /* -------------------------------------------- */

  /**
   * The documentId and documentCollection fields have been replaced with a single uuid field.
   * @param {object} data
   * @deprecated since V13
   */
  static #migrateDocumentUuid(data) {
    const hasRealProperty = p => Object.hasOwn(data, p) && !Object.getOwnPropertyDescriptor(data, p).get;
    if ( ["documentId", "documentCollection"].every(p => hasRealProperty(p)) ) {
      if ( data.type === CONST.TABLE_RESULT_TYPES.DOCUMENT ) {
        data.name = data.text;
        data.text = "";
        const [documentName, pack] = CONST.COMPENDIUM_DOCUMENT_TYPES.includes(data.documentCollection)
          ? [data.documentCollection, undefined]
          : [null, data.documentCollection];
        data.documentUuid = buildUuid({id: data.documentId, documentName, pack});
      }
      delete data.documentId;
      delete data.documentCollection;
    }
    this._addDataFieldMigration(data, "text", "description");
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static shimData(data, options) {
    if ( Object.isSealed(data) || Object.hasOwn(data, "documentId") ) {
      return super.shimData(data, options);
    }
    BaseTableResult.#shimDocumentUuid(data);
    this._addDataFieldShim(data, "text", "description", {since: 13, until: 15});
    return super.shimData(data, options);
  }

  /* -------------------------------------------- */

  /**
   * Provide accessors for documentId and documentCollection, attempting to preserve a well-formed uuid on set.
   * @param {object} data
   */
  static #shimDocumentUuid(data) {
    const obj = "TableResultData";
    Object.defineProperties(data, {
      documentId: {
        get() {
          const message = `${obj}#documentId is deprecated. Consult ${obj}#documentUuid instead.`;
          foundry.utils.logCompatibilityWarning(message, {since: 13, until: 15, once: true});
          return parseUuid(data.documentUuid)?.id ?? null;
        },
        set(id) {
          const message = `${obj}#documentId is deprecated. Update ${obj}#documentUuid instead.`;
          foundry.utils.logCompatibilityWarning(message, {since: 13, until: 15, once: true});
          const [documentName, pack] = CONST.WORLD_DOCUMENT_TYPES.includes(data.documentCollection)
            ? [data.documentCollection, undefined]
            : [null, data.documentCollection];
          data.documentUuid = buildUuid({id, documentName, pack});
        }
      },
      documentCollection: {
        get() {
          const message = `${obj}#documentCollection is deprecated. Consult ${obj}#documentUuid instead.`;
          foundry.utils.logCompatibilityWarning(message, {since: 13, until: 15});
          const parsedUuid = parseUuid(data.documentUuid);
          return parsedUuid?.collection?.metadata?.id ?? parsedUuid.type ?? "";
        },
        set(value) {
          const message = `${obj}#documentCollection is deprecated. Update ${obj}#documentUuid instead.`;
          foundry.utils.logCompatibilityWarning(message, {since: 13, until: 15, once: true});
          data.documentUuid = CONST.WORLD_DOCUMENT_TYPES.includes(value)
            ? buildUuid({id: data.documentId, documentName: value})
            : buildUuid({id: data.documentId, pack: value});
        }
      }
    });
  }
}

/**
 * @import {TileData} from "./_types.mjs";
 */

/**
 * The Tile Document.
 * Defines the DataSchema and common behaviors for a Tile which are shared between both client and server.
 * @extends {Document<TileData>}
 * @mixes TileData
 * @category Documents
 */
class BaseTile extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "Tile",
    collection: "tiles",
    label: "DOCUMENT.Tile",
    labelPlural: "DOCUMENT.Tiles",
    schemaVersion: "13.341"
  }, {inplace: false}));

  /** @inheritdoc */
  static defineSchema() {
    const occlusionModes = Object.entries(OCCLUSION_MODES).reduce((modes, entry) => {
      modes[entry[1]] = `TILE.OcclusionMode${entry[0].titleCase()}`;
      return modes;
    }, {});
    return {
      _id: new DocumentIdField(),
      texture: new TextureData({}, {initial: {anchorX: 0.5, anchorY: 0.5, alphaThreshold: 0.75}}),
      width: new NumberField({required: true, min: 0, integer: true, nullable: false}),
      height: new NumberField({required: true, min: 0, integer: true, nullable: false}),
      x: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
      y: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
      elevation: new NumberField({required: true, nullable: false, initial: 0}),
      sort: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
      rotation: new AngleField(),
      alpha: new AlphaField(),
      hidden: new BooleanField(),
      locked: new BooleanField(),
      restrictions: new SchemaField({
        light: new BooleanField(),
        weather: new BooleanField()
      }),
      occlusion: new SchemaField({
        mode: new NumberField({choices: occlusionModes, initial: OCCLUSION_MODES.NONE,
          validationError: "must be a value in CONST.TILE_OCCLUSION_MODES"}),
        alpha: new AlphaField({initial: 0})
      }),
      video: new SchemaField({
        loop: new BooleanField({initial: true}),
        autoplay: new BooleanField({initial: true}),
        volume: new AlphaField({initial: 0, step: 0.01})
      }),
      flags: new DocumentFlagsField()
    };
  }

  /** @override */
  static LOCALIZATION_PREFIXES = ["DOCUMENT", "TILE"];

  /* ---------------------------------------- */
  /*  Deprecations and Compatibility          */
  /* ---------------------------------------- */

  /** @inheritdoc */
  static migrateData(data) {
    /**
     * V12 migration to elevation and sort
     * @deprecated since v12
     */
    this._addDataFieldMigration(data, "z", "sort");

    /**
     * V12 migration from roof to restrictions.light and restrictions.weather
     * @deprecated since v12
     */
    if ( foundry.utils.hasProperty(data, "roof") ) {
      const value = foundry.utils.getProperty(data, "roof");
      if ( !foundry.utils.hasProperty(data, "restrictions.light") ) foundry.utils.setProperty(data, "restrictions.light", value);
      if ( !foundry.utils.hasProperty(data, "restrictions.weather") ) foundry.utils.setProperty(data, "restrictions.weather", value);
      delete data.roof;
    }

    return super.migrateData(data);
  }

  /* ---------------------------------------- */

  /** @inheritdoc */
  static shimData(data, options) {
    this._addDataFieldShim(data, "z", "sort", {since: 12, until: 14});
    return super.shimData(data, options);
  }

  /* ---------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  set roof(enabled) {
    this.constructor._logDataFieldMigration("roof", "restrictions.{light|weather}", {since: 12, until: 14});
    this.restrictions.light = enabled;
    this.restrictions.weather = enabled;
  }

  /**
   * @deprecated since v12
   * @ignore
   */
  get roof() {
    this.constructor._logDataFieldMigration("roof", "restrictions.{light|weather}", {since: 12, until: 14});
    return this.restrictions.light && this.restrictions.weather;
  }

  /* ---------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  get z() {
    this.constructor._logDataFieldMigration("z", "sort", {since: 12, until: 14});
    return this.sort;
  }

  /* ---------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  get overhead() {
    foundry.utils.logCompatibilityWarning(`${this.constructor.name}#overhead is deprecated.`, {since: 12, until: 14});
    return this.elevation >= this.parent?.foregroundElevation;
  }
}

/**
 * @import {Point, ElevatedPoint, DeepReadonly} from "../_types.mjs";
 * @import {TokenShapeType} from "../constants.mjs";
 * @import {TokenHexagonalOffsetsData, TokenHexagonalShapeData, TokenDimensions, TokenPosition} from "./_types.mjs";
 * @import {GridOffset2D, GridOffset3D} from "../grid/_types.mjs";
 * @import {TokenData} from "./_types.mjs";
 * @import {SquareGrid} from "../grid/_module.mjs";
 * @import {DataModelUpdateOptions, DocumentPermissionTest} from "@common/abstract/_types.mjs";
 */

/**
 * The Token Document.
 * Defines the DataSchema and common behaviors for a Token which are shared between both client and server.
 * @extends {Document<TokenData>}
 * @mixes TokenData
 * @category Documents
 */
class BaseToken extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritdoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "Token",
    collection: "tokens",
    label: "DOCUMENT.Token",
    labelPlural: "DOCUMENT.Tokens",
    isEmbedded: true,
    embedded: {
      ActorDelta: "delta"
    },
    permissions: {
      create: "TOKEN_CREATE",
      update: this.#canUpdate,
      delete: "TOKEN_DELETE"
    },
    schemaVersion: "13.341"
  }, {inplace: false}));

  /* -------------------------------------------- */

  /** @inheritdoc */
  static defineSchema() {
    const documents = foundry.documents;
    return {
      _id: new DocumentIdField(),
      name: new StringField({required: true, blank: true, textSearch: true}),
      displayName: new NumberField({required: true, initial: TOKEN_DISPLAY_MODES.NONE,
        choices: Object.values(TOKEN_DISPLAY_MODES),
        validationError: "must be a value in CONST.TOKEN_DISPLAY_MODES"
      }),
      actorId: new ForeignDocumentField(documents.BaseActor, {idOnly: true}),
      actorLink: new BooleanField(),
      delta: new ActorDeltaField(documents.BaseActorDelta),
      width: new NumberField({required: true, nullable: false, positive: true, initial: 1}),
      height: new NumberField({required: true, nullable: false, positive: true, initial: 1}),
      texture: new TextureData({}, {initial: {src: () => this.DEFAULT_ICON, anchorX: 0.5, anchorY: 0.5, fit: "contain",
        alphaThreshold: 0.75}, wildcard: true}),
      shape: new NumberField({initial: TOKEN_SHAPES.RECTANGLE_1,
        choices: Object.values(TOKEN_SHAPES)}),
      x: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
      y: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
      elevation: new NumberField({required: true, nullable: false, initial: 0}),
      sort: new NumberField({required: true, integer: true, nullable: false, initial: 0}),
      locked: new BooleanField(),
      lockRotation: new BooleanField(),
      rotation: new AngleField(),
      alpha: new AlphaField(),
      hidden: new BooleanField(),
      disposition: new NumberField({required: true, choices: Object.values(TOKEN_DISPOSITIONS),
        initial: TOKEN_DISPOSITIONS.HOSTILE,
        validationError: "must be a value in CONST.TOKEN_DISPOSITIONS"
      }),
      displayBars: new NumberField({required: true, choices: Object.values(TOKEN_DISPLAY_MODES),
        initial: TOKEN_DISPLAY_MODES.NONE,
        validationError: "must be a value in CONST.TOKEN_DISPLAY_MODES"
      }),
      bar1: new SchemaField({
        attribute: new StringField({required: true, nullable: true, blank: false,
          initial: () => game?.system.primaryTokenAttribute || null})
      }),
      bar2: new SchemaField({
        attribute: new StringField({required: true, nullable: true, blank: false,
          initial: () => game?.system.secondaryTokenAttribute || null})
      }),
      light: new EmbeddedDataField(LightData),
      sight: new SchemaField({
        enabled: new BooleanField({initial: data => Number(data?.sight?.range) > 0}),
        range: new NumberField({required: true, nullable: true, min: 0, step: 0.01, initial: 0}),
        angle: new AngleField({initial: 360, normalize: false}),
        visionMode: new StringField({required: true, blank: false, initial: "basic"}),
        color: new ColorField(),
        attenuation: new AlphaField({initial: 0.1}),
        brightness: new NumberField({required: true, nullable: false, initial: 0, min: -1, max: 1}),
        saturation: new NumberField({required: true, nullable: false, initial: 0, min: -1, max: 1}),
        contrast: new NumberField({required: true, nullable: false, initial: 0, min: -1, max: 1})
      }),
      detectionModes: new ArrayField(new SchemaField({
        id: new StringField(),
        enabled: new BooleanField({initial: true}),
        range: new NumberField({required: true, min: 0, step: 0.01})
      }), {
        validate: BaseToken.#validateDetectionModes
      }),
      occludable: new SchemaField({
        radius: new NumberField({required: true, nullable: false, min: 0, step: 0.01, initial: 0})
      }),
      ring: new SchemaField({
        enabled: new BooleanField(),
        colors: new SchemaField({
          ring: new ColorField(),
          background: new ColorField()
        }),
        effects: new NumberField({required: true, nullable: false, integer: true, initial: 1, min: 0,
          max: 8388607}),
        subject: new SchemaField({
          scale: new NumberField({required: true, nullable: false, initial: 1, min: 0.5}),
          texture: new FilePathField({categories: ["IMAGE"]})
        })
      }),
      turnMarker: new SchemaField({
        mode: new NumberField({required: true, choices: Object.values(TOKEN_TURN_MARKER_MODES),
          initial: TOKEN_TURN_MARKER_MODES.DEFAULT,
          validationError: "must be a value in CONST.TOKEN_TURN_MARKER_MODES"
        }),
        animation: new StringField({required: true, blank: false, nullable: true}),
        src: new FilePathField({categories: ["IMAGE", "VIDEO"]}),
        disposition: new BooleanField()
      }),
      movementAction: new StringField({required: true, blank: false, nullable: true, initial: null,
        choices: CONFIG.Token.movement?.actions}),
      /** @internal */
      _movementHistory: new ArrayField(new SchemaField({
        x: new NumberField({required: true, nullable: false, integer: true, initial: undefined}),
        y: new NumberField({required: true, nullable: false, integer: true, initial: undefined}),
        elevation: new NumberField({required: true, nullable: false, initial: undefined}),
        width: new NumberField({required: true, nullable: false, positive: true, initial: undefined}),
        height: new NumberField({required: true, nullable: false, positive: true, initial: undefined}),
        shape: new NumberField({required: true, initial: undefined, choices: Object.values(TOKEN_SHAPES)}),
        action: new StringField({required: true, blank: false, initial: undefined}),
        terrain: CONFIG.Token.movement?.TerrainData ? new EmbeddedDataField(CONFIG.Token.movement.TerrainData,
          {nullable: true, initial: undefined}) : new ObjectField({nullable: true, initial: undefined}),
        snapped: new BooleanField({initial: undefined}),
        explicit: new BooleanField({initial: undefined}),
        checkpoint: new BooleanField({initial: undefined}),
        intermediate: new BooleanField({initial: undefined}),
        userId: new ForeignDocumentField(documents.BaseUser, {idOnly: true, required: true, initial: undefined}),
        movementId: new StringField({required: true, blank: false, initial: undefined,
          validate: value => {
            if ( !foundry.data.validators.isValidId(value) ) throw new Error("must be a valid 16-character alphanumeric ID");
          }
        }),
        cost: new NumberField({required: true, nullable: true, min: 0, initial: undefined})
      })),
      /** @internal */
      _regions: new ArrayField(new ForeignDocumentField(documents.BaseRegion, {idOnly: true})),
      flags: new DocumentFlagsField()
    };
  }

  /* -------------------------------------------- */

  /** @override */
  static LOCALIZATION_PREFIXES = ["DOCUMENT", "TOKEN"];

  /**
   * The fields of the data model for which changes count as a movement action.
   * @type {Readonly<["x", "y", "elevation", "width", "height", "shape"]>}
   * @readonly
   */
  static MOVEMENT_FIELDS = Object.freeze(["x", "y", "elevation", "width", "height", "shape"]);

  /* -------------------------------------------- */

  /**
   * Are the given positions equal?
   * @param {TokenPosition} position1
   * @param {TokenPosition} position2
   * @returns {boolean}
   */
  static arePositionsEqual(position1, position2) {
    return (position1 === position2) || this.MOVEMENT_FIELDS.every(k => position1[k] === position2[k]);
  }

  /* -------------------------------------------- */

  /**
   * Validate the structure of the detection modes array
   * @param {object[]} modes    Configured detection modes
   * @throws                    An error if the array is invalid
   */
  static #validateDetectionModes(modes) {
    const seen = new Set();
    for ( const mode of modes ) {
      if ( mode.id === "" ) continue;
      if ( seen.has(mode.id) ) {
        throw new Error(`may not have more than one configured detection mode of type "${mode.id}"`);
      }
      seen.add(mode.id);
    }
  }

  /* -------------------------------------------- */

  /**
   * The default icon used for newly created Token documents
   * @type {string}
   */
  static DEFAULT_ICON = DEFAULT_TOKEN;

  /* -------------------------------------------- */

  /**
   * Is a User able to update an existing Token? One can update a Token embedded in a World Scene if they own the
   * corresponding Actor.
   * @type {DocumentPermissionTest}
   */
  static #canUpdate(user, doc, data) {
    if ( user.isGM ) return true;                     // GM users can do anything
    if ( doc.inCompendium ) return doc.testUserPermission(user, "OWNER");
    if ( doc.actor ) {                                // You can update Tokens for Actors you control
      return doc.actor.testUserPermission(user, "OWNER");
    }
    return !doc.actorId;                              // Actor-less Tokens can be updated by anyone
  }

  /* -------------------------------------------- */
  /*  Data Management                             */
  /* -------------------------------------------- */

  /**
   * Prepare changes to a descendent delta collection.
   * @param {object} changes                  Candidate source changes.
   * @param {DataModelUpdateOptions} options  Options which determine how the new data is merged.
   * @internal
   */
  _prepareDeltaUpdate(changes={}, options={}) {
    if ( changes.delta && this.delta ) this.delta._prepareDeltaUpdate(changes.delta, options);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  updateSource(changes={}, options={}) {
    this._prepareDeltaUpdate(changes, options);
    return super.updateSource(changes, options);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  clone(data={}, context={}) {
    const clone = super.clone(data, context);
    if ( (clone instanceof Promise) || clone.actorLink ) return clone;
    // Extra care needs to be taken when temporarily cloning an unlinked TokenDocument.
    // Preparation of the clone's synthetic Actor using the embedded ActorDelta can easily enter an infinite loop.
    // In this case we need to eagerly evaluate the clone ActorDelta instance so it is available immediately.
    clone.delta; // Resolve lazy getter
    return clone;
  }

  /* -------------------------------------------- */
  /*  Token Methods                               */
  /* -------------------------------------------- */

  /**
   * Get the snapped position of the Token.
   * @param {Partial<ElevatedPoint & TokenDimensions>} [data] The position and dimensions
   * @returns {ElevatedPoint}                                 The snapped position
   */
  getSnappedPosition(data={}) {
    const grid = this.parent?.grid ?? BaseScene.defaultGrid;
    const x = data.x ?? this.x;
    const y = data.y ?? this.y;
    let elevation = data.elevation ?? this.elevation;
    const unsnapped = {x, y, elevation};

    // Gridless grid
    if ( grid.isGridless ) return unsnapped;

    // Get position and elevation
    elevation = Math.round(elevation / grid.distance) * grid.distance;

    // Round width and height to nearest multiple of 0.5
    const width = Math.round((data.width ?? this.width) * 2) / 2;
    const height = Math.round((data.height ?? this.height) * 2) / 2;
    const shape = data.shape ?? this.shape;

    // Square grid
    let snapped;
    if ( grid.isSquare ) snapped = BaseToken.#getSnappedPositionInSquareGrid(grid, unsnapped, width, height);

    // Hexagonal grid
    else snapped = BaseToken.#getSnappedPositionInHexagonalGrid(grid, unsnapped, width, height, shape);
    return {x: snapped.x, y: snapped.y, elevation};
  }

  /* -------------------------------------------- */

  /**
   * Get the snapped position on a square grid.
   * @param {SquareGrid} grid     The square grid
   * @param {Point} position      The position that is snapped or grid offset
   * @param {number} width        The width in grid spaces (positive)
   * @param {number} height       The height in grid spaces (positive)
   * @returns {Point}             The snapped position
   */
  static #getSnappedPositionInSquareGrid(grid, position, width, height) {

    // Small tokens snap to any vertex of the subgrid with resolution 4
    // where the token is fully contained within the grid space
    const isSmall = ((width === 0.5) && (height <= 1)) || ((width <= 1) && (height === 0.5));
    if ( isSmall ) {
      let x = position.x / grid.size;
      let y = position.y / grid.size;
      if ( width === 1 ) x = Math.round(x);
      else {
        x = Math.floor(x * 8);
        const k = ((x % 8) + 8) % 8;
        if ( k >= 6 ) x = Math.ceil(x / 8);
        else if ( k === 5 ) x = Math.floor(x / 8) + 0.5;
        else x = Math.round(x / 2) / 4;
      }
      if ( height === 1 ) y = Math.round(y);
      else {
        y = Math.floor(y * 8);
        const k = ((y % 8) + 8) % 8;
        if ( k >= 6 ) y = Math.ceil(y / 8);
        else if ( k === 5 ) y = Math.floor(y / 8) + 0.5;
        else y = Math.round(y / 2) / 4;
      }
      x *= grid.size;
      y *= grid.size;
      return {x, y};
    }

    const M = GRID_SNAPPING_MODES;
    const modeX = Number.isInteger(width) ? M.VERTEX : M.VERTEX | M.EDGE_MIDPOINT | M.CENTER;
    const modeY = Number.isInteger(height) ? M.VERTEX : M.VERTEX | M.EDGE_MIDPOINT | M.CENTER;
    if ( modeX === modeY ) return grid.getSnappedPoint(position, {mode: modeX});
    return {
      x: grid.getSnappedPoint(position, {mode: modeX}).x,
      y: grid.getSnappedPoint(position, {mode: modeY}).y
    };
  }

  /* -------------------------------------------- */

  /**
   * Get the snapped position on a hexagonal grid.
   * @param {SquareGrid} grid       The hexagonal grid
   * @param {Point} position        The position that is snapped or grid offset
   * @param {number} width          The width in grid spaces (positive)
   * @param {number} height         The height in grid spaces (positive)
   * @param {TokenShapeType} shape  The shape (one of {@link CONST.TOKEN_SHAPES})
   * @returns {Point}               The snapped position
   */
  static #getSnappedPositionInHexagonalGrid(grid, position, width, height, shape) {

    // Hexagonal shape
    const hexagonalShape = BaseToken.#getHexagonalShape(width, height, shape, grid.columns);
    if ( hexagonalShape ) {
      const offsetX = hexagonalShape.anchor.x * grid.sizeX;
      const offsetY = hexagonalShape.anchor.y * grid.sizeY;
      position = grid.getCenterPoint({x: position.x + offsetX, y: position.y + offsetY});
      position.x -= offsetX;
      position.y -= offsetY;
      return position;
    }

    // Rectagular shape
    const M = GRID_SNAPPING_MODES;
    return grid.getSnappedPoint(position, {mode: M.CENTER | M.VERTEX | M.CORNER | M.SIDE_MIDPOINT});
  }

  /* -------------------------------------------- */

  /**
   * Get the top-left grid offset of the Token.
   * @param {Partial<ElevatedPoint & TokenDimensions>} [data]      The position and dimensions
   * @returns {GridOffset3D}                                       The top-left grid offset
   * @internal
   */
  _positionToGridOffset(data={}) {
    let x = Math.round(data.x ?? this.x);
    let y = Math.round(data.y ?? this.y);
    const elevation = data.elevation ?? this.elevation;

    // Gridless grid
    const grid = this.parent?.grid ?? BaseScene.defaultGrid;
    if ( grid.isGridless ) return grid.getOffset({x, y, elevation});

    // Round width and height to nearest multiple of 0.5
    const width = Math.round((data.width ?? this.width) * 2) / 2;
    const height = Math.round((data.height ?? this.height) * 2) / 2;

    // Square grid
    if ( grid.isSquare ) {
      x += (grid.size * (Number.isInteger(width) ? 0.5 : 0.25));
      y += (grid.size * (Number.isInteger(height) ? 0.5 : 0.25));
    }

    // Hexagonal grid
    else {
      const {anchor} = BaseToken._getHexagonalOffsets(width, height, data.shape ?? this.shape, grid.columns);
      x += (grid.sizeX * anchor.x);
      y += (grid.sizeY * anchor.y);
    }

    return grid.getOffset({x, y, elevation});
  }

  /* -------------------------------------------- */

  /**
   * Get the position of the Token from the top-left grid offset.
   * @param {GridOffset3D } offset                 The top-left grid offset
   * @param {Partial<TokenDimensions>} [data]      The dimensions that override the current dimensions
   * @returns {ElevatedPoint}                      The snapped position
   * @internal
   */
  _gridOffsetToPosition(offset, data={}) {
    const grid = this.parent?.grid ?? BaseScene.defaultGrid;

    // Gridless grid
    if ( grid.isGridless ) return {x: offset.j, y: offset.i, elevation: offset.k / grid.size * grid.distance};

    // Round width and height to nearest multiple of 0.5
    const width = Math.round((data.width ?? this.width) * 2) / 2;
    const height = Math.round((data.height ?? this.height) * 2) / 2;

    let x;
    let y;

    // Square grid
    if ( grid.isSquare ) {
      x = offset.j;
      y = offset.i;
      const isSmall = ((width === 0.5) && (height <= 1)) || ((width <= 1) && (height === 0.5));
      if ( isSmall ) {
        if ( width === 0.5 ) x += 0.25;
        if ( height === 0.5 ) y += 0.25;
      }
      x *= grid.size;
      y *= grid.size;
    }

    // Hexagonal grid
    else {
      const {anchor} = BaseToken._getHexagonalOffsets(width, height, data.shape ?? this.shape, grid.columns);
      const position = grid.getCenterPoint(offset);
      x = position.x - (anchor.x * grid.sizeX);
      y = position.y - (anchor.y * grid.sizeY);
    }

    return {x, y, elevation: offset.k * grid.distance};
  }

  /* -------------------------------------------- */

  /**
   * Get the width and height of the Token in pixels.
   * @param {Partial<{width: number; height: number}>} [data] The width and/or height in grid units (must be positive)
   * @returns {{width: number; height: number}} The width and height in pixels
   */
  getSize(data={}) {
    let width = data.width ?? this.width;
    let height = data.height ?? this.height;

    const grid = this.parent?.grid ?? BaseScene.defaultGrid;
    if ( grid.isHexagonal ) {
      if ( grid.columns ) width = (0.75 * Math.floor(width)) + (0.5 * (width % 1)) + 0.25;
      else height = (0.75 * Math.floor(height)) + (0.5 * (height % 1)) + 0.25;
    }

    width *= grid.sizeX;
    height *= grid.sizeY;
    return {width, height};
  }

  /* -------------------------------------------- */

  /**
   * Get the center point of the Token.
   * @param {Partial<ElevatedPoint & TokenDimensions>} [data] The position and dimensions
   * @returns {ElevatedPoint}                                 The center point
   */
  getCenterPoint(data={}) {
    const x = data.x ?? this.x;
    const y = data.y ?? this.y;
    const elevation = data.elevation ?? this.elevation;

    // Hexagonal shape
    const grid = this.parent?.grid ?? BaseScene.defaultGrid;
    if ( grid.isHexagonal ) {
      const width = data.width ?? this.width;
      const height = data.height ?? this.height;
      const hexagonalShape = BaseToken.#getHexagonalShape(width, height, data.shape ?? this.shape, grid.columns);
      if ( hexagonalShape ) {
        const center = hexagonalShape.center;
        return {x: x + (center.x * grid.sizeX), y: y + (center.y * grid.sizeY), elevation};
      }

      // No hexagonal shape for this combination of shape type, width, and height.
      // Fallback to the center of the rectangle.
    }

    // Rectangular shape
    const {width, height} = this.getSize(data);
    return {x: x + (width / 2), y: y + (height / 2), elevation};
  }

  /* -------------------------------------------- */

  /**
   * Get the grid space polygon of the Token.
   * Returns undefined in gridless grids because there are no grid spaces.
   * @param {Partial<TokenDimensions>} [data] The dimensions
   * @returns {Point[]|void}                  The grid space polygon or undefined if gridless
   */
  getGridSpacePolygon(data={}) {
    const grid = this.parent?.grid ?? BaseScene.defaultGrid;

    // Gridless grid
    if ( grid.isGridless ) return;

    // Hexagonal shape
    if ( grid.isHexagonal ) {
      const width = data.width ?? this.width;
      const height = data.height ?? this.height;
      const hexagonalShape = BaseToken.#getHexagonalShape(width, height, data.shape ?? this.shape, grid.columns);
      if ( hexagonalShape ) {
        const points = [];
        for ( let i = 0; i < hexagonalShape.points.length; i += 2 ) {
          points.push({x: hexagonalShape.points[i] * grid.sizeX, y: hexagonalShape.points[i + 1] * grid.sizeY});
        }
        return points;
      }

      // No hexagonal shape for this combination of shape type, width, and height.
      // Fallback to rectangular shape.
    }

    // Rectangular shape
    const {width, height} = this.getSize(data);
    return [{x: 0, y: 0}, {x: width, y: 0}, {x: width, y: height}, {x: 0, y: height}];
  }

  /* -------------------------------------------- */

  /**
   * Get the offsets of grid spaces that are occupied by this Token at the current or given position.
   * The grid spaces the Token occupies are those that are covered by the Token's shape in the snapped position.
   * Returns an empty array in gridless grids.
   * @param {Partial<Point & TokenDimensions>} [data] The position and dimensions
   * @returns {GridOffset2D[]}                        The offsets of occupied grid spaces
   */
  getOccupiedGridSpaceOffsets(data={}) {
    const offsets = [];
    const grid = this.parent?.grid ?? BaseScene.defaultGrid;

    // No grid spaces in gridless grids
    if ( grid.isGridless ) return offsets;

    // Get the top-left grid offset
    const {i: i0, j: j0} = this._positionToGridOffset(data);

    // Round width and height to nearest multiple of 0.5
    const width = Math.round((data.width ?? this.width) * 2) / 2;
    const height = Math.round((data.height ?? this.height) * 2) / 2;

    // Square grid
    if ( grid.isSquare ) {
      const i1 = i0 + Math.ceil(height);
      const j1 = j0 + Math.ceil(width);
      for ( let i = i0; i < i1; i++ ) {
        for ( let j = j0; j < j1; j++ ) {
          offsets.push({i, j});
        }
      }
    }

    // Hexagonal grid
    else {
      const {even: offsetsEven, odd: offsetsOdd} = BaseToken._getHexagonalOffsets(
        width, height, data.shape ?? this.shape, grid.columns);
      const isEven = ((grid.columns ? j0 : i0) % 2 === 0) === grid.even;
      for ( const {i: di, j: dj} of (isEven ? offsetsEven : offsetsOdd) ) {
        offsets.push({i: i0 + di, j: j0 + dj});
      }
    }

    return offsets;
  }

  /* -------------------------------------------- */
  /*  Hexagonal Helpers                           */
  /* -------------------------------------------- */

  /**
   * The cache of hexagonal offsets.
   * @type {Map<string, DeepReadonly<TokenHexagonalOffsetsData>>}
   */
  static #hexagonalOffsets = new Map();

  /* -------------------------------------------- */

  /**
   * Get the hexagonal offsets given the type, width, and height.
   * @param {number} width                                 The width of the Token (positive)
   * @param {number} height                                The height of the Token (positive)
   * @param {TokenShapeType} shape                         The shape (one of {@link CONST.TOKEN_SHAPES})
   * @param {boolean} columns                              Column-based instead of row-based hexagonal grid?
   * @returns {DeepReadonly<TokenHexagonalOffsetsData>}    The hexagonal offsets
   * @internal
   */
  static _getHexagonalOffsets(width, height, shape, columns) {
    // TODO: can we set a max of 2^13 on width and height so that we may use an integer key?
    const key = `${width},${height},${shape}${columns ? "C" : "R"}`;
    let offsets = BaseToken.#hexagonalOffsets.get(key);
    if ( offsets ) return offsets;

    let anchor;
    let data = BaseToken.#getHexagonalShape(width, height, shape, columns);

    // Hexagonal shape
    if ( data ) anchor = data.anchor;

    // Fallback for non-hexagonal shapes
    else {
      if ( columns ) {
        height += 0.5;
        width = Math.round(width);
        if ( width === 1 ) height = Math.floor(height);
        else if ( height === 1 ) height += 0.5;
      } else {
        width += 0.5;
        height = Math.round(height);
        if ( height === 1 ) width = Math.floor(width);
        else if ( width === 1 ) width += 0.5;
      }
      data = BaseToken.#getHexagonalShape(width, height, TOKEN_SHAPES.RECTANGLE_1, columns);
      anchor = {x: data.anchor.x - 0.25, y: data.anchor.y - 0.25};
    }

    // Cache the offsets
    offsets = foundry.utils.deepFreeze({
      even: data.offsets.even,
      odd: data.offsets.odd,
      anchor
    });
    BaseToken.#hexagonalOffsets.set(key, offsets);
    return offsets;
  }

  /* -------------------------------------------- */

  /**
   * The cache of hexagonal shapes.
   * @type {Map<string, DeepReadonly<TokenHexagonalShapeData>>}
   */
  static #hexagonalShapes = new Map();

  /* -------------------------------------------- */

  /**
   * Get the hexagonal shape given the type, width, and height.
   * @param {number} width                                    The width of the Token (positive)
   * @param {number} height                                   The height of the Token (positive)
   * @param {TokenShapeType} shape                            The shape (one of {@link CONST.TOKEN_SHAPES})
   * @param {boolean} columns                                 Column-based instead of row-based hexagonal grid?
   * @returns {DeepReadonly<TokenHexagonalShapeData>|null}    The hexagonal shape or null if there is no shape
   *                                                          for the given combination of arguments
   */
  static #getHexagonalShape(width, height, shape, columns) {
    if ( !Number.isInteger(width * 2) || !Number.isInteger(height * 2) ) return null;

    // TODO: can we set a max of 2^13 on width and height so that we may use an integer key?
    const key = `${width},${height},${shape}${columns ? "C" : "R"}`;
    let data = BaseToken.#hexagonalShapes.get(key);
    if ( data ) return data;

    // Hexagon symmetry
    if ( columns ) {
      const rowData = BaseToken.#getHexagonalShape(height, width, shape, false);
      if ( !rowData ) return null;

      // Transpose the offsets/points of the shape in row orientation
      const offsets = {even: [], odd: []};
      for ( const {i, j} of rowData.offsets.even ) offsets.even.push({i: j, j: i});
      for ( const {i, j} of rowData.offsets.odd ) offsets.odd.push({i: j, j: i});
      offsets.even.sort(({i: i0, j: j0}, {i: i1, j: j1}) => (j0 - j1) || (i0 - i1));
      offsets.odd.sort(({i: i0, j: j0}, {i: i1, j: j1}) => (j0 - j1) || (i0 - i1));
      const points = [];
      for ( let i = rowData.points.length; i > 0; i -= 2 ) {
        points.push(rowData.points[i - 1], rowData.points[i - 2]);
      }
      data = {
        offsets,
        points,
        center: {x: rowData.center.y, y: rowData.center.x},
        anchor: {x: rowData.anchor.y, y: rowData.anchor.x}
      };
    }

    // Small hexagon
    else if ( (width === 0.5) && (height === 0.5) ) {
      data = {
        offsets: {even: [{i: 0, j: 0}], odd: [{i: 0, j: 0}]},
        points: [0.25, 0.0, 0.5, 0.125, 0.5, 0.375, 0.25, 0.5, 0.0, 0.375, 0.0, 0.125],
        center: {x: 0.25, y: 0.25},
        anchor: {x: 0.25, y: 0.25}
      };
    }

    // Normal hexagon
    else if ( (width === 1) && (height === 1) ) {
      data = {
        offsets: {even: [{i: 0, j: 0}], odd: [{i: 0, j: 0}]},
        points: [0.5, 0.0, 1.0, 0.25, 1, 0.75, 0.5, 1.0, 0.0, 0.75, 0.0, 0.25],
        center: {x: 0.5, y: 0.5},
        anchor: {x: 0.5, y: 0.5}
      };
    }

    // Hexagonal ellipse or trapezoid
    else if ( shape <= TOKEN_SHAPES.TRAPEZOID_2 ) {
      data = BaseToken.#createHexagonalEllipseOrTrapezoid(width, height, shape);
    }

    // Hexagonal rectangle
    else if ( shape <= TOKEN_SHAPES.RECTANGLE_2 ) {
      data = BaseToken.#createHexagonalRectangle(width, height, shape);
    }

    // Cache the shape
    if ( data ) {
      foundry.utils.deepFreeze(data);
      BaseToken.#hexagonalShapes.set(key, data);
    }
    return data;
  }

  /* -------------------------------------------- */

  /**
   * Create the row-based hexagonal ellipse/trapezoid given the type, width, and height.
   * @param {number} width                   The width of the Token (positive)
   * @param {number} height                  The height of the Token (positive)
   * @param {number} shape                   The shape type (must be ELLIPSE_1, ELLIPSE_1, TRAPEZOID_1, or TRAPEZOID_2)
   * @returns {TokenHexagonalShapeData|null} The hexagonal shape or null if there is no shape for the given combination
   *                                         of arguments
   */
  static #createHexagonalEllipseOrTrapezoid(width, height, shape) {
    if ( !Number.isInteger(width) || !Number.isInteger(height) ) return null;
    const points = [];
    let top;
    let bottom;
    switch ( shape ) {
      case TOKEN_SHAPES.ELLIPSE_1:
        if ( height >= 2 * width ) return null;
        top = Math.floor(height / 2);
        bottom = Math.floor((height - 1) / 2);
        break;
      case TOKEN_SHAPES.ELLIPSE_2:
        if ( height >= 2 * width ) return null;
        top = Math.floor((height - 1) / 2);
        bottom = Math.floor(height / 2);
        break;
      case TOKEN_SHAPES.TRAPEZOID_1:
        if ( height > width ) return null;
        top = height - 1;
        bottom = 0;
        break;
      case TOKEN_SHAPES.TRAPEZOID_2:
        if ( height > width ) return null;
        top = 0;
        bottom = height - 1;
        break;
    }
    const offsets = {even: [], odd: []};
    for ( let i = bottom; i > 0; i-- ) {
      for ( let j = 0; j < width - i; j++ ) {
        offsets.even.push({i: bottom - i, j: j + (((bottom & 1) + i + 1) >> 1)});
        offsets.odd.push({i: bottom - i, j: j + (((bottom & 1) + i) >> 1)});
      }
    }
    for ( let i = 0; i <= top; i++ ) {
      for ( let j = 0; j < width - i; j++ ) {
        offsets.even.push({i: bottom + i, j: j + (((bottom & 1) + i + 1) >> 1)});
        offsets.odd.push({i: bottom + i, j: j + (((bottom & 1) + i) >> 1)});
      }
    }
    let x = 0.5 * bottom;
    let y = 0.25;
    for ( let k = width - bottom; k--; ) {
      points.push(x, y);
      x += 0.5;
      y -= 0.25;
      points.push(x, y);
      x += 0.5;
      y += 0.25;
    }
    points.push(x, y);
    for ( let k = bottom; k--; ) {
      y += 0.5;
      points.push(x, y);
      x += 0.5;
      y += 0.25;
      points.push(x, y);
    }
    y += 0.5;
    for ( let k = top; k--; ) {
      points.push(x, y);
      x -= 0.5;
      y += 0.25;
      points.push(x, y);
      y += 0.5;
    }
    for ( let k = width - top; k--; ) {
      points.push(x, y);
      x -= 0.5;
      y += 0.25;
      points.push(x, y);
      x -= 0.5;
      y -= 0.25;
    }
    points.push(x, y);
    for ( let k = top; k--; ) {
      y -= 0.5;
      points.push(x, y);
      x -= 0.5;
      y -= 0.25;
      points.push(x, y);
    }
    y -= 0.5;
    for ( let k = bottom; k--; ) {
      points.push(x, y);
      x += 0.5;
      y -= 0.25;
      points.push(x, y);
      y -= 0.5;
    }
    return {
      offsets,
      points,
      // We use the centroid of the polygon for ellipse and trapzoid shapes
      center: foundry.utils.polygonCentroid(points),
      anchor: bottom % 2 ? {x: 0.0, y: 0.5} : {x: 0.5, y: 0.5}
    };
  }

  /* -------------------------------------------- */

  /**
   * Create the row-based hexagonal rectangle given the type, width, and height.
   * @param {number} width                      The width of the Token (positive)
   * @param {number} height                     The height of the Token (positive)
   * @param {TokenShapeType} shape              The shape type (must be RECTANGLE_1 or RECTANGLE_2)
   * @returns {TokenHexagonalShapeData|null}    The hexagonal shape or null if there is no shape
   *                                            for the given combination of arguments
   */
  static #createHexagonalRectangle(width, height, shape) {
    if ( (width < 1) || !Number.isInteger(height) ) return null;
    if ( (width === 1) && (height > 1) ) return null;
    if ( !Number.isInteger(width) && (height === 1) ) return null;
    const even = (shape === TOKEN_SHAPES.RECTANGLE_1) || (height === 1);
    const offsets = {even: [], odd: []};
    for ( let i = 0; i < height; i++) {
      const j0 = even ? 0 : (i + 1) & 1;
      const j1 = ((width + ((i & 1) * 0.5)) | 0) - (even ? (i & 1) : 0);
      for ( let j = j0; j < j1; j++ ) {
        offsets.even.push({i, j: j + (i & 1)});
        offsets.odd.push({i, j});
      }
    }
    let x = even ? 0.0 : 0.5;
    let y = 0.25;
    const points = [x, y];
    while ( x + 1 <= width ) {
      x += 0.5;
      y -= 0.25;
      points.push(x, y);
      x += 0.5;
      y += 0.25;
      points.push(x, y);
    }
    if ( x !== width ) {
      y += 0.5;
      points.push(x, y);
      x += 0.5;
      y += 0.25;
      points.push(x, y);
    }
    while ( y + 1.5 <= 0.75 * height ) {
      y += 0.5;
      points.push(x, y);
      x -= 0.5;
      y += 0.25;
      points.push(x, y);
      y += 0.5;
      points.push(x, y);
      x += 0.5;
      y += 0.25;
      points.push(x, y);
    }
    if ( y + 0.75 < 0.75 * height ) {
      y += 0.5;
      points.push(x, y);
      x -= 0.5;
      y += 0.25;
      points.push(x, y);
    }
    y += 0.5;
    points.push(x, y);
    while ( x - 1 >= 0 ) {
      x -= 0.5;
      y += 0.25;
      points.push(x, y);
      x -= 0.5;
      y -= 0.25;
      points.push(x, y);
    }
    if ( x !== 0 ) {
      y -= 0.5;
      points.push(x, y);
      x -= 0.5;
      y -= 0.25;
      points.push(x, y);
    }
    while ( y - 1.5 > 0 ) {
      y -= 0.5;
      points.push(x, y);
      x += 0.5;
      y -= 0.25;
      points.push(x, y);
      y -= 0.5;
      points.push(x, y);
      x -= 0.5;
      y -= 0.25;
      points.push(x, y);
    }
    if ( y - 0.75 > 0 ) {
      y -= 0.5;
      points.push(x, y);
      x += 0.5;
      y -= 0.25;
      points.push(x, y);
    }
    return {
      offsets,
      points,
      // We use center of the rectangle (and not the centroid of the polygon) for the rectangle shapes
      center: {
        x: width / 2,
        y: ((0.75 * Math.floor(height)) + (0.5 * (height % 1)) + 0.25) / 2
      },
      anchor: even ? {x: 0.5, y: 0.5} : {x: 0.0, y: 0.5}
    };
  }

  /* -------------------------------------------- */
  /*  Document Methods                            */
  /* -------------------------------------------- */

  /** @inheritDoc */
  getUserLevel(user) {
    if ( this.actor ) return this.actor.getUserLevel(user);
    return super.getUserLevel(user);
  }

  /* -------------------------------------------- */

  /** @inheritdoc */
  toObject(source=true) {
    const obj = super.toObject(source);
    obj.delta = obj.actorLink ? null : this.delta.toObject(source);
    return obj;
  }

  /* -------------------------------------------- */
  /*  Deprecations and Compatibility              */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static migrateData(data) {
    // Remember that any migrations defined here may also be required for the PrototypeToken model.
    /**
     * Migration of hexagonalShape to shape.
     * @deprecated since v13
     */
    if ( ("hexagonalShape" in data) && !("shape" in data) ) {
      data.shape = data.hexagonalShape;
      delete data.hexagonalShape;
    }
    return super.migrateData(data);
  }

  /* ----------------------------------------- */

  /** @inheritdoc */
  static shimData(data, options) {
    // Remember that any shims defined here may also be required for the PrototypeToken model.
    this._addDataFieldShim(data, "effects", undefined, {value: [], since: 12, until: 14,
      warning: "TokenDocument#effects is deprecated in favor of using ActiveEffect documents on the associated Actor"});
    this._addDataFieldShim(data, "overlayEffect", undefined, {value: "", since: 12, until: 14,
      warning: "TokenDocument#overlayEffect is deprecated in favor of using"
        + " ActiveEffect documents on the associated Actor"});
    this._addDataFieldShim(data, "hexagonalShape", "shape", {since: 13, until: 15,
      warning: "TokenDocument#hexagonalShape is deprecated in favor of TokenDocument#shape."});
    return super.shimData(data, options);
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  get effects() {
    foundry.utils.logCompatibilityWarning("TokenDocument#effects is deprecated in favor of using ActiveEffect"
      + " documents on the associated Actor", {since: 12, until: 14, once: true});
    return [];
  }

  /**
   * @deprecated since v12
   * @ignore
   */
  get overlayEffect() {
    foundry.utils.logCompatibilityWarning("TokenDocument#overlayEffect is deprecated in favor of using"
      + " ActiveEffect documents on the associated Actor", {since: 12, until: 14, once: true});
    return "";
  }

  /**
   * @deprecated since v13
   * @ignore
   */
  get hexagonalShape() {
    foundry.utils.logCompatibilityWarning("TokenDocument#hexagonalShape is deprecated in favor of TokenDocument#shape.",
      {since: 13, until: 15, once: true});
    return this.shape;
  }
}

/* -------------------------------------------- */

/**
 * A special subclass of EmbeddedDocumentField which allows construction of the ActorDelta to be lazily evaluated.
 */
class ActorDeltaField extends EmbeddedDocumentField {
  /** @inheritdoc */
  initialize(value, model, options = {}) {
    if ( !value ) return value;
    const descriptor = Object.getOwnPropertyDescriptor(model, this.name);
    if ( (descriptor === undefined) || (!descriptor.get && !descriptor.value) ) {
      return () => {
        const m = new this.model(value, {...options, parent: model, parentCollection: this.name});
        Object.defineProperty(m, "schema", {value: this});
        Object.defineProperty(model, this.name, {
          value: m,
          configurable: true,
          writable: true
        });
        return m;
      };
    }
    else if ( descriptor.get instanceof Function ) return descriptor.get;
    model[this.name]._initialize(options);
    return model[this.name];
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _updateCommit(source, key, value, diff, options) {
    super._updateCommit(source, key, value, diff, options);
    options._deltaModel?.updateSyntheticActor?.();
  }
}

/**
 * @import {WallData} from "./_types.mjs";
 * @import {DocumentPermissionTest} from "@common/abstract/_types.mjs";
 */

/**
 * The Wall Document.
 * Defines the DataSchema and common behaviors for a Wall which are shared between both client and server.
 * @extends {Document<WallData>}
 * @mixes WallData
 * @category Documents
 */
class BaseWall extends Document {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static metadata = Object.freeze(mergeObject(super.metadata, {
    name: "Wall",
    collection: "walls",
    label: "DOCUMENT.Wall",
    labelPlural: "DOCUMENT.Walls",
    permissions: {
      update: this.#canUpdate
    },
    schemaVersion: "13.341"
  }, {inplace: false}));

  /** @inheritDoc */
  static defineSchema() {
    const choices = [
      ["senseTypes", WALL_SENSE_TYPES, "SenseTypes"],
      ["moveTypes", WALL_MOVEMENT_TYPES, "SenseTypes"],
      ["directions", WALL_DIRECTIONS, "Directions"],
      ["doorTypes", WALL_DOOR_TYPES, "DoorTypes"],
      ["doorStates", WALL_DOOR_STATES, "DoorStates"]
    ].reduce((outer, [key, record, labelObj]) => {
      outer[key] = Object.entries(record).reduce((inner, [labelKey, value]) => {
        inner[value] = `WALL.${labelObj}.${labelKey}`;
        return inner;
      }, {});
      return outer;
    }, {});
    return {
      _id: new DocumentIdField(),
      c: new ArrayField(new NumberField({required: true, integer: true, nullable: false}), {
        validate: c => (c.length === 4),
        validationError: "must be a length-4 array of integer coordinates"}),
      light: new NumberField({required: true, choices: choices.senseTypes,
        initial: WALL_SENSE_TYPES.NORMAL,
        validationError: "must be a value in CONST.WALL_SENSE_TYPES"}),
      move: new NumberField({required: true, choices: choices.moveTypes,
        initial: WALL_MOVEMENT_TYPES.NORMAL,
        validationError: "must be a value in CONST.WALL_MOVEMENT_TYPES"}),
      sight: new NumberField({required: true, choices: choices.senseTypes,
        initial: WALL_SENSE_TYPES.NORMAL,
        validationError: "must be a value in CONST.WALL_SENSE_TYPES"}),
      sound: new NumberField({required: true, choices: choices.senseTypes,
        initial: WALL_SENSE_TYPES.NORMAL,
        validationError: "must be a value in CONST.WALL_SENSE_TYPES"}),
      dir: new NumberField({required: true, choices: choices.directions,
        initial: WALL_DIRECTIONS.BOTH,
        validationError: "must be a value in CONST.WALL_DIRECTIONS"}),
      door: new NumberField({required: true, choices: choices.doorTypes,
        initial: WALL_DOOR_TYPES.NONE,
        validationError: "must be a value in CONST.WALL_DOOR_TYPES"}),
      ds: new NumberField({required: true, choices: choices.doorStates,
        initial: WALL_DOOR_STATES.CLOSED,
        validationError: "must be a value in CONST.WALL_DOOR_STATES"}),
      doorSound: new StringField({required: false, blank: true, initial: undefined}),
      threshold: new SchemaField({
        light: new NumberField({required: true, nullable: true, initial: null, positive: true}),
        sight: new NumberField({required: true, nullable: true, initial: null, positive: true}),
        sound: new NumberField({required: true, nullable: true, initial: null, positive: true}),
        attenuation: new BooleanField()
      }),
      animation: new SchemaField({
        direction: new NumberField({choices: [-1, 1], initial: 1}),
        double: new BooleanField({initial: false}),
        duration: new NumberField({positive: true, integer: true, initial: 750}),
        flip: new BooleanField({initial: false}),
        strength: new NumberField({initial: 1.0, min: 0, max: 2.0, step: 0.05}),
        texture: new FilePathField({categories: ["IMAGE"], virtual: true}),
        type: new StringField({initial: "swing", blank: true}) // Allow any value to be persisted
      }, {required: true, nullable: true, initial: null}),
      flags: new DocumentFlagsField()
    };
  }

  /** @override */
  static LOCALIZATION_PREFIXES = ["DOCUMENT", "WALL"];

  /* -------------------------------------------- */

  /**
   * Is a user able to update an existing Wall?
   * @type {DocumentPermissionTest}
   */
  static #canUpdate(user, doc, data) {
    if ( user.isGM ) return true;                     // GM users can do anything
    const dsOnly = Object.keys(data).every(k => ["_id", "ds"].includes(k));
    if ( dsOnly && (doc.ds !== WALL_DOOR_STATES.LOCKED) && (data.ds !== WALL_DOOR_STATES.LOCKED) ) {
      return user.hasRole("PLAYER");                  // Players may open and close unlocked doors
    }
    return false;
  }
}

/**
 * @import {ElevatedPoint, REGION_MOVEMENT_SEGMENTS, RegionMovementSegmentType, TOKEN_SHAPES,
 *   TokenShapeType} from "@common/constants.mjs";
 * @import {EffectDurationData,  TokenPosition, TokenMovementWaypoint} from "@client/documents/_types.mjs";
 * @import Roll from "@client/dice/roll.mjs";
 * @import {GridMeasurePathCostFunction3D, GridOffset3D} from "@common/grid/_types.mjs";
 * @import {DataModel, Document} from "@common/abstract/_module.mjs";
 * @import {DeepReadonly, TokenConstrainMovementPathOptions, TokenMovementActionConfig} from "../_types.mjs";
 * @import {Combat, Combatant, Folder, RegionDocument, TableResult, TokenDocument, User} from "./_module.mjs";
 */

/**
 * @typedef AdventureImportData
 * The data that is planned to be imported for the adventure, categorized into new documents that will be created and
 * existing documents that will be updated.
 * @property {Record<string, object[]>} toCreate    Arrays of document data to create, organized by document name
 * @property {Record<string, object[]>} toUpdate    Arrays of document data to update, organized by document name
 * @property {number} documentCount                 The total count of documents to import
 */

/**
 * @callback AdventurePreImportCallback
 * A callback function that is invoked and awaited during import data preparation before the adventure import proceeds.
 * This can be used to perform custom pre-processing on the import data.
 * @param {AdventureImportData} data
 * @param {AdventureImportOptions} options
 * @returns {Promise<void>}
 */

/**
 * @typedef AdventureImportOptions
 * Options which customize how the adventure import process is orchestrated.
 * Modules can use the preImportAdventure hook to extend these options by adding preImport or postImport callbacks.
 * @property {boolean} [dialog=true]                Display a warning dialog if existing documents would be overwritten
 * @property {string[]} [importFields]              A subset of adventure fields to import
 * @property {AdventurePreImportCallback[]} [preImport]   An array of awaited pre-import callbacks
 * @property {AdventurePostImportCallback[]} [postImport] An array of awaited post-import callbacks
 */

/**
 * @typedef AdventureImportResult
 * A report of the world Document instances that were created or updated during the import process.
 * @property {Record<string, Document[]>} created Documents created as a result of the import, grouped by document name
 * @property {Record<string, Document[]>} updated Documents updated as a result of the import, grouped by document name
 */

/**
 * @callback AdventurePostImportCallback
 * A callback function that is invoked and awaited after import but before the overall import workflow concludes.
 * This can be used to perform additional custom adventure setup steps.
 * @param {AdventureImportResult} result
 * @param {AdventureImportOptions} options
 * @returns {Promise<void>}
 */

/**
 * @typedef _ActiveEffectDuration
 * @property {string} type            The duration type, either "seconds", "turns", or "none"
 * @property {number|null} duration   The total effect duration, in seconds of world time or as a decimal
 *                                    number with the format {rounds}.{turns}
 * @property {number|null} remaining  The remaining effect duration, in seconds of world time or as a decimal
 *                                    number with the format {rounds}.{turns}
 * @property {string} label           A formatted string label that represents the remaining duration
 * @property {number} [_worldTime]    An internal flag used determine when to recompute seconds-based duration
 * @property {number} [_combatTime]   An internal flag used determine when to recompute turns-based duration
 */

/**
 * @typedef {EffectDurationData & _ActiveEffectDuration} ActiveEffectDuration
 */

/**
 * @typedef FolderChildNode
 * A node of a Folder-content tree
 * @property {boolean} root               Whether this is the root node of a tree
 * @property {Folder} folder              The Folder document represented by this node
 * @property {number} depth               This node's depth number in the tree
 * @property {boolean} visible            Whether the Folder is visible to the current User
 * @property {FolderChildNode[]} children Child nodes of this node
 * @property {Document[]|CompendiumCollection[]} entries Loose contents in this node
 */

/**
 * @typedef CombatHistoryData
 * @property {number} round
 * @property {number|null} turn
 * @property {string|null} tokenId
 * @property {string|null} combatantId
 */

/**
 * @typedef CombatTurnEventContext
 * @property {number} round       The round
 * @property {number} turn        The turn
 * @property {boolean} skipped    Was skipped?
 */

/**
 * @typedef {Omit<CombatTurnEventContext, "turn">} CombatRoundEventContext
 */

/**
 * @template [Data=object]
 * @typedef RegionEvent
 * @property {string} name                The name of the event
 * @property {object} data                The data of the event
 * @property {RegionDocument} region      The Region the event was triggered on
 * @property {User} user                  The User that triggered the event
 */

/**
 * @typedef {RegionEvent<{}>} RegionRegionBoundaryEvent
 * @typedef {RegionEvent<{}>} RegionBehaviorActivatedEvent
 * @typedef {RegionEvent<{}>} RegionBehaviorDeactivatedEvent
 * @typedef {RegionEvent<{}>} RegionBehaviorViewedEvent
 * @typedef {RegionEvent<{}>} RegionBehaviorUnviewedEvent
 */

/**
 * @typedef RegionTokenEnterExitEventData
 * @property {TokenDocument} token                  The Token that entered/exited the Region
 * @property {TokenMovementOperation|null} movement The movement if the Token entered/exited by moving out of the Region
 *
 * @typedef {RegionEvent<RegionTokenEnterExitEventData>} RegionTokenEnterExitEvent
 * @typedef {RegionTokenEnterExitEvent} RegionTokenEnterEvent
 * @typedef {RegionTokenEnterExitEvent} RegionTokenExitEvent
 */

/**
 * @typedef RegionTokenMoveEventData
 * @property {TokenDocument} token                The Token that moved into/out of/within the Region
 * @property {TokenMovementOperation} movement    The movement
 *
 * @typedef {RegionEvent<RegionTokenMoveEventData>} RegionTokenMoveEvent
 * @typedef {RegionTokenMoveEvent} RegionTokenMoveInEvent
 * @typedef {RegionTokenMoveEvent} RegionTokenMoveOutEvent
 * @typedef {RegionTokenMoveEvent} RegionTokenMoveWithinEvent
 */

/**
 * @typedef RegionTokenAnimateEventData
 * @property {TokenDocument} token       The Token that animated into/out of the Region
 * @property {TokenPosition} position    The position of the Token when it moved into/out of the Region
 *
 * @typedef {RegionEvent<RegionTokenAnimateEventData>} RegionTokenAnimateEvent
 * @typedef {RegionTokenAnimateEvent} RegionTokenAnimateInEvent
 * @typedef {RegionTokenAnimateEvent} RegionTokenAnimateOutEvent
 */

/**
 * @typedef RegionTokenTurnEventData
 * @property {TokenDocument} token    The Token that started/ended its Combat turn
 * @property {Combatant} combatant    The Combatant of the Token that started/ended its Combat turn
 * @property {Combat} combat          The Combat
 * @property {number} round           The round of this turn
 * @property {number} turn            The turn that started/ended
 * @property {boolean} skipped        Was the turn skipped?
 *
 * @typedef {RegionEvent<RegionTokenTurnEventData>} RegionTokenTurnEvent
 * @typedef {RegionTokenTurnEvent} RegionTokenTurnStartEvent
 * @typedef {RegionTokenTurnEvent} RegionTokenTurnEndEvent
 */

/**
 * @typedef RegionTokenRoundEventData
 * @property {TokenDocument} token    The Token
 * @property {Combatant} combatant    The Combatant of the Token
 * @property {Combat} combat          The Combat
 * @property {number} round           The round that started/ended
 * @property {boolean} skipped        Was the round skipped?
 *
 * @typedef {RegionEvent<RegionTokenRoundEventData>} RegionTokenRoundEvent
 * @typedef {RegionTokenRoundEvent} RegionTokenRoundStartEvent
 * @typedef {RegionTokenRoundEvent} RegionTokenRoundEndEvent
 */

/**
 * @typedef RegionMovementSegment
 * @property {RegionMovementSegmentType} type   The type of this segment (see {@link CONST.REGION_MOVEMENT_SEGMENTS}).
 * @property {ElevatedPoint} from               The waypoint that this segment starts from.
 * @property {ElevatedPoint} to                 The waypoint that this segment goes to.
 * @property {boolean} teleport                 Teleport between the waypoints?
 */

/**
 * @typedef RegionSegmentizeMovementPathWaypoint
 * @property {number} x                         The x-coordinate in pixels (integer).
 * @property {number} y                         The y-coordinate in pixels (integer).
 * @property {number} elevation                 The elevation in grid units.
 * @property {boolean} [teleport=false]         Teleport from the previous to this waypoint? Default: `false`.
 */

/**
 * @typedef RollTableDraw
 * An object containing the executed Roll and the produced results
 * @property {Roll} roll                The Dice roll which generated the draw
 * @property {TableResult[]} results    An array of drawn TableResult documents
 */

/**
 * @typedef SceneDimensions
 * @property {number} width        The width of the canvas.
 * @property {number} height       The height of the canvas.
 * @property {number} size         The grid size.
 * @property {PIXI.Rectangle} rect      The canvas rectangle.
 * @property {number} sceneX       The X coordinate of the scene rectangle within the larger canvas.
 * @property {number} sceneY       The Y coordinate of the scene rectangle within the larger canvas.
 * @property {number} sceneWidth   The width of the scene.
 * @property {number} sceneHeight  The height of the scene.
 * @property {PIXI.Rectangle} sceneRect The scene rectangle.
 * @property {number} distance     The number of distance units in a single grid space.
 * @property {number} distancePixels  The factor to convert distance units to pixels.
 * @property {string} units        The units of distance.
 * @property {number} ratio        The aspect ratio of the scene rectangle.
 * @property {number} maxR         The length of the longest line that can be drawn on the canvas.
 * @property {number} rows         The number of grid rows on the canvas.
 * @property {number} columns      The number of grid columns on the canvas.
 */

/**
 * @typedef TrackedAttributesDescription
 * @property {string[][]} bar    A list of property path arrays to attributes with both a value and a max property.
 * @property {string[][]} value  A list of property path arrays to attributes that have only a value property.
 */

/**
 * @typedef TokenMeasuredMovementWaypoint
 * @property {number} x                  The top-left x-coordinate in pixels (integer).
 * @property {number} y                  The top-left y-coordinate in pixels (integer).
 * @property {number} elevation          The elevation in grid units.
 * @property {number} width              The width in grid spaces (positive).
 * @property {number} height             The height in grid spaces (positive).
 * @property {TokenShapeType} shape      The shape type (see {@link CONST.TOKEN_SHAPES}).
 * @property {string} action             The movement action from the previous to this waypoint.
 * @property {DataModel|null} terrain    The terrain data from the previous to this waypoint.
 * @property {boolean} snapped           Was this waypoint snapped to the grid?
 * @property {boolean} explicit          Was this waypoint explicitly placed by the user?
 * @property {boolean} checkpoint        Is this waypoint a checkpoint?
 * @property {boolean} intermediate      Is this waypoint intermediate?
 * @property {string} userId             The ID of the user that moved the token to from the previous to this waypoint.
 * @property {string} movementId         The ID of the movement from the previous to this waypoint.
 * @property {number} cost               The movement cost from the previous to this waypoint (nonnegative).
 */

/**
 * @typedef {Omit<TokenMeasuredMovementWaypoint, "terrain"|"intermediate"|"userId"|"movementId"
 *   |"cost">} TokenMovementWaypoint
 */

/**
 * @typedef {Pick<TokenMeasuredMovementWaypoint, "width"|"height"|"shape"|"action"|"terrain">
 *   & {actionConfig: TokenMovementActionConfig, teleport: boolean}} TokenMovementSegmentData
 */

/**
 * @typedef TokenMeasureMovementPathWaypoint
 * @property {number} [x]                                 The top-left x-coordinate in pixels (integer).
 *                                                        Default: the previous or source x-coordinate.
 * @property {number} [y]                                 The top-left y-coordinate in pixels (integer).
 *                                                        Default: the previous or source y-coordinate.
 * @property {number} [elevation]                         The elevation in grid units.
 *                                                        Default: the previous or source elevation.
 * @property {number} [width]                             The width in grid spaces (positive).
 *                                                        Default: the previous or source width.
 * @property {number} [height]                            The height in grid spaces (positive).
 *                                                        Default: the previous or source height.
 * @property {TokenShapeType} [shape]                     The shape type (see {@link CONST.TOKEN_SHAPES}).
 *                                                        Default: the previous or source shape.
 * @property {string} [action]                            The movement action from the previous to this waypoint.
 *                                                        Default: the previous or prepared movement action.
 * @property {DataModel|null} [terrain=null]              The terrain data of this segment. Default: `null`.
 * @property {number|TokenMovementCostFunction} [cost]    A predetermined cost (nonnegative) or cost function
 *                                                        to be used instead of `options.cost`.
 */

/**
 * @typedef TokenMeasureMovementPathOptions
 * @property {boolean} [preview=false]    Measure a preview path? Default: `false`.
 */

/**
 * @typedef {GridMeasurePathCostFunction3D<TokenMovementSegmentData>} TokenMovementCostFunction
 */

/**
 * @callback TokenMovementCostAggregator
 * @param {Array<DeepReadonly<{from: GridOffset3D, to: GridOffset3D, cost: number}>>} results
 *                                                           The results of the cost function calls.
 *                                                           The array may be sorted but otherwise not be mutated.
 * @param {number} distance                                  The distance between the grid spaces.
 * @param {DeepReadonly<TokenMovementSegmentData>} segment   The properties of the segment.
 * @returns {number}                                         The aggregated cost.
 */

/**
 * @typedef TokenGetCompleteMovementPathWaypoint
 * @property {number} [x]                       The top-left x-coordinate in pixels (integer).
 *                                              Default: the previous or source x-coordinate.
 * @property {number} [y]                       The top-left y-coordinate in pixels (integer).
 *                                              Default: the previous or source y-coordinate.
 * @property {number} [elevation]               The elevation in grid units.
 *                                              Default: the previous or source elevation.
 * @property {number} [width]                   The width in grid spaces (positive).
 *                                              Default: the previous or source width.
 * @property {number} [height]                  The height in grid spaces (positive).
 *                                              Default: the previous or source height.
 * @property {TokenShapeType} [shape]           The shape type (see {@link CONST.TOKEN_SHAPES}).
 *                                              Default: the previous or source shape.
 * @property {string} [action]                  The movement action from the previous to this waypoint.
 *                                              Default: the previous or prepared movement action.
 * @property {DataModel|null} [terrain=null]    The terrain data of this segment. Default: `null`.
 * @property {boolean} [snapped=false]          Was this waypoint snapped to the grid? Default: `false`.
 * @property {boolean} [explicit=false]         Was this waypoint explicitly placed by the user? Default: `false`.
 * @property {boolean} [checkpoint=false]       Is this waypoint a checkpoint? Default: `false`.
 * @property {boolean} [intermediate=false]     Is this waypoint intermediate? Default: `false`.
 */

/**
 * @typedef {Omit<TokenMeasuredMovementWaypoint, "userId"|"movementId"|"cost">} TokenCompleteMovementWaypoint
 */

/**
 * @typedef TokenSegmentizeMovementWaypoint
 * @property {number} [x]                       The x-coordinate in pixels (integer).
 *                                              Default: the previous or source x-coordinate.
 * @property {number} [y]                       The y-coordinate in pixels (integer).
 *                                              Default: the previous or source y-coordinate.
 * @property {number} [elevation]               The elevation in grid units.
 *                                              Default: the previous or source elevation.
 * @property {number} [width]                   The width in grid spaces (positive).
 *                                              Default: the previous or source width.
 * @property {number} [height]                  The height in grid spaces (positive).
 *                                              Default: the previous or source height.
 * @property {TokenShapeType} [shape]           The shape type (see {@link CONST.TOKEN_SHAPES}).
 *                                              Default: the previous or source shape.
 * @property {string} [action]                  The movement action from the previous to this waypoint.
 *                                              Default: the previous or prepared movement action.
 * @property {DataModel|null} [terrain=null]    The terrain data of this segment. Default: `null`.
 * @property {boolean} [snapped=false]          Was this waypoint snapped to the grid? Default: `false`.
 */

/**
 * @typedef {TokenPosition} TokenRegionMovementWaypoint
 */

/**
 * @typedef TokenRegionMovementSegment
 * @property {RegionMovementSegmentType} type     The type of this segment (see {@link CONST.REGION_MOVEMENT_SEGMENTS}).
 * @property {TokenRegionMovementWaypoint} from   The waypoint that this segment starts from.
 * @property {TokenRegionMovementWaypoint} to     The waypoint that this segment goes to.
 * @property {string} action                      The movement action between the waypoints.
 * @property {DataModel|null} terrain             The terrain data of this segment.
 * @property {boolean} snapped                    Is the destination snapped to the grid?
 */


/**
 * @typedef TokenMovementSectionData
 * @property {TokenMeasuredMovementWaypoint[]} waypoints    The waypoints of the movement path
 * @property {number} distance                              The distance of the movement path
 * @property {number} cost                                  The cost of the movement path
 * @property {number} spaces                                The number of spaces moved along the path
 * @property {number} diagonals                             The number of diagonals moved along the path
 */

/**
 * @typedef TokenMovementHistoryData
 * @property {TokenMovementSectionData} recorded            The recorded waypoints of the movement path
 * @property {TokenMovementSectionData} unrecorded          The unrecored waypoints of the movement path
 * @property {number} distance                              The distance of the combined movement path
 * @property {number} cost                                  The cost of the combined movement path
 * @property {number} spaces                                The number of spaces moved along the combined path
 * @property {number} diagonals                             The number of diagonals moved along the combined path
 */

/**
 * @typedef {"api"|"config"|"dragging"|"keyboard"|"paste"|"undo"} TokenMovementMethod
 */

/**
 * @typedef {"completed"|"paused"|"pending"|"stopped"} TokenMovementState
 */

/**
 * @typedef TokenMovementData
 * @property {string} id         The ID of the movement
 * @property {string[]} chain    The chain of prior movement IDs that this movement is a continuation of
 * @property {TokenPosition} origin                The origin of movement
 * @property {TokenPosition} destination           The destination of movement
 * @property {TokenMovementSectionData} passed     The waypoints and measurements of the passed path
 * @property {TokenMovementSectionData} pending    The waypoints and measurements of the pending path
 * @property {TokenMovementHistoryData} history    The waypoints and measurements of the history path
 * @property {boolean} constrained                 Was the movement constrained?
 * @property {boolean} recorded                    Was the movement recorded in the movement history?
 * @property {TokenMovementMethod} method          The method of movement
 * @property {Omit<TokenConstrainMovementPathOptions, "preview"|"history">} constrainOptions
 *                                         The options to constrain movement
 * @property {boolean} autoRotate          Automatically rotate the token in the direction of movement?
 * @property {boolean} showRuler           Show the ruler during the movement animation of the token?
 * @property {User} user                   The user that moved the token
 * @property {TokenMovementState} state    The state of the movement
 * @property {object} updateOptions        The update options of the movement operation
 */

/**
 * @typedef {Omit<TokenMovementData, "user"|"state"|"updateOptions">} TokenMovementOperation
 */

/**
 * @typedef TokenMovementContinuationData
 * @property {string} movementId                        The movement ID
 * @property {number} continueCounter                   The number of continuations
 * @property {boolean} continued                        Was continued?
 * @property {Promise<boolean>|null} continuePromise    The continuation promise
 * @property {Promise<void>} waitPromise                The promise to wait for before continuing movement
 * @property {() => {}|undefined} resolveWaitPromise    Resolve function of the wait promise
 * @property {Promise<void>} postWorkflowPromise        The promise that resolves after the update workflow
 * @property {{[movementId: string]: {handles: Map<string|symbol, TokenMovementContinuationHandle>;
 *   callbacks: Array<(continued: boolean) => void>; pending: Set<string>}}} states    The movement continuation states
 */

/**
 * @typedef TokenMovementContinuationHandle
 * @property {string} movementId                             The movement ID
 * @property {Promise<boolean>|undefined} continuePromise    The continuation promise
 */

/**
 * @callback TokenResumeMovementCallback
 * @returns {Promise<boolean>}    A promise that resolves to true if the movement was resumed.
 *                                If it wasn't resumed, it resolves to false.
 */

var _types$i = /*#__PURE__*/Object.freeze({
  __proto__: null
});

/**
 * @typedef HookedFunction
 * @property {string} hook
 * @property {number} id
 * @property {Function} fn
 * @property {boolean} once
 */

/**
 * A simple event framework used throughout Foundry Virtual Tabletop.
 * When key actions or events occur, a "hook" is defined where user-defined callback functions can execute.
 * This class manages the registration and execution of hooked callback functions.
 */
let Hooks$1 = class Hooks {

  /**
   * A mapping of hook events which have functions registered to them.
   * @type {Record<string, HookedFunction[]>}
   */
  static get events() {
    return this.#events;
  }

  /** @type {Record<string, HookedFunction[]>} */
  static #events = {};

  /**
   * A mapping of hooked functions by their assigned ID
   * @type {Map<number, HookedFunction>}
   */
  static #ids = new Map();

  /**
   * An incrementing counter for assigned hooked function IDs
   * @type {number}
   */
  static #id = 1;

  /* -------------------------------------------- */

  /**
   * Register a callback handler which should be triggered when a hook is triggered.
   * @param {string} hook     The unique name of the hooked event
   * @param {Function} fn     The callback function which should be triggered when the hook event occurs
   * @param {object} options  Options which customize hook registration
   * @param {boolean} options.once  Only trigger the hooked function once
   * @returns {number}      An ID number of the hooked function which can be used to turn off the hook later
   */
  static on(hook, fn, {once=false}={}) {
    console.debug(`${CONST.vtt} | Registered callback for ${hook} hook`);
    const id = this.#id++;
    if ( !(hook in this.#events) ) {
      Object.defineProperty(this.#events, hook, {value: [], writable: false});
    }
    const entry = {hook, id, fn, once};
    this.#events[hook].push(entry);
    this.#ids.set(id, entry);
    return id;
  }

  /* -------------------------------------------- */

  /**
   * Register a callback handler for an event which is only triggered once the first time the event occurs.
   * An alias for Hooks.on with {once: true}
   * @param {string} hook   The unique name of the hooked event
   * @param {Function} fn   The callback function which should be triggered when the hook event occurs
   * @returns {number}      An ID number of the hooked function which can be used to turn off the hook later
   */
  static once(hook, fn) {
    return this.on(hook, fn, {once: true});
  }

  /* -------------------------------------------- */

  /**
   * Unregister a callback handler for a particular hook event
   * @param {string} hook           The unique name of the hooked event
   * @param {Function|number} fn    The function, or ID number for the function, that should be turned off
   */
  static off(hook, fn) {
    let entry;

    // Provided an ID
    if ( typeof fn === "number" ) {
      const id = fn;
      entry = this.#ids.get(id);
      if ( !entry ) return;
      this.#ids.delete(id);
      const event = this.#events[entry.hook];
      event.findSplice(h => h.id === id);
    }

    // Provided a Function
    else {
      const event = this.#events[hook];
      if ( !event ) return;
      const entry = event.findSplice(h => h.fn === fn);
      if ( !entry ) return;
      this.#ids.delete(entry.id);
    }
    console.debug(`${CONST.vtt} | Unregistered callback for ${hook} hook`);
  }

  /* -------------------------------------------- */

  /**
   * Call all hook listeners in the order in which they were registered
   * Hooks called this way can not be handled by returning false and will always trigger every hook callback.
   *
   * @param {string} hook   The hook being triggered
   * @param {...*} args     Arguments passed to the hook callback functions
   */
  static callAll(hook, ...args) {
    if ( CONFIG.debug.hooks ) {
      console.log(`DEBUG | Calling ${hook} hook with args:`);
      console.log(args);
    }
    if ( !(hook in this.#events) ) return;
    for ( const entry of Array.from(this.#events[hook]) ) {
      this.#call(entry, args);
    }
  }

  /* -------------------------------------------- */

  /**
   * Call hook listeners in the order in which they were registered.
   * Continue calling hooks until either all have been called or one returns false.
   *
   * Hook listeners which return false denote that the original event has been adequately handled and no further
   * hooks should be called.
   *
   * @param {string} hook   The hook being triggered
   * @param {...*} args     Arguments passed to the hook callback functions
   * @returns {boolean}     Were all hooks called without execution being prevented?
   */
  static call(hook, ...args) {
    if ( CONFIG.debug.hooks ) {
      console.log(`DEBUG | Calling ${hook} hook with args:`);
      console.log(args);
    }
    if ( !(hook in this.#events) ) return true;
    for ( const entry of Array.from(this.#events[hook]) ) {
      const callAdditional = this.#call(entry, args);
      if ( callAdditional === false ) return false;
    }
    return true;
  }

  /* -------------------------------------------- */

  /**
   * Call a hooked function using provided arguments and perhaps unregister it.
   * @param {HookedFunction} entry    The hooked function entry
   * @param {any[]} args              Arguments to be passed
   */
  static #call(entry, args) {
    const {hook, id, fn, once} = entry;
    if ( once ) this.off(hook, id);
    try {
      return entry.fn(...args);
    } catch(err) {
      const msg = `Error thrown in hooked function '${fn?.name}' for hook '${hook}'`;
      console.warn(`${CONST.vtt} | ${msg}`);
      if ( hook !== "error" ) this.onError("Hooks.#call", err, {msg, hook, fn, log: "error"});
    }
  }

  /* --------------------------------------------- */

  /**
   * Notify subscribers that an error has occurred within foundry.
   * @param {string} location                The method where the error was caught.
   * @param {Error} error                    The error.
   * @param {object} [options={}]            Additional options to configure behaviour.
   * @param {string} [options.msg=""]        A message which should prefix the resulting error or notification.
   * @param {?string} [options.log=null]     The level at which to log the error to console (if at all).
   * @param {?string} [options.notify=null]  The level at which to spawn a notification in the UI (if at all).
   * @param {object} [options.data={}]       Additional data to pass to the hook subscribers.
   */
  static onError(location, error, {msg="", notify=null, log=null, ...data}={}) {
    if ( !(error instanceof Error) ) return;
    if ( msg ) error = new Error(`${msg}. ${error.message}`, { cause: error });
    if ( log ) console[log]?.(error);
    if ( notify ) ui.notifications[notify]?.(foundry.utils.escapeHTML(msg || error.message));
    Hooks.callAll("error", location, error, data);
  }
};

/**
 * @import {Application, ApplicationV2} from "@client/applications/api/_module.mjs";
 * @import {SearchableField} from "@client/_types.mjs";
 * @import Document from "@common/abstract/document.mjs";
 * @import {DatabaseAction, DatabaseOperation} from "@common/abstract/_types.mjs";
 * @import User from "../user.mjs";
 */

/**
 * An abstract subclass of the Collection container which defines a collection of Document instances.
 * @abstract
 * @category Collections
 * @template {Document} TDocument
 * @extends Collection<string, TDocument>
 */
class DocumentCollection extends Collection {
  /**
   * @param {object[]} data      An array of data objects from which to create document instances
   */
  constructor(data=[]) {
    super();

    /**
     * The source data array from which the Documents in the WorldCollection are created
     * @type {object[]}
     * @internal
     */
    Object.defineProperty(this, "_source", {
      value: data,
      writable: false
    });

    /**
     * An Array of application references which will be automatically updated when the collection content changes
     * @type {(Application|ApplicationV2)[]}
     */
    this.apps = [];

    // Initialize data
    this._initialize();
  }

  /* -------------------------------------------- */

  /**
   * Initialize the DocumentCollection by constructing any initially provided Document instances
   * @protected
   */
  _initialize() {
    this.clear();
    for ( const d of this._source ) {
      let doc;
      if ( game.issues ) game.issues._countDocumentSubType(this.documentClass, d);
      try {
        doc = this.documentClass.fromSource(d, {strict: true, dropInvalidEmbedded: true});
        super.set(doc.id, doc);
      } catch(err) {
        this.invalidDocumentIds.add(d._id);
        if ( game.issues ) game.issues._trackValidationFailure(this, d, err);
        Hooks$1.onError(`${this.constructor.name}#_initialize`, err, {
          msg: `Failed to initialize ${this.documentName} [${d._id}]`,
          log: "error",
          id: d._id
        });
      }
    }
  }

  /* -------------------------------------------- */
  /*  Collection Properties                       */
  /* -------------------------------------------- */

  /**
   * A reference to the Document class definition which is contained within this DocumentCollection.
   * @type {typeof Document}
   */
  get documentClass() {
    return foundry.utils.getDocumentClass(this.documentName);
  }

  /** @inheritDoc */
  get documentName() {
    const name = this.constructor.documentName;
    if ( !name ) throw new Error("A subclass of DocumentCollection must define its static documentName");
    return name;
  }

  /**
   * The base Document type which is contained within this DocumentCollection
   * @type {string}
   */
  static documentName;

  /**
   * Record the set of document ids where the Document was not initialized because of invalid source data
   * @type {Set<string>}
   */
  invalidDocumentIds = new Set();

  /**
   * The Collection class name
   * @type {string}
   */
  get name() {
    return this.constructor.name;
  }

  /* -------------------------------------------- */
  /*  Collection Methods                          */
  /* -------------------------------------------- */

  /**
   * Instantiate a Document for inclusion in the Collection.
   * @param {object} data       The Document data.
   * @param {object} [context]  Document creation context.
   * @returns {TDocument}
   */
  createDocument(data, context={}) {
    return new this.documentClass(data, context);
  }

  /* -------------------------------------------- */

  /**
   * Obtain a temporary Document instance for a document id which currently has invalid source data.
   * @param {string} id                      A document ID with invalid source data.
   * @param {object} [options]               Additional options to configure retrieval.
   * @param {boolean} [options.strict=true]  Throw an Error if the requested ID is not in the set of invalid IDs for
   *                                         this collection.
   * @returns {TDocument|void}               An in-memory instance for the invalid Document
   * @throws {Error}                         If strict is true and the requested ID is not in the set of invalid IDs
   *                                         for this collection.
   */
  getInvalid(id, {strict=true}={}) {
    if ( !this.invalidDocumentIds.has(id) ) {
      if ( strict ) throw new Error(`${this.constructor.documentName} id [${id}] is not in the set of invalid ids`);
      return;
    }
    const data = this._source.find(d => d._id === id);
    return this.documentClass.fromSource(foundry.utils.deepClone(data));
  }

  /* -------------------------------------------- */

  /**
   * Get an element from the DocumentCollection by its ID.
   * @param {string} id                        The ID of the Document to retrieve.
   * @param {object} [options]                 Additional options to configure retrieval.
   * @param {boolean} [options.strict=false]   Throw an Error if the requested Document does not exist.
   * @param {boolean} [options.invalid=false]  Allow retrieving an invalid Document.
   * @returns {TDocument}
   * @throws {Error}                           If strict is true and the Document cannot be found.
   */
  get(id, {invalid=false, strict=false}={}) {
    let result = super.get(id);
    if ( !result && invalid ) result = this.getInvalid(id, { strict: false });
    if ( !result && strict ) throw new Error(`${this.constructor.documentName} id [${id}] does not exist in the `
      + `${this.constructor.name} collection.`);
    return result;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  set(id, document) {
    const cls = this.documentClass;
    if (!(document instanceof cls)) {
      throw new Error(`You may only push instances of ${cls.documentName} to the ${this.name} collection`);
    }
    const replacement = this.has(document.id);
    super.set(document.id, document);
    if ( replacement ) this._source.findSplice(e => e._id === id, document.toObject());
    else this._source.push(document.toObject());
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  delete(id) {
    super.delete(id);
    const removed = this._source.findSplice(e => e._id === id);
    return !!removed;
  }

  /* -------------------------------------------- */

  /**
   * Render any Applications associated with this DocumentCollection.
   * @param {boolean} [force=false]     Force rendering
   * @param {object} [options={}]       Optional options
   */
  render(force=false, options={}) {
    for ( const app of this.apps ) {
      const opts = foundry.utils.deepClone(options);
      if ( app instanceof foundry.applications.api.ApplicationV2 ) {
        opts.force = force;
        app.render(opts);
      }
      else app.render(force, opts);
    }
  }

  /* -------------------------------------------- */

  /**
   * The cache of search fields for each data model
   * @type {Map<string, Record<string, SearchableField>>}
   */
  static #dataModelSearchFieldsCache = new Map();

  /**
   * Get the searchable fields for a given document or index, based on its data model
   * @param {string} documentName         The document name
   * @param {string} [type]               A document subtype
   * @returns {Record<string, SearchableField>} A record of searchable DataField definitions
   */
  static getSearchableFields(documentName, type) {
    const searchFields = DocumentCollection.#getSearchableFields(documentName);
    if ( type ) {
      const systemFields = DocumentCollection.#getSearchableFields(documentName, type);
      if ( !foundry.utils.isEmpty(systemFields) ) Object.assign(searchFields, systemFields);
    }
    return searchFields;
  }

  /* -------------------------------------------- */

  /**
   * Identify and cache the searchable fields for a DataModel.
   * @param {string} documentName
   * @param {string} [type]
   * @returns {Record<string, SearchableField>}
   */
  static #getSearchableFields(documentName, type) {
    const isSubtype = !!type;
    const cacheName = isSubtype ? `${documentName}.${type}` : documentName;

    // If this already exists in the cache, return it
    const cached = DocumentCollection.#dataModelSearchFieldsCache.get(cacheName);
    if ( cached ) return cached;

    // Reference the Document model
    const docConfig = CONFIG[documentName];
    if ( !docConfig ) throw new Error(`Could not find configuration for ${documentName}`);
    const model = isSubtype ? docConfig.dataModels?.[type] : docConfig.documentClass;
    if ( !model ) return {};

    // Get fields for the base model
    let searchFields = {};
    model.schema.apply(function() {
      if ( (this instanceof foundry.data.fields.StringField) && this.textSearch ) searchFields[this.fieldPath] = this;
    });
    searchFields = foundry.utils.expandObject(searchFields);
    DocumentCollection.#dataModelSearchFieldsCache.set(cacheName, searchFields);
    return searchFields;
  }

  /* -------------------------------------------- */

  /**
   * Find all Documents which match a given search term using a full-text search against their indexed HTML fields
   * and their name. If filters are provided, results are filtered to only those that match the provided values.
   * @param {object} search                      An object configuring the search
   * @param {string} [search.query]              A case-insensitive search string
   * @param {FieldFilter[]} [search.filters]     An array of filters to apply
   * @param {string[]} [search.exclude]          An array of document IDs to exclude from search results
   * @returns {TDocument[]|object[]}
   */
  search({query= "", filters=[], exclude=[]}) {
    query = foundry.applications.ux.SearchFilter.cleanQuery(query);
    const regex = new RegExp(RegExp.escape(query), "i");

    // Iterate over all index members or documents
    const results = [];
    for ( const doc of this.index ?? this.contents ) {
      if ( exclude.includes(doc._id) ) continue; // Explicitly exclude this document
      let matched = !query;

      // Do a full-text search against any searchable fields based on metadata
      if ( query ) {
        const searchFields = DocumentCollection.getSearchableFields(this.documentName, doc.type);
        const match = DocumentCollection.#searchTextFields(doc, searchFields, regex);
        if ( !match ) continue; // Query did not match, no need to continue
        matched = true;
      }

      // Apply filters
      for ( const filter of filters ) {
        const match = foundry.applications.ux.SearchFilter.evaluateFilter(doc, filter);
        if ( !match ) {
          matched = false;
          break; // Filter did not match, no need to continue
        }
      }
      if ( matched ) results.push(doc);
    }
    return results;
  }

  /* -------------------------------------------- */

  /**
   * Recursively search text fields.
   * @param {object} data
   * @param {Record<string, SearchableField>} searchFields
   * @param {RegExp} rgx
   * @param {DOMParser} [domParser]
   */
  static #searchTextFields(data, searchFields, rgx, domParser) {
    for ( const [k, field] of Object.entries(searchFields) ) {
      let v = data[k];
      if ( !v ) continue;
      if ( typeof v === "string" ) {
        if ( field instanceof foundry.data.fields.HTMLField ) {
          domParser ??= new DOMParser();
          // TODO: Ideally we would search the text content of enriched HTML
          v = domParser.parseFromString(v, "text/html").body.textContent;
        }
        if ( foundry.applications.ux.SearchFilter.testQuery(rgx, v) ) return true;
      }
      else if ( Array.isArray(v) ) {
        if ( v.some(x => foundry.applications.ux.SearchFilter.testQuery(rgx, x)) ) return true;
      }
      else if ( typeof v === "object" ) {
        const m = DocumentCollection.#searchTextFields(v, field, rgx, domParser);
        if ( m ) return true;
      }
    }
    return false;
  }

  /* -------------------------------------------- */
  /*  Database Operations                         */
  /* -------------------------------------------- */

  /**
   * Update all objects in this DocumentCollection with a provided transformation.
   * Conditionally filter to only apply to Entities which match a certain condition.
   * @param {Function|object} transformation    An object of data or function to apply to all matched objects
   * @param {Function|null}  condition          A function which tests whether to target each object
   * @param {object} [options]                  Additional options passed to Document.updateDocuments
   * @returns {Promise<TDocument[]>}            An array of updated data once the operation is complete
   */
  async updateAll(transformation, condition=null, options={}) {
    const hasTransformer = transformation instanceof Function;
    if ( !hasTransformer && (foundry.utils.getType(transformation) !== "Object") ) {
      throw new Error("You must provide a data object or transformation function");
    }
    const hasCondition = condition instanceof Function;
    const updates = [];
    for ( const doc of this ) {
      if ( hasCondition && !condition(doc) ) continue;
      const update = hasTransformer ? transformation(doc) : foundry.utils.deepClone(transformation);
      update._id = doc.id;
      updates.push(update);
    }
    return this.documentClass.updateDocuments(updates, options);
  }

  /* -------------------------------------------- */

  /**
   * Follow-up actions to take when a database operation modifies Documents in this DocumentCollection.
   * @param {DatabaseAction} action                   The database action performed
   * @param {TDocument[]} documents                   The array of modified Documents
   * @param {any[]} result                            The result of the database operation
   * @param {DatabaseOperation} operation             Database operation details
   * @param {User} user                               The User who performed the operation
   * @internal
   */
  _onModifyContents(action, documents, result, operation, user) {
    if ( operation.render && !operation.parent && documents.length ) {
      this.render(false, {renderContext: `${action}${documents[0].documentName}`, renderData: result});
    }
  }
}

/**
 * @import Collection from "@common/utils/collection.mjs";
 * @import Folder from "../folder.mjs";
 */

/**
 * A mixin which adds directory functionality to a DocumentCollection, such as folders, tree structures, and sorting.
 * @category Mixins
 * @param {typeof Collection} BaseCollection      The base collection class to extend
 */
function DirectoryCollectionMixin(BaseCollection) {

  /**
   * An extension of the Collection class which adds behaviors specific to tree-based collections of entries and
   * folders.
   */
  return class DirectoryCollection extends BaseCollection {

    /**
     * Reference the set of Folders which contain documents in this collection
     * @type {Collection<string, Folder>}
     */
    get folders() {
      throw new Error("You must implement the folders getter for this DirectoryCollection");
    }

    /* -------------------------------------------- */

    /**
     * The built tree structure of the DocumentCollection
     * @type {object}
     */
    get tree() {
      return this.#tree;
    }

    /**
     * The built tree structure of the DocumentCollection. Lazy initialized.
     * @type {object}
     */
    #tree;

    /* -------------------------------------------- */

    /**
     * The current search mode for this collection
     * @type {string}
     */
    get searchMode() {
      const searchModes = game.settings.get("core", "collectionSearchModes");
      return searchModes[this.collection ?? this.name] || CONST.DIRECTORY_SEARCH_MODES.NAME;
    }

    /**
     * Toggle the search mode for this collection between "name" and "full" text search
     */
    toggleSearchMode() {
      const name = this.collection ?? this.name;
      const searchModes = game.settings.get("core", "collectionSearchModes");
      searchModes[name] = searchModes[name] === CONST.DIRECTORY_SEARCH_MODES.FULL
        ? CONST.DIRECTORY_SEARCH_MODES.NAME
        : CONST.DIRECTORY_SEARCH_MODES.FULL;
      game.settings.set("core", "collectionSearchModes", searchModes);
    }

    /* -------------------------------------------- */

    /**
     * The current sort mode used to order the top level entries in this collection
     * @type {string}
     */
    get sortingMode() {
      const sortingModes = game.settings.get("core", "collectionSortingModes");
      return sortingModes[this.collection ?? this.name] || "a";
    }

    /**
     * Toggle the sorting mode for this collection between "a" (Alphabetical) and "m" (Manual by sort property)
     */
    toggleSortingMode() {
      const name = this.collection ?? this.name;
      const sortingModes = game.settings.get("core", "collectionSortingModes");
      const updatedSortingMode = this.sortingMode === "a" ? "m" : "a";
      sortingModes[name] = updatedSortingMode;
      game.settings.set("core", "collectionSortingModes", sortingModes);
      this.initializeTree();
    }

    /* -------------------------------------------- */

    /**
     * The maximum depth of folder nesting which is allowed in this collection
     * @returns {number}
     */
    get maxFolderDepth() {
      return CONST.FOLDER_MAX_DEPTH;
    }

    /* -------------------------------------------- */

    /**
     * Return a reference to list of entries which are visible to the User in this tree
     * @returns {Array<*>}
     * @protected
     */
    _getVisibleTreeContents() {
      return this.contents;
    }

    /* -------------------------------------------- */

    /**
     * Initialize the tree by categorizing folders and entries into a hierarchical tree structure.
     */
    initializeTree() {
      const folders = this.folders.contents;
      const entries = this._getVisibleTreeContents();
      this.#tree = this.#buildTree(folders, entries);
    }

    /* -------------------------------------------- */

    /**
     * Given a list of Folders and a list of Entries, set up the Folder tree
     * @param {Folder[]} folders        The Array of Folder objects to organize
     * @param {object[]} entries        The Array of Entries objects to organize
     * @returns {object}                A tree structure containing the folders and entries
     */
    #buildTree(folders, entries) {
      const handled = new Set();
      const createNode = (root, folder, depth) => {
        return {root, folder, depth, visible: false, children: [], entries: []};
      };

      // Create the tree structure
      const tree = createNode(true, null, 0);
      const depths = [[tree]];

      // Iterate by folder depth, populating content
      for ( let depth = 1; depth <= this.maxFolderDepth + 1; depth++ ) {
        const allowChildren = depth <= this.maxFolderDepth;
        depths[depth] = [];
        const nodes = depths[depth - 1];
        if ( !nodes.length ) break;
        for ( const node of nodes ) {
          const folder = node.folder;
          if ( !node.root ) { // Ensure we don't encounter any infinite loop
            if ( handled.has(folder.id) ) continue;
            handled.add(folder.id);
          }

          // Classify content for this folder
          const classified = this.#classifyFolderContent(folder, folders, entries, {allowChildren});
          node.entries = classified.entries;
          node.children = classified.folders.map(folder => createNode(false, folder, depth));
          depths[depth].push(...node.children);

          // Update unassigned content
          folders = classified.unassignedFolders;
          entries = classified.unassignedEntries;
        }
      }

      // Populate left-over folders at the root level of the tree
      for ( const folder of folders ) {
        const node = createNode(false, folder, 1);
        const classified = this.#classifyFolderContent(folder, folders, entries, {allowChildren: false});
        node.entries = classified.entries;
        entries = classified.unassignedEntries;
        depths[1].push(node);
      }

      // Populate left-over entries at the root level of the tree
      if ( entries.length ) {
        tree.entries.push(...entries);
      }

      // Sort the top level entries and folders
      const sort = this.sortingMode === "a" ? this.constructor._sortAlphabetical : this.constructor._sortStandard;
      tree.entries.sort(sort);
      tree.children.sort((a, b) => sort(a.folder, b.folder));

      // Recursively filter visibility of the tree
      const filterChildren = node => {
        node.children = node.children.filter(child => {
          filterChildren(child);
          return child.visible;
        });
        node.visible = node.root || game.user.isGM || ((node.children.length + node.entries.length) > 0);

        // Populate some attributes of the Folder document
        if ( node.folder ) {
          node.folder.displayed = node.visible;
          node.folder.depth = node.depth;
          node.folder.children = node.children;
        }
      };
      filterChildren(tree);
      return tree;
    }

    /* -------------------------------------------- */

    /**
     * Creates the list of Folder options in this Collection in hierarchical order
     * for populating the options of a select tag.
     * @returns {{id: string, name: string}[]}
     * @internal
     */
    _formatFolderSelectOptions() {
      const options = [];
      const traverse = node => {
        if ( !node ) return;
        const folder = node.folder;
        if ( folder?.visible ) options.push({
          id: folder.id,
          name: `${"─".repeat(folder.depth - 1)} ${folder.name}`.trim()
        });
        node.children.forEach(traverse);
      };
      traverse(this.tree);
      return options;
    }

    /* -------------------------------------------- */

    /**
     * Populate a single folder with child folders and content
     * This method is called recursively when building the folder tree
     * @param {Folder|null} folder                    A parent folder being populated or null for the root node
     * @param {Folder[]} folders                      Remaining unassigned folders which may be children of this one
     * @param {object[]} entries                      Remaining unassigned entries which may be children of this one
     * @param {object} [options={}]                   Options which configure population
     * @param {boolean} [options.allowChildren=true]  Allow additional child folders
     */
    #classifyFolderContent(folder, folders, entries, {allowChildren = true} = {}) {
      const sort = folder?.sorting === "a" ? this.constructor._sortAlphabetical : this.constructor._sortStandard;

      // Determine whether an entry belongs to a folder, via folder ID or folder reference
      const folderMatches = entry => {
        if ( entry.folder?._id ) return entry.folder._id === folder?._id;
        return (entry.folder === folder) || (entry.folder === folder?._id);
      };

      // Partition folders into children and unassigned folders
      const [unassignedFolders, subfolders] = folders.partition(f => allowChildren && folderMatches(f));
      subfolders.sort(sort);

      // Partition entries into folder contents and unassigned entries
      const [unassignedEntries, contents] = entries.partition(e => folderMatches(e));
      contents.sort(sort);

      // Return the classified content
      return {folders: subfolders, entries: contents, unassignedFolders, unassignedEntries};
    }

    /* -------------------------------------------- */

    /**
     * Sort two Entries by name, alphabetically.
     * @param {Object} a    Some Entry
     * @param {Object} b    Some other Entry
     * @returns {number}    The sort order between entries a and b
     * @protected
     */
    static _sortAlphabetical(a, b) {
      return (a.name ?? "").localeCompare(b.name ?? "", game.i18n.lang);
    }

    /* -------------------------------------------- */

    /**
     * Sort two Entries using their numeric sort fields.
     * @param {Object} a    Some Entry
     * @param {Object} b    Some other Entry
     * @returns {number}    The sort order between Entries a and b
     * @protected
     */
    static _sortStandard(a, b) {
      return (a.sort ?? 0) - (b.sort ?? 0);
    }

    /* -------------------------------------------- */

    /** @inheritDoc */
    _onModifyContents(action, documents, result, operation, user) {
      if ( !operation.parent ) this.initializeTree();
      super._onModifyContents(action, documents, result, operation, user);
    }
  };
}

/**
 * @import Folder from "../folder.mjs";
 * @import CompendiumCollection from "./compendium-collection.mjs";
 */

/**
 * A Collection of Folder documents within a Compendium pack.
 * @extends {DocumentCollection<Folder>}
 * @category Collections
 */
class CompendiumFolderCollection extends DocumentCollection {
  constructor(pack, data=[]) {
    super(data);
    this.pack = pack;
  }

  /**
   * The CompendiumCollection instance that contains this CompendiumFolderCollection
   * @type {CompendiumCollection}
   */
  pack;

  /* -------------------------------------------- */

  /** @inheritDoc */
  get documentName() {
    return "Folder";
  }

  /* -------------------------------------------- */

  /** @override */
  render(force, options) {
    this.pack.render(force, options);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async updateAll(transformation, condition=null, options={}) {
    options.pack = this.pack.collection;
    return super.updateAll(transformation, condition, options);
  }

  /* -------------------------------------------- */

  /** @override */
  _onModifyContents(action, documents, result, operation, user) {
    this.pack._onModifyContents(action, documents, result, operation, user);
  }
}

/**
 * @import Folder from "../folder.mjs";
 * @import {ManageCompendiumResponse, WorldCompendiumConfiguration,
 *   WorldCompendiumPackConfiguration} from "@client/_types.mjs";
 * @import WorldCollection from "../abstract/world-collection.mjs";
 * @import Document from "@common/abstract/document.mjs";
 */

/**
 * A collection of Document objects contained within a specific compendium pack.
 * Each Compendium pack has its own associated instance of the CompendiumCollection class which contains its contents.
 *
 * ### Hook Events
 * - {@link hookEvents.updateCompendium}
 *
 * @template {Document} TDocument
 * @extends {DocumentCollection<TDocument>}
 * @category Collections
 *
 * @see {@link foundry.Game#packs}
 */
class CompendiumCollection extends DirectoryCollectionMixin(DocumentCollection) {
  /**
   * @param {object} metadata   The compendium metadata, an object provided by game.data
   */
  constructor(metadata) {
    super();

    // Cache the world setting if not already populated
    CompendiumCollection.#config ??= foundry.utils.deepClone(
      game.settings.get("core", CompendiumCollection.CONFIG_SETTING)
    );

    /**
     * The compendium metadata which defines the compendium content and location
     * @type {object}
     */
    this.metadata = metadata;

    /**
     * A subsidiary collection which contains the more minimal index of the pack
     * @type {Collection<string, object>}
     */
    this.index = new foundry.utils.Collection();

    /**
     * A subsidiary collection which contains the folders within the pack
     * @type {Collection<string, Folder>}
     */
    this.#folders = new CompendiumFolderCollection(this);

    // Initialize a provided Compendium index
    this.#indexedFields = new Set(this.documentClass.metadata.compendiumIndexFields);
    for ( const i of metadata.index ) {
      i.uuid = this.getUuid(i._id);
      this.index.set(i._id, i);
    }
    delete metadata.index;
    const Folder = foundry.utils.getDocumentClass("Folder");
    for ( const f of metadata.folders.sort((a, b) => a.sort - b.sort) ) {
      this.#folders.set(f._id, new Folder(f, {pack: this.collection}));
    }
    delete metadata.folders;
  }

  /* -------------------------------------------- */

  /**
   * The amount of time that Document instances within this CompendiumCollection are held in memory.
   * Accessing the contents of the Compendium pack extends the duration of this lifetime.
   * @type {number}
   */
  static CACHE_LIFETIME_SECONDS = 300;

  /**
   * The named game setting which contains Compendium configurations.
   * @type {string}
   */
  static CONFIG_SETTING = "compendiumConfiguration";

  /**
   * The DataField definition for the configuration Setting
   * @type {foundry.data.fields.TypedObjectField}
   */
  static get CONFIG_FIELD() {
    if ( CompendiumCollection.#CONFIG_FIELD ) return CompendiumCollection.#CONFIG_FIELD;
    const ownershipChoices = Object.keys(CONST.DOCUMENT_OWNERSHIP_LEVELS);
    return CompendiumCollection.#CONFIG_FIELD = new TypedObjectField(new SchemaField({
      folder: new StringField({required: true, blank: false, nullable: true,
        validate: f => isValidId(f)}),
      sort: new NumberField({required: false, nullable: false, integer: true, min: 0, initial: undefined}),
      locked: new BooleanField({required: false, initial: undefined}),
      ownership: new SchemaField({
        GAMEMASTER: new StringField({required: true, choices: ["OWNER"], initial: "OWNER"}),
        ASSISTANT: new StringField({required: true, choices: ownershipChoices, initial: "OWNER"}),
        TRUSTED: new StringField({required: true, choices: ownershipChoices, initial: "INHERIT"}),
        PLAYER: new StringField({required: true, choices: ownershipChoices, initial: "INHERIT"})
      }, {required: false, initial: undefined})
    }));
  }

  static #CONFIG_FIELD;

  /**
   * The cached value of the compendiumConfiguration setting.
   * @type {WorldCompendiumConfiguration}
   */
  static #config;

  /* -------------------------------------------- */

  /**
   * The canonical Compendium name - comprised of the originating package and the pack name
   * @type {string}
   */
  get collection() {
    return this.metadata.id;
  }

  /**
   * The banner image for this Compendium pack, or the default image for the pack type if no image is set.
   * @returns {string|null|void}
   */
  get banner() {
    if ( this.metadata.banner === undefined ) return CONFIG[this.metadata.type]?.compendiumBanner;
    return this.metadata.banner;
  }

  /**
   * A reference to the Application class which provides an interface to interact with this compendium content.
   * @type {typeof foundry.appv1.api.Application|typeof foundry.applications.api.ApplicationV2}
   */
  applicationClass = foundry.applications.sidebar.apps.Compendium;

  /**
   * The set of Compendium Folders
   */
  #folders;

  get folders() {
    return this.#folders;
  }

  /** @override */
  get maxFolderDepth() {
    return super.maxFolderDepth - 1;
  }

  /* -------------------------------------------- */

  /**
   * Get the Folder that this Compendium is displayed within
   * @returns {Folder|null}
   */
  get folder() {
    return game.folders.get(this.config.folder) ?? null;
  }

  /* -------------------------------------------- */

  /**
   * Assign this CompendiumCollection to be organized within a specific Folder.
   * @param {Folder|string|null} folder     The desired Folder within the World or null to clear the folder
   * @returns {Promise<void>}               A promise which resolves once the transaction is complete
   */
  async setFolder(folder) {
    const current = this.config.folder;

    // Clear folder
    if ( folder === null ) {
      if ( current === null ) return;
      return this.configure({folder: null});
    }

    // Set folder
    if ( typeof folder === "string" ) folder = game.folders.get(folder);
    if ( !(folder instanceof foundry.documents.Folder) ) throw new Error("You must pass a valid Folder or Folder ID.");
    if ( folder.type !== "Compendium" ) throw new Error(`Folder "${folder.id}" is not of the required Compendium type`);
    if ( folder.id === current ) return;
    await this.configure({folder: folder.id});
  }

  /* -------------------------------------------- */

  /**
   * Get the sort order for this Compendium
   * @returns {number}
   */
  get sort() {
    return this.config.sort ?? 0;
  }

  /* -------------------------------------------- */

  /** @override */
  _getVisibleTreeContents() {
    return this.index.contents;
  }

  /**
   * Access the compendium configuration data for this pack
   * @type {object}
   */
  get config() {
    return CompendiumCollection.#config[this.collection] || {};
  }

  /** @inheritDoc */
  get documentName() {
    return this.metadata.type;
  }

  /**
   * Track whether the Compendium Collection is locked for editing
   * @type {boolean}
   */
  get locked() {
    return this.config.locked ?? (this.metadata.packageType !== "world");
  }

  /**
   * The visibility configuration of this compendium pack.
   * @type {WorldCompendiumPackConfiguration["ownership"]}
   */
  get ownership() {
    return this.config.ownership ?? this.metadata.ownership ?? {
      ...foundry.packages.Module.schema.getField("packs.ownership").initial
    };
  }

  /**
   * Is this Compendium pack visible to the current game User?
   * @type {boolean}
   */
  get visible() {
    return this.getUserLevel() >= CONST.DOCUMENT_OWNERSHIP_LEVELS.OBSERVER;
  }

  /**
   * A convenience reference to the label which should be used as the title for the Compendium pack.
   * @type {string}
   */
  get title() {
    return this.metadata.label;
  }

  /**
   * The index fields which should be loaded for this compendium pack
   * @type {Set<string>}
   */
  get indexFields() {
    const coreFields = this.documentClass.metadata.compendiumIndexFields;
    const configFields = CONFIG[this.documentName].compendiumIndexFields || [];
    return new Set([...coreFields, ...configFields]);
  }

  /**
   * Track which document fields have been indexed for this compendium pack
   * @type {Set<string>}
   */
  #indexedFields;

  /**
   * Has this compendium pack been fully indexed?
   * @type {boolean}
   */
  get indexed() {
    return this.indexFields.isSubsetOf(this.#indexedFields);
  }

  /* -------------------------------------------- */

  /**
   * A debounced function which will clear the contents of the Compendium pack if it is not accessed frequently.
   * @type {Function}
   */
  #flush = foundry.utils.debounce(this.clear.bind(this), this.constructor.CACHE_LIFETIME_SECONDS * 1000);

  /* -------------------------------------------- */
  /*  Methods                                     */
  /* -------------------------------------------- */

  /** @inheritDoc */
  get(key, options) {
    this.#flush();
    return super.get(key, options);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  set(id, document) {
    if ( document instanceof foundry.documents.Folder ) {
      return this.#folders.set(id, document);
    }
    this.#flush();
    this.indexDocument(document);
    return super.set(id, document);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  delete(id) {
    this.index.delete(id);
    return super.delete(id);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  clear() {
    for ( const doc of this.values() ) {
      if ( !Object.values(doc.apps).some(app => app.rendered) ) super.delete(doc.id);
    }
  }

  /* -------------------------------------------- */

  /**
   * Load the Compendium index and cache it as the keys and values of the Collection.
   * @param {object} [options]    Options which customize how the index is created
   * @param {string[]} [options.fields]  An array of fields to return as part of the index
   * @returns {Promise<Collection>}
   */
  async getIndex({fields=[]}={}) {
    const cls = this.documentClass;

    // Maybe reuse the existing index if we have already indexed all fields
    const indexFields = new Set([...this.indexFields, ...fields]);
    if ( indexFields.isSubsetOf(this.#indexedFields) ) return this.index;

    // Request the new index from the server
    const index = await cls.database.get(cls, {
      query: {},
      index: true,
      indexFields: Array.from(indexFields),
      pack: this.collection
    }, game.user);

    // Assign the index to the collection
    const restoreArt = this.documentName === "Actor";
    for ( const i of index ) {
      const x = this.index.get(i._id);
      const indexed = x ? foundry.utils.mergeObject(x, i) : i;
      indexed.uuid = this.getUuid(indexed._id);
      // Restore compendium art if previously assigned
      if ( restoreArt ) {
        const img = game.compendiumArt.get(indexed.uuid)?.actor ?? indexed.img;
        indexed.img = img;
      }
      this.index.set(i._id, indexed);
    }

    // Record that the pack has been indexed
    console.log(`${CONST.vtt} | Constructed index of ${this.collection} Compendium containing ${this.index.size} entries`);
    this.#indexedFields = indexFields;
    return this.index;
  }

  /* -------------------------------------------- */

  /**
   * Get a single Document from this Compendium by ID.
   * The document may already be locally cached, otherwise it is retrieved from the server.
   * @param {string} id               The requested Document id
   * @returns {Promise<TDocument>|undefined}     The retrieved Document instance
   */
  async getDocument(id) {
    if ( !id ) return undefined;
    const cached = this.get(id);
    if ( cached instanceof foundry.abstract.Document ) return cached;
    const documents = await this.getDocuments({_id: id});
    return documents.length ? documents.shift() : null;
  }

  /* -------------------------------------------- */

  /**
   * Load multiple documents from the Compendium pack using a provided query object. The available query options are
   * shown below.
   * @param {object} query            A database query used to retrieve documents from the underlying database
   * @returns {Promise<TDocument[]>}   The retrieved Document instances
   *
   * @example Get Documents that match the given value only.
   * ```js
   * await pack.getDocuments({ type: "weapon" });
   * ```
   *
   * @example Get all Documents that do not have the given value.
   * ```js
   * await pack.getDocuments({ type__ne: "weapon" });
   * ```
   *
   * @example Get several Documents by their IDs.
   * ```js
   * await pack.getDocuments({ _id__in: arrayOfIds });
   * ```
   *
   * @example Get Documents by their sub-types.
   * ```js
   * await pack.getDocuments({ type__in: ["weapon", "armor"] });
   * ```
   */
  async getDocuments(query={}) {
    const cls = this.documentClass;
    const documents = await cls.database.get(cls, {query, pack: this.collection}, game.user);
    for ( const d of documents ) {
      if ( d.invalid && !this.invalidDocumentIds.has(d.id) ) {
        this.invalidDocumentIds.add(d.id);
        this._source.push(d);
      }
      else this.set(d.id, d);
    }
    return documents;
  }

  /* -------------------------------------------- */

  /**
   * Get the ownership level that a User has for this Compendium pack.
   * @param {documents.User} user     The user being tested
   * @returns {number}                The ownership level in CONST.DOCUMENT_OWNERSHIP_LEVELS
   */
  getUserLevel(user=game.user) {
    const levels = CONST.DOCUMENT_OWNERSHIP_LEVELS;
    let level = levels.NONE;
    for ( const [role, l] of Object.entries(this.ownership) ) {
      if ( user.hasRole(role) ) level = Math.max(level, levels[l]);
    }
    return level;
  }

  /* -------------------------------------------- */

  /**
   * Test whether a certain User has a requested permission level (or greater) over the Compendium pack
   * @param {documents.BaseUser} user       The User being tested
   * @param {string|number} permission      The permission level from DOCUMENT_OWNERSHIP_LEVELS to test
   * @param {object} options                Additional options involved in the permission test
   * @param {boolean} [options.exact=false]     Require the exact permission level requested?
   * @returns {boolean}                      Does the user have this permission level over the Compendium pack?
   */
  testUserPermission(user, permission, {exact=false}={}) {
    const perms = CONST.DOCUMENT_OWNERSHIP_LEVELS;
    const level = user.isGM ? perms.OWNER : this.getUserLevel(user);
    const target = (typeof permission === "string") ? (perms[permission] ?? perms.OWNER) : permission;
    return exact ? level === target : level >= target;
  }

  /* -------------------------------------------- */

  /**
   * Import a Document into this Compendium Collection.
   * @param {Document} document     The existing Document you wish to import
   * @param {object} [options]      Additional options which modify how the data is imported.
   *                                See ClientDocumentMixin#toCompendium.
   * @returns {Promise<TDocument>}   The imported Document instance
   */
  async importDocument(document, options={}) {
    if ( !(document instanceof this.documentClass) && !(document instanceof foundry.documents.Folder) ) {
      const err = Error(`You may not import a ${document.constructor.name} Document into the ${this.collection}
      Compendium which contains ${this.documentClass.name} Documents.`);
      ui.notifications.error(err.message);
      throw err;
    }
    options.clearOwnership = options.clearOwnership ?? (this.metadata.packageType === "world");
    const data = document.toCompendium(this, options);

    return document.constructor.create(data, {pack: this.collection});
  }

  /* -------------------------------------------- */

  /**
   * Import a Folder into this Compendium Collection.
   * @param {Folder} folder                         The existing Folder you wish to import
   * @param {object} [options]                      Additional options which modify how the data is imported.
   * @param {boolean} [options.importParents=true]  Import any parent folders which are not already present in the
   *                                                Compendium.
   * @returns {Promise<void>}
   */
  async importFolder(folder, {importParents=true, ...options}={}) {
    if ( !(folder instanceof foundry.documents.Folder) ) {
      const err = Error(`You may not import a ${folder.constructor.name} Document into the folders collection of
      the ${this.collection} Compendium.`);
      ui.notifications.error(err.message);
      throw err;
    }

    const toCreate = [folder];
    if ( importParents ) toCreate.push(...folder.getParentFolders().filter(f => !this.folders.has(f.id)));
    await foundry.documents.Folder.implementation.createDocuments(toCreate, {pack: this.collection, keepId: true});
  }

  /* -------------------------------------------- */

  /**
   * Import an array of Folders into this Compendium Collection.
   * @param {Folder[]} folders                      The existing Folders you wish to import
   * @param {object} [options]                      Additional options which modify how the data is imported.
   * @param {boolean} [options.importParents=true]  Import any parent folders which are not already present in the
   *                                                Compendium.
   * @returns {Promise<void>}
   */
  async importFolders(folders, {importParents=true, ...options}={}) {
    if ( folders.some(f => !(f instanceof foundry.documents.Folder)) ) {
      const err = Error(`You can only import Folder documents into the folders collection of the ${this.collection}
      Compendium.`);
      ui.notifications.error(err.message);
      throw err;
    }

    const toCreate = new Set(folders);
    if ( importParents ) {
      for ( const f of folders ) {
        for ( const p of f.getParentFolders() ) {
          if ( !this.folders.has(p.id) ) toCreate.add(p);
        }
      }
    }
    await foundry.documents.Folder.implementation.createDocuments(Array.from(toCreate),
      {pack: this.collection, keepId: true});
  }

  /* -------------------------------------------- */

  /**
   * Fully import the contents of a Compendium pack into a World folder.
   * @param {object} [options={}]     Options which modify the import operation. Additional options are forwarded to
   *                                  {@link foundry.documents.abstract.WorldCollection#fromCompendium} and
   *                                  {@link foundry.abstract.Document.createDocuments}
   * @param {string|null} [options.folderId]  An existing Folder _id to use.
   * @param {string} [options.folderName]     A new Folder name to create.
   * @returns {Promise<TDocument[]>}   The imported Documents, now existing within the World
   */
  async importAll({folderId=null, folderName="", ...options}={}) {
    let parentFolder;

    // Optionally, create a top level folder
    if ( CONST.FOLDER_DOCUMENT_TYPES.includes(this.documentName) ) {

      // Re-use an existing folder
      if ( folderId ) parentFolder = game.folders.get(folderId, {strict: true});

      // Create a new Folder
      if ( !parentFolder ) {
        parentFolder = await foundry.documents.Folder.implementation.create({
          name: folderName || this.title,
          type: this.documentName,
          parent: null,
          color: this.folder?.color ?? null
        });
      }
    }

    // Load all content
    const folders = this.folders;
    const documents = await this.getDocuments();
    ui.notifications.info("COMPENDIUM.ImportAll.Start", {format: {number: documents.length, folderNumber: folders.size,
      type: game.i18n.localize(this.documentClass.metadata.label), folder: parentFolder.name}});

    // Create any missing Folders
    const folderCreateData = folders.map(f => {
      if ( game.folders.has(f.id) ) return null;
      const data = f.toObject();
      // If this folder has no parent folder, assign it to the new folder
      if ( !data.folder ) data.folder = parentFolder.id;
      return data;
    }).filter(f => f);
    await foundry.documents.Folder.implementation.createDocuments(folderCreateData, {keepId: true});

    // Prepare import data
    const collection = game.collections.get(this.documentName);
    const createData = documents.map(doc => {
      const data = collection.fromCompendium(doc, options);

      // If this document has no folder, assign it to the new folder
      if ( !data.folder) data.folder = parentFolder.id;
      return data;
    });

    // Create World Documents in batches
    const chunkSize = 100;
    const nBatches = Math.ceil(createData.length / chunkSize);
    let created = [];
    for ( let n=0; n<nBatches; n++ ) {
      const chunk = createData.slice(n*chunkSize, (n+1)*chunkSize);
      const docs = await this.documentClass.createDocuments(chunk, options);
      created = created.concat(docs);
    }

    // Notify of success
    ui.notifications.info("COMPENDIUM.ImportAll.Finish", {format: {number: created.length, folderNumber: folders.size,
      folder: parentFolder.name, type: game.i18n.localize(this.documentClass.metadata.label)}});
    return created;
  }

  /* -------------------------------------------- */

  /**
   * Provide a dialog form that prompts the user to import the full contents of a Compendium pack into the World.
   * @param {object} [options={}] Additional options passed to the DialogV2.confirm method
   * @returns {Promise<TDocument[]|boolean|null>} A promise which resolves in the following ways: an array of imported
   *                            Documents if the "yes" button was pressed, false if the "no" button was pressed, or
   *                            null if the dialog was closed without making a choice.
   */
  async importDialog(options={}) {
    // Render the HTML form
    const collection = CONFIG[this.documentName]?.collection?.instance;
    const html = await foundry.applications.handlebars.renderTemplate("templates/sidebar/apps/compendium-import.hbs", {
      folderName: this.title,
      keepId: options.keepId ?? false,
      folders: collection?._formatFolderSelectOptions() ?? []
    });
    const content = document.createElement("div");
    content.innerHTML = html;

    // Present the Dialog
    return foundry.applications.api.DialogV2.confirm(foundry.utils.mergeObject({
      window: {
        title: game.i18n.format("COMPENDIUM.ImportAll.Title", {compendium: this.title}), // FIXME: double localization
        icon: "fa-solid fa-download"
      },
      content,
      render: event => {
        const form = event.target.element.querySelector("form");
        form.elements.folder.addEventListener("change", event => {
          form.elements.folderName.disabled = !!event.currentTarget.value;
        }, {passive: true});
      },
      yes: {
        label: "COMPENDIUM.ImportAll.Submit",
        callback: event => {
          const form = event.currentTarget.querySelector("form");
          return this.importAll({
            folderId: form.elements.folder.value,
            folderName: form.folderName.value,
            keepId: form.keepId.checked
          });
        }
      },
      no: {label: "Cancel"}
    }, options));
  }

  /* -------------------------------------------- */

  /**
   * Add a Document to the index, capturing its relevant index attributes
   * @param {TDocument} document       The document to index
   */
  indexDocument(document) {
    let index = this.index.get(document.id);
    const data = document.toObject();
    if ( index ) foundry.utils.mergeObject(index, data, {insertKeys: false, insertValues: false});
    else {
      index = this.#indexedFields.reduce((obj, field) => {
        foundry.utils.setProperty(obj, field, foundry.utils.getProperty(data, field));
        return obj;
      }, {});
    }
    index.img = data.thumb ?? data.img;
    index._id = data._id;
    index.uuid = document.uuid;
    this.index.set(document.id, index);
  }

  /* -------------------------------------------- */

  /**
   * Prompt the gamemaster with a dialog to configure ownership of this Compendium pack.
   * @returns {Promise<Record<string, string>>}   The configured ownership for the pack
   */
  async configureOwnershipDialog() {
    if ( !game.user.isGM ) throw new Error("You do not have permission to configure ownership for this Compendium pack");
    const current = this.ownership;
    const levels = {
      "": game.i18n.localize("COMPENDIUM.OwnershipInheritBelow"),
      NONE: game.i18n.localize("OWNERSHIP.NONE"),
      LIMITED: game.i18n.localize("OWNERSHIP.LIMITED"),
      OBSERVER: game.i18n.localize("OWNERSHIP.OBSERVER"),
      OWNER: game.i18n.localize("OWNERSHIP.OWNER")
    };
    const roles = {
      ASSISTANT: {label: "USER.RoleAssistant", value: current.ASSISTANT, levels: { ...levels }},
      TRUSTED: {label: "USER.RoleTrusted", value: current.TRUSTED, levels: { ...levels }},
      PLAYER: {label: "USER.RolePlayer", value: current.PLAYER, levels: { ...levels }}
    };
    delete roles.PLAYER.levels[""];
    await foundry.applications.api.DialogV2.wait({
      id: "compendium-ownership",
      window: {
        title: game.i18n.format("OWNERSHIP.Title", {object: game.i18n.localize(this.metadata.label)}), // FIXME: double localization
        icon: "fa-solid fa-user-lock"
      },
      position: {width: 480},
      content: await foundry.applications.handlebars.renderTemplate("templates/sidebar/apps/compendium-ownership.hbs", {roles}),
      buttons: [
        {
          action: "reset",
          label: "COMPENDIUM.OwnershipReset",
          icon: "fa-solid fa-arrow-rotate-left",
          callback: () => this.configure({ ownership: undefined })
        },
        {
          action: "ok",
          label: "OWNERSHIP.Configure",
          icon: "fa-solid fa-check",
          callback: async event => {
            const fd = new foundry.applications.ux.FormDataExtended(event.currentTarget.querySelector("form"));
            const ownership = Object.entries(fd.object).reduce((obj, [r, l]) => {
              if ( l ) obj[r] = l;
              return obj;
            }, {});
            ownership.GAMEMASTER = "OWNER";
            await this.configure({ownership});
          },
          default: true
        }
      ]
    });
    return this.ownership;
  }

  /* -------------------------------------------- */
  /*  Compendium Management                       */
  /* -------------------------------------------- */

  /**
   * Activate the Socket event listeners used to receive responses to compendium management events.
   * @param {Socket} socket  The active game socket.
   * @internal
   */
  static _activateSocketListeners(socket) {
    socket.on("manageCompendium", response => {
      const { request } = response;
      switch ( request.action ) {
        case "create":
          CompendiumCollection.#handleCreateCompendium(response);
          break;
        case "delete":
          CompendiumCollection.#handleDeleteCompendium(response);
          break;
        default:
          throw new Error(`Invalid Compendium modification action ${request.action} provided.`);
      }
    });
  }

  /**
   * Create a new Compendium Collection using provided metadata.
   * @param {object} metadata   The compendium metadata used to create the new pack
   * @param {object} options   Additional options which modify the Compendium creation request
   * @returns {Promise<CompendiumCollection>}
   */
  static async createCompendium(metadata, options={}) {
    if ( !game.user.isGM ) return ui.notifications.error("You do not have permission to modify this compendium pack");
    const response = await foundry.helpers.SocketInterface.dispatch("manageCompendium", {
      action: "create",
      data: metadata,
      options: options
    });

    return this.#handleCreateCompendium(response);
  }

  /* -------------------------------------------- */

  /**
   * Generate a UUID for a given primary document ID within this Compendium pack
   * @param {string} id     The document ID to generate a UUID for
   * @returns {string}      The generated UUID, in the form of "Compendium.<collection>.<documentName>.<id>"
   */
  getUuid(id) {
    const {documentName, collection: pack} = this;
    return foundry.utils.buildUuid({id, documentName, pack});
  }

  /* ----------------------------------------- */

  /**
   * Assign configuration metadata settings to the compendium pack
   * @param {object} configuration  The object of compendium settings to define
   * @returns {Promise<void>}       A Promise which resolves once the setting is updated
   */
  async configure(configuration={}) {
    const settings = foundry.utils.deepClone(CompendiumCollection.#config);
    settings[this.collection] ||= {};
    const config = settings[this.collection];
    for ( const [k, v] of Object.entries(configuration) ) {
      if ( v === undefined ) delete config[k];
      else config[k] = v;
    }
    await game.settings.set("core", this.constructor.CONFIG_SETTING, settings);
  }

  /* ----------------------------------------- */

  /**
   * Delete an existing world-level Compendium Collection.
   * This action may only be performed for world-level packs by a Gamemaster User.
   * @returns {Promise<CompendiumCollection>}
   */
  async deleteCompendium() {
    this.#assertUserCanManage();
    this.apps.forEach(app => app.close());
    const response = await foundry.helpers.SocketInterface.dispatch("manageCompendium", {
      action: "delete",
      data: this.metadata.name
    });

    return CompendiumCollection.#handleDeleteCompendium(response);
  }

  /* ----------------------------------------- */

  /**
   * Duplicate a compendium pack to the current World.
   * @param {string} label    A new Compendium label
   * @returns {Promise<CompendiumCollection>}
   */
  async duplicateCompendium({label}={}) {
    this.#assertUserCanManage({requireUnlocked: false});
    label = label || this.title;
    const metadata = foundry.utils.mergeObject(this.metadata, {
      name: label.slugify({strict: true}),
      label: label
    }, {inplace: false});
    return this.constructor.createCompendium(metadata, {source: this.collection});
  }

  /* ----------------------------------------- */

  /**
   * Validate that the current user is able to modify content of this Compendium pack
   * @param {object} [options]
   * @param {boolean} [options.requireUnlocked=true] Throw if the compendium is locked.
   * @returns {boolean}
   */
  #assertUserCanManage({requireUnlocked=true}={}) {
    const config = this.config;
    let err;
    if ( !game.user.isGM ) err = new Error("You do not have permission to modify this compendium pack");
    if ( requireUnlocked && config.locked ) {
      err = new Error("You cannot modify content in this compendium pack because it is locked.");
    }
    if ( err ) {
      ui.notifications.error(err.message);
      throw err;
    }
    return true;
  }

  /* -------------------------------------------- */

  /**
   * Migrate a compendium pack.
   * This operation re-saves all documents within the compendium pack to disk, applying the current data model.
   * If the document type has system data, the latest system data template will also be applied to all documents.
   * @param {object} [options]
   * @param {boolean} [options.notify=true]  Display notifications
   * @returns {Promise<CompendiumCollection>}
   */
  async migrate({ notify=true }={}) {
    this.#assertUserCanManage();
    if ( notify ) {
      ui.notifications.info("COMPENDIUM.Migration.Begin", { format: { collection: this.collection } });
    }
    await foundry.helpers.SocketInterface.dispatch("manageCompendium", {
      type: this.collection,
      action: "migrate",
      data: this.collection,
      options: { broadcast: false }
    });
    if ( notify ) ui.notifications.info("COMPENDIUM.Migration.Complete", { format: { collection: this.collection } });
    return this;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async updateAll(transformation, condition=null, options={}) {
    await this.getDocuments();
    options.pack = this.collection;
    return super.updateAll(transformation, condition, options);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  render(force, options) {
    super.render(force, options);
    if ( options?.renderContext === "updateConfiguration" ) {
      for ( const document of this.contents ) document.render(false);
    }
  }

  /* -------------------------------------------- */
  /*  Event Handlers                              */
  /* -------------------------------------------- */

  /** @inheritDoc */
  _onModifyContents(action, documents, result, operation, user) {
    super._onModifyContents(action, documents, result, operation, user);
    Hooks$1.callAll("updateCompendium", this, documents, operation, user.id);
  }

  /* -------------------------------------------- */

  /**
   * Handle a response from the server where a compendium was created.
   * @param {ManageCompendiumResponse} response  The server response.
   * @returns {CompendiumCollection}
   */
  static #handleCreateCompendium({ result }) {
    game.data.packs.push(result);
    const pack = new this(result);
    game.packs.set(pack.collection, pack);
    pack.initializeTree();
    pack.apps.push(new pack.applicationClass({collection: pack}));
    game.packs.initializeTree();
    ui.compendium.render();
    return pack;
  }

  /* -------------------------------------------- */

  /**
   * Handle a response from the server where a compendium was deleted.
   * @param {ManageCompendiumResponse} response  The server response.
   * @returns {CompendiumCollection}
   */
  static #handleDeleteCompendium({ result }) {
    const pack = game.packs.get(result);
    if ( !pack ) throw new Error(`Compendium pack '${result}' did not exist to be deleted.`);
    game.data.packs.findSplice(p => p.id === result);
    game.packs.delete(result);
    ui.compendium.render();
    return pack;
  }

  /* -------------------------------------------- */

  /**
   * Handle changes to the world compendium configuration setting.
   * @param {WorldCompendiumConfiguration} config
   */
  static _onConfigure(config) {
    const prior = CompendiumCollection.#config;
    CompendiumCollection.#config = foundry.utils.deepClone(config);
    const diff = foundry.utils.diffObject(prior, config);
    let folderChanged = false;
    let sortChanged = false;
    for ( const [id, delta] of Object.entries(diff) ) {
      const pack = game.packs.get(id);
      if ( !pack ) continue;
      if ( "sort" in delta ) sortChanged = true;
      if ( "folder" in delta ) folderChanged = true;
      if ( "ownership" in delta ) pack.initializeTree();
      if ( "locked" in delta ) pack.render(false, { renderContext: "updateConfiguration", renderData: delta });
    }
    if ( folderChanged || sortChanged ) game.packs.initializeTree();
    ui.compendium.render();
  }
}

/**
 * @typedef ProseMirrorHistory
 * @property {string} userId  The ID of the user who submitted the step.
 * @property {Step} step      The step that was submitted.
 */

/**
 * A class responsible for managing state and collaborative editing of a single ProseMirror instance.
 */
class ProseMirrorEditor {
  /**
   * @param {string} uuid                        A string that uniquely identifies this ProseMirror instance.
   * @param {EditorView} view                    The ProseMirror EditorView.
   * @param {Plugin} isDirtyPlugin               The plugin to track the dirty state of the editor.
   * @param {boolean} collaborate                Whether this is a collaborative editor.
   * @param {object} [options]                   Additional options.
   * @param {ClientDocument} [options.document]  A document associated with this editor.
   */
  constructor(uuid, view, isDirtyPlugin, collaborate, options={}) {
    /**
     * A string that uniquely identifies this ProseMirror instance.
     * @type {string}
     */
    Object.defineProperty(this, "uuid", {value: uuid, writable: false});

    /**
     * The ProseMirror EditorView.
     * @type {EditorView}
     */
    Object.defineProperty(this, "view", {value: view, writable: false});

    /**
     * Whether this is a collaborative editor.
     * @type {boolean}
     */
    Object.defineProperty(this, "collaborate", {value: collaborate, writable: false});

    this.options = options;
    this.#isDirtyPlugin = isDirtyPlugin;
  }

  /* -------------------------------------------- */

  /**
   * A list of active editor instances by their UUIDs.
   * @type {Map<string, ProseMirrorEditor>}
   */
  static #editors = new Map();

  /* -------------------------------------------- */

  /**
   * The plugin to track the dirty state of the editor.
   * @type {Plugin}
   */
  #isDirtyPlugin;

  /* -------------------------------------------- */

  /**
   * Retire this editor instance and clean up.
   */
  destroy() {
    ProseMirrorEditor.#editors.delete(this.uuid);
    this.view.destroy();
    if ( this.collaborate ) game.socket.emit("pm.endSession", this.uuid);
  }

  /* -------------------------------------------- */

  /**
   * Have the contents of the editor been edited by the user?
   * @returns {boolean}
   */
  isDirty() {
    return this.#isDirtyPlugin.getState(this.view.state);
  }

  /* -------------------------------------------- */

  /**
   * Handle new editing steps supplied by the server.
   * @param {string} offset                 The offset into the history, representing the point at which it was last
   *                                        truncated.
   * @param {ProseMirrorHistory[]} history  The entire edit history.
   * @protected
   */
  _onNewSteps(offset, history) {
    this._disableSourceCodeEditing();
    this.options.document?.sheet?._onNewSteps?.();
    const version = ProseMirror.collab.getVersion(this.view.state);
    const newSteps = history.slice(version - offset);

    // Flatten out the data into a format that ProseMirror.collab.receiveTransaction can understand.
    const [steps, ids] = newSteps.reduce(([steps, ids], entry) => {
      steps.push(ProseMirror.Step.fromJSON(ProseMirror.defaultSchema, entry.step));
      ids.push(entry.userId);
      return [steps, ids];
    }, [[], []]);

    const tr = ProseMirror.collab.receiveTransaction(this.view.state, steps, ids);
    this.view.dispatch(tr);
  }

  /* -------------------------------------------- */

  /**
   * Disable source code editing if the user was editing it when new steps arrived.
   * @protected
   */
  _disableSourceCodeEditing() {
    const htmlEditor = this.view.dom.closest(".editor")?.querySelector(":scope > code-mirror[language=html]");
    if ( !htmlEditor ) return;
    htmlEditor.disabled = true;
    ui.notifications.warn("EDITOR.EditingHTMLWarning", {localize: true, permanent: true});
  }

  /* -------------------------------------------- */

  /**
   * The state of this ProseMirror editor has fallen too far behind the central authority's and must be re-synced.
   * @protected
   */
  _resync() {
    // Copy the editor's current state to the clipboard to avoid data loss.
    const existing = this.view.dom;
    existing.contentEditable = false;
    const selection = document.getSelection();
    selection.removeAllRanges();
    const range = document.createRange();
    range.selectNode(existing);
    selection.addRange(range);
    // We cannot use navigator.clipboard.write here as it is disabled or not fully implemented in some browsers.
    // https://developer.mozilla.org/en-US/docs/Mozilla/Add-ons/WebExtensions/Interact_with_the_clipboard
    document.execCommand("copy");
    ui.notifications.warn("EDITOR.Resync", {localize: true, permanent: true});
    this.destroy();
    this.options.document?.sheet?.render(true, {resync: true});
  }

  /* -------------------------------------------- */

  /**
   * Handle users joining or leaving collaborative editing.
   * @param {string[]} users  The IDs of users currently editing (including ourselves).
   * @protected
   */
  _updateUserDisplay(users) {
    const editor = this.view.dom.closest(".editor");
    editor.classList.toggle("collaborating", users.length > 1);
    const pips = users.map(id => {
      const user = game.users.get(id);
      if ( !user ) return "";
      return `
        <span class="scene-player" style="background: ${user.color}; border: 1px solid ${user.border.css};">
          ${user.name[0]}
        </span>
      `;
    }).join("");
    const collaborating = editor.querySelector("menu .concurrent-users");
    collaborating.dataset.tooltipText = users.map(id => game.users.get(id)?.name).join(", ");
    collaborating.innerHTML = `
      <i class="fa-solid fa-user-group"></i>
      ${pips}
    `;
  }

  /* -------------------------------------------- */

  /**
   * Handle an autosave update for an already-open editor.
   * @param {string} html  The updated editor contents.
   * @protected
   */
  _handleAutosave(html) {
    this.options.document?.sheet?._onAutosave?.(html);
  }

  /* -------------------------------------------- */

  /**
   * Create a ProseMirror editor instance.
   * @param {HTMLElement} target                     An HTML element to mount the editor to.
   * @param {string} [content=""]                    Content to populate the editor with.
   * @param {object} [options]                       Additional options to configure the ProseMirror instance.
   * @param {string} [options.uuid]                  A string to uniquely identify this ProseMirror instance. Ignored
   *                                                 for a collaborative editor.
   * @param {ClientDocument} [options.document]      A Document whose content is being edited. Required for
   *                                                 collaborative editing and relative UUID generation.
   * @param {string} [options.fieldName]             The field within the Document that is being edited. Required for
   *                                                 collaborative editing.
   * @param {Record<string, Plugin>} [options.plugins]       Plugins to include with the editor.
   * @param {boolean} [options.collaborate=false]    Whether collaborative editing enabled.
   * @param {boolean} [options.relativeLinks=false]  Whether to generate relative UUID links to Documents that are
   *                                                 dropped on the editor.
   * @param {object} [options.props]                 Additional ProseMirror editor properties.
   * @returns {Promise<ProseMirrorEditor>}
   */
  static async create(target, content="", {uuid, document, fieldName, plugins={}, collaborate=false,
    relativeLinks=false, props={}}={}) {

    if ( collaborate && (!document || !fieldName) ) {
      throw new Error("A document and fieldName must be provided when creating an editor with collaborative editing.");
    }

    uuid = collaborate ? `${document.uuid}#${fieldName}` : uuid ?? `ProseMirror.${foundry.utils.randomID()}`;
    const state = ProseMirror.EditorState.create({doc: ProseMirror.dom.parseString(content)});
    plugins = Object.assign({}, ProseMirror.defaultPlugins, plugins);
    plugins.contentLinks = ProseMirror.ProseMirrorContentLinkPlugin.build(ProseMirror.defaultSchema, {
      document, relativeLinks
    });

    if ( document ) {
      plugins.images = ProseMirror.ProseMirrorImagePlugin.build(ProseMirror.defaultSchema, {document});
    }

    const options = {state};
    Hooks$1.callAll("createProseMirrorEditor", uuid, plugins, options);

    const view = collaborate
      ? await this._createCollaborativeEditorView(uuid, target, options.state, Object.values(plugins), props)
      : this._createLocalEditorView(target, options.state, Object.values(plugins), props);
    const editor = new ProseMirrorEditor(uuid, view, plugins.isDirty, collaborate, {document});
    ProseMirrorEditor.#editors.set(uuid, editor);
    return editor;
  }

  /* -------------------------------------------- */

  /**
   * Create an EditorView with collaborative editing enabled.
   * @param {string} uuid         The ProseMirror instance UUID.
   * @param {HTMLElement} target  An HTML element to mount the editor view to.
   * @param {EditorState} state   The ProseMirror editor state.
   * @param {Plugin[]} plugins    The ProseMirror editor plugins to load.
   * @param {object} props        Additional ProseMirror editor properties.
   * @returns {Promise<EditorView>}
   * @protected
   */
  static async _createCollaborativeEditorView(uuid, target, state, plugins, props) {
    const authority = await new Promise((resolve, reject) => {
      game.socket.emit("pm.editDocument", uuid, state, authority => {
        if ( authority.state ) resolve(authority);
        else reject();
      });
    });
    return new ProseMirror.EditorView({mount: target}, {
      ...props,
      state: ProseMirror.EditorState.fromJSON({
        schema: ProseMirror.defaultSchema,
        plugins: [
          ...plugins,
          ProseMirror.collab.collab({version: authority.version, clientID: game.userId})
        ]
      }, authority.state),
      dispatchTransaction(tr) {
        const newState = this.state.apply(tr);
        this.updateState(newState);
        const sendable = ProseMirror.collab.sendableSteps(newState);
        if ( sendable ) game.socket.emit("pm.receiveSteps", uuid, sendable.version, sendable.steps);
      }
    });
  }

  /* -------------------------------------------- */

  /**
   * Create a plain EditorView without collaborative editing.
   * @param {HTMLElement} target  An HTML element to mount the editor view to.
   * @param {EditorState} state   The ProseMirror editor state.
   * @param {Plugin[]} plugins    The ProseMirror editor plugins to load.
   * @param {object} props        Additional ProseMirror editor properties.
   * @returns {EditorView}
   * @protected
   */
  static _createLocalEditorView(target, state, plugins, props) {
    return new ProseMirror.EditorView({mount: target}, {
      ...props, state: ProseMirror.EditorState.create({doc: state.doc, plugins}),
    });
  }

  /* -------------------------------------------- */
  /*  Socket Handlers                             */
  /* -------------------------------------------- */

  /**
   * Handle new editing steps supplied by the server.
   * @param {string} uuid                   The UUID that uniquely identifies the ProseMirror instance.
   * @param {number} offset                 The offset into the history, representing the point at which it was last
   *                                        truncated.
   * @param {ProseMirrorHistory[]} history  The entire edit history.
   * @protected
   */
  static _onNewSteps(uuid, offset, history) {
    const editor = ProseMirrorEditor.#editors.get(uuid);
    if ( editor ) editor._onNewSteps(offset, history);
    else {
      console.warn(`New steps were received for UUID '${uuid}' which is not a ProseMirror editor instance.`);
    }
  }

  /* -------------------------------------------- */

  /**
   * Our client is too far behind the central authority's state and must be re-synced.
   * @param {string} uuid  The UUID that uniquely identifies the ProseMirror instance.
   * @protected
   */
  static _onResync(uuid) {
    const editor = ProseMirrorEditor.#editors.get(uuid);
    if ( editor ) editor._resync();
    else {
      console.warn(`A resync request was received for UUID '${uuid}' which is not a ProseMirror editor instance.`);
    }
  }

  /* -------------------------------------------- */

  /**
   * Handle users joining or leaving collaborative editing.
   * @param {string} uuid       The UUID that uniquely identifies the ProseMirror instance.
   * @param {string[]} users    The IDs of the users editing (including ourselves).
   * @protected
   */
  static _onUsersEditing(uuid, users) {
    const editor = ProseMirrorEditor.#editors.get(uuid);
    if ( editor ) editor._updateUserDisplay(users);
    else {
      console.warn(`A user update was received for UUID '${uuid}' which is not a ProseMirror editor instance.`);
    }
  }

  /* -------------------------------------------- */

  /**
   * Update client state when the editor contents are autosaved server-side.
   * @param {string} uuid  The UUID that uniquely identifies the ProseMirror instance.
   * @param {string} html  The updated editor contents.
   * @protected
   */
  static async _onAutosave(uuid, html) {
    const editor = ProseMirrorEditor.#editors.get(uuid);
    const [docUUID, field] = uuid?.split("#") ?? [];
    const doc = await foundry.utils.fromUuid(docUUID);
    if ( doc ) doc.updateSource({[field]: html});
    if ( editor ) editor._handleAutosave(html);
    else doc.render(false);
  }

  /* -------------------------------------------- */

  /**
   * Listen for ProseMirror collaboration events.
   * @param {Socket} socket  The open websocket.
   * @internal
   */
  static _activateSocketListeners(socket) {
    socket.on("pm.newSteps", this._onNewSteps.bind(this));
    socket.on("pm.resync", this._onResync.bind(this));
    socket.on("pm.usersEditing", this._onUsersEditing.bind(this));
    socket.on("pm.autosave", this._onAutosave.bind(this));
  }
}

/**
 * @import Application from "@client/appv1/api/application-v1.mjs"
 */

/**
 * @typedef ContextMenuEntry
 * @property {string} name                              The context menu label. Can be localized.
 * @property {string} [icon]                            A string containing an HTML icon element for the menu item.
 * @property {string} [classes]                         Additional CSS classes to apply to this menu item.
 * @property {string} [group]                           An identifier for a group this entry belongs to.
 * @property {ContextMenuJQueryCallback} callback       The function to call when the menu item is clicked.
 * @property {ContextMenuCondition|boolean} [condition] A function to call or boolean value to determine if this entry
 *                                                      appears in the menu.
 */

/**
 * @callback ContextMenuCondition
 * @param {jQuery|HTMLElement} html                     The element of the context menu entry.
 * @returns {boolean}                                   Whether the entry should be rendered in the context menu.
 */

/**
 * @callback ContextMenuCallback
 * @param {HTMLElement} target                          The element that the context menu has been triggered for.
 * @returns {unknown}
 */

/**
 * @callback ContextMenuJQueryCallback
 * @param {HTMLElement|jQuery} target                   The element that the context menu has been triggered for. Will
 *                                                      either be a jQuery object or an HTMLElement instance, depending
 *                                                      on how the ContextMenu was configured.
 * @returns {unknown}
 */

/**
 * @typedef ContextMenuOptions
 * @property {string} [eventName="contextmenu"] Optionally override the triggering event which can spawn the menu. If
 *                                              the menu is using fixed positioning, this event must be a MouseEvent.
 * @property {ContextMenuCallback} [onOpen]     A function to call when the context menu is opened.
 * @property {ContextMenuCallback} [onClose]    A function to call when the context menu is closed.
 * @property {boolean} [fixed=false]            If true, the context menu is given a fixed position rather than being
 *                                              injected into the target.
 * @property {boolean} [jQuery=true]            If true, callbacks will be passed jQuery objects instead of HTMLElement
 *                                              instances.
 */

/**
 * @typedef ContextMenuRenderOptions
 * @property {Event} [event]           The event that triggered the context menu opening.
 * @property {boolean} [animate=true]  Animate the context menu opening.
 */

/**
 * Display a right-click activated Context Menu which provides a dropdown menu of options.
 * A ContextMenu is constructed by designating a parent HTML container and a target selector.
 * An Array of menuItems defines the entries of the menu which is displayed.
 */
class ContextMenu {
  /**
   * @param {HTMLElement|jQuery} container              The HTML element that contains the context menu targets.
   * @param {string} selector                           A CSS selector which activates the context menu.
   * @param {ContextMenuEntry[]} menuItems              An Array of entries to display in the menu
   * @param {ContextMenuOptions} [options]              Additional options to configure the context menu.
   */
  constructor(container, selector, menuItems, {eventName="contextmenu", onOpen, onClose, jQuery, fixed=false}={}) {
    if ( jQuery === undefined ) {
      foundry.utils.logCompatibilityWarning("ContextMenu is changing to no longer transact jQuery objects for menu"
        + " item callbacks. Because the jQuery option provided to the ContextMenu constructor was undefined, your"
        + " callbacks will receive jQuery objects. You may opt-out and receive HTMLElement references instead by"
        + " passing jQuery: false to the constructor. This parameter will be false by default in v14 and deprecated"
        + " entirely in v15 at which point only HTMLElement references will be used.",
      { since: 13, until: 15, once: true });
      jQuery = true;
    }

    // Accept HTMLElement or jQuery (for now)
    if ( !(container instanceof HTMLElement) ) {
      foundry.utils.logCompatibilityWarning("ContextMenu is changing to no longer transact jQuery objects."
        + " You must begin passing an HTMLElement instead.", { since: 13, until: 15, once: true });
      container = container[0];
    }

    // Assign attributes
    this.#container = container;
    this.#selector = selector || container.id;
    this.#eventName = eventName;
    this.menuItems = menuItems;
    this.onOpen = onOpen;
    this.onClose = onClose;
    this.#fixed = fixed;
    /** @deprecated since v13 until v15 */
    this.#jQuery = jQuery;

    // Bind to the container.
    this.#container.addEventListener(this.eventName, this._onActivate.bind(this));
  }

  /* -------------------------------------------- */

  /**
   * Create a ContextMenu for this Application and dispatch hooks.
   * @param {Application} app                           The Application this ContextMenu belongs to.
   * @param {JQuery|HTMLElement} html                   The Application's rendered HTML.
   * @param {string} selector                           The target CSS selector which activates the menu.
   * @param {ContextMenuEntry[]} menuItems              The array of menu items being rendered.
   * @param {object} [options]                          Additional options to configure context menu initialization.
   * @param {string} [options.hookName="EntryContext"]  The name of the hook to call.
   * @returns {ContextMenu}
   * @deprecated since v13
   */
  static create(app, html, selector, menuItems, {hookName="EntryContext", ...options}={}) {
    if ( app instanceof foundry.applications.api.ApplicationV2 ) {
      throw new Error("ContextMenu.create is deprecated and only supports Application (v1) instances.");
    }
    app._callHooks?.(className => `get${className}${hookName}`, menuItems);
    return new this(html, selector, menuItems, options);
  }

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * The HTML element that contains the context menu targets.
   * @type {HTMLElement}
   */
  #container;

  /**
   * The menu element.
   * @type {HTMLElement}
   */
  get element() {
    return this.#element;
  }

  #element;

  /**
   * A CSS selector to identify context menu targets.
   * @type {string}
   */
  get selector() {
    return this.#selector;
  }

  #selector;

  /**
   * The event name to listen for.
   * @type {string}
   */
  get eventName() {
    return this.#eventName;
  }

  #eventName;

  /**
   * The array of menu items to render.
   * @type {Array<ContextMenuEntry & {element: HTMLElement}>}
   */
  menuItems;

  /**
   * A function to call when the context menu is opened.
   * @type {ContextMenuCallback}
   */
  onOpen;

  /**
   * A function to call when the context menu is closed.
   * @type {ContextMenuCallback}
   */
  onClose;

  /**
   * Check which direction the menu is expanded in.
   * @type {boolean}
   */
  get expandUp() {
    return this.#expandUp;
  }

  #expandUp = false;

  /**
   * Whether to position the context menu as a fixed element, or inject it into the target.
   * @type {boolean}
   */
  get fixed() {
    return this.#fixed;
  }

  #fixed;

  /**
   * Whether to pass jQuery objects or HTMLElement instances to callback.
   * @type {boolean}
   */
  #jQuery;

  /**
   * The parent HTML element to which the context menu is attached
   * @type {HTMLElement}
   */
  get target() {
    return this.#target;
  }

  #target;

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /**
   * Animate the context menu's height when opening or closing.
   * @param {boolean} open      Whether the menu is opening or closing.
   * @returns {Promise<void>}   A Promise that resolves when the animation completes.
   * @protected
   */
  async _animate(open=false) {
    const { height } = this.#element.getBoundingClientRect();
    const from = open ? "0" : `${height}px`;
    const to = open ? `${height}px` : "0";
    Object.assign(this.#element.style, { height: from, padding: "0", overflow: "hidden" });
    await this.#element.animate({ height: [from, to] }, { duration: 200, easing: "ease", fill: "forwards" }).finished;
    if ( open ) Object.assign(this.#element.style, { height: "", padding: "", overflow: "" });
  }

  /* -------------------------------------------- */

  /**
   * Closes the menu and removes it from the DOM.
   * @param {object} [options]                Options to configure the closing behavior.
   * @param {boolean} [options.animate=true]  Animate the context menu closing.
   * @param {HTMLElement} [options.target]    The target element to close on.
   * @returns {Promise<void>}
   */
  async close({animate=true, target}={}) {
    if ( animate ) await this._animate(false);
    this._close({ target });
  }

  /* -------------------------------------------- */

  /**
   * Close the menu and remove it from the DOM.
   * @param {object} [options]
   * @param {HTMLElement} [options.target]  The target element to close on.
   * @protected
   */
  _close({ target }={}) {
    for ( const item of this.menuItems ) delete item.element;
    this.#element.remove();
    document.querySelectorAll(".context").forEach(el => el.classList.remove("context"));
    if ( ui.context === this ) delete ui.context;
    this.onClose?.(target ?? this.#target);
  }

  /* -------------------------------------------- */

  /**
   * Called before the context menu begins rendering.
   * @param {HTMLElement} target  The context target.
   * @param {ContextMenuRenderOptions} [options]
   * @returns {Promise<void>}
   * @protected
   */
  async _preRender(target, options={}) {}

  /* -------------------------------------------- */

  /**
   * Render the Context Menu by iterating over the menuItems it contains.
   * Check the visibility of each menu item, and only render ones which are allowed by the item's logical condition.
   * Attach a click handler to each item which is rendered.
   * @param {HTMLElement} target  The target element to which the context menu is attached.
   * @param {ContextMenuRenderOptions} [options]
   * @returns {Promise<void>}     A Promise that resolves when the open animation has completed.
   */
  async render(target, options={}) {
    await this._preRender(target, options);
    this.#element?.remove();
    const html = this.#element = document.createElement("nav");
    html.id = "context-menu";
    const menu = document.createElement("menu");
    menu.classList.add("context-items");
    html.replaceChildren(menu);

    if ( !this.menuItems.length ) return;

    /** @type {Record<string, ContextMenuEntry>} */
    const groups = this.menuItems.reduce((acc, entry) => {
      const group = entry.group ?? "_none";
      acc[group] ??= [];
      // Determine menu item visibility (display unless false)
      let display = true;
      if ( entry.condition !== undefined ) {
        if ( entry.condition instanceof Function ) display = entry.condition(this.#jQuery ? $(target) : target);
        else display = entry.condition;
      }
      if ( display ) acc[group].push(entry);
      return acc;
    }, {});

    for ( const [group, entries] of Object.entries(groups) ) {
      let parent = menu;
      if ( (group !== "_none") && entries.length ) {
        const item = document.createElement("li");
        item.classList.add("context-group");
        item.dataset.groupId = group;
        const list = document.createElement("ol");
        item.append(list);
        menu.append(item);
        parent = list;
      }
      for ( const item of entries ) {
        // Construct and add the menu item
        const name = game.i18n.localize(item.name);
        const classes = ["context-item", item.classes].filterJoin(" ");
        const entry = document.createElement("li");
        entry.className = classes;
        if ( item.icon ) {
          entry.insertAdjacentHTML("afterbegin", item.icon);
          entry.querySelector("i")?.classList.add("fa-fw");
        }
        const span = document.createElement("span");
        span.append(name);
        entry.append(span);
        parent.append(entry);

        // Record a reference to the element.
        item.element = entry;
      }
    }

    // Bail out if there are no children
    if ( !menu.children.length ) return;

    // Append to target
    this._setPosition(html, target, options);

    // Apply interactivity
    this.activateListeners(html);

    // Deactivate global tooltip
    game.tooltip.deactivate();

    // Animate open the menu
    if ( options.animate !== false ) await this._animate(true);
    return this._onRender(options);
  }

  /* -------------------------------------------- */

  /**
   * Called after the context menu has finished rendering and animating open.
   * @param {ContextMenuRenderOptions} [options]
   * @returns {Promise<void>}
   * @protected
   */
  async _onRender(options={}) {}

  /* -------------------------------------------- */

  /**
   * Set the position of the context menu, taking into consideration whether the menu should expand upward or downward
   * @param {HTMLElement} menu       The context menu element.
   * @param {HTMLElement} target     The element that the context menu was spawned on.
   * @param {object} [options]
   * @param {Event} [options.event]  The event that triggered the context menu opening.
   * @protected
   */
  _setPosition(menu, target, { event }={}) {
    if ( this.#fixed ) this._setFixedPosition(menu, target, { event });
    else this._injectMenu(menu, target);
  }

  /* -------------------------------------------- */

  /**
   * Inject the menu inside the target.
   * @param {HTMLElement} menu    The menu element.
   * @param {HTMLElement} target  The context target.
   * @protected
   */
  _injectMenu(menu, target) {
    const container = target.parentElement;

    // Append to target and get the context bounds
    target.style.position = "relative";
    menu.style.visibility = "hidden";
    target.append(menu);
    const menuRect = menu.getBoundingClientRect();
    const parentRect = target.getBoundingClientRect();
    const containerRect = container.getBoundingClientRect();

    // Determine whether to expand upwards
    const menuTop = parentRect.top - menuRect.height;
    const menuBottom = parentRect.bottom + menuRect.height;
    const canOverflowUp = (menuTop > containerRect.top) || (getComputedStyle(container).overflowY === "visible");

    // If it overflows the container bottom, but not the container top
    const containerUp = (menuBottom > containerRect.bottom) && (menuTop >= containerRect.top);
    const windowUp = (menuBottom > window.innerHeight) && (menuTop > 0) && canOverflowUp;
    this.#expandUp = containerUp || windowUp;

    // Display the menu
    menu.classList.toggle("expand-up", this.#expandUp);
    menu.classList.toggle("expand-down", !this.#expandUp);
    menu.style.visibility = "";
    target.classList.add("context");
  }

  /* -------------------------------------------- */

  /**
   * Set the context menu at a fixed position in the viewport.
   * @param {HTMLElement} menu       The menu element.
   * @param {HTMLElement} target     The context target.
   * @param {object} [options]
   * @param {Event} [options.event]  The event that triggered the context menu opening.
   * @protected
   */
  _setFixedPosition(menu, target, { event }={}) {
    let { clientX, clientY } = event ?? {};

    // Bail early if it won't be possible to position the menu.
    const needsCoords = [clientX, clientY].includes(undefined) || (!event.isTrusted && [clientX, clientY].includes(0));
    if ( needsCoords && !target.checkVisibility() ) return;

    menu.setAttribute("popover", "manual");
    document.body.appendChild(menu);
    menu.showPopover();
    const { clientWidth, clientHeight } = document.documentElement;
    const { width, height } = menu.getBoundingClientRect();

    if ( needsCoords ) {
      // If an event was either not provided or without meaningful clientX/clientY co-ordinates, set the co-ordinates to
      // the bottom-left of the target.
      ({ left: clientX, bottom: clientY } = target.getBoundingClientRect());
    }

    menu.style.left = `${(Math.min(clientX, clientWidth - width))}px`;
    this.#expandUp = (clientY + height) > clientHeight;
    if ( this.#expandUp ) menu.style.bottom = `${clientHeight - clientY}px`;
    else menu.style.top = `${clientY}px`;
    menu.classList.toggle("expand-up", this.#expandUp);
    menu.classList.toggle("expand-down", !this.#expandUp);
    target.classList.add("context");

    const nearestThemed = target.closest(".themed") ?? document.body;
    const [, theme] = nearestThemed.className.match(/(?:^|\s)(theme-\w+)/) ?? [];
    if ( theme ) menu.classList.add("themed", theme);
  }

  /* -------------------------------------------- */
  /*  Event Listeners & Handlers                  */
  /* -------------------------------------------- */

  /**
   * Local listeners which apply to each ContextMenu instance which is created.
   * @param {HTMLElement} menu  The context menu element.
   */
  activateListeners(menu) {
    menu.addEventListener("click", this.#onClickItem.bind(this));
  }

  /* -------------------------------------------- */

  /**
   * Handle context menu activation.
   * @param {Event} event  The triggering event.
   * @protected
   */
  _onActivate(event) {
    const matching = event.target.closest(this.#selector);
    if ( !matching ) return;
    event.preventDefault();
    const priorTarget = this.#target;
    this.#target = matching;

    // Remove existing context UI.
    if ( this.#target.classList.contains("context") ) return this.close();

    // If the menu is already open, call its close handler on its original target.
    const closeOptions = { animate: ui.context !== this };
    if ( ui.context === this ) closeOptions.target = priorTarget;
    ui.context?.close(closeOptions);

    // Render a new context menu.
    event.stopImmediatePropagation();
    ui.context = this;
    this.onOpen?.(this.#target);
    return this.render(this.#target, { event });
  }

  /* -------------------------------------------- */

  /**
   * Handle click events on context menu items.
   * @param {PointerEvent} event      The click event
   */
  #onClickItem(event) {
    event.preventDefault();
    event.stopPropagation();
    const element = event.target.closest(".context-item");
    if ( !element ) return;
    const item = this.menuItems.find(i => i.element === element);
    item?.callback(this.#jQuery ? $(this.#target) : this.#target);
    this.close();
  }

  /* -------------------------------------------- */

  /**
   * Global listeners which apply once only to the document.
   */
  static eventListeners() {
    document.addEventListener("click", () => ui.context?.close(), { passive: true });
  }

  /* -------------------------------------------- */
  /*  Factory Methods                             */
  /* -------------------------------------------- */

  /**
   * Retrieve the configured DragDrop implementation.
   * @type {typeof ContextMenu}
   */
  static get implementation() {
    let Class = CONFIG.ux.ContextMenu;
    if ( !foundry.utils.isSubclass(Class, ContextMenu) ) {
      console.warn("Configured ContextMenu override must be a subclass of ContextMenu.");
      Class = ContextMenu;
    }
    return Class;
  }

  /* -------------------------------------------- */
  /*  Deprecations                                */
  /* -------------------------------------------- */

  /**
   * @deprecated since v13 until v15
   * @ignore
   */
  get _expandUp() {
    foundry.utils.logCompatibilityWarning("ContextMenu#_expandUp is deprecated. Please use ContextMenu#expandUp "
      + "instead.", { since: 13, until: 15, once: true });
    return this.#expandUp;
  }

  /* -------------------------------------------- */

  /**
   * @deprecated since v13 until v15
   * @ignore
   */
  get menu() {
    foundry.utils.logCompatibilityWarning("ContextMenu#menu is deprecated. "
      + "Please use ContextMenu#element instead.", { since: 13, until: 15, once: true });
    return $("#context-menu");
  }
}

/**
 * An extension of the native FormData implementation.
 *
 * This class functions the same way that the default FormData does, but it is more opinionated about how
 * input fields of certain types should be evaluated and handled.
 *
 * It also adds support for certain Foundry VTT specific concepts including:
 *  Support for defined data types and type conversion
 *  Support for TinyMCE editors
 *  Support for editable HTML elements
 *
 * @extends {FormData}
 *
 * @param {HTMLFormElement} form          The form being processed
 * @param {object} options                Options which configure form processing
 * @param {Record<string, object>} [options.editors]      A record of TinyMCE editor metadata objects, indexed by their update key
 * @param {Record<string, string>} [options.dtypes]       A mapping of data types for form fields
 * @param {boolean} [options.disabled=false]      Include disabled fields?
 * @param {boolean} [options.readonly=false]      Include readonly fields?
 */
class FormDataExtended extends FormData {
  constructor(form, {dtypes={}, editors={}, disabled=false, readonly=true}={}) {
    super();

    /**
     * A mapping of data types requested for each form field.
     * @type {{string, string}}
     */
    this.dtypes = dtypes;

    /**
     * A record of TinyMCE editors which are linked to this form.
     * @type {Record<string, object>}
     */
    this.editors = editors;

    /**
     * The object representation of the form data, available once processed.
     * @type {object}
     */
    Object.defineProperty(this, "object", {value: {}, writable: false, enumerable: false});

    // Process the provided form
    this.process(form, {disabled, readonly});
  }

  /* -------------------------------------------- */

  /**
   * Process the HTML form element to populate the FormData instance.
   * @param {HTMLFormElement} form    The HTML form being processed
   * @param {object} options          Options forwarded from the constructor
   */
  process(form, options) {
    this.#processFormFields(form, options);
    this.#processEditableHTML(form, options);
    this.#processEditors();

    // Emit the formdata event for compatibility with the parent FormData class
    form.dispatchEvent(new FormDataEvent("formdata", {formData: this}));
  }

  /* -------------------------------------------- */

  /**
   * Assign a value to the FormData instance which always contains JSON strings.
   * Also assign the cast value in its preferred data type to the parsed object representation of the form data.
   * @param {string} name     The field name
   * @param {any} value       The raw extracted value from the field
   * @override
   */
  set(name, value) {
    this.object[name] = value;
    if ( value instanceof Array ) value = JSON.stringify(value);
    super.set(name, value);
  }

  /* -------------------------------------------- */

  /**
   * Append values to the form data, adding them to an array.
   * @param {string} name     The field name to append to the form
   * @param {any} value       The value to append to the form data
   * @override
   */
  append(name, value) {
    if ( name in this.object ) {
      if ( !Array.isArray(this.object[name]) ) this.object[name] = [this.object[name]];
    }
    else this.object[name] = [];
    this.object[name].push(value);
    super.append(name, value);
  }

  /* -------------------------------------------- */

  /**
   * Process all standard HTML form field elements from the form.
   * @param {HTMLFormElement} form    The form being processed
   * @param {object} options          Options forwarded from the constructor
   * @param {boolean} [options.disabled]    Process disabled fields?
   * @param {boolean} [options.readonly]    Process readonly fields?
   */
  #processFormFields(form, {disabled, readonly}={}) {
    if ( !disabled && form.hasAttribute("disabled") ) return;
    const mceEditorIds = Object.values(this.editors).map(e => e.mce?.id);
    for ( const element of form.elements ) {
      const name = element.name;

      // Skip fields which are unnamed or already handled
      if ( !name || this.has(name) ) continue;

      // Skip buttons and editors
      if ( (element.tagName === "BUTTON") || mceEditorIds.includes(name) ) continue;

      // Skip disabled or read-only fields
      if ( !disabled && element.matches(":disabled") ) continue;
      if ( !readonly && element.readOnly ) continue;

      // Extract and process the value of the field
      const field = form.elements.namedItem(name);
      const value = this.#getFieldValue(name, field);
      this.set(name, value);
    }
  }

  /* -------------------------------------------- */

  /**
   * Process editable HTML elements (ones with a [data-edit] attribute).
   * @param {HTMLFormElement} form    The form being processed
   * @param {object} options          Options forwarded from the constructor
   * @param {boolean} [options.disabled]    Process disabled fields?
   * @param {boolean} [options.readonly]    Process readonly fields?
   */
  #processEditableHTML(form, {disabled, readonly}={}) {
    const editableElements = form.querySelectorAll("[data-edit]");
    for ( const element of editableElements ) {
      const name = element.dataset.edit;
      if ( this.has(name) || (name in this.editors) ) continue;
      if ( (!disabled && element.disabled) || (!readonly && element.readOnly) ) continue;
      let value;
      if (element.tagName === "IMG") value = element.getAttribute("src");
      else value = element.innerHTML.trim();
      this.set(name, value);
    }
  }

  /* -------------------------------------------- */

  /**
   * Process TinyMCE editor instances which are present in the form.
   */
  #processEditors() {
    for ( const [name, editor] of Object.entries(this.editors) ) {
      if ( !editor.instance ) continue;
      if ( editor.options.engine === "tinymce" ) {
        const content = editor.instance.getContent();
        this.delete(editor.mce.id); // Delete hidden MCE inputs
        this.set(name, content);
      } else if ( editor.options.engine === "prosemirror" ) {
        this.set(name, ProseMirror.dom.serializeString(editor.instance.view.state.doc.content));
      }
    }
  }

  /* -------------------------------------------- */

  /**
   * Obtain the parsed value of a field conditional on its element type and requested data type.
   * @param {string} name                       The field name being processed
   * @param {HTMLElement|RadioNodeList} field   The HTML field or a RadioNodeList of multiple fields
   * @returns {*}                               The processed field value
   */
  #getFieldValue(name, field) {

    // Multiple elements with the same name
    if ( field instanceof RadioNodeList ) {
      const fields = Array.from(field);
      if ( fields.every(f => f.type === "radio") ) {
        const chosen = fields.find(f => f.checked);
        return chosen ? this.#getFieldValue(name, chosen) : undefined;
      }
      return Array.from(field).map(f => this.#getFieldValue(name, f));
    }

    // Record requested data type
    const dataType = field.dataset.dtype || this.dtypes[name];

    // Checkbox
    if ( field.type === "checkbox" ) {

      // Non-boolean checkboxes with an explicit value attribute yield that value or null
      if ( field.hasAttribute("value") && (dataType !== "Boolean") ) {
        return this.#castType(field.checked ? field.value : null, dataType);
      }

      // Otherwise, true or false based on the checkbox checked state
      return this.#castType(field.checked, dataType);
    }

    // Number and Range
    if ( ["number", "range"].includes(field.type) ) {
      if ( field.value === "" ) return null;
      else return this.#castType(field.value, dataType || "Number");
    }

    // Multi-Select
    if ( field.type === "select-multiple" ) {
      return Array.from(field.options).reduce((chosen, opt) => {
        if ( opt.selected ) chosen.push(this.#castType(opt.value, dataType));
        return chosen;
      }, []);
    }

    // Radio Select
    if ( field.type === "radio" ) {
      return field.checked ? this.#castType(field.value, dataType) : null;
    }

    // Other field types
    return this.#castType(field.value, dataType);
  }

  /* -------------------------------------------- */

  /**
   * Cast a processed value to a desired data type.
   * @param {any} value         The raw field value
   * @param {string} dataType   The desired data type
   * @returns {any}             The resulting data type
   */
  #castType(value, dataType) {
    if ( value instanceof Array ) return value.map(v => this.#castType(v, dataType));
    if ( [undefined, null].includes(value) || (dataType === "String") ) return value;

    // Boolean
    if ( dataType === "Boolean" ) {
      if ( value === "false" ) return false;
      return Boolean(value);
    }

    // Number
    else if ( dataType === "Number" ) {
      if ( (value === "") || (value === "null") ) return null;
      return Number(value);
    }

    // Serialized JSON
    else if ( dataType === "JSON" ) {
      return JSON.parse(value);
    }

    // Other data types
    if ( window[dataType] instanceof Function ) {
      try {
        return window[dataType](value);
      } catch(err) {
        console.warn(`The form field value "${value}" was not able to be cast to the requested data type ${dataType}`);
      }
    }
    return value;
  }
}

/**
 * @import {
 *   ApplicationClosingOptions,
 *   ApplicationConfiguration,
 *   ApplicationHeaderControlsEntry,
 *   ApplicationPosition,
 *   ApplicationRenderContext,
 *   ApplicationRenderOptions,
 *   ApplicationTab,
 *   ApplicationTabsConfiguration
 * } from "../_types.mjs";
 */

/**
 * @typedef {typeof ApplicationV2.RENDER_STATES[keyof typeof ApplicationV2.RENDER_STATES]} ApplicationRenderState
 */

/**
 * The Application class is responsible for rendering an HTMLElement into the Foundry Virtual Tabletop user interface.
 * @template {ApplicationConfiguration} [Configuration=ApplicationConfiguration]
 * @template {ApplicationRenderOptions} [RenderOptions=ApplicationRenderOptions]
 */
class ApplicationV2 extends EventEmitterMixin() {

  /**
   * Applications are constructed by providing an object of configuration options.
   * @param {Partial<Configuration>} [options]    Options used to configure the Application instance
   */
  constructor(options={}) {
    super();

    // Configure Application Options
    this.options = Object.freeze(this._initializeApplicationOptions(options));
    this.#id = this.options.id.replace("{id}", this.options.uniqueId);
    Object.assign(this.#position, this.options.position);

    // Verify the Application class is renderable
    this.#renderable = (this._renderHTML !== ApplicationV2.prototype._renderHTML)
      && (this._replaceHTML !== ApplicationV2.prototype._replaceHTML);
  }

  /**
   * Designates which upstream Application class in this class' inheritance chain is the base application.
   * Any DEFAULT_OPTIONS of super-classes further upstream of the BASE_APPLICATION are ignored.
   * Hook events for super-classes further upstream of the BASE_APPLICATION are not dispatched.
   * @type {typeof ApplicationV2}
   */
  static BASE_APPLICATION = ApplicationV2;

  /**
   * The default configuration options which are assigned to every instance of this Application class.
   * @type {Partial<Configuration>}
   */
  static DEFAULT_OPTIONS = {
    id: "app-{id}",
    classes: [],
    tag: "div",
    window: {
      frame: true,
      positioned: true,
      title: "",
      icon: "",
      controls: [],
      minimizable: true,
      resizable: false,
      contentTag: "section",
      contentClasses: []
    },
    actions: {},
    form: {
      handler: undefined,
      submitOnChange: false,
      closeOnSubmit: false
    },
    position: {
      width: "auto",
      height: "auto"
    }
  };

  /**
   * Configuration of application tabs, with an entry per tab group.
   * @type {Record<string, ApplicationTabsConfiguration>}
   */
  static TABS = {};

  /**
   * The sequence of rendering states that describe the Application life-cycle.
   * @type {Record<string, number>}
   */
  static RENDER_STATES = Object.freeze({
    ERROR: -3,
    CLOSING: -2,
    CLOSED: -1,
    NONE: 0,
    RENDERING: 1,
    RENDERED: 2
  });

  /**
   * An incrementing integer Application ID.
   * @type {number}
   * @internal
   */
  static _appId = 0;

  /**
   * The current maximum z-index of any displayed Application.
   * @type {number}
   * @internal
   */
  static _maxZ = Number(getComputedStyle(document.body).getPropertyValue("--z-index-window") ?? 100);

  /**
   * Which application is currently "in front" with the maximum z-index
   * @type {ApplicationV2}
   */
  static #frontApp;

  /** @override */
  static emittedEvents = Object.freeze(["render", "close", "position"]);

  /**
   * Initial values of the #window object.
   * @type {object}
   */
  static #INITIAL_WINDOW_VALUES = {
    header: undefined,
    title: undefined,
    icon: undefined,
    resize: undefined,
    close: undefined,
    content: undefined,
    controls: undefined,
    controlsDropdown: undefined,
    pointerStartPosition: undefined,
    pointerMoveThrottle: false
  };

  /* -------------------------------------------- */

  /**
   * Application instance configuration options.
   * @type {Readonly<Configuration>}
   */
  options;

  /**
   * @type {string}
   */
  #id;

  /**
   * Flag that this Application instance is renderable.
   * Applications are not renderable unless a subclass defines the _renderHTML and _replaceHTML methods.
   */
  #renderable = true;

  /**
   * The outermost HTMLElement of this rendered Application.
   * For window applications this is ApplicationV2##frame.
   * For non-window applications this ApplicationV2##content.
   * @type {HTMLElement}
   */
  #element;

  /**
   * The HTMLElement within which inner HTML is rendered.
   * For non-window applications this is the same as ApplicationV2##element.
   * @type {HTMLElement}
   */
  #content;

  /**
   * Data pertaining to the minimization status of the Application.
   * @type {{
   *  active: boolean,
   *  priorWidth?: number,
   *  priorHeight?: number,
   *  priorBoundingWidth?: number,
   *  priorBoundingHeight?: number
   * }}
   */
  #minimization = Object.seal({
    active: false,
    priorWidth: undefined,
    priorHeight: undefined,
    priorBoundingWidth: undefined,
    priorBoundingHeight: undefined
  });

  /**
   * The rendered position of the Application.
   * @type {ApplicationPosition}
   */
  #position = Object.seal({
    top: undefined,
    left: undefined,
    width: undefined,
    height: "auto",
    scale: 1,
    zIndex: ApplicationV2._maxZ
  });

  /**
   * @type {ApplicationRenderState}
   */
  #state = ApplicationV2.RENDER_STATES.NONE;

  /**
   * A Semaphore used to enqueue asynchronous operations.
   * @type {Semaphore}
   */
  #semaphore = new Semaphore(1);

  /**
   * Convenience references to window header elements.
   * @type {{
   *  header: HTMLElement,
   *  resize: HTMLElement,
   *  title: HTMLHeadingElement,
   *  icon: HTMLElement,
   *  close: HTMLButtonElement,
   *  controls: HTMLButtonElement,
   *  content: HTMLElement,
   *  controlsDropdown: HTMLDivElement,
   *  onDrag: Function,
   *  onResize: Function,
   *  pointerStartPosition: ApplicationPosition,
   *  pointerMoveThrottle: boolean
   * }}
   */
  get window() {
    return this.#window;
  }

  #window = {
    ...ApplicationV2.#INITIAL_WINDOW_VALUES,
    onDrag: this.#onWindowDragMove.bind(this),
    onResize: this.#onWindowResizeMove.bind(this)
  };

  /**
   * If this Application uses tabbed navigation groups, this mapping is updated whenever the changeTab method is called.
   * Reports the active tab for each group, with a value of `null` indicating no tab is active.
   * Subclasses may override this property to define default tabs for each group.
   * @type {Record<string, string|null>}
   */
  tabGroups = Object.entries(this.constructor.TABS).reduce((obj, [id, {initial}]) => {
    obj[id] = initial || null;
    return obj;
  }, {});

  /* -------------------------------------------- */
  /*  Application Properties                      */
  /* -------------------------------------------- */

  /**
   * The CSS class list of this Application instance
   * @type {DOMTokenList}
   */
  get classList() {
    return this.#element?.classList;
  }

  /**
   * The HTML element ID of this Application instance.
   * This provides a readonly view into the internal ID used by this application.
   * This getter should not be overridden by subclasses, which should instead configure the ID in `DEFAULT_OPTIONS` or
   * by defining a `uniqueId` during `_initializeApplicationOptions`.
   * @type {string}
   */
  get id() {
    return this.#id;
  }

  /**
   * A convenience reference to the title of the Application window.
   * @type {string}
   */
  get title() {
    return game.i18n.localize(this.options.window.title);
  }

  /**
   * The HTMLElement which renders this Application into the DOM.
   * @type {HTMLElement}
   */
  get element() {
    return this.#element;
  }

  /**
   * Does this Application have a top-level form element?
   * @type {HTMLFormElement|null}
   */
  get form() {
    if ( this.options.tag === "form" ) return this.#element;
    if ( this.options.window?.contentTag === "form" ) return this.#content;
    return null;
  }

  /**
   * Is this Application instance currently minimized?
   * @type {boolean}
   */
  get minimized() {
    return this.rendered && this.#minimization.active;
  }

  /**
   * The current position of the application with respect to the window.document.body.
   * @type {ApplicationPosition}
   */
  position = new Proxy(this.#position, {
    set: (obj, prop, value) => {
      if ( prop in obj ) {
        obj[prop] = value;
        this._updatePosition(this.#position);
        return value;
      }
    }
  });

  /**
   * Is this Application instance currently rendered?
   * @type {boolean}
   */
  get rendered() {
    return this.#state === ApplicationV2.RENDER_STATES.RENDERED;
  }

  /**
   * The current render state of the Application.
   * @type {ApplicationRenderState}
   */
  get state() {
    return this.#state;
  }

  /**
   * Does this Application instance render within an outer window frame?
   * @type {boolean}
   */
  get hasFrame() {
    return this.options.window.frame;
  }

  /* -------------------------------------------- */
  /*  Initialization                              */
  /* -------------------------------------------- */

  /**
   * Iterate over the inheritance chain of this Application.
   * The chain includes this Application itself and all parents until the base application is encountered.
   * @see {@link ApplicationV2.BASE_APPLICATION}
   * @yields {typeof ApplicationV2}
   */
  static *inheritanceChain() {
    let cls = this;
    while ( cls ) {
      yield cls;
      if ( cls === this.BASE_APPLICATION ) return;
      cls = Object.getPrototypeOf(cls);
    }
  }

  /* -------------------------------------------- */

  /**
   * Initialize configuration options for the Application instance.
   * The default behavior of this method is to intelligently merge options for each class with those of their parents.
   * - Array-based options are concatenated
   * - Inner objects are merged
   * - Otherwise, properties in the subclass replace those defined by a parent
   * @param {Partial<ApplicationConfiguration>} options      Options provided directly to the constructor
   * @returns {ApplicationConfiguration}                     Configured options for the application instance
   * @protected
   */
  _initializeApplicationOptions(options) {

    // Options initialization order
    const order = [options];
    for ( const cls of this.constructor.inheritanceChain() ) {
      if ( cls.hasOwnProperty("DEFAULT_OPTIONS") ) order.unshift(cls.DEFAULT_OPTIONS);
    }

    // Intelligently merge with parent class options
    /** @type {ApplicationConfiguration} */
    const applicationOptions = {};
    for ( const opts of order ) {
      ApplicationV2.#mergeApplicationOptions(applicationOptions, opts);
    }

    // Unique application ID
    applicationOptions.uniqueId = String(++ApplicationV2._appId);

    // Constrain some options into mutual coherence
    if ( !applicationOptions.window.frame ) applicationOptions.window.minimizable = false;

    // Special handling for classes
    if ( applicationOptions.window.frame ) applicationOptions.classes.unshift("application");
    applicationOptions.classes = Array.from(new Set(applicationOptions.classes));
    return applicationOptions;
  }

  /* -------------------------------------------- */

  /**
   * Merge Application options with logic as described by ApplicationV2#_initializeApplicationOptions.
   * @param {object} options
   * @param {object} opts
   */
  static #mergeApplicationOptions(options, opts) {
    for ( const [k, v] of Object.entries(opts) ) {
      const v1 = foundry.utils.deepClone(v);
      if ( (k in options) ) {
        const v0 = options[k];
        if ( Array.isArray(v0) ) options[k].push(...v1);                          // Concatenate arrays
        else if ( foundry.utils.getType(v0) === "Object") {                       // Merge objects
          ApplicationV2.#mergeApplicationOptions(v0, v1);
        }
        else options[k] = v1;                                                     // Replace option
      }
      else options[k] = v1;                                                       // Define option
    }
  }

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /**
   * Render the Application, creating its HTMLElement and replacing its innerHTML.
   * Add it to the DOM if it is not currently rendered and rendering is forced. Otherwise, re-render its contents.
   * @param {boolean|RenderOptions} [options]            Options which configure application rendering behavior.
   *                                                      A boolean is interpreted as the "force" option.
   * @param {RenderOptions} [_options]                   Legacy options for backwards-compatibility with the original
   *                                                      ApplicationV1#render signature.
   * @returns {Promise<this>}            A Promise which resolves to the rendered Application instance
   */
  async render(options={}, _options={}) {
    if ( typeof options === "boolean" ) options = Object.assign(_options, {force: options});
    return this.#semaphore.add(this.#render.bind(this), options);
  }

  /* -------------------------------------------- */

  /**
   * Manage the rendering step of the Application life-cycle.
   * This private method delegates out to several protected methods which can be defined by the subclass.
   * @param {RenderOptions} [options]             Options which configure application rendering behavior
   * @returns {Promise<this>}            A Promise which resolves to the rendered Application instance
   */
  async #render(options) {
    const states = ApplicationV2.RENDER_STATES;
    if ( !this.#renderable ) throw new Error(`The ${this.constructor.name} Application class is not renderable because`
      + " it does not implement the abstract methods _renderHTML and _replaceHTML. Consider using a mixin such as"
      + " foundry.applications.api.HandlebarsApplicationMixin for this purpose.");

    // Verify that the Application is allowed to be rendered
    try {
      const canRender = this._canRender(options);
      if ( canRender === false ) return this;
    } catch(err) {
      ui.notifications.warn(err.message);
      return this;
    }
    options.isFirstRender = this.#state <= states.NONE;
    if ( options.isFirstRender && !options.force ) return this;

    // Prepare rendering context data
    this._configureRenderOptions(options);
    const context = await this._prepareContext(options);

    // Pre-render life-cycle events (awaited)
    const handlerArgs = [context, options];
    if ( options.isFirstRender ) {
      await this._doEvent(this._preFirstRender, {async: true, handlerArgs, debugText: "Before first render"});
    }
    await this._doEvent(this._preRender, {async: true, handlerArgs, debugText: "Before render"});

    // Render the Application frame
    this.#state = states.RENDERING;
    if ( options.isFirstRender ) {
      this.#element = await this._renderFrame(options);
      this.#content = this.hasFrame ? this.#element.querySelector(".window-content") : this.#element;
      this._attachFrameListeners();
    }

    // Render Application content
    try {
      const result = await this._renderHTML(context, options);
      this._replaceHTML(result, this.#content, options);
    }
    catch(err) {
      if ( this.#element ) {
        this.#element.remove();
        this.#element = null;
      }
      this.#state = states.ERROR;
      throw new Error(`Failed to render Application "${this.id}":\n${err.message}`, { cause: err });
    }

    // Register the rendered Application
    if ( options.isFirstRender ) {
      foundry.applications.instances.set(this.#id, this);
      this._insertElement(this.#element);
    }
    if ( this.hasFrame ) this._updateFrame(options);
    this.#state = states.RENDERED;

    // Post-render life-cycle events (not awaited)
    if ( options.isFirstRender ) {
      await this._doEvent(this._onFirstRender, {handlerArgs, debugText: "After first render", async: true});
    }
    await this._doEvent(this._onRender, {
      handlerArgs, debugText: "After render", eventName: "render", hookName: "render",
      hookArgs: [this.#element, ...handlerArgs], async: true
    });

    // Finalize render after hooks have run.
    await this._doEvent(this._postRender, {handlerArgs, debugText: "Render finalization", async: true});

    // Update application position
    if ( "position" in options ) this.setPosition(options.position);
    if ( options.force ) this.maximize().then(() => this.bringToFront());
    return this;
  }

  /* -------------------------------------------- */

  /**
   * Modify the provided options passed to a render request.
   * @param {RenderOptions} options                 Options which configure application rendering behavior
   * @protected
   */
  _configureRenderOptions(options) {
    const {window, position} = this.options;

    // Initial frame options
    if ( options.isFirstRender ) {
      if ( this.hasFrame ) {
        options.window ??= {};
        options.window.title = (options.window.title || this.title).replace(/\s+/g, " ").trim();
        options.window.icon ||= window.icon;
        options.window.controls = true;
        options.window.resizable = window.resizable;
      }
    }

    // Automatic repositioning
    if ( options.isFirstRender ) options.position = Object.assign(this.#position, options.position);
    else {
      if ( position.width === "auto" ) options.position = Object.assign({width: "auto"}, options.position);
      if ( position.height === "auto" ) options.position = Object.assign({height: "auto"}, options.position);
    }

    // Tabs
    if ( options.tab ) {
      const tabType = foundry.utils.getType(options.tab);
      if ( tabType === "string" ) this.tabGroups[Object.keys(this.constructor.TABS)[0]] = options.tab;
      else if ( tabType === "Object" ) {
        for ( const [tabGroup, tabId] of Object.entries(options.tab) ) this.tabGroups[tabGroup] = tabId;
      }
    }
  }

  /* -------------------------------------------- */

  /**
   * Prepare application rendering context data for a given render request. If exactly one tab group is configured for
   * this application, it will be prepared automatically.
   * @param {RenderOptions} options                 Options which configure application rendering behavior
   * @returns {Promise<ApplicationRenderContext>}   Context data for the render operation
   * @protected
   */
  async _prepareContext(options) {
    const tabGroupIds = Object.keys(this.constructor.TABS);
    return tabGroupIds.length === 1 ? {tabs: this._prepareTabs(tabGroupIds[0])} : {};
  }

  /* -------------------------------------------- */

  /**
   * Prepare application tab data for a single tab group.
   * @param {string} group The ID of the tab group to prepare
   * @returns {Record<string, ApplicationTab>}
   * @protected
   */
  _prepareTabs(group) {
    const {tabs, labelPrefix, initial=null} = this._getTabsConfig(group) ?? {tabs: []};
    this.tabGroups[group] ??= initial;
    return tabs.reduce((prepared, {id, cssClass, ...tabConfig}) => {
      const active = this.tabGroups[group] === id;
      if ( active ) cssClass = [cssClass, "active"].filterJoin(" ");
      const tab = {group, id, active, cssClass, ...tabConfig};
      if ( labelPrefix ) tab.label ??= `${labelPrefix}.${id}`;
      prepared[id] = tab;
      return prepared;
    }, {});
  }

  /* -------------------------------------------- */

  /**
   * Get the configuration for a tabs group.
   * @param {string} group The ID of a tabs group
   * @returns {ApplicationTabsConfiguration|null}
   * @protected
   */
  _getTabsConfig(group) {
    return this.constructor.TABS[group] ?? null;
  }

  /* -------------------------------------------- */

  /**
   * Configure the array of header control menu options
   * @returns {ApplicationHeaderControlsEntry[]}
   * @protected
   */
  _getHeaderControls() {
    return this.options.window.controls?.slice() ?? [];
  }

  /* -------------------------------------------- */

  /**
   * Iterate over header control buttons, filtering for controls which are visible for the current client.
   * @returns {Generator<ApplicationHeaderControlsEntry>}
   * @yields {ApplicationHeaderControlsEntry}
   * @protected
   */
  *_headerControlButtons() {
    const controls = this._doEvent(this._getHeaderControls, {
      async: false,
      debugText: "Header Control Buttons",
      hookName: "getHeaderControls",
      hookResponse: true
    });
    for ( const control of controls ) {
      const visible = typeof control.visible === "function" ? control.visible.call(this) : control.visible ?? true;
      if ( visible ) yield control;
    }
  }

  /* -------------------------------------------- */

  /**
   * Render an HTMLElement for the Application.
   * An Application subclass must implement this method in order for the Application to be renderable.
   * @param {ApplicationRenderContext} context      Context data for the render operation
   * @param {RenderOptions} options                 Options which configure application rendering behavior
   * @returns {Promise<any>}                        The result of HTML rendering may be implementation specific.
   *                                                Whatever value is returned here is passed to _replaceHTML
   * @abstract
   */
  async _renderHTML(context, options) {}

  /* -------------------------------------------- */

  /**
   * Replace the HTML of the application with the result provided by the rendering backend.
   * An Application subclass should implement this method in order for the Application to be renderable.
   * @param {any} result                            The result returned by the application rendering backend
   * @param {HTMLElement} content                   The content element into which the rendered result must be inserted
   * @param {RenderOptions} options                 Options which configure application rendering behavior
   * @protected
   */
  _replaceHTML(result, content, options) {}

  /* -------------------------------------------- */

  /**
   * Render the outer framing HTMLElement which wraps the inner HTML of the Application.
   * @param {RenderOptions} options                 Options which configure application rendering behavior
   * @returns {Promise<HTMLElement>}
   * @protected
   */
  async _renderFrame(options) {
    const frame = document.createElement(this.options.tag);
    frame.id = this.#id;
    if ( this.options.classes.length ) frame.className = this.options.classes.join(" ");
    if ( !this.hasFrame ) return frame;

    // Window applications
    const labels = {
      controls: game.i18n.localize("APPLICATION.TOOLS.ControlsMenu"),
      toggleControls: game.i18n.localize("APPLICATION.TOOLS.ToggleControls"),
      close: game.i18n.localize("APPLICATION.TOOLS.Close")
    };
    frame.innerHTML = `<header class="window-header">
      <i class="window-icon hidden" inert></i>
      <h1 class="window-title"></h1>
      <button type="button" class="header-control icon fa-solid fa-ellipsis-vertical"
              data-tooltip="${labels.toggleControls}" aria-label="${labels.toggleControls}"
              data-action="toggleControls"></button>
      <button type="button" class="header-control icon fa-solid fa-xmark"
              data-tooltip="${labels.close}" aria-label="${labels.close}" data-action="close"></button>
    </header>
    <menu class="controls-dropdown"></menu>`;
    const content = document.createElement(this.options.window.contentTag);
    content.classList.add("window-content", ...this.options.window.contentClasses);
    frame.appendChild(content);
    if ( this.options.window.resizable ) frame.insertAdjacentHTML("beforeend", '<div class="window-resize-handle"></div>');

    // Reference elements
    this.#window.header = frame.querySelector(".window-header");
    this.#window.title = frame.querySelector(".window-title");
    this.#window.icon = frame.querySelector(".window-icon");
    this.#window.resize = frame.querySelector(".window-resize-handle");
    this.#window.close = frame.querySelector("button[data-action=close]");
    this.#window.content = content;
    this.#window.controls = frame.querySelector("button[data-action=toggleControls]");
    this.#window.controlsDropdown = frame.querySelector(".controls-dropdown");
    return frame;
  }

  /* -------------------------------------------- */

  /**
   * Render a header control button.
   * @param {ApplicationHeaderControlsEntry} control
   * @returns {HTMLLIElement}
   * @protected
   */
  _renderHeaderControl(control) {
    const li = document.createElement("li");
    li.className = "header-control";
    li.dataset.action = control.action;
    const button = document.createElement("button");
    button.type = "button";
    button.className = "control";
    const i = document.createElement("i");
    i.className = `control-icon fa-fw ${control.icon}`;
    const span = document.createElement("span");
    span.className = "control-label";
    span.innerText = game.i18n.localize(control.label);
    button.append(i, span);
    li.append(button);
    if ( typeof control.onClick === "function" ) {
      button.addEventListener("click", event => {
        event.preventDefault();
        control.onClick(event);
      });
    }
    return li;
  }

  /* -------------------------------------------- */

  /**
   * When the Application is rendered, optionally update aspects of the window frame.
   * @param {RenderOptions} options               Options provided at render-time
   * @protected
   */
  _updateFrame(options) {
    const window = options.window;
    if ( !window ) return;
    if ( "title" in window ) this.#window.title.innerText = window.title;
    if ( "icon" in window ) this.#window.icon.className = `window-icon fa-fw ${window.icon || "hidden"}`;

    // Window header controls
    const controls = [];
    for ( const c of this._headerControlButtons() ) {
      controls.push(this._renderHeaderControl(c));
    }
    this.#window.controlsDropdown.replaceChildren(...controls);
    this.#window.controls.classList.toggle("hidden", !controls.length);
  }

  /* -------------------------------------------- */

  /**
   * Insert the application HTML element into the DOM.
   * Subclasses may override this method to customize how the application is inserted.
   * @param {HTMLElement} element                 The element to insert
   * @protected
   */
  _insertElement(element) {
    const existing = document.getElementById(element.id);
    if ( existing ) existing.replaceWith(element);
    else document.body.append(element);
  }

  /* -------------------------------------------- */
  /*  Closing                                     */
  /* -------------------------------------------- */

  /**
   * Close the Application, removing it from the DOM.
   * @param {Partial<ApplicationClosingOptions>} [options]  Options which modify how the application is closed.
   * @returns {Promise<this>}                               A Promise which resolves to the closed Application instance
   */
  async close(options={}) {
    return this.#semaphore.add(this.#close.bind(this), options);
  }

  /* -------------------------------------------- */

  /**
   * Manage the closing step of the Application life-cycle.
   * This private method delegates out to several protected methods which can be defined by the subclass.
   * @param {Partial<ApplicationClosingOptions>} options    Options which modify how the application is closed
   * @returns {Promise<this>}                               A Promise which resolves to the closed Application instance
   */
  async #close(options) {
    const states = ApplicationV2.RENDER_STATES;
    if ( !this.#element ) {
      this.#state = states.CLOSED;
      return this;
    }

    // Pre-close life-cycle events (awaited)
    await this._doEvent(this._preClose, {async: true, handlerArgs: [options], debugText: "Before close"});

    // Toggle controls
    await this.toggleControls(false, {animate: false});

    // Set explicit dimensions for the transition.
    if ( options.animate !== false ) {
      const { width, height } = this.#element.getBoundingClientRect();
      this.#applyPosition({ ...this.#position, width, height });
    }

    // Animate the window closing
    this.#element.classList.add("minimizing");
    this.#element.style.maxHeight = "0px";
    this.#state = states.CLOSING;
    if ( options.animate !== false ) await this._awaitTransition(this.#element, 1000);

    // Tearing down and removing the instance
    this._tearDown(options);
    this.#state = states.CLOSED;
    foundry.applications.instances.delete(this.#id);

    // Reset minimization state and restore original size
    if ( this.minimized ) {
      const {priorWidth, priorHeight} = this.#minimization;
      if ( priorWidth ) this.#position.width = priorWidth;
      if ( priorHeight ) this.#position.height = priorHeight;
      this.#minimization.active = false;
    }

    // Post-close life-cycle events (not awaited)
    this._doEvent(this._onClose, {handlerArgs: [options], debugText: "After close", eventName: "close",
      hookName: "close"});
    return this;
  }

  /* -------------------------------------------- */

  /**
   * Remove the application HTML element from the DOM.
   * Subclasses may override this method to customize how the application element is removed.
   * @param {HTMLElement} element                 The element to be removed
   * @protected
   */
  _removeElement(element) {
    element.remove();
  }

  /* -------------------------------------------- */

  /**
   * Remove elements from the DOM and trigger garbage collection as part of application closure.
   * @param {ApplicationClosingOptions} options
   * @protected
   */
  _tearDown(options) {
    this._removeElement(this.#element);
    this.#element = null;
    this.#content = null;
    Object.assign(this.#window, ApplicationV2.#INITIAL_WINDOW_VALUES);
  }

  /* -------------------------------------------- */
  /*  Positioning                                 */
  /* -------------------------------------------- */

  /**
   * Update the Application element position using provided data which is merged with the prior position.
   * @param {Partial<ApplicationPosition>} [position] New Application positioning data
   * @returns {ApplicationPosition|void}              The updated application position
   */
  setPosition(position) {
    if ( !this.options.window.positioned ) return;
    position = Object.assign(this.#position, position);
    this._doEvent(this._prePosition, {handlerArgs: [position], debugText: "Before reposition"});

    // Update resolved position
    const updated = this._updatePosition(position);
    Object.assign(this.#position, updated);

    // Assign CSS styles
    this.#applyPosition(updated);
    this._doEvent(this._onPosition, {handlerArgs: [position], debugText: "After reposition", eventName: "position"});
    return position;
  }

  /* -------------------------------------------- */

  /**
   * Translate a requested application position updated into a resolved allowed position for the Application.
   * Subclasses may override this method to implement more advanced positioning behavior.
   * @param {ApplicationPosition} position        Requested Application positioning data
   * @returns {ApplicationPosition}               Resolved Application positioning data
   * @protected
   */
  _updatePosition(position) {
    if ( !this.#element ) return position;
    const el = this.#element;
    let {width, height, left, top, scale} = position;
    scale ??= 1.0;
    const computedStyle = getComputedStyle(el);
    let minWidth = ApplicationV2.parseCSSDimension(computedStyle.minWidth, el.parentElement.offsetWidth) || 0;
    let maxWidth = ApplicationV2.parseCSSDimension(computedStyle.maxWidth, el.parentElement.offsetWidth) || Infinity;
    let minHeight = ApplicationV2.parseCSSDimension(computedStyle.minHeight, el.parentElement.offsetHeight) || 0;
    let maxHeight = ApplicationV2.parseCSSDimension(computedStyle.maxHeight, el.parentElement.offsetHeight) || Infinity;
    let bounds = el.getBoundingClientRect();
    const {clientWidth, clientHeight} = document.documentElement;

    // Explicit width
    const autoWidth = width === "auto";
    if ( !autoWidth ) {
      const targetWidth = Number(width || bounds.width);
      minWidth = parseInt(minWidth) || 0;
      maxWidth = parseInt(maxWidth) || (clientWidth / scale);
      width = Math.clamp(targetWidth, minWidth, maxWidth);
    }

    // Explicit height
    const autoHeight = height === "auto";
    if ( !autoHeight ) {
      const targetHeight = Number(height || bounds.height);
      minHeight = parseInt(minHeight) || 0;
      maxHeight = parseInt(maxHeight) || (clientHeight / scale);
      height = Math.clamp(targetHeight, minHeight, maxHeight);
    }

    // Implicit height
    if ( autoHeight ) {
      Object.assign(el.style, {width: `${width}px`, height: ""});
      bounds = el.getBoundingClientRect();
      height = bounds.height;
    }

    // Implicit width
    if ( autoWidth ) {
      Object.assign(el.style, {height: `${height}px`, width: ""});
      bounds = el.getBoundingClientRect();
      width = bounds.width;
    }

    // Left Offset
    const scaledWidth = width * scale;
    const targetLeft = left ?? ((clientWidth - scaledWidth) / 2);
    const maxLeft = Math.max(clientWidth - scaledWidth, 0);
    left = Math.clamp(targetLeft, 0, maxLeft);

    // Top Offset
    const scaledHeight = height * scale;
    const targetTop = top ?? ((clientHeight - scaledHeight) / 2);
    const maxTop = Math.max(clientHeight - scaledHeight, 0);
    top = Math.clamp(targetTop, 0, maxTop);

    // Scale
    scale ??= 1.0;
    return {width: autoWidth ? "auto" : width, height: autoHeight ? "auto" : height, left, top, scale};
  }

  /* -------------------------------------------- */

  /**
   * Apply validated position changes to the element.
   * @param {ApplicationPosition} position  The new position data to apply.
   */
  #applyPosition(position) {
    Object.assign(this.#element.style, {
      width: position.width === "auto" ? "" : `${position.width}px`,
      height: position.height === "auto" ? "" : `${position.height}px`,
      left: `${position.left}px`,
      top: `${position.top}px`,
      transform: position.scale === 1 ? "" : `scale(${position.scale})`
    });
  }

  /* -------------------------------------------- */
  /*  Other Public Methods                        */
  /* -------------------------------------------- */

  /**
   * Is the window control buttons menu currently expanded?
   * @type {boolean}
   */
  #controlsExpanded = false;

  /**
   * Toggle display of the Application controls menu.
   * Only applicable to window Applications.
   * @param {boolean} [expanded]      Set the controls visibility to a specific state.
   *                                  Otherwise, the visible state is toggled from its current value
   * @param {object} [options]                Options to configure the toggling behavior.
   * @param {boolean} [options.animate=true]  Animate the controls toggling.
   * @returns {Promise<void>}         A Promise which resolves once the control expansion animation is complete
   */
  async toggleControls(expanded, {animate=true}={}) {
    expanded ??= !this.#controlsExpanded;
    if ( expanded === this.#controlsExpanded ) return;
    const dropdown = this.#element.querySelector(".controls-dropdown");
    game.tooltip.deactivate();
    this.#controlsExpanded = expanded;
    dropdown.classList.remove("expanded");
    if ( animate ) {
      const transitionClass = expanded ? "expanding" : "collapsing";
      dropdown.classList.add(transitionClass);
      await this._awaitTransition(dropdown, 1000);
      dropdown.classList.remove(transitionClass);
    }
    if ( expanded ) dropdown.classList.add("expanded");
  }

  /* -------------------------------------------- */

  /**
   * Minimize the Application, collapsing it to a minimal header.
   * @returns {Promise<void>}
   */
  async minimize() {
    if ( this.minimized || !this.rendered || !this.options.window.minimizable ) return;
    this.#minimization.active = true;

    // Set explicit dimensions for the transition.
    const { width, height } = this.#element.getBoundingClientRect();
    this.#applyPosition({ ...this.#position, width, height });

    // Record pre-minimization data
    this.#minimization.priorWidth = this.#position.width;
    this.#minimization.priorHeight = this.#position.height;
    this.#minimization.priorBoundingWidth = width;
    this.#minimization.priorBoundingHeight = height;

    // Animate to collapsed size
    this.#element.classList.add("minimizing");
    this.#element.style.maxWidth = "var(--minimized-width)";
    this.#element.style.maxHeight = "var(--header-height)";
    await this._awaitTransition(this.#element, 1000);
    this.#element.classList.add("minimized");
    this.#element.classList.remove("minimizing");
  }

  /* -------------------------------------------- */

  /**
   * Restore the Application to its original dimensions.
   * @returns {Promise<void>}
   */
  async maximize() {
    if ( !this.minimized ) return;
    this.#minimization.active = false;

    // Animate back to full size
    this.#element.classList.remove("minimized");
    this.#element.classList.add("maximizing");
    this.#element.style.maxWidth = "";
    this.#element.style.maxHeight = "";
    await this._awaitTransition(this.#element, 1000);

    // Set position
    const {priorBoundingWidth: width, priorBoundingHeight: height} = this.#minimization;
    this.setPosition({width, height});
    this.#element.classList.remove("maximizing");
  }

  /* -------------------------------------------- */

  /**
   * Bring this Application window to the front of the rendering stack by increasing its z-index.
   * Once ApplicationV1 is deprecated we should switch from _maxZ to ApplicationV2#maxZ
   * We should also eliminate ui.activeWindow in favor of only ApplicationV2#frontApp
   */
  bringToFront() {
    if ( !this.options.window?.frame ) return;
    if ( !((ApplicationV2.#frontApp === this) && (ui.activeWindow === this)) ) {
      this.#position.zIndex = ++ApplicationV2._maxZ;
    }
    this.#element.style.zIndex = String(this.#position.zIndex);
    ApplicationV2.#frontApp = this;
    ui.activeWindow = this; // ApplicationV1 compatibility
  }

  /* -------------------------------------------- */

  /**
   * Change the active tab within a tab group in this Application instance.
   * @param {string} tab        The name of the tab which should become active
   * @param {string} group      The name of the tab group which defines the set of tabs
   * @param {object} [options]  Additional options which affect tab navigation
   * @param {Event} [options.event]                 An interaction event which caused the tab change, if any
   * @param {HTMLElement} [options.navElement]      An explicit navigation element being modified
   * @param {boolean} [options.force=false]         Force changing the tab even if the new tab is already active
   * @param {boolean} [options.updatePosition=true] Update application position after changing the tab?
   */
  changeTab(tab, group, {event, navElement, force=false, updatePosition=true}={}) {
    if ( !tab || !group ) throw new Error("You must pass both the tab and tab group identifier");
    if ( (this.tabGroups[group] === tab) && !force ) return;  // No change necessary
    const tabElement = this.#content.querySelector(`.tabs [data-group="${group}"][data-tab="${tab}"]`);
    if ( !tabElement ) throw new Error(`No matching tab element found for group "${group}" and tab "${tab}"`);

    // Update tab navigation
    for ( const t of this.#content.querySelectorAll(`.tabs [data-group="${group}"]`) ) {
      t.classList.toggle("active", t.dataset.tab === tab);
      if ( t instanceof HTMLButtonElement ) t.ariaPressed = `${t.dataset.tab === tab}`;
    }

    // Update tab contents
    for ( const section of this.#content.querySelectorAll(`.tab[data-group="${group}"]`) ) {
      section.classList.toggle("active", section.dataset.tab === tab);
    }
    this.tabGroups[group] = tab;

    // Update automatic width or height
    if ( !updatePosition ) return;
    const positionUpdate = {};
    if ( this.options.position.width === "auto" ) positionUpdate.width = "auto";
    if ( this.options.position.height === "auto" ) positionUpdate.height = "auto";
    if ( !foundry.utils.isEmpty(positionUpdate) ) this.setPosition(positionUpdate);
  }

  /* -------------------------------------------- */

  /**
   * Programmatically submit an ApplicationV2 instance which implements a single top-level form.
   * @param {object} [submitOptions]  Arbitrary options which are supported by and provided to the configured form
   *                                  submission handler.
   * @returns {Promise<*>}            A promise that resolves to the returned result of the form submission handler,
   *                                  if any.
   */
  async submit(submitOptions={}) {
    const formConfig = this.options.form;
    if ( !formConfig?.handler ) throw new Error(`The ${this.constructor.name} Application does not support a`
      + " single top-level form element.");
    const form = this.form;
    const event = new SubmitEvent("submit", {cancelable: true});
    const formData = new FormDataExtended(form);
    return formConfig.handler.call(this, event, form, formData, submitOptions);
  }

  /* -------------------------------------------- */
  /*  Life-Cycle Handlers                         */
  /* -------------------------------------------- */

  /**
   * Perform an event in the application life-cycle.
   * Await an internal life-cycle method defined by the class.
   * Optionally dispatch an event for any registered listeners.
   * @param {Function} handler        A handler function to call
   * @param {object} options          Options which configure event handling
   * @param {boolean} [options.async]         Await the result of the handler function?
   * @param {any[]} [options.handlerArgs]     Arguments passed to the handler function
   * @param {string} [options.debugText]      Debugging text to log for the event
   * @param {string} [options.eventName]      An event name to dispatch for registered listeners
   * @param {string} [options.hookName]       A hook name to dispatch for this and all parent classes
   * @param {any[]} [options.hookArgs]        Arguments passed to the requested hook function
   * @param {boolean} [options.hookResponse=false]  Add the handler response to hookArgs
   * @param {boolean} [options.parentClassHooks=true] Call hooks for parent classes in the inheritance chain?
   * @returns {Promise<void>|void}    A promise which resoles once the handler is complete if async is true
   * @internal
   */
  _doEvent(handler, {async=false, handlerArgs=[], debugText, eventName, hookName, hookArgs=[],
    hookResponse=false, parentClassHooks=true}={}) {

    // Debug logging
    if ( debugText && CONFIG.debug.applications ) {
      console.debug(`${this.constructor.name} | ${debugText}`);
    }

    // Async Events
    const response = handler.call(this, ...handlerArgs);
    if ( async && (response instanceof Promise) ) return response.then(r => {
      if ( hookResponse ) hookArgs = [...hookArgs, r];
      this.#dispatchEvent(eventName, hookName, hookArgs, parentClassHooks);
      return r;
    });

    // Sync Events
    else {
      if ( hookResponse ) hookArgs = [...hookArgs, response];
      this.#dispatchEvent(eventName, hookName, hookArgs, parentClassHooks);
      return response;
    }
  }

  /* -------------------------------------------- */

  /**
   * Dispatch downstream workflows after either an async or sync event.
   * @param {string} eventName
   * @param {string} hookName
   * @param {any[]} hookArgs
   * @param {boolean} parentClassHooks
   */
  #dispatchEvent(eventName, hookName, hookArgs, parentClassHooks) {

    // Dispatch event for this Application instance
    if ( eventName ) this.dispatchEvent(new Event(eventName, { bubbles: true, cancelable: true }));

    // Call hooks for this Application class
    if ( hookName ) this.#callHooks(hookName, hookArgs, parentClassHooks);
  }

  /* -------------------------------------------- */

  /**
   * Call hooks for this Application class.
   * @param {string} hookName             The hook name.
   * @param {any[]} hookArgs              Arguments passed to the hook.
   * @param {boolean} parentClassHooks    Call hooks for parent classes in the inheritance chain?
   */
  #callHooks(hookName, hookArgs, parentClassHooks) {
    if ( parentClassHooks && !hookName.includes("{}") ) hookName += "{}";
    const classes = parentClassHooks ? this.constructor.inheritanceChain() : [this.constructor];
    for ( const cls of classes ) {
      if ( !cls.name ) continue;
      Hooks$1.callAll(hookName.replace("{}", cls.name), this, ...hookArgs);
    }
  }

  /* -------------------------------------------- */
  /*  Rendering Life-Cycle Methods                */
  /* -------------------------------------------- */

  /**
   * Test whether this Application is allowed to be rendered.
   * @param {RenderOptions} options                 Provided render options
   * @returns {false|void}                          Return false to prevent rendering
   * @throws {Error}                                An Error to display a warning message
   * @protected
   */
  _canRender(options) {}

  /* -------------------------------------------- */

  /**
   * Actions performed before a first render of the Application.
   * @param {ApplicationRenderContext} context      Prepared context data
   * @param {RenderOptions} options                 Provided render options
   * @returns {Promise<void>}
   * @protected
   */
  async _preFirstRender(context, options) {}

  /* -------------------------------------------- */

  /**
   * Actions performed after a first render of the Application.
   * @param {ApplicationRenderContext} context      Prepared context data
   * @param {RenderOptions} options                 Provided render options
   * @returns {Promise<void>}
   * @protected
   */
  async _onFirstRender(context, options) {}

  /* -------------------------------------------- */

  /**
   * Actions performed before any render of the Application.
   * Pre-render steps are awaited by the render process.
   * @param {ApplicationRenderContext} context      Prepared context data
   * @param {RenderOptions} options                 Provided render options
   * @returns {Promise<void>}
   * @protected
   */
  async _preRender(context, options) {}

  /* -------------------------------------------- */

  /**
   * Actions performed after any render of the Application.
   * @param {ApplicationRenderContext} context      Prepared context data
   * @param {RenderOptions} options                 Provided render options
   * @returns {Promise<void>}
   * @protected
   */
  async _onRender(context, options) {}

  /* -------------------------------------------- */

  /**
   * Perform post-render finalization actions.
   * @param {ApplicationRenderContext} context  Prepared context data.
   * @param {RenderOptions} options             Provided render options.
   * @returns {Promise<void>}
   * @protected
   */
  async _postRender(context, options) {
    if ( options.isFirstRender ) this.#element.querySelector("[autofocus]")?.focus();
  }

  /* -------------------------------------------- */

  /**
   * Actions performed before closing the Application.
   * Pre-close steps are awaited by the close process.
   * @param {RenderOptions} options                 Provided render options
   * @returns {Promise<void>}
   * @protected
   */
  async _preClose(options) {}

  /* -------------------------------------------- */

  /**
   * Actions performed after closing the Application.
   * Post-close steps are not awaited by the close process.
   * @param {RenderOptions} options Provided render options
   * @protected
   */
  _onClose(options) {}

  /* -------------------------------------------- */

  /**
   * Actions performed before the Application is re-positioned.
   * Pre-position steps are not awaited because setPosition is synchronous.
   * @param {ApplicationPosition} position          The requested application position
   * @protected
   */
  _prePosition(position) {}

  /* -------------------------------------------- */

  /**
   * Actions performed after the Application is re-positioned.
   * @param {ApplicationPosition} position          The requested application position
   * @protected
   */
  _onPosition(position) {}

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /**
   * Attach event listeners to the Application frame.
   * @protected
   */
  _attachFrameListeners() {

    // Application Click Events
    this.#element.addEventListener("pointerdown", this.#onPointerDown.bind(this), {capture: true});
    const click = this.#onClick.bind(this);
    this.#element.addEventListener("click", click);
    this.#element.addEventListener("contextmenu", click);

    if ( this.hasFrame ) {
      this.bringToFront();
      this.#window.header.addEventListener("pointerdown", this.#onWindowDragStart.bind(this));
      this.#window.header.addEventListener("dblclick", this.#onWindowDoubleClick.bind(this));
      this.#window.resize?.addEventListener("pointerdown", this.#onWindowResizeStart.bind(this));
    }

    // Form handlers
    const form = this.form;
    if ( form ) {
      form.autocomplete = "off";
      form.addEventListener("submit", this._onSubmitForm.bind(this, this.options.form));
      form.addEventListener("change", this._onChangeForm.bind(this, this.options.form));
    }
  }

  /* -------------------------------------------- */

  /**
   * Handle initial pointerdown events inside a rendered Application.
   * @param {PointerEvent} event
   */
  async #onPointerDown(event) {
    if ( this.hasFrame ) this.bringToFront();
  }

  /* -------------------------------------------- */

  /**
   * Centralized handling of click events which occur on or within the Application frame.
   * @param {PointerEvent} event
   */
  async #onClick(event) {
    const target = event.target;
    const actionButton = target.closest("[data-action]");
    if ( actionButton ) return this.#onClickAction(event, actionButton);
    this.toggleControls(false);
  }

  /* -------------------------------------------- */

  /**
   * Handle a click event on an element which defines a [data-action] handler.
   * @param {PointerEvent} event      The originating click event
   * @param {HTMLElement} target      The capturing HTML element which defined a [data-action]
   */
  #onClickAction(event, target) {
    const action = target.dataset.action;
    switch ( action ) {
      case "close":
        event.stopPropagation();
        event.preventDefault();
        if ( event.button === 0 ) this.close();
        else this.toggleControls(false);
        break;
      case "tab":
        this.toggleControls(false);
        this._onClickTab(event);
        break;
      case "toggleControls":
        event.stopPropagation();
        event.preventDefault();
        this.toggleControls(event.button === 0 ? undefined : false);
        break;
      default: {
        let handler = this.options.actions[action];

        // Toggle controls dropdown
        this.toggleControls(false);

        // No defined handler
        if ( !handler ) {
          this._onClickAction(event, target);
          break;
        }

        // Defined handler
        let buttons = [0];
        if ( typeof handler === "object" ) {
          buttons = handler.buttons;
          handler = handler.handler;
        }
        if ( buttons.includes(event.button) ) handler?.call(this, event, target);
        break;
      }
    }
  }

  /* -------------------------------------------- */

  /**
   * Handle click events on a tab within the Application.
   * @param {PointerEvent} event
   * @protected
   */
  _onClickTab(event) {
    const button = event.target;
    const tab = button.dataset.tab;
    if ( !tab || button.classList.contains("active") || (event.button !== 0) ) return;
    const group = button.dataset.group;
    const navElement = button.closest(".tabs");
    this.changeTab(tab, group, {event, navElement});
  }

  /* -------------------------------------------- */

  /**
   * A generic event handler for action clicks which can be extended by subclasses.
   * Action handlers defined in DEFAULT_OPTIONS are called first. This method is only called for actions which have
   * no defined handler.
   * @param {PointerEvent} event      The originating click event
   * @param {HTMLElement} target      The capturing HTML element which defined a [data-action]
   * @protected
   */
  _onClickAction(event, target) {}

  /* -------------------------------------------- */

  /**
   * Begin capturing pointer events on the application frame.
   * @param {PointerEvent} event  The triggering event.
   * @param {Function} callback   The callback to attach to pointer move events.
   */
  #startPointerCapture(event, callback) {
    this.#window.pointerStartPosition = Object.assign(foundry.utils.deepClone(this.#position), {
      clientX: event.clientX, clientY: event.clientY
    });
    this.#element.addEventListener("pointermove", callback, { passive: true });
    this.#element.addEventListener("pointerup", event => this.#endPointerCapture(event, callback), {
      capture: true, once: true
    });
  }

  /* -------------------------------------------- */

  /**
   * End capturing pointer events on the application frame.
   * @param {PointerEvent} event  The triggering event.
   * @param {Function} callback   The callback to remove from pointer move events.
   */
  #endPointerCapture(event, callback) {
    this.#element.releasePointerCapture(event.pointerId);
    this.#element.removeEventListener("pointermove", callback);
    delete this.#window.pointerStartPosition;
    this.#window.pointerMoveThrottle = false;
  }

  /* -------------------------------------------- */

  /**
   * Handle a pointer move event while dragging or resizing the window frame.
   * @param {PointerEvent} event
   * @returns {{dx: number, dy: number}|void}  The amount the cursor has moved since the last frame, or undefined if
   *                                           the movement occurred between frames.
   */
  #onPointerMove(event) {
    if ( this.#window.pointerMoveThrottle ) return;
    this.#window.pointerMoveThrottle = true;
    const dx = event.clientX - this.#window.pointerStartPosition.clientX;
    const dy = event.clientY - this.#window.pointerStartPosition.clientY;
    requestAnimationFrame(() => this.#window.pointerMoveThrottle = false);
    return { dx, dy };
  }

  /* -------------------------------------------- */

  /**
   * Begin dragging the Application position.
   * @param {PointerEvent} event
   */
  #onWindowDragStart(event) {
    if ( event.target.closest(".header-control") ) return;
    this.#endPointerCapture(event, this.#window.onDrag);
    this.#startPointerCapture(event, this.#window.onDrag);
  }

  /* -------------------------------------------- */

  /**
   * Begin resizing the Application.
   * @param {PointerEvent} event
   */
  #onWindowResizeStart(event) {
    for ( const dim of ["width", "height"] ) {
      if ( this.#position[dim] === "auto" ) {
        this.#position[dim] = this.#element[`client${dim.titleCase()}`];
        this.options.position[dim] = this.#position[dim];
      }
    }
    this.#endPointerCapture(event, this.#window.onResize);
    this.#startPointerCapture(event, this.#window.onResize);
  }

  /* -------------------------------------------- */

  /**
   * Drag the Application position during mouse movement.
   * @param {PointerEvent} event
   */
  #onWindowDragMove(event) {
    if ( !this.#window.header.hasPointerCapture(event.pointerId) ) {
      this.#window.header.setPointerCapture(event.pointerId);
    }
    const delta = this.#onPointerMove(event);
    if ( !delta ) return;
    const { pointerStartPosition } = this.#window;
    let { top, left, height, width } = pointerStartPosition;
    left += delta.dx;
    top += delta.dy;
    this.setPosition({ top, left, height, width });
  }

  /* -------------------------------------------- */

  /**
   * Resize the Application during mouse movement.
   * @param {PointerEvent} event
   */
  #onWindowResizeMove(event) {
    if ( !this.#window.resize.hasPointerCapture(event.pointerId) ) {
      this.#window.resize.setPointerCapture(event.pointerId);
    }
    const delta = this.#onPointerMove(event);
    if ( !delta ) return;
    const { scale } = this.#position;
    const { pointerStartPosition } = this.#window;
    let { top, left, height, width } = pointerStartPosition;
    width += delta.dx / scale;
    height += delta.dy / scale;
    this.setPosition({ top, left, width, height });
  }

  /* -------------------------------------------- */

  /**
   * Double-click events on the window title are used to minimize or maximize the application.
   * @param {PointerEvent} event
   */
  #onWindowDoubleClick(event) {
    event.preventDefault();
    if ( event.target.dataset.action ) return; // Ignore double clicks on buttons which perform an action
    if ( !this.options.window.minimizable ) return;
    if ( this.minimized ) this.maximize();
    else this.minimize();
  }

  /* -------------------------------------------- */

  /**
   * Handle submission for an Application which uses the form element.
   * @param {ApplicationFormConfiguration} formConfig     The form configuration for which this handler is bound
   * @param {Event|SubmitEvent} event                     The form submission event
   * @returns {Promise<void>}
   * @protected
   */
  async _onSubmitForm(formConfig, event) {
    event.preventDefault();
    const form = event.currentTarget;
    const {handler, closeOnSubmit} = formConfig;
    const formData = new FormDataExtended(form);
    if ( handler instanceof Function ) {
      try {
        await handler.call(this, event, form, formData);
      } catch(err) {
        ui.notifications.error(err, {console: true});
        return; // Do not close
      }
    }
    if ( closeOnSubmit ) await this.close({submitted: true});
  }

  /* -------------------------------------------- */

  /**
   * Handle changes to an input element within the form.
   * @param {ApplicationFormConfiguration} formConfig     The form configuration for which this handler is bound
   * @param {Event} event                                 An input change event within the form
   * @protected
   */
  _onChangeForm(formConfig, event) {
    const { RENDERED, CLOSING } = ApplicationV2.RENDER_STATES;
    const open = (this.#state === RENDERED) || (this.#state === CLOSING);
    if ( open && formConfig.submitOnChange ) this._onSubmitForm(formConfig, event);
  }

  /* -------------------------------------------- */
  /*  Helper Methods                              */
  /* -------------------------------------------- */

  /**
   * Parse a CSS style rule into a number of pixels which apply to that dimension.
   * @param {string} style            The CSS style rule
   * @param {number} parentDimension  The relevant dimension of the parent element
   * @returns {number|void}           The parsed style dimension in pixels
   */
  static parseCSSDimension(style, parentDimension) {
    if ( style.includes("px") ) return parseInt(style.replace("px", ""));
    if ( style.includes("%") ) {
      const p = parseInt(style.replace("%", "")) / 100;
      return parentDimension * p;
    }
  }

  /* -------------------------------------------- */

  /**
   * Wait for a CSS transition to complete for an element.
   * @param {HTMLElement} element         The element which is transitioning
   * @param {number} timeout              A timeout in milliseconds in case the transitionend event does not occur
   * @returns {Promise<void>}
   * @internal
   */
  async _awaitTransition(element, timeout) {
    let listener;
    await Promise.race([
      new Promise(resolve => {
        listener = event => { if ( event.target === element ) resolve(); };
        element.addEventListener("transitionend", listener);
      }),
      new Promise(resolve => window.setTimeout(resolve, timeout))
    ]);
    element.removeEventListener("transitionend", listener);
  }

  /* -------------------------------------------- */

  /**
   * Create a ContextMenu instance used in this Application.
   * @param {() => ContextMenuEntry[]} handler  A handler function that provides initial context options
   * @param {string} selector                   A CSS selector to which the ContextMenu will be bound
   * @param {object} [options]                  Additional options which affect ContextMenu construction
   * @param {HTMLElement} [options.container]   A parent HTMLElement which contains the selector target
   * @param {string} [options.hookName]         The hook name
   * @param {boolean} [options.parentClassHooks=true]  Whether to call hooks for the parent classes in the inheritance
   *                                                   chain.
   * @returns {ContextMenu|null}                A created ContextMenu or null if no menu items were defined
   * @protected
   */
  _createContextMenu(handler, selector, {container, hookName, parentClassHooks, ...options}={}) {
    container ??= this.element;
    hookName ??= "get{}ContextOptions";
    const menuItems = this._doEvent(handler, {hookName, parentClassHooks, hookResponse: true});
    if ( !menuItems.length ) return null;
    return new ContextMenu.implementation(container, selector, menuItems, {jQuery: false, ...options});
  }

  /* -------------------------------------------- */

  /**
   * Wait for any images in the given element to load.
   * @param {HTMLElement} element  The element.
   * @returns {Promise<void>}
   */
  static async waitForImages(element) {
    const images = Array.from(element.querySelectorAll("img")).filter(img => !img.complete);
    if ( !images.length ) return;
    let loaded = 0;
    const { promise, resolve } = Promise.withResolvers();
    const onLoad = img => {
      loaded++;
      img.onload = img.onerror = null;
      if ( loaded >= images.length ) resolve();
    };
    for ( const img of images ) img.onload = img.onerror = onLoad.bind(img, img);
    return promise;
  }

  /* -------------------------------------------- */
  /*  Deprecations and Compatibility              */
  /* -------------------------------------------- */

  /**
   * @deprecated since v12
   * @ignore
   */
  bringToTop() {
    foundry.utils.logCompatibilityWarning(`ApplicationV2#bringToTop is not a valid function and redirects to
      ApplicationV2#bringToFront. This shim will be removed in v14.`, {since: 12, until: 14});
    return this.bringToFront();
  }
}

/**
 * @import ApplicationV2 from "./application.mjs";
 * @import {Constructor} from "@common/_types.mjs";
 * @import {ApplicationConfiguration, ApplicationFormConfiguration} from "../_types.mjs";
 */

/**
 * @typedef HandlebarsRenderOptions
 * @property {string[]} parts                       An array of named template parts to render
 */

/**
 * @typedef HandlebarsTemplatePart
 * @property {string} template                      The template entry-point for the part
 * @property {string} [id]                          A CSS id to assign to the top-level element of the rendered part.
 *                                                  This id string is automatically prefixed by the application id.
 * @property {boolean} [root=false]                 Does this rendered contents of this template part replace the
 *                                                  children of the root element?
 * @property {string[]} [classes]                   An array of CSS classes to apply to the top-level element of the
 *                                                  rendered part.
 * @property {string[]} [templates]                 An array of additional templates that are required to render the
 *                                                  part. If omitted, only the entry-point is inferred as required.
 * @property {string[]} [scrollable]                An array of selectors within this part whose scroll positions should
 *                                                  be persisted during a re-render operation. A blank string is used
 *                                                  to denote that the root level of the part is scrollable.
 * @property {Record<string, ApplicationFormConfiguration>} [forms] A registry of forms selectors and submission
 *                                                                  handlers.
 */

/**
 * Augment an Application class with [Handlebars](https://handlebarsjs.com) template rendering behavior.
 * @param {Constructor<ApplicationV2>} BaseApplication
 */
function HandlebarsApplicationMixin(BaseApplication) {
  /**
   * The mixed application class augmented with [Handlebars](https://handlebarsjs.com) template rendering behavior.
   * @extends {ApplicationV2<ApplicationConfiguration, HandlebarsRenderOptions>}
   */
  class HandlebarsApplication extends BaseApplication {
    constructor(...args) {
      super(...args);
      Object.values(this.constructor.PARTS).forEach((part, i) => {
        if ( part.root && (i > 0) ) throw new Error("A root template must be first in the PARTS record.");
      });
    }

    /**
     * Configure a registry of template parts which are supported for this application for partial rendering.
     * @type {Record<string, HandlebarsTemplatePart>}
     */
    static PARTS = {};

    /**
     * A record of all rendered template parts.
     * @returns {Record<string, HTMLElement>}
     */
    get parts() {
      return this.#parts;
    }

    #parts = {};

    /**
     * Dynamically configured part descriptors.
     * @type {Readonly<Record<string, HandlebarsTemplatePart>>}
     */
    #partDescriptors;

    /* -------------------------------------------- */

    /** @inheritDoc */
    _configureRenderOptions(options) {
      super._configureRenderOptions(options);
      this.#partDescriptors = Object.freeze(this._configureRenderParts(options));
      // Render all parts if we are rendering the root element
      if ( options.parts && options.parts.some(id => this.#partDescriptors[id]?.root) ) options.parts = undefined;
      options.parts ??= Object.keys(this.#partDescriptors);
    }

    /* -------------------------------------------- */

    /**
     * Allow subclasses to dynamically configure render parts.
     * @param {HandlebarsRenderOptions} options
     * @returns {Record<string, HandlebarsTemplatePart>}
     * @protected
     */
    _configureRenderParts(options) {
      const parts = foundry.utils.deepClone(this.constructor.PARTS);
      Object.values(parts).forEach(p => p.templates ??= []);
      return parts;
    }

    /* -------------------------------------------- */

    /** @inheritDoc */
    async _preRender(context, options) {
      await super._preRender(context, options);
      const allTemplates = new Set();
      for ( const part of Object.values(this.#partDescriptors) ) {
        allTemplates.add(part.template);
        part.templates ??= [];
        for ( const template of part.templates ) allTemplates.add(template);
      }
      await foundry.applications.handlebars.loadTemplates(Array.from(allTemplates));
    }

    /* -------------------------------------------- */

    /**
     * Render each configured application part using Handlebars templates.
     * @param {ApplicationRenderContext} context        Context data for the render operation
     * @param {HandlebarsRenderOptions} options         Options which configure application rendering behavior
     * @returns {Promise<Record<string, HTMLElement>>}  A single rendered HTMLElement for each requested part
     * @protected
     * @override
     */
    async _renderHTML(context, options) {
      const rendered = {};
      const partInfo = this.#partDescriptors;
      for ( const partId of options.parts ) {
        const part = partInfo[partId];
        if ( !part ) {
          ui.notifications.warn(`Part "${partId}" is not a supported template part for ${this.constructor.name}`);
          continue;
        }
        const partContext = await this._preparePartContext(partId, context, options);
        try {
          const htmlString = await foundry.applications.handlebars.renderTemplate(part.template, partContext);
          rendered[partId] = this.#parsePartHTML(partId, part, htmlString);
        } catch(err) {
          throw new Error(`Failed to render template part "${partId}":\n${err.message}`, {cause: err});
        }
      }
      return rendered;
    }

    /* -------------------------------------------- */

    /**
     * Prepare context that is specific to only a single rendered part.
     *
     * It is recommended to augment or mutate the shared context so that downstream methods like _onRender have
     * visibility into the data that was used for rendering. It is acceptable to return a different context object
     * rather than mutating the shared context at the expense of this transparency.
     *
     * @param {strin