import { initConfig } from "./config.js";
import { getSetting, registerSettings } from "./settings.js";
import { GetRollData } from "./app/getRollData.js";
import { EpicRoll } from "./app/EpicRoll.js";
import { Socket } from "./lib/socket.js";
import { APIQueue } from "./lib/api.js";

export const MODULE_ID = "epic-rolls-5e";

const API_REQUEST_QUEUE = new APIQueue();

Hooks.on("init", () => {
    globalThis.ui.EpicRolls5e = {
        EpicRoll,
        GetRollData,
        Socket,
        _queue: [],
        requestRoll: async (data) => {
            const firstActiveUser = game.users.find((u) => u.active);
            const res = await Socket.routeRequest(data, { users: [firstActiveUser.id] });
            return res[0].response;
        },
    };
    initConfig();
    registerSettings();
});

Hooks.on("ready", () => {
    Socket.register("updateEpicRoll", (data) => {
        ui.EpicRolls5e._currentRoll.update(data);
    });
    Socket.register(
        "routeRequest",
        async (data) => {
            const response = await API_REQUEST_QUEUE.queueResponse(data);
            if (response.error) return { error: response.error };
            if (response.queue) {
                return await ui.EpicRolls5e._queue[response.index].promise;
            }
            return await ui.EpicRolls5e._currentRoll.promise;
        },
        { response: true },
    );
    Socket.register(
        "dispatchEpicRoll",
        (data) => {
            if (data.actors) data.actors = data.actors.map((a) => a?.uuid || a);
            if (data.contestants) data.contestants = data.contestants.map((a) => a?.uuid || a);
            if (ui.EpicRolls5e._currentRoll) {
                const er = new EpicRoll(data);
                ui.EpicRolls5e._queue.push(er);
                const index = ui.EpicRolls5e._queue.length - 1;
                return { queue: true, index };
            }
            new EpicRoll(data).init().render(true);
            return { response: true };
        },
        { response: true },
    );
    Socket.register("endEpicRoll", (data) => {
        ui.EpicRolls5e._currentRoll.endEpicRoll(data);
    });
    Socket.register("toggleRollButton", (data) => {
        ui.EpicRolls5e._currentRoll.toggleRollButton(data.uuid, data.rolling);
    });
    Socket.register(
        "recoverQueue",
        (data) => {
            let currentRollData = null;
            if (ui.EpicRolls5e._currentRoll) {
                const data = { ...ui.EpicRolls5e._currentRoll.rollData };
                const results = ui.EpicRolls5e._currentRoll._results;
                const rolls = ui.EpicRolls5e._currentRoll._rolls;
                const messageIds = Array.from(ui.EpicRolls5e._currentRoll._messageIds);
                data.recovered = {
                    results,
                    rolls,
                    messageIds,
                };
                currentRollData = data;
            }
            return { queue: ui.EpicRolls5e._queue.map((er) => er.rollData), current: currentRollData };
        },
        { response: true },
    );

    recoverQueue();
});

async function recoverQueue() {
    const res = await Socket.recoverQueue({});
    const queues = res.map((r) => r.response.queue);
    const current = res.map((r) => r.response.current).find((r) => r);
    //find longest queue
    if (queues.length) {
        let longestQueue = queues[0];
        queues.forEach((queue) => {
            if (queue.length > longestQueue.length) {
                longestQueue = queue;
            }
        });
        //restore queue
        longestQueue.forEach((data) => {
            ui.EpicRolls5e._queue.push(new EpicRoll(data));
        });
    }
    //restore current roll
    if (current) {
        new EpicRoll(current).init().render(true);
    }
}

Hooks.on("renderChatLog", () => {
    injectButton();
});

Hooks.on("changeSidebarTab", () => {
    injectButton();
});

Hooks.on("renderChatInput", () => {
    injectButton();
});

function injectButton() {
    if (!game.user.isGM) return;
    const html = ui.chat.element;
    if (!html) return;
    const target = html.querySelector(".control-buttons");
    const existing = target?.querySelector(".epic-roll-chat-control");
    if (!target || existing) return;
    const label = document.createElement("button");
    label.type = "button";
    label.classList.add("epic-roll-chat-control", "ui-control", "icon", "fa-solid", "fa-dice-d12");
    label.dataset.tooltip = game.i18n.localize(`${MODULE_ID}.button-tooltip`);
    label.dataset.tooltipDirection = "LEFT";
    label.addEventListener("click", () => {
        new GetRollData().render(true);
    });
    label.addEventListener("contextmenu", (e) => {
        const existingMenu = document.querySelector(".er5e-recent-rolls");
        if (existingMenu) return;
        const recentRolls = getSetting("recentRolls");
        const wrapper = document.createElement("div");
        const ul = document.createElement("ul");
        wrapper.appendChild(ul);
        wrapper.classList.add("er5e-recent-rolls");

        //Epic Initiative
        const epicInitiative = document.createElement("li");
        epicInitiative.innerHTML = `<i class="fad fa-swords"></i> ` + " " + game.i18n.localize("COMBAT.InitiativeRoll");
        ul.appendChild(epicInitiative);
        epicInitiative.addEventListener("click", (e) => {
            const isAlt = e.altKey;
            const selected = canvas.tokens.controlled.map(t => t.actor.uuid);
            const combatants = game.combat ? game.combat.combatants.map(c => c.actor.uuid) : [];
            const actors = selected.length ? selected : combatants;
            if (!actors.length) return ui.notifications.error("No Tokens selected or in Combat");
            ui.EpicRolls5e.requestRoll({ actors: actors.filter((a) => fromUuidSync(a).hasPlayerOwner), contestants: actors.filter((a) => !fromUuidSync(a).hasPlayerOwner), type: "initiative.initiative", contest: "initiative.initiative", options: { formula: "", DC: 0, showDC: false, useAverage: isAlt, allowReroll: false, showRollResults: false, blindRoll: false, hideNames: false, autoColor: true, color: "0", customLabel: "", noMessage: true } });
        });

        recentRolls.forEach((roll) => {
            const li = document.createElement("li");
            li.innerHTML = EpicRoll.getRollLabel(roll.type, roll.options.DC, roll.contest, roll.options);
            ul.appendChild(li);
            li.addEventListener("click", () => {
                new GetRollData(roll).render(true);
            });
        });
        const labelPos = label.getBoundingClientRect();
        wrapper.style.bottom = `${window.innerHeight - labelPos.bottom + labelPos.height + 2}px`;
        wrapper.style.right = `${window.innerWidth - labelPos.right}px`;
        document.body.appendChild(wrapper);
        const listener = () => {
            wrapper.remove();
            document.removeEventListener("click", listener);
        };
        document.addEventListener("click", listener);
    });
    html.querySelector(".control-buttons").prepend(label);
}