import { deepClone, HandlebarsApplication, l } from "../lib/utils.js";
import { getSetting, setSetting } from "../settings.js";
import { SidebarVideoPlayer } from "./app.js";

const MODULE_ID = "fvtt-youtube-player";

export class SoundLibrary extends HandlebarsApplication {
    constructor() {
        super();
        this.searchQuery = "";
    }

    static DEFAULT_OPTIONS = {
        classes: [this.APP_ID],
        tag: "div",
        window: {
            frame: true,
            positioned: true,
            title: `${MODULE_ID}.${this.APP_ID}.title`,
            icon: "fa-duotone fa-books",
            controls: [
                {
                    icon: "fas fa-tag",
                    label: `${MODULE_ID}.tags-configuration.title`,
                    action: "configure-tags",
                },
                {
                    icon: "fas fa-plus",
                    label: `${MODULE_ID}.${this.APP_ID}.add-tracks`,
                    action: "add-tracks",
                },
            ],
            minimizable: true,
            resizable: true,
            contentTag: "section",
            contentClasses: [],
        },
        actions: {},
        form: {
            handler: undefined,
            submitOnChange: false,
            closeOnSubmit: false,
        },
        position: {
            width: 560,
            height: "auto",
        },
        actions: {
            "configure-tags": () => SoundLibrary.configureTags(),
            "add-tracks": () => SoundLibrary.importTracksDialog(),
        },
    };

    static PARTS = {
        search: {
            template: "modules/fvtt-youtube-player/templates/search.hbs",
            classes: ["search-library"],
        },
        content: {
            template: `modules/${MODULE_ID}/templates/${this.APP_ID}.hbs`,
            classes: ["scrollable"],
            scrollable: [""],
        },
    };

    static get APP_ID() {
        return this.name
            .split(/(?=[A-Z])/)
            .join("-")
            .toLowerCase();
    }

    get APP_ID() {
        return this.constructor.APP_ID;
    }

    async _prepareContext(options) {
        const tracks = deepClone(getSetting("soundsLibrary"));
        const tags = getSetting("tags");
        Object.values(tracks).forEach((track) => {
            track.tags ??= [];
            track.tags = track.tags.map((tag) => tags[tag]).filter(t=>!!t).sort((a, b) => a.label.localeCompare(b.label));
        });
        return {
            tracks,
            tags,
            searchQuery: this.searchQuery,
        };
    }

    _onRender(context, options) {
        super._onRender(context, options);
        const html = this.element;
        html.querySelectorAll(".configure-track").forEach((button) => {
            button.addEventListener("click", (event) => {
                event.preventDefault();
                const trackId = button.dataset.trackid;
                SoundLibrary.addTrack(trackId);
            });
        });
        html.querySelectorAll(".delete-track").forEach((button) => {
            button.addEventListener("click", (event) => {
                event.preventDefault();
                const trackId = button.dataset.trackid;
                const tracks = getSetting("soundsLibrary");
                delete tracks[trackId];
                setSetting("soundsLibrary", tracks).then(() => {
                    this.render(true);
                });
            });
        });
        html.querySelectorAll(".track-title").forEach((title) => {
            title.addEventListener("click", (event) => {
                event.preventDefault();
                const trackId = title.dataset.trackid;
                const track = getSetting("soundsLibrary")[trackId];
                setSetting("currentVideo", track.url);
            });
        });
        html.querySelector('input[type="search"]').addEventListener("input", this.onSearch.bind(this));
        html.querySelectorAll(".search-library .tag").forEach((tag) => {
            tag.addEventListener("click", (event) => {
                event.preventDefault();
                const tag = event.currentTarget.dataset.tag;
                const tags = getSetting("tags");
                tags[tag].disabled = !tags[tag].disabled;
                setSetting("tags", tags).then(() => {
                    this.render(true);
                });
            });
        });
        this.onSearch();
        this._initContextMenu();
    }

    _initContextMenu() {
        if(this._contextMenu) return;
        this._contextMenu = new ContextMenu(this.element, ".track", [
            {
                name: `${MODULE_ID}.TOOLTIP.add-to-playlist`,
                icon: '<i class="fas fa-plus"></i>',
                callback: (el) => {
                    el = el[0] ?? el;
                    const trackId = el.dataset.trackid;
                    const track = getSetting("soundsLibrary")[trackId];
                    const url = `https://www.youtube.com/watch?v=${track.url}`;
                    ui.sidebarVideoPlayer.addTrackToPlaylist(url);
                },
            },
            {
                name: `${MODULE_ID}.TOOLTIP.copy-url`,
                icon: '<i class="fas fa-copy"></i>',
                callback: (el) => {
                    el = el[0] ?? el;
                    const trackId = el.dataset.trackid;
                    const track = getSetting("soundsLibrary")[trackId];
                    const url = `https://www.youtube.com/watch?v=${track.url}`;
                    game.clipboard.copyPlainText(url);
                },
            },
            {
                name: `${MODULE_ID}.TOOLTIP.configure-track`,
                icon: '<i class="fas fa-cog"></i>',
                callback: (el) => {
                    el = el[0] ?? el;
                    const trackId = el.dataset.trackid;
                    SoundLibrary.addTrack(trackId);
                },
            },
            {
                name: `${MODULE_ID}.TOOLTIP.delete-track`,
                icon: '<i class="fas fa-trash"></i>',
                classes: ["delete"],
                callback: (el) => {
                    el = el[0] ?? el;
                    const trackId = el.dataset.trackid;
                    const tracks = getSetting("soundsLibrary");
                    delete tracks[trackId];
                    setSetting("soundsLibrary", tracks).then(() => {
                        this.render(true);
                    });
                },
            },
        ]);
    }

    onSearch(event) {
        this.searchQuery = event ? event.currentTarget.value.toLowerCase() : this.searchQuery;
        const tags = getSetting("tags");
        const activeTags = Object.entries(tags)
            .filter(([key, tag]) => !tag.disabled)
            .map(([key, tag]) => key);
        const tracks = getSetting("soundsLibrary");
        for (const [key, track] of Object.entries(tracks)) {
            const hasTags = !track.tags.length || track.tags.filter((tag) => activeTags.includes(tag)).length;
            this.element.querySelector(`.track[data-trackid="${key}"]`).classList.toggle("hidden", !(hasTags && track.title.toLowerCase().includes(this.searchQuery)));
        }
    }

    _onClose(options) {
        super._onClose(options);
    }

    static addTrack(track) {
        new TrackConfiguration(track).render(true);
    }

    static configureTags() {
        new TagsConfiguration().render(true);
    }

    static open() {
        new SoundLibrary().render(true);
    }

    static async importTracksDialog() {
        new foundry.applications.api.DialogV2({
            window: { title: `${MODULE_ID}.${this.APP_ID}.import-tracks` },
            content: `
            <div class="form-group stacked">
                <label for="file-input">${l(`${MODULE_ID}.${this.APP_ID}.import-tracks-textarea`)}</label>
                <textarea id="track-list" rows="10" ></textarea>
            `,
            buttons: [
                {
                    icon: "fas fa-file-import",
                    label: "Import",
                    callback: async (e, b, dialog) => {
                        async function processData() {
                            const allVideos = [];
                            const trackUrls = dialog.element.querySelector("#track-list").value.split("\n");
                            for (const url of trackUrls) {
                                const videos = await SidebarVideoPlayer.getInstance().processUrl(url);
                                console.log(videos);
                                allVideos.push(...videos);
                            }
                            const tracks = getSetting("soundsLibrary");
                            for (const video of allVideos) {
                                tracks[foundry.utils.randomID()] = { title: video.title, url: video.id, tags: [] };
                            }
                            setSetting("soundsLibrary", tracks).then(() => {
                                foundry.applications.instances.forEach((app) => {
                                    if (app instanceof SoundLibrary) app.render(true);
                                });
                            });
                        }
                        processData();
                    },
                },
            ],
        }).render(true);
    }

    static refresh() {
        foundry.applications.instances.forEach((app) => {
            if (app instanceof SoundLibrary) app.render(true);
        });
    }
}

export class TagsConfiguration extends HandlebarsApplication {
    constructor() {
        super();
    }

    static DEFAULT_OPTIONS = {
        classes: [this.APP_ID],
        tag: "form",
        window: {
            frame: true,
            positioned: true,
            title: `${MODULE_ID}.${this.APP_ID}.title`,
            icon: "fas fa-tags",
            controls: [],
            minimizable: true,
            resizable: false,
            contentTag: "section",
            contentClasses: ["standard-form"],
        },
        actions: {},
        form: {
            handler: this.#onSubmit,
            submitOnChange: false,
            closeOnSubmit: true,
        },
        position: {
            width: 560,
            height: "auto",
        },
        actions: {},
    };

    static PARTS = {
        content: {
            template: `modules/${MODULE_ID}/templates/${this.APP_ID}.hbs`,
            classes: ["standard-form"],
        },

        footer: {
            template: "templates/generic/form-footer.hbs",
        },
    };

    static get APP_ID() {
        return this.name
            .split(/(?=[A-Z])/)
            .join("-")
            .toLowerCase();
    }

    get APP_ID() {
        return this.constructor.APP_ID;
    }

    async _prepareContext(options) {
        const tags = getSetting("tags");
        for (const [key, value] of Object.entries(tags)) {
            if (typeof value !== "object") delete tags[key];
        }

        return {
            ...this.tags,
            tagsList: Object.entries(tags).map(([id, data]) => ({ key: id, label: data.label, color: data.color })),
            randomColor: Color.fromHSL([Math.random(), 0.5, 0.5]).css,
            buttons: [
                {
                    type: "submit",
                    action: "submit",
                    icon: "far fa-save",
                    label: "Save",
                },
            ],
        };
    }

    _onRender(context, options) {
        super._onRender(context, options);
        const html = this.element;
        html.querySelector("#create-tag").addEventListener("click", () => {
            const tagLabel = html.querySelector('[name="label"]').value;
            const tagColor = html.querySelector('[name="color"]').value;
            if (!tagLabel || !tagColor) ui.notifications.error(`${MODULE_ID}.tags-configuration.create-tag-error`, { localize: true });
            const tags = getSetting("tags");
            tags[foundry.utils.randomID()] = { label: tagLabel, color: tagColor };
            setSetting("tags", tags).then(() => {
                this.render(true);
            });
        });
        //delete-tag
        html.querySelectorAll(".delete-tag").forEach((button) => {
            button.addEventListener("click", (event) => {
                event.preventDefault();
                const tagId = button.dataset.tagid;
                const tags = getSetting("tags");
                delete tags[tagId];
                setSetting("tags", tags).then(() => {
                    this.render(true);
                });
            });
        });
    }

    static async #onSubmit(event, form, formData) {
        const data = foundry.utils.expandObject(formData.object);
        delete data.color;
        delete data.label;
        const tags = getSetting("tags");
        //Purge bad data
        for (const [key, value] of Object.entries(tags)) {
            if (typeof value !== "object") delete tags[key];
        }
        await setSetting("tags", foundry.utils.mergeObject(tags, data));
    }

    _onClose(options) {
        super._onClose(options);
    }
}

export class TrackConfiguration extends HandlebarsApplication {
    constructor(track) {
        super();
        this.edit = typeof track === "string" ? track : false;
        this.track = this.edit ? getSetting("soundsLibrary")[track] : track;
        this.track.tags ??= [];
    }

    static DEFAULT_OPTIONS = {
        classes: [this.APP_ID],
        tag: "form",
        window: {
            frame: true,
            positioned: true,
            title: `${MODULE_ID}.${this.APP_ID}.title`,
            icon: "fas fa-music",
            controls: [],
            minimizable: true,
            resizable: false,
            contentTag: "section",
            contentClasses: ["standard-form"],
        },
        actions: {},
        form: {
            handler: this.#onSubmit,
            submitOnChange: false,
            closeOnSubmit: true,
        },
        position: {
            width: 560,
            height: "auto",
        },
        actions: {},
    };

    static PARTS = {
        content: {
            template: `modules/${MODULE_ID}/templates/${this.APP_ID}.hbs`,
            classes: ["standard-form"],
        },

        footer: {
            template: "templates/generic/form-footer.hbs",
        },
    };

    static get APP_ID() {
        return this.name
            .split(/(?=[A-Z])/)
            .join("-")
            .toLowerCase();
    }

    get APP_ID() {
        return this.constructor.APP_ID;
    }

    async _prepareContext(options) {
        return {
            ...this.track,
            tagsList: getSetting("tags"),
            buttons: [
                {
                    type: "submit",
                    action: "submit",
                    icon: "far fa-save",
                    label: this.edit ? "Save" : "Create",
                },
            ],
        };
    }

    _onRender(context, options) {
        super._onRender(context, options);
        const html = this.element;
    }

    static async #onSubmit(event, form, formData) {
        const data = foundry.utils.expandObject(formData.object);
        const library = getSetting("soundsLibrary");
        if (this.edit) {
            library[this.edit] = { title: data.title, url: data.url, tags: data.tags };
            await setSetting("soundsLibrary", library);
        } else {
            library[foundry.utils.randomID()] = { name: data.name, url: data.url, tags: data.tags };
            await setSetting("soundsLibrary", library);
        }
    }

    _onClose(options) {
        super._onClose(options);
    }
}
