import {getCellData} from "../app/HexplorerApp.js";
import { MODULE_ID } from "../main.js";
import { getSetting } from "../settings.js";
import { GridUtils } from "./GridUtils.js";

export class Fog {
    constructor(scene, hexplorer) {
        this.scene = scene;
        this.hexplorer = hexplorer;
        this.init();
        this.draw = foundry.utils.throttle(this.draw.bind(this), 100);
    }

    get alpha() {
        return game.user.isGM ? 0.5 : 1;
    }

    get showColor() {
        return getSetting("showColor");
    }

    init() {
        this.fog = new PIXI.Graphics();
        this.fog.sortLayer = foundry.canvas.groups.PrimaryCanvasGroup.SORT_LAYERS.TILES + 1;
        this.fog.elevation = 0;
        this.fog.sort = 1e100;
        canvas.primary.addChild(this.fog);
        //add blur filter
        const blurFilter = new PIXI.BlurFilter();
        blurFilter.blur = 5;
        this.blurFilter = blurFilter;
        //this.fog.filters = [blurFilter];
        if (this.scene.tokenVision) {
            this.fogMask = new PIXI.Graphics();
            canvas.masks.vision.addChild(this.fogMask);
            this.fogMask.filters = [blurFilter];
        }
        this.draw();
    }

    async setCells(cells, revealed = true, adjacentDepth = 0) {
        cells = GridUtils.getCells(cells, adjacentDepth);
        const current = this.scene.getFlag(MODULE_ID, "exploration") ?? {};
        const updatePermissionPages = [];
        const cellData = cells.reduce((acc, cell) => {
            const key = GridUtils.getKey(cell);
            const journalEntry = current[key]?.journalEntry ?? "";
            if (revealed && journalEntry) updatePermissionPages.push(journalEntry);
            acc[key] = { revealed: revealed };
            return acc;
        }, {});
        const exploration = foundry.utils.mergeObject(current, cellData);
        await this.scene.setFlag(MODULE_ID, "exploration", exploration);
        const pages = updatePermissionPages.map((uuid) => fromUuidSync(uuid));
        for (const page of pages) {
            await page.update({"ownership.default": CONST.DOCUMENT_OWNERSHIP_LEVELS.OBSERVER});
        }
        return exploration;
    }

    async processPath(start, end, tokenDocument) {
        const depth = tokenDocument.flags[MODULE_ID]?.revealRadius ?? 0;
        const startOffset = canvas.grid.getOffset(start);
        const cells = canvas.grid.getDirectPath([start, end]);
        for (const cell of cells) {
            if (cell.i === startOffset.i && cell.j === startOffset.j) continue;
            await this.hexplorer.moveIntoCell(cell, this.scene, tokenDocument);
            if(getSetting("autoTriggerExplore")) await this.hexplorer.exploreCell(cell, this.scene, tokenDocument);
        }
        await this.revealPath(start, end, depth);
    }

    async revealPath(start, end, adjacentDepth = 0) {
        const cells = canvas.grid.getDirectPath([start, end]);
        const cellsWithDepth = cells.map((c) => {
            const cellData = getCellData(c);
            const depth = cellData.revealRadius >= 0 ? cellData.revealRadius : adjacentDepth;
            return {cell: c, depth};
        });
        const depthMap = {};
        for (const {cell, depth} of cellsWithDepth) {
            depthMap[depth] ??= [];
            depthMap[depth].push(cell);
        }
        for(const [depth, cells] of Object.entries(depthMap)) {
            await this.reveal(cells, parseInt(depth));
        }
    }

    async reveal(cells, adjacentDepth = 0) {
        const connectedReveal = [];
        const exploration = this.scene.getFlag(MODULE_ID, "exploration") ?? {};
        for (const cell of cells) {
            const key = GridUtils.getKey(cell);
            const cellData = exploration[key] ?? {};
            if (cellData.reveal) {
                const connected = cellData.reveal
                    .split(",")
                    .map((c) => c.trim())
                    .filter((c) => c)
                    .map((c) => GridUtils.getOffsetFromKey(c));
                connectedReveal.push(...connected);
            }
        }
        if (connectedReveal.length) await this.reveal(connectedReveal, 0);
        return this.setCells(cells, true, adjacentDepth);
    }

    async hide(cells, adjacentDepth = 0) {
        return this.setCells(cells, false, adjacentDepth);
    }

    async reset() {
        const exploration = this.scene.getFlag(MODULE_ID, "exploration") ?? {};
        Object.keys(exploration).forEach((key) => {
            if (exploration[key].revealed) exploration[key].revealed = false;
        });
        await this.scene.setFlag(MODULE_ID, "exploration", exploration);
    }

    get drawColor() {
        const canSeeColor = game.user.isGM || this.scene.getFlag(MODULE_ID, "config")?.showColor;
        const showColor = this.showColor;
        return canSeeColor && showColor;
    }

    draw() {
        const fog = this.fog;
        const alpha = this.alpha;
        const gridColor = this.scene.grid.color ?? "#000000";
        const gridThickness = this.scene.grid.thickness ?? 1;
        fog.clear();
        const exploration = this.scene.getFlag(MODULE_ID, "exploration") ?? {};
        const rows = canvas.scene.dimensions.rows;
        const cols = canvas.scene.dimensions.columns;
        const minSize = Math.min(canvas.grid.sizeX, canvas.grid.sizeY);
        const cellShape = canvas.grid.getShape();
        const unexploredColor = canvas.scene.fog.colors.unexplored?.css || "#000000";
        for (let i = 0; i < rows; i++) {
            for (let j = 0; j < cols; j++) {
                const key = GridUtils.getKey({ i, j });
                const topLeft = canvas.grid.getCenterPoint({ i, j });
                const offsetShape = cellShape.map((p) => ({ x: p.x + topLeft.x, y: p.y + topLeft.y }));
                if (!exploration[key]?.revealed) {
                    fog.lineStyle(alpha, unexploredColor, alpha);
                    fog.beginFill(unexploredColor, alpha);
                    fog.drawPolygon(offsetShape);
                } else if (this.drawColor && exploration[key].color && exploration[key].color !== "#000000") {
                    fog.lineStyle(0, 0x000000, 0);
                    fog.beginFill(exploration[key].color, 0.5);
                    fog.drawPolygon(offsetShape);
                }
                if (exploration[key]?.journalEntry) {
                    //draw circle
                    const journal = fromUuidSync(exploration[key].journalEntry);
                    if(journal && journal.testUserPermission(game.user, CONST.DOCUMENT_OWNERSHIP_LEVELS.OBSERVER)) {
                        fog.lineStyle( 2, 0x000000, 1);
                        fog.beginFill(exploration[key].color, 1);
                        fog.drawCircle(topLeft.x, topLeft.y, minSize / 8);
                    }
                }
                /*
                if (exploration[key]?.discovered) {
                    fog.lineStyle( gridThickness, gridColor, 1);
                    fog.beginFill(gridColor, 0);
                    fog.drawPolygon(offsetShape);
                }
                */
                fog.endFill();
            }
        }
        this.drawFogMask();
    }

    drawFogMask() {
        const fogMask = this.fogMask;
        if(!fogMask) return;
        const alpha = 1;
        fogMask.clear();
        const exploration = this.scene.getFlag(MODULE_ID, "exploration") ?? {};
        const rows = canvas.scene.dimensions.rows;
        const cols = canvas.scene.dimensions.columns;
        const cellShape = canvas.grid.getShape();
        fogMask.lineStyle(0, 0x000000, 0);
        for (let i = 0; i < rows; i++) {
            for (let j = 0; j < cols; j++) {
                const key = GridUtils.getKey({ i, j });
                const topLeft = canvas.grid.getCenterPoint({ i, j });
                const offsetShape = cellShape.map((p) => ({ x: p.x + topLeft.x, y: p.y + topLeft.y }));
                fogMask.beginFill(exploration[key]?.revealed ? 0xffffff : 0x000000, alpha);
                fogMask.drawPolygon(offsetShape);
                fogMask.endFill();
            }
        }
    }

    destroy() {
        this.fog.removeFromParent();
        if(!this.fog.destroyed) this.fog.destroy(true);
        this.fogMask?.removeFromParent();
        if(!this.fogMask?.destroyed) this.fogMask?.destroy(true);
    }
}
