export async function moveToToken({ x, y, elevation }, token, sourceToken) {
    if (game.paused) return ui.notifications.error("GAME.PausedWarning", {localize : true});
    let occupiedPositionBox;
    if (token) {
        occupiedPositionBox = {
            x: token.document.x,
            y: token.document.y,
            width: token.document.width * canvas.grid.size,
            height: token.document.height * canvas.grid.size,
        };
        x = token.document.x;
        y = token.document.y;
        elevation = token.document.elevation;
    } else {
        occupiedPositionBox = {
            x: x,
            y: y,
            width: canvas.grid.size,
            height: canvas.grid.size,
        };
    }

    const sourceOccupiedPositionBox = {
        x: sourceToken.document.x,
        y: sourceToken.document.y,
        width: sourceToken.document.width * canvas.grid.size,
        height: sourceToken.document.height * canvas.grid.size,
    };

    if (occupiedPositionBox.x === sourceOccupiedPositionBox.x && occupiedPositionBox.y === sourceOccupiedPositionBox.y) {
        return;
    }

    //find the closest grid space to the target

    const targetEdge = {
        x: occupiedPositionBox.x - sourceOccupiedPositionBox.width,
        y: occupiedPositionBox.y - sourceOccupiedPositionBox.height,
        width: occupiedPositionBox.width + sourceOccupiedPositionBox.width * 2,
        height: occupiedPositionBox.height + sourceOccupiedPositionBox.height * 2,
    };

    //loop all grid spaces in the target edge
    let closestGridSpace = null;
    let closestDistance = Infinity;
    const possiblePositions = [];

    const computeDistance = (x, y, debug) => {
        const gridCenter = {
            x: x + canvas.grid.size / 2,
            y: y + canvas.grid.size / 2,
        };
        //check if the grid space is occupied
        if (
            canvas.tokens.placeables.some((t) => {
                if (t.id === sourceToken.id) {
                    return false;
                }
                const otherOccupiedPositionBox = {
                    x: t.document.x,
                    y: t.document.y,
                    width: t.document.width * canvas.grid.size,
                    height: t.document.height * canvas.grid.size,
                };
                if (gridCenter.x < otherOccupiedPositionBox.x + otherOccupiedPositionBox.width && gridCenter.x > otherOccupiedPositionBox.x && gridCenter.y < otherOccupiedPositionBox.y + otherOccupiedPositionBox.height && gridCenter.y > otherOccupiedPositionBox.y) {
                    return true;
                }
            })
        )
            return;

        const sourceCenter = {
            x: sourceOccupiedPositionBox.x + sourceOccupiedPositionBox.width / 2,
            y: sourceOccupiedPositionBox.y + sourceOccupiedPositionBox.height / 2,
        };
        const targetCenter = {
            x: x + occupiedPositionBox.width / 2,
            y: y + occupiedPositionBox.height / 2,
        };
        const distance = Math.sqrt(Math.pow(targetCenter.x - sourceCenter.x, 2) + Math.pow(targetCenter.y - sourceCenter.y, 2));
        if (distance < closestDistance) {
            closestDistance = distance;
            closestGridSpace = {x, y};
            possiblePositions.push({
                x,
                y,
                distance,
            });
        }
    };

    const checkCollision = (x, y, elevation) => {
        const gridCenter = {
            x: x + canvas.grid.size / 2,
            y: y + canvas.grid.size / 2,
        };
        return sourceToken.checkCollision({ x: gridCenter.x, y: gridCenter.y, elevation });
    }

    const completeMove = async (x, y, elevation) => {
        return await sourceToken.document.update({ x, y, elevation });
    };

    if (!token) {
        computeDistance(occupiedPositionBox.x, occupiedPositionBox.y);
        if (Number.isFinite(closestDistance) && !checkCollision(closestGridSpace.x, closestGridSpace.y, elevation)) {
            return await completeMove(closestGridSpace.x, closestGridSpace.y, elevation);
        }
    }

    //loop the top row
    for (let x = targetEdge.x; x <= targetEdge.x + targetEdge.width - sourceOccupiedPositionBox.width; x += canvas.grid.size) {
        computeDistance(x, targetEdge.y, true);
    }

    //loop the bottom row
    for (let x = targetEdge.x; x <= targetEdge.x + targetEdge.width - sourceOccupiedPositionBox.width; x += canvas.grid.size) {
        computeDistance(x, targetEdge.y + targetEdge.height - sourceOccupiedPositionBox.height, true);
    }

    //loop the left column minus the corners
    for (let y = targetEdge.y; y < targetEdge.y + targetEdge.height - sourceOccupiedPositionBox.height + canvas.grid.size; y += canvas.grid.size) {
        computeDistance(targetEdge.x, y);
    }

    //loop the right column minus the corners
    for (let y = targetEdge.y; y < targetEdge.y + targetEdge.height - sourceOccupiedPositionBox.height + canvas.grid.size; y += canvas.grid.size) {
        computeDistance(targetEdge.x + targetEdge.width - sourceOccupiedPositionBox.width, y);
    }

    if (possiblePositions.length === 0) return ui.notifications.error("RULER.MovementNotAllowed", {localize : true});

    //sort the possible positions by distance
    possiblePositions.sort((a, b) => a.distance - b.distance);
    
    //find the first valid position with no collision
    const finalPosition = possiblePositions.find((position) => !checkCollision(position.x, position.y, elevation));

    if (!finalPosition) return ui.notifications.error("RULER.MovementCollision", {localize : true});
    //move the token to the closest grid space
    return await completeMove(finalPosition.x, finalPosition.y, elevation);
}
