import { Socket } from "./lib/socket.js";

export class Quickdraw {
    constructor(data) {
        this.points = data?.points || [];
        this.canceled = false;
        this.debounce = false;
        this.pollRate = game.settings.get("quickdraw", "pollrate");
        this.fadeoutTimer = game.settings.get("quickdraw", "fadeoutDelay") || data?.fadeOutTimer || 1000;
        this.color = data?.color || game?.user?.color?.css?.replace("#", "0x") || 0xffffff;
        this.width = data?.width || game.settings.get("quickdraw", "linethicc") || 2;
        this.preview = new PIXI.Graphics();
        canvas.controls.debug.addChild(this.preview);
        this.id = data?.id || foundry.utils.randomID();
    }

    start() {
        $(document).on("mousemove", this._onMouseMove);
    }

    end() {
        $(document).off("mousemove", this._onMouseMove);
        canvas.controls.debug.removeChild(this.preview);
        game.QuickdrawHandler.previews[this.id]?.clear();
        if (this.canceled) return;
        Socket.draw( {
            points: this.points,
            color: this.color,
            width: this.width,
            id: this.id,
            fadeoutTimer: this.fadeoutTimer,
        });
    }

    draw() {
        const points = this.points;
        const line = new PIXI.Graphics();
        line.lineStyle(this.width, this.color, 1);
        line.moveTo(points[0], points[1]);
        for (let i = 2; i < points.length; i += 2) {
            line.lineTo(points[i], points[i + 1]);
        }
        const s = new PIXI.Sprite();
        s.addChild(line);
        function fadeOut() {
            s.alpha -= 0.05;
            if (s.alpha <= 0) {
                canvas.app.ticker.remove(fadeOut);
                s.destroy();
            }
        }
        setTimeout(() => {
            canvas.app.ticker.add(fadeOut);
        }, this.fadeoutTimer);

        canvas.controls.debug.addChild(s);
        game.QuickdrawHandler.previews[this.id]?.clear();
    }

    _onMouseMove = (e) => {
        if (!this.debounce) {
            this.debounce = true;
            setTimeout(() => {
                this.debounce = false;
            }, this.pollRate);
            const [x, y] = [e.clientX, e.clientY];
            const t = canvas.stage.worldTransform;
            const cX = (x - t.tx) / canvas.stage.scale.x;
            const cy = (y - t.ty) / canvas.stage.scale.y;
            this.points.push(cX, cy);
            Socket.drawPreview({
                points: this.points,
                color: this.color,
                width: this.width,
                id: this.id,
            });
        }
    };

    drawPreview() {
        this.preview.clear();
        const points = this.points;
        const line = this.preview;
        line.lineStyle(this.width, this.color, 1);
        line.moveTo(points[0], points[1]);
        for (let i = 2; i < points.length; i += 2) {
            line.lineTo(points[i], points[i + 1]);
        }
    }
}

export class QuickdrawHandler {
    constructor() {
        Socket.register("draw", QuickdrawHandler.socketQuickdraw);
        Socket.register("drawPreview", QuickdrawHandler.socketQuickdrawPreview);
        globalThis.game.QuickdrawHandler = this;
        this.current = null;
        this.hotkey = null;
        this.previews = {};
        //this.activateListeners();
    }

    static socketQuickdraw(data) {
        new Quickdraw(data).draw();
    }

    static socketQuickdrawPreview(data) {
        game.QuickdrawHandler.previews[data.id]?.clear();
        if (!game.QuickdrawHandler.previews[data.id]) game.QuickdrawHandler.previews[data.id] = new PIXI.Graphics();
        const points = data.points;
        const line = game.QuickdrawHandler.previews[data.id];
        line.lineStyle(data.width, data.color, 1);
        line.moveTo(points[0], points[1]);
        for (let i = 2; i < points.length; i += 2) {
            line.lineTo(points[i], points[i + 1]);
        }
        canvas.controls.debug.addChild(line);
    }
}
