import { Socket } from "../lib/socket.js";
import { MODULE_ID } from "../main.js";
import { getSetting } from "../settings.js";

export class EpicRoll extends Application {
    constructor(rollData) {
        super();
        this.rollData = rollData;
        this._resolve;
        this._reject;
        this.promise = new Promise((resolve, reject) => {
            this._resolve = resolve;
            this._reject = reject;
        });
    }

    init() {
        const recovered = this.rollData.recovered;
        this.prepareData();
        ui.EpicRolls5e._currentRoll = this;
        this._results = {};
        this._rolls = {};
        this._messageIds = new Set();
        if (recovered) {
            this._recovered = true;
            this._results = recovered.results;
            this._rolls = recovered.rolls;
            this._messageIds = new Set(recovered.messageIds);
        }
        return this;
    }

    static get APP_ID() {
        return this.name
            .split(/(?=[A-Z])/)
            .join("-")
            .toLowerCase();
    }

    get APP_ID() {
        return this.constructor.APP_ID;
    }

    static get defaultOptions() {
        return foundry.utils.mergeObject(super.defaultOptions, {
            id: this.APP_ID,
            template: `modules/${MODULE_ID}/templates/${this.APP_ID}.hbs`,
            popOut: false,
            title: game.i18n.localize(`${MODULE_ID}.${this.APP_ID}.title`),
        });
    }

    prepareData() {
        this.actors = this.rollData.actors.map((actor) => fromUuidSync(actor));
        this.contestants = this.rollData.contestants.map((actor) => fromUuidSync(actor));
        this.rollOptions = this.rollData.options;
        this.rollOptions.showDC = this.rollOptions.showDC || game.user.isGM;
        this.rollOptions.hideNames = this.rollOptions.hideNames;
        this.type = this.rollData.type;
        this.contest = this.rollData.contest;
        const autoColor = this.rollOptions.autoColor ? this.getAutoColor() : null;
        const color = autoColor ?? this.rollOptions.color ?? 0;
        document.documentElement.style.setProperty("--epic-rolls-banner-hue", `${color}deg`);
    }

    getAutoColor() {
        const [type, key] = this.rollData.type.split(".");
        const ability = CONFIG.DND5E.skills[key]?.ability || Object.keys(CONFIG.DND5E.abilities).find((a) => a === key);
        let colorKey = ability ?? type;
        return ROLL_COLORS[colorKey];
    }

    async getData() {
        const showDC = this.rollOptions.showDC || game.user.isGM;
        const introLabel = EpicRoll.getRollLabel(this.rollData.type, showDC ? this.rollData.options.DC : null, this.rollData.contest, this.rollData.options);
        this._introLabel = introLabel;
        return { introLabel, actors: this.actors, contestants: this.contestants, options: this.rollOptions, isGM: game.user.isGM };
    }

    activateListeners(html) {
        super.activateListeners(html);
        html = html[0] ?? html;
        this.executeIntroAnimation(html);
        html.querySelectorAll("span.adv, span.dis").forEach((span) => {
            span.addEventListener("click", this._onClickAdvDis.bind(this));
        });
        html.querySelectorAll(".roll").forEach((roll) => {
            roll.addEventListener("click", this.roll.bind(this));
        });
        if (this.rollOptions.allowReroll || game.user.isGM) {
            html.querySelectorAll(".result").forEach((roll) => {
                roll.addEventListener("click", this.roll.bind(this));
            });
        }
        html.querySelector(".end-epic-roll").addEventListener("click", () => {
            Socket.endEpicRoll({ abort: true });
        });
        html.querySelector(".end-epic-roll-manual").addEventListener("click", (e) => {
            e.preventDefault();
            Socket.endEpicRoll({ button: true });
            e.currentTarget.classList.add("er5e-hidden-2");
        });
        this.recoverState();
        this.setAdvDis();
    }

    setAdvDis() {
        const rollSettings = this.rollOptions.rollSettings;
        if (!rollSettings?.length) return;
        for (const rollSetting of rollSettings) {
            const uuid = rollSetting.uuid;
            const actorCard = this.element[0].querySelector(`.actor-card[data-uuid="${uuid}"]`);
            if (!actorCard) continue;
            const rollBadge = actorCard.querySelector(".roll-badge");
            if (!rollBadge) continue;
            const adv = rollBadge.querySelector("span.adv");
            const dis = rollBadge.querySelector("span.dis");
            if (rollSetting.advantage) adv.classList.add("active");
            if (rollSetting.disadvantage) dis.classList.add("active");
            if (rollSetting.autoRoll && game.users.activeGM === game.user) {
                setTimeout(() => {
                    this.roll({ currentTarget: actorCard.querySelector(".roll") });
                }, 1000);
            }
        }
    }

    _onClickAdvDis(e) {
        const span = e.currentTarget;
        const other = Array.from(span.closest(".roll-badge").querySelectorAll("span.adv, span.dis")).find((s) => s !== span);
        if (other.classList.contains("active")) {
            other.classList.remove("active");
        }
        span.classList.toggle("active");
    }

    async executeIntroAnimation(html) {
        const soundPath = getSetting("introSound");
        if (soundPath) {
            foundry.audio.AudioHelper.play({ src: soundPath, volume: 0.8, autoplay: true, loop: false }, false);
        }

        const introText = html.querySelector(".intro-text");
        //remove it when the animation is done
        const introTextPromise = new Promise((resolve) => {
            introText.addEventListener("animationend", () => {
                introText.remove();
                resolve();
            });
        });
        await introTextPromise;
        //unhide actor cards
        const actorCardsContainer = html.querySelector(".actor-cards");
        actorCardsContainer.classList.remove("er5e-hidden");
        //set transform translate to 0 on actor cards in sequence
        const actorCards = actorCardsContainer.querySelectorAll(".actor-card");
        //animate the cards from transform translate -100vw to 0
        const cardCount = actorCards.length;
        actorCards.forEach((card, index) => {
            card.animate([{ transform: `translateX(-${100 * (index + 1)}vw)` }, { transform: "translateX(0)" }], {
                duration: 700,
                easing: "ease-in-out",
                fill: "forwards",
                delay: (cardCount - index) * 150,
            });
        });
    }

    async executeOutroAnimation(html, isSuccess) {
        const soundPath = isSuccess ? getSetting("successSound") : getSetting("failureSound");

        //wait 2 seconds before starting the outro animation
        await EpicRoll.wait(2000);
        const actorCardsContainer = html.querySelector(".actor-cards");
        const actorCards = actorCardsContainer.querySelectorAll(".actor-card");
        //animate the cards from transform translate 0 to -100vw
        const cardCount = actorCards.length;
        actorCards.forEach((card, index) => {
            card.animate([{ transform: "translateX(0)" }, { transform: `translateX(100vw)` }], {
                duration: 500,
                easing: "ease-in-out",
                fill: "forwards",
                delay: (cardCount - index) * 100,
            });
        });
        //wait for all animations to finish
        await EpicRoll.wait(500 + cardCount * 100);
        actorCardsContainer.remove();

        //display end text
        if (this.rollOptions.showRollResults && isSuccess !== undefined) {
            const outroText = html.querySelector(".outro-text");
            outroText.textContent = game.i18n.localize(`${MODULE_ID}.${isSuccess ? "success" : "failure"}`) + "!";
            outroText.classList.remove("er5e-hidden");
            //play fade in animation on outro text
            outroText.animate([{ opacity: 0 }, { opacity: 1 }], {
                duration: 300,
                easing: "ease-in-out",
                fill: "forwards",
            });
            if (soundPath) foundry.audio.AudioHelper.play({ src: soundPath, volume: 0.8, autoplay: true, loop: false }, false);
            await EpicRoll.wait(3000);
        }
        //fade out the html
        html.animate([{ opacity: 1 }, { opacity: 0 }], {
            duration: 500,
            easing: "ease-in-out",
            fill: "forwards",
        });
        await EpicRoll.wait(500);
        await this.close();
        return true;
    }

    async roll(e) {
        const rollButton = e.currentTarget;
        const uuid = rollButton.closest(".actor-card").dataset.uuid;
        const adv = rollButton.closest(".roll-badge").querySelector("span.adv").classList.contains("active");
        const dis = rollButton.closest(".roll-badge").querySelector("span.dis").classList.contains("active");
        Socket.toggleRollButton({ uuid, rolling: true });
        const ff = adv || dis || !e.shiftKey;
        if (ff) e = null;
        const actor = fromUuidSync(uuid);
        const isContestant = this.contestants.some((c) => c.uuid === uuid);
        const rollType = isContestant ? this.contest : this.type;
        const [type, key] = rollType.split(".");
        const rollOptions = { event: e, fastForward: ff, advantage: adv, disadvantage: dis, chatMessage: false };

        const isBlind = this.rollOptions.blindRoll;

        rollOptions.rollMode = isBlind ? "blindroll" : "publicroll";

        let result;

        if (type === "skill") {
            result = await actor.rollSkill({skill: key, ...rollOptions}, {}, {create: false});
        } else if (type === "save") {
            if (key === "death") {
                result = await actor.rollDeathSave({legacy: false, ...rollOptions}, {}, {create: false});
            } else {
                result = await actor.rollSavingThrow({ability: key, ...rollOptions}, {}, {create: false});
            }
        } else if (type === "check") {
            result = await actor.rollAbilityTest({ability: key, ...rollOptions}, {}, {create: false});
        } else if (type === "tool") {
            result = await actor.rollToolCheck({tool: key, ...rollOptions}, {}, {create: false});
        } else if (type === "custom") {
            const roll = new Roll(this.rollOptions.formula, actor.system);
            result = await roll.roll();
        } else if (type === "initiative") {
            const token = actor.token ?? actor.getActiveTokens()[0];
            const tokenDocument = token.document ?? token;
            if (!tokenDocument.inCombat) await tokenDocument.toggleCombatant();
            const combatant = tokenDocument.combatant;
            const roll = combatant.getInitiativeRoll();
            await roll.evaluate();
            await combatant.update({ "initiative": roll.total });
            result = roll;
        }

        if (!result) {
            Socket.toggleRollButton({ uuid, rolling: false });
            return;
        }

        result = Array.isArray(result) ? result[0] : result;

        const value = result.total;

        const message = await result.toMessage({ speaker: ChatMessage.getSpeaker({ actor }) });

        if (!isBlind) await this.waitFor3DDice(message.id);

        Socket.toggleRollButton({ uuid, rolling: false });

        Socket.updateEpicRoll({ uuid, value, isCritical: result.isCritical, isFumble: result.isFumble, rollData: result, messageId: message?.id });
    }

    async waitFor3DDice(messageId) {
        if (!game.modules.get("dice-so-nice")?.active) return true;
        await EpicRoll.wait(10);
        const msgEl = document.querySelector(`#chat-log .message[data-message-id="${messageId}"]`);
        if (!msgEl?.classList.contains("dsn-hide")) return true;
        return await game.dice3d.waitFor3DAnimationByMessageID(messageId);
    }

    toggleRollButton(uuid, rolling = false) {
        const actorCard = this.element[0].querySelector(`.actor-card[data-uuid="${uuid}"]`);
        const rollButton = actorCard.querySelector(".roll");
        rollButton.style.pointerEvents = rolling ? "none" : "auto";
        rollButton.classList.toggle("fa-shake", rolling);
    }

    recoverState() {
        if (!this._recovered) return;
        for (const [uuid, value] of Object.entries(this._results)) {
            const actorCard = this.element[0].querySelector(`.actor-card[data-uuid="${uuid}"]`);
            const data = this._rolls[uuid];
            const resultEl = actorCard.querySelector(".result");
            const hideResult = this.rollOptions.blindRoll && !game.user.isGM;
            resultEl.textContent = hideResult ? "?" : value;
            resultEl.classList.remove("er5e-hidden");
            if (data.isCritical && !hideResult) resultEl.classList.add("critical");
            if (data.isFumble && !hideResult) resultEl.classList.add("fumble");
            actorCard.querySelector(".roll").classList.add("er5e-hidden");
            if (this.rollOptions.blindRoll) resultEl.classList.add("blind");
        }
        delete this._recovered;
    }

    update(data) {
        this._results[data.uuid] = data.value;
        this._rolls[data.uuid] = data.rollData;
        if (data.messageId) this._messageIds.add(data.messageId);
        const actorCard = this.element[0].querySelector(`.actor-card[data-uuid="${data.uuid}"]`);
        const resultEl = actorCard.querySelector(".result");
        resultEl.textContent = this.rollOptions.blindRoll && !game.user.isGM ? "?" : data.value;
        resultEl.classList.remove("er5e-hidden");
        actorCard.querySelector(".roll").classList.add("er5e-hidden");
        if (data.isCritical) resultEl.classList.add("critical");
        if (data.isFumble) resultEl.classList.add("fumble");
        if (this.rollOptions.blindRoll) resultEl.classList.add("blind");
        //play animation on actor card, scale up then slowly scale back to normal
        actorCard.animate([{ transform: "scale(1.0)" }, { transform: "scale(1.1)" }, { transform: "scale(1.0)" }], {
            duration: 500,
            easing: "cubic-bezier(0.22, 1, 0.36, 1)",
            fill: "forwards",
        });
        //check if all results are in
        const allActors = this.actors.concat(this.contestants);
        const allResults = allActors.every((actor) => Number.isNumeric(this._results[actor.uuid]));
        if (allResults) {
            Socket.endEpicRoll({ abort: false });
        }
    }

    async endEpicRoll({ abort, button }) {
        if (this._ending) return;
        this._ending = true;
        if (abort) {
            this.resolveRoll(abort);
            return this.close();
        }
        const allowReroll = this.rollOptions.allowReroll;
        if (!button && allowReroll) {
            if (game.user.isGM) this.element[0].querySelector(".end-epic-roll-manual").classList.remove("er5e-hidden-2");
            this._ending = false;
            return;
        }
        const isSuccess = this.computeSuccess();
        this.resolveRoll(abort, isSuccess);
        this.computeInitiative();
        await this.executeOutroAnimation(this.element[0], isSuccess);
    }

    async computeInitiative() {
        const isInitiative = this.type === "initiative.initiative";
        if (!isInitiative || !this.rollOptions.useAverage) return;
        const playerOwned = this.actors.filter((a) => a.hasPlayerOwner).map((a) => a.token?.document ?? a.getActiveTokens()[0]);
        const nonPlayerOwned = this.actors.filter((a) => !a.hasPlayerOwner).map((a) => a.token?.document ?? a.getActiveTokens()[0]);
        const playerAverageInitiative = playerOwned.reduce((acc, t) => acc + t.combatant.initiative, 0) / playerOwned.length;
        const nonPlayerAverageInitiative = nonPlayerOwned.reduce((acc, t) => acc + t.combatant.initiative, 0) / nonPlayerOwned.length;
        const updates = [];
        for (const playerOwnedToken of playerOwned) {
            updates.push({
                _id: playerOwnedToken.combatant.id,
                "initiative": playerAverageInitiative,
            });
        }
        for (const nonPlayerOwnedToken of nonPlayerOwned) {
            updates.push({
                _id: nonPlayerOwnedToken.combatant.id,
                "initiative": nonPlayerAverageInitiative,
            });
        }
        await game.combat.updateEmbeddedDocuments("Combatant", updates);
    }

    computeSuccess() {
        if (game.user === game.users.activeGM && getSetting("cleanupMessages")) {
            ChatMessage.deleteDocuments(Array.from(this._messageIds));
        }
        if (!Number.isNumeric(this.rollOptions.DC) && !this.contestants.length) {
            this.createChatRecap(null, null);
            return undefined;
        }
        const dc = this.contestants.length ? this.contestants.reduce((acc, c) => acc + this._results[c.uuid], 0) / this.contestants.length : this.rollOptions.DC;
        let success;
        if (this.rollOptions.useAverage) {
            const average = this.actors.reduce((acc, a) => acc + this._results[a.uuid], 0) / this.actors.length;
            success = average >= dc;
        } else {
            const successCount = this.actors.filter((a) => this._results[a.uuid] >= dc).length;
            const half = Math.ceil(this.actors.length / 2);
            success = successCount >= half;
        }
        this.createChatRecap(dc, success);
        return success;
    }

    async createChatRecap(dc, success) {
        if (game.user !== game.users.activeGM) return;
        if ((this, this.rollOptions.noMessage)) return;
        const recapSetting = getSetting("recapMessage");

        if (recapSetting === "none") return;

        const resultLabel = game.i18n.localize(`${MODULE_ID}.${success ? "success" : "failure"}`);

        const actorEntries = [];
        const contestantEntries = [];
        for (const actor of this.actors) {
            const result = this._results[actor.uuid];
            const roll = this._rolls[actor.uuid];
            const entry = {
                actor,
                result,
                roll,
                success: result >= dc,
            };
            actorEntries.push(entry);
        }
        for (const contestant of this.contestants) {
            const result = this._results[contestant.uuid];
            const roll = this._rolls[contestant.uuid];
            const entry = {
                actor: contestant,
                result,
                roll,
                success: result >= dc,
            };
            contestantEntries.push(entry);
        }

        const template = `modules/${MODULE_ID}/templates/chat-recap.hbs`;
        const html = await renderTemplate(template, {
            label: this._introLabel,
            dc,
            successLabel: success !== undefined ? resultLabel : null,
            success,
            actors: actorEntries,
            contestants: contestantEntries,
            noDC: !Number.isNumeric(dc),
        });

        ChatMessage.create({
            user: game.user.id,
            whisper: recapSetting === "gm" || this.rollOptions.hideNames ? ChatMessage.getWhisperRecipients("GM") : null,
            speaker: { alias: game.i18n.localize(`${MODULE_ID}.epicRoll`) },
            content: html,
        });
    }

    async resolveRoll(abort, isSuccess) {
        const results = [];
        for (const uuid of Object.keys(this._results)) {
            const actor = fromUuidSync(uuid);
            results.push({ actor, value: this._results[uuid], roll: this._rolls[uuid] });
        }
        this._resolve({ canceled: abort, results, success: isSuccess });
    }

    static async wait(ms) {
        return new Promise((resolve) => setTimeout(resolve, ms));
    }

    static getRollLabel(rollKey, dc, vs, options = {}) {
        if (options.customLabel) return options.customLabel;
        if (rollKey === "initiative.initiative") return game.i18n.localize("COMBAT.InitiativeRoll");
        const formula = options.formula;
        let rollName = "";
        if (Number.isNumeric(dc) && !vs) {
            rollName = game.i18n.format(`DND5E.SaveDC`, { dc: dc, ability: "" });
            rollName += " ";
        }

        const getLabel = (rKey) => {
            const [type, key] = rKey.split(".");
            let label = "";
            if (type === "skill") {
                label += CONFIG.DND5E.skills[key].label;
            }
            if (type === "save" || type === "check") {
                label += CONFIG.DND5E.abilities[key]?.label ?? game.i18n.localize(`${MODULE_ID}.death`);
            }
            if (type === "tool") {
                label += key;
            }
            if(formula) {
                label += formula;
            }

            label += " ";

            if (type === "skill" || type === "check" || type === "tool" || type === "custom") {
                label += game.i18n.localize(`${MODULE_ID}.check`);
            }
            if (type === "save") {
                label += game.i18n.localize(`${MODULE_ID}.save`);
            }

            //remove double spaces
            label = label.replace(/\s+/g, " ");
            return label;
        };

        if (vs) {
            rollName += getLabel(rollKey) + " vs " + getLabel(vs);
        } else {
            rollName += getLabel(rollKey);
        }
        return rollName;
    }

    async close(...args) {
        ui.EpicRolls5e._currentRoll = null;
        const res = await super.close(...args);
        //advance the queue if there is one
        if (ui.EpicRolls5e._queue.length) {
            const next = ui.EpicRolls5e._queue.shift();
            next.init().render(true);
        }
        return res;
    }
}

const ROLL_COLORS = {
    str: 0,
    dex: 60,
    con: 120,
    int: 180,
    wis: 240,
    cha: 300,
    death: 0,
    check: 180,
    save: 0,
    skill: 300,
    tool: 120,
};
