import { getCellData } from "../app/HexplorerApp.js";
import { MODULE_ID } from "../main.js";
import { GridUtils } from "./GridUtils.js";

export class HexplorerDrawPreviewLayer {
    constructor(scene) {
        this.scene = scene;
        this.init();
        this.previewCells = new Map();
        this.hoverCells = new Map();
    }

    get cells() {
        return Array.from(this.previewCells.values()).map(({cell}) => cell);
    }

    init() {
        this.overlay = new PIXI.Graphics();
        this.overlay.zIndex = 20;
        canvas.primary.addChild(this.overlay);
    }

    addPreview(cells, color) {
        cells = Array.isArray(cells) ? cells : [cells];
        let changed = false;
        for (const cell of cells) {
            const key = GridUtils.getKey(cell);
            if (this.previewCells.has(key)) continue;
            this.previewCells.set(key, {cell, color});
            changed = true;
        }
        if (changed) this.draw();
    }

    #hoverCacheKey = "";

    get hoverCacheKey() {
        return this.#hoverCacheKey;
    }

    set hoverCacheKey(key) {
        this.#hoverCacheKey = key;
    }

    generateHoverCacheKey(cell, radius, color) {
        return `${GridUtils.getKey(cell)}-${radius}-${color}`;
    }

    updateHoverPreview(centerCell, color, radius) {
        const currentKey = this.hoverCacheKey;
        const key = this.generateHoverCacheKey(centerCell, radius, color);
        if (currentKey === key) return;
        const cells = GridUtils.getCells(centerCell, radius);
        this.hoverCacheKey = key;
        this.hoverCells.clear();
        for (const cell of cells) {
            const key = GridUtils.getKey(cell);
            this.hoverCells.set(key, {cell, color});
        }
        this.draw();
    }

    clear() {
        this.previewCells.clear();
        this.hoverCells.clear();
        this.hoverCacheKey = "";
        this.draw();
    }

    draw() {
        const overlay = this.overlay;
        overlay.clear();
        const cellShape = canvas.grid.getShape();

        const drawArray = Array.from(this.previewCells.values());

        for (const {cell, color} of drawArray) {
            overlay.lineStyle(0, 0x000000, 0);
            overlay.beginFill(color, 1);
            const topLeft = canvas.grid.getCenterPoint(cell);
            const offsetShape = cellShape.map((p) => ({ x: p.x + topLeft.x, y: p.y + topLeft.y }));
            overlay.drawPolygon(offsetShape);
            overlay.endFill();
        }

        const hoverArray = Array.from(this.hoverCells.values());
        for (const {cell, color} of hoverArray) {
            overlay.lineStyle(0, 0x000000, 0);
            overlay.beginFill(color, 1);
            const topLeft = canvas.grid.getCenterPoint(cell);
            const offsetShape = cellShape.map((p) => ({ x: p.x + topLeft.x, y: p.y + topLeft.y }));
            overlay.drawPolygon(offsetShape);
            overlay.endFill();
        }
    }

    destroy() {
        this.overlay.removeFromParent();
        if(!this.overlay.destroyed) this.overlay.destroy(true);
    }
}
