import {getSetting} from "../settings";

export class GridUtils {
    static getCells(cells, adjacentDepth = 0) {
        cells = Array.isArray(cells) ? cells : [cells];
        cells = cells.map(c => canvas.grid.getOffset(c));
        if (adjacentDepth > 0) {
            let cellsToCheck = cells;
            const checked = new Set();
            for (let i = 0; i < adjacentDepth; i++) {
                const adjacentCells = [];
                cellsToCheck.forEach((cell) => {
                    const key = GridUtils.getKey(cell);
                    if (checked.has(key)) return;
                    checked.add(key);
                    const adjacent = canvas.grid.getAdjacentOffsets(cell);
                    adjacentCells.push(...adjacent);
                });
                cells = cells.concat(adjacentCells);
                cellsToCheck = adjacentCells;
            }
        }
        cells = Object.values(cells.reduce((acc, cell) => {
            acc[GridUtils.getKey(cell)] = cell;
            return acc;
        }, {}));
        return cells;
    }

    static getKey(cell) {
        return `${cell.i}-${cell.j}`;
    }

    static getOffsetFromKey(key) {
        const [i, j] = key.split("-");
        return { i: parseInt(i), j: parseInt(j) };
    }

    static async executeSegmentedMovement(points, tokenDocument) {
        const offsetOrigin = GridUtils.offsetToCenter({x: tokenDocument.x, y: tokenDocument.y}, tokenDocument);

        const startOffset = canvas.grid.getOffset(offsetOrigin);
        const cells = [offsetOrigin, ...points];
        const pointsOnPath = [];
        for (const cell of cells) {
            if (cell.i === startOffset.i && cell.j === startOffset.j) continue;
            pointsOnPath.push(canvas.grid.getTopLeftPoint(cell));
        }
        for (const point of pointsOnPath) {
            await tokenDocument.update({x: point.x, y: point.y}, {hexplorerConfirmed: true, animation: {duration: getSetting("hexTime")}});
            const animations = Array.from(tokenDocument.object.animationContexts.values()).map(a => a.promise);
            await Promise.all(animations);
        }
        
    }

    static offsetToCenter(point, tokenDocument) {
        const sizeOffset = {x: tokenDocument.width * canvas.grid.sizeX / 2, y: tokenDocument.height * canvas.grid.sizeY / 2};
        return {x: point.x + sizeOffset.x, y: point.y + sizeOffset.y};
    }
}