export class DistanceTooltip {
  constructor(originToken, destinationToken) {
    this.destinationToken = destinationToken;
    this.distance = canvas.grid.measurePath([{ x: originToken.center.x, y: originToken.center.y },
      { x: destinationToken.center.x, y: destinationToken.center.y }], {
      gridSpaces: true,
    }).distance;
  }

  static draw() {
    this.tooltip.visible=false
    if(!this.distanceTooltip.parent){
      this.addChild(this.distanceTooltip);
    }
    const origin = canvas?.tokens?.controlled[0];
    if (!origin || origin.id == this.id || this?.document?.disposition === CONST.TOKEN_DISPOSITIONS.SECRET) return;
    const ray = new Ray(
      { x: this.center.x, y: this.center.y },
      { x: origin.center.x, y: origin.center.y }
    );
    const distance = DistanceTooltip.measureMinTokenDistance(this,origin)
    this.distanceTooltip.removeChildren().forEach((c) => c.destroy());
    const band = DistanceTooltip.getRangeBand(distance)
    // Create the tooltip Text
    let tip;
    if (!band) {
      tip = `${distance} ${canvas?.scene?.grid?.units}.`;
    } else {
      const rangeBandsNumbers = game.settings.get("hover-distance", "rangeBandsNumbers");
      tip = rangeBandsNumbers ? `${band} (${distance} ${canvas?.scene?.grid?.units}).` : `${band}`;
    }
    if (!tip.length) return;
    const style = CONFIG.canvasTextStyle.clone();
    style.fontSize = Math.max(Math.round(canvas.dimensions.size * 0.36 * 12) / 12, 36);
    const text = new PreciseText(tip, style);
    this.distanceTooltip.addChild(text);

    // Add the tooltip at the top of the parent Token container
    const distanceTooltipPosition = game.settings.get("hover-distance", "tooltipPosition");
    switch (distanceTooltipPosition) {
      case "top":
        this.distanceTooltip.position.set(this.w / 2, -2);
        text.anchor.set(0.5, 1);
        break;
      case "bottom":
        this.distanceTooltip.position.set(this.w / 2, this.h + 2);
        text.anchor.set(0.5, 0);
        break;
    }
    
    DistanceTooltip.highlightGrid(this)
  }

  static clear() {
    this.tooltip.visible=true
    this.distanceTooltip?.removeChildren()?.forEach((c) => c.destroy());
    DistanceTooltip.clearGrid(this);
  }
  static async tokenDraw(wrapped, ...args) {
    await wrapped(...args);
    this.distanceTooltip = this.addChild(new PIXI.Container());
    return this;
  }

  static highlightGrid(token) {
    if(!game.settings.get("hover-distance", "highlightGrid") || !token.isVisible) return;
    if (!token.HDHighlight) {
      token.HDHighlight = new PIXI.Graphics();
      token.addChildAt(token.HDHighlight, 0);
    }
    const highlight = token.HDHighlight;
    highlight.clear();
    const shape = new PIXI.Rectangle(0,0,token.w,token.h);
    const color = 0x0000ff;
    const border = 0x000000;
    const alpha = 0.25;
    highlight.beginFill(color, alpha);
    if (border) highlight.lineStyle(2, border, Math.min(alpha * 1.5, 1.0));
    highlight.drawShape(shape).endFill();
  }
  static clearGrid(token){
    const highlight = token?.HDHighlight;
    if(highlight && !highlight._destroyed) highlight?.clear();
  }

  static getDistEuclidean(p1, p2){
    const unitsToPixel = canvas.dimensions.size / canvas.dimensions.distance;
    const d =
    Math.sqrt(
      Math.pow(p2.x - p1.x, 2) + Math.pow(p2.y - p1.y, 2) + Math.pow(p2.z - p1.z, 2)
    ) / unitsToPixel;
  return d;
  }

  static measureMinTokenDistance(token1, token2){
    const unitsToPixel = canvas.dimensions.size / canvas.dimensions.distance;
    const square = (canvas.scene.dimensions.size)
    const halfSquare = square/2;
    const generatePoints = (token) => {
        const tokenHeight = ((token.losHeight ?? (token.document.elevation+0.00001))-token.document.elevation)/canvas.scene.dimensions.distance;
        const tokenPositions = [];
        const tokenStart = {
          x: token.center.x,
          y: token.document.elevation*unitsToPixel,
          z: token.center.y
        }
        tokenStart.x += -token.document.width*halfSquare+halfSquare
        tokenStart.y += halfSquare
        tokenStart.z += -token.document.height*halfSquare+halfSquare

        for(let i = 0; i < token.document.width; i++){
            for(let j = 0; j < token.document.height; j++){
                for(let k = 0; k < tokenHeight; k++){
                    const position = {
                        x: tokenStart.x + i*square,
                        y: tokenStart.y + k*square,
                        z: tokenStart.z + j*square
                    };
                    tokenPositions.push(position);
                }
            }
        }
        return tokenPositions;
    }
    const measurements = [];
    const originPoints = generatePoints(token1);
    const targetPoints = generatePoints(token2);
    const verticalSetting = game.settings.get("hover-distance", "verticalDistance");
    for(const oPoint of originPoints){
        for(const tPoint of targetPoints){
          const rulerDistance = canvas.grid.measurePath([{ x: oPoint.x, y: oPoint.z },
            { x: tPoint.x, y: tPoint.z }], {gridSpaces: true}).distance;
          const euclideanDistance = DistanceTooltip.getDistEuclidean(oPoint, tPoint);
          const verticalDistance = Math.abs(oPoint.y - tPoint.y) / unitsToPixel;
          let distance = null;
          if (verticalDistance < 0.01) {
            distance = rulerDistance;
          } else {            
            if (verticalSetting === "euclidean") distance = euclideanDistance;
            if (verticalSetting === "useHighest") distance = Math.max(rulerDistance, verticalDistance);
            if (verticalSetting === "ignore") distance = rulerDistance;
          }

          measurements.push(distance);
        }
    }
    const rounding = game.settings.get("hover-distance", "roundMeasurement");
    if (!rounding) return Math.floor(Math.min(...measurements));
    let min = Math.min(...measurements);
    const roundingValue = rounding;
    const remainder = min % roundingValue;
    if(remainder > roundingValue/2) min += roundingValue - remainder;
    else min -= remainder;
    min = Math.round(min * 1000000) / 1000000;
    return min;
    
  }

  static getRangeBand(distance) { 
    const rangeBands = game.settings.get("hover-distance", "rangeBands");
    if(!rangeBands) return;
    const bands = rangeBands.split(",");
    const processedBands = []
    for (let i = 0; i < bands.length; i+=2) {
      const bandDist = parseFloat(bands[i]);
      const bandName = bands[i + 1];
      processedBands.push({dist: bandDist, name: bandName});
    }
    for (let i = 0; i < processedBands.length; i++) {
      const band = processedBands[i];
      const nextBand = processedBands[i + 1];
      if (!nextBand) return band.name;
      if(distance >= band.dist && distance < nextBand.dist ) return band.name;
    }

  }

}
