class ImageContext {
    constructor(src, e) {
        this.src = src;
        this.e = e;
    }

    static onContextMenu(e) {
        const html = $(e.target.outerHTML);
        const src = html.attr("src");
        new ImageContext(src, e).showContextMenu();
    }

    static show() {
        const src = $(this).attr("data-src");
        new ImagePopout(src, { shareable: true }).render(true);
        this.parentElement.remove();
    }

    static toChat() {
        const src = $(this).attr("data-src");
        this.parentElement.remove();
        if (game.settings.get("image-context", "advancedToChat")) {
            ImageContext.toChatWithDialog(src);
            return;
        }

        ChatMessage.create({
            content: `<img class="image-context-chat-message" data-src="${src}" src="${src}">`,
        });
    }

    static theaterOfTheMind() {
        const src = $(this).attr("data-src");
        game.settings.set("simple-quest", "ttmSrc", { src });
        this.parentElement.remove();
    }

    static toChatWithDialog(src) {
        new Dialog({
            title: "Send Image to Chat",
            content: ImageContext.getDialogContent(),
            buttons: {
                all: {
                    icon: '<i class="fas fa-users"></i>',
                    label: "Show to All",
                    callback: () => {
                        ChatMessage.create({
                            content: `<img class="image-context-chat-message" data-src="${src}" src="${src}">`,
                        });
                    },
                },
                wisp: {
                    icon: '<i class="fas fa-user-check"></i>',
                    label: "Wisper",
                    callback: (html) => {
                        let players = [];
                        html.find(".image-context-dialog-checkbox:checked").each((i, e) => players.push($(e).attr("data-player")));
                        ChatMessage.create({
                            content: `<img class="image-context-chat-message" data-src="${src}" src="${src}">`,
                            whisper: players,
                        });
                    },
                },
                canc: {
                    icon: '<i class="fas fa-times"></i>',
                    label: "Cancel",
                    callback: () => {},
                },
            },
            default: "canc",
            render: (html) => {
                html.on("click", "label", (e) => $(e.target).prev().click());
            },
        }).render(true);
    }

    static getDialogContent() {
        const players = game.users.players;
        const container = $(`<div class="image-context-dialog-container"></div>`);
        container.append(`<div class="image-context-dialog-header"><strong>Send Image to:</strong></div>`);
        players.forEach((player) => {
            const checkbox = $(`<input type="checkbox" class="image-context-dialog-checkbox" data-player="${player.id}">`);
            const label = $(`<label class="image-context-dialog-label">${player.name}</label>`);
            const div = $(`<div class="image-context-dialog-player"></div>`);
            div.append(checkbox);
            div.append(label);
            container.append(div);
        });
        return container.html();
    }

    static copyURL() {
        const src = $(this).attr("data-src");
        game.clipboard.copyPlainText(src);
        ui.notifications.info(`URL: '${src}'' copied to clipboard`);
        this.parentElement.remove();
    }

    showContextMenu() {
        $(".image-context").remove();
        const contextmenu = $(`<div class="image-context"></div>`);
        const buttons = ImageContext.getButtons();
        buttons.forEach((button) => {
            const buttonElement = $(`<div class="image-context-button" data-src="${this.src}">${button.icon} ${button.name}</div>`);
            buttonElement.click(button.callback);
            contextmenu.append(buttonElement);
        });
        contextmenu.css({
            top: this.e.clientY,
            left: this.e.clientX,
        });
        $("body").append(contextmenu);
    }

    static getButtons() {
        //ttmSrc
        const buttons = [
            {
                name: "Show",
                icon: '<i class="fas fa-eye"></i>',
                callback: ImageContext.show,
            },
            {
                name: "Send to Chat",
                icon: '<i class="fas fa-share"></i>',
                callback: ImageContext.toChat,
            },
            {
                name: "Copy URL",
                icon: '<i class="fas fa-link"></i>',
                callback: ImageContext.copyURL,
            },
        ];

        if (game.modules.get("simple-quest")?.active) {
            const sqButton = {
                name: "Theater of the Mind",
                icon: '<i class="fas fa-theater-masks"></i>',
                callback: ImageContext.theaterOfTheMind,
            };
            buttons.splice(buttons.length - 1, 0, sqButton);
        }

        return buttons;
    }

    static async showPasteMenu() {
        const b64 = await ImageContext.imageFromClipboard();
        if (!b64) return;
        const d = new Dialog({
            title: "From Clipboard",
            content: `<img src="" style="width: 400px; max-width: 400px;"`,
            buttons: {
                uploadShare: {
                    icon: '<i class="fas fa-share"></i>',
                    label: "Upload & Share",
                    callback: async () => {
                        ImageContext.uploadB64andGetPath(b64).then((path) => {
                            ImageContext.toChatWithDialog(path);
                        });
                    },
                },
                journalShare: {
                    icon: '<i class="fas fa-book-open"></i>',
                    label: "To Journal and Share",
                    callback: async () => {
                        const path = await ImageContext.uploadB64andGetPath(b64);
                        let journal = game.journal.getName("Image Context");
                        if (!journal) {
                            journal = await JournalEntry.create({
                                name: "Image Context"
                            });
                        }
                        journal.createEmbeddedDocuments("JournalEntryPage", [{name: `image-${Date.now()}`, src: path, type: "image"}]);
                        ImageContext.toChatWithDialog(path);
                    },
                },
            },
            render: (html) => {
                const img = html.find("img");
                img.attr("src", b64);
                img[0].onload = () => {
                    d.setPosition({height: "auto", width: "auto"});
                }
                d.setPosition({height: "auto", width: "auto"});
            },
            default: "canc",
        })
        d.render(true);
    }

    static async uploadB64andGetPath(b64) {
        const targetFolder = game.settings.get("image-context", "uploadLocation");
        //infer data directory
        const [source, target] = new FilePicker()._inferCurrentDirectory(targetFolder);
        //see if the directory exists
        try {
            await FilePicker.browse(source, target);
        } catch (error) {
            await FilePicker.createDirectory(source, target);
        }

        const res = await fetch(b64);
        const blob = await res.blob();
        const filename = `image-${Date.now()}.png`;
        const file = new File([blob], filename, {type: "image/png"});
        const f = await FilePicker.upload(source, target, file);
        return f.path;
    }
    
    static async imageFromClipboard() {
        return new Promise(async (resolve, reject) => {
            try {
                const clipboardContents = await navigator.clipboard.read();
                for (const item of clipboardContents) {
                    if (!item.types.includes("image/png")) {
                        throw new Error("Clipboard does not contain PNG image data.");
                    }
                    const blob = await item.getType("image/png");
                    //return base64 string
                    const reader = new FileReader();
                    reader.readAsDataURL(blob);
                    reader.onloadend = function () {
                        const base64data = reader.result;
                        resolve(base64data);
                    };
                }
            } catch (error) {
                resolve(null);
            }
        });
    }

    static async clipboardHasImage() {
        const clipboardContents = await navigator.clipboard.read();
        for (const item of clipboardContents) {
            if (item.types.includes("image/png")) {
                return true;
            }
        }
        return false;
    }
}
