import { MODULE_ID } from "./module.js";
import { convertURL } from "./optimizer.js";

export class FileConverter extends FormApplication {
    constructor(files, path) {
        super();
        this.files = files.filter((file) => {
            const lc = file.toLowerCase();
            if (lc.endsWith(".webp") || lc.endsWith(".ogg") || lc.endsWith(".webm")) return false;
            return true;
        });
        this.path = decodeURIComponent(path);
    }

    static get defaultOptions() {
        return {
            ...super.defaultOptions,
            title: game.i18n.localize(`${MODULE_ID}.file-converter.title`),
            id: "media-optimizer-file-converter",
            template: `modules/media-optimizer/templates/file-converter.hbs`,
            popOut: true,
            width: 600,
        };
    }

    getData() {
        const fileData = this.files.map((file) => {
            return {
                fullPath: file,
                fileName: decodeURIComponent(file.split("/").pop()),
            };
        });
        return {
            files: fileData,
        };
    }

    activateListeners(html) {
        super.activateListeners(html);
        html = html[0];
        html.querySelector(`#select-all`).addEventListener("click", (e) => {
            e.preventDefault();
            const checkboxes = html.querySelectorAll(`input[type="checkbox"]`);
            checkboxes.forEach((checkbox) => {
                checkbox.checked = true;
            });
            this.updateButton();
        });
        html.querySelector(`#select-none`).addEventListener("click", (e) => {
            e.preventDefault();
            const checkboxes = html.querySelectorAll(`input[type="checkbox"]`);
            checkboxes.forEach((checkbox) => {
                checkbox.checked = false;
            });
            this.updateButton();
        });
        html.querySelectorAll(`input[type="checkbox"]`).forEach((checkbox) => {
            checkbox.addEventListener("change", (e) => {
                this.updateButton();
            });
        });
        this.updateButton();
    }

    updateButton() {
        const checkboxes = this.element[0].querySelectorAll(`input[type="checkbox"]`);
        const checkedCount = Array.from(checkboxes).filter((checkbox) => checkbox.checked).length;
        const button = this.element[0].querySelector(`button[type="submit"]`);
        button.innerHTML = `<i class="far fa-compress"></i> ${game.i18n.localize("media-optimizer.file-converter.buttons.convert")} (${checkedCount})`;
    }

    async _onSubmit(event) {
        event.preventDefault();
        const formData = this._getSubmitData(event);
        const filesToConvert = [];
        for (const [file, convert] of Object.entries(formData)) {
            if (!file.includes(".")) continue;
            if (convert) filesToConvert.push(file);
        }
        this.convert(filesToConvert);
        this.close();
    }

    async convert(filesToConvert) {
        for (const file of filesToConvert) {
            await convertURL(file, this.path);
        }
    }
}

