import { mergeTiles } from "./optimizer.js";
import { MODULE_ID } from "./module.js";

export class TileCombiner extends FormApplication {
    constructor(scene) {
        super();
        this.scene = scene ?? canvas.scene;
        this.fileName = "combined-" + foundry.utils.randomID(20);
    }

    static get defaultOptions() {
        return {
            ...super.defaultOptions,
            title: game.i18n.localize(`${MODULE_ID}.tile-combiner.title`),
            id: "media-optimizer-tile-combiner",
            template: `modules/media-optimizer/templates/tile-combiner.hbs`,
            popOut: true,
            width: 500,
        };
    }

    getData() {
        return {
            ...game.settings.get(MODULE_ID, "combinerSettings"),
            fileName: this.fileName,
        };
    }

    activateListeners(html) {
        super.activateListeners(html);
        html[0].querySelector(`button[data-action="cancel"]`).addEventListener("click", (e) => {
            e.preventDefault();
            this.close();
        });
    }

    async _onSubmit(event) {
        event.preventDefault();
        const doBackup = event.submitter.dataset.action === "backup-combine";
        const combineData = this._getSubmitData(event);
        combineData.fileName = combineData.fileName || this.fileName;
        if (doBackup) await this.scene.clone({ name: `${this.scene._source.name} (Copy)` }, { save: true });
        this.combine(combineData);
        this.close();
    }

    async uploadFile(file, path) {
        const fp = new FilePicker();
        const [source, folder] = fp._inferCurrentDirectory(path);
        await FilePicker.upload(source, folder, file);
    }

    async confirmCombine(backup = false) {
        let backupText = "";
        if (!backup) {
            backupText = `<p class="notification error">${game.i18n.localize(`${MODULE_ID}.tile-combiner.confirm.backup`)}</p>`;
        }
        return new Promise((resolve, reject) => {
            Dialog.confirm({
                title: game.i18n.localize(`${MODULE_ID}.tile-combiner.confirm.title`),
                content: backupText + `<div>${game.i18n.localize(`${MODULE_ID}.tile-combiner.confirm.content`)}</div>`,
                yes: () => resolve(true),
                no: () => resolve(false),
                defaultYes: false,
            });
        });
    }

    async combine(combineData) {
        if (!(await this.confirmCombine(combineData.backup))) return;
        let tiles;
        const FGElevation = canvas.scene.foregroundElevation;
        switch (combineData.mode) {
            case "all":
                tiles = canvas.tiles.placeables;
                break;
            case "selected":
                tiles = canvas.tiles.controlled;
                break;
            case "background":
                tiles = canvas.tiles.placeables.filter((t) => t.document.elevation !== FGElevation);
                break;
            case "foreground":
                tiles = canvas.tiles.placeables.filter((t) => t.document.elevation === FGElevation);
                break;
        }
        if (!tiles.length) return;
        const combined = await mergeTiles(tiles, this.scene, combineData);
        await this.uploadFile(combined.file, combineData.folderPath);
        const newTileData = {
            texture: {
                src: combineData.folderPath + "/" + combineData.fileName + ".webp",
            },
            width: combined.width,
            height: combined.height,
            x: combined.x,
            y: combined.y,
        };
        if (combineData.background) {
            await this.scene.update({ "background.src": combineData.folderPath + "/" + combineData.fileName + ".webp" });
        } else {
            await this.scene.createEmbeddedDocuments("Tile", [newTileData]);
        }
        if (combineData.deleteTiles) {
            await canvas.scene.deleteEmbeddedDocuments(
                "Tile",
                tiles.map((t) => t.id),
            );
        }
    }

    async close(...args) {
        const combineData = this._getSubmitData();
        delete combineData.fileName;
        game.settings.set(MODULE_ID, "combinerSettings", combineData);
        super.close(...args);
    }
}
