/**
 * Converts web colors to base 16
 * @param n {Hex}               Web format color, f.x. #FF0000
 * @return {Hex}                Base 16 format color, f.x. 0xFF0000
 */
function webToHex (n) {
	return n.replace("#", "0x");
}

/**
 * Converts a base16 color into a web color
 * @param n {Hex}               Base 16 Color, f.x. 0xFF0000
 * @return {Hex}                Web format color, f.x. #FF0000
 */
function hexToWeb (n) {
	return (`${n}`).replace("0x", "#");
}

/**
 * Converts an object containing coordinate pair arrays into a single array of points for PIXI
 * @param hex {Object}  An object containing a set of [x,y] pairs
 */
function hexObjsToArr (hex) {
	const a = [];
	hex.forEach((point) => {
		a.push(point.x);
		a.push(point.y);
	});
	// Append first point to end of array to complete the shape
	a.push(hex[0].x);
	a.push(hex[0].y);
	return a;
}

/**
 * Prints formatted console msg if string, otherwise dumps object
 * @param data {String | Object} Output to be dumped
 * @param force {Boolean}        Log output even if CONFIG.debug.polmap = false
 */
function polmapLog (data, force = false) {
	if (CONFIG.debug.polmap || force) {
		// eslint-disable-next-line no-console
		if (typeof data === "string") console.log(`Political Map Overlay | ${data}`);
		// eslint-disable-next-line no-console
		else console.log("Political Map Overlay |", data);
	}
}

const MODULE_ID = "polmap";

/* -------------------------------------------- */

const RGB_RED = {name: "Red", color: "#ff4500"};
const RGB_ORANGE = {name: "Orange", color: "#ffa800"};
const RGB_YELLOW = {name: "Yellow", color: "#ffd635"};
const RGB_DARK_GREEN = {name: "Dark green", color: "#00a368"};
const RGB_LIGHT_GREEN = {name: "Light green", color: "#7eed56"};
const RGB_DARK_BLUE = {name: "Dark blue", color: "#2450a4"};
const RGB_BLUE = {name: "Blue", color: "#3690ea"};
const RGB_LIGHT_BLUE = {name: "Light blue", color: "#51e9f4"};
const RGB_DARK_PURPLE = {name: "Dark purple", color: "#811e9f"};
const RGB_PURPLE = {name: "Purple", color: "#b44ac0"};
const RGB_LIGHT_PINK = {name: "Light pink", color: "#ff99aa"};
const RGB_BROWN = {name: "Brown", color: "#9c6926"};
const RGB_WHITE = {name: "White", color: "#ffffff"};
const RGB_LIGHT_GRAY = {name: "Light gray", color: "#d4d7d9"};
const RGB_GRAY = {name: "Gray", color: "#898d90"};
const RGB_BLACK = {name: "Black", color: "#000000"};

const RGB_BASIC = [
	RGB_RED,
	RGB_ORANGE,
	RGB_YELLOW,
	RGB_DARK_GREEN,
	RGB_LIGHT_GREEN,
	RGB_DARK_BLUE,
	RGB_BLUE,
	RGB_LIGHT_BLUE,
	RGB_DARK_PURPLE,
	RGB_PURPLE,
	RGB_LIGHT_PINK,
	RGB_BROWN,
	RGB_WHITE,
	RGB_LIGHT_GRAY,
	RGB_GRAY,
	RGB_BLACK,
];

/**
 * Queue and execute history updates in a serial fashion, from all connected GMs and players.
 *
 * Only the main, connected, GM client (the one Socketlib deems the "responsible" GM) makes modifications to the history
 * data. This avoids any race conditions (assuming client activity detection is reliable, which it *mostly* is).
 */
class HistorySocketInterface {
	static _SOCKET = null;

	static registerSocketBindings (socket) {
		this._SOCKET = socket;

		socket.register("commitHistory", this._pCommitHistory.bind(this));
		socket.register("resetHistory", this._pResetHistory.bind(this));
	}

	/* -------------------------------------------- */

	/**
	 * From Socketlib:
	 * https://github.com/manuelVo/foundryvtt-socketlib/blob/develop/src/socketlib.js#L313C1-L315C2
	 */
	static _isActiveGM (user) {
		return user.active && user.isGM;
	}

	/**
	 * From Socketlib:
	 * https://github.com/manuelVo/foundryvtt-socketlib/blob/develop/src/socketlib.js#L306
	 */
	static _isResponsibleGM () {
		if (!game.user.isGM) return false;
		const connectedGMs = game.users.filter(this._isActiveGM.bind(this));
		return !connectedGMs.some(other => other.id < game.user.id);
	}

	static _isAnyActiveGm () {
		return game.users.some(this._isActiveGM.bind(this));
	}

	/* -------------------------------------------- */

	static _SEMAPHORE_HISTORY = new Semaphore(1);

	static _validateActiveGM () {
		if (this._isAnyActiveGm()) return true;
		ui.notifications.error(`Could not perform operation \u2014 no active GM!`);
		return false;
	}

	static _validateUserPermissions (userId) {
		if (game.user.isGM) return true;
		if (game.settings.get(MODULE_ID, "isPlayerEditable") && game.users.get(userId)?.can("DRAWING_CREATE")) return true;
		ui.notifications.error(`Could not perform operation \u2014 did not have permissions!`);
		return false;
	}

	/* ----- */

	static async pCommitHistory (userId, historyBuffer) {
		if (!this._validateUserPermissions(userId) || !this._validateActiveGM()) return false;
		await this._SOCKET.executeForAllGMs("commitHistory", historyBuffer);
		return true;
	}

	static async _pCommitHistory (historyBuffer) {
		if (historyBuffer?.length === 0) return;
		if (!this._isResponsibleGM()) return;
		await this._SEMAPHORE_HISTORY.add(this._pCommitHistoryTask.bind(this), historyBuffer);
	}

	static async _pCommitHistoryTask (historyBuffer) {
		if (!historyBuffer?.length) return;

		const history = canvas.scene.getFlag(MODULE_ID, "history")
			// If history storage doesn't exist, create it
			|| {
				events: [],
				pointer: 0,
			};

		// If pointer is less than history length (f.x. user undo), truncate history
		history.events = history.events.slice(0, history.pointer);

		// Push the new history buffer to the scene
		history.events.push(historyBuffer);
		history.pointer = history.events.length;

		await canvas.scene.setFlag(MODULE_ID, "history", history);
		polmapLog(`Pushed ${historyBuffer.length} updates.`);
	}

	/* ----- */

	static async pResetHistory (userId) {
		if (!this._validateUserPermissions(userId) || !this._validateActiveGM()) return false;
		await this._SOCKET.executeForAllGMs("resetHistory");
		return true;
	}

	static async _pResetHistory () {
		if (!this._isResponsibleGM()) return;
		await this._SEMAPHORE_HISTORY.add(this._pResetHistoryTask.bind(this));
	}

	static async _pResetHistoryTask () {
		await canvas.scene.unsetFlag(MODULE_ID, "history");
		await canvas.scene.setFlag(MODULE_ID, "history", {
			events: [],
			pointer: 0,
		});
		polmapLog(`Reset history.`);
	}
}

class _SocketManager {
	static onSocketlibReady () {
		const socket = socketlib.registerModule(MODULE_ID);

		HistorySocketInterface.registerSocketBindings(socket);
	}
}

Hooks.once("socketlib.ready", () => {
	_SocketManager.onSocketlibReady();
});

class PoliticalMapMigrations {
	static async check () {
		if (!game.user.isGM) return;
		polmapLog("Checking migrations");

		let ver = game.settings.get(MODULE_ID, "migrationVersion");

		if (ver == null || !isNaN(ver)) return; // Disable example migration--remove as required

		ver = await this.migration1(ver);
		// ... etc. ...
	}

	static async migration1 (ver) {
		if (ver > 1) return;
		polmapLog("Performing migration #1", true);
		// (Implement as required)
		await game.settings.set(MODULE_ID, "migrationVersion", 1);
		return 1;
	}
}

class OverlayLayer extends InteractionLayer {
	static _TINT_ERASER = 0xFF00FF;

	constructor () {
		super();
		this._historyBuffer = [];
		this._pointer = 0;
		this._gridLayout = {};
		this._BRUSH_TYPES = {
			BOX: 1,
			POLYGON: 3,
		};
		this._DEFAULTS = {
			visible: false,
		};
		this._tempSettings = {};
		this._layerTexture = null;
		this._layer = null;

		// Allow zIndex prop to function for items on this layer
		this.sortableChildren = true;
	}

	static get layerOptions () {
		return mergeObject(super.layerOptions, {
			baseClass: InteractionLayer,
			zIndex: 19, // Below drawings (which is 20)
		});
	}

	// So you can hit escape on the keyboard and it will bring up the menu
	get controlled () { return {}; }

	/* -------------------------------------------- */
	/*  Init                                        */
	/* -------------------------------------------- */

	/**
   * Called on canvas init, creates the canvas layers and various objects and registers listeners
   *
   * Important objects:
   *
   * layer        - PIXI Sprite which holds all the region elements
   * layerTexture - renderable texture that holds the actual region data
   */
	async pInitOverlay () {
		// Check if layer is flagged visible
		let v = this.getSetting("visible");
		if (v === undefined) v = false;
		this.visible = v;

		await this.setAlpha(this.getAlpha());

		this._layerTexture = this.constructor._getLayerTexture();
		this._layer = new PIXI.Sprite(this._layerTexture);
		this.setClear();

		// Render initial history stack
		await this.pRenderStack();
	}

	/* -------------------------------------------- */
	/*  History & Buffer                            */
	/* -------------------------------------------- */

	static _getLayerTexture () {
		const d = canvas.dimensions;
		let res = 1.0;
		if (d.width * d.height > 16000 ** 2) res = 0.25;
		else if (d.width * d.height > 8000 ** 2) res = 0.5;

		return PIXI.RenderTexture.create({
			width: canvas.dimensions.width,
			height: canvas.dimensions.height,
			resolution: res,
		});
	}

	/**
   * Gets and sets various layer wide properties
   * Some properties have different values depending on if user is a GM or player
   */
	getSetting (name, {scene = null} = {}) {
		scene = scene || canvas.scene;

		let setting = scene.getFlag(MODULE_ID, name);
		if (setting === undefined) setting = this.getUserSetting(name);
		if (setting === undefined) setting = this._DEFAULTS[name];
		return setting;
	}

	async setSetting (name, value, {scene = null} = {}) {
		scene = scene || canvas.scene;

		return scene.setFlag(MODULE_ID, name, value);
	}

	getUserSetting (name) {
		let setting = game.user.getFlag(MODULE_ID, name);
		if (setting === undefined) setting = this._DEFAULTS[name];
		return setting;
	}

	async setUserSetting (name, value) {
		return game.user.setFlag(MODULE_ID, name, value);
	}

	getTempSetting (name) {
		return this._tempSettings[name];
	}

	setTempSetting (name, value) {
		this._tempSettings[name] = value;
	}

	/**
   * Renders the history stack to the regions
   * @param history {Array}       A collection of history events
   * @param start {Number}        The position in the history stack to begin rendering from
   * @param stop {Number}        The position in the history stack to stop rendering
   */
	async pRenderStack (
		history = canvas.scene.getFlag(MODULE_ID, "history"),
		start = this._pointer,
		stop = undefined,
	) {
		// If history is blank, do nothing
		if (history === undefined) return;

		const events = history.events.filter(arr => arr?.length);

		// If history is zero, reset scene overlay
		if (events.length === 0) return this.pResetLayer(false);

		if (start === undefined) start = 0;
		stop = stop === undefined ? events.length : stop;

		// If pointer precedes the stop, reset and start from 0
		if (stop <= this._pointer) {
			await this.pResetLayer(false);
			start = 0;
		}

		polmapLog(`Rendering from: ${start} to ${stop}`);
		// Render all ops starting from pointer
		for (let i = start; i < stop; i += 1) {
			for (let j = 0; j < events[i].length; j += 1) {
				this._renderBrushGraphic(events[i][j], false);
			}
		}
		// Update local pointer
		this._pointer = stop;
	}

	/**
   * Add buffered history stack to scene flag and clear buffer
   */
	async commitHistory () {
		// Do nothing if no history to be committed, otherwise get history
		if (this._historyBuffer.length === 0) return;

		const isApproved = await HistorySocketInterface.pCommitHistory(game.userId, this._historyBuffer);
		polmapLog(`Pushed ${this._historyBuffer.length} updates (via GM socket).`);
		this._historyBuffer = [];

		if (!isApproved) await this.pRenderStack();
	}

	/**
   * Resets the layer
   * @param save {Boolean} If true, also resets the layer history
   */
	async pResetLayer (save = true) {
		// Clear the layer
		this.setClear();

		if (!save) return;

		// If save, also unset history and reset pointer
		const isApproved = await HistorySocketInterface.pResetHistory(game.userId);
		this._pointer = 0;

		if (!isApproved) await this.pRenderStack();
	}

	/* -------------------------------------------- */
	/*  Shapes, sprites and PIXI objs               */
	/* -------------------------------------------- */

	/**
   * Creates a PIXI graphic using the given brush parameters
   * @param data {Object}       A collection of brush parameters
   * @returns {Object}          PIXI.Graphics() instance
   *
   * @example
   * const myBrush = this.brush({
   *      shape: <brush type>,
   *      x: 0,
   *      y: 0,
   *      fill: 0x000000,
   *      width: 50,
   *      height: 50,
   *      alpha: 1,
   *      visible: true
   * });
   */
	_getBrushGraphic (data) {
		// Get new graphic & begin filling
		const alpha = typeof data.alpha === "undefined" ? 1 : data.alpha;
		const visible = typeof data.visible === "undefined" ? true : data.visible;
		const brush = new PIXI.Graphics();
		brush.beginFill(data.fill);
		// Draw the shape depending on type of brush
		switch (data.shape) {
			case this._BRUSH_TYPES.BOX:
				brush.drawRect(0, 0, data.width, data.height);
				break;
			case this._BRUSH_TYPES.POLYGON:
				brush.drawPolygon(data.vertices);
				break;
		}
		// End fill and set the basic props
		brush.endFill();
		brush.alpha = alpha;
		brush.visible = visible;
		brush.x = data.x;
		brush.y = data.y;
		brush.zIndex = data.zIndex;
		if (data.blend) brush.blendMode = PIXI.BLEND_MODES[data.blend];
		return brush;
	}

	/**
   * Gets a brush using the given parameters, renders it to layer and saves the event to history
   * @param data {Object}       A collection of brush parameters
   * @param save {Boolean}      If true, will add the operation to the history buffer
   */
	_renderBrushGraphic (data, save = true) {
		const brush = this._getBrushGraphic(data);
		this._doRenderBrushToLayer(brush);
		brush.destroy();
		if (save) this._historyBuffer.push(data);
	}

	/**
	 * Renders the given brush to the layer
	 * @param brush {Object}       PIXI Object to be used as brush
	 * @param isClear              If layer should be cleared.
	 */
	_doRenderBrushToLayer (brush, {isClear = false} = {}) {
		canvas.app.renderer.render(brush, {
			renderTexture: this._layerTexture,
			clear: isClear,
			transform: null,
			skipUpdateTransform: false,
		});
	}

	/**
   * Clears the layer
   */
	setClear () {
		const fill = new PIXI.Graphics();
		this._doRenderBrushToLayer(fill, {isClear: true});
		fill.destroy();
	}

	/**
   * Toggles visibility of primary layer
   */
	toggle () {
		const v = this.getSetting("visible");
		this.visible = !v;
		this.setSetting("visible", !v);
	}

	/**
   * Actions upon layer becoming active
   */
	activate () {
		super.activate();
		this.eventMode = "static";
	}

	/**
   * Actions upon layer becoming inactive
   */
	deactivate () {
		super.deactivate();
		this.eventMode = "passive";
	}

	async draw () {
		const out = await super.draw();
		await this.pInitOverlay();
		this.addChild(this._layer);
		return out;
	}

	static refreshZIndex () {
		canvas.polmap.zIndex = game.settings.get(MODULE_ID, "zIndex");
	}
}

/*
 * Global PolMap Configuration Options
 */


var config = [
	{
		name: "migrationVersion",
		data: {
			name: "Political Map Overlay Migration Version",
			scope: "world",
			config: false,
			type: Number,
			default: 0,
		},
	},
	{
		name: "zIndex",
		data: {
			name: "Political Map Overlay Z-Index",
			hint: "The z-index determines the order in which various layers are rendered within the Foundry canvas.  A higher number will be rendered on top of lower numbered layers (and the objects on that layer).  This allows for the adjustment of the z-index to allow for the Political Map Overlay to be rendered above/below other layers; particularly ones added by other modules. Going below 200 will intermingle with Foundry layers such as the foreground image (200), tokens (100), etc...  (Default: 215)",
			scope: "world",
			config: true,
			default: 215,
			type: Number,
			onChange: OverlayLayer.refreshZIndex,
		},
	},
	{
		name: "isPlayerEditable",
		data: {
			name: "Player Editable",
			hint: "If enabled, non-GM users will be able to draw on the political map layer.",
			scope: "world",
			config: true,
			default: false,
			type: Boolean,
			onChange: () => {
				if (ui.controls) ui.controls.initialize();
			},
		},
	},
];

/* eslint-disable */
// Generated code -- CC0 -- No Rights Reserved -- http://www.redblobgames.com/grids/hexagons/
class Point {
  constructor(x, y) {
    this.x = x;
    this.y = y;
  }
}
class Hex {
  constructor(q, r, s) {
    this.q = q;
    this.r = r;
    this.s = s;
    if (Math.round(q + r + s) !== 0)
      throw "q + r + s must be 0";
  }
  add(b) {
    return new Hex(this.q + b.q, this.r + b.r, this.s + b.s);
  }
  subtract(b) {
    return new Hex(this.q - b.q, this.r - b.r, this.s - b.s);
  }
  scale(k) {
    return new Hex(this.q * k, this.r * k, this.s * k);
  }
  rotateLeft() {
    return new Hex(-this.s, -this.q, -this.r);
  }
  rotateRight() {
    return new Hex(-this.r, -this.s, -this.q);
  }
  static direction(direction) {
    return Hex.directions[direction];
  }
  neighbor(direction) {
    return this.add(Hex.direction(direction));
  }
  diagonalNeighbor(direction) {
    return this.add(Hex.diagonals[direction]);
  }
  len() {
    return (Math.abs(this.q) + Math.abs(this.r) + Math.abs(this.s)) / 2;
  }
  distance(b) {
    return this.subtract(b).len();
  }
  round() {
    var qi = Math.round(this.q);
    var ri = Math.round(this.r);
    var si = Math.round(this.s);
    var q_diff = Math.abs(qi - this.q);
    var r_diff = Math.abs(ri - this.r);
    var s_diff = Math.abs(si - this.s);
    if (q_diff > r_diff && q_diff > s_diff) {
      qi = -ri - si;
    }
    else if (r_diff > s_diff) {
      ri = -qi - si;
    }
    else {
      si = -qi - ri;
    }
    return new Hex(qi, ri, si);
  }
  lerp(b, t) {
    return new Hex(this.q * (1.0 - t) + b.q * t, this.r * (1.0 - t) + b.r * t, this.s * (1.0 - t) + b.s * t);
  }
  linedraw(b) {
    var N = this.distance(b);
    var a_nudge = new Hex(this.q + 1e-06, this.r + 1e-06, this.s - 2e-06);
    var b_nudge = new Hex(b.q + 1e-06, b.r + 1e-06, b.s - 2e-06);
    var results = [];
    var step = 1.0 / Math.max(N, 1);
    for (var i = 0; i <= N; i++) {
      results.push(a_nudge.lerp(b_nudge, step * i).round());
    }
    return results;
  }
}
Hex.directions = [new Hex(1, 0, -1), new Hex(1, -1, 0), new Hex(0, -1, 1), new Hex(-1, 0, 1), new Hex(-1, 1, 0), new Hex(0, 1, -1)];
Hex.diagonals = [new Hex(2, -1, -1), new Hex(1, -2, 1), new Hex(-1, -1, 2), new Hex(-2, 1, 1), new Hex(-1, 2, -1), new Hex(1, 1, -2)];
class Orientation {
  constructor(f0, f1, f2, f3, b0, b1, b2, b3, start_angle) {
    this.f0 = f0;
    this.f1 = f1;
    this.f2 = f2;
    this.f3 = f3;
    this.b0 = b0;
    this.b1 = b1;
    this.b2 = b2;
    this.b3 = b3;
    this.start_angle = start_angle;
  }
}
class Layout {
  constructor(orientation, size, origin) {
    this.orientation = orientation;
    this.size = size;
    this.origin = origin;
  }
  hexToPixel(h) {
    var M = this.orientation;
    var size = this.size;
    var origin = this.origin;
    var x = (M.f0 * h.q + M.f1 * h.r) * size.x;
    var y = (M.f2 * h.q + M.f3 * h.r) * size.y;
    return new Point(x + origin.x, y + origin.y);
  }
  pixelToHex(p) {
    var M = this.orientation;
    var size = this.size;
    var origin = this.origin;
    var pt = new Point((p.x - origin.x) / size.x, (p.y - origin.y) / size.y);
    var q = M.b0 * pt.x + M.b1 * pt.y;
    var r = M.b2 * pt.x + M.b3 * pt.y;
    return new Hex(q, r, -q - r);
  }
  hexCornerOffset(corner) {
    var M = this.orientation;
    var size = this.size;
    var angle = 2.0 * Math.PI * (M.start_angle - corner) / 6.0;
    return new Point(size.x * Math.cos(angle), size.y * Math.sin(angle));
  }
  polygonCorners(h) {
    var corners = [];
    var center = this.hexToPixel(h);
    for (var i = 0; i < 6; i++) {
      var offset = this.hexCornerOffset(i);
      corners.push(new Point(center.x + offset.x, center.y + offset.y));
    }
    return corners;
  }
}
Layout.pointy = new Orientation(Math.sqrt(3.0), Math.sqrt(3.0) / 2.0, 0.0, 3.0 / 2.0, Math.sqrt(3.0) / 3.0, -1.0 / 3.0, 0.0, 2.0 / 3.0, 0.5);
Layout.flat = new Orientation(3.0 / 2.0, 0.0, Math.sqrt(3.0) / 2.0, Math.sqrt(3.0), 2.0 / 3.0, 0.0, -1.0 / 3.0, Math.sqrt(3.0) / 3.0, 0.0);

class PoliticalMapLayer extends OverlayLayer {
	static _LINE_OPTS_ERASE = {
		width: 7,
		color: 0xFF0000,
		cap: PIXI.LINE_CAP.ROUND,
		alpha: 1,
	};

	constructor () {
		super();

		// Register event listeners
		this._registerMouseListeners();

		this._DEFAULTS = Object.assign(this._DEFAULTS, {
			gmAlpha: 0.5,
			playerAlpha: 0.5,
			previewColor: "0x00FFFF",
			paletteColor: "0xFFFFFF",
		});

		// React to changes to current scene
		Hooks.on("updateScene", (scene, data) => this._updateScene(scene, data));

		this._boxPreview = null;
		this._polygonPreview = null;
		this._erasePreview = null;

		// Right-click tracking
		this._rcDist = null;
		this._rcPos = null;

		this._dupes = [];
	}

	init () {
		// Preview brush objects
		this._boxPreview = this._getBrushGraphic({
			shape: this._BRUSH_TYPES.BOX,
			x: 0,
			y: 0,
			fill: 0xFFFFFF,
			alpha: 1,
			width: 100,
			height: 100,
			visible: false,
			zIndex: 10,
		});
		this._polygonPreview = this._getBrushGraphic({
			shape: this._BRUSH_TYPES.POLYGON,
			x: 0,
			y: 0,
			vertices: [],
			fill: 0xFFFFFF,
			alpha: 1,
			visible: false,
			zIndex: 10,
		});
		this._erasePreview = this._getBrushGraphic({
			shape: this._BRUSH_TYPES.POLYGON,
			x: 0,
			y: 0,
			vertices: [],
			fill: 0xFFFFFF,
			alpha: 1,
			visible: false,
			zIndex: 10,
		});
	}

	canvasInit () {
		// Set default flags if they dont exist already
		Object.keys(this._DEFAULTS).forEach((key) => {
			if (!game.user.isGM) return;
			// Check for existing scene specific setting
			if (this.getSetting(key) !== undefined) return;
			// Check for custom default
			const def = this.getUserSetting(key);
			// If user has custom default, set it for scene
			if (def !== undefined) this.setSetting(key, def);
			// Otherwise fall back to module default
			else this.setSetting(key, this._DEFAULTS[key]);
		});
	}

	/* -------------------------------------------- */
	/*  Misc helpers                                */
	/* -------------------------------------------- */

	static _isSquareGrid (gridType) {
		return gridType === 1;
	}

	static _isHexGrid (gridType) {
		return [2, 3, 4, 5].includes(gridType);
	}

	/* -------------------------------------------- */
	/*  Getters and setters for layer props         */
	/* -------------------------------------------- */

	// Alpha has special cases because it can differ between GM & Players

	getAlpha () {
		let alpha;
		if (game.user.isGM) alpha = this.getSetting("gmAlpha");
		else alpha = this.getSetting("playerAlpha");
		if (!alpha) {
			if (game.user.isGM) alpha = this._DEFAULTS.gmAlpha;
			else alpha = this._DEFAULTS.playerAlpha;
		}
		return alpha;
	}

	/**
	* Sets the scene's alpha for the primary layer.
	* @param alpha {Number} 0-1 opacity representation
	*/
	async setAlpha (alpha) {
		this.alpha = alpha;
	}

	/* -------------------------------------------- */
	/*  Event Listeners and Handlers                */
	/* -------------------------------------------- */

	/**
	* React to updates of canvas.scene flags
	*/
	async _updateScene (scene, data) {
		// Check if update applies to current viewed scene
		if (!scene._view) return;
		// React to visibility change
		if (hasProperty(data, `flags.${MODULE_ID}.visible`)) {
			canvas[MODULE_ID].visible = data.flags[MODULE_ID].visible;
		}
		// React to brush history change
		if (hasProperty(data, `flags.${MODULE_ID}.history`)) {
			await canvas[MODULE_ID].pRenderStack(data.flags[MODULE_ID].history);
		}
		// React to alpha/tint changes
		if (!game.user.isGM && hasProperty(data, `flags.${MODULE_ID}.playerAlpha`)) {
			await canvas[MODULE_ID].setAlpha(data.flags[MODULE_ID].playerAlpha);
		}
		if (game.user.isGM && hasProperty(data, `flags.${MODULE_ID}.gmAlpha`)) {
			await canvas[MODULE_ID].setAlpha(data.flags[MODULE_ID].gmAlpha);
		}
	}

	/**
	* Adds the mouse listeners to the layer
	*/
	_registerMouseListeners () {
		this.addListener("pointerdown", this._pointerDown);
		this.addListener("pointerup", this._pointerUp);
		this.addListener("pointermove", this._pointerMove);
	}

	/**
	 * Sets the active tool & shows preview for grid tool.
	 */
	setActiveTool (tool) {
		this.clearActiveTool();
		this.activeTool = tool;
		this.setPreviewTint();

		// Note: this `setActiveTool` method gets called on changing scene or scene dimensions, as the scene controls
		//   are reset, and the user has to click on the polmap tools again. This may break in a future Foundry update; if
		//   it does, wire this up to e.g. a "scene update" event.
		this._initGrid();

		switch (tool) {
			case "grid": {
				if (this.constructor._isSquareGrid(canvas.scene.grid.type)) {
					this._boxPreview.width = canvas.scene.grid.size;
					this._boxPreview.height = canvas.scene.grid.size;
					this._boxPreview.visible = true;
					break;
				}

				if (this.constructor._isHexGrid(canvas.scene.grid.type)) {
					this._polygonPreview.visible = true;
				}
			}
		}
	}

	setPreviewTint () {
		const previews = [
			this._boxPreview,
			this._polygonPreview,
		];

		if (this.getTempSetting("isErasing") || this.getTempSetting("isErasingRightClick")) {
			previews.forEach(preview => preview.tint = this.constructor._TINT_ERASER);
			return;
		}

		const tint = this.getSetting("paletteColor");
		previews.forEach(preview => preview.tint = tint);
	}

	/**
	* Aborts any active drawing tools
	*/
	clearActiveTool () {
		// Box preview
		if (this._boxPreview) this._boxPreview.visible = false;
		// Shape preview
		if (this._polygonPreview) {
			this._polygonPreview.clear();
			this._polygonPreview.visible = false;
		}
		// Erase preview
		if (this._erasePreview) {
			this._erasePreview.clear();
			this._erasePreview.visible = false;
		}
		// Cancel op flag
		this.op = false;
		// Clear history buffer
		this._historyBuffer = [];
	}

	/**
	 * Get mouse position translated to canvas coords
	 */
	_getRoundedLocalPosition (evt) {
		const p = evt.data.getLocalPosition(canvas.app.stage);
		// Round positions to nearest pixel
		p.x = Math.round(p.x);
		p.y = Math.round(p.y);
		return p;
	}

	/**
	* Mouse handlers for canvas layer interactions
	*/
	_pointerDown (evt) {
		// Don't allow new action if history push still in progress
		if (this._historyBuffer.length > 0) {
			console.warn(`Discarded input; still got ${this._historyBuffer.length} to sync :(`);
			return;
		}

		switch (evt.data.button) {
			// LMB
			case 0: {
				const p = this._getRoundedLocalPosition(evt);

				this.op = true;
				// Check active tool
				switch (this.activeTool) {
					case "grid": this._pointerDownGrid(p, evt); break;
				}
				// Call _pointermove so single click will still draw brush if mouse does not move
				this._pointerMove(evt);
				break;
			}

			// RMB
			case 2: {
				this._rcDist = 0;
				this._rcPos = this._getRoundedLocalPosition(evt);
			}
		}
	}

	_pointerMove (evt) {
		const p = this._getRoundedLocalPosition(evt);

		switch (this.activeTool) {
			case "grid": this._pointerMoveGrid(p, evt); break;
		}

		if (this._rcPos != null) {
			const dx = Math.abs(p.x - this._rcPos.x);
			const dy = Math.abs(p.y - this._rcPos.y);
			this._rcDist += dx + dy;
		}
	}

	_pointerUp (evt) {
		switch (evt.data.button) {
			// LMB
			case 0: {
				// Translate click to canvas position
				const p = evt.data.getLocalPosition(canvas.app.stage);
				// Round positions to nearest pixel
				p.x = Math.round(p.x);
				p.y = Math.round(p.y);
				// Reset operation
				this.op = false;
				// Push the history buffer
				this.commitHistory();
				break;
			}

			// RMB
			case 2: {
				// If the mouse has moved too great a distance between starting and finishing the right-click, ignore it
				const isClick = this._rcDist < 5;
				this._rcDist = null;
				this._rcPos = null;
				if (!isClick) return;

				// Allow right-click to erase
				this.op = "grid";
				this.setTempSetting("isErasingRightClick", true);
				this._pointerMove(evt);
				this.setTempSetting("isErasingRightClick", false);
				// Reset operation
				this.op = false;
				// Push the history buffer
				this.commitHistory();
			}
		}
	}

	/**
	* Grid Tool
	*/
	_pointerDownGrid () {
		this.op = "grid";
		this._dupes = [];
	}

	_pointerMoveGrid (p) {
		const {size: grid, type: gridType} = canvas.scene.grid;

		const isErasingVisual = this.getTempSetting("isErasing");
		const isErasing = isErasingVisual || this.getTempSetting("isErasingRightClick");

		const fill = isErasingVisual
			? 1
			: this.getSetting("paletteColor");

		if (this.constructor._isSquareGrid(gridType)) {
			const gridx = Math.floor(p.x / grid);
			const gridy = Math.floor(p.y / grid);
			const x = gridx * grid;
			const y = gridy * grid;

			this._boxPreview.visible = !isErasingVisual;
			this._erasePreview.visible = isErasingVisual;

			if (isErasingVisual) {
				this._erasePreview.clear();
				this._erasePreview
					.lineStyle(this.constructor._LINE_OPTS_ERASE)
					.moveTo(x + grid, y + grid)
					.lineTo(x, y);
			} else {
				this._boxPreview.x = x;
				this._boxPreview.y = y;
				this._boxPreview.width = grid;
				this._boxPreview.height = grid;
			}

			// If drag operation has not started, bail out
			if (!this.op) return;

			// Avoid duplicating data within a single drag
			const coord = `${x},${y}`;
			if (this._dupes.includes(coord)) return;
			this._dupes.push(coord);

			// Save info to history
			const brush = {
				shape: this._BRUSH_TYPES.BOX,
				x,
				y,
				width: grid,
				height: grid,
				fill,
			};
			if (isErasing) brush.blend = "ERASE";
			this._renderBrushGraphic(brush);

			return;
		}

		if (this.constructor._isHexGrid(gridType)) {
			// Convert pixel coord to hex coord
			const qr = this._gridLayout.pixelToHex(p).round();
			// Get current grid coord verts
			const vertices = this._gridLayout.polygonCorners({ q: qr.q, r: qr.r });
			// Convert to array of individual verts
			const vertexArray = hexObjsToArr(vertices);

			// Update the preview shape
			this._polygonPreview.visible = !isErasingVisual;
			this._erasePreview.visible = isErasingVisual;

			if (isErasingVisual) {
				this._erasePreview.clear();
				this._erasePreview
					.lineStyle(this.constructor._LINE_OPTS_ERASE)
					.moveTo(vertexArray[2], vertexArray[3])
					.lineTo(vertexArray[8], vertexArray[9]);
			} else {
				this._polygonPreview.clear();
				this._polygonPreview.beginFill(0xFFFFFF);
				this._polygonPreview.drawPolygon(vertexArray);
				this._polygonPreview.endFill();
			}

			// If drag operation has not started, bail out
			if (!this.op) return;

			// Avoid duplicating data within a single drag
			const coord = `${qr.q},${qr.r}`;
			if (this._dupes.includes(coord)) return;
			this._dupes.push(coord);

			// Save info to history
			const brush = {
				shape: this._BRUSH_TYPES.POLYGON,
				vertices: vertexArray,
				x: 0,
				y: 0,
				fill,
			};
			if (isErasing) brush.blend = "ERASE";
			this._renderBrushGraphic(brush);
		}
	}

	/*
	* Checks grid type, creating a hex grid layout if required
	*/
	_initGrid () {
		const grid = canvas.scene.grid.size;

		if (canvas.scene.flags.core?.legacyHex) {
			switch (canvas.scene.grid.type) {
				// Pointy Hex Odd
				case 2:
					this._gridLayout = new Layout(
						Layout.pointy,
						{ x: grid / 2, y: grid / 2 },
						{ x: 0, y: grid / 2 },
					);
					break;
				// Pointy Hex Even
				case 3:
					this._gridLayout = new Layout(
						Layout.pointy,
						{ x: grid / 2, y: grid / 2 },
						{ x: Math.sqrt(3) * grid / 4, y: grid / 2 },
					);
					break;
				// Flat Hex Odd
				case 4:
					this._gridLayout = new Layout(
						Layout.flat,
						{ x: grid / 2, y: grid / 2 },
						{ x: grid / 2, y: 0 },
					);
					break;
				// Flat Hex Even
				case 5:
					this._gridLayout = new Layout(
						Layout.flat,
						{ x: grid / 2, y: grid / 2 },
						{ x: grid / 2, y: Math.sqrt(3) * grid / 4 },
					);
					break;
			}
		} else {
			switch (canvas.scene.grid.type) {
				// Pointy Hex Odd
				case 2:
					this._gridLayout = new Layout(
						Layout.pointy,
						{ x: grid / Math.sqrt(3), y: grid / Math.sqrt(3)},
						{ x: 0, y: grid / Math.sqrt(3)},
					);
					break;
				// Pointy Hex Even
				case 3:
					this._gridLayout = new Layout(
						Layout.pointy,
						{ x: grid / Math.sqrt(3), y: grid / Math.sqrt(3)},
						{ x: grid / 2, y: grid / Math.sqrt(3) },
					);
					break;
				// Flat Hex Odd
				case 4:
					this._gridLayout = new Layout(
						Layout.flat,
						{ x: grid / Math.sqrt(3), y: grid / Math.sqrt(3)},
						{ x: grid / Math.sqrt(3), y: 0 },
					);
					break;
				// Flat Hex Even
				case 5:
					this._gridLayout = new Layout(
						Layout.flat,
						{ x: grid / Math.sqrt(3), y: grid / Math.sqrt(3)},
						{ x: grid / Math.sqrt(3), y: grid / 2},
					);
					break;
			}
		}
	}

	async draw () {
		const out = await super.draw();
		this.init();
		this.addChild(this._boxPreview);
		this.addChild(this._polygonPreview);
		this.addChild(this._erasePreview);
		return out;
	}
}

Hooks.once("init", () => {
	polmapLog("Initializing polmap", true);

	// Register global module settings
	config.forEach((cfg) => {
		game.settings.register(MODULE_ID, cfg.name, cfg.data);
	});

	CONFIG.Canvas.layers.polmap = {group: "interface", layerClass: PoliticalMapLayer};

	Object.defineProperty(canvas, MODULE_ID, {
		value: new PoliticalMapLayer(),
		configurable: true,
		writable: true,
		enumerable: false,
	});
});

Hooks.once("canvasInit", (cvs) => {
	canvas.polmap.canvasInit(cvs);
});

Hooks.once("ready", () => {
	PoliticalMapMigrations.check().then(() => polmapLog("Migrations complete!"));

	OverlayLayer.refreshZIndex();
});

class PoliticalMapConfig extends FormApplication {
	constructor ({scene}) {
		super();
		this._scene = scene;
	}

	static get defaultOptions () {
		return mergeObject(super.defaultOptions, {
			classes: ["form"],
			closeOnSubmit: false,
			submitOnChange: true,
			submitOnClose: true,
			popOut: true,
			editable: game.user.isGM,
			width: 500,
			template: "modules/polmap/templates/scene-config.hbs",
			id: "polmap-scene-config",
			title: game.i18n.localize("POLMAP.Political Map Options"),
		});
	}

	/* -------------------------------------------- */

	/**
   * Obtain module metadata and merge it with game settings which track current module visibility
   * @return {Object}   The data provided to the template when rendering the form
   */
	getData () {
		// Return data to the template
		return {
			gmAlpha: Math.round(canvas.polmap.getSetting("gmAlpha", {scene: this._scene}) * 100),
			playerAlpha: Math.round(canvas.polmap.getSetting("playerAlpha", {scene: this._scene}) * 100),
		};
	}

	/* -------------------------------------------- */
	/*  Event Listeners and Handlers                */
	/* -------------------------------------------- */

	/**
   * This method is called upon form submission after form data is validated
   * @param event {Event}       The initial triggering submission event
   * @param formData {Object}   The object of validated form data with which to update the object
   * @private
   */
	async _updateObject (event, formData) {
		await Promise.allSettled(
			Object.entries(formData)
				.map(async ([key, val]) => {
					// If setting is an opacity slider, convert from 1-100 to 0-1
					if (["gmAlpha", "playerAlpha"].includes(key)) val /= 100;
					// Save settings to scene
					await canvas.polmap.setSetting(key, val, {scene: this._scene});
					// If saveDefaults button clicked, also save to user's defaults
					if (event.submitter?.name === "saveDefaults") {
						canvas.polmap.setUserSetting(key, val);
					}
				}),
		);

		// If save button was clicked, close app
		if (event.submitter?.name === "submit") {
			Object.values(ui.windows).forEach((val) => {
				if (val.id === "polmap-scene-config") val.close();
			});
		}
	}
}

class PaletteControls extends FormApplication {
	constructor (...args) {
		super(...args);

		this._isErasing = false;
	}

	static get defaultOptions () {
		return mergeObject(super.defaultOptions, {
			classes: ["form"],
			closeOnSubmit: false,
			submitOnChange: true,
			submitOnClose: true,
			popOut: false,
			editable: true,
			template: "modules/polmap/templates/palette-controls.hbs",
			id: "filter-config",
			title: game.i18n.localize("POLMAP.Political Map Options"),
		});
	}

	/* -------------------------------------------- */

	/**
   * Obtain module metadata and merge it with game settings which track current module visibility
   * @return {Object}   The data provided to the template when rendering the form
   */
	getData () {
		// Return data to the template
		return {
			colors: RGB_BASIC,
			paletteColor: hexToWeb(canvas.polmap.getUserSetting("paletteColor")),
		};
	}

	/* -------------------------------------------- */
	/*  Event Listeners and Handlers                */
	/* -------------------------------------------- */

	/** @override */
	activateListeners ($html) {
		super.activateListeners($html);

		const toggleErasing = (val) => {
			this._isErasing = val ?? !this._isErasing;
			$btnErase.toggleClass(`polmap__btn-erase--active`, this._isErasing);
		};

		const $iptColor = $(`[name="paletteColor"]`)
			.change(async evt => {
				toggleErasing(false);
				await this._onSubmit(evt, {preventClose: true, preventRender: true});
			});

		$(`[name="btn-color"]`)
			.click(async evt => {
				toggleErasing(false);
				const rgb = evt.currentTarget.dataset.hex;
				$iptColor.val(rgb);
				await this._onSubmit(evt, {preventClose: true, preventRender: true});
			});

		const $btnErase = $(`[name="btn-erase"]`)
			.click(async evt => {
				toggleErasing();
				await this._onSubmit(evt, {preventClose: true, preventRender: true});
			});
	}

	/**
   * This method is called upon form submission after form data is validated
   * @param event {Event}       The initial triggering submission event
   * @param formData {Object}   The object of validated form data with which to update the object
   * @private
   */
	async _updateObject (event, formData) {
		await canvas.polmap.setUserSetting("paletteColor", webToHex(formData.paletteColor));
		canvas.polmap.setTempSetting("isErasing", this._isErasing);
		canvas.polmap.setPreviewTint();
	}
}

let $wrpPaletteControls;

const _doCacheControls = () => {
	if (!$wrpPaletteControls?.length) $wrpPaletteControls = $("#polmap-palette");
};

/**
 * Add control buttons
 */
Hooks.on("getSceneControlButtons", (controls) => {
	const toolMarker = {
		name: "grid",
		title: game.i18n.localize("POLMAP.Marker Tool"),
		icon: "fas fa-border-none",
	};

	controls.push({
		name: "polmap",
		title: game.i18n.localize("POLMAP.Political Map Overlay"),
		icon: "fas fa-handshake",
		layer: MODULE_ID,
		activeTool: "grid",
		visible: game.user.can("DRAWING_CREATE")
			&& (game.user.isGM || game.settings.get(MODULE_ID, "isPlayerEditable")),
		tools: game.user.isGM
			? [
				{
					name: "polmaptoggle",
					title: game.i18n.localize("POLMAP.Enable/Disable Political Map Overlay"),
					icon: "fas fa-eye",
					onClick: () => {
						canvas.polmap.toggle();
					},
					active: canvas.polmap?.visible,
					toggle: true,
				},
				toolMarker,
				{
					name: "sceneConfig",
					title: game.i18n.localize("POLMAP.Scene Configuration"),
					icon: "fas fa-cog",
					onClick: () => {
						new PoliticalMapConfig({scene: canvas.scene}).render(true);
					},
					button: true,
				},
				{
					name: "clearfog",
					title: game.i18n.localize("POLMAP.Reset Political Map Overlay"),
					icon: "fas fa-trash",
					onClick: () => {
						const dg = new Dialog({
							title: game.i18n.localize("POLMAP.Reset Political Map Overlay"),
							content: game.i18n.localize("POLMAP.Are you sure? Political map areas will be reset."),
							buttons: {
								blank: {
									icon: `<i class="fas fa-eye"></i>`,
									label: "Blank",
									callback: () => canvas.polmap.pResetLayer(),
								},
								cancel: {
									icon: `<i class="fas fa-times"></i>`,
									label: "Cancel",
								},
							},
							default: "reset",
						});
						dg.render(true);
					},
					button: true,
				},
			]
			: [
				toolMarker,
			]
		,
	});
});

/**
 * Handles adding the custom brush controls palette
 * and switching active brush flag
 */
Hooks.on("renderSceneControls", (controls) => {
	// Switching to layer
	if (canvas.polmap == null) return;

	if (controls.activeControl === "polmap" && controls.activeTool != null) {
		// Open brush tools if not already open
		_doCacheControls();
		if (!$wrpPaletteControls.length) new PaletteControls().render(true);
		// Set active tool
		const tool = controls.controls.find((control) => control.name === "polmap").activeTool;
		canvas.polmap.setActiveTool(tool);
		return;
	}

	// region Switching away from layer
	// Clear active tool
	canvas.polmap.clearActiveTool();
	// Remove brush tools if open
	_doCacheControls();
	$wrpPaletteControls.remove();
	$wrpPaletteControls = null;
	// endregion
});

/**
 * Sets Y position of the brush controls to account for scene navigation buttons
 */
const setPaletteControlPos = () => {
	_doCacheControls();
	if (!$wrpPaletteControls.length) return;

	const h = $("#navigation").height();
	$wrpPaletteControls.css({top: `${h + 40}px`});
};

// Reset position when brush controls are rendered or sceneNavigation changes
Hooks.on("renderPaletteControls", setPaletteControlPos);
Hooks.on("renderSceneNavigation", setPaletteControlPos);
