class ProgressTracker extends Application {
  constructor(id) {
    super();
    this.data = game.settings.get("progress-tracker", "sliders")[id];
    this.sliders = this.data.progress || {};
  }

  static get defaultOptions() {
    return {
      ...super.defaultOptions,
      title: `Progress Tracker`,
      id: "progress-tracker",
      template: `modules/progress-tracker/templates/progress-tracker.hbs`,
      resizable: true,
      width: 1000,
      height: 100,
      dragDrop: [{ dragSelector: null, dropSelector: null }],
    };
  }

  getData() {
    return {};
  }

  setPosition({left, top, width, height, scale}={}) {
    super.setPosition({left, top, width, height, scale});
    //restore the slider position
    this.loadProgress();
    for(let slider of Object.values(this.sliders)){
      const $slider = $(`.slider[data-name="${slider}"]`);
      const percent = this.sliders[slider];
      const sliderLeft = this.element.width() * percent / 100;
      $slider.css("left", sliderLeft);
    }
  }

  activateListeners(html) {
    $("body").off("mouseup", this.saveOnUp)
    const _this = this;
    const pb = this.generateProgressBar();
    html.append(pb);
    const pos = game.settings.get("progress-tracker", "windowpos");
    this.setPosition(pos);
    $("#progress-tracker").find(".window-title").text(`${this.data.name}`);
    if(!game.user.isGM) return;
    html.on("mousemove", (e) => {
      if(!this.sliding) return
      //move the slider
      const slider = _this.activeSlider;
      const name = slider.data("name");
      let x = e.pageX - html.offset().left -slider.width()/2;
      const min = 0;
      const max = html.width() - slider.width() - 5;
      x = Math.max(min, Math.min(max, x));
      slider.css("left", x);
      //get percentage
      const percent = Math.round(x / html.width() * 100);
      _this.sliders[name] = percent;
    });
    html.on("mousedown", ".slider", (e)=>{
      _this.sliding = true
      _this.activeSlider = $(e.currentTarget);
    })
    $("body").on("mouseup", this.saveOnUp)
  }

  saveOnUp = (e)=>{
    this.sliding = false
    this.saveData();
  }

  loadProgress() {
    const progress = this.data.progress;
    if(!progress) return;
    $(this.element).find(".slider").each((i, slider)=>{
      const name = $(slider).data("name");
      const percent = progress[name];
      const sliderLeft = this.element.width() * percent / 100;
      $(slider).css("left", sliderLeft);
    });

  }

  generateProgressBar() {
    let waypoints = "";
    for (let waypoint of this.data.waypoints) {
      let style = "";
      if(waypoint.percent == 100) style = `right: 0;`
      else if(waypoint.percent == 0) style = `left: 0;`
      else style = `left: ${waypoint.percent}%; transform: translateX(-50%)`
      waypoints += `<img class="waypoint" src="${waypoint.img}" style="${style}"></img>`;
    }
    let sliders = "";
    for(let slider of this.data.sliders) {
      sliders += `<img class="slider" src="${slider.img}" data-name="${slider.name}"></img>`
    }

    const $progressBar =
      $(`<div class="progress-bar">
      <img class="progress-bar-bg" src="${this.data.bg}"></img>
      ${sliders}
      ${waypoints}      
      </div>`);
      console.log($progressBar);
    return $progressBar;
  }

  saveData(){
    if(!game.user.isGM) return;
    const settings = game.settings.get("progress-tracker", "sliders")
    settings[this.data.id].progress = this.sliders;
    game.settings.set("progress-tracker", "sliders", settings);
  }

  close(){
    $("body").off("mouseup", this.saveOnUp)
    //get position
    const window = $("#progress-tracker")
    const left = window.offset().left;
    const top = window.offset().top;
    const width = window.width();
    const height = window.height();
    const pos = {left, top, width, height};
    game.settings.set("progress-tracker", "windowpos", pos);
    super.close();
  }
}
