import { MODULE_ID, JOURNAL_NAME, MAX_TEXT_LENGTH } from "./main.js";
import { getSetting } from "./settings.js";

export function initConfig() {
    CONFIG.TextEditor.enrichers.push({
        id: MODULE_ID,
        pattern: /@INFO\[(.*?)\]/g,
        enricher: (match, content) => {
            try {
                const theme = getSetting("theme");
                const addTitle = getSetting("addTitle");
                const infoJournalTitle = match[1];
                const infoJournal = game.journal.getName(JOURNAL_NAME);
                const infoJournalPage = infoJournal.pages.getName(infoJournalTitle);
                const a = document.createElement("a");
                a.classList.add(`${MODULE_ID}-info-link`);
                a.dataset.tooltip = (addTitle ? `<h2>${infoJournalTitle}</h2>` : "") + infoJournalPage.text.content;
                a.dataset.tooltipDirection = TooltipManager.TOOLTIP_DIRECTIONS.UP;
                a.dataset.tooltipClass = `rich-info-tooltip ${theme}`;
                a.innerText = infoJournalTitle;
                return a;
            } catch (e) {
                const a = document.createElement("a");
                a.innerHTML = `<i class="fas fa-times"></i> Info Not Found: ${match[1]}`;
                return a;
            }
        },
    });
}

export function initAutoTooltips() {
    if (!getSetting("autoTooltips")) return;
    let currentElement = null;
    const wait = (ms) => new Promise((resolve) => setTimeout(resolve, ms));
    $(document).on("mouseleave", `.content-link`, (event) => {
        currentElement = null;
    });
    $(document).on("mouseenter", `.content-link`, async (event) => {
        if((event.currentTarget.dataset.tooltip ?? "").includes("<")) return;
        if (!game.tooltip.element) {
            currentElement = event.currentTarget;
            await wait(TooltipManager.TOOLTIP_ACTIVATION_MS);
            const isHovering = currentElement === event.currentTarget || currentElement?.closest(".content-link") === event.currentTarget;
            if (!isHovering) return;
        }
        const uuid = event.currentTarget.dataset.uuid;
        const document = await fromUuid(uuid);
        showTooltipForDocument(event.currentTarget, document);
    });

    $(document).on("mouseenter", ".macro", async (event) => {
        const itemName = event.currentTarget.dataset.tooltip;
        const actors = [canvas.tokens.controlled.map((t) => t.actor), game.user.character].flat().filter((a) => a);
        if (actors.length) {
            let item = null;
            for (let actor of actors) {
                item = actor.items.getName(itemName);
                if (item) break;
            }
            if (item) return showTooltipForDocument(event.currentTarget, item);
        }
        const macro = game.macros.get(event.currentTarget.dataset.macroId);
        if(!macro) return;
        const macroCommand = macro.command;
        const uuid = macroCommand.substring(macroCommand.indexOf('"') + 1, macroCommand.lastIndexOf('"'));
        const document = await fromUuid(uuid);
        if (document) showTooltipForDocument(event.currentTarget, document);
    });
}

async function showTooltipForDocument(element, document) {
    if (!document || document.permission < CONST.DOCUMENT_OWNERSHIP_LEVELS.OBSERVER) return;
    const text = document.system?.description?.value ?? document.text?.content;
    if (!text) return;

    const textLength = text.length;

    const addTitle = getSetting("addTitle");

    let finalText = addTitle ? `<h2>${document.name}</h2>${text}` : text;

    if (getSetting("useImageBackground")) finalText = `<div class="rich-tooltip-image" style="background-image: url(${document.img})"></div>${finalText}`;

    if (textLength > MAX_TEXT_LENGTH) {
        finalText = finalText.substring(0, MAX_TEXT_LENGTH) + "...";
    }

    const enriched = await TextEditor.enrichHTML(finalText);

    const theme = getSetting("theme");

    const cssClass = `rich-info-tooltip ${theme}`;

    game.tooltip.activate(element, { text: enriched, cssClass, direction: TooltipManager.TOOLTIP_DIRECTIONS.UP });
}
