/**
 * Custom control icon used to display Map Location journal pages when pinned to the map.
 */
class MapLocationControlIcon extends PIXI.Container {
  constructor({code, size=40, ...style}={}, ...args) {
    super(...args);

    this.code = code;
    this.size = size;
    this.style = style;

    this.renderMarker();
    this.refresh();
  }

  /* -------------------------------------------- */

  /**
   * Perform the actual rendering of the marker.
   */
  renderMarker() {
    this.radius = this.size / 2;
    this.circle = [this.radius, this.radius, this.radius + 8];
    this.backgroundColor = this.style.backgroundColor;
    this.borderColor = this.style.borderHoverColor;

    // Define hit area
    this.eventMode = "static";
    this.interactiveChildren = false;
    this.hitArea = new PIXI.Circle(...this.circle);
    this.cursor = "pointer";

    // Drop Shadow
    this.shadow = this.addChild(new PIXI.Graphics());
    this.shadow.clear()
      .beginFill(this.style.shadowColor, 0.65)
      .drawCircle(this.radius + 8, this.radius + 8, this.radius + 10)
      .endFill();
    this.shadow.filters = [new PIXI.filters.BlurFilter(16)];

    // 3D Effect
    this.extrude = this.addChild(new PIXI.Graphics());
    this.extrude.clear()
      .beginFill(this.style.borderColor, 1.0)
      .drawCircle(this.radius + 2, this.radius + 2, this.radius + 9)
      .endFill();

    // Background
    this.bg = this.addChild(new PIXI.Graphics());
    this.bg.clear()
      .beginFill(this.backgroundColor, 1.0)
      .lineStyle(2, this.style.borderColor, 1.0)
      .drawCircle(...this.circle)
      .endFill();

    // Text
    this.text = new PreciseText(this.code, this._getTextStyle(this.code.length, this.size));
    this.text.anchor.set(0.5, 0.5);
    this.text.position.set(this.radius, this.radius);
    this.addChild(this.text);

    // Border
    this.border = this.addChild(new PIXI.Graphics());
    this.border.visible = false;
  }

  /* -------------------------------------------- */

  /**
   * Code text to be rendered.
   * @type {string}
   */
  code;

  /* -------------------------------------------- */

  /** @inheritDoc */
  refresh({ visible, iconColor, borderColor, borderVisible }={}) {
    if ( borderColor ) this.borderColor = borderColor;
    this.border.clear().lineStyle(2, this.borderColor, 1.0).drawCircle(...this.circle).endFill();
    if ( borderVisible !== undefined ) this.border.visible = borderVisible;
    if ( visible !== undefined ) this.visible = visible;
    return this;
  }

  /* -------------------------------------------- */

  /**
   * Define PIXI TestStyle object for rendering the map location code.
   * @param {number} characterCount  Number of characters in the code.
   * @param {number} size            Size of the icon in the Scene.
   * @returns {PIXI.TextStyle}
   * @protected
   */
  _getTextStyle(characterCount, size) {
    const style = CONFIG.canvasTextStyle.clone();
    style.dropShadow = false;
    style.fill = Color.from(this.style.textColor);
    style.strokeThickness = 0;
    style.fontFamily = ["Signika"];
    if ( this.style.fontFamily ) style.fontFamily.unshift(this.style.fontFamily);
    style.fontSize = characterCount > 2 ? size * .5 : size * .6;
    return style;
  }
}

const {
  Coin, DiceTerm: DiceTerm$3, Die: Die$1, FunctionTerm: FunctionTerm$1, NumericTerm: NumericTerm$2, OperatorTerm: OperatorTerm$3, ParentheticalTerm: ParentheticalTerm$1, RollTerm: RollTerm$1
} = foundry.dice.terms;

/**
 * A standardized helper function for simplifying the constant parts of a multipart roll formula.
 *
 * @param {string} formula                          The original roll formula.
 * @param {object} [options]                        Formatting options.
 * @param {boolean} [options.preserveFlavor=false]  Preserve flavor text in the simplified formula.
 * @param {boolean} [options.deterministic]         Strip any non-deterministic terms from the result.
 *
 * @returns {string}  The resulting simplified formula.
 */
function simplifyRollFormula(formula, { preserveFlavor=false, deterministic=false } = {}) {
  // Create a new roll and verify that the formula is valid before attempting simplification.
  let roll;
  try { roll = new Roll(formula); }
  catch(err) { console.warn(`Unable to simplify formula '${formula}': ${err}`); }
  Roll.validate(roll.formula);

  // Optionally strip flavor annotations.
  if ( !preserveFlavor ) roll.terms = Roll.parse(roll.formula.replace(RollTerm$1.FLAVOR_REGEXP, ""));

  if ( deterministic ) {
    // Perform arithmetic simplification to simplify parsing through the terms.
    roll.terms = _simplifyOperatorTerms(roll.terms);

    // Remove non-deterministic terms, their preceding operators, and dependent operators/terms.
    const terms = [];
    let temp = [];
    let multiplicative = false;
    let determ;

    for ( let i = roll.terms.length - 1; i >= 0; ) {
      let paren;
      let term = roll.terms[i];
      if ( term instanceof ParentheticalTerm$1 ) {
        paren = simplifyRollFormula(term.term, { preserveFlavor, deterministic });
      }
      if ( Number.isNumeric(paren) ) {
        const termData = { number: paren };
        if ( preserveFlavor ) termData.options = { flavor: term.flavor };
        term = new NumericTerm$2(termData);
      }
      determ = term.isDeterministic && (!multiplicative || determ);
      if ( determ ) temp.unshift(term);
      else temp = [];
      term = roll.terms[--i];
      while ( term instanceof OperatorTerm$3 ) {
        if ( determ ) temp.unshift(term);
        if ( (term.operator === "*") || (term.operator === "/") || (term.operator === "%") ) multiplicative = true;
        else {
          multiplicative = false;
          while ( temp.length ) terms.unshift(temp.pop());
        }
        term = roll.terms[--i];
      }
    }
    if ( determ ) {
      while ( temp.length ) terms.unshift(temp.pop());
    }
    roll.terms = terms;
  }

  // Perform arithmetic simplification on the existing roll terms.
  roll.terms = _simplifyOperatorTerms(roll.terms);

  // If the formula contains multiplication or division we cannot easily simplify
  if ( /[*/]/.test(roll.formula) ) {
    if ( roll.isDeterministic && !/d\(/.test(roll.formula) && (!/\[/.test(roll.formula) || !preserveFlavor) ) {
      return String(new Roll(roll.formula).evaluateSync().total);
    }
    else return roll.constructor.getFormula(roll.terms);
  }

  // Flatten the roll formula and eliminate string terms.
  roll.terms = _expandParentheticalTerms(roll.terms);
  roll.terms = Roll.simplifyTerms(roll.terms);

  // Group terms by type and perform simplifications on various types of roll term.
  let { poolTerms, diceTerms, mathTerms, numericTerms } = _groupTermsByType(roll.terms);
  numericTerms = _simplifyNumericTerms(numericTerms ?? []);
  diceTerms = _simplifyDiceTerms(diceTerms ?? []);

  // Recombine the terms into a single term array and remove an initial + operator if present.
  const simplifiedTerms = [diceTerms, poolTerms, mathTerms, numericTerms].flat().filter(Boolean);
  if ( simplifiedTerms[0]?.operator === "+" ) simplifiedTerms.shift();
  return roll.constructor.getFormula(simplifiedTerms);
}

/* -------------------------------------------- */

/**
 * A helper function to perform arithmetic simplification and remove redundant operator terms.
 * @param {RollTerm[]} terms  An array of roll terms.
 * @returns {RollTerm[]}      A new array of roll terms with redundant operators removed.
 */
function _simplifyOperatorTerms(terms) {
  return terms.reduce((acc, term) => {
    const prior = acc[acc.length - 1];
    const ops = new Set([prior?.operator, term.operator]);

    // If one of the terms is not an operator, add the current term as is.
    if ( ops.has(undefined) ) acc.push(term);

    // Replace consecutive "+ -" operators with a "-" operator.
    else if ( (ops.has("+")) && (ops.has("-")) ) acc.splice(-1, 1, new OperatorTerm$3({ operator: "-" }));

    // Replace double "-" operators with a "+" operator.
    else if ( (ops.has("-")) && (ops.size === 1) ) acc.splice(-1, 1, new OperatorTerm$3({ operator: "+" }));

    // Don't include "+" operators that directly follow "+", "*", or "/". Otherwise, add the term as is.
    else if ( !ops.has("+") ) acc.push(term);

    return acc;
  }, []);
}

/* -------------------------------------------- */

/**
 * A helper function for combining unannotated numeric terms in an array into a single numeric term.
 * @param {object[]} terms  An array of roll terms.
 * @returns {object[]}      A new array of terms with unannotated numeric terms combined into one.
 */
function _simplifyNumericTerms(terms) {
  const simplified = [];
  const { annotated, unannotated } = _separateAnnotatedTerms(terms);

  // Combine the unannotated numerical bonuses into a single new NumericTerm.
  if ( unannotated.length ) {
    const staticBonus = Roll.safeEval(Roll.getFormula(unannotated));
    if ( staticBonus === 0 ) return [...annotated];

    // If the staticBonus is greater than 0, add a "+" operator so the formula remains valid.
    if ( staticBonus > 0 ) simplified.push(new OperatorTerm$3({ operator: "+"}));
    simplified.push(new NumericTerm$2({ number: staticBonus} ));
  }
  return [...simplified, ...annotated];
}

/* -------------------------------------------- */

/**
 * A helper function to group dice of the same size and sign into single dice terms.
 * @param {object[]} terms  An array of DiceTerms and associated OperatorTerms.
 * @returns {object[]}      A new array of simplified dice terms.
 */
function _simplifyDiceTerms(terms) {
  const { annotated, unannotated } = _separateAnnotatedTerms(terms);

  // Split the unannotated terms into different die sizes and signs
  const diceQuantities = unannotated.reduce((obj, curr, i) => {
    if ( curr instanceof OperatorTerm$3 ) return obj;
    const isCoin = curr.constructor?.name === "Coin";
    const face = isCoin ? "c" : curr.faces;
    const modifiers = isCoin ? "" : curr.modifiers.filterJoin("");
    const key = `${unannotated[i - 1].operator}${face}${modifiers}`;
    obj[key] ??= {};
    if ( (curr._number instanceof Roll) && (curr._number.isDeterministic) ) curr._number.evaluateSync();
    obj[key].number = (obj[key].number ?? 0) + curr.number;
    if ( !isCoin ) obj[key].modifiers = (obj[key].modifiers ?? []).concat(curr.modifiers);
    return obj;
  }, {});

  // Add new die and operator terms to simplified for each die size and sign
  const simplified = Object.entries(diceQuantities).flatMap(([key, {number, modifiers}]) => ([
    new OperatorTerm$3({ operator: key.charAt(0) }),
    key.slice(1) === "c"
      ? new Coin({ number: number })
      : new Die$1({ number, faces: parseInt(key.slice(1)), modifiers: [...new Set(modifiers)] })
  ]));
  return [...simplified, ...annotated];
}

/* -------------------------------------------- */

/**
 * A helper function to extract the contents of parenthetical terms into their own terms.
 * @param {object[]} terms  An array of roll terms.
 * @returns {object[]}      A new array of terms with no parenthetical terms.
 */
function _expandParentheticalTerms(terms) {
  terms = terms.reduce((acc, term) => {
    if ( term instanceof ParentheticalTerm$1 ) {
      if ( term.isDeterministic ) {
        const roll = new Roll(term.term);
        term = new NumericTerm$2({ number: roll.evaluateSync().total });
      } else {
        const subterms = new Roll(term.term).terms;
        term = _expandParentheticalTerms(subterms);
      }
    }
    acc.push(term);
    return acc;
  }, []);
  return _simplifyOperatorTerms(terms.flat());
}

/* -------------------------------------------- */

/**
 * A helper function to group terms into PoolTerms, DiceTerms, FunctionTerms, and NumericTerms.
 * FunctionTerms are included as NumericTerms if they are deterministic.
 * @param {RollTerm[]} terms  An array of roll terms.
 * @returns {object}          An object mapping term types to arrays containing roll terms of that type.
 */
function _groupTermsByType(terms) {
  // Add an initial operator so that terms can be rearranged arbitrarily.
  if ( !(terms[0] instanceof OperatorTerm$3) ) terms.unshift(new OperatorTerm$3({ operator: "+" }));

  return terms.reduce((obj, term, i) => {
    let type;
    if ( term instanceof DiceTerm$3 ) type = DiceTerm$3;
    else if ( (term instanceof FunctionTerm$1) && (term.isDeterministic) ) type = NumericTerm$2;
    else type = term.constructor;
    const key = `${type.name.charAt(0).toLowerCase()}${type.name.substring(1)}s`;

    // Push the term and the preceding OperatorTerm.
    (obj[key] = obj[key] ?? []).push(terms[i - 1], term);
    return obj;
  }, {});
}

/* -------------------------------------------- */

/**
 * A helper function to separate annotated terms from unannotated terms.
 * @param {object[]} terms     An array of DiceTerms and associated OperatorTerms.
 * @returns {Array | Array[]}  A pair of term arrays, one containing annotated terms.
 */
function _separateAnnotatedTerms(terms) {
  return terms.reduce((obj, curr, i) => {
    if ( curr instanceof OperatorTerm$3 ) return obj;
    obj[curr.flavor ? "annotated" : "unannotated"].push(terms[i - 1], curr);
    return obj;
  }, { annotated: [], unannotated: [] });
}

/* -------------------------------------------- */
/*  Formatters                                  */
/* -------------------------------------------- */

/**
 * Format a Challenge Rating using the proper fractional symbols.
 * @param {number} value                   CR value to format.
 * @param {object} [options={}]
 * @param {boolean} [options.narrow=true]  Use narrow fractions (e.g. ⅛) rather than wide ones (e.g. 1/8).
 * @returns {string}
 */
function formatCR(value, { narrow=true }={}) {
  if ( value === null ) return "—";
  const fractions = narrow ? { 0.125: "⅛", 0.25: "¼", 0.5: "½" } : { 0.125: "1/8", 0.25: "1/4", 0.5: "1/2" };
  return fractions[value] ?? formatNumber(value);
}

/* -------------------------------------------- */

/**
 * Form a number using the provided length unit.
 * @param {number} value         The length to format.
 * @param {string} unit          Length unit as defined in `CONFIG.DND5E.movementUnits`.
 * @param {object} [options={}]  Formatting options passed to `formatNumber`.
 * @returns {string}
 */
function formatLength(value, unit, options={}) {
  return _formatSystemUnits(value, unit, CONFIG.DND5E.movementUnits[unit], options);
}

/* -------------------------------------------- */

/**
 * Format a modifier for display with its sign separate.
 * @param {number} mod  The modifier.
 * @returns {Handlebars.SafeString}
 */
function formatModifier(mod) {
  if ( !Number.isFinite(mod) ) return new Handlebars.SafeString("—");
  return new Handlebars.SafeString(`<span class="sign">${mod < 0 ? "-" : "+"}</span>${Math.abs(mod)}`);
}

/* -------------------------------------------- */

/**
 * A helper for using Intl.NumberFormat within handlebars.
 * @param {number} value    The value to format.
 * @param {object} options  Options forwarded to {@link https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Intl/NumberFormat}
 * @param {boolean} [options.numerals]  Format the number as roman numerals.
 * @param {boolean} [options.ordinal]   Use ordinal formatting.
 * @param {boolean} [options.words]     Write out number as full word, if possible.
 * @returns {string}
 */
function formatNumber(value, { numerals, ordinal, words, ...options }={}) {
  if ( words && game.i18n.has(`DND5E.NUMBER.${value}`, false) ) return game.i18n.localize(`DND5E.NUMBER.${value}`);
  if ( numerals ) return _formatNumberAsNumerals(value);
  if ( ordinal ) return _formatNumberAsOrdinal(value, options);
  const formatter = new Intl.NumberFormat(game.i18n.lang, options);
  return formatter.format(value);
}

/**
 * Roman numerals.
 * @type {Record<string, number>}
 */
const _roman = {
  M: 1000, CM: 900, D: 500, CD: 400, C: 100, XC: 90, L: 50, XL: 40, X: 10, IX: 9, V: 5, IV: 4, I: 1
};

/**
 * Format a number as roman numerals.
 * @param {number} n  The number to format.
 * @returns {string}
 */
function _formatNumberAsNumerals(n) {
  let out = "";
  if ( (n < 1) || !Number.isInteger(n) ) return out;
  for ( const [numeral, decimal] of Object.entries(_roman) ) {
    const quotient = Math.floor(n / decimal);
    n -= quotient * decimal;
    out += numeral.repeat(quotient);
  }
  return out;
}

/* -------------------------------------------- */

/**
 * Format a number using an ordinal format.
 * @param {number} n        The number to format.
 * @param {object} options  Options forwarded to `formatNumber`.
 * @returns {string}
 */
function _formatNumberAsOrdinal(n, options={}) {
  const pr = getPluralRules({ type: "ordinal" }).select(n);
  const number = formatNumber(n, options);
  return game.i18n.has(`DND5E.ORDINAL.${pr}`) ? game.i18n.format(`DND5E.ORDINAL.${pr}`, { number }) : number;
}

/* -------------------------------------------- */

/**
 * Produce a number with the parts wrapped in their own spans.
 * @param {number} value      A number for format.
 * @param {object} [options]  Formatting options.
 * @returns {string}
 */
function formatNumberParts(value, options) {
  if ( options.numerals ) throw new Error("Cannot segment numbers when formatted as numerals.");
  return new Intl.NumberFormat(game.i18n.lang, options).formatToParts(value)
    .reduce((str, { type, value }) => `${str}<span class="${type}">${value}</span>`, "");
}

/* -------------------------------------------- */

/**
 * A helper for using Intl.NumberFormat within handlebars for format a range.
 * @param {number} min      The lower end of the range.
 * @param {number} max      The upper end of the range.
 * @param {object} options  Options forwarded to {@link https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Intl/NumberFormat}
 * @returns {string}
 */
function formatRange(min, max, options) {
  const formatter = new Intl.NumberFormat(game.i18n.lang, options);
  return formatter.formatRange(min, max);
}

/* -------------------------------------------- */

/**
 * A helper function to format textarea text to HTML with linebreaks.
 * @param {string} value  The text to format.
 * @returns {Handlebars.SafeString}
 */
function formatText(value) {
  return new Handlebars.SafeString(value?.replaceAll("\n", "<br>") ?? "");
}

/* -------------------------------------------- */

/**
 * A helper function that formats a time in a human-readable format.
 * @param {number} value         Time to display.
 * @param {string} unit          Units as defined in `CONFIG.DND5E.timeUnits`.
 * @param {object} [options={}]  Formatting options passed to `formatNumber`.
 * @returns {string}
 */
function formatTime(value, unit, options={}) {
  options.maximumFractionDigits ??= 0;
  options.unitDisplay ??= "long";
  const config = CONFIG.DND5E.timeUnits[unit];
  if ( config?.counted ) {
    if ( (options.unitDisplay === "narrow") && game.i18n.has(`${config.counted}.narrow`) ) {
      return game.i18n.format(`${config.counted}.narrow`, { number: formatNumber(value, options) });
    } else {
      const pr = new Intl.PluralRules(game.i18n.lang);
      return game.i18n.format(`${config.counted}.${pr.select(value)}`, { number: formatNumber(value, options) });
    }
  }
  try {
    return formatNumber(value, { ...options, style: "unit", unit });
  } catch(err) {
    return formatNumber(value, options);
  }
}

/* -------------------------------------------- */

/**
 * Form a number using the provided volume unit.
 * @param {number} value         The volume to format.
 * @param {string} unit          Volume unit as defined in `CONFIG.DND5E.volumeUnits`.
 * @param {object} [options={}]  Formatting options passed to `formatNumber`.
 * @returns {string}
 */
function formatVolume(value, unit, options={}) {
  return _formatSystemUnits(value, unit, CONFIG.DND5E.volumeUnits[unit], options);
}

/* -------------------------------------------- */

/**
 * Form a number using the provided weight unit.
 * @param {number} value         The weight to format.
 * @param {string} unit          Weight unit as defined in `CONFIG.DND5E.weightUnits`.
 * @param {object} [options={}]  Formatting options passed to `formatNumber`.
 * @returns {string}
 */
function formatWeight(value, unit, options={}) {
  return _formatSystemUnits(value, unit, CONFIG.DND5E.weightUnits[unit], options);
}

/* -------------------------------------------- */

/**
 * Format a number using one of core's built-in unit types.
 * @param {number} value                   Value to display.
 * @param {string} unit                    Name of the unit to use.
 * @param {UnitConfiguration} config       Configuration data for the unit.
 * @param {object} [options={}]            Formatting options passed to `formatNumber`.
 * @param {boolean} [options.parts=false]  Format to parts.
 * @returns {string}
 */
function _formatSystemUnits(value, unit, config, { parts=false, ...options }={}) {
  options.unitDisplay ??= "short";
  if ( config?.counted ) {
    const localizationKey = `${config.counted}.${options.unitDisplay}.${getPluralRules().select(value)}`;
    return game.i18n.format(localizationKey, { number: formatNumber(value, options) });
  }
  unit = config?.formattingUnit ?? unit;
  if ( isValidUnit(unit) ) {
    options.style ??= "unit";
    options.unit ??= unit;
  }
  return (parts ? formatNumberParts : formatNumber)(value, options);
}

/* -------------------------------------------- */

/**
 * Cached store of Intl.PluralRules instances.
 * @type {Record<string, Intl.PluralRules>}
 */
const _pluralRules = {};

/**
 * Get a PluralRules object, fetching from cache if possible.
 * @param {object} [options={}]
 * @param {string} [options.type=cardinal]
 * @returns {Intl.PluralRules}
 */
function getPluralRules({ type="cardinal" }={}) {
  _pluralRules[type] ??= new Intl.PluralRules(game.i18n.lang, { type });
  return _pluralRules[type];
}

/* -------------------------------------------- */
/*  Formulas                                    */
/* -------------------------------------------- */

/**
 * Return whether a string is a valid reroll, explosion, min, or max dice modifier.
 * @param {string} mod      The modifier to test.
 * @returns {boolean}
 */
function isValidDieModifier(mod) {
  const regex = {
    reroll: /rr?([0-9]+)?([<>=]+)?([0-9]+)?/i,
    explode: /xo?([0-9]+)?([<>=]+)?([0-9]+)?/i,
    minimum: /(?:min)([0-9]+)/i,
    maximum: /(?:max)([0-9]+)/i,
    dropKeep: /[dk]([hl])?([0-9]+)?/i,
    count: /(?:c[sf])([<>=]+)?([0-9]+)?/i
  };
  return Object.values(regex).some(rgx => rgx.test(mod));
}

/* -------------------------------------------- */

/**
 * Handle a delta input for a number value from a form.
 * @param {HTMLInputElement} input  Input that contains the modified value.
 * @param {Document} target         Target document to be updated.
 * @returns {number|void}
 */
function parseInputDelta(input, target) {
  let value = input.value;
  if ( ["+", "-"].includes(value[0]) ) {
    const delta = parseFloat(value);
    value = Number(foundry.utils.getProperty(target, input.dataset.name ?? input.name)) + delta;
  }
  else if ( value[0] === "=" ) value = Number(value.slice(1));
  if ( Number.isNaN(value) ) return;
  input.value = value;
  return value;
}

/* -------------------------------------------- */

/**
 * Prepare the final formula value for a model field.
 * @param {ItemDataModel|BaseActivityData} model  Model for which the value is being prepared.
 * @param {string} keyPath                        Path to the field within the model.
 * @param {string} label                          Label to use in preparation warnings.
 * @param {object} rollData                       Roll data to use when replacing formula values.
 */
function prepareFormulaValue(model, keyPath, label, rollData) {
  const value = foundry.utils.getProperty(model, keyPath);
  if ( !value ) return;
  const item = model.item ?? model.parent;
  const property = game.i18n.localize(label);
  try {
    const formula = replaceFormulaData(value, rollData, { item, property });
    const roll = new Roll(formula);
    foundry.utils.setProperty(model, keyPath, roll.evaluateSync().total);
  } catch(err) {
    if ( item.isEmbedded ) {
      const message = game.i18n.format("DND5E.FormulaMalformedError", { property, name: model.name ?? item.name });
      item.actor._preparationWarnings.push({ message, link: item.uuid, type: "error" });
      console.error(message, err);
    }
  }
}

/* -------------------------------------------- */

/**
 * Replace referenced data attributes in the roll formula with values from the provided data.
 * If the attribute is not found in the provided data, display a warning on the actor.
 * @param {string} formula           The original formula within which to replace.
 * @param {object} data              The data object which provides replacements.
 * @param {object} [options={}]
 * @param {Actor5e} [options.actor]            Actor for which the value is being prepared.
 * @param {Item5e} [options.item]              Item for which the value is being prepared.
 * @param {string|null} [options.missing="0"]  Value to use when replacing missing references, or `null` to not replace.
 * @param {string} [options.property]          Name of the property to which this formula belongs.
 * @returns {string}                 Formula with replaced data.
 */
function replaceFormulaData(formula, data, { actor, item, missing="0", property }={}) {
  const dataRgx = new RegExp(/@([a-z.0-9_-]+)/gi);
  const missingReferences = new Set();
  formula = String(formula).replace(dataRgx, (match, term) => {
    let value = foundry.utils.getProperty(data, term);
    if ( value == null ) {
      missingReferences.add(match);
      return missing ?? match[0];
    }
    return String(value).trim();
  });
  actor ??= item?.parent;
  if ( (missingReferences.size > 0) && actor && property ) {
    const listFormatter = new Intl.ListFormat(game.i18n.lang, { style: "long", type: "conjunction" });
    const message = game.i18n.format("DND5E.FormulaMissingReferenceWarn", {
      property, name: item?.name ?? actor.name, references: listFormatter.format(missingReferences)
    });
    actor._preparationWarnings.push({ message, link: item?.uuid ?? actor.uuid, type: "warning" });
  }
  return formula;
}

/* -------------------------------------------- */

/**
 * Convert a bonus value to a simple integer for displaying on the sheet.
 * @param {number|string|null} bonus  Bonus formula.
 * @param {object} [data={}]          Data to use for replacing @ strings.
 * @returns {number}                  Simplified bonus as an integer.
 * @protected
 */
function simplifyBonus(bonus, data={}) {
  if ( !bonus ) return 0;
  if ( Number.isNumeric(bonus) ) return Number(bonus);
  try {
    const roll = new Roll(bonus, data);
    return roll.isDeterministic ? roll.evaluateSync().total : 0;
  } catch(error) {
    console.error(error);
    return 0;
  }
}

/* -------------------------------------------- */
/*  IDs                                         */
/* -------------------------------------------- */

/**
 * Create an ID from the input truncating or padding the value to make it reach 16 characters.
 * @param {string} id
 * @returns {string}
 */
function staticID(id) {
  if ( id.length >= 16 ) return id.substring(0, 16);
  return id.padEnd(16, "0");
}

/* -------------------------------------------- */
/*  Keybindings Helper                          */
/* -------------------------------------------- */

/**
 * Track which KeyboardEvent#code presses associate with each modifier.
 * Added support for treating Meta separate from Control.
 * @enum {string[]}
 */
const MODIFIER_CODES = {
  Alt: KeyboardManager.MODIFIER_CODES.Alt,
  Control: KeyboardManager.MODIFIER_CODES.Control.filter(k => k.startsWith("Control")),
  Meta: KeyboardManager.MODIFIER_CODES.Control.filter(k => !k.startsWith("Control")),
  Shift: KeyboardManager.MODIFIER_CODES.Shift
};

/**
 * Based on the provided event, determine if the keys are pressed to fulfill the specified keybinding.
 * @param {Event} event    Triggering event.
 * @param {string} action  Keybinding action within the `dnd5e` namespace.
 * @returns {boolean}      Is the keybinding triggered?
 */
function areKeysPressed(event, action) {
  if ( !event ) return false;
  const activeModifiers = {};
  const addModifiers = (key, pressed) => {
    activeModifiers[key] = pressed;
    MODIFIER_CODES[key].forEach(n => activeModifiers[n] = pressed);
  };
  addModifiers(KeyboardManager.MODIFIER_KEYS.ALT, event.altKey);
  addModifiers(KeyboardManager.MODIFIER_KEYS.CONTROL, event.ctrlKey);
  addModifiers("Meta", event.metaKey);
  addModifiers(KeyboardManager.MODIFIER_KEYS.SHIFT, event.shiftKey);
  return game.keybindings.get("dnd5e", action).some(b => {
    if ( game.keyboard.downKeys.has(b.key) && b.modifiers.every(m => activeModifiers[m]) ) return true;
    if ( b.modifiers.length ) return false;
    return activeModifiers[b.key];
  });
}

/* -------------------------------------------- */
/*  Logging                                     */
/* -------------------------------------------- */

/**
 * Log a console message with the "D&D 5e" prefix and styling.
 * @param {string} message                    Message to display.
 * @param {object} [options={}]
 * @param {string} [options.color="#6e0000"]  Color to use for the log.
 * @param {any[]} [options.extras=[]]         Extra options passed to the logging method.
 * @param {string} [options.level="log"]      Console logging method to call.
 */
function log(message, { color="#6e0000", extras=[], level="log" }={}) {
  console[level](
    `%cD&D 5e | %c${message}`, `color: ${color}; font-variant: small-caps`, "color: revert", ...extras
  );
}

/* -------------------------------------------- */
/*  Object Helpers                              */
/* -------------------------------------------- */

/**
 * Transform an object, returning only the keys which match the provided filter.
 * @param {object} obj         Object to transform.
 * @param {Function} [filter]  Filtering function. If none is provided, it will just check for truthiness.
 * @returns {string[]}         Array of filtered keys.
 */
function filteredKeys(obj, filter) {
  filter ??= e => e;
  return Object.entries(obj).filter(e => filter(e[1])).map(e => e[0]);
}

/* -------------------------------------------- */

/**
 * Check whether an object exists without transversing any getters, preventing any deprecation warnings from triggering.
 * @param {object} object
 * @param {string} keyPath
 * @returns {boolean}
 */
function safePropertyExists(object, keyPath) {
  const parts = keyPath.split(".");
  for ( const part of parts ) {
    const descriptor = Object.getOwnPropertyDescriptor(object, part);
    if ( !descriptor || !("value" in descriptor) ) return false;
    object = object[part];
  }
  return true;
}

/* -------------------------------------------- */

/**
 * Sort the provided object by its values or by an inner sortKey.
 * @param {object} obj                 The object to sort.
 * @param {string|Function} [sortKey]  An inner key upon which to sort or sorting function.
 * @returns {object}                   A copy of the original object that has been sorted.
 */
function sortObjectEntries(obj, sortKey) {
  let sorted = Object.entries(obj);
  const sort = (lhs, rhs) => foundry.utils.getType(lhs) === "string" ? lhs.localeCompare(rhs, game.i18n.lang) : lhs - rhs;
  if ( foundry.utils.getType(sortKey) === "function" ) sorted = sorted.sort((lhs, rhs) => sortKey(lhs[1], rhs[1]));
  else if ( sortKey ) sorted = sorted.sort((lhs, rhs) => sort(lhs[1][sortKey], rhs[1][sortKey]));
  else sorted = sorted.sort((lhs, rhs) => sort(lhs[1], rhs[1]));
  return Object.fromEntries(sorted);
}

/* -------------------------------------------- */

/**
 * Retrieve the indexed data for a Document using its UUID. Will never return a result for embedded documents.
 * @param {string} uuid  The UUID of the Document index to retrieve.
 * @returns {object}     Document's index if one could be found.
 */
function indexFromUuid(uuid) {
  const parts = uuid.split(".");
  let index;

  // Compendium Documents
  if ( parts[0] === "Compendium" ) {
    const [, scope, packName, id] = parts;
    const pack = game.packs.get(`${scope}.${packName}`);
    index = pack?.index.get(id);
  }

  // World Documents
  else if ( parts.length < 3 ) {
    const [docName, id] = parts;
    const collection = CONFIG[docName].collection.instance;
    index = collection.get(id);
  }

  return index || null;
}

/* -------------------------------------------- */

/**
 * Creates an HTML document link for the provided UUID.
 * Try to build links to compendium content synchronously to avoid DB lookups.
 * @param {string} uuid                    UUID for which to produce the link.
 * @param {object} [options]
 * @param {string} [options.tooltip]       Tooltip to add to the link.
 * @param {string} [options.renderBroken]  If a UUID cannot found, render it as a broken link instead of returning the
 *                                         empty string.
 * @returns {string}                       Link to the item or empty string if item wasn't found.
 */
function linkForUuid(uuid, { tooltip, renderBroken }={}) {
  let doc = fromUuidSync(uuid);
  if ( !doc ) {
    if ( renderBroken ) return `
      <a class="content-link broken" data-uuid="${uuid}">
        <i class="fas fa-unlink"></i> ${game.i18n.localize("Unknown")}
      </a>
    `;
    return "";
  }
  if ( uuid.startsWith("Compendium.") && !(doc instanceof foundry.abstract.Document) ) {
    const {collection} = foundry.utils.parseUuid(uuid);
    const cls = collection.documentClass;
    // Minimal "shell" of a document using index data
    doc = new cls(foundry.utils.deepClone(doc), {pack: collection.metadata.id});
  }
  const a = doc.toAnchor();
  if ( tooltip ) a.dataset.tooltip = tooltip;
  return a.outerHTML;
}

/* -------------------------------------------- */
/*  Targeting                                   */
/* -------------------------------------------- */

/**
 * Important information on a targeted token.
 *
 * @typedef {object} TargetDescriptor5e
 * @property {string} uuid  The UUID of the target.
 * @property {string} img   The target's image.
 * @property {string} name  The target's name.
 * @property {number} ac    The target's armor class, if applicable.
 */

/**
 * Grab the targeted tokens and return relevant information on them.
 * @returns {TargetDescriptor[]}
 */
function getTargetDescriptors() {
  const targets = new Map();
  for ( const token of game.user.targets ) {
    const { name } = token;
    const { img, system, uuid, statuses } = token.actor ?? {};
    if ( uuid ) {
      const ac = statuses.has("coverTotal") ? null : system.attributes?.ac?.value;
      targets.set(uuid, { name, img, uuid, ac: ac ?? null });
    }
  }
  return Array.from(targets.values());
}

/* -------------------------------------------- */

/**
 * Get currently selected tokens in the scene or user's character's tokens.
 * @returns {Token5e[]}
 */
function getSceneTargets() {
  let targets = canvas.tokens?.controlled.filter(t => t.actor) ?? [];
  if ( !targets.length && game.user.character ) targets = game.user.character.getActiveTokens();
  return targets;
}

/* -------------------------------------------- */
/*  Conversions                                 */
/* -------------------------------------------- */

/**
 * Convert the provided length to another unit.
 * @param {number} value                   The length being converted.
 * @param {string} from                    The initial units.
 * @param {string} to                      The final units.
 * @param {object} [options={}]
 * @param {boolean} [options.strict=true]  Throw an error if either unit isn't found.
 * @returns {number}
 */
function convertLength(value, from, to, { strict=true }={}) {
  const message = unit => `Length unit ${unit} not defined in CONFIG.DND5E.movementUnits`;
  return _convertSystemUnits(value, from, to, CONFIG.DND5E.movementUnits, { message, strict });
}

/* -------------------------------------------- */

/**
 * Convert the provided time value to another unit. If no final unit is provided, then will convert it to the largest
 * unit that can still represent the value as a whole number.
 * @param {number} value                    The time being converted.
 * @param {string} from                     The initial unit as defined in `CONFIG.DND5E.timeUnits`.
 * @param {object} [options={}]
 * @param {boolean} [options.combat=false]  Use combat units when auto-selecting units, rather than normal units.
 * @param {boolean} [options.strict=true]   Throw an error if from unit isn't found.
 * @param {string} [options.to]             The final units, if explicitly provided.
 * @returns {{ value: number, unit: string }}
 */
function convertTime(value, from, { combat=false, strict=true, to }={}) {
  const base = value * (CONFIG.DND5E.timeUnits[from]?.conversion ?? 1);
  if ( !to ) {
    // Find unit with largest conversion value that can still display the value
    const unitOptions = Object.entries(CONFIG.DND5E.timeUnits)
      .reduce((arr, [key, v]) => {
        if ( ((v.combat ?? false) === combat) && ((base % v.conversion === 0) || (base >= v.conversion * 2)) ) {
          arr.push({ key, conversion: v.conversion });
        }
        return arr;
      }, [])
      .sort((lhs, rhs) => rhs.conversion - lhs.conversion);
    to = unitOptions[0]?.key ?? from;
  }

  const message = unit => `Time unit ${unit} not defined in CONFIG.DND5E.timeUnits`;
  return { value: _convertSystemUnits(value, from, to, CONFIG.DND5E.timeUnits, { message, strict }), unit: to };
}

/* -------------------------------------------- */

/**
 * Convert the provided weight to another unit.
 * @param {number} value                   The weight being converted.
 * @param {string} from                    The initial unit as defined in `CONFIG.DND5E.weightUnits`.
 * @param {string} to                      The final units.
 * @param {object} [options={}]
 * @param {boolean} [options.strict=true]  Throw an error if either unit isn't found.
 * @returns {number}      Weight in the specified units.
 */
function convertWeight(value, from, to, { strict=true }={}) {
  const message = unit => `Weight unit ${unit} not defined in CONFIG.DND5E.weightUnits`;
  return _convertSystemUnits(value, from, to, CONFIG.DND5E.weightUnits, { message, strict });
}

/* -------------------------------------------- */

/**
 * Convert from one unit to another using one of core's built-in unit types.
 * @param {number} value                                Value to display.
 * @param {string} from                                 The initial unit.
 * @param {string} to                                   The final unit.
 * @param {UnitConfiguration} config                    Configuration data for the unit.
 * @param {object} options
 * @param {function(string): string} [options.message]  Method used to produce the error message if unit not found.
 * @param {boolean} [options.strict]                    Throw an error if either unit isn't found.
 * @returns {string}
 */
function _convertSystemUnits(value, from, to, config, { message, strict }) {
  if ( from === to ) return value;
  if ( strict && !config[from] ) throw new Error(message(from));
  if ( strict && !config[to] ) throw new Error(message(to));
  return value * (config[from]?.conversion ?? 1) / (config[to]?.conversion ?? 1);
}

/* -------------------------------------------- */

/**
 * Default units to use depending on system setting.
 * @param {"length"|"weight"} type  Type of units to select.
 * @returns {string}
 */
function defaultUnits(type) {
  return CONFIG.DND5E.defaultUnits[type]?.[
    game.settings.get("dnd5e", `metric${type.capitalize()}Units`) ? "metric" : "imperial"
  ];
}

/* -------------------------------------------- */
/*  Validators                                  */
/* -------------------------------------------- */

/**
 * Ensure the provided string contains only the characters allowed in identifiers.
 * @param {string} identifier
 * @returns {boolean}
 */
function isValidIdentifier(identifier) {
  return /^([a-z0-9_-]+)$/i.test(identifier);
}

const validators = {
  isValidIdentifier: isValidIdentifier
};

/* -------------------------------------------- */

/**
 * Determine whether the provided unit is usable within `Intl.NumberFormat`.
 * @param {string} unit
 * @returns {boolean}
 */
function isValidUnit(unit) {
  if ( unit?.includes("-per-") ) return unit.split("-per-").every(u => isValidUnit(u));
  return Intl.supportedValuesOf("unit").includes(unit);
}

/* -------------------------------------------- */
/*  Handlebars Template Helpers                 */
/* -------------------------------------------- */

/**
 * Define a set of template paths to pre-load. Pre-loaded templates are compiled and cached for fast access when
 * rendering. These paths will also be available as Handlebars partials by using the file name
 * (e.g. "dnd5e.actor-traits").
 * @returns {Promise}
 */
async function preloadHandlebarsTemplates() {
  const partials = [
    // Shared Partials
    "systems/dnd5e/templates/shared/active-effects.hbs",
    "systems/dnd5e/templates/shared/active-effects2.hbs",
    "systems/dnd5e/templates/shared/inventory.hbs",
    "systems/dnd5e/templates/shared/inventory2.hbs",
    "systems/dnd5e/templates/apps/parts/trait-list.hbs",
    "systems/dnd5e/templates/apps/parts/traits-list.hbs",

    // Actor Sheet Partials
    "systems/dnd5e/templates/actors/parts/actor-classes.hbs",
    "systems/dnd5e/templates/actors/parts/actor-trait-pills.hbs",
    "systems/dnd5e/templates/actors/parts/actor-traits.hbs",
    "systems/dnd5e/templates/actors/parts/actor-features.hbs",
    "systems/dnd5e/templates/actors/parts/actor-inventory.hbs",
    "systems/dnd5e/templates/actors/parts/actor-spellbook.hbs",
    "systems/dnd5e/templates/actors/parts/actor-warnings.hbs",
    "systems/dnd5e/templates/actors/parts/actor-warnings-dialog.hbs",
    "systems/dnd5e/templates/actors/parts/biography-textbox.hbs",
    "systems/dnd5e/templates/actors/tabs/character-bastion.hbs",
    "systems/dnd5e/templates/actors/tabs/character-biography.hbs",
    "systems/dnd5e/templates/actors/tabs/character-details.hbs",
    "systems/dnd5e/templates/actors/tabs/creature-features.hbs",
    "systems/dnd5e/templates/actors/tabs/creature-special-traits.hbs",
    "systems/dnd5e/templates/actors/tabs/creature-spells.hbs",
    "systems/dnd5e/templates/actors/tabs/group-members.hbs",
    "systems/dnd5e/templates/actors/tabs/npc-biography.hbs",

    // Actor Sheet Item Summary Columns
    "systems/dnd5e/templates/actors/parts/columns/column-feature-controls.hbs",
    "systems/dnd5e/templates/actors/parts/columns/column-formula.hbs",
    "systems/dnd5e/templates/actors/parts/columns/column-recovery.hbs",
    "systems/dnd5e/templates/actors/parts/columns/column-roll.hbs",
    "systems/dnd5e/templates/actors/parts/columns/column-uses.hbs",

    // Chat Message Partials
    "systems/dnd5e/templates/chat/parts/card-activities.hbs",
    "systems/dnd5e/templates/chat/parts/card-deltas.hbs",

    // Item Sheet Partials
    "systems/dnd5e/templates/items/details/details-background.hbs",
    "systems/dnd5e/templates/items/details/details-class.hbs",
    "systems/dnd5e/templates/items/details/details-consumable.hbs",
    "systems/dnd5e/templates/items/details/details-container.hbs",
    "systems/dnd5e/templates/items/details/details-equipment.hbs",
    "systems/dnd5e/templates/items/details/details-facility.hbs",
    "systems/dnd5e/templates/items/details/details-feat.hbs",
    "systems/dnd5e/templates/items/details/details-loot.hbs",
    "systems/dnd5e/templates/items/details/details-mountable.hbs",
    "systems/dnd5e/templates/items/details/details-species.hbs",
    "systems/dnd5e/templates/items/details/details-spell.hbs",
    "systems/dnd5e/templates/items/details/details-spellcasting.hbs",
    "systems/dnd5e/templates/items/details/details-starting-equipment.hbs",
    "systems/dnd5e/templates/items/details/details-subclass.hbs",
    "systems/dnd5e/templates/items/details/details-tool.hbs",
    "systems/dnd5e/templates/items/details/details-weapon.hbs",
    "systems/dnd5e/templates/items/parts/item-action.hbs",
    "systems/dnd5e/templates/items/parts/item-activation.hbs",
    "systems/dnd5e/templates/items/parts/item-activities.hbs",
    "systems/dnd5e/templates/items/parts/item-advancement.hbs",
    "systems/dnd5e/templates/items/parts/item-advancement2.hbs",
    "systems/dnd5e/templates/items/parts/item-description.hbs",
    "systems/dnd5e/templates/items/parts/item-description2.hbs",
    "systems/dnd5e/templates/items/parts/item-details.hbs",
    "systems/dnd5e/templates/items/parts/item-mountable.hbs",
    "systems/dnd5e/templates/items/parts/item-spellcasting.hbs",
    "systems/dnd5e/templates/items/parts/item-source.hbs",
    "systems/dnd5e/templates/items/parts/item-summary.hbs",
    "systems/dnd5e/templates/items/parts/item-tooltip.hbs",
    "systems/dnd5e/templates/items/parts/spell-block.hbs",

    // Field Partials
    "systems/dnd5e/templates/shared/fields/field-activation.hbs",
    "systems/dnd5e/templates/shared/fields/field-damage.hbs",
    "systems/dnd5e/templates/shared/fields/field-duration.hbs",
    "systems/dnd5e/templates/shared/fields/field-range.hbs",
    "systems/dnd5e/templates/shared/fields/field-targets.hbs",
    "systems/dnd5e/templates/shared/fields/field-uses.hbs",
    "systems/dnd5e/templates/shared/fields/fieldlist.hbs",

    // Journal Partials
    "systems/dnd5e/templates/journal/parts/journal-legacy-traits.hbs",
    "systems/dnd5e/templates/journal/parts/journal-modern-traits.hbs",
    "systems/dnd5e/templates/journal/parts/journal-table.hbs",

    // Activity Partials
    "systems/dnd5e/templates/activity/columns/activity-column-controls.hbs",
    "systems/dnd5e/templates/activity/columns/activity-column-formula.hbs",
    "systems/dnd5e/templates/activity/columns/activity-column-price.hbs",
    "systems/dnd5e/templates/activity/columns/activity-column-quantity.hbs",
    "systems/dnd5e/templates/activity/columns/activity-column-range.hbs",
    "systems/dnd5e/templates/activity/columns/activity-column-recovery.hbs",
    "systems/dnd5e/templates/activity/columns/activity-column-roll.hbs",
    "systems/dnd5e/templates/activity/columns/activity-column-school.hbs",
    "systems/dnd5e/templates/activity/columns/activity-column-target.hbs",
    "systems/dnd5e/templates/activity/columns/activity-column-time.hbs",
    "systems/dnd5e/templates/activity/columns/activity-column-uses.hbs",
    "systems/dnd5e/templates/activity/columns/activity-column-weight.hbs",
    "systems/dnd5e/templates/activity/activity-row-summary.hbs",
    "systems/dnd5e/templates/activity/parts/activity-usage-notes.hbs",

    // Advancement Partials
    "systems/dnd5e/templates/advancement/parts/advancement-ability-score-control.hbs",
    "systems/dnd5e/templates/advancement/parts/advancement-controls.hbs",
    "systems/dnd5e/templates/advancement/parts/advancement-spell-config.hbs"
  ];

  const paths = {};
  for ( const path of partials ) {
    paths[path.replace(".hbs", ".html")] = path;
    paths[`dnd5e.${path.split("/").pop().replace(".hbs", "")}`] = path;
  }

  return loadTemplates(paths);
}

/* -------------------------------------------- */

/**
 * A helper that converts the provided object into a series of `data-` entries.
 * @param {object} object   Object to convert into dataset entries.
 * @param {object} options  Handlebars options.
 * @returns {string}
 */
function dataset(object, options) {
  const entries = [];
  for ( let [key, value] of Object.entries(object ?? {}) ) {
    if ( value === undefined ) continue;
    key = key.replace(/[A-Z]+(?![a-z])|[A-Z]/g, (a, b) => (b ? "-" : "") + a.toLowerCase());
    entries.push(`data-${key}="${value}"`);
  }
  return new Handlebars.SafeString(entries.join(" "));
}

/* -------------------------------------------- */

/**
 * A helper to create a set of <option> elements in a <select> block grouped together
 * in <optgroup> based on the provided categories.
 *
 * @param {SelectChoices} choices          Choices to format.
 * @param {object} [options]
 * @param {boolean} [options.localize]     Should the label be localized?
 * @param {string} [options.blank]         Name for the empty option, if one should be added.
 * @param {string} [options.labelAttr]     Attribute pointing to label string.
 * @param {string} [options.chosenAttr]    Attribute pointing to chosen boolean.
 * @param {string} [options.childrenAttr]  Attribute pointing to array of children.
 * @returns {Handlebars.SafeString}        Formatted option list.
 */
function groupedSelectOptions(choices, options) {
  const localize = options.hash.localize ?? false;
  const blank = options.hash.blank ?? null;
  const labelAttr = options.hash.labelAttr ?? "label";
  const chosenAttr = options.hash.chosenAttr ?? "chosen";
  const childrenAttr = options.hash.childrenAttr ?? "children";

  // Create an option
  const option = (name, label, chosen) => {
    if ( localize ) label = game.i18n.localize(label);
    html += `<option value="${name}" ${chosen ? "selected" : ""}>${label}</option>`;
  };

  // Create a group
  const group = category => {
    let label = category[labelAttr];
    if ( localize ) game.i18n.localize(label);
    html += `<optgroup label="${label}">`;
    children(category[childrenAttr]);
    html += "</optgroup>";
  };

  // Add children
  const children = children => {
    for ( let [name, child] of Object.entries(children) ) {
      if ( child[childrenAttr] ) group(child);
      else option(name, child[labelAttr], child[chosenAttr] ?? false);
    }
  };

  // Create the options
  let html = "";
  if ( blank !== null ) option("", blank);
  children(choices);
  return new Handlebars.SafeString(html);
}

/* -------------------------------------------- */

/**
 * A helper that fetch the appropriate item context from root and adds it to the first block parameter.
 * @param {object} context  Current evaluation context.
 * @param {object} options  Handlebars options.
 * @returns {string}
 */
function itemContext(context, options) {
  if ( arguments.length !== 2 ) throw new Error("#dnd5e-itemContext requires exactly one argument");
  if ( foundry.utils.getType(context) === "function" ) context = context.call(this);

  const ctx = options.data.root.itemContext?.[context.id];
  if ( !ctx ) {
    const inverse = options.inverse(this);
    if ( inverse ) return options.inverse(this);
  }

  return options.fn(context, { data: options.data, blockParams: [ctx] });
}

/* -------------------------------------------- */

/**
 * Conceal a section and display a notice if unidentified.
 * @param {boolean} conceal  Should the section be concealed?
 * @param {object} options   Handlebars options.
 * @returns {string}
 */
function concealSection(conceal, options) {
  let content = options.fn(this);
  if ( !conceal ) return content;

  content = `<div inert>
    ${content}
  </div>
  <div class="unidentified-notice">
      <div>
          <strong>${game.i18n.localize("DND5E.Unidentified.Title")}</strong>
          <p>${game.i18n.localize("DND5E.Unidentified.Notice")}</p>
      </div>
  </div>`;
  return content;
}

/* -------------------------------------------- */

/**
 * Construct an object from the provided arguments.
 * @param {object} options       Handlebars options.
 * @param {object} options.hash
 * @returns {object}
 */
function makeObject({ hash }) {
  return hash;
}

/* -------------------------------------------- */

/**
 * Register custom Handlebars helpers used by 5e.
 */
function registerHandlebarsHelpers() {
  Handlebars.registerHelper({
    getProperty: foundry.utils.getProperty,
    "dnd5e-concealSection": concealSection,
    "dnd5e-dataset": dataset,
    "dnd5e-formatCR": formatCR,
    "dnd5e-formatModifier": formatModifier,
    "dnd5e-groupedSelectOptions": groupedSelectOptions,
    "dnd5e-itemContext": itemContext,
    "dnd5e-linkForUuid": (uuid, options) => linkForUuid(uuid, options.hash),
    "dnd5e-numberFormat": (context, options) => formatNumber(context, options.hash),
    "dnd5e-numberParts": (context, options) => formatNumberParts(context, options.hash),
    "dnd5e-object": makeObject,
    "dnd5e-textFormat": formatText
  });
}

/* -------------------------------------------- */
/*  Config Pre-Localization                     */
/* -------------------------------------------- */

/**
 * Storage for pre-localization configuration.
 * @type {object}
 * @private
 */
const _preLocalizationRegistrations = {};

/**
 * Mark the provided config key to be pre-localized during the init stage.
 * @param {string} configKeyPath          Key path within `CONFIG.DND5E` to localize.
 * @param {object} [options={}]
 * @param {string} [options.key]          If each entry in the config enum is an object,
 *                                        localize and sort using this property.
 * @param {string[]} [options.keys=[]]    Array of localization keys. First key listed will be used for sorting
 *                                        if multiple are provided.
 * @param {boolean} [options.sort=false]  Sort this config enum, using the key if set.
 */
function preLocalize(configKeyPath, { key, keys=[], sort=false }={}) {
  if ( key ) keys.unshift(key);
  _preLocalizationRegistrations[configKeyPath] = { keys, sort };
}

/* -------------------------------------------- */

/**
 * Execute previously defined pre-localization tasks on the provided config object.
 * @param {object} config  The `CONFIG.DND5E` object to localize and sort. *Will be mutated.*
 */
function performPreLocalization(config) {
  for ( const [keyPath, settings] of Object.entries(_preLocalizationRegistrations) ) {
    const target = foundry.utils.getProperty(config, keyPath);
    if ( !target ) continue;
    _localizeObject(target, settings.keys);
    if ( settings.sort ) foundry.utils.setProperty(config, keyPath, sortObjectEntries(target, settings.keys[0]));
  }

  // Localize & sort status effects
  CONFIG.statusEffects.forEach(s => s.name = game.i18n.localize(s.name));
  CONFIG.statusEffects.sort((lhs, rhs) =>
    lhs.order || rhs.order ? (lhs.order ?? Infinity) - (rhs.order ?? Infinity)
      : lhs.name.localeCompare(rhs.name, game.i18n.lang)
  );
}

/* -------------------------------------------- */

/**
 * Localize the values of a configuration object by translating them in-place.
 * @param {object} obj       The configuration object to localize.
 * @param {string[]} [keys]  List of inner keys that should be localized if this is an object.
 * @private
 */
function _localizeObject(obj, keys) {
  for ( const [k, v] of Object.entries(obj) ) {
    const type = typeof v;
    if ( type === "string" ) {
      obj[k] = game.i18n.localize(v);
      continue;
    }

    if ( type !== "object" ) {
      console.error(new Error(
        `Pre-localized configuration values must be a string or object, ${type} found for "${k}" instead.`
      ));
      continue;
    }
    if ( !keys?.length ) {
      console.error(new Error(
        "Localization keys must be provided for pre-localizing when target is an object."
      ));
      continue;
    }

    for ( const key of keys ) {
      const value = foundry.utils.getProperty(v, key);
      if ( !value ) continue;
      foundry.utils.setProperty(v, key, game.i18n.localize(value));
    }
  }
}

/* -------------------------------------------- */
/*  Localization                                */
/* -------------------------------------------- */

/**
 * A cache of already-fetched labels for faster lookup.
 * @type {Record<string, Map<string, string>>}
 */
const _attributeLabelCache = {
  activity: new Map(),
  actor: new Map(),
  item: new Map()
};

/**
 * Convert an attribute path to a human-readable label. Assumes paths are on an actor unless an reference item
 * is provided.
 * @param {string} attr              The attribute path.
 * @param {object} [options]
 * @param {Actor5e} [options.actor]  An optional reference actor.
 * @param {Item5e} [options.item]    An optional reference item.
 * @returns {string|void}
 */
function getHumanReadableAttributeLabel(attr, { actor, item }={}) {
  if ( attr.startsWith("system.") ) attr = attr.slice(7);

  // Check any actor-specific names first.
  if ( attr.startsWith("resources.") && actor ) {
    const key = attr.replace(/\.value$/, "");
    const resource = foundry.utils.getProperty(actor, `system.${key}`);
    if ( resource?.label ) return resource.label;
  }

  if ( (attr === "details.xp.value") && (actor?.type === "npc") ) {
    return game.i18n.localize("DND5E.ExperiencePoints.Value");
  }

  if ( attr.startsWith(".") && actor ) {
    // TODO: Remove `strict: false` when https://github.com/foundryvtt/foundryvtt/issues/11214 is resolved
    // Only necessary when opening the token config for an actor in a compendium
    const item = fromUuidSync(attr, { relative: actor, strict: false });
    return item?.name ?? attr;
  }

  // Check if the attribute is already in cache.
  let label = item ? null : _attributeLabelCache.actor.get(attr);
  if ( label ) return label;
  let name;
  let type = "actor";

  const getSchemaLabel = (attr, type, doc) => {
    if ( doc ) return doc.system.schema.getField(attr)?.label;
    for ( const model of Object.values(CONFIG[type].dataModels) ) {
      const field = model.schema.getField(attr);
      if ( field ) return field.label;
    }
  };

  // Activity labels
  if ( item && attr.startsWith("activities.") ) {
    let [, activityId, ...keyPath] = attr.split(".");
    const activity = item.system.activities?.get(activityId);
    if ( !activity ) return attr;
    attr = keyPath.join(".");
    name = `${item.name}: ${activity.name}`;
    type = "activity";
    if ( _attributeLabelCache.activity.has(attr) ) label = _attributeLabelCache.activity.get(attr);
    else if ( attr === "uses.spent" ) label = "DND5E.Uses";
  }

  // Item labels
  else if ( item ) {
    name = item.name;
    type = "item";
    if ( _attributeLabelCache.item.has(attr) ) label = _attributeLabelCache.item.get(attr);
    else if ( attr === "hd.spent" ) label = "DND5E.HitDice";
    else if ( attr === "uses.spent" ) label = "DND5E.Uses";
    else label = getSchemaLabel(attr, "Item", item);
  }

  // Derived fields.
  else if ( attr === "attributes.init.total" ) label = "DND5E.InitiativeBonus";
  else if ( (attr === "attributes.ac.value") || (attr === "attributes.ac.flat") ) label = "DND5E.ArmorClass";
  else if ( attr === "attributes.spell.dc" ) label = "DND5E.SpellDC";

  // Abilities.
  else if ( attr.startsWith("abilities.") ) {
    const [, key] = attr.split(".");
    label = game.i18n.format("DND5E.AbilityScoreL", { ability: CONFIG.DND5E.abilities[key].label });
  }

  // Skills.
  else if ( attr.startsWith("skills.") ) {
    const [, key] = attr.split(".");
    label = game.i18n.format("DND5E.SkillPassiveScore", { skill: CONFIG.DND5E.skills[key].label });
  }

  // Spell slots.
  else if ( attr.startsWith("spells.") ) {
    const [, key] = attr.split(".");
    if ( !/spell\d+/.test(key) ) label = `DND5E.SpellSlots${key.capitalize()}`;
    else {
      const plurals = new Intl.PluralRules(game.i18n.lang, { type: "ordinal" });
      const level = Number(key.slice(5));
      label = game.i18n.format(`DND5E.SpellSlotsN.${plurals.select(level)}`, { n: level });
    }
  }

  // Currency
  else if ( attr.startsWith("currency.") ) {
    const [, key] = attr.split(".");
    label = CONFIG.DND5E.currencies[key]?.label;
  }

  // Attempt to find the attribute in a data model.
  if ( !label ) label = getSchemaLabel(attr, "Actor", actor);

  if ( label ) {
    label = game.i18n.localize(label);
    _attributeLabelCache[type].set(attr, label);
    if ( name ) label = `${name} ${label}`;
  }

  return label;
}

/* -------------------------------------------- */

/**
 * Perform pre-localization on the contents of a SchemaField. Necessary because the `localizeSchema` method
 * on `Localization` is private.
 * @param {SchemaField} schema
 * @param {string[]} prefixes
 */
function localizeSchema(schema, prefixes) {
  Localization.localizeDataModel({ schema }, { prefixes });
}

/* -------------------------------------------- */

/**
 * Split a semi-colon-separated list and clean out any empty entries.
 * @param {string} input
 * @returns {string[]}
 */
function splitSemicolons(input) {
  return input.split(";").map(t => t.trim()).filter(t => t);
}

/* -------------------------------------------- */
/*  Migration                                   */
/* -------------------------------------------- */

/**
 * Synchronize the spells for all Actors in some collection with source data from an Item compendium pack.
 * @param {CompendiumCollection} actorPack      An Actor compendium pack which will be updated
 * @param {CompendiumCollection} spellsPack     An Item compendium pack which provides source data for spells
 * @returns {Promise<void>}
 */
async function synchronizeActorSpells(actorPack, spellsPack) {

  // Load all actors and spells
  const actors = await actorPack.getDocuments();
  const spells = await spellsPack.getDocuments();
  const spellsMap = spells.reduce((obj, item) => {
    obj[item.name] = item;
    return obj;
  }, {});

  // Unlock the pack
  await actorPack.configure({locked: false});

  // Iterate over actors
  SceneNavigation.displayProgressBar({label: "Synchronizing Spell Data", pct: 0});
  for ( const [i, actor] of actors.entries() ) {
    const {toDelete, toCreate} = _synchronizeActorSpells(actor, spellsMap);
    if ( toDelete.length ) await actor.deleteEmbeddedDocuments("Item", toDelete);
    if ( toCreate.length ) await actor.createEmbeddedDocuments("Item", toCreate, {keepId: true});
    console.debug(`${actor.name} | Synchronized ${toCreate.length} spells`);
    SceneNavigation.displayProgressBar({label: actor.name, pct: ((i / actors.length) * 100).toFixed(0)});
  }

  // Re-lock the pack
  await actorPack.configure({locked: true});
  SceneNavigation.displayProgressBar({label: "Synchronizing Spell Data", pct: 100});
}

/* -------------------------------------------- */

/**
 * A helper function to synchronize spell data for a specific Actor.
 * @param {Actor5e} actor
 * @param {Object<string,Item5e>} spellsMap
 * @returns {{toDelete: string[], toCreate: object[]}}
 * @private
 */
function _synchronizeActorSpells(actor, spellsMap) {
  const spells = actor.itemTypes.spell;
  const toDelete = [];
  const toCreate = [];
  if ( !spells.length ) return {toDelete, toCreate};

  for ( const spell of spells ) {
    const source = spellsMap[spell.name];
    if ( !source ) {
      console.warn(`${actor.name} | ${spell.name} | Does not exist in spells compendium pack`);
      continue;
    }

    // Combine source data with the preparation and uses data from the actor
    const spellData = source.toObject();
    const {preparation, uses, save} = spell.toObject().system;
    Object.assign(spellData.system, {preparation, uses});
    spellData.system.save.dc = save.dc;
    foundry.utils.setProperty(spellData, "_stats.compendiumSource", source.uuid);

    // Record spells to be deleted and created
    toDelete.push(spell.id);
    toCreate.push(spellData);
  }
  return {toDelete, toCreate};
}

var utils = /*#__PURE__*/Object.freeze({
  __proto__: null,
  areKeysPressed: areKeysPressed,
  convertLength: convertLength,
  convertTime: convertTime,
  convertWeight: convertWeight,
  defaultUnits: defaultUnits,
  filteredKeys: filteredKeys,
  formatCR: formatCR,
  formatLength: formatLength,
  formatModifier: formatModifier,
  formatNumber: formatNumber,
  formatNumberParts: formatNumberParts,
  formatRange: formatRange,
  formatText: formatText,
  formatTime: formatTime,
  formatVolume: formatVolume,
  formatWeight: formatWeight,
  getHumanReadableAttributeLabel: getHumanReadableAttributeLabel,
  getPluralRules: getPluralRules,
  getSceneTargets: getSceneTargets,
  getTargetDescriptors: getTargetDescriptors,
  indexFromUuid: indexFromUuid,
  isValidDieModifier: isValidDieModifier,
  isValidUnit: isValidUnit,
  linkForUuid: linkForUuid,
  localizeSchema: localizeSchema,
  log: log,
  parseInputDelta: parseInputDelta,
  performPreLocalization: performPreLocalization,
  preLocalize: preLocalize,
  preloadHandlebarsTemplates: preloadHandlebarsTemplates,
  prepareFormulaValue: prepareFormulaValue,
  registerHandlebarsHelpers: registerHandlebarsHelpers,
  replaceFormulaData: replaceFormulaData,
  safePropertyExists: safePropertyExists,
  simplifyBonus: simplifyBonus,
  sortObjectEntries: sortObjectEntries,
  splitSemicolons: splitSemicolons,
  staticID: staticID,
  synchronizeActorSpells: synchronizeActorSpells,
  validators: validators
});

/**
 * @typedef {StringFieldOptions} FormulaFieldOptions
 * @property {boolean} [deterministic=false]  Is this formula not allowed to have dice values?
 */

/**
 * Special case StringField which represents a formula.
 *
 * @param {FormulaFieldOptions} [options={}]  Options which configure the behavior of the field.
 * @property {boolean} deterministic=false    Is this formula not allowed to have dice values?
 */
class FormulaField extends foundry.data.fields.StringField {

  /** @inheritDoc */
  static get _defaults() {
    return foundry.utils.mergeObject(super._defaults, {
      deterministic: false
    });
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _validateType(value) {
    const roll = new Roll(value.replace(/@([a-z.0-9_-]+)/gi, "1"));
    roll.evaluateSync({ strict: false });
    if ( this.options.deterministic && !roll.isDeterministic ) throw new Error(`must not contain dice terms: ${value}`);
    super._validateType(value);
  }

  /* -------------------------------------------- */
  /*  Active Effect Integration                   */
  /* -------------------------------------------- */

  /** @override */
  _castChangeDelta(delta) {
    return this._cast(delta).trim();
  }

  /* -------------------------------------------- */

  /** @override */
  _applyChangeAdd(value, delta, model, change) {
    if ( !value ) return delta;
    const operator = delta.startsWith("-") ? "-" : "+";
    delta = delta.replace(/^[+-]/, "").trim();
    return `${value} ${operator} ${delta}`;
  }

  /* -------------------------------------------- */

  /** @override */
  _applyChangeMultiply(value, delta, model, change) {
    if ( !value ) return delta;
    const terms = new Roll(value).terms;
    if ( terms.length > 1 ) return `(${value}) * ${delta}`;
    return `${value} * ${delta}`;
  }

  /* -------------------------------------------- */

  /** @override */
  _applyChangeUpgrade(value, delta, model, change) {
    if ( !value ) return delta;
    const terms = new Roll(value).terms;
    if ( (terms.length === 1) && (terms[0].fn === "max") ) return current.replace(/\)$/, `, ${delta})`);
    return `max(${value}, ${delta})`;
  }

  /* -------------------------------------------- */

  /** @override */
  _applyChangeDowngrade(value, delta, model, change) {
    if ( !value ) return delta;
    const terms = new Roll(value).terms;
    if ( (terms.length === 1) && (terms[0].fn === "min") ) return current.replace(/\)$/, `, ${delta})`);
    return `min(${value}, ${delta})`;
  }
}

const { ArrayField: ArrayField$p, EmbeddedDataField: EmbeddedDataField$6, SchemaField: SchemaField$W, StringField: StringField$1c } = foundry.data.fields;

/**
 * Field for holding one or more consumption targets.
 */
class ConsumptionTargetsField extends ArrayField$p {
  constructor(options={}) {
    super(new EmbeddedDataField$6(ConsumptionTargetData), options);
  }
}

/**
 * Embedded data model for storing consumption target data and handling consumption.
 *
 * @property {string} type             Type of consumption (e.g. activity uses, item uses, hit die, spell slot).
 * @property {string} target           Target of the consumption depending on the selected type (e.g. item's ID, hit
 *                                     die denomination, spell slot level).
 * @property {string} value            Formula that determines amount consumed or recovered.
 * @property {object} scaling
 * @property {string} scaling.mode     Scaling mode (e.g. no scaling, scale target amount, scale spell level).
 * @property {string} scaling.formula  Specific scaling formula if not automatically calculated from target's value.
 */
class ConsumptionTargetData extends foundry.abstract.DataModel {
  /** @override */
  static defineSchema() {
    return {
      type: new StringField$1c(),
      target: new StringField$1c(),
      value: new FormulaField({ initial: "1" }),
      scaling: new SchemaField$W({
        mode: new StringField$1c(),
        formula: new FormulaField()
      })
    };
  }

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * Activity to which this consumption target belongs.
   * @type {Activity}
   */
  get activity() {
    return this.parent;
  }

  /* -------------------------------------------- */

  /**
   * Actor containing this consumption target, if embedded.
   * @type {Actor5e}
   */
  get actor() {
    return this.activity.actor;
  }

  /* -------------------------------------------- */

  /**
   * Should this consumption only be performed during initiative? This will return `true` if consuming activity or item
   * uses and those uses only recover on "combat" periods.
   * @type {boolean}
   */
  get combatOnly() {
    let recovery;
    switch ( this.type ) {
      case "activityUses":
        recovery = this.activity.uses.recovery;
        break;
      case "itemUses":
        recovery = (this.target ? this.actor?.items.get(this.target) : this.item)?.system.uses.recovery;
        break;
      default: return false;
    }
    if ( !recovery?.length ) return false;
    return recovery.every(r => CONFIG.DND5E.limitedUsePeriods[r.period]?.type === "combat");
  }

  /* -------------------------------------------- */

  /**
   * Item to which this consumption target's activity belongs.
   * @type {Item5e}
   */
  get item() {
    return this.activity.item;
  }

  /* -------------------------------------------- */

  /**
   * List of valid targets within the current context.
   * @type {FormSelectOption[]|null}
   */
  get validTargets() {
    const config = CONFIG.DND5E.activityConsumptionTypes[this.type];
    if ( !config?.validTargets || (!this.item.isEmbedded && (config.targetRequiresEmbedded === true)) ) return null;
    return config.validTargets.call(this);
  }

  /* -------------------------------------------- */
  /*  Consumption                                 */
  /* -------------------------------------------- */

  /**
   * Perform consumption according to the target type.
   * @param {ActivityUseConfiguration} config  Configuration data for the activity usage.
   * @param {ActivityUsageUpdates} updates     Updates to be performed.
   * @throws ConsumptionError
   */
  async consume(config, updates) {
    const typeConfig = CONFIG.DND5E.activityConsumptionTypes[this.type];
    if ( !typeConfig?.consume ) throw new Error("Consumption types must define consumption method.");
    await typeConfig.consume.call(this, config, updates);
  }

  /* -------------------------------------------- */

  /**
   * Prepare consumption updates for "Activity Uses" consumption type.
   * @this {ConsumptionTargetData}
   * @param {ActivityUseConfiguration} config  Configuration data for the activity usage.
   * @param {ActivityUsageUpdates} updates     Updates to be performed.
   * @throws ConsumptionError
   */
  static async consumeActivityUses(config, updates) {
    const result = await this._usesConsumption(config, {
      uses: this.activity.uses,
      type: game.i18n.format("DND5E.CONSUMPTION.Type.ActivityUses.Warning", {
        activity: this.activity.name, item: this.item.name
      }),
      rolls: updates.rolls,
      delta: { item: this.item.id, keyPath: `system.activities.${this.activity.id}.uses.spent` }
    });
    if ( result ) foundry.utils.mergeObject(updates.activity, { "uses.spent": result.spent });
  }

  /* -------------------------------------------- */

  /**
   * Prepare consumption updates for "Attribute" consumption type.
   * @this {ConsumptionTargetData}
   * @param {ActivityUseConfiguration} config  Configuration data for the activity usage.
   * @param {ActivityUsageUpdates} updates     Updates to be performed.
   * @throws ConsumptionError
   */
  static async consumeAttribute(config, updates) {
    const keyPath = `system.${this.target}`;
    const cost = (await this.resolveCost({ config, delta: { keyPath }, rolls: updates.rolls })).total;

    if ( !foundry.utils.hasProperty(this.actor, keyPath) ) throw new ConsumptionError(
      game.i18n.format("DND5E.CONSUMPTION.Warning.MissingAttribute", {
        activity: this.activity.name, attribute: this.target, item: this.item.name
      })
    );
    const current = foundry.utils.getProperty(this.actor, keyPath);

    let warningMessage;
    if ( (cost > 0) && !current ) warningMessage = "DND5E.CONSUMPTION.Warning.None";
    else if ( current < cost ) warningMessage = "DND5E.CONSUMPTION.Warning.NotEnough";
    if ( warningMessage ) throw new ConsumptionError(game.i18n.format(warningMessage, {
      available: formatNumber(current), cost: formatNumber(cost),
      type: game.i18n.format("DND5E.CONSUMPTION.Type.Attribute.Warning", { attribute: this.target })
    }));

    updates.actor[keyPath] = current - cost;
  }

  /* -------------------------------------------- */

  /**
   * Prepare consumption updates for "Hit Dice" consumption type.
   * @this {ConsumptionTargetData}
   * @param {ActivityUseConfiguration} config  Configuration data for the activity usage.
   * @param {ActivityUsageUpdates} updates     Updates to be performed.
   * @throws ConsumptionError
   */
  static async consumeHitDice(config, updates) {
    const cost = (await this.resolveCost({ config, rolls: updates.rolls })).total;

    const denom = !["smallest", "largest"].includes(this.target) ? this.target : false;
    const validClasses = Object.values(this.actor.classes).filter(cls => {
      return !denom || (cls.system.hd.denomination === denom);
    });
    const total = validClasses.reduce((count, cls) => count + cls.system.hd.value, 0);

    if ( !denom ) validClasses.sort((lhs, rhs) => {
      const sort = lhs.system.hd.denomination.localeCompare(rhs.system.hd.denomination, "en", { numeric: true });
      return (this.target === "smallest") ? sort : sort * -1;
    });

    let warningMessage;
    if ( !validClasses.length ) warningMessage = "DND5E.CONSUMPTION.Warning.MissingHitDice";
    else if ( (cost > 0) && !total ) warningMessage = "DND5E.CONSUMPTION.Warning.None";
    else if ( total < cost ) warningMessage = "DND5E.CONSUMPTION.Warning.NotEnough";
    if ( warningMessage ) {
      const denomination = !["smallest", "largest"].includes(this.target) ? this.target : "";
      throw new ConsumptionError(game.i18n.format(warningMessage, {
        available: formatNumber(total), cost: formatNumber(cost), denomination,
        type: game.i18n.format("DND5E.CONSUMPTION.Type.HitDice.Warning", { denomination })
      }));
    }

    let toConsume = cost;
    for ( const cls of validClasses ) {
      const available = toConsume > 0 ? cls.system.hd.value : 0;
      const delta = toConsume > 0 ? Math.min(toConsume, available) : Math.max(toConsume, available);
      const itemUpdate = { "system.hd.spent": cls.system.hd.spent + delta };
      if ( delta !== 0 ) {
        const itemIndex = updates.item.findIndex(i => i._id === cls.id);
        if ( itemIndex === -1 ) updates.item.push({ _id: cls.id, ...itemUpdate });
        else foundry.utils.mergeObject(updates.item[itemIndex], itemUpdate);
        toConsume -= delta;
        if ( toConsume === 0 ) break;
      }
    }
  }

  /* -------------------------------------------- */

  /**
   * Prepare consumption updates for "Item Uses" consumption type.
   * @this {ConsumptionTargetData}
   * @param {ActivityUseConfiguration} config  Configuration data for the activity usage.
   * @param {ActivityUsageUpdates} updates     Updates to be performed.
   * @throws ConsumptionError
   */
  static async consumeItemUses(config, updates) {
    const item = this.target ? this.actor.items.get(this.target) : this.item;
    if ( !item ) throw new ConsumptionError(game.i18n.format("DND5E.CONSUMPTION.Warning.MissingItem", {
      activity: this.activity.name, item: this.item.name
    }));

    const result = await this._usesConsumption(config, {
      uses: item.system.uses,
      type: game.i18n.format("DND5E.CONSUMPTION.Type.ItemUses.Warning", { name: this.item.name }),
      rolls: updates.rolls,
      delta: { item: item.id, keyPath: "system.uses.spent" }
    });
    if ( !result ) return;

    const itemUpdate = {};
    if ( item.system.uses.autoDestroy && (result.spent === item.system.uses.max) ) {
      const newQuantity = item.system.quantity - 1;
      if ( newQuantity === 0 ) {
        updates.delete.push(item.id);
        return;
      } else {
        itemUpdate["system.uses.spent"] = 0;
        itemUpdate["system.quantity"] = newQuantity;
      }
    } else {
      itemUpdate["system.uses.spent"] = result.spent;
    }

    const itemIndex = updates.item.findIndex(i => i._id === item.id);
    if ( itemIndex === -1 ) updates.item.push({ _id: item.id, ...itemUpdate });
    else foundry.utils.mergeObject(updates.item[itemIndex], itemUpdate);
  }

  /* -------------------------------------------- */

  /**
   * Prepare consumption updates for "Material" consumption type.
   * @this {ConsumptionTargetData}
   * @param {ActivityUseConfiguration} config  Configuration data for the activity usage.
   * @param {ActivityUsageUpdates} updates     Updates to be performed.
   * @throws ConsumptionError
   */
  static async consumeMaterial(config, updates) {
    const item = this.target ? this.actor.items.get(this.target) : this.item;
    if ( !item ) throw new ConsumptionError(game.i18n.format("DND5E.CONSUMPTION.Warning.MissingItem", {
      activity: this.activity.name, item: this.item.name
    }));

    const delta = { item: item.id, keyPath: "system.quantity" };
    const cost = (await this.resolveCost({ config, delta, rolls: updates.rolls })).total;

    let warningMessage;
    if ( cost > 0 && !item.system.quantity ) warningMessage = "DND5E.CONSUMPTION.Warning.None";
    else if ( cost > item.system.quantity ) warningMessage = "DND5E.CONSUMPTION.Warning.NotEnough";
    if ( warningMessage ) throw new ConsumptionError(game.i18n.format(warningMessage, {
      available: formatNumber(item.system.quantity), cost: formatNumber(cost),
      type: game.i18n.format("DND5E.CONSUMPTION.Type.Material.Warning", { name: item.name })
    }));

    const newQuantity = item.system.quantity - cost;
    if ( (newQuantity === 0) && item.system.uses?.autoDestroy ) {
      updates.delete.push(item.id);
    } else {
      const itemUpdate = { "system.quantity": newQuantity };
      const itemIndex = updates.item.findIndex(i => i._id === item.id);
      if ( itemIndex === -1 ) updates.item.push({ _id: item.id, ...itemUpdate });
      else foundry.utils.mergeObject(updates.item[itemIndex], itemUpdate);
    }
  }

  /* -------------------------------------------- */

  /**
   * Prepare consumption updates for "Spell Slots" consumption type.
   * @this {ConsumptionTargetData}
   * @param {ActivityUseConfiguration} config  Configuration data for the activity usage.
   * @param {ActivityUsageUpdates} updates     Updates to be performed.
   * @throws ConsumptionError
   */
  static async consumeSpellSlots(config, updates) {
    const levelNumber = Math.clamp(
      this.resolveLevel({ config, rolls: updates.rolls }), 1, Object.keys(CONFIG.DND5E.spellLevels).length - 1
    );
    const keyPath = `system.spells.spell${levelNumber}.value`;
    const cost = (await this.resolveCost({ config, delta: { keyPath }, rolls: updates.rolls })).total;

    // Check to see if enough slots are available at the specified level
    const levelData = this.actor.system.spells?.[`spell${levelNumber}`];
    const newValue = (levelData?.value ?? 0) - cost;
    let warningMessage;
    if ( !levelData?.max ) warningMessage = "DND5E.CONSUMPTION.Warning.MissingSpellSlot";
    else if ( (cost > 0) && !levelData.value ) warningMessage = "DND5E.CONSUMPTION.Warning.None";
    else if ( newValue < 0 ) warningMessage = "DND5E.CONSUMPTION.Warning.NotEnough";
    if ( warningMessage ) {
      const level = CONFIG.DND5E.spellLevels[levelNumber];
      const type = game.i18n.format("DND5E.CONSUMPTION.Type.SpellSlots.Warning", { level });
      throw new ConsumptionError(game.i18n.format(warningMessage, {
        type, level, cost: formatNumber(cost), available: formatNumber(levelData.value)
      }));
    }

    updates.actor[keyPath] = Math.max(0, newValue);
  }

  /* -------------------------------------------- */

  /**
   * Calculate updates to activity or item uses.
   * @param {ActivityUseConfiguration} config  Configuration data for the activity usage.
   * @param {object} options
   * @param {UsesData} options.uses            Uses data to consume.
   * @param {string} options.type              Type label to be used in warning messages.
   * @param {BasicRoll[]} options.rolls        Rolls performed as part of the usages.
   * @param {object} [options.delta]           Delta information stored in roll options.
   * @returns {{ spent: number, quantity: number }|null}
   * @internal
   */
  async _usesConsumption(config, { uses, type, rolls, delta }) {
    const cost = (await this.resolveCost({ config, delta, rolls })).total;

    let warningMessage;
    if ( cost > 0 && !uses.value ) warningMessage = "DND5E.CONSUMPTION.Warning.None";
    else if ( cost > uses.value ) warningMessage = "DND5E.CONSUMPTION.Warning.NotEnough";
    if ( warningMessage ) throw new ConsumptionError(
      game.i18n.format(warningMessage, { type, cost: formatNumber(cost), available: formatNumber(uses.value) })
    );

    return { spent: uses.spent + cost };
  }

  /* -------------------------------------------- */
  /*  Consumption Hints                           */
  /* -------------------------------------------- */

  /**
   * Create label and hint text indicating how much of this resource will be consumed/recovered.
   * @param {ActivityUseConfiguration} config  Configuration data for the activity usage.
   * @param {object} [options={}]
   * @param {boolean} [options.consumed]       Is this consumption currently set to be consumed?
   * @returns {ConsumptionLabels}
   */
  getConsumptionLabels(config, options={}) {
    const typeConfig = CONFIG.DND5E.activityConsumptionTypes[this.type];
    if ( !typeConfig?.consumptionLabels ) return "";
    return typeConfig.consumptionLabels.call(this, config, options);
  }

  /* -------------------------------------------- */

  /**
   * Create hint text indicating how much of this resource will be consumed/recovered.
   * @this {ConsumptionTargetData}
   * @param {ActivityUseConfiguration} config  Configuration data for the activity usage.
   * @param {object} [options={}]
   * @param {boolean} [options.consumed]       Is this consumption currently set to be consumed?
   * @returns {ConsumptionLabels}
   */
  static consumptionLabelsActivityUses(config, { consumed }={}) {
    const { cost, simplifiedCost, increaseKey, pluralRule } = this._resolveHintCost(config);
    const uses = this.activity.uses;
    const usesPluralRule = new Intl.PluralRules(game.i18n.lang).select(uses.value);
    return {
      label: game.i18n.localize(`DND5E.CONSUMPTION.Type.ActivityUses.Prompt${increaseKey}`),
      hint: game.i18n.format(
        `DND5E.CONSUMPTION.Type.ActivityUses.PromptHint${increaseKey}`,
        {
          cost,
          use: game.i18n.localize(`DND5E.CONSUMPTION.Type.Use.${pluralRule}`),
          available: formatNumber(uses.value),
          availableUse: game.i18n.localize(`DND5E.CONSUMPTION.Type.Use.${usesPluralRule}`)
        }
      ),
      warn: simplifiedCost > uses.value
    };
  }

  /* -------------------------------------------- */

  /**
   * Create hint text indicating how much of this resource will be consumed/recovered.
   * @this {ConsumptionTargetData}
   * @param {ActivityUseConfiguration} config  Configuration data for the activity usage.
   * @param {object} [options={}]
   * @param {boolean} [options.consumed]       Is this consumption currently set to be consumed?
   * @returns {ConsumptionLabels}
   */
  static consumptionLabelsAttribute(config, { consumed }={}) {
    const { cost, simplifiedCost, increaseKey } = this._resolveHintCost(config);
    const current = foundry.utils.getProperty(this.actor.system, this.target);
    return {
      label: game.i18n.localize(`DND5E.CONSUMPTION.Type.Attribute.Prompt${increaseKey}`),
      hint: game.i18n.format(
        `DND5E.CONSUMPTION.Type.Attribute.PromptHint${increaseKey}`,
        { cost, attribute: this.target, current: formatNumber(current) }
      ),
      warn: simplifiedCost > current
    };
  }

  /* -------------------------------------------- */

  /**
   * Create hint text indicating how much of this resource will be consumed/recovered.
   * @this {ConsumptionTargetData}
   * @param {ActivityUseConfiguration} config  Configuration data for the activity usage.
   * @param {object} [options={}]
   * @param {boolean} [options.consumed]       Is this consumption currently set to be consumed?
   * @returns {ConsumptionLabels}
   */
  static consumptionLabelsHitDice(config, { consumed }={}) {
    const { cost, simplifiedCost, increaseKey, pluralRule } = this._resolveHintCost(config);
    let denomination;
    if ( this.target === "smallest" ) denomination = game.i18n.localize("DND5E.ConsumeHitDiceSmallest");
    else if ( this.target === "largest" ) denomination = game.i18n.localize("DND5E.ConsumeHitDiceLargest");
    else denomination = this.target;
    const available = (["smallest", "largest"].includes(this.target)
      ? this.actor.system.attributes?.hd?.value : this.actor.system.attributes?.hd?.bySize?.[this.target]) ?? 0;
    return {
      label: game.i18n.localize(`DND5E.CONSUMPTION.Type.HitDice.Prompt${increaseKey}`),
      hint: game.i18n.format(
        `DND5E.CONSUMPTION.Type.HitDice.PromptHint${increaseKey}`,
        {
          cost, denomination: denomination.toLowerCase(),
          die: game.i18n.localize(`DND5E.CONSUMPTION.Type.HitDie.${pluralRule}`),
          available: formatNumber(available)
        }
      ),
      warn: simplifiedCost > available
    };
  }

  /* -------------------------------------------- */

  /**
   * Create hint text indicating how much of this resource will be consumed/recovered.
   * @this {ConsumptionTargetData}
   * @param {ActivityUseConfiguration} config  Configuration data for the activity usage.
   * @param {object} [options={}]
   * @param {boolean} [options.consumed]       Is this consumption currently set to be consumed?
   * @returns {ConsumptionLabels}
   */
  static consumptionLabelsItemUses(config, { consumed }={}) {
    const { cost, simplifiedCost, increaseKey, pluralRule } = this._resolveHintCost(config);
    const item = this.actor.items.get(this.target);
    const itemName = item ? item.name : game.i18n.localize("DND5E.CONSUMPTION.Target.ThisItem").toLowerCase();
    const uses = (item ?? this.item).system.uses;
    const usesPluralRule = new Intl.PluralRules(game.i18n.lang).select(uses.value);

    const notes = [];
    let warn = false;
    if ( simplifiedCost > uses.value ) warn = true;
    else if ( (simplifiedCost > 0) && (uses.value - simplifiedCost === 0) && uses.autoDestroy ) notes.push({
      type: "warn",
      message: game.i18n.format("DND5E.CONSUMPTION.Warning.WillDestroy", { item: itemName })
    });

    return {
      label: game.i18n.localize(`DND5E.CONSUMPTION.Type.ItemUses.Prompt${increaseKey}`),
      hint: game.i18n.format(
        `DND5E.CONSUMPTION.Type.ItemUses.PromptHint${increaseKey}`,
        {
          cost,
          use: game.i18n.localize(`DND5E.CONSUMPTION.Type.Use.${pluralRule}`),
          available: formatNumber(uses.value),
          availableUse: game.i18n.localize(`DND5E.CONSUMPTION.Type.Use.${usesPluralRule}`),
          item: item ? `<em>${itemName}</em>` : itemName
        }
      ),
      notes: consumed ? notes : null,
      warn
    };
  }

  /* -------------------------------------------- */

  /**
   * Create hint text indicating how much of this resource will be consumed/recovered.
   * @this {ConsumptionTargetData}
   * @param {ActivityUseConfiguration} config  Configuration data for the activity usage.
   * @param {object} [options={}]
   * @param {boolean} [options.consumed]       Is this consumption currently set to be consumed?
   * @returns {ConsumptionLabels}
   */
  static consumptionLabelsMaterial(config, { consumed }={}) {
    const { cost, simplifiedCost, increaseKey } = this._resolveHintCost(config);
    const item = this.actor.items.get(this.target);
    const quantity = (item ?? this.item).system.quantity;
    return {
      label: game.i18n.localize(`DND5E.CONSUMPTION.Type.Material.Prompt${increaseKey}`),
      hint: game.i18n.format(
        `DND5E.CONSUMPTION.Type.Material.PromptHint${increaseKey}`,
        {
          cost,
          item: item ? `<em>${item.name}</em>` : game.i18n.localize("DND5E.CONSUMPTION.Target.ThisItem").toLowerCase(),
          quantity: formatNumber(quantity)
        }
      ),
      warn: simplifiedCost > quantity
    };
  }

  /* -------------------------------------------- */

  /**
   * Create hint text indicating how much of this resource will be consumed/recovered.
   * @this {ConsumptionTargetData}
   * @param {ActivityUseConfiguration} config  Configuration data for the activity usage.
   * @param {object} [options={}]
   * @param {boolean} [options.consumed]       Is this consumption currently set to be consumed?
   * @returns {ConsumptionLabels}
   */
  static consumptionLabelsSpellSlots(config, { consumed }={}) {
    const { cost, simplifiedCost, increaseKey, pluralRule } = this._resolveHintCost(config);
    const levelNumber = Math.clamp(this.resolveLevel({ config }), 1, Object.keys(CONFIG.DND5E.spellLevels).length - 1);
    const level = CONFIG.DND5E.spellLevels[levelNumber].toLowerCase();
    const available = this.actor.system.spells?.[`spell${levelNumber}`]?.value ?? 0;
    return {
      label: game.i18n.localize(`DND5E.CONSUMPTION.Type.SpellSlots.Prompt${increaseKey}`),
      hint: game.i18n.format(
        `DND5E.CONSUMPTION.Type.SpellSlots.PromptHint${increaseKey}`,
        {
          cost,
          slot: game.i18n.format(`DND5E.CONSUMPTION.Type.SpellSlot.${pluralRule}`, { level }),
          available: formatNumber(available)
        }
      ),
      warn: simplifiedCost > available
    };
  }

  /* -------------------------------------------- */

  /**
   * Resolve the cost for the consumption hint.
   * @param {ActivityUseConfiguration} config  Configuration data for the activity usage.
   * @returns {{ cost: string, simplifiedCost: number, increaseKey: string, pluralRule: string }}
   * @internal
   */
  _resolveHintCost(config) {
    const costRoll = this.resolveCost({ config, evaluate: false });
    let cost = costRoll.isDeterministic
      ? String(costRoll.evaluateSync().total)
      : simplifyRollFormula(costRoll.formula);
    const simplifiedCost = simplifyBonus(cost);
    const isNegative = cost.startsWith("-");
    if ( isNegative ) cost = cost.replace("-", "");
    let pluralRule;
    if ( costRoll.isDeterministic ) pluralRule = new Intl.PluralRules(game.i18n.lang).select(Number(cost));
    else pluralRule = "other";
    return { cost, simplifiedCost, increaseKey: isNegative ? "Increase" : "Decrease", pluralRule };
  }

  /* -------------------------------------------- */
  /*  Valid Targets                               */
  /* -------------------------------------------- */

  /**
   * Generate a list of targets for the "Attribute" consumption type.
   * @this {ConsumptionTargetData}
   * @returns {FormSelectOption[]}
   */
  static validAttributeTargets() {
    if ( !this.actor ) return [];
    return TokenDocument.implementation.getConsumedAttributes(this.actor.type).map(attr => {
      let group;
      if ( attr.startsWith("abilities.") ) group = game.i18n.localize("DND5E.AbilityScorePl");
      else if ( attr.startsWith("currency.") ) group = game.i18n.localize("DND5E.Currency");
      else if ( attr.startsWith("spells.") ) group = game.i18n.localize("DND5E.CONSUMPTION.Type.SpellSlots.Label");
      else if ( attr.startsWith("attributes.movement.") ) group = game.i18n.localize("DND5E.Speed");
      else if ( attr.startsWith("attributes.senses.") ) group = game.i18n.localize("DND5E.Senses");
      else if ( attr.startsWith("resources.") ) group = game.i18n.localize("DND5E.Resources");
      return { group, value: attr, label: getHumanReadableAttributeLabel(attr, { actor: this.actor }) || attr };
    });
  }

  /* -------------------------------------------- */

  /**
   * Generate a list of targets for the "Hit Dice" consumption type.
   * @this {ConsumptionTargetData}
   * @returns {FormSelectOption[]}
   */
  static validHitDiceTargets() {
    return [
      { value: "smallest", label: game.i18n.localize("DND5E.ConsumeHitDiceSmallest") },
      ...CONFIG.DND5E.hitDieTypes.map(d => ({ value: d, label: d })),
      { value: "largest", label: game.i18n.localize("DND5E.ConsumeHitDiceLargest") }
    ];
  }

  /* -------------------------------------------- */

  /**
   * Generate a list of targets for the "Item Uses" consumption type.
   * @this {ConsumptionTargetData}
   * @returns {FormSelectOption[]}
   */
  static validItemUsesTargets() {
    const makeLabel = (name, item) => {
      let label;
      const uses = item.system.uses;
      if ( uses.max && (uses.recovery?.length === 1) && (uses.recovery[0].type === "recoverAll")
        && (uses.recovery[0].period !== "recharge") ) {
        const per = CONFIG.DND5E.limitedUsePeriods[uses.recovery[0].period]?.abbreviation;
        label = game.i18n.format("DND5E.AbilityUseConsumableLabel", { max: uses.max, per });
      }
      else label = game.i18n.format("DND5E.AbilityUseChargesLabel", { value: uses.value });
      return `${name} (${label})`;
    };
    return [
      { value: "", label: makeLabel(game.i18n.localize("DND5E.CONSUMPTION.Target.ThisItem"), this.item) },
      { rule: true },
      ...(this.actor?.items ?? [])
        .filter(i => i.system.uses?.max && (i !== this.item))
        .map(i => ({ value: i.id, label: makeLabel(i.name, i) }))
    ];
  }

  /* -------------------------------------------- */

  /**
   * Generate a list of targets for the "Material" consumption type.
   * @this {ConsumptionTargetData}
   * @returns {FormSelectOption[]}
   */
  static validMaterialTargets() {
    return (this.actor?.items ?? [])
      .filter(i => ["consumable", "loot"].includes(i.type) && !i.system.activities?.size)
      .map(i => ({ value: i.id, label: `${i.name} (${formatNumber(i.system.quantity)})` }));
  }

  /* -------------------------------------------- */

  /**
   * Generate a list of targets for the "Spell Slots" consumption type.
   * @this {ConsumptionTargetData}
   * @returns {FormSelectOption[]}
   */
  static validSpellSlotsTargets() {
    return Object.entries(CONFIG.DND5E.spellLevels).reduce((arr, [value, label]) => {
      if ( value !== "0" ) arr.push({ value, label });
      return arr;
    }, []);
  }

  /* -------------------------------------------- */
  /*  Helpers                                     */
  /* -------------------------------------------- */

  /**
   * Resolve the amount to consume, taking scaling into account.
   * @param {object} [options={}]
   * @param {ActivityUseConfiguration} [options.config]  Usage configuration.
   * @param {boolean} [options.evaluate=true]            Should the cost roll be evaluated?
   * @param {BasicRoll[]} [options.rolls]                Rolls performed as part of the usages.
   * @returns {Promise<BasicRoll>|BasicRoll}             Returns Promise if evaluate is `true`.
   */
  resolveCost({ config={}, ...options }={}) {
    return this._resolveScaledRoll(this.value, this.scaling.mode === "amount" ? config.scaling ?? 0 : 0, options);
  }

  /* -------------------------------------------- */

  /**
   * Resolve the spell level to consume, taking scaling into account.
   * @param {object} [options={}]
   * @param {ActivityUseConfiguration} [options.config]  Usage configuration.
   * @param {BasicRoll[]} [options.rolls]                Rolls performed as part of the usages.
   * @returns {number}
   */
  resolveLevel({ config={}, ...options }={}) {
    const roll = this._resolveScaledRoll(
      this.target, this.scaling.mode === "level" ? config.scaling ?? 0 : 0, { ...options, evaluate: false }
    );
    roll.evaluateSync();
    return roll.total;
  }

  /* -------------------------------------------- */

  /**
   * Resolve a scaling consumption value formula.
   * @param {string} formula                   Formula for the initial value.
   * @param {number} scaling                   Amount to scale the formula.
   * @param {object} [options={}]
   * @param {object} [options.delta]           Delta information stored in roll options.
   * @param {boolean} [options.evaluate=true]  Should the slot roll be evaluated?
   * @param {BasicRoll[]} [options.rolls]      Rolls performed as part of the usages.
   * @returns {Promise<BasicRoll>|BasicRoll}
   * @internal
   */
  _resolveScaledRoll(formula, scaling, { delta, evaluate=true, rolls }={}) {
    const rollData = this.activity.getRollData();
    const roll = new CONFIG.Dice.BasicRoll(formula, rollData, { delta });

    if ( scaling ) {
      // If a scaling formula is provided, multiply it and add to the end of the initial formula
      if ( this.scaling.formula ) {
        const scalingRoll = new Roll(this.scaling.formula, rollData);
        scalingRoll.alter(scaling, undefined, { multiplyNumeric: true });
        roll.terms.push(new foundry.dice.terms.OperatorTerm({ operator: "+" }), ...scalingRoll.terms);
      }

      // Otherwise increase the number of dice and the numeric term for each scaling step
      else roll.terms = roll.terms.map(term => {
        if ( term instanceof foundry.dice.terms.DiceTerm ) return term.alter(undefined, scaling);
        else if ( term instanceof foundry.dice.terms.NumericTerm ) term.number += scaling;
        return term;
      });

      roll.resetFormula();
    }

    if ( evaluate ) return roll.evaluate().then(roll => {
      if ( rolls && !roll.isDeterministic ) rolls.push(roll);
      return roll;
    });
    if ( rolls && !roll.isDeterministic ) rolls.push(roll);
    return roll;
  }
}

/**
 * Error to throw when consumption cannot be achieved.
 */
class ConsumptionError extends Error {
  constructor(...args) {
    super(...args);
    this.name = "ConsumptionError";
  }
}

const { HandlebarsApplicationMixin } = foundry.applications.api;

/**
 * @typedef {ApplicationContainerParts}
 * @property {object} [container]
 * @property {string} [container.id]         ID of the container. Containers with the same ID will be grouped together.
 * @property {string[]} [container.classes]  Classes to add to the container.
 */

/**
 * Mixin method for ApplicationV2-based 5e applications.
 * @template {ApplicationV2} T
 * @param {typeof T} Base   Application class being extended.
 * @returns {typeof BaseApplication5e}
 * @mixin
 */
function ApplicationV2Mixin(Base) {
  class BaseApplication5e extends HandlebarsApplicationMixin(Base) {
    /** @override */
    static DEFAULT_OPTIONS = {
      classes: ["dnd5e2"],
      window: {
        subtitle: ""
      }
    };

    /* -------------------------------------------- */

    /**
     * @type {Record<string, HandlebarsTemplatePart & ApplicationContainerParts>}
     */
    static PARTS = {};

    /* -------------------------------------------- */
    /*  Properties                                  */
    /* -------------------------------------------- */

    /**
     * A reference to the window subtitle.
     * @type {string}
     */
    get subtitle() {
      return game.i18n.localize(this.options.window.subtitle ?? "");
    }

    /* -------------------------------------------- */
    /*  Initialization                              */
    /* -------------------------------------------- */

    /** @inheritDoc */
    _initializeApplicationOptions(options) {
      const applicationOptions = super._initializeApplicationOptions(options);
      // Fix focus bug caused by the use of UUIDs in application IDs
      // TODO: Remove once https://github.com/foundryvtt/foundryvtt/issues/11742 is fixed
      applicationOptions.uniqueId = applicationOptions.uniqueId.replace(/\./g, "-");
      return applicationOptions;
    }

    /* -------------------------------------------- */
    /*  Rendering                                   */
    /* -------------------------------------------- */

    /** @inheritDoc */
    _configureRenderOptions(options) {
      super._configureRenderOptions(options);
      if ( options.isFirstRender && this.hasFrame ) {
        options.window ||= {};
        options.window.subtitle ||= this.subtitle;
      }
    }

    /* -------------------------------------------- */

    /** @inheritDoc */
    _onFirstRender(context, options) {
      super._onFirstRender(context, options);
      const containers = {};
      for ( const [part, config] of Object.entries(this.constructor.PARTS) ) {
        if ( !config.container?.id ) continue;
        const element = this.element.querySelector(`[data-application-part="${part}"]`);
        if ( !element ) continue;
        if ( !containers[config.container.id] ) {
          const div = document.createElement("div");
          div.dataset.containerId = config.container.id;
          div.classList.add(...config.container.classes ?? []);
          containers[config.container.id] = div;
          element.replaceWith(div);
        }
        containers[config.container.id].append(element);
      }
    }

    /* -------------------------------------------- */

    /** @inheritDoc */
    async _prepareContext(options) {
      const context = await super._prepareContext(options);
      context.CONFIG = CONFIG.DND5E;
      context.inputs = { ...foundry.applications.fields, ...dnd5e.applications.fields };
      return context;
    }

    /* -------------------------------------------- */

    /** @inheritDoc */
    async _preparePartContext(partId, context, options) {
      return { ...await super._preparePartContext(partId, context, options) };
    }

    /* -------------------------------------------- */

    /** @inheritDoc */
    async _renderFrame(options) {
      const frame = await super._renderFrame(options);

      // Subtitles
      const subtitle = document.createElement("h2");
      subtitle.classList.add("window-subtitle");
      frame?.querySelector(".window-title")?.insertAdjacentElement("afterend", subtitle);

      // Icon
      if ( (options.window?.icon ?? "").includes(".") ) {
        const icon = frame.querySelector(".window-icon");
        const newIcon = document.createElement(options.window.icon?.endsWith(".svg") ? "dnd5e-icon" : "img");
        newIcon.classList.add("window-icon");
        newIcon.src = options.window.icon;
        icon.replaceWith(newIcon);
      }

      return frame;
    }

    /* -------------------------------------------- */

    /** @inheritDoc */
    _updateFrame(options) {
      super._updateFrame(options);
      if ( options.window && ("subtitle" in options.window) ) {
        this.element.querySelector(".window-header > .window-subtitle").innerText = options.window.subtitle;
      }
    }

    /* -------------------------------------------- */

    /** @inheritDoc */
    _onRender(context, options) {
      super._onRender(context, options);

      // Allow multi-select tags to be removed when the whole tag is clicked.
      this.element.querySelectorAll("multi-select").forEach(select => {
        if ( select.disabled ) return;
        select.querySelectorAll(".tag").forEach(tag => {
          tag.classList.add("remove");
          tag.querySelector(":scope > span")?.classList.add("remove");
        });
      });

      // Add special styling for label-top hints.
      this.element.querySelectorAll(".label-top > p.hint").forEach(hint => {
        const label = hint.parentElement.querySelector(":scope > label");
        if ( !label ) return;
        hint.ariaLabel = hint.innerText;
        hint.dataset.tooltip = hint.innerHTML;
        hint.innerHTML = "";
        label.insertAdjacentElement("beforeend", hint);
      });
    }

    /* -------------------------------------------- */

    /**
     * Disable form fields that aren't marked with the `interface-only` class.
     */
    _disableFields() {
      const selector = `.window-content :is(${[
        "INPUT", "SELECT", "TEXTAREA", "BUTTON", "DND5E-CHECKBOX", "COLOR-PICKER", "DOCUMENT-TAGS",
        "FILE-PICKER", "HUE-SLIDER", "MULTI-SELECT", "PROSE-MIRROR", "RANGE-PICKER", "STRING-TAGS"
      ].join(", ")}):not(.interface-only)`;
      for ( const element of this.element.querySelectorAll(selector) ) {
        if ( element.tagName === "TEXTAREA" ) element.readOnly = true;
        else element.disabled = true;
      }
    }
  }
  return BaseApplication5e;
}

const { ApplicationV2 } = foundry.applications.api;

/**
 * Base application from which all system applications should be based.
 */
class Application5e extends ApplicationV2Mixin(ApplicationV2) {}

/**
 * Default sheet for activities.
 */
class PseudoDocumentSheet extends Application5e {
  constructor(options={}) {
    super(options);
    this.#documentId = options.document.id;
    this.#documentType = options.document.metadata.name;
    this.#item = options.document.item;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static DEFAULT_OPTIONS = {
    classes: ["pseudo-document", "sheet", "standard-form"],
    tag: "form",
    document: null,
    viewPermission: CONST.DOCUMENT_OWNERSHIP_LEVELS.LIMITED,
    editPermission: CONST.DOCUMENT_OWNERSHIP_LEVELS.OWNER,
    actions: {
      copyUuid: { handler: PseudoDocumentSheet.#onCopyUuid, buttons: [0, 2] }
    },
    form: {
      handler: PseudoDocumentSheet.#onSubmitForm,
      submitOnChange: true
    }
  };

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * The PseudoDocument associated with this application.
   * @type {PseudoDocument}
   */
  get document() {
    return this.item.getEmbeddedDocument(this.#documentType, this.#documentId);
  }

  /**
   * ID of this PseudoDocument on the parent item.
   * @type {string}
   */
  #documentId;

  /**
   * Collection representing this PseudoDocument.
   * @type {string}
   */
  #documentType;

  /* -------------------------------------------- */

  /**
   * Is this PseudoDocument sheet visible to the current user?
   * @type {boolean}
   */
  get isVisible() {
    return this.item.testUserPermission(game.user, this.options.viewPermission);
  }

  /* -------------------------------------------- */

  /**
   * Is this PseudoDocument sheet editable by the current User?
   * This is governed by the editPermission threshold configured for the class.
   * @type {boolean}
   */
  get isEditable() {
    if ( game.packs.get(this.item.pack)?.locked ) return false;
    return this.item.testUserPermission(game.user, this.options.editPermission);
  }

  /* -------------------------------------------- */

  /**
   * Parent item to which this PseudoDocument belongs.
   * @type {Item5e}
   */
  #item;

  get item() {
    return this.#item;
  }

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareContext(options) {
    return {
      ...await super._prepareContext(options),
      document: this.document,
      editable: this.isEditable,
      options: this.options
    };
  }

  /* -------------------------------------------- */
  /*  Life-Cycle Handlers                         */
  /* -------------------------------------------- */

  /** @override */
  _canRender(options) {
    if ( !this.isVisible ) throw new Error(game.i18n.format("SHEETS.DocumentSheetPrivate", {
      type: game.i18n.localize(this.document.metadata.label)
    }));
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _onFirstRender(context, options) {
    super._onFirstRender(context, options);
    this.document.constructor._registerApp(this.document, this);
    this.item.apps[this.id] = this;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _onRender(context, options) {
    super._onRender(context, options);
    if ( !this.isEditable ) this._disableFields();
  }

  /* -------------------------------------------- */

  /** @override */
  _onClose(_options) {
    this.document?.constructor._unregisterApp(this.document, this);
    delete this.item.apps[this.id];
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _renderFrame(options) {
    const frame = await super._renderFrame(options);
    frame.autocomplete = "off";

    // Add document ID copy
    const copyLabel = game.i18n.localize("SHEETS.CopyUuid");
    const copyId = `<button type="button" class="header-control fa-solid fa-passport icon" data-action="copyUuid"
                            data-tooltip="${copyLabel}" aria-label="${copyLabel}"></button>`;
    this.window.close.insertAdjacentHTML("beforebegin", copyId);

    return frame;
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /**
   * Handle click events to copy the UUID of this document to clipboard.
   * @this {PseudoDocumentSheet}
   * @param {Event} event         Triggering click event.
   * @param {HTMLElement} target  Button that was clicked.
   * @this {PseudoDocumentSheet}
   */
  static #onCopyUuid(event, target) {
    event.preventDefault();
    event.stopPropagation();
    if ( event.detail > 1 ) return;
    const id = event.button === 2 ? this.document.id : this.document.uuid;
    const type = event.button === 2 ? "id" : "uuid";
    const label = game.i18n.localize(this.document.metadata.label);
    game.clipboard.copyPlainText(id);
    ui.notifications.info(game.i18n.format("DOCUMENT.IdCopiedClipboard", { label, type, id }));
  }

  /* -------------------------------------------- */
  /*  Form Handling                               */
  /* -------------------------------------------- */

  /**
   * Handle form submission.
   * @param {SubmitEvent} event          Triggering submit event.
   * @param {HTMLFormElement} form       The form that was submitted.
   * @param {FormDataExtended} formData  Data from the submitted form.
   */
  static async #onSubmitForm(event, form, formData) {
    const submitData = this._prepareSubmitData(event, formData);
    await this._processSubmitData(event, submitData);
  }

  /* -------------------------------------------- */

  /**
   * Perform any pre-processing of the form data to prepare it for updating.
   * @param {SubmitEvent} event          Triggering submit event.
   * @param {FormDataExtended} formData  Data from the submitted form.
   * @returns {object}
   */
  _prepareSubmitData(event, formData) {
    const submitData = foundry.utils.expandObject(formData.object);
    // Workaround for https://github.com/foundryvtt/foundryvtt/issues/11610
    this.element.querySelectorAll("fieldset legend :is(input, select, dnd5e-checkbox)").forEach(input => {
      foundry.utils.setProperty(submitData, input.name, input.value);
    });
    return submitData;
  }

  /* -------------------------------------------- */

  /**
   * Handle updating the PseudoDocument based on processed submit data.
   * @param {SubmitEvent} event  Triggering submit event.
   * @param {object} submitData  Prepared object for updating.
   */
  async _processSubmitData(event, submitData) {
    await this.document.update(submitData);
  }

  /* -------------------------------------------- */

  /**
   * Programmatically submit a PseudoDocumentSheet instance, providing additional data to be merged with form data.
   * @param {object} options
   * @param {object} [options.updateData]  Additional data merged with processed form data.
   */
  async submit({ updateData={} }={}) {
    if ( !this.options.form?.handler ) throw new Error(
      `The ${this.constructor.name} PseudoDocumentSheet does not support a single top-level form element.`
    );
    const event = new Event("submit", { cancelable: true });
    const formData = new FormDataExtended(this.element);
    const submitData = await this._prepareSubmitData(event, formData);
    foundry.utils.mergeObject(submitData, updateData, { inplace: true });
    await this._processSubmitData(event, submitData);
  }
}

/**
 * Default sheet for activities.
 */
class ActivitySheet extends PseudoDocumentSheet {
  /** @inheritDoc */
  static DEFAULT_OPTIONS = {
    classes: ["activity"],
    window: {
      icon: "fa-solid fa-gauge"
    },
    actions: {
      addConsumption: ActivitySheet.#addConsumption,
      addDamagePart: ActivitySheet.#addDamagePart,
      addEffect: ActivitySheet.#addEffect,
      addRecovery: ActivitySheet.#addRecovery,
      deleteConsumption: ActivitySheet.#deleteConsumption,
      deleteDamagePart: ActivitySheet.#deleteDamagePart,
      deleteEffect: ActivitySheet.#deleteEffect,
      deleteRecovery: ActivitySheet.#deleteRecovery,
      dissociateEffect: ActivitySheet.#dissociateEffect,
      toggleCollapsed: ActivitySheet.#toggleCollapsed
    },
    position: {
      width: 500,
      height: "auto"
    }
  };

  /* -------------------------------------------- */

  /** @override */
  static PARTS = {
    tabs: {
      template: "templates/generic/tab-navigation.hbs"
    },
    identity: {
      template: "systems/dnd5e/templates/activity/identity.hbs",
      templates: [
        "systems/dnd5e/templates/activity/parts/activity-identity.hbs"
      ]
    },
    activation: {
      template: "systems/dnd5e/templates/activity/activation.hbs",
      templates: [
        "systems/dnd5e/templates/activity/parts/activity-time.hbs",
        "systems/dnd5e/templates/activity/parts/activity-targeting.hbs",
        "systems/dnd5e/templates/activity/parts/activity-consumption.hbs"
      ]
    },
    effect: {
      template: "systems/dnd5e/templates/activity/effect.hbs",
      templates: [
        "systems/dnd5e/templates/activity/parts/activity-effects.hbs"
      ]
    }
  };

  /* -------------------------------------------- */

  /**
   * Key paths to the parts of the submit data stored in arrays that will need special handling on submission.
   * @type {string[]}
   */
  static CLEAN_ARRAYS = ["consumption.targets", "damage.parts", "effects", "uses.recovery"];

  /* -------------------------------------------- */

  /** @override */
  tabGroups = {
    sheet: "identity",
    activation: "time"
  };

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * The Activity associated with this application.
   * @type {Activity}
   */
  get activity() {
    return this.document;
  }

  /* -------------------------------------------- */

  /**
   * Expanded states for additional settings sections.
   * @type {Map<string, boolean>}
   */
  #expandedSections = new Map();

  get expandedSections() {
    return this.#expandedSections;
  }

  /* -------------------------------------------- */

  /** @override */
  get title() {
    return this.activity.name;
  }

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareContext(options) {
    return {
      ...await super._prepareContext(options),
      activity: this.activity,
      fields: this.activity.schema.fields,
      inferred: this.activity._inferredSource,
      source: this.activity.toObject(),
      tabs: this._getTabs()
    };
  }

  /* -------------------------------------------- */

  /** @override */
  async _preparePartContext(partId, context) {
    switch ( partId ) {
      case "activation": return this._prepareActivationContext(context);
      case "effect": return this._prepareEffectContext(context);
      case "identity": return this._prepareIdentityContext(context);
    }
    return context;
  }

  /* -------------------------------------------- */

  /**
   * Prepare rendering context for the activation tab.
   * @param {ApplicationRenderContext} context  Context being prepared.
   * @returns {ApplicationRenderContext}
   * @protected
   */
  async _prepareActivationContext(context) {
    context.tab = context.tabs.activation;

    context.data = {};
    context.disabled = {};
    for ( const field of ["activation", "duration", "range", "target", "uses"] ) {
      if ( !this.activity[field] ) continue;
      context.data[field] = this.activity[field].override ? context.source[field] : context.inferred[field];
      context.disabled[field] = this.activity[field].canOverride && !this.activity[field].override;
    }

    context.activationTypes = [
      ...Object.entries(CONFIG.DND5E.activityActivationTypes).map(([value, config]) => ({
        value,
        label: game.i18n.localize(config.label),
        group: game.i18n.localize(config.group)
      })),
      { value: "", label: game.i18n.localize("DND5E.NoneActionLabel") }
    ];
    context.affectsPlaceholder = game.i18n.localize(
      `DND5E.TARGET.Count.${context.data.target?.template?.type ? "Every" : "Any"}`
    );
    context.durationUnits = [
      { value: "inst", label: game.i18n.localize("DND5E.TimeInst") },
      ...Object.entries(CONFIG.DND5E.scalarTimePeriods).map(([value, label]) => ({
        value, label, group: game.i18n.localize("DND5E.DurationTime")
      })),
      ...Object.entries(CONFIG.DND5E.permanentTimePeriods).map(([value, label]) => ({
        value, label, group: game.i18n.localize("DND5E.DurationPermanent")
      })),
      { value: "spec", label: game.i18n.localize("DND5E.Special") }
    ];
    context.rangeUnits = [
      ...Object.entries(CONFIG.DND5E.rangeTypes).map(([value, label]) => ({ value, label })),
      ...Object.entries(CONFIG.DND5E.movementUnits).map(([value, { label }]) => ({
        value, label, group: game.i18n.localize("DND5E.RangeDistance")
      }))
    ];

    // Consumption targets
    const canScale = this.activity.canConfigureScaling;
    const consumptionTypeOptions = Array.from(this.activity.validConsumptionTypes).map(value => ({
      value,
      label: CONFIG.DND5E.activityConsumptionTypes[value].label
    }));
    context.consumptionTargets = context.source.consumption.targets.map((data, index) => {
      const typeConfig = CONFIG.DND5E.activityConsumptionTypes[data.type] ?? {};
      const showTextTarget = typeConfig.targetRequiresEmbedded && !this.item.isEmbedded;
      const target = new ConsumptionTargetData(data, { parent: this.activity });
      return {
        data,
        fields: this.activity.schema.fields.consumption.fields.targets.element.fields,
        prefix: `consumption.targets.${index}.`,
        source: context.source.consumption.targets[index] ?? data,
        typeOptions: consumptionTypeOptions,
        scalingModes: canScale ? [
          { value: "", label: game.i18n.localize("DND5E.CONSUMPTION.Scaling.None") },
          { value: "amount", label: game.i18n.localize("DND5E.CONSUMPTION.Scaling.Amount") },
          ...(typeConfig.scalingModes ?? []).map(({ value, label }) => ({ value, label: game.i18n.localize(label) }))
        ] : null,
        showTargets: "validTargets" in typeConfig,
        targetPlaceholder: data.type === "itemUses" ? game.i18n.localize("DND5E.CONSUMPTION.Target.ThisItem") : null,
        validTargets: showTextTarget ? null : target.validTargets
      };
    });
    context.showConsumeSpellSlot = this.activity.isSpell && (this.item.system.level !== 0);
    context.showScaling = !this.activity.isSpell;

    // Uses recovery
    context.recoveryPeriods = CONFIG.DND5E.limitedUsePeriods.recoveryOptions;
    context.recoveryTypes = [
      { value: "recoverAll", label: game.i18n.localize("DND5E.USES.Recovery.Type.RecoverAll") },
      { value: "loseAll", label: game.i18n.localize("DND5E.USES.Recovery.Type.LoseAll") },
      { value: "formula", label: game.i18n.localize("DND5E.USES.Recovery.Type.Formula") }
    ];
    context.usesRecovery = context.source.uses.recovery.map((data, index) => ({
      data,
      fields: this.activity.schema.fields.uses.fields.recovery.element.fields,
      prefix: `uses.recovery.${index}.`,
      source: context.source.uses.recovery[index] ?? data,
      formulaOptions: data.period === "recharge" ? Array.fromRange(5, 2).reverse().map(min => ({
        value: min,
        label: game.i18n.format("DND5E.USES.Recovery.Recharge.Range", {
          range: min === 6 ? formatNumber(6) : formatRange(min, 6)
        })
      })) : null
    }));

    // Template dimensions
    context.dimensions = context.activity.target?.template?.dimensions;

    return context;
  }

  /* -------------------------------------------- */

  /**
   * Prepare a specific applied effect if present in the activity data.
   * @param {ApplicationRenderContext} context  Context being prepared.
   * @param {object} effect                     Applied effect context being prepared.
   * @returns {object}
   * @protected
   */
  _prepareAppliedEffectContext(context, effect) {
    return effect;
  }

  /* -------------------------------------------- */

  /**
   * Prepare a specific damage part if present in the activity data.
   * @param {ApplicationRenderContext} context  Context being prepared.
   * @param {object} part                       Damage part context being prepared.
   * @returns {object}
   * @protected
   */
  _prepareDamagePartContext(context, part) {
    return part;
  }

  /* -------------------------------------------- */

  /**
   * Prepare rendering context for the effect tab.
   * @param {ApplicationRenderContext} context  Context being prepared.
   * @returns {ApplicationRenderContext}
   * @protected
   */
  async _prepareEffectContext(context) {
    context.tab = context.tabs.effect;

    if ( context.activity.effects ) {
      const appliedEffects = new Set(context.activity.effects?.map(e => e._id) ?? []);
      context.allEffects = this.item.effects
        .filter(e => e.type !== "enchantment")
        .map(effect => ({
          value: effect.id, label: effect.name, selected: appliedEffects.has(effect.id)
        }));
      context.appliedEffects = context.activity.effects.reduce((arr, data) => {
        if ( !data.effect ) return arr;
        const effect = {
          data,
          collapsed: this.expandedSections.get(`effects.${data._id}`) ? "" : "collapsed",
          effect: data.effect,
          fields: this.activity.schema.fields.effects.element.fields,
          prefix: `effects.${data._index}.`,
          source: context.source.effects[data._index] ?? data,
          contentLink: data.effect.toAnchor().outerHTML,
          additionalSettings: null
        };
        arr.push(this._prepareAppliedEffectContext(context, effect));
        return arr;
      }, []);
    }

    context.denominationOptions = [
      { value: "", label: "" },
      ...CONFIG.DND5E.dieSteps.map(value => ({ value, label: `d${value}` }))
    ];
    if ( context.activity.damage?.parts ) {
      const scaleKey = (this.item.type === "spell") && (this.item.system.level === 0) ? "labelCantrip" : "label";
      const scalingOptions = [
        { value: "", label: game.i18n.localize("DND5E.DAMAGE.Scaling.None") },
        ...Object.entries(CONFIG.DND5E.damageScalingModes).map(([value, { [scaleKey]: label }]) => ({ value, label }))
      ];
      const typeOptions = Object.entries(CONFIG.DND5E.damageTypes).map(([value, { label }]) => ({ value, label }));
      const makePart = (data, index) => this._prepareDamagePartContext(context, {
        data, index, scalingOptions, typeOptions,
        canScale: this.activity.canScaleDamage,
        fields: this.activity.schema.fields.damage.fields.parts.element.fields,
        prefix: index !== undefined ? `damage.parts.${index}.` : "_.",
        source: data
      });
      context.damageParts = [
        ...context.activity.damage.parts
          .filter(p => p._index === undefined)
          .map((data, index) => makePart(data)),
        ...context.source.damage.parts.map((data, index) => makePart(data, index))
      ];
    }

    return context;
  }

  /* -------------------------------------------- */

  /**
   * Prepare rendering context for the identity tab.
   * @param {ApplicationRenderContext} context  Context being prepared.
   * @returns {ApplicationRenderContext}
   * @protected
   */
  async _prepareIdentityContext(context) {
    context.tab = context.tabs.identity;
    context.placeholder = {
      name: game.i18n.localize(this.activity.metadata.title),
      img: this.activity.metadata.img
    };
    return context;
  }

  /* -------------------------------------------- */

  /**
   * Prepare the tab information for the sheet.
   * @returns {Record<string, Partial<ApplicationTab>>}
   * @protected
   */
  _getTabs() {
    return this._markTabs({
      identity: {
        id: "identity", group: "sheet", icon: "fa-solid fa-tag",
        label: "DND5E.ACTIVITY.SECTIONS.Identity"
      },
      activation: {
        id: "activation", group: "sheet", icon: "fa-solid fa-clapperboard",
        label: "DND5E.ACTIVITY.SECTIONS.Activation",
        tabs: {
          time: {
            id: "time", group: "activation", icon: "fa-solid fa-clock",
            label: "DND5E.ACTIVITY.SECTIONS.Time"
          },
          consumption: {
            id: "consumption", group: "activation", icon: "fa-solid fa-boxes-stacked",
            label: "DND5E.CONSUMPTION.FIELDS.consumption.label"
          },
          targeting: {
            id: "activation-targeting", group: "activation", icon: "fa-solid fa-bullseye",
            label: "DND5E.TARGET.FIELDS.target.label"
          }
        }
      },
      effect: {
        id: "effect", group: "sheet", icon: "fa-solid fa-sun",
        label: "DND5E.ACTIVITY.SECTIONS.Effect"
      }
    });
  }

  /* -------------------------------------------- */

  /**
   * Helper to mark the tabs data structure with the appropriate CSS class if it is active.
   * @param {Record<string, Partial<ApplicationTab>>} tabs  Tabs definition to modify.
   * @returns {Record<string, Partial<ApplicationTab>>}
   * @internal
   */
  _markTabs(tabs) {
    for ( const v of Object.values(tabs) ) {
      v.active = this.tabGroups[v.group] === v.id;
      v.cssClass = v.active ? "active" : "";
      if ( "tabs" in v ) this._markTabs(v.tabs);
    }
    return tabs;
  }

  /* -------------------------------------------- */
  /*  Life-Cycle Handlers                         */
  /* -------------------------------------------- */

  /** @inheritDoc */
  _onRender(context, options) {
    super._onRender(context, options);
    for ( const element of this.element.querySelectorAll("[data-expand-id]") ) {
      element.querySelector(".collapsible")?.classList
        .toggle("collapsed", !this.#expandedSections.get(element.dataset.expandId));
    }
    this.#toggleNestedTabs();
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /** @inheritDoc */
  changeTab(tab, group, options={}) {
    super.changeTab(tab, group, options);
    if ( group !== "sheet" ) return;
    this.#toggleNestedTabs();
  }

  /* -------------------------------------------- */

  /**
   * Apply nested tab classes.
   */
  #toggleNestedTabs() {
    const primary = this.element.querySelector('.window-content > [data-application-part="tabs"]');
    const active = this.element.querySelector('.tab.active[data-group="sheet"]');
    if ( !primary || !active ) return;
    primary.classList.toggle("nested-tabs", active.querySelector(":scope > .sheet-tabs"));
  }

  /* -------------------------------------------- */

  /**
   * Handle adding a new entry to the consumption list.
   * @this {ActivitySheet}
   * @param {Event} event         Triggering click event.
   * @param {HTMLElement} target  Button that was clicked.
   */
  static #addConsumption(event, target) {
    const types = this.activity.validConsumptionTypes;
    const existingTypes = new Set(this.activity.consumption.targets.map(t => t.type));
    const filteredTypes = types.difference(existingTypes);
    this.activity.update({
      "consumption.targets": [
        ...this.activity.toObject().consumption.targets,
        { type: filteredTypes.first() ?? types.first() }
      ]
    });
  }

  /* -------------------------------------------- */

  /**
   * Handle adding a new entry to the damage parts list.
   * @this {ActivitySheet}
   * @param {Event} event         Triggering click event.
   * @param {HTMLElement} target  Button that was clicked.
   */
  static #addDamagePart(event, target) {
    if ( !this.activity.damage?.parts ) return;
    this.activity.update({ "damage.parts": [...this.activity.toObject().damage.parts, {}] });
  }

  /* -------------------------------------------- */

  /**
   * Handle creating a new active effect and adding it to the applied effects list.
   * @this {ActivitySheet}
   * @param {Event} event         Triggering click event.
   * @param {HTMLElement} target  Button that was clicked.
   */
  static async #addEffect(event, target) {
    if ( !this.activity.effects ) return;
    const effectData = this._addEffectData();
    const [created] = await this.item.createEmbeddedDocuments("ActiveEffect", [effectData], { render: false });
    this.activity.update({ effects: [...this.activity.toObject().effects, { _id: created.id }] });
  }

  /* -------------------------------------------- */

  /**
   * The data for a newly created applied effect.
   * @returns {object}
   * @protected
   */
  _addEffectData() {
    return {
      name: this.item.name,
      img: this.item.img,
      origin: this.item.uuid,
      transfer: false
    };
  }

  /* -------------------------------------------- */

  /**
   * Handle adding a new entry to the uses recovery list.
   * @this {ActivitySheet}
   * @param {Event} event         Triggering click event.
   * @param {HTMLElement} target  Button that was clicked.
   */
  static #addRecovery(event, target) {
    const periods = new Set(
      Object.entries(CONFIG.DND5E.limitedUsePeriods).filter(([, config]) => !config.deprecated).map(([k]) => k)
    );
    const existingPeriods = new Set(this.activity.uses.recovery.map(t => t.period));
    const filteredPeriods = periods.difference(existingPeriods);
    this.activity.update({
      "uses.recovery": [
        ...this.activity.toObject().uses.recovery,
        { period: filteredPeriods.first() ?? periods.first() }
      ]
    });
  }

  /* -------------------------------------------- */

  /**
   * Handle removing an entry from the consumption targets list.
   * @this {ActivitySheet}
   * @param {Event} event         Triggering click event.
   * @param {HTMLElement} target  Button that was clicked.
   */
  static #deleteConsumption(event, target) {
    const consumption = this.activity.toObject().consumption.targets;
    consumption.splice(target.closest("[data-index]").dataset.index, 1);
    this.activity.update({ "consumption.targets": consumption });
  }

  /* -------------------------------------------- */

  /**
   * Handle removing an entry from the damage parts list.
   * @this {ActivitySheet}
   * @param {Event} event         Triggering click event.
   * @param {HTMLElement} target  Button that was clicked.
   */
  static #deleteDamagePart(event, target) {
    if ( !this.activity.damage?.parts ) return;
    const parts = this.activity.toObject().damage.parts;
    parts.splice(target.closest("[data-index]").dataset.index, 1);
    this.activity.update({ "damage.parts": parts });
  }

  /* -------------------------------------------- */

  /**
   * Handle deleting an active effect and removing it from the applied effects list.
   * @this {ActivitySheet}
   * @param {Event} event         Triggering click event.
   * @param {HTMLElement} target  Button that was clicked.
   */
  static async #deleteEffect(event, target) {
    if ( !this.activity.effects ) return;
    const effectId = target.closest("[data-effect-id]")?.dataset.effectId;
    const result = await this.item.effects.get(effectId)?.deleteDialog({}, { render: false });
    if ( result instanceof ActiveEffect ) {
      const effects = this.activity.toObject().effects.filter(e => e._id !== effectId);
      this.activity.update({ effects });
    }
  }

  /* -------------------------------------------- */

  /**
   * Handle removing an entry from the uses recovery list.
   * @this {ActivitySheet}
   * @param {Event} event         Triggering click event.
   * @param {HTMLElement} target  Button that was clicked.
   */
  static #deleteRecovery(event, target) {
    const recovery = this.activity.toObject().uses.recovery;
    recovery.splice(target.closest("[data-index]").dataset.index, 1);
    this.activity.update({ "uses.recovery": recovery });
  }

  /* -------------------------------------------- */

  /**
   * Handle dissociating an Active Effect from this Activity.
   * @this {ActivitySheet}
   * @param {PointerEvent} event  The triggering click event.
   * @param {HTMLElement} target  The button that was clicked.
   */
  static #dissociateEffect(event, target) {
    const { effectId } = target.closest("[data-effect-id]")?.dataset ?? {};
    if ( !this.activity.effects || !effectId ) return;
    const effects = this.activity.toObject().effects.filter(e => e._id !== effectId);
    this.activity.update({ effects });
  }

  /* -------------------------------------------- */

  /**
   * Handle toggling the collapsed state of an additional settings section.
   * @this {ActivitySheet}
   * @param {Event} event         Triggering click event.
   * @param {HTMLElement} target  Button that was clicked.
   */
  static #toggleCollapsed(event, target) {
    if ( event.target.closest(".collapsible-content") ) return;
    target.classList.toggle("collapsed");
    this.#expandedSections.set(
      target.closest("[data-expand-id]")?.dataset.expandId,
      !target.classList.contains("collapsed")
    );
  }

  /* -------------------------------------------- */
  /*  Form Handling                               */
  /* -------------------------------------------- */

  /**
   * Perform any pre-processing of the form data to prepare it for updating.
   * @param {SubmitEvent} event          Triggering submit event.
   * @param {FormDataExtended} formData  Data from the submitted form.
   * @returns {object}
   */
  _prepareSubmitData(event, formData) {
    const submitData = super._prepareSubmitData(event, formData);
    for ( const keyPath of this.constructor.CLEAN_ARRAYS ) {
      const data = foundry.utils.getProperty(submitData, keyPath);
      if ( data ) foundry.utils.setProperty(submitData, keyPath, Object.values(data));
    }
    if ( foundry.utils.hasProperty(submitData, "appliedEffects") ) {
      const effects = submitData.effects ?? this.activity.toObject().effects;
      submitData.effects = effects.filter(e => submitData.appliedEffects.includes(e._id));
      for ( const _id of submitData.appliedEffects ) {
        if ( submitData.effects.find(e => e._id === _id) ) continue;
        submitData.effects.push({ _id });
      }
    }
    return submitData;
  }
}

/**
 * Application for creating dnd5e dialogs.
 */
class Dialog5e extends Application5e {
  /** @override */
  static DEFAULT_OPTIONS = {
    tag: "dialog",
    window: {
      contentTag: "form",
      contentClasses: ["standard-form"],
      minimizable: false
    }
  };

  /* -------------------------------------------- */

  /** @override */
  static PARTS = {
    content: {
      template: ""
    },
    footer: {
      template: "templates/generic/form-footer.hbs"
    }
  };

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * Form element within the dialog.
   * @type {HTMLFormElement|void}
   */
  get form() {
    return this.options.tag === "form" ? this.element : this.element.querySelector("form");
  }

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _preparePartContext(partId, context, options) {
    context = { ...(await super._preparePartContext(partId, context, options)) };
    if ( partId === "content" ) return this._prepareContentContext(context, options);
    if ( partId === "footer" ) return this._prepareFooterContext(context, options);
    return context;
  }

  /* -------------------------------------------- */

  /**
   * Prepare rendering context for the content section.
   * @param {ApplicationRenderContext} context  Context being prepared.
   * @param {HandlebarsRenderOptions} options   Options which configure application rendering behavior.
   * @returns {Promise<ApplicationRenderContext>}
   * @protected
   */
  async _prepareContentContext(context, options) {
    return context;
  }

  /* -------------------------------------------- */

  /**
   * Prepare rendering context for the footer.
   * @param {ApplicationRenderContext} context  Context being prepared.
   * @param {HandlebarsRenderOptions} options   Options which configure application rendering behavior.
   * @returns {Promise<ApplicationRenderContext>}
   * @protected
   */
  async _prepareFooterContext(context, options) {
    context.buttons = this.options.buttons?.map(button => ({
      ...button, cssClass: button.class
    }));
    return context;
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /** @inheritDoc */
  _attachFrameListeners() {
    super._attachFrameListeners();

    // Add event listeners to the form manually (see https://github.com/foundryvtt/foundryvtt/issues/11621)
    if ( this.options.tag !== "form" ) {
      this.form?.addEventListener("submit", this._onSubmitForm.bind(this, this.options.form));
      this.form?.addEventListener("change", this._onChangeForm.bind(this, this.options.form));
    }
  }
}

const { BooleanField: BooleanField$I, NumberField: NumberField$K, StringField: StringField$1b } = foundry.data.fields;

/**
 * Dialog for configuring the usage of an activity.
 */
class ActivityUsageDialog extends Dialog5e {
  constructor(options={}) {
    super(options);
    this.#activityId = options.activity.id;
    this.#item = options.activity.item;
    this.#config = options.config;
  }

  /* -------------------------------------------- */

  /** @override */
  static DEFAULT_OPTIONS = {
    classes: ["activity-usage"],
    actions: {
      use: ActivityUsageDialog.#onUse
    },
    activity: null,
    button: {
      icon: null,
      label: null
    },
    config: null,
    display: {
      all: true
    },
    form: {
      handler: ActivityUsageDialog.#onSubmitForm,
      submitOnChange: true
    },
    position: {
      width: 420
    }
  };

  /* -------------------------------------------- */

  /** @override */
  static PARTS = {
    scaling: {
      template: "systems/dnd5e/templates/activity/activity-usage-scaling.hbs"
    },
    concentration: {
      template: "systems/dnd5e/templates/activity/activity-usage-concentration.hbs"
    },
    consumption: {
      template: "systems/dnd5e/templates/activity/activity-usage-consumption.hbs"
    },
    creation: {
      template: "systems/dnd5e/templates/activity/activity-usage-creation.hbs"
    },
    footer: {
      template: "templates/generic/form-footer.hbs"
    }
  };

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * ID of the activity being activated.
   * @type {Activity}
   */
  #activityId;

  /**
   * Activity being activated.
   * @type {Activity}
   */
  get activity() {
    return this.item.system.activities.get(this.#activityId);
  }

  /* -------------------------------------------- */

  /**
   * Actor using this activity.
   * @type {Actor5e}
   */
  get actor() {
    return this.item.actor;
  }

  /* -------------------------------------------- */

  /**
   * Activity usage configuration data.
   * @type {ActivityUseConfiguration}
   */
  #config;

  get config() {
    return this.#config;
  }

  /* -------------------------------------------- */

  /**
   * Item that contains the activity.
   * @type {Item5e}
   */
  #item;

  get item() {
    return this.#item;
  }

  /* -------------------------------------------- */

  /** @override */
  get title() {
    return this.item.name;
  }

  /* -------------------------------------------- */

  /** @override */
  get subtitle() {
    return this.activity.name;
  }

  /* -------------------------------------------- */

  /**
   * Was the use button clicked?
   * @type {boolean}
   */
  #used = false;

  get used() {
    return this.#used;
  }

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  _configureRenderOptions(options) {
    super._configureRenderOptions(options);
    if ( options.isFirstRender ) options.window.icon ||= this.activity.img;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareContext(options) {
    if ( "scaling" in this.config ) this.#item = this.#item.clone({ "flags.dnd5e.scaling": this.config.scaling });
    return {
      ...await super._prepareContext(options),
      activity: this.activity,
      linkedActivity: this.config.cause ? this.activity.getLinkedActivity(this.config.cause.activity) : null
    };
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _preparePartContext(partId, context, options) {
    context = await super._preparePartContext(partId, context, options);
    switch ( partId ) {
      case "concentration": return this._prepareConcentrationContext(context, options);
      case "consumption": return this._prepareConsumptionContext(context, options);
      case "creation": return this._prepareCreationContext(context, options);
      case "scaling": return this._prepareScalingContext(context, options);
    }
    return context;
  }

  /* -------------------------------------------- */

  /**
   * Prepare rendering context for the concentration section.
   * @param {ApplicationRenderContext} context  Context being prepared.
   * @param {HandlebarsRenderOptions} options   Options which configure application rendering behavior.
   * @returns {Promise<ApplicationRenderContext>}
   * @protected
   */
  async _prepareConcentrationContext(context, options) {
    if ( !this.activity.requiresConcentration || game.settings.get("dnd5e", "disableConcentration")
      || !this._shouldDisplay("concentration") ) return context;
    context.hasConcentration = true;
    context.notes = [];

    context.fields = [{
      field: new BooleanField$I({ label: game.i18n.localize("DND5E.Concentration") }),
      name: "concentration.begin",
      value: this.config.concentration?.begin,
      input: context.inputs.createCheckboxInput
    }];
    if ( this.config.concentration?.begin ) {
      const existingConcentration = Array.from(this.actor.concentration.effects).map(effect => {
        const data = effect.getFlag("dnd5e", "item");
        return {
          value: effect.id,
          label: data?.data?.name ?? this.actor.items.get(data?.id)?.name
            ?? game.i18n.localize("DND5E.ConcentratingItemless")
        };
      });
      if ( existingConcentration.length ) {
        const optional = existingConcentration.length < (this.actor.system.attributes?.concentration?.limit ?? 0);
        context.fields.push({
          field: new StringField$1b({ label: game.i18n.localize("DND5E.ConcentratingEnd") }),
          name: "concentration.end",
          value: this.config.concentration?.end,
          options: optional ? [{ value: "", label: "—" }, ...existingConcentration] : existingConcentration
        });
        context.notes.push({
          type: "info", message: game.i18n.localize(`DND5E.ConcentratingWarnLimit${optional ? "Optional" : ""}`)
        });
      } else if ( !this.actor.system.attributes?.concentration?.limit ) {
        context.notes.push({
          type: "warn", message: game.i18n.localize("DND5E.ConcentratingWarnLimitZero")
        });
      }
    }

    return context;
  }

  /* -------------------------------------------- */

  /**
   * Prepare rendering context for the consumption section.
   * @param {ApplicationRenderContext} context  Context being prepared.
   * @param {HandlebarsRenderOptions} options   Options which configure application rendering behavior.
   * @returns {Promise<ApplicationRenderContext>}
   * @protected
   */
  async _prepareConsumptionContext(context, options) {
    context.fields = [];
    context.notes = [];

    const containsLegendaryConsumption = this.activity.consumption.targets
      .find(t => (t.type === "attribute") && (t.target === "resources.legact.value"));
    if ( (this.activity.activation.type === "legendary") && this.actor.system.resources?.legact
      && this._shouldDisplay("consume.action") && !containsLegendaryConsumption ) {
      const pr = new Intl.PluralRules(game.i18n.lang);
      const value = (this.config.consume !== false) && (this.config.consume?.action !== false);
      const warn = (this.actor.system.resources.legact.value < this.activity.activation.value) && value;
      context.fields.push({
        field: new BooleanField$I({
          label: game.i18n.format("DND5E.CONSUMPTION.Type.Action.Prompt", {
            type: game.i18n.localize("DND5E.LegendaryAction.Label")
          }),
          hint: game.i18n.format("DND5E.CONSUMPTION.Type.Action.PromptHint", {
            available: game.i18n.format(
              `DND5E.ACTIVATION.Type.Legendary.Counted.${pr.select(this.actor.system.resources.legact.value)}`,
              { number: `<strong>${formatNumber(this.actor.system.resources.legact.value)}</strong>` }
            ),
            cost: game.i18n.format(
              `DND5E.ACTIVATION.Type.Legendary.Counted.${pr.select(this.activity.activation.value)}`,
              { number: `<strong>${formatNumber(this.activity.activation.value)}</strong>` }
            )
          })
        }),
        input: context.inputs.createCheckboxInput,
        name: "consume.action",
        value, warn
      });
    }

    if ( this.activity.requiresSpellSlot && this.activity.consumption.spellSlot
      && this._shouldDisplay("consume.spellSlot") && !this.config.cause ) context.fields.push({
      field: new BooleanField$I({ label: game.i18n.localize("DND5E.SpellCastConsume") }),
      input: context.inputs.createCheckboxInput,
      name: "consume.spellSlot",
      value: this.config.consume?.spellSlot
    });

    if ( this._shouldDisplay("consume.resources") ) {
      const addResources = (targets, keyPath) => {
        const consume = foundry.utils.getProperty(this.config, keyPath);
        const isArray = foundry.utils.getType(consume) === "Array";
        for ( const [index, target] of targets.entries() ) {
          const value = (isArray && consume.includes(index))
            || (!isArray && (consume !== false) && (this.config.consume !== false));
          const { label, hint, notes, warn } = target.getConsumptionLabels(this.config, value);
          if ( notes?.length ) context.notes.push(...notes);
          context.fields.push({
            field: new BooleanField$I({ label, hint }),
            input: context.inputs.createCheckboxInput,
            name: `${keyPath}.${index}`,
            value,
            warn: value ? warn : false
          });
        }
      };
      addResources(this.activity.consumption.targets, "consume.resources");
      if ( context.linkedActivity && (!this.activity.isSpell || this.activity.consumption.spellSlot) ) {
        addResources(context.linkedActivity.consumption.targets, "cause.resources");
      }
    }

    context.hasConsumption = context.fields.length > 0;

    return context;
  }

  /* -------------------------------------------- */

  /**
   * Prepare rendering context for the creation section.
   * @param {ApplicationRenderContext} context  Context being prepared.
   * @param {HandlebarsRenderOptions} options   Options which configure application rendering behavior.
   * @returns {Promise<ApplicationRenderContext>}
   * @protected
   */
  async _prepareCreationContext(context, options) {
    context.hasCreation = false;
    if ( this.activity.target?.template?.type && this._shouldDisplay("create.measuredTemplate") ) {
      context.hasCreation = true;
      context.template = {
        field: new BooleanField$I({ label: game.i18n.localize("DND5E.TARGET.Action.PlaceTemplate") }),
        name: "create.measuredTemplate",
        value: this.config.create?.measuredTemplate
      };
    }
    return context;
  }

  /* -------------------------------------------- */

  /**
   * Prepare rendering context for the footer.
   * @param {ApplicationRenderContext} context  Context being prepared.
   * @param {HandlebarsRenderOptions} options   Options which configure application rendering behavior.
   * @returns {Promise<ApplicationRenderContext>}
   * @protected
   */
  async _prepareFooterContext(context, options) {
    context.buttons = [{
      action: "use",
      icon: this.options.button.icon ?? `fa-solid fa-${this.activity.isSpell ? "magic" : "fist-raised"}`,
      label: this.options.button.label ?? `DND5E.AbilityUse${this.activity.isSpell ? "Cast" : "Use"}`,
      type: "button"
    }];
    return context;
  }

  /* -------------------------------------------- */

  /**
   * Prepare rendering context for the scaling section.
   * @param {ApplicationRenderContext} context  Context being prepared.
   * @param {HandlebarsRenderOptions} options   Options which configure application rendering behavior.
   * @returns {Promise<ApplicationRenderContext>}
   * @protected
   */
  async _prepareScalingContext(context, options) {
    context.hasScaling = true;
    context.notes = [];
    if ( !this._shouldDisplay("scaling") ) {
      context.hasScaling = false;
      return context;
    }

    const scale = (context.linkedActivity ?? this.activity).consumption.scaling;
    const rollData = (context.linkedActivity ?? this.activity).getRollData({ deterministic: true });

    if ( this.activity.requiresSpellSlot && context.linkedActivity && (this.config.scaling !== false) ) {
      const max = simplifyBonus(scale.max, rollData);
      const minimumLevel = context.linkedActivity.spell?.level ?? this.item.system.level ?? 1;
      const maximumLevel = scale.allowed ? scale.max ? minimumLevel + max - 1 : Infinity : minimumLevel;
      const spellSlotOptions = Object.entries(CONFIG.DND5E.spellLevels).map(([level, label]) => {
        if ( (Number(level) < minimumLevel) || (Number(level) > maximumLevel) ) return null;
        return { value: `spell${level}`, label };
      }).filter(_ => _);
      context.spellSlots = {
        field: new StringField$1b({ label: game.i18n.localize("DND5E.SpellCastUpcast") }),
        name: "spell.slot",
        value: this.config.spell?.slot,
        options: spellSlotOptions
      };
    }

    else if ( this.activity.requiresSpellSlot && (this.config.scaling !== false) ) {
      const minimumLevel = this.item.system.level ?? 1;
      const maximumLevel = Object.values(this.actor.system.spells)
        .reduce((max, d) => d.max ? Math.max(max, d.level) : max, 0);

      const consumeSlot = (this.config.consume === true) || this.config.consume?.spellSlot;
      let spellSlotValue = this.actor.system.spells[this.config.spell?.slot]?.value || !consumeSlot
        ? this.config.spell.slot : null;
      const spellSlotOptions = Object.entries(this.actor.system.spells).map(([value, slot]) => {
        if ( (slot.level < minimumLevel) || (slot.level > maximumLevel) || !slot.type ) return null;
        let label;
        if ( slot.type === "leveled" ) {
          label = game.i18n.format("DND5E.SpellLevelSlot", { level: slot.label, n: slot.value });
        } else {
          label = game.i18n.format(`DND5E.SpellLevel${slot.type.capitalize()}`, { level: slot.level, n: slot.value });
        }
        // Set current value if applicable.
        const disabled = (slot.value === 0) && consumeSlot;
        if ( !disabled && !spellSlotValue ) spellSlotValue = value;
        return { value, label, disabled, selected: spellSlotValue === value };
      }).filter(_ => _);

      context.spellSlots = {
        field: new StringField$1b({ label: game.i18n.localize("DND5E.SpellCastUpcast") }),
        name: "spell.slot",
        value: spellSlotValue,
        options: spellSlotOptions
      };

      if ( !spellSlotOptions.some(o => !o.disabled) ) context.notes.push({
        type: "warn", message: game.i18n.format("DND5E.SpellCastNoSlotsLeft", {
          name: this.item.name
        })
      });
    }

    else if ( scale.allowed && (this.config.scaling !== false) ) {
      const max = scale.max ? simplifyBonus(scale.max, rollData) : Infinity;
      if ( max > 1 ) context.scaling = {
        field: new NumberField$K({ min: 1, max, label: game.i18n.localize("DND5E.ScalingValue") }),
        name: "scalingValue",
        // Config stores the scaling increase, but scaling value (increase + 1) is easier to understand in the UI
        value: Math.clamp((this.config.scaling ?? 0) + 1, 1, max),
        max,
        showRange: max <= 20
      };
      else context.hasScaling = false;
    }

    else {
      context.hasScaling = false;
    }

    return context;
  }

  /* -------------------------------------------- */

  /**
   * Determine whether a particular element should be displayed based on the `display` options.
   * @param {string} section  Key path describing the section to be displayed.
   * @returns {boolean}
   */
  _shouldDisplay(section) {
    const display = this.options.display;
    if ( foundry.utils.hasProperty(display, section) ) return foundry.utils.getProperty(display, section);
    const [group] = section.split(".");
    if ( (group !== section) && (group in display) ) return display[group];
    return this.options.display.all ?? true;
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /**
   * Handle form submission.
   * @this {ActivityUsageDialog}
   * @param {SubmitEvent} event          Triggering submit event.
   * @param {HTMLFormElement} form       The form that was submitted.
   * @param {FormDataExtended} formData  Data from the submitted form.
   */
  static async #onSubmitForm(event, form, formData) {
    const submitData = await this._prepareSubmitData(event, formData);
    await this._processSubmitData(event, submitData);
  }

  /* -------------------------------------------- */

  /**
   * Handle clicking the use button.
   * @this {ActivityUsageDialog}
   * @param {Event} event         Triggering click event.
   * @param {HTMLElement} target  Button that was clicked.
   */
  static async #onUse(event, target) {
    const formData = new FormDataExtended(this.element.querySelector("form"));
    const submitData = await this._prepareSubmitData(event, formData);
    foundry.utils.mergeObject(this.#config, submitData);
    this.#used = true;
    this.close();
  }

  /* -------------------------------------------- */

  /**
   * Perform any pre-processing of the form data to prepare it for updating.
   * @param {SubmitEvent} event          Triggering submit event.
   * @param {FormDataExtended} formData  Data from the submitted form.
   * @returns {Promise<object>}
   */
  async _prepareSubmitData(event, formData) {
    const submitData = foundry.utils.expandObject(formData.object);
    if ( foundry.utils.hasProperty(submitData, "spell.slot") ) {
      const level = this.actor.system.spells?.[submitData.spell.slot]?.level ?? 0;
      submitData.scaling = Math.max(0, level - this.item.system.level);
    } else if ( "scalingValue" in submitData ) {
      submitData.scaling = submitData.scalingValue - 1;
      delete submitData.scalingValue;
    }
    for ( const key of ["consume", "cause"] ) {
      if ( foundry.utils.getType(submitData[key]?.resources) === "Object" ) {
        submitData[key].resources = filteredKeys(submitData[key].resources).map(i => Number(i));
      }
    }
    return submitData;
  }

  /* -------------------------------------------- */

  /**
   * Handle updating the usage configuration based on processed submit data.
   * @param {SubmitEvent} event  Triggering submit event.
   * @param {object} submitData  Prepared object for updating.
   */
  async _processSubmitData(event, submitData) {
    foundry.utils.mergeObject(this.#config, submitData);
    this.render();
  }

  /* -------------------------------------------- */
  /*  Factory Methods                             */
  /* -------------------------------------------- */

  /**
   * Display the activity usage dialog.
   * @param {Activity} activity                Activity to use.
   * @param {ActivityUseConfiguration} config  Configuration data for the usage.
   * @param {object} options                   Additional options for the application.
   * @returns {Promise<object|null>}           Form data object with results of the activation.
   */
  static async create(activity, config, options) {
    if ( !activity.item.isOwned ) throw new Error("Cannot activate an activity that is not owned.");

    return new Promise((resolve, reject) => {
      const dialog = new this({ activity, config, ...options });
      dialog.addEventListener("close", event => {
        if ( dialog.used ) resolve(dialog.config);
        else reject();
      }, { once: true });
      dialog.render({ force: true });
    });
  }
}

/**
 * A helper class for building MeasuredTemplates for 5e spells and abilities
 */
class AbilityTemplate extends (foundry.canvas?.placeables?.MeasuredTemplate ?? MeasuredTemplate) {

  /**
   * Track the timestamp when the last mouse move event was captured.
   * @type {number}
   */
  #moveTime = 0;

  /* -------------------------------------------- */

  /**
   * Current token that is highlighted when using adjusted size template.
   * @type {Token5e}
   */
  #hoveredToken;

  /* -------------------------------------------- */

  /**
   * The initially active CanvasLayer to re-activate after the workflow is complete.
   * @type {CanvasLayer}
   */
  #initialLayer;

  /* -------------------------------------------- */

  /**
   * Track the bound event handlers so they can be properly canceled later.
   * @type {object}
   */
  #events;

  /* -------------------------------------------- */

  /**
   * A factory method to create an AbilityTemplate instance using provided data from an Activity instance.
   * @param {Activity} activity         The Activity for which to construct the template.
   * @param {object} [options={}]       Options to modify the created template.
   * @returns {AbilityTemplate[]|null}  The template objects, or null if the item does not produce a template.
   */
  static fromActivity(activity, options={}) {
    const target = activity.target?.template ?? {};
    const templateShape = dnd5e.config.areaTargetTypes[target.type]?.template;
    if ( !templateShape ) return null;

    // Prepare template data
    const rollData = activity.getRollData();
    const templateData = foundry.utils.mergeObject({
      t: templateShape,
      user: game.user.id,
      distance: target.size,
      direction: 0,
      x: 0,
      y: 0,
      fillColor: game.user.color,
      flags: { dnd5e: {
        dimensions: {
          size: target.size,
          width: target.width,
          height: target.height,
          adjustedSize: target.type === "radius"
        },
        item: activity.item.uuid,
        origin: activity.uuid,
        spellLevel: rollData.item.level
      } }
    }, options);

    // Additional type-specific data
    switch ( templateShape ) {
      case "cone":
        templateData.angle = CONFIG.MeasuredTemplate.defaults.angle;
        break;
      case "rect": // 5e rectangular AoEs are always cubes
        templateData.width = target.size;
        if ( game.settings.get("dnd5e", "gridAlignedSquareTemplates") ) {
          templateData.distance = Math.hypot(target.size, target.size);
          templateData.direction = 45;
        } else {
          // Override as 'ray' to make the template able to be rotated without morphing its shape
          templateData.t = "ray";
        }
        break;
      case "ray": // 5e rays are most commonly 1 square (5 ft) in width
        templateData.width = target.width ?? canvas.dimensions.distance;
        break;
    }

    /**
     * A hook event that fires before a template is created for an Activity.
     * @function dnd5e.preCreateActivityTemplate
     * @memberof hookEvents
     * @param {Activity} activity    Activity for which the template is being placed.
     * @param {object} templateData  Data used to create the new template.
     * @returns {boolean}            Explicitly return `false` to prevent the template from being placed.
     */
    if ( Hooks.call("dnd5e.preCreateActivityTemplate", activity, templateData) === false ) return null;

    if ( "dnd5e.preCreateItemTemplate" in Hooks.events ) {
      foundry.utils.logCompatibilityWarning(
        "The `dnd5e.preCreateItemTemplate` hook has been deprecated and replaced with `dnd5e.preCreateActivityTemplate`.",
        { since: "DnD5e 4.0", until: "DnD5e 4.4" }
      );
      if ( Hooks.call("dnd5e.preCreateItemTemplate", activity.item, templateData) === false ) return null;
    }

    // Construct the templates from activity data
    const cls = CONFIG.MeasuredTemplate.documentClass;
    const created = Array.fromRange(target.count || 1).map(() => {
      const template = new cls(foundry.utils.deepClone(templateData), { parent: canvas.scene });
      const object = new this(template);
      object.activity = activity;
      object.item = activity.item;
      object.actorSheet = activity.actor?.sheet || null;
      return object;
    });

    /**
     * A hook event that fires after a template are created for an Activity.
     * @function dnd5e.createActivityTemplate
     * @memberof hookEvents
     * @param {Activity} activity            Activity for which the template is being placed.
     * @param {AbilityTemplate[]} templates  The templates being placed.
     */
    Hooks.callAll("dnd5e.createActivityTemplate", activity, created);

    if ( "dnd5e.createItemTemplate" in Hooks.events ) {
      foundry.utils.logCompatibilityWarning(
        "The `dnd5e.createItemTemplate` hook has been deprecated and replaced with `dnd5e.createActivityTemplate`.",
        { since: "DnD5e 4.0", until: "DnD5e 4.4" }
      );
      Hooks.callAll("dnd5e.createItemTemplate", activity.item, created[0]);
    }

    return created;
  }

  /* -------------------------------------------- */

  /**
   * A factory method to create an AbilityTemplate instance using provided data from an Item5e instance
   * @param {Item5e} item               The Item object for which to construct the template
   * @param {object} [options={}]       Options to modify the created template.
   * @returns {AbilityTemplate|null}    The template object, or null if the item does not produce a template
   * @deprecated since DnD5e 4.0, available until DnD5e 4.4
   */
  static fromItem(item, options={}) {
    foundry.utils.logCompatibilityWarning(
      "The `AbilityTemplate#fromItem` method has been deprecated and replaced with `fromActivity`.",
      { since: "DnD5e 4.0", until: "DnD5e 4.4" }
    );
    const activity = this.system.activities?.contents[0];
    if ( activity ) return this.fromActivity(activity, options)?.[0] ?? null;
    return null;
  }

  /* -------------------------------------------- */

  /**
   * Creates a preview of the spell template.
   * @returns {Promise}  A promise that resolves with the final measured template if created.
   */
  drawPreview() {
    const initialLayer = canvas.activeLayer;

    // Draw the template and switch to the template layer
    this.draw();
    this.layer.activate();
    this.layer.preview.addChild(this);

    // Hide the sheet that originated the preview
    this.actorSheet?.minimize();

    // Activate interactivity
    return this.activatePreviewListeners(initialLayer);
  }

  /* -------------------------------------------- */

  /**
   * Activate listeners for the template preview
   * @param {CanvasLayer} initialLayer  The initially active CanvasLayer to re-activate after the workflow is complete
   * @returns {Promise}                 A promise that resolves with the final measured template if created.
   */
  activatePreviewListeners(initialLayer) {
    return new Promise((resolve, reject) => {
      this.#initialLayer = initialLayer;
      this.#events = {
        cancel: this._onCancelPlacement.bind(this),
        confirm: this._onConfirmPlacement.bind(this),
        move: this._onMovePlacement.bind(this),
        resolve,
        reject,
        rotate: this._onRotatePlacement.bind(this)
      };

      // Activate listeners
      canvas.stage.on("mousemove", this.#events.move);
      canvas.stage.on("mousedown", this.#events.confirm);
      canvas.app.view.oncontextmenu = this.#events.cancel;
      canvas.app.view.onwheel = this.#events.rotate;
    });
  }

  /* -------------------------------------------- */

  /**
   * Shared code for when template placement ends by being confirmed or canceled.
   * @param {Event} event  Triggering event that ended the placement.
   */
  async _finishPlacement(event) {
    this.layer._onDragLeftCancel(event);
    canvas.stage.off("mousemove", this.#events.move);
    canvas.stage.off("mousedown", this.#events.confirm);
    canvas.app.view.oncontextmenu = null;
    canvas.app.view.onwheel = null;
    if ( this.#hoveredToken ) {
      this.#hoveredToken._onHoverOut(event);
      this.#hoveredToken = null;
    }
    this.#initialLayer.activate();
    await this.actorSheet?.maximize();
  }

  /* -------------------------------------------- */

  /**
   * Move the template preview when the mouse moves.
   * @param {Event} event  Triggering mouse event.
   */
  _onMovePlacement(event) {
    event.stopPropagation();
    const now = Date.now(); // Apply a 20ms throttle
    if ( now - this.#moveTime <= 20 ) return;
    const center = event.data.getLocalPosition(this.layer);
    const updates = this.getSnappedPosition(center);

    // Adjust template size to take hovered token into account if `adjustedSize` is set
    const baseDistance = this.document.flags.dnd5e?.dimensions?.size;
    if ( this.document.flags.dnd5e?.dimensions?.adjustedSize && baseDistance ) {
      const rectangle = new PIXI.Rectangle(center.x, center.y, 1, 1);
      const hoveredToken = canvas.tokens.quadtree.getObjects(rectangle, {
        collisionTest: ({ t }) => t.visible && !t.document.isSecret }).first();
      if ( hoveredToken && (hoveredToken !== this.#hoveredToken) ) {
        this.#hoveredToken = hoveredToken;
        this.#hoveredToken._onHoverIn(event);
        const size = Math.max(hoveredToken.document.width, hoveredToken.document.height);
        updates.distance = baseDistance + (size * canvas.grid.distance / 2);
      } else if ( !hoveredToken && this.#hoveredToken ) {
        this.#hoveredToken._onHoverOut(event);
        this.#hoveredToken = null;
        updates.distance = baseDistance;
      }
    }

    this.document.updateSource(updates);
    this.refresh();
    this.#moveTime = now;
  }

  /* -------------------------------------------- */

  /**
   * Rotate the template preview by 3˚ increments when the mouse wheel is rotated.
   * @param {Event} event  Triggering mouse event.
   */
  _onRotatePlacement(event) {
    if ( event.ctrlKey ) event.preventDefault(); // Avoid zooming the browser window
    event.stopPropagation();
    const delta = canvas.grid.type > CONST.GRID_TYPES.SQUARE ? 30 : 15;
    const snap = event.shiftKey ? delta : 5;
    const update = {direction: this.document.direction + (snap * Math.sign(event.deltaY))};
    this.document.updateSource(update);
    this.refresh();
  }

  /* -------------------------------------------- */

  /**
   * Confirm placement when the left mouse button is clicked.
   * @param {Event} event  Triggering mouse event.
   */
  async _onConfirmPlacement(event) {
    await this._finishPlacement(event);
    const destination = canvas.templates.getSnappedPoint({ x: this.document.x, y: this.document.y });
    this.document.updateSource(destination);
    this.#events.resolve(canvas.scene.createEmbeddedDocuments("MeasuredTemplate", [this.document.toObject()]));
  }

  /* -------------------------------------------- */

  /**
   * Cancel placement when the right mouse button is clicked.
   * @param {Event} event  Triggering mouse event.
   */
  async _onCancelPlacement(event) {
    await this._finishPlacement(event);
    this.#events.reject();
  }

}

/**
 * A mixin which extends a DataModel to provide behavior shared between activities & advancements.
 * @template {DataModel} T
 * @param {typeof T} Base  The base DataModel to be mixed.
 * @returns {typeof PseudoDocument}
 * @mixin
 */
function PseudoDocumentMixin(Base) {
  class PseudoDocument extends Base {
    constructor(data, { parent=null, ...options }={}) {
      if ( parent instanceof Item ) parent = parent.system;
      super(data, { parent, ...options });
    }

    /* -------------------------------------------- */

    /**
     * Mapping of PseudoDocument UUID to the apps they should re-render.
     * @type {Map<string, Set<Application|ApplicationV2>>}
     * @internal
     */
    static _apps = new Map();

    /* -------------------------------------------- */

    /**
     * Existing sheets of a specific type for a specific document.
     * @type {Map<[PseudoDocument, typeof ApplicationV2], ApplicationV2>}
     */
    static _sheets = new Map();

    /* -------------------------------------------- */
    /*  Model Configuration                         */
    /* -------------------------------------------- */

    /**
     * Configuration information for PseudoDocuments.
     *
     * @typedef PseudoDocumentsMetadata
     * @property {string} name        Base type name of this PseudoDocument (e.g. "Activity", "Advancement").
     * @property {string} label       Localized name for this PseudoDocument type.
     */

    /**
     * Configuration information for PseudoDocuments.
     * @type {PseudoDocumentsMetadata}
     */
    get metadata() {
      return this.constructor.metadata;
    }

    /* -------------------------------------------- */

    /**
     * Configuration object that defines types.
     * @type {object}
     */
    static get documentConfig() {
      return CONFIG.DND5E[`${this.documentName.toLowerCase()}Types`];
    }

    get documentConfig() {
      return this.constructor.documentConfig;
    }

    /* -------------------------------------------- */

    /**
     * The canonical name of this PseudoDocument type, for example "Activity".
     * @type {string}
     */
    static get documentName() {
      return this.metadata.name;
    }

    get documentName() {
      return this.constructor.documentName;
    }

    /* -------------------------------------------- */
    /*  Instance Properties                         */
    /* -------------------------------------------- */

    /**
     * Unique identifier for this PseudoDocument within its item.
     * @type {string}
     */
    get id() {
      return this._id;
    }

    /* -------------------------------------------- */

    /**
     * Unique ID for this PseudoDocument on an actor.
     * @type {string}
     */
    get relativeID() {
      return `${this.item.id}.${this.id}`;
    }

    /* -------------------------------------------- */

    /**
     * Globally unique identifier for this PseudoDocument.
     * @type {string}
     */
    get uuid() {
      return `${this.item.uuid}.${this.documentName}.${this.id}`;
    }

    /* -------------------------------------------- */

    /**
     * Item to which this PseudoDocument belongs.
     * @type {Item5e}
     */
    get item() {
      return this.parent.parent;
    }

    /* -------------------------------------------- */

    /**
     * Actor to which this PseudoDocument's item belongs, if the item is embedded.
     * @type {Actor5e|null}
     */
    get actor() {
      return this.item.parent ?? null;
    }

    /* -------------------------------------------- */

    /**
     * Lazily obtain a Application instance used to configure this PseudoDocument, or null if no sheet is available.
     * @type {Application|ApplicationV2|null}
     */
    get sheet() {
      const cls = this.constructor.metadata.sheetClass ?? this.constructor.metadata.apps?.config;
      if ( !cls ) return null;
      if ( !this.constructor._sheets.has(this.uuid) ) {
        let sheet;
        if ( Application.isPrototypeOf(cls) ) sheet = new cls(this);
        else sheet = new cls({ document: this });
        this.constructor._sheets.set(this.uuid, sheet);
      }
      return this.constructor._sheets.get(this.uuid);
    }

    /* -------------------------------------------- */
    /*  Display Methods                             */
    /* -------------------------------------------- */

    /**
     * Render all the Application instances which are connected to this PseudoDocument.
     * @param {ApplicationRenderOptions} [options]  Rendering options.
     */
    render(options) {
      for ( const app of this.constructor._apps.get(this.uuid) ?? [] ) {
        app.render({ window: { title: app.title }, ...options });
      }
    }

    /* -------------------------------------------- */

    /**
     * Register an application to respond to updates to a certain document.
     * @param {PseudoDocument} doc  Pseudo document to watch.
     * @param {Application} app     Application to update.
     * @internal
     */
    static _registerApp(doc, app) {
      if ( !this._apps.has(doc.uuid) ) this._apps.set(doc.uuid, new Set());
      this._apps.get(doc.uuid).add(app);
    }

    /* -------------------------------------------- */

    /**
     * Remove an application from the render registry.
     * @param {PseudoDocument} doc  Pseudo document being watched.
     * @param {Application} app     Application to stop watching.
     */
    static _unregisterApp(doc, app) {
      this._apps.get(doc?.uuid)?.delete(app);
    }

    /* -------------------------------------------- */
    /*  Editing Methods                             */
    /* -------------------------------------------- */

    /**
     * Update this PseudoDocument.
     * @param {object} updates             Updates to apply to this PseudoDocument.
     * @param {object} [options={}]        Additional context which customizes the update workflow.
     * @returns {Promise<PseudoDocument>}  This PseudoDocument after updates have been applied.
     */
    async update(updates, options={}) {
      const result = await this.item[`update${this.documentName}`](this.id, updates, options);
      this.render();
      return result;
    }

    /* -------------------------------------------- */

    /**
     * Update this PseudoDocument's data on the item without performing a database commit.
     * @param {object} updates    Updates to apply to this PseudoDocument.
     * @returns {PseudoDocument}  This PseudoDocument after updates have been applied.
     */
    updateSource(updates) {
      super.updateSource(updates);
      return this;
    }

    /* -------------------------------------------- */

    /**
     * Delete this PseudoDocument, removing it from the database.
     * @param {object} [options={}]        Additional context which customizes the deletion workflow.
     * @returns {Promise<PseudoDocument>}  The deleted PseudoDocument instance.
     */
    async delete(options={}) {
      return await this.item[`delete${this.documentName}`](this.id, options);
    }

    /* -------------------------------------------- */

    /**
     * Present a Dialog form to confirm deletion of this PseudoDocument.
     * @param {object} [options]           Positioning and sizing options for the resulting dialog.
     * @returns {Promise<PseudoDocument>}  A Promise which resolves to the deleted PseudoDocument.
     */
    async deleteDialog(options={}) {
      const type = game.i18n.localize(this.metadata.label);
      return Dialog.confirm({
        title: `${game.i18n.format("DOCUMENT.Delete", { type })}: ${this.name || this.title}`,
        content: `<h4>${game.i18n.localize("AreYouSure")}</h4><p>${game.i18n.format("SIDEBAR.DeleteWarning", {
          type
        })}</p>`,
        yes: this.delete.bind(this),
        options: options
      });
    }

    /* -------------------------------------------- */

    /**
     * Serialize salient information for this PseudoDocument when dragging it.
     * @returns {object}  An object of drag data.
     */
    toDragData() {
      const dragData = { type: this.documentName, data: this.toObject() };
      if ( this.id ) dragData.uuid = this.uuid;
      return dragData;
    }

    /* -------------------------------------------- */
    /*  Importing and Exporting                     */
    /* -------------------------------------------- */

    /**
     * Spawn a dialog for creating a new Activity.
     * @param {object} [data]  Data to pre-populate the Activity with.
     * @param {object} context
     * @param {Item5e} context.parent        A parent for the Activity.
     * @param {string[]|null} [context.types]  A list of types to restrict the choices to, or null for no restriction.
     * @returns {Promise<Item5e|null>}
     */
    static async createDialog(data={}, { parent, types=null, ...options }={}) {
      types ??= Object.keys(this.documentConfig);
      if ( !types.length || !parent ) return null;

      const label = game.i18n.localize(`DOCUMENT.DND5E.${this.documentName}`);
      const title = game.i18n.format("DOCUMENT.Create", { type: label });
      let type = data.type;

      if ( !types.includes(type) ) type = types[0];
      const content = await renderTemplate("systems/dnd5e/templates/apps/document-create.hbs", {
        name, type,
        types: types.reduce((arr, type) => {
          const label = this.documentConfig[type]?.documentClass?.metadata?.title;
          arr.push({
            type,
            label: game.i18n.has(label) ? game.i18n.localize(label) : type,
            icon: this.documentConfig[type]?.documentClass?.metadata?.img
          });
          return arr;
        }, []).sort((a, b) => a.label.localeCompare(b.label, game.i18n.lang))
      });
      return Dialog.prompt({
        title, content,
        label: title,
        render: html => {
          const app = html.closest(".app");
          const folder = app.querySelector("select");
          if ( folder ) app.querySelector(".dialog-buttons").insertAdjacentElement("afterbegin", folder);
          app.querySelectorAll(".window-header .header-button").forEach(btn => {
            const label = btn.innerText;
            const icon = btn.querySelector("i");
            btn.innerHTML = icon.outerHTML;
            btn.dataset.tooltip = label;
            btn.setAttribute("aria-label", label);
          });
          app.querySelector(".document-name").select();
        },
        callback: html => {
          const form = html.querySelector("form");
          if ( !form.checkValidity() ) {
            throw new Error(game.i18n.format("DOCUMENT.DND5E.Warning.SelectType", { name: label }));
          }
          const fd = new FormDataExtended(form);
          const createData = foundry.utils.mergeObject(data, fd.object, { inplace: false });
          if ( !createData.name?.trim() ) delete createData.name;
          parent[`create${this.documentName}`](createData.type, createData);
        },
        rejectClose: false,
        options: { ...options, jQuery: false, width: 350, classes: ["dnd5e2", "create-document", "dialog"] }
      });
    }
  }
  return PseudoDocument;
}

/**
 * @import { PseudoDocumentsMetadata } from "../mixins/pseudo-document.mjs";
 */

/**
 * Mixin used to provide base logic to all activities.
 * @template {BaseActivityData} T
 * @param {typeof T} Base  The base activity data class to wrap.
 * @returns {typeof Activity}
 * @mixin
 */
function ActivityMixin(Base) {
  class Activity extends PseudoDocumentMixin(Base) {
    /**
     * Configuration information for Activities.
     *
     * @typedef {PseudoDocumentsMetadata} ActivityMetadata
     * @property {string} type                              Type name of this activity.
     * @property {string} img                               Default icon.
     * @property {string} title                             Default title.
     * @property {typeof ActivitySheet} sheetClass          Sheet class used to configure this activity.
     * @property {object} usage
     * @property {Record<string, Function>} usage.actions   Actions that can be triggered from the chat card.
     * @property {string} usage.chatCard                    Template used to render the chat card.
     * @property {typeof ActivityUsageDialog} usage.dialog  Default usage prompt.
     */

    /**
     * Configuration information for this PseudoDocument.
     * @type {Readonly<ActivityMetadata>}
     */
    static metadata = Object.freeze({
      name: "Activity",
      label: "DOCUMENT.DND5E.Activity",
      sheetClass: ActivitySheet,
      usage: {
        actions: {},
        chatCard: "systems/dnd5e/templates/chat/activity-card.hbs",
        dialog: ActivityUsageDialog
      }
    });

    /* -------------------------------------------- */

    /**
     * Perform the pre-localization of this data model.
     */
    static localize() {
      Localization.localizeDataModel(this);
      const fields = this.schema.fields;
      if ( fields.damage?.fields.parts ) {
        localizeSchema(fields.damage.fields.parts.element, ["DND5E.DAMAGE.FIELDS.damage.parts"]);
      }
      if ( fields.consumption ) {
        localizeSchema(fields.consumption.fields.targets.element, ["DND5E.CONSUMPTION.FIELDS.consumption.targets"]);
      }
      if ( fields.uses ) localizeSchema(fields.uses.fields.recovery.element, ["DND5E.USES.FIELDS.uses.recovery"]);
    }

    /* -------------------------------------------- */

    /**
     * Perform pre-localization on the contents of a SchemaField. Necessary because the `localizeSchema` method
     * on `Localization` is private.
     * @param {SchemaField} schema
     * @param {string[]} prefixes
     * @internal
     */
    static _localizeSchema(schema, prefixes) {
      localizeSchema(schema, prefixes);
    }

    /* -------------------------------------------- */
    /*  Properties                                  */
    /* -------------------------------------------- */

    /**
     * Should this activity be able to be used?
     * @type {boolean}
     */
    get canUse() {
      return true;
    }

    /* -------------------------------------------- */

    /**
     * Description used in chat message flavor for messages created with `rollDamage`.
     * @type {string}
     */
    get damageFlavor() {
      return game.i18n.localize("DND5E.DamageRoll");
    }

    /* -------------------------------------------- */

    /**
     * Create the data added to messages flags.
     * @type {object}
     */
    get messageFlags() {
      return {
        activity: { type: this.type, id: this.id, uuid: this.uuid },
        item: { type: this.item.type, id: this.item.id, uuid: this.item.uuid },
        targets: getTargetDescriptors()
      };
    }

    /* -------------------------------------------- */

    /**
     * Relative UUID for this activity on an actor.
     * @type {string}
     */
    get relativeUUID() {
      return `.Item.${this.item.id}.Activity.${this.id}`;
    }

    /* -------------------------------------------- */

    /**
     * Consumption targets that can be use for this activity.
     * @type {Set<string>}
     */
    get validConsumptionTypes() {
      const types = new Set(Object.keys(CONFIG.DND5E.activityConsumptionTypes));
      if ( this.isSpell ) types.delete("spellSlots");
      return types;
    }

    /* -------------------------------------------- */
    /*  Activation                                  */
    /* -------------------------------------------- */

    /**
     * Configuration data for an activity usage being prepared.
     *
     * @typedef {object} ActivityUseConfiguration
     * @property {object|false} create
     * @property {boolean} create.measuredTemplate     Should this item create a template?
     * @property {object} concentration
     * @property {boolean} concentration.begin         Should this usage initiate concentration?
     * @property {string|null} concentration.end       ID of an active effect to end concentration on.
     * @property {object|false} consume
     * @property {boolean} consume.action              Should action economy be tracked? Currently only handles
     *                                                 legendary actions.
     * @property {boolean|number[]} consume.resources  Set to `true` or `false` to enable or disable all resource
     *                                                 consumption or provide a list of consumption target indexes
     *                                                 to only enable those targets.
     * @property {boolean} consume.spellSlot           Should this spell consume a spell slot?
     * @property {Event} event                         The browser event which triggered the item usage, if any.
     * @property {boolean|number} scaling              Number of steps above baseline to scale this usage, or `false` if
     *                                                 scaling is not allowed.
     * @property {object} spell
     * @property {number} spell.slot                   The spell slot to consume.
     * @property {boolean} [subsequentActions=true]    Trigger subsequent actions defined by this activity.
     * @property {object} [cause]
     * @property {string} [cause.activity]             Relative UUID to the activity that caused this one to be used.
     *                                                 Activity must be on the same actor as this one.
     * @property {boolean|number[]} [cause.resources]  Control resource consumption on linked item.
     */

    /**
     * Data for the activity activation configuration dialog.
     *
     * @typedef {object} ActivityDialogConfiguration
     * @property {boolean} [configure=true]  Display a configuration dialog for the item usage, if applicable?
     * @property {typeof ActivityUsageDialog} [applicationClass]  Alternate activation dialog to use.
     * @property {object} [options]          Options passed through to the dialog.
     */

    /**
     * Message configuration for activity usage.
     *
     * @typedef {object} ActivityMessageConfiguration
     * @property {boolean} [create=true]     Whether to automatically create a chat message (if true) or simply return
     *                                       the prepared chat message data (if false).
     * @property {object} [data={}]          Additional data used when creating the message.
     * @property {boolean} [hasConsumption]  Was consumption available during activation.
     * @property {string} [rollMode]         The roll display mode with which to display (or not) the card.
     */

    /**
     * Details of final changes performed by the usage.
     *
     * @typedef {object} ActivityUsageResults
     * @property {ActiveEffect5e[]} effects              Active effects that were created or deleted.
     * @property {ChatMessage5e|object} message          The chat message created for the activation, or the message
     *                                                   data if `create` in ActivityMessageConfiguration was `false`.
     * @property {MeasuredTemplateDocument[]} templates  Created measured templates.
     * @property {ActivityUsageUpdates} updates          Updates to the actor & items.
     */

    /**
     * Activate this activity.
     * @param {ActivityUseConfiguration} usage        Configuration info for the activation.
     * @param {ActivityDialogConfiguration} dialog    Configuration info for the usage dialog.
     * @param {ActivityMessageConfiguration} message  Configuration info for the created chat message.
     * @returns {Promise<ActivityUsageResults|void>}  Details on the usage process if not canceled.
     */
    async use(usage={}, dialog={}, message={}) {
      if ( !this.item.isEmbedded || this.item.pack ) return;
      if ( !this.item.isOwner ) {
        ui.notifications.error("DND5E.DocumentUseWarn", { localize: true });
        return;
      }
      if ( !this.canUse ) {
        ui.notifications.error("DND5E.ACTIVITY.Warning.UsageNotAllowed", { localize: true });
        return;
      }

      // Create an item clone to work with throughout the rest of the process
      let item = this.item.clone({}, { keepId: true });
      let activity = item.system.activities.get(this.id);

      const usageConfig = activity._prepareUsageConfig(usage);

      const dialogConfig = foundry.utils.mergeObject({
        configure: true,
        applicationClass: this.metadata.usage.dialog
      }, dialog);

      const messageConfig = foundry.utils.mergeObject({
        create: true,
        data: {
          flags: {
            dnd5e: {
              ...this.messageFlags,
              messageType: "usage",
              use: {
                effects: this.applicableEffects?.map(e => e.id)
              }
            }
          }
        },
        hasConsumption: usageConfig.hasConsumption
      }, message);

      /**
       * A hook event that fires before an activity usage is configured.
       * @function dnd5e.preUseActivity
       * @memberof hookEvents
       * @param {Activity} activity                           Activity being used.
       * @param {ActivityUseConfiguration} usageConfig        Configuration info for the activation.
       * @param {ActivityDialogConfiguration} dialogConfig    Configuration info for the usage dialog.
       * @param {ActivityMessageConfiguration} messageConfig  Configuration info for the created chat message.
       * @returns {boolean}  Explicitly return `false` to prevent activity from being used.
       */
      if ( Hooks.call("dnd5e.preUseActivity", activity, usageConfig, dialogConfig, messageConfig) === false ) return;

      if ( "dnd5e.preUseItem" in Hooks.events ) {
        foundry.utils.logCompatibilityWarning(
          "The `dnd5e.preUseItem` hook has been deprecated and replaced with `dnd5e.preUseActivity`.",
          { since: "DnD5e 4.0", until: "DnD5e 4.4" }
        );
        const { config, options } = this._createDeprecatedConfigs(usageConfig, dialogConfig, messageConfig);
        if ( Hooks.call("dnd5e.preUseItem", item, config, options) === false ) return;
        this._applyDeprecatedConfigs(usageConfig, dialogConfig, messageConfig, config, options);
      }

      // Display configuration window if necessary
      if ( dialogConfig.configure && activity._requiresConfigurationDialog(usageConfig) ) {
        try {
          await dialogConfig.applicationClass.create(activity, usageConfig, dialogConfig.options);
        } catch(err) {
          return;
        }
      }

      // Handle scaling
      await activity._prepareUsageScaling(usageConfig, messageConfig, item);
      activity = item.system.activities.get(this.id);

      // Handle consumption
      const updates = await activity.consume(usageConfig, messageConfig);
      if ( updates === false ) return;
      const results = { effects: [], templates: [], updates };

      // Create concentration effect & end previous effects
      if ( usageConfig.concentration?.begin ) {
        const effect = await item.actor.beginConcentrating(activity, { "flags.dnd5e.scaling": usageConfig.scaling });
        if ( effect ) {
          results.effects ??= [];
          results.effects.push(effect);
          foundry.utils.setProperty(messageConfig.data, "flags.dnd5e.use.concentrationId", effect.id);
        }
        if ( usageConfig.concentration?.end ) {
          const deleted = await item.actor.endConcentration(usageConfig.concentration.end);
          results.effects.push(...deleted);
        }
      }

      // Create chat message
      messageConfig.data.rolls = (messageConfig.data.rolls ?? []).concat(updates.rolls);
      results.message = await activity._createUsageMessage(messageConfig);

      // Perform any final usage steps
      await activity._finalizeUsage(usageConfig, results);

      /**
       * A hook event that fires when an activity is activated.
       * @function dnd5e.postUseActivity
       * @memberof hookEvents
       * @param {Activity} activity                     Activity being activated.
       * @param {ActivityUseConfiguration} usageConfig  Configuration data for the activation.
       * @param {ActivityUsageResults} results          Final details on the activation.
       * @returns {boolean}  Explicitly return `false` to prevent any subsequent actions from being triggered.
       */
      if ( Hooks.call("dnd5e.postUseActivity", activity, usageConfig, results) === false ) return results;

      if ( "dnd5e.useItem" in Hooks.events ) {
        foundry.utils.logCompatibilityWarning(
          "The `dnd5e.useItem` hook has been deprecated and replaced with `dnd5e.postUseActivity`.",
          { since: "DnD5e 4.0", until: "DnD5e 4.4" }
        );
        const { config, options } = this._createDeprecatedConfigs(usageConfig, dialogConfig, messageConfig);
        Hooks.callAll("dnd5e.itemUsageConsumption", item, config, options, results.templates, results.effects, null);
      }

      // Trigger any primary action provided by this activity
      if ( usageConfig.subsequentActions !== false ) {
        activity._triggerSubsequentActions(usageConfig, results);
      }

      return results;
    }

    /* -------------------------------------------- */

    /**
     * Consume this activation's usage.
     * @param {ActivityUseConfiguration} usageConfig        Usage configuration.
     * @param {ActivityMessageConfiguration} messageConfig  Configuration data for the chat message.
     * @returns {ActivityUsageUpdates|false}
     */
    async consume(usageConfig, messageConfig) {
      /**
       * A hook event that fires before an item's resource consumption is calculated.
       * @function dnd5e.preActivityConsumption
       * @memberof hookEvents
       * @param {Activity} activity                           Activity being activated.
       * @param {ActivityUseConfiguration} usageConfig        Configuration data for the activation.
       * @param {ActivityMessageConfiguration} messageConfig  Configuration info for the created chat message.
       * @returns {boolean}  Explicitly return `false` to prevent activity from being activated.
       */
      if ( Hooks.call("dnd5e.preActivityConsumption", this, usageConfig, messageConfig) === false ) return false;

      if ( "dnd5e.preItemUsageConsumption" in Hooks.events ) {
        foundry.utils.logCompatibilityWarning(
          "The `dnd5e.preItemUsageConsumption` hook has been deprecated and replaced with `dnd5e.preActivityConsumption`.",
          { since: "DnD5e 4.0", until: "DnD5e 4.4" }
        );
        const { config, options } = this._createDeprecatedConfigs(usageConfig, {}, messageConfig);
        if ( Hooks.call("dnd5e.preItemUsageConsumption", this.item, config, options) === false ) return false;
        this._applyDeprecatedConfigs(usageConfig, {}, messageConfig, config, options);
      }

      const updates = await this._prepareUsageUpdates(usageConfig);
      if ( !updates ) return false;

      /**
       * A hook event that fires after an item's resource consumption is calculated, but before any updates are
       * performed.
       * @function dnd5e.activityConsumption
       * @memberof hookEvents
       * @param {Activity} activity                           Activity being activated.
       * @param {ActivityUseConfiguration} usageConfig        Configuration data for the activation.
       * @param {ActivityMessageConfiguration} messageConfig  Configuration info for the created chat message.
       * @param {ActivityUsageUpdates} updates                Updates to apply to the actor and other documents.
       * @returns {boolean}  Explicitly return `false` to prevent activity from being activated.
       */
      if ( Hooks.call("dnd5e.activityConsumption", this, usageConfig, messageConfig, updates) === false ) return false;

      if ( "dnd5e.itemUsageConsumption" in Hooks.events ) {
        foundry.utils.logCompatibilityWarning(
          "The `dnd5e.itemUsageConsumption` hook has been deprecated and replaced with `dnd5e.activityConsumption`.",
          { since: "DnD5e 4.0", until: "DnD5e 4.4" }
        );
        const { config, options } = this._createDeprecatedConfigs(usageConfig, {}, messageConfig);
        const usage = {
          actorUpdates: updates.actor,
          deleteIds: updates.delete,
          itemUpdates: updates.item.find(i => i._id === this.item.id),
          resourceUpdates: updates.item.filter(i => i._id !== this.item.id)
        };
        if ( Hooks.call("dnd5e.itemUsageConsumption", this.item, config, options, usage) === false ) return false;
        this._applyDeprecatedConfigs(usageConfig, {}, messageConfig, config, options);
        updates.actor = usage.actorUpdates;
        updates.delete = usage.deleteIds;
        updates.item = usage.resourceUpdates;
        if ( !foundry.utils.isEmpty(usage.itemUpdates) ) updates.item.push({ _id: this.item.id, ...usage.itemUpdates });
      }

      const consumed = await this.#applyUsageUpdates(updates);
      if ( !foundry.utils.isEmpty(consumed) ) {
        foundry.utils.setProperty(messageConfig, "data.flags.dnd5e.use.consumed", consumed);
      }
      if ( usageConfig.cause?.activity ) {
        foundry.utils.setProperty(messageConfig, "data.flags.dnd5e.use.cause", usageConfig.cause.activity);
      }

      /**
       * A hook event that fires after an item's resource consumption is calculated and applied.
       * @function dnd5e.postActivityConsumption
       * @memberof hookEvents
       * @param {Activity} activity                           Activity being activated.
       * @param {ActivityUseConfiguration} usageConfig        Configuration data for the activation.
       * @param {ActivityMessageConfiguration} messageConfig  Configuration info for the created chat message.
       * @param {ActivityUsageUpdates} updates                Applied updates to the actor and other documents.
       * @returns {boolean}  Explicitly return `false` to prevent activity from being activated.
       */
      if ( Hooks.call("dnd5e.postActivityConsumption", this, usageConfig, messageConfig, updates) === false ) return false;

      return updates;
    }

    /* -------------------------------------------- */

    /**
     * @typedef ActivityConsumptionDescriptor
     * @property {{ keyPath: string, delta: number }[]} actor                 Changes for the actor.
     * @property {Record<string, { keyPath: string, delta: number }[]>} item  Changes for each item grouped by ID.
     */

    /**
     * Refund previously used consumption for an activity.
     * @param {ActivityConsumptionDescriptor} consumed  Data on the consumption that occurred.
     */
    async refund(consumed) {
      const updates = {
        activity: {}, actor: {}, create: consumed.deleted ?? [], delete: consumed.created ?? [], item: []
      };
      for ( const { keyPath, delta } of consumed.actor ?? [] ) {
        const value = foundry.utils.getProperty(this.actor, keyPath) - delta;
        if ( !Number.isNaN(value) ) updates.actor[keyPath] = value;
      }
      for ( const [id, changes] of Object.entries(consumed.item ?? {}) ) {
        const item = this.actor.items.get(id);
        if ( !item ) continue;
        const itemUpdate = {};
        for ( const { keyPath, delta } of changes ) {
          let currentValue;
          if ( keyPath.startsWith("system.activities") ) {
            const [id, ...kp] = keyPath.slice(18).split(".");
            currentValue = foundry.utils.getProperty(item.system.activities?.get(id) ?? {}, kp.join("."));
          } else currentValue = foundry.utils.getProperty(item, keyPath);
          const value = currentValue - delta;
          if ( !Number.isNaN(value) ) itemUpdate[keyPath] = value;
        }
        if ( !foundry.utils.isEmpty(itemUpdate) ) {
          itemUpdate._id = id;
          updates.item.push(itemUpdate);
        }
      }
      await this.#applyUsageUpdates(updates);
    }

    /* -------------------------------------------- */

    /**
     * Merge activity updates into the appropriate item updates and apply.
     * @param {ActivityUsageUpdates} updates
     * @returns {ActivityConsumptionDescriptor}  Information on consumption performed to store in message flag.
     */
    async #applyUsageUpdates(updates) {
      this._mergeActivityUpdates(updates);

      // Ensure no existing items are created again & no non-existent items try to be deleted
      updates.create = updates.create?.filter(i => !this.actor.items.has(i));
      updates.delete = updates.delete?.filter(i => this.actor.items.has(i));

      // Create the consumed flag
      const getDeltas = (document, updates) => {
        updates = foundry.utils.flattenObject(updates);
        return Object.entries(updates).map(([keyPath, value]) => {
          let currentValue;
          if ( keyPath.startsWith("system.activities") ) {
            const [id, ...kp] = keyPath.slice(18).split(".");
            currentValue = foundry.utils.getProperty(document.system.activities?.get(id) ?? {}, kp.join("."));
          } else currentValue = foundry.utils.getProperty(document, keyPath);
          const delta = value - currentValue;
          if ( delta && !Number.isNaN(delta) ) return { keyPath, delta };
          return null;
        }).filter(_ => _);
      };
      const consumed = {
        actor: getDeltas(this.actor, updates.actor),
        item: updates.item.reduce((obj, { _id, ...changes }) => {
          const deltas = getDeltas(this.actor.items.get(_id), changes);
          if ( deltas.length ) obj[_id] = deltas;
          return obj;
        }, {})
      };
      if ( foundry.utils.isEmpty(consumed.actor) ) delete consumed.actor;
      if ( foundry.utils.isEmpty(consumed.item) ) delete consumed.item;
      if ( updates.create?.length ) consumed.created = updates.create;
      if ( updates.delete?.length ) consumed.deleted = updates.delete.map(i => this.actor.items.get(i).toObject());

      // Update documents with consumption
      if ( !foundry.utils.isEmpty(updates.actor) ) await this.actor.update(updates.actor);
      if ( !foundry.utils.isEmpty(updates.create) ) {
        await this.actor.createEmbeddedDocuments("Item", updates.create, { keepId: true });
      }
      if ( !foundry.utils.isEmpty(updates.delete) ) await this.actor.deleteEmbeddedDocuments("Item", updates.delete);
      if ( !foundry.utils.isEmpty(updates.item) ) await this.actor.updateEmbeddedDocuments("Item", updates.item);

      return consumed;
    }

    /* -------------------------------------------- */

    /**
     * Translate new config objects back into old config objects for deprecated hooks.
     * @param {ActivityUseConfiguration} usageConfig
     * @param {ActivityDialogConfiguration} dialogConfig
     * @param {ActivityMessageConfiguration} messageConfig
     * @returns {{ config: ItemUseConfiguration, options: ItemUseOptions }}
     * @internal
     */
    _createDeprecatedConfigs(usageConfig, dialogConfig, messageConfig) {
      let consumeResource;
      let consumeUsage;
      if ( (usageConfig.consume === true) || (usageConfig.consume?.resources === true) ) {
        consumeResource = consumeUsage = true;
      } else if ( (usageConfig.consume === false) || (usageConfig.comsume?.resources === false) ) {
        consumeResource = consumeUsage = false;
      } else if ( foundry.utils.getType(usageConfig.consume?.resources) === "Array" ) {
        for ( const index of usageConfig.consume.resources ) {
          if ( ["activityUses", "itemUses"].includes(this.consumption.targets[index]?.type) ) consumeUsage = true;
          else consumeResource = true;
        }
      }
      return {
        config: {
          createMeasuredTemplate: usageConfig.create?.measuredTemplate ?? null,
          consumeResource,
          consumeSpellSlot: usageConfig.consume?.spellSlot !== false ?? null,
          consumeUsage,
          slotLevel: usageConfig.spell?.slot ?? null,
          resourceAmount: usageConfig.scaling ?? null,
          beginConcentrating: usageConfig.concentration?.begin ?? false,
          endConcentration: usageConfig.concentration?.end ?? null
        },
        options: {
          configureDialog: dialogConfig.configure,
          rollMode: messageConfig.rollMode,
          createMessage: messageConfig.create,
          flags: messageConfig.data?.flags,
          event: usageConfig.event
        }
      };
    }

    /* -------------------------------------------- */

    /**
     * Apply changes from old config objects back onto new config objects.
     * @param {ActivityUseConfiguration} usageConfig
     * @param {ActivityDialogConfiguration} dialogConfig
     * @param {ActivityMessageConfiguration} messageConfig
     * @param {ItemUseConfiguration} config
     * @param {ItemUseOptions} options
     * @internal
     */
    _applyDeprecatedConfigs(usageConfig, dialogConfig, messageConfig, config, options) {
      const { resourceIndices, usageIndices } = this.consumption.targets.reduce((o, data, index) => {
        if ( ["activityUses", "itemUses"].includes(data.type) ) o.usageIndices.push(index);
        else o.resourceIndices.push(index);
        return o;
      }, { resourceIndices: [], usageIndices: [] });
      let resources;
      if ( config.consumeResource && config.consumeUsage ) resources = true;
      else if ( config.consumeResource && (config.consumeUsage === false) ) resources = resourceIndices;
      else if ( (config.consumeResource === false) && config.consumeUsage ) resources = usageIndices;

      // Set property so long as the value is not undefined
      // Avoids problems with `mergeObject` overwriting values with `undefined`
      const set = (config, keyPath, value) => {
        if ( value === undefined ) return;
        foundry.utils.setProperty(config, keyPath, value);
      };

      set(usageConfig, "create.measuredTemplate", config.createMeasuredTemplate);
      set(usageConfig, "concentration.begin", config.beginConcentrating);
      set(usageConfig, "concentration.end", config.endConcentration);
      set(usageConfig, "consume.resources", resources);
      set(usageConfig, "consume.spellSlot", config.consumeSpellSlot);
      set(usageConfig, "scaling", config.resourceAmount);
      set(usageConfig, "spell.slot", config.slotLevel);
      set(dialogConfig, "configure", options.configureDialog);
      set(messageConfig, "create", options.createMessage);
      set(messageConfig, "rollMode", options.rollMode);
      if ( options.flags ) {
        messageConfig.data ??= {};
        messageConfig.data.flags = foundry.utils.mergeObject(messageConfig.data.flags ?? {}, options.flags);
      }
    }

    /* -------------------------------------------- */

    /**
     * Prepare usage configuration with the necessary defaults.
     * @param {ActivityUseConfiguration} config  Configuration object passed to the `use` method.
     * @returns {ActivityUseConfiguration}
     * @protected
     */
    _prepareUsageConfig(config) {
      config = foundry.utils.deepClone(config);
      const linked = this.getLinkedActivity(config.cause?.activity);

      if ( config.create !== false ) {
        config.create ??= {};
        config.create.measuredTemplate ??= !!this.target.template.type && this.target.prompt;
        // TODO: Handle permissions checks in `ActivityUsageDialog`
      }

      const ignoreLinkedConsumption = this.isSpell && !this.consumption.spellSlot;
      if ( config.consume !== false ) {
        const hasActionConsumption = this.activation.type === "legendary";
        const hasResourceConsumption = this.consumption.targets.length > 0;
        const hasLinkedConsumption = (linked?.consumption.targets.length > 0) && !ignoreLinkedConsumption;
        const hasSpellSlotConsumption = this.requiresSpellSlot && this.consumption.spellSlot;
        config.consume ??= {};
        config.consume.action ??= hasActionConsumption;
        config.consume.resources ??= Array.from(this.consumption.targets.entries())
          .filter(([, target]) => !target.combatOnly || this.actor.inCombat)
          .map(([index]) => index);
        config.consume.spellSlot ??= !linked && hasSpellSlotConsumption;
        config.hasConsumption = hasActionConsumption || hasResourceConsumption || hasLinkedConsumption
          || (!linked && hasSpellSlotConsumption);
      }

      const levelingFlag = this.item.getFlag("dnd5e", "spellLevel");
      if ( levelingFlag ) {
        // Handle fixed scaling from spell scrolls
        config.scaling = false;
        config.spell ??= {};
        config.spell.slot = levelingFlag.value;
      }

      else {
        const canScale = linked ? linked.consumption.scaling.allowed : this.canScale;
        const linkedDelta = (linked?.spell?.level ?? Infinity) - this.item.system.level;
        if ( canScale ) config.scaling ??= Number.isFinite(linkedDelta) ? linkedDelta : 0;
        else config.scaling = false;

        if ( this.requiresSpellSlot ) {
          const mode = this.item.system.preparation.mode;
          config.spell ??= {};
          config.spell.slot ??= linked?.spell?.level ? `spell${linked.spell.level}`
            : (mode in this.actor.system.spells) ? mode : `spell${this.item.system.level}`;
        }
      }

      if ( this.requiresConcentration && !game.settings.get("dnd5e", "disableConcentration") ) {
        config.concentration ??= {};
        config.concentration.begin ??= true;
        const { effects } = this.actor.concentration;
        const limit = this.actor.system.attributes?.concentration?.limit ?? 0;
        if ( limit && (limit <= effects.size) ) config.concentration.end ??= effects.find(e => {
          const data = e.flags.dnd5e?.item?.data ?? {};
          return (data === this.id) || (data._id === this.id);
        })?.id ?? effects.first()?.id ?? null;
      }

      if ( linked ) {
        config.cause ??= {};
        config.cause.activity ??= linked.relativeUUID;
        config.cause.resources ??= (linked.consumption.targets.length > 0) && !ignoreLinkedConsumption;
      }

      return config;
    }

    /* -------------------------------------------- */

    /**
     * Determine scaling values and update item clone if necessary.
     * @param {ActivityUseConfiguration} usageConfig        Configuration data for the activation.
     * @param {ActivityMessageConfiguration} messageConfig  Configuration data for the chat message.
     * @param {Item5e} item                                 Clone of the item that contains this activity.
     * @protected
     */
    async _prepareUsageScaling(usageConfig, messageConfig, item) {
      const levelingFlag = this.item.getFlag("dnd5e", "spellLevel");
      if ( levelingFlag ) {
        usageConfig.scaling = Math.max(0, levelingFlag.value - levelingFlag.base);
      } else if ( this.isSpell ) {
        const level = this.actor.system.spells?.[usageConfig.spell?.slot]?.level;
        if ( level ) {
          usageConfig.scaling = level - item.system.level;
          foundry.utils.setProperty(messageConfig, "data.flags.dnd5e.use.spellLevel", level);
        }
      }

      if ( usageConfig.scaling ) {
        foundry.utils.setProperty(messageConfig, "data.flags.dnd5e.scaling", usageConfig.scaling);
        item.updateSource({ "flags.dnd5e.scaling": usageConfig.scaling });
        item.prepareFinalAttributes();
      }
    }

    /* -------------------------------------------- */

    /**
     * Update data produced by activity usage.
     *
     * @typedef {object} ActivityUsageUpdates
     * @property {object} activity  Updates applied to activity that performed the activation.
     * @property {object} actor     Updates applied to the actor that performed the activation.
     * @property {object[]} create  Full data for Items to create (with IDs maintained).
     * @property {string[]} delete  IDs of items to be deleted from the actor.
     * @property {object[]} item    Updates applied to items on the actor that performed the activation.
     * @property {Roll[]} rolls     Any rolls performed as part of the activation.
     */

    /**
     * Calculate changes to actor, items, & this activity based on resource consumption.
     * @param {ActivityUseConfiguration} config                  Usage configuration.
     * @param {object} [options={}]
     * @param {boolean} [options.returnErrors=false]             Return array of errors, rather than displaying them.
     * @returns {ActivityUsageUpdates|ConsumptionError[]|false}  Updates to perform, an array of ConsumptionErrors,
     *                                                           or `false` if a consumption error occurred.
     * @protected
     */
    async _prepareUsageUpdates(config, { returnErrors=false }={}) {
      const updates = { activity: {}, actor: {}, create: [], delete: [], item: [], rolls: [] };
      if ( config.consume === false ) return updates;
      const errors = [];

      // Handle action economy
      if ( ((config.consume === true) || config.consume.action) && (this.activation.type === "legendary") ) {
        const containsLegendaryConsumption = this.consumption.targets
          .find(t => (t.type === "attribute") && (t.target === "resources.legact.value"));
        const count = this.activation.value ?? 1;
        const legendary = this.actor.system.resources?.legact;
        if ( legendary && !containsLegendaryConsumption ) {
          let message;
          if ( legendary.value === 0 ) message = "DND5E.ACTIVATION.Warning.NoActions";
          else if ( count > legendary.value ) message = "DND5E.ACTIVATION.Warning.NotEnoughActions";
          if ( message ) {
            const err = new ConsumptionError(game.i18n.format(message, {
              type: game.i18n.localize("DND5E.LegendaryAction.Label"),
              required: formatNumber(count),
              available: formatNumber(legendary.value)
            }));
            errors.push(err);
          } else {
            updates.actor["system.resources.legact.value"] = legendary.value - count;
          }
        }
      }

      // Handle consumption targets
      if ( (config.consume === true) || config.consume.resources ) {
        const indexes = (config.consume === true) || (config.consume.resources === true)
          ? this.consumption.targets.keys() : config.consume.resources;
        for ( const index of indexes ) {
          const target = this.consumption.targets[index];
          try {
            await target.consume(config, updates);
          } catch(err) {
            if ( err instanceof ConsumptionError ) errors.push(err);
            else throw err;
          }
        }
      }

      // Handle consumption on a linked activity
      if ( config.cause ) {
        const linkedActivity = this.getLinkedActivity(config.cause.activity);
        if ( linkedActivity ) {
          const consume = {
            resources: (config.consume === true) || (config.cause?.resources === true)
              ? linkedActivity.consumption.targets.keys() : config.cause?.resources,
            spellSlot: false
          };
          const usageConfig = foundry.utils.mergeObject(config, { consume, cause: false }, { inplace: false });
          const results = await linkedActivity._prepareUsageUpdates(usageConfig, { returnErrors: true });
          if ( foundry.utils.getType(results) === "Object" ) {
            linkedActivity._mergeActivityUpdates(results);
            foundry.utils.mergeObject(updates.actor, results.actor);
            updates.delete.push(...results.delete);
            updates.item.push(...results.item);
            updates.rolls.push(...results.rolls);
            // Mark this item for deletion if it is linked to a cast activity that will be deleted
            if ( updates.delete.includes(linkedActivity.item.id)
              && (this.item.getFlag("dnd5e", "cachedFor") === linkedActivity.relativeUUID) ) {
              updates.delete.push(this.item.id);
            }
          } else if ( results?.length ) {
            errors.push(...results);
          }
        }
      }

      // Handle spell slot consumption
      else if ( ((config.consume === true) || config.consume.spellSlot)
        && this.requiresSpellSlot && this.consumption.spellSlot ) {
        const mode = this.item.system.preparation.mode;
        const isLeveled = ["always", "prepared"].includes(mode);
        const effectiveLevel = this.item.system.level + (config.scaling ?? 0);
        const slot = config.spell?.slot ?? (isLeveled ? `spell${effectiveLevel}` : mode);
        const slotData = this.actor.system.spells?.[slot];
        if ( slotData ) {
          if ( slotData.value ) {
            const newValue = Math.max(slotData.value - 1, 0);
            foundry.utils.mergeObject(updates.actor, { [`system.spells.${slot}.value`]: newValue });
          } else {
            const err = new ConsumptionError(game.i18n.format("DND5E.SpellCastNoSlots", {
              name: this.item.name, level: slotData.label
            }));
            errors.push(err);
          }
        }
      }

      // Ensure concentration can be handled
      if ( config.concentration?.begin ) {
        const { effects } = this.actor.concentration;
        // Ensure existing concentration effect exists when replacing concentration
        if ( config.concentration.end ) {
          const replacedEffect = effects.find(i => i.id === config.concentration.end);
          if ( !replacedEffect ) errors.push(
            new ConsumptionError(game.i18n.localize("DND5E.ConcentratingMissingItem"))
          );
        }

        // Cannot begin more concentrations than the limit
        else if ( effects.size >= this.actor.system.attributes?.concentration?.limit ) errors.push(
          new ConsumptionError(game.i18n.localize("DND5E.ConcentratingLimited"))
        );
      }

      if ( !returnErrors ) errors.forEach(err => ui.notifications.error(err.message, { console: false }));
      return errors.length ? returnErrors ? errors : false : updates;
    }

    /* -------------------------------------------- */

    /**
     * Determine if the configuration dialog is required based on the configuration options. Does not guarantee a dialog
     * is shown if the dialog is suppressed in the activation dialog configuration.
     * @param {ActivityUseConfiguration} config
     * @returns {boolean}
     * @protected
     */
    _requiresConfigurationDialog(config) {
      const checkObject = obj => (foundry.utils.getType(obj) === "Object")
        && Object.values(obj).some(v => v === true || v?.length);
      return config.concentration?.begin === true
        || checkObject(config.create)
        || ((checkObject(config.consume) || (config.cause?.resources === true)) && config.hasConsumption)
        || (config.scaling !== false);
    }

    /* -------------------------------------------- */

    /**
     * Prepare the context used to render the usage chat card.
     * @param {ActivityMessageConfiguration} message  Configuration info for the created message.
     * @returns {object}
     * @protected
     */
    async _usageChatContext(message) {
      const data = await this.item.system.getCardData({ activity: this });
      const properties = [...(data.tags ?? []), ...(data.properties ?? [])];
      const supplements = [];
      if ( this.activation.condition ) {
        supplements.push(`<strong>${game.i18n.localize("DND5E.Trigger")}</strong> ${this.activation.condition}`);
      }
      if ( data.materials?.value ) {
        supplements.push(`<strong>${game.i18n.localize("DND5E.Materials")}</strong> ${data.materials.value}`);
      }
      const buttons = this._usageChatButtons(message);

      // Include spell level in the subtitle.
      if ( this.item.type === "spell" ) {
        const spellLevel = foundry.utils.getProperty(message, "data.flags.dnd5e.use.spellLevel");
        const { spellLevels, spellSchools } = CONFIG.DND5E;
        data.subtitle = [spellLevels[spellLevel], spellSchools[this.item.system.school]?.label].filterJoin(" &bull; ");
      }

      return {
        activity: this,
        actor: this.item.actor,
        item: this.item,
        token: this.item.actor?.token,
        buttons: buttons.length ? buttons : null,
        description: data.description,
        properties: properties.length ? properties : null,
        subtitle: this.description.chatFlavor || data.subtitle,
        supplements
      };
    }

    /* -------------------------------------------- */

    /**
     * @typedef {object} ActivityUsageChatButton
     * @property {string} label    Label to display on the button.
     * @property {string} icon     Icon to display on the button.
     * @property {string} classes  Classes for the button.
     * @property {object} dataset  Data attributes attached to the button.
     */

    /**
     * Create the buttons that will be displayed in chat.
     * @param {ActivityMessageConfiguration} message  Configuration info for the created message.
     * @returns {ActivityUsageChatButton[]}
     * @protected
     */
    _usageChatButtons(message) {
      const buttons = [];

      if ( this.target?.template?.type ) buttons.push({
        label: game.i18n.localize("DND5E.TARGET.Action.PlaceTemplate"),
        icon: '<i class="fas fa-bullseye" inert></i>',
        dataset: {
          action: "placeTemplate"
        }
      });

      if ( message.hasConsumption ) buttons.push({
        label: game.i18n.localize("DND5E.CONSUMPTION.Action.ConsumeResource"),
        icon: '<i class="fa-solid fa-cubes-stacked" inert></i>',
        dataset: {
          action: "consumeResource"
        }
      }, {
        label: game.i18n.localize("DND5E.CONSUMPTION.Action.RefundResource"),
        icon: '<i class="fa-solid fa-clock-rotate-left"></i>',
        dataset: {
          action: "refundResource"
        }
      });

      return buttons;
    }

    /* -------------------------------------------- */

    /**
     * Determine whether the provided button in a chat message should be visible.
     * @param {HTMLButtonElement} button  The button to check.
     * @param {ChatMessage5e} message     Chat message containing the button.
     * @returns {boolean}
     */
    shouldHideChatButton(button, message) {
      const flag = message.getFlag("dnd5e", "use.consumed");
      switch ( button.dataset.action ) {
        case "consumeResource": return !!flag;
        case "refundResource": return !flag;
        case "placeTemplate": return !game.user.can("TEMPLATE_CREATE") || !game.canvas.scene;
      }
      return false;
    }

    /* -------------------------------------------- */

    /**
     * Display a chat message for this usage.
     * @param {ActivityMessageConfiguration} message  Configuration info for the created message.
     * @returns {Promise<ChatMessage5e|object>}
     * @protected
     */
    async _createUsageMessage(message) {
      const context = await this._usageChatContext(message);
      const messageConfig = foundry.utils.mergeObject({
        rollMode: game.settings.get("core", "rollMode"),
        data: {
          content: await renderTemplate(this.metadata.usage.chatCard, context),
          speaker: ChatMessage.getSpeaker({ actor: this.item.actor }),
          flags: {
            core: { canPopout: true }
          }
        }
      }, message);

      /**
       * A hook event that fires before an activity usage card is created.
       * @function dnd5e.preCreateUsageMessage
       * @memberof hookEvents
       * @param {Activity} activity                     Activity for which the card will be created.
       * @param {ActivityMessageConfiguration} message  Configuration info for the created message.
       */
      Hooks.callAll("dnd5e.preCreateUsageMessage", this, messageConfig);

      ChatMessage.applyRollMode(messageConfig.data, messageConfig.rollMode);
      const card = messageConfig.create === false ? messageConfig.data : await ChatMessage.create(messageConfig.data);

      /**
       * A hook event that fires after an activity usage card is created.
       * @function dnd5e.postCreateUsageMessage
       * @memberof hookEvents
       * @param {Activity} activity          Activity for which the card was created.
       * @param {ChatMessage5e|object} card  Created card or configuration data if not created.
       */
      Hooks.callAll("dnd5e.postCreateUsageMessage", this, card);

      return card;
    }

    /* -------------------------------------------- */

    /**
     * Perform any final steps of the activation including creating measured templates.
     * @param {ActivityUseConfiguration} config  Configuration data for the activation.
     * @param {ActivityUsageResults} results     Final details on the activation.
     * @protected
     */
    async _finalizeUsage(config, results) {
      results.templates = config.create?.measuredTemplate ? await this.#placeTemplate() : [];
    }

    /* -------------------------------------------- */

    /**
     * Trigger a primary activation action defined by the activity (such as opening the attack dialog for attack rolls).
     * @param {ActivityUseConfiguration} config  Configuration data for the activation.
     * @param {ActivityUsageResults} results     Final details on the activation.
     * @protected
     */
    async _triggerSubsequentActions(config, results) {}

    /* -------------------------------------------- */
    /*  Rolling                                     */
    /* -------------------------------------------- */

    /**
     * Perform a damage roll.
     * @param {Partial<DamageRollProcessConfiguration>} config  Configuration information for the roll.
     * @param {Partial<BasicRollDialogConfiguration>} dialog    Configuration for the roll dialog.
     * @param {Partial<BasicRollMessageConfiguration>} message  Configuration for the roll message.
     * @returns {Promise<DamageRoll[]|void>}
     */
    async rollDamage(config={}, dialog={}, message={}) {
      const rollConfig = this.getDamageConfig(config);
      rollConfig.hookNames = [...(config.hookNames ?? []), "damage"];
      rollConfig.subject = this;

      const dialogConfig = foundry.utils.mergeObject({
        options: {
          position: {
            width: 400,
            top: config.event ? config.event.clientY - 80 : null,
            left: window.innerWidth - 710
          },
          window: {
            title: this.damageFlavor,
            subtitle: this.item.name,
            icon: this.item.img
          }
        }
      }, dialog);

      const messageConfig = foundry.utils.mergeObject({
        create: true,
        data: {
          flavor: `${this.item.name} - ${this.damageFlavor}`,
          flags: {
            dnd5e: {
              ...this.messageFlags,
              messageType: "roll",
              roll: { type: "damage" }
            }
          },
          speaker: ChatMessage.getSpeaker({ actor: this.actor })
        }
      }, message);

      let returnMultiple = rollConfig.returnMultiple ?? true;
      if ( "dnd5e.preRollDamage" in Hooks.events ) {
        foundry.utils.logCompatibilityWarning(
          "The `dnd5e.preRollDamage` hook has been deprecated and replaced with `dnd5e.preRollDamageV2`.",
          { since: "DnD5e 4.0", until: "DnD5e 4.4" }
        );
        const oldRollConfig = {
          actor: this.actor,
          rollConfigs: rollConfig.rolls.map((r, _index) => ({
            _index,
            parts: r.parts,
            type: r.options?.type,
            types: r.options?.types,
            properties: r.options?.properties
          })),
          data: rollConfig.rolls[0]?.data ?? {},
          event: rollConfig.event,
          returnMultiple,
          allowCritical: rollConfig.rolls[0]?.critical?.allow ?? rollConfig.critical?.allow ?? true,
          critical: rollConfig.rolls[0]?.isCritical,
          criticalBonusDice: rollConfig.rolls[0]?.critical?.bonusDice ?? rollConfig.critical?.bonusDice,
          criticalMultiplier: rollConfig.rolls[0]?.critical?.multiplier ?? rollConfig.critical?.multiplier,
          multiplyNumeric: rollConfig.rolls[0]?.critical?.multiplyNumeric ?? rollConfig.critical?.multiplyNumeric,
          powerfulCritical: rollConfig.rolls[0]?.critical?.powerfulCritical ?? rollConfig.critical?.powerfulCritical,
          criticalBonusDamage: rollConfig.rolls[0]?.critical?.bonusDamage ?? rollConfig.critical?.bonusDamage,
          title: `${this.item.name} - ${this.damageFlavor}`,
          dialogOptions: dialogConfig.options,
          chatMessage: messageConfig.create,
          messageData: messageConfig.data,
          rollMode: messageConfig.rollMode,
          flavor: messageConfig.data.flavor
        };
        if ( "configure" in dialogConfig ) oldRollConfig.fastForward = !dialogConfig.configure;
        if ( Hooks.call("dnd5e.preRollDamage", this.item, oldRollConfig) === false ) return;
        rollConfig.rolls = rollConfig.rolls.map((roll, index) => {
          const otherConfig = oldRollConfig.rollConfigs.find(r => r._index === index);
          if ( !otherConfig ) return null;
          roll.data = oldRollConfig.data;
          roll.parts = otherConfig.parts;
          roll.isCritical = oldRollConfig.critical;
          roll.options.type = otherConfig.type;
          roll.options.types = otherConfig.types;
          roll.options.properties = otherConfig.properties;
          return roll;
        }, [])
          .filter(_ => _)
          .concat(oldRollConfig.rollConfigs.filter(r => r._index === undefined));
        returnMultiple = oldRollConfig.returnMultiple;
        rollConfig.critical ??= {};
        rollConfig.critical.allow = oldRollConfig.allowCritical;
        if ( "fastForward" in oldRollConfig ) dialogConfig.configure = !oldRollConfig.fastForward;
        dialogConfig.options = oldRollConfig.dialogOptions;
        messageConfig.create = oldRollConfig.chatMessage;
        messageConfig.data = oldRollConfig.messageData;
        messageConfig.rollMode = oldRollConfig.rollMode;
        messageConfig.data.flavor = oldRollConfig.flavor;
      }

      const rolls = await CONFIG.Dice.DamageRoll.build(rollConfig, dialogConfig, messageConfig);
      if ( !rolls?.length ) return;
      const lastDamageTypes = rolls.reduce((obj, roll, index) => {
        if ( roll.options.type ) obj[index] = roll.options.type;
        return obj;
      }, {});
      if ( !foundry.utils.isEmpty(lastDamageTypes) && this.actor.items.has(this.item.id) ) {
        await this.item.setFlag("dnd5e", `last.${this.id}.damageType`, lastDamageTypes);
      }

      /**
       * A hook event that fires after damage has been rolled.
       * @function dnd5e.rollDamageV2
       * @memberof hookEvents
       * @param {DamageRoll[]} rolls       The resulting rolls.
       * @param {object} [data]
       * @param {Activity} [data.subject]  The activity that performed the roll.
       */
      Hooks.callAll("dnd5e.rollDamageV2", rolls, { subject: this });

      if ( "dnd5e.rollDamage" in Hooks.events ) {
        foundry.utils.logCompatibilityWarning(
          "The `dnd5e.rollDamage` hook has been deprecated and replaced with `dnd5e.rollDamageV2`.",
          { since: "DnD5e 4.0", until: "DnD5e 4.4" }
        );
        Hooks.callAll("dnd5e.rollDamage", this.item, returnMultiple ? rolls : rolls[0]);
      }

      return rolls;
    }

    /* -------------------------------------------- */
    /*  Event Listeners and Handlers                */
    /* -------------------------------------------- */

    /**
     * Activate listeners on a chat message.
     * @param {ChatMessage} message  Associated chat message.
     * @param {HTMLElement} html     Element in the chat log.
     */
    activateChatListeners(message, html) {
      html.addEventListener("click", event => {
        const target = event.target.closest("[data-action]");
        if ( target ) this.#onChatAction(event, target, message);
      });
    }

    /* -------------------------------------------- */

    /**
     * Construct context menu options for this Activity.
     * @returns {ContextMenuEntry[]}
     */
    getContextMenuOptions() {
      const entries = [];
      const compendiumLocked = this.item[game.release.generation < 13 ? "compendium" : "collection"]?.locked;

      if ( this.item.isOwner && !compendiumLocked ) {
        entries.push({
          name: "DND5E.ContextMenuActionEdit",
          icon: '<i class="fas fa-pen-to-square fa-fw"></i>',
          callback: () => this.sheet.render({ force: true })
        }, {
          name: "DND5E.ContextMenuActionDuplicate",
          icon: '<i class="fas fa-copy fa-fw"></i>',
          callback: () => {
            const createData = this.toObject();
            delete createData._id;
            this.item.createActivity(createData.type, createData, { renderSheet: false });
          }
        }, {
          name: "DND5E.ContextMenuActionDelete",
          icon: '<i class="fas fa-trash fa-fw"></i>',
          callback: () => this.deleteDialog()
        });
      } else {
        entries.push({
          name: "DND5E.ContextMenuActionView",
          icon: '<i class="fas fa-eye fa-fw"></i>',
          callback: () => this.sheet.render({ force: true })
        });
      }

      if ( "favorites" in (this.actor?.system ?? {}) ) {
        const uuid = `${this.item.getRelativeUUID(this.actor)}.Activity.${this.id}`;
        const isFavorited = this.actor.system.hasFavorite(uuid);
        entries.push({
          name: isFavorited ? "DND5E.FavoriteRemove" : "DND5E.Favorite",
          icon: '<i class="fas fa-bookmark fa-fw"></i>',
          condition: () => this.item.isOwner && !compendiumLocked,
          callback: () => {
            if ( isFavorited ) this.actor.system.removeFavorite(uuid);
            else this.actor.system.addFavorite({ type: "activity", id: uuid });
          },
          group: "state"
        });
      }

      return entries;
    }

    /* -------------------------------------------- */

    /**
     * Handle an action activated from an activity's chat message.
     * @param {PointerEvent} event     Triggering click event.
     * @param {HTMLElement} target     The capturing HTML element which defined a [data-action].
     * @param {ChatMessage5e} message  Message associated with the activation.
     */
    async #onChatAction(event, target, message) {
      const scaling = message.getFlag("dnd5e", "scaling") ?? 0;
      const item = scaling ? this.item.clone({ "flags.dnd5e.scaling": scaling }, { keepId: true }) : this.item;
      const activity = item.system.activities.get(this.id);

      const action = target.dataset.action;
      const handler = this.metadata.usage?.actions?.[action];
      target.disabled = true;
      try {
        if ( handler ) await handler.call(activity, event, target, message);
        else if ( action === "consumeResource" ) await this.#consumeResource(event, target, message);
        else if ( action === "refundResource" ) await this.#refundResource(event, target, message);
        else if ( action === "placeTemplate" ) await this.#placeTemplate();
        else await activity._onChatAction(event, target, message);
      } catch(err) {
        Hooks.onError("Activity#onChatAction", err, { log: "error", notify: "error" });
      } finally {
        target.disabled = false;
      }
    }

    /* -------------------------------------------- */

    /**
     * Handle an action activated from an activity's chat message. Action handlers in metadata are called first.
     * This method is only called for actions which have no defined handler.
     * @param {PointerEvent} event     Triggering click event.
     * @param {HTMLElement} target     The capturing HTML element which defined a [data-action].
     * @param {ChatMessage5e} message  Message associated with the activation.
     * @protected
     */
    async _onChatAction(event, target, message) {}

    /* -------------------------------------------- */

    /**
     * Handle context menu events on activities.
     * @param {Item5e} item         The Item the Activity belongs to.
     * @param {HTMLElement} target  The element the menu was triggered on.
     */
    static onContextMenu(item, target) {
      const { activityId } = target.closest("[data-activity-id]")?.dataset ?? {};
      const activity = item.system.activities?.get(activityId);
      if ( !activity ) return;
      const menuItems = activity.getContextMenuOptions();

      /**
       * A hook even that fires when the context menu for an Activity is opened.
       * @function dnd5e.getItemActivityContext
       * @memberof hookEvents
       * @param {Activity} activity             The Activity.
       * @param {HTMLElement} target            The element that menu was triggered on.
       * @param {ContextMenuEntry[]} menuItems  The context menu entries.
       */
      Hooks.callAll("dnd5e.getItemActivityContext", activity, target, menuItems);
      ui.context.menuItems = menuItems;
    }

    /* -------------------------------------------- */

    /**
     * Handle consuming resources from the chat card.
     * @param {PointerEvent} event     Triggering click event.
     * @param {HTMLElement} target     The capturing HTML element which defined a [data-action].
     * @param {ChatMessage5e} message  Message associated with the activation.
     */
    async #consumeResource(event, target, message) {
      const messageConfig = {};
      const scaling = message.getFlag("dnd5e", "scaling");
      const usageConfig = { consume: true, event, scaling };
      const linkedActivity = this.getLinkedActivity(message.getFlag("dnd5e", "use.cause"));
      if ( linkedActivity ) usageConfig.cause = {
        activity: linkedActivity.relativeUUID, resources: linkedActivity.consumption.targets.length > 0
      };
      await this.consume(usageConfig, messageConfig);
      if ( !foundry.utils.isEmpty(messageConfig.data) ) await message.update(messageConfig.data);
    }

    /* -------------------------------------------- */

    /**
     * Handle refunding consumption from a chat card.
     * @param {PointerEvent} event     Triggering click event.
     * @param {HTMLElement} target     The capturing HTML element which defined a [data-action].
     * @param {ChatMessage5e} message  Message associated with the activation.
     */
    async #refundResource(event, target, message) {
      const consumed = message.getFlag("dnd5e", "use.consumed");
      if ( !foundry.utils.isEmpty(consumed) ) {
        await this.refund(consumed);
        await message.unsetFlag("dnd5e", "use.consumed");
      }
    }

    /* -------------------------------------------- */

    /**
     * Handle placing a measured template in the scene.
     * @returns {MeasuredTemplateDocument[]}
     */
    async #placeTemplate() {
      const templates = [];
      try {
        for ( const template of AbilityTemplate.fromActivity(this) ) {
          const result = await template.drawPreview();
          if ( result ) templates.push(result);
        }
      } catch(err) {
        Hooks.onError("Activity#placeTemplate", err, {
          msg: game.i18n.localize("DND5E.TARGET.Warning.PlaceTemplate"),
          log: "error",
          notify: "error"
        });
      }
      return templates;
    }

    /* -------------------------------------------- */
    /*  Helpers                                     */
    /* -------------------------------------------- */

    /**
     * Prepare activity favorite data.
     * @returns {Promise<FavoriteData5e>}
     */
    async getFavoriteData() {
      return {
        img: this.img,
        title: this.name,
        subtitle: [this.labels.activation, this.labels.recovery],
        range: this.range,
        uses: { ...this.uses, name: "uses.value" }
      };
    }

    /* -------------------------------------------- */

    /**
     * Retrieve a linked activity based on the provided relative UUID, or the stored `cachedFor` value.
     * @param {string} relativeUUID  Relative UUID for an activity on this actor.
     * @returns {Activity|null}
     */
    getLinkedActivity(relativeUUID) {
      if ( !this.actor ) return null;
      relativeUUID ??= this.item.getFlag("dnd5e", "cachedFor");
      return fromUuidSync(relativeUUID, { relative: this.actor, strict: false });
    }

    /* -------------------------------------------- */

    /**
     * Prepare a data object which defines the data schema used by dice roll commands against this Activity.
     * @param {object} [options]
     * @param {boolean} [options.deterministic]  Whether to force deterministic values for data properties that could
     *                                           be either a die term or a flat term.
     * @returns {object}
     */
    getRollData(options) {
      const rollData = this.item.getRollData(options);
      rollData.activity = { ...this };
      rollData.mod = this.actor?.system.abilities?.[this.ability]?.mod ?? 0;
      return rollData;
    }

    /* -------------------------------------------- */

    /**
     * Merge the activity updates into this activity's item updates.
     * @param {ActivityUsageUpdates} updates
     * @internal
     */
    _mergeActivityUpdates(updates) {
      if ( foundry.utils.isEmpty(updates.activity) ) return;
      const itemIndex = updates.item.findIndex(i => i._id === this.item.id);
      const keyPath = `system.activities.${this.id}`;
      const activityUpdates = foundry.utils.expandObject(updates.activity);
      if ( itemIndex === -1 ) updates.item.push({ _id: this.item.id, [keyPath]: activityUpdates });
      else updates.item[itemIndex][keyPath] = activityUpdates;
    }
  }
  return Activity;
}

/**
 * Sheet for the attack activity.
 */
class AttackSheet extends ActivitySheet {

  /** @inheritDoc */
  static DEFAULT_OPTIONS = {
    classes: ["attack-activity"]
  };

  /* -------------------------------------------- */

  /** @inheritDoc */
  static PARTS = {
    ...super.PARTS,
    identity: {
      template: "systems/dnd5e/templates/activity/attack-identity.hbs",
      templates: [
        ...super.PARTS.identity.templates,
        "systems/dnd5e/templates/activity/parts/attack-identity.hbs"
      ]
    },
    effect: {
      template: "systems/dnd5e/templates/activity/attack-effect.hbs",
      templates: [
        ...super.PARTS.effect.templates,
        "systems/dnd5e/templates/activity/parts/attack-damage.hbs",
        "systems/dnd5e/templates/activity/parts/attack-details.hbs",
        "systems/dnd5e/templates/activity/parts/damage-part.hbs",
        "systems/dnd5e/templates/activity/parts/damage-parts.hbs"
      ]
    }
  };

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareEffectContext(context) {
    context = await super._prepareEffectContext(context);

    const availableAbilities = this.activity.availableAbilities;
    context.abilityOptions = [
      {
        value: "", label: game.i18n.format("DND5E.DefaultSpecific", {
          default: this.activity.attack.type.classification === "spell"
            ? game.i18n.localize("DND5E.Spellcasting").toLowerCase()
            : availableAbilities.size
              ? game.i18n.getListFormatter({ style: "short", type: "disjunction" }).format(
                Array.from(availableAbilities).map(a => CONFIG.DND5E.abilities[a].label.toLowerCase())
              )
              : game.i18n.localize("DND5E.None").toLowerCase()
        })
      },
      { rule: true },
      { value: "none", label: game.i18n.localize("DND5E.None") },
      { value: "spellcasting", label: game.i18n.localize("DND5E.Spellcasting") },
      ...Object.entries(CONFIG.DND5E.abilities).map(([value, config]) => ({
        value, label: config.label, group: game.i18n.localize("DND5E.Abilities")
      }))
    ];

    context.hasBaseDamage = this.item.system.offersBaseDamage;

    return context;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareIdentityContext(context) {
    context = await super._prepareIdentityContext(context);

    context.attackTypeOptions = Object.entries(CONFIG.DND5E.attackTypes)
      .map(([value, config]) => ({ value, label: config.label }));
    if ( this.item.system.validAttackTypes?.size ) context.attackTypeOptions.unshift({
      value: "",
      label: game.i18n.format("DND5E.DefaultSpecific", {
        default: game.i18n.getListFormatter({ type: "disjunction" }).format(
          Array.from(this.item.system.validAttackTypes).map(t => CONFIG.DND5E.attackTypes[t].label.toLowerCase())
        )
      })
    });

    context.attackClassificationOptions = Object.entries(CONFIG.DND5E.attackClassifications)
      .map(([value, config]) => ({ value, label: config.label }));
    if ( this.item.system.attackClassification ) context.attackClassificationOptions.unshift({
      value: "",
      label: game.i18n.format("DND5E.DefaultSpecific", {
        default: CONFIG.DND5E.attackClassifications[this.item.system.attackClassification].label.toLowerCase()
      })
    });

    return context;
  }
}

const { DiceTerm: DiceTerm$2 } = foundry.dice.terms;

/**
 * Dialog rendering options for a roll configuration dialog.
 *
 * @typedef {object} BasicRollConfigurationDialogOptions
 * @property {typeof BasicRoll} rollType              Roll type to use when constructing final roll.
 * @property {object} [default]
 * @property {number} [default.rollMode]              Default roll mode to have selected.
 * @property {RollBuildConfigCallback} [buildConfig]  Callback to handle additional build configuration.
 * @property {BasicRollConfigurationDialogRenderOptions} [rendering]
 */

/**
 * @callback RollBuildConfigCallback
 * @param {BasicRollProcessConfiguration} process  Configuration for the entire rolling process.
 * @param {BasicRollConfiguration} config          Configuration for a specific roll.
 * @param {FormDataExtended} [formData]            Any data entered into the rolling prompt.
 * @param {number} index                           Index of the roll within all rolls being prepared.
 */

/**
 * @typedef BasicRollConfigurationDialogRenderOptions
 * @property {object} [dice]
 * @property {number} [dice.max=5]               The maximum number of dice to display in the large dice breakdown. If
 *                                               the given rolls contain more dice than this, then the large breakdown
 *                                               is not shown.
 * @property {Set<string>} [dice.denominations]  Valid die denominations to display in the large dice breakdown. If any
 *                                               of the given rolls contain an invalid denomination, then the large
 *                                               breakdown is not shown.
 */

/**
 * Dialog for configuring one or more rolls.
 *
 * @param {BasicRollProcessConfiguration} [config={}]         Initial roll configuration.
 * @param {BasicRollMessageConfiguration} [message={}]        Message configuration.
 * @param {BasicRollConfigurationDialogOptions} [options={}]  Dialog rendering options.
 */
class RollConfigurationDialog extends Dialog5e {
  constructor(config={}, message={}, options={}) {
    super(options);

    this.#config = config;
    this.#message = message;
    this.#buildRolls(foundry.utils.deepClone(this.#config));
  }

  /* -------------------------------------------- */

  /** @override */
  static DEFAULT_OPTIONS = {
    classes: ["roll-configuration"],
    window: {
      title: "DND5E.RollConfiguration.Title",
      icon: "fa-solid fa-dice"
    },
    form: {
      handler: RollConfigurationDialog.#handleFormSubmission
    },
    position: {
      width: 400
    },
    buildConfig: null,
    rendering: {
      dice: {
        max: 5,
        denominations: new Set(["d4", "d6", "d8", "d10", "d12", "d20"])
      }
    }
  };

  /* -------------------------------------------- */

  /** @override */
  static PARTS = {
    formulas: {
      template: "systems/dnd5e/templates/dice/roll-formulas.hbs"
    },
    configuration: {
      template: "systems/dnd5e/templates/dice/roll-configuration.hbs"
    },
    buttons: {
      template: "systems/dnd5e/templates/dice/roll-buttons.hbs"
    }
  };

  /* -------------------------------------------- */

  /**
   * Roll type to use when constructing the rolls.
   * @type {typeof BasicRoll}
   */
  static get rollType() {
    return CONFIG.Dice.BasicRoll;
  }

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * Roll configuration.
   * @type {BasicRollProcessConfiguration}
   */
  #config;

  get config() {
    return this.#config;
  }

  /* -------------------------------------------- */

  /**
   * Configuration information for the roll message.
   * @type {BasicRollMessageConfiguration}
   */
  #message;

  get message() {
    return this.#message;
  }

  /* -------------------------------------------- */

  /**
   * The rolls being configured.
   * @type {BasicRoll[]}
   */
  #rolls;

  get rolls() {
    return this.#rolls;
  }

  /* -------------------------------------------- */

  /**
   * Roll type to use when constructing the rolls.
   * @type {typeof BasicRoll}
   */
  get rollType() {
    return this.options.rollType ?? this.constructor.rollType;
  }

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /**
   * Identify DiceTerms in this app's rolls.
   * @returns {{ icon: string, label: string }[]}
   * @protected
   */
  _identifyDiceTerms() {
    let dice = [];
    let shouldDisplay = true;

    /**
     * Determine if a given term is displayable.
     * @param {RollTerm} term  The term.
     * @returns {boolean|void}
     */
    const identifyTerm = term => {
      if ( !(term instanceof DiceTerm$2) ) return;
      // If any of the terms have complex components, do not attempt to display only some dice, bail out entirely.
      if ( !Number.isFinite(term.number) || !Number.isFinite(term.faces) ) return shouldDisplay = false;
      // If any of the terms are of an unsupported denomination, do not attempt to display only some dice, bail out
      // entirely.
      if ( !this.options.rendering.dice.denominations.has(term.denomination) ) return shouldDisplay = false;
      for ( let i = 0; i < term.number; i++ ) dice.push({
        icon: `systems/dnd5e/icons/svg/dice/${term.denomination}.svg`,
        label: term.denomination,
        denomination: term.denomination
      });
    };

    /**
     * Identify any DiceTerms in the given terms.
     * @param {RollTerm[]} terms  The terms.
     */
    const identifyDice = (terms=[]) => {
      for ( const term of terms ) {
        identifyTerm(term);
        if ( "dice" in term ) identifyDice(term.dice);
      }
    };

    this.rolls.forEach(roll => identifyDice(roll.terms));
    if ( dice.length > this.options.rendering.dice.max ) {
      // Compact dice display.
      const byDenom = dice.reduce((obj, { icon, denomination }) => {
        obj[denomination] ??= { icon, count: 0 };
        obj[denomination].count++;
        return obj;
      }, {});
      dice = Object.entries(byDenom).map(([d, { icon, count }]) => ({ icon, label: `${count}${d}` }));
      if ( dice.length > this.options.rendering.dice.max ) shouldDisplay = false;
    }
    else if ( !dice.length ) shouldDisplay = false;
    return shouldDisplay ? dice : [];
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _preparePartContext(partId, context, options) {
    context = await super._preparePartContext(partId, context, options);
    switch ( partId ) {
      case "buttons":
        return this._prepareButtonsContext(context, options);
      case "configuration":
        return this._prepareConfigurationContext(context, options);
      case "formulas":
        return this._prepareFormulasContext(context, options);
      default:
        return context;
    }
  }

  /* -------------------------------------------- */

  /**
   * Prepare the context for the buttons.
   * @param {ApplicationRenderContext} context  Shared context provided by _prepareContext.
   * @param {HandlebarsRenderOptions} options   Options which configure application rendering behavior.
   * @returns {Promise<ApplicationRenderContext>}
   * @protected
   */
  async _prepareButtonsContext(context, options) {
    context.buttons = {
      roll: {
        default: true,
        icon: '<i class="fa-solid fa-dice" inert></i>',
        label: game.i18n.localize("DND5E.Roll")
      }
    };
    return context;
  }

  /* -------------------------------------------- */

  /**
   * Prepare the context for the roll configuration section.
   * @param {ApplicationRenderContext} context  Shared context provided by _prepareContext.
   * @param {HandlebarsRenderOptions} options   Options which configure application rendering behavior.
   * @returns {Promise<ApplicationRenderContext>}
   * @protected
   */
  async _prepareConfigurationContext(context, options) {
    context.fields = [{
      field: new foundry.data.fields.StringField({ label: game.i18n.localize("DND5E.RollMode") }),
      name: "rollMode",
      value: this.message.rollMode ?? this.options.default?.rollMode ?? game.settings.get("core", "rollMode"),
      options: Object.entries(CONFIG.Dice.rollModes)
        .map(([value, l]) => ({ value, label: game.i18n.localize(`${game.release.generation < 13 ? l : l.label}`) }))
    }];
    return context;
  }

  /* -------------------------------------------- */

  /**
   * Prepare the context for the formulas list.
   * @param {ApplicationRenderContext} context  Shared context provided by _prepareContext.
   * @param {HandlebarsRenderOptions} options   Options which configure application rendering behavior.
   * @returns {Promise<ApplicationRenderContext>}
   * @protected
   */
  async _prepareFormulasContext(context, options) {
    context.rolls = this.rolls.map(roll => ({ roll }));
    context.dice = this._identifyDiceTerms() || [];
    return context;
  }

  /* -------------------------------------------- */
  /*  Roll Handling                               */
  /* -------------------------------------------- */

  /**
   * Build a roll from the provided configuration objects.
   * @param {BasicRollProcessConfiguration} config  Roll configuration data.
   * @param {FormDataExtended} [formData]           Any data entered into the rolling prompt.
   */
  #buildRolls(config, formData) {
    const RollType = this.rollType;
    this.#rolls = config.rolls?.map((config, index) =>
      RollType.fromConfig(this._buildConfig(config, formData, index), this.config)
    ) ?? [];
  }

  /* -------------------------------------------- */

  /**
   * Prepare individual configuration object before building a roll.
   * @param {BasicRollConfiguration} config  Roll configuration data.
   * @param {FormDataExtended} [formData]    Any data entered into the rolling prompt.
   * @param {number} index                   Index of the roll within all rolls being prepared.
   * @returns {BasicRollConfiguration}
   * @protected
   */
  _buildConfig(config, formData, index) {
    config = foundry.utils.mergeObject({ parts: [], data: {}, options: {} }, config);

    /**
     * A hook event that fires when a roll config is built using the roll prompt.
     * @function dnd5e.buildRollConfig
     * @memberof hookEvents
     * @param {RollConfigurationDialog} app    Roll configuration dialog.
     * @param {BasicRollConfiguration} config  Roll configuration data.
     * @param {FormDataExtended} [formData]    Any data entered into the rolling prompt.
     * @param {number} index                   Index of the roll within all rolls being prepared.
     */
    Hooks.callAll("dnd5e.buildRollConfig", this, config, formData, index);

    const situational = formData?.get(`roll.${index}.situational`);
    if ( situational && (config.situational !== false) ) {
      config.parts.push("@situational");
      config.data.situational = situational;
    } else {
      config.parts.findSplice(v => v === "@situational");
    }

    this.options.buildConfig?.(this.config, config, formData, index);

    return config;
  }

  /* -------------------------------------------- */

  /**
   * Make any final modifications to rolls based on the button clicked.
   * @param {string} action  Action on the button clicked.
   * @returns {BasicRoll[]}
   * @protected
   */
  _finalizeRolls(action) {
    return this.rolls;
  }

  /* -------------------------------------------- */

  /**
   * Rebuild rolls based on an updated config and re-render the dialog.
   */
  rebuild() {
    this._onChangeForm(this.options.form, new Event("change"));
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /**
   * Handle submission of the dialog using the form buttons.
   * @this {RollConfigurationDialog}
   * @param {Event|SubmitEvent} event    The form submission event.
   * @param {HTMLFormElement} form       The submitted form.
   * @param {FormDataExtended} formData  Data from the dialog.
   */
  static async #handleFormSubmission(event, form, formData) {
    if ( formData.has("rollMode") ) this.message.rollMode = formData.get("rollMode");
    this.#rolls = this._finalizeRolls(event.submitter?.dataset?.action);
    await this.close({ dnd5e: { submitted: true } });
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _onChangeForm(formConfig, event) {
    super._onChangeForm(formConfig, event);

    const formData = new FormDataExtended(this.form);
    if ( formData.has("rollMode") ) this.message.rollMode = formData.get("rollMode");
    this.#buildRolls(foundry.utils.deepClone(this.#config), formData);
    this.render({ parts: ["formulas"] });
  }

  /* -------------------------------------------- */

  /** @override */
  _onClose(options={}) {
    if ( !options.dnd5e?.submitted ) this.#rolls = [];
  }

  /* -------------------------------------------- */
  /*  Factory Methods                             */
  /* -------------------------------------------- */

  /**
   * A helper to handle displaying and responding to the dialog.
   * @param {BasicRollProcessConfiguration} [config]   Initial roll configuration.
   * @param {BasicRollDialogConfiguration} [dialog]    Dialog configuration options.
   * @param {BasicRollMessageConfiguration} [message]  Message configuration.
   * @returns {Promise<BasicRoll[]>}
   */
  static async configure(config={}, dialog={}, message={}) {
    return new Promise(resolve => {
      const app = new this(config, message, dialog.options);
      app.addEventListener("close", () => resolve(app.rolls), { once: true });
      app.render({ force: true });
    });
  }
}

/**
 * Dialog for configuring d20 rolls.
 *
 * @param {D20RollProcessConfiguration} [config={}]           Initial roll configuration.
 * @param {BasicRollMessageConfiguration} [message={}]        Message configuration.
 * @param {BasicRollConfigurationDialogOptions} [options={}]  Dialog rendering options.
 */
class D20RollConfigurationDialog extends RollConfigurationDialog {

  /** @override */
  static get rollType() {
    return CONFIG.Dice.D20Roll;
  }

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @override */
  async _prepareButtonsContext(context, options) {
    let defaultButton = this.options.defaultButton;
    if ( !defaultButton ) {
      let advantage = false;
      let disadvantage = false;
      for ( const roll of this.config.rolls ) {
        if ( !roll.options ) continue;
        if ( roll.options.advantageMode === CONFIG.Dice.D20Roll.ADV_MODE.ADVANTAGE ) advantage = true;
        else if ( roll.options.advantageMode === CONFIG.Dice.D20Roll.ADV_MODE.DISADVANTAGE ) disadvantage = true;
        else if ( roll.options.advantage && !roll.options.disadvantage ) advantage = true;
        else if ( !roll.options.advantage && roll.options.disadvantage ) disadvantage = true;
      }
      if ( advantage && !disadvantage ) defaultButton = "advantage";
      else if ( !advantage && disadvantage ) defaultButton = "disadvantage";
    }
    context.buttons = {
      advantage: {
        default: defaultButton === "advantage",
        label: game.i18n.localize("DND5E.Advantage")
      },
      normal: {
        default: !["advantage", "disadvantage"].includes(defaultButton),
        label: game.i18n.localize("DND5E.Normal")
      },
      disadvantage: {
        default: defaultButton === "disadvantage",
        label: game.i18n.localize("DND5E.Disadvantage")
      }
    };
    return context;
  }

  /* -------------------------------------------- */
  /*  Roll Handling                               */
  /* -------------------------------------------- */

  /** @override */
  _finalizeRolls(action) {
    let advantageMode = CONFIG.Dice.D20Roll.ADV_MODE.NORMAL;
    if ( action === "advantage" ) advantageMode = CONFIG.Dice.D20Roll.ADV_MODE.ADVANTAGE;
    else if ( action === "disadvantage" ) advantageMode = CONFIG.Dice.D20Roll.ADV_MODE.DISADVANTAGE;
    return this.rolls.map(roll => {
      roll.options.advantageMode = advantageMode;
      roll.configureModifiers();
      return roll;
    });
  }
}

/**
 * @typedef {BasicRollConfigurationDialogOptions} AttackRollConfigurationDialogOptions
 * @property {FormSelectOption[]} ammunitionOptions  Ammunition that can be used with the attack.
 * @property {FormSelectOption[]} attackModeOptions  Different modes of attack.
 * @property {FormSelectOption[]} masteryOptions     Available masteries for the attacking weapon.
 */

/**
 * Extended roll configuration dialog that allows selecting attack mode, ammunition, and weapon mastery.
 */
class AttackRollConfigurationDialog extends D20RollConfigurationDialog {
  /** @override */
  static DEFAULT_OPTIONS = {
    ammunitionOptions: [],
    attackModeOptions: [],
    masteryOptions: []
  };

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareConfigurationContext(context, options) {
    context = await super._prepareConfigurationContext(context, options);
    const optionsFields = [
      { key: "attackMode", label: "DND5E.ATTACK.Mode.Label", options: this.options.attackModeOptions },
      { key: "ammunition", label: "DND5E.CONSUMABLE.Type.Ammunition.Label", options: this.options.ammunitionOptions },
      { key: "mastery", label: "DND5E.WEAPON.Mastery.Label", options: this.options.masteryOptions }
    ];
    context.fields = [
      ...optionsFields.map(({ key, label, options }) => options.length ? {
        field: new foundry.data.fields.StringField({ label: game.i18n.localize(label) }),
        name: key,
        options,
        value: this.config[key]
      } : null).filter(_ => _),
      ...context.fields
    ];
    return context;
  }
}

/**
 * Lightweight class containing scaling information for an item that is used in roll data to ensure it is available
 * in the correct format in roll formulas: `@scaling` is the scaling value, and `@scaling.increase` as the scaling
 * steps above baseline.
 *
 * @param {number} increase  Scaling steps above baseline.
 */
class Scaling {
  constructor(increase) {
    this.#increase = increase;
  }

  /* -------------------------------------------- */

  /**
   * Scaling steps above baseline.
   * @type {number}
   */
  #increase;

  get increase() {
    return this.#increase;
  }

  /* -------------------------------------------- */

  /**
   * Value of the scaling starting 1.
   * @type {string}
   */
  get value() {
    return this.#increase + 1;
  }

  /* -------------------------------------------- */

  /** @override */
  toString() {
    return this.value;
  }
}

const { BooleanField: BooleanField$H, EmbeddedDataField: EmbeddedDataField$5, NumberField: NumberField$J, SchemaField: SchemaField$V, SetField: SetField$w, StringField: StringField$1a } = foundry.data.fields;

/**
 * Field for storing damage data.
 */
class DamageField extends EmbeddedDataField$5 {
  constructor(options) {
    super(DamageData, options);
  }
}

/* -------------------------------------------- */

/**
 * Data model that stores information on a single damage part.
 *
 * @property {number} number           Number of dice to roll.
 * @property {number} denomination     Die denomination to roll.
 * @property {string} bonus            Bonus added to the damage.
 * @property {Set<string>} types       One or more damage types. If multiple are selected, then the user will be able to
 *                                     select from those types.
 * @property {object} custom
 * @property {boolean} custom.enabled  Should the custom formula be used?
 * @property {string} custom.formula   Custom damage formula.
 * @property {object} scaling
 * @property {string} scaling.mode     How the damage scales in relation with levels.
 * @property {number} scaling.number   Number of dice to add per scaling level.
 * @property {string} scaling.formula  Arbitrary scaling formula which will be multiplied by scaling increase.
 */
class DamageData extends foundry.abstract.DataModel {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @override */
  static defineSchema() {
    return {
      number: new NumberField$J({ min: 0, integer: true }),
      denomination: new NumberField$J({ min: 0, integer: true }),
      bonus: new FormulaField(),
      types: new SetField$w(new StringField$1a()),
      custom: new SchemaField$V({
        enabled: new BooleanField$H(),
        formula: new FormulaField()
      }),
      scaling: new SchemaField$V({
        mode: new StringField$1a(),
        number: new NumberField$J({ initial: 1, min: 0, integer: true }),
        formula: new FormulaField()
      })
    };
  }

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * The default damage formula.
   * @type {string}
   */
  get formula() {
    if ( this.custom.enabled ) return this.custom.formula ?? "";
    return this._automaticFormula();
  }

  /* -------------------------------------------- */
  /*  Helpers                                     */
  /* -------------------------------------------- */

  /**
   * Produce the auto-generated formula from the `number`, `denomination`, and `bonus`.
   * @param {number} [increase=0]  Amount to increase the die count.
   * @returns {string}
   * @protected
   */
  _automaticFormula(increase=0) {
    let formula;
    const number = (this.number ?? 0) + increase;
    if ( number && this.denomination ) formula = `${number}d${this.denomination}`;
    if ( this.bonus ) formula = formula ? `${formula} + ${this.bonus}` : this.bonus;
    return formula ?? "";
  }

  /* -------------------------------------------- */

  /**
   * Scale the damage by a number of steps using its configured scaling configuration.
   * @param {number|Scaling} increase  Number of steps above base damage to scaling.
   * @returns {string}
   */
  scaledFormula(increase) {
    if ( increase instanceof Scaling ) increase = increase.increase;

    switch ( this.scaling.mode ) {
      case "whole": break;
      case "half": increase = Math.floor(increase * .5); break;
      default: increase = 0; break;
    }
    if ( !increase ) return this.formula;
    let formula;

    // If dice count scaling, increase the count on the first die rolled
    const dieIncrease = (this.scaling.number ?? 0) * increase;
    if ( this.custom.enabled ) {
      formula = this.custom.formula;
      formula = formula.replace(/^(\d)+d/, (match, number) => `${Number(number) + dieIncrease}d`);
    } else {
      formula = this._automaticFormula(dieIncrease);
    }

    // If custom scaling included, modify to match increase and append for formula
    if ( this.scaling.formula ) {
      let roll = new Roll(this.scaling.formula);
      roll = roll.alter(increase, 0, { multiplyNumeric: true });
      formula = formula ? `${formula} + ${roll.formula}` : roll.formula;
    }

    return formula;
  }

  /* -------------------------------------------- */

  /**
   * Step the die denomination up or down by a number of steps, sticking to proper die sizes. Will return `null` if
   * stepping reduced the denomination below minimum die size.
   * @param {number} [steps=1]  Number of steps to increase or decrease the denomination.
   * @returns {number|null}
   */
  steppedDenomination(steps=1) {
    return CONFIG.DND5E.dieSteps[Math.min(
      CONFIG.DND5E.dieSteps.indexOf(this.denomination) + steps,
      CONFIG.DND5E.dieSteps.length - 1
    )] ?? null;
  }
}

const { NumberField: NumberField$I, SchemaField: SchemaField$U, StringField: StringField$19 } = foundry.data.fields;

/**
 * Field for storing activation data.
 *
 * @property {string} type            Activation type (e.g. action, legendary action, minutes).
 * @property {number} value           Scalar value associated with the activation.
 * @property {string} condition       Condition required to activate this activity.
 */
class ActivationField extends SchemaField$U {
  constructor(fields={}, options={}) {
    fields = {
      type: new StringField$19({ initial: "action" }),
      value: new NumberField$I({ min: 0, integer: true }),
      condition: new StringField$19(),
      ...fields
    };
    super(fields, options);
  }

  /* -------------------------------------------- */
  /*  Data Preparation                            */
  /* -------------------------------------------- */

  /**
   * Prepare data for this field. Should be called during the `prepareFinalData` stage.
   * @this {ItemDataModel|BaseActivityData}
   * @param {object} rollData  Roll data used for formula replacements.
   * @param {object} [labels]  Object in which to insert generated labels.
   */
  static prepareData(rollData, labels) {
    this.activation.scalar = CONFIG.DND5E.activityActivationTypes[this.activation.type]?.scalar ?? false;
    if ( !this.activation.scalar ) this.activation.value = null;

    if ( labels && this.activation.type ) {
      labels.activation = [
        this.activation.value, CONFIG.DND5E.activityActivationTypes[this.activation.type]?.label
      ].filterJoin(" ");
    }
  }
}

const { SchemaField: SchemaField$T, StringField: StringField$18 } = foundry.data.fields;

/**
 * Field for storing duration data.
 *
 * @property {string} value             Scalar value for the activity's duration.
 * @property {string} units             Units that are used for the duration.
 * @property {string} special           Description of any special duration details.
 */
class DurationField extends SchemaField$T {
  constructor(fields={}, options={}) {
    fields = {
      value: new FormulaField({ deterministic: true }),
      units: new StringField$18({ initial: "inst" }),
      special: new StringField$18(),
      ...fields
    };
    super(fields, options);
  }

  /* -------------------------------------------- */
  /*  Data Preparation                            */
  /* -------------------------------------------- */

  /**
   * Prepare data for this field. Should be called during the `prepareFinalData` stage.
   * @this {ItemDataModel|BaseActivityData}
   * @param {object} rollData  Roll data used for formula replacements.
   * @param {object} [labels]  Object in which to insert generated labels.
   */
  static prepareData(rollData, labels) {
    this.duration.scalar = this.duration.units in CONFIG.DND5E.scalarTimePeriods;
    if ( this.duration.scalar ) {
      prepareFormulaValue(this, "duration.value", "DND5E.DURATION.FIELDS.duration.value.label", rollData);
    } else this.duration.value = null;

    if ( labels && this.duration.units ) {
      let duration = CONFIG.DND5E.timePeriods[this.duration.units] ?? "";
      if ( this.duration.value ) duration = `${this.duration.value} ${duration.toLowerCase()}`;
      labels.duration = duration;
      // TODO: Allow activities to indicate they require concentration regardless of the base item
      labels.concentrationDuration = this.properties?.has("concentration")
        ? game.i18n.format("DND5E.ConcentrationDuration", { duration }) : duration;
    }

    Object.defineProperty(this.duration, "getEffectData", {
      value: DurationField.getEffectDuration.bind(this.duration),
      configurable: true
    });
  }

  /* -------------------------------------------- */

  /**
   * Create duration data usable for an active effect based on this duration.
   * @this {DurationData}
   * @returns {EffectDurationData}
   */
  static getEffectDuration() {
    if ( !Number.isNumeric(this.value) ) return {};
    switch ( this.units ) {
      case "turn": return { turns: this.value };
      case "round": return { rounds: this.value };
      case "minute": return { seconds: this.value * 60 };
      case "hour": return { seconds: this.value * 60 * 60 };
      case "day": return { seconds: this.value * 60 * 60 * 24 };
      case "year": return { seconds: this.value * 60 * 60 * 24 * 365 };
      default: return {};
    }
  }
}

const { SchemaField: SchemaField$S, StringField: StringField$17 } = foundry.data.fields;

/**
 * Field for storing range data.
 *
 * @property {string} value                Scalar value for the activity's range.
 * @property {string} units                Units that are used for the range.
 * @property {string} special              Description of any special range details.
 */
class RangeField extends SchemaField$S {
  constructor(fields={}, options={}) {
    fields = {
      value: new FormulaField({ deterministic: true }),
      units: new StringField$17(),
      special: new StringField$17(),
      ...fields
    };
    super(fields, options);
  }

  /* -------------------------------------------- */
  /*  Data Preparation                            */
  /* -------------------------------------------- */

  /**
   * Prepare data for this field. Should be called during the `prepareFinalData` stage.
   * @this {ItemDataModel|BaseActivityData}
   * @param {object} rollData  Roll data used for formula replacements.
   * @param {object} [labels]  Object in which to insert generated labels.
   */
  static prepareData(rollData, labels) {
    this.range.scalar = this.range.units in CONFIG.DND5E.movementUnits;
    if ( this.range.scalar ) {
      prepareFormulaValue(this, "range.value", "DND5E.RANGE.FIELDS.range.value.label", rollData);
    } else this.range.value = null;

    if ( labels && this.range.units ) {
      if ( this.range.scalar && this.range.value ) {
        labels.range = formatLength(this.range.value, this.range.units);
        labels.rangeParts = formatLength(this.range.value, this.range.units, { parts: true });
      } else if ( !this.range.scalar ) {
        labels.range = CONFIG.DND5E.distanceUnits[this.range.units];
      }
    } else if ( labels ) labels.range = game.i18n.localize("DND5E.DistSelf");
  }
}

const { BooleanField: BooleanField$G, SchemaField: SchemaField$R, StringField: StringField$16 } = foundry.data.fields;

/**
 * @typedef {object} TargetData
 * @property {object} template
 * @property {string} template.count        Number of templates created.
 * @property {boolean} template.contiguous  Must all created areas be connected to one another?
 * @property {string} template.type         Type of area of effect caused by this activity.
 * @property {string} template.size         Size of the activity's area of effect on its primary axis.
 * @property {string} template.width        Width of line area of effect.
 * @property {string} template.height       Height of cylinder area of effect.
 * @property {string} template.units        Units used to measure the area of effect sizes.
 * @property {object} affects
 * @property {string} affects.count         Number of individual targets that can be affected.
 * @property {string} affects.type          Type of targets that can be affected (e.g. creatures, objects, spaces).
 * @property {boolean} affects.choice       When targeting an area, can the user choose who it affects?
 * @property {string} affects.special       Description of special targeting.
 */

/**
 * Field for storing target data.
 */
class TargetField extends SchemaField$R {
  constructor(fields={}, options={}) {
    fields = {
      template: new SchemaField$R({
        count: new FormulaField({ deterministic: true }),
        contiguous: new BooleanField$G(),
        type: new StringField$16(),
        size: new FormulaField({ deterministic: true }),
        width: new FormulaField({ deterministic: true }),
        height: new FormulaField({ deterministic: true }),
        units: new StringField$16({ initial: () => defaultUnits("length") })
      }),
      affects: new SchemaField$R({
        count: new FormulaField({ deterministic: true }),
        type: new StringField$16(),
        choice: new BooleanField$G(),
        special: new StringField$16()
      }),
      ...fields
    };
    super(fields, options);
  }

  /* -------------------------------------------- */
  /*  Data Preparation                            */
  /* -------------------------------------------- */

  /**
   * Prepare data for this field. Should be called during the `prepareFinalData` stage.
   * @this {ItemDataModel|BaseActivityData}
   * @param {object} rollData  Roll data used for formula replacements.
   * @param {object} [labels]  Object in which to insert generated labels.
   */
  static prepareData(rollData, labels) {
    this.target.affects.scalar = this.target.affects.type
      && (CONFIG.DND5E.individualTargetTypes[this.target.affects.type]?.scalar !== false);
    if ( this.target.affects.scalar ) {
      prepareFormulaValue(this, "target.affects.count", "DND5E.TARGET.FIELDS.target.affects.count.label", rollData);
    } else this.target.affects.count = null;

    this.target.template.dimensions = TargetField.templateDimensions(this.target.template.type);

    if ( this.target.template.type ) {
      this.target.template.count ||= "1";
      if ( this.target.template.dimensions.width ) this.target.template.width ||= "5";
      if ( this.target.template.dimensions.height ) this.target.template.height ||= "5";
      prepareFormulaValue(this, "target.template.count", "DND5E.TARGET.FIELDS.target.template.count.label", rollData);
      prepareFormulaValue(this, "target.template.size", "DND5E.TARGET.FIELDS.target.template.size.label", rollData);
      prepareFormulaValue(this, "target.template.width", "DND5E.TARGET.FIELDS.target.template.width.label", rollData);
      prepareFormulaValue(this, "target.template.height", "DND5E.TARGET.FIELDS.target.template.height.label", rollData);
    } else {
      this.target.template.count = null;
      this.target.template.size = null;
      this.target.template.width = null;
      this.target.template.height = null;
    }

    const pr = getPluralRules();

    // Generate the template label
    const templateConfig = CONFIG.DND5E.areaTargetTypes[this.target.template.type];
    if ( templateConfig ) {
      const parts = [];
      if ( this.target.template.count > 1 ) parts.push(`${this.target.template.count} ×`);
      if ( this.target.template.units in CONFIG.DND5E.movementUnits ) {
        parts.push(formatLength(this.target.template.size, this.target.template.units));
      }
      this.target.template.label = game.i18n.format(
        `${templateConfig.counted}.${pr.select(this.target.template.count || 1)}`, { number: parts.filterJoin(" ") }
      ).trim().capitalize();
    } else this.target.template.label = "";

    // Generate the affects label
    const affectsConfig = CONFIG.DND5E.individualTargetTypes[this.target.affects.type];
    this.target.affects.labels = {
      sheet: affectsConfig?.counted ? game.i18n.format(
        `${affectsConfig.counted}.${this.target.affects.count ? pr.select(this.target.affects.count) : "other"}`, {
          number: this.target.affects.count ? formatNumber(this.target.affects.count)
            : game.i18n.localize(`DND5E.TARGET.Count.${this.target.template.type ? "Every" : "Any"}`)
        }
      ).trim().capitalize() : (affectsConfig?.label ?? ""),
      statblock: game.i18n.format(
        `${affectsConfig?.counted ?? "DND5E.TARGET.Type.Target.Counted"}.${pr.select(this.target.affects.count || 1)}`,
        { number: formatNumber(this.target.affects.count || 1, { words: true }) }
      )
    };

    if ( labels ) labels.target = this.target.template.label || this.target.affects.labels.sheet;
  }

  /* -------------------------------------------- */

  /**
   * Create the template dimensions labels for a template type.
   * @param {string} type  Area of effect type.
   * @returns {{ size: string, [width]: string, [height]: string }}
   */
  static templateDimensions(type) {
    const sizes = CONFIG.DND5E.areaTargetTypes[type]?.sizes;
    const dimensions = { size: "DND5E.AreaOfEffect.Size.Label" };
    if ( sizes ) {
      dimensions.width = sizes.includes("width") && (sizes.includes("length") || sizes.includes("radius"));
      dimensions.height = sizes.includes("height");
      if ( sizes.includes("radius") ) dimensions.size = "DND5E.AreaOfEffect.Size.Radius";
      else if ( sizes.includes("length") ) dimensions.size = "DND5E.AreaOfEffect.Size.Length";
      else if ( sizes.includes("width") ) dimensions.size = "DND5E.AreaOfEffect.Size.Width";
      if ( sizes.includes("thickness") ) dimensions.width = "DND5E.AreaOfEffect.Size.Thickness";
      else if ( dimensions.width ) dimensions.width = "DND5E.AreaOfEffect.Size.Width";
      if ( dimensions.height ) dimensions.height = "DND5E.AreaOfEffect.Size.Height";
    }
    return dimensions;
  }
}

const { ArrayField: ArrayField$o, NumberField: NumberField$H, SchemaField: SchemaField$Q, StringField: StringField$15 } = foundry.data.fields;

/**
 * @import {
 *   BasicRollProcessConfiguration, BasicRollDialogConfiguration, BasicRollMessageConfiguration
 * } from "../../dice/basic-roll.mjs";
 */

/**
 * @typedef {object} UsesData
 * @property {number} spent                 Number of uses that have been spent.
 * @property {string} max                   Formula for the maximum number of uses.
 * @property {UsesRecoveryData[]} recovery  Recovery profiles for this activity's uses.
 */

/**
 * Data for a recovery profile for an activity's uses.
 *
 * @typedef {object} UsesRecoveryData
 * @property {string} period   Period at which this profile is activated.
 * @property {string} type     Whether uses are reset to full, reset to zero, or recover a certain number of uses.
 * @property {string} formula  Formula used to determine recovery if type is not reset.
 */

/**
 * Field for storing uses data.
 */
class UsesField extends SchemaField$Q {
  constructor(fields={}, options={}) {
    fields = {
      spent: new NumberField$H({ initial: 0, min: 0, integer: true }),
      max: new FormulaField({ deterministic: true }),
      recovery: new ArrayField$o(
        new SchemaField$Q({
          period: new StringField$15({ initial: "lr" }),
          type: new StringField$15({ initial: "recoverAll" }),
          formula: new FormulaField()
        })
      ),
      ...fields
    };
    super(fields, options);
  }

  /* -------------------------------------------- */
  /*  Data Preparation                            */
  /* -------------------------------------------- */

  /**
   * Prepare data for this field. Should be called during the `prepareFinalData` stage.
   * @this {ItemDataModel|BaseActivityData}
   * @param {object} rollData  Roll data used for formula replacements.
   * @param {object} [labels]  Object in which to insert generated labels.
   */
  static prepareData(rollData, labels) {
    prepareFormulaValue(this, "uses.max", "DND5E.USES.FIELDS.uses.max.label", rollData);
    this.uses.value = this.uses.max ? Math.clamp(this.uses.max - this.uses.spent, 0, this.uses.max) : 0;

    const periods = [];
    for ( const recovery of this.uses.recovery ) {
      if ( recovery.period === "recharge" ) {
        recovery.formula ??= "6";
        recovery.type = "recoverAll";
        recovery.recharge = {
          options: Array.fromRange(5, 2).reverse().map(min => ({
            value: min,
            label: game.i18n.format("DND5E.USES.Recovery.Recharge.Range", {
              range: min === 6 ? formatNumber(6) : formatRange(min, 6)
            })
          }))
        };
        if ( labels ) labels.recharge ??= `${game.i18n.localize("DND5E.Recharge")} [${
          recovery.formula}${parseInt(recovery.formula) < 6 ? "+" : ""}]`;
      } else if ( recovery.period in CONFIG.DND5E.limitedUsePeriods ) {
        const config = CONFIG.DND5E.limitedUsePeriods[recovery.period];
        periods.push(config.abbreviation ?? config.label);
      }
    }
    if ( labels ) labels.recovery = game.i18n.getListFormatter({ style: "narrow" }).format(periods);

    this.uses.label = UsesField.getStatblockLabel.call(this);

    Object.defineProperty(this.uses, "rollRecharge", {
      value: UsesField.rollRecharge.bind(this.parent?.system ? this.parent : this),
      configurable: true
    });
  }

  /* -------------------------------------------- */

  /**
   * Create a label for uses data that matches the style seen on NPC stat blocks. Complex recovery data might result
   * in no label being generated if it doesn't represent recovery that can be normally found on a NPC.
   * @this {ItemDataModel|BaseActivityData}
   * @returns {string}
   */
  static getStatblockLabel() {
    if ( !this.uses.max || (this.uses.recovery.length !== 1) ) return "";
    const recovery = this.uses.recovery[0];

    // Recharge X–Y
    if ( recovery.period === "recharge" ) {
      const value = parseInt(recovery.formula);
      return `${game.i18n.localize("DND5E.Recharge")} ${value === 6 ? "6" : `${value}–6`}`;
    }

    // Recharge after a Short or Long Rest
    if ( ["lr", "sr"].includes(recovery.period) && (this.uses.max === 1) ) {
      return game.i18n.localize(`DND5E.Recharge${recovery.period === "sr" ? "Short" : "Long"}`);
    }

    // X/Day
    const period = CONFIG.DND5E.limitedUsePeriods[recovery.period === "sr" ? "sr" : "day"]?.label ?? "";
    if ( !period ) return "";
    return `${this.uses.max}/${period}`;
  }

  /* -------------------------------------------- */
  /*  Helpers                                     */
  /* -------------------------------------------- */

  /**
   * Determine uses recovery.
   * @this {ItemDataModel|BaseActivityData}
   * @param {string[]} periods  Recovery periods to check.
   * @param {object} rollData   Roll data to use when evaluating recover formulas.
   * @returns {Promise<{ updates: object, rolls: BasicRoll[] }|false>}
   */
  static async recoverUses(periods, rollData) {
    if ( !this.uses?.recovery.length ) return false;

    // Search the recovery profiles in order to find the first matching period,
    // and then find the first profile that uses that recovery period
    let profile;
    findPeriod: {
      for ( const period of periods ) {
        for ( const recovery of this.uses.recovery ) {
          if ( recovery.period === period ) {
            profile = recovery;
            break findPeriod;
          }
        }
      }
    }
    if ( !profile ) return false;

    const updates = {};
    const rolls = [];
    const item = this.item ?? this.parent;

    if ( profile.type === "recoverAll" ) updates.spent = 0;
    else if ( profile.type === "loseAll" ) updates.spent = this.uses.max;
    else if ( profile.formula ) {
      let roll;
      let total;
      try {
        const delta = this.parent instanceof Item ? { item: this.parent.id, keyPath: "system.uses.spent" }
          : { item: this.item.id, keyPath: `system.activities.${this.id}.uses.spent` };
        roll = new CONFIG.Dice.BasicRoll(profile.formula, rollData, { delta });
        if ( ["day", "dawn", "dusk"].includes(profile.period)
          && (game.settings.get("dnd5e", "restVariant") === "gritty") ) {
          roll.alter(7, 0, { multiplyNumeric: true });
        }
        total = (await roll.evaluate()).total;
      } catch(err) {
        Hooks.onError("UsesField#recoverUses", err, {
          msg: game.i18n.format("DND5E.ItemRecoveryFormulaWarning", {
            name: item.name, formula: profile.formula, uuid: this.uuid ?? item.uuid
          }),
          log: "error",
          notify: "error"
        });
        return false;
      }

      const newSpent = Math.clamp(this.uses.spent - total, 0, this.uses.max);
      if ( newSpent !== this.uses.spent ) {
        updates.spent = newSpent;
        if ( !roll.isDeterministic ) rolls.push(roll);
      }
    }

    return { updates, rolls };
  }

  /* -------------------------------------------- */

  /**
   * @typedef {BasicRollProcessConfiguration} RechargeRollProcessConfiguration
   * @property {boolean} [apply]  Apply the uses updates back to the item or activity. If set to `false`, then the
   *                              `dnd5e.postRollRecharge` hook won't be called.
   */

  /**
   * Rolls a recharge test for an Item or Activity that uses the d6 recharge mechanic.
   * @this {Item5e|Activity}
   * @param {RechargeRollProcessConfiguration} config  Configuration information for the roll.
   * @param {BasicRollDialogConfiguration} dialog      Configuration for the roll dialog.
   * @param {BasicRollMessageConfiguration} message    Configuration for the roll message.
   * @returns {Promise<BasicRoll[]|{ rolls: BasicRoll[], updates: object }|void>}  The created Roll instances, update
   *                                                                               data, or nothing if not rolled.
   */
  static async rollRecharge(config={}, dialog={}, message={}) {
    const uses = this.system ? this.system.uses : this.uses;
    const recharge = uses?.recovery.find(({ period }) => period === "recharge");
    if ( !recharge || !uses?.spent ) return;

    let oldReturn = false;
    if ( config.apply === undefined ) {
      foundry.utils.logCompatibilityWarning(
        "The `apply` parameter should be passed to `rollRecharge` to opt-in to the new return behavior.",
        { since: "DnD5e 4.3", until: "DnD5e 5.0" }
      );
      oldReturn = config.apply = true;
    }

    const rollConfig = foundry.utils.mergeObject({
      rolls: [{
        parts: ["1d6"],
        data: this.getRollData(),
        options: {
          delta: this instanceof Item ? { item: this.id, keyPath: "system.uses.spent" }
            : { item: this.item.id, keyPath: `system.activities.${this.id}.uses.spent` },
          target: parseInt(recharge.formula)
        }
      }]
    }, config);
    rollConfig.hookNames = [...(config.hookNames ?? []), "recharge"];
    rollConfig.subject = this;

    const dialogConfig = foundry.utils.mergeObject({ configure: false }, dialog);

    const messageConfig = foundry.utils.mergeObject({
      create: true,
      data: {
        speaker: ChatMessage.getSpeaker({ actor: this.actor, token: this.actor.token })
      },
      rollMode: game.settings.get("core", "rollMode")
    }, message);

    if ( "dnd5e.preRollRecharge" in Hooks.events ) {
      foundry.utils.logCompatibilityWarning(
        "The `dnd5e.preRollRecharge` hook has been deprecated and replaced with `dnd5e.preRollRechargeV2`.",
        { since: "DnD5e 4.0", until: "DnD5e 4.4" }
      );
      const hookData = {
        formula: rollConfig.rolls[0].parts[0], data: rollConfig.rolls[0].data,
        target: rollConfig.rolls[0].options.target, chatMessage: messageConfig.create
      };
      if ( Hooks.call("dnd5e.preRollRecharge", this, hookData) === false ) return;
      rollConfig.rolls[0].parts[0] = hookData.formula;
      rollConfig.rolls[0].data = hookData.data;
      rollConfig.rolls[0].options.target = hookData.target;
      messageConfig.create = hookData.chatMessage;
    }

    const rolls = await CONFIG.Dice.BasicRoll.buildConfigure(rollConfig, dialogConfig, messageConfig);
    await CONFIG.Dice.BasicRoll.buildEvaluate(rolls, rollConfig, messageConfig);
    if ( !rolls.length ) return;
    messageConfig.data.flavor = game.i18n.format("DND5E.ItemRechargeCheck", {
      name: this.name,
      result: game.i18n.localize(`DND5E.ItemRecharge${rolls[0].isSuccess ? "Success" : "Failure"}`)
    });
    await CONFIG.Dice.BasicRoll.buildPost(rolls, rollConfig, messageConfig);

    const updates = {};
    if ( rolls[0].isSuccess ) {
      if ( this instanceof Item ) updates["system.uses.spent"] = 0;
      else updates["uses.spent"] = 0;
    }

    /**
     * A hook event that fires after an Item or Activity has rolled to recharge, but before any usage changes have
     * been made.
     * @function dnd5e.rollRechargeV2
     * @memberof hookEvents
     * @param {BasicRoll[]} rolls             The resulting rolls.
     * @param {object} data
     * @param {Item5e|Activity} data.subject  Item or Activity for which the roll was performed.
     * @param {object} data.updates           Updates to be applied to the subject.
     * @returns {boolean}                     Explicitly return `false` to prevent updates from being performed.
     */
    if ( Hooks.call("dnd5e.rollRechargeV2", rolls, { subject: this, updates }) === false ) return rolls;

    if ( "dnd5e.rollRecharge" in Hooks.events ) {
      foundry.utils.logCompatibilityWarning(
        "The `dnd5e.rollRecharge` hook has been deprecated and replaced with `dnd5e.rollRechargeV2`.",
        { since: "DnD5e 4.0", until: "DnD5e 4.4" }
      );
      if ( Hooks.call("dnd5e.rollRecharge", this, rolls[0]) === false ) return rolls;
    }

    if ( rollConfig.apply && !foundry.utils.isEmpty(updates) ) await this.update(updates);

    /**
     * A hook event that fires after an Item or Activity has rolled recharge and usage updates have been performed.
     * @function dnd5e.postRollRecharge
     * @memberof hookEvents
     * @param {BasicRoll[]} rolls     The resulting rolls.
     * @param {object} data
     * @param {Actor5e} data.subject  Item or Activity for which the roll was performed.
     */
    Hooks.callAll("dnd5e.postRollRecharge", rolls, { subject: this });

    return oldReturn ? rolls : { rolls, updates };
  }
}

const { DocumentIdField: DocumentIdField$a, SchemaField: SchemaField$P } = foundry.data.fields;

/**
 * Field for storing an active effects applied by an activity.
 *
 * @property {string} _id  ID of the effect to apply.
 */
class AppliedEffectField extends SchemaField$P {
  constructor(fields={}, options={}) {
    super({
      _id: new DocumentIdField$a(),
      ...fields
    }, options);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  initialize(value, model, options={}) {
    const obj = super.initialize(value, model, options);
    const item = model.item;

    Object.defineProperty(obj, "effect", {
      get() { return item?.effects.get(this._id); },
      configurable: true
    });

    return obj;
  }
}

const {
  ArrayField: ArrayField$n, BooleanField: BooleanField$F, DocumentIdField: DocumentIdField$9, FilePathField: FilePathField$2, IntegerSortField: IntegerSortField$2, SchemaField: SchemaField$O, StringField: StringField$14
} = foundry.data.fields;

/**
 * Data for effects that can be applied.
 *
 * @typedef {object} EffectApplicationData
 * @property {string} _id  ID of the effect to apply.
 */

/**
 * Data model for activities.
 *
 * @property {string} _id                        Unique ID for the activity on an item.
 * @property {string} type                       Type name of the activity used to build a specific activity class.
 * @property {string} name                       Name for this activity.
 * @property {string} img                        Image that represents this activity.
 * @property {ActivationField} activation        Activation time & conditions.
 * @property {boolean} activation.override       Override activation values inferred from item.
 * @property {object} consumption
 * @property {object} consumption.scaling
 * @property {boolean} consumption.scaling.allowed          Can this non-spell activity be activated at higher levels?
 * @property {string} consumption.scaling.max               Maximum number of scaling levels for this item.
 * @property {boolean} consumption.spellSlot                If this is on a spell, should it consume a spell slot?
 * @property {ConsumptionTargetData[]} consumption.targets  Collection of consumption targets.
 * @property {object} description
 * @property {string} description.chatFlavor     Extra text displayed in the activation chat message.
 * @property {DurationField} duration            Duration of the effect.
 * @property {boolean} duration.concentration    Does this effect require concentration?
 * @property {boolean} duration.override         Override duration values inferred from item.
 * @property {EffectApplicationData[]} effects   Linked effects that can be applied.
 * @property {object} range
 * @property {boolean} range.override            Override range values inferred from item.
 * @property {TargetData} target
 * @property {boolean} target.override           Override target values inferred from item.
 * @property {boolean} target.prompt             Should the player be prompted to place the template?
 * @property {UsesData} uses                     Uses available to this activity.
 */
class BaseActivityData extends foundry.abstract.DataModel {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @override */
  static LOCALIZATION_PREFIXES = [
    "DND5E.ACTIVITY", "DND5E.ACTIVATION", "DND5E.CONSUMPTION",
    "DND5E.DURATION", "DND5E.RANGE", "DND5E.TARGET", "DND5E.USES"
  ];

  /* -------------------------------------------- */

  /** @inheritDoc */
  static defineSchema() {
    return {
      _id: new DocumentIdField$9({ initial: () => foundry.utils.randomID() }),
      type: new StringField$14({
        blank: false, required: true, readOnly: true, initial: () => this.metadata.type
      }),
      name: new StringField$14({ initial: undefined }),
      img: new FilePathField$2({ initial: undefined, categories: ["IMAGE"], base64: false }),
      sort: new IntegerSortField$2(),
      activation: new ActivationField({
        override: new BooleanField$F()
      }),
      consumption: new SchemaField$O({
        scaling: new SchemaField$O({
          allowed: new BooleanField$F(),
          max: new FormulaField({ deterministic: true })
        }),
        spellSlot: new BooleanField$F({ initial: true }),
        targets: new ConsumptionTargetsField()
      }),
      description: new SchemaField$O({
        chatFlavor: new StringField$14()
      }),
      duration: new DurationField({
        concentration: new BooleanField$F(),
        override: new BooleanField$F()
      }),
      effects: new ArrayField$n(new AppliedEffectField()),
      range: new RangeField({
        override: new BooleanField$F()
      }),
      target: new TargetField({
        override: new BooleanField$F(),
        prompt: new BooleanField$F({ initial: true })
      }),
      uses: new UsesField()
    };
  }

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * The primary ability for this activity that will be available as `@mod` in roll data.
   * @type {string|null}
   */
  get ability() {
    return this.isSpell ? this.spellcastingAbility : null;
  }

  /* -------------------------------------------- */

  /**
   * Helper property to translate this activity type into the old `actionType`.
   * @type {string}
   */
  get actionType() {
    return this.metadata.data;
  }

  /* -------------------------------------------- */

  /**
   * A specific set of activation-specific labels displayed in chat cards.
   * @type {object|null}
   */
  get activationLabels() {
    if ( !this.activation.type || this.isSpell ) return null;
    const { activation, duration, range, reach, target } = this.labels;
    return { activation, duration, range, reach, target };
  }

  /* -------------------------------------------- */

  /**
   * Effects that can be applied from this activity.
   * @type {ActiveEffect5e[]|null}
   */
  get applicableEffects() {
    return this.effects?.map(e => e.effect).filter(e => e) ?? null;
  }

  /* -------------------------------------------- */

  /**
   * Can consumption scaling be configured?
   * @type {boolean}
   */
  get canConfigureScaling() {
    return this.consumption.scaling.allowed || (this.isSpell && (this.item.system.level > 0));
  }

  /* -------------------------------------------- */

  /**
   * Is scaling possible with this activity?
   * @type {boolean}
   */
  get canScale() {
    return this.consumption.scaling.allowed || (this.isSpell && this.item.system.level > 0
      && CONFIG.DND5E.spellPreparationModes[this.item.system.preparation.mode]?.upcast);
  }

  /* -------------------------------------------- */

  /**
   * Can this activity's damage be scaled?
   * @type {boolean}
   */
  get canScaleDamage() {
    return this.consumption.scaling.allowed || this.isScaledScroll || this.isSpell;
  }

  /* -------------------------------------------- */

  /**
   * Is this activity on a spell scroll that is scaled.
   * @type {boolean}
   */
  get isScaledScroll() {
    return !!this.item.getFlag("dnd5e", "spellLevel");
  }

  /* -------------------------------------------- */

  /**
   * Is this activity on a spell?
   * @type {boolean}
   */
  get isSpell() {
    return this.item.type === "spell";
  }

  /* -------------------------------------------- */

  /**
   * Does this activity or its item require concentration?
   * @type {boolean}
   */
  get requiresConcentration() {
    return this.duration.concentration;
  }

  /* -------------------------------------------- */

  /**
   * Does activating this activity consume a spell slot?
   * @type {boolean}
   */
  get requiresSpellSlot() {
    if ( !this.isSpell || !this.actor?.system.spells ) return false;
    return this.canScale;
  }

  /* -------------------------------------------- */

  /**
   * Retrieve the spellcasting ability that can be used with this activity.
   * @type {string|null}
   */
  get spellcastingAbility() {
    let ability;
    if ( this.isSpell ) ability = this.item.system.availableAbilities?.first();
    return ability ?? this.actor?.system.attributes?.spellcasting ?? null;
  }

  /* -------------------------------------------- */
  /*  Data Migrations                             */
  /* -------------------------------------------- */

  /**
   * Static ID used for the automatically generated activity created during migration.
   * @type {string}
   */
  static INITIAL_ID = staticID("dnd5eactivity");

  /* -------------------------------------------- */

  /**
   * Migrate data from the item to a newly created activity.
   * @param {object} source              Item's candidate source data.
   * @param {object} [options={}]
   * @param {number} [options.offset=0]  Adjust the default ID using this number when creating multiple activities.
   */
  static createInitialActivity(source, { offset=0, ...options }={}) {
    const activityData = this.transformTypeData(source, {
      _id: this.INITIAL_ID.replace("0", offset),
      type: this.metadata.type,
      activation: this.transformActivationData(source, options),
      consumption: this.transformConsumptionData(source, options),
      description: this.transformDescriptionData(source, options),
      duration: this.transformDurationData(source, options),
      effects: this.transformEffectsData(source, options),
      range: this.transformRangeData(source, options),
      target: this.transformTargetData(source, options),
      uses: this.transformUsesData(source, options)
    }, options);
    foundry.utils.setProperty(source, `system.activities.${activityData._id}`, activityData);
    foundry.utils.setProperty(source, "flags.dnd5e.persistSourceMigration", true);
  }

  /* -------------------------------------------- */

  /**
   * Fetch data from the item source and transform it into an activity's activation object.
   * @param {object} source   Item's candidate source data to transform.
   * @param {object} options  Additional options passed to the creation process.
   * @returns {object}        Creation data for new activity.
   */
  static transformActivationData(source, options) {
    if ( source.type === "spell" ) return {};
    return {
      type: source.system.activation?.type === "none" ? "" : (source.system.activation?.type ?? ""),
      value: source.system.activation?.cost ?? null,
      condition: source.system.activation?.condition ?? ""
    };
  }

  /* -------------------------------------------- */

  /**
   * Fetch data from the item source and transform it into an activity's consumption object.
   * @param {object} source   Item's candidate source data to transform.
   * @param {object} options  Additional options passed to the creation process.
   * @returns {object}        Creation data for new activity.
   */
  static transformConsumptionData(source, options) {
    const targets = [];

    const type = {
      attribute: "attribute",
      hitDice: "hitDice",
      material: "material",
      charges: "itemUses"
    }[source.system.consume?.type];

    if ( type ) targets.push({
      type,
      target: source.system.consume?.target ?? "",
      value: source.system.consume?.amount ?? "1",
      scaling: {
        mode: source.system.consume?.scale ? "amount" : "",
        formula: ""
      }
    });

    // If no target type set but this item has max uses, set consumption type to itemUses with blank target
    else if ( source.system.uses?.max ) targets.push({
      type: "itemUses",
      target: "",
      value: "1",
      scaling: {
        mode: source.system.consume?.scale ? "amount" : "",
        formula: ""
      }
    });

    if ( source.system.recharge?.value && source.system.uses?.per ) targets.push({
      type: source.system.uses?.max ? "activityUses" : "itemUses",
      target: "",
      value: "1",
      scaling: { mode: "", formula: "" }
    });

    return {
      targets,
      scaling: {
        allowed: source.system.consume?.scale ?? false,
        max: ""
      }
    };
  }

  /* -------------------------------------------- */

  /**
   * Transform an old damage part into the new damage part format.
   * @param {object} source  Item's candidate source data to transform.
   * @param {string[]} part  The damage part to transform.
   * @returns {object}       Creation data for new activity.
   */
  static transformDamagePartData(source, [formula, type]) {
    const data = {
      number: null,
      denomination: null,
      bonus: "",
      types: type ? [type] : [],
      custom: {
        enabled: false,
        formula: ""
      },
      scaling: {
        mode: source?.system.scaling?.mode !== "none" ? "whole" : "",
        number: null,
        formula: source?.system.scaling?.formula ?? ""
      }
    };

    const parsed = (formula ?? "").match(/^\s*(\d+)d(\d+)(?:\s*([+|-])\s*(@?[\w\d.-]+))?\s*$/i);
    if ( parsed && CONFIG.DND5E.dieSteps.includes(Number(parsed[2])) ) {
      data.number = Number(parsed[1]);
      data.denomination = Number(parsed[2]);
      if ( parsed[4] ) data.bonus = parsed[3] === "-" ? `-${parsed[4]}` : parsed[4];
    } else if ( formula ) {
      data.custom.enabled = true;
      data.custom.formula = formula;
    }

    // If scaling denomination matches the damage denomination, set scaling using number rather than formula
    const scaling = data.scaling.formula.match(/^\s*(\d+)d(\d+)\s*$/i);
    if ( (scaling && (Number(scaling[2]) === data.denomination)) || (source.system.scaling?.mode === "cantrip") ) {
      data.scaling.number = Number(scaling?.[1] || 1);
      data.scaling.formula = "";
    }

    return data;
  }

  /* -------------------------------------------- */

  /**
   * Fetch data from the item source and transform it into an activity's description object.
   * @param {object} source   Item's candidate source data to transform.
   * @param {object} options  Additional options passed to the creation process.
   * @returns {object}        Creation data for new activity.
   */
  static transformDescriptionData(source, options) {
    return {
      chatFlavor: source.system.chatFlavor ?? ""
    };
  }

  /* -------------------------------------------- */

  /**
   * Fetch data from the item source and transform it into an activity's duration object.
   * @param {object} source   Item's candidate source data to transform.
   * @param {object} options  Additional options passed to the creation process.
   * @returns {object}        Creation data for new activity.
   */
  static transformDurationData(source, options) {
    if ( source.type === "spell" ) return {};
    const concentration = !!source.system.properties?.findSplice(p => p === "concentration");
    return {
      concentration,
      value: source.system.duration?.value ?? null,
      units: source.system.duration?.units ?? "inst",
      special: ""
    };
  }

  /* -------------------------------------------- */

  /**
   * Fetch data from the item source and transform it into an activity's effects array.
   * @param {object} source   Item's candidate source data to transform.
   * @param {object} options  Additional options passed to the creation process.
   * @returns {object[]}      Creation data for new activity.
   */
  static transformEffectsData(source, options) {
    return source.effects
      .filter(e => !e.transfer && (e.type !== "enchantment") && (e.flags?.dnd5e?.type !== "enchantment"))
      .map(e => ({ _id: e._id }));
  }

  /* -------------------------------------------- */

  /**
   * Fetch data from the item source and transform it into an activity's range object.
   * @param {object} source   Item's candidate source data to transform.
   * @param {object} options  Additional options passed to the creation process.
   * @returns {object}        Creation data for new activity.
   */
  static transformRangeData(source, options) {
    if ( source.type === "spell" ) return {};
    return {
      value: source.system.range?.value ?? null,
      units: source.system.range?.units ?? "",
      special: ""
    };
  }

  /* -------------------------------------------- */

  /**
   * Fetch data from the item source and transform it into an activity's target object.
   * @param {object} source   Item's candidate source data to transform.
   * @param {object} options  Additional options passed to the creation process.
   * @returns {object}        Creation data for new activity.
   */
  static transformTargetData(source, options) {
    if ( source.type === "spell" ) return {
      prompt: source.system.target?.prompt ?? true
    };

    const data = {
      template: {
        count: "",
        contiguous: false,
        type: "",
        size: "",
        width: "",
        height: "",
        units: source.system.target?.units ?? "ft"
      },
      affects: {
        count: "",
        type: "",
        choice: false,
        special: ""
      },
      prompt: source.system.target?.prompt ?? true
    };

    if ( source.system.target?.type in CONFIG.DND5E.areaTargetTypes ) foundry.utils.mergeObject(data, {
      template: {
        type: source.system.target?.type ?? "",
        size: source.system.target?.value ?? "",
        width: source.system.target?.width ?? ""
      }
    });

    else foundry.utils.mergeObject(data, {
      affects: {
        count: source.system.target?.value ?? "",
        type: source.system.target?.type ?? ""
      }
    });

    return data;
  }

  /* -------------------------------------------- */

  /**
   * Perform any type-specific data transformations.
   * @param {object} source        Item's candidate source data to transform.
   * @param {object} activityData  In progress creation data.
   * @param {object} options       Additional options passed to the creation process.
   * @returns {object}             Creation data for new activity.
   */
  static transformTypeData(source, activityData, options) {
    return activityData;
  }

  /* -------------------------------------------- */

  /**
   * Fetch data from the item source and transform it into an activity's uses object.
   * @param {object} source   Item's candidate source data to transform.
   * @param {object} options  Additional options passed to the creation process.
   * @returns {object}        Creation data for new activity.
   */
  static transformUsesData(source, options) {
    // Do not add a recharge recovery to the activity if the parent item would already get recharge recovery.
    if ( !source.system.recharge?.value || !source.system.uses?.max || !source.system.uses?.per ) {
      return { spent: 0, max: "", recovery: [] };
    }
    return {
      spent: source.system.recharge.charged ? 0 : 1,
      max: "1",
      recovery: [{
        period: "recharge",
        type: "recoverAll",
        formula: String(source.system.recharge.value)
      }]
    };
  }

  /* -------------------------------------------- */
  /*  Data Preparation                            */
  /* -------------------------------------------- */

  /**
   * Prepare data related to this activity.
   */
  prepareData() {
    this.name = this.name || game.i18n.localize(this.metadata?.title);
    this.img = this.img || this.metadata?.img;
    this.labels ??= {};
    const addBaseIndices = data => data?.forEach((d, idx) => Object.defineProperty(d, "_index", { value: idx }));
    addBaseIndices(this.consumption?.targets);
    addBaseIndices(this.damage?.parts);
    addBaseIndices(this.effects);
    addBaseIndices(this.uses?.recovery);
  }

  /* -------------------------------------------- */

  /**
   * Perform final preparation after containing item is prepared.
   * @param {object} [rollData]  Deterministic roll data from the activity.
   */
  prepareFinalData(rollData) {
    rollData ??= this.getRollData({ deterministic: true });

    if ( this.activation ) this._setOverride("activation");
    if ( this.duration ) this._setOverride("duration");
    if ( this.range ) this._setOverride("range");
    if ( this.target ) this._setOverride("target");

    Object.defineProperty(this, "_inferredSource", {
      value: Object.freeze(this.toObject(false)),
      configurable: false,
      enumerable: false,
      writable: false
    });

    if ( this.activation ) ActivationField.prepareData.call(this, rollData, this.labels);
    if ( this.duration ) DurationField.prepareData.call(this, rollData, this.labels);
    if ( this.range ) RangeField.prepareData.call(this, rollData, this.labels);
    if ( this.target ) TargetField.prepareData.call(this, rollData, this.labels);
    if ( this.uses ) UsesField.prepareData.call(this, rollData, this.labels);

    const actor = this.item.actor;
    if ( !actor || !("consumption" in this) ) return;
    for ( const target of this.consumption.targets ) {
      if ( !["itemUses", "material"].includes(target.type) || !target.target ) continue;

      // Re-link UUIDs in consumption fields to explicit items on the actor
      if ( target.target.includes(".") ) {
        const item = actor.sourcedItems?.get(target.target, { legacy: false })?.first();
        if ( item ) target.target = item.id;
      }

      // If targeted item isn't found, display preparation warning
      if ( !actor.items.get(target.target) ) {
        const message = game.i18n.format("DND5E.CONSUMPTION.Warning.MissingItem", {
          activity: this.name, item: this.item.name
        });
        actor._preparationWarnings.push({ message, link: this.uuid, type: "warning" });
      }
    }
  }

  /* -------------------------------------------- */

  /**
   * Prepare the label for a compiled and simplified damage formula.
   * @param {DamageData[]} parts  Damage parts to create labels for.
   * @param {object} rollData     Deterministic roll data from the item.
   */
  prepareDamageLabel(parts, rollData) {
    this.labels.damage = parts.map((part, index) => {
      let formula;
      try {
        formula = part.formula;
        if ( part.base ) {
          if ( this.item.system.magicAvailable ) formula += ` + ${this.item.system.magicalBonus ?? 0}`;
          if ( (this.item.type === "weapon") && !/@mod\b/.test(formula) ) formula += " + @mod";
        }
        if ( !index && this.item.system.damageBonus ) formula += ` + ${this.item.system.damageBonus}`;
        const roll = new CONFIG.Dice.BasicRoll(formula, rollData);
        roll.simplify();
        formula = simplifyRollFormula(roll.formula, { preserveFlavor: true });
      } catch(err) {
        console.warn(`Unable to simplify formula for ${this.name} in item ${this.item.name}${
          this.actor ? ` on ${this.actor.name} (${this.actor.id})` : ""
        } (${this.uuid})`, err);
      }

      let label = formula;
      if ( part.types.size ) {
        label = `${formula} ${game.i18n.getListFormatter({ type: "conjunction" }).format(
          Array.from(part.types)
            .map(p => CONFIG.DND5E.damageTypes[p]?.label ?? CONFIG.DND5E.healingTypes[p]?.label)
            .filter(t => t)
        )}`;
      }

      return { formula, damageType: part.types.size === 1 ? part.types.first() : null, label, base: part.base };
    });
  }

  /* -------------------------------------------- */
  /*  Socket Event Handlers                       */
  /* -------------------------------------------- */

  /**
   * Perform preliminary operations before an Activity is created.
   * @param {object} data     The initial data object provided to the document creation request.
   * @returns {boolean|void}  A return value of false indicates the creation operation should be cancelled.
   * @protected
   */
  _preCreate(data) {}

  /* -------------------------------------------- */
  /*  Helpers                                     */
  /* -------------------------------------------- */

  /**
   * Retrieve the action type reflecting changes based on the provided attack mode.
   * @param {string} [attackMode=""]
   * @returns {string}
   */
  getActionType(attackMode="") {
    let actionType = this.actionType;
    if ( (actionType === "mwak") && (attackMode?.startsWith("thrown") || (attackMode === "ranged")) ) return "rwak";
    return actionType;
  }

  /* -------------------------------------------- */

  /**
   * Get the roll parts used to create the damage rolls.
   * @param {Partial<DamageRollProcessConfiguration>} [config={}]
   * @returns {DamageRollProcessConfiguration}
   */
  getDamageConfig(config={}) {
    if ( !this.damage?.parts ) return foundry.utils.mergeObject({ rolls: [] }, config);

    const rollConfig = foundry.utils.mergeObject({ scaling: 0 }, config);
    const rollData = this.getRollData();
    rollConfig.rolls = this.damage.parts
      .map((d, index) => this._processDamagePart(d, rollConfig, rollData, index))
      .filter(d => d.parts.length)
      .concat(config.rolls ?? []);

    return rollConfig;
  }

  /* -------------------------------------------- */

  /**
   * Process a single damage part into a roll configuration.
   * @param {DamageData} damage                                   Damage to prepare for the roll.
   * @param {Partial<DamageRollProcessConfiguration>} rollConfig  Roll configuration being built.
   * @param {object} rollData                                     Roll data to populate with damage data.
   * @param {number} [index=0]                                    Index of the damage part.
   * @returns {DamageRollConfiguration}
   * @protected
   */
  _processDamagePart(damage, rollConfig, rollData, index=0) {
    const scaledFormula = damage.scaledFormula(rollData.scaling);
    const parts = scaledFormula ? [scaledFormula] : [];
    const data = { ...rollData };

    if ( index === 0 ) {
      const actionType = this.getActionType(rollConfig.attackMode);
      const bonus = foundry.utils.getProperty(this.actor ?? {}, `system.bonuses.${actionType}.damage`);
      if ( bonus && !/^0+$/.test(bonus) ) parts.push(bonus);
      if ( this.item.system.damageBonus ) parts.push(String(this.item.system.damageBonus));
    }

    const lastType = this.item.getFlag("dnd5e", `last.${this.id}.damageType.${index}`);

    return {
      data, parts,
      options: {
        type: (damage.types.has(lastType) ? lastType : null) ?? damage.types.first(),
        types: Array.from(damage.types),
        properties: Array.from(this.item.system.properties ?? [])
          .filter(p => CONFIG.DND5E.itemProperties[p]?.isPhysical)
      }
    };
  }

  /* -------------------------------------------- */

  /**
   * Add an `canOverride` property to the provided object and, if `override` is `false`, replace the data on the
   * activity with data from the item.
   * @param {string} keyPath  Path of the property to set on the activity.
   * @internal
   */
  _setOverride(keyPath) {
    const obj = foundry.utils.getProperty(this, keyPath);
    Object.defineProperty(obj, "canOverride", {
      value: safePropertyExists(this.item.system, keyPath),
      configurable: true,
      enumerable: false
    });
    if ( obj.canOverride && !obj.override ) {
      foundry.utils.mergeObject(obj, foundry.utils.getProperty(this.item.system, keyPath));
    }
  }
}

const { ArrayField: ArrayField$m, BooleanField: BooleanField$E, NumberField: NumberField$G, SchemaField: SchemaField$N, StringField: StringField$13 } = foundry.data.fields;

/**
 * Data model for an attack activity.
 *
 * @property {object} attack
 * @property {string} attack.ability              Ability used to make the attack and determine damage.
 * @property {string} attack.bonus                Arbitrary bonus added to the attack.
 * @property {object} attack.critical
 * @property {number} attack.critical.threshold   Minimum value on the D20 needed to roll a critical hit.
 * @property {boolean} attack.flat                Should the bonus be used in place of proficiency & ability modifier?
 * @property {object} attack.type
 * @property {string} attack.type.value           Is this a melee or ranged attack?
 * @property {string} attack.type.classification  Is this a unarmed, weapon, or spell attack?
 * @property {object} damage
 * @property {object} damage.critical
 * @property {string} damage.critical.bonus       Extra damage applied when a critical is rolled. Added to the base
 *                                                damage or first damage part.
 * @property {boolean} damage.includeBase         Should damage defined by the item be included with other damage parts?
 * @property {DamageData[]} damage.parts          Parts of damage to inflict.
 */
class AttackActivityData extends BaseActivityData {
  /** @inheritDoc */
  static defineSchema() {
    return {
      ...super.defineSchema(),
      attack: new SchemaField$N({
        ability: new StringField$13(),
        bonus: new FormulaField(),
        critical: new SchemaField$N({
          threshold: new NumberField$G({ integer: true, positive: true })
        }),
        flat: new BooleanField$E(),
        type: new SchemaField$N({
          value: new StringField$13(),
          classification: new StringField$13()
        })
      }),
      damage: new SchemaField$N({
        critical: new SchemaField$N({
          bonus: new FormulaField()
        }),
        includeBase: new BooleanField$E({ initial: true }),
        parts: new ArrayField$m(new DamageField())
      })
    };
  }

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /** @override */
  get ability() {
    if ( this.attack.ability === "none" ) return null;
    if ( this.attack.ability === "spellcasting" ) return this.spellcastingAbility;
    if ( this.attack.ability in CONFIG.DND5E.abilities ) return this.attack.ability;

    const availableAbilities = this.availableAbilities;
    if ( !availableAbilities?.size ) return null;
    if ( availableAbilities?.size === 1 ) return availableAbilities.first();
    const abilities = this.actor?.system.abilities ?? {};
    return availableAbilities.reduce((largest, ability) =>
      (abilities[ability]?.mod ?? -Infinity) > (abilities[largest]?.mod ?? -Infinity) ? ability : largest
    , availableAbilities.first());
  }

  /* -------------------------------------------- */

  /** @override */
  get actionType() {
    const type = this.attack.type;
    return `${type.value === "ranged" ? "r" : "m"}${type.classification === "spell" ? "sak" : "wak"}`;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  get activationLabels() {
    const labels = super.activationLabels;
    if ( labels && (this.item.type === "weapon") && !this.range.override ) {
      if ( this.item.labels?.range ) labels.range = this.item.labels.range;
      if ( this.item.labels?.reach ) labels.reach = this.item.labels.reach;
    }
    return labels;
  }

  /* -------------------------------------------- */

  /**
   * Abilities that could potentially be used with this attack. Unless a specific ability is specified then
   * whichever ability has the highest modifier will be selected when making an attack.
   * @type {Set<string>}
   */
  get availableAbilities() {
    // Defer to item if available and matching attack classification
    if ( this.item.system.availableAbilities && (this.item.type === this.attack.type.classification) ) {
      return this.item.system.availableAbilities;
    }

    // Spell attack not associated with a single class, use highest spellcasting ability on actor
    if ( this.attack.type.classification === "spell" ) return new Set(
      this.actor?.system.attributes?.spellcasting
        ? [this.actor.system.attributes.spellcasting]
        : Object.values(this.actor?.spellcastingClasses ?? {}).map(c => c.spellcasting.ability)
    );

    // Weapon & unarmed attacks uses melee or ranged ability depending on type, or both if actor is an NPC
    const melee = CONFIG.DND5E.defaultAbilities.meleeAttack;
    const ranged = CONFIG.DND5E.defaultAbilities.rangedAttack;
    return new Set([this.attack.type.value === "melee" ? melee : ranged]);
  }

  /* -------------------------------------------- */

  /**
   * Critical threshold for attacks with this activity.
   * @type {number}
   */
  get criticalThreshold() {
    // TODO: Fetch threshold from ammo
    const threshold = Math.min(
      this.attack.critical.threshold ?? Infinity,
      this.item.system.criticalThreshold ?? Infinity,
      Infinity
    );
    return threshold < Infinity ? threshold : 20;
  }

  /* -------------------------------------------- */

  /**
   * Potential attack types when attacking with this activity.
   * @type {Set<string>}
   */
  get validAttackTypes() {
    const sourceType = this._source.attack.type.value;
    if ( sourceType ) return new Set([sourceType]);
    return this.item.system.validAttackTypes ?? new Set();
  }

  /* -------------------------------------------- */
  /*  Data Migrations                             */
  /* -------------------------------------------- */

  /** @override */
  static transformTypeData(source, activityData, options) {
    // For weapons and ammunition, separate the first part from the rest to be used as the base damage and keep the rest
    let damageParts = source.system.damage?.parts ?? [];
    const hasBase = (source.type === "weapon")
      || ((source.type === "consumable") && (source.system?.type?.value === "ammo"));
    if ( hasBase && damageParts.length && !source.system.damage?.base ) {
      const [base, ...rest] = damageParts;
      source.system.damage.parts = [base];
      damageParts = rest;
    }

    return foundry.utils.mergeObject(activityData, {
      attack: {
        ability: source.system.ability ?? "",
        bonus: source.system.attack?.bonus ?? "",
        critical: {
          threshold: source.system.critical?.threshold
        },
        flat: source.system.attack?.flat ?? false,
        type: {
          value: source.system.actionType.startsWith("m") ? "melee" : "ranged",
          classification: source.system.actionType.endsWith("wak") ? "weapon" : "spell"
        }
      },
      damage: {
        critical: {
          bonus: source.system.critical?.damage
        },
        includeBase: true,
        parts: damageParts.map(part => this.transformDamagePartData(source, part)) ?? []
      }
    });
  }

  /* -------------------------------------------- */
  /*  Data Preparation                            */
  /* -------------------------------------------- */

  /** @inheritDoc */
  prepareData() {
    super.prepareData();
    this.attack.type.value ||= this.item.system.attackType ?? "melee";
    this.attack.type.classification ||= this.item.system.attackClassification ?? "weapon";
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  prepareFinalData(rollData) {
    if ( this.damage.includeBase && this.item.system.offersBaseDamage && this.item.system.damage.base.formula ) {
      const basePart = this.item.system.damage.base.clone(this.item.system.damage.base.toObject(false));
      basePart.base = true;
      basePart.locked = true;
      this.damage.parts.unshift(basePart);
    }

    rollData ??= this.getRollData({ deterministic: true });
    super.prepareFinalData(rollData);
    this.prepareDamageLabel(this.damage.parts, rollData);

    const { data, parts } = this.getAttackData();
    const roll = new Roll(parts.join("+"), data);
    this.labels.modifier = simplifyRollFormula(roll.formula, { deterministic: true }) || "0";
    const formula = simplifyRollFormula(roll.formula) || "0";
    this.labels.toHit = !/^[+-]/.test(formula) ? `+${formula}` : formula;
  }

  /* -------------------------------------------- */
  /*  Helpers                                     */
  /* -------------------------------------------- */

  /**
   * The game term label for this attack.
   * @param {string} [attackMode]  The mode the attack was made with.
   * @returns {string}
   */
  getActionLabel(attackMode) {
    let attackModeLabel;
    if ( attackMode ) {
      const key = attackMode.split("-").map(s => s.capitalize()).join("");
      attackModeLabel = game.i18n.localize(`DND5E.ATTACK.Mode.${key}`);
    }
    const actionType = this.getActionType(attackMode);
    let actionTypeLabel = game.i18n.localize(`DND5E.Action${actionType.toUpperCase()}`);
    const isLegacy = game.settings.get("dnd5e", "rulesVersion") === "legacy";
    const isUnarmed = this.attack.type.classification === "unarmed";
    if ( isUnarmed ) attackModeLabel = game.i18n.localize("DND5E.ATTACK.Classification.Unarmed");
    const isSpell = (actionType === "rsak") || (actionType === "msak");
    if ( isLegacy || isSpell ) return [actionTypeLabel, attackModeLabel].filterJoin(" &bull; ");
    actionTypeLabel = game.i18n.localize(`DND5E.ATTACK.Attack.${actionType}`);
    if ( isUnarmed ) return [actionTypeLabel, attackModeLabel].filterJoin(" &bull; ");
    const weaponType = CONFIG.DND5E.weaponTypeMap[this.item.system.type?.value];
    const weaponTypeLabel = weaponType
      ? game.i18n.localize(`DND5E.ATTACK.Weapon.${weaponType.capitalize()}`)
      : CONFIG.DND5E.weaponTypes[this.item.system.type?.value];
    return [actionTypeLabel, weaponTypeLabel, attackModeLabel].filterJoin(" &bull; ");
  }

  /* -------------------------------------------- */

  /**
   * Get the roll parts used to create the attack roll.
   * @param {object} [config={}]
   * @param {string} [config.ammunition]
   * @param {string} [config.attackMode]
   * @param {string} [config.situational]
   * @returns {{ data: object, parts: string[] }}
   */
  getAttackData({ ammunition, attackMode, situational }={}) {
    const rollData = this.getRollData();
    if ( this.attack.flat ) return CONFIG.Dice.BasicRoll.constructParts({ toHit: this.attack.bonus }, rollData);

    const weapon = this.item.system;
    const ammo = this.actor?.items.get(ammunition)?.system;
    const { parts, data } = CONFIG.Dice.BasicRoll.constructParts({
      mod: this.attack.ability !== "none" ? rollData.mod : null,
      prof: weapon.prof?.term,
      bonus: this.attack.bonus,
      weaponMagic: weapon.magicAvailable ? weapon.magicalBonus : null,
      ammoMagic: ammo?.magicAvailable ? ammo.magicalBonus : null,
      actorBonus: this.actor?.system.bonuses?.[this.getActionType(attackMode)]?.attack,
      situational
    }, rollData);

    // Add exhaustion reduction
    this.actor?.addRollExhaustion(parts, data);

    return { data, parts };
  }

  /* -------------------------------------------- */

  /**
   * @typedef {AttackDamageRollProcessConfiguration} [config={}]
   * @property {Item5e} ammunition  Ammunition used with the attack.
   * @property {"oneHanded"|"twoHanded"|"offhand"|"thrown"|"thrown-offhand"} attackMode  Attack mode.
   */

  /**
   * Get the roll parts used to create the damage rolls.
   * @param {Partial<AttackDamageRollProcessConfiguration>} [config={}]
   * @returns {AttackDamageRollProcessConfiguration}
   */
  getDamageConfig(config={}) {
    const rollConfig = super.getDamageConfig(config);

    // Handle ammunition
    const ammo = config.ammunition?.system;
    if ( ammo ) {
      const properties = Array.from(ammo.properties).filter(p => CONFIG.DND5E.itemProperties[p]?.isPhysical);
      if ( this.item.system.properties?.has("mgc") && !properties.includes("mgc") ) properties.push("mgc");

      // Add any new physical properties from the ammunition to the damage properties
      for ( const roll of rollConfig.rolls ) {
        for ( const property of properties ) {
          if ( !roll.options.properties.includes(property) ) roll.options.properties.push(property);
        }
      }

      // Add the ammunition's damage
      if ( ammo.damage.base.formula ) {
        const basePartIndex = rollConfig.rolls.findIndex(i => i.base);
        const damage = ammo.damage.base.clone(ammo.damage.base);
        const rollData = this.getRollData();

        // If mode is "replace" and base part is present, replace the base part
        if ( ammo.damage.replace & (basePartIndex !== -1) ) {
          damage.base = true;
          rollConfig.rolls.splice(basePartIndex, 1, this._processDamagePart(damage, config, rollData, basePartIndex));
        }

        // Otherwise stick the ammo damage after base part (or as first part)
        else {
          damage.ammo = true;
          rollConfig.rolls.splice(
            basePartIndex + 1, 0, this._processDamagePart(damage, rollConfig, rollData, basePartIndex + 1)
          );
        }
      }
    }

    if ( this.damage.critical.bonus && !rollConfig.rolls[0]?.options?.critical?.bonusDamage ) {
      foundry.utils.setProperty(rollConfig.rolls[0], "options.critical.bonusDamage", this.damage.critical.bonus);
    }

    return rollConfig;
  }

  /* -------------------------------------------- */

  /**
   * Create a label based on this activity's settings and, if contained in a weapon, additional details from the weapon.
   * @returns {string}
   */
  getRangeLabel() {
    if ( this.item.type !== "weapon" ) return this.labels?.range ?? "";

    const parts = [];

    // Add reach for melee weapons, unless the activity is explicitly specified as a ranged attack
    if ( this.validAttackTypes.has("melee") ) {
      let { reach, units } = this.item.system.range;
      if ( !reach ) reach = convertLength(5, "ft", units);
      parts.push(game.i18n.format("DND5E.RANGE.Formatted.Reach", {
        reach: formatLength(reach, units, { strict: false })
      }));
    }

    // Add range for ranged or thrown weapons, unless the activity is explicitly specified as melee
    if ( this.validAttackTypes.has("ranged") ) {
      let range;
      if ( this.range.override ) range = `${this.range.value} ${this.range.units ?? ""}`;
      else {
        const { value, long, units } = this.item.system.range;
        if ( long && (value !== long) ) range = `${value}/${formatLength(long, units, { strict: false })}`;
        else range = formatLength(value, units, { strict: false });
      }
      parts.push(game.i18n.format("DND5E.RANGE.Formatted.Range", { range }));
    }

    return game.i18n.getListFormatter({ type: "disjunction" }).format(parts.filter(_ => _));
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _processDamagePart(damage, rollConfig, rollData, index=0) {
    if ( !damage.base ) return super._processDamagePart(damage, rollConfig, rollData, index);

    // Swap base damage for versatile if two-handed attack is made on versatile weapon
    if ( this.item.system.isVersatile && (rollConfig.attackMode === "twoHanded") ) {
      const versatile = this.item.system.damage.versatile.clone(this.item.system.damage.versatile);
      versatile.base = true;
      versatile.denomination ||= damage.steppedDenomination();
      versatile.number ||= damage.number;
      versatile.types = damage.types;
      damage = versatile;
    }

    const roll = super._processDamagePart(damage, rollConfig, rollData, index);
    roll.base = true;

    if ( this.item.type === "weapon" ) {
      // Ensure `@mod` is present in damage unless it is positive and an off-hand attack or damage is a flat value
      const isDeterministic = new Roll(roll.parts[0]).isDeterministic;
      const includeMod = (!rollConfig.attackMode?.endsWith("offhand") || (roll.data.mod < 0)) && !isDeterministic;
      if ( includeMod && !roll.parts.some(p => p.includes("@mod")) ) roll.parts.push("@mod");

      // Add magical bonus
      if ( this.item.system.magicalBonus && this.item.system.magicAvailable ) {
        roll.parts.push("@magicalBonus");
        roll.data.magicalBonus = this.item.system.magicalBonus;
      }

      // Add ammunition bonus
      const ammo = rollConfig.ammunition?.system;
      if ( ammo?.magicAvailable && ammo.magicalBonus ) {
        roll.parts.push("@ammoBonus");
        roll.data.ammoBonus = ammo.magicalBonus;
      }
    }

    const criticalBonusDice = this.actor?.getFlag("dnd5e", "meleeCriticalDamageDice") ?? 0;
    if ( (this.getActionType(rollConfig.attackMode) === "mwak") && (parseInt(criticalBonusDice) !== 0) ) {
      foundry.utils.setProperty(roll, "options.critical.bonusDice", criticalBonusDice);
    }

    return roll;
  }
}

/**
 * @typedef {BasicRollConfigurationDialogOptions} SkillToolRollConfigurationDialogOptions
 * @property {boolean} chooseAbility  Should the ability be selectable?
 */

/**
 * Extended roll configuration dialog that allows selecting abilities.
 */
class SkillToolRollConfigurationDialog extends D20RollConfigurationDialog {
  /** @override */
  static DEFAULT_OPTIONS = {
    chooseAbility: true
  };

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareConfigurationContext(context, options) {
    context = await super._prepareConfigurationContext(context, options);
    if ( this.options.chooseAbility ) context.fields.unshift({
      field: new foundry.data.fields.StringField({ label: game.i18n.localize("DND5E.Abilities") }),
      name: "ability",
      options: Object.entries(CONFIG.DND5E.abilities).map(([value, { label }]) => ({ value, label })),
      value: this.config.ability
    });
    return context;
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /** @inheritDoc */
  _onChangeForm(formConfig, event) {
    super._onChangeForm(formConfig, event);
    if ( this.config.skill && (event.target?.name === "ability") ) {
      const skillLabel = CONFIG.DND5E.skills[this.config.skill]?.label ?? "";
      const ability = event.target.value ?? this.config.ability;
      const abilityLabel = CONFIG.DND5E.abilities[ability]?.label ?? "";
      const flavor = game.i18n.format("DND5E.SkillPromptTitle", { skill: skillLabel, ability: abilityLabel });
      foundry.utils.setProperty(this.message, "data.flavor", flavor);
      this._updateFrame({ window: { title: flavor } });
    }
  }
}

const { DiceTerm: DiceTerm$1, NumericTerm: NumericTerm$1 } = foundry.dice.terms;

/**
 * Configuration data for the process of creating one or more basic rolls.
 *
 * @typedef {object} BasicRollProcessConfiguration
 * @property {BasicRollConfiguration[]} rolls  Configuration data for individual rolls.
 * @property {boolean} [evaluate=true]         Should the rolls be evaluated? If set to `false`, then no chat message
 *                                             will be created regardless of message configuration.
 * @property {Event} [event]                   Event that triggered the rolls.
 * @property {string[]} [hookNames]            Name suffixes for configuration hooks called.
 * @property {Document} [subject]              Document that initiated this roll.
 * @property {number} [target]                 Default target value for all rolls.
 */

/**
 * Configuration data for an individual roll.
 *
 * @typedef {object} BasicRollConfiguration
 * @property {string[]} [parts=[]]         Parts used to construct the roll formula.
 * @property {object} [data={}]            Data used to resolve placeholders in the formula.
 * @property {boolean} [situational=true]  Whether the situational bonus can be added to this roll in the prompt.
 * @property {BasicRollOptions} [options]  Additional options passed through to the created roll.
 */

/**
 * Options allowed on a basic roll.
 *
 * @typedef {object} BasicRollOptions
 * @property {number} [target]  The total roll result that must be met for the roll to be considered a success.
 */

/* -------------------------------------------- */

/**
 * Configuration data for the roll prompt.
 *
 * @typedef {object} BasicRollDialogConfiguration
 * @property {boolean} [configure=true]  Display a configuration dialog for the rolling process.
 * @property {typeof RollConfigurationDialog} [applicationClass]  Alternate configuration application to use.
 * @property {BasicRollConfigurationDialogOptions} [options]      Additional options passed to the dialog.
 */

/* -------------------------------------------- */

/**
 * Configuration data for creating a roll message.
 *
 * @typedef {object} BasicRollMessageConfiguration
 * @property {boolean} [create=true]     Create a message when the rolling is complete.
 * @property {ChatMessage5e} [document]  Final created chat message document once process is completed.
 * @property {string} [rollMode]         The roll mode to apply to this message from `CONFIG.Dice.rollModes`.
 * @property {object} [data={}]          Additional data used when creating the message.
 */

/* -------------------------------------------- */

/**
 * Custom base roll type with methods for building rolls, presenting prompts, and creating messages.
 */
class BasicRoll extends Roll {

  /**
   * Default application used for the roll configuration prompt.
   * @type {typeof RollConfigurationDialog}
   */
  static DefaultConfigurationDialog = RollConfigurationDialog;

  /* -------------------------------------------- */
  /*  Static Construction                         */
  /* -------------------------------------------- */

  /**
   * Create a roll instance from a roll config.
   * @param {BasicRollConfiguration} config          Configuration info for the roll.
   * @param {BasicRollProcessConfiguration} process  Configuration info for the whole rolling process.
   * @returns {BasicRoll}
   */
  static fromConfig(config, process) {
    const formula = (config.parts ?? []).join(" + ");
    config.options ??= {};
    config.options.target ??= process.target;
    return new this(formula, config.data, config.options);
  }

  /* -------------------------------------------- */

  /**
   * Construct roll parts and populate its data object.
   * @param {object} parts   Information on the parts to be constructed.
   * @param {object} [data]  Roll data to use and populate while constructing the parts.
   * @returns {{ parts: string[], data: object }}
   */
  static constructParts(parts, data={}) {
    const finalParts = [];
    for ( const [key, value] of Object.entries(parts) ) {
      if ( !value && (value !== 0) ) continue;
      finalParts.push(`@${key}`);
      foundry.utils.setProperty(
        data, key, foundry.utils.getType(value) === "string" ? Roll.replaceFormulaData(value, data) : value
      );
    }
    return { parts: finalParts, data };
  }

  /* -------------------------------------------- */

  /**
   * Construct and perform a roll through the standard workflow.
   * @param {BasicRollProcessConfiguration} [config={}]   Configuration for the rolls.
   * @param {BasicRollDialogConfiguration} [dialog={}]    Configuration for roll prompt.
   * @param {BasicRollMessageConfiguration} [message={}]  Configuration for message creation.
   * @returns {BasicRoll[]}
   */
  static async build(config={}, dialog={}, message={}) {
    const rolls = await this.buildConfigure(config, dialog, message);
    await this.buildEvaluate(rolls, config, message);
    await this.buildPost(rolls, config, message);
    return rolls;
  }

  /* -------------------------------------------- */

  /**
   * Stage one of the standard rolling workflow, configuring the roll.
   * @param {BasicRollProcessConfiguration} [config={}]   Configuration for the rolls.
   * @param {BasicRollDialogConfiguration} [dialog={}]    Configuration for roll prompt.
   * @param {BasicRollMessageConfiguration} [message={}]  Configuration for message creation.
   * @returns {Promise<BasicRoll[]>}
   */
  static async buildConfigure(config={}, dialog={}, message={}) {
    config.hookNames = [...(config.hookNames ?? []), ""];

    /**
     * A hook event that fires before a roll is performed. Multiple hooks may be called depending on the rolling
     * method (e.g. `dnd5e.preRollSkillV2`, `dnd5e.preRollAbilityCheckV2`, `dnd5e.preRollV2`). Exact contents of the
     * configuration object will also change based on the roll type, but the same objects will always be present.
     * @function dnd5e.preRollV2
     * @memberof hookEvents
     * @param {BasicRollProcessConfiguration} config   Configuration data for the pending roll.
     * @param {BasicRollDialogConfiguration} dialog    Presentation data for the roll configuration dialog.
     * @param {BasicRollMessageConfiguration} message  Configuration data for the roll's message.
     * @returns {boolean}                              Explicitly return `false` to prevent the roll.
     */
    for ( const hookName of config.hookNames ) {
      if ( Hooks.call(`dnd5e.preRoll${hookName.capitalize()}V2`, config, dialog, message) === false ) return [];
    }

    this.applyKeybindings(config, dialog, message);

    let rolls;
    if ( dialog.configure === false ) {
      rolls = config.rolls?.map((r, index) => {
        dialog.options?.buildConfig?.(config, r, null, index);
        return this.fromConfig(r, config);
      }) ?? [];
    } else {
      const DialogClass = dialog.applicationClass ?? this.DefaultConfigurationDialog;
      rolls = await DialogClass.configure(config, dialog, message);
    }

    // Store the roll type in roll.options so it can be accessed from only the roll
    const rollType = foundry.utils.getProperty(message, "data.flags.dnd5e.roll.type");
    if ( rollType ) rolls.forEach(roll => roll.options.rollType ??= rollType);

    /**
     * A hook event that fires after roll configuration is complete, but before the roll is evaluated.
     * Multiple hooks may be called depending on the rolling method (e.g. `dnd5e.postSkillCheckRollConfiguration`,
     * `dnd5e.postAbilityTestRollConfiguration`, and `dnd5e.postRollConfiguration` for skill checks). Exact contents of
     * the configuration object will also change based on the roll type, but the same objects will always be present.
     * @function dnd5e.postRollConfiguration
     * @memberof hookEvents
     * @param {BasicRoll[]} rolls                      Rolls that have been constructed but not evaluated.
     * @param {BasicRollProcessConfiguration} config   Configuration information for the roll.
     * @param {BasicRollDialogConfiguration} dialog    Configuration for the roll dialog.
     * @param {BasicRollMessageConfiguration} message  Configuration for the roll message.
     * @returns {boolean}                              Explicitly return `false` to prevent rolls.
     */
    for ( const hookName of config.hookNames ) {
      const name = `dnd5e.post${hookName.capitalize()}RollConfiguration`;
      if ( Hooks.call(name, rolls, config, dialog, message) === false ) return [];
    }

    return rolls;
  }

  /* -------------------------------------------- */

  /**
   * Stage two of the standard rolling workflow, evaluating the rolls.
   * @param {BasicRoll[]} rolls                           Rolls to evaluate.
   * @param {BasicRollProcessConfiguration} [config={}]   Configuration for the rolls.
   * @param {BasicRollMessageConfiguration} [message={}]  Configuration for message creation.
   */
  static async buildEvaluate(rolls, config={}, message={}) {
    if ( config.evaluate !== false ) {
      for ( const roll of rolls ) await roll.evaluate();
    }
  }

  /* -------------------------------------------- */

  /**
   * Stage three of the standard rolling workflow, posting a message to chat.
   * @param {BasicRoll[]} rolls                      Rolls to evaluate.
   * @param {BasicRollProcessConfiguration} config   Configuration for the rolls.
   * @param {BasicRollMessageConfiguration} message  Configuration for message creation.
   * @returns {ChatMessage5e|void}
   */
  static async buildPost(rolls, config, message) {
    message.data = foundry.utils.expandObject(message.data ?? {});
    const messageId = config.event?.target.closest("[data-message-id]")?.dataset.messageId;
    if ( messageId ) foundry.utils.setProperty(message.data, "flags.dnd5e.originatingMessage", messageId);

    if ( rolls?.length && (config.evaluate !== false) && (message.create !== false) ) {
      message.document = await this.toMessage(rolls, message.data, { rollMode: message.rollMode });
    }

    return message.document;
  }

  /* -------------------------------------------- */

  /**
   * Determines whether the roll process should be fast forwarded.
   * @param {BasicRollProcessConfiguration} config   Roll configuration data.
   * @param {BasicRollDialogConfiguration} dialog    Data for the roll configuration dialog.
   * @param {BasicRollMessageConfiguration} message  Message configuration data.
   */
  static applyKeybindings(config, dialog, message) {
    dialog.configure ??= true;
  }

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * Is the result of this roll a failure? Returns `undefined` if roll isn't evaluated.
   * @type {boolean|void}
   */
  get isFailure() {
    if ( !this._evaluated ) return;
    if ( !Number.isNumeric(this.options.target) ) return false;
    return this.total < this.options.target;
  }

  /* -------------------------------------------- */

  /**
   * Is the result of this roll a success? Returns `undefined` if roll isn't evaluated.
   * @type {boolean|void}
   */
  get isSuccess() {
    if ( !this._evaluated ) return;
    if ( !Number.isNumeric(this.options.target) ) return false;
    return this.total >= this.options.target;
  }

  /* -------------------------------------------- */
  /*  Chat Messages                               */
  /* -------------------------------------------- */

  /**
   * Transform a Roll instance into a ChatMessage, displaying the roll result.
   * This function can either create the ChatMessage directly, or return the data object that will be used to create it.
   *
   * @param {BasicRoll[]} rolls              Rolls to add to the message.
   * @param {object} messageData             The data object to use when creating the message.
   * @param {options} [options]              Additional options which modify the created message.
   * @param {string} [options.rollMode]      The template roll mode to use for the message from CONFIG.Dice.rollModes
   * @param {boolean} [options.create=true]  Whether to automatically create the chat message, or only return the
   *                                         prepared chatData object.
   * @returns {Promise<ChatMessage|object>}  A promise which resolves to the created ChatMessage document if create is
   *                                         true, or the Object of prepared chatData otherwise.
   */
  static async toMessage(rolls, messageData={}, { rollMode, create=true }={}) {
    for ( const roll of rolls ) {
      if ( !roll._evaluated ) await roll.evaluate({ allowInteractive: rollMode !== CONST.DICE_ROLL_MODES.BLIND });
      rollMode ??= roll.options.rollMode;
    }

    // Prepare chat data
    messageData = foundry.utils.mergeObject({ sound: CONFIG.sounds.dice }, messageData);
    messageData.rolls = rolls;
    this._prepareMessageData(rolls, messageData);

    // Process the chat data
    const cls = getDocumentClass("ChatMessage");
    const msg = new cls(messageData);

    // Either create or return the data
    if ( create ) return cls.create(msg.toObject(), { rollMode });
    else {
      if ( rollMode ) msg.applyRollMode(rollMode);
      return msg.toObject();
    }
  }

  /* -------------------------------------------- */

  /**
   * Perform specific changes to message data before creating message.
   * @param {BasicRoll[]} rolls   Rolls to add to the message.
   * @param {object} messageData  The data object to use when creating the message.
   * @protected
   */
  static _prepareMessageData(rolls, messageData) {}

  /* -------------------------------------------- */
  /*  Evaluate Methods                            */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async evaluate(options={}) {
    this.preCalculateDiceTerms(options);
    return super.evaluate(options);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  evaluateSync(options={}) {
    this.preCalculateDiceTerms(options);
    return super.evaluateSync(options);
  }

  /* -------------------------------------------- */
  /*  Roll Formula Parsing                        */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static replaceFormulaData(formula, data, options) {
    // This looks for the pattern `$!!$` and replaces it with just the value between the marks (the bang has
    // been added to ensure this is a deliberate shim from the system, not a unintentional usage that should
    // show an error).
    return super.replaceFormulaData(formula, data, options).replaceAll(/\$"?!(.+?)!"?\$/g, "$1");
  }

  /* -------------------------------------------- */
  /*  Maximize/Minimize Methods                   */
  /* -------------------------------------------- */

  /**
   * Replaces all dice terms that have modifiers with their maximum/minimum value.
   *
   * @param {object} [options={}]            Extra optional arguments which describe or modify the BasicRoll.
   */
  preCalculateDiceTerms(options={}) {
    if ( this._evaluated || (!options.maximize && !options.minimize) ) return;
    this.terms = this.terms.map(term => {
      if ( (term instanceof DiceTerm$1) && term.modifiers.length ) {
        const minimize = !options.maximize;
        const number = this.constructor.preCalculateTerm(term, { minimize });
        if ( Number.isFinite(number) ) return new NumericTerm$1({ number, options: term.options });
      }
      return term;
    });
  }

  /* -------------------------------------------- */

  /**
   * Gets information from passed die and calculates the maximum or minimum value that could be rolled.
   *
   * @param {DiceTerm} die                            DiceTerm to get the maximum/minimum value.
   * @param {object} [preCalculateOptions={}]         Additional options to modify preCalculate functionality.
   * @param {boolean} [preCalculateOptions.minimize=false]  Calculate the minimum value instead of the maximum.
   * @returns {number|null}                                 Maximum/Minimum value that could be rolled as an integer, or
   *                                                        null if the modifiers could not be precalculated.
   */
  static preCalculateTerm(die, { minimize=false }={}) {
    let face = minimize ? 1 : die.faces;
    let number = die.number;
    const currentModifiers = foundry.utils.deepClone(die.modifiers);
    const keep = new Set(["k", "kh", "kl"]);
    const drop = new Set(["d", "dh", "dl"]);
    const validModifiers = new Set([...keep, ...drop, "max", "min"]);
    let matchedModifier = false;

    for ( const modifier of currentModifiers ) {
      const rgx = /(m[ai][xn]|[kd][hl]?)(\d+)?/i;
      const match = modifier.match(rgx);
      if ( !match ) continue;
      if ( match[0].length < match.input.length ) currentModifiers.push(match.input.slice(match[0].length));
      let [, command, value] = match;
      command = command.toLowerCase();
      if ( !validModifiers.has(command) ) continue;

      matchedModifier = true;
      const amount = parseInt(value) || (command === "max" || command === "min" ? -1 : 1);
      if ( amount > 0 ) {
        if ( (command === "max" && minimize) || (command === "min" && !minimize) ) continue;
        else if ( (command === "max" || command === "min") ) face = Math.min(die.faces, amount);
        else if ( keep.has(command) ) number = Math.min(number, amount);
        else if ( drop.has(command) ) number = Math.max(1, number - amount);
      }
    }

    return matchedModifier ? face * number : null;
  }

  /* -------------------------------------------- */
  /*  Simplification Methods                      */
  /* -------------------------------------------- */

  /**
   * Replace number and faces of dice terms with numeric values where possible.
   */
  simplify() {
    for ( const die of this.dice ) {
      const n = die._number;
      if ( (n instanceof BasicRoll) && n.isDeterministic ) die._number = n.evaluateSync().total;
      const f = die._faces;
      if ( (f instanceof BasicRoll) && f.isDeterministic ) die._faces = f.evaluateSync().total;

      // Preserve flavor.
      if ( f.terms?.[0]?.flavor ) die.options.flavor = f.terms[0].flavor;
    }

    this.resetFormula();
  }

  /* -------------------------------------------- */
  /*  Helpers                                     */
  /* -------------------------------------------- */

  /**
   * Merge two roll configurations.
   * @param {Partial<BasicRollConfiguration>} original  The initial configuration that will be merged into.
   * @param {Partial<BasicRollConfiguration>} other     The configuration to merge.
   * @returns {Partial<BasicRollConfiguration>}         The original instance.
   */
  static mergeConfigs(original, other={}) {
    if ( other.data ) {
      original.data ??= {};
      Object.assign(original.data, other.data);
    }

    if ( other.parts?.length ) {
      original.parts ??= [];
      original.parts.unshift(...other.parts);
    }

    if ( other.options ) {
      original.options ??= {};
      foundry.utils.mergeObject(original.options, other.options);
    }

    return original;
  }
}

/**
 * Configuration data for the process of rolling d20 rolls.
 *
 * @typedef {BasicRollProcessConfiguration} D20RollProcessConfiguration
 * @property {boolean} [advantage]             Apply advantage to each roll.
 * @property {boolean} [disadvantage]          Apply disadvantage to each roll.
 * @property {boolean} [elvenAccuracy]         Use three dice when rolling with advantage.
 * @property {boolean} [halflingLucky]         Add a re-roll once modifier to the d20 die.
 * @property {boolean} [reliableTalent]        Set the minimum for the d20 roll to 10.
 * @property {D20RollConfiguration[]} rolls    Configuration data for individual rolls.
 */

/**
 * D20 roll configuration data.
 *
 * @typedef {BasicRollConfiguration} D20RollConfiguration
 * @property {string[]} parts          Parts used to construct the roll formula, not including the d20 die.
 * @property {D20RollOptions} options  Options passed through to the roll.
 */

/**
 * Options that describe a d20 roll.
 *
 * @typedef {BasicRollOptions} D20RollOptions
 * @property {boolean} [advantage]       Does this roll potentially have advantage?
 * @property {boolean} [disadvantage]    Does this roll potentially have disadvantage?
 * @property {D20Roll.ADV_MODE} [advantageMode]  Final advantage mode.
 * @property {number} [criticalSuccess]  The value of the d20 die to be considered a critical success.
 * @property {number} [criticalFailure]  The value of the d20 die to be considered a critical failure.
 * @property {boolean} [elvenAccuracy]   Use three dice when rolling with advantage.
 * @property {boolean} [halflingLucky]   Add a re-roll once modifier to the d20 die.
 * @property {number} [maximum]          Maximum number the d20 die can roll.
 * @property {number} [minimum]          Minimum number the d20 die can roll.
 */

/* -------------------------------------------- */

/**
 * A type of Roll specific to a d20-based check, save, or attack roll in the 5e system.
 * @param {string} formula          The string formula to parse.
 * @param {object} data             The data object against which to parse attributes within the formula.
 * @param {D20RollOptions} options  Additional options that describe the d20 roll.
 */
class D20Roll extends BasicRoll {
  constructor(formula, data, options) {
    super(formula, data, options);
    this.#createD20Die();
    if ( !this.options.configured ) this.configureModifiers();
  }

  /* -------------------------------------------- */

  /**
   * Advantage mode of a 5e d20 roll
   * @enum {number}
   */
  static ADV_MODE = {
    NORMAL: 0,
    ADVANTAGE: 1,
    DISADVANTAGE: -1
  };

  /* -------------------------------------------- */

  /** @inheritDoc */
  static DefaultConfigurationDialog = D20RollConfigurationDialog;

  /* -------------------------------------------- */
  /*  Static Construction                         */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static fromConfig(config, process) {
    const formula = [new CONFIG.Dice.D20Die().formula].concat(config.parts ?? []).join(" + ");
    config.options.criticalSuccess ??= CONFIG.Dice.D20Die.CRITICAL_SUCCESS_TOTAL;
    config.options.criticalFailure ??= CONFIG.Dice.D20Die.CRITICAL_FAILURE_TOTAL;
    config.options.elvenAccuracy ??= process.elvenAccuracy;
    config.options.halflingLucky ??= process.halflingLucky;
    config.options.reliableTalent ??= process.reliableTalent;
    config.options.target ??= process.target;
    return new this(formula, config.data, config.options);
  }

  /* -------------------------------------------- */

  /**
   * Create a D20Roll from a standard Roll instance.
   * @param {Roll} roll
   * @returns {D20Roll}
   */
  static fromRoll(roll) {
    const newRoll = new this(roll.formula, roll.data, roll.options);
    Object.assign(newRoll, roll);
    return newRoll;
  }

  /* -------------------------------------------- */

  /**
   * Determines whether the roll should be fast forwarded and what the default advantage mode should be.
   * @param {D20RollProcessConfiguration} config     Roll configuration data.
   * @param {BasicRollDialogConfiguration} dialog    Data for the roll configuration dialog.
   * @param {BasicRollMessageConfiguration} message  Configuration data that guides roll message creation.
   */
  static applyKeybindings(config, dialog, message) {
    const keys = {
      normal: areKeysPressed(config.event, "skipDialogNormal"),
      advantage: areKeysPressed(config.event, "skipDialogAdvantage"),
      disadvantage: areKeysPressed(config.event, "skipDialogDisadvantage")
    };

    // Should the roll configuration dialog be displayed?
    dialog.configure ??= !Object.values(keys).some(k => k);

    // Determine advantage mode
    for ( const roll of config.rolls ?? [] ) {
      const advantage = roll.options.advantage || config.advantage || keys.advantage;
      const disadvantage = roll.options.disadvantage || config.disadvantage || keys.disadvantage;
      if ( advantage && !disadvantage ) roll.options.advantageMode = this.ADV_MODE.ADVANTAGE;
      else if ( !advantage && disadvantage ) roll.options.advantageMode = this.ADV_MODE.DISADVANTAGE;
      else roll.options.advantageMode = this.ADV_MODE.NORMAL;
    }
  }

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * The primary die used in this d20 roll.
   * @type {D20Die|void}
   */
  get d20() {
    if ( !(this.terms[0] instanceof foundry.dice.terms.Die) ) return;
    if ( !(this.terms[0] instanceof CONFIG.Dice.D20Die) ) this.#createD20Die();
    return this.terms[0];
  }

  /* -------------------------------------------- */

  /**
   * Set the d20 for this roll.
   */
  set d20(die) {
    if ( !(die instanceof CONFIG.Dice.D20Die) ) throw new Error(
      `D20 die must be an instance of ${CONFIG.Dice.D20Die.name}, instead a ${die.constructor.name} was provided.`
    );
    this.terms[0] = die;
  }

  /* -------------------------------------------- */

  /**
   * A convenience reference for whether this D20Roll has advantage.
   * @type {boolean}
   */
  get hasAdvantage() {
    return this.options.advantageMode === this.constructor.ADV_MODE.ADVANTAGE;
  }

  /* -------------------------------------------- */

  /**
   * A convenience reference for whether this D20Roll has disadvantage.
   * @type {boolean}
   */
  get hasDisadvantage() {
    return this.options.advantageMode === this.constructor.ADV_MODE.DISADVANTAGE;
  }

  /* -------------------------------------------- */

  /**
   * Is this roll a critical success? Returns undefined if roll isn't evaluated.
   * @type {boolean|void}
   */
  get isCritical() {
    return this.d20.isCriticalSuccess;
  }

  /* -------------------------------------------- */

  /**
   * Is this roll a critical failure? Returns undefined if roll isn't evaluated.
   * @type {boolean|void}
   */
  get isFumble() {
    return this.d20.isCriticalFailure;
  }

  /* -------------------------------------------- */

  /**
   * Does this roll start with a d20?
   * @type {boolean}
   */
  get validD20Roll() {
    return (this.d20 instanceof CONFIG.Dice.D20Die) && this.d20.isValid;
  }

  /* -------------------------------------------- */
  /*  Chat Messages                               */
  /* -------------------------------------------- */

  /** @override */
  static _prepareMessageData(rolls, messageData) {
    let advantage = true;
    let disadvantage = true;

    const rtLabel = game.i18n.localize("DND5E.FlagsReliableTalent");
    for ( const roll of rolls ) {
      if ( !roll.validD20Roll ) continue;
      if ( !roll.hasAdvantage ) advantage = false;
      if ( !roll.hasDisadvantage ) disadvantage = false;
      if ( roll.options.reliableTalent && roll.d20.results.every(r => !r.active || (r.result < 10)) ) {
        roll.d20.options.flavor = roll.d20.options.flavor ? `${roll.d20.options.flavor} (${rtLabel})` : rtLabel;
      }
    }

    messageData.flavor ??= "";
    if ( advantage ) messageData.flavor += ` (${game.i18n.localize("DND5E.Advantage")})`;
    else if ( disadvantage ) messageData.flavor += ` (${game.i18n.localize("DND5E.Disadvantage")})`;
  }

  /* -------------------------------------------- */
  /*  Roll Configuration                          */
  /* -------------------------------------------- */

  /**
   * Apply optional modifiers which customize the behavior of the d20term
   * @private
   */
  configureModifiers() {
    if ( !this.validD20Roll ) return;

    if ( this.options.advantageMode === undefined ) {
      const { advantage, disadvantage } = this.options;
      if ( advantage && !disadvantage ) this.options.advantageMode = this.constructor.ADV_MODE.ADVANTAGE;
      else if ( !advantage && disadvantage ) this.options.advantageMode = this.constructor.ADV_MODE.DISADVANTAGE;
      else this.options.advantageMode = this.constructor.ADV_MODE.NORMAL;
    }

    // Determine minimum, taking reliable talent into account
    let minimum = this.options.minimum;
    if ( this.options.reliableTalent ) minimum = Math.max(minimum ?? -Infinity, 10);

    // Directly modify the d20
    this.d20.applyFlag("elvenAccuracy", this.options.elvenAccuracy === true);
    this.d20.applyFlag("halflingLucky", this.options.halflingLucky === true);
    this.d20.applyAdvantage(this.options.advantageMode);
    this.d20.applyRange({ minimum, maximum: this.options.maximum });

    // Assign critical and fumble thresholds
    if ( this.options.criticalSuccess ) this.d20.options.criticalSuccess = this.options.criticalSuccess;
    if ( this.options.criticalFailure ) this.d20.options.criticalFailure = this.options.criticalFailure;
    if ( this.options.target ) this.d20.options.target = this.options.target;

    // Re-compile the underlying formula
    this.resetFormula();

    // Mark configuration as complete
    this.options.configured = true;
  }

  /* -------------------------------------------- */

  /**
   * Ensure the d20 die for this roll is actually a D20Die instance.
   */
  #createD20Die() {
    if ( this.terms[0] instanceof CONFIG.Dice.D20Die ) return;
    if ( !(this.terms[0] instanceof foundry.dice.terms.Die) ) return;
    const { number, faces, ...data } = this.terms[0];
    this.terms[0] = new CONFIG.Dice.D20Die({ ...data, number, faces });
  }

  /* -------------------------------------------- */
  /*  Configuration Dialog                        */
  /* -------------------------------------------- */

  /**
   * Create a Dialog prompt used to configure evaluation of an existing D20Roll instance.
   * @param {object} data                     Dialog configuration data
   * @param {string} [data.title]             The title of the shown dialog window
   * @param {number} [data.defaultRollMode]   The roll mode that the roll mode select element should default to
   * @param {number} [data.defaultAction]     The button marked as default
   * @param {FormSelectOption[]} [data.ammunitionOptions]  Selectable ammunition options.
   * @param {FormSelectOption[]} [data.attackModes]        Selectable attack modes.
   * @param {boolean} [data.chooseModifier]   Choose which ability modifier should be applied to the roll?
   * @param {string} [data.defaultAbility]    For tool rolls, the default ability modifier applied to the roll
   * @param {FormSelectOption[]} [data.masteryOptions]     Selectable weapon masteries.
   * @param {string} [data.template]          A custom path to an HTML template to use instead of the default
   * @param {object} options                  Additional Dialog customization options
   * @returns {Promise<D20Roll|null>}         A resulting D20Roll object constructed with the dialog, or null if the
   *                                          dialog was closed
   */
  async configureDialog({
    title, defaultRollMode, defaultAction=D20Roll.ADV_MODE.NORMAL, ammunitionOptions,
    attackModes, chooseModifier=false, defaultAbility, masteryOptions, template
  }={}, options={}) {
    foundry.utils.logCompatibilityWarning(
      "The `configureDialog` on D20Roll has been deprecated and is now handled through `D20Roll.build`.",
      { since: "DnD5e 4.1", until: "DnD5e 4.5" }
    );
    let DialogClass = this.constructor.DefaultConfigurationDialog;
    if ( chooseModifier ) DialogClass = SkillToolRollConfigurationDialog;
    else if ( ammunitionOptions || attackModes || masteryOptions ) DialogClass = AttackRollConfigurationDialog;
    const defaultButton = {
      [D20Roll.ADV_MODE.NORMAL]: "normal",
      [D20Roll.ADV_MODE.ADVANTAGE]: "advantage",
      [D20Roll.ADV_MODE.DISADVANTAGE]: "disadvantage"
    }[String(defaultAction ?? "0")];
    return await DialogClass.configure(
      { rolls: [{ parts: [this.formula.replace(roll.d20.formula, "")], options: this.options }] },
      { options: { ammunitionOptions, attackModes, defaultButton, masteryOptions, title } },
      { rollMode: defaultRollMode }
    );
  }
}

/* -------------------------------------------- */

/**
 * Translate new config objects back into old config objects for deprecated hooks.
 * @param {D20RollProcessConfiguration} rollConfig
 * @param {BasicRollDialogConfiguration} dialogConfig
 * @param {BasicRollMessageConfiguration} messageConfig
 * @returns {DeprecatedD20RollConfiguration}
 * @internal
 */
function _createDeprecatedD20Config(rollConfig, dialogConfig, messageConfig) {
  const oldConfig = {
    parts: rollConfig.rolls[0].parts,
    data: rollConfig.rolls[0].data,
    event: rollConfig.event,
    advantage: rollConfig.rolls[0].options?.advantage,
    disadvantage: rollConfig.rolls[0].options?.disadvantage,
    critical: rollConfig.rolls[0].options?.criticalSuccess,
    fumble: rollConfig.rolls[0].options?.criticalFailure,
    targetValue: rollConfig.target,
    ammunition: rollConfig.ammunition,
    attackMode: rollConfig.attackMode,
    mastery: rollConfig.mastery,
    elvenAccuracy: rollConfig.elvenAccuracy,
    halflingLucky: rollConfig.halflingLucky,
    reliableTalent: rollConfig.reliableTalent,
    ammunitionOptions: dialogConfig.options?.ammunitionOptions,
    attackModes: dialogConfig.options?.attackModeOptions,
    chooseModifier: dialogConfig.options?.chooseAbility,
    masteryOptions: dialogConfig?.options?.masteryOptions,
    title: dialogConfig.options?.title,
    dialogOptions: dialogConfig.options,
    chatMessage: messageConfig.create,
    messageData: messageConfig.data,
    rollMode: messageConfig.rollMode,
    flavor: messageConfig.data?.flavor
  };
  if ( "configure" in dialogConfig ) oldConfig.fastForward = !dialogConfig.configure;
  return oldConfig;
}

/* -------------------------------------------- */

/**
 * Apply changes from old config objects back onto new config objects.
 * @param {D20RollProcessConfiguration} rollConfig
 * @param {BasicRollDialogConfiguration} dialogConfig
 * @param {BasicRollMessageConfiguration} messageConfig
 * @param {DeprecatedD20RollConfiguration} options
 * @internal
 */
function _applyDeprecatedD20Configs(rollConfig, dialogConfig, messageConfig, options) {
  const set = (config, keyPath, value) => {
    if ( value === undefined ) return;
    foundry.utils.setProperty(config, keyPath, value);
  };

  let roll = rollConfig.rolls?.[0] ?? {};
  set(roll, "parts", options.parts);
  set(roll, "data", options.data);
  set(rollConfig, "event", options.event);
  set(roll, "options.advantage", options.advantage);
  set(roll, "options.disadvantage", options.disadvantage);
  set(roll, "options.criticalSuccess", options.critical);
  set(roll, "options.criticalFailure", options.fumble);
  set(rollConfig, "target", options.targetValue);
  set(rollConfig, "ammunition", options.ammunition);
  set(rollConfig, "attackMode", options.attackMode);
  set(rollConfig, "mastery", options.mastery);
  set(rollConfig, "elvenAccuracy", options.elvenAccuracy);
  set(rollConfig, "halflingLucky", options.halflingLucky);
  set(rollConfig, "reliableTalent", options.reliableTalent);
  if ( "fastForward" in options ) dialogConfig.configure = !options.fastForward;
  set(dialogConfig, "options", options.dialogOptions);
  set(dialogConfig, "options.ammunitionOptions", options.ammunitionOptions);
  set(dialogConfig, "options.attackModeOptions", options.attackModes);
  set(dialogConfig, "options.chooseAbility", options.chooseModifier);
  set(dialogConfig, "options.masteryOptions", options.masteryOptions);
  set(dialogConfig, "options.title", options.title);
  set(messageConfig, "create", options.chatMessage);
  set(messageConfig, "data", options.messageData);
  set(messageConfig, "rollMode", options.rollMode);
  set(messageConfig, "data.flavor", options.flavor);

  if ( !foundry.utils.isEmpty(roll) ) {
    rollConfig.rolls ??= [];
    if ( rollConfig.rolls[0] ) rollConfig.rolls[0] = roll;
    else rollConfig.rolls.push(roll);
  }
}

/**
 * Activity for making attacks and rolling damage.
 */
class AttackActivity extends ActivityMixin(AttackActivityData) {
  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static LOCALIZATION_PREFIXES = [...super.LOCALIZATION_PREFIXES, "DND5E.ATTACK"];

  /* -------------------------------------------- */

  /** @inheritDoc */
  static metadata = Object.freeze(
    foundry.utils.mergeObject(super.metadata, {
      type: "attack",
      img: "systems/dnd5e/icons/svg/activity/attack.svg",
      title: "DND5E.ATTACK.Title.one",
      sheetClass: AttackSheet,
      usage: {
        actions: {
          rollAttack: AttackActivity.#rollAttack,
          rollDamage: AttackActivity.#rollDamage
        }
      }
    }, { inplace: false })
  );

  /* -------------------------------------------- */
  /*  Activation                                  */
  /* -------------------------------------------- */

  /** @override */
  _usageChatButtons(message) {
    const buttons = [{
      label: game.i18n.localize("DND5E.Attack"),
      icon: '<i class="dnd5e-icon" data-src="systems/dnd5e/icons/svg/trait-weapon-proficiencies.svg" inert></i>',
      dataset: {
        action: "rollAttack"
      }
    }];
    if ( this.damage.parts.length || this.item.system.properties?.has("amm") ) buttons.push({
      label: game.i18n.localize("DND5E.Damage"),
      icon: '<i class="fa-solid fa-burst" inert></i>',
      dataset: {
        action: "rollDamage"
      }
    });
    return buttons.concat(super._usageChatButtons(message));
  }

  /* -------------------------------------------- */

  /** @override */
  async _triggerSubsequentActions(config, results) {
    this.rollAttack({ event: config.event }, {}, { data: { "flags.dnd5e.originatingMessage": results.message?.id } });
  }

  /* -------------------------------------------- */
  /*  Rolling                                     */
  /* -------------------------------------------- */

  /**
   * @typedef {D20RollProcessConfiguration} AttackRollProcessConfiguration
   * @property {string|boolean} [ammunition]  Specific ammunition to consume, or `false` to prevent any ammo usage.
   * @property {string} [attackMode]          Mode to use for making the attack and rolling damage.
   * @property {string} [mastery]             Weapon mastery option to use.
   */

  /**
   * @typedef {BasicRollDialogConfiguration} AttackRollDialogConfiguration
   * @property {AttackRollConfigurationDialogOptions} [options]  Configuration options.
   */

  /**
   * @typedef {object} AmmunitionUpdate
   * @property {string} id        ID of the ammunition item to update.
   * @property {boolean} destroy  Will the ammunition item be deleted?
   * @property {number} quantity  New quantity after the ammunition is spent.
   */

  /**
   * Perform an attack roll.
   * @param {AttackRollProcessConfiguration} config  Configuration information for the roll.
   * @param {AttackRollDialogConfiguration} dialog   Configuration for the roll dialog.
   * @param {BasicRollMessageConfiguration} message  Configuration for the roll message.
   * @returns {Promise<D20Roll[]|null>}
   */
  async rollAttack(config={}, dialog={}, message={}) {
    const targets = getTargetDescriptors();

    if ( (this.item.type === "weapon") && (this.item.system.quantity === 0) ) {
      ui.notifications.warn("DND5E.ATTACK.Warning.NoQuantity", { localize: true });
    }

    const buildConfig = this._buildAttackConfig.bind(this);

    const rollConfig = foundry.utils.mergeObject({
      ammunition: this.item.getFlag("dnd5e", `last.${this.id}.ammunition`),
      attackMode: this.item.getFlag("dnd5e", `last.${this.id}.attackMode`),
      elvenAccuracy: this.actor?.getFlag("dnd5e", "elvenAccuracy")
        && CONFIG.DND5E.characterFlags.elvenAccuracy.abilities.includes(this.ability),
      halflingLucky: this.actor?.getFlag("dnd5e", "halflingLucky"),
      mastery: this.item.getFlag("dnd5e", `last.${this.id}.mastery`),
      target: targets.length === 1 ? targets[0].ac : undefined
    }, config);

    const ammunitionOptions = this.item.system.ammunitionOptions ?? [];
    if ( ammunitionOptions.length ) ammunitionOptions.unshift({ value: "", label: "" });
    if ( rollConfig.ammunition === undefined ) rollConfig.ammunition = ammunitionOptions?.[1]?.value;
    else if ( !ammunitionOptions?.find(m => m.value === rollConfig.ammunition) ) {
      rollConfig.ammunition = ammunitionOptions?.[0]?.value;
    }
    const attackModeOptions = this.item.system.attackModes;
    if ( !attackModeOptions?.find(m => m.value === rollConfig.attackMode) ) {
      rollConfig.attackMode = attackModeOptions?.[0]?.value;
    }
    const masteryOptions = this.item.system.masteryOptions;
    if ( !masteryOptions?.find(m => m.value === rollConfig.mastery) ) {
      rollConfig.mastery = masteryOptions?.[0]?.value;
    }

    rollConfig.hookNames = [...(config.hookNames ?? []), "attack", "d20Test"];
    rollConfig.rolls = [BasicRoll.mergeConfigs({
      options: {
        ammunition: rollConfig.ammunition,
        attackMode: rollConfig.attackMode,
        criticalSuccess: this.criticalThreshold,
        mastery: rollConfig.mastery
      }
    }, config.rolls?.shift())].concat(config.rolls ?? []);
    rollConfig.subject = this;

    const dialogConfig = foundry.utils.mergeObject({
      applicationClass: AttackRollConfigurationDialog,
      options: {
        ammunitionOptions: rollConfig.ammunition !== false ? ammunitionOptions : [],
        attackModeOptions,
        buildConfig,
        masteryOptions: (masteryOptions?.length > 1) && !config.mastery ? masteryOptions : [],
        position: {
          top: config.event ? config.event.clientY - 80 : null,
          left: window.innerWidth - 710
        },
        window: {
          title: game.i18n.localize("DND5E.AttackRoll"),
          subtitle: this.item.name,
          icon: this.item.img
        }
      }
    }, dialog);

    const messageConfig = foundry.utils.mergeObject({
      create: true,
      data: {
        flavor: `${this.item.name} - ${game.i18n.localize("DND5E.AttackRoll")}`,
        flags: {
          dnd5e: {
            ...this.messageFlags,
            messageType: "roll",
            roll: { type: "attack" }
          }
        },
        speaker: ChatMessage.getSpeaker({ actor: this.actor })
      }
    }, message);

    if ( "dnd5e.preRollAttack" in Hooks.events ) {
      foundry.utils.logCompatibilityWarning(
        "The `dnd5e.preRollAttack` hook has been deprecated and replaced with `dnd5e.preRollAttackV2`.",
        { since: "DnD5e 4.0", until: "DnD5e 4.4" }
      );
      const oldConfig = _createDeprecatedD20Config(rollConfig, dialogConfig, messageConfig);
      if ( Hooks.call("dnd5e.preRollAttack", this.item, oldConfig) === false ) return null;
      _applyDeprecatedD20Configs(rollConfig, dialogConfig, messageConfig, oldConfig);
    }

    const rolls = await CONFIG.Dice.D20Roll.buildConfigure(rollConfig, dialogConfig, messageConfig);
    await CONFIG.Dice.D20Roll.buildEvaluate(rolls, rollConfig, messageConfig);
    if ( !rolls.length ) return null;
    for ( const key of ["ammunition", "attackMode", "mastery"] ) {
      if ( !rolls[0].options[key] ) continue;
      foundry.utils.setProperty(messageConfig.data, `flags.dnd5e.roll.${key}`, rolls[0].options[key]);
    }
    await CONFIG.Dice.D20Roll.buildPost(rolls, rollConfig, messageConfig);

    const flags = {};
    let ammoUpdate = null;

    if ( rolls[0].options.ammunition ) {
      const ammo = this.actor?.items.get(rolls[0].options.ammunition);
      if ( ammo ) {
        if ( !ammo.system.properties?.has("ret") ) {
          ammoUpdate = { id: ammo.id, quantity: Math.max(0, ammo.system.quantity - 1) };
          ammoUpdate.destroy = ammo.system.uses.autoDestroy && (ammoUpdate.quantity === 0);
        }
        flags.ammunition = rolls[0].options.ammunition;
      }
    } else if ( rolls[0].options.attackMode?.startsWith("thrown") && !this.item.system.properties?.has("ret") ) {
      ammoUpdate = { id: this.item.id, quantity: Math.max(0, this.item.system.quantity - 1) };
    } else if ( !rolls[0].options.ammunition && dialogConfig.options?.ammunitionOptions?.length ) {
      flags.ammunition = "";
    }
    if ( rolls[0].options.attackMode ) flags.attackMode = rolls[0].options.attackMode;
    else if ( rollConfig.attackMode ) rolls[0].options.attackMode = rollConfig.attackMode;
    if ( rolls[0].options.mastery ) flags.mastery = rolls[0].options.mastery;
    if ( !foundry.utils.isEmpty(flags) && this.actor.items.has(this.item.id) ) {
      await this.item.setFlag("dnd5e", `last.${this.id}`, flags);
    }

    /**
     * A hook event that fires after an attack has been rolled but before any ammunition is consumed.
     * @function dnd5e.rollAttackV2
     * @memberof hookEvents
     * @param {D20Roll[]} rolls                        The resulting rolls.
     * @param {object} data
     * @param {AttackActivity|null} data.subject       The Activity that performed the attack.
     * @param {AmmunitionUpdate|null} data.ammoUpdate  Any updates related to ammo consumption for this attack.
     */
    Hooks.callAll("dnd5e.rollAttackV2", rolls, { subject: this, ammoUpdate });

    if ( "dnd5e.rollAttack" in Hooks.events ) {
      foundry.utils.logCompatibilityWarning(
        "The `dnd5e.rollAttack` hook has been deprecated and replaced with `dnd5e.rollAttackV2`.",
        { since: "DnD5e 4.0", until: "DnD5e 4.4" }
      );
      const oldAmmoUpdate = ammoUpdate ? [{ _id: ammoUpdate.id, "system.quantity": ammoUpdate.quantity }] : [];
      Hooks.callAll("dnd5e.rollAttack", this.item, rolls[0], oldAmmoUpdate);
      if ( oldAmmoUpdate[0] ) {
        ammoUpdate.id = oldAmmoUpdate[0]._id;
        ammoUpdate.quantity = foundry.utils.getProperty(oldAmmoUpdate[0], "system.quantity");
      }
    }

    // Commit ammunition consumption on attack rolls resource consumption if the attack roll was made
    if ( ammoUpdate?.destroy ) {
      // If ammunition was deleted, store a copy of it in the roll message
      const data = this.actor.items.get(ammoUpdate.id).toObject();
      const messageId = messageConfig.data?.flags?.dnd5e?.originatingMessage
        ?? rollConfig.event?.target.closest("[data-message-id]")?.dataset.messageId;
      const attackMessage = dnd5e.registry.messages.get(messageId, "attack")?.pop();
      await attackMessage?.setFlag("dnd5e", "roll.ammunitionData", data);
      await this.actor.deleteEmbeddedDocuments("Item", [ammoUpdate.id]);
    }
    else if ( ammoUpdate ) await this.actor?.updateEmbeddedDocuments("Item", [
      { _id: ammoUpdate.id, "system.quantity": ammoUpdate.quantity }
    ]);

    /**
     * A hook event that fires after an attack has been rolled and ammunition has been consumed.
     * @function dnd5e.postRollAttack
     * @memberof hookEvents
     * @param {D20Roll[]} rolls                   The resulting rolls.
     * @param {object} data
     * @param {AttackActivity|null} data.subject  The activity that performed the attack.
     */
    Hooks.callAll("dnd5e.postRollAttack", rolls, { subject: this });

    return rolls;
  }

  /* -------------------------------------------- */

  /**
   * Configure a roll config for each roll performed as part of the attack process. Will be called once per roll
   * in the process each time an option is changed in the roll configuration interface.
   * @param {D20RollProcessConfiguration} process          Configuration for the entire rolling process.
   * @param {D20RollConfiguration} config                  Configuration for a specific roll.
   * @param {FormDataExtended} [formData]                  Any data entered into the rolling prompt.
   * @param {number} index                                 Index of the roll within all rolls being prepared.
   */
  _buildAttackConfig(process, config, formData, index) {
    const ammunition = formData?.get("ammunition") ?? process.ammunition;
    const attackMode = formData?.get("attackMode") ?? process.attackMode;
    const mastery = formData?.get("mastery") ?? process.mastery;

    let { parts, data } = this.getAttackData({ ammunition, attackMode });
    const options = config.options ?? {};
    if ( ammunition !== undefined ) options.ammunition = ammunition;
    if ( attackMode !== undefined ) options.attackMode = attackMode;
    if ( mastery !== undefined ) options.mastery = mastery;

    config.parts = [...(config.parts ?? []), ...parts];
    config.data = { ...data, ...(config.data ?? {}) };
    config.options = options;
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /**
   * Handle performing an attack roll.
   * @this {AttackActivity}
   * @param {PointerEvent} event     Triggering click event.
   * @param {HTMLElement} target     The capturing HTML element which defined a [data-action].
   * @param {ChatMessage5e} message  Message associated with the activation.
   */
  static #rollAttack(event, target, message) {
    this.rollAttack({ event });
  }

  /* -------------------------------------------- */

  /**
   * Handle performing a damage roll.
   * @this {AttackActivity}
   * @param {PointerEvent} event     Triggering click event.
   * @param {HTMLElement} target     The capturing HTML element which defined a [data-action].
   * @param {ChatMessage5e} message  Message associated with the activation.
   */
  static #rollDamage(event, target, message) {
    const lastAttack = message.getAssociatedRolls("attack").pop();
    const attackMode = lastAttack?.getFlag("dnd5e", "roll.attackMode");

    // Fetch the ammunition used with the last attack roll
    let ammunition;
    const actor = lastAttack?.getAssociatedActor();
    if ( actor ) {
      const storedData = lastAttack.getFlag("dnd5e", "roll.ammunitionData");
      ammunition = storedData
        ? new Item.implementation(storedData, { parent: actor })
        : actor.items.get(lastAttack.getFlag("dnd5e", "roll.ammunition"));
    }

    const isCritical = lastAttack?.rolls[0]?.isCritical;
    const dialogConfig = {};
    if ( isCritical ) dialogConfig.options = { defaultButton: "critical" };

    this.rollDamage({ event, ammunition, attackMode, isCritical }, dialogConfig);
  }

  /* -------------------------------------------- */
  /*  Helpers                                     */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async getFavoriteData() {
    return foundry.utils.mergeObject(await super.getFavoriteData(), { modifier: this.labels.modifier });
  }
}

/**
 * Sheet for the cast activity.
 */
class CastSheet extends ActivitySheet {

  /** @inheritDoc */
  static DEFAULT_OPTIONS = {
    classes: ["cast-activity"],
    actions: {
      removeSpell: CastSheet.#removeSpell
    }
  };

  /* -------------------------------------------- */

  /** @inheritDoc */
  static PARTS = {
    ...super.PARTS,
    identity: {
      template: "systems/dnd5e/templates/activity/cast-identity.hbs",
      templates: super.PARTS.identity.templates
    },
    effect: {
      template: "systems/dnd5e/templates/activity/cast-effect.hbs",
      templates: [
        "systems/dnd5e/templates/activity/parts/cast-spell.hbs",
        "systems/dnd5e/templates/activity/parts/cast-details.hbs"
      ]
    }
  };

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareContext(options) {
    return {
      ...await super._prepareContext(options),
      spell: await fromUuid(this.activity.spell.uuid)
    };
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareEffectContext(context) {
    context = await super._prepareEffectContext(context);

    if ( context.spell ) {
      context.contentLink = context.spell.toAnchor().outerHTML;
      if ( context.spell.system.level > 0 ) context.levelOptions = Object.entries(CONFIG.DND5E.spellLevels)
        .filter(([level]) => Number(level) >= context.spell.system.level)
        .map(([value, label]) => ({ value, label }));
    }

    context.abilityOptions = [
      { value: "", label: game.i18n.localize("DND5E.Spellcasting") },
      { rule: true },
      ...Object.entries(CONFIG.DND5E.abilities).map(([value, { label }]) => ({ value, label }))
    ];
    context.propertyOptions = Array.from(CONFIG.DND5E.validProperties.spell).map(value => ({
      value, label: CONFIG.DND5E.itemProperties[value]?.label ?? ""
    }));

    return context;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareIdentityContext(context) {
    context = await super._prepareIdentityContext(context);
    if ( context.spell ) context.placeholder = { name: context.spell.name, img: context.spell.img };
    return context;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _getTabs() {
    const tabs = super._getTabs();
    tabs.effect.label = "DND5E.CAST.SECTIONS.Spell";
    tabs.effect.icon = "fa-solid fa-wand-sparkles";
    return tabs;
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /**
   * Handle removing the associated spell.
   * @this {CastSheet}
   * @param {Event} event         Triggering click event.
   * @param {HTMLElement} target  Button that was clicked.
   */
  static #removeSpell(event, target) {
    this.activity.update({ "spell.uuid": null });
  }
}

const { BooleanField: BooleanField$D, DocumentUUIDField: DocumentUUIDField$4, NumberField: NumberField$F, SchemaField: SchemaField$M, SetField: SetField$v, StringField: StringField$12 } = foundry.data.fields;

/**
 * Data model for a Cast activity.
 *
 * @property {object} spell
 * @property {string} spell.ability              Ability to override default spellcasting ability.
 * @property {object} spell.challenge
 * @property {number} spell.challenge.attack     Flat to hit bonus in place of the spell's normal attack bonus.
 * @property {number} spell.challenge.save       Flat DC to use in place of the spell's normal save DC.
 * @property {boolean} spell.challenge.override  Use custom attack bonus & DC rather than creature's.
 * @property {number} spell.level                Base level at which to cast the spell.
 * @property {Set<string>} spell.properties      Spell components & tags to ignore while casting.
 * @property {boolean} spell.spellbook           Display spell in the Spells tab of the character sheet.
 * @property {string} spell.uuid                 UUID of the spell to cast.
 */
class CastActivityData extends BaseActivityData {
  /** @inheritDoc */
  static defineSchema() {
    const schema = super.defineSchema();
    delete schema.effects;
    return {
      ...schema,
      spell: new SchemaField$M({
        ability: new StringField$12(),
        challenge: new SchemaField$M({
          attack: new NumberField$F(),
          save: new NumberField$F(),
          override: new BooleanField$D()
        }),
        level: new NumberField$F(),
        properties: new SetField$v(new StringField$12(), { initial: ["vocal", "somatic", "material"] }),
        spellbook: new BooleanField$D({ initial: true }),
        uuid: new DocumentUUIDField$4()
      })
    };
  }

  /* -------------------------------------------- */
  /*  Data Preparation                            */
  /* -------------------------------------------- */

  /** @inheritDoc */
  prepareData() {
    const spell = fromUuidSync(this.spell.uuid);
    if ( spell ) {
      this.name = this.name || spell.name;
      this.img = this.img || spell.img;
    }
    super.prepareData();
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  prepareFinalData(rollData) {
    super.prepareFinalData(rollData);

    for ( const field of ["activation", "duration", "range", "target"] ) {
      Object.defineProperty(this[field], "canOverride", {
        value: true,
        configurable: true,
        enumerable: false
      });
    }
  }
}

/**
 * Activity for casting a spell from another item.
 */
class CastActivity extends ActivityMixin(CastActivityData) {
  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /**
   * Static ID used for the enchantment that modifies spell data.
   */
  static ENCHANTMENT_ID = staticID("dnd5espellchanges");

  /* -------------------------------------------- */

  /** @inheritDoc */
  static LOCALIZATION_PREFIXES = [...super.LOCALIZATION_PREFIXES, "DND5E.CAST"];

  /* -------------------------------------------- */

  /** @inheritDoc */
  static metadata = Object.freeze(
    foundry.utils.mergeObject(super.metadata, {
      type: "cast",
      img: "systems/dnd5e/icons/svg/activity/cast.svg",
      title: "DND5E.CAST.Title",
      sheetClass: CastSheet
    }, { inplace: false })
  );

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * Cached copy of the associated spell stored on the actor.
   * @type {Item5e|void}
   */
  get cachedSpell() {
    return this.actor?.sourcedItems.get(this.spell.uuid, { legacy: false })
      ?.find(i => i.getFlag("dnd5e", "cachedFor") === this.relativeUUID);
  }

  /* -------------------------------------------- */

  /**
   * Should this spell be listed in the actor's spellbook?
   * @type {boolean}
   */
  get displayInSpellbook() {
    return (this.item.system.magicAvailable !== false) && this.spell.spellbook;
  }

  /* -------------------------------------------- */
  /*  Activation                                  */
  /* -------------------------------------------- */

  /** @override */
  async use(usage={}, dialog={}, message={}) {
    if ( !this.item.isEmbedded || this.item.pack ) return;
    if ( !this.item.isOwner ) {
      ui.notifications.error("DND5E.DocumentUseWarn", { localize: true });
      return;
    }

    /**
     * A hook event that fires before a linked spell is used by a Cast activity.
     * @function dnd5e.preUseLinkedSpell
     * @memberof hookEvents
     * @param {CastActivity} activity                                Cast activity being used.
     * @param {Partial<ActivityUseConfiguration>} usageConfig        Configuration info for the activation.
     * @param {Partial<ActivityDialogConfiguration>} dialogConfig    Configuration info for the usage dialog.
     * @param {Partial<ActivityMessageConfiguration>} messageConfig  Configuration info for the created chat message.
     * @returns {boolean}  Explicitly return `false` to prevent activity from being used.
     */
    if ( Hooks.call("dnd5e.preUseLinkedSpell", this, usage, dialog, message) === false ) return;

    let spell = this.cachedSpell;
    if ( !spell ) {
      [spell] = await this.actor.createEmbeddedDocuments("Item", [await this.getCachedSpellData()]);
    }

    const results = await spell.use({ ...usage, legacy: false }, dialog, message);

    /**
     * A hook event that fires after a linked spell is used by a Cast activity.
     * @function dnd5e.postUseLinkedSpell
     * @memberof hookEvents
     * @param {CastActivity} activity                          Activity being activated.
     * @param {Partial<ActivityUseConfiguration>} usageConfig  Configuration data for the activation.
     * @param {ActivityUsageResults} results                   Final details on the activation.
     */
    if ( results ) Hooks.callAll("dnd5e.postUseLinkedSpell", this, usage, results);

    return results;
  }

  /* -------------------------------------------- */
  /*  Helpers                                     */
  /* -------------------------------------------- */

  /**
   * Prepare the data for the cached spell to store on the actor.
   * @returns {Promise<object|void>}
   */
  async getCachedSpellData() {
    const originalSpell = await fromUuid(this.spell.uuid);
    if ( !originalSpell ) return;
    return originalSpell.clone({
      effects: [
        ...originalSpell.effects.map(e => e.toObject()),
        {
          _id: this.constructor.ENCHANTMENT_ID,
          type: "enchantment",
          name: game.i18n.localize("DND5E.CAST.Enchantment.Name"),
          img: "systems/dnd5e/icons/svg/activity/cast.svg",
          origin: this.uuid,
          changes: this.getSpellChanges()
        }
      ],
      flags: {
        dnd5e: {
          cachedFor: this.relativeUUID
        }
      },
      _stats: { compendiumSource: this.spell.uuid }
    }).toObject();
  }

  /* -------------------------------------------- */

  /**
   * Create spell changes based on the activity's configuration.
   * @returns {object[]}
   */
  getSpellChanges() {
    const changes = [];
    const source = this.toObject();

    // Override spell details
    for ( const type of ["activation", "duration", "range", "target"] ) {
      if ( !this[type].override ) continue;
      const data = source[type];
      delete data.override;
      changes.push({ key: `system.${type}`, mode: CONST.ACTIVE_EFFECT_MODES.OVERRIDE, value: JSON.stringify(data) });
    }

    // Set the casting ability
    if ( this.spell.ability ) changes.push({
      key: "system.ability", mode: CONST.ACTIVE_EFFECT_MODES.OVERRIDE, value: this.spell.ability
    });

    // Remove ignored properties
    for ( const property of this.spell.properties ) {
      changes.push({ key: "system.properties", mode: CONST.ACTIVE_EFFECT_MODES.ADD, value: `-${property}` });
    }

    // Set challenge overrides
    const challenge = this.spell.challenge;
    if ( challenge.override && challenge.attack ) changes.push(
      { key: "activities[attack].attack.bonus", mode: CONST.ACTIVE_EFFECT_MODES.OVERRIDE, value: challenge.attack },
      { key: "activities[attack].attack.flat", mode: CONST.ACTIVE_EFFECT_MODES.OVERRIDE, value: true }
    );
    if ( challenge.override && challenge.save ) changes.push(
      { key: "activities[save].save.dc.calculation", mode: CONST.ACTIVE_EFFECT_MODES.OVERRIDE, value: "" },
      { key: "activities[save].save.dc.formula", mode: CONST.ACTIVE_EFFECT_MODES.OVERRIDE, value: challenge.save }
    );

    return changes;
  }
}

/**
 * Object representing a nested set of choices to be displayed in a grouped select list or a trait selector.
 *
 * @typedef {object} SelectChoicesEntry
 * @property {string} label              Label, either pre- or post-localized.
 * @property {boolean} [chosen]          Has this choice been selected?
 * @property {boolean} [sorting=true]    Should this value be sorted? If there are a mixture of this value at
 *                                       a level, unsorted values are listed first followed by sorted values.
 * @property {SelectChoices} [children]  Nested choices. If wildcard filtering support is desired, then trait keys
 *                                       should be provided prefixed for children (e.g. `parent:child`, rather than
 *                                       just `child`).
 */

/**
 * Object with a number of methods for performing actions on a nested set of choices.
 *
 * @param {Object<string, SelectChoicesEntry>} [choices={}]  Initial choices for the object.
 */
class SelectChoices {
  constructor(choices={}) {
    const clone = foundry.utils.deepClone(choices);
    for ( const value of Object.values(clone) ) {
      if ( !value.children || (value.children instanceof SelectChoices) ) continue;
      value.category = true;
      value.children = new this.constructor(value.children);
    }
    Object.assign(this, clone);
  }

  /* -------------------------------------------- */

  /**
   * Create a set of available choice keys.
   * @param {Set<string>} [set]  Existing set to which the values will be added.
   * @returns {Set<string>}
   */
  asSet(set) {
    set ??= new Set();
    for ( const [key, choice] of Object.entries(this) ) {
      if ( choice.children ) choice.children.asSet(set);
      else set.add(key);
    }
    return set;
  }

  /* -------------------------------------------- */

  /**
   * Create a clone of this object.
   * @returns {SelectChoices}
   */
  clone() {
    const newData = {};
    for ( const [key, value] of Object.entries(this) ) {
      newData[key] = foundry.utils.deepClone(value);
      if ( value.children ) newData[key].children = value.children.clone();
    }
    const clone = new this.constructor(newData);
    return clone;
  }

  /* -------------------------------------------- */

  /**
   * Find key and value for the provided key or key suffix.
   * @param {string} key  Full prefixed key (e.g. `tool:art:alchemist`) or just the suffix (e.g. `alchemist`).
   * @returns {[string, SelectChoicesEntry]|null}  An array with the first value being the matched key,
   *                                               and the second being the value.
   */
  find(key) {
    for ( const [k, v] of Object.entries(this) ) {
      if ( (k === key) || k.endsWith(`:${key}`) ) {
        return [k, v];
      } else if ( v.children ) {
        const result = v.children.find(key);
        if ( result ) return result;
      }
    }
    return null;
  }

  /* -------------------------------------------- */

  /**
   * Execute the provided function for each entry in the object.
   * @param {Function} func  Function to execute on each entry. Receives the trait key and value.
   */
  forEach(func) {
    for ( const [key, value] of Object.entries(this) ) {
      func(key, value);
      if ( value.children ) value.children.forEach(func);
    }
  }

  /* -------------------------------------------- */

  /**
   * Merge another SelectChoices object into this one.
   * @param {SelectChoices} other
   * @param {object} [options={}]
   * @param {boolean} [options.inplace=true]  Should this SelectChoices be mutated or a new one returned?
   * @returns {SelectChoices}
   */
  merge(other, { inplace=true }={}) {
    if ( !inplace ) return this.clone().merge(other);
    return foundry.utils.mergeObject(this, other);
  }

  /* -------------------------------------------- */

  /**
   * Internal sorting method.
   * @param {object} lhs
   * @param {object} rhs
   * @returns {number}
   * @protected
   */
  _sort(lhs, rhs) {
    if ( (lhs.sorting === false) && (rhs.sorting === false) ) return 0;
    if ( lhs.sorting === false ) return -1;
    if ( rhs.sorting === false ) return 1;
    return lhs.label.localeCompare(rhs.label, game.i18n.lang);
  }

  /* -------------------------------------------- */

  /**
   * Sort the entries using the label.
   * @param {object} [options={}]
   * @param {boolean} [options.inplace=true]  Should this SelectChoices be mutated or a new one returned?
   * @returns {SelectChoices}
   */
  sort({ inplace=true }={}) {
    const sorted = new SelectChoices(sortObjectEntries(this, this._sort));

    if ( inplace ) {
      for ( const key of Object.keys(this) ) delete this[key];
      this.merge(sorted);
      for ( const entry of Object.values(this) ) {
        if ( entry.children ) entry.children.sort();
      }
      return this;
    }

    else {
      for ( const entry of Object.values(sorted) ) {
        if ( entry.children ) entry.children = entry.children.sort({ inplace });
      }
      return sorted;
    }
  }

  /* -------------------------------------------- */

  /**
   * Filters choices in place to only include the provided keys.
   * @param {Set<string>|SelectChoices} filter   Keys of traits to retain or another SelectChoices object.
   * @param {object} [options={}]
   * @param {boolean} [options.inplace=true]     Should this SelectChoices be mutated or a new one returned?
   * @returns {SelectChoices}                    This SelectChoices with filter applied.
   *
   * @example
   * const choices = new SelectChoices({
   *   categoryOne: { label: "One" },
   *   categoryTwo: { label: "Two", children: {
   *     childOne: { label: "Child One" },
   *     childTwo: { label: "Child Two" }
   *   } }
   * });
   *
   * // Results in only categoryOne
   * choices.filter(new Set(["categoryOne"]));
   *
   * // Results in only categoryTwo, but none if its children
   * choices.filter(new Set(["categoryTwo"]));
   *
   * // Results in categoryTwo and all of its children
   * choices.filter(new Set(["categoryTwo:*"]));
   *
   * // Results in categoryTwo with only childOne
   * choices.filter(new Set(["categoryTwo:childOne"]));
   *
   * // Results in categoryOne, plus categoryTwo with only childOne
   * choices.filter(new Set(["categoryOne", "categoryTwo:childOne"]));
   *
   * @example
   * const choices = new SelectChoices({
   *   "type:categoryOne": { label: "One" },
   *   "type:categoryTwo": { label: "Two", children: {
   *     "type:categoryOne:childOne": { label: "Child One" },
   *     "type:categoryOne:childTwo": { label: "Child Two" }
   *   } }
   * });
   *
   * // Results in no changes
   * choices.filter(new Set(["type:*"]));
   *
   * // Results in only categoryOne
   * choices.filter(new Set(["type:categoryOne"]));
   *
   * // Results in categoryTwo and all of its children
   * choices.filter(new Set(["type:categoryTwo:*"]));
   *
   * // Results in categoryTwo with only childOne
   * choices.filter(new Set(["type:categoryTwo:childOne"]));
   */
  filter(filter, { inplace=true }={}) {
    if ( !inplace ) return this.clone().filter(filter);
    if ( filter instanceof SelectChoices ) filter = filter.asSet();

    for ( const [key, trait] of Object.entries(this) ) {
      // Remove children if direct match and no wildcard for this category present
      const wildcardKey = key.replace(/(:|^)(\w+)$/, "$1*");
      if ( filter.has(key) && !filter.has(wildcardKey) ) {
        if ( trait.children ) delete trait.children;
      }

      // Check children, remove entry if no children match filter
      else if ( !filter.has(wildcardKey) && !filter.has(`${key}:*`) ) {
        if ( trait.children ) trait.children.filter(filter);
        if ( !Object.keys(trait.children ?? {}).length ) delete this[key];
      }
    }

    return this;
  }

  /* -------------------------------------------- */

  /**
   * Removes in place any traits or categories the keys of which are included in the exclusion set.
   * Note: Wildcard keys are not supported with this method.
   * @param {Set<string>} keys                Set of keys to remove from the choices.
   * @param {object} [options={}]
   * @param {boolean} [options.inplace=true]  Should this SelectChoices be mutated or a new one returned?
   * @returns {SelectChoices}                 This SelectChoices with excluded keys removed.
   *
   * @example
   * const choices = new SelectChoices({
   *   categoryOne: { label: "One" },
   *   categoryTwo: { label: "Two", children: {
   *     childOne: { label: "Child One" },
   *     childTwo: { label: "Child Two" }
   *   } }
   * });
   *
   * // Results in categoryOne being removed
   * choices.exclude(new Set(["categoryOne"]));
   *
   * // Results in categoryOne and childOne being removed, but categoryTwo and childTwo remaining
   * choices.exclude(new Set(["categoryOne", "categoryTwo:childOne"]));
   */
  exclude(keys, { inplace=true }={}) {
    if ( !inplace ) return this.clone().exclude(keys);
    for ( const [key, trait] of Object.entries(this) ) {
      if ( keys.has(key) ) delete this[key];
      else if ( trait.children ) trait.children = trait.children.exclude(keys);
    }
    return this;
  }
}

/**
 * Cached version of the base items compendia indices with the needed subtype fields.
 * @type {object}
 * @private
 */
const _cachedIndices = {};

/**
 * Determine the appropriate label to use for a trait category.
 * @param {object|string} data  Category for which to fetch the label.
 * @param {object} config       Trait configuration data.
 * @returns {string}
 * @private
 */
function _innerLabel(data, config) {
  return foundry.utils.getType(data) === "Object"
    ? foundry.utils.getProperty(data, config.labelKeyPath ?? "label") : data;
}

/* -------------------------------------------- */
/*  Application                                 */
/* -------------------------------------------- */

/**
 * Get the schema fields for this trait on the actor.
 * @param {Actor5e} actor  Actor for which to get the fields.
 * @param {string} trait   Trait as defined in `CONFIG.DND5E.traits`.
 * @returns {object|void}
 */
function actorFields(actor, trait) {
  const keyPath = actorKeyPath(trait);
  return (keyPath.startsWith("system.")
    ? actor.system.schema.getField(keyPath.slice(7))
    : actor.schema.getField(keyPath))?.fields;
}

/* -------------------------------------------- */

/**
 * Get the key path to the specified trait on an actor.
 * @param {string} trait  Trait as defined in `CONFIG.DND5E.traits`.
 * @returns {string}      Key path to this trait's object within an actor's system data.
 */
function actorKeyPath(trait) {
  const traitConfig = CONFIG.DND5E.traits[trait];
  if ( traitConfig.actorKeyPath ) return traitConfig.actorKeyPath;
  return `system.traits.${trait}`;
}

/* -------------------------------------------- */

/**
 * Get the current trait values for the provided actor.
 * @param {Actor5e} actor  Actor from which to retrieve the values.
 * @param {string} trait   Trait as defined in `CONFIG.DND5E.traits`.
 * @returns {Object<number>}
 */
async function actorValues(actor, trait) {
  const keyPath = actorKeyPath(trait);
  const data = foundry.utils.getProperty(actor, keyPath);
  if ( !data ) return {};
  const values = {};
  const traitChoices = await choices(trait, {prefixed: true});

  const setValue = (k, v) => {
    const result = traitChoices.find(k);
    if ( result ) values[result[0]] = v;
  };

  if ( ["skills", "tool"].includes(trait) ) {
    Object.entries(data).forEach(([k, d]) => setValue(k, d.value));
  } else if ( trait === "saves" ) {
    Object.entries(data).forEach(([k, d]) => setValue(k, d.proficient));
  } else if ( trait === "dm" ) {
    Object.entries(data.amount).forEach(([k, d]) => setValue(k, d));
  } else {
    data.value?.forEach(v => setValue(v, 1));
  }

  if ( trait === "weapon" ) data.mastery?.value?.forEach(v => setValue(v, 2));

  return values;
}

/* -------------------------------------------- */

/**
 * Calculate the change key path for a provided trait key.
 * @param {string} key      Key for a trait to set.
 * @param {string} [trait]  Trait as defined in `CONFIG.DND5E.traits`, only needed if key isn't prefixed.
 * @returns {string|void}
 */
function changeKeyPath(key, trait) {
  const split = key.split(":");
  if ( !trait ) trait = split.shift();

  const traitConfig = CONFIG.DND5E.traits[trait];
  if ( !traitConfig ) return;

  let keyPath = actorKeyPath(trait);

  if ( trait === "saves" ) {
    return `${keyPath}.${split.pop()}.proficient`;
  } else if ( ["skills", "tool"].includes(trait) ) {
    return `${keyPath}.${split.pop()}.value`;
  } else {
    return `${keyPath}.value`;
  }
}

/* -------------------------------------------- */
/*  Trait Lists                                 */
/* -------------------------------------------- */

/**
 * Build up a trait structure containing all of the children gathered from config & base items.
 * @param {string} trait       Trait as defined in `CONFIG.DND5E.traits`.
 * @returns {Promise<object>}  Object with trait categories and children.
 */
async function categories(trait) {
  const traitConfig = CONFIG.DND5E.traits[trait];
  const config = foundry.utils.deepClone(CONFIG.DND5E[traitConfig.configKey ?? trait]);

  for ( const key of Object.keys(config) ) {
    if ( foundry.utils.getType(config[key]) !== "Object" ) config[key] = { label: config[key] };
    if ( traitConfig.children?.[key] ) {
      const children = config[key].children ??= {};
      for ( const [childKey, value] of Object.entries(CONFIG.DND5E[traitConfig.children[key]]) ) {
        if ( foundry.utils.getType(value) !== "Object" ) children[childKey] = { label: value };
        else children[childKey] = { ...value };
      }
    }
  }

  if ( traitConfig.subtypes ) {
    const map = CONFIG.DND5E[`${trait}ProficienciesMap`];

    // Merge all ID lists together
    const ids = traitConfig.subtypes.ids.reduce((obj, key) => {
      foundry.utils.mergeObject(obj, CONFIG.DND5E[key] ?? {});
      return obj;
    }, {});

    // Fetch base items for all IDs
    const baseItems = await Promise.all(Object.entries(ids).map(async ([key, id]) => {
      const index = await getBaseItem(id);
      return [key, index];
    }));

    // Sort base items as children of categories based on subtypes
    for ( const [key, index] of baseItems ) {
      if ( !index ) continue;

      // Get the proper subtype, using proficiency map if needed
      let type = index.system.type.value;
      if ( map?.[type] ) type = map[type];

      // No category for this type, add at top level
      if ( !config[type] ) config[key] = { label: index.name };

      // Add as child of appropriate category
      else {
        config[type].children ??= {};
        config[type].children[key] = { label: index.name };
      }
    }
  }

  return config;
}

/* -------------------------------------------- */

/**
 * Get a list of choices for a specific trait.
 * @param {string} trait                      Trait as defined in `CONFIG.DND5E.traits`.
 * @param {object} [options={}]
 * @param {Set<string>} [options.chosen=[]]   Optional list of keys to be marked as chosen.
 * @param {boolean} [options.prefixed=false]  Should keys be prefixed with trait type?
 * @param {boolean} [options.any=false]       Should the "Any" option be added to each category?
 * @returns {Promise<SelectChoices>}          Object mapping proficiency ids to choice objects.
 */
async function choices(trait, { chosen=new Set(), prefixed=false, any=false }={}) {
  const traitConfig = CONFIG.DND5E.traits[trait];
  if ( !traitConfig ) return new SelectChoices();
  if ( foundry.utils.getType(chosen) === "Array" ) chosen = new Set(chosen);
  const categoryData = await categories(trait);

  let result = {};

  if ( traitConfig.labels?.all && !any ) {
    const key = prefixed ? `${trait}:ALL` : "ALL";
    result[key] = { label: traitConfig.labels.all, chosen: chosen.has(key), sorting: false };
  }

  if ( prefixed && any ) {
    const key = `${trait}:*`;
    result[key] = {
      label: keyLabel(key).titleCase(),
      chosen: chosen.has(key), sorting: false, wildcard: true
    };
  }

  const prepareCategory = (key, data, result, prefix, topLevel=false) => {
    let label = _innerLabel(data, traitConfig);
    if ( !label ) label = key;
    if ( prefixed ) key = `${prefix}:${key}`;
    result[key] = {
      label: game.i18n.localize(label),
      chosen: data.selectable !== false ? chosen.has(key) : false,
      selectable: data.selectable !== false,
      sorting: topLevel ? traitConfig.sortCategories === true : true
    };
    if ( data.children ) {
      const children = result[key].children = {};
      if ( prefixed && any ) {
        const anyKey = `${key}:*`;
        children[anyKey] = {
          label: keyLabel(anyKey).titleCase(),
          chosen: chosen.has(anyKey), sorting: false, wildcard: true
        };
      }
      Object.entries(data.children).forEach(([k, v]) => prepareCategory(k, v, children, key));
    }
  };

  Object.entries(categoryData).forEach(([k, v]) => prepareCategory(k, v, result, trait, true));

  return new SelectChoices(result).sort();
}

/* -------------------------------------------- */

/**
 * Prepare an object with all possible choices from a set of keys. These choices will be grouped by
 * trait type if more than one type is present.
 * @param {Set<string>} keys  Prefixed trait keys.
 * @returns {Promise<SelectChoices>}
 */
async function mixedChoices(keys) {
  if ( !keys.size ) return new SelectChoices();
  const types = {};
  for ( const key of keys ) {
    const split = key.split(":");
    const trait = split.shift();
    const selectChoices = (await choices(trait, { prefixed: true })).filter(new Set([key]));
    types[trait] ??= { label: traitLabel(trait), children: new SelectChoices() };
    types[trait].children.merge(selectChoices);
  }
  if ( Object.keys(types).length > 1 ) return new SelectChoices(types);
  return Object.values(types)[0].children;
}

/* -------------------------------------------- */

/**
 * Fetch an item for the provided ID. If the provided ID contains a compendium pack name
 * it will be fetched from that pack, otherwise it will be fetched from the compendium defined
 * in `DND5E.sourcePacks.ITEMS`.
 * @param {string} identifier            Simple ID or compendium name and ID separated by a dot.
 * @param {object} [options]
 * @param {boolean} [options.indexOnly]  If set to true, only the index data will be fetched (will never return
 *                                       Promise).
 * @param {boolean} [options.fullItem]   If set to true, the full item will be returned as long as `indexOnly` is
 *                                       false.
 * @returns {Promise<Item5e>|object}     Promise for a `Document` if `indexOnly` is false & `fullItem` is true,
 *                                       otherwise else a simple object containing the minimal index data.
 */
function getBaseItem(identifier, { indexOnly=false, fullItem=false }={}) {
  const uuid = getBaseItemUUID(identifier);
  const { collection, documentId: id } = foundry.utils.parseUuid(uuid);
  const pack = collection?.metadata.id;

  // Full Item5e document required, always async.
  if ( fullItem && !indexOnly ) return collection?.getDocument(id);

  const cache = _cachedIndices[pack];
  const loading = cache instanceof Promise;

  // Return extended index if cached, otherwise normal index, guaranteed to never be async.
  if ( indexOnly ) {
    const index = collection?.index.get(id);
    return loading ? index : cache?.[id] ?? index;
  }

  // Returned cached version of extended index if available.
  if ( loading ) return cache.then(() => _cachedIndices[pack][id]);
  else if ( cache ) return cache[id];
  if ( !collection ) return;

  // Build the extended index and return a promise for the data
  const fields = traitIndexFields();
  const promise = collection.getIndex({ fields }).then(index => {
    const store = index.reduce((obj, entry) => {
      obj[entry._id] = entry;
      return obj;
    }, {});
    _cachedIndices[pack] = store;
    return store[id];
  });
  _cachedIndices[pack] = promise;
  return promise;
}

/* -------------------------------------------- */

/**
 * Construct a proper UUID for the provided base item ID.
 * @param {string} identifier  Simple ID, compendium name and ID separated by a dot, or proper UUID.
 * @returns {string}
 */
function getBaseItemUUID(identifier) {
  if ( identifier.startsWith("Compendium.") ) return identifier;
  let pack = CONFIG.DND5E.sourcePacks.ITEMS;
  let [scope, collection, id] = identifier.split(".");
  if ( scope && collection ) pack = `${scope}.${collection}`;
  if ( !id ) id = identifier;
  return `Compendium.${pack}.Item.${id}`;
}

/* -------------------------------------------- */

/**
 * List of fields on items that should be indexed for retrieving subtypes.
 * @returns {string[]}  Index list to pass to `Compendium#getIndex`.
 * @protected
 */
function traitIndexFields() {
  const fields = ["system.type.value"];
  for ( const traitConfig of Object.values(CONFIG.DND5E.traits) ) {
    if ( !traitConfig.subtypes ) continue;
    fields.push(`system.${traitConfig.subtypes.keyPath}`);
  }
  return fields;
}

/* -------------------------------------------- */
/*  Localized Formatting Methods                */
/* -------------------------------------------- */

/**
 * Get the localized label for a specific trait type.
 * @param {string} trait    Trait as defined in `CONFIG.DND5E.traits`.
 * @param {number} [count]  Count used to determine pluralization. If no count is provided, will default to
 *                          the 'other' pluralization.
 * @returns {string}        Localized label.
 */
function traitLabel(trait, count) {
  const traitConfig = CONFIG.DND5E.traits[trait];
  const pluralRule = (count !== undefined) ? new Intl.PluralRules(game.i18n.lang).select(count) : "other";
  if ( !traitConfig ) return game.i18n.localize(`DND5E.TraitGenericPlural.${pluralRule}`);
  return game.i18n.localize(`${traitConfig.labels.localization}.${pluralRule}`);
}

/* -------------------------------------------- */

/**
 * Retrieve the proper display label for the provided key. Will return a promise unless a categories
 * object is provided in config.
 * @param {string} key              Key for which to generate the label.
 * @param {object} [config={}]
 * @param {number} [config.count]   Number to display, only if a wildcard is used as final part of key.
 * @param {string} [config.trait]   Trait as defined in `CONFIG.DND5E.traits` if not using a prefixed key.
 * @param {boolean} [config.final]  Is this the final in a list?
 * @returns {string}                Retrieved label.
 *
 * @example
 * // Returns "Tool Proficiency"
 * keyLabel("tool");
 *
 * @example
 * // Returns "Artisan's Tools"
 * keyLabel("tool:art");
 *
 * @example
 * // Returns "any Artisan's Tools"
 * keyLabel("tool:art:*");
 *
 * @example
 * // Returns "any 2 Artisan's Tools"
 * keyLabel("tool:art:*", { count: 2 });
 *
 * @example
 * // Returns "2 other Artisan's Tools"
 * keyLabel("tool:art:*", { count: 2, final: true });
 *
 * @example
 * // Returns "Gaming Sets"
 * keyLabel("tool:game");
 *
 * @example
 * // Returns "Land Vehicle"
 * keyLabel("tool:vehicle:land");
 *
 * @example
 * // Returns "Shortsword"
 * keyLabel("weapon:shortsword");
 * keyLabel("weapon:simple:shortsword");
 * keyLabel("shortsword", { trait: "weapon" });
 */
function keyLabel(key, config={}) {
  let { count, trait, final } = config;

  let parts = key.split(":");
  const pluralRules = new Intl.PluralRules(game.i18n.lang);

  if ( !trait ) trait = parts.shift();
  const traitConfig = CONFIG.DND5E.traits[trait];
  if ( !traitConfig ) return key;
  const traitData = CONFIG.DND5E[traitConfig.configKey ?? trait] ?? {};
  let categoryLabel = game.i18n.localize(`${traitConfig.labels.localization}.${
    pluralRules.select(count ?? 1)}`);

  // Trait (e.g. "Tool Proficiency")
  const lastKey = parts.pop();
  if ( !lastKey ) return categoryLabel;

  // All (e.g. "All Languages")
  if ( lastKey === "ALL" ) return traitConfig.labels?.all ?? key;

  // Wildcards (e.g. "Artisan's Tools", "any Artisan's Tools", "any 2 Artisan's Tools", or "2 other Artisan's Tools")
  else if ( lastKey === "*" ) {
    let type;
    if ( parts.length ) {
      let category = traitData;
      do {
        category = (category.children ?? category)[parts.shift()];
        if ( !category ) return key;
      } while ( parts.length );
      type = _innerLabel(category, traitConfig);
    } else type = categoryLabel.toLowerCase();
    const localization = `DND5E.TraitConfigChoose${final ? "Other" : `Any${count ? "Counted" : "Uncounted"}`}`;
    return game.i18n.format(localization, { count: count ?? 1, type });
  }

  else {
    // Category (e.g. "Gaming Sets")
    const category = traitData[lastKey];
    if ( category ) return _innerLabel(category, traitConfig);

    // Child (e.g. "Land Vehicle")
    for ( const childrenKey of Object.values(traitConfig.children ?? {}) ) {
      const childLabel = CONFIG.DND5E[childrenKey]?.[lastKey];
      if ( childLabel ) return childLabel;
    }

    // Base item (e.g. "Shortsword")
    for ( const idsKey of traitConfig.subtypes?.ids ?? [] ) {
      const baseItemId = CONFIG.DND5E[idsKey]?.[lastKey];
      if ( !baseItemId ) continue;
      const index = getBaseItem(baseItemId, { indexOnly: true });
      if ( index ) return index.name;
      break;
    }

    // Explicit categories (e.g. languages)
    const searchCategory = (data, key) => {
      for ( const [k, v] of Object.entries(data) ) {
        if ( k === key ) return v;
        if ( v.children ) {
          const result = searchCategory(v.children, key);
          if ( result ) return result;
        }
      }
    };
    const config = searchCategory(traitData, lastKey);
    return config ? _innerLabel(config, traitConfig) : key;
  }
}

/* -------------------------------------------- */

/**
 * Create a human readable description of the provided choice.
 * @param {TraitChoice} choice             Data for a specific choice.
 * @param {object} [options={}]
 * @param {boolean} [options.only=false]   Is this choice on its own, or part of a larger list?
 * @param {boolean} [options.final=false]  If this choice is part of a list of other grants or choices,
 *                                         is it in the final position?
 * @returns {string}
 *
 * @example
 * // Returns "any three skill proficiencies"
 * choiceLabel({ count: 3, pool: new Set(["skills:*"]) });
 *
 * @example
 * // Returns "three other skill proficiencies"
 * choiceLabel({ count: 3, pool: new Set(["skills:*"]) }, { final: true });
 *
 * @example
 * // Returns "any skill proficiency"
 * choiceLabel({ count: 1, pool: new Set(["skills:*"]) }, { only: true });
 *
 * @example
 * // Returns "Thieves Tools or any skill"
 * choiceLabel({ count: 1, pool: new Set(["tool:thief", "skills:*"]) }, { only: true });
 *
 * @example
 * // Returns "Thieves' Tools or any artisan tool"
 * choiceLabel({ count: 1, pool: new Set(["tool:thief", "tool:art:*"]) }, { only: true });
 *
 * @example
 * // Returns "2 from Thieves' Tools or any skill proficiency"
 * choiceLabel({ count: 2, pool: new Set(["tool:thief", "skills:*"]) });
 *
 */
function choiceLabel(choice, { only=false, final=false }={}) {
  if ( !choice.pool.size ) return "";

  // Single entry in pool (e.g. "any three skill proficiencies" or "three other skill proficiencies")
  if ( choice.pool.size === 1 ) {
    return keyLabel(choice.pool.first(), {
      count: (choice.count > 1 || !only) ? choice.count : null, final: final && !only
    });
  }

  const listFormatter = new Intl.ListFormat(game.i18n.lang, { type: "disjunction" });

  // Singular count (e.g. "any skill", "Thieves Tools or any skill", or "Thieves' Tools or any artisan tool")
  if ( (choice.count === 1) && only ) {
    return listFormatter.format(choice.pool.map(key => keyLabel(key)));
  }

  // Select from a list of options (e.g. "2 from Thieves' Tools or any skill proficiency")
  const choices = choice.pool.map(key => keyLabel(key));
  return game.i18n.format("DND5E.TraitConfigChooseList", {
    count: choice.count,
    list: listFormatter.format(choices)
  });
}

/* -------------------------------------------- */

/**
 * Create a human readable description of trait grants & choices.
 * @param {object} config
 * @param {Set<string>} [config.grants]        Guaranteed trait grants.
 * @param {TraitChoice[]} [config.choices=[]]  Trait choices.
 * @returns {string}
 *
 * @example
 * // Returns "Acrobatics and Athletics"
 * localizedList({ grants: new Set(["skills:acr", "skills:ath"]) });
 *
 * @example
 * // Returns "Acrobatics and one other skill proficiency"
 * localizedList({ grants: new Set(["skills:acr"]), choices: [{ count: 1, pool: new Set(["skills:*"])}] });
 *
 * @example
 * // Returns "Choose any skill proficiency"
 * localizedList({ choices: [{ count: 1, pool: new Set(["skills:*"])}] });
 */
function localizedList({ grants=new Set(), choices=[] }) {
  const sections = Array.from(grants).map(g => keyLabel(g));

  for ( const [index, choice] of choices.entries() ) {
    const final = index === choices.length - 1;
    sections.push(choiceLabel(choice, { final, only: !grants.size && choices.length === 1 }));
  }

  const listFormatter = new Intl.ListFormat(game.i18n.lang, { style: "long", type: "conjunction" });
  if ( !sections.length || grants.size ) return listFormatter.format(sections);
  return game.i18n.format("DND5E.TraitConfigChooseWrapper", {
    choices: listFormatter.format(sections)
  });
}

var trait = /*#__PURE__*/Object.freeze({
  __proto__: null,
  actorFields: actorFields,
  actorKeyPath: actorKeyPath,
  actorValues: actorValues,
  categories: categories,
  changeKeyPath: changeKeyPath,
  choiceLabel: choiceLabel,
  choices: choices,
  getBaseItem: getBaseItem,
  getBaseItemUUID: getBaseItemUUID,
  keyLabel: keyLabel,
  localizedList: localizedList,
  mixedChoices: mixedChoices,
  traitIndexFields: traitIndexFields,
  traitLabel: traitLabel
});

/**
 * Sheet for the check activity.
 */
class CheckSheet extends ActivitySheet {

  /** @inheritDoc */
  static DEFAULT_OPTIONS = {
    classes: ["check-activity"]
  };

  /* -------------------------------------------- */

  /** @inheritDoc */
  static PARTS = {
    ...super.PARTS,
    effect: {
      template: "systems/dnd5e/templates/activity/check-effect.hbs",
      templates: [
        ...super.PARTS.effect.templates,
        "systems/dnd5e/templates/activity/parts/check-details.hbs"
      ]
    }
  };

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareEffectContext(context) {
    context = await super._prepareEffectContext(context);

    const group = game.i18n.localize("DND5E.Abilities");
    context.abilityOptions = [
      { value: "", label: "" },
      { rule: true },
      { value: "spellcasting", label: game.i18n.localize("DND5E.SpellAbility") },
      ...Object.entries(CONFIG.DND5E.abilities).map(([value, config]) => ({ value, label: config.label, group }))
    ];
    let ability;
    const associated = this.activity.check.associated;
    if ( (this.item.type === "tool") && !associated.size ) {
      ability = CONFIG.DND5E.abilities[this.item.system.ability]?.label?.toLowerCase();
    } else if ( (associated.size === 1) && (associated.first() in CONFIG.DND5E.skills) ) {
      ability = CONFIG.DND5E.abilities[CONFIG.DND5E.skills[associated.first()].ability]?.label?.toLowerCase();
    }
    if ( ability ) context.abilityOptions[0].label = game.i18n.format("DND5E.DefaultSpecific", { default: ability });

    context.associatedOptions = [
      ...Object.entries(CONFIG.DND5E.skills).map(([value, { label }]) => ({
        value, label, group: game.i18n.localize("DND5E.Skills")
      })),
      ...Object.keys(CONFIG.DND5E.toolIds).map(value => ({
        value, label: keyLabel(value, { trait: "tool" }), group: game.i18n.localize("TYPES.Item.toolPl")
      })).sort((lhs, rhs) => lhs.label.localeCompare(rhs.label, game.i18n.lang))
    ];

    context.calculationOptions = [
      { value: "", label: game.i18n.localize("DND5E.SAVE.FIELDS.save.dc.CustomFormula") },
      { rule: true },
      { value: "spellcasting", label: game.i18n.localize("DND5E.SpellAbility") },
      ...Object.entries(CONFIG.DND5E.abilities).map(([value, config]) => ({ value, label: config.label, group }))
    ];

    return context;
  }
}

const { SchemaField: SchemaField$L, SetField: SetField$u, StringField: StringField$11 } = foundry.data.fields;

/**
 * Data model for a check activity.
 *
 * @property {object} check
 * @property {string} check.ability          Ability used with the check.
 * @property {Set<string>} check.associated  Skills or tools that can contribute to the check.
 * @property {object} check.dc
 * @property {string} check.dc.calculation   Method or ability used to calculate the difficulty class of the check.
 * @property {string} check.dc.formula       Custom DC formula or flat value.
 */
class CheckActivityData extends BaseActivityData {
  /** @inheritDoc */
  static defineSchema() {
    return {
      ...super.defineSchema(),
      check: new SchemaField$L({
        ability: new StringField$11(),
        associated: new SetField$u(new StringField$11()),
        dc: new SchemaField$L({
          calculation: new StringField$11(),
          formula: new FormulaField({ deterministic: true })
        })
      })
    };
  }

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /** @override */
  get ability() {
    if ( this.check.dc.calculation in CONFIG.DND5E.abilities ) return this.check.dc.calculation;
    if ( this.check.dc.calculation === "spellcasting" ) return this.spellcastingAbility;
    return this.check.ability;
  }

  /* -------------------------------------------- */
  /*  Data Migrations                             */
  /* -------------------------------------------- */

  /** @override */
  static transformTypeData(source, activityData, options) {
    return foundry.utils.mergeObject(activityData, {
      check: {
        ability: source.system.ability ?? Object.keys(CONFIG.DND5E.abilities)[0]
      }
    });
  }

  /* -------------------------------------------- */
  /*  Data Preparation                            */
  /* -------------------------------------------- */

  /** @inheritDoc */
  prepareFinalData(rollData) {
    rollData ??= this.getRollData({ deterministic: true });
    super.prepareFinalData(rollData);

    if ( this.check.ability === "spellcasting" ) this.check.ability = this.spellcastingAbility;

    let ability;
    if ( this.check.dc.calculation ) ability = this.ability;
    else this.check.dc.value = simplifyBonus(this.check.dc.formula, rollData);
    if ( ability ) this.check.dc.value = this.actor?.system.abilities?.[ability]?.dc
      ?? 8 + (this.actor?.system.attributes?.prof ?? 0);

    if ( !this.check.dc.value ) this.check.dc.value = null;
  }

  /* -------------------------------------------- */
  /*  Helpers                                     */
  /* -------------------------------------------- */

  /**
   * Get the ability to use with an associated value.
   * @param {string} associated  Skill or tool ID.
   * @returns {string|null}      Ability to use.
   */
  getAbility(associated) {
    if ( this.check.ability ) return this.check.ability;
    if ( associated in CONFIG.DND5E.skills ) return CONFIG.DND5E.skills[associated]?.ability ?? null;
    else if ( associated in CONFIG.DND5E.tools ) {
      if ( (this.item.type === "tool") && this.item.system.ability ) return this.item.system.ability;
      return CONFIG.DND5E.tools[associated]?.ability ?? null;
    }
    return null;
  }
}

/**
 * Activity for making ability checks.
 */
class CheckActivity extends ActivityMixin(CheckActivityData) {
  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static LOCALIZATION_PREFIXES = [...super.LOCALIZATION_PREFIXES, "DND5E.CHECK"];

  /* -------------------------------------------- */

  /** @inheritDoc */
  static metadata = Object.freeze(
    foundry.utils.mergeObject(super.metadata, {
      type: "check",
      img: "systems/dnd5e/icons/svg/activity/check.svg",
      title: "DND5E.CHECK.Title",
      sheetClass: CheckSheet,
      usage: {
        actions: {
          rollCheck: CheckActivity.#rollCheck
        }
      }
    }, { inplace: false })
  );

  /* -------------------------------------------- */
  /*  Activation                                  */
  /* -------------------------------------------- */

  /** @override */
  _usageChatButtons(message) {
    const buttons = [];
    const dc = this.check.dc.value;

    const createButton = (abilityKey, associated) => {
      const ability = CONFIG.DND5E.abilities[abilityKey]?.label;
      const checkType = (associated in CONFIG.DND5E.skills) ? "skill"
        : (associated in CONFIG.DND5E.toolIds) ? "tool": "ability";
      const dataset = { ability: abilityKey, action: "rollCheck", visibility: "all" };
      if ( dc ) dataset.dc = dc;
      if ( checkType !== "ability" ) dataset[checkType] = associated;

      let label = ability;
      let type;
      if ( checkType === "skill" ) type = CONFIG.DND5E.skills[associated]?.label;
      else if ( checkType === "tool" ) type = keyLabel(associated, { trait: "tool" });
      if ( type ) label = game.i18n.format("EDITOR.DND5E.Inline.SpecificCheck", { ability, type });
      else label = ability;

      buttons.push({
        label: dc ? `
          <span class="visible-dc">${game.i18n.format("EDITOR.DND5E.Inline.DC", { dc, check: wrap(label) })}</span>
          <span class="hidden-dc">${wrap(label)}</span>
        ` : wrap(label),
        icon: checkType === "tool" ? '<i class="fa-solid fa-hammer" inert></i>'
          : '<i class="dnd5e-icon" data-src="systems/dnd5e/icons/svg/ability-score-improvement.svg" inert></i>',
        dataset
      });
    };
    const wrap = check => game.i18n.format("EDITOR.DND5E.Inline.CheckShort", { check });

    const associated = Array.from(this.check.associated);
    if ( !associated.length && (this.item.type === "tool") ) associated.push(this.item.system.type.baseItem);
    if ( associated.length ) associated.forEach(a => {
      const ability = this.getAbility(a);
      if ( ability ) createButton(ability, a);
    });
    else if ( this.check.ability ) createButton(this.check.ability);

    return buttons.concat(super._usageChatButtons(message));
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /**
   * Handle performing an ability check.
   * @this {CheckActivity}
   * @param {PointerEvent} event     Triggering click event.
   * @param {HTMLElement} target     The capturing HTML element which defined a [data-action].
   * @param {ChatMessage5e} message  Message associated with the activation.
   */
  static async #rollCheck(event, target, message) {
    const targets = getSceneTargets();
    if ( !targets.length && game.user.character ) targets.push(game.user.character);
    if ( !targets.length ) ui.notifications.warn("DND5E.ActionWarningNoToken", { localize: true });
    let { ability, dc, skill, tool } = target.dataset;
    dc = parseInt(dc);
    const rollData = { event, target: Number.isFinite(dc) ? dc : this.check.dc.value };
    if ( ability in CONFIG.DND5E.abilities ) rollData.ability = ability;

    for ( const token of targets ) {
      const actor = token instanceof Actor ? token : token.actor;
      const speaker = ChatMessage.getSpeaker({ actor, scene: canvas.scene, token: token.document });
      const messageData = { data: { speaker } };
      if ( skill ) await actor.rollSkill({ ...rollData, skill }, {}, messageData);
      else if ( tool ) {
        rollData.tool = tool;
        if ( (this.item.type === "tool") && !this.check.associated.size ) {
          rollData.bonus = this.item.system.bonus;
          rollData.prof = this.item.system.prof;
          rollData.item = this.item;
        }
        await actor.rollToolCheck(rollData, {}, messageData);
      }
      else await actor.rollAbilityCheck(rollData, {}, messageData);
    }
  }
}

/**
 * Sheet for the damage activity.
 */
class DamageSheet extends ActivitySheet {

  /** @inheritDoc */
  static DEFAULT_OPTIONS = {
    classes: ["damage-activity"]
  };

  /* -------------------------------------------- */

  /** @inheritDoc */
  static PARTS = {
    ...super.PARTS,
    effect: {
      template: "systems/dnd5e/templates/activity/damage-effect.hbs",
      templates: [
        ...super.PARTS.effect.templates,
        "systems/dnd5e/templates/activity/parts/damage-damage.hbs",
        "systems/dnd5e/templates/activity/parts/damage-part.hbs",
        "systems/dnd5e/templates/activity/parts/damage-parts.hbs"
      ]
    }
  };
}

const { ArrayField: ArrayField$l, BooleanField: BooleanField$C, SchemaField: SchemaField$K } = foundry.data.fields;

/**
 * Data model for an damage activity.
 *
 * @property {object} damage
 * @property {boolean} damage.critical.allow  Can this damage be critical?
 * @property {string} damage.critical.bonus   Extra damage applied to the first damage part when a critical is rolled.
 * @property {DamageData[]} damage.parts      Parts of damage to inflict.
 */
class DamageActivityData extends BaseActivityData {
  /** @inheritDoc */
  static defineSchema() {
    return {
      ...super.defineSchema(),
      damage: new SchemaField$K({
        critical: new SchemaField$K({
          allow: new BooleanField$C(),
          bonus: new FormulaField()
        }),
        parts: new ArrayField$l(new DamageField())
      })
    };
  }

  /* -------------------------------------------- */
  /*  Data Migrations                             */
  /* -------------------------------------------- */

  /** @override */
  static transformTypeData(source, activityData, options) {
    return foundry.utils.mergeObject(activityData, {
      damage: {
        critical: {
          allow: false,
          bonus: source.system.critical?.damage ?? ""
        },
        parts: options.versatile
          ? [this.transformDamagePartData(source, [source.system.damage?.versatile, ""])]
          : (source.system.damage?.parts?.map(part => this.transformDamagePartData(source, part)) ?? [])
      }
    });
  }

  /* -------------------------------------------- */
  /*  Data Preparation                            */
  /* -------------------------------------------- */

  /** @inheritDoc */
  prepareFinalData(rollData) {
    rollData ??= this.getRollData({ deterministic: true });
    super.prepareFinalData(rollData);
    this.prepareDamageLabel(this.damage.parts, rollData);
  }

  /* -------------------------------------------- */
  /*  Helpers                                     */
  /* -------------------------------------------- */

  /** @inheritDoc */
  getDamageConfig(config={}) {
    const rollConfig = super.getDamageConfig(config);

    rollConfig.critical ??= {};
    rollConfig.critical.allow ??= this.damage.critical.allow;
    rollConfig.critical.bonusDamage ??= this.damage.critical.bonus;

    return rollConfig;
  }
}

/**
 * Activity for rolling damage.
 */
class DamageActivity extends ActivityMixin(DamageActivityData) {
  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static LOCALIZATION_PREFIXES = [...super.LOCALIZATION_PREFIXES, "DND5E.DAMAGE"];

  /* -------------------------------------------- */

  /** @inheritDoc */
  static metadata = Object.freeze(
    foundry.utils.mergeObject(super.metadata, {
      type: "damage",
      img: "systems/dnd5e/icons/svg/activity/damage.svg",
      title: "DND5E.DAMAGE.Title",
      sheetClass: DamageSheet,
      usage: {
        actions: {
          rollDamage: DamageActivity.#rollDamage
        }
      }
    }, { inplace: false })
  );

  /* -------------------------------------------- */
  /*  Activation                                  */
  /* -------------------------------------------- */

  /** @override */
  _usageChatButtons(message) {
    if ( !this.damage.parts.length ) return super._usageChatButtons(message);
    return [{
      label: game.i18n.localize("DND5E.Damage"),
      icon: '<i class="fa-solid fa-burst" inert></i>',
      dataset: {
        action: "rollDamage"
      }
    }].concat(super._usageChatButtons(message));
  }

  /* -------------------------------------------- */

  /** @override */
  async _triggerSubsequentActions(config, results) {
    this.rollDamage({ event: config.event }, {}, { data: { "flags.dnd5e.originatingMessage": results.message?.id } });
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /**
   * Handle performing a damage roll.
   * @this {DamageActivity}
   * @param {PointerEvent} event     Triggering click event.
   * @param {HTMLElement} target     The capturing HTML element which defined a [data-action].
   * @param {ChatMessage5e} message  Message associated with the activation.
   */
  static #rollDamage(event, target, message) {
    this.rollDamage({ event });
  }
}

/**
 * Sheet for the enchant activity.
 */
class EnchantSheet extends ActivitySheet {

  /** @inheritDoc */
  static DEFAULT_OPTIONS = {
    classes: ["enchant-activity"]
  };

  /* -------------------------------------------- */

  /** @inheritDoc */
  static PARTS = {
    ...super.PARTS,
    effect: {
      template: "systems/dnd5e/templates/activity/enchant-effect.hbs",
      templates: [
        "systems/dnd5e/templates/activity/parts/enchant-enchantments.hbs",
        "systems/dnd5e/templates/activity/parts/enchant-restrictions.hbs"
      ]
    }
  };

  /* -------------------------------------------- */

  /** @override */
  tabGroups = {
    sheet: "identity",
    activation: "time",
    effect: "enchantments"
  };

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @override */
  _prepareAppliedEffectContext(context, effect) {
    effect.activityOptions = this.item.system.activities
      .filter(a => a.id !== this.activity.id)
      .map(a => ({ value: a.id, label: a.name, selected: effect.data.riders.activity.has(a.id) }));
    effect.effectOptions = context.allEffects.map(e => ({
      ...e, selected: effect.data.riders.effect.has(e.value)
    }));
    return effect;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareEffectContext(context) {
    context = await super._prepareEffectContext(context);

    const appliedEnchantments = new Set(context.activity.effects?.map(e => e._id) ?? []);
    context.allEnchantments = this.item.effects
      .filter(e => e.type === "enchantment")
      .map(effect => ({
        value: effect.id, label: effect.name, selected: appliedEnchantments.has(effect.id)
      }));

    const enchantableTypes = this.activity.enchantableTypes;
    context.typeOptions = [
      { value: "", label: game.i18n.localize("DND5E.ENCHANT.FIELDS.restrictions.type.Any") },
      ...Object.keys(CONFIG.Item.dataModels)
        .filter(t => enchantableTypes.has(t))
        .map(value => ({ value, label: game.i18n.localize(CONFIG.Item.typeLabels[value]) }))
    ];

    const type = context.source.restrictions.type;
    const typeDataModel = CONFIG.Item.dataModels[type];
    if ( typeDataModel ) context.categoryOptions = Object.entries(typeDataModel.itemCategories ?? {})
      .map(([value, config]) => ({ value, label: foundry.utils.getType(config) === "string" ? config : config.label }));

    context.propertyOptions = (CONFIG.DND5E.validProperties[type] ?? [])
      .map(value => ({ value, label: CONFIG.DND5E.itemProperties[value]?.label ?? value }));

    return context;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _getTabs() {
    const tabs = super._getTabs();
    tabs.effect.label = "DND5E.ENCHANT.SECTIONS.Enchanting";
    tabs.effect.icon = "fa-solid fa-wand-sparkles";
    tabs.effect.tabs = this._markTabs({
      enchantments: {
        id: "enchantments", group: "effect", icon: "fa-solid fa-star",
        label: "DND5E.ENCHANT.SECTIONS.Enchantments"
      },
      restrictions: {
        id: "restrictions", group: "effect", icon: "fa-solid fa-ban",
        label: "DND5E.ENCHANT.SECTIONS.Restrictions"
      }
    });
    return tabs;
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /** @override */
  _addEffectData() {
    return {
      type: "enchantment",
      name: this.item.name,
      img: this.item.img,
      disabled: true
    };
  }
}

const { StringField: StringField$10 } = foundry.data.fields;

/**
 * Dialog for configuring the usage of an activity.
 */
class EnchantUsageDialog extends ActivityUsageDialog {

  /** @inheritDoc */
  static PARTS = {
    ...super.PARTS,
    creation: {
      template: "systems/dnd5e/templates/activity/enchant-usage-creation.hbs"
    }
  };

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareCreationContext(context, options) {
    context = await super._prepareCreationContext(context, options);

    const enchantments = this.activity.availableEnchantments;
    if ( (enchantments.length > 1) && this._shouldDisplay("create.enchantment") ) {
      context.hasCreation = true;
      context.enchantment = {
        field: new StringField$10({ label: game.i18n.localize("DND5E.ENCHANTMENT.Label") }),
        name: "enchantmentProfile",
        value: this.config.enchantmentProfile,
        options: enchantments.map(e => ({ value: e._id, label: e.effect.name }))
      };
    } else if ( enchantments.length ) {
      context.enchantment = enchantments[0]?._id ?? false;
    }

    return context;
  }
}

/**
 * Special case StringField that includes automatic validation for identifiers.
 */
class IdentifierField extends foundry.data.fields.StringField {
  /** @override */
  _validateType(value) {
    if ( !dnd5e.utils.validators.isValidIdentifier(value) ) {
      throw new Error(game.i18n.localize("DND5E.IdentifierError"));
    }
  }
}

const {
  ArrayField: ArrayField$k, BooleanField: BooleanField$B, DocumentIdField: DocumentIdField$8, DocumentUUIDField: DocumentUUIDField$3, NumberField: NumberField$E, SchemaField: SchemaField$J, SetField: SetField$t, StringField: StringField$$
} = foundry.data.fields;

/**
 * @typedef {EffectApplicationData} EnchantEffectApplicationData
 * @property {object} level
 * @property {number} level.min             Minimum level at which this profile can be used.
 * @property {number} level.max             Maximum level at which this profile can be used.
 * @property {object} riders
 * @property {Set<string>} riders.activity  IDs of other activities on this item that will be added when enchanting.
 * @property {Set<string>} riders.effect    IDs of other effects on this item that will be added when enchanting.
 * @property {Set<string>} riders.item      UUIDs of items that will be added with this enchantment.
 */

/**
 * Data model for a enchant activity.
 *
 * @property {object} enchant
 * @property {string} enchant.identifier    Class identifier that will be used to determine applicable level.
 * @property {object} restrictions
 * @property {boolean} restrictions.allowMagical    Allow enchantments to be applied to items that are already magical.
 * @property {Set<string>} restrictions.categories  Item categories to restrict to.
 * @property {Set<string>} restrictions.properties  Item properties to restrict to.
 * @property {string} restrictions.type             Item type to which this enchantment can be applied.
 */
class EnchantActivityData extends BaseActivityData {
  /** @inheritDoc */
  static defineSchema() {
    return {
      ...super.defineSchema(),
      effects: new ArrayField$k(new AppliedEffectField({
        level: new SchemaField$J({
          min: new NumberField$E({ min: 0, integer: true }),
          max: new NumberField$E({ min: 0, integer: true })
        }),
        riders: new SchemaField$J({
          activity: new SetField$t(new DocumentIdField$8()),
          effect: new SetField$t(new DocumentIdField$8()),
          item: new SetField$t(new DocumentUUIDField$3())
        })
      })),
      enchant: new SchemaField$J({
        identifier: new IdentifierField()
      }),
      restrictions: new SchemaField$J({
        allowMagical: new BooleanField$B(),
        categories: new SetField$t(new StringField$$()),
        properties: new SetField$t(new StringField$$()),
        type: new StringField$$()
      })
    };
  }

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /** @override */
  get actionType() {
    return "ench";
  }

  /* -------------------------------------------- */

  /** @override */
  get applicableEffects() {
    return null;
  }

  /* -------------------------------------------- */

  /**
   * Enchantments that have been applied by this activity.
   * @type {ActiveEffect5e[]}
   */
  get appliedEnchantments() {
    return dnd5e.registry.enchantments.applied(this.uuid);
  }

  /* -------------------------------------------- */

  /**
   * Enchantments that can be applied based on spell/character/class level.
   * @type {EnchantEffectApplicationData[]}
   */
  get availableEnchantments() {
    const keyPath = (this.item.type === "spell") && (this.item.system.level > 0) ? "item.level"
      : this.classIdentifier ? `classes.${this.classIdentifier}.levels` : "details.level";
    const level = foundry.utils.getProperty(this.getRollData(), keyPath) ?? 0;
    return this.effects
      .filter(e => e.effect && ((e.level.min ?? -Infinity) <= level) && (level <= (e.level.max ?? Infinity)));
  }

  /* -------------------------------------------- */

  /**
   * List of item types that are enchantable.
   * @type {Set<string>}
   */
  static get enchantableTypes() {
    return Object.entries(CONFIG.Item.dataModels).reduce((set, [k, v]) => {
      if ( v.metadata?.enchantable ) set.add(k);
      return set;
    }, new Set());
  }

  /* -------------------------------------------- */
  /*  Data Migrations                             */
  /* -------------------------------------------- */

  /** @override */
  static transformEffectsData(source, options) {
    const effects = [];
    for ( const effect of source.effects ) {
      if ( (effect.type !== "enchantment") && (effect.flags?.dnd5e?.type !== "enchantment") ) continue;
      effects.push({ _id: effect._id, ...(effect.flags?.dnd5e?.enchantment ?? {}) });
      delete effect.flags?.dnd5e?.enchantment;
    }
    return effects;
  }

  /* -------------------------------------------- */

  /** @override */
  static transformTypeData(source, activityData) {
    return foundry.utils.mergeObject(activityData, {
      enchant: {
        identifier: source.system.enchantment?.classIdentifier ?? ""
      },
      restrictions: source.system.enchantment?.restrictions ?? []
    });
  }
}

/**
 * Activity for enchanting items.
 */
class EnchantActivity extends ActivityMixin(EnchantActivityData) {
  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static LOCALIZATION_PREFIXES = [...super.LOCALIZATION_PREFIXES, "DND5E.ENCHANT"];

  /* -------------------------------------------- */

  /** @inheritDoc */
  static metadata = Object.freeze(
    foundry.utils.mergeObject(super.metadata, {
      type: "enchant",
      img: "systems/dnd5e/icons/svg/activity/enchant.svg",
      title: "DND5E.ENCHANT.Title",
      sheetClass: EnchantSheet,
      usage: {
        dialog: EnchantUsageDialog
      }
    }, { inplace: false })
  );

  /* -------------------------------------------- */

  /** @inheritDoc */
  static localize() {
    super.localize();
    this._localizeSchema(this.schema.fields.effects.element, ["DND5E.ENCHANT.FIELDS.effects"]);
  }

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * List of item types that are enchantable.
   * @type {Set<string>}
   */
  get enchantableTypes() {
    return Object.entries(CONFIG.Item.dataModels).reduce((set, [k, v]) => {
      if ( v.metadata?.enchantable ) set.add(k);
      return set;
    }, new Set());
  }

  /* -------------------------------------------- */
  /*  Activation                                  */
  /* -------------------------------------------- */

  /**
   * @typedef {ActivityUseConfiguration} EnchantUseConfiguration
   * @property {string} enchantmentProfile
   */

  /** @inheritDoc */
  _createDeprecatedConfigs(usageConfig, dialogConfig, messageConfig) {
    const config = super._createDeprecatedConfigs(usageConfig, dialogConfig, messageConfig);
    config.enchantmentProfile = usageConfig.enchantmentProfile ?? null;
    return config;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _applyDeprecatedConfigs(usageConfig, dialogConfig, messageConfig, config, options) {
    super._applyDeprecatedConfigs(usageConfig, dialogConfig, messageConfig, config, options);
    if ( config.enchantmentProfile ) usageConfig.enchantmentProfile = config.enchantmentProfile;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _prepareUsageConfig(config) {
    config = super._prepareUsageConfig(config);
    config.enchantmentProfile ??= this.availableEnchantments[0]?._id;
    return config;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareUsageScaling(usageConfig, messageConfig, item) {
    await super._prepareUsageScaling(usageConfig, messageConfig, item);

    // Store selected enchantment profile in message flag
    if ( usageConfig.enchantmentProfile ) foundry.utils.setProperty(
      messageConfig, "data.flags.dnd5e.use.enchantmentProfile", usageConfig.enchantmentProfile
    );
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _requiresConfigurationDialog(config) {
    return super._requiresConfigurationDialog(config) || (this.availableEnchantments.length > 1);
  }

  /* -------------------------------------------- */
  /*  Helpers                                     */
  /* -------------------------------------------- */

  /**
   * Determine whether the provided item can be enchanted based on this enchantment's restrictions.
   * @param {Item5e} item  Item that might be enchanted.
   * @returns {true|EnchantmentError[]}
   */
  canEnchant(item) {
    const errors = [];

    if ( !this.restrictions.allowMagical && item.system.properties?.has("mgc") ) {
      errors.push(new EnchantmentError$1(game.i18n.localize("DND5E.ENCHANT.Warning.NoMagicalItems")));
    }

    if ( this.restrictions.type && (item.type !== this.restrictions.type) ) {
      errors.push(new EnchantmentError$1(game.i18n.format("DND5E.ENCHANT.Warning.WrongType", {
        incorrectType: game.i18n.localize(CONFIG.Item.typeLabels[item.type]),
        allowedType: game.i18n.localize(CONFIG.Item.typeLabels[this.restrictions.type])
      })));
    }

    if ( this.restrictions.categories.size && !this.restrictions.categories.has(item.system.type?.value) ) {
      const getLabel = key => {
        const config = CONFIG.Item.dataModels[this.restrictions.type]?.itemCategories[key];
        if ( !config ) return key;
        if ( foundry.utils.getType(config) === "string" ) return config;
        return config.label;
      };
      errors.push(new EnchantmentError$1(game.i18n.format(
        `DND5E.ENCHANT.Warning.${item.system.type?.value ? "WrongType" : "NoSubtype"}`,
        {
          allowedType: game.i18n.getListFormatter({ type: "disjunction" }).format(
            Array.from(this.restrictions.categories).map(c => getLabel(c).toLowerCase())
          ),
          incorrectType: getLabel(item.system.type?.value)
        }
      )));
    }

    if ( this.restrictions.properties.size
      && !this.restrictions.properties.intersection(item.system.properties ?? new Set()).size ) {
      errors.push(new EnchantmentError$1(game.i18n.format("DND5E.Enchantment.Warning.MissingProperty", {
        validProperties: game.i18n.getListFormatter({ type: "disjunction" }).format(
          Array.from(this.restrictions.properties).map(p => CONFIG.DND5E.itemProperties[p]?.label ?? p)
        )
      })));
    }

    /**
     * A hook event that fires while validating whether an enchantment can be applied to a specific item.
     * @function dnd5e.canEnchant
     * @memberof hookEvents
     * @param {EnchantActivity} activity   The activity performing the enchanting.
     * @param {Item5e} item                Item to which the enchantment will be applied.
     * @param {EnchantmentError[]} errors  List of errors containing failed restrictions. The item will be enchanted
     *                                     so long as no errors are listed, otherwise the provided errors will be
     *                                     displayed to the user.
     */
    Hooks.callAll("dnd5e.canEnchant", this, item, errors);

    return errors.length ? errors : true;
  }
}

/**
 * Error to throw when an item cannot be enchanted.
 */
let EnchantmentError$1 = class EnchantmentError extends Error {
  constructor(...args) {
    super(...args);
    this.name = "EnchantmentError";
  }
};

/**
 * Sheet for the forward activity.
 */
class ForwardSheet extends ActivitySheet {

  /** @inheritDoc */
  static DEFAULT_OPTIONS = {
    classes: ["forward-activity"]
  };

  /* -------------------------------------------- */

  /** @inheritDoc */
  static PARTS = {
    ...super.PARTS,
    identity: {
      template: "systems/dnd5e/templates/activity/forward-identity.hbs",
      templates: super.PARTS.identity.templates
    },
    activation: {
      template: "systems/dnd5e/templates/activity/forward-activation.hbs",
      templates: [
        "systems/dnd5e/templates/activity/parts/activity-consumption.hbs"
      ]
    },
    effect: {
      template: "systems/dnd5e/templates/activity/forward-effect.hbs"
    }
  };

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareActivationContext(context) {
    context = await super._prepareActivationContext(context);
    context.showConsumeSpellSlot = false;
    context.showScaling = true;
    return context;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareEffectContext(context) {
    context = await super._prepareEffectContext(context);
    context.activityOptions = [
      { value: "", label: "" },
      ...this.item.system.activities.contents
        .filter(a => (a.type !== "forward") && (CONFIG.DND5E.activityTypes[a.type] !== false))
        .map(activity => ({ value: activity.id, label: activity.name }))
    ];
    return context;
  }

  /* -------------------------------------------- */

  /**
   * Prepare the tab information for the sheet.
   * @returns {Record<string, Partial<ApplicationTab>>}
   * @protected
   */
  _getTabs() {
    return this._markTabs({
      identity: {
        id: "identity", group: "sheet", icon: "fa-solid fa-tag",
        label: "DND5E.ACTIVITY.SECTIONS.Identity"
      },
      activation: {
        id: "activation", group: "sheet", icon: "fa-solid fa-boxes-stacked",
        label: "DND5E.CONSUMPTION.FIELDS.consumption.label"
      },
      effect: {
        id: "effect", group: "sheet", icon: "fa-solid fa-sun",
        label: "DND5E.ACTIVITY.SECTIONS.Effect"
      }
    });
  }
}

const { DocumentIdField: DocumentIdField$7, SchemaField: SchemaField$I } = foundry.data.fields;

/**
 * Data model for a Forward activity.
 *
 * @property {object} activity
 * @property {string} activity.id  ID of the activity to forward to.
 */
class ForwardActivityData extends BaseActivityData {
  /** @inheritDoc */
  static defineSchema() {
    const schema = super.defineSchema();
    delete schema.duration;
    delete schema.effects;
    delete schema.range;
    delete schema.target;
    return {
      ...schema,
      activity: new SchemaField$I({
        id: new DocumentIdField$7()
      })
    };
  }
}

/**
 * Activity for triggering another activity with modified consumption.
 */
class ForwardActivity extends ActivityMixin(ForwardActivityData) {
  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static LOCALIZATION_PREFIXES = [...super.LOCALIZATION_PREFIXES, "DND5E.FORWARD"];

  /* -------------------------------------------- */

  /** @inheritDoc */
  static metadata = Object.freeze(
    foundry.utils.mergeObject(super.metadata, {
      type: "forward",
      img: "systems/dnd5e/icons/svg/activity/forward.svg",
      title: "DND5E.FORWARD.Title",
      sheetClass: ForwardSheet
    }, { inplace: false })
  );

  /* -------------------------------------------- */
  /*  Activation                                  */
  /* -------------------------------------------- */

  /** @override */
  async use(usage={}, dialog={}, message={}) {
    const usageConfig = foundry.utils.mergeObject({
      cause: {
        activity: this.relativeUUID
      },
      consume: {
        resources: false,
        spellSlot: false
      }
    }, usage);

    const activity = this.item.system.activities.get(this.activity.id);
    if ( !activity ) ui.notifications.error("DND5E.FORWARD.Warning.NoActivity", { localize: true });
    return activity?.use(usageConfig, dialog, message);
  }
}

/**
 * Sheet for the healing activity.
 */
class HealSheet extends ActivitySheet {

  /** @inheritDoc */
  static DEFAULT_OPTIONS = {
    classes: ["heal-activity"]
  };

  /* -------------------------------------------- */

  /** @inheritDoc */
  static PARTS = {
    ...super.PARTS,
    effect: {
      template: "systems/dnd5e/templates/activity/heal-effect.hbs",
      templates: [
        ...super.PARTS.effect.templates,
        "systems/dnd5e/templates/activity/parts/damage-part.hbs",
        "systems/dnd5e/templates/activity/parts/heal-healing.hbs"
      ]
    }
  };

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareEffectContext(context) {
    context = await super._prepareEffectContext(context);
    context.typeOptions = Object.entries(CONFIG.DND5E.healingTypes).map(([value, config]) => ({
      value, label: config.label, selected: context.activity.healing.types.has(value)
    }));
    const scaleKey = (this.item.type === "spell" && this.item.system.level === 0) ? "labelCantrip" : "label";
    context.scalingOptions = [
      { value: "", label: game.i18n.localize("DND5E.DAMAGE.Scaling.None") },
      ...Object.entries(CONFIG.DND5E.damageScalingModes).map(([value, { [scaleKey]: label }]) => ({ value, label }))
    ];
    return context;
  }
}

/**
 * Data model for an heal activity.
 *
 * @property {DamageData} healing
 */
class HealActivityData extends BaseActivityData {
  /** @inheritDoc */
  static defineSchema() {
    return {
      ...super.defineSchema(),
      healing: new DamageField()
    };
  }

  /* -------------------------------------------- */
  /*  Data Migrations                             */
  /* -------------------------------------------- */

  /** @override */
  static transformTypeData(source, activityData, options) {
    return foundry.utils.mergeObject(activityData, {
      healing: this.transformDamagePartData(source, source.system.damage?.parts?.[0] ?? ["", ""])
    });
  }

  /* -------------------------------------------- */
  /*  Data Preparation                            */
  /* -------------------------------------------- */

  /** @inheritDoc */
  prepareFinalData(rollData) {
    rollData ??= this.getRollData({ deterministic: true });
    super.prepareFinalData(rollData);
    this.prepareDamageLabel([this.healing], rollData);
  }

  /* -------------------------------------------- */
  /*  Helpers                                     */
  /* -------------------------------------------- */

  /** @override */
  getDamageConfig(config={}) {
    if ( !this.healing.formula ) return foundry.utils.mergeObject({ rolls: [] }, config);

    const rollConfig = foundry.utils.mergeObject({ critical: { allow: false }, scaling: 0 }, config);
    const rollData = this.getRollData();
    rollConfig.rolls = [this._processDamagePart(this.healing, rollConfig, rollData)].concat(config.rolls ?? []);

    return rollConfig;
  }
}

/**
 * Activity for rolling healing.
 */
class HealActivity extends ActivityMixin(HealActivityData) {
  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static LOCALIZATION_PREFIXES = [...super.LOCALIZATION_PREFIXES, "DND5E.HEAL"];

  /* -------------------------------------------- */

  /** @inheritDoc */
  static metadata = Object.freeze(
    foundry.utils.mergeObject(super.metadata, {
      type: "heal",
      img: "systems/dnd5e/icons/svg/activity/heal.svg",
      title: "DND5E.HEAL.Title",
      sheetClass: HealSheet,
      usage: {
        actions: {
          rollHealing: HealActivity.#rollHealing
        }
      }
    }, { inplace: false })
  );

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /** @override */
  get damageFlavor() {
    return game.i18n.localize("DND5E.HealingRoll");
  }

  /* -------------------------------------------- */
  /*  Activation                                  */
  /* -------------------------------------------- */

  /** @override */
  _usageChatButtons(message) {
    if ( !this.healing.formula ) return super._usageChatButtons(message);
    return [{
      label: game.i18n.localize("DND5E.Healing"),
      icon: '<i class="dnd5e-icon" data-src="systems/dnd5e/icons/svg/damage/healing.svg"></i>',
      dataset: {
        action: "rollHealing"
      }
    }].concat(super._usageChatButtons(message));
  }

  /* -------------------------------------------- */

  /** @override */
  async _triggerSubsequentActions(config, results) {
    this.rollDamage({ event: config.event }, {}, { data: { "flags.dnd5e.originatingMessage": results.message?.id } });
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /**
   * Handle performing a healing roll.
   * @this {HealActivity}
   * @param {PointerEvent} event     Triggering click event.
   * @param {HTMLElement} target     The capturing HTML element which defined a [data-action].
   * @param {ChatMessage5e} message  Message associated with the activation.
   */
  static #rollHealing(event, target, message) {
    this.rollDamage({ event });
  }
}

const { DocumentIdField: DocumentIdField$6, FilePathField: FilePathField$1, StringField: StringField$_ } = foundry.data.fields;

/**
 * Data model for an order activity.
 * @property {string} order  The issued order.
 */
class OrderActivityData extends BaseActivityData {
  /** @override */
  static defineSchema() {
    return {
      _id: new DocumentIdField$6({ initial: () => foundry.utils.randomID() }),
      type: new StringField$_({
        blank: false, required: true, readOnly: true, initial: () => this.metadata.type
      }),
      name: new StringField$_({ initial: undefined }),
      img: new FilePathField$1({ initial: undefined, categories: ["IMAGE"], base64: false }),
      order: new StringField$_({ required: true, blank: false, nullable: false })
    };
  }

  /* -------------------------------------------- */
  /*  Data Preparation                            */
  /* -------------------------------------------- */

  /** @inheritDoc */
  prepareData() {
    super.prepareData();
    this.img = CONFIG.DND5E.facilities.orders[this.order]?.icon || this.metadata?.img;
  }
}

const { BooleanField: BooleanField$A, DocumentUUIDField: DocumentUUIDField$2, NumberField: NumberField$D, StringField: StringField$Z } = foundry.data.fields;

/**
 * Dialog for configuring the usage of an order activity.
 */
class OrderUsageDialog extends ActivityUsageDialog {
  /** @override */
  static DEFAULT_OPTIONS = {
    actions: {
      deleteOccupant: OrderUsageDialog.#onDeleteOccupant,
      removeCraft: OrderUsageDialog.#onRemoveCraft
    }
  };

  /** @override */
  static PARTS = {
    order: {
      template: "systems/dnd5e/templates/activity/order-usage.hbs"
    },
    footer: {
      template: "templates/generic/form-footer.hbs"
    }
  };

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /**
   * Prepare render context for the build section.
   * @param {ApplicationRenderContext} context  Render context.
   * @param {HandlebarsRenderOptions} options   Render options.
   * @protected
   */
  _prepareBuildContext(context, options) {
    context.build = {
      choices: CONFIG.DND5E.facilities.sizes,
      field: new StringField$Z({ nullable: false, blank: false, label: "DND5E.FACILITY.FIELDS.size.label" }),
      name: "building.size",
      value: this.config.building?.size ?? "cramped"
    };
  }

  /* -------------------------------------------- */

  /**
   * Prepare render context for the costs section.
   * @param {ApplicationRenderContext} context  Render context.
   * @param {HandlebarsRenderOptions} options   Render options.
   * @param {number} options.days               The cost in days.
   * @param {number} options.gold               The cost in gold.
   * @protected
   */
  _prepareCostsContext(context, { days, gold }) {
    const { duration } = game.settings.get("dnd5e", "bastionConfiguration");
    context.costs = {
      days: {
        field: new NumberField$D({ nullable: true, integer: true, min: 0, label: "DND5E.TimeDay" }),
        name: "costs.days",
        value: this.config.costs?.days ?? days ?? duration
      },
      gold: {
        field: new NumberField$D({ nullable: true, integer: true, min: 0, label: "DND5E.CurrencyGP" }),
        name: "costs.gold",
        value: this.config.costs?.gold ?? gold ?? 0
      }
    };
  }

  /* -------------------------------------------- */

  /**
   * Prepare render context for the craft section.
   * @param {ApplicationRenderContext} context  Render context.
   * @param {HandlebarsRenderOptions} options   Render options.
   * @protected
   */
  async _prepareCraftContext(context, options) {
    const { craft } = this.item.system;
    context.craft = {
      legend: game.i18n.localize(`DND5E.FACILITY.Orders.${this.activity.order}.present`),
      item: {
        field: new DocumentUUIDField$2(),
        name: "craft.item",
        value: this.config.craft?.item ?? ""
      }
    };

    if ( this.activity.order === "harvest" ) {
      context.craft.isHarvesting = true;
      context.craft.item.value = this.config.craft?.item ?? craft.item ?? "";
      context.craft.quantity = {
        field: new NumberField$D({ nullable: false, integer: true, positive: true }),
        name: "craft.quantity",
        value: this.config.craft?.quantity ?? craft.quantity ?? 1
      };
    } else {
      context.craft.baseItem = {
        field: new BooleanField$A({
          label: "DND5E.FACILITY.Craft.BaseItem.Label",
          hint: "DND5E.FACILITY.Craft.BaseItem.Hint"
        }),
        name: "craft.buyBaseItem",
        value: this.config.craft?.buyBaseItem ?? false
      };
    }

    if ( context.craft.item.value ) {
      const item = await fromUuid(context.craft.item.value);
      context.craft.value = {
        img: item.img,
        name: item.name,
        contentLink: item.toAnchor().outerHTML
      };
    }
  }

  /* -------------------------------------------- */

  /**
   * Prepare render context for the enlarge order.
   * @param {ApplicationRenderContext} context  Render context.
   * @param {HandlebarsRenderOptions} options   Render options.
   * @returns {{ days: number, gold: number }}  The costs associated with performing this order.
   * @protected
   */
  _prepareEnlargeContext(context, options) {
    const sizes = Object.entries(CONFIG.DND5E.facilities.sizes).sort((a, b) => a.value - b.value);
    const index = sizes.findIndex(([size]) => size === this.item.system.size);
    const [, current] = sizes[index];
    const [, target] = sizes[index + 1];
    context.description = `
      <span class="current">${current.label}</span>
      <span class="separator">➡</span>
      <span class="target">${target.label}</span>
    `;
    const days = this.item.system.type.value === "basic" ? target.days - current.days : 0;
    return { days, gold: target.value - current.value };
  }

  /* -------------------------------------------- */

  /** @override */
  async _prepareFooterContext(context, options) {
    context.buttons = [{
      action: "use",
      type: "button",
      icon: "fas fa-hand-point-right",
      label: "DND5E.FACILITY.Order.Execute"
    }];
    return context;
  }

  /* -------------------------------------------- */

  /**
   * Prepare render context for orders.
   * @param {ApplicationRenderContext} context  Render context.
   * @param {HandlebarsRenderOptions} options   Render options.
   * @protected
   */
  async _prepareOrderContext(context, options) {
    if ( this.activity.order === "enlarge" ) {
      const { days, gold } = this._prepareEnlargeContext(context, options);
      this._prepareCostsContext(context, { ...options, days, gold });
      return;
    }

    if ( this.activity.order === "build" ) {
      const { days, value: gold } = CONFIG.DND5E.facilities.sizes.cramped;
      this._prepareBuildContext(context, options);
      this._prepareCostsContext(context, { ...options, days, gold });
      return;
    }

    let { duration } = game.settings.get("dnd5e", "bastionConfiguration");
    if ( (this.activity.order === "craft") || (this.activity.order === "harvest") ) {
      await this._prepareCraftContext(context, options);
    }
    else if ( this.activity.order === "trade" ) await this._prepareTradeContext(context, options);
    else {
      const config = CONFIG.DND5E.facilities.orders[this.activity.order];
      if ( config?.duration ) duration = config.duration;
    }

    this._prepareCostsContext(context, { ...options, days: duration });
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _preparePartContext(partId, context, options) {
    context = await super._preparePartContext(partId, context, options);
    switch ( partId ) {
      case "order": await this._prepareOrderContext(context, options); break;
    }
    return context;
  }

  /* -------------------------------------------- */

  /**
   * Prepare render context for the trade order.
   * @param {ApplicationRenderContext} context  Render context.
   * @param {HandlebarsRenderOptions} options   Render options.
   * @protected
   */
  async _prepareTradeContext(context, options) {
    const { trade } = this.item.system;
    if ( !trade.creatures.max && !trade.stock.max ) {
      context.trade = {
        stocked: {
          field: new BooleanField$A({
            label: "DND5E.FACILITY.Trade.Stocked.Label",
            hint: "DND5E.FACILITY.Trade.Stocked.Hint"
          }),
          name: "trade.stock.stocked",
          value: this.config.trade?.stock?.stocked ?? false
        }
      };
    } else {
      const isSelling = this.config.trade?.sell ?? false;
      context.trade = {
        sell: {
          field: new BooleanField$A({ label: "DND5E.FACILITY.Trade.Sell.Label" }),
          name: "trade.sell",
          value: isSelling
        }
      };

      if ( trade.stock.max ) {
        const max = isSelling ? trade.stock.value || 0 : trade.stock.max - (trade.stock.value ?? 0);
        const label = `DND5E.FACILITY.Trade.Stock.${isSelling ? "Sell" : "Buy"}`;
        context.trade.stock = {
          field: new NumberField$D({ label, max, min: 0, step: 1, nullable: false }),
          name: "trade.stock.value",
          value: this.config.trade?.stock?.value ?? 0
        };
      } else if ( trade.creatures.max ) {
        const sell = await Promise.all(trade.creatures.value.map(async (uuid, i) => {
          const doc = await fromUuid(uuid);
          return {
            contentLink: doc.toAnchor().outerHTML,
            field: new BooleanField$A(),
            name: "trade.creatures.sell",
            value: this.config.trade?.creatures?.sell?.[i] ?? false
          };
        }));
        const buy = await Promise.all(Array.fromRange(trade.creatures.max).map(async i => {
          let removable = true;
          let uuid = trade.creatures.value[i];
          if ( uuid ) removable = false;
          else uuid = this.config.trade?.creatures?.buy?.[i];
          const doc = await fromUuid(uuid);
          if ( doc ) return { removable, uuid, img: doc.img, name: doc.name };
          return { empty: true };
        }));
        context.trade.creatures = {
          buy, sell,
          hint: "DND5E.FACILITY.Trade.Creatures.Buy",
          price: {
            field: new NumberField$D({
              nullable: false, min: 0, integer: true,
              label: "DND5E.FACILITY.Trade.Price.Label",
              hint: "DND5E.FACILITY.Trade.Price.Hint"
            }),
            name: "trade.creatures.price",
            value: this.config.trade?.creatures?.price ?? 0
          }
        };
      }
    }
  }

  /* -------------------------------------------- */
  /*  Event Listeners & Handlers                  */
  /* -------------------------------------------- */

  /** @inheritDoc */
  _attachFrameListeners() {
    super._attachFrameListeners();
    this.element.addEventListener("drop", this._onDrop.bind(this));
  }

  /* -------------------------------------------- */

  /**
   * Handle drops onto the dialog.
   * @param {DragEvent} event  The drag-drop event.
   * @protected
   */
  _onDrop(event) {
    const data = TextEditor.getDragEventData(event);
    if ( (data.type !== "Actor") || !data.uuid ) return;
    const { trade } = this.item.system;
    if ( !this.config.trade?.creatures?.buy ) {
      this.config.trade ??= {};
      this.config.trade.creatures ??= {};
      this.config.trade.creatures.buy = [];
    }
    const index = Math.max(trade.creatures.value.length, this.config.trade.creatures.buy.length);
    if ( index + 1 > trade.creatures.max ) return;
    this.config.trade.creatures.buy[index] = data.uuid;
    this.render();
  }

  /* -------------------------------------------- */

  /**
   * Prepare submission data for build orders.
   * @param {object} submitData  Submission data.
   * @protected
   */
  _prepareBuildData(submitData) {
    if ( (this.config.building?.size ?? "cramped") !== submitData.building?.size ) {
      const { days, value: gold } = CONFIG.DND5E.facilities.sizes[submitData.building.size];
      Object.assign(submitData.costs, { days, gold });
    }
  }

  /* -------------------------------------------- */

  /**
   * Prepare submission data for craft orders.
   * @param {object} submitData  Submission data.
   * @returns {Promise<void>}
   * @protected
   */
  async _prepareCraftData(submitData) {
    let recalculateCosts = submitData.craft.item !== this.config.craft?.item;
    recalculateCosts ||= submitData.craft.buyBaseItem !== this.config.craft?.buyBaseItem;
    if ( (this.activity.order === "craft") && recalculateCosts ) {
      const item = await fromUuid(submitData.craft.item);
      const { days, gold } = await item.system.getCraftCost({
        baseItem: submitData.craft.buyBaseItem ? "buy" : "craft"
      });
      Object.assign(submitData.costs, { days, gold });
    }
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareSubmitData(event, formData) {
    const submitData = await super._prepareSubmitData(event, formData);
    if ( "building" in submitData ) this._prepareBuildData(submitData);
    if ( submitData.craft?.item ) await this._prepareCraftData(submitData);
    if ( "trade" in submitData ) await this._prepareTradeData(submitData);
    return submitData;
  }

  /* -------------------------------------------- */

  /**
   * Prepare submission data for trade orders.
   * @param {object} submitData  Submission data.
   * @returns {Promise<void>}
   * @protected
   */
  async _prepareTradeData(submitData) {
    // Clear data when toggling trade mode.
    if ( ("trade" in this.config) && (submitData.trade.sell !== this.config.trade?.sell) ) {
      delete this.config.trade.stock;
      delete this.config.trade.creatures;
      submitData.costs.gold = 0;
    }

    if ( ("stock" in submitData.trade) && !submitData.trade.sell ) {
      submitData.costs.gold = submitData.trade.stock.value;
    }

    if ( "creatures" in submitData.trade && !submitData.trade.sell ) {
      const buy = [];
      const { creatures } = submitData.trade;
      Object.keys(creatures.buy ?? {}).forEach(k => buy[k] = creatures.buy[k]);
      submitData.trade.creatures.buy = buy;
    }
  }

  /* -------------------------------------------- */

  /**
   * Handle removing a configured occupant.
   * @this {OrderUsageDialog}
   * @param {PointerEvent} event  The triggering event.
   * @param {HTMLElement} target  The event target.
   */
  static #onDeleteOccupant(event, target) {
    const { index } = target.closest("[data-index]")?.dataset ?? {};
    this.config.trade.creatures.buy.splice(index, 1);
    this.render();
  }

  /* -------------------------------------------- */

  /**
   * Handle clearing the currently configured item for crafting.
   * @this {OrderUsageDialog}
   */
  static #onRemoveCraft() {
    delete this.config.craft.item;
    this.render();
  }
}

/**
 * @typedef AwardOptions
 * @property {Record<string, number>|null} currency  Amount of each currency to award.
 * @property {boolean} each                          Distribute full award to each destination, rather than dividing it
 *                                                   among the destinations.
 * @property {Set<string>} savedDestinations         Set of IDs for previously selected destinations.
 * @property {number|null} xp                        Amount of experience points to award.
 */

/**
 * Application for awarding XP and currency to players.
 */
class Award extends Application5e {
  constructor(options, _options={}) {
    if ( options instanceof foundry.abstract.Document ) {
      foundry.utils.logCompatibilityWarning(
        "The `Award` origin actor should now be passed within the options object as `origin`.",
        { since: "DnD5e 4.3", until: "DnD5e 4.5" }
      );
      _options.origin = options;
      options = _options;
    }

    if ( !options ) options = _options;
    for ( const key of ["currency", "each", "savedDestinations", "xp"] ) {
      if ( !(key in options) ) continue;
      options.award ??= {};
      options.award[key] = options[key];
      delete options[key];
      foundry.utils.logCompatibilityWarning(
        `The \`${key}\` option in \`Award\` has been moved to \`award.${key}\`.`,
        { since: "DnD5e 4.3", until: "DnD5e 4.5" }
      );
    }

    super(options);
  }

  /* -------------------------------------------- */

  /** @override */
  static DEFAULT_OPTIONS = {
    award: {
      currency: null,
      each: false,
      savedDestinations: new Set(),
      xp: null
    },
    classes: ["award", "standard-form"],
    form: {
      handler: Award.#handleFormSubmission
    },
    origin: null,
    position: {
      width: 350
    },
    tag: "form",
    window: {
      title: "DND5E.Award.Title"
    }
  };

  /* -------------------------------------------- */

  /** @override */
  static PARTS = {
    award: {
      template: "systems/dnd5e/templates/apps/award.hbs"
    }
  };

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * Award options.
   * @type {AwardOptions}
   */
  get award() {
    return this.options.award;
  }

  /* -------------------------------------------- */

  /**
   * Group actor from which this award is being granted.
   * @type {Actor5e}
   */
  get origin() {
    return this.options.origin;
  }

  /* -------------------------------------------- */

  /**
   * Destinations to which XP & currency can be awarded.
   * @type {Actor5e[]}
   */
  get transferDestinations() {
    if ( this.isPartyAward ) return this.origin.system.transferDestinations ?? [];
    if ( !game.user.isGM ) return [];
    const primaryParty = game.settings.get("dnd5e", "primaryParty")?.actor;
    return primaryParty
      ? [primaryParty, ...primaryParty.system.transferDestinations]
      : game.users.map(u => u.character).filter(c => c);
  }

  /* -------------------------------------------- */

  /**
   * Is this award coming from a party group actor rather than the /award command?
   * @type {boolean}
   */
  get isPartyAward() {
    return this.origin?.system.type?.value === "party";
  }

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareContext(options) {
    const context = await super._prepareContext(options);

    context.currency = Object.entries(CONFIG.DND5E.currencies).reduce((obj, [k, { label, icon }]) => {
      obj[k] = {
        label, icon,
        value: this.award.currency ? this.award.currency[k] : this.origin?.system.currency[k]
      };
      return obj;
    }, {});
    context.destinations = Award.prepareDestinations(this.transferDestinations, this.award.savedDestinations);
    context.each = this.award.each ?? false;
    context.hideXP = game.settings.get("dnd5e", "levelingMode") === "noxp";
    context.noPrimaryParty = !game.settings.get("dnd5e", "primaryParty")?.actor && !this.isPartyAward;
    context.xp = this.award.xp ?? this.origin?.system.details.xp.value ?? this.origin?.system.details.xp.derived;

    return context;
  }

  /* -------------------------------------------- */

  /**
   * Apply type icons to transfer destinations and prepare them for display in the list.
   * @param {Document[]} destinations          Destination documents to prepare.
   * @param {Set<string>} [savedDestinations]  IDs of targets to pre-check.
   * @returns {{doc: Document, icon: string}[]}
   */
  static prepareDestinations(destinations, savedDestinations) {
    const icons = {
      container: '<dnd5e-icon class="fa-fw" src="systems/dnd5e/icons/svg/backpack.svg"></dnd5e-icon>',
      group: '<i class="fa-solid fa-people-group"></i>',
      vehicle: '<i class="fa-solid fa-sailboat"></i>'
    };
    return destinations.map(doc => ({
      doc, checked: savedDestinations?.has(doc.id), icon: icons[doc.type] ?? '<i class="fa-solid fa-fw fa-user"></i>'
    }));
  }

  /* -------------------------------------------- */
  /*  Life-Cycle Handlers                         */
  /* -------------------------------------------- */

  /** @inheritDoc */
  _onRender(context, options) {
    super._onRender(context, options);
    this._validateForm();
  }

  /* -------------------------------------------- */
  /*  Form Handling                               */
  /* -------------------------------------------- */

  /** @inheritDoc */
  _onChangeForm(formConfig, event) {
    super._onChangeForm(formConfig, event);
    this._validateForm();
  }

  /* -------------------------------------------- */

  /**
   * Ensure the award form is in a valid form to be submitted.
   * @protected
   */
  _validateForm() {
    const formData = new FormDataExtended(this.element);
    const data = foundry.utils.expandObject(formData.object);
    let valid = true;
    if ( !filteredKeys(data.amount ?? {}).length && !data.xp ) valid = false;
    if ( !filteredKeys(data.destination ?? {}).length ) valid = false;
    this.element.querySelector('button[name="transfer"]').disabled = !valid;
  }

  /* -------------------------------------------- */

  /**
   * Handle submitting the award form.
   * @this {Award}
   * @param {Event|SubmitEvent} event    The form submission event.
   * @param {HTMLFormElement} form       The submitted form.
   * @param {FormDataExtended} formData  Data from the dialog.
   */
  static async #handleFormSubmission(event, form, formData) {
    const data = foundry.utils.expandObject(formData.object);
    const destinations = this.transferDestinations.filter(d => data.destination[d.id]);
    const each = data.each;
    this._saveDestinations(destinations);
    const results = new Map();
    await this.constructor.awardCurrency(data.amount, destinations, { each, origin: this.origin, results });
    await this.constructor.awardXP(data.xp, destinations, { each, origin: this.origin, results });
    this.constructor.displayAwardMessages(results);
    this.close();
  }

  /* -------------------------------------------- */

  /**
   * Save the selected destination IDs to either the current group's flags or the user's flags.
   * @param {Actor5e[]} destinations  Selected destinations to save.
   * @protected
   */
  _saveDestinations(destinations) {
    const target = this.isPartyAward ? this.origin : game.user;
    target.setFlag("dnd5e", "awardDestinations", destinations);
  }

  /* -------------------------------------------- */
  /*  Awarding Methods                            */
  /* -------------------------------------------- */

  /**
   * Award currency, optionally transferring between one document and another.
   * @param {Record<string, number>} amounts   Amount of each denomination to transfer.
   * @param {(Actor5e|Item5e)[]} destinations  Documents that should receive the currency.
   * @param {object} [config={}]
   * @param {boolean} [config.each=false]      Award the specified amount to each player, rather than splitting it.
   * @param {Actor5e|Item5e} [config.origin]   Document from which to move the currency, if not a freeform award.
   * @param {Map<Actor5e|Item5e, object>} [config.results]  Results of the award operation.
   */
  static async awardCurrency(amounts, destinations, { each=false, origin, results=new Map() }={}) {
    if ( !destinations.length ) return;
    const originCurrency = origin ? foundry.utils.deepClone(origin.system.currency) : null;

    for ( const k of Object.keys(amounts) ) {
      if ( each ) amounts[k] = amounts[k] * destinations.length;
      if ( origin ) amounts[k] = Math.min(amounts[k], originCurrency[k] ?? 0);
    }

    let remainingDestinations = destinations.length;
    for ( const destination of destinations ) {
      const destinationUpdates = {};
      if ( !results.has(destination) ) results.set(destination, {});
      const result = results.get(destination).currency ??= {};

      for ( let [key, amount] of Object.entries(amounts) ) {
        if ( !amount ) continue;
        amount = Math.clamp(
          // Divide amount between remaining destinations
          Math.floor(amount / remainingDestinations),
          // Ensure negative amounts aren't more than is contained in destination
          -destination.system.currency[key],
          // Ensure positive amounts aren't more than is contained in origin
          originCurrency ? originCurrency[key] : Infinity
        );
        amounts[key] -= amount;
        if ( originCurrency ) originCurrency[key] -= amount;
        destinationUpdates[`system.currency.${key}`] = destination.system.currency[key] + amount;
        result[key] = amount;
      }

      await destination.update(destinationUpdates);
      remainingDestinations -= 1;
    }

    if ( origin ) await origin.update({ "system.currency": originCurrency });
  }

  /* -------------------------------------------- */

  /**
   * Award XP split across the provided destination actors.
   * @param {number} amount            Amount of XP to award.
   * @param {Actor5e[]} destinations   Actors that should receive the XP.
   * @param {object} [config={}]
   * @param {boolean} [config.each=false]      Award the specified amount to each player, rather than splitting it.
   * @param {Actor5e} [config.origin]  Group actor from which to transfer the XP.
   * @param {Map<Actor5e|Item5e, object>} [config.results]  Results of the award operation.
   */
  static async awardXP(amount, destinations, { each=false, origin, results=new Map() }={}) {
    destinations = destinations.filter(d => ["character", "group"].includes(d.type));
    if ( !amount || !destinations.length ) return;

    const xp = origin?.system.details.xp;
    let originUpdate = origin ? (xp.value ?? xp.derived ?? 0) : Infinity;
    if ( each ) amount = amount * destinations.length;
    const perDestination = Math.floor(Math.min(amount, originUpdate) / destinations.length);
    originUpdate -= amount;
    for ( const destination of destinations ) {
      await destination.update({ "system.details.xp.value": destination.system.details.xp.value + perDestination });
      if ( !results.has(destination) ) results.set(destination, {});
      const result = results.get(destination);
      result.xp = perDestination;
    }

    if ( Number.isFinite(originUpdate) ) await origin.update({ "system.details.xp.value": originUpdate });
  }

  /* -------------------------------------------- */

  /**
   * Display chat messages for awarded currency and XP.
   * @param {Map<Actor5e|Item5e, object>} results  Results of any award operations.
   */
  static async displayAwardMessages(results) {
    const cls = getDocumentClass("ChatMessage");
    const messages = [];
    for ( const [destination, result] of results ) {
      const entries = [];
      for ( const [key, amount] of Object.entries(result.currency ?? {}) ) {
        const label = CONFIG.DND5E.currencies[key].label;
        entries.push(`
          <span class="award-entry">
            ${formatNumber(amount)} <i class="currency ${key}" data-tooltip="${label}" aria-label="${label}"></i>
          </span>
        `);
      }
      if ( result.xp ) entries.push(`
        <span class="award-entry">
          ${formatNumber(result.xp)} ${game.i18n.localize("DND5E.ExperiencePoints.Abbreviation")}
        </span>
      `);
      if ( !entries.length ) continue;

      const content = game.i18n.format("DND5E.Award.Message", {
        name: destination.name, award: `<span class="dnd5e2">${game.i18n.getListFormatter().format(entries)}</span>`
      });

      const whisperTargets = game.users.filter(user => destination.testUserPermission(user, "OWNER"));
      const whisper = whisperTargets.length !== game.users.size;
      const messageData = {
        content,
        whisper: whisper ? whisperTargets : []
      };
      messages.push(messageData);
    }
    if ( messages.length ) cls.createDocuments(messages);
  }

  /* -------------------------------------------- */
  /*  Chat Command                                */
  /* -------------------------------------------- */

  /**
   * Regular expression used to match the /award command in chat messages.
   * @type {RegExp}
   */
  static COMMAND_PATTERN = new RegExp(/^\/award(?:\s|$)/i);

  /* -------------------------------------------- */

  /**
   * Regular expression used to split currency & xp values from their labels.
   * @type {RegExp}
   */
  static VALUE_PATTERN = new RegExp(/^(.+?)(\D+)$/);

  /* -------------------------------------------- */

  /**
   * Use the `chatMessage` hook to determine if an award command was typed.
   * @param {string} message   Text of the message being posted.
   * @returns {boolean|void}   Returns `false` to prevent the message from continuing to parse.
   */
  static chatMessage(message) {
    if ( !this.COMMAND_PATTERN.test(message) ) return;
    this.handleAward(message);
    return false;
  }

  /* -------------------------------------------- */

  /**
   * Parse the award command and grant an award.
   * @param {string} message  Award command typed in chat.
   */
  static async handleAward(message) {
    if ( !game.user.isGM ) {
      ui.notifications.error("DND5E.Award.NotGMError", { localize: true });
      return;
    }

    try {
      const { currency, xp, party, each } = this.parseAwardCommand(message);

      for ( const [key, formula] of Object.entries(currency) ) {
        const roll = new Roll(formula);
        await roll.evaluate();
        currency[key] = roll.total;
      }

      // If the party command is set, a primary party is set, and the award isn't empty, skip the UI
      const primaryParty = game.settings.get("dnd5e", "primaryParty")?.actor;
      if ( party && primaryParty && (xp || filteredKeys(currency).length) ) {
        const destinations = each ? primaryParty.system.playerCharacters : [primaryParty];
        const results = new Map();
        await this.awardCurrency(currency, destinations, { each, results });
        await this.awardXP(xp, destinations, { each, results });
        this.displayAwardMessages(results);
      }

      // Otherwise show the UI with defaults
      else {
        const savedDestinations = game.user.getFlag("dnd5e", "awardDestinations");
        const app = new Award({ award: { currency, xp, each, savedDestinations } });
        app.render({ force: true });
      }
    } catch(err) {
      ui.notifications.warn(err.message);
    }
  }

  /* -------------------------------------------- */

  /**
   * Parse the award command.
   * @param {string} message  Award command typed in chat.
   * @returns {{currency: Record<string, number>, xp: number, party: boolean}}
   */
  static parseAwardCommand(message) {
    const command = message.replace(this.COMMAND_PATTERN, "").toLowerCase();

    const currency = {};
    let each = false;
    let party = false;
    let xp;
    const unrecognized = [];
    for ( const part of command.split(" ") ) {
      if ( !part ) continue;
      let [, amount, label] = part.match(this.VALUE_PATTERN) ?? [];
      label = label?.toLowerCase();
      try {
        new Roll(amount);
        if ( label in CONFIG.DND5E.currencies ) currency[label] = amount;
        else if ( label === "xp" ) xp = Number(amount);
        else if ( part === "each" ) each = true;
        else if ( part === "party" ) party = true;
        else throw new Error();
      } catch(err) {
        unrecognized.push(part);
      }
    }

    // Display warning about an unrecognized commands
    if ( unrecognized.length ) throw new Error(game.i18n.format("DND5E.Award.UnrecognizedWarning", {
      commands: game.i18n.getListFormatter().format(unrecognized.map(u => `"${u}"`))
    }));

    return { currency, xp, each, party };
  }
}

/**
 * Application for performing currency conversions & transfers.
 */
class CurrencyManager extends Application5e {
  constructor(options, _options={}) {
    if ( options instanceof foundry.abstract.Document ) {
      foundry.utils.logCompatibilityWarning(
        "The `CurrencyManager` document should now be passed within the options object as `document`.",
        { since: "DnD5e 4.3", until: "DnD5e 4.5" }
      );
      _options.document = options;
      options = _options;
    }
    super(options);
  }

  /* -------------------------------------------- */

  /** @override */
  static DEFAULT_OPTIONS = {
    actions: {
      setAll: CurrencyManager.#setTransferValue,
      setHalf: CurrencyManager.#setTransferValue
    },
    classes: ["currency-manager", "standard-form"],
    document: null,
    form: {
      closeOnSubmit: true,
      handler: CurrencyManager.#handleFormSubmission
    },
    position: {
      width: 350
    },
    tag: "form",
    window: {
      title: "DND5E.CurrencyManager.Title"
    }
  };

  /* -------------------------------------------- */

  /** @override */
  static PARTS = {
    tabs: {
      template: "templates/generic/tab-navigation.hbs"
    },
    convert: {
      template: "systems/dnd5e/templates/apps/currency-manager-convert.hbs"
    },
    transfer: {
      template: "systems/dnd5e/templates/apps/currency-manager-transfer.hbs"
    }
  };

  /* -------------------------------------------- */

  /** @override */
  tabGroups = {
    primary: "transfer"
  };

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * Document for which the currency is being managed.
   * @type {Actor5e|Item5e}
   */
  get document() {
    return this.options.document;
  }

  /* -------------------------------------------- */

  /**
   * Destinations to which currency can be transferred.
   * @type {(Actor5e|Item5e)[]}
   */
  get transferDestinations() {
    const destinations = [];
    const actor = this.document instanceof Actor ? this.document : this.document.parent;
    if ( actor && (actor !== this.document) ) destinations.push(actor);
    destinations.push(...(actor?.system.transferDestinations ?? []));
    destinations.push(...(actor?.itemTypes.container.filter(b => b !== this.document) ?? []));
    if ( game.user.isGM ) {
      const primaryParty = game.settings.get("dnd5e", "primaryParty")?.actor;
      if ( primaryParty && (this.document !== primaryParty) && !destinations.includes(primaryParty) ) {
        destinations.push(primaryParty);
      }
    }
    return destinations;
  }

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareContext(options) {
    const context = await super._prepareContext(options);

    context.currency = this.document.system.currency;
    context.destinations = Award.prepareDestinations(this.transferDestinations);
    context.tabs = this._getTabs();

    return context;
  }

  /* -------------------------------------------- */

  /** @override */
  async _preparePartContext(partId, context) {
    context = await super._preparePartContext(partId, context);
    context.tab = context.tabs[partId];
    return context;
  }

  /* -------------------------------------------- */

  /**
   * Prepare the tab information for the sheet.
   * @returns {Record<string, Partial<ApplicationTab>>}
   * @protected
   */
  _getTabs() {
    return {
      convert: {
        id: "convert", group: "primary", icon: "fa-solid fa-arrow-up-short-wide",
        label: "DND5E.CurrencyManager.Convert.Label",
        active: this.tabGroups.primary === "convert",
        cssClass: this.tabGroups.primary === "convert" ? "active" : ""
      },
      transfer: {
        id: "transfer", group: "primary", icon: "fa-solid fa-reply-all fa-flip-horizontal",
        label: "DND5E.CurrencyManager.Transfer.Label",
        active: this.tabGroups.primary === "transfer",
        cssClass: this.tabGroups.primary === "transfer" ? "active" : ""
      }
    };
  }

  /* -------------------------------------------- */
  /*  Event Handling                              */
  /* -------------------------------------------- */

  /**
   * Handle setting the transfer amount based on the buttons.
   * @this {CurrencyManager}
   * @param {Event} event         Triggering click event.
   * @param {HTMLElement} target  Button that was clicked.
   * @protected
   */
  static #setTransferValue(event, target) {
    for ( let [key, value] of Object.entries(this.document.system.currency) ) {
      if ( target.dataset.action === "setHalf" ) value = Math.floor(value / 2);
      const input = this.element.querySelector(`[name="amount.${key}"]`);
      if ( input && value ) input.value = value;
    }
    this._validateForm();
  }

  /* -------------------------------------------- */
  /*  Form Handling                               */
  /* -------------------------------------------- */

  /** @inheritDoc */
  _onChangeForm(formConfig, event) {
    super._onChangeForm(formConfig, event);
    this._validateForm();
  }

  /* -------------------------------------------- */

  /**
   * Ensure the transfer form is in a valid form to be submitted.
   * @protected
   */
  _validateForm() {
    const formData = new FormDataExtended(this.element);
    const data = foundry.utils.expandObject(formData.object);
    let valid = true;
    if ( !filteredKeys(data.amount ?? {}).length ) valid = false;
    if ( !filteredKeys(data.destination ?? {}).length ) valid = false;
    this.element.querySelector('button[name="transfer"]').disabled = !valid;
  }

  /* -------------------------------------------- */

  /**
   * Handle submitting the currency manager form.
   * @this {Award}
   * @param {Event|SubmitEvent} event    The form submission event.
   * @param {HTMLFormElement} form       The submitted form.
   * @param {FormDataExtended} formData  Data from the dialog.
   */
  static async #handleFormSubmission(event, form, formData) {
    const data = foundry.utils.expandObject(formData.object);
    switch ( event.submitter?.name ) {
      case "convert":
        await this.constructor.convertCurrency(this.document);
        break;
      case "transfer":
        const destinations = this.transferDestinations.filter(d => data.destination[d.id]);
        await this.constructor.transferCurrency(this.document, destinations, data.amount);
        break;
    }
  }

  /* -------------------------------------------- */
  /*  Currency Operations                         */
  /* -------------------------------------------- */

  /**
   * Convert all carried currency to the highest possible denomination using configured conversion rates.
   * See CONFIG.DND5E.currencies for configuration.
   * @param {Actor5e|Item5e} doc  Actor or container item to convert.
   * @returns {Promise<Actor5e|Item5e>}
   */
  static convertCurrency(doc) {
    const currency = foundry.utils.deepClone(doc.system.currency);

    const currencies = Object.entries(CONFIG.DND5E.currencies);
    currencies.sort((a, b) => a[1].conversion - b[1].conversion);

    // Count total converted units of the base currency
    let basis = currencies.reduce((change, [denomination, config]) => {
      if ( !config.conversion ) return change;
      return change + (currency[denomination] / config.conversion);
    }, 0);

    // Convert base units into the highest denomination possible
    for ( const [denomination, config] of currencies) {
      if ( !config.conversion ) continue;
      const amount = Math.floor(basis * config.conversion);
      currency[denomination] = amount;
      basis -= (amount / config.conversion);
    }

    // Save the updated currency object
    return doc.update({ "system.currency": currency });
  }

  /* -------------------------------------------- */

  /**
   * Deduct a certain amount of currency from a given Actor.
   * @param {Actor5e} actor                          The actor.
   * @param {number} amount                          The amount of currency.
   * @param {string} denomination                    The currency's denomination.
   * @param {object} [options]
   * @param {boolean} [options.recursive=false]      Deduct currency from containers as well as the base Actor. TODO
   * @param {"high"|"low"} [options.priority="low"]  Prioritize higher denominations before lower, or vice-versa.
   * @param {boolean} [options.exact=true]           Prioritize deducting the requested denomination first.
   * @throws {Error} If the Actor does not have sufficient currency.
   * @returns {Promise<Actor5e>|void}
   */
  static deductActorCurrency(actor, amount, denomination, options={}) {
    if ( amount <= 0 ) return;
    // eslint-disable-next-line no-unused-vars
    const { item, remainder, ...updates } = this.getActorCurrencyUpdates(actor, amount, denomination, options);
    if ( remainder ) throw new Error(game.i18n.format("DND5E.CurrencyManager.Error.InsufficientFunds", {
      denomination,
      amount: new Intl.NumberFormat(game.i18n.lang).format(amount),
      name: actor.name
    }));
    return actor.update(updates);
  }

  /* -------------------------------------------- */

  /**
   * Determine model updates for deducting a certain amount of currency from a given Actor.
   * @param {Actor5e} actor                          The actor.
   * @param {number} amount                          The amount of currency.
   * @param {string} denomination                    The currency's denomination.
   * @param {object} [options]
   * @param {boolean} [options.recursive=false]      Deduct currency from containers as well as the base Actor. TODO
   * @param {"high"|"low"} [options.priority="low"]  Prioritize higher denominations before lower, or vice-versa.
   * @param {boolean} [options.exact=true]           Prioritize deducting the requested denomination first.
   * @returns {{ item: object[], remainder: number, [p: string]: any }}
   */
  static getActorCurrencyUpdates(actor, amount, denomination, { recursive=false, priority="low", exact=true }={}) {
    const { currency } = actor.system;
    const updates = { system: { currency: { ...currency } }, remainder: amount, item: [] };
    if ( amount <= 0 ) return updates;

    const currencies = Object.entries(CONFIG.DND5E.currencies).map(([denom, { conversion }]) => {
      return [denom, conversion];
    }).sort(([, a], [, b]) => priority === "high" ? a - b : b - a);
    const baseConversion = CONFIG.DND5E.currencies[denomination].conversion;

    if ( exact ) currencies.unshift([denomination, baseConversion]);
    for ( const [denom, conversion] of currencies ) {
      const multiplier = conversion / baseConversion;
      const deduct = Math.min(updates.system.currency[denom], Math.floor(updates.remainder * multiplier));
      updates.remainder -= deduct / multiplier;
      updates.system.currency[denom] -= deduct;
      if ( !updates.remainder ) return updates;
    }

    return updates;
  }

  /* -------------------------------------------- */

  /**
   * Transfer currency between one document and another.
   * @param {Actor5e|Item5e} origin       Document from which to move the currency.
   * @param {Document[]} destinations     Documents that should receive the currency.
   * @param {object[]} amounts            Amount of each denomination to transfer.
   */
  static async transferCurrency(origin, destinations, amounts) {
    Award.awardCurrency(amounts, destinations, { origin });
  }
}

/**
 * @typedef {ActivityUseConfiguration} OrderUseConfiguration
 * @property {object} [building]
 * @property {string} [building.size]            The size of facility to build.
 * @property {object} [costs]
 * @property {number} [costs.days]               The cost of executing the order, in days.
 * @property {number} [costs.gold]               The cost of executing the order, in gold.
 * @property {boolean} [costs.paid]              Whether the gold cost has been paid.
 * @property {object} [craft]
 * @property {string} [craft.item]               The item being crafted or harvested.
 * @property {number} [craft.quantity]           The quantity of items to harvest.
 * @property {object} [trade]
 * @property {boolean} [trade.sell]              Whether the trade was a sell operation.
 * @property {object} [trade.stock]
 * @property {boolean} [trade.stock.stocked]     Whether the order was to fully stock the inventory.
 * @property {boolean} [trade.stock.value]       The base value of goods transacted.
 * @property {object} [trade.creatures]
 * @property {string[]} [trade.creatures.buy]    Additional animals purchased.
 * @property {boolean[]} [trade.creatures.sell]  Whether a creature in a given slot was sold.
 * @property {number} [trade.creatures.price]    The base value of the animals sold.
 */

/**
 * An activity for issuing an order to a facility.
 */
class OrderActivity extends ActivityMixin(OrderActivityData) {
  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static metadata = Object.freeze(foundry.utils.mergeObject(super.metadata, {
    type: "order",
    img: "systems/dnd5e/icons/svg/activity/order.svg",
    title: "DND5E.FACILITY.Order.Issue",
    usage: {
      actions: {
        pay: OrderActivity.#onPayOrder
      },
      chatCard: "systems/dnd5e/templates/chat/order-activity-card.hbs",
      dialog: OrderUsageDialog
    }
  }, { inplace: false }));

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /** @inheritDoc */
  get canUse() {
    return super.canUse
      // Don't allow usage if facility is already executing the same order
      && !this.inProgress
      // Enlarge order cannot be executed if facility is already maximum size
      && ((this.order !== "enlarge") || (this.parent.size !== "vast"));
  }

  /* -------------------------------------------- */

  /**
   * Is this order currently in the process of being executed by its facility?
   * @type {boolean}
   */
  get inProgress() {
    if ( this.parent.progress.order !== this.order ) return false;
    // TODO: Ideally this would also check to see if the order has already been paid,
    // but that information is only part of the chat message and there isn't a clean
    // way to retrieve it at the moment
    return this.parent.progress.value > 0;
  }

  /* -------------------------------------------- */
  /*  Activation                                  */
  /* -------------------------------------------- */

  /**
   * Update building configuration.
   * @param {OrderUseConfiguration} usageConfig  Order configuration.
   * @param {object} updates                     Item updates.
   * @protected
   */
  _finalizeBuild(usageConfig, updates) {
    updates["system.building.size"] = usageConfig.building.size;
  }

  /* -------------------------------------------- */

  /**
   * Update costs.
   * @param {OrderUseConfiguration} usageConfig  Order configuration.
   * @param {object} updates                     Item updates.
   * @protected
   */
  _finalizeCosts(usageConfig, updates) {
    const { costs } = usageConfig;
    if ( costs.days ) updates["system.progress"] = { value: 0, max: costs.days, order: this.order };
  }

  /* -------------------------------------------- */

  /**
   * Update crafting configuration.
   * @param {OrderUseConfiguration} usageConfig  Order configuration.
   * @param {object} updates                     Item updates.
   * @protected
   */
  _finalizeCraft(usageConfig, updates) {
    const { craft } = usageConfig;
    updates["system.craft"] = { item: craft.item, quantity: 1 };
    if ( this.order === "harvest" ) updates["system.craft"].quantity = craft.quantity;
  }

  /* -------------------------------------------- */

  /**
   * Update facility size.
   * @param {OrderUseConfiguration} usageConfig  Order configuration.
   * @param {object} updates                     Item updates.
   * @protected
   */
  _finalizeEnlarge(usageConfig, updates) {
    // Special facilities enlarge immediately.
    if ( (this.item.system.type.value !== "special") || (this.item.system.size === "vast") ) return;
    const sizes = Object.entries(CONFIG.DND5E.facilities.sizes).sort((a, b) => a.value - b.value);
    const index = sizes.findIndex(([size]) => size === this.item.system.size);
    updates["system.size"] = sizes[index + 1][0];
  }

  /* -------------------------------------------- */

  /**
   * Update trading configuration.
   * @param {OrderUseConfiguration} usageConfig  Order configuration.
   * @param {object} updates                     Item updates.
   * @protected
   */
  _finalizeTrade(usageConfig, updates) {
    const { costs, trade } = usageConfig;
    const { system } = this.item;
    updates["system.trade.pending.operation"] = trade.sell ? "sell" : "buy";
    updates["system.trade.pending.creatures"] = [];
    updates["system.trade.pending.value"] = null;
    if ( trade.stock ) {
      if ( "stocked" in trade.stock ) {
        updates["system.trade.pending.stocked"] = trade.stock.stocked;
        updates["system.trade.pending.operation"] = trade.stock.stocked ? "buy" : null;
      }
      else updates["system.trade.pending.value"] = trade.stock.value;
    }
    if ( trade.creatures ) {
      let creatures = (trade.creatures.buy ?? []).filter(_ => _);
      if ( trade.sell ) {
        creatures = [];
        for ( let i = 0; i < trade.creatures.sell?.length ?? 0; i++ ) {
          const sold = trade.creatures.sell[i];
          if ( sold ) creatures.push(system.trade.creatures.value[i]);
        }
      }
      updates["system.trade.pending.value"] = trade.sell ? (trade.creatures.price ?? 0) : costs.gold;
      updates["system.trade.pending.creatures"] = creatures;
      // TODO: We need a way to visualize 'pending' animal purchases/sales. For now update immediately.
      updates["system.trade.creatures.value"] = trade.sell
        ? system.trade.creatures.value.filter((_, i) => !trade.creatures.sell[i])
        : system.trade.creatures.value.concat(creatures);
    }
  }

  /* -------------------------------------------- */

  /** @override */
  async _finalizeUsage(usageConfig, results) {
    const updates = {};
    switch ( this.order ) {
      case "build": this._finalizeBuild(usageConfig, updates); break;
      case "craft":
      case "harvest":
        this._finalizeCraft(usageConfig, updates);
        break;
      case "enlarge": this._finalizeEnlarge(usageConfig, updates); break;
      case "trade": this._finalizeTrade(usageConfig, updates); break;
    }
    this._finalizeCosts(usageConfig, updates);
    return this.item.update(updates);
  }

  /* -------------------------------------------- */

  /** @override */
  _prepareUsageConfig(config) {
    config.consume = false;
    return config;
  }

  /* -------------------------------------------- */

  /** @override */
  _prepareUsageScaling(usageConfig, messageConfig, item) {
    // FIXME: No scaling happening here, but this is the only context we have both usageConfig and messageConfig.
    const { costs, craft, trade } = usageConfig;
    messageConfig.data.flags.dnd5e.order = { costs, craft, trade };
  }

  /* -------------------------------------------- */

  /** @override */
  _requiresConfigurationDialog(config) {
    return true;
  }

  /* -------------------------------------------- */

  /** @override */
  _usageChatButtons(message) {
    const { costs } = message.data.flags.dnd5e.order;
    if ( !costs.gold || costs.paid ) return [];
    return [{
      label: game.i18n.localize("DND5E.FACILITY.Costs.Automatic"),
      icon: '<i class="fas fa-coins"></i>',
      dataset: { action: "pay", method: "automatic" }
    }, {
      label: game.i18n.localize("DND5E.FACILITY.Costs.Manual"),
      icon: '<i class="fas fa-clipboard-check"></i>',
      dataset: { action: "pay", method: "manual" }
    }];
  }

  /* -------------------------------------------- */

  /** @override */
  async _usageChatContext(message) {
    const { costs, craft, trade } = message.data.flags.dnd5e.order;
    const { type } = this.item.system;
    const supplements = [];
    if ( costs.days ) supplements.push(`
      <strong>${game.i18n.localize("DND5E.DurationTime")}</strong>
      ${game.i18n.format("DND5E.FACILITY.Costs.Days", { days: costs.days })}
    `);
    if ( costs.gold ) supplements.push(`
      <strong>${game.i18n.localize("DND5E.CurrencyGP")}</strong>
      ${formatNumber(costs.gold)}
      (${game.i18n.localize(`DND5E.FACILITY.Costs.${costs.paid ? "Paid" : "Unpaid"}`)})
    `);
    if ( craft?.item ) {
      const item = await fromUuid(craft.item);
      supplements.push(`
        <strong>${game.i18n.localize("DOCUMENT.Items")}</strong>
        ${craft.quantity > 1 ? `${craft.quantity}&times;` : ""}
        ${item.toAnchor().outerHTML}
      `);
    }
    if ( trade?.stock?.value && trade.sell ) supplements.push(`
      <strong>${game.i18n.localize("DND5E.FACILITY.Trade.Sell.Supplement")}</strong>
      ${formatNumber(trade.stock.value)}
      ${CONFIG.DND5E.currencies.gp?.abbreviation ?? ""}
    `);
    if ( trade?.creatures ) {
      const creatures = [];
      if ( trade.sell ) {
        for ( let i = 0; i < trade.creatures.sell.length; i++ ) {
          const sold = trade.creatures.sell[i];
          if ( sold ) creatures.push(await fromUuid(this.item.system.trade.creatures.value[i]));
        }
      }
      else creatures.push(...await Promise.all(trade.creatures.buy.filter(_ => _).map(fromUuid)));
      supplements.push(`
        <strong>${game.i18n.localize(`DND5E.FACILITY.Trade.${trade.sell ? "Sell" : "Buy"}.Supplement`)}</strong>
        ${game.i18n.getListFormatter({ style: "narrow" }).format(creatures.map(a => a.toAnchor().outerHTML))}
      `);
    }
    const facilityType = game.i18n.localize(`DND5E.FACILITY.Types.${type.value.titleCase()}.Label.one`);
    const buttons = this._usageChatButtons(message);
    return {
      supplements,
      buttons: buttons.length ? buttons : null,
      description: game.i18n.format("DND5E.FACILITY.Use.Description", {
        order: game.i18n.localize(`DND5E.FACILITY.Orders.${this.order}.inf`),
        link: this.item.toAnchor().outerHTML,
        facilityType: facilityType.toLocaleLowerCase(game.i18n.lang)
      })
    };
  }

  /* -------------------------------------------- */
  /*  Event Listeners & Handlers                  */
  /* -------------------------------------------- */

  /**
   * Handle deducting currency for the order.
   * @this {OrderActivity}
   * @param {PointerEvent} event     The triggering event.
   * @param {HTMLElement} target     The button that was clicked.
   * @param {ChatMessage5e} message  The message associated with the activation.
   * @returns {Promise<void>}
   */
  static async #onPayOrder(event, target, message) {
    const { method } = target.dataset;
    const order = message.getFlag("dnd5e", "order");
    const config = foundry.utils.expandObject({ "data.flags.dnd5e.order": order });
    if ( method === "automatic" ) {
      try {
        await CurrencyManager.deductActorCurrency(this.actor, order.costs.gold, "gp", {
          recursive: true,
          priority: "high"
        });
      } catch(err) {
        ui.notifications.error(err.message);
        return;
      }
    }
    foundry.utils.setProperty(config, "data.flags.dnd5e.order.costs.paid", true);
    const context = await this._usageChatContext(config);
    const content = await renderTemplate(this.metadata.usage.chatCard, context);
    await message.update({ content, flags: config.data.flags });
  }
}

/**
 * Sheet for the save activity.
 */
class SaveSheet extends ActivitySheet {

  /** @inheritDoc */
  static DEFAULT_OPTIONS = {
    classes: ["save-activity"]
  };

  /* -------------------------------------------- */

  /** @inheritDoc */
  static PARTS = {
    ...super.PARTS,
    effect: {
      template: "systems/dnd5e/templates/activity/save-effect.hbs",
      templates: [
        ...super.PARTS.effect.templates,
        "systems/dnd5e/templates/activity/parts/damage-part.hbs",
        "systems/dnd5e/templates/activity/parts/damage-parts.hbs",
        "systems/dnd5e/templates/activity/parts/save-damage.hbs",
        "systems/dnd5e/templates/activity/parts/save-details.hbs",
        "systems/dnd5e/templates/activity/parts/save-effect-settings.hbs"
      ]
    }
  };

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @override */
  _prepareAppliedEffectContext(context, effect) {
    effect.additionalSettings = "systems/dnd5e/templates/activity/parts/save-effect-settings.hbs";
    return effect;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareEffectContext(context) {
    context = await super._prepareEffectContext(context);

    context.abilityOptions = Object.entries(CONFIG.DND5E.abilities).map(([value, config]) => ({
      value, label: config.label
    }));
    context.calculationOptions = [
      { value: "", label: game.i18n.localize("DND5E.SAVE.FIELDS.save.dc.CustomFormula") },
      { rule: true },
      { value: "spellcasting", label: game.i18n.localize("DND5E.SpellAbility") },
      ...Object.entries(CONFIG.DND5E.abilities).map(([value, config]) => ({
        value, label: config.label, group: game.i18n.localize("DND5E.Abilities")
      }))
    ];
    context.onSaveOptions = [
      { value: "none", label: game.i18n.localize("DND5E.SAVE.FIELDS.damage.onSave.None") },
      { value: "half", label: game.i18n.localize("DND5E.SAVE.FIELDS.damage.onSave.Half") },
      { value: "full", label: game.i18n.localize("DND5E.SAVE.FIELDS.damage.onSave.Full") }
    ];

    return context;
  }
}

const { ArrayField: ArrayField$j, BooleanField: BooleanField$z, SchemaField: SchemaField$H, SetField: SetField$s, StringField: StringField$Y } = foundry.data.fields;

/**
 * @typedef {EffectApplicationData} SaveEffectApplicationData
 * @property {boolean} onSave  Should this effect still be applied on a successful save?
 */

/**
 * Data model for an save activity.
 *
 * @property {object} damage
 * @property {string} damage.onSave                 How much damage is done on a successful save?
 * @property {DamageData[]} damage.parts            Parts of damage to inflict.
 * @property {SaveEffectApplicationData[]} effects  Linked effects that can be applied.
 * @property {object} save
 * @property {Set<string>} save.ability             Make the saving throw with one of these abilities.
 * @property {object} save.dc
 * @property {string} save.dc.calculation           Method or ability used to calculate the difficulty class.
 * @property {string} save.dc.formula               Custom DC formula or flat value.
 */
class SaveActivityData extends BaseActivityData {
  /** @inheritDoc */
  static defineSchema() {
    return {
      ...super.defineSchema(),
      damage: new SchemaField$H({
        onSave: new StringField$Y(),
        parts: new ArrayField$j(new DamageField())
      }),
      effects: new ArrayField$j(new AppliedEffectField({
        onSave: new BooleanField$z()
      })),
      save: new SchemaField$H({
        ability: new SetField$s(new StringField$Y()),
        dc: new SchemaField$H({
          calculation: new StringField$Y({ initial: "initial" }),
          formula: new FormulaField({ deterministic: true })
        })
      })
    };
  }

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /** @override */
  get ability() {
    if ( this.save.dc.calculation in CONFIG.DND5E.abilities ) return this.save.dc.calculation;
    if ( this.save.dc.calculation === "spellcasting" ) return this.spellcastingAbility;
    return this.save.ability.first() ?? null;
  }

  /* -------------------------------------------- */
  /*  Data Migrations                             */
  /* -------------------------------------------- */

  /** @override */
  static migrateData(source) {
    if ( foundry.utils.getType(source.save?.ability) === "string" ) {
      if ( source.save.ability ) source.save.ability = [source.save.ability];
      else source.save.ability = [];
    }
  }

  /* -------------------------------------------- */

  /** @override */
  static transformTypeData(source, activityData, options) {
    let calculation = source.system.save?.scaling;
    if ( calculation === "flat" ) calculation = "";
    else if ( calculation === "spell" ) calculation = "spellcasting";

    return foundry.utils.mergeObject(activityData, {
      damage: {
        onSave: (source.type === "spell") && (source.system.level === 0) ? "none" : "half",
        parts: source.system.damage?.parts?.map(part => this.transformDamagePartData(source, part)) ?? []
      },
      save: {
        ability: [source.system.save?.ability || Object.keys(CONFIG.DND5E.abilities)[0]],
        dc: {
          calculation,
          formula: String(source.system.save?.dc ?? "")
        }
      }
    });
  }

  /* -------------------------------------------- */
  /*  Data Preparation                            */
  /* -------------------------------------------- */

  /** @inheritDoc */
  prepareData() {
    super.prepareData();
    if ( !this.damage.onSave ) this.damage.onSave = this.isSpell && (this.item.system.level === 0) ? "none" : "half";
    if ( this.save.dc.calculation === "initial" ) this.save.dc.calculation = this.isSpell ? "spellcasting" : "";
    this.save.dc.bonus = "";
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  prepareFinalData(rollData) {
    rollData ??= this.getRollData({ deterministic: true });
    super.prepareFinalData(rollData);
    this.prepareDamageLabel(this.damage.parts, rollData);

    const bonus = this.save.dc.bonus ? simplifyBonus(this.save.dc.bonus, rollData) : 0;

    let ability;
    if ( this.save.dc.calculation ) ability = this.ability;
    else this.save.dc.value = simplifyBonus(this.save.dc.formula, rollData);
    this.save.dc.value ??= this.actor?.system.abilities?.[ability]?.dc
      ?? 8 + (this.actor?.system.attributes?.prof ?? 0);
    this.save.dc.value += bonus;

    if ( this.save.dc.value ) this.labels.save = game.i18n.format("DND5E.SaveDC", {
      dc: this.save.dc.value,
      ability: CONFIG.DND5E.abilities[ability]?.label ?? ""
    });
  }

  /* -------------------------------------------- */
  /*  Helpers                                     */
  /* -------------------------------------------- */

  /** @inheritDoc */
  getDamageConfig(config={}) {
    const rollConfig = super.getDamageConfig(config);

    rollConfig.critical ??= {};
    rollConfig.critical.allow ??= false;

    return rollConfig;
  }
}

/**
 * Activity for making saving throws and rolling damage.
 */
class SaveActivity extends ActivityMixin(SaveActivityData) {
  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static LOCALIZATION_PREFIXES = [...super.LOCALIZATION_PREFIXES, "DND5E.SAVE"];

  /* -------------------------------------------- */

  /** @inheritDoc */
  static metadata = Object.freeze(
    foundry.utils.mergeObject(super.metadata, {
      type: "save",
      img: "systems/dnd5e/icons/svg/activity/save.svg",
      title: "DND5E.SAVE.Title.one",
      sheetClass: SaveSheet,
      usage: {
        actions: {
          rollDamage: SaveActivity.#rollDamage,
          rollSave: SaveActivity.#rollSave
        }
      }
    }, { inplace: false })
  );

  /* -------------------------------------------- */
  /*  Activation                                  */
  /* -------------------------------------------- */

  /** @override */
  _usageChatButtons(message) {
    const buttons = [];
    const dc = this.save.dc.value;

    for ( const abilityId of this.save.ability ) {
      const ability = CONFIG.DND5E.abilities[abilityId]?.label ?? "";
      buttons.push({
        label: `
          <span class="visible-dc">${game.i18n.format("DND5E.SavingThrowDC", { dc, ability })}</span>
          <span class="hidden-dc">${game.i18n.format("DND5E.SavePromptTitle", { ability })}</span>
        `,
        icon: '<i class="fa-solid fa-shield-heart" inert></i>',
        dataset: {
          dc,
          ability: abilityId,
          action: "rollSave",
          visibility: "all"
        }
      });
    }

    if ( this.damage.parts.length ) buttons.push({
      label: game.i18n.localize("DND5E.Damage"),
      icon: '<i class="fas fa-burst" inert></i>',
      dataset: {
        action: "rollDamage"
      }
    });
    return buttons.concat(super._usageChatButtons(message));
  }

  /* -------------------------------------------- */
  /*  Rolling                                     */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async rollDamage(config={}, dialog={}, message={}) {
    message = foundry.utils.mergeObject({
      "data.flags.dnd5e.roll": {
        damageOnSave: this.damage.onSave
      }
    }, message);
    return super.rollDamage(config, dialog, message);
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /**
   * Handle performing a damage roll.
   * @this {SaveActivity}
   * @param {PointerEvent} event     Triggering click event.
   * @param {HTMLElement} target     The capturing HTML element which defined a [data-action].
   * @param {ChatMessage5e} message  Message associated with the activation.
   */
  static #rollDamage(event, target, message) {
    this.rollDamage({ event });
  }

  /* -------------------------------------------- */

  /**
   * Handle performing a saving throw.
   * @this {SaveActivity}
   * @param {PointerEvent} event     Triggering click event.
   * @param {HTMLElement} target     The capturing HTML element which defined a [data-action].
   * @param {ChatMessage5e} message  Message associated with the activation.
   */
  static async #rollSave(event, target, message) {
    const targets = getSceneTargets();
    if ( !targets.length && game.user.character ) targets.push(game.user.character);
    if ( !targets.length ) ui.notifications.warn("DND5E.ActionWarningNoToken", { localize: true });
    const dc = parseInt(target.dataset.dc);
    for ( const token of targets ) {
      const actor = token instanceof Actor ? token : token.actor;
      const speaker = ChatMessage.getSpeaker({ actor, scene: canvas.scene, token: token.document });
      await actor.rollSavingThrow({
        event,
        ability: target.dataset.ability ?? this.save.ability.first(),
        target: Number.isFinite(dc) ? dc : this.save.dc.value
      }, {}, { data: { speaker } });
    }
  }

  /* -------------------------------------------- */
  /*  Helpers                                     */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async getFavoriteData() {
    return foundry.utils.mergeObject(await super.getFavoriteData(), { save: this.save });
  }
}

/**
 * Sheet for the summon activity.
 */
class SummonSheet extends ActivitySheet {

  /** @inheritDoc */
  static DEFAULT_OPTIONS = {
    classes: ["summon-activity"],
    actions: {
      addProfile: SummonSheet.#addProfile,
      deleteProfile: SummonSheet.#deleteProfile
    }
  };

  /* -------------------------------------------- */

  /** @inheritDoc */
  static PARTS = {
    ...super.PARTS,
    identity: {
      template: "systems/dnd5e/templates/activity/summon-identity.hbs",
      templates: super.PARTS.identity.templates
    },
    effect: {
      template: "systems/dnd5e/templates/activity/summon-effect.hbs",
      templates: [
        "systems/dnd5e/templates/activity/parts/activity-effects.hbs",
        "systems/dnd5e/templates/activity/parts/summon-changes.hbs",
        "systems/dnd5e/templates/activity/parts/summon-profiles.hbs"
      ]
    }
  };

  /* -------------------------------------------- */

  /** @inheritDoc */
  static CLEAN_ARRAYS = [...super.CLEAN_ARRAYS, "profiles"];

  /* -------------------------------------------- */

  /** @override */
  tabGroups = {
    sheet: "identity",
    activation: "time",
    effect: "profiles"
  };

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareEffectContext(context) {
    context = await super._prepareEffectContext(context);

    context.abilityOptions = [
      { value: "", label: this.activity.isSpell ? game.i18n.localize("DND5E.Spellcasting") : "" },
      { rule: true },
      ...Object.entries(CONFIG.DND5E.abilities).map(([value, { label }]) => ({ value, label }))
    ];
    context.creatureSizeOptions = Object.entries(CONFIG.DND5E.actorSizes).map(([value, config]) => ({
      value, label: config.label, selected: this.activity.creatureSizes.has(value)
    }));
    context.creatureTypeOptions = Object.entries(CONFIG.DND5E.creatureTypes).map(([value, config]) => ({
      value, label: config.label, selected: this.activity.creatureTypes.has(value)
    }));

    context.profileModes = [
      { value: "", label: game.i18n.localize("DND5E.SUMMON.FIELDS.summon.mode.Direct") },
      { value: "cr", label: game.i18n.localize("DND5E.SUMMON.FIELDS.summon.mode.CR") }
    ];
    context.profiles = this.activity.profiles.map((data, index) => ({
      data, index,
      collapsed: this.expandedSections.get(`profiles.${data._id}`) ? "" : "collapsed",
      fields: this.activity.schema.fields.profiles.element.fields,
      prefix: `profiles.${index}.`,
      source: context.source.profiles[index] ?? data,
      document: data.uuid ? fromUuidSync(data.uuid) : null,
      mode: this.activity.summon.mode,
      typeOptions: this.activity.summon.mode === "cr" ? context.creatureTypeOptions.map(t => ({
        ...t, selected: data.types.has(t.value)
      })) : null
    })).sort((lhs, rhs) =>
      (lhs.name || lhs.document?.name || "").localeCompare(rhs.name || rhs.document?.name || "", game.i18n.lang)
    );

    return context;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _getTabs() {
    const tabs = super._getTabs();
    tabs.effect.label = "DND5E.SUMMON.SECTIONS.Summoning";
    tabs.effect.icon = "fa-solid fa-spaghetti-monster-flying";
    tabs.effect.tabs = this._markTabs({
      profiles: {
        id: "profiles", group: "effect", icon: "fa-solid fa-address-card",
        label: "DND5E.SUMMON.SECTIONS.Profiles"
      },
      changes: {
        id: "changes", group: "effect", icon: "fa-solid fa-sliders",
        label: "DND5E.SUMMON.SECTIONS.Changes"
      }
    });
    return tabs;
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /** @inheritDoc */
  _onRender() {
    super._onRender();
    this.element.querySelector(".activity-profiles").addEventListener("drop", this.#onDrop.bind(this));
  }

  /* -------------------------------------------- */

  /**
   * Handle adding a new entry to the summoning profiles list.
   * @this {ActivityConfig}
   * @param {Event} event         Triggering click event.
   * @param {HTMLElement} target  Button that was clicked.
   */
  static #addProfile(event, target) {
    this.activity.update({ profiles: [...this.activity.toObject().profiles, {}] });
  }

  /* -------------------------------------------- */

  /**
   * Handle removing an entry from the summoning profiles list.
   * @this {ActivityConfig}
   * @param {Event} event         Triggering click event.
   * @param {HTMLElement} target  Button that was clicked.
   */
  static #deleteProfile(event, target) {
    const profiles = this.activity.toObject().profiles;
    profiles.splice(target.closest("[data-index]").dataset.index, 1);
    this.activity.update({ profiles });
  }

  /* -------------------------------------------- */
  /*  Drag & Drop                                 */
  /* -------------------------------------------- */

  /**
   * Handle dropping actors onto the sheet.
   * @param {Event} event  Triggering drop event.
   */
  async #onDrop(event) {
    // Try to extract the data
    const data = TextEditor.getDragEventData(event);

    // Handle dropping linked items
    if ( data?.type !== "Actor" ) return;
    const actor = await Actor.implementation.fromDropData(data);

    // If dropped onto existing profile, add or replace link
    const profileId = event.target.closest("[data-profile-id]")?.dataset.profileId;
    if ( profileId ) {
      const profiles = this.activity.toObject().profiles;
      const profile = profiles.find(p => p._id === profileId);
      profile.uuid = actor.uuid;
      this.activity.update({ profiles });
    }

    // Otherwise create a new profile
    else this.activity.update({ profiles: [...this.activity.toObject().profiles, { uuid: actor.uuid }] });
  }
}

const { BooleanField: BooleanField$y, StringField: StringField$X } = foundry.data.fields;

/**
 * Dialog for configuring the usage of the summon activity.
 */
class SummonUsageDialog extends ActivityUsageDialog {

  /** @inheritDoc */
  static PARTS = {
    ...super.PARTS,
    creation: {
      template: "systems/dnd5e/templates/activity/summon-usage-creation.hbs"
    }
  };

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareCreationContext(context, options) {
    context = await super._prepareCreationContext(context, options);

    const profiles = this.activity.availableProfiles;
    if ( this._shouldDisplay("create.summons") && (profiles.length || (this.activity.creatureSizes.size > 1)
      || (this.activity.creatureTypes.size > 1)) ) {
      context.hasCreation = true;
      context.summonsFields = [];

      if ( !foundry.utils.hasProperty(this.options.display, "create.summons") ) context.summonsFields.push({
        field: new BooleanField$y({ label: game.i18n.localize("DND5E.SUMMON.Action.Place") }),
        name: "create.summons",
        value: this.config.create?.summons,
        input: context.inputs.createCheckboxInput
      });

      if ( this.config.create?.summons ) {
        const rollData = this.activity.getRollData();
        if ( profiles.length > 1 ) {
          let options = profiles.map(profile => ({
            value: profile._id, label: this.getProfileLabel(profile, rollData)
          }));
          if ( options.every(o => o.label.startsWith("1 × ")) ) {
            options = options.map(({ value, label }) => ({ value, label: label.replace("1 × ", "") }));
          }
          context.summonsFields.push({
            field: new StringField$X({ label: game.i18n.localize("DND5E.SUMMON.Profile.Label") }),
            name: "summons.profile",
            value: this.config.summons?.profile,
            options
          });
        } else context.summonsProfile = profiles[0]._id;

        if ( this.activity.creatureSizes.size > 1 ) context.summonsFields.push({
          field: new StringField$X({ label: game.i18n.localize("DND5E.Size") }),
          name: "summons.creatureSize",
          value: this.config.summons?.creatureSize,
          options: Array.from(this.activity.creatureSizes)
            .map(value => ({ value, label: CONFIG.DND5E.actorSizes[value]?.label }))
            .filter(k => k)
        });

        if ( this.activity.creatureTypes.size > 1 ) context.summonsFields.push({
          field: new StringField$X({ label: game.i18n.localize("DND5E.CreatureType") }),
          name: "summons.creatureType",
          value: this.config.summons?.creatureType,
          options: Array.from(this.activity.creatureTypes)
            .map(value => ({ value, label: CONFIG.DND5E.creatureTypes[value]?.label }))
            .filter(k => k)
        });
      }
    }

    return context;
  }

  /* -------------------------------------------- */

  /**
   * Determine the label for a profile in the ability use dialog.
   * @param {SummonsProfile} profile  Profile for which to generate the label.
   * @param {object} rollData         Roll data used to prepare the count.
   * @returns {string}
   */
  getProfileLabel(profile, rollData) {
    let label;
    if ( profile.name ) label = profile.name;
    else {
      switch ( this.activity.summon.mode ) {
        case "cr":
          const cr = simplifyBonus(profile.cr, rollData);
          label = game.i18n.format("DND5E.SUMMON.Profile.ChallengeRatingLabel", { cr: formatCR(cr) });
          break;
        default:
          const doc = fromUuidSync(profile.uuid);
          if ( doc ) label = doc.name;
          break;
      }
    }
    label ??= "—";

    let count = simplifyRollFormula(Roll.replaceFormulaData(profile.count ?? "1", rollData));
    if ( Number.isNumeric(count) ) count = parseInt(count);
    if ( count ) label = `${count} × ${label}`;

    return label;
  }
}

/**
 * A filter description.
 *
 * @typedef {object} FilterDescription
 * @property {string} k        Key on the data object to check.
 * @property {any} v           Value to compare.
 * @property {string} [o="_"]  Operator or comparison function to use.
 */

/**
 * Check some data against a filter to determine if it matches.
 * @param {object} data                                   Data to check.
 * @param {FilterDescription|FilterDescription[]} filter  Filter to compare against.
 * @returns {boolean}
 * @throws
 */
function performCheck(data, filter=[]) {
  if ( foundry.utils.getType(filter) === "Array" ) return AND(data, filter);
  return _check(data, filter.k, filter.v, filter.o);
}

/* -------------------------------------------- */

/**
 * Determine the unique keys referenced by a set of filters.
 * @param {FilterDescription[]} filter  Filter to examine.
 * @returns {Set<string>}
 */
function uniqueKeys(filter=[]) {
  const keys = new Set();
  const _uniqueKeys = filters => {
    for ( const f of filters ) {
      const operator = f.o in OPERATOR_FUNCTIONS;
      if ( operator && (foundry.utils.getType(f.v) === "Array") ) _uniqueKeys(f.v);
      else if ( f.o === "NOT" ) _uniqueKeys([f.v]);
      else if ( !operator ) keys.add(f.k);
    }
  };
  _uniqueKeys(filter);
  return keys;
}

/* -------------------------------------------- */

/**
 * Internal check implementation.
 * @param {object} data             Data to check.
 * @param {string} [keyPath]        Path to individual piece within data to check.
 * @param {*} value                 Value to compare against or additional filters.
 * @param {string} [operation="_"]  Checking function to use.
 * @returns {boolean}
 * @internal
 * @throws
 */
function _check(data, keyPath, value, operation="_") {
  const operator = OPERATOR_FUNCTIONS[operation];
  if ( operator ) return operator(data, value);

  const comparison = COMPARISON_FUNCTIONS[operation];
  if ( !comparison ) throw new Error(`Comparison function "${operation}" could not be found.`);
  return comparison(foundry.utils.getProperty(data, keyPath), value);
}

/* -------------------------------------------- */
/*  Operator Functions                          */
/* -------------------------------------------- */

/**
 * Operator functions.
 * @enum {Function}
 */
const OPERATOR_FUNCTIONS = {
  AND, NAND, OR, NOR, XOR, NOT
};

/* -------------------------------------------- */

/**
 * Perform an AND check against all filters.
 * @param {object} data                 Data to check.
 * @param {FilterDescription[]} filter  Filter to compare against.
 * @returns {boolean}
 */
function AND(data, filter) {
  return filter.every(({k, v, o}) => _check(data, k, v, o));
}

/* -------------------------------------------- */

/**
 * Perform an NAND check against all filters.
 * @param {object} data                 Data to check.
 * @param {FilterDescription[]} filter  Filter to compare against.
 * @returns {boolean}
 */
function NAND(data, filter) {
  return !filter.every(({k, v, o}) => _check(data, k, v, o));
}

/* -------------------------------------------- */

/**
 * Perform an OR check against all filters.
 * @param {object} data                 Data to check.
 * @param {FilterDescription[]} filter  Filter to compare against.
 * @returns {boolean}
 */
function OR(data, filter) {
  return filter.some(({k, v, o}) => _check(data, k, v, o));
}

/* -------------------------------------------- */

/**
 * Perform an NOR check against all filters.
 * @param {object} data                 Data to check.
 * @param {FilterDescription[]} filter  Filter to compare against.
 * @returns {boolean}
 */
function NOR(data, filter) {
  return !filter.some(({k, v, o}) => _check(data, k, v, o));
}

/* -------------------------------------------- */

/**
 * Perform an XOR check against all filters.
 * @param {object} data                 Data to check.
 * @param {FilterDescription[]} filter  Filter to compare against.
 * @returns {boolean}
 */
function XOR(data, filter) {
  if ( !filter.length ) return false;
  let currentResult = _check(data, filter[0].k, filter[0].v, filter[0].o);
  for ( let i = 1; i < filter.length; i++ ) {
    const { k, v, o } = filter[i];
    currentResult ^= _check(data, k, v, o);
  }
  return Boolean(currentResult);
}

/* -------------------------------------------- */

/**
 * Invert the result of a nested check,
 * @param {object} data               Data to check.
 * @param {FilterDescription} filter  Filter to compare against.
 * @returns {boolean}
 */
function NOT(data, filter) {
  const { k, v, o } = filter;
  return !_check(data, k, v, o);
}

/* -------------------------------------------- */
/*  Comparison Functions                        */
/* -------------------------------------------- */

/**
 * Currently supported comparison functions.
 * @enum {Function}
 */
const COMPARISON_FUNCTIONS = {
  _: exact, exact, contains, icontains, startswith, istartswith, endswith,
  has, hasany, hasall, in: in_, gt, gte, lt, lte
};

/* -------------------------------------------- */

/**
 * Check for an exact match. The default comparison mode if none is provided.
 * @param {*} data
 * @param {*} value
 * @returns {boolean}
 */
function exact(data, value) {
  return data === value;
}

/* -------------------------------------------- */

/**
 * Check that data contains value.
 * @param {*} data
 * @param {*} value
 * @returns {boolean}
 */
function contains(data, value) {
  return String(data).includes(String(value));
}

/* -------------------------------------------- */

/**
 * Case-insensitive check that data contains value.
 * @param {*} data
 * @param {*} value
 * @returns {boolean}
 */
function icontains(data, value) {
  return contains(String(data).toLocaleLowerCase(game.i18n.lang), String(value).toLocaleLowerCase(game.i18n.lang));
}

/* -------------------------------------------- */

/**
 * Check that data starts with value.
 * @param {*} data
 * @param {*} value
 * @returns {boolean}
 */
function startswith(data, value) {
  return String(data).startsWith(String(value));
}

/* -------------------------------------------- */

/**
 * Case-insensitive check that data starts with value.
 * @param {*} data
 * @param {*} value
 * @returns {boolean}
 */
function istartswith(data, value) {
  return startswith(String(data).toLocaleLowerCase(game.i18n.lang), String(value).toLocaleLowerCase(game.i18n.lang));
}

/* -------------------------------------------- */

/**
 * Check that data ends with value.
 * @param {*} data
 * @param {*} value
 * @returns {boolean}
 */
function endswith(data, value) {
  return String(data).endsWith(String(value));
}

/* -------------------------------------------- */

/**
 * Check that the data collection has the provided value.
 * @param {*} data
 * @param {*} value
 * @returns {boolean}
 */
function has(data, value) {
  // If the value is another filter description, apply that check against each member of the collection
  if ( foundry.utils.getType(value) === "Object" ) {
    switch ( foundry.utils.getType(data) ) {
      case "Array":
      case "Set": return !!data.find(d => performCheck(d, value));
      default: return false;
    }
  } else return in_(value, data);
}

/* -------------------------------------------- */

/**
 * Check that the data collection has any of the provided values.
 * @param {*} data
 * @param {*} value
 * @returns {boolean}
 */
function hasany(data, value) {
  return Array.from(value).some(v => has(data, v));
}

/* -------------------------------------------- */

/**
 * Check that the data collection has all of the provided values.
 * @param {*} data
 * @param {*} value
 * @returns {boolean}
 */
function hasall(data, value) {
  return Array.from(value).every(v => has(data, v));
}

/* -------------------------------------------- */

/**
 * Check that data matches one of the provided values.
 * @param {*} data
 * @param {*} value
 * @returns {boolean}
 */
function in_(data, value) {
  switch ( foundry.utils.getType(value) ) {
    case "Array": return value.includes(data);
    case "Set": return value.has(data);
    default: return false;
  }
}

/* -------------------------------------------- */

/**
 * Check that value is greater than data.
 * @param {*} data
 * @param {*} value
 * @returns {boolean}
 */
function gt(data, value) {
  return data > value;
}

/* -------------------------------------------- */

/**
 * Check that value is greater than or equal to data.
 * @param {*} data
 * @param {*} value
 * @returns {boolean}
 */
function gte(data, value) {
  return data >= value;
}

/* -------------------------------------------- */

/**
 * Check that value is less than data.
 * @param {*} data
 * @param {*} value
 * @returns {boolean}
 */
function lt(data, value) {
  return data < value;
}

/* -------------------------------------------- */

/**
 * Check that value is less than or equal to data.
 * @param {*} data
 * @param {*} value
 * @returns {boolean}
 */
function lte(data, value) {
  return data <= value;
}

var Filter = /*#__PURE__*/Object.freeze({
  __proto__: null,
  AND: AND,
  COMPARISON_FUNCTIONS: COMPARISON_FUNCTIONS,
  NAND: NAND,
  NOR: NOR,
  NOT: NOT,
  OPERATOR_FUNCTIONS: OPERATOR_FUNCTIONS,
  OR: OR,
  XOR: XOR,
  contains: contains,
  endswith: endswith,
  exact: exact,
  gt: gt,
  gte: gte,
  has: has,
  hasall: hasall,
  hasany: hasany,
  icontains: icontains,
  in_: in_,
  istartswith: istartswith,
  lt: lt,
  lte: lte,
  performCheck: performCheck,
  startswith: startswith,
  uniqueKeys: uniqueKeys
});

const { NumberField: NumberField$C, SchemaField: SchemaField$G, StringField: StringField$W } = foundry.data.fields;

/**
 * @typedef {object} SourceData
 * @property {string} book      Book/publication where the item originated.
 * @property {string} page      Page or section where the item can be found.
 * @property {string} custom    Fully custom source label.
 * @property {string} license   Type of license that covers this item.
 * @property {number} revision  Revision count for this item.
 * @property {string} rules     Version of the rules for this document (e.g. 2014 vs. 2024).
 */

/**
 * Data fields that stores information on the adventure or sourcebook where this document originated.
 */
class SourceField extends SchemaField$G {
  constructor(fields={}, options={}) {
    fields = {
      book: new StringField$W(),
      page: new StringField$W(),
      custom: new StringField$W(),
      license: new StringField$W(),
      revision: new NumberField$C({ initial: 1 }),
      rules: new StringField$W({ initial: "2024" }),
      ...fields
    };
    Object.entries(fields).forEach(([k, v]) => !v ? delete fields[k] : null);
    super(fields, { label: "DND5E.SOURCE.FIELDS.source.label", ...options });
  }

  /* -------------------------------------------- */
  /*  Data Preparation                            */
  /* -------------------------------------------- */

  /**
   * Prepare the source label.
   * @this {SourceData}
   * @param {string} uuid  Compendium source or document UUID.
   */
  static prepareData(uuid) {
    const pkg = SourceField.getPackage(uuid);
    this.bookPlaceholder = SourceField.getModuleBook(pkg) ?? "";
    if ( !this.book ) this.book = this.bookPlaceholder;

    if ( this.custom ) this.label = this.custom;
    else {
      const page = Number.isNumeric(this.page)
        ? game.i18n.format("DND5E.SOURCE.Display.Page", { page: this.page }) : (this.page ?? "");
      this.label = game.i18n.format("DND5E.SOURCE.Display.Full", { book: this.book, page }).trim();
    }

    this.value = this.book || (pkg?.title ?? "");
    this.slug = this.value.slugify({ strict: true });

    Object.defineProperty(this, "directlyEditable", {
      value: (this.custom ?? "") === this.label,
      configurable: true,
      enumerable: false
    });
  }

  /* -------------------------------------------- */

  /**
   * Check if the provided package has any source books registered in its manifest. If it has only one, then return
   * that book's key.
   * @param {ClientPackage} pkg  The package.
   * @returns {string|null}
   */
  static getModuleBook(pkg) {
    if ( !pkg ) return null;
    const sourceBooks = pkg.flags?.dnd5e?.sourceBooks;
    const keys = Object.keys(sourceBooks ?? {});
    if ( keys.length !== 1 ) return null;
    return keys[0];
  }

  /* -------------------------------------------- */

  /**
   * Get the package associated with the given UUID, if any.
   * @param {string} uuid  The UUID.
   * @returns {ClientPackage|null}
   */
  static getPackage(uuid) {
    if ( !uuid ) return null;
    const pack = foundry.utils.parseUuid(uuid)?.collection?.metadata;
    switch ( pack?.packageType ) {
      case "module": return game.modules.get(pack.packageName);
      case "system": return game.system;
      case "world": return game.world;
    }
    return null;
  }

  /* -------------------------------------------- */
  /*  Shims                                       */
  /* -------------------------------------------- */

  /**
   * Add a shim for the old source path.
   * @this {ActorDataModel}
   */
  static shimActor() {
    const source = this.source;
    Object.defineProperty(this.details, "source", {
      get() {
        foundry.utils.logCompatibilityWarning(
          "The source data for actors has been moved to `system.source`.",
          { since: "DnD5e 4.0", until: "DnD5e 4.4" }
        );
        return source;
      }
    });
  }
}

/**
 * @typedef {ApplicationConfiguration} CompendiumBrowserSourceConfiguration
 * @property {string} [selected]  The initially-selected package.
 */

/**
 * @typedef CompendiumSourceConfig5e
 * @property {object} packages
 * @property {CompendiumSourcePackageConfig5e} packages.world
 * @property {CompendiumSourcePackageConfig5e} packages.system
 * @property {Record<string, CompendiumSourcePackageConfig5e>} packages.modules
 * @property {object} packs
 * @property {CompendiumSourcePackGroup5e} packs.items
 * @property {CompendiumSourcePackGroup5e} packs.actors
 */

/**
 * @typedef CompendiumSourcePackageConfig5e
 * @property {string} title           The package title.
 * @property {string} id              The package ID.
 * @property {number} count           The number of packs provided by this package.
 * @property {boolean} checked        True if all the packs are included.
 * @property {boolean} indeterminate  True if only some of the packs are included.
 * @property {boolean} active         True if the package is currently selected.
 * @property {string} filter          The normalized package title for filtering.
 */

/**
 * @typedef CompendiumSourcePackGroup5e
 * @property {boolean} checked        True if all members of this pack group are included.
 * @property {boolean} indeterminate  True if only some of this pack group are included.
 * @property {CompendiumSourcePackConfig5e[]} entries
 */

/**
 * @typedef CompendiumSourcePackConfig5e
 * @property {string} title     The pack title.
 * @property {string} id        The pack ID.
 * @property {boolean} checked  True if the pack is included.
 */

/**
 * An application for configuring which compendium packs contribute their content to the compendium browser.
 * @extends Application5e<CompendiumBrowserSourceConfiguration>
 */
class CompendiumBrowserSettingsConfig extends Application5e {
  constructor(options) {
    super(options);
    this.#selected = this.options.selected;
  }

  /* -------------------------------------------- */

  /** @override */
  static DEFAULT_OPTIONS = {
    id: "compendium-browser-source-config",
    classes: ["dialog-lg"],
    tag: "form",
    window: {
      title: "DND5E.CompendiumBrowser.Sources.Label",
      icon: "fas fa-book-open-reader",
      resizable: true
    },
    position: {
      width: 800,
      height: 650
    },
    actions: {
      clearFilter: CompendiumBrowserSettingsConfig.#onClearPackageFilter,
      selectPackage: CompendiumBrowserSettingsConfig.#onSelectPackage
    },
    selected: "system"
  };

  /* -------------------------------------------- */

  /** @override */
  static PARTS = {
    sidebar: {
      id: "sidebar",
      template: "systems/dnd5e/templates/compendium/sources-sidebar.hbs"
    },
    packs: {
      id: "packs",
      template: "systems/dnd5e/templates/compendium/sources-packs.hbs"
    }
  };

  /* -------------------------------------------- */

  /**
   * The number of milliseconds to delay between user keypresses before executing the package filter.
   * @type {number}
   */
  static FILTER_DELAY = 200;

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * The current package filter.
   * @type {string}
   */
  #filter = "";

  /* -------------------------------------------- */

  /**
   * The currently selected package.
   * @type {string}
   */
  #selected;

  /* -------------------------------------------- */

  _debouncedFilter = foundry.utils.debounce(this._onFilterPackages.bind(this), this.constructor.FILTER_DELAY);

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareContext(options) {
    const sources = this.constructor.collateSources();
    const byPackage = { world: new Set(), system: new Set() };

    for ( const { collection, documentName, metadata } of game.packs ) {
      if ( (documentName !== "Actor") && (documentName !== "Item") ) continue;
      let entry;
      if ( (metadata.packageType === "world") || (metadata.packageType === "system") ) {
        entry = byPackage[metadata.packageType];
      }
      else entry = byPackage[`module.${metadata.packageName}`] ??= new Set();
      entry.add(collection);
    }

    const packages = { };
    packages.world = this._preparePackageContext("world", game.world, byPackage.world, sources);
    packages.system = this._preparePackageContext("system", game.system, byPackage.system, sources);

    const modules = Object.entries(byPackage).reduce((arr, [k, packs]) => {
      if ( (k === "world") || (k === "system") ) return arr;
      const id = k.slice(7);
      const module = game.modules.get(id);
      arr.push(this._preparePackageContext(k, module, packs, sources));
      return arr;
    }, []);
    modules.sort((a, b) => a.title.localeCompare(b.title, game.i18n.lang));
    packages.modules = Object.fromEntries(modules.map(m => [m.id, m]));

    const packs = { actors: {}, items: {} };
    [["actors", "Actor"], ["items", "Item"]].forEach(([p, type]) => {
      packs[p] = this._preparePackGroupContext(type, byPackage[this.#selected], sources);
    });

    return {
      ...await super._prepareContext(options),
      packages, packs,
      filter: this.#filter
    };
  }

  /* -------------------------------------------- */

  /**
   * Prepare render context for packages.
   * @param {string} id            The package identifier.
   * @param {ClientPackage} pkg    The package.
   * @param {Set<string>} packs    The packs belonging to this package.
   * @param {Set<string>} sources  The packs currently selected for inclusion.
   * @returns {CompendiumSourcePackageConfig5e}
   * @protected
   */
  _preparePackageContext(id, pkg, packs, sources) {
    const { title } = pkg;
    const all = packs.isSubsetOf(sources);
    const indeterminate = !all && packs.intersects(sources);
    return {
      id, title, indeterminate,
      checked: indeterminate || all,
      count: packs.size,
      active: this.#selected === id,
      filter: title.replace(/[^\p{L} ]/gu, "").toLocaleLowerCase(game.i18n.lang)
    };
  }

  /* -------------------------------------------- */

  /**
   * Prepare render context for pack groups.
   * @param {string} documentType    The pack group's Document type.
   * @param {Set<string>} packs      The packs provided by the selected package.
   * @param {Set<string>} sources    The packs currently selected for inclusion.
   * @returns {CompendiumSourcePackGroup5e}
   * @protected
   */
  _preparePackGroupContext(documentType, packs, sources) {
    packs = packs.filter(id => {
      const pack = game.packs.get(id);
      return pack.documentName === documentType;
    });
    const all = packs.isSubsetOf(sources);
    const indeterminate = !all && packs.intersects(sources);
    return {
      indeterminate,
      checked: indeterminate || all,
      entries: Array.from(packs.map(id => {
        const { collection, title } = game.packs.get(id);
        return {
          title,
          id: collection,
          checked: sources.has(id)
        };
      })).sort((a, b) => a.title.localeCompare(b.title, game.i18n.lang))
    };
  }

  /* -------------------------------------------- */
  /*  Event Listeners & Handlers                  */
  /* -------------------------------------------- */

  /** @inheritDoc */
  _attachFrameListeners() {
    super._attachFrameListeners();
    this.element.addEventListener("keydown", this._debouncedFilter, { passive: true });
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _attachPartListeners(partId, htmlElement, options) {
    super._attachPartListeners(partId, htmlElement, options);
    if ( partId === "sidebar" ) this._filterPackages();
  }

  /* -------------------------------------------- */

  /**
   * Execute the package list filter.
   * @protected
   */
  _filterPackages() {
    const query = this.#filter.replace(/[^\p{L} ]/gu, "").toLocaleLowerCase(game.i18n.lang);
    this.element.querySelectorAll(".package-list.modules > li").forEach(item => {
      item.toggleAttribute("hidden", query && !item.dataset.filter.includes(query));
    });
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _onChangeForm(formConfig, event) {
    super._onChangeForm(formConfig, event);
    if ( event.target.dataset.type ) this._onToggleSource(event.target);
  }

  /* -------------------------------------------- */

  /**
   * Handle filtering the package sidebar.
   * @param {KeyboardEvent} event  The triggering event.
   * @protected
   */
  _onFilterPackages(event) {
    if ( !event.target.matches("search > input") ) return;
    this.#filter = event.target.value;
    this._filterPackages();
  }

  /* -------------------------------------------- */

  /**
   * Handle toggling a compendium browser source pack.
   * @param {CheckboxElement} target  The element that was toggled.
   * @returns {Record<string, boolean>}
   * @protected
   */
  _onTogglePack(target) {
    const packs = {};
    const { name, checked, indeterminate } = target;
    if ( (name === "all-items") || (name === "all-actors") ) {
      const [, documentType] = name.split("-");
      const pkg = this.#selected === "world"
        ? game.world
        : this.#selected === "system"
          ? game.system
          : game.modules.get(this.#selected.slice(7));
      for ( const { id, type } of pkg.packs ) {
        if ( game[documentType].documentName === type ) packs[id] = indeterminate ? false : checked;
      }
    }
    else packs[name] = checked;
    return packs;
  }

  /* -------------------------------------------- */

  /**
   * Handle toggling a compendium browser source package.
   * @param {CheckboxElement} target  The element that was toggled.
   * @returns {Record<string, boolean>}
   * @protected
   */
  _onTogglePackage(target) {
    const packs = {};
    const { name, checked, indeterminate } = target;
    const pkg = name === "world" ? game.world : name === "system" ? game.system : game.modules.get(name.slice(7));
    for ( const { id } of pkg.packs ) packs[id] = indeterminate ? false : checked;
    return packs;
  }

  /* -------------------------------------------- */

  /**
   * Toggle a compendium browser source.
   * @param {CheckboxElement} target  The element that was toggled.
   * @protected
   */
  async _onToggleSource(target) {
    let packs;
    switch ( target.dataset.type ) {
      case "pack": packs = this._onTogglePack(target); break;
      case "package": packs = this._onTogglePackage(target); break;
      default: return;
    }
    const setting = { ...game.settings.get("dnd5e", "packSourceConfiguration"), ...packs };
    await game.settings.set("dnd5e", "packSourceConfiguration", setting);
    this.render();
  }

  /* -------------------------------------------- */

  /**
   * Handle clearing the package filter.
   * @this {CompendiumBrowserSettingsConfig}
   * @param {PointerEvent} event  The originating click event.
   * @param {HTMLElement} target  The target of the click event.
   */
  static #onClearPackageFilter(event, target) {
    const input = target.closest("search").querySelector(":scope > input");
    input.value = this.#filter = "";
    this._filterPackages();
  }

  /* -------------------------------------------- */

  /**
   * Handle selecting a package.
   * @this {CompendiumBrowserSettingsConfig}
   * @param {PointerEvent} event  The originating click event.
   * @param {HTMLElement} target  The target of the click event.
   */
  static #onSelectPackage(event, target) {
    const { packageId } = target.closest("[data-package-id]")?.dataset ?? {};
    if ( !packageId ) return;
    this.#selected = packageId;
    this.render();
  }

  /* -------------------------------------------- */
  /*  Helpers                                     */
  /* -------------------------------------------- */

  /**
   * Collate sources for inclusion in the compendium browser.
   * @returns {Set<string>}  The set of packs that should be included in the compendium browser.
   */
  static collateSources() {
    const sources = new Set();
    const setting = game.settings.get("dnd5e", "packSourceConfiguration");
    for ( const { collection, documentName } of game.packs ) {
      if ( (documentName !== "Actor") && (documentName !== "Item") ) continue;
      if ( setting[collection] !== false ) sources.add(collection);
    }
    return sources;
  }
}

/**
 * @typedef {ApplicationConfiguration} CompendiumBrowserConfiguration
 * @property {{locked: CompendiumBrowserFilters, initial: CompendiumBrowserFilters}} filters  Filters to set to start.
 *                                              Locked filters won't be able to be changed by the user. Initial filters
 *                                              will be set to start but can be changed.
 * @property {CompendiumBrowserSelectionConfiguration} selection  Configuration used to define document selections.
 */

/**
 * @typedef {object} CompendiumBrowserSelectionConfiguration
 * @property {number|null} min                  Minimum number of documents that must be selected.
 * @property {number|null} max                  Maximum number of documents that must be selected.
 */

/**
 * @typedef {object} CompendiumBrowserFilters
 * @property {string} [documentClass]  Document type to fetch (e.g. Actor or Item).
 * @property {Set<string>} [types]     Individual document subtypes to filter upon (e.g. "loot", "class", "npc").
 * @property {object} [additional]     Additional type-specific filters applied.
 * @property {FilterDescription[]} [arbitrary]  Additional arbitrary filters to apply, not displayed in the UI.
 *                                     Only available as part of locked filters.
 * @property {string} [name]           A substring to filter by Document name.
 */

/**
 * Filter definition object for additional filters in the Compendium Browser.
 *
 * @typedef {object} CompendiumBrowserFilterDefinitionEntry
 * @property {string} label                                   Localizable label for the filter.
 * @property {"boolean"|"range"|"set"} type                   Type of filter control to display.
 * @property {object} config                                  Type-specific configuration data.
 * @property {CompendiumBrowserCreateFilters} [createFilter]  Method that can be called to create filters.
 */

/**
 * @callback CompendiumBrowserFilterCreateFilters
 * @param {FilterDescription[]} filters                        Array of filters to be applied that should be mutated.
 * @param {*} value                                            Value of the filter.
 * @param {CompendiumBrowserFilterDefinitionEntry} definition  Definition for this filter.
 */

/**
 * @typedef {Map<string, CompendiumBrowserFilterDefinitionEntry>} CompendiumBrowserFilterDefinition
 */

/**
 * Application for browsing, filtering, and searching for content between multiple compendiums.
 * @extends Application5e
 * @template CompendiumBrowserConfiguration
 */
class CompendiumBrowser extends Application5e {
  constructor(...args) {
    super(...args);

    this.#filters = this.options.filters?.initial ?? {};

    if ( "mode" in this.options ) {
      this._mode = this.options.mode;
      this._applyModeFilters(this.options.mode);
    }

    const isAdvanced = this._mode === this.constructor.MODES.ADVANCED;
    const tab = this.constructor.TABS.find(t => t.tab === this.options.tab);
    if ( !tab || (!!tab.advanced !== isAdvanced) ) this.options.tab = isAdvanced ? "actors" : "classes";
    this._applyTabFilters(this.options.tab);
  }

  /* -------------------------------------------- */

  /** @override */
  static DEFAULT_OPTIONS = {
    id: "compendium-browser-{id}",
    classes: ["compendium-browser", "vertical-tabs"],
    tag: "form",
    window: {
      title: "DND5E.CompendiumBrowser.Title",
      minimizable: true,
      resizable: true
    },
    actions: {
      configureSources: CompendiumBrowser.#onConfigureSources,
      clearName: CompendiumBrowser.#onClearName,
      openLink: CompendiumBrowser.#onOpenLink,
      setFilter: CompendiumBrowser.#onSetFilter,
      setType: CompendiumBrowser.#onSetType,
      toggleCollapse: CompendiumBrowser.#onToggleCollapse,
      toggleMode: CompendiumBrowser.#onToggleMode
    },
    form: {
      handler: CompendiumBrowser.#onHandleSubmit,
      closeOnSubmit: true
    },
    position: {
      width: 850,
      height: 700
    },
    filters: {
      locked: {},
      initial: {
        documentClass: "Item",
        types: new Set(["class"])
      }
    },
    selection: {
      min: null,
      max: null
    },
    tab: "classes"
  };

  /* -------------------------------------------- */

  /** @override */
  static PARTS = {
    header: {
      id: "header",
      classes: ["header"],
      template: "systems/dnd5e/templates/compendium/browser-header.hbs"
    },
    search: {
      id: "sidebar-search",
      classes: ["filter-element"],
      container: { id: "sidebar", classes: ["sidebar", "flexcol"] },
      template: "systems/dnd5e/templates/compendium/browser-sidebar-search.hbs"
    },
    types: {
      id: "sidebar-types",
      container: { id: "sidebar", classes: ["sidebar", "flexcol"] },
      template: "systems/dnd5e/templates/compendium/browser-sidebar-types.hbs"
    },
    filters: {
      id: "sidebar-filters",
      container: { id: "sidebar", classes: ["sidebar", "flexcol"] },
      template: "systems/dnd5e/templates/compendium/browser-sidebar-filters.hbs",
      templates: ["systems/dnd5e/templates/compendium/browser-sidebar-filter-set.hbs"]
    },
    results: {
      id: "results",
      classes: ["results"],
      template: "systems/dnd5e/templates/compendium/browser-results.hbs",
      templates: ["systems/dnd5e/templates/compendium/browser-entry.hbs"],
      scrollable: [""]
    },
    footer: {
      id: "footer",
      classes: ["footer"],
      template: "systems/dnd5e/templates/compendium/browser-footer.hbs"
    },
    tabs: {
      id: "tabs",
      classes: ["tabs", "tabs-left"],
      template: "systems/dnd5e/templates/compendium/browser-tabs.hbs"
    }
  };

  /* -------------------------------------------- */

  /**
   * @typedef {SheetTabDescriptor5e} CompendiumBrowserTabDescriptor5e
   * @property {string} documentClass  The class of Documents this tab contains.
   * @property {string[]} [types]      The sub-types of Documents this tab contains, otherwise all types of the Document
   *                                   class are assumed.
   * @property {boolean} [advanced]    Is this tab only available in the advanced browsing mode.
   */

  /**
   * Application tabs.
   * @type {CompendiumBrowserTabDescriptor5e[]}
   */
  static TABS = [
    {
      tab: "classes",
      label: "TYPES.Item.classPl",
      svg: "systems/dnd5e/icons/svg/items/class.svg",
      documentClass: "Item",
      types: ["class"]
    },
    {
      tab: "subclasses",
      label: "TYPES.Item.subclassPl",
      svg: "systems/dnd5e/icons/svg/items/subclass.svg",
      documentClass: "Item",
      types: ["subclass"]
    },
    {
      tab: "races",
      label: "TYPES.Item.racePl",
      svg: "systems/dnd5e/icons/svg/items/race.svg",
      documentClass: "Item",
      types: ["race"]
    },
    {
      tab: "feats",
      label: "DND5E.CompendiumBrowser.Tabs.Feat.other",
      icon: "fas fa-star",
      documentClass: "Item",
      types: ["feat"]
    },
    {
      tab: "backgrounds",
      label: "TYPES.Item.backgroundPl",
      svg: "systems/dnd5e/icons/svg/items/background.svg",
      documentClass: "Item",
      types: ["background"]
    },
    {
      tab: "physical",
      label: "DND5E.CompendiumBrowser.Tabs.Item.other",
      svg: "systems/dnd5e/icons/svg/backpack.svg",
      documentClass: "Item",
      types: ["physical"]
    },
    {
      tab: "spells",
      label: "TYPES.Item.spellPl",
      icon: "fas fa-book",
      documentClass: "Item",
      types: ["spell"]
    },
    {
      tab: "monsters",
      label: "DND5E.CompendiumBrowser.Tabs.Monster.other",
      svg: "systems/dnd5e/icons/svg/monster.svg",
      documentClass: "Actor",
      types: ["npc"]
    },
    {
      tab: "vehicles",
      label: "TYPES.Actor.vehiclePl",
      svg: "systems/dnd5e/icons/svg/vehicle.svg",
      documentClass: "Actor",
      types: ["vehicle"]
    },
    {
      tab: "actors",
      label: "DOCUMENT.Actors",
      svg: "systems/dnd5e/icons/svg/monster.svg",
      documentClass: "Actor",
      advanced: true
    },
    {
      tab: "items",
      label: "DOCUMENT.Items",
      svg: "systems/dnd5e/icons/svg/backpack.svg",
      documentClass: "Item",
      advanced: true
    }
  ];

  /* -------------------------------------------- */

  /**
   * Available filtering modes.
   * @enum {number}
   */
  static MODES = {
    BASIC: 1,
    ADVANCED: 2
  };

  /* -------------------------------------------- */

  /**
   * Batching configuration.
   * @type {Record<string, number>}
   */
  static BATCHING = {
    /**
     * The number of pixels before reaching the end of the scroll container to begin loading additional entries.
     */
    MARGIN: 50,

    /**
     * The number of entries to load per batch.
     */
    SIZE: 50
  };

  /* -------------------------------------------- */

  /**
   * The number of milliseconds to delay between user keypresses before executing a search.
   * @type {number}
   */
  static SEARCH_DELAY = 200;

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * Should the selection controls be displayed?
   * @type {boolean}
   */
  get displaySelection() {
    return !!this.options.selection.min || !!this.options.selection.max;
  }

  /* -------------------------------------------- */

  /**
   * Currently defined filters.
   */
  #filters;

  /**
   * Current filters selected.
   * @type {CompendiumBrowserFilters}
   */
  get currentFilters() {
    const filters = foundry.utils.mergeObject(
      this.#filters,
      this.options.filters.locked,
      { inplace: false }
    );
    filters.documentClass ??= "Item";
    if ( filters.additional?.source ) {
      filters.additional.source = Object.entries(filters.additional.source).reduce((obj, [k, v]) => {
        obj[k.slugify({ strict: true })] = v;
        return obj;
      }, {});
    }
    return filters;
  }

  /* -------------------------------------------- */

  /**
   * Fetched results.
   * @type {Promise<object[]|Document[]>|object[]|Document[]}
   */
  #results;

  /* -------------------------------------------- */

  /**
   * The index of the next result to render as part of batching.
   * @type {number}
   */
  #resultIndex = -1;

  /* -------------------------------------------- */

  /**
   * Whether rendering is currently throttled.
   * @type {boolean}
   */
  #renderThrottle = false;

  /* -------------------------------------------- */

  /**
   * UUIDs of currently selected documents.
   * @type {Set<string>}
   */
  #selected = new Set();

  get selected() {
    return this.#selected;
  }

  /* -------------------------------------------- */

  /**
   * Suffix used for localization selection messages based on min and max values.
   * @type {string|null}
   */
  get #selectionLocalizationSuffix() {
    const max = this.options.selection.max;
    const min = this.options.selection.min;
    if ( !min && !max ) return null;
    if ( !min && max ) return "Max";
    if ( min && !max ) return "Min";
    if ( min !== max ) return "Range";
    return "Single";
  }

  /* -------------------------------------------- */

  /**
   * The cached set of available sources to filter on.
   * @type {Record<string, string>}
   */
  #sources;

  /* -------------------------------------------- */

  /**
   * The mode the browser is currently in.
   * @type {CompendiumBrowser.MODES}
   */
  _mode = this.constructor.MODES.BASIC;

  /* -------------------------------------------- */

  /**
   * The function to invoke when searching results by name.
   * @type {Function}
   */
  _debouncedSearch = foundry.utils.debounce(this._onSearchName.bind(this), this.constructor.SEARCH_DELAY);

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  _configureRenderOptions(options) {
    super._configureRenderOptions(options);
    if ( options.isFirstRender ) {
      const tab = this.constructor.TABS.find(t => t.tab === this.options.tab);
      if ( tab ) foundry.utils.setProperty(options, "dnd5e.browser.types", tab.types);
    }
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareContext(options) {
    const context = await super._prepareContext(options);
    context.filters = this.currentFilters;

    let dataModels = Object.entries(CONFIG[context.filters.documentClass].dataModels);
    if ( context.filters.types?.size ) dataModels = dataModels.filter(([type]) => context.filters.types.has(type));
    context.filterDefinitions = dataModels
      .map(([, d]) => d.compendiumBrowserFilters ?? new Map())
      .reduce((first, second) => {
        if ( !first ) return second;
        return CompendiumBrowser.intersectFilters(first, second);
      }, null) ?? new Map();
    context.filterDefinitions.set("source", {
      label: "DND5E.SOURCE.FIELDS.source.label",
      type: "set",
      config: {
        keyPath: "system.source.slug",
        choices: foundry.utils.mergeObject(
          this.#sources ?? {},
          Object.fromEntries(Object.keys(this.options.filters?.locked?.additional?.source ?? {}).map(k => {
            return [k.slugify({ strict: true }), CONFIG.DND5E.sourceBooks[k] ?? k];
          })), { inplace: false }
        )
      }
    });
    return context;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _preparePartContext(partId, context, options) {
    await super._preparePartContext(partId, context, options);
    switch ( partId ) {
      case "documentClass":
      case "types":
      case "filters": return this._prepareSidebarContext(partId, context, options);
      case "results": return this._prepareResultsContext(context, options);
      case "footer": return this._prepareFooterContext(context, options);
      case "tabs": return this._prepareTabsContext(context, options);
      case "header": return this._prepareHeaderContext(context, options);
    }
    return context;
  }

  /* -------------------------------------------- */

  /**
   * Prepare the footer context.
   * @param {ApplicationRenderContext} context     Shared context provided by _prepareContext.
   * @param {HandlebarsRenderOptions} options      Options which configure application rendering behavior.
   * @returns {Promise<ApplicationRenderContext>}  Context data for a specific part.
   * @protected
   */
  async _prepareFooterContext(context, options) {
    const value = this.#selected.size;
    const { max, min } = this.options.selection;

    context.displaySelection = this.displaySelection;
    context.invalid = (value < (min || -Infinity)) || (value > (max || Infinity)) ? "invalid" : "";
    const suffix = this.#selectionLocalizationSuffix;
    context.summary = suffix ? game.i18n.format(
      `DND5E.CompendiumBrowser.Selection.Summary.${suffix}`, { max, min, value }
    ) : value;
    return context;
  }

  /* -------------------------------------------- */

  /**
   * Prepare the header context.
   * @param {ApplicationRenderContext} context  Shared context provided by _prepareContext.
   * @param {HandlebarsRenderOptions} options   Options which configure rendering behavior.
   * @returns {Promise<ApplicationRenderContext>}
   * @protected
   */
  async _prepareHeaderContext(context, options) {
    context.showModeToggle = foundry.utils.isEmpty(this.options.filters.locked);
    context.isAdvanced = this._mode === this.constructor.MODES.ADVANCED;
    return context;
  }

  /* -------------------------------------------- */

  /**
   * Prepare the sidebar context.
   * @param {string} partId                        The part being rendered.
   * @param {ApplicationRenderContext} context     Shared context provided by _prepareContext.
   * @param {HandlebarsRenderOptions} options      Options which configure application rendering behavior.
   * @returns {Promise<ApplicationRenderContext>}  Context data for a specific part.
   * @protected
   */
  async _prepareSidebarContext(partId, context, options) {
    context.isLocked = {};
    context.isLocked.filters = ("additional" in this.options.filters.locked);
    context.isLocked.types = ("types" in this.options.filters.locked) || context.isLocked.filters;
    context.isLocked.documentClass = ("documentClass" in this.options.filters.locked) || context.isLocked.types;
    const types = foundry.utils.getProperty(options, "dnd5e.browser.types") ?? [];

    if ( partId === "types" ) {
      context.showTypes = (types.length !== 1) || (types[0] === "physical");
      context.types = CONFIG[context.filters.documentClass].documentClass.compendiumBrowserTypes({
        chosen: context.filters.types
      });

      // Special case handling for 'Items' tab in basic mode.
      if ( types[0] === "physical" ) context.types = context.types.physical.children;

      if ( context.isLocked.types ) {
        for ( const [key, value] of Object.entries(context.types) ) {
          if ( !value.children && !value.chosen ) delete context.types[key];
          else if ( value.children ) {
            for ( const [k, v] of Object.entries(value.children) ) {
              if ( !v.chosen ) delete value.children[k];
            }
            if ( foundry.utils.isEmpty(value.children) ) delete context.types[key];
          }
        }
      }
    }

    else if ( partId === "filters" ) {
      context.additional = Array.from(context.filterDefinitions?.entries() ?? []).reduce((arr, [key, data]) => {
        // Special case handling for 'Feats' tab in basic mode.
        if ( (types[0] === "feat") && ((key === "category") || (key === "subtype")) ) return arr;

        let sort = 0;
        switch ( data.type ) {
          case "boolean": sort = 1; break;
          case "range": sort = 2; break;
          case "set": sort = 3; break;
        }

        arr.push(foundry.utils.mergeObject(data, {
          key, sort,
          value: context.filters.additional?.[key],
          locked: this.options.filters.locked?.additional?.[key]
        }, { inplace: false }));
        return arr;
      }, []);

      context.additional.sort((a, b) => a.sort - b.sort);
    }

    return context;
  }

  /* -------------------------------------------- */

  /**
   * Prepare the results context.
   * @param {ApplicationRenderContext} context     Shared context provided by _prepareContext.
   * @param {HandlebarsRenderOptions} options      Options which configure application rendering behavior.
   * @returns {Promise<ApplicationRenderContext>}  Context data for a specific part.
   * @protected
   */
  async _prepareResultsContext(context, options) {
    // TODO: Determine if new set of results need to be fetched, otherwise use old results and re-sort as necessary
    // Sorting changes alone shouldn't require a re-fetch, but any change to filters will
    const filters = CompendiumBrowser.applyFilters(context.filterDefinitions, context.filters.additional);
    // Add the name & arbitrary filters
    if ( this.#filters.name?.length ) filters.push({ k: "name", o: "icontains", v: this.#filters.name });
    if ( context.filters.arbitrary?.length ) filters.push(...context.filters.arbitrary);
    this.#results = CompendiumBrowser.fetch(CONFIG[context.filters.documentClass].documentClass, {
      filters,
      types: context.filters.types,
      indexFields: new Set(["system.source"])
    });
    context.displaySelection = this.displaySelection;
    return context;
  }

  /* -------------------------------------------- */

  /**
   * Prepare the tabs context.
   * @param {ApplicationRenderContext} context  Shared context provided by _prepareContext.
   * @param {HandlebarsRenderOptions} options   Options which configure application rendering behavior.
   * @returns {Promise<ApplicationRenderContext>}
   * @protected
   */
  async _prepareTabsContext(context, options) {
    // If we are locked to a particular filter, do not show tabs.
    if ( !foundry.utils.isEmpty(this.options.filters.locked) ) {
      context.tabs = [];
      return context;
    }

    const advanced = this._mode === this.constructor.MODES.ADVANCED;
    context.tabs = foundry.utils.deepClone(this.constructor.TABS.filter(t => !!t.advanced === advanced));
    const tab = options.isFirstRender ? this.options.tab : this.tabGroups.primary;
    const activeTab = context.tabs.find(t => t.tab === tab) ?? context.tabs[0];
    activeTab.active = true;

    return context;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _renderFrame(options) {
    const frame = await super._renderFrame(options);
    if ( game.user.isGM ) {
      frame.querySelector('[data-action="close"]').insertAdjacentHTML("beforebegin", `
        <button type="button" class="header-control fas fa-cog icon" data-action="configureSources"
                data-tooltip="DND5E.CompendiumBrowser.Sources.Label"
                aria-label="${game.i18n.localize("DND5E.CompendiumBrowser.Sources.Label")}"></button>
      `);
    }
    return frame;
  }

  /* -------------------------------------------- */

  /**
   * Render a single result entry.
   * @param {object|Document} entry  The entry.
   * @param {string} documentClass   The entry's Document class.
   * @returns {Promise<HTMLElement>}
   * @protected
   */
  async _renderResult(entry, documentClass) {
    const { img, name, type, uuid, system } = entry;
    // TODO: Provide more useful subtitles.
    const subtitle = CONFIG[documentClass].typeLabels[type] ?? "";
    const source = system?.source?.value ?? "";
    const context = {
      entry: { img, name, subtitle, uuid, source },
      displaySelection: this.displaySelection,
      selected: this.#selected.has(uuid)
    };
    const html = await renderTemplate("systems/dnd5e/templates/compendium/browser-entry.hbs", context);
    const template = document.createElement("template");
    template.innerHTML = html;
    const element = template.content.firstElementChild;
    if ( documentClass !== "Item" ) return element;
    element.dataset.tooltip = `
      <section class="loading" data-uuid="${uuid}">
        <i class="fa-solid fa-spinner fa-spin-pulse" inert></i>
      </section>
    `;
    element.dataset.tooltipClass = "dnd5e2 dnd5e-tooltip item-tooltip";
    element.dataset.tooltipDirection ??= "RIGHT";
    return element;
  }

  /* -------------------------------------------- */

  /**
   * Render results once loaded to avoid holding up initial app display.
   * @protected
   */
  async _renderResults() {
    let rendered = [];
    const { documentClass } = this.currentFilters;
    const results = await this.#results;
    this.#results = results;
    const batchEnd = Math.min(this.constructor.BATCHING.SIZE, results.length);
    for ( let i = 0; i < batchEnd; i++ ) {
      rendered.push(this._renderResult(results[i], documentClass));
    }
    this.element.querySelector(".results-loading").hidden = true;
    this.element.querySelector('[data-application-part="results"] .item-list')
      .replaceChildren(...(await Promise.all(rendered)));
    this.#resultIndex = batchEnd;
  }

  /* -------------------------------------------- */

  /**
   * Show a list of applicable source filters for the available results.
   * @protected
   */
  async _renderSourceFilters() {
    const sources = [];
    for ( const result of this.#results ) {
      const source = foundry.utils.getProperty(result, "system.source");
      if ( foundry.utils.getType(source) !== "Object" ) continue;
      const { slug, value } = source;
      sources.push({ slug, value: CONFIG.DND5E.sourceBooks[value] ?? value });
    }
    sources.sort((a, b) => a.value.localeCompare(b.value, game.i18n.lang));
    this.#sources = Object.fromEntries(sources.map(({ slug, value }) => [slug, value]));
    const filters = this.element.querySelector('[data-application-part="filters"]');
    filters.querySelector('[data-filter-id="source"]')?.remove();
    if ( !sources.length ) return;
    const locked = Object.entries(this.options.filters?.locked?.additional?.source ?? {}).reduce((obj, [k, v]) => {
      obj[k.slugify({ strict: true })] = v;
      return obj;
    }, {});
    const filter = await renderTemplate("systems/dnd5e/templates/compendium/browser-sidebar-filter-set.hbs", {
      locked,
      value: locked,
      key: "source",
      label: "DND5E.SOURCE.FIELDS.source.label",
      config: { choices: this.#sources }
    });
    filters.insertAdjacentHTML("beforeend", filter);
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /** @inheritDoc */
  changeTab(tab, group, options={}) {
    super.changeTab(tab, group, options);
    const target = this.element.querySelector(`nav.tabs [data-group="${group}"][data-tab="${tab}"]`);
    let { types } = target.dataset;
    types = types ? types.split(",") : [];
    this._applyTabFilters(tab);
    this.render({ parts: ["results", "filters", "types"], dnd5e: { browser: { types } }, changedTab: true });
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _attachFrameListeners() {
    super._attachFrameListeners();
    this.element.addEventListener("scroll", this._onScrollResults.bind(this), { capture: true, passive: true });
    this.element.addEventListener("dragstart", this._onDragStart.bind(this));
    this.element.addEventListener("keydown", this._debouncedSearch, { passive: true });
    this.element.addEventListener("keydown", this._onKeyAction.bind(this), { passive: true });
    this.element.addEventListener("pointerdown", event => {
      if ( (event.button === 1) && document.getElementById("tooltip")?.classList.contains("active") ) {
        event.preventDefault();
      }
    });
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _attachPartListeners(partId, htmlElement, options) {
    super._attachPartListeners(partId, htmlElement, options);
    if ( partId === "results" ) this._renderResults().then(() => {
      if ( options.isFirstRender || options.changedTab ) this._renderSourceFilters();
    });
    else if ( partId === "types" ) this.#adjustCheckboxStates(htmlElement);
  }

  /* -------------------------------------------- */

  /**
   * Apply filters based on the compendium browser's mode.
   * @param {CompendiumBrowser.MODES} mode  The mode.
   * @protected
   */
  _applyModeFilters(mode) {
    const isAdvanced = mode === this.constructor.MODES.ADVANCED;
    delete this.#filters.types;
    delete this.#filters.additional;
    if ( isAdvanced ) this.#filters.documentClass = "Actor";
    else {
      this.#filters.documentClass = "Item";
      this.#filters.types = new Set(["class"]);
    }
  }

  /* -------------------------------------------- */

  /**
   * Apply filters based on the selected tab.
   * @param {string} id  The tab ID.
   * @protected
   */
  _applyTabFilters(id) {
    const tab = this.constructor.TABS.find(t => t.tab === id);
    if ( !tab ) return;
    const { documentClass, types } = tab;
    delete this.#filters.additional;
    this.#filters.documentClass = documentClass;
    this.#filters.types = new Set(types);

    // Special case handling for 'Items' tab in basic mode.
    if ( id === "physical" ) {
      const physical = Item.implementation.compendiumBrowserTypes().physical.children;
      Object.keys(physical).forEach(this.#filters.types.add, this.#filters.types);
    }

    // Special case handling for 'Feats' tab in basic mode.
    if ( id === "feats" ) this.#filters.additional = { category: { feat: 1 } };
  }

  /* -------------------------------------------- */

  /**
   * Adjust the states of group checkboxes to make then indeterminate if only some of their children are selected.
   * @param {HTMLElement} htmlElement  Element within which to find groups.
   */
  #adjustCheckboxStates(htmlElement) {
    for ( const groupArea of htmlElement.querySelectorAll(".type-group") ) {
      const group = groupArea.querySelector(".type-group-header dnd5e-checkbox");
      const children = groupArea.querySelectorAll(".wrapper dnd5e-checkbox");
      if ( Array.from(children).every(e => e.checked) ) {
        group.checked = true;
        group.indeterminate = false;
      } else {
        group.checked = group.indeterminate = Array.from(children).some(e => e.checked);
      }
    }
  }

  /* -------------------------------------------- */

  /** @override */
  _onChangeForm(formConfig, event) {
    if ( event.target.name === "selected" ) {
      if ( event.target.checked ) this.#selected.add(event.target.value);
      else this.#selected.delete(event.target.value);
      event.target.closest(".item").classList.toggle("selected", event.target.checked);
      this.render({ parts: ["footer"] });
    }
    if ( event.target.name?.startsWith("additional.") ) CompendiumBrowser.#onSetFilter.call(this, event, event.target);
  }

  /* -------------------------------------------- */

  /**
   * Handle dragging an entry.
   * @param {DragEvent} event  The drag event.
   * @protected
   */
  _onDragStart(event) {
    const { uuid } = event.target.closest("[data-uuid]")?.dataset ?? {};
    try {
      const { type } = foundry.utils.parseUuid(uuid);
      event.dataTransfer.setData("text/plain", JSON.stringify({ type, uuid }));
    } catch(e) {
      console.error(e);
    }
  }

  /* -------------------------------------------- */

  /**
   * Handle triggering an action via keyboard.
   * @param {KeyboardEvent} event  The originating event.
   * @protected
   */
  _onKeyAction(event) {
    const target = event.target.closest("[data-action]");
    if ( (event.key !== " ") || !target ) return;
    const { action } = target.dataset;
    const handler = this.options.actions[action];
    if ( handler ) handler.call(this, event, target);
  }

  /* -------------------------------------------- */

  /**
   * Handle rendering a new batch of results when the user scrolls to the bottom of the list.
   * @param {Event} event  The originating scroll event.
   * @protected
   */
  async _onScrollResults(event) {
    if ( this.#renderThrottle || !event.target.matches('[data-application-part="results"]') ) return;
    if ( (this.#results instanceof Promise) || (this.#resultIndex >= this.#results.length) ) return;
    const { scrollTop, scrollHeight, clientHeight } = event.target;
    if ( scrollTop + clientHeight < scrollHeight - this.constructor.BATCHING.MARGIN ) return;
    this.#renderThrottle = true;
    const { documentClass } = this.currentFilters;
    const rendered = [];
    const batchStart = this.#resultIndex;
    const batchEnd = Math.min(batchStart + this.constructor.BATCHING.SIZE, this.#results.length);
    for ( let i = batchStart; i < batchEnd; i++ ) {
      rendered.push(this._renderResult(this.#results[i], documentClass));
    }
    this.element.querySelector('[data-application-part="results"] .item-list').append(...(await Promise.all(rendered)));
    this.#resultIndex = batchEnd;
    this.#renderThrottle = false;
  }

  /* -------------------------------------------- */

  /**
   * Handle searching for a Document by name.
   * @param {KeyboardEvent} event  The triggering event.
   * @protected
   */
  _onSearchName(event) {
    if ( !event.target.matches("search > input") ) return;
    this.#filters.name = event.target.value;
    this.render({ parts: ["results"] });
  }

  /* -------------------------------------------- */

  /**
   * Handle configuring compendium browser sources.
   * @this {CompendiumBrowser}
   */
  static #onConfigureSources() {
    new CompendiumBrowserSettingsConfig().render({ force: true });
  }

  /* -------------------------------------------- */

  /**
   * Handle clearing the name filter.
   * @this {CompendiumBrowser}
   * @param {PointerEvent} event  The originating click event.
   * @param {HTMLElement} target  The target of the click event.
   */
  static async #onClearName(event, target) {
    const input = target.closest("search").querySelector(":scope > input");
    input.value = this.#filters.name = "";
    this.render({ parts: ["results"] });
  }

  /* -------------------------------------------- */

  /**
   * Handle form submission with selection.
   * @this {CompendiumBrowser}
   * @param {SubmitEvent} event          The form submission event.
   * @param {HTMLFormElement} form       The submitted form element.
   * @param {FormDataExtended} formData  The data from the submitted form.
   */
  static async #onHandleSubmit(event, form, formData) {
    if ( !this.displaySelection ) return;

    const value = this.#selected.size;
    const { max, min } = this.options.selection;
    if ( (value < (min || -Infinity)) || (value > (max || Infinity)) ) {
      const suffix = this.#selectionLocalizationSuffix;
      const pr = new Intl.PluralRules(game.i18n.lang);
      throw new Error(game.i18n.format(`DND5E.CompendiumBrowser.Selection.Warning.${suffix}`, {
        max, min, value,
        document: game.i18n.localize(`DND5E.CompendiumBrowser.Selection.Warning.Document.${pr.select(max || min)}`)
      }));
    }

    /**
     * Hook event that calls when a compendium browser is submitted with selected items.
     * @function dnd5e.compendiumBrowserSelection
     * @memberof hookEvents
     * @param {CompendiumBrowser} browser  Compendium Browser application being submitted.
     * @param {Set<string>} selected       Set of document UUIDs that are selected.
     */
    Hooks.callAll("dnd5e.compendiumBrowserSelection", this, this.#selected);
  }

  /* -------------------------------------------- */

  /**
   * Handle opening a link to an item.
   * @this {CompendiumBrowser}
   * @param {PointerEvent} event  The originating click event.
   * @param {HTMLElement} target  The capturing HTML element which defined a [data-action].
   */
  static async #onOpenLink(event, target) {
    (await fromUuid(target.closest("[data-uuid]")?.dataset.uuid))?.sheet?.render(true);
  }

  /* -------------------------------------------- */

  /**
   * Handle setting the document class or a filter.
   * @this {CompendiumBrowser}
   * @param {PointerEvent} event  The originating click event.
   * @param {HTMLElement} target  The capturing HTML element which defined a [data-action].
   */
  static async #onSetFilter(event, target) {
    const name = target.name;
    const value = target.value;
    const existingValue = foundry.utils.getProperty(this.#filters, name);
    if ( value === existingValue ) return;
    foundry.utils.setProperty(this.#filters, name, value === "" ? undefined : value);

    if ( target.tagName === "BUTTON" ) for ( const button of this.element.querySelectorAll(`[name="${name}"]`) ) {
      button.ariaPressed = button.value === value;
    }

    this.render({ parts: ["results"] });
  }

  /* -------------------------------------------- */

  /**
   * Handle setting a type restriction.
   * @this {CompendiumBrowser}
   * @param {PointerEvent} event  The originating click event.
   * @param {HTMLElement} target  The capturing HTML element which defined a [data-action].
   */
  static async #onSetType(event, target) {
    this.#filters.types ??= new Set();

    if ( target.defaultValue ) {
      if ( target.checked ) this.#filters.types.add(target.defaultValue);
      else this.#filters.types.delete(target.defaultValue);
      this.#adjustCheckboxStates(target.closest(".sidebar"));
    }

    else {
      target.indeterminate = false;
      for ( const child of target.closest(".type-group").querySelectorAll("dnd5e-checkbox[value]") ) {
        child.checked = target.checked;
        if ( target.checked ) this.#filters.types.add(child.defaultValue);
        else this.#filters.types.delete(child.defaultValue);
      }
    }

    this.render({ parts: ["filters", "results"] });
  }

  /* -------------------------------------------- */

  /**
   * Handle toggling the collapsed state of a collapsible section.
   * @this {CompendiumBrowser}
   * @param {PointerEvent} event  The originating click event.
   * @param {HTMLElement} target  The capturing HTML element which defined a [data-action].
   */
  static async #onToggleCollapse(event, target) {
    target.closest(".collapsible")?.classList.toggle("collapsed");
  }

  /* -------------------------------------------- */

  /**
   * Handle toggling the compendium browser mode.
   * @this {CompendiumBrowser}
   * @param {PointerEvent} event  The originating click event.
   * @param {HTMLElement} target  The element that was clicked.
   */
  static #onToggleMode(event, target) {
    // TODO: Consider persisting this choice in a client setting.
    this._mode = target.checked ? this.constructor.MODES.ADVANCED : this.constructor.MODES.BASIC;
    const tabs = foundry.utils.deepClone(this.constructor.TABS.filter(t => !!t.advanced === target.checked));
    const activeTab = tabs.find(t => t.tab === this.tabGroups.primary) ?? tabs[0];
    const types = target.checked ? [] : (activeTab?.types ?? ["class"]);
    this._applyModeFilters(this._mode);
    this._applyTabFilters(activeTab?.tab);
    this.render({ parts: ["results", "filters", "types", "tabs"], dnd5e: { browser: { types } }, changedTab: true });
  }

  /* -------------------------------------------- */
  /*  Database Access                             */
  /* -------------------------------------------- */

  /**
   * Retrieve a listing of documents from all compendiums for a specific Document type, with additional filters
   * optionally applied.
   * @param {typeof Document} documentClass  Document type to fetch (e.g. Actor or Item).
   * @param {object} [options={}]
   * @param {Set<string>} [options.types]    Individual document subtypes to filter upon (e.g. "loot", "class", "npc").
   * @param {FilterDescription[]} [options.filters]  Filters to provide further filters.
   * @param {boolean} [options.index=true]   Should only the index for each document be returned, or the whole thing?
   * @param {Set<string>} [options.indexFields]  Key paths for fields to index.
   * @param {boolean|string|Function} [options.sort=true]  Should the contents be sorted? By default sorting will be
   *                                         performed using document names, but a key path can be provided to sort on
   *                                         a specific property or a function to provide more advanced sorting.
   * @returns {object[]|Document[]}
   */
  static async fetch(documentClass, { types=new Set(), filters=[], index=true, indexFields=new Set(), sort=true }={}) {
    // Nothing within containers should be shown
    filters.push({ k: "system.container", o: "in", v: [null, undefined] });

    // If filters are provided, merge their keys with any other fields needing to be indexed
    if ( filters.length ) indexFields = indexFields.union(uniqueKeys(filters));

    // Do not attempt to index derived fields as this will throw an error server-side.
    indexFields.delete("system.source.slug");

    // Collate compendium sources.
    const sources = CompendiumBrowserSettingsConfig.collateSources();

    // Iterate over all packs
    let documents = game.packs

      // Skip packs that have the wrong document class
      .filter(p => (p.metadata.type === documentClass.metadata.name)

        // Do not show entries inside compendia that are not visible to the current user.
        && p.visible

        && sources.has(p.collection)

        // If types are set and specified in compendium flag, only include those that include the correct types
        && (!types.size || !p.metadata.flags.dnd5e?.types || new Set(p.metadata.flags.dnd5e.types).intersects(types)))

      // Generate an index based on the needed fields
      .map(async p => await Promise.all((await p.getIndex({ fields: Array.from(indexFields) })

        // Apply module art to the new index
        .then(index => game.dnd5e.moduleArt.apply(index)))

        // Derive source values
        .map(i => {
          const source = foundry.utils.getProperty(i, "system.source");
          if ( (foundry.utils.getType(source) === "Object") && i.uuid ) SourceField.prepareData.call(source, i.uuid);
          return i;
        })

        // Remove any documents that don't match the specified types or the provided filters
        .filter(i => (!types.size || types.has(i.type)) && (!filters.length || performCheck(i, filters)))

        // If full documents are required, retrieve those, otherwise stick with the indices
        .map(async i => index ? i : await fromUuid(i.uuid))
      ));

    // Wait for everything to finish loading and flatten the arrays
    documents = (await Promise.all(documents)).flat();

    if ( sort ) {
      if ( sort === true ) sort = "name";
      const sortFunc = foundry.utils.getType(sort) === "function" ? sort : (lhs, rhs) =>
        String(foundry.utils.getProperty(lhs, sort))
          .localeCompare(String(foundry.utils.getProperty(rhs, sort)), game.i18n.lang);
      documents.sort(sortFunc);
    }

    return documents;
  }

  /* -------------------------------------------- */
  /*  Factory Methods                             */
  /* -------------------------------------------- */

  /**
   * Factory method used to spawn a compendium browser and wait for the results of a selection.
   * @param {Partial<CompendiumBrowserConfiguration>} [options]
   * @returns {Promise<Set<string>|null>}
   */
  static async select(options={}) {
    return new Promise((resolve, reject) => {
      const browser = new CompendiumBrowser(options);
      browser.addEventListener("close", event => {
        resolve(browser.selected?.size ? browser.selected : null);
      }, { once: true });
      browser.render({ force: true });
    });
  }

  /* -------------------------------------------- */

  /**
   * Factory method used to spawn a compendium browser and return a single selected item or null if canceled.
   * @param {Partial<CompendiumBrowserConfiguration>} [options]
   * @returns {Promise<string|null>}
   */
  static async selectOne(options={}) {
    const result = await this.select(
      foundry.utils.mergeObject(options, { selection: { min: 1, max: 1 } }, { inplace: false })
    );
    return result?.size ? result.first() : null;
  }

  /* -------------------------------------------- */
  /*  Helpers                                     */
  /* -------------------------------------------- */

  /**
   * Transform filter definition and additional filters values into the final filters to apply.
   * @param {CompendiumBrowserFilterDefinition} definition  Filter definition provided by type.
   * @param {object} values                                 Values of currently selected filters.
   * @returns {FilterDescription[]}
   */
  static applyFilters(definition, values) {
    const filters = [];
    for ( const [key, value] of Object.entries(values ?? {}) ) {
      const def = definition.get(key);
      if ( !def ) continue;
      if ( foundry.utils.getType(def.createFilter) === "function" ) {
        def.createFilter(filters, value, def);
        continue;
      }
      switch ( def.type ) {
        case "boolean":
          if ( value ) filters.push({ k: def.config.keyPath, v: value === 1 });
          break;
        case "range":
          const min = Number(value.min);
          const max = Number(value.max);
          if ( Number.isFinite(min) ) filters.push({ k: def.config.keyPath, o: "gte", v: min });
          if ( Number.isFinite(max) ) filters.push({ k: def.config.keyPath, o: "lte", v: max });
          break;
        case "set":
          const choices = foundry.utils.deepClone(def.config.choices);
          if ( def.config.blank ) choices._blank = "";
          const [positive, negative] = Object.entries(value ?? {}).reduce(([positive, negative], [k, v]) => {
            if ( k in choices ) {
              if ( k === "_blank" ) k = "";
              if ( v === 1 ) positive.push(k);
              else if ( v === -1 ) negative.push(k);
            }
            return [positive, negative];
          }, [[], []]);
          if ( positive.length ) filters.push(
            { k: def.config.keyPath, o: def.config.multiple ? "hasall" : "in", v: positive }
          );
          if ( negative.length ) filters.push(
            { o: "NOT", v: { k: def.config.keyPath, o: def.config.multiple ? "hasany" : "in", v: negative } }
          );
          break;
        default:
          console.warn(`Filter type ${def.type} not handled.`);
          break;
      }
    }
    return filters;
  }

  /* -------------------------------------------- */

  /**
   * Inject the compendium browser button into the compendium sidebar.
   * @param {HTMLElement} html  HTML of the sidebar being rendered.
   */
  static injectSidebarButton(html) {
    const button = document.createElement("button");
    button.type = "button";
    button.classList.add("open-compendium-browser");
    button.innerHTML = `
      <i class="fa-solid fa-book-open-reader" inert></i>
      ${game.i18n.localize("DND5E.CompendiumBrowser.Action.Open")}
    `;
    button.addEventListener("click", event => (new CompendiumBrowser()).render({ force: true }));

    let headerActions = html.querySelector(".header-actions");
    // FIXME: Workaround for 336 bug. Remove when 337 released.
    if ( !headerActions ) {
      headerActions = document.createElement("div");
      headerActions.className = "header-actions action-buttons flexrow";
      html.querySelector(":scope > header").insertAdjacentElement("afterbegin", headerActions);
    }
    headerActions.append(button);
  }

  /* -------------------------------------------- */

  /**
   * Take two filter sets and find only the filters that match between the two.
   * @param {CompendiumBrowserFilterDefinition} first
   * @param {CompendiumBrowserFilterDefinition>} second
   * @returns {CompendiumBrowserFilterDefinition}
   */
  static intersectFilters(first, second) {
    const final = new Map();

    // Iterate over all keys in first map
    for ( const [key, firstConfig] of first.entries() ) {
      const secondConfig = second.get(key);
      if ( firstConfig.type !== secondConfig?.type ) continue;
      const finalConfig = foundry.utils.deepClone(firstConfig);

      switch ( firstConfig.type ) {
        case "range":
          if ( ("min" in firstConfig.config) || ("min" in secondConfig.config) ) {
            if ( !("min" in firstConfig.config) || !("min" in secondConfig.config) ) continue;
            finalConfig.config.min = Math.max(firstConfig.config.min, secondConfig.config.min);
          }
          if ( ("max" in firstConfig.config) || ("max" in secondConfig.config) ) {
            if ( !("max" in firstConfig.config) || !("max" in secondConfig.config) ) continue;
            finalConfig.config.max = Math.min(firstConfig.config.max, secondConfig.config.max);
          }
          if ( ("min" in finalConfig.config) && ("max" in finalConfig.config)
            && (finalConfig.config.min > finalConfig.config.max) ) continue;
          break;
        case "set":
          Object.keys(finalConfig.config.choices).forEach(k => {
            if ( !(k in secondConfig.config.choices) ) delete finalConfig.config.choices[k];
          });
          if ( foundry.utils.isEmpty(finalConfig.config.choices) ) continue;
          break;
      }

      final.set(key, finalConfig);
    }
    return final;
  }
}

/**
 * Configuration information for a token placement operation.
 *
 * @typedef {object} TokenPlacementConfiguration
 * @property {PrototypeToken[]} tokens  Prototype token information for rendering.
 */

/**
 * Data for token placement on the scene.
 *
 * @typedef {object} PlacementData
 * @property {PrototypeToken} prototypeToken
 * @property {object} index
 * @property {number} index.total             Index of the placement across all placements.
 * @property {number} index.unique            Index of the placement across placements with the same original token.
 * @property {number} x
 * @property {number} y
 * @property {number} rotation
 */

/**
 * Class responsible for placing one or more tokens onto the scene.
 * @param {TokenPlacementConfiguration} config  Configuration information for placement.
 */
class TokenPlacement {
  constructor(config) {
    this.config = config;
  }

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * Configuration information for the placements.
   * @type {TokenPlacementConfiguration}
   */
  config;

  /* -------------------------------------------- */

  /**
   * Index of the token configuration currently being placed in the scene.
   * @param {number}
   */
  #currentPlacement = -1;

  /* -------------------------------------------- */

  /**
   * Track the bound event handlers so they can be properly canceled later.
   * @type {object}
   */
  #events;

  /* -------------------------------------------- */

  /**
   * Track the timestamp when the last mouse move event was captured.
   * @type {number}
   */
  #moveTime = 0;

  /* -------------------------------------------- */

  /**
   * Placements that have been generated.
   * @type {PlacementData[]}
   */
  #placements;

  /* -------------------------------------------- */

  /**
   * Preview tokens. Should match 1-to-1 with placements.
   * @type {Token[]}
   */
  #previews;

  /* -------------------------------------------- */

  /**
   * Is the system currently being throttled to the next animation frame?
   * @type {boolean}
   */
  #throttle = false;

  /* -------------------------------------------- */
  /*  Placement                                   */
  /* -------------------------------------------- */

  /**
   * Perform the placement, asking player guidance when necessary.
   * @param {TokenPlacementConfiguration} config
   * @returns {Promise<PlacementData[]>}
   */
  static place(config) {
    const placement = new this(config);
    return placement.place();
  }

  /**
   * Perform the placement, asking player guidance when necessary.
   * @returns {Promise<PlacementData[]>}
   */
  async place() {
    this.#createPreviews();
    try {
      const placements = [];
      let total = 0;
      const uniqueTokens = new Map();
      while ( this.#currentPlacement < this.config.tokens.length - 1 ) {
        this.#currentPlacement++;
        const obj = canvas.tokens.preview.addChild(this.#previews[this.#currentPlacement].object);
        await obj.draw();
        obj.eventMode = "none";
        const placement = await this.#requestPlacement();
        if ( placement ) {
          const actorId = placement.prototypeToken.parent.id;
          uniqueTokens.set(actorId, (uniqueTokens.get(actorId) ?? -1) + 1);
          placement.index = { total: total++, unique: uniqueTokens.get(actorId) };
          placements.push(placement);
        } else obj.clear();
      }
      return placements;
    } finally {
      this.#destroyPreviews();
    }
  }

  /* -------------------------------------------- */

  /**
   * Create token previews based on the prototype tokens in config.
   */
  #createPreviews() {
    this.#placements = [];
    this.#previews = [];
    for ( const prototypeToken of this.config.tokens ) {
      const tokenData = prototypeToken.toObject();
      tokenData.sight.enabled = false;
      tokenData._id = foundry.utils.randomID();
      if ( tokenData.randomImg ) tokenData.texture.src = prototypeToken.actor.img;
      const cls = getDocumentClass("Token");
      const doc = new cls(tokenData, { parent: canvas.scene });
      this.#placements.push({ prototypeToken, x: 0, y: 0, rotation: tokenData.rotation ?? 0 });
      this.#previews.push(doc);
    }
  }

  /* -------------------------------------------- */

  /**
   * Clear any previews from the scene.
   */
  #destroyPreviews() {
    this.#previews.forEach(p => p.object.destroy());
  }

  /* -------------------------------------------- */
  /*  Event Handlers                              */
  /* -------------------------------------------- */

  /**
   * Activate listeners for the placement preview.
   * @returns {Promise<PlacementData|false>}  A promise that resolves with the final placement if created,
   *                                          or false if the placement was skipped.
   */
  #requestPlacement() {
    return new Promise((resolve, reject) => {
      this.#events = {
        confirm: this.#onConfirmPlacement.bind(this),
        move: this.#onMovePlacement.bind(this),
        resolve,
        reject,
        rotate: this.#onRotatePlacement.bind(this),
        skip: this.#onSkipPlacement.bind(this)
      };

      // Activate listeners
      canvas.stage.on("mousemove", this.#events.move);
      canvas.stage.on("mousedown", this.#events.confirm);
      canvas.app.view.oncontextmenu = this.#events.skip;
      canvas.app.view.onwheel = this.#events.rotate;
    });
  }

  /* -------------------------------------------- */

  /**
   * Shared code for when token placement ends by being confirmed or canceled.
   * @param {Event} event  Triggering event that ended the placement.
   */
  async #finishPlacement(event) {
    canvas.stage.off("mousemove", this.#events.move);
    canvas.stage.off("mousedown", this.#events.confirm);
    canvas.app.view.oncontextmenu = null;
    canvas.app.view.onwheel = null;
  }

  /* -------------------------------------------- */

  /**
   * Move the token preview when the mouse moves.
   * @param {Event} event  Triggering mouse event.
   */
  #onMovePlacement(event) {
    event.stopPropagation();
    if ( this.#throttle ) return;
    this.#throttle = true;
    const idx = this.#currentPlacement;
    const preview = this.#previews[idx];
    const clone = preview.object;
    const local = event.data.getLocalPosition(canvas.tokens);
    local.x = local.x - (clone.w / 2);
    local.y = local.y - (clone.h / 2);
    const dest = !event.shiftKey ? clone.getSnappedPosition(local) : local;
    preview.updateSource({x: dest.x, y: dest.y});
    this.#placements[idx].x = preview.x;
    this.#placements[idx].y = preview.y;
    canvas.tokens.preview.children[this.#currentPlacement]?.refresh();
    requestAnimationFrame(() => this.#throttle = false);
  }

  /* -------------------------------------------- */

  /**
   * Rotate the token preview by 3˚ increments when the mouse wheel is rotated.
   * @param {Event} event  Triggering mouse event.
   */
  #onRotatePlacement(event) {
    if ( event.ctrlKey ) event.preventDefault(); // Avoid zooming the browser window
    event.stopPropagation();
    const delta = canvas.grid.type > CONST.GRID_TYPES.SQUARE ? 30 : 15;
    const snap = event.shiftKey ? delta : 5;
    const preview = this.#previews[this.#currentPlacement];
    this.#placements[this.#currentPlacement].rotation += snap * Math.sign(event.deltaY);
    preview.updateSource({ rotation: this.#placements[this.#currentPlacement].rotation });
    canvas.tokens.preview.children[this.#currentPlacement]?.refresh();
  }

  /* -------------------------------------------- */

  /**
   * Confirm placement when the left mouse button is clicked.
   * @param {Event} event  Triggering mouse event.
   */
  async #onConfirmPlacement(event) {
    await this.#finishPlacement(event);
    this.#events.resolve(this.#placements[this.#currentPlacement]);
  }

  /* -------------------------------------------- */

  /**
   * Skip placement when the right mouse button is clicked.
   * @param {Event} event  Triggering mouse event.
   */
  async #onSkipPlacement(event) {
    await this.#finishPlacement(event);
    this.#events.resolve(false);
  }

  /* -------------------------------------------- */
  /*  Helpers                                     */
  /* -------------------------------------------- */

  /**
   * Adjust the appended number on an unlinked token to account for multiple placements.
   * @param {TokenDocument|object} tokenDocument  Document or data object to adjust.
   * @param {PlacementData} placement             Placement data associated with this token document.
   */
  static adjustAppendedNumber(tokenDocument, placement) {
    const regex = new RegExp(/\((\d+)\)$/);
    const match = tokenDocument.name?.match(regex);
    if ( !match ) return;
    const name = tokenDocument.name.replace(regex, `(${Number(match[1]) + placement.index.unique})`);
    if ( tokenDocument instanceof TokenDocument ) tokenDocument.updateSource({ name });
    else tokenDocument.name = name;
  }
}

const {
  ArrayField: ArrayField$i, BooleanField: BooleanField$x, DocumentIdField: DocumentIdField$5, NumberField: NumberField$B, SchemaField: SchemaField$F, SetField: SetField$r, StringField: StringField$V
} = foundry.data.fields;

/**
 * Information for a single summoned creature.
 *
 * @typedef {object} SummonsProfile
 * @property {string} _id         Unique ID for this profile.
 * @property {string} count       Formula for the number of creatures to summon.
 * @property {string} cr          Formula for the CR of summoned creatures if in CR mode.
 * @property {object} level
 * @property {number} level.min   Minimum level at which this profile can be used.
 * @property {number} level.max   Maximum level at which this profile can be used.
 * @property {string} name        Display name for this profile if it differs from actor's name.
 * @property {Set<string>} types  Types of summoned creatures if in CR mode.
 * @property {string} uuid        UUID of the actor to summon if in default mode.
 */

/**
 * Data model for a summon activity.
 *
 * @property {object} bonuses
 * @property {string} bonuses.ac            Formula for armor class bonus on summoned actor.
 * @property {string} bonuses.hd            Formula for bonus hit dice to add to each summoned NPC.
 * @property {string} bonuses.hp            Formula for bonus hit points to add to each summoned actor.
 * @property {string} bonuses.attackDamage  Formula for bonus added to damage for attacks.
 * @property {string} bonuses.saveDamage    Formula for bonus added to damage for saving throws.
 * @property {string} bonuses.healing       Formula for bonus added to healing.
 * @property {Set<string>} creatureSizes    Set of creature sizes that will be set on summoned creature.
 * @property {Set<string>} creatureTypes    Set of creature types that will be set on summoned creature.
 * @property {object} match
 * @property {string} match.ability         Ability to use for calculating match values.
 * @property {boolean} match.attacks        Match the to hit values on summoned actor's attack to the summoner.
 * @property {boolean} match.proficiency    Match proficiency on summoned actor to the summoner.
 * @property {boolean} match.saves          Match the save DC on summoned actor's abilities to the summoner.
 * @property {SummonsProfile[]} profiles    Information on creatures that can be summoned.
 * @property {object} summon
 * @property {string} summon.identifier     Class identifier that will be used to determine applicable level.
 * @property {""|"cr"} summon.mode          Method of determining what type of creature is summoned.
 * @property {boolean} summon.prompt        Should the player be prompted to place the summons?
 */
class SummonActivityData extends BaseActivityData {
  /** @inheritDoc */
  static defineSchema() {
    return {
      ...super.defineSchema(),
      bonuses: new SchemaField$F({
        ac: new FormulaField(),
        hd: new FormulaField(),
        hp: new FormulaField(),
        attackDamage: new FormulaField(),
        saveDamage: new FormulaField(),
        healing: new FormulaField()
      }),
      creatureSizes: new SetField$r(new StringField$V()),
      creatureTypes: new SetField$r(new StringField$V()),
      match: new SchemaField$F({
        ability: new StringField$V(),
        attacks: new BooleanField$x(),
        proficiency: new BooleanField$x(),
        saves: new BooleanField$x()
      }),
      profiles: new ArrayField$i(new SchemaField$F({
        _id: new DocumentIdField$5({ initial: () => foundry.utils.randomID() }),
        count: new FormulaField(),
        cr: new FormulaField({ deterministic: true }),
        level: new SchemaField$F({
          min: new NumberField$B({ integer: true, min: 0 }),
          max: new NumberField$B({ integer: true, min: 0 })
        }),
        name: new StringField$V(),
        types: new SetField$r(new StringField$V()),
        uuid: new StringField$V()
      })),
      summon: new SchemaField$F({
        identifier: new IdentifierField(),
        mode: new StringField$V(),
        prompt: new BooleanField$x({ initial: true })
      })
    };
  }

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /** @inheritDoc */
  get ability() {
    return this.match.ability || super.ability;
  }

  /* -------------------------------------------- */

  /** @override */
  get actionType() {
    return "summ";
  }

  /* -------------------------------------------- */

  /** @override */
  get applicableEffects() {
    return null;
  }

  /* -------------------------------------------- */

  /**
   * Summons that can be performed based on spell/character/class level.
   * @type {SummonsProfile[]}
   */
  get availableProfiles() {
    const level = this.relevantLevel;
    return this.profiles.filter(e => ((e.level.min ?? -Infinity) <= level) && (level <= (e.level.max ?? Infinity)));
  }

  /* -------------------------------------------- */

  /**
   * Determine the level used to determine profile limits, based on the spell level for spells or either the
   * character or class level, depending on whether `classIdentifier` is set.
   * @type {number}
   */
  get relevantLevel() {
    const keyPath = (this.item.type === "spell") && (this.item.system.level > 0) ? "item.level"
      : this.summon.identifier ? `classes.${this.summon.identifier}.levels` : "details.level";
    return foundry.utils.getProperty(this.getRollData(), keyPath) ?? 0;
  }

  /* -------------------------------------------- */

  /**
   * Creatures summoned by this activity.
   * @type {Actor5e[]}
   */
  get summonedCreatures() {
    if ( !this.actor ) return [];
    return dnd5e.registry.summons.creatures(this.actor)
      .filter(i => i?.getFlag("dnd5e", "summon.origin") === this.uuid);
  }

  /* -------------------------------------------- */
  /*  Data Migrations                             */
  /* -------------------------------------------- */

  /** @override */
  static transformTypeData(source, activityData, options) {
    return foundry.utils.mergeObject(activityData, {
      bonuses: source.system.summons?.bonuses ?? {},
      creatureSizes: source.system.summons?.creatureSizes ?? [],
      creatureTypes: source.system.summons?.creatureTypes ?? [],
      match: {
        ...(source.system.summons?.match ?? {}),
        ability: source.system.ability
      },
      profiles: source.system.summons?.profiles ?? [],
      summon: {
        identifier: source.system.summons?.classIdentifier ?? "",
        mode: source.system.summons?.mode ?? "",
        prompt: source.system.summons?.prompt ?? true
      }
    });
  }
}

/**
 * Activity for summoning creatures.
 */
class SummonActivity extends ActivityMixin(SummonActivityData) {
  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static LOCALIZATION_PREFIXES = [...super.LOCALIZATION_PREFIXES, "DND5E.SUMMON"];

  /* -------------------------------------------- */

  /** @inheritDoc */
  static metadata = Object.freeze(
    foundry.utils.mergeObject(super.metadata, {
      type: "summon",
      img: "systems/dnd5e/icons/svg/activity/summon.svg",
      title: "DND5E.SUMMON.Title",
      sheetClass: SummonSheet,
      usage: {
        actions: {
          placeSummons: SummonActivity.#placeSummons
        },
        dialog: SummonUsageDialog
      }
    }, { inplace: false })
  );

  /* -------------------------------------------- */

  /** @inheritDoc */
  static localize() {
    super.localize();
    this._localizeSchema(this.schema.fields.profiles.element, ["DND5E.SUMMON.FIELDS.profiles"]);
  }

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * Does the user have permissions to summon?
   * @type {boolean}
   */
  get canSummon() {
    return game.user.can("TOKEN_CREATE") && (game.user.isGM || game.settings.get("dnd5e", "allowSummoning"));
  }

  /* -------------------------------------------- */
  /*  Activation                                  */
  /* -------------------------------------------- */

  /**
   * @typedef {ActivityUseConfiguration} SummonUseConfiguration
   * @property {object|false} create
   * @property {string} create.summons                    Should a summoned creature be created?
   * @property {Partial<SummoningConfiguration>} summons  Options for configuring summoning behavior.
   */

  /**
   * Configuration data for summoning behavior.
   *
   * @typedef {object} SummoningConfiguration
   * @property {string} profile         ID of the summoning profile to use.
   * @property {string} [creatureSize]  Selected creature size if multiple are available.
   * @property {string} [creatureType]  Selected creature type if multiple are available.
   */

  /**
   * @typedef {ActivityUsageResults} SummonUsageResults
   * @property {Token5e[]} summoned  Summoned tokens.
   */

  /** @inheritDoc */
  _createDeprecatedConfigs(usageConfig, dialogConfig, messageConfig) {
    const config = super._createDeprecatedConfigs(usageConfig, dialogConfig, messageConfig);
    config.createSummons = usageConfig.create?.summons ?? null;
    config.summonsProfile = usageConfig.summons?.profile ?? null;
    config.summonsOptions = {
      creatureSize: usageConfig.summons?.creatureSize,
      creatureType: usageConfig.summons?.creatureType
    };
    return config;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _applyDeprecatedConfigs(usageConfig, dialogConfig, messageConfig, config, options) {
    super._applyDeprecatedConfigs(usageConfig, dialogConfig, messageConfig, config, options);
    const set = (config, keyPath, value) => {
      if ( value === undefined ) return;
      foundry.utils.setProperty(config, keyPath, value);
    };
    set(usageConfig, "create.summons", config.createSummons);
    set(usageConfig, "summons.profile", config.summonsProfile);
    set(usageConfig, "summons.creatureSize", config.summonsOptions?.creatureSize);
    set(usageConfig, "summons.creatureType", config.summonsOptions?.creatureType);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _prepareUsageConfig(config) {
    config = super._prepareUsageConfig(config);
    const summons = this.availableProfiles;
    config.create ??= {};
    config.create.summons ??= this.canSummon && canvas.scene && summons.length && this.summon.prompt;
    config.summons ??= {};
    config.summons.profile ??= summons[0]?._id ?? null;
    config.summons.creatureSize ??= this.creatureSizes.first() ?? null;
    config.summons.creatureType ??= this.creatureTypes.first() ?? null;
    return config;
  }

  /* -------------------------------------------- */

  /** @override */
  _usageChatButtons(message) {
    if ( !this.availableProfiles.length ) return super._usageChatButtons(message);
    return [{
      label: game.i18n.localize("DND5E.SUMMON.Action.Summon"),
      icon: '<i class="fa-solid fa-spaghetti-monster-flying" inert></i>',
      dataset: {
        action: "placeSummons"
      }
    }].concat(super._usageChatButtons(message));
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  shouldHideChatButton(button, message) {
    if ( button.dataset.action === "placeSummons" ) return !this.canSummon;
    return super.shouldHideChatButton(button, message);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _finalizeUsage(config, results) {
    await super._finalizeUsage(config, results);
    if ( config.create?.summons ) {
      try {
        results.summoned = await this.placeSummons(config.summons);
      } catch(err) {
        results.summoned = [];
        Hooks.onError("SummonActivity#use", err, { log: "error", notify: "error" });
      }
    }
  }

  /* -------------------------------------------- */
  /*  Summoning                                   */
  /* -------------------------------------------- */

  /**
   * Process for summoning actor to the scene.
   * @param {SummoningConfiguration} options  Configuration data for summoning behavior.
   * @returns {Token5e[]|void}
   */
  async placeSummons(options) {
    if ( !this.canSummon || !canvas.scene ) return;

    const profile = this.profiles.find(p => p._id === options?.profile);
    if ( !profile ) throw new Error(
      game.i18n.format("DND5E.SUMMON.Warning.NoProfile", { profileId: options.profile, item: this.item.name })
    );

    /**
     * A hook event that fires before summoning is performed.
     * @function dnd5e.preSummon
     * @memberof hookEvents
     * @param {SummonActivity} activity         The activity that is performing the summoning.
     * @param {SummonsProfile} profile          Profile used for summoning.
     * @param {SummoningConfiguration} options  Additional summoning options.
     * @returns {boolean}                       Explicitly return `false` to prevent summoning.
     */
    if ( Hooks.call("dnd5e.preSummon", this, profile, options) === false ) return;

    // Fetch the actor that will be summoned
    const summonUuid = this.summon.mode === "cr" ? await this.queryActor(profile) : profile.uuid;
    if ( !summonUuid ) return;
    const actor = await this.fetchActor(summonUuid);

    // Verify ownership of actor
    if ( !actor.isOwner ) {
      throw new Error(game.i18n.format("DND5E.SUMMON.Warning.NoOwnership", { actor: actor.name }));
    }

    const tokensData = [];
    const minimized = !this.actor?.sheet._minimized;
    await this.actor?.sheet.minimize();
    try {
      // Figure out where to place the summons
      const placements = await this.getPlacement(actor.prototypeToken, profile, options);

      for ( const placement of placements ) {
        // Prepare changes to actor data, re-calculating per-token for potentially random values
        const tokenUpdateData = {
          actor,
          placement,
          ...(await this.getChanges(actor, profile, options))
        };

        /**
         * A hook event that fires before a specific token is summoned. After placement has been determined but before
         * the final token data is constructed.
         * @function dnd5e.preSummonToken
         * @memberof hookEvents
         * @param {SummonActivity} activity         The activity that is performing the summoning.
         * @param {SummonsProfile} profile          Profile used for summoning.
         * @param {TokenUpdateData} config          Configuration for creating a modified token.
         * @param {SummoningConfiguration} options  Additional summoning options.
         * @returns {boolean}                       Explicitly return `false` to prevent this token from being summoned.
         */
        if ( Hooks.call("dnd5e.preSummonToken", this, profile, tokenUpdateData, options) === false ) continue;

        // Create a token document and apply updates
        const tokenData = await this.getTokenData(tokenUpdateData);

        /**
         * A hook event that fires after token creation data is prepared, but before summoning occurs.
         * @function dnd5e.summonToken
         * @memberof hookEvents
         * @param {SummonActivity} activity         The activity that is performing the summoning.
         * @param {SummonsProfile} profile          Profile used for summoning.
         * @param {object} tokenData                Data for creating a token.
         * @param {SummoningConfiguration} options  Additional summoning options.
         */
        Hooks.callAll("dnd5e.summonToken", this, profile, tokenData, options);

        tokensData.push(tokenData);
      }
    } finally {
      if ( minimized ) this.actor?.sheet.maximize();
    }

    const createdTokens = await canvas.scene.createEmbeddedDocuments("Token", tokensData);

    /**
     * A hook event that fires when summoning is complete.
     * @function dnd5e.postSummon
     * @memberof hookEvents
     * @param {SummonActivity} activity         The activity that is performing the summoning.
     * @param {SummonsProfile} profile          Profile used for summoning.
     * @param {Token5e[]} tokens                Tokens that have been created.
     * @param {SummoningConfiguration} options  Additional summoning options.
     */
    Hooks.callAll("dnd5e.postSummon", this, profile, createdTokens, options);

    return createdTokens;
  }

  /* -------------------------------------------- */

  /**
   * If actor to be summoned is in a compendium, create a local copy or use an already imported version if present.
   * @param {string} uuid  UUID of actor that will be summoned.
   * @returns {Actor5e}    Local copy of actor.
   */
  async fetchActor(uuid) {
    const actor = await fromUuid(uuid);
    if ( !actor ) throw new Error(game.i18n.format("DND5E.SUMMON.Warning.NoActor", { uuid }));

    const actorLink = actor.prototypeToken.actorLink;
    if ( !actor.pack && (!actorLink || actor.getFlag("dnd5e", "summon.origin") === this.item?.uuid )) return actor;

    // Search world actors to see if any usable summoned actor instances are present from prior summonings.
    // Linked actors must match the summoning origin (activity) to be considered.
    const localActor = game.actors.find(a =>
      // Has been cloned for summoning use
      a.getFlag("dnd5e", "summonedCopy")
      // Sourced from the desired actor UUID
      && (a._stats?.compendiumSource === uuid)
      // Unlinked or created from this activity's parent item specifically
      && ((a.getFlag("dnd5e", "summon.origin") === this.item?.uuid) || !a.prototypeToken.actorLink)
    );
    if ( localActor ) return localActor;

    // Check permissions to create actors before importing
    if ( !game.user.can("ACTOR_CREATE") ) throw new Error(game.i18n.localize("DND5E.SUMMON.Warning.CreateActor"));

    // No suitable world actor was found, create a new actor for this summoning instance.
    if ( actor.pack ) {
      // Template actor resides only in compendium, import the actor into the world and set the flag.
      return game.actors.importFromCompendium(game.packs.get(actor.pack), actor.id, {
        "flags.dnd5e.summonedCopy": true
      });
    } else {
      // Template actor (linked) found in world, create a copy for this user's item.
      return actor.clone({
        "flags.dnd5e.summonedCopy": true,
        "_stats.compendiumSource": actor.uuid
      }, {save: true});
    }
  }

  /* -------------------------------------------- */

  /**
   * Request a specific actor to summon from the player.
   * @param {SummonsProfile} profile  Profile used for summoning.
   * @returns {Promise<string|null>}  UUID of the concrete actor to summon or `null` if canceled.
   */
  async queryActor(profile) {
    const locked = {
      documentClass: "Actor",
      types: new Set(["npc"]),
      additional: {
        cr: { max: simplifyBonus(profile.cr, this.getRollData({ deterministic: true })) }
      }
    };
    if ( profile.types.size ) locked.additional.type = Array.from(profile.types).reduce((obj, type) => {
      obj[type] = 1;
      return obj;
    }, {});
    return CompendiumBrowser.selectOne({ filters: { locked } });
  }

  /* -------------------------------------------- */

  /**
   * Prepare the updates to apply to the summoned actor and its token.
   * @param {Actor5e} actor                   Actor that will be modified.
   * @param {SummonsProfile} profile          Summoning profile used to summon the actor.
   * @param {SummoningConfiguration} options  Configuration data for summoning behavior.
   * @returns {Promise<{actorChanges: object, tokenChanges: object}>}  Changes that will be applied to the actor,
   *                                                                   its items, and its token.
   */
  async getChanges(actor, profile, options) {
    const actorUpdates = { effects: [], items: [] };
    const tokenUpdates = {};
    const rollData = { ...this.getRollData(), summon: actor.getRollData() };
    const prof = rollData.attributes?.prof ?? 0;

    // Add flags
    actorUpdates["flags.dnd5e.summon"] = {
      level: this.relevantLevel,
      mod: rollData.mod,
      origin: this.item.uuid,
      activity: this.id,
      profile: profile._id
    };

    // Match proficiency
    if ( this.match.proficiency ) {
      const proficiencyEffect = new ActiveEffect({
        _id: staticID("dnd5eMatchProficiency"),
        changes: [{
          key: "system.attributes.prof",
          mode: CONST.ACTIVE_EFFECT_MODES.OVERRIDE,
          value: prof
        }],
        disabled: false,
        icon: "icons/skills/targeting/crosshair-bars-yellow.webp",
        name: game.i18n.localize("DND5E.SUMMON.FIELDS.match.proficiency.label")
      });
      actorUpdates.effects.push(proficiencyEffect.toObject());
    }

    // Add bonus to AC
    if ( this.bonuses.ac ) {
      const acBonus = new Roll(this.bonuses.ac, rollData);
      await acBonus.evaluate();
      if ( acBonus.total ) {
        if ( actor.system.attributes.ac.calc === "flat" ) {
          actorUpdates["system.attributes.ac.flat"] = (actor.system.attributes.ac.flat ?? 0) + acBonus.total;
        } else {
          actorUpdates.effects.push((new ActiveEffect({
            _id: staticID("dnd5eACBonus"),
            changes: [{
              key: "system.attributes.ac.bonus",
              mode: CONST.ACTIVE_EFFECT_MODES.ADD,
              value: acBonus.total
            }],
            disabled: false,
            icon: "icons/magic/defensive/shield-barrier-blue.webp",
            name: game.i18n.localize("DND5E.SUMMON.FIELDS.bonuses.ac.label")
          })).toObject());
        }
      }
    }

    // Add bonus to HD
    if ( this.bonuses.hd && (actor.type === "npc") ) {
      const hdBonus = new Roll(this.bonuses.hd, rollData);
      await hdBonus.evaluate();
      if ( hdBonus.total ) {
        actorUpdates.effects.push((new ActiveEffect({
          _id: staticID("dnd5eHDBonus"),
          changes: [{
            key: "system.attributes.hd.max",
            mode: CONST.ACTIVE_EFFECT_MODES.ADD,
            value: hdBonus.total
          }],
          disabled: false,
          icon: "icons/sundries/gaming/dice-runed-brown.webp",
          name: game.i18n.localize("DND5E.SUMMON.FIELDS.bonuses.hd.label")
        })).toObject());
      }
    }

    // Add bonus to HP
    if ( this.bonuses.hp ) {
      const hpBonus = new Roll(this.bonuses.hp, rollData);
      await hpBonus.evaluate();

      // If non-zero hp bonus, apply as needed for this actor.
      // Note: Only unlinked actors will have their current HP set to their new max HP
      if ( hpBonus.total ) {

        // Helper function for modifying max HP ('bonuses.overall' or 'max')
        const maxHpEffect = hpField => {
          return (new ActiveEffect({
            _id: staticID("dnd5eHPBonus"),
            changes: [{
              key: `system.attributes.hp.${hpField}`,
              mode: CONST.ACTIVE_EFFECT_MODES.ADD,
              value: hpBonus.total
            }],
            disabled: false,
            icon: "icons/magic/life/heart-glowing-red.webp",
            name: game.i18n.localize("DND5E.SUMMON.FIELDS.bonuses.hp.label")
          })).toObject();
        };

        if ( !foundry.utils.isEmpty(actor.classes) && !actor._source.system.attributes.hp.max ) {
          // Actor has classes without a hard-coded max -- apply bonuses to 'overall'
          actorUpdates.effects.push(maxHpEffect("bonuses.overall"));
        } else if ( actor.prototypeToken.actorLink ) {
          // Otherwise, linked actors boost HP via 'max' AE
          actorUpdates.effects.push(maxHpEffect("max"));
        } else {
          // Unlinked actors assumed to always be "fresh" copies with bonus HP added to both
          // Max HP and Current HP
          actorUpdates["system.attributes.hp.max"] = actor.system.attributes.hp.max + hpBonus.total;
          actorUpdates["system.attributes.hp.value"] = actor.system.attributes.hp.value + hpBonus.total;
        }
      }
    }

    // Change creature size
    if ( this.creatureSizes.size ) {
      const size = this.creatureSizes.has(options.creatureSize) ? options.creatureSize : this.creatureSizes.first();
      const config = CONFIG.DND5E.actorSizes[size];
      if ( config ) {
        actorUpdates["system.traits.size"] = size;
        tokenUpdates.width = config.token ?? 1;
        tokenUpdates.height = config.token ?? 1;
      }
    }

    // Change creature type
    if ( this.creatureTypes.size ) {
      const type = this.creatureTypes.has(options.creatureType) ? options.creatureType : this.creatureTypes.first();
      if ( actor.system.details?.race instanceof Item ) {
        actorUpdates.items.push({ _id: actor.system.details.race.id, "system.type.value": type });
      } else {
        actorUpdates["system.details.type.value"] = type;
      }
    }

    const attackDamageBonus = Roll.replaceFormulaData(this.bonuses.attackDamage ?? "", rollData);
    const saveDamageBonus = Roll.replaceFormulaData(this.bonuses.saveDamage ?? "", rollData);
    const healingBonus = Roll.replaceFormulaData(this.bonuses.healing ?? "", rollData);
    for ( const item of actor.items ) {
      if ( !item.system.activities?.size ) continue;
      const changes = [];

      // Match attacks
      if ( this.match.attacks && item.system.hasAttack ) {
        const ability = this.ability ?? this.item.abilityMod ?? rollData.attributes?.spellcasting;
        const actionType = item.system.activities.getByType("attack")[0].actionType;
        const typeMapping = { mwak: "msak", rwak: "rsak" };
        const parts = [
          rollData.abilities?.[ability]?.mod,
          prof,
          rollData.bonuses?.[typeMapping[actionType] ?? actionType]?.attack
        ].filter(p => p);
        changes.push({
          key: "system.attack.bonus",
          mode: CONST.ACTIVE_EFFECT_MODES.OVERRIDE,
          value: parts.join(" + ")
        }, {
          key: "system.attack.flat",
          mode: CONST.ACTIVE_EFFECT_MODES.OVERRIDE,
          value: true
        });
      }

      // Match saves
      if ( this.match.saves && item.hasSave ) {
        let dc = rollData.abilities?.[this.ability]?.dc ?? rollData.attributes.spell.dc;
        if ( this.item.type === "spell" ) {
          const ability = this.item.system.availableAbilities?.first();
          if ( ability ) dc = rollData.abilities[ability]?.dc ?? dc;
        }
        changes.push({
          key: "system.save.dc",
          mode: CONST.ACTIVE_EFFECT_MODES.OVERRIDE,
          value: dc
        }, {
          key: "system.save.scaling",
          mode: CONST.ACTIVE_EFFECT_MODES.OVERRIDE,
          value: "flat"
        });
      }

      // Damage bonus
      let damageBonus;
      if ( item.hasAttack ) damageBonus = attackDamageBonus;
      else if ( item.hasSave ) damageBonus = saveDamageBonus;
      else if ( item.isHealing ) damageBonus = healingBonus;
      if ( damageBonus && item.system.activities.find(a => a.damage?.parts?.length || a.healing?.formula) ) {
        changes.push({
          key: "system.damage.bonus",
          mode: CONST.ACTIVE_EFFECT_MODES.ADD,
          value: damageBonus
        });
      }

      if ( changes.length ) {
        const effect = (new ActiveEffect({
          _id: staticID("dnd5eItemChanges"),
          changes,
          disabled: false,
          icon: "icons/skills/melee/strike-slashes-orange.webp",
          name: game.i18n.localize("DND5E.SUMMON.ItemChanges.Label"),
          origin: this.uuid,
          type: "enchantment"
        })).toObject();
        actorUpdates.items.push({ _id: item.id, effects: [effect] });
      }
    }

    // Add applied effects
    actorUpdates.effects.push(...this.effects.map(e => e.effect?.toObject()).filter(e => e));

    return { actorUpdates, tokenUpdates };
  }

  /* -------------------------------------------- */

  /**
   * Determine where the summons should be placed on the scene.
   * @param {PrototypeToken} token            Token to be placed.
   * @param {SummonsProfile} profile          Profile used for summoning.
   * @param {SummoningConfiguration} options  Additional summoning options.
   * @returns {Promise<PlacementData[]>}
   */
  async getPlacement(token, profile, options) {
    // Ensure the token matches the final size
    if ( this.creatureSizes.size ) {
      const size = this.creatureSizes.has(options.creatureSize) ? options.creatureSize : this.creatureSizes.first();
      const config = CONFIG.DND5E.actorSizes[size];
      if ( config ) token = token.clone({ width: config.token ?? 1, height: config.token ?? 1 });
    }

    const rollData = this.getRollData();
    const count = new Roll(profile.count || "1", rollData);
    await count.evaluate();
    return TokenPlacement.place({ tokens: Array(parseInt(count.total)).fill(token) });
  }

  /* -------------------------------------------- */

  /**
   * Configuration for creating a modified token.
   *
   * @typedef {object} TokenUpdateData
   * @property {Actor5e} actor            Original actor from which the token will be created.
   * @property {PlacementData} placement  Information on the location to summon the token.
   * @property {object} tokenUpdates      Additional updates that will be applied to token data.
   * @property {object} actorUpdates      Updates that will be applied to actor delta.
   */

  /**
   * Create token data ready to be summoned.
   * @param {config} TokenUpdateData  Configuration for creating a modified token.
   * @returns {object}
   */
  async getTokenData({ actor, placement, tokenUpdates, actorUpdates }) {
    if ( actor.prototypeToken.randomImg && !game.user.can("FILES_BROWSE") ) {
      tokenUpdates.texture ??= {};
      tokenUpdates.texture.src ??= actor.img;
      ui.notifications.warn("DND5E.SUMMON.Warning.Wildcard", { localize: true });
    }

    delete placement.prototypeToken;
    const tokenDocument = await actor.getTokenDocument(foundry.utils.mergeObject(placement, tokenUpdates));

    // Linked summons require more explicit updates before token creation.
    // Unlinked summons can take actor delta directly.
    if ( tokenDocument.actorLink ) {
      const { effects, items, ...rest } = actorUpdates;
      await tokenDocument.actor.update(rest);
      await tokenDocument.actor.updateEmbeddedDocuments("Item", items);

      const { newEffects, oldEffects } = effects.reduce((acc, curr) => {
        const target = tokenDocument.actor.effects.get(curr._id) ? "oldEffects" : "newEffects";
        acc[target].push(curr);
        return acc;
      }, { newEffects: [], oldEffects: [] });

      await tokenDocument.actor.updateEmbeddedDocuments("ActiveEffect", oldEffects);
      await tokenDocument.actor.createEmbeddedDocuments("ActiveEffect", newEffects, {keepId: true});
    } else {
      tokenDocument.delta.updateSource(actorUpdates);
      if ( actor.prototypeToken.appendNumber ) TokenPlacement.adjustAppendedNumber(tokenDocument, placement);
    }

    return tokenDocument.toObject();
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /**
   * Handle placing a summons from the chat card.
   * @this {SummonActivity}
   * @param {PointerEvent} event     Triggering click event.
   * @param {HTMLElement} target     The capturing HTML element which defined a [data-action].
   * @param {ChatMessage5e} message  Message associated with the activation.
   */
  static async #placeSummons(event, target, message) {
    const config = {
      create: { summons: true },
      summons: {}
    };
    let needsConfiguration = false;

    // No profile specified and only one profile on item, use that one
    const profiles = this.availableProfiles;
    if ( profiles.length === 1 ) config.summons.profile = profiles[0]._id;
    else needsConfiguration = true;

    // More than one creature size or type requires configuration
    if ( (this.creatureSizes.size > 1) || (this.creatureTypes.size > 1) ) needsConfiguration = true;

    if ( needsConfiguration ) {
      try {
        await SummonUsageDialog.create(this, config, {
          button: {
            icon: "fa-solid fa-spaghetti-monster-flying",
            label: "DND5E.SUMMON.Action.Summon"
          },
          display: {
            all: false,
            create: { summons: true }
          }
        });
      } catch(err) {
        return;
      }
    }

    try {
      await this.placeSummons(config.summons);
    } catch(err) {
      Hooks.onError("SummonsActivity#placeSummons", err, { log: "error", notify: "error" });
    }
  }
}

/**
 * Sheet for the utility activity.
 */
class UtilitySheet extends ActivitySheet {

  /** @inheritDoc */
  static DEFAULT_OPTIONS = {
    classes: ["utility-activity"]
  };

  /* -------------------------------------------- */

  /** @inheritDoc */
  static PARTS = {
    ...super.PARTS,
    identity: {
      template: "systems/dnd5e/templates/activity/utility-identity.hbs",
      templates: super.PARTS.identity.templates
    },
    effect: {
      template: "systems/dnd5e/templates/activity/utility-effect.hbs",
      templates: super.PARTS.effect.templates
    }
  };
}

const { BooleanField: BooleanField$w, SchemaField: SchemaField$E, StringField: StringField$U } = foundry.data.fields;

/**
 * Data model for an utility activity.
 *
 * @property {object} roll
 * @property {string} roll.formula   Arbitrary formula that can be rolled.
 * @property {string} roll.name      Label for the rolling button.
 * @property {boolean} roll.prompt   Should the roll configuration dialog be displayed?
 * @property {boolean} roll.visible  Should the rolling button be visible to all players?
 */
class UtilityActivityData extends BaseActivityData {
  /** @inheritDoc */
  static defineSchema() {
    return {
      ...super.defineSchema(),
      roll: new SchemaField$E({
        formula: new FormulaField(),
        name: new StringField$U(),
        prompt: new BooleanField$w(),
        visible: new BooleanField$w()
      })
    };
  }

  /* -------------------------------------------- */
  /*  Data Migrations                             */
  /* -------------------------------------------- */

  /** @override */
  static transformTypeData(source, activityData, options) {
    return foundry.utils.mergeObject(activityData, {
      roll: {
        formula: source.system.formula ?? "",
        name: "",
        prompt: false,
        visible: false
      }
    });
  }
}

/**
 * Generic activity for applying effects and rolling an arbitrary die.
 */
class UtilityActivity extends ActivityMixin(UtilityActivityData) {
  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static LOCALIZATION_PREFIXES = [...super.LOCALIZATION_PREFIXES, "DND5E.UTILITY"];

  /* -------------------------------------------- */

  /** @inheritDoc */
  static metadata = Object.freeze(
    foundry.utils.mergeObject(super.metadata, {
      type: "utility",
      img: "systems/dnd5e/icons/svg/activity/utility.svg",
      title: "DND5E.UTILITY.Title",
      sheetClass: UtilitySheet,
      usage: {
        actions: {
          rollFormula: UtilityActivity.#rollFormula
        }
      }
    }, { inplace: false })
  );

  /* -------------------------------------------- */
  /*  Activation                                  */
  /* -------------------------------------------- */

  /** @override */
  _usageChatButtons(message) {
    if ( !this.roll.formula ) return super._usageChatButtons(message);
    return [{
      label: this.roll.name || game.i18n.localize("DND5E.Roll"),
      icon: '<i class="fa-solid fa-dice" inert></i>',
      dataset: {
        action: "rollFormula",
        visibility: this.roll.visible ? "all" : undefined
      }
    }].concat(super._usageChatButtons(message));
  }

  /* -------------------------------------------- */
  /*  Rolling                                     */
  /* -------------------------------------------- */

  /**
   * Roll the formula attached to this utility.
   * @param {BasicRollProcessConfiguration} [config]   Configuration information for the roll.
   * @param {BasicRollDialogConfiguration} [dialog]    Configuration for the roll dialog.
   * @param {BasicRollMessageConfiguration} [message]  Configuration for the roll message.
   * @returns {Promise<BasicRoll[]|void>}              The created Roll instances.
   */
  async rollFormula(config={}, dialog={}, message={}) {
    if ( !this.roll.formula ) {
      console.warn(`No formula defined for the activity ${this.name} on ${this.item.name} (${this.uuid}).`);
      return;
    }

    const rollConfig = foundry.utils.deepClone(config);
    rollConfig.hookNames = [...(config.hookNames ?? []), "formula"];
    rollConfig.rolls = [{ parts: [this.roll.formula], data: this.getRollData() }].concat(config.rolls ?? []);
    rollConfig.subject = this;

    const dialogConfig = foundry.utils.mergeObject({
      configure: this.roll.prompt,
      options: {
        window: {
          title: this.item.name,
          subtitle: "DND5E.RollConfiguration.Title",
          icon: this.item.img
        }
      }
    }, dialog);

    const messageConfig = foundry.utils.mergeObject({
      create: true,
      data: {
        flavor: `${this.item.name} - ${this.roll.label || game.i18n.localize("DND5E.OtherFormula")}`,
        flags: {
          dnd5e: {
            ...this.messageFlags,
            messageType: "roll",
            roll: { type: "generic" }
          }
        }
      }
    }, message);

    if ( "dnd5e.preRollFormula" in Hooks.events ) {
      foundry.utils.logCompatibilityWarning(
        "The `dnd5e.preRollFormula` hook has been deprecated and replaced with `dnd5e.preRollFormulaV2`.",
        { since: "DnD5e 4.0", until: "DnD5e 4.4" }
      );
      const hookData = {
        formula: rollConfig.rolls[0].parts[0], data: rollConfig.rolls[0].data, chatMessage: messageConfig.create
      };
      if ( Hooks.call("dnd5e.preRollFormula", this.item, hookData) === false ) return;
      rollConfig.rolls[0].parts[0] = hookData.formula;
      rollConfig.rolls[0].data = hookData.data;
      messageConfig.create = hookData.chatMessage;
    }

    const rolls = await CONFIG.Dice.BasicRoll.build(rollConfig, dialogConfig, messageConfig);
    if ( !rolls.length ) return;

    /**
     * A hook event that fires after a formula has been rolled for a Utility activity.
     * @function dnd5e.rollFormulaV2
     * @memberof hookEvents
     * @param {BasicRoll[]} rolls             The resulting rolls.
     * @param {object} data
     * @param {UtilityActivity} data.subject  The Activity that performed the roll.
     */
    Hooks.callAll("dnd5e.rollFormulaV2", rolls, { subject: this });

    if ( "dnd5e.rollFormula" in Hooks.events ) {
      foundry.utils.logCompatibilityWarning(
        "The `dnd5e.rollFormula` hook has been deprecated and replaced with `dnd5e.rollFormulaV2`.",
        { since: "DnD5e 4.0", until: "DnD5e 4.4" }
      );
      Hooks.callAll("dnd5e.rollFormula", this.item, rolls[0]);
    }

    return rolls;
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /**
   * Handle rolling the formula attached to this utility.
   * @this {UtilityActivity}
   * @param {PointerEvent} event     Triggering click event.
   * @param {HTMLElement} target     The capturing HTML element which defined a [data-action].
   * @param {ChatMessage5e} message  Message associated with the activation.
   */
  static #rollFormula(event, target, message) {
    this.rollFormula({ event });
  }
}

var _module$s = /*#__PURE__*/Object.freeze({
  __proto__: null,
  ActivityMixin: ActivityMixin,
  AttackActivity: AttackActivity,
  CastActivity: CastActivity,
  CheckActivity: CheckActivity,
  DamageActivity: DamageActivity,
  EnchantActivity: EnchantActivity,
  EnchantmentError: EnchantmentError$1,
  ForwardActivity: ForwardActivity,
  HealActivity: HealActivity,
  OrderActivity: OrderActivity,
  SaveActivity: SaveActivity,
  SummonActivity: SummonActivity,
  UtilityActivity: UtilityActivity
});

/**
 * Base configuration application for advancements that can be extended by other types to implement custom
 * editing interfaces.
 */
let AdvancementConfig$1 = class AdvancementConfig extends PseudoDocumentSheet {
  constructor(advancement={}, options={}) {
    if ( advancement instanceof dnd5e.documents.advancement.Advancement ) {
      options.document = advancement;
      // TODO: Add deprecation warning for this calling pattern once system has switched over to using the sheet
      // getter on Advancement, rather than creating separately
    } else options = { ...advancement, ...options };
    super(options);
  }

  /* -------------------------------------------- */

  /** @override */
  static DEFAULT_OPTIONS = {
    classes: ["advancement", "grid-columns"],
    window: {
      icon: "fa-solid fa-person-rays"
    },
    actions: {
      deleteItem: AdvancementConfig.#deleteDroppedItem
    },
    dropKeyPath: null,
    position: {
      width: 400
    }
  };

  /* -------------------------------------------- */

  /** @override */
  static PARTS = {
    config: {
      template: "systems/dnd5e/templates/advancement/advancement-controls-section.hbs"
    }
  };

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * The advancement being created or edited.
   * @type {Advancement}
   */
  get advancement() {
    return this.document;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  get title() {
    return this.advancement.constructor.metadata.title;
  }

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareContext(options) {
    const levels = Array.fromRange(CONFIG.DND5E.maxLevel + 1).map(l => ({ value: l, label: l }));
    if ( ["class", "subclass"].includes(this.item.type) ) delete levels[0];
    else levels[0].label = game.i18n.localize("DND5E.ADVANCEMENT.Config.AnyLevel");
    const context = {
      ...(await super._prepareContext(options)),
      advancement: this.advancement,
      configuration: {
        data: this.advancement.configuration,
        fields: this.advancement.configuration?.schema?.fields
      },
      fields: this.advancement.schema.fields,
      source: this.advancement._source,
      default: {
        title: this.advancement.constructor.metadata.title,
        icon: this.advancement.constructor.metadata.icon,
        hint: ""
      },
      levels,
      classRestrictionOptions: [
        { value: "", label: game.i18n.localize("DND5E.AdvancementClassRestrictionNone") },
        { value: "primary", label: game.i18n.localize("DND5E.AdvancementClassRestrictionPrimary") },
        { value: "secondary", label: game.i18n.localize("DND5E.AdvancementClassRestrictionSecondary") }
      ],
      showClassRestrictions: this.item.type === "class",
      showLevelSelector: !this.advancement.constructor.metadata.multiLevel
    };
    return context;
  }

  /* -------------------------------------------- */
  /*  Life-Cycle Handlers                         */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _onRender(context, options) {
    await super._onRender(context, options);
    new DragDrop({
      dragSelector: ".draggable",
      dropSelector: null,
      callbacks: {
        dragstart: this._onDragStart.bind(this),
        drop: this._onDrop.bind(this)
      }
    }).bind(this.element);
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /**
   * Handle deleting an existing Item entry from the Advancement.
   * @this {AdvancementConfig}
   * @param {Event} event         Triggering click event.
   * @param {HTMLElement} target  Button that was clicked.
   */
  static async #deleteDroppedItem(event, target) {
    const uuidToDelete = target.closest("[data-item-uuid]")?.dataset.itemUuid;
    if ( !uuidToDelete ) return;
    const items = foundry.utils.getProperty(this.advancement.configuration, this.options.dropKeyPath);
    const updates = { configuration: await this.prepareConfigurationUpdate({
      [this.options.dropKeyPath]: items.filter(i => i.uuid !== uuidToDelete)
    }) };
    await this.advancement.update(updates);
  }

  /* -------------------------------------------- */
  /*  Form Handling                               */
  /* -------------------------------------------- */

  /**
   * Perform any changes to configuration data before it is saved to the advancement.
   * @param {object} configuration  Configuration object.
   * @returns {object}              Modified configuration.
   */
  async prepareConfigurationUpdate(configuration) {
    return configuration;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _processSubmitData(event, submitData) {
    submitData.configuration ??= {};
    submitData.configuration = await this.prepareConfigurationUpdate(submitData.configuration);
    await this.advancement.update(submitData);
  }

  /* -------------------------------------------- */

  /**
   * Helper method to take an object and apply updates that remove any empty keys.
   * @param {object} object  Object to be cleaned.
   * @returns {object}       Copy of object with only non false-ish values included and others marked
   *                         using `-=` syntax to be removed by update process.
   * @protected
   */
  static _cleanedObject(object) {
    return Object.entries(object).reduce((obj, [key, value]) => {
      let keep = false;
      if ( foundry.utils.getType(value) === "Object" ) keep = Object.values(value).some(v => v);
      else if ( value ) keep = true;
      if ( keep ) obj[key] = value;
      else obj[`-=${key}`] = null;
      return obj;
    }, {});
  }

  /* -------------------------------------------- */
  /*  Drag & Drop                                 */
  /* -------------------------------------------- */

  /**
   * Handle beginning drag events on the sheet.
   * @param {DragEvent} event  The initiating drag start event.
   * @protected
   */
  async _onDragStart(event) {}

  /* -------------------------------------------- */

  /**
   * Handle dropping items onto the sheet.
   * @param {DragEvent} event  The concluding drag event.
   * @protected
   */
  async _onDrop(event) {
    if ( !this.options.dropKeyPath ) return;

    // Try to extract the data
    const data = TextEditor.getDragEventData(event);

    if ( data?.type !== "Item" ) return;
    const item = await Item.implementation.fromDropData(data);

    try {
      this._validateDroppedItem(event, item);
    } catch(err) {
      ui.notifications.error(err.message);
      return;
    }

    const existingItems = foundry.utils.getProperty(this.advancement.configuration, this.options.dropKeyPath);

    // Abort if this uuid is the parent item
    if ( item.uuid === this.item.uuid ) {
      ui.notifications.error("DND5E.ADVANCEMENT.ItemGrant.Warning.Recursive", {localize: true});
      return;
    }

    // Abort if this uuid exists already
    if ( existingItems.find(i => i.uuid === item.uuid) ) {
      ui.notifications.warn("DND5E.ADVANCEMENT.ItemGrant.Warning.Duplicate", {localize: true});
      return;
    }

    await this.advancement.update({[`configuration.${this.options.dropKeyPath}`]: [
      ...existingItems, { uuid: item.uuid }
    ]});
  }

  /* -------------------------------------------- */

  /**
   * Called when an item is dropped to validate the Item before it is saved. An error should be thrown
   * if the item is invalid.
   * @param {Event} event  Triggering drop event.
   * @param {Item5e} item  The materialized Item that was dropped.
   * @throws An error if the item is invalid.
   * @protected
   */
  _validateDroppedItem(event, item) {}
};

/**
 * Base class for the advancement interface displayed by the advancement prompt that should be subclassed by
 * individual advancement types.
 *
 * @param {Item5e} item           Item to which the advancement belongs.
 * @param {string} advancementId  ID of the advancement this flow modifies.
 * @param {number} level          Level for which to configure this flow.
 * @param {object} [options={}]   Application rendering options.
 */
class AdvancementFlow extends FormApplication {
  constructor(item, advancementId, level, options={}) {
    super({}, options);

    /**
     * The item that houses the Advancement.
     * @type {Item5e}
     */
    this.item = item;

    /**
     * ID of the advancement this flow modifies.
     * @type {string}
     * @private
     */
    this._advancementId = advancementId;

    /**
     * Level for which to configure this flow.
     * @type {number}
     */
    this.level = level;

    /**
     * Data retained by the advancement manager during a reverse step. If restoring data using Advancement#restore,
     * this data should be used when displaying the flow's form.
     * @type {object|null}
     */
    this.retainedData = null;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static get defaultOptions() {
    return foundry.utils.mergeObject(super.defaultOptions, {
      template: "systems/dnd5e/templates/advancement/advancement-flow.hbs",
      popOut: false
    });
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static _customElements = super._customElements.concat(["dnd5e-checkbox"]);

  /* -------------------------------------------- */

  /** @inheritDoc */
  get id() {
    return `actor-${this.advancement.item.id}-advancement-${this.advancement.id}-${this.level}`;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  get title() {
    return this.advancement.title;
  }

  /* -------------------------------------------- */

  /**
   * The Advancement object this flow modifies.
   * @type {Advancement|null}
   */
  get advancement() {
    return this.item.advancement?.byId[this._advancementId] ?? null;
  }

  /* -------------------------------------------- */

  /**
   * Set the retained data for this flow. This method gives the flow a chance to do any additional prep
   * work required for the retained data before the application is rendered.
   * @param {object} data  Retained data associated with this flow.
   */
  async retainData(data) {
    this.retainedData = data;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  getData() {
    return {
      appId: this.id,
      advancement: this.advancement,
      type: this.advancement.constructor.typeName,
      title: this.title,
      hint: this.advancement.hint,
      summary: this.advancement.summaryForLevel(this.level),
      level: this.level
    };
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _render(...args) {
    await super._render(...args);

    // Call setPosition on manager to adjust for size changes
    this.options.manager?.setPosition();
  }

  /* -------------------------------------------- */

  /**
   * Retrieve automatic application data from the advancement, if supported.
   * @returns {object|false}  Data to pass to the apply method, or `false` if advancement requirers user intervention.
   */
  getAutomaticApplicationValue() {
    return this.advancement.automaticApplicationValue(this.level);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _updateObject(event, formData) {
    await this.advancement.apply(this.level, formData);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _canDragDrop(selector) {
    return true;
  }

}

/**
 * Object describing the proficiency for a specific ability or skill.
 *
 * @param {number} proficiency   Actor's flat proficiency bonus based on their current level.
 * @param {number} multiplier    Value by which to multiply the actor's base proficiency value.
 * @param {boolean} [roundDown]  Should half-values be rounded up or down?
 */
class Proficiency {
  constructor(proficiency, multiplier, roundDown=true) {

    /**
     * Base proficiency value of the actor.
     * @type {number}
     * @private
     */
    this._baseProficiency = Number(proficiency ?? 0);

    /**
     * Value by which to multiply the actor's base proficiency value.
     * @type {number}
     */
    this.multiplier = Number(multiplier ?? 0);

    /**
     * Direction decimal results should be rounded ("up" or "down").
     * @type {string}
     */
    this.rounding = roundDown ? "down" : "up";
  }

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * Should only deterministic proficiency be returned, regardless of system settings?
   * @type {boolean}
   */
  deterministic = false;

  /* -------------------------------------------- */

  /**
   * Flat proficiency value regardless of proficiency mode.
   * @type {number}
   */
  get flat() {
    const roundMethod = (this.rounding === "down") ? Math.floor : Math.ceil;
    return roundMethod(this.multiplier * this._baseProficiency);
  }

  /* -------------------------------------------- */

  /**
   * Dice-based proficiency value regardless of proficiency mode.
   * @type {string}
   */
  get dice() {
    if ( (this._baseProficiency === 0) || (this.multiplier === 0) ) return "0";
    const roundTerm = (this.rounding === "down") ? "floor" : "ceil";
    if ( this.multiplier === 0.5 ) {
      return `${roundTerm}(1d${this._baseProficiency * 2} / 2)`;
    } else {
      return `${this.multiplier}d${this._baseProficiency * 2}`;
    }
  }

  /* -------------------------------------------- */

  /**
   * Either flat or dice proficiency term based on configured setting.
   * @type {string}
   */
  get term() {
    return (game.settings.get("dnd5e", "proficiencyModifier") === "dice") && !this.deterministic
      ? this.dice : String(this.flat);
  }

  /* -------------------------------------------- */

  /**
   * Whether the proficiency is greater than zero.
   * @type {boolean}
   */
  get hasProficiency() {
    return (this._baseProficiency > 0) && (this.multiplier > 0);
  }

  /* -------------------------------------------- */
  /*  Methods                                     */
  /* -------------------------------------------- */

  /**
   * Calculate an actor's proficiency modifier based on level or CR.
   * @param {number} level  Level or CR To use for calculating proficiency modifier.
   * @returns {number}      Proficiency modifier.
   */
  static calculateMod(level) {
    return Math.floor((level + 7) / 4);
  }

  /* -------------------------------------------- */

  /**
   * Return a clone of this proficiency with any changes applied.
   * @param {object} [updates={}]
   * @param {number} updates.proficiency  Actor's flat proficiency bonus based on their current level.
   * @param {number} updates.multiplier   Value by which to multiply the actor's base proficiency value.
   * @param {boolean} updates.roundDown   Should half-values be rounded up or down?
   * @returns {Proficiency}
   */
  clone({ proficiency, multiplier, roundDown }={}) {
    proficiency ??= this._baseProficiency;
    multiplier ??= this.multiplier;
    roundDown ??= this.rounding === "down";
    return new this.constructor(proficiency, multiplier, roundDown);
  }

  /* -------------------------------------------- */

  /**
   * Override the default `toString` method to return flat proficiency for backwards compatibility in formula.
   * @returns {string}  Either flat or dice proficiency term based on configured setting.
   */
  toString() {
    return this.term;
  }
}

/**
 * Data Model variant with some extra methods to support template mix-ins.
 *
 * **Note**: This uses some advanced Javascript techniques that are not necessary for most data models.
 * Please refer to the [advancement data models]{@link BaseAdvancement} for an example of a more typical usage.
 *
 * In template.json, each Actor or Item type can incorporate several templates which are chunks of data that are
 * common across all the types that use them. One way to represent them in the schema for a given Document type is to
 * duplicate schema definitions for the templates and write them directly into the Data Model for the Document type.
 * This works fine for small templates or systems that do not need many Document types but for more complex systems
 * this boilerplate can become prohibitive.
 *
 * Here we have opted to instead create a separate Data Model for each template available. These define their own
 * schemas which are then mixed-in to the final schema for the Document type's Data Model. A Document type Data Model
 * can define its own schema unique to it, and then add templates in direct correspondence to those in template.json
 * via SystemDataModel.mixin.
 */
class SystemDataModel extends foundry.abstract.TypeDataModel {

  /** @inheritDoc */
  static _enableV10Validation = true;

  /**
   * System type that this system data model represents (e.g. "character", "npc", "vehicle").
   * @type {string}
   */
  static _systemType;

  /* -------------------------------------------- */

  /**
   * Base templates used for construction.
   * @type {*[]}
   * @private
   */
  static _schemaTemplates = [];

  /* -------------------------------------------- */

  /**
   * The field names of the base templates used for construction.
   * @type {Set<string>}
   * @private
   */
  static get _schemaTemplateFields() {
    const fieldNames = Object.freeze(new Set(this._schemaTemplates.map(t => t.schema.keys()).flat()));
    Object.defineProperty(this, "_schemaTemplateFields", {
      value: fieldNames,
      writable: false,
      configurable: false
    });
    return fieldNames;
  }

  /* -------------------------------------------- */

  /**
   * A list of properties that should not be mixed-in to the final type.
   * @type {Set<string>}
   * @private
   */
  static _immiscible = new Set(["length", "mixed", "name", "prototype", "cleanData", "_cleanData",
    "_initializationOrder", "validateJoint", "_validateJoint", "migrateData", "_migrateData",
    "shimData", "_shimData", "defineSchema"]);

  /* -------------------------------------------- */

  /**
   * @typedef {object} SystemDataModelMetadata
   * @property {typeof DataModel} [systemFlagsModel]  Model that represents flags data within the dnd5e namespace.
   */

  /**
   * Metadata that describes this DataModel.
   * @type {SystemDataModelMetadata}
   */
  static metadata = Object.freeze({
    systemFlagsModel: null
  });

  get metadata() {
    return this.constructor.metadata;
  }

  /* -------------------------------------------- */

  /**
   * Filters available for this item type when using the compendium browser.
   * @returns {CompendiumBrowserFilterDefinition}
   */
  static get compendiumBrowserFilters() {
    return new Map();
  }

  /* -------------------------------------------- */

  /**
   * Key path to the description used for default embeds.
   * @type {string|null}
   */
  get embeddedDescriptionKeyPath() {
    return null;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static defineSchema() {
    const schema = {};
    for ( const template of this._schemaTemplates ) {
      if ( !template.defineSchema ) {
        throw new Error(`Invalid dnd5e template mixin ${template} defined on class ${this.constructor}`);
      }
      this.mergeSchema(schema, template.defineSchema());
    }
    return schema;
  }

  /* -------------------------------------------- */

  /**
   * Merge two schema definitions together as well as possible.
   * @param {DataSchema} a  First schema that forms the basis for the merge. *Will be mutated.*
   * @param {DataSchema} b  Second schema that will be merged in, overwriting any non-mergeable properties.
   * @returns {DataSchema}  Fully merged schema.
   */
  static mergeSchema(a, b) {
    Object.assign(a, b);
    return a;
  }

  /* -------------------------------------------- */
  /*  Data Cleaning                               */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static cleanData(source, options) {
    this._cleanData(source, options);
    return super.cleanData(source, options);
  }

  /* -------------------------------------------- */

  /**
   * Performs cleaning without calling DataModel.cleanData.
   * @param {object} [source]         The source data
   * @param {object} [options={}]     Additional options (see DataModel.cleanData)
   * @protected
   */
  static _cleanData(source, options) {
    for ( const template of this._schemaTemplates ) {
      template._cleanData(source, options);
    }
  }

  /* -------------------------------------------- */
  /*  Data Initialization                         */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static *_initializationOrder() {
    for ( const template of this._schemaTemplates ) {
      for ( const entry of template._initializationOrder() ) {
        entry[1] = this.schema.get(entry[0]);
        yield entry;
      }
    }
    for ( const entry of this.schema.entries() ) {
      if ( this._schemaTemplateFields.has(entry[0]) ) continue;
      yield entry;
    }
  }

  /* -------------------------------------------- */
  /*  Socket Event Handlers                       */
  /* -------------------------------------------- */

  /**
   * Pre-creation logic for this system data.
   * @param {object} data               The initial data object provided to the document creation request.
   * @param {object} options            Additional options which modify the creation request.
   * @param {User} user                 The User requesting the document creation.
   * @returns {Promise<boolean|void>}   A return value of false indicates the creation operation should be cancelled.
   * @see {Document#_preCreate}
   * @protected
   */
  async _preCreate(data, options, user) {
    const actor = this.parent.actor;
    if ( (actor?.type !== "character") || !this.metadata?.singleton ) return;
    if ( actor.itemTypes[data.type]?.length ) {
      ui.notifications.error(game.i18n.format("DND5E.ActorWarningSingleton", {
        itemType: game.i18n.localize(CONFIG.Item.typeLabels[data.type]),
        actorType: game.i18n.localize(CONFIG.Actor.typeLabels[actor.type])
      }));
      return false;
    }
  }

  /* -------------------------------------------- */
  /*  Data Validation                             */
  /* -------------------------------------------- */

  /** @inheritDoc */
  validate(options={}) {
    if ( this.constructor._enableV10Validation === false ) return true;
    return super.validate(options);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static validateJoint(data) {
    this._validateJoint(data);
    return super.validateJoint(data);
  }

  /* -------------------------------------------- */

  /**
   * Performs joint validation without calling DataModel.validateJoint.
   * @param {object} data     The source data
   * @throws                  An error if a validation failure is detected
   * @protected
   */
  static _validateJoint(data) {
    for ( const template of this._schemaTemplates ) {
      template._validateJoint(data);
    }
  }

  /* -------------------------------------------- */
  /*  Data Migration                              */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static migrateData(source) {
    this._migrateData(source);
    return super.migrateData(source);
  }

  /* -------------------------------------------- */

  /**
   * Performs migration without calling DataModel.migrateData.
   * @param {object} source     The source data
   * @protected
   */
  static _migrateData(source) {
    for ( const template of this._schemaTemplates ) {
      template._migrateData(source);
    }
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static shimData(data, options) {
    this._shimData(data, options);
    return super.shimData(data, options);
  }

  /* -------------------------------------------- */

  /**
   * Performs shimming without calling DataModel.shimData.
   * @param {object} data         The source data
   * @param {object} [options]    Additional options (see DataModel.shimData)
   * @protected
   */
  static _shimData(data, options) {
    for ( const template of this._schemaTemplates ) {
      template._shimData(data, options);
    }
  }

  /* -------------------------------------------- */
  /*  Mixins                                      */
  /* -------------------------------------------- */

  /**
   * Mix multiple templates with the base type.
   * @param {...*} templates            Template classes to mix.
   * @returns {typeof SystemDataModel}  Final prepared type.
   */
  static mixin(...templates) {
    for ( const template of templates ) {
      if ( !(template.prototype instanceof SystemDataModel) ) {
        throw new Error(`${template.name} is not a subclass of SystemDataModel`);
      }
    }

    const Base = class extends this {};
    Object.defineProperty(Base, "_schemaTemplates", {
      value: Object.seal([...this._schemaTemplates, ...templates]),
      writable: false,
      configurable: false
    });

    for ( const template of templates ) {
      // Take all static methods and fields from template and mix in to base class
      for ( const [key, descriptor] of Object.entries(Object.getOwnPropertyDescriptors(template)) ) {
        if ( this._immiscible.has(key) ) continue;
        Object.defineProperty(Base, key, descriptor);
      }

      // Take all instance methods and fields from template and mix in to base class
      for ( const [key, descriptor] of Object.entries(Object.getOwnPropertyDescriptors(template.prototype)) ) {
        if ( ["constructor"].includes(key) ) continue;
        Object.defineProperty(Base.prototype, key, descriptor);
      }
    }

    return Base;
  }

  /* -------------------------------------------- */
  /*  Helpers                                     */
  /* -------------------------------------------- */

  /** @override */
  async toEmbed(config, options={}) {
    const keyPath = this.embeddedDescriptionKeyPath;
    if ( !keyPath || !foundry.utils.hasProperty(this, keyPath) ) return null;
    const enriched = await TextEditor.enrichHTML(foundry.utils.getProperty(this, keyPath), {
      ...options,
      relativeTo: this.parent
    });
    const container = document.createElement("div");
    container.innerHTML = enriched;
    return container.children;
  }
}

/* -------------------------------------------- */

/**
 * Variant of the SystemDataModel with some extra actor-specific handling.
 */
class ActorDataModel extends SystemDataModel {

  /**
   * @typedef {SystemDataModelMetadata} ActorDataModelMetadata
   * @property {boolean} supportsAdvancement  Can advancement be performed for this actor type?
   */

  /** @type {ActorDataModelMetadata} */
  static metadata = Object.freeze(foundry.utils.mergeObject(super.metadata, {
    supportsAdvancement: false
  }, {inplace: false}));

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /** @override */
  get embeddedDescriptionKeyPath() {
    return "details.biography.value";
  }

  /* -------------------------------------------- */

  /**
   * Other actors that are available for currency transfers from this actor.
   * @type {Actor5e[]}
   */
  get transferDestinations() {
    const primaryParty = game.settings.get("dnd5e", "primaryParty")?.actor;
    if ( !primaryParty?.system.members.ids.has(this.parent.id) ) return [];
    const destinations = primaryParty.system.members.map(m => m.actor).filter(a => a.isOwner && a !== this.parent);
    if ( primaryParty.isOwner ) destinations.unshift(primaryParty);
    return destinations;
  }

  /* -------------------------------------------- */
  /*  Data Preparation                            */
  /* -------------------------------------------- */

  /**
   * Data preparation steps to perform after item data has been prepared, but before active effects are applied.
   */
  prepareEmbeddedData() {
    this._prepareScaleValues();
  }

  /* -------------------------------------------- */

  /**
   * Derive any values that have been scaled by the Advancement system.
   * Mutates the value of the `system.scale` object.
   * @protected
   */
  _prepareScaleValues() {
    this.scale = this.parent.items.reduce((scale, item) => {
      if ( CONFIG.DND5E.advancementTypes.ScaleValue.validItemTypes.has(item.type) ) {
        scale[item.identifier] = item.scaleValues;
      }
      return scale;
    }, {});
  }

  /* -------------------------------------------- */
  /*  Helpers                                     */
  /* -------------------------------------------- */

  /**
   * Prepare a data object which defines the data schema used by dice roll commands against this Actor.
   * @param {object} [options]
   * @param {boolean} [options.deterministic] Whether to force deterministic values for data properties that could be
   *                                          either a die term or a flat term.
   * @returns {object}
   */
  getRollData({ deterministic=false }={}) {
    const data = { ...this };
    data.prof = new Proficiency(this.attributes?.prof ?? 0, 1);
    data.prof.deterministic = deterministic;
    return data;
  }

  /* -------------------------------------------- */

  /**
   * Reset combat-related uses.
   * @param {string[]} periods               Which recovery periods should be considered.
   * @param {CombatRecoveryResults} results  Updates to perform on the actor and containing items.
   */
  async recoverCombatUses(periods, results) {}
}

/* -------------------------------------------- */

/**
 * Variant of the SystemDataModel with support for rich item tooltips.
 */
class ItemDataModel extends SystemDataModel {

  /**
   * @typedef {SystemDataModelMetadata} ItemDataModelMetadata
   * @property {boolean} enchantable    Can this item be modified by enchantment effects?
   * @property {boolean} inventoryItem  Should this item be listed with an actor's inventory?
   * @property {number} inventoryOrder  Order this item appears in the actor's inventory, smaller numbers are earlier.
   * @property {boolean} singleton      Should only a single item of this type be allowed on an actor?
   */

  /** @type {ItemDataModelMetadata} */
  static metadata = Object.freeze(foundry.utils.mergeObject(super.metadata, {
    enchantable: false,
    inventoryItem: false,
    inventoryOrder: Infinity,
    singleton: false
  }, {inplace: false}));

  /**
   * The handlebars template for rendering item tooltips.
   * @type {string}
   */
  static ITEM_TOOLTIP_TEMPLATE = "systems/dnd5e/templates/items/parts/item-tooltip.hbs";

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * Modes that can be used when making an attack with this item.
   * @type {FormSelectOption[]}
   */
  get attackModes() {
    return [];
  }

  /* -------------------------------------------- */

  /**
   * Set of abilities that can automatically be associated with this item.
   * @type {Set<string>|null}
   */
  get availableAbilities() {
    return null;
  }

  /* -------------------------------------------- */

  /** @override */
  get embeddedDescriptionKeyPath() {
    return game.user.isGM || (this.identified !== false) ? "description.value" : "unidentified.description";
  }

  /* -------------------------------------------- */

  /**
   * Scaling increase for this item type.
   * @type {number|null}
   */
  get scalingIncrease() {
    return null;
  }

  /* -------------------------------------------- */
  /*  Data Preparation                            */
  /* -------------------------------------------- */

  /** @inheritDoc */
  prepareBaseData() {
    if ( this.parent.isEmbedded && this.parent.actor?.items.has(this.parent.id) ) {
      const sourceId = this.parent.flags.dnd5e?.sourceId ?? this.parent._stats.compendiumSource
        ?? this.parent.flags.core?.sourceId;
      if ( sourceId ) this.parent.actor.sourcedItems?.set(sourceId, this.parent);
    }
  }

  /* -------------------------------------------- */
  /*  Helpers                                     */
  /* -------------------------------------------- */

  /**
   * Render a rich tooltip for this item.
   * @param {EnrichmentOptions} [enrichmentOptions={}]  Options for text enrichment.
   * @returns {{content: string, classes: string[]}}
   */
  async richTooltip(enrichmentOptions={}) {
    return {
      content: await renderTemplate(
        this.constructor.ITEM_TOOLTIP_TEMPLATE, await this.getCardData(enrichmentOptions)
      ),
      classes: ["dnd5e2", "dnd5e-tooltip", "item-tooltip"]
    };
  }

  /* -------------------------------------------- */

  /**
   * Prepare item card template data.
   * @param {EnrichmentOptions} [enrichmentOptions={}]  Options for text enrichment.
   * @param {Activity} [enrichmentOptions.activity]     Specific activity on item to use for customizing the data.
   * @returns {Promise<object>}
   */
  async getCardData({ activity, ...enrichmentOptions }={}) {
    const { name, type, img } = this.parent;
    let {
      price, weight, uses, identified, unidentified, description, school, materials
    } = this;
    const rollData = (activity ?? this.parent).getRollData();
    const isIdentified = identified !== false;
    const chat = isIdentified ? description.chat || description.value : unidentified?.description;
    description = game.user.isGM || isIdentified ? description.value : unidentified?.description;
    uses = this.hasLimitedUses && (game.user.isGM || identified) ? uses : null;
    price = game.user.isGM || identified ? price : null;

    const subtitle = [this.type?.label ?? game.i18n.localize(CONFIG.Item.typeLabels[this.parent.type])];
    const context = {
      name, type, img, price, weight, uses, school, materials,
      config: CONFIG.DND5E,
      controlHints: game.settings.get("dnd5e", "controlHints"),
      labels: foundry.utils.deepClone((activity ?? this.parent).labels),
      tags: this.parent.labels?.components?.tags,
      subtitle: subtitle.filterJoin(" &bull; "),
      description: {
        value: await TextEditor.enrichHTML(description ?? "", {
          rollData, relativeTo: this.parent, ...enrichmentOptions
        }),
        chat: await TextEditor.enrichHTML(chat ?? "", {
          rollData, relativeTo: this.parent, ...enrichmentOptions
        }),
        concealed: game.user.isGM && game.settings.get("dnd5e", "concealItemDescriptions") && !description.chat
      }
    };

    context.properties = [];

    if ( game.user.isGM || isIdentified ) {
      context.properties.push(
        ...this.cardProperties ?? [],
        ...Object.values((activity ? activity?.activationLabels : this.parent.labels.activations?.[0]) ?? {}),
        ...this.equippableItemCardProperties ?? []
      );
    }

    context.properties = context.properties.filter(_ => _);
    context.hasProperties = context.tags?.length || context.properties.length;
    return context;
  }

  /* -------------------------------------------- */

  /**
   * Determine the cost to craft this Item.
   * @param {object} [options]
   * @param {"buy"|"craft"|"none"} [options.baseItem="craft"]  Ignore base item if "none". Include full base item gold
   *                                                           price if "buy". Include base item craft costs if "craft".
   * @returns {Promise<{ days: number, gold: number }>}
   */
  async getCraftCost({ baseItem="craft" }={}) {
    let days = 0;
    let gold = 0;
    if ( !("price" in this) ) return { days, gold };
    const { price, type, rarity } = this;

    // Mundane Items
    if ( !this.properties.has("mgc") || !rarity ) {
      const { mundane } = CONFIG.DND5E.crafting;
      const valueInGP = price.valueInGP ?? 0;
      return { days: Math.ceil(valueInGP * mundane.days), gold: Math.floor(valueInGP * mundane.gold) };
    }

    const base = await getBaseItem(type.identifier ?? "", { fullItem: true });
    if ( base && (baseItem !== "none") ) {
      if ( baseItem === "buy" ) gold += base.system.price.valueInGP ?? 0;
      else {
        const costs = await base.system.getCraftCost();
        days += costs.days;
        gold += costs.gold;
      }
    }

    const { magic } = CONFIG.DND5E.crafting;
    if ( !(rarity in magic) ) return { days, gold };
    const costs = magic[rarity];
    return { days: days + costs.days, gold: gold + costs.gold };
  }

  /* -------------------------------------------- */

  /**
   * @typedef {object} FavoriteData5e
   * @property {string} img                  The icon path.
   * @property {string} title                The title.
   * @property {string|string[]} [subtitle]  An optional subtitle or several subtitle parts.
   * @property {number} [value]              A single value to display.
   * @property {number} [quantity]           The item's quantity.
   * @property {string|number} [modifier]    A modifier associated with the item.
   * @property {number} [passive]            A passive score associated with the item.
   * @property {object} [range]              The item's range.
   * @property {number} [range.value]        The first range increment.
   * @property {number|null} [range.long]    The second range increment.
   * @property {string} [range.units]        The range units.
   * @property {object} [save]               The item's saving throw.
   * @property {string} [save.ability]       The saving throw ability.
   * @property {number} [save.dc]            The saving throw DC.
   * @property {object} [uses]               Data on an item's uses.
   * @property {number} [uses.value]         The current available uses.
   * @property {number} [uses.max]           The maximum available uses.
   * @property {string} [uses.name]          The property to update on the item. If none is provided, the property will
   *                                         not be updatable.
   * @property {boolean} [toggle]            The effect's toggle state.
   * @property {boolean} [suppressed]        Whether the favorite is suppressed.
   */

  /**
   * Prepare item favorite data.
   * @returns {Promise<FavoriteData5e>}
   */
  async getFavoriteData() {
    return {
      img: this.parent.img,
      title: this.parent.name,
      subtitle: game.i18n.localize(CONFIG.Item.typeLabels[this.parent.type])
    };
  }

  /* -------------------------------------------- */

  /**
   * Prepare type-specific data for the Item sheet.
   * @param {object} context  Sheet context data.
   * @returns {Promise<void>}
   */
  async getSheetData(context) {}

  /* -------------------------------------------- */

  /**
   * Prepare a data object which defines the data schema used by dice roll commands against this Item.
   * @param {object} [options]
   * @param {boolean} [options.deterministic] Whether to force deterministic values for data properties that could be
   *                                          either a die term or a flat term.
   * @returns {object}
   */
  getRollData({ deterministic=false }={}) {
    const actorRollData = this.parent.actor?.getRollData({ deterministic }) ?? {};
    const data = { ...actorRollData, item: { ...this } };
    return data;
  }
}

/* -------------------------------------------- */

/**
 * Data Model variant that does not export fields with an `undefined` value during `toObject(true)`.
 */
class SparseDataModel extends foundry.abstract.DataModel {
  /** @inheritDoc */
  toObject(source=true) {
    if ( !source ) return super.toObject(source);
    const clone = foundry.utils.flattenObject(this._source);
    // Remove any undefined keys from the source data
    Object.keys(clone).filter(k => clone[k] === undefined).forEach(k => delete clone[k]);
    return foundry.utils.expandObject(clone);
  }
}

/**
 * Data field that automatically selects the Advancement-specific configuration or value data models.
 *
 * @param {Advancement} advancementType  Advancement class to which this field belongs.
 */
class AdvancementDataField extends foundry.data.fields.ObjectField {
  constructor(advancementType, options={}) {
    super(options);
    this.advancementType = advancementType;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static get _defaults() {
    return foundry.utils.mergeObject(super._defaults, {required: true});
  }

  /**
   * Get the DataModel definition for the specified field as defined in metadata.
   * @returns {typeof DataModel|null}  The DataModel class, or null.
   */
  getModel() {
    return this.advancementType.metadata?.dataModels?.[this.name];
  }

  /* -------------------------------------------- */

  /**
   * Get the defaults object for the specified field as defined in metadata.
   * @returns {object}
   */
  getDefaults() {
    return this.advancementType.metadata?.defaults?.[this.name] ?? {};
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _cleanType(value, options) {
    if ( !(typeof value === "object") ) value = {};

    // Use a defined DataModel
    const cls = this.getModel();
    if ( cls ) return cls.cleanData(value, options);
    if ( options.partial ) return value;

    // Use the defined defaults
    const defaults = this.getDefaults();
    return foundry.utils.mergeObject(defaults, value, {inplace: false});
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  initialize(value, model, options={}) {
    const cls = this.getModel();
    if ( cls ) return new cls(value, {parent: model, ...options});
    return foundry.utils.deepClone(value);
  }

  /* -------------------------------------------- */

  /**
   * Migrate this field's candidate source data.
   * @param {object} sourceData   Candidate source data of the root model
   * @param {any} fieldData       The value of this field within the source data
   */
  migrateSource(sourceData, fieldData) {
    const cls = this.getModel();
    if ( cls ) cls.migrateDataSafe(fieldData);
  }
}

const { DocumentIdField: DocumentIdField$4, FilePathField, NumberField: NumberField$A, StringField: StringField$T } = foundry.data.fields;

/**
 * Base data model for advancement.
 *
 * @property {string} _id               The advancement's ID.
 * @property {string} type              Type of advancement.
 * @property {*} configuration          Type-specific configuration data.
 * @property {*} value                  Type-specific value data after the advancement is applied.
 * @property {number} level             For single-level advancement, the level at which it should apply.
 * @property {string} title             Optional custom title.
 * @property {string} hint              Brief description of what the advancement does or guidance for the player.
 * @property {string} icon              Optional custom icon.
 * @property {string} classRestriction  Should this advancement apply at all times, only when on the first class on
 *                                      an actor, or only on a class that is multi-classing?
 */
class BaseAdvancement extends SparseDataModel {

  /**
   * Name of this advancement type that will be stored in config and used for lookups.
   * @type {string}
   * @protected
   */
  static get typeName() {
    return this.name.replace(/Advancement$/, "");
  }

  /* -------------------------------------------- */

  /** @override */
  static defineSchema() {
    return {
      _id: new DocumentIdField$4({initial: () => foundry.utils.randomID()}),
      type: new StringField$T({
        required: true, initial: this.typeName, validate: v => v === this.typeName,
        validationError: `must be the same as the Advancement type name ${this.typeName}`
      }),
      configuration: new AdvancementDataField(this, {required: true}),
      value: new AdvancementDataField(this, {required: true}),
      level: new NumberField$A({
        integer: true, initial: this.metadata?.multiLevel ? undefined : 0, min: 0, label: "DND5E.Level"
      }),
      title: new StringField$T({initial: undefined, label: "DND5E.AdvancementCustomTitle"}),
      hint: new StringField$T({label: "DND5E.AdvancementHint"}),
      icon: new FilePathField({
        initial: undefined, categories: ["IMAGE"], label: "DND5E.AdvancementCustomIcon", base64: true
      }),
      classRestriction: new StringField$T({
        initial: undefined, choices: ["primary", "secondary"], label: "DND5E.AdvancementClassRestriction"
      })
    };
  }

  /* -------------------------------------------- */
  /*  Data Migration                              */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static migrateData(source) {
    super.migrateData(source);
    if ( source.configuration?.hint ) source.hint = source.configuration.hint;
    return source;
  }
}

/**
 * @import { PseudoDocumentsMetadata } from "../mixins/pseudo-document.mjs";
 */

/**
 * Error that can be thrown during the advancement update preparation process.
 */
class AdvancementError extends Error {
  constructor(...args) {
    super(...args);
    this.name = "AdvancementError";
  }
}

/**
 * Abstract base class which various advancement types can subclass.
 * @param {Item5e} item          Item to which this advancement belongs.
 * @param {object} [data={}]     Raw data stored in the advancement object.
 * @param {object} [options={}]  Options which affect DataModel construction.
 * @abstract
 */
class Advancement extends PseudoDocumentMixin(BaseAdvancement) {
  constructor(data, {parent=null, ...options}={}) {
    if ( parent instanceof Item ) parent = parent.system;
    super(data, {parent, ...options});

    /**
     * A collection of Application instances which should be re-rendered whenever this document is updated.
     * The keys of this object are the application ids and the values are Application instances. Each
     * Application in this object will have its render method called by {@link Document#render}.
     * @type {Object<Application>}
     */
    Object.defineProperty(this, "apps", {
      value: {},
      writable: false,
      enumerable: false
    });
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _initialize(options) {
    super._initialize(options);
    return this.prepareData();
  }

  static ERROR = AdvancementError;

  /* -------------------------------------------- */

  /**
   * Information on how an advancement type is configured.
   *
   * @typedef {PseudoDocumentsMetadata} AdvancementMetadata
   * @property {object} dataModels
   * @property {DataModel} configuration  Data model used for validating configuration data.
   * @property {DataModel} value          Data model used for validating value data.
   * @property {number} order          Number used to determine default sorting order of advancement items.
   * @property {string} icon           Icon used for this advancement type if no user icon is specified.
   * @property {string} typeIcon       Icon used when selecting this advancement type during advancement creation.
   * @property {string} title          Title to be displayed if no user title is specified.
   * @property {string} hint           Description of this type shown in the advancement selection dialog.
   * @property {boolean} multiLevel    Can this advancement affect more than one level? If this is set to true,
   *                                   the level selection control in the configuration window is hidden and the
   *                                   advancement should provide its own implementation of `Advancement#levels`
   *                                   and potentially its own level configuration interface.
   * @property {Set<string>} validItemTypes  Set of types to which this advancement can be added. (deprecated)
   * @property {object} apps
   * @property {*} apps.config         Subclass of AdvancementConfig that allows for editing of this advancement type.
   * @property {*} apps.flow           Subclass of AdvancementFlow that is displayed while fulfilling this advancement.
   */

  /**
   * Configuration information for this advancement type.
   * @type {AdvancementMetadata}
   */
  static get metadata() {
    return {
      name: "Advancement",
      label: "DOCUMENT.DND5E.Advancement",
      order: 100,
      icon: "icons/svg/upgrade.svg",
      typeIcon: "icons/svg/upgrade.svg",
      title: game.i18n.localize("DND5E.AdvancementTitle"),
      hint: "",
      multiLevel: false,
      validItemTypes: new Set(["background", "class", "race", "subclass"]),
      apps: {
        config: AdvancementConfig$1,
        flow: AdvancementFlow
      }
    };
  }

  /* -------------------------------------------- */

  /**
   * Perform the pre-localization of this data model.
   */
  static localize() {
    Localization.localizeDataModel(this);
    if ( this.metadata.dataModels?.configuration ) {
      Localization.localizeDataModel(this.metadata.dataModels.configuration);
    }
    if ( this.metadata.dataModels?.value ) {
      Localization.localizeDataModel(this.metadata.dataModels.value);
    }
  }

  /* -------------------------------------------- */
  /*  Instance Properties                         */
  /* -------------------------------------------- */

  /**
   * List of levels in which this advancement object should be displayed. Will be a list of class levels if this
   * advancement is being applied to classes or subclasses, otherwise a list of character levels.
   * @returns {number[]}
   */
  get levels() {
    return this.level !== undefined ? [this.level] : [];
  }

  /* -------------------------------------------- */

  /**
   * Should this advancement be applied to a class based on its class restriction setting? This will always return
   * true for advancements that are not within an embedded class item.
   * @type {boolean}
   * @protected
   */
  get appliesToClass() {
    const originalClass = this.item.isOriginalClass;
    return !this.classRestriction
      || (this.classRestriction === "primary" && [true, null].includes(originalClass))
      || (this.classRestriction === "secondary" && !originalClass);
  }

  /* -------------------------------------------- */
  /*  Preparation Methods                         */
  /* -------------------------------------------- */

  /**
   * Prepare data for the Advancement.
   */
  prepareData() {
    this.title = this.title || this.constructor.metadata.title;
    this.icon = this.icon || this.constructor.metadata.icon;
  }

  /* -------------------------------------------- */

  /**
   * Perform preliminary operations before an Advancement is created.
   * @param {object} data      The initial data object provided to the document creation request.
   * @returns {boolean|void}   A return value of false indicates the creation operation should be cancelled.
   * @protected
   */
  _preCreate(data) {
    if ( !["class", "subclass"].includes(this.item.type)
      || foundry.utils.hasProperty(data, "level")
      || this.constructor.metadata.multiLevel ) return;
    this.updateSource({level: 1});
  }

  /* -------------------------------------------- */
  /*  Display Methods                             */
  /* -------------------------------------------- */

  /**
   * Has the player made choices for this advancement at the specified level?
   * @param {number} level  Level for which to check configuration.
   * @returns {boolean}     Have any available choices been made?
   */
  configuredForLevel(level) {
    return true;
  }

  /* -------------------------------------------- */

  /**
   * Value used for sorting this advancement at a certain level.
   * @param {number} level  Level for which this entry is being sorted.
   * @returns {string}      String that can be used for sorting.
   */
  sortingValueForLevel(level) {
    return `${this.constructor.metadata.order.paddedString(4)} ${this.titleForLevel(level)}`;
  }

  /* -------------------------------------------- */

  /**
   * Title displayed in advancement list for a specific level.
   * @param {number} level                           Level for which to generate a title.
   * @param {object} [options={}]
   * @param {boolean} [options.legacyDisplay=false]  Use legacy formatting?
   * @param {boolean} [options.configMode=false]     Is the advancement's item sheet in configuration mode? When in
   *                                                 config mode, the choices already made on this actor should not
   *                                                 be displayed.
   * @returns {string}                               HTML title with any level-specific information.
   */
  titleForLevel(level, options={}) {
    return this.title;
  }

  /* -------------------------------------------- */

  /**
   * Summary content displayed beneath the title in the advancement list.
   * @param {number} level                           Level for which to generate the summary.
   * @param {object} [options={}]
   * @param {boolean} [options.legacyDisplay=false]  Use legacy formatting?
   * @param {boolean} [options.configMode=false]     Is the advancement's item sheet in configuration mode? When in
   *                                                 config mode, the choices already made on this actor should not
   *                                                 be displayed.
   * @returns {string}                               HTML content of the summary.
   */
  summaryForLevel(level, options={}) {
    return "";
  }

  /* -------------------------------------------- */
  /*  Editing Methods                             */
  /* -------------------------------------------- */

  /**
   * Can an advancement of this type be added to the provided item?
   * @param {Item5e} item  Item to check against.
   * @returns {boolean}    Should this be enabled as an option on the `AdvancementSelection` dialog?
   */
  static availableForItem(item) {
    return true;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async delete(options={}) {
    if ( this.item.actor?.system.metadata?.supportsAdvancement
        && !game.settings.get("dnd5e", "disableAdvancements") ) {
      const manager = dnd5e.applications.advancement.AdvancementManager
        .forDeletedAdvancement(this.item.actor, this.item.id, this.id);
      if ( manager.steps.length ) return manager.render(true);
    }
    return super.delete(options);
  }

  /* -------------------------------------------- */
  /*  Application Methods                         */
  /* -------------------------------------------- */

  /**
   * Locally apply this advancement to the actor.
   * @param {number} level   Level being advanced.
   * @param {object} data    Data from the advancement form.
   * @abstract
   */
  async apply(level, data) { }


  /* -------------------------------------------- */

  /**
   * Retrieves the data to pass to the apply method in order to apply this advancement automatically, if possible.
   * @param {number} level    Level being advanced.
   * @returns {object|false}  Data to pass to the apply method, or `false` if advancement requirers user intervention.
   */
  automaticApplicationValue(level) {
    return false;
  }

  /* -------------------------------------------- */

  /**
   * Locally apply this advancement from stored data, if possible. If stored data can not be restored for any reason,
   * throw an AdvancementError to display the advancement flow UI.
   * @param {number} level  Level being advanced.
   * @param {object} data   Data from `Advancement#reverse` needed to restore this advancement.
   * @abstract
   */
  async restore(level, data) { }

  /* -------------------------------------------- */

  /**
   * Locally remove this advancement's changes from the actor.
   * @param {number} level  Level being removed.
   * @returns {object}      Data that can be passed to the `Advancement#restore` method to restore this reversal.
   * @abstract
   */
  async reverse(level) { }

  /* -------------------------------------------- */

  /**
   * Fetch an item and create a clone with the proper flags.
   * @param {string} uuid  UUID of the item to fetch.
   * @param {string} [id]  Optional ID to use instead of a random one.
   * @returns {object|null}
   */
  async createItemData(uuid, id) {
    const source = await fromUuid(uuid);
    if ( !source ) return null;
    const { _stats } = game.items.fromCompendium(source);
    return source.clone({
      _stats,
      _id: id ?? foundry.utils.randomID(),
      "flags.dnd5e.sourceId": uuid,
      "flags.dnd5e.advancementOrigin": `${this.item.id}.${this.id}`
    }, { keepId: true }).toObject();
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /**
   * Construct context menu options for this Activity.
   * @returns {ContextMenuEntry[]}
   */
  getContextMenuOptions() {
    if ( this.item.isOwner && !this.item[game.release.generation < 13 ? "compendium" : "collection"]?.locked ) return [
      {
        name: "DND5E.ADVANCEMENT.Action.Edit",
        icon: "<i class='fas fa-edit fa-fw'></i>",
        callback: () => this.sheet?.render(true)
      },
      {
        name: "DND5E.ADVANCEMENT.Action.Duplicate",
        icon: "<i class='fas fa-copy fa-fw'></i>",
        condition: li => this?.constructor.availableForItem(this.item),
        callback: () => {
          const createData = this.toObject();
          delete createData._id;
          this.item.createAdvancement(createData.type, createData, { renderSheet: false });
        }
      },
      {
        name: "DND5E.ADVANCEMENT.Action.Delete",
        icon: "<i class='fas fa-trash fa-fw'></i>",
        callback: () => this.deleteDialog()
      }
    ];

    return [{
      name: "DND5E.ADVANCEMENT.Action.View",
      icon: "<i class='fas fa-eye fa-fw'></i>",
      callback: () => this.sheet?.render(true)
    }];
  }

  /* -------------------------------------------- */

  /**
   * Handle context menu events on activities.
   * @param {Item5e} item         The Item the Activity belongs to.
   * @param {HTMLElement} target  The element the menu was triggered on.
   */
  static onContextMenu(item, target) {
    const { id } = target.closest("[data-id]")?.dataset ?? {};
    const advancement = item.advancement?.byId[id];
    if ( !advancement ) return;
    const menuItems = advancement.getContextMenuOptions();

    /**
     * A hook even that fires when the context menu for an Advancement is opened.
     * @function dnd5e.getItemAdvancementContext
     * @memberof hookEvents
     * @param {Advancement} advancement       The Advancement.
     * @param {HTMLElement} target            The element that menu was triggered on.
     * @param {ContextMenuEntry[]} menuItems  The context menu entries.
     */
    Hooks.callAll("dnd5e.getItemAdvancementContext", advancement, target, menuItems);
    ui.context.menuItems = menuItems;
  }
}

/**
 * Configuration application for ability score improvements.
 */
class AbilityScoreImprovementConfig extends AdvancementConfig$1 {
  /** @override */
  static DEFAULT_OPTIONS = {
    classes: ["ability-score-improvement"],
    actions: {
      decrease: AbilityScoreImprovementConfig.#adjustValue,
      increase: AbilityScoreImprovementConfig.#adjustValue,
      lock: AbilityScoreImprovementConfig.#lockValue
    }
  };

  /* -------------------------------------------- */

  /** @inheritDoc */
  static PARTS = {
    ...super.PARTS,
    details: {
      template: "systems/dnd5e/templates/advancement/ability-score-improvement-config-details.hbs"
    },
    scores: {
      template: "systems/dnd5e/templates/advancement/ability-score-improvement-config-scores.hbs",
      templates: ["systems/dnd5e/templates/advancement/parts/advancement-ability-score-control-v2.hbs"]
    }
  };

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareContext(options) {
    const context = await super._prepareContext(options);

    context.abilities = Object.entries(CONFIG.DND5E.abilities).reduce((obj, [key, data]) => {
      if ( !this.advancement.canImprove(key) ) return obj;
      const fixed = this.advancement.configuration.fixed[key] ?? 0;
      const locked = this.advancement.configuration.locked.has(key);
      obj[key] = {
        key,
        name: `configuration.fixed.${key}`,
        label: data.label,
        locked: {
          value: locked,
          hint: `DND5E.ADVANCEMENT.AbilityScoreImprovement.FIELDS.locked.${locked ? "locked" : "unlocked"}`
        },
        value: fixed,
        canIncrease: true,
        canDecrease: true
      };
      return obj;
    }, {});

    context.points = {
      key: "points",
      name: "configuration.points",
      label: game.i18n.localize("DND5E.ADVANCEMENT.AbilityScoreImprovement.FIELDS.points.label"),
      min: 0,
      value: this.advancement.configuration.points,
      canIncrease: true,
      canDecrease: this.advancement.configuration.points > 0
    };

    return context;
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /**
   * Handle clicking the plus and minus buttons.
   * @this {AbilityScoreImprovementConfig}
   * @param {Event} event         Triggering click event.
   * @param {HTMLElement} target  Button that was clicked.
   */
  static #adjustValue(event, target) {
    const action = target.dataset.action;
    const input = target.closest("li").querySelector("input[type=number]");

    if ( action === "decrease" ) input.valueAsNumber -= 1;
    else if ( action === "increase" ) input.valueAsNumber += 1;

    this.submit();
  }

  /* -------------------------------------------- */

  /**
   * Handle locking or unlocking an ability.
   * @this {AbilityScoreImprovementConfig}
   * @param {PointerEvent} event  The triggering event.
   * @param {HTMLElement} target  The action target.
   */
  static #lockValue(event, target) {
    const parent = target.closest("[data-score]");
    const { score } = parent.dataset;
    const input = parent.querySelector(`[name="configuration.locked.${score}"]`);
    input.value = input.value === "true" ? "false" : "true";
    this.submit();
  }

  /* -------------------------------------------- */
  /*  Form Handling                               */
  /* -------------------------------------------- */

  /** @override */
  async prepareConfigurationUpdate(configuration) {
    configuration.locked = Object.entries(configuration.locked).reduce((arr, [k, v]) => {
      if ( v ) arr.push(k);
      return arr;
    }, []);
    return configuration;
  }
}

/**
 * Inline application that presents the player with a choice between ability score improvement and taking a feat.
 */
class AbilityScoreImprovementFlow extends AdvancementFlow {

  /**
   * Player assignments to abilities.
   * @type {Object<string, number>}
   */
  assignments = {};

  /* -------------------------------------------- */

  /**
   * The dropped feat item.
   * @type {Item5e}
   */
  feat;

  /* -------------------------------------------- */

  /** @inheritDoc */
  static _customElements = super._customElements.concat(["dnd5e-checkbox"]);

  /* -------------------------------------------- */

  /** @inheritDoc */
  static get defaultOptions() {
    return foundry.utils.mergeObject(super.defaultOptions, {
      dragDrop: [{ dropSelector: "form" }],
      template: "systems/dnd5e/templates/advancement/ability-score-improvement-flow.hbs"
    });
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async retainData(data) {
    await super.retainData(data);
    this.assignments = this.retainedData.assignments ?? {};
    const featUuid = Object.values(this.retainedData.feat ?? {})[0];
    if ( featUuid ) this.feat = await fromUuid(featUuid);
    else if ( !foundry.utils.isEmpty(this.assignments) ) this.feat = { isASI: true };
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async getData() {
    const points = {
      assigned: Object.keys(CONFIG.DND5E.abilities).reduce((assigned, key) => {
        if ( !this.advancement.canImprove(key) || this.advancement.configuration.locked.has(key) ) return assigned;
        return assigned + (this.assignments[key] ?? 0);
      }, 0),
      cap: this.advancement.configuration.cap ?? Infinity,
      total: this.advancement.configuration.points
    };
    points.available = points.total - points.assigned;

    const formatter = new Intl.NumberFormat(game.i18n.lang, { signDisplay: "always" });

    const lockImprovement = this.feat && !this.feat.isASI;
    const abilities = Object.entries(CONFIG.DND5E.abilities).reduce((obj, [key, data]) => {
      if ( !this.advancement.canImprove(key) ) return obj;
      const ability = this.advancement.actor.system.abilities[key];
      const assignment = this.assignments[key] ?? 0;
      const fixed = this.advancement.configuration.fixed[key] ?? 0;
      const locked = this.advancement.configuration.locked.has(key);
      const value = Math.min(ability.value + fixed + assignment, ability.max);
      const max = locked ? value : Math.min(value + points.available, ability.max);
      const min = Math.min(ability.value + fixed, ability.max);
      obj[key] = {
        key, max, min, value,
        name: `abilities.${key}`,
        label: data.label,
        initial: ability.value + fixed,
        delta: (value - ability.value) ? formatter.format(value - ability.value) : null,
        showDelta: true,
        isDisabled: lockImprovement,
        isLocked: !!locked || (ability.value >= ability.max),
        canIncrease: (value < max) && ((fixed + assignment) < points.cap) && !locked && !lockImprovement,
        canDecrease: (value > (ability.value + fixed)) && !locked && !lockImprovement
      };
      return obj;
    }, {});

    const modernRules = game.settings.get("dnd5e", "rulesVersion") === "modern";
    const pluralRules = new Intl.PluralRules(game.i18n.lang);
    return foundry.utils.mergeObject(super.getData(), {
      abilities, lockImprovement, points,
      feat: this.feat,
      pointCap: game.i18n.format(
        `DND5E.ADVANCEMENT.AbilityScoreImprovement.CapDisplay.${pluralRules.select(points.cap)}`, { points: points.cap }
      ),
      pointsRemaining: game.i18n.format(
        `DND5E.ADVANCEMENT.AbilityScoreImprovement.PointsRemaining.${pluralRules.select(points.available)}`,
        {points: points.available}
      ),
      showASIFeat: modernRules && this.advancement.allowFeat,
      showImprovement: !modernRules || !this.advancement.allowFeat || this.feat?.isASI,
      staticIncrease: !this.advancement.configuration.points
    });
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  activateListeners(html) {
    super.activateListeners(html);
    html.find(".adjustment-button").click(this._onClickButton.bind(this));
    html.find("[data-action='browse']").click(this._onBrowseCompendium.bind(this));
    html.find("[data-action='delete']").click(this._onItemDelete.bind(this));
    html.find("[data-action='viewItem']").click(this._onClickFeature.bind(this));
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _onChangeInput(event) {
    super._onChangeInput(event);
    const input = event.currentTarget;
    if ( input.name === "asi-selected" ) {
      if ( input.checked ) this.feat = { isASI: true };
      else {
        if ( this.feat?.isASI ) this.assignments = {};
        this.feat = null;
      }
    } else {
      const key = input.closest("[data-score]").dataset.score;
      if ( isNaN(input.valueAsNumber) ) this.assignments[key] = 0;
      else {
        this.assignments[key] = Math.min(
          Math.clamp(input.valueAsNumber, Number(input.min), Number(input.max)) - Number(input.dataset.initial),
          (this.advancement.configuration.cap - (this.advancement.configuration.fixed[key] ?? 0)) ?? Infinity
        );
      }
    }
    this.render();
  }

  /* -------------------------------------------- */

  /**
   * Handle opening the compendium browser and displaying the result.
   * @param {PointerEvent} event  The triggering event.
   * @protected
   */
  async _onBrowseCompendium(event) {
    event.preventDefault();
    const filters = {
      locked: {
        additional: { category: { feat: 1 } },
        types: new Set(["feat"])
      }
    };
    const result = await CompendiumBrowser.selectOne({ filters, tab: "feats" });
    if ( result ) this.feat = await fromUuid(result);
    this.render();
  }

  /* -------------------------------------------- */

  /**
   * Handle clicking the plus and minus buttons.
   * @param {Event} event  Triggering click event.
   */
  _onClickButton(event) {
    event.preventDefault();
    const action = event.currentTarget.dataset.action;
    const key = event.currentTarget.closest("li").dataset.score;

    this.assignments[key] ??= 0;
    if ( action === "decrease" ) this.assignments[key] -= 1;
    else if ( action === "increase" ) this.assignments[key] += 1;
    else return;

    this.render();
  }

  /* -------------------------------------------- */

  /**
   * Handle clicking on a feature during item grant to preview the feature.
   * @param {MouseEvent} event  The triggering event.
   * @protected
   */
  async _onClickFeature(event) {
    event.preventDefault();
    const uuid = event.target.closest("[data-uuid]")?.dataset.uuid;
    const item = await fromUuid(uuid);
    item?.sheet.render(true);
  }

  /* -------------------------------------------- */

  /**
   * Handle deleting a dropped feat.
   * @param {Event} event  The originating click event.
   * @protected
   */
  async _onItemDelete(event) {
    event.preventDefault();
    this.feat = null;
    this.render();
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _updateObject(event, formData) {
    await this.advancement.apply(this.level, {
      type: (this.feat && !this.feat.isASI) ? "feat" : "asi",
      assignments: this.assignments,
      featUuid: this.feat?.uuid,
      retainedItems: this.retainedData?.retainedItems
    });
  }

  /* -------------------------------------------- */
  /*  Drag & Drop                                 */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _onDrop(event) {
    if ( !this.advancement.allowFeat ) return false;

    // Try to extract the data
    let data;
    try {
      data = JSON.parse(event.dataTransfer.getData("text/plain"));
    } catch(err) {
      return false;
    }

    if ( data.type !== "Item" ) return false;
    const item = await Item.implementation.fromDropData(data);

    if ( (item.type !== "feat") || (item.system.type.value !== "feat") ) {
      ui.notifications.error("DND5E.ADVANCEMENT.AbilityScoreImprovement.Warning.Type", {localize: true});
      return null;
    }

    // If a feat has a level pre-requisite, make sure it is less than or equal to current character level
    if ( (item.system.prerequisites?.level ?? -Infinity) > this.advancement.actor.system.details.level ) {
      ui.notifications.error(game.i18n.format("DND5E.ADVANCEMENT.AbilityScoreImprovement.Warning.Level", {
        level: item.system.prerequisites.level
      }));
      return null;
    }

    this.feat = item;
    this.render();
  }
}

/**
 * @callback MappingFieldInitialValueBuilder
 * @param {string} key       The key within the object where this new value is being generated.
 * @param {*} initial        The generic initial data provided by the contained model.
 * @param {object} existing  Any existing mapping data.
 * @returns {object}         Value to use as default for this key.
 */

/**
 * @typedef {DataFieldOptions} MappingFieldOptions
 * @property {string[]} [initialKeys]       Keys that will be created if no data is provided.
 * @property {MappingFieldInitialValueBuilder} [initialValue]  Function to calculate the initial value for a key.
 * @property {boolean} [initialKeysOnly=false]  Should the keys in the initialized data be limited to the keys provided
 *                                              by `options.initialKeys`?
 */

/**
 * A subclass of ObjectField that represents a mapping of keys to the provided DataField type.
 *
 * @param {DataField} model                    The class of DataField which should be embedded in this field.
 * @param {MappingFieldOptions} [options={}]   Options which configure the behavior of the field.
 * @property {string[]} [initialKeys]          Keys that will be created if no data is provided.
 * @property {MappingFieldInitialValueBuilder} [initialValue]  Function to calculate the initial value for a key.
 * @property {boolean} [initialKeysOnly=false]  Should the keys in the initialized data be limited to the keys provided
 *                                              by `options.initialKeys`?
 */
class MappingField extends foundry.data.fields.ObjectField {
  constructor(model, options) {
    if ( !(model instanceof foundry.data.fields.DataField) ) {
      throw new Error("MappingField must have a DataField as its contained element");
    }
    super(options);

    /**
     * The embedded DataField definition which is contained in this field.
     * @type {DataField}
     */
    this.model = model;
    model.parent = this;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static get _defaults() {
    return foundry.utils.mergeObject(super._defaults, {
      initialKeys: null,
      initialValue: null,
      initialKeysOnly: false
    });
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _cleanType(value, options) {
    Object.entries(value).forEach(([k, v]) => {
      if ( k.startsWith("-=") ) return;
      value[k] = this.model.clean(v, options);
    });
    return value;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  getInitialValue(data) {
    let keys = this.initialKeys;
    const initial = super.getInitialValue(data);
    if ( !keys || !foundry.utils.isEmpty(initial) ) return initial;
    if ( !(keys instanceof Array) ) keys = Object.keys(keys);
    for ( const key of keys ) initial[key] = this._getInitialValueForKey(key);
    return initial;
  }

  /* -------------------------------------------- */

  /**
   * Get the initial value for the provided key.
   * @param {string} key       Key within the object being built.
   * @param {object} [object]  Any existing mapping data.
   * @returns {*}              Initial value based on provided field type.
   */
  _getInitialValueForKey(key, object) {
    const initial = this.model.getInitialValue();
    return this.initialValue?.(key, initial, object) ?? initial;
  }

  /* -------------------------------------------- */

  /** @override */
  _validateType(value, options={}) {
    if ( foundry.utils.getType(value) !== "Object" ) throw new Error("must be an Object");
    const errors = this._validateValues(value, options);
    if ( !foundry.utils.isEmpty(errors) ) {
      const failure = new foundry.data.validation.DataModelValidationFailure();
      failure.elements = Object.entries(errors).map(([id, failure]) => ({ id, failure }));
      throw failure.asError();
    }
  }

  /* -------------------------------------------- */

  /**
   * Validate each value of the object.
   * @param {object} value     The object to validate.
   * @param {object} options   Validation options.
   * @returns {Record<string, Error>}  An object of value-specific errors by key.
   */
  _validateValues(value, options) {
    const errors = {};
    for ( const [k, v] of Object.entries(value) ) {
      if ( k.startsWith("-=") ) continue;
      const error = this.model.validate(v, options);
      if ( error ) errors[k] = error;
    }
    return errors;
  }

  /* -------------------------------------------- */

  /** @override */
  initialize(value, model, options={}) {
    if ( !value ) return value;
    const obj = {};
    const initialKeys = (this.initialKeys instanceof Array) ? this.initialKeys : Object.keys(this.initialKeys ?? {});
    const keys = this.initialKeysOnly ? initialKeys : Object.keys(value);
    for ( const key of keys ) {
      const data = value[key] ?? this._getInitialValueForKey(key, value);
      obj[key] = this.model.initialize(data, model, options);
    }
    return obj;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _getField(path) {
    if ( path.length === 0 ) return this;
    else if ( path.length === 1 ) return this.model;
    path.shift();
    return this.model._getField(path);
  }
}

const { NumberField: NumberField$z, SetField: SetField$q, StringField: StringField$S } = foundry.data.fields;

/**
 * Data model for the Ability Score Improvement advancement configuration.
 *
 * @property {number} cap                    Maximum number of points that can be assigned to a single score.
 * @property {Set<string>} locked            Abilities that cannot be changed by this advancement.
 * @property {Object<string, number>} fixed  Number of points automatically assigned to a certain score.
 * @property {number} points                 Number of points that can be assigned to any score.
 */
class AbilityScoreImprovementConfigurationData extends foundry.abstract.DataModel {

  /** @override */
  static LOCALIZATION_PREFIXES = ["DND5E.ADVANCEMENT.AbilityScoreImprovement"];

  /* -------------------------------------------- */

  /** @inheritDoc */
  static defineSchema() {
    return {
      cap: new NumberField$z({ integer: true, min: 1, initial: 2 }),
      fixed: new MappingField(new NumberField$z({ nullable: false, integer: true, initial: 0 })),
      locked: new SetField$q(new StringField$S()),
      points: new NumberField$z({ integer: true, min: 0, initial: 0 })
    };
  }
}

/**
 * Data model for the Ability Score Improvement advancement value.
 *
 * @property {string} type             When on a class, whether the player chose ASI or a Feat.
 * @property {Object<string, number>}  Points assigned to individual scores.
 * @property {Object<string, string>}  Feat that was selected.
 */
class AbilityScoreImprovementValueData extends SparseDataModel {
  /** @inheritDoc */
  static defineSchema() {
    return {
      type: new StringField$S({ required: true, initial: "asi", choices: ["asi", "feat"] }),
      assignments: new MappingField(new NumberField$z({
        nullable: false, integer: true
      }), { required: false, initial: undefined }),
      feat: new MappingField(new StringField$S(), { required: false, initial: undefined, label: "DND5E.Feature.Feat" })
    };
  }
}

/**
 * Advancement that presents the player with the option of improving their ability scores or selecting a feat.
 */
class AbilityScoreImprovementAdvancement extends Advancement {

  /** @inheritDoc */
  static get metadata() {
    return foundry.utils.mergeObject(super.metadata, {
      dataModels: {
        configuration: AbilityScoreImprovementConfigurationData,
        value: AbilityScoreImprovementValueData
      },
      order: 20,
      icon: "icons/magic/symbols/star-solid-gold.webp",
      typeIcon: "systems/dnd5e/icons/svg/ability-score-improvement.svg",
      title: game.i18n.localize("DND5E.ADVANCEMENT.AbilityScoreImprovement.Title"),
      hint: game.i18n.localize("DND5E.ADVANCEMENT.AbilityScoreImprovement.Hint"),
      apps: {
        config: AbilityScoreImprovementConfig,
        flow: AbilityScoreImprovementFlow
      }
    });
  }

  /* -------------------------------------------- */
  /*  Preparation Methods                         */
  /* -------------------------------------------- */

  /** @inheritDoc */
  _preCreate(data) {
    if ( super._preCreate(data) === false ) return false;
    if ( this.item.type !== "class" || foundry.utils.hasProperty(data, "configuration.points") ) return;
    this.updateSource({"configuration.points": 2});
  }

  /* -------------------------------------------- */
  /*  Instance Properties                         */
  /* -------------------------------------------- */

  /**
   * Does this advancement allow feats, or just ability score improvements?
   * @type {boolean}
   */
  get allowFeat() {
    return (this.item.type === "class") && (game.settings.get("dnd5e", "allowFeats")
      || game.settings.get("dnd5e", "rulesVersion") === "modern");
  }

  /* -------------------------------------------- */

  /**
   * Information on the ASI points available.
   * @type {{ assigned: number, total: number }}
   */
  get points() {
    return {
      assigned: Object.entries(this.value.assignments ?? {}).reduce((n, [abl, c]) => {
        if ( this.canImprove(abl) ) n += c;
        return n;
      }, 0),
      total: this.configuration.points + Object.entries(this.configuration.fixed).reduce((t, [abl, v]) => {
        if ( this.canImprove(abl) ) t += v;
        return t;
      }, 0)
    };
  }

  /* -------------------------------------------- */
  /*  Instance Methods                            */
  /* -------------------------------------------- */

  /**
   * Is this ability allowed to be improved?
   * @param {string} ability  The ability key.
   * @returns {boolean}
   */
  canImprove(ability) {
    return CONFIG.DND5E.abilities[ability]?.improvement !== false;
  }

  /* -------------------------------------------- */
  /*  Display Methods                             */
  /* -------------------------------------------- */

  /** @inheritDoc */
  titleForLevel(level, { configMode=false }={}) {
    if ( this.value.selected !== "feat" ) return this.title;
    return game.i18n.localize("DND5E.Feature.Feat");
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  summaryForLevel(level, { configMode=false }={}) {
    const formatter = new Intl.NumberFormat(game.i18n.lang, { signDisplay: "always" });
    if ( configMode ) {
      const entries = Object.entries(this.configuration.fixed).map(([key, value]) => {
        if ( !value ) return null;
        const name = CONFIG.DND5E.abilities[key]?.label ?? key;
        return `<span class="tag">${name} <strong>${formatter.format(value)}</strong></span>`;
      });
      if ( this.configuration.points ) entries.push(`<span class="tag">${
        game.i18n.localize("DND5E.ADVANCEMENT.AbilityScoreImprovement.FIELDS.points.label")}: <strong>${
        this.configuration.points}</strong></span>`
      );
      return entries.filterJoin("\n");
    }

    else if ( (this.value.type === "feat") && this.value.feat ) {
      const id = Object.keys(this.value.feat)[0];
      const feat = this.actor.items.get(id);
      if ( feat ) return feat.toAnchor({classes: ["content-link"]}).outerHTML;
    }

    else if ( (this.value.type === "asi") && this.value.assignments ) {
      return Object.entries(this.value.assignments).reduce((html, [key, value]) => {
        const name = CONFIG.DND5E.abilities[key]?.label ?? key;
        html += `<span class="tag">${name} <strong>${formatter.format(value)}</strong></span>\n`;
        return html;
      }, "");
    }

    return "";
  }

  /* -------------------------------------------- */
  /*  Application Methods                         */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async apply(level, data) {
    if ( data.type === "asi" ) {
      const assignments = Object.keys(CONFIG.DND5E.abilities).reduce((obj, key) => {
        obj[key] = (this.configuration.fixed[key] ?? 0) + (data.assignments[key] ?? 0);
        return obj;
      }, {});
      const updates = {};
      for ( const key of Object.keys(assignments) ) {
        const ability = this.actor.system.abilities[key];
        const source = this.actor.system.toObject().abilities[key] ?? {};
        if ( !ability || !this.canImprove(key) ) continue;
        assignments[key] = Math.min(assignments[key], ability.max - source.value);
        if ( assignments[key] ) updates[`system.abilities.${key}.value`] = source.value + assignments[key];
        else delete assignments[key];
      }
      data.assignments = assignments;
      data.feat = null;
      this.actor.updateSource(updates);
    }

    else {
      let itemData = data.retainedItems?.[data.featUuid];
      if ( !itemData ) itemData = await this.createItemData(data.featUuid);
      data.assignments = null;
      if ( itemData ) {
        data.feat = { [itemData._id]: data.featUuid };
        this.actor.updateSource({items: [itemData]});
      }
    }

    delete data.featUuid;
    delete data.retainedItems;
    this.updateSource({value: data});
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  restore(level, data) {
    data.featUuid = Object.values(data.feat ?? {})[0];
    this.apply(level, data);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  reverse(level) {
    const source = this.value.toObject();

    if ( this.value.type === "asi" ) {
      const updates = {};
      for ( const [key, change] of Object.entries(this.value.assignments ?? {}) ) {
        const ability = this.actor.system.toObject().abilities[key];
        if ( !ability || !this.canImprove(key) ) continue;
        updates[`system.abilities.${key}.value`] = ability.value - change;
        source.assignments[key] -= (this.configuration.fixed[key] ?? 0);
      }
      this.actor.updateSource(updates);
    }

    else {
      const [id, uuid] = Object.entries(this.value.feat ?? {})[0] ?? [];
      const item = this.actor.items.get(id);
      if ( item ) source.retainedItems = {[uuid]: item.toObject()};
      this.actor.items.delete(id);
    }

    this.updateSource({ "value.assignments": null, "value.feat": null });
    return source;
  }
}

/**
 * Configuration application for hit points.
 */
let HitPointsConfig$1 = class HitPointsConfig extends AdvancementConfig$1 {
  /** @override */
  static DEFAULT_OPTIONS = {
    classes: ["hit-points"]
  };

  /* -------------------------------------------- */

  /** @inheritDoc */
  static PARTS = {
    ...super.PARTS,
    hitPoints: {
      template: "systems/dnd5e/templates/advancement/hit-points-config.hbs"
    }
  };

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareContext(options) {
    const context = await super._prepareContext(options);
    context.hitDie = this.advancement.hitDie;
    return context;
  }
};

/**
 * Inline application that presents hit points selection upon level up.
 */
class HitPointsFlow extends AdvancementFlow {

  /** @inheritDoc */
  static get defaultOptions() {
    return foundry.utils.mergeObject(super.defaultOptions, {
      template: "systems/dnd5e/templates/advancement/hit-points-flow.hbs"
    });
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  getData() {
    const source = this.retainedData ?? this.advancement.value;
    const value = source[this.level];

    // If value is empty, `useAverage` should default to the value selected at the previous level
    let useAverage = value === "avg";
    if ( !value ) {
      const lastValue = source[this.level - 1];
      if ( lastValue === "avg" ) useAverage = true;
    }

    return foundry.utils.mergeObject(super.getData(), {
      isFirstClassLevel: (this.level === 1) && this.advancement.item.isOriginalClass,
      hitDie: this.advancement.hitDie,
      dieValue: this.advancement.hitDieValue,
      data: {
        value: Number.isInteger(value) ? value : "",
        useAverage
      }
    });
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  activateListeners(html) {
    this.form.querySelector(".average-checkbox")?.addEventListener("change", event => {
      this.form.querySelector(".roll-result").disabled = event.target.checked;
      this.form.querySelector(".roll-button").disabled = event.target.checked;
      this._updateRollResult();
    });
    this.form.querySelector(".roll-button")?.addEventListener("click", async () => {
      const roll = await this.advancement.actor.rollClassHitPoints(this.advancement.item);
      this.form.querySelector(".roll-result").value = roll.total;
    });
    this._updateRollResult();
  }

  /* -------------------------------------------- */

  /**
   * Update the roll result display when the average result is taken.
   * @protected
   */
  _updateRollResult() {
    if ( !this.form.elements.useAverage?.checked ) return;
    this.form.elements.value.value = (this.advancement.hitDieValue / 2) + 1;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _updateObject(event, formData) {
    let value;
    if ( formData.useMax ) value = "max";
    else if ( formData.useAverage ) value = "avg";
    else if ( Number.isInteger(formData.value) ) value = parseInt(formData.value);

    if ( value !== undefined ) return this.advancement.apply(this.level, { [this.level]: value });

    this.form.querySelector(".rollResult")?.classList.add("error");
    const errorType = formData.value ? "Invalid" : "Empty";
    throw new Advancement.ERROR(game.i18n.localize(`DND5E.ADVANCEMENT.HitPoints.Warning.${errorType}`));
  }

}

/**
 * Advancement that presents the player with the option to roll hit points at each level or select the average value.
 * Keeps track of player hit point rolls or selection for each class level. **Can only be added to classes and each
 * class can only have one.**
 */
class HitPointsAdvancement extends Advancement {

  /** @inheritDoc */
  static get metadata() {
    return foundry.utils.mergeObject(super.metadata, {
      order: 10,
      icon: "icons/magic/life/heart-pink.webp",
      typeIcon: "systems/dnd5e/icons/svg/hit-points.svg",
      title: game.i18n.localize("DND5E.ADVANCEMENT.HitPoints.Title"),
      hint: game.i18n.localize("DND5E.ADVANCEMENT.HitPoints.Hint"),
      multiLevel: true,
      apps: {
        config: HitPointsConfig$1,
        flow: HitPointsFlow
      }
    });
  }

  /* -------------------------------------------- */
  /*  Instance Properties                         */
  /* -------------------------------------------- */

  /**
   * The amount gained if the average is taken.
   * @type {number}
   */
  get average() {
    return (this.hitDieValue / 2) + 1;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  get levels() {
    return Array.fromRange(CONFIG.DND5E.maxLevel + 1).slice(1);
  }

  /* -------------------------------------------- */

  /**
   * Shortcut to the hit die used by the class.
   * @returns {string}
   */
  get hitDie() {
    if ( this.actor?.type === "npc" ) return `d${this.actor.system.attributes.hd.denomination}`;
    return this.item.system.hd.denomination;
  }

  /* -------------------------------------------- */

  /**
   * The face value of the hit die used.
   * @returns {number}
   */
  get hitDieValue() {
    return Number(this.hitDie.substring(1));
  }

  /* -------------------------------------------- */
  /*  Display Methods                             */
  /* -------------------------------------------- */

  /** @inheritDoc */
  configuredForLevel(level) {
    return this.valueForLevel(level) !== null;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  titleForLevel(level, { configMode=false, legacyDisplay=false }={}) {
    const hp = this.valueForLevel(level);
    if ( !hp || configMode || !legacyDisplay ) return this.title;
    return `${this.title}: <strong>${hp}</strong>`;
  }

  /* -------------------------------------------- */

  /**
   * Hit points given at the provided level.
   * @param {number} level   Level for which to get hit points.
   * @returns {number|null}  Hit points for level or null if none have been taken.
   */
  valueForLevel(level) {
    return this.constructor.valueForLevel(this.value, this.hitDieValue, level);
  }

  /* -------------------------------------------- */

  /**
   * Hit points given at the provided level.
   * @param {object} data         Contents of `value` used to determine this value.
   * @param {number} hitDieValue  Face value of the hit die used by this advancement.
   * @param {number} level        Level for which to get hit points.
   * @returns {number|null}       Hit points for level or null if none have been taken.
   */
  static valueForLevel(data, hitDieValue, level) {
    const value = data[level];
    if ( !value ) return null;

    if ( value === "max" ) return hitDieValue;
    if ( value === "avg" ) return (hitDieValue / 2) + 1;
    return value;
  }

  /* -------------------------------------------- */

  /**
   * Total hit points provided by this advancement.
   * @returns {number}  Hit points currently selected.
   */
  total() {
    return Object.keys(this.value).reduce((total, level) => total + this.valueForLevel(parseInt(level)), 0);
  }

  /* -------------------------------------------- */

  /**
   * Total hit points taking the provided ability modifier into account, with a minimum of 1 per level.
   * @param {number} mod  Modifier to add per level.
   * @returns {number}    Total hit points plus modifier.
   */
  getAdjustedTotal(mod) {
    return Object.keys(this.value).reduce((total, level) => {
      return total + Math.max(this.valueForLevel(parseInt(level)) + mod, 1);
    }, 0);
  }

  /* -------------------------------------------- */
  /*  Editing Methods                             */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static availableForItem(item) {
    return !item.advancement.byType.HitPoints?.length;
  }

  /* -------------------------------------------- */
  /*  Application Methods                         */
  /* -------------------------------------------- */

  /**
   * Add the ability modifier and any bonuses to the provided hit points value to get the number to apply.
   * @param {number} value  Hit points taken at a given level.
   * @returns {number}      Hit points adjusted with ability modifier and per-level bonuses.
   */
  #getApplicableValue(value) {
    const abilityId = CONFIG.DND5E.defaultAbilities.hitPoints || "con";
    value = Math.max(value + (this.actor.system.abilities[abilityId]?.mod ?? 0), 1);
    value += simplifyBonus(this.actor.system.attributes.hp.bonuses?.level, this.actor.getRollData());
    return value;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  apply(level, data) {
    let value = this.constructor.valueForLevel(data, this.hitDieValue, level);
    if ( value === undefined ) return;
    this.actor.updateSource({
      "system.attributes.hp.value": this.actor.system.attributes.hp.value + this.#getApplicableValue(value)
    });
    this.updateSource({ value: data });
  }

  /* -------------------------------------------- */

  /** @override */
  automaticApplicationValue(level) {
    if ( (level === 1) && this.item.isOriginalClass ) return { [level]: "max" };
    if ( this.value[level - 1] === "avg" ) return { [level]: "avg" };
    return false;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  restore(level, data) {
    this.apply(level, data);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  reverse(level) {
    let value = this.valueForLevel(level);
    if ( value === undefined ) return;
    this.actor.updateSource({
      "system.attributes.hp.value": this.actor.system.attributes.hp.value - this.#getApplicableValue(value)
    });
    const source = { [level]: this.value[level] };
    this.updateSource({ [`value.-=${level}`]: null });
    return source;
  }
}

/**
 * Configuration application for item choices.
 */
class ItemChoiceConfig extends AdvancementConfig$1 {
  /** @override */
  static DEFAULT_OPTIONS = {
    classes: ["item-choice", "three-column"],
    dropKeyPath: "pool",
    position: {
      width: 800
    }
  };

  /* -------------------------------------------- */

  /** @inheritDoc */
  static PARTS = {
    config: {
      container: { classes: ["column-container"], id: "column-left" },
      template: "systems/dnd5e/templates/advancement/advancement-controls-section.hbs"
    },
    details: {
      container: { classes: ["column-container"], id: "column-left" },
      template: "systems/dnd5e/templates/advancement/item-choice-config-details.hbs"
    },
    spellConfig: {
      container: { classes: ["column-container"], id: "column-left" },
      template: "systems/dnd5e/templates/advancement/advancement-spell-config-section.hbs"
    },
    items: {
      container: { classes: ["column-container"], id: "column-center" },
      template: "systems/dnd5e/templates/advancement/item-choice-config-items.hbs"
    },
    levels: {
      container: { classes: ["column-container"], id: "column-right" },
      template: "systems/dnd5e/templates/advancement/item-choice-config-levels.hbs"
    }
  };

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareContext(options) {
    const context = await super._prepareContext(options);

    context.items = this.advancement.configuration.pool.map(data => ({
      data,
      fields: this.advancement.configuration.schema.fields.pool.element.fields,
      index: fromUuidSync(data.uuid)
    }));

    context.abilityOptions = Object.entries(CONFIG.DND5E.abilities).map(([value, { label }]) => ({ value, label }));
    context.choices = context.levels.reduce((obj, { value, label }) => {
      obj[value] = { label, ...this.advancement.configuration.choices[value] };
      return obj;
    }, {});
    context.levelRestrictionOptions = [
      { value: "", label: "" },
      {
        value: "available",
        label: game.i18n.localize("DND5E.ADVANCEMENT.ItemChoice.FIELDS.restriction.level.Available")
      },
      { rule: true },
      ...Object.entries(CONFIG.DND5E.spellLevels).map(([value, label]) => ({ value, label }))
    ];
    context.showContainerWarning = context.items.some(i => i.index?.type === "container");
    context.showSpellConfig = this.advancement.configuration.type === "spell";
    context.showRequireSpellSlot = !this.advancement.configuration.spell?.preparation
      || CONFIG.DND5E.spellPreparationModes[this.advancement.configuration.spell?.preparation]?.upcast;
    context.typeOptions = [
      { value: "", label: game.i18n.localize("DND5E.ADVANCEMENT.ItemChoice.FIELDS.type.Any") },
      { rule: true },
      ...this.advancement.constructor.VALID_TYPES
        .map(value => ({ value, label: game.i18n.localize(CONFIG.Item.typeLabels[value]) }))
    ];

    if ( this.advancement.configuration.type === "feat" ) {
      const selectedType = CONFIG.DND5E.featureTypes[this.advancement.configuration.restriction.type];
      context.typeRestriction = {
        typeLabel: game.i18n.localize("DND5E.ItemFeatureType"),
        typeOptions: [
          { value: "", label: "" },
          ...Object.entries(CONFIG.DND5E.featureTypes).map(([value, { label }]) => ({ value, label }))
        ],
        subtypeLabel: game.i18n.format("DND5E.ItemFeatureSubtype", {category: selectedType?.label}),
        subtypeOptions: selectedType?.subtypes ? [
          { value: "", label: "" },
          ...Object.entries(selectedType.subtypes).map(([value, label]) => ({ value, label }))
        ] : null
      };
    }

    return context;
  }

  /* -------------------------------------------- */
  /*  Form Handling                               */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async prepareConfigurationUpdate(configuration) {
    if ( configuration.choices ) configuration.choices = this.constructor._cleanedObject(configuration.choices);
    if ( configuration.spell ) configuration.spell.ability ??= [];
    if ( configuration.pool ) configuration.pool = Object.values(configuration.pool);

    // Ensure items are still valid if type restriction or spell restriction are changed
    const pool = [];
    for ( const item of (configuration.pool ?? this.advancement.configuration.pool) ) {
      if ( this.advancement._validateItemType(await fromUuid(item.uuid), {
        type: configuration.type, restriction: configuration.restriction ?? {}, strict: false
      }) ) pool.push(item);
    }
    configuration.pool = pool;

    return configuration;
  }

  /* -------------------------------------------- */
  /*  Drag & Drop                                 */
  /* -------------------------------------------- */

  /** @inheritDoc */
  _validateDroppedItem(event, item) {
    this.advancement._validateItemType(item);
  }
}

const { BooleanField: BooleanField$v } = foundry.data.fields;

/**
 * Dialog with shared resting functionality.
 */
class BaseRestDialog extends Dialog5e {
  constructor(options={}, _dialogData={}, _options={}) {
    if ( options instanceof Actor ) {
      foundry.utils.logCompatibilityWarning(
        "The rest dialogs now take a single options object with `document` and `config` options.",
        { since: "DnD5e 4.2", until: "DnD5e 4.4" }
      );
      options = { ..._options, data: _dialogData, document: options };
    }
    super(options);
    this.actor = options.document;
    this.#config = options.config;
  }

  /* -------------------------------------------- */

  /** @override */
  static DEFAULT_OPTIONS = {
    classes: ["rest"],
    config: null,
    document: null,
    form: {
      handler: BaseRestDialog.#handleFormSubmission
    },
    position: {
      width: 380
    }
  };

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * The actor being rested.
   * @type {Actor5e}
   */
  actor;

  /* -------------------------------------------- */

  /**
   * The rest configuration.
   * @type {RestConfiguration}
   */
  #config;

  get config() {
    return this.#config;
  }

  /* -------------------------------------------- */

  /**
   * Should the user be prompted as to whether a new day has occurred?
   * @type {boolean}
   */
  get promptNewDay() {
    const duration = CONFIG.DND5E.restTypes[this.config.type]
      ?.duration?.[game.settings.get("dnd5e", "restVariant")] ?? 0;
    // Only prompt if rest is longer than 10 minutes and less than 24 hours
    return (duration > 10) && (duration < 1440);
  }

  /* -------------------------------------------- */

  /**
   * Was the rest button pressed?
   * @type {boolean}
   */
  #rested = false;

  get rested() {
    return this.#rested;
  }

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareContext(options) {
    const context = {
      ...await super._prepareContext(options),
      actor: this.actor,
      config: this.config,
      fields: [],
      result: this.result,
      hd: this.actor.system.attributes?.hd,
      hp: this.actor.system.attributes?.hp,
      isGroup: this.actor.type === "group",
      variant: game.settings.get("dnd5e", "restVariant")
    };
    if ( this.promptNewDay ) context.fields.push({
      field: new BooleanField$v({
        label: game.i18n.localize("DND5E.REST.NewDay.Label"),
        hint: game.i18n.localize("DND5E.REST.NewDay.Hint")
      }),
      input: context.inputs.createCheckboxInput,
      name: "newDay",
      value: context.config.newDay
    });
    return context;
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /**
   * Handle submission of the dialog using the form buttons.
   * @this {BaseRestDialog}
   * @param {Event|SubmitEvent} event    The form submission event.
   * @param {HTMLFormElement} form       The submitted form.
   * @param {FormDataExtended} formData  Data from the dialog.
   */
  static async #handleFormSubmission(event, form, formData) {
    foundry.utils.mergeObject(this.config, formData.object);
    this.#rested = true;
    await this.close();
  }

  /* -------------------------------------------- */
  /*  Factory Methods                             */
  /* -------------------------------------------- */

  /**
   * A helper to handle displaying and responding to the dialog.
   * @param {Actor5e} actor              The actor that is resting.
   * @param {RestConfiguration}  config  Configuration information for the rest.
   * @returns {Promise<RestConfiguration>}
   */
  static async configure(actor, config) {
    return new Promise((resolve, reject) => {
      const app = new this({
        config,
        buttons: [
          {
            default: true,
            icon: "fa-solid fa-bed",
            label: game.i18n.localize("DND5E.REST.Label"),
            name: "rest",
            type: "submit"
          }
        ],
        document: actor
      });
      app.addEventListener("close", () => app.rested ? resolve(app.config) : reject(), { once: true });
      app.render({ force: true });
    });
  }
}

const { BooleanField: BooleanField$u } = foundry.data.fields;

/**
 * Dialog for configuring a short rest.
 */
class ShortRestDialog extends BaseRestDialog {
  /** @override */
  static DEFAULT_OPTIONS = {
    classes: ["short-rest"],
    actions: {
      rollHitDie: ShortRestDialog.#rollHitDie
    },
    window: {
      title: "DND5E.REST.Short.Label"
    }
  };

  /* -------------------------------------------- */

  /** @inheritDoc */
  static PARTS = {
    ...super.PARTS,
    content: {
      template: "systems/dnd5e/templates/actors/rest/short-rest.hbs"
    }
  };

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * Currently selected hit dice denomination.
   * @type {string}
   */
  #denom;

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareContext(options) {
    const context = await super._prepareContext(options);
    context.autoRoll = new BooleanField$u({
      label: game.i18n.localize("DND5E.REST.HitDice.AutoSpend.Label"),
      hint: game.i18n.localize("DND5E.REST.HitDice.AutoSpend.Hint")
    });

    if ( this.actor.type === "npc" ) {
      const hd = this.actor.system.attributes.hd;
      context.hitDice = {
        canRoll: hd.value > 0,
        denomination: `d${hd.denomination}`,
        options: [{
          value: `d${hd.denomination}`,
          label: `d${hd.denomination} (${game.i18n.format("DND5E.HITDICE.Available", { number: hd.value })})`
        }]
      };
    }

    else if ( foundry.utils.hasProperty(this.actor, "system.attributes.hd") ) {
      context.hitDice = {
        canRoll: this.actor.system.attributes.hd.value > 0,
        options: Object.entries(this.actor.system.attributes.hd.bySize).map(([value, number]) => ({
          value, label: `${value} (${game.i18n.format("DND5E.HITDICE.Available", { number })})`, number
        }))
      };
      context.denomination = (this.actor.system.attributes.hd.bySize[this.#denom] > 0)
        ? this.#denom : context.hitDice.options.find(o => o.number > 0)?.value;
    }

    else context.fields.unshift({
      field: context.autoRoll,
      input: context.inputs.createCheckboxInput,
      name: "autoHD",
      value: context.config.autoHD
    });

    return context;
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /**
   * Handle rolling a hit die.
   * @this {ShortRestDialog}
   * @param {Event} event         Triggering click event.
   * @param {HTMLElement} target  Button that was clicked.
   */
  static async #rollHitDie(event, target) {
    this.#denom = this.form.denom.value;
    await this.actor.rollHitDie({ denomination: this.#denom });
    foundry.utils.mergeObject(this.config, new FormDataExtended(this.form).object);
    this.render();
  }

  /* -------------------------------------------- */
  /*  Factory Methods                             */
  /* -------------------------------------------- */

  /**
   * A helper constructor function which displays the Short Rest dialog and returns a Promise once its workflow has
   * been resolved.
   * @param {object} [options={}]
   * @param {Actor5e} [options.actor]  Actor that is taking the short rest.
   * @returns {Promise}                Promise that resolves when the rest is completed or rejects when canceled.
   */
  static async shortRestDialog({ actor } = {}) {
    foundry.utils.logCompatibilityWarning(
      "The `shortRestDialog` method on `ShortRestDialog` has been renamed `configure`.",
      { since: "DnD5e 4.2", until: "DnD5e 4.4" }
    );
    return this.configure(actor, { type: "short" });
  }
}

const { BooleanField: BooleanField$t } = foundry.data.fields;

/**
 * Dialog for configuring a long rest.
 */
class LongRestDialog extends BaseRestDialog {
  /** @override */
  static DEFAULT_OPTIONS = {
    classes: ["long-rest"],
    window: {
      title: "DND5E.REST.Long.Label"
    }
  };

  /* -------------------------------------------- */

  /** @inheritDoc */
  static PARTS = {
    ...super.PARTS,
    content: {
      template: "systems/dnd5e/templates/actors/rest/long-rest.hbs"
    }
  };

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareContext(options) {
    const context = await super._prepareContext(options);

    const { enabled } = game.settings.get("dnd5e", "bastionConfiguration");
    if ( game.user.isGM && context.isGroup && enabled ) context.fields.unshift({
      field: new BooleanField$t({ label: game.i18n.localize("DND5E.Bastion.Action.BastionTurn") }),
      input: context.inputs.createCheckboxInput,
      name: "advanceBastionTurn",
      value: context.config.advanceBastionTurn
    });

    return context;
  }

  /* -------------------------------------------- */
  /*  Factory Methods                             */
  /* -------------------------------------------- */

  /**
   * A helper constructor function which displays the Long Rest confirmation dialog and returns a Promise once its
   * workflow has been resolved.
   * @param {object} [options={}]
   * @param {Actor5e} [options.actor]  Actor that is taking the long rest.
   * @returns {Promise}                Promise that resolves when the rest is completed or rejects when canceled.
   */
  static async longRestDialog({ actor } = {}) {
    foundry.utils.logCompatibilityWarning(
      "The `longRestDialog` method on `LongRestDialog` has been renamed `configure`.",
      { since: "DnD5e 4.2", until: "DnD5e 4.4" }
    );
    return this.configure(actor, { type: "long" });
  }
}

/**
 * Description for a single part of a property attribution.
 * @typedef {object} AttributionDescription
 * @property {string} label               Descriptive label that will be displayed. If the label is in the form
 *                                        of an @ property, the system will try to turn it into a human-readable label.
 * @property {number} mode                Application mode for this step as defined in
 *                           [CONST.ACTIVE_EFFECT_MODES](https://foundryvtt.com/api/module-constants.html#.ACTIVE_EFFECT_MODES).
 * @property {number} value               Value of this step.
 * @property {ActiveEffect5e} [document]  Active effect applying this attribution, if any.
 */

/**
 * Interface for viewing what factors went into determining a specific property.
 *
 * @param {Document} object                        The Document that owns the property being attributed.
 * @param {AttributionDescription[]} attributions  An array of all the attribution data.
 * @param {string} property                        Dot separated path to the property.
 * @param {object} [options={}]                    Application rendering options.
 */
class PropertyAttribution extends Application5e {
  constructor(object, attributions, property, options={}) {
    super(options);
    this.object = object;
    this.attributions = attributions;
    this.property = property;
  }

  /* -------------------------------------------- */

  /** @override */
  static DEFAULT_OPTIONS = {
    classes: ["property-attribution"],
    window: {
      frame: false,
      positioned: false
    }
  };

  /* -------------------------------------------- */

  /** @override */
  static PARTS = {
    attribution: {
      template: "systems/dnd5e/templates/apps/property-attribution.hbs"
    }
  };

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /**
   * Prepare tooltip contents.
   * @returns {Promise<string>}
   */
  async renderTooltip() {
    await this.render({ force: true });
    return this.element.innerHTML;
  }

  /* -------------------------------------------- */

  /** @override */
  _insertElement(element) {}

  /* -------------------------------------------- */

  /** @override */
  async _prepareContext(options) {
    const property = foundry.utils.getProperty(this.object.system, this.property);
    let total;
    if ( Number.isNumeric(property)) total = property;
    else if ( typeof property === "object" && Number.isNumeric(property.value) ) total = property.value;
    const sources = foundry.utils.duplicate(this.attributions);
    return {
      caption: game.i18n.localize(this.options.title),
      sources: sources.map(entry => {
        if ( entry.label.startsWith("@") ) entry.label = this.getPropertyLabel(entry.label.slice(1));
        if ( (entry.mode === CONST.ACTIVE_EFFECT_MODES.ADD) && (entry.value < 0) ) {
          entry.negative = true;
          entry.value = entry.value * -1;
        }
        return entry;
      }),
      total: total
    };
  }

  /* -------------------------------------------- */
  /*  Helpers                                     */
  /* -------------------------------------------- */

  /**
   * Produce a human-readable and localized name for the provided property.
   * @param {string} property  Dot separated path to the property.
   * @returns {string}         Property name for display.
   */
  getPropertyLabel(property) {
    const parts = property.split(".");
    if ( parts[0] === "abilities" && parts[1] ) {
      return CONFIG.DND5E.abilities[parts[1]]?.label ?? property;
    } else if ( (property === "attributes.ac.dex") && CONFIG.DND5E.abilities.dex ) {
      return CONFIG.DND5E.abilities.dex.label;
    } else if ( (parts[0] === "prof") || (property === "attributes.prof") ) {
      return game.i18n.localize("DND5E.Proficiency");
    }
    return property;
  }
}

const { SetField: SetField$p, StringField: StringField$R } = foundry.data.fields;

/**
 * @typedef {Set<string>} ActivationsData
 */

/**
 * A field for storing relative UUIDs to activations on the actor.
 */
class ActivationsField extends SetField$p {
  constructor() {
    super(new StringField$R());
  }

  /* -------------------------------------------- */

  /**
   * Find any activity relative UUIDs on this actor that can be used during a set of periods.
   * @param {Actor5e} actor
   * @param {string[]} periods
   * @returns {string[]}
   */
  static getActivations(actor, periods) {
    return actor.items
      .map(i => i.system.activities?.filter(a => periods.includes(a.activation?.type)).map(a => a.relativeUUID) ?? [])
      .flat();
  }

  /* -------------------------------------------- */

  /**
   * Prepare activations for display on chat card.
   * @this {ActivationsData}
   * @param {Actor5e} actor  Actor to which this activations can be used.
   * @returns {Activity[]}
   */
  static processActivations(actor) {
    return Array.from(this)
      .map(uuid => fromUuidSync(uuid, { relative: actor, strict: false }))
      .filter(_ => _)
      .sort((lhs, rhs) => (lhs.item.sort - rhs.item.sort) || (lhs.sort - rhs.sort));
  }
}

const { ArrayField: ArrayField$h, NumberField: NumberField$y, SchemaField: SchemaField$D, StringField: StringField$Q } = foundry.data.fields;

/**
 * @typedef ActorDeltasData
 * @property {IndividualDeltaData[]} actor                 Changes for the actor.
 * @property {Record<string, IndividualDeltaData[]>} item  Changes for each item grouped by ID.
 */

/**
 * @typedef DeltaDisplayContext
 * @property {string} type              Type of document to which the delta applies.
 * @property {string} delta             The formatted numeric change.
 * @property {Actor5e|Item5e} document  The document to which the delta applies.
 * @property {string} label             The formatted label for the attribute.
 * @property {Roll[]} [rolls]           Any rolls associated with the delta.
 */

/**
 * A field for storing deltas made to an actor or embedded items.
 */
class ActorDeltasField extends SchemaField$D {
  constructor() {
    super({
      actor: new ArrayField$h(new IndividualDeltaField()),
      item: new MappingField(new ArrayField$h(new IndividualDeltaField()))
    });
  }

  /* -------------------------------------------- */

  /**
   * Calculate delta information for an actor document from the given updates.
   * @param {Actor5e} actor                              Actor for which to calculate the deltas.
   * @param {{ actor: object, item: object[] }} updates  Updates to apply to the actor and contained items.
   * @returns {ActorDeltasData}
   */
  static getDeltas(actor, updates) {
    return {
      actor: IndividualDeltaField.getDeltas(actor, updates.actor),
      item: updates.item.reduce((obj, { _id, ...changes }) => {
        const deltas = IndividualDeltaField.getDeltas(actor.items.get(_id), changes);
        if ( deltas.length ) obj[_id] = deltas;
        return obj;
      }, {})
    };
  }

  /* -------------------------------------------- */

  /**
   * Prepare deltas for display in a chat message.
   * @this {ActorDeltasData}
   * @param {Actor5e} actor   Actor to which this delta applies.
   * @param {Roll[]} [rolls]  Rolls that may be associated with a delta.
   * @returns {DeltaDisplayContext[]}
   */
  static processDeltas(actor, rolls=[]) {
    return [
      ...this.actor.map(d => IndividualDeltaField.processDelta.call(d, actor, rolls
        .filter(r => !r.options.delta?.item && (r.options.delta?.keyPath === d.keyPath)))),
      ...Object.entries(this.item).flatMap(([id, deltas]) =>
        deltas.map(d => IndividualDeltaField.processDelta.call(d, actor.items.get(id), rolls
          .filter(r => (r.options.delta?.item === id) && (r.options.delta?.keyPath === d.keyPath))))
      )
    ];
  }
}

/**
 * @typedef IndividualDeltaData
 * @property {number} delta    The change in the specified field.
 * @property {string} keyPath  Path to the changed field on the document.
 */

/**
 * A field that stores a delta for an individual property on an actor or item.
 */
class IndividualDeltaField extends SchemaField$D {
  constructor() {
    super({ delta: new NumberField$y(), keyPath: new StringField$Q() });
  }

  /* -------------------------------------------- */

  /**
   * Calculate delta information for a document from the given updates.
   * @param {DataModel} dataModel      Document for which to calculate the deltas.
   * @param {object} updates           Updates that are to be applied.
   * @returns {IndividualDeltaData[]}
   */
  static getDeltas(dataModel, updates) {
    updates = foundry.utils.flattenObject(updates);
    const deltas = [];
    for ( const [keyPath, value] of Object.entries(updates) ) {
      let currentValue;
      if ( keyPath.startsWith("system.activities") ) {
        const [id, ...kp] = keyPath.slice(18).split(".");
        currentValue = foundry.utils.getProperty(dataModel.system.activities?.get(id) ?? {}, kp.join("."));
      }
      else currentValue = foundry.utils.getProperty(dataModel, keyPath);

      const delta = value - currentValue;
      if ( delta && !Number.isNaN(delta) ) deltas.push({ keyPath, delta });
    }
    return deltas;
  }

  /* -------------------------------------------- */

  /**
   * Prepare a delta for display in a chat message.
   * @this {IndividualDeltaData}
   * @param {Actor5e|Item5e} doc  Actor or item to which this delta applies.
   * @param {Roll[]} [rolls]      Rolls that may be associated with a delta.
   * @returns {DeltaDisplayContext}
   */
  static processDelta(doc, rolls=[]) {
    const type = doc instanceof Actor ? "actor" : "item";
    const value = this.keyPath.endsWith(".spent") ? -this.delta : this.delta;
    return {
      type,
      delta: formatNumber(value, { signDisplay: "always" }),
      document: doc,
      label: getHumanReadableAttributeLabel(this.keyPath, { [type]: doc }) ?? this.keyPath,
      rolls: rolls.map(roll => ({ roll, anchor: roll.toAnchor().outerHTML.replace(`${roll.total}</a>`, "</a>") }))
    };
  }
}

/**
 * Attempt to create a macro from the dropped data. Will use an existing macro if one exists.
 * @param {object} dropData     The dropped data
 * @param {number} slot         The hotbar slot to use
 * @returns {Promise}
 */
async function create5eMacro(dropData, slot) {
  const macroData = { type: "script", scope: "actor" };
  switch ( dropData.type ) {
    case "Activity":
      const activity = await fromUuid(dropData.uuid);
      if ( !activity ) {
        ui.notifications.warn("MACRO.5eUnownedWarn", { localize: true });
        return null;
      }
      foundry.utils.mergeObject(macroData, {
        name: `${activity.item.name}: ${activity.name}`,
        img: activity.img,
        command: `dnd5e.documents.macro.rollItem("${activity.item._source.name}", { activityName: "${
          activity._source.name}", event });`,
        flags: { "dnd5e.itemMacro": true }
      });
      break;
    case "Item":
      const itemData = await Item.implementation.fromDropData(dropData);
      if ( !itemData ) {
        ui.notifications.warn("MACRO.5eUnownedWarn", { localize: true });
        return null;
      }
      foundry.utils.mergeObject(macroData, {
        name: itemData.name,
        img: itemData.img,
        command: `dnd5e.documents.macro.rollItem("${itemData._source.name}", { event })`,
        flags: { "dnd5e.itemMacro": true }
      });
      break;
    case "ActiveEffect":
      const effectData = await ActiveEffect.implementation.fromDropData(dropData);
      if ( !effectData ) {
        ui.notifications.warn("MACRO.5eUnownedWarn", { localize: true });
        return null;
      }
      foundry.utils.mergeObject(macroData, {
        name: effectData.name,
        img: effectData.img,
        command: `dnd5e.documents.macro.toggleEffect("${effectData.name}")`,
        flags: { "dnd5e.effectMacro": true }
      });
      break;
    default:
      return;
  }

  // Assign the macro to the hotbar
  const macro = game.macros.find(m => {
    return (m.name === macroData.name) && (m.command === macroData.command) && m.isAuthor;
  }) || await Macro.create(macroData);
  game.user.assignHotbarMacro(macro, slot);
}

/* -------------------------------------------- */

/**
 * Find a document of the specified name and type on an assigned or selected actor.
 * @param {string} name          Document name to locate.
 * @param {string} documentType  Type of embedded document (e.g. "Item" or "ActiveEffect").
 * @returns {Document}           Document if found, otherwise nothing.
 */
function getMacroTarget(name, documentType) {
  let actor;
  const speaker = ChatMessage.getSpeaker();
  if ( speaker.token ) actor = game.actors.tokens[speaker.token];
  actor ??= game.actors.get(speaker.actor);
  if ( !actor ) {
    ui.notifications.warn("MACRO.5eNoActorSelected", {localize: true});
    return null;
  }

  const collection = (documentType === "Item") ? actor.items : Array.from(actor.allApplicableEffects());

  // Find item in collection
  const documents = collection.filter(i => i._source.name === name);
  const type = game.i18n.localize(`DOCUMENT.${documentType}`);
  if ( documents.length === 0 ) {
    ui.notifications.warn(game.i18n.format("MACRO.5eMissingTargetWarn", { actor: actor.name, type, name }));
    return null;
  }
  if ( documents.length > 1 ) {
    ui.notifications.warn(game.i18n.format("MACRO.5eMultipleTargetsWarn", { actor: actor.name, type, name }));
  }
  return documents[0];
}

/* -------------------------------------------- */

/**
 * Trigger an item to be used when a macro is clicked.
 * @param {string} itemName                Name of the item on the selected actor to trigger.
 * @param {object} [options={}]
 * @param {string} [options.activityName]  Name of a specific activity on the item to trigger.
 * @param {Event} [options.event]          The triggering event.
 * @returns {Promise<ChatMessage|object>}  Usage result.
 */
function rollItem(itemName, { activityName, event }={}) {
  let target = getMacroTarget(itemName, "Item");
  if ( activityName ) target = target?.system.activities?.getName(activityName);
  return target?.use({ event, legacy: false });
}

/* -------------------------------------------- */

/**
 * Toggle an effect on and off when a macro is clicked.
 * @param {string} effectName        Name of the effect to be toggled.
 * @returns {Promise<ActiveEffect>}  The effect after it has been toggled.
 */
function toggleEffect(effectName) {
  const effect = getMacroTarget(effectName, "ActiveEffect");
  return effect?.update({ disabled: !effect.disabled });
}

var macro = /*#__PURE__*/Object.freeze({
  __proto__: null,
  create5eMacro: create5eMacro,
  rollItem: rollItem,
  toggleEffect: toggleEffect
});

const slugify$1 = value => value?.slugify().replaceAll("-", "").replaceAll("(", "").replaceAll(")", "");

/**
 * Set up custom text enrichers.
 */
function registerCustomEnrichers() {
  const stringNames = [
    "attack", "award", "check", "concentration", "damage", "heal", "healing", "item", "save", "skill", "tool"
  ];
  CONFIG.TextEditor.enrichers.push({
    pattern: new RegExp(`\\[\\[/(?<type>${stringNames.join("|")})(?<config> .*?)?]](?!])(?:{(?<label>[^}]+)})?`, "gi"),
    enricher: enrichString
  },
  {
    pattern: /\[\[(?<type>lookup) (?<config>[^\]]+)]](?:{(?<label>[^}]+)})?/gi,
    enricher: enrichString
  },
  {
    pattern: /&(?<type>Reference)\[(?<config>[^\]]+)](?:{(?<label>[^}]+)})?/gi,
    enricher: enrichString
  });

  document.body.addEventListener("click", applyAction);
  document.body.addEventListener("click", awardAction);
  document.body.addEventListener("click", rollAction);
}

/* -------------------------------------------- */

/**
 * Parse the enriched string and provide the appropriate content.
 * @param {RegExpMatchArray} match       The regular expression match result.
 * @param {EnrichmentOptions} options    Options provided to customize text enrichment.
 * @returns {Promise<HTMLElement|null>}  An HTML element to insert in place of the matched text or null to
 *                                       indicate that no replacement should be made.
 */
async function enrichString(match, options) {
  let { type, config, label } = match.groups;
  config = parseConfig(config, { multiple: ["damage", "heal", "healing"].includes(type) });
  config._input = match[0];
  config._rules = _getRulesVersion(options);
  switch ( type.toLowerCase() ) {
    case "attack": return enrichAttack(config, label, options);
    case "award": return enrichAward(config, label);
    case "heal":
    case "healing": config._isHealing = true;
    case "damage": return enrichDamage(config, label, options);
    case "check":
    case "skill":
    case "tool": return enrichCheck(config, label, options);
    case "lookup": return enrichLookup(config, label, options);
    case "concentration": config._isConcentration = true;
    case "save": return enrichSave(config, label, options);
    case "item": return enrichItem(config, label, options);
    case "reference": return enrichReference(config, label);
  }
  return null;
}

/* -------------------------------------------- */

/**
 * Parse a roll string into a configuration object.
 * @param {string} match  Matched configuration string.
 * @param {object} [options={}]
 * @param {boolean} [options.multiple=false]  Support splitting configuration by "&" into multiple sub-configurations.
 *                                            If set to `true` then an array of configs will be returned.
 * @returns {object|object[]}
 */
function parseConfig(match="", { multiple=false }={}) {
  if ( multiple ) return match.split("&").map(s => parseConfig(s));
  const config = { _config: match, values: [] };
  for ( const part of match.match(/(?:[^\s"]+|"[^"]*")+/g) ?? [] ) {
    if ( !part ) continue;
    const [key, value] = part.split("=");
    const valueLower = value?.toLowerCase();
    if ( value === undefined ) config.values.push(key.replace(/(^"|"$)/g, ""));
    else if ( ["true", "false"].includes(valueLower) ) config[key] = valueLower === "true";
    else if ( Number.isNumeric(value) ) config[key] = Number(value);
    else config[key] = value.replace(/(^"|"$)/g, "");
  }
  return config;
}

/* -------------------------------------------- */

/**
 * Determine the appropriate rules version based on the provided document or system setting.
 * @param {EnrichmentOptions} options  Options provided to customize text enrichment.
 * @returns {string}
 */
function _getRulesVersion(options) {
  // Select from actor data first, then item data, and then fall back to system setting
  return options.relativeTo?.parent?.system?.source?.rules
    || options.relativeTo?.system?.source?.rules
    || (game.settings.get("dnd5e", "rulesVersion") === "modern" ? "2024" : "2014");
}

/* -------------------------------------------- */
/*  Attack Enricher                             */
/* -------------------------------------------- */

/**
 * Enrich an attack link using a pre-set to hit value.
 * @param {object} config              Configuration data.
 * @param {string} [label]             Optional label to replace default text.
 * @param {EnrichmentOptions} options  Options provided to customize text enrichment.
 * @returns {HTMLElement|null}         An HTML link if the attack could be built, otherwise null.
 *
 * @example Create an attack link using a fixed to hit:
 * ```[[/attack +5]]``` or ```[[/attack formula=5]]```
 * becomes
 * ```html
 * <a class="roll-action" data-type="attack" data-formula="+5">
 *   <i class="fa-solid fa-dice-d20" inert></i> +5
 * </a>
 * ```
 *
 * @example Create an attack link using a specific attack mode:
 * ```[[/attack +5]]``` or ```[[/attack formula=5 attackMode=thrown]]```
 * becomes
 * ```html
 * <a class="roll-action" data-type="attack" data-formula="+5" data-attack-mode="thrown">
 *   <i class="fa-solid fa-dice-d20" inert></i> +5
 * </a>
 * ```
 *
 * @example Link an enricher to an attack activity, either explicitly or automatically:
 * ```[[/attack activity=RLQlsLo5InKHZadn]]``` or ```[[/attack]]```
 * becomes
 * ```html
 * <a class="roll-action" data-type="attack" data-formula="+8" data-activity-uuid="...uuid...">
 *   <i class="fa-solid fa-dice-d20" inert"></i> +8
 * </a>
 * ```
 *
 * @example Display the full attack section:
 * ```[[/attack format=extended]]``` or ```[[/attack extended]]```
 * becomes
 * ```html
 * <span class="attack-extended">
 *   <em>Melee Attack Roll</em>:
 *   <span class="roll-link-group" data-type="attack" data-formula="+16" data-activity-uuid="...uuid...">
 *     <a class="roll-link"><i class="fa-solid fa-dice-d20" inert"></i> +16</a>
 *   </span>, reach 15 ft
 * </span>
 * ```
 */
async function enrichAttack(config, label, options) {
  if ( config.activity && config.formula ) {
    console.warn(`Activity ID and formula found while enriching ${config._input}, only one is supported.`);
    return null;
  }

  const formulaParts = [];
  if ( config.formula ) formulaParts.push(config.formula);
  for ( const value of config.values ) {
    if ( value in CONFIG.DND5E.attackModes ) config.attackMode = value;
    else if ( value === "extended" ) config.format = "extended";
    else formulaParts.push(value);
  }
  config.formula = Roll.defaultImplementation.replaceFormulaData(formulaParts.join(" "), options.rollData ?? {});

  const activity = config.activity ? options.relativeTo?.system?.activities?.get(config.activity)
    : !config.formula ? options.relativeTo?.system?.activities?.getByType("attack")[0] : null;

  if ( activity ) {
    if ( activity.type !== "attack" ) {
      console.warn(`Attack enricher linked to non-attack activity when enriching ${config._input}`);
      return null;
    }

    config.activityUuid = activity.uuid;
    const attackConfig = activity.getAttackData({ attackMode: config.attackMode });
    config.formula = simplifyRollFormula(
      Roll.defaultImplementation.replaceFormulaData(attackConfig.parts.join(" + "), attackConfig.data)
    );
    delete config.activity;
  }

  if ( !config.activityUuid && !config.formula ) {
    console.warn(`No formula or linked activity found while enriching ${config._input}.`);
    return null;
  }

  config.type = "attack";
  if ( label ) return createRollLink(label, config, { classes: "roll-link-group roll-link" });

  let displayFormula = simplifyRollFormula(config.formula) || "+0";
  if ( !displayFormula.startsWith("+") && !displayFormula.startsWith("-") ) displayFormula = `+${displayFormula}`;

  const span = document.createElement("span");
  span.className = "roll-link-group";
  _addDataset(span, config);
  span.innerHTML = game.i18n.format(`EDITOR.DND5E.Inline.Attack${config._rules === "2014" ? "Long" : "Short"}`, {
    formula: createRollLink(displayFormula).outerHTML
  });

  if ( config.format === "extended" ) {
    const type = game.i18n.format(`DND5E.ATTACK.Formatted.${config._rules}`, {
      type: game.i18n.getListFormatter({ type: "disjunction" }).format(
        Array.from(activity?.validAttackTypes ?? []).map(t => CONFIG.DND5E.attackTypes[t]?.label)
      ),
      classification: CONFIG.DND5E.attackClassifications[activity?.attack.type.classification]?.label ?? ""
    }).trim();
    const parts = [span.outerHTML, activity?.getRangeLabel(config.attackMode)];
    if ( config._rules === "2014" ) parts.push(activity?.target?.affects.labels?.statblock);

    const full = document.createElement("span");
    full.className = "attack-extended";
    full.innerHTML = game.i18n.format("EDITOR.DND5E.Inline.AttackExtended", {
      type, parts: game.i18n.getListFormatter({ type: "unit" }).format(parts.filter(_ => _))
    });
    return full;
  }

  return span;
}

/* -------------------------------------------- */
/*  Award Enricher                              */
/* -------------------------------------------- */

/**
 * Enrich an award block displaying amounts for each part granted with a GM-control for awarding to the party.
 * @param {object} config              Configuration data.
 * @param {string} [label]             Optional label to replace default text.
 * @param {EnrichmentOptions} options  Options provided to customize text enrichment.
 * @returns {HTMLElement|null}         An HTML link if the check could be built, otherwise null.
 */
async function enrichAward(config, label, options) {
  const command = config._config;
  let parsed;
  try {
    parsed = Award.parseAwardCommand(command);
  } catch(err) {
    console.warn(err.message);
    return null;
  }

  const block = document.createElement("span");
  block.classList.add("award-block", "dnd5e2");
  block.dataset.awardCommand = command;

  const entries = [];
  for ( let [key, amount] of Object.entries(parsed.currency) ) {
    const label = CONFIG.DND5E.currencies[key].label;
    amount = Number.isNumeric(amount) ? formatNumber(amount) : amount;
    entries.push(`
      <span class="award-entry">
        ${amount} <i class="currency ${key}" data-tooltip="${label}" aria-label="${label}"></i>
      </span>
    `);
  }
  if ( parsed.xp ) entries.push(`
    <span class="award-entry">
      ${formatNumber(parsed.xp)} ${game.i18n.localize("DND5E.ExperiencePoints.Abbreviation")}
    </span>
  `);

  let award = game.i18n.getListFormatter({ type: "unit" }).format(entries);
  if ( parsed.each ) award = game.i18n.format("EDITOR.DND5E.Inline.AwardEach", { award });

  block.innerHTML += `
    ${award}
    <a class="award-link" data-action="awardRequest">
      <i class="fa-solid fa-trophy"></i> ${label ?? game.i18n.localize("DND5E.Award.Action")}
    </a>
  `;

  return block;
}

/* -------------------------------------------- */
/*  Check & Save Enrichers                      */
/* -------------------------------------------- */

/**
 * Enrich an ability check link to perform a specific ability or skill check. If an ability is provided
 * along with a skill, then the skill check will always use the provided ability. Otherwise it will use
 * the character's default ability for that skill.
 * @param {object} config              Configuration data.
 * @param {string} [label]             Optional label to replace default text.
 * @param {EnrichmentOptions} options  Options provided to customize text enrichment.
 * @returns {HTMLElement|null}         An HTML link if the check could be built, otherwise null.
 *
 * @example Create a dexterity check:
 * ```[[/check ability=dex]]```
 * becomes
 * ```html
 * <a class="roll-action" data-type="check" data-ability="dex">
 *   <i class="fa-solid fa-dice-d20" inert></i> Dexterity check
 * </a>
 * ```
 *
 * @example Create an acrobatics check with a DC and default ability:
 * ```[[/check skill=acr dc=20]]```
 * becomes
 * ```html
 * <a class="roll-action" data-type="check" data-skill="acr" data-dc="20">
 *   <i class="fa-solid fa-dice-d20" inert></i> DC 20 Dexterity (Acrobatics) check
 * </a>
 * ```
 *
 * @example Create an acrobatics check using strength:
 * ```[[/check ability=str skill=acr]]```
 * becomes
 * ```html
 * <a class="roll-action" data-type="check" data-ability="str" data-skill="acr">
 *   <i class="fa-solid fa-dice-d20" inert></i> Strength (Acrobatics) check
 * </a>
 * ```
 *
 * @example Create a tool check:
 * ```[[/check tool=thief ability=int]]```
 * becomes
 * ```html
 * <a class="roll-action" data-type="check" data-ability="int" data-tool="thief">
 *   <i class="fa-solid fa-dice-d20" inert></i> Intelligence (Thieves' Tools) check
 * </a>
 * ```
 *
 * @example Formulas used for DCs will be resolved using data provided to the description (not the roller):
 * ```[[/check ability=cha dc=@abilities.int.dc]]```
 * becomes
 * ```html
 * <a class="roll-action" data-type="check" data-ability="cha" data-dc="15">
 *   <i class="fa-solid fa-dice-d20" inert></i> DC 15 Charisma check
 * </a>
 * ```
 *
 * @example Use multiple skills in a check using default abilities:
 * ```[[/check skill=acr/ath dc=15]]```
 * ```[[/check acrobatics athletics 15]]```
 * becomes
 * ```html
 * <span class="roll-link-group" data-type="check" data-skill="acr|ath" data-dc="15">
 *   DC 15
 *   <a class="roll-action" data-ability="dex" data-skill="acr">
 *     <i class="fa-solid fa-dice-d20" inert></i> Dexterity (Acrobatics)
 *   </a> or
 *   <a class="roll-action" data-ability="dex">
 *     <i class="fa-solid fa-dice-d20" inert></i> Strength (Athletics)
 *   </a>
 *   <a class="enricher-action" data-action="request" ...><!-- request link --></a>
 * </span>
 * ```
 *
 * @example Use multiple skills with a fixed ability:
 * ```[[/check ability=str skill=dec/per dc=15]]```
 * ```[[/check strength deception persuasion 15]]```
 * becomes
 * ```html
 * <span class="roll-link-group" data-type="check" data-ability="str" data-skill="dec|per" data-dc="15">
 *   DC 15 Strength
 *   (<a class="roll-action" data-skill="dec"><i class="fa-solid fa-dice-d20" inert></i> Deception</a> or
 *   <a class="roll-action" data-ability="per"><i class="fa-solid fa-dice-d20" inert></i> Persuasion</a>)
 *   <a class="enricher-action" data-action="request" ...><!-- request link --></a>
 * </span>
 * ```
 *
 * @example Link an enricher to an check activity, either explicitly or automatically
 * ```[[/check activity=RLQlsLo5InKHZadn]]``` or ```[[/check]]```
 * becomes
 * ```html
 * <span class="roll-link-group" data-type="check" data-ability="dex" data-dc="20" data-activity-uuid="...">
 *   <a class="roll-action"><i class="fa-solid fa-dice-d20" inert></i> DC 20 Dexterity</a>
 *   <a class="enricher-action" data-action="request" ...><!-- request link --></a>
 * </span>
 * ```
 */
async function enrichCheck(config, label, options) {
  config.skill = config.skill?.replaceAll("/", "|").split("|") ?? [];
  config.tool = config.tool?.replaceAll("/", "|").split("|") ?? [];
  for ( let value of config.values ) {
    const slug = foundry.utils.getType(value) === "string" ? slugify$1(value) : value;
    if ( slug in CONFIG.DND5E.enrichmentLookup.abilities ) config.ability = slug;
    else if ( slug in CONFIG.DND5E.enrichmentLookup.skills ) config.skill.push(slug);
    else if ( slug in CONFIG.DND5E.enrichmentLookup.tools ) config.tool.push(slug);
    else if ( Number.isNumeric(value) ) config.dc = Number(value);
    else config[value] = true;
  }
  delete config.values;

  const groups = new Map();
  let invalid = false;

  const anything = config.ability || config.skill.length || config.tool.length;
  const activity = config.activity ? options.relativeTo?.system?.activities?.get(config.activity)
    : !anything ? options.relativeTo?.system?.activities?.getByType("check")[0] : null;

  if ( activity ) {
    if ( activity.type !== "check" ) {
      console.warn(`Check enricher linked to non-check activity when enriching ${config._input}.`);
      return null;
    }

    if ( activity.check.ability ) config.ability = activity.check.ability;
    config.activityUuid = activity.uuid;
    config.dc = activity.check.dc.value;
    config.skill = [];
    config.tool = [];
    for ( const associated of activity.check.associated ) {
      if ( associated in CONFIG.DND5E.skills ) config.skill.push(associated);
      else if ( associated in CONFIG.DND5E.tools ) config.tool.push(associated);
    }
    delete config.activity;
  }

  // TODO: Support "spellcasting" ability
  let abilityConfig = CONFIG.DND5E.enrichmentLookup.abilities[slugify$1(config.ability)];
  if ( config.ability && !abilityConfig ) {
    console.warn(`Ability "${config.ability}" not found while enriching ${config._input}.`);
    invalid = true;
  } else if ( abilityConfig?.key ) config.ability = abilityConfig.key;

  for ( let [index, skill] of config.skill.entries() ) {
    const skillConfig = CONFIG.DND5E.enrichmentLookup.skills[slugify$1(skill)];
    if ( skillConfig ) {
      if ( skillConfig.key ) skill = config.skill[index] = skillConfig.key;
      const ability = config.ability || skillConfig.ability;
      if ( !groups.has(ability) ) groups.set(ability, []);
      groups.get(ability).push({ key: skill, type: "skill", label: skillConfig.label });
    } else {
      console.warn(`Skill "${skill}" not found while enriching ${config._input}.`);
      invalid = true;
    }
  }

  for ( const tool of config.tool ) {
    const toolConfig = CONFIG.DND5E.tools[slugify$1(tool)];
    const toolUUID = CONFIG.DND5E.enrichmentLookup.tools[slugify$1(tool)];
    const toolIndex = toolUUID ? getBaseItem(toolUUID, { indexOnly: true }) : null;
    if ( toolIndex ) {
      const ability = config.ability || toolConfig?.ability;
      if ( ability ) {
        if ( !groups.has(ability) ) groups.set(ability, []);
        groups.get(ability).push({ key: tool, type: "tool", label: toolIndex.name });
      } else {
        console.warn(`Tool "${tool}" found without specified or default ability while enriching ${config._input}.`);
        invalid = true;
      }
    } else {
      console.warn(`Tool "${tool}" not found while enriching ${config._input}.`);
      invalid = true;
    }
  }

  if ( !abilityConfig && !groups.size ) {
    console.warn(`No ability, skill, tool, or linked activity provided while enriching ${config._input}.`);
    invalid = true;
  }

  const complex = (config.skill.length + config.tool.length) > 1;
  if ( config.passive && complex ) {
    console.warn(`Multiple skills or tools and passive flag found while enriching ${config._input}, which aren't supported together.`);
    invalid = true;
  }
  if ( label && complex ) {
    console.warn(`Multiple skills or tools and a custom label found while enriching ${config._input}, which aren't supported together.`);
    invalid = true;
  }

  if ( config.dc && !Number.isNumeric(config.dc) ) config.dc = simplifyBonus(config.dc, options.rollData);

  if ( invalid ) return null;

  if ( complex ) {
    const formatter = game.i18n.getListFormatter({ type: "disjunction" });
    const parts = [];
    for ( const [ability, associated] of groups.entries() ) {
      const makeConfig = ({ key, type }) => ({ type, [type]: key, ability: groups.size > 1 ? ability : undefined });

      // Multiple associated proficiencies, link each individually
      if ( associated.length > 1 ) parts.push(
        game.i18n.format("EDITOR.DND5E.Inline.SpecificCheck", {
          ability: CONFIG.DND5E.enrichmentLookup.abilities[ability].label,
          type: formatter.format(associated.map(a => createRollLink(a.label, makeConfig(a)).outerHTML ))
        })
      );

      // Only single associated proficiency, wrap whole thing in roll link
      else {
        const associatedConfig = makeConfig(associated[0]);
        parts.push(createRollLink(createRollLabel({ ...associatedConfig, ability }), associatedConfig).outerHTML);
      }
    }
    label = formatter.format(parts);
    if ( config.dc && !config.hideDC ) {
      label = game.i18n.format("EDITOR.DND5E.Inline.DC", { dc: config.dc, check: label });
    }
    label = game.i18n.format(`EDITOR.DND5E.Inline.Check${config.format === "long" ? "Long" : "Short"}`, { check: label });
    const template = document.createElement("template");
    template.innerHTML = label;
    return createRequestLink(template, {
      type: "check", ...config, skill: config.skill.join("|"), tool: config.tool.join("|")
    });
  }

  const type = config.skill.length ? "skill" : config.tool.length ? "tool" : "check";
  config = { type, ability: Array.from(groups.keys())[0], ...config, skill: config.skill[0], tool: config.tool[0] };
  if ( !label ) label = createRollLabel(config);
  return config.passive ? createPassiveTag(label, config) : createRequestLink(createRollLink(label), config);
}

/* -------------------------------------------- */

/**
 * Create the buttons for a check requested in chat.
 * @param {object} dataset
 * @returns {object[]}
 */
function createCheckRequestButtons(dataset) {
  const skills = dataset.skill?.split("|") ?? [];
  const tools = dataset.tool?.split("|") ?? [];
  if ( (skills.length + tools.length) <= 1 ) return [createRequestButton(dataset)];
  const baseDataset = { ...dataset };
  delete baseDataset.skill;
  delete baseDataset.tool;
  return [
    ...skills.map(skill => createRequestButton({
      ability: CONFIG.DND5E.skills[skill].ability, ...baseDataset, format: "short", skill, type: "skill"
    })),
    ...tools.map(tool => createRequestButton({
      ability: CONFIG.DND5E.tools[tool]?.ability, ...baseDataset, format: "short", tool, type: "tool"
    }))
  ];
}

/* -------------------------------------------- */

/**
 * Enrich a saving throw link.
 * @param {object} config              Configuration data.
 * @param {string} [label]             Optional label to replace default text.
 * @param {EnrichmentOptions} options  Options provided to customize text enrichment.
 * @returns {HTMLElement|null}         An HTML link if the save could be built, otherwise null.
 *
 * @example Create a dexterity saving throw:
 * ```[[/save ability=dex]]```
 * becomes
 * ```html
 * <span class="roll-link-group" data-type="save" data-ability="dex">
 *   <a class="roll-action"><i class="fa-solid fa-dice-d20" inert></i> Dexterity</a>
 *   <a class="enricher-action" data-action="request" ...><!-- request link --></a>
 * </span>
 * ```
 *
 * @example Add a DC to the save:
 * ```[[/save ability=dex dc=20]]```
 * becomes
 * ```html
 * <span class="roll-link-group" data-type="save" data-ability="dex" data-dc="20">
 *   <a class="roll-action"><i class="fa-solid fa-dice-d20" inert></i> DC 20 Dexterity</a>
 *   <a class="enricher-action" data-action="request" ...><!-- request link --></a>
 * </span>
 * ```
 *
 * @example Specify multiple abilities:
 * ```[[/save ability=str/dex dc=20]]```
 * ```[[/save strength dexterity 20]]```
 * becomes
 * ```html
 * <span class="roll-link-group" data-type="save" data-ability="str|dex" data-dc="20">
 *   DC 20
 *   <a class="roll-action" data-ability="str"><i class="fa-solid fa-dice-d20" inert></i> Strength</a> or
 *   <a class="roll-action" data-ability="dex"><i class="fa-solid fa-dice-d20" inert></i> Dexterity</a>
 *   <a class="enricher-action" data-action="request" ...><!-- request link --></a>
 * </span>
 * ```
 *
 * @example Create a concentration saving throw:
 * ```[[/concentration 10]]```
 * becomes
 * ```html
 * <span class="roll-link-group" data-type="concentration" data-dc=10>
 *   <a class="roll-action"><i class="fa-solid fa-dice-d20" inert></i> DC 10 concentration</a>
 *   <a class="enricher-action" data-action="request" ...><!-- request link --></a>
 * </span>
 * ```
 *
 * @example Link an enricher to an save activity, either explicitly or automatically
 * ```[[/save activity=RLQlsLo5InKHZadn]]``` or ```[[/save]]```
 * becomes
 * ```html
 * <span class="roll-link-group" data-type="save" data-ability="dex" data-dc="20" data-activity-uuid="...">
 *   <a class="roll-action"><i class="fa-solid fa-dice-d20" inert></i> DC 20 Dexterity</a>
 *   <a class="enricher-action" data-action="request" ...><!-- request link --></a>
 * </span>
 * ```
 */
async function enrichSave(config, label, options) {
  config.ability = config.ability?.replace("/", "|").split("|") ?? [];
  for ( let value of config.values ) {
    const slug = foundry.utils.getType(value) === "string" ? slugify$1(value) : value;
    if ( slug in CONFIG.DND5E.enrichmentLookup.abilities ) config.ability.push(slug);
    else if ( Number.isNumeric(value) ) config.dc = Number(value);
    else config[value] = true;
  }
  config.ability = config.ability
    .filter(a => a in CONFIG.DND5E.enrichmentLookup.abilities)
    .map(a => CONFIG.DND5E.enrichmentLookup.abilities[a].key ?? a);

  const activity = config.activity ? options.relativeTo?.system?.activities?.get(config.activity)
    : !config.ability.length ? options.relativeTo?.system?.activities?.getByType("save")[0] : null;

  if ( activity ) {
    if ( activity.type !== "save" ) {
      console.warn(`Save enricher linked to non-save activity when enriching ${config._input}`);
      return null;
    }

    config.ability = Array.from(activity.save.ability);
    config.activityUuid = activity.uuid;
    config.dc = activity.save.dc.value;
    delete config.activity;
  }

  if ( !config.ability.length && !config._isConcentration ) {
    console.warn(`No ability or linked activity found while enriching ${config._input}.`);
    return null;
  }

  if ( config.dc && !Number.isNumeric(config.dc) ) config.dc = simplifyBonus(config.dc, options.rollData);

  if ( config.ability.length > 1 && label ) {
    console.warn(`Multiple abilities and custom label found while enriching ${config._input}, which aren't supported together.`);
    return null;
  }

  config = { type: config._isConcentration ? "concentration" : "save", ...config };
  if ( label ) label = createRollLink(label);
  else if ( config.ability.length <= 1 ) label = createRollLink(createRollLabel(config));
  else {
    label = game.i18n.getListFormatter({ type: "disjunction" }).format(config.ability.map(ability =>
      createRollLink(createRollLabel({ type: "save", ability }), { ability }).outerHTML
    ));
    if ( config.dc && !config.hideDC ) {
      label = game.i18n.format("EDITOR.DND5E.Inline.DC", { dc: config.dc, check: label });
    }
    label = game.i18n.format(`EDITOR.DND5E.Inline.Save${config.format === "long" ? "Long" : "Short"}`, { save: label });
    const template = document.createElement("template");
    template.innerHTML = label;
    label = template;
  }
  return createRequestLink(label, { ...config, ability: config.ability.join("|") });
}

/* -------------------------------------------- */

/**
 * Create the buttons for a save requested in chat.
 * @param {object} dataset
 * @returns {object[]}
 */
function createSaveRequestButtons(dataset) {
  return (dataset.ability?.split("|") ?? [])
    .map(ability => createRequestButton({ ...dataset, format: "long", ability }));
}

/* -------------------------------------------- */
/*  Damage Enricher                             */
/* -------------------------------------------- */

/**
 * Enrich a damage link.
 * @param {object[]} configs           Configuration data.
 * @param {string} [label]             Optional label to replace default text.
 * @param {EnrichmentOptions} options  Options provided to customize text enrichment.
 * @returns {HTMLElement|null}         An HTML link if the save could be built, otherwise null.
 *
 * @example Create a damage link:
 * ```[[/damage 2d6 type=bludgeoning]]``
 * becomes
 * ```html
 * <a class="roll-link-group" data-type="damage" data-formulas="2d6" data-damage-types="bludgeoning">
 *   <span class="roll-link"><i class="fa-solid fa-dice-d20"></i> 2d6</span> bludgeoning
 * </a>
 * ````
 *
 * @example Display the average:
 * ```[[/damage 2d6 type=bludgeoning average=true]]``
 * becomes
 * ```html
 * 7 (<a class="roll-link-group" data-type="damage" data-formulas="2d6" data-damage-types="bludgeoning">
 *   <span class="roll-link"><i class="fa-solid fa-dice-d20"></i> 2d6</span>
 * </a>) bludgeoning
 * ````
 *
 * @example Manually set the average & don't prefix the type:
 * ```[[/damage 8d4dl force average=666]]``
 * becomes
 * ```html
 * 666 (<a class="roll-link-group" data-type="damage" data-formulas="8d4dl" data-damage-types="force">
 *   <span class="roll-link"><i class="fa-solid fa-dice-d20"></i> 8d4dl</span>
 * </a> force
 * ````
 *
 * @example Create a healing link:
 * ```[[/heal 2d6]]``` or ```[[/damage 2d6 healing]]```
 * becomes
 * ```html
 * <a class="roll-link-group" data-type="damage" data-formulas="2d6" data-damage-types="healing">
 *   <span class="roll-link"><i class="fa-solid fa-dice-d20"></i> 2d6</span>
 * </a> healing
 * ```
 *
 * @example Specify variable damage types:
 * ```[[/damage 2d6 type=fire|cold]]``` or ```[[/damage 2d6 type=fire/cold]]```
 * becomes
 * ```html
 * <a class="roll-link-group" data-type="damage" data-formulas="2d6" data-damage-types="fire|cold">
 *   <span class="roll-link"><i class="fa-solid fa-dice-d20"></i> 2d6</span>
 * </a> fire or cold
 * ```
 *
 * @example Add multiple damage parts
 * ```[[/damage 1d6 fire & 1d6 cold]]```
 * becomes
 * ```html
 * <a class="roll-link-group" data-type="damage" data-formulas="1d6&1d6" data-damage-types="fire&cold">
 *   <span class="roll-link"><i class="fa-solid fa-dice-d20"></i> 1d6</span> fire and
 *   <span class="roll-link"><i class="fa-solid fa-dice-d20"></i> 1d6</span> cold
 * </a>
 * ```
 *
 * @example Link an enricher to an damage activity, either explicitly or automatically
 * ```[[/damage activity=RLQlsLo5InKHZadn]]``` or ```[[/damage]]```
 * becomes
 * ```html
 * <a class="roll-link-group" data-type="damage" data-formulas="1d6&1d6" data-damage-types="fire&cold"
 *    data-activity-uuid="...">
 *   <span class="roll-link"><i class="fa-solid fa-dice-d20"></i> 1d6</span> fire and
 *   <span class="roll-link"><i class="fa-solid fa-dice-d20"></i> 1d6</span> cold
 * </a>
 * ```
 *
 * @example Displaying the full hit section:
 * ```[[/damage extended]]``
 * becomes
 * ```html
 * <span class="damage-extended">
 *   <em>Hit:</em>
 *   <a class="roll-link-group" data-type="damage" data-formulas="2d6" data-damage-types="bludgeoning"
 *      data-activity-uuid="...">
 *     7 (<span class="roll-link"><i class="fa-solid fa-dice-d20"></i> 2d6</span></a>) Bludgeoning damage
 *   </a>
 * </span>
 * ````
 */
async function enrichDamage(configs, label, options) {
  const config = { type: "damage", formulas: [], damageTypes: [], rollType: configs._isHealing ? "healing" : "damage" };
  for ( const c of configs ) {
    const formulaParts = [];
    if ( c.activity ) config.activity = c.activity;
    if ( c.attackMode ) config.attackMode = c.attackMode;
    if ( c.average ) config.average = c.average;
    if ( c.format ) config.format = c.format;
    if ( c.formula ) formulaParts.push(c.formula);
    c.type = c.type?.replaceAll("/", "|").split("|") ?? [];
    for ( const value of c.values ) {
      if ( value in CONFIG.DND5E.damageTypes ) c.type.push(value);
      else if ( value in CONFIG.DND5E.healingTypes ) c.type.push(value);
      else if ( value in CONFIG.DND5E.attackModes ) config.attackMode = value;
      else if ( value === "average" ) config.average = true;
      else if ( value === "extended" ) config.format = "extended";
      else if ( value === "temp" ) c.type.push("temphp");
      else formulaParts.push(value);
    }
    c.formula = Roll.defaultImplementation.replaceFormulaData(formulaParts.join(" "), options.rollData ?? {});
    if ( configs._isHealing && !c.type.length ) c.type.push("healing");
    if ( c.formula ) {
      config.formulas.push(c.formula);
      config.damageTypes.push(c.type.join("|"));
    }
  }
  config.damageTypes = config.damageTypes.map(t => t?.replace("/", "|"));
  if ( config.format === "extended" ) config.average ??= true;

  if ( config.activity && config.formulas.length ) {
    console.warn(`Activity ID and formulas found while enriching ${config._input}, only one is supported.`);
    return null;
  }

  let activity = options.relativeTo?.system?.activities?.get(config.activity);
  if ( !activity && !config.formulas.length ) {
    const types = configs._isHealing ? ["heal"] : ["attack", "damage", "save"];
    for ( const a of options.relativeTo?.system?.activities?.getByTypes(...types) ?? [] ) {
      if ( a.damage?.parts.length || a.healing?.formula ) {
        activity = a;
        break;
      }
    }
  }

  if ( activity ) {
    config.activityUuid = activity.uuid;
    const damageConfig = activity.getDamageConfig({ attackMode: config.attackMode });
    for ( const roll of damageConfig.rolls ) {
      config.formulas.push(simplifyRollFormula(
        Roll.defaultImplementation.replaceFormulaData(roll.parts.join(" + "), roll.data)
      ));
      config.damageTypes.push(roll.options.types?.join("|") ?? roll.options.type);
    }
    delete config.activity;
  }

  if ( !config.activityUuid && !config.formulas.length ) {
    console.warn(`No formula or linked activity found while enriching ${config._input}.`);
    return null;
  }

  const formulas = config.formulas.join("&");
  const damageTypes = config.damageTypes.join("&");

  if ( !config.formulas.length ) return null;
  if ( label ) {
    return createRollLink(label, { ...config, formulas, damageTypes }, { classes: "roll-link-group roll-link" });
  }

  const parts = [];
  for ( const [idx, formula] of config.formulas.entries() ) {
    const type = config.damageTypes[idx];
    const types = type?.split("|")
      .map(t => CONFIG.DND5E.damageTypes[t]?.label ?? CONFIG.DND5E.healingTypes[t]?.label)
      .filter(_ => _);
    const localizationData = {
      formula: createRollLink(formula, {}, { tag: "span" }).outerHTML,
      type: game.i18n.getListFormatter({ type: "disjunction" }).format(types)
    };
    if ( configs._rules === "2014" ) localizationData.type = localizationData.type.toLowerCase();

    let localizationType = "Short";
    if ( config.average ) {
      localizationType = "Long";
      if ( config.average === true ) {
        const minRoll = Roll.create(formula).evaluate({ minimize: true });
        const maxRoll = Roll.create(formula).evaluate({ maximize: true });
        localizationData.average = Math.floor(((await minRoll).total + (await maxRoll).total) / 2);
      } else if ( Number.isNumeric(config.average) ) {
        localizationData.average = config.average;
      } else {
        localizationType = "Short";
      }
    }

    parts.push(game.i18n.format(`EDITOR.DND5E.Inline.Damage${localizationType}`, localizationData));
  }

  const link = document.createElement("a");
  link.className = "roll-link-group";
  _addDataset(link, { ...config, formulas, damageTypes });
  if ( config.average && (parts.length === 2) ) {
    link.innerHTML = game.i18n.format("EDITOR.DND5E.Inline.DamageDouble", { first: parts[0], second: parts[1] });
  } else {
    link.innerHTML = game.i18n.getListFormatter().format(parts);
  }

  if ( config.format === "extended" ) {
    const span = document.createElement("span");
    span.className = "damage-extended";
    span.innerHTML = game.i18n.format("EDITOR.DND5E.Inline.DamageExtended", { damage: link.outerHTML });
    return span;
  }

  return link;
}

/* -------------------------------------------- */
/*  Lookup Enricher                             */
/* -------------------------------------------- */

/**
 * Enrich a property lookup.
 * @param {object} config              Configuration data.
 * @param {string} [fallback]          Optional fallback if the value couldn't be found.
 * @param {EnrichmentOptions} options  Options provided to customize text enrichment.
 * @returns {HTMLElement|null}         An HTML element if the lookup could be built, otherwise null.
 *
 * @example Include a creature's name in its description:
 * ```[[lookup @name]]```
 * becomes
 * ```html
 * <span class="lookup-value">Adult Black Dragon</span>
 * ```
 *
 * @example Lookup a property within an activity:
 * ```[[lookup @target.template.size activity=dnd5eactivity000]]```
 * becomes
 * ```html
 * <span class="lookup-value">120</span>
 * ```
 */
function enrichLookup(config, fallback, options) {
  let keyPath = config.path;
  let style = config.style;
  for ( const value of config.values ) {
    if ( value === "capitalize" ) style ??= "capitalize";
    else if ( value === "lowercase" ) style ??= "lowercase";
    else if ( value === "uppercase" ) style ??= "uppercase";
    else if ( value.startsWith("@") ) keyPath ??= value;
  }

  let activity = options.relativeTo?.system?.activities?.get(config.activity);
  if ( config.activity && !activity ) {
    console.warn(`Activity not found when enriching ${config._input}.`);
    return null;
  }

  if ( !keyPath ) {
    console.warn(`Lookup path must be defined to enrich ${config._input}.`);
    return null;
  }

  const data = activity ? activity.getRollData().activity : options.rollData
    ?? options.relativeTo?.getRollData?.() ?? {};
  let value = foundry.utils.getProperty(data, keyPath.substring(1)) ?? fallback;
  if ( value && style ) {
    if ( style === "capitalize" ) value = value.capitalize();
    else if ( style === "lowercase" ) value = value.toLowerCase();
    else if ( style === "uppercase" ) value = value.toUpperCase();
  }

  const span = document.createElement("span");
  span.classList.add("lookup-value");
  if ( !value && (options.documents === false) ) return null;
  if ( !value ) span.classList.add("not-found");
  span.innerText = value ?? keyPath;
  return span;
}

/* -------------------------------------------- */
/*  Reference Enricher                          */
/* -------------------------------------------- */

/**
 * Enrich a reference link.
 * @param {object} config              Configuration data.
 * @param {string} [label]             Optional label to replace default text.
 * @param {EnrichmentOptions} options  Options provided to customize text enrichment.
 * @returns {HTMLElement|null}         An HTML link to the Journal Entry Page for the given reference.
 *
 * @example Create a content link to the relevant reference:
 * ```&Reference[condition=unconscious]{Label}```
 * becomes
 * ```html
 * <span class="reference-link">
 *   <a class="content-link" draggable="true"
 *      data-uuid="Compendium.dnd5e.rules.JournalEntry.w7eitkpD7QQTB6j0.JournalEntryPage.UWw13ISmMxDzmwbd"
 *      data-type="JournalEntryPage" data-tooltip="Text Page">
 *     <i class="fas fa-book-open"></i> Label
 *   </a>
 *   <a class="enricher-action" data-action="apply" data-status="unconscious"
 *      data-tooltip="EDITOR.DND5E.Inline.ApplyStatus" aria-label="Apply Status to Selected Tokens">
 *     <i class="fas fa-fw fa-reply-all fa-flip-horizontal"></i>
 *   </a>
 * </span>
 * ```
 */
async function enrichReference(config, label, options) {
  let key;
  let source;
  let isCondition = "condition" in config;
  const type = Object.keys(config).find(k => k in CONFIG.DND5E.ruleTypes);
  if ( type ) {
    key = slugify$1(config[type]);
    source = foundry.utils.getProperty(CONFIG.DND5E, CONFIG.DND5E.ruleTypes[type].references)?.[key];
  } else if ( config.values.length ) {
    key = slugify$1(config.values.join(""));
    for ( const [type, { references }] of Object.entries(CONFIG.DND5E.ruleTypes) ) {
      source = foundry.utils.getProperty(CONFIG.DND5E, references)[key];
      if ( source ) {
        if ( type === "condition" ) isCondition = true;
        break;
      }
    }
  }
  if ( !source ) {
    console.warn(`No valid rule found while enriching ${config._input}.`);
    return null;
  }
  const uuid = foundry.utils.getType(source) === "Object" ? source.reference : source;
  if ( !uuid ) return null;
  const doc = await fromUuid(uuid);
  const span = document.createElement("span");
  span.classList.add("reference-link");
  span.append(doc.toAnchor({ name: label || doc.name }));
  if ( isCondition && (config.apply !== false) ) {
    const apply = document.createElement("a");
    apply.classList.add("enricher-action");
    apply.dataset.action = "apply";
    apply.dataset.status = key;
    apply.dataset.tooltip = "EDITOR.DND5E.Inline.ApplyStatus";
    apply.setAttribute("aria-label", game.i18n.localize(apply.dataset.tooltip));
    apply.innerHTML = '<i class="fas fa-fw fa-reply-all fa-flip-horizontal"></i>';
    span.append(apply);
  }
  return span;
}

/* -------------------------------------------- */
/*  Helpers                                     */
/* -------------------------------------------- */

/**
 * Enrich an item use link to roll an item on the selected token.
 * @param {string[]} config              Configuration data.
 * @param {string} [label]               Optional label to replace default text.
 * @param {EnrichmentOptions} options  Options provided to customize text enrichment.
 * @returns {Promise<HTMLElement|null>}  An HTML link if the item link could be built, otherwise null.
 *
 * @example Use an Item from a name:
 * ```[[/item Heavy Crossbow]]```
 * becomes
 * ```html
 * <a class="roll-action" data-type="item" data-roll-item-name="Heavy Crossbow">
 *   <i class="fa-solid fa-dice-d20"></i> Heavy Crossbow
 * </a>
 * ```
 *
 * @example Use an Item from a UUID:
 * ```[[/item Actor.M4eX4Mu5IHCr3TMf.Item.amUUCouL69OK1GZU]]```
 * becomes
 * ```html
 * <a class="roll-action" data-type="item" data-roll-item-uuid="Actor.M4eX4Mu5IHCr3TMf.Item.amUUCouL69OK1GZU">
 *   <i class="fa-solid fa-dice-d20"></i> Bite
 * </a>
 * ```
 *
 * @example Use an Item from an ID:
 * ```[[/item amUUCouL69OK1GZU]]```
 * becomes
 * ```html
 * <a class="roll-action" data-type="item" data-roll-item-uuid="Actor.M4eX4Mu5IHCr3TMf.Item.amUUCouL69OK1GZU">
 *   <i class="fa-solid fa-dice-d20"></i> Bite
 * </a>
 * ```
 *
 * @example Use an Activity on an Item from a name:
 * ```[[/item Heavy Crossbow activity=Poison]]```
 * becomes
 * ```html
 * <a class="roll-action" data-type="item" data-roll-item-name="Heavy Crossbow" data-roll-activity-name="Poison">
 *   <i class="fa-solid fa-dice-d20"></i> Heavy Crossbow: Poison
 * </a>
 * ```
 *
 * @example Use an Activity on an Item:
 * ```[[/item amUUCouL69OK1GZU activity=G8ng63Tjqy5W52OP]]```
 * becomes
 * ```html
 * <a class="roll-action" data-type="item"
 *    data-roll-activity-uuid="Actor.M4eX4Mu5IHCr3TMf.Item.amUUCouL69OK1GZU.Activity.G8ng63Tjqy5W52OP">
 *   <i class="fa-solid fa-dice-d20"></i> Bite: Save
 * </a>
 * ```
 */
async function enrichItem(config, label, options) {
  const givenItem = config.values.join(" ");
  // If config is a UUID
  const itemUuidMatch = givenItem.match(
    /^(?<synthid>Scene\.\w{16}\.Token\.\w{16}\.)?(?<actorid>Actor\.\w{16})(?<itemid>\.?Item(?<relativeId>\.\w{16}))$/
  );

  const makeLink = (label, dataset) => {
    const span = document.createElement("span");
    span.classList.add("roll-link-group");
    _addDataset(span, dataset);
    span.append(createRollLink(label));
    return span;
  };

  if ( itemUuidMatch ) {
    const ownerActor = itemUuidMatch.groups.actorid.trim();
    if ( !label ) {
      const item = await fromUuid(givenItem);
      if ( !item ) {
        console.warn(`Item not found while enriching ${config._input}.`);
        return null;
      }
      label = item.name;
    }
    return makeLink(label, { type: "item", rollItemActor: ownerActor, rollItemUuid: givenItem });
  }

  let foundItem;
  const foundActor = options.relativeTo instanceof Item
    ? options.relativeTo.parent
    : options.relativeTo instanceof Actor ? options.relativeTo : null;

  // If config is an Item ID
  if ( /^\w{16}$/.test(givenItem) && foundActor ) foundItem = foundActor.items.get(givenItem);

  // If config is a relative UUID
  if ( givenItem.startsWith(".") ) {
    try {
      foundItem = await fromUuid(givenItem, { relative: options.relativeTo });
    } catch(err) { return null; }
  }

  if ( foundItem ) {
    let foundActivity;
    if ( config.activity ) {
      foundActivity = foundItem.system.activities?.get(config.activity)
        ?? foundItem.system.activities?.getName(config.activity);
      if ( !foundActivity ) {
        console.warn(`Activity ${config.activity} not found on ${foundItem.name} while enriching ${config._input}.`);
        return null;
      }
      if ( !label ) label = `${foundItem.name}: ${foundActivity.name}`;
      return makeLink(label, { type: "item", rollActivityUuid: foundActivity.uuid });
    }

    if ( !label ) label = foundItem.name;
    return makeLink(label, { type: "item", rollItemUuid: foundItem.uuid });
  }

  // Finally, if config is an item name
  if ( !label ) label = config.activity ? `${givenItem}: ${config.activity}` : givenItem;
  return makeLink(label, {
    type: "item", rollItemActor: foundActor?.uuid, rollItemName: givenItem, rollActivityName: config.activity
  });
}

/* -------------------------------------------- */

/**
 * Add a dataset object to the provided element.
 * @param {HTMLElement} element  Element to modify.
 * @param {object} dataset       Data properties to add.
 * @private
 */
function _addDataset(element, dataset) {
  for ( const [key, value] of Object.entries(dataset) ) {
    if ( !key.startsWith("_") && (key !== "values") && value ) element.dataset[key] = value;
  }
}

/* -------------------------------------------- */

/**
 * Create a passive skill tag.
 * @param {string} label    Label to display.
 * @param {object} dataset  Data that will be added to the tag.
 * @returns {HTMLElement}
 */
function createPassiveTag(label, dataset) {
  const span = document.createElement("span");
  span.classList.add("passive-check");
  _addDataset(span, {
    ...dataset,
    tooltip: `
      <section class="loading" data-passive><i class="fas fa-spinner fa-spin-pulse"></i></section>
    `
  });
  span.innerText = label;
  return span;
}

/* -------------------------------------------- */

/**
 * Create a label for a roll message.
 * @param {object} config  Configuration data.
 * @returns {string}
 */
function createRollLabel(config) {
  const { label: ability, abbreviation } = CONFIG.DND5E.abilities[config.ability] ?? {};
  const skill = CONFIG.DND5E.skills[config.skill]?.label;
  const toolUUID = CONFIG.DND5E.enrichmentLookup.tools[config.tool];
  const tool = toolUUID ? getBaseItem(toolUUID, { indexOnly: true })?.name : null;
  const longSuffix = config.format === "long" ? "Long" : "Short";
  const showDC = config.dc && !config.hideDC;

  let label;
  switch ( config.type ) {
    case "check":
    case "skill":
    case "tool":
      if ( ability && (skill || tool) ) {
        label = game.i18n.format("EDITOR.DND5E.Inline.SpecificCheck", { ability, type: skill ?? tool });
      } else {
        label = ability;
      }
      if ( config.passive ) {
        label = game.i18n.format(
          `EDITOR.DND5E.Inline.${showDC ? "DC" : ""}Passive${longSuffix}`, { dc: config.dc, check: label }
        );
      } else {
        if ( showDC ) label = game.i18n.format("EDITOR.DND5E.Inline.DC", { dc: config.dc, check: label });
        label = game.i18n.format(`EDITOR.DND5E.Inline.Check${longSuffix}`, { check: label });
      }
      break;
    case "concentration":
    case "save":
      if ( config.type === "save" ) label = ability;
      else label = `${game.i18n.localize("DND5E.Concentration")} ${ability ? `(${abbreviation})` : ""}`;
      if ( showDC ) label = game.i18n.format("EDITOR.DND5E.Inline.DC", { dc: config.dc, check: label });
      label = game.i18n.format(`EDITOR.DND5E.Inline.Save${longSuffix}`, { save: label });
      break;
    default:
      return "";
  }

  if ( config.icon ) {
    switch ( config.type ) {
      case "check":
      case "skill":
        label = `<i class="dnd5e-icon" data-src="systems/dnd5e/icons/svg/ability-score-improvement.svg"></i>${label}`;
        break;
      case "tool":
        label = `<i class="fas fa-hammer"></i>${label}`;
        break;
      case "concentration":
      case "save":
        label = `<i class="fas fa-shield-heart"></i>${label}`;
        break;
    }
  }

  return label;
}

/* -------------------------------------------- */

/**
 * Create a rollable link with a request section for GMs.
 * @param {HTMLElement|string} label  Label to display
 * @param {object} dataset            Data that will be added to the link for the rolling method.
 * @returns {HTMLElement}
 */
function createRequestLink(label, dataset) {
  const span = document.createElement("span");
  span.classList.add("roll-link-group");
  _addDataset(span, dataset);
  if ( label instanceof HTMLTemplateElement ) span.append(label.content);
  else span.append(label);

  // Add chat request link for GMs
  if ( game.user.isGM ) {
    const gmLink = document.createElement("a");
    gmLink.classList.add("enricher-action");
    gmLink.dataset.action = "request";
    gmLink.dataset.tooltip = "EDITOR.DND5E.Inline.RequestRoll";
    gmLink.setAttribute("aria-label", game.i18n.localize(gmLink.dataset.tooltip));
    gmLink.insertAdjacentHTML("afterbegin", '<i class="fa-solid fa-comment-dots"></i>');
    span.insertAdjacentElement("beforeend", gmLink);
  }

  return span;
}

/* -------------------------------------------- */

/**
 * Create a rollable link.
 * @param {string} label                           Label to display.
 * @param {object} [dataset={}]                    Data that will be added to the link for the rolling method.
 * @param {object} [options={}]
 * @param {boolean} [options.classes="roll-link"]  Class to add to the link.
 * @param {string} [options.tag="a"]               Tag to use for the main link.
 * @returns {HTMLElement}
 */
function createRollLink(label, dataset={}, { classes="roll-link", tag="a" }={}) {
  const link = document.createElement(tag);
  link.className = classes;
  link.insertAdjacentHTML("afterbegin", '<i class="fa-solid fa-dice-d20" inert></i>');
  link.append(label);
  _addDataset(link, dataset);
  return link;
}

/* -------------------------------------------- */
/*  Actions                                     */
/* -------------------------------------------- */

/**
 * Toggle status effects on selected tokens.
 * @param {PointerEvent} event  The triggering event.
 * @returns {Promise<void>}
 */
async function applyAction(event) {
  const target = event.target.closest('[data-action="apply"][data-status]');
  const status = target?.dataset.status;
  if ( !status ) return;
  event.stopPropagation();
  const actors = new Set();
  for ( const { actor } of canvas.tokens.controlled ) {
    if ( !actor || actors.has(actor) ) continue;
    await actor.toggleStatusEffect(status);
    actors.add(actor);
  }
}

/* -------------------------------------------- */

/**
 * Forward clicks on award requests to the Award application.
 * @param {Event} event  The click event triggering the action.
 * @returns {Promise<void>}
 */
async function awardAction(event) {
  const target = event.target.closest('[data-action="awardRequest"]');
  const command = target?.closest("[data-award-command]")?.dataset.awardCommand;
  if ( !command ) return;
  event.stopPropagation();
  Award.handleAward(command);
}

/* -------------------------------------------- */

/**
 * Perform the provided roll action.
 * @param {Event} event  The click event triggering the action.
 * @returns {Promise}
 */
async function rollAction(event) {
  const target = event.target.closest('.roll-link-group, [data-action="rollRequest"], [data-action="concentration"]');
  if ( !target ) return;
  event.stopPropagation();

  const dataset = {
    ...((event.target.closest(".roll-link-group") ?? target)?.dataset ?? {}),
    ...(event.target.closest(".roll-link")?.dataset ?? {})
  };
  const { type, ability, skill, tool, dc } = dataset;
  const options = { event };
  if ( ability in CONFIG.DND5E.abilities ) options.ability = ability;
  if ( dc ) options.target = Number(dc);

  const action = event.target.closest("a")?.dataset.action ?? "roll";
  const link = event.target.closest("a") ?? event.target;

  // Direct roll
  if ( (action === "roll") || !game.user.isGM ) {
    link.disabled = true;
    try {
      switch ( type ) {
        case "attack": return await rollAttack(event);
        case "damage": return await rollDamage(event);
        case "item": return await useItem(dataset);
      }

      const actors = getSceneTargets().map(t => t.actor);
      if ( !actors.length && game.user.character ) actors.push(game.user.character);
      if ( !actors.length ) {
        ui.notifications.warn("EDITOR.DND5E.Inline.Warning.NoActor", { localize: true });
        return;
      }

      for ( const actor of actors ) {
        switch ( type ) {
          case "check":
            await actor.rollAbilityCheck(options);
            break;
          case "concentration":
            await actor.rollConcentration({ ...options, legacy: false });
            break;
          case "save":
            await actor.rollSavingThrow(options);
            break;
          case "skill":
            await actor.rollSkill({ skill, ...options });
            break;
          case "tool":
            await actor.rollToolCheck({ tool, ...options });
            break;
        }
      }
    } finally {
      link.disabled = false;
    }
  }

  // Roll request
  else {
    const MessageClass = getDocumentClass("ChatMessage");

    let buttons;
    if ( dataset.type === "check" ) buttons = createCheckRequestButtons(dataset);
    else if ( dataset.type === "save" ) buttons = createSaveRequestButtons(dataset);
    else buttons = [createRequestButton({ ...dataset, format: "short" })];

    const chatData = {
      user: game.user.id,
      content: await renderTemplate("systems/dnd5e/templates/chat/request-card.hbs", { buttons }),
      flavor: game.i18n.localize("EDITOR.DND5E.Inline.RollRequest"),
      speaker: MessageClass.getSpeaker({user: game.user})
    };
    return MessageClass.create(chatData);
  }
}

/* -------------------------------------------- */

/**
 * Create a button for a chat request.
 * @param {object} dataset
 * @returns {object}
 */
function createRequestButton(dataset) {
  return {
    buttonLabel: createRollLabel({ ...dataset, icon: true }),
    hiddenLabel: createRollLabel({ ...dataset, icon: true, hideDC: true }),
    dataset: { ...dataset, action: "rollRequest", visibility: "all" }
  };
}

/* -------------------------------------------- */

/**
 * Perform an attack roll.
 * @param {Event} event     The click event triggering the action.
 * @returns {Promise|void}
 */
async function rollAttack(event) {
  const target = event.target.closest(".roll-link-group");
  const { activityUuid, attackMode, formula } = target.dataset;

  if ( activityUuid ) {
    const activity = await fromUuid(activityUuid);
    if ( activity ) return activity.rollAttack({ attackMode, event });
  }

  const targets = getTargetDescriptors();
  const rollConfig = {
    attackMode, event,
    hookNames: ["attack", "d20Test"],
    rolls: [{
      parts: [formula.replace(/^\s*\+\s*/, "")],
      options: {
        target: targets.length === 1 ? targets[0].ac : undefined
      }
    }]
  };

  const dialogConfig = {
    applicationClass: AttackRollConfigurationDialog
  };

  const messageConfig = {
    data: {
      flags: {
        dnd5e: {
          messageType: "roll",
          roll: { type: "attack" }
        }
      },
      flavor: game.i18n.localize("DND5E.AttackRoll"),
      speaker: ChatMessage.implementation.getSpeaker()
    }
  };

  const rolls = await CONFIG.Dice.D20Roll.build(rollConfig, dialogConfig, messageConfig);
  if ( rolls?.length ) {
    Hooks.callAll("dnd5e.rollAttackV2", rolls, { subject: null, ammoUpdate: null });
    Hooks.callAll("dnd5e.postRollAttack", rolls, { subject: null });
  }
}

/* -------------------------------------------- */

/**
 * Perform a damage roll.
 * @param {Event} event  The click event triggering the action.
 * @returns {Promise<void>}
 */
async function rollDamage(event) {
  const target = event.target.closest(".roll-link-group");
  let { activityUuid, attackMode, formulas, damageTypes, rollType } = target.dataset;

  if ( activityUuid ) {
    const activity = await fromUuid(activityUuid);
    if ( activity ) return activity.rollDamage({ attackMode, event });
  }

  formulas = formulas?.split("&") ?? [];
  damageTypes = damageTypes?.split("&") ?? [];

  const rollConfig = {
    attackMode, event,
    hookNames: ["damage"],
    rolls: formulas.map((formula, idx) => {
      const types = damageTypes[idx]?.split("|") ?? [];
      return {
        parts: [formula],
        options: { type: types[0], types }
      };
    })
  };

  const messageConfig = {
    create: true,
    data: {
      flags: {
        dnd5e: {
          messageType: "roll",
          roll: { type: rollType },
          targets: getTargetDescriptors()
        }
      },
      flavor: game.i18n.localize(`DND5E.${rollType === "healing" ? "Healing" : "Damage"}Roll`),
      speaker: ChatMessage.implementation.getSpeaker()
    }
  };

  const rolls = await CONFIG.Dice.DamageRoll.build(rollConfig, {}, messageConfig);
  if ( !rolls?.length ) return;
  Hooks.callAll("dnd5e.rollDamageV2", rolls);
}

/* -------------------------------------------- */

/**
 * Use an Item from an Item enricher.
 * @param {object} [options]
 * @param {string} [options.rollActivityUuid]  Lookup the Activity by UUID.
 * @param {string} [options.rollActivityName]  Lookup the Activity by name.
 * @param {string} [options.rollItemUuid]      Lookup the Item by UUID.
 * @param {string} [options.rollItemName]      Lookup the Item by name.
 * @param {string} [options.rollItemActor]     The UUID of a specific Actor that should use the Item.
 * @returns {Promise}
 */
async function useItem({ rollActivityUuid, rollActivityName, rollItemUuid, rollItemName, rollItemActor }={}) {
  // If UUID is provided, always roll that item directly
  if ( rollActivityUuid ) return (await fromUuid(rollActivityUuid))?.use();
  if ( rollItemUuid ) return (await fromUuid(rollItemUuid))?.use({ legacy: false });

  if ( !rollItemName ) return;
  const actor = rollItemActor ? await fromUuid(rollItemActor) : null;

  // If no actor is specified or player isn't owner, fall back to the macro rolling logic
  if ( !actor?.isOwner ) return rollItem(rollItemName, { activityName: rollActivityName });
  const token = canvas.tokens.controlled[0];

  // If a token is controlled, and it has an item with the correct name, activate it
  let item = token?.actor.items.getName(rollItemName);

  // Otherwise check the specified actor for the item
  if ( !item ) {
    item = actor.items.getName(rollItemName);

    // Display a warning to indicate the item wasn't rolled from the controlled actor
    if ( item && canvas.tokens.controlled.length ) ui.notifications.warn(
      game.i18n.format("MACRO.5eMissingTargetWarn", {
        actor: token.name, name: rollItemName, type: game.i18n.localize("DOCUMENT.Item")
      })
    );
  }

  if ( item ) {
    if ( rollActivityName ) {
      const activity = item.system.activities?.getName(rollActivityName);
      if ( activity ) return activity.use();

      // If no activity could be found at all, display a warning
      else ui.notifications.warn(game.i18n.format("EDITOR.DND5E.Inline.Warning.NoActivityOnItem", {
        activity: rollActivityName, actor: actor.name, item: rollItemName
      }));
    }

    else return item.use({ legacy: false });
  }

  // If no item could be found at all, display a warning
  else ui.notifications.warn(game.i18n.format("EDITOR.DND5E.Inline.Warning.NoItemOnActor", {
    actor: actor.name, item: rollItemName
  }));
}

var enrichers = /*#__PURE__*/Object.freeze({
  __proto__: null,
  createRollLabel: createRollLabel,
  registerCustomEnrichers: registerCustomEnrichers
});

/**
 * FIXME: Remove when v12 support dropped or https://github.com/foundryvtt/foundryvtt/issues/11991 backported.
 * Should NOT be exported for general use.
 * @ignore
 */
function parseUuid(uuid, {relative}={}) {
  if ( game.release.generation > 12 ) return foundry.utils.parseUuid(uuid, { relative });
  if ( !uuid ) throw new Error("A UUID string is required.");
  if ( uuid.startsWith(".") && relative ) return _resolveRelativeUuid(uuid, relative);
  const parsed = foundry.utils.parseUuid(uuid, {relative});
  if ( !parsed?.collection ) return parsed;
  const remappedUuid = uuid.startsWith("Compendium") ? [
    "Compendium",
    parsed.collection.metadata.id,
    parsed.primaryType ?? parsed.documentType,
    parsed.primaryId ?? parsed.documentId,
    ...parsed.embedded
  ].join(".") : uuid;
  return { ...parsed, uuid: remappedUuid };
}

/** @ignore */
function _resolveRelativeUuid(uuid, relative) {
  if ( !(relative instanceof foundry.abstract.Document) ) {
    throw new Error("A relative Document instance must be provided to _resolveRelativeUuid");
  }
  uuid = uuid.substring(1);
  const parts = uuid.split(".");
  if ( !parts.length ) throw new Error("Invalid relative UUID");
  let id;
  let type;
  let root;
  let primaryType;
  let primaryId;
  let collection;

  // Identify the root document and its collection
  const getRoot = doc => {
    if ( doc.parent ) parts.unshift(doc.documentName, doc.id);
    return doc.parent ? getRoot(doc.parent) : doc;
  };

  // Even-numbered parts include an explicit child document type
  if ( (parts.length % 2) === 0 ) {
    root = getRoot(relative);
    id = parts.at(-1);
    type = parts.at(-2);
    primaryType = root.documentName;
    primaryId = root.id;
    uuid = [primaryType, primaryId, ...parts].join(".");
  }

  // Relative Embedded Document
  else if ( relative.parent ) {
    id = parts.at(-1);
    type = relative.documentName;
    parts.unshift(type);
    root = getRoot(relative.parent);
    primaryType = root.documentName;
    primaryId = root.id;
    uuid = [primaryType, primaryId, ...parts].join(".");
  }

  // Relative Document
  else {
    root = relative;
    id = parts.pop();
    type = relative.documentName;
    uuid = [type, id].join(".");
  }

  // Recreate fully-qualified UUID and return the resolved result
  collection = root.pack ? root.compendium : root.collection;
  if ( root.pack ) uuid = `Compendium.${root.pack}.${uuid}`;
  return {uuid, type, id, collection, primaryType, primaryId, embedded: parts,
    documentType: primaryType ?? type, documentId: primaryId ?? id};
}

const { ObjectField, SchemaField: SchemaField$C, SetField: SetField$o, StringField: StringField$P } = foundry.data.fields;

/**
 * Extend the base ActiveEffect class to implement system-specific logic.
 */
class ActiveEffect5e extends ActiveEffect {
  /**
   * Static ActiveEffect ID for various conditions.
   * @type {Record<string, string>}
   */
  static ID = {
    BLOODIED: staticID("dnd5ebloodied"),
    ENCUMBERED: staticID("dnd5eencumbered"),
    EXHAUSTION: staticID("dnd5eexhaustion")
  };

  /* -------------------------------------------- */

  /**
   * Additional key paths to properties added during base data preparation that should be treated as formula fields.
   * @type {Set<string>}
   */
  static FORMULA_FIELDS = new Set([
    "system.attributes.ac.bonus",
    "system.attributes.ac.min",
    "system.attributes.encumbrance.bonuses.encumbered",
    "system.attributes.encumbrance.bonuses.heavilyEncumbered",
    "system.attributes.encumbrance.bonuses.maximum",
    "system.attributes.encumbrance.bonuses.overall",
    "system.attributes.encumbrance.multipliers.encumbered",
    "system.attributes.encumbrance.multipliers.heavilyEncumbered",
    "system.attributes.encumbrance.multipliers.maximum",
    "system.attributes.encumbrance.multipliers.overall",
    "save.dc.bonus"
  ]);

  /* -------------------------------------------- */

  /** @inheritdoc */
  static LOCALIZATION_PREFIXES = [...super.LOCALIZATION_PREFIXES, "DND5E.ACTIVEEFFECT"];

  /* -------------------------------------------- */

  /**
   * Is this effect an enchantment on an item that accepts enchantment?
   * @type {boolean}
   */
  get isAppliedEnchantment() {
    return (this.type === "enchantment") && !!this.origin && (this.origin !== this.parent.uuid);
  }

  /* -------------------------------------------- */

  /**
   * Should this status effect be hidden from the current user?
   * @type {boolean}
   */
  get isConcealed() {
    if ( this.target?.testUserPermission(game.user, "OBSERVER") ) return false;

    // Hide bloodied status effect from players unless the token is friendly
    if ( (this.id === this.constructor.ID.BLOODIED) && (game.settings.get("dnd5e", "bloodied") === "player") ) {
      return this.target?.token?.disposition !== foundry.CONST.TOKEN_DISPOSITIONS.FRIENDLY;
    }

    return false;
  }

  /* -------------------------------------------- */

  /**
   * Is this active effect currently suppressed?
   * @type {boolean}
   */
  isSuppressed = false;

  /* -------------------------------------------- */

  /** @inheritDoc */
  get isTemporary() {
    return super.isTemporary && !this.isConcealed;
  }

  /* -------------------------------------------- */

  /**
   * Retrieve the source Actor or Item, or null if it could not be determined.
   * @returns {Promise<Actor5e|Item5e|null>}
   */
  async getSource() {
    if ( (this.target instanceof dnd5e.documents.Actor5e) && (this.parent instanceof dnd5e.documents.Item5e) ) {
      return this.parent;
    }
    return fromUuid(this.origin);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static async _fromStatusEffect(statusId, { reference, ...effectData }, options) {
    if ( !("description" in effectData) && reference ) effectData.description = `@Embed[${reference} inline]`;
    return super._fromStatusEffect?.(statusId, effectData, options) ?? new this(effectData, options);
  }

  /* -------------------------------------------- */
  /*  Data Migration                              */
  /* -------------------------------------------- */

  /** @inheritDoc */
  _initializeSource(data, options={}) {
    if ( data instanceof foundry.abstract.DataModel ) data = data.toObject();

    if ( data.flags?.dnd5e?.type === "enchantment" ) {
      data.type = "enchantment";
      delete data.flags.dnd5e.type;
    }

    return super._initializeSource(data, options);
  }

  /* -------------------------------------------- */
  /*  Effect Application                          */
  /* -------------------------------------------- */

  /** @inheritDoc */
  apply(doc, change) {
    // Ensure changes targeting flags use the proper types
    if ( change.key.startsWith("flags.dnd5e.") ) change = this._prepareFlagChange(doc, change);

    // Properly handle formulas that don't exist as part of the data model
    if ( ActiveEffect5e.FORMULA_FIELDS.has(change.key) ) {
      const field = new FormulaField({ deterministic: true });
      return { [change.key]: this.constructor.applyField(doc, change, field) };
    }

    // Handle activity-targeted changes
    if ( (change.key.startsWith("activities[") || change.key.startsWith("system.activities."))
      && (doc instanceof Item) ) return this.applyActivity(doc, change);

    return super.apply(doc, change);
  }

  /* -------------------------------------------- */

  /**
   * Apply a change to activities on this item.
   * @param {Item5e} item              The Item to whom this change should be applied.
   * @param {EffectChangeData} change  The change data being applied.
   * @returns {Record<string, *>}      An object of property paths and their updated values.
   */
  applyActivity(item, change) {
    const changes = {};
    const apply = (activity, key) => {
      const c = this.apply(activity, { ...change, key });
      Object.entries(c).forEach(([k, v]) => changes[`system.activities.${activity.id}.${k}`] = v);
    };
    if ( change.key.startsWith("system.activities.") ) {
      const [, , id, ...keyPath] = change.key.split(".");
      const activity = item.system.activities?.get(id);
      if ( activity ) apply(activity, keyPath.join("."));
    } else {
      const { type, key } = change.key.match(/activities\[(?<type>[^\]]+)]\.(?<key>.+)/)?.groups ?? {};
      item.system.activities?.getByType(type)?.forEach(activity => apply(activity, key));
    }
    return changes;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static applyField(model, change, field) {
    field ??= model.schema.getField(change.key);
    change = foundry.utils.deepClone(change);
    const current = foundry.utils.getProperty(model, change.key);
    const modes = CONST.ACTIVE_EFFECT_MODES;

    // Replace value when using string interpolation syntax
    if ( (field instanceof StringField$P) && (change.mode === modes.OVERRIDE) && change.value.includes("{}") ) {
      change.value = change.value.replace("{}", current ?? "");
    }

    // If current value is `null`, UPGRADE & DOWNGRADE should always just set the value
    if ( (current === null) && [modes.UPGRADE, modes.DOWNGRADE].includes(change.mode) ) change.mode = modes.OVERRIDE;

    // Handle removing entries from sets
    if ( (field instanceof SetField$o) && (change.mode === modes.ADD) && (foundry.utils.getType(current) === "Set") ) {
      for ( const value of field._castChangeDelta(change.value) ) {
        const neg = value.replace(/^\s*-\s*/, "");
        if ( neg !== value ) current.delete(neg);
        else current.add(value);
      }
      return current;
    }

    // If attempting to apply active effect to empty MappingField entry, create it
    if ( (current === undefined) && change.key.startsWith("system.") ) {
      let keyPath = change.key;
      let mappingField = field;
      while ( !(mappingField instanceof MappingField) && mappingField ) {
        if ( mappingField.name ) keyPath = keyPath.substring(0, keyPath.length - mappingField.name.length - 1);
        mappingField = mappingField.parent;
      }
      if ( mappingField && (foundry.utils.getProperty(model, keyPath) === undefined) ) {
        const created = mappingField.model.initialize(mappingField.model.getInitialValue(), mappingField);
        foundry.utils.setProperty(model, keyPath, created);
      }
    }

    // Parse any JSON provided when targeting an object
    if ( (field instanceof ObjectField) || (field instanceof SchemaField$C) ) {
      change = { ...change, value: this.prototype._parseOrString(change.value) };
    }

    return super.applyField(model, change, field);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _applyAdd(actor, change, current, delta, changes) {
    if ( current instanceof Set ) {
      const handle = v => {
        const neg = v.replace(/^\s*-\s*/, "");
        if ( neg !== v ) current.delete(neg);
        else current.add(v);
      };
      if ( Array.isArray(delta) ) delta.forEach(item => handle(item));
      else handle(delta);
      return;
    }
    super._applyAdd(actor, change, current, delta, changes);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _applyLegacy(actor, change, changes) {
    if ( this.system._applyLegacy?.(actor, change, changes) === false ) return;
    super._applyLegacy(actor, change, changes);
  }

  /* --------------------------------------------- */

  /** @inheritDoc */
  _applyUpgrade(actor, change, current, delta, changes) {
    if ( current === null ) return this._applyOverride(actor, change, current, delta, changes);
    return super._applyUpgrade(actor, change, current, delta, changes);
  }

  /* --------------------------------------------- */

  /**
   * Transform the data type of the change to match the type expected for flags.
   * @param {Actor5e} actor            The Actor to whom this effect should be applied.
   * @param {EffectChangeData} change  The change being applied.
   * @returns {EffectChangeData}       The change with altered types if necessary.
   */
  _prepareFlagChange(actor, change) {
    const { key, value } = change;
    const data = CONFIG.DND5E.characterFlags[key.replace("flags.dnd5e.", "")];
    if ( !data ) return change;

    // Set flag to initial value if it isn't present
    const current = foundry.utils.getProperty(actor, key) ?? null;
    if ( current === null ) {
      let initialValue = null;
      if ( data.placeholder ) initialValue = data.placeholder;
      else if ( data.type === Boolean ) initialValue = false;
      else if ( data.type === Number ) initialValue = 0;
      foundry.utils.setProperty(actor, key, initialValue);
    }

    // Coerce change data into the correct type
    if ( data.type === Boolean ) {
      if ( value === "false" ) change.value = false;
      else change.value = Boolean(value);
    }
    return change;
  }

  /* --------------------------------------------- */

  /**
   * Determine whether this Active Effect is suppressed or not.
   */
  determineSuppression() {
    this.isSuppressed = false;
    if ( this.type === "enchantment" ) return;
    if ( this.parent instanceof dnd5e.documents.Item5e ) this.isSuppressed = this.parent.areEffectsSuppressed;
  }

  /* -------------------------------------------- */
  /*  Lifecycle                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  prepareDerivedData() {
    super.prepareDerivedData();
    if ( this.id === this.constructor.ID.EXHAUSTION ) this._prepareExhaustionLevel();
    if ( this.isAppliedEnchantment ) dnd5e.registry.enchantments.track(this.origin, this.uuid);
  }

  /* -------------------------------------------- */

  /**
   * Modify the ActiveEffect's attributes based on the exhaustion level.
   * @protected
   */
  _prepareExhaustionLevel() {
    const config = CONFIG.DND5E.conditionTypes.exhaustion;
    let level = this.getFlag("dnd5e", "exhaustionLevel");
    if ( !Number.isFinite(level) ) level = 1;
    this.img = this.constructor._getExhaustionImage(level);
    this.name = `${game.i18n.localize("DND5E.Exhaustion")} ${level}`;
    if ( level >= config.levels ) {
      this.statuses.add("dead");
      CONFIG.DND5E.statusEffects.dead.statuses?.forEach(s => this.statuses.add(s));
    }
  }

  /* -------------------------------------------- */

  /**
   * Prepare effect favorite data.
   * @returns {Promise<FavoriteData5e>}
   */
  async getFavoriteData() {
    return {
      img: this.img,
      title: this.name,
      subtitle: this.duration.remaining ? this.duration.label : "",
      toggle: !this.disabled,
      suppressed: this.isSuppressed
    };
  }

  /* -------------------------------------------- */

  /**
   * Create conditions that are applied separately from an effect.
   * @returns {Promise<ActiveEffect5e[]>}      Created rider effects.
   */
  async createRiderConditions() {
    const riders = new Set();

    for ( const status of this.getFlag("dnd5e", "riders.statuses") ?? [] ) {
      riders.add(status);
    }

    for ( const status of this.statuses ) {
      const r = CONFIG.statusEffects.find(e => e.id === status)?.riders ?? [];
      for ( const p of r ) riders.add(p);
    }

    if ( !riders.size ) return [];

    const createRider = async id => {
      const existing = this.parent.effects.get(staticID(`dnd5e${id}`));
      if ( existing ) return;
      const effect = await ActiveEffect5e.fromStatusEffect(id);
      return effect.toObject();
    };

    const effectData = await Promise.all(Array.from(riders).map(createRider));
    return ActiveEffect5e.createDocuments(effectData.filter(_ => _), { keepId: true, parent: this.parent });
  }

  /* -------------------------------------------- */

  /**
   * Create additional activities, effects, and items that are applied separately from an enchantment.
   * @param {object} options  Options passed to the effect creation.
   */
  async createRiderEnchantments(options={}) {
    let item;
    let profile;
    const { chatMessageOrigin } = options;
    const { enchantmentProfile, activityId } = options.dnd5e ?? {};

    if ( chatMessageOrigin ) {
      const message = game.messages.get(options?.chatMessageOrigin);
      item = message?.getAssociatedItem();
      const activity = message?.getAssociatedActivity();
      profile = activity?.effects.find(e => e._id === message?.getFlag("dnd5e", "use.enchantmentProfile"));
    } else if ( enchantmentProfile && activityId ) {
      let activity;
      const origin = await fromUuid(this.origin);
      if ( origin instanceof dnd5e.documents.activity.EnchantActivity ) {
        activity = origin;
        item = activity.item;
      } else if ( origin instanceof Item ) {
        item = origin;
        activity = item.system.activities?.get(activityId);
      }
      profile = activity?.effects.find(e => e._id === enchantmentProfile);
    }

    if ( !profile || !item ) return;

    // Create Activities
    const riderActivities = {};
    let riderEffects = [];
    for ( const id of profile.riders.activity ) {
      const activityData = item.system.activities.get(id)?.toObject();
      if ( !activityData ) continue;
      activityData._id = foundry.utils.randomID();
      riderActivities[activityData._id] = activityData;
    }
    let createdActivities = [];
    if ( !foundry.utils.isEmpty(riderActivities) ) {
      await this.parent.update({ "system.activities": riderActivities });
      createdActivities = Object.keys(riderActivities).map(id => this.parent.system.activities?.get(id));
      createdActivities.forEach(a => a.effects?.forEach(e => {
        if ( !this.parent.effects.has(e._id) ) riderEffects.push(item.effects.get(e._id)?.toObject());
      }));
    }

    // Create Effects
    riderEffects.push(...profile.riders.effect.map(id => {
      const effectData = item.effects.get(id)?.toObject();
      if ( effectData ) {
        delete effectData._id;
        delete effectData.flags?.dnd5e?.rider;
        effectData.origin = this.origin;
      }
      return effectData;
    }));
    riderEffects = riderEffects.filter(_ => _);
    const createdEffects = await this.parent.createEmbeddedDocuments("ActiveEffect", riderEffects, { keepId: true });

    // Create Items
    let createdItems = [];
    if ( this.parent.isEmbedded ) {
      const riderItems = await Promise.all(profile.riders.item.map(async uuid => {
        const itemData = (await fromUuid(uuid))?.toObject();
        if ( itemData ) {
          delete itemData._id;
          foundry.utils.setProperty(itemData, "flags.dnd5e.enchantment", { origin: this.uuid });
        }
        return itemData;
      }));
      createdItems = await this.parent.actor.createEmbeddedDocuments("Item", riderItems.filter(i => i));
    }

    if ( createdActivities.length || createdEffects.length || createdItems.length ) {
      this.addDependent(...createdActivities, ...createdEffects, ...createdItems);
    }
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  toDragData() {
    const data = super.toDragData();
    const activity = this.parent?.system.activities?.getByType("enchant").find(a => {
      return a.effects.some(e => e._id === this.id);
    });
    if ( activity ) data.activityId = activity.id;
    return data;
  }

  /* -------------------------------------------- */
  /*  Socket Event Handlers                       */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _preCreate(data, options, user) {
    if ( await super._preCreate(data, options, user) === false ) return false;
    if ( options.keepOrigin === false ) this.updateSource({ origin: this.parent.uuid });

    // Enchantments cannot be added directly to actors
    if ( (this.type === "enchantment") && (this.parent instanceof Actor) ) {
      ui.notifications.error("DND5E.ENCHANTMENT.Warning.NotOnActor", { localize: true });
      return false;
    }

    if ( this.isAppliedEnchantment ) {
      const origin = await fromUuid(this.origin);
      const errors = origin?.canEnchant?.(this.parent);
      if ( errors?.length ) {
        errors.forEach(err => console.error(err));
        return false;
      }
      this.updateSource({ disabled: false });
    }
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _onCreate(data, options, userId) {
    super._onCreate(data, options, userId);
    if ( userId === game.userId ) {
      if ( this.active && (this.parent instanceof Actor) ) await this.createRiderConditions();
      if ( this.isAppliedEnchantment ) await this.createRiderEnchantments(options);
    }
    if ( options.chatMessageOrigin ) {
      document.body.querySelectorAll(`[data-message-id="${options.chatMessageOrigin}"] enchantment-application`)
        .forEach(element => element.buildItemList());
    }
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _onUpdate(data, options, userId) {
    super._onUpdate(data, options, userId);
    const originalLevel = foundry.utils.getProperty(options, "dnd5e.originalExhaustion");
    const newLevel = foundry.utils.getProperty(data, "flags.dnd5e.exhaustionLevel");
    const originalEncumbrance = foundry.utils.getProperty(options, "dnd5e.originalEncumbrance");
    const newEncumbrance = data.statuses?.[0];
    const name = this.name;

    // Display proper scrolling status effects for exhaustion
    if ( (this.id === this.constructor.ID.EXHAUSTION) && Number.isFinite(newLevel) && Number.isFinite(originalLevel) ) {
      if ( newLevel === originalLevel ) return;
      // Temporarily set the name for the benefit of _displayScrollingTextStatus. We should improve this method to
      // accept a name parameter instead.
      if ( newLevel < originalLevel ) this.name = `Exhaustion ${originalLevel}`;
      this._displayScrollingStatus(newLevel > originalLevel);
      this.name = name;
    }

    // Display proper scrolling status effects for encumbrance
    else if ( (this.id === this.constructor.ID.ENCUMBERED) && originalEncumbrance && newEncumbrance ) {
      if ( newEncumbrance === originalEncumbrance ) return;
      const increase = !originalEncumbrance || ((originalEncumbrance === "encumbered") && newEncumbrance)
        || (newEncumbrance === "exceedingCarryingCapacity");
      if ( !increase ) this.name = CONFIG.DND5E.encumbrance.effects[originalEncumbrance].name;
      this._displayScrollingStatus(increase);
      this.name = name;
    }
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _preDelete(options, user) {
    const dependents = this.getDependents();
    if ( dependents.length && !game.users.activeGM ) {
      ui.notifications.warn("DND5E.ConcentrationBreakWarning", { localize: true });
      return false;
    }
    return super._preDelete(options, user);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _onDelete(options, userId) {
    super._onDelete(options, userId);
    if ( game.user === game.users.activeGM ) this.getDependents().forEach(e => e.delete());
    if ( this.isAppliedEnchantment ) dnd5e.registry.enchantments.untrack(this.origin, this.uuid);
    document.body.querySelectorAll(`enchantment-application:has([data-enchantment-uuid="${this.uuid}"]`)
      .forEach(element => element.buildItemList());
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _displayScrollingStatus(enabled) {
    if ( this.isConcealed ) return;
    super._displayScrollingStatus(enabled);
  }

  /* -------------------------------------------- */
  /*  Exhaustion and Concentration Handling       */
  /* -------------------------------------------- */

  /**
   * Create effect data for concentration on an actor.
   * @param {Activity} activity  The Activity on which to begin concentrating.
   * @param {object} [data]      Additional data provided for the effect instance.
   * @returns {object}           Created data for the ActiveEffect.
   */
  static createConcentrationEffectData(activity, data={}) {
    if ( activity instanceof Item ) {
      foundry.utils.logCompatibilityWarning(
        "The `createConcentrationEffectData` method on ActiveEffect5e now takes an Activity, rather than an Item.",
        { since: "DnD5e 4.0", until: "DnD5e 4.4" }
      );
      activity = activity.system.activities?.contents[0];
    }

    const item = activity?.item;
    if ( !item?.isEmbedded || !activity.duration.concentration ) {
      throw new Error("You may not begin concentrating on this item!");
    }

    const statusEffect = CONFIG.statusEffects.find(e => e.id === CONFIG.specialStatusEffects.CONCENTRATING);
    const effectData = foundry.utils.mergeObject({
      ...statusEffect,
      name: `${game.i18n.localize("EFFECT.DND5E.StatusConcentrating")}: ${item.name}`,
      description: `<p>${game.i18n.format("DND5E.ConcentratingOn", {
        name: item.name,
        type: game.i18n.localize(`TYPES.Item.${item.type}`)
      })}</p><hr><p>@Embed[${item.uuid} inline]</p>`,
      duration: activity.duration.getEffectData(),
      "flags.dnd5e": {
        activity: {
          type: activity.type, id: activity.id, uuid: activity.uuid
        },
        item: {
          type: item.type, id: item.id, uuid: item.uuid,
          data: !item.actor.items.has(item.id) ? item.toObject() : undefined
        }
      },
      origin: item.uuid,
      statuses: [statusEffect.id].concat(statusEffect.statuses ?? [])
    }, data, {inplace: false});
    delete effectData.id;
    if ( item.type === "spell" ) effectData["flags.dnd5e.spellLevel"] = item.system.level;

    return effectData;
  }

  /* -------------------------------------------- */

  /**
   * Register listeners for custom handling in the TokenHUD.
   */
  static registerHUDListeners() {
    Hooks.on("renderTokenHUD", this.onTokenHUDRender);
    document.addEventListener("click", this.onClickTokenHUD.bind(this), { capture: true });
    document.addEventListener("contextmenu", this.onClickTokenHUD.bind(this), { capture: true });
  }

  /* -------------------------------------------- */

  /**
   * Add modifications to the core ActiveEffect config.
   * @param {ActiveEffectConfig} app   The ActiveEffect config.
   * @param {jQuery|HTMLElement} html  The ActiveEffect config element.
   */
  static onRenderActiveEffectConfig(app, html) {
    if ( game.release.generation < 13 ) html = html[0];
    const element = new foundry.data.fields.SetField(new foundry.data.fields.StringField(), {}).toFormGroup({
      label: game.i18n.localize("DND5E.CONDITIONS.RiderConditions.label"),
      hint: game.i18n.localize("DND5E.CONDITIONS.RiderConditions.hint")
    }, {
      name: "flags.dnd5e.riders.statuses",
      value: app.document.getFlag("dnd5e", "riders.statuses") ?? [],
      options: CONFIG.statusEffects.map(se => ({ value: se.id, label: se.name }))
    });
    // TODO: Temporary fix to work around https://github.com/foundryvtt/foundryvtt/issues/11567
    // Replace with `after` when switched to V13-only
    html.querySelector("[data-tab=details] > .form-group:has([name=statuses])")
      ?.insertAdjacentHTML("afterend", element.outerHTML);

    if ( game.release.generation < 13 ) {
      html.querySelector(".form-fields:has([name=statuses])").insertAdjacentHTML("afterend", `
        <p class="hint">${app.document.schema.fields.statuses.hint}</p>
      `);
      app.setPosition();
    }
  }

  /* -------------------------------------------- */

  /**
   * Adjust exhaustion icon display to match current level.
   * @param {Application} app  The TokenHUD application.
   * @param {jQuery} html      The TokenHUD HTML.
   */
  static onTokenHUDRender(app, html) {
    const actor = app.object.actor;
    const level = foundry.utils.getProperty(actor, "system.attributes.exhaustion");
    if ( Number.isFinite(level) && (level > 0) ) {
      const img = ActiveEffect5e._getExhaustionImage(level);
      html.find('[data-status-id="exhaustion"]').css({
        objectPosition: "-100px",
        background: `url('${img}') no-repeat center / contain`
      });
    }
  }

  /* -------------------------------------------- */

  /**
   * Get the image used to represent exhaustion at this level.
   * @param {number} level
   * @returns {string}
   */
  static _getExhaustionImage(level) {
    const split = CONFIG.DND5E.conditionTypes.exhaustion.icon.split(".");
    const ext = split.pop();
    const path = split.join(".");
    return `${path}-${level}.${ext}`;
  }

  /* -------------------------------------------- */

  /**
   * Map the duration of an item to an active effect duration.
   * @param {Item5e} item           An item with a duration.
   * @returns {EffectDurationData}  The active effect duration.
   */
  static getEffectDurationFromItem(item) {
    foundry.utils.logCompatibilityWarning(
      "The `getEffectDurationFromItem` method on ActiveEffect5e has been deprecated and replaced with `getEffectData` within Item or Activity duration.",
      { since: "DnD5e 4.0", until: "DnD5e 4.4" }
    );
    return item.system.duration?.getEffectData?.() ?? {};
  }

  /* -------------------------------------------- */

  /**
   * Implement custom behavior for select conditions on the token HUD.
   * @param {PointerEvent} event        The triggering event.
   */
  static onClickTokenHUD(event) {
    const { target } = event;
    if ( !target.classList?.contains("effect-control") ) return;

    const actor = canvas.hud.token.object?.actor;
    if ( !actor ) return;

    const id = target.dataset?.statusId;
    if ( id === "exhaustion" ) ActiveEffect5e._manageExhaustion(event, actor);
    else if ( id === "concentrating" ) ActiveEffect5e._manageConcentration(event, actor);
  }

  /* -------------------------------------------- */

  /**
   * Manage custom exhaustion cycling when interacting with the token HUD.
   * @param {PointerEvent} event        The triggering event.
   * @param {Actor5e} actor             The actor belonging to the token.
   */
  static _manageExhaustion(event, actor) {
    let level = foundry.utils.getProperty(actor, "system.attributes.exhaustion");
    if ( !Number.isFinite(level) ) return;
    event.preventDefault();
    event.stopPropagation();
    if ( event.button === 0 ) level++;
    else level--;
    const max = CONFIG.DND5E.conditionTypes.exhaustion.levels;
    actor.update({ "system.attributes.exhaustion": Math.clamp(level, 0, max) });
  }

  /* -------------------------------------------- */

  /**
   * Manage custom concentration handling when interacting with the token HUD.
   * @param {PointerEvent} event        The triggering event.
   * @param {Actor5e} actor             The actor belonging to the token.
   */
  static _manageConcentration(event, actor) {
    const { effects } = actor.concentration;
    if ( effects.size < 1 ) return;
    event.preventDefault();
    event.stopPropagation();
    if ( effects.size === 1 ) {
      actor.endConcentration(effects.first());
      return;
    }
    const choices = effects.reduce((acc, effect) => {
      const data = effect.getFlag("dnd5e", "item.data");
      acc[effect.id] = data?.name ?? actor.items.get(data)?.name ?? game.i18n.localize("DND5E.ConcentratingItemless");
      return acc;
    }, {});
    const options = HandlebarsHelpers.selectOptions(choices, { hash: { sort: true } });
    const content = `
    <form class="dnd5e">
      <p>${game.i18n.localize("DND5E.ConcentratingEndChoice")}</p>
      <div class="form-group">
        <label>${game.i18n.localize("DND5E.SOURCE.FIELDS.source.label")}</label>
        <div class="form-fields">
          <select name="source">${options}</select>
        </div>
      </div>
    </form>`;
    Dialog.prompt({
      content: content,
      callback: ([html]) => {
        const source = new FormDataExtended(html.querySelector("FORM")).object.source;
        if ( source ) actor.endConcentration(source);
      },
      rejectClose: false,
      title: game.i18n.localize("DND5E.Concentration"),
      label: game.i18n.localize("DND5E.Confirm")
    });
  }

  /* -------------------------------------------- */

  /**
   * Record another effect as a dependent of this one.
   * @param {...ActiveEffect5e} dependent  One or more dependent effects.
   * @returns {Promise<ActiveEffect5e>}
   */
  addDependent(...dependent) {
    const dependents = this.getFlag("dnd5e", "dependents") ?? [];
    dependents.push(...dependent.map(d => ({ uuid: d.uuid })));
    return this.setFlag("dnd5e", "dependents", dependents);
  }

  /* -------------------------------------------- */

  /**
   * Retrieve a list of dependent effects.
   * @returns {Array<ActiveEffect5e|Item5e>}
   */
  getDependents() {
    return (this.getFlag("dnd5e", "dependents") || []).reduce((arr, { uuid }) => {
      let effect;
      // TODO: Remove this special casing once https://github.com/foundryvtt/foundryvtt/issues/11214 is resolved
      if ( this.parent.pack && uuid.includes(this.parent.uuid) ) {
        const [, embeddedName, id] = uuid.replace(this.parent.uuid, "").split(".");
        effect = this.parent.getEmbeddedDocument(embeddedName, id);
      }
      else effect = fromUuidSync(uuid, { strict: false });
      if ( effect ) arr.push(effect);
      return arr;
    }, []);
  }

  /* -------------------------------------------- */
  /*  Helpers                                     */
  /* -------------------------------------------- */

  /**
   * Helper method to add choices that have been overridden by an active effect. Used to determine what fields might
   * need to be disabled because they are overridden by an active effect in a way not easily determined by looking at
   * the `Document#overrides` data structure.
   * @param {Actor5e|Item5e} doc  Document from which to determine the overrides.
   * @param {string} prefix       The initial form prefix under which the choices are grouped.
   * @param {string} path         Path in document data.
   * @param {string[]} overrides  The list of fields that are currently modified by Active Effects. *Will be mutated.*
   */
  static addOverriddenChoices(doc, prefix, path, overrides) {
    const source = new Set(foundry.utils.getProperty(doc._source, path) ?? []);
    const current = foundry.utils.getProperty(doc, path) ?? new Set();
    const delta = current.symmetricDifference(source);
    for ( const choice of delta ) overrides.push(`${prefix}.${choice}`);
  }

  /* -------------------------------------------- */

  /**
   * Render a rich tooltip for this effect.
   * @param {EnrichmentOptions} [enrichmentOptions={}]  Options for text enrichment.
   * @returns {Promise<{content: string, classes: string[]}>}
   */
  async richTooltip(enrichmentOptions={}) {
    const properties = [];
    if ( this.isSuppressed ) properties.push("DND5E.EffectType.Unavailable");
    else if ( this.disabled ) properties.push("DND5E.EffectType.Inactive");
    else if ( this.isTemporary ) properties.push("DND5E.EffectType.Temporary");
    else properties.push("DND5E.EffectType.Passive");
    if ( this.type === "enchantment" ) properties.push("DND5E.ENCHANTMENT.Label");

    return {
      content: await renderTemplate(
        "systems/dnd5e/templates/effects/parts/effect-tooltip.hbs", {
          effect: this,
          description: await TextEditor.enrichHTML(this.description ?? "", { relativeTo: this, ...enrichmentOptions }),
          durationParts: this.duration.remaining ? this.duration.label.split(", ") : [],
          properties: properties.map(p => game.i18n.localize(p))
        }
      ),
      classes: ["dnd5e2", "dnd5e-tooltip", "effect-tooltip"]
    };
  }

  /* -------------------------------------------- */

  /** @override */
  async deleteDialog(dialogOptions={}, operation={}) {
    const type = game.i18n.localize(this.constructor.metadata.label);
    return foundry.applications.api.DialogV2.confirm(foundry.utils.mergeObject({
      window: { title: `${game.i18n.format("DOCUMENT.Delete", { type })}: ${this.name}` },
      position: { width: 400 },
      content: `
        <p>
            <strong>${game.i18n.localize("AreYouSure")}</strong> ${game.i18n.format("SIDEBAR.DeleteWarning", { type })}
        </p>
      `,
      yes: { callback: () => this.delete(operation) }
    }, dialogOptions));
  }
}

/**
 * Dialog for choosing an activity to use on an Item.
 */
class ActivityChoiceDialog extends Application5e {
  /**
   * @param {Item5e} item                         The Item whose activities are being chosen.
   * @param {ApplicationConfiguration} [options]  Application configuration options.
   */
  constructor(item, options={}) {
    super(options);
    this.#item = item;
  }

  /** @override */
  static DEFAULT_OPTIONS = {
    classes: ["activity-choice"],
    actions: {
      choose: ActivityChoiceDialog.#onChooseActivity
    },
    position: {
      width: 350
    }
  };

  static PARTS = {
    activities: {
      template: "systems/dnd5e/templates/activity/activity-choices.hbs"
    }
  };

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * The chosen activity.
   * @type {Activity|null}
   */
  get activity() {
    return this.#activity ?? null;
  }

  #activity;

  /* -------------------------------------------- */

  /**
   * The Item whose activities are being chosen.
   * @type {Item5e}
   */
  get item() {
    return this.#item;
  }

  #item;

  /* -------------------------------------------- */

  /** @override */
  get title() {
    return this.#item.name;
  }

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  _configureRenderOptions(options) {
    super._configureRenderOptions(options);
    if ( options.isFirstRender ) options.window.icon ||= this.#item.img;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareContext(options) {
    let controlHint;
    if ( game.settings.get("dnd5e", "controlHints") ) {
      controlHint = game.i18n.localize("DND5E.Controls.Activity.FastForwardHint");
      controlHint = controlHint.replace(
        "<left-click>",
        `<img src="systems/dnd5e/icons/svg/mouse-left.svg" alt="${game.i18n.localize("DND5E.Controls.LeftClick")}">`
      );
    }
    const activities = this.#item.system.activities
      .filter(a => !this.#item.getFlag("dnd5e", "riders.activity")?.includes(a.id) && a.canUse)
      .map(this._prepareActivityContext.bind(this))
      .sort((a, b) => a.sort - b.sort);
    return {
      ...await super._prepareContext(options),
      controlHint, activities
    };
  }

  /* -------------------------------------------- */

  /**
   * @typedef ActivityChoiceDialogContext
   * @property {string} id
   * @property {string} name
   * @property {number} sort
   * @property {object} icon
   * @property {string} icon.src
   * @property {boolean} icon.svg
   */

  /**
   * Prepare rendering context for a given activity.
   * @param {Activity} activity  The activity.
   * @returns {ActivityChoiceDialogContext}
   * @protected
   */
  _prepareActivityContext(activity) {
    const { id, name, img, sort } = activity;
    return {
      id, name, sort,
      icon: {
        src: img,
        svg: img.endsWith(".svg")
      }
    };
  }

  /* -------------------------------------------- */
  /*  Event Listeners & Handlers                  */
  /* -------------------------------------------- */

  /**
   * Handle choosing an activity.
   * @this {ActivityChoiceDialog}
   * @param {PointerEvent} event  The triggering click event.
   * @param {HTMLElement} target  The activity button that was clicked.
   */
  static async #onChooseActivity(event, target) {
    const { activityId } = target.dataset;
    this.#activity = this.#item.system.activities.get(activityId);
    this.close();
  }

  /* -------------------------------------------- */
  /*  Factory Methods                             */
  /* -------------------------------------------- */

  /**
   * Display the activity choice dialog.
   * @param {Item5e} item                         The Item whose activities are being chosen.
   * @param {ApplicationConfiguration} [options]  Application configuration options.
   * @returns {Promise<Activity|null>}            The chosen activity, or null if the dialog was dismissed.
   */
  static create(item, options) {
    return new Promise(resolve => {
      const dialog = new this(item, options);
      dialog.addEventListener("close", () => resolve(dialog.activity), { once: true });
      dialog.render({ force: true });
    });
  }
}

/**
 * Internal type used to manage each step within the advancement process.
 *
 * @typedef {object} AdvancementStep
 * @property {string} type                Step type from "forward", "reverse", "restore", or "delete".
 * @property {AdvancementFlow} [flow]     Flow object for the advancement being applied by this step. In the case of
 *                                        "delete" steps, this flow indicates the advancement flow that originally
 *                                        deleted the item.
 * @property {Item5e} [item]              For "delete" steps only, the item to be removed.
 * @property {object} [class]             Contains data on class if step was triggered by class level change.
 * @property {Item5e} [class.item]        Class item that caused this advancement step.
 * @property {number} [class.level]       Level the class should be during this step.
 * @property {boolean} [automatic=false]  Should the manager attempt to apply this step without user interaction?
 * @property {boolean} [synthetic=false]  Was this step created as a result of an item introduced or deleted?
 */

/**
 * @typedef AdvancementManagerConfiguration
 * @property {boolean} [automaticApplication=false]  Apply advancement steps automatically if no user input is required.
 * @property {boolean} [showVisualizer=false]        Display the step debugging application.
 */

/**
 * Application for controlling the advancement workflow and displaying the interface.
 *
 * @param {Actor5e} actor        Actor on which this advancement is being performed.
 * @param {object} [options={}]  Additional application options.
 */
class AdvancementManager extends Application5e {
  constructor(actor, options={}) {
    super(options);
    this.actor = actor;
    this.clone = actor.clone();
    if ( this.options.showVisualizer ) this.#visualizer = new AdvancementVisualizer({ manager: this });
  }

  /* -------------------------------------------- */

  /** @override */
  static DEFAULT_OPTIONS = {
    classes: ["advancement", "manager"],
    window: {
      icon: "fa-solid fa-forward",
      title: "DND5E.ADVANCEMENT.Manager.Title.Default"
    },
    actions: {
      complete: AdvancementManager.#process,
      next: AdvancementManager.#process,
      previous: AdvancementManager.#process,
      restart: AdvancementManager.#process
    },
    position: {
      width: 460
    },
    automaticApplication: false,
    showVisualizer: false
  };

  /* -------------------------------------------- */

  /** @override */
  static PARTS = {
    manager: {
      template: "systems/dnd5e/templates/advancement/advancement-manager.hbs"
    }
  };

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * The original actor to which changes will be applied when the process is complete.
   * @type {Actor5e}
   */
  actor;

  /* -------------------------------------------- */

  /**
   * Is the prompt currently advancing through un-rendered steps?
   * @type {boolean}
   */
  #advancing = false;

  /* -------------------------------------------- */

  /**
   * A clone of the original actor to which the changes can be applied during the advancement process.
   * @type {Actor5e}
   */
  clone;

  /* -------------------------------------------- */

  /** @inheritDoc */
  get subtitle() {
    const parts = [];

    // Item Name
    const item = this.step.flow.item;
    parts.push(item.name);

    // Class/Subclass level
    let level = this.step.flow.level;
    if ( this.step.class && ["class", "subclass"].includes(item.type) ) level = this.step.class.level;
    if ( level ) parts.push(game.i18n.format("DND5E.AdvancementLevelHeader", { level }));

    // Step Count
    const visibleSteps = this.steps.filter(s => !s.automatic);
    const visibleIndex = visibleSteps.indexOf(this.step);
    if ( visibleIndex >= 0 ) parts.push(game.i18n.format("DND5E.ADVANCEMENT.Manager.Steps", {
      current: visibleIndex + 1,
      total: visibleSteps.length
    }));

    return parts.join(" • ");
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  get id() {
    return `actor-${this.actor.id}-advancement`;
  }

  /* -------------------------------------------- */

  /**
   * Get the step that is currently in progress.
   * @type {object|null}
   */
  get step() {
    return this.steps[this.#stepIndex] ?? null;
  }

  /* -------------------------------------------- */

  /**
   * Step being currently displayed.
   * @type {number|null}
   */
  #stepIndex = null;

  /* -------------------------------------------- */

  /**
   * Individual steps that will be applied in order.
   * @type {AdvancementStep[]}
   */
  steps = [];

  /* -------------------------------------------- */

  /**
   * Get the step before the current one.
   * @type {object|null}
   */
  get previousStep() {
    return this.steps[this.#stepIndex - 1] ?? null;
  }

  /* -------------------------------------------- */

  /**
   * Get the step after the current one.
   * @type {object|null}
   */
  get nextStep() {
    const nextIndex = this.#stepIndex === null ? 0 : this.#stepIndex + 1;
    return this.steps[nextIndex] ?? null;
  }

  /* -------------------------------------------- */

  /**
   * Side application for debugging advancement steps.
   * @type {AdvancementVisualizer}
   */
  #visualizer;

  /* -------------------------------------------- */
  /*  Factory Methods                             */
  /* -------------------------------------------- */

  /**
   * Construct a manager for a newly added advancement from drag-drop.
   * @param {Actor5e} actor               Actor from which the advancement should be updated.
   * @param {string} itemId               ID of the item to which the advancements are being dropped.
   * @param {Advancement[]} advancements  Dropped advancements to add.
   * @param {object} [options={}]         Rendering options passed to the application.
   * @returns {AdvancementManager}  Prepared manager. Steps count can be used to determine if advancements are needed.
   */
  static forNewAdvancement(actor, itemId, advancements, options={}) {
    const manager = new this(actor, options);
    const clonedItem = manager.clone.items.get(itemId);
    if ( !clonedItem || !advancements.length ) return manager;

    const currentLevel = this.currentLevel(clonedItem, manager.clone);
    const minimumLevel = advancements.reduce((min, a) => Math.min(a.levels[0] ?? Infinity, min), Infinity);
    if ( minimumLevel > currentLevel ) return manager;

    const oldFlows = Array.fromRange(currentLevel + 1).slice(minimumLevel)
      .flatMap(l => this.flowsForLevel(clonedItem, l));

    // Revert advancements through minimum level
    oldFlows.reverse().forEach(flow => manager.steps.push({ type: "reverse", flow, automatic: true }));

    // Add new advancements
    const advancementArray = clonedItem.toObject().system.advancement;
    advancementArray.push(...advancements.map(a => {
      const obj = a.toObject();
      if ( obj.constructor.dataModels?.value ) a.value = (new a.constructor.metadata.dataModels.value()).toObject();
      else obj.value = foundry.utils.deepClone(a.constructor.metadata.defaults?.value ?? {});
      return obj;
    }));
    clonedItem.updateSource({"system.advancement": advancementArray});

    const newFlows = Array.fromRange(currentLevel + 1).slice(minimumLevel)
      .flatMap(l => this.flowsForLevel(clonedItem, l));

    // Restore existing advancements and apply new advancements
    newFlows.forEach(flow => {
      const matchingFlow = oldFlows.find(f => (f.advancement.id === flow.advancement.id) && (f.level === flow.level));
      if ( matchingFlow ) manager.steps.push({ type: "restore", flow: matchingFlow, automatic: true });
      else manager.steps.push({ type: "forward", flow });
    });

    return manager;
  }

  /* -------------------------------------------- */

  /**
   * Construct a manager for a newly added item.
   * @param {Actor5e} actor         Actor to which the item is being added.
   * @param {object} itemData       Data for the item being added.
   * @param {object} [options={}]   Rendering options passed to the application.
   * @returns {AdvancementManager}  Prepared manager. Steps count can be used to determine if advancements are needed.
   */
  static forNewItem(actor, itemData, options={}) {
    const manager = new this(actor, options);

    // Prepare data for adding to clone
    const dataClone = foundry.utils.deepClone(itemData);
    dataClone._id = foundry.utils.randomID();
    if ( itemData.type === "class" ) {
      dataClone.system.levels = 0;
      if ( !manager.clone.system.details.originalClass ) {
        manager.clone.updateSource({"system.details.originalClass": dataClone._id});
      }
    }

    // Add item to clone & get new instance from clone
    manager.clone.updateSource({items: [dataClone]});
    const clonedItem = manager.clone.items.get(dataClone._id);

    // For class items, prepare level change data
    if ( itemData.type === "class" ) {
      return manager.createLevelChangeSteps(clonedItem, itemData.system?.levels ?? 1);
    }

    // All other items, just create some flows up to current character level (or class level for subclasses)
    let targetLevel = manager.clone.system.details.level ?? 0;
    if ( clonedItem.type === "subclass" ) targetLevel = clonedItem.class?.system.levels ?? 0;
    Array.fromRange(targetLevel + 1)
      .flatMap(l => this.flowsForLevel(clonedItem, l))
      .forEach(flow => manager.steps.push({ type: "forward", flow }));

    return manager;
  }

  /* -------------------------------------------- */

  /**
   * Construct a manager for modifying choices on an item at a specific level.
   * @param {Actor5e} actor         Actor from which the choices should be modified.
   * @param {object} itemId         ID of the item whose choices are to be changed.
   * @param {number} level          Level at which the choices are being changed.
   * @param {object} [options={}]   Rendering options passed to the application.
   * @returns {AdvancementManager}  Prepared manager. Steps count can be used to determine if advancements are needed.
   */
  static forModifyChoices(actor, itemId, level, options={}) {
    const manager = new this(actor, options);
    const clonedItem = manager.clone.items.get(itemId);
    if ( !clonedItem ) return manager;

    const flows = Array.fromRange(this.currentLevel(clonedItem, manager.clone) + 1).slice(level)
      .flatMap(l => this.flowsForLevel(clonedItem, l));

    // Revert advancements through changed level
    flows.reverse().forEach(flow => manager.steps.push({ type: "reverse", flow, automatic: true }));

    // Create forward advancements for level being changed
    flows.reverse().filter(f => f.level === level).forEach(flow => manager.steps.push({ type: "forward", flow }));

    // Create restore advancements for other levels
    flows.filter(f => f.level > level).forEach(flow => manager.steps.push({ type: "restore", flow, automatic: true }));

    return manager;
  }

  /* -------------------------------------------- */

  /**
   * Construct a manager for an advancement that needs to be deleted.
   * @param {Actor5e} actor         Actor from which the advancement should be unapplied.
   * @param {string} itemId         ID of the item from which the advancement should be deleted.
   * @param {string} advancementId  ID of the advancement to delete.
   * @param {object} [options={}]   Rendering options passed to the application.
   * @returns {AdvancementManager}  Prepared manager. Steps count can be used to determine if advancements are needed.
   */
  static forDeletedAdvancement(actor, itemId, advancementId, options={}) {
    const manager = new this(actor, options);
    const clonedItem = manager.clone.items.get(itemId);
    const advancement = clonedItem?.advancement.byId[advancementId];
    if ( !advancement ) return manager;

    const minimumLevel = advancement.levels[0];
    const currentLevel = this.currentLevel(clonedItem, manager.clone);

    // If minimum level is greater than current level, no changes to remove
    if ( (minimumLevel > currentLevel) || !advancement.appliesToClass ) return manager;

    advancement.levels
      .reverse()
      .filter(l => l <= currentLevel)
      .map(l => new advancement.constructor.metadata.apps.flow(clonedItem, advancementId, l))
      .forEach(flow => manager.steps.push({ type: "reverse", flow, automatic: true }));

    if ( manager.steps.length ) manager.steps.push({ type: "delete", advancement, automatic: true });

    return manager;
  }

  /* -------------------------------------------- */

  /**
   * Construct a manager for an item that needs to be deleted.
   * @param {Actor5e} actor         Actor from which the item should be deleted.
   * @param {string} itemId         ID of the item to be deleted.
   * @param {object} [options={}]   Rendering options passed to the application.
   * @returns {AdvancementManager}  Prepared manager. Steps count can be used to determine if advancements are needed.
   */
  static forDeletedItem(actor, itemId, options={}) {
    const manager = new this(actor, options);
    const clonedItem = manager.clone.items.get(itemId);
    if ( !clonedItem ) return manager;

    // For class items, prepare level change data
    if ( clonedItem.type === "class" ) {
      return manager.createLevelChangeSteps(clonedItem, clonedItem.system.levels * -1);
    }

    // All other items, just create some flows down from current character level
    Array.fromRange((manager.clone.system.details.level ?? 0) + 1)
      .flatMap(l => this.flowsForLevel(clonedItem, l))
      .reverse()
      .forEach(flow => manager.steps.push({ type: "reverse", flow, automatic: true }));

    // Add a final step to remove the item only if there are advancements to apply
    if ( manager.steps.length ) manager.steps.push({ type: "delete", item: clonedItem, automatic: true });
    return manager;
  }

  /* -------------------------------------------- */

  /**
   * Construct a manager for a change in a class's levels.
   * @param {Actor5e} actor         Actor whose level has changed.
   * @param {string} classId        ID of the class being changed.
   * @param {number} levelDelta     Levels by which to increase or decrease the class.
   * @param {object} options        Rendering options passed to the application.
   * @returns {AdvancementManager}  Prepared manager. Steps count can be used to determine if advancements are needed.
   */
  static forLevelChange(actor, classId, levelDelta, options={}) {
    const manager = new this(actor, options);
    const clonedItem = manager.clone.items.get(classId);
    if ( !clonedItem ) return manager;
    return manager.createLevelChangeSteps(clonedItem, levelDelta);
  }

  /* -------------------------------------------- */

  /**
   * Create steps based on the provided level change data.
   * @param {string} classItem      Class being changed.
   * @param {number} levelDelta     Levels by which to increase or decrease the class.
   * @returns {AdvancementManager}  Manager with new steps.
   */
  createLevelChangeSteps(classItem, levelDelta) {
    const raceItem = this.clone.system?.details?.race instanceof Item ? this.clone.system.details.race : null;
    const pushSteps = (flows, data) => this.steps.push(...flows.map(flow => ({ flow, ...data })));
    const getItemFlows = characterLevel => this.clone.items.contents.flatMap(i => {
      if ( ["class", "subclass", "race"].includes(i.type) ) return [];
      return this.constructor.flowsForLevel(i, characterLevel);
    });

    // Level increased
    for ( let offset = 1; offset <= levelDelta; offset++ ) {
      const classLevel = classItem.system.levels + offset;
      const characterLevel = (this.actor.system.details.level ?? 0) + offset;
      const stepData = { type: "forward", class: {item: classItem, level: classLevel} };
      pushSteps(this.constructor.flowsForLevel(raceItem, characterLevel), stepData);
      pushSteps(this.constructor.flowsForLevel(classItem, classLevel), stepData);
      pushSteps(this.constructor.flowsForLevel(classItem.subclass, classLevel), stepData);
      pushSteps(getItemFlows(characterLevel), stepData);
    }

    // Level decreased
    for ( let offset = 0; offset > levelDelta; offset-- ) {
      const classLevel = classItem.system.levels + offset;
      const characterLevel = (this.actor.system.details.level ?? 0) + offset;
      const stepData = { type: "reverse", class: {item: classItem, level: classLevel}, automatic: true };
      pushSteps(getItemFlows(characterLevel).reverse(), stepData);
      pushSteps(this.constructor.flowsForLevel(classItem.subclass, classLevel).reverse(), stepData);
      pushSteps(this.constructor.flowsForLevel(classItem, classLevel).reverse(), stepData);
      pushSteps(this.constructor.flowsForLevel(raceItem, characterLevel).reverse(), stepData);
      if ( classLevel === 1 ) this.steps.push({ type: "delete", item: classItem, automatic: true });
    }

    // Ensure the class level ends up at the appropriate point
    this.steps.push({
      type: "forward", automatic: true,
      class: { item: classItem, level: classItem.system.levels += levelDelta }
    });

    return this;
  }

  /* -------------------------------------------- */

  /**
   * Creates advancement flows for all advancements at a specific level.
   * @param {Item5e} item                               Item that has advancement.
   * @param {number} level                              Level in question.
   * @param {object} [options={}]
   * @param {AdvancementStep[]} [options.findExisting]  Find if an existing matching flow exists.
   * @returns {AdvancementFlow[]}                       Created or matched flow applications.
   */
  static flowsForLevel(item, level, { findExisting }={}) {
    const match = (advancement, step) => (step.flow?.item.id === item.id)
      && (step.flow?.advancement.id === advancement.id)
      && (step.flow?.level === level);
    return (item?.advancement.byLevel[level] ?? [])
      .filter(a => a.appliesToClass)
      .map(a => {
        const existing = findExisting?.find(s => match(a, s))?.flow;
        if ( !existing ) return new a.constructor.metadata.apps.flow(item, a.id, level);
        existing.item = item;
        return existing;
      });
  }

  /* -------------------------------------------- */

  /**
   * Determine the proper working level either from the provided item or from the cloned actor.
   * @param {Item5e} item    Item being advanced. If class or subclass, its level will be used.
   * @param {Actor5e} actor  Actor being advanced.
   * @returns {number}       Working level.
   */
  static currentLevel(item, actor) {
    return item.system.levels ?? item.class?.system.levels ?? actor.system.details.level ?? 0;
  }

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  _configureRenderOptions(options) {
    super._configureRenderOptions(options);
    options.window ??= {};
    options.window.subtitle ??= this.subtitle;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareContext(options) {
    const context = await super._prepareContext(options);
    if ( !this.step ) return context;

    const visibleSteps = this.steps.filter(s => !s.automatic);
    const visibleIndex = visibleSteps.indexOf(this.step);

    return {
      ...context,
      actor: this.clone,
      // Keep styles from non-converted flow applications functioning
      // Should be removed when V1 of `AdvancementFlow` is deprecated
      flowClasses: this.step.flow instanceof Application ? "dnd5e advancement flow" : "",
      flowId: this.step.flow.id,
      steps: {
        current: visibleIndex + 1,
        total: visibleSteps.length,
        hasPrevious: visibleIndex > 0,
        hasNext: visibleIndex < visibleSteps.length - 1
      }
    };
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  render(forced=false, options={}) {
    if ( this.steps.length && (this.#stepIndex === null) ) this.#stepIndex = 0;

    // Ensure the level on the class item matches the specified level
    if ( this.step?.class ) {
      let level = this.step.class.level;
      if ( this.step.type === "reverse" ) level -= 1;
      this.step.class.item.updateSource({"system.levels": level});
      this.clone.reset();
    }

    /**
     * A hook event that fires when an AdvancementManager is about to be processed.
     * @function dnd5e.preAdvancementManagerRender
     * @memberof hookEvents
     * @param {AdvancementManager} advancementManager The advancement manager about to be rendered
     */
    const allowed = Hooks.call("dnd5e.preAdvancementManagerRender", this);

    // Abort if not allowed
    if ( allowed === false ) return this;

    const automaticData = (this.options.automaticApplication && (options.direction !== "backward"))
      ? this.step?.flow?.getAutomaticApplicationValue() : false;

    if ( this.step?.automatic || (automaticData !== false) ) {
      if ( this.#advancing ) return this;
      this.#forward({ automaticData });
      return this;
    }

    return super.render(forced, options);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _onRender(context, options) {
    super._onRender(context, options);
    if ( !this.rendered || !this.step ) return;
    this.#visualizer?.render({ force: true });

    // Render the step
    this.step.flow._element = null;
    this.step.flow.options.manager ??= this;
    await this.step.flow._render(true, options);
    this.setPosition();
  }

  /* -------------------------------------------- */
  /*  Life-Cycle Handlers                         */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async close(options={}) {
    if ( !options.skipConfirmation ) {
      return new Dialog({
        title: `${game.i18n.localize("DND5E.ADVANCEMENT.Manager.ClosePrompt.Title")}: ${this.actor.name}`,
        content: game.i18n.localize("DND5E.ADVANCEMENT.Manager.ClosePrompt.Message"),
        buttons: {
          close: {
            icon: '<i class="fas fa-times" inert></i>',
            label: game.i18n.localize("DND5E.ADVANCEMENT.Manager.ClosePrompt.Action.Stop"),
            callback: () => {
              this.#visualizer?.close();
              super.close(options);
            }
          },
          continue: {
            icon: '<i class="fas fa-chevron-right" inert></i>',
            label: game.i18n.localize("DND5E.ADVANCEMENT.Manager.ClosePrompt.Action.Continue")
          }
        },
        default: "close"
      }).render(true);
    }
    this.#visualizer?.close();
    await super.close(options);
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /**
   * Handle one of the buttons for moving through the process.
   * @this {AdvancementManager}
   * @param {Event} event         Triggering click event.
   * @param {HTMLElement} target  Button that was clicked.
   */
  static async #process(event, target) {
    target.disabled = true;
    this.element.querySelector(".error")?.classList.remove("error");
    try {
      switch ( target.dataset.action ) {
        case "restart":
          if ( this.previousStep ) await this.#restart(event);
          break;
        case "previous":
          if ( this.previousStep ) await this.#backward(event);
          break;
        case "next":
        case "complete":
          await this.#forward(event);
          break;
      }
    } finally {
      target.disabled = false;
    }
  }

  /* -------------------------------------------- */
  /*  Process                                     */
  /* -------------------------------------------- */

  /**
   * Advance through the steps until one requiring user interaction is encountered.
   * @param {object} config
   * @param {object} [config.automaticData]  Data provided to handle automatic application.
   * @param {Event} [config.event]           Triggering click event if one occurred.
   * @returns {Promise}
   */
  async #forward({ automaticData, event }) {
    this.#advancing = true;
    try {
      do {
        const flow = this.step.flow;
        const type = this.step.type;
        const preEmbeddedItems = Array.from(this.clone.items);

        // Apply changes based on step type
        if ( (type === "delete") && this.step.item ) {
          if ( this.step.flow?.retainedData?.retainedItems ) {
            this.step.flow.retainedData.retainedItems[this.step.item.flags.dnd5e?.sourceId] = this.step.item.toObject();
          }
          this.clone.items.delete(this.step.item.id);
        } else if ( (type === "delete") && this.step.advancement ) {
          this.step.advancement.item.deleteAdvancement(this.step.advancement.id, { source: true });
        }
        else if ( type === "restore" ) await flow.advancement.restore(flow.level, flow.retainedData);
        else if ( type === "reverse" ) await flow.retainData(await flow.advancement.reverse(flow.level));
        else if ( automaticData && flow ) await flow.advancement.apply(flow.level, automaticData);
        else if ( flow ) await flow._updateObject(event, flow._getSubmitData());

        this.#synthesizeSteps(preEmbeddedItems);
        this.#stepIndex++;

        // Ensure the level on the class item matches the specified level
        if ( this.step?.class ) {
          let level = this.step.class.level;
          if ( this.step.type === "reverse" ) level -= 1;
          this.step.class.item.updateSource({"system.levels": level});
        }
        this.clone.reset();
      } while ( this.step?.automatic );
    } catch(error) {
      if ( !(error instanceof Advancement.ERROR) ) throw error;
      ui.notifications.error(error.message);
      this.step.automatic = false;
      if ( this.step.type === "restore" ) this.step.type = "forward";
    } finally {
      this.#advancing = false;
    }

    if ( this.step ) this.render({ force: true, direction: "forward" });
    else this.#complete();
  }

  /* -------------------------------------------- */

  /**
   * Add synthetic steps for any added or removed items with advancement.
   * @param {Item5e[]} preEmbeddedItems  Items present before the current step was applied.
   */
  #synthesizeSteps(preEmbeddedItems) {
    // Build a set of item IDs for non-synthetic steps
    const initialIds = this.steps.reduce((ids, step) => {
      if ( step.synthetic || !step.flow?.item ) return ids;
      ids.add(step.flow.item.id);
      return ids;
    }, new Set());

    const preIds = new Set(preEmbeddedItems.map(i => i.id));
    const postIds = new Set(this.clone.items.map(i => i.id));
    const addedIds = postIds.difference(preIds).difference(initialIds);
    const deletedIds = preIds.difference(postIds).difference(initialIds);

    for ( const addedId of addedIds ) {
      const item = this.clone.items.get(addedId);
      if ( !item.hasAdvancement ) continue;

      let handledLevel = 0;
      for ( let idx = this.#stepIndex; idx < this.steps.length; idx++ ) {
        // Find spots where the level increases
        const thisLevel = this.steps[idx].flow?.level ?? this.steps[idx].class?.level;
        const nextLevel = this.steps[idx + 1]?.flow?.level ?? this.steps[idx + 1]?.class?.level;
        if ( (thisLevel < handledLevel) || (thisLevel === nextLevel) ) continue;

        // Determine if there is any advancement to be done for the added item to this level
        // from the previously handled level
        const steps = Array.fromRange(thisLevel - handledLevel + 1, handledLevel)
          .flatMap(l => this.constructor.flowsForLevel(item, l, { findExisting: this.steps }))
          .map(flow => ({ type: "forward", flow, synthetic: true }));

        // Add new steps at the end of the level group
        this.steps.splice(idx + 1, 0, ...steps);
        idx += steps.length;

        handledLevel = nextLevel ?? handledLevel;
      }
    }

    if ( (this.step.type === "delete") && this.step.synthetic ) return;
    for ( const deletedId of deletedIds ) {
      let item = preEmbeddedItems.find(i => i.id === deletedId);
      if ( !item?.hasAdvancement ) continue;

      // Temporarily add the item back
      this.clone.updateSource({items: [item.toObject()]});
      item = this.clone.items.get(item.id);

      // Check for advancement from the maximum level handled by this manager to zero
      let steps = [];
      Array.fromRange(this.clone.system.details.level + 1)
        .flatMap(l => this.constructor.flowsForLevel(item, l))
        .reverse()
        .forEach(flow => steps.push({ type: "reverse", flow, automatic: true, synthetic: true }));

      // Add a new remove item step to the end of the synthetic steps to finally get rid of this item
      steps.push({ type: "delete", flow: this.step.flow, item, automatic: true, synthetic: true });

      // Add new steps after the current step
      this.steps.splice(this.#stepIndex + 1, 0, ...steps);
    }
  }

  /* -------------------------------------------- */

  /**
   * Reverse through the steps until one requiring user interaction is encountered.
   * @param {Event} [event]                  Triggering click event if one occurred.
   * @param {object} [options]               Additional options to configure behavior.
   * @param {boolean} [options.render=true]  Whether to render the Application after the step has been reversed. Used
   *                                         by the restart workflow.
   * @returns {Promise}
   */
  async #backward(event, { render=true }={}) {
    this.#advancing = true;
    try {
      do {
        this.#stepIndex--;
        if ( !this.step ) break;
        const flow = this.step.flow;
        const type = this.step.type;
        const preEmbeddedItems = Array.from(this.clone.items);

        // Reverse step based on step type
        if ( (type === "delete") && this.step.item ) this.clone.updateSource({items: [this.step.item]});
        else if ( (type === "delete") && this.step.advancement ) this.advancement.item.createAdvancement(
          this.advancement.typeName, this.advancement._source, { source: true }
        );
        else if ( type === "reverse" ) await flow.advancement.restore(flow.level, flow.retainedData);
        else if ( flow ) await flow.retainData(await flow.advancement.reverse(flow.level));

        this.#clearSyntheticSteps(preEmbeddedItems);
        this.clone.reset();
      } while ( this.step?.automatic );
    } catch(error) {
      if ( !(error instanceof Advancement.ERROR) ) throw error;
      ui.notifications.error(error.message);
      this.step.automatic = false;
    } finally {
      this.#advancing = false;
    }

    if ( !render ) return;
    if ( this.step ) this.render(true, { direction: "backward" });
    else this.close({ skipConfirmation: true });
  }

  /* -------------------------------------------- */

  /**
   * Remove synthetic steps for any added or removed items.
   * @param {Item5e[]} preEmbeddedItems  Items present before the current step was applied.
   */
  #clearSyntheticSteps(preEmbeddedItems) {
    // Create a disjoint union of the before and after items
    const preIds = new Set(preEmbeddedItems.map(i => i.id));
    const postIds = new Set(this.clone.items.map(i => i.id));
    const modifiedIds = postIds.symmetricDifference(preIds);

    // Remove any synthetic steps after the current step if their item has been modified
    for ( const [idx, element] of Array.from(this.steps.entries()).reverse() ) {
      if ( idx <= this.#stepIndex ) break;
      if ( element.synthetic && modifiedIds.has(element.flow?.item?.id) ) this.steps.splice(idx, 1);
    }
  }

  /* -------------------------------------------- */

  /**
   * Reset back to the manager's initial state.
   * @param {MouseEvent} [event]  The triggering click event if one occurred.
   * @returns {Promise}
   */
  async #restart(event) {
    const restart = await Dialog.confirm({
      title: game.i18n.localize("DND5E.ADVANCEMENT.Manager.RestartPrompt.Title"),
      content: game.i18n.localize("DND5E.ADVANCEMENT.Manager.RestartPrompt.Message")
    });
    if ( !restart ) return;
    // While there is still a renderable step.
    while ( this.steps.slice(0, this.#stepIndex).some(s => !s.automatic) ) {
      await this.#backward(event, {render: false});
    }
    this.render(true);
  }

  /* -------------------------------------------- */

  /**
   * Apply changes to actual actor after all choices have been made.
   * @param {Event} event  Button click that triggered the change.
   * @returns {Promise}
   */
  async #complete(event) {
    const updates = this.clone.toObject();
    const items = updates.items;
    delete updates.items;

    // Gather changes to embedded items
    const { toCreate, toUpdate, toDelete } = items.reduce((obj, item) => {
      if ( !this.actor.items.get(item._id) ) {
        obj.toCreate.push(item);
      } else {
        obj.toUpdate.push(item);
        obj.toDelete.findSplice(id => id === item._id);
      }
      return obj;
    }, { toCreate: [], toUpdate: [], toDelete: this.actor.items.map(i => i.id) });

    /**
     * A hook event that fires at the final stage of a character's advancement process, before actor and item updates
     * are applied.
     * @function dnd5e.preAdvancementManagerComplete
     * @memberof hookEvents
     * @param {AdvancementManager} advancementManager  The advancement manager.
     * @param {object} actorUpdates                    Updates to the actor.
     * @param {object[]} toCreate                      Items that will be created on the actor.
     * @param {object[]} toUpdate                      Items that will be updated on the actor.
     * @param {string[]} toDelete                      IDs of items that will be deleted on the actor.
     */
    if ( Hooks.call("dnd5e.preAdvancementManagerComplete", this, updates, toCreate, toUpdate, toDelete) === false ) {
      log("AdvancementManager completion was prevented by the 'preAdvancementManagerComplete' hook.");
      return this.close({ skipConfirmation: true });
    }

    // Apply changes from clone to original actor
    await Promise.all([
      this.actor.update(updates, { isAdvancement: true }),
      this.actor.createEmbeddedDocuments("Item", toCreate, { keepId: true, isAdvancement: true }),
      this.actor.updateEmbeddedDocuments("Item", toUpdate, { isAdvancement: true }),
      this.actor.deleteEmbeddedDocuments("Item", toDelete, { isAdvancement: true })
    ]);

    /**
     * A hook event that fires when an AdvancementManager is done modifying an actor.
     * @function dnd5e.advancementManagerComplete
     * @memberof hookEvents
     * @param {AdvancementManager} advancementManager The advancement manager that just completed
     */
    Hooks.callAll("dnd5e.advancementManagerComplete", this);

    // Close prompt
    return this.close({ skipConfirmation: true });
  }
}

/* -------------------------------------------- */

/**
 * Debug application for visualizing advancement steps.
 * Note: Intentionally not localized due to its nature as a debug application.
 */
class AdvancementVisualizer extends Application5e {
  /** @override */
  static DEFAULT_OPTIONS = {
    classes: ["advancement-visualizer"],
    window: {
      title: "Advancement Steps"
    },
    position: {
      top: 50,
      left: 50,
      width: 440
    },
    manager: null
  };

  /* -------------------------------------------- */

  /** @override */
  static PARTS = {
    steps: {
      template: "systems/dnd5e/templates/advancement/advancement-visualizer.hbs"
    }
  };

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * The advancement manager that this is visualizing.
   * @type {AdvancementManager}
   */
  get manager() {
    return this.options.manager;
  }

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareContext(options) {
    const context = await super._prepareContext(options);
    context.steps = this.manager.steps.map(step => ({
      ...step,
      current: step === this.manager.step
    }));
    return context;
  }
}

/**
 * Dialog to confirm the deletion of an embedded item with advancement or decreasing a class level.
 */
class AdvancementConfirmationDialog extends Dialog {

  /** @inheritDoc */
  static get defaultOptions() {
    return foundry.utils.mergeObject(super.defaultOptions, {
      template: "systems/dnd5e/templates/advancement/advancement-confirmation-dialog.hbs",
      jQuery: false
    });
  }

  /* -------------------------------------------- */

  /**
   * A helper function that displays the dialog prompting for an item deletion.
   * @param {Item5e} item  Item to be deleted.
   * @returns {Promise<boolean|null>}  Resolves with whether advancements should be unapplied. Rejects with null.
   */
  static forDelete(item) {
    return this.createDialog(
      item,
      game.i18n.localize("DND5E.AdvancementDeleteConfirmationTitle"),
      game.i18n.localize("DND5E.AdvancementDeleteConfirmationMessage"),
      {
        icon: '<i class="fas fa-trash"></i>',
        label: game.i18n.localize("Delete")
      }
    );
  }

  /* -------------------------------------------- */

  /**
   * A helper function that displays the dialog prompting for leveling down.
   * @param {Item5e} item  The class whose level is being changed.
   * @returns {Promise<boolean|null>}  Resolves with whether advancements should be unapplied. Rejects with null.
   */
  static forLevelDown(item) {
    return this.createDialog(
      item,
      game.i18n.localize("DND5E.AdvancementLevelDownConfirmationTitle"),
      game.i18n.localize("DND5E.AdvancementLevelDownConfirmationMessage"),
      {
        icon: '<i class="fas fa-sort-numeric-down-alt"></i>',
        label: game.i18n.localize("DND5E.LevelActionDecrease")
      }
    );
  }

  /* -------------------------------------------- */

  /**
   * A helper constructor function which displays the confirmation dialog.
   * @param {Item5e} item              Item to be changed.
   * @param {string} title             Localized dialog title.
   * @param {string} message           Localized dialog message.
   * @param {object} continueButton    Object containing label and icon for the action button.
   * @returns {Promise<boolean|null>}  Resolves with whether advancements should be unapplied. Rejects with null.
   */
  static createDialog(item, title, message, continueButton) {
    return new Promise((resolve, reject) => {
      const dialog = new this({
        title: `${title}: ${item.name}`,
        content: message,
        buttons: {
          continue: foundry.utils.mergeObject(continueButton, {
            callback: html => {
              const checkbox = html.querySelector('input[name="apply-advancement"]');
              resolve(checkbox.checked);
            }
          }),
          cancel: {
            icon: '<i class="fas fa-times"></i>',
            label: game.i18n.localize("Cancel"),
            callback: html => reject(null)
          }
        },
        default: "continue",
        close: () => reject(null)
      });
      dialog.render(true);
    });
  }

}

const { NumberField: NumberField$x, StringField: StringField$O } = foundry.data.fields;

/**
 * Application for configuration spell scroll creation.
 */
class CreateScrollDialog extends Dialog5e {
  constructor(options={}) {
    super(options);
    this.#config = options.config;
    this.#spell = options.spell;
  }

  /** @override */
  static DEFAULT_OPTIONS = {
    classes: ["create-scroll"],
    window: {
      title: "DND5E.Scroll.CreateScroll",
      icon: "fa-solid fa-scroll"
    },
    form: {
      handler: CreateScrollDialog.#handleFormSubmission
    },
    position: {
      width: 420
    },
    buttons: [{
      action: "create",
      label: "DND5E.Scroll.CreateScroll",
      icon: "fa-solid fa-check",
      default: true
    }],
    config: null,
    spell: null
  };

  /** @inheritDoc */
  static PARTS = {
    ...super.PARTS,
    content: {
      template: "systems/dnd5e/templates/apps/spell-scroll-dialog.hbs"
    }
  };

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * Configuration options for scroll creation.
   * @type {SpellScrollConfiguration}
   */
  #config;

  get config() {
    return this.#config;
  }

  /* -------------------------------------------- */

  /**
   * Spell from which the scroll will be created.
   * @type {Item5e|object}
   */
  #spell;

  get spell() {
    return this.#spell;
  }

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /**
   * Prepare rendering context for the content section.
   * @param {ApplicationRenderContext} context  Context being prepared.
   * @param {HandlebarsRenderOptions} options   Options which configure application rendering behavior.
   * @returns {Promise<ApplicationRenderContext>}
   * @protected
   */
  async _prepareContentContext(context, options) {
    context.anchor = this.spell instanceof Item ? this.spell.toAnchor().outerHTML : `<span>${this.spell.name}</span>`;
    context.config = this.config;
    context.fields = [{
      field: new StringField$O({
        label: game.i18n.localize("DND5E.Scroll.Explanation.Label"),
        hint: game.i18n.localize("DND5E.Scroll.Explanation.Hint")
      }),
      name: "explanation",
      options: [
        { value: "full", label: game.i18n.localize("DND5E.Scroll.Explanation.Complete") },
        { value: "reference", label: game.i18n.localize("DND5E.Scroll.Explanation.Reference") },
        { value: "none", label: game.i18n.localize("DND5E.None") }
      ],
      value: this.config.explanation ?? "reference"
    }, {
      field: new NumberField$x({ label: game.i18n.localize("DND5E.SpellLevel") }),
      name: "level",
      options: Object.entries(CONFIG.DND5E.spellLevels)
        .map(([value, label]) => ({ value, label }))
        .filter(l => Number(l.value) >= this.spell.system.level),
      value: this.config.level ?? this.spell.system.level
    }];
    context.values = {
      bonus: new NumberField$x({ label: game.i18n.localize("DND5E.BonusAttack") }),
      dc: new NumberField$x({ label: game.i18n.localize("DND5E.Scroll.SaveDC") })
    };
    context.valuePlaceholders = {};
    for ( const level of Array.fromRange(this.config.level + 1).reverse() ) {
      context.valuePlaceholders = CONFIG.DND5E.spellScrollValues[level];
      if ( context.valuePlaceholders ) break;
    }
    return context;
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /**
   * Handle submission of the dialog using the form buttons.
   * @this {CreateScrollDialog}
   * @param {Event|SubmitEvent} event    The form submission event.
   * @param {HTMLFormElement} form       The submitted form.
   * @param {FormDataExtended} formData  Data from the dialog.
   */
  static async #handleFormSubmission(event, form, formData) {
    foundry.utils.mergeObject(this.#config, formData.object);
    this.#config.level = Number(this.#config.level);
    await this.close({ dnd5e: { submitted: true } });
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _onChangeForm(formConfig, event) {
    super._onChangeForm(formConfig, event);
    const formData = new FormDataExtended(this.form);
    foundry.utils.mergeObject(this.#config, formData.object);
    this.#config.level = Number(this.#config.level);
    this.render({ parts: ["content"] });
  }

  /* -------------------------------------------- */

  /** @override */
  _onClose(options={}) {
    if ( !options.dnd5e?.submitted ) this.#config = null;
  }

  /* -------------------------------------------- */
  /*  Factory Methods                             */
  /* -------------------------------------------- */

  /**
   * Display the create spell scroll dialog.
   * @param {Item5e|object} spell              The spell or item data to be made into a scroll.
   * @param {SpellScrollConfiguration} config  Configuration options for scroll creation.
   * @param {object} [options={}]              Additional options for the application.
   * @returns {Promise<object|null>}           Form data object with results of the dialog.
   */
  static async create(spell, config, options={}) {
    return new Promise(resolve => {
      const dialog = new this({ spell, config, ...options });
      dialog.addEventListener("close", event => resolve(dialog.config), { once: true });
      dialog.render({ force: true });
    });
  }
}

const { BooleanField: BooleanField$s, StringField: StringField$N } = foundry.data.fields;

/**
 * Configuration application for traits.
 */
class TraitConfig extends AdvancementConfig$1 {
  constructor(...args) {
    super(...args);
    this.selected = (this.config.choices.length && !this.config.grants.size) ? 0 : -1;
    this.trait = this.types.first() ?? "skills";
  }

  /* -------------------------------------------- */

  /** @override */
  static DEFAULT_OPTIONS = {
    classes: ["trait", "trait-selector"],
    actions: {
      addChoice: TraitConfig.#addChoice,
      removeChoice: TraitConfig.#removeChoice
    },
    position: {
      width: 680
    }
  };

  /* -------------------------------------------- */

  /** @inheritDoc */
  static PARTS = {
    config: {
      container: { classes: ["column-container"], id: "column-left" },
      template: "systems/dnd5e/templates/advancement/advancement-controls-section.hbs"
    },
    details: {
      container: { classes: ["column-container"], id: "column-left" },
      template: "systems/dnd5e/templates/advancement/trait-config-details.hbs"
    },
    guaranteed: {
      container: { classes: ["column-container"], id: "column-left" },
      template: "systems/dnd5e/templates/advancement/trait-config-guaranteed.hbs"
    },
    choices: {
      container: { classes: ["column-container"], id: "column-left" },
      template: "systems/dnd5e/templates/advancement/trait-config-choices.hbs"
    },
    traits: {
      container: { classes: ["column-container"], id: "column-right" },
      template: "systems/dnd5e/templates/advancement/trait-config-traits.hbs"
    }
  };

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * Shortcut to the configuration data on the advancement.
   * @type {object}
   */
  get config() {
    return this.advancement.configuration;
  }

  /* -------------------------------------------- */

  /**
   * Index of the selected configuration, `-1` means `grants` array, any other number is equal
   * to an index in `choices` array.
   * @type {number}
   */
  selected;

  /* -------------------------------------------- */

  /**
   * Trait type to display in the selector interface.
   * @type {string}
   */
  trait;

  /* -------------------------------------------- */

  /**
   * List of trait types for the current selected configuration.
   * @type {Set<string>}
   */
  get types() {
    const pool = this.selected === -1 ? this.config.grants : this.config.choices[this.selected].pool;
    return this.advancement.representedTraits([pool]);
  }

  /* -------------------------------------------- */
  /*  Rendering                                   */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async _prepareContext(options) {
    const context = await super._prepareContext(options);

    context.grants = {
      label: localizedList({ grants: this.config.grants }) || "—",
      data: this.config.grants,
      selected: this.selected === -1
    };
    context.choices = this.config.choices.map((choice, index) => ({
      label: choiceLabel(choice, { only: true }).capitalize() || "—",
      data: choice,
      selected: this.selected === index
    }));
    const chosen = (this.selected === -1) ? context.grants.data : context.choices[this.selected].data.pool;
    if ( this.selected !== -1 ) context.count = {
      field: context.configuration.fields.choices.element.fields.count,
      value: context.choices[this.selected]?.data.count
    };
    context.selectedIndex = this.selected;

    const rep = this.advancement.representedTraits();
    context.disableAllowReplacements = rep.size > 1;
    const traitConfig = rep.size === 1 ? CONFIG.DND5E.traits[rep.first()] : null;
    if ( traitConfig ) {
      context.default.title = traitConfig.labels.title;
      context.default.icon = traitConfig.icon;
    } else {
      context.default.title = game.i18n.localize("DND5E.TraitGenericPlural.other");
      context.default.icon = this.advancement.constructor.metadata.icon;
    }
    context.default.hint = localizedList({ grants: this.config.grants, choices: this.config.choices });

    context.trait = {
      field: new BooleanField$s(),
      input: context.inputs.createCheckboxInput,
      options: await choices(this.trait, { chosen, prefixed: true, any: this.selected !== -1 }),
      selected: this.trait,
      selectedHeader: `${CONFIG.DND5E.traits[this.trait].labels.localization}.other`,
      typeField: new StringField$N({ label: game.i18n.localize("DND5E.ADVANCEMENT.Trait.TraitType") }),
      typeOptions: Object.entries(CONFIG.DND5E.traits)
        .filter(([, config]) => ((this.config.mode === "default") || (this.config.mode === "mastery"
          ? config.mastery : config.expertise)) && (config.dataType !== Number))
        .map(([value, config]) => ({ value, label: config.labels.title }))
    };

    // Disable selecting categories in mastery mode
    if ( this.advancement.configuration.mode === "mastery" ) {
      context.trait.options.forEach((key, value) => value.disabled = !!value.children);
    }

    context.mode = {
      hint: CONFIG.DND5E.traitModes[this.advancement.configuration.mode].hint,
      options: Object.entries(CONFIG.DND5E.traitModes).map(([value, { label }]) => ({ value, label }))
    };

    return context;
  }

  /* -------------------------------------------- */
  /*  Life-Cycle Handlers                         */
  /* -------------------------------------------- */

  /** @inheritDoc */
  _onRender(context, options) {
    super._onRender(context, options);
    // Handle selecting & disabling category children when a category is selected
    for ( const checkbox of this.element.querySelectorAll(".trait-list dnd5e-checkbox[checked]") ) {
      const toCheck = (checkbox.name.endsWith("*") || checkbox.name.endsWith("ALL"))
        ? checkbox.closest("ol").querySelectorAll(`dnd5e-checkbox:not([name="${checkbox.name}"])`)
        : checkbox.closest("li").querySelector("ol")?.querySelectorAll("dnd5e-checkbox");
      toCheck?.forEach(i => i.checked = i.disabled = true);
    }
  }

  /* -------------------------------------------- */
  /*  Event Listeners and Handlers                */
  /* -------------------------------------------- */

  /**
   * Handle adding a new choice.
   * @this {TraitConfig}
   * @param {Event} event         Triggering click event.
   * @param {HTMLElement} target  Button that was clicked.
   */
  static async #addChoice(event, target) {
    this.config.choices.push({ count: 1 });
    this.selected = this.config.choices.length - 1;
    await this.advancement.update({ configuration: await this.prepareConfigurationUpdate() });
  }

  /* -------------------------------------------- */

  /**
   * Handle removing a choice.
   * @this {TraitConfig}
   * @param {Event} event         Triggering click event.
   * @param {HTMLElement} target  Button that was clicked.
   */
  static async #removeChoice(event, target) {
    const input = target.closest("li").querySelector("[name='selectedIndex']");
    const selectedIndex = Number(input.value);
    this.config.choices.splice(selectedIndex, 1);
    if ( selectedIndex <= this.selected ) this.selected -= 1;
    await this.advancement.update({ configuration: await this.prepareConfigurationUpdate() });
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _onChangeForm(formConfig, event) {
    // Display new set of trait choices
    if ( event.target.name === "selectedTrait" ) {
      this.trait = event.target.value;
      return this.render();
    }

    // Change selected configuration set
    else if ( event.target.name === "selectedIndex" ) {
      this.selected = Number(event.target.value ?? -1);
      const types = this.types;
      if ( types.size && !types.has(this.trait) ) this.trait = types.first();
      return this.render();
    }

    // If mode is changed from default to one of the others, change selected type if current type is not valid
    if ( (event.target.name === "configuration.mode")
      && (event.target.value !== "default")
      && (event.target.value !== this.config.mode) ) {
      const checkKey = event.target.value === "mastery" ? "mastery" : "expertise";
      const validTraitTypes = filteredKeys(CONFIG.DND5E.traits, c => c[checkKey]);
      if ( !validTraitTypes.includes(this.trait) ) this.trait = validTraitTypes[0];
    }

    super._onChangeForm(formConfig, event);
  }

  /* -------------------------------------------- */
  /*  Form Handling                               */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async prepareConfigurationUpdate(configuration={}) {
    const choicesCollection = foundry.utils.deepClone(this.config.choices);

    if ( configuration.checked ) {
      const prefix = `${this.trait}:`;
      const filteredSelected = filteredKeys(configuration.checked);

      // Update grants
      if ( this.selected === -1 ) {
        const filteredPrevious = this.config.grants.filter(k => !k.startsWith(prefix));
        configuration.grants = [...filteredPrevious, ...filteredSelected];
      }

      // Update current choice pool
      else {
        const current = choicesCollection[this.selected];
        const filteredPrevious = current.pool.filter(k => !k.startsWith(prefix));
        current.pool = [...filteredPrevious, ...filteredSelected];
      }
      delete configuration.checked;
    }

    if ( configuration.count ) {
      choicesCollection[this.selected].count = configuration.count;
      delete configuration.count;
    }

    // TODO: Remove when https://github.com/foundryvtt/foundryvtt/issues/7706 is resolved
    choicesCollection.forEach(c => {
      if ( !c.pool ) return;
      c.pool = Array.from(c.pool);
    });
    configuration.choices = choicesCollection;
    configuration.grants ??= Array.from(this.config.grants);

    // If one of the expertise modes is selected, filter out any traits that are not of a valid type
    if ( (configuration.mode ?? this.config.mode) !== "default" ) {
      const checkKey = (configuration.mode ?? this.config.mode) === "mastery" ? "mastery" : "expertise";
      const validTraitTypes = filteredKeys(CONFIG.DND5E.traits, c => c[checkKey]);
      configuration.grants = configuration.grants.filter(k => validTraitTypes.some(t => k.startsWith(t)));
      configuration.choices.forEach(c => c.pool = c.pool?.filter(k => validTraitTypes.some(t => k.startsWith(t))));
    }

    return configuration;
  }
}

/**
 * Inline application that presents the player with a trait choices.
 */
class TraitFlow extends AdvancementFlow {

  /**
   * Array of trait keys currently chosen.
   * @type {Set<string>}
   */
  chosen;

  /* -------------------------------------------- */

  /** @inheritDoc */
  static get defaultOptions() {
    return foundry.utils.mergeObject(super.defaultOptions, {
      template: "systems/dnd5e/templates/advancement/trait-flow.hbs"
    });
  }

  /* -------------------------------------------- */

  /**
   * Trait configuration from `CONFIG.TRAITS` for this advancement's trait type.
   * @type {TraitConfiguration}
   */
  get traitConfig() {
    return CONFIG.DND5E.traits[this.advancement.configuration.type];
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async getData() {
    this.chosen ??= await this.prepareInitialValue();
    const available = await this.advancement.availableChoices(this.chosen);
    return foundry.utils.mergeObject(super.getData(), {
      hint: this.advancement.hint ? this.advancement.hint : localizedList({
        grants: this.advancement.configuration.grants, choices: this.advancement.configuration.choices
      }),
      slots: this.prepareTraitSlots(available),
      available
    });
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  activateListeners(html) {
    this.form.querySelectorAll("select").forEach(s => s.addEventListener("change", this._onSelectTrait.bind(this)));
    this.form.querySelectorAll(".remove").forEach(s => s.addEventListener("click", this._onRemoveTrait.bind(this)));
  }

  /* -------------------------------------------- */

  /**
   * Add a trait to the value when one is selected.
   * @param {Event} event  Triggering change to a select input.
   */
  _onSelectTrait(event) {
    const addedTrait = event.target.value;
    if ( addedTrait === "" ) return;
    this.chosen.add(addedTrait);
    this.render();
  }

  /* -------------------------------------------- */

  /**
   * Remove a trait for the value when the remove button is clicked.
   * @param {Event} event  Triggering click.
   */
  _onRemoveTrait(event) {
    const tag = event.target.closest(".trait-slot");
    this.chosen.delete(tag.dataset.key);
    this.render();
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _updateObject(event, formData) {
    if ( formData.chosen && !Array.isArray(formData.chosen) ) formData.chosen = [formData.chosen];
    super._updateObject(event, formData);
  }

  /* -------------------------------------------- */
  /*  Data Preparation Methods                    */
  /* -------------------------------------------- */

  /**
   * When only a single choice is available, automatically select that choice provided value is empty.
   * @returns {Set<string>}
   */
  async prepareInitialValue() {
    const existingChosen = this.retainedData?.chosen ?? this.advancement.value.chosen;
    if ( existingChosen?.size ) return new Set(existingChosen);
    const { available } = await this.advancement.unfulfilledChoices();
    const chosen = new Set();
    for ( const { choices } of available ) {
      const set = choices.asSet();
      if ( set.size === 1 ) chosen.add(set.first());
    }
    return chosen;
  }

  /* -------------------------------------------- */

  /**
   * Prepare the list of slots to be populated by traits.
   * @param {object} available  Trait availability returned by `prepareAvailableTraits`.
   * @returns {object[]}
   */
  prepareTraitSlots(available) {
    const config = this.advancement.configuration;
    const count = config.choices.reduce((count, c) => count + c.count, config.grants.size);
    const chosen = Array.from(this.chosen);
    let selectorShown = false;
    const slots = [];
    for ( let i = 1; i <= count; i++ ) {
      const key = chosen.shift();
      if ( selectorShown || (!key && !available) ) continue;
      selectorShown = !key;
      slots.push({
        key,
        label: key ? keyLabel(key, { type: config.type }) : null,
        showDelete: !this.advancement.configuration.grants.has(key),
        showSelector: !key
      });
    }
    return slots;
  }
}

const { ArrayField: ArrayField$g, BooleanField: BooleanField$r, NumberField: NumberField$w, SetField: SetField$n, SchemaField: SchemaField$B, StringField: StringField$M } = foundry.data.fields;

/**
 * Map language category changes.
 * @type {Record<string, string>}
 */
const _MAP = {
  "languages:exotic:draconic": "languages:standard:draconic",
  "languages:cant": "languages:exotic:cant",
  "languages:druidic": "languages:exotic:druidic"
};

const LANGUAGE_MAP = { modern: _MAP, legacy: foundry.utils.invertObject(_MAP) };

/**
 * Configuration for a specific trait choice.
 *
 * @typedef {object} TraitChoice
 * @property {number} count     Number of traits that can be selected.
 * @property {string[]} [pool]  List of trait or category keys that can be chosen. If no choices are provided,
 *                              any trait of the specified type can be selected.
 */

/**
 * Configuration data for the TraitAdvancement.
 *
 * @property {boolean} allowReplacements  Whether all potential choices should be presented to the user if there
 *                                        are no more choices available in a more limited set.
 * @property {TraitChoice[]} choices      Choices presented to the user.
 * @property {string[]} grants            Keys for traits granted automatically.
 * @property {string} mode                Method by which this advancement modifies the actor's traits.
 */
class TraitConfigurationData extends foundry.abstract.DataModel {

  /* -------------------------------------------- */
  /*  Model Configuration                         */
  /* -------------------------------------------- */

  /** @override */
  static LOCALIZATION_PREFIXES = ["DND5E.ADVANCEMENT.Trait"];

  /* -------------------------------------------- */

  static defineSchema() {
    return {
      allowReplacements: new BooleanField$r({ required: true }),
      choices: new ArrayField$g(new SchemaField$B({
        count: new NumberField$w({ required: true, positive: true, integer: true, initial: 1 }),
        pool: new SetField$n(new StringField$M(), { required: false })
      })),
      grants: new SetField$n(new StringField$M(), { required: true }),
      mode: new StringField$M({ required: true, blank: false, initial: "default" })
    };
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static migrateData(source) {
    super.migrateData(source);
    const version = game.settings.get("dnd5e", "rulesVersion");
    const languageMap = LANGUAGE_MAP[version] ?? {};
    if ( source.grants?.length ) source.grants = source.grants.map(t => languageMap[t] ?? t);
    if ( source.choices?.length ) source.choices.forEach(c => c.pool = c.pool.map(t => languageMap[t] ?? t));
    return source;
  }
}

/**
 * Value data for the TraitAdvancement.
 *
 * @property {Set<string>} chosen  Trait keys that have been chosen.
 */
class TraitValueData extends foundry.abstract.DataModel {
  static defineSchema() {
    return {
      chosen: new SetField$n(new StringField$M(), { required: false })
    };
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  static migrateData(source) {
    super.migrateData(source);
    const version = game.settings.get("dnd5e", "rulesVersion");
    const languageMap = LANGUAGE_MAP[version] ?? {};
    if ( source.chosen?.length ) source.chosen = source.chosen.map(t => languageMap[t] ?? t);
    return source;
  }
}

/**
 * Advancement that grants the player with certain traits or presents them with a list of traits from which
 * to choose.
 */
class TraitAdvancement extends Advancement {

  /** @inheritDoc */
  static get metadata() {
    return foundry.utils.mergeObject(super.metadata, {
      dataModels: {
        configuration: TraitConfigurationData,
        value: TraitValueData
      },
      order: 30,
      icon: "icons/sundries/scrolls/scroll-yellow-teal.webp",
      typeIcon: "systems/dnd5e/icons/svg/trait.svg",
      title: game.i18n.localize("DND5E.ADVANCEMENT.Trait.Title"),
      hint: game.i18n.localize("DND5E.ADVANCEMENT.Trait.Hint"),
      apps: {
        config: TraitConfig,
        flow: TraitFlow
      }
    });
  }

  /* -------------------------------------------- */

  /**
   * Perform the pre-localization of this data model.
   */
  static localize() {
    super.localize();
    localizeSchema(
      this.metadata.dataModels.configuration.schema.fields.choices.element,
      ["DND5E.ADVANCEMENT.Trait.FIELDS.choices"]
    );
  }

  /* -------------------------------------------- */

  /**
   * The maximum number of traits granted by this advancement. The number of traits actually granted may be lower if
   * actor already has some traits.
   * @type {number}
   */
  get maxTraits() {
    const { grants, choices } = this.configuration;
    return grants.size + choices.reduce((acc, choice) => acc + choice.count, 0);
  }

  /* -------------------------------------------- */
  /*  Preparation Methods                         */
  /* -------------------------------------------- */

  /**
   * Prepare data for the Advancement.
   */
  prepareData() {
    const rep = this.representedTraits();
    const traitConfig = rep.size === 1 ? CONFIG.DND5E.traits[rep.first()] : null;
    this.title = this.title || traitConfig?.labels.title || this.constructor.metadata.title;
    this.icon = this.icon || traitConfig?.icon || this.constructor.metadata.icon;
  }

  /* -------------------------------------------- */
  /*  Display Methods                             */
  /* -------------------------------------------- */

  /** @inheritDoc */
  configuredForLevel(level) {
    return !!this.value.chosen?.size;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  sortingValueForLevel(levels) {
    const traitOrder = Object.keys(CONFIG.DND5E.traits).findIndex(k => k === this.representedTraits().first());
    const modeOrder = Object.keys(CONFIG.DND5E.traitModes).findIndex(k => k === this.configuration.mode);
    const order = traitOrder + (modeOrder * 100);
    return `${this.constructor.metadata.order.paddedString(4)} ${order.paddedString(4)} ${this.titleForLevel(levels)}`;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  summaryForLevel(level, { configMode=false }={}) {
    if ( configMode ) {
      if ( this.hint ) return `<p>${this.hint}</p>`;
      return `<p>${localizedList({
        grants: this.configuration.grants, choices: this.configuration.choices
      })}</p>`;
    } else {
      return Array.from(this.value.chosen).map(k => `<span class="tag">${keyLabel(k)}</span>`).join(" ");
    }
  }

  /* -------------------------------------------- */
  /*  Application Methods                         */
  /* -------------------------------------------- */

  /** @inheritDoc */
  async apply(level, data) {
    const updates = {};
    if ( !data.chosen ) return;

    for ( const key of data.chosen ) {
      const keyPath = this.configuration.mode === "mastery" ? "system.traits.weaponProf.mastery.value"
        : changeKeyPath(key);
      let existingValue = updates[keyPath] ?? foundry.utils.getProperty(this.actor, keyPath);

      if ( ["Array", "Set"].includes(foundry.utils.getType(existingValue)) ) {
        existingValue = new Set(existingValue);
        existingValue.add(key.split(":").pop());
        updates[keyPath] = Array.from(existingValue);
      } else if ( (this.configuration.mode !== "expertise") || (existingValue !== 0) ) {
        updates[keyPath] = (this.configuration.mode === "default")
          || ((this.configuration.mode === "upgrade") && (existingValue === 0)) ? 1 : 2;
      }

      if ( key.startsWith("tool") ) {
        const toolId = key.split(":").pop();
        const ability = CONFIG.DND5E.tools[toolId]?.ability;
        const kp = `system.tools.${toolId}.ability`;
        if ( ability && !foundry.utils.hasProperty(this.actor, kp) ) updates[kp] = ability;
      }
    }

    this.actor.updateSource(updates);
    this.updateSource({ "value.chosen": Array.from(data.chosen) });
  }

  /* -------------------------------------------- */

  /** @override */
  automaticApplicationValue(level) {
    // TODO: Ideally this would be able to detect situations where choices are automatically fulfilled because
    // they only have one valid option, but that is an async process and cannot be called from within `render`
    if ( this.configuration.choices.length || this.configuration.allowReplacements ) return false;
    return { chosen: Array.from(this.configuration.grants) };
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async restore(level, data) {
    this.apply(level, data);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async reverse(level) {
    const updates = {};
    if ( !this.value.chosen ) return;

    for ( const key of this.value.chosen ) {
      const keyPath = this.configuration.mode === "mastery" ? "system.traits.weaponProf.mastery.value"
        : changeKeyPath(key);
      let existingValue = updates[keyPath] ?? foundry.utils.getProperty(this.actor, keyPath);

      if ( ["Array", "Set"].includes(foundry.utils.getType(existingValue)) ) {
        existingValue = new Set(existingValue);
        existingValue.delete(key.split(":").pop());
        updates[keyPath] = Array.from(existingValue);
      }

      else if ( this.configuration.mode === "expertise" ) updates[keyPath] = 1;
      else if ( this.configuration.mode === "upgrade" ) updates[keyPath] = existingValue === 1 ? 0 : 1;
      else updates[keyPath] = 0;
      // NOTE: When using forced expertise mode, this will not return to original value
      // if the value before being applied is 1.
    }

    const retainedData = foundry.utils.deepClone(this.value);
    this.actor.updateSource(updates);
    this.updateSource({ "value.chosen": [] });
    return retainedData;
  }

  /* -------------------------------------------- */
  /*  Helper Methods                              */
  /* -------------------------------------------- */

  /**
   * Two sets of keys based on actor data, one that is considered "selected" and thus unavailable to be chosen
   * and another that is "available". This is based off configured advancement mode.
   * @returns {{selected: Set<string>, available: Set<string>}}
   */
  async actorSelected() {
    const selected = new Set();
    const available = new Set();

    // If "default" mode is selected, return all traits
    // If any other mode is selected, only return traits that support expertise or mastery
    const traitTypes = this.configuration.mode === "default" ? Object.keys(CONFIG.DND5E.traits).filter(k => k !== "dm")
      : filteredKeys(CONFIG.DND5E.traits, t => t[this.configuration.mode === "mastery" ? "mastery" : "expertise"]);

    for ( const trait$1 of traitTypes ) {
      const actorValues$1 = await actorValues(this.actor, trait$1);
      const choices$1 = await choices(trait$1, { prefixed: true });
      for ( const key of choices$1.asSet() ) {
        const value = actorValues$1[key] ?? 0;
        if ( this.configuration.mode === "default" ) {
          if ( value >= 1 ) selected.add(key);
          else available.add(key);
        } else if ( this.configuration.mode === "mastery" ) {
          const split = key.split(":");
          split.pop();
          const category = split.join(":");
          if ( value === 2 ) selected.add(key);
          if ( (value === 1) || (actorValues$1[category] === 1) ) available.add(key);
        } else {
          if ( value === 2 ) selected.add(key);
          if ( (this.configuration.mode === "expertise") && (value === 1) ) available.add(key);
          else if ( (this.configuration.mode !== "expertise") && (value < 2) ) available.add(key);
        }
      }
    }

    return { selected, available };
  }

  /* -------------------------------------------- */

  /**
   * Guess the trait type from the grants & choices on this advancement.
   * @param {Set<string>[]} [pools]  Trait pools to use when figuring out the type.
   * @returns {Set<string>}
   */
  representedTraits(pools) {
    const set = new Set();
    pools ??= [this.configuration.grants, ...this.configuration.choices.map(c => c.pool)];
    for ( const pool of pools ) {
      for ( const key of pool ) {
        const [type] = key.split(":");
        set.add(type);
      }
    }
    return set;
  }

  /* -------------------------------------------- */

  /**
   * Prepare the list of available traits from which the player can choose.
   * @param {Set<string>} [chosen]  Traits already chosen on the advancement. If not set then it will
   *                                be retrieved from advancement's value.
   * @returns {{choices: SelectChoices, label: string}|null}
   */
  async availableChoices(chosen) {
    // TODO: Still shows "Choose 1 x" even if not possible due to mode restriction
    let { available, choices } = await this.unfulfilledChoices(chosen);

    // If all traits of this type are already assigned, then nothing new can be selected
    if ( foundry.utils.isEmpty(choices) ) return null;

    // Remove any grants that have no choices remaining
    let unfilteredLength = available.length;
    available = available.filter(a => a.choices.asSet().size > 0);

    // If replacements are allowed and there are grants with zero choices from their limited set,
    // display all remaining choices as an option
    if ( this.configuration.allowReplacements && (unfilteredLength > available.length) ) {
      const rep = this.representedTraits();
      if ( rep.size === 1 ) return {
        choices: choices.filter(this.representedTraits().map(t => `${t}:*`), { inplace: false }),
        label: game.i18n.format("DND5E.ADVANCEMENT.Trait.ChoicesRemaining", {
          count: unfilteredLength,
          type: traitLabel(rep.first(), unfilteredLength)
        })
      };
      // TODO: This works well for types without categories like skills where it is primarily intended,
      // but perhaps there could be some improvements elsewhere. For example, if I have an advancement
      // that grants proficiency in the Bagpipes and allows replacements, but the character already has
      // Bagpipe proficiency. In this example this just lets them choose from any other tool proficiency
      // as their replacement, but it might make sense to only show other musical instruments unless
      // they already have proficiency in all musical instruments. Might not be worth the effort.
    }

    if ( !available.length ) return null;

    // Create a choices object featuring a union of choices from all remaining grants
    const remainingSet = new Set(available.flatMap(a => Array.from(a.choices.asSet())));
    choices.filter(remainingSet);

    const rep = this.representedTraits(available.map(a => a.choices.asSet()));
    return {
      choices,
      label: game.i18n.format("DND5E.ADVANCEMENT.Trait.ChoicesRemaining", {
        count: available.length,
        type: traitLabel(rep.size === 1 ? rep.first() : null, available.length)
      })
    };
  }

  /* -------------------------------------------- */

  /**
   * The advancement configuration is flattened into separate options for the user that are chosen step-by-step. Some
   * are automatically picked for them if they are 'grants' or if there is only one option after the character's
   * existing traits have been taken into account.
   * @typedef {object} TraitChoices
   * @property {"grant"|"choice"} type  Whether this trait is automatically granted or is chosen from some options.
   * @property {number} [choiceIdx]     An index that groups each separate choice into the groups that they originally
   *                                    came from.
   * @property {SelectChoices} choices  The available traits to pick from. Grants have only 0 or 1, depending on whether
   *                                    the character already has the granted trait.
   */

  /**
   * Determine which of the provided grants, if any, still needs to be fulfilled.
   * @param {Set<string>} [chosen]  Traits already chosen on the advancement. If not set then it will
   *                                be retrieved from advancement's value.
   * @returns {{ available: TraitChoices[], choices: SelectChoices }}
   */
  async unfulfilledChoices(chosen) {
    const actorData = await this.actorSelected();
    const selected = {
      actor: actorData.selected,
      item: chosen ?? this.value.selected ?? new Set()
    };

    // If everything has already been selected, no need to go further
    if ( this.maxTraits <= selected.item.size ) {
      return { available: [], choices: new SelectChoices() };
    }

    const available = await Promise.all([
      ...this.configuration.grants.map(async g => ({
        type: "grant",
        choices: await mixedChoices(new Set([g]))
      })),
      ...this.configuration.choices.reduce((arr, choice, index) => {
        return arr.concat(Array.fromRange(choice.count).map(async () => ({
          type: "choice",
          choiceIdx: index,
          choices: await mixedChoices(choice.pool)
        })));
      }, [])
    ]);

    available.sort((lhs, rhs) => lhs.choices.asSet().size - rhs.choices.asSet().size);

    // Remove any fulfilled grants
    for ( const key of selected.item ) available.findSplice(grant => grant.choices.asSet().has(key));

    // Merge all possible choices into a single SelectChoices
    const allChoices = await mixedChoices(actorData.available);
    allChoices.exclude(new Set([...(selected.actor ?? []), ...selected.item]));
    available.forEach(a => a.choices = allChoices.filter(a.choices, { inplace: false }));

    return { available, choices: allChoices };
  }
}

/**
 * Data field that selects the appropriate advancement data model if available, otherwise defaults to generic
 * `ObjectField` to prevent issues with custom advancement types that aren't currently loaded.
 */
class AdvancementField extends foundry.data.fields.ObjectField {

  /**
   * Get the BaseAdvancement definition for the specified advancement type.
   * @param {string} type                    The Advancement type.
   * @returns {typeof BaseAdvancement|null}  The BaseAdvancement class, or null.
   */
  getModelForType(type) {
    return CONFIG.DND5E.advancementTypes[type]?.documentClass ?? null;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _cleanType(value, options) {
    if ( !(typeof value === "object") ) value = {};

    const cls = this.getModelForType(value.type);
    if ( cls ) return cls.cleanData(value, options);
    return value;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  initialize(value, model, options={}) {
    const cls = this.getModelForType(value.type);
    if ( cls ) return new cls(value, {parent: model, ...options});
    return foundry.utils.deepClone(value);
  }

  /* -------------------------------------------- */

  /**
   * Migrate this field's candidate source data.
   * @param {object} sourceData   Candidate source data of the root model
   * @param {any} fieldData       The value of this field within the source data
   */
  migrateSource(sourceData, fieldData) {
    const cls = this.getModelForType(fieldData.type);
    if ( cls ) cls.migrateDataSafe(fieldData);
  }
}

const { SchemaField: SchemaField$A, StringField: StringField$L } = foundry.data.fields;

/**
 * Data field for class & subclass spellcasting information.
 *
 * @property {string} progression          Spellcasting progression (e.g. full, half, pact).
 * @property {string} ability              Ability used for spell attacks and save DCs.
 * @property {object} preparation
 * @property {string} preparation.formula  Formula used to calculate max prepared spells, if a prepared caster.
 */
class SpellcastingField extends SchemaField$A {
  constructor(fields={}, options={}) {
    fields = {
      progression: new StringField$L({
        initial: "none",
        blank: false,
        label: "DND5E.SpellProgression"
      }),
      ability: new StringField$L({ label: "DND5E.SpellAbility" }),
      preparation: new SchemaField$A({
        formula: new FormulaField({ label: "DND5E.SpellPreparation.Formula" })
      }),
      ...fields
    };
    Object.entries(fields).forEach(([k, v]) => !v ? delete fields[k] : null);
    super(fields, { label: "DND5E.Spellcasting", ...options });
  }

  /* -------------------------------------------- */
  /*  Data Preparation                            */
  /* -------------------------------------------- */

  /**
   * Prepare data for this field. Should be called during the `prepareFinalData` stage.
   * @this {ItemDataModel}
   * @param {object} rollData  Roll data used for formula replacements.
   */
  static prepareData(rollData) {
    this.spellcasting.preparation.max = simplifyBonus(this.spellcasting.preparation.formula, rollData);

    // Temp method for determining spellcasting type until this data is available directly using advancement
    if ( CONFIG.DND5E.spellcastingTypes[this.spellcasting.progression] ) {
      this.spellcasting.type = this.spellcasting.progression;
    } else this.spellcasting.type = Object.entries(CONFIG.DND5E.spellcastingTypes).find(([, { progression }]) =>
      progression?.[this.spellcasting.progression]
    )?.[0];

    const actor = this.parent.actor;
    if ( !actor ) return;
    this.spellcasting.levels = this.levels ?? this.parent.class?.system.levels;

    // Prepare attack bonus and save DC
    const ability = actor.system.abilities?.[this.spellcasting.ability];
    const mod = ability?.mod ?? 0;
    const modProf = mod + (actor.system.attributes?.prof ?? 0);
    const msak = simplifyBonus(actor.system.bonuses?.msak?.attack, rollData);
    const rsak = simplifyBonus(actor.system.bonuses?.rsak?.attack, rollData);
    this.spellcasting.attack = modProf + (msak === rsak ? msak : 0);
    this.spellcasting.save = ability?.dc ?? (8 + modProf);
  }
}

const { SchemaField: SchemaField$z, HTMLField: HTMLField$8 } = foundry.data.fields;

/**
 * Data model template with item description & source.
 *
 * @property {object} description               Various item descriptions.
 * @property {string} description.value         Full item description.
 * @property {string} description.chat          Description displayed in chat card.
 * @property {string} identifier                Identifier slug for this item.
 * @property {SourceData} source                Adventure or sourcebook where this item originated.
 * @mixin
 */
class ItemDescriptionTemplate extends SystemDataModel {
  /** @inheritDoc */
  static defineSchema() {
    return {
      description: new SchemaField$z({
        value: new HTMLField$8({required: true, nullable: true, label: "DND5E.Description"}),
        chat: new HTMLField$8({required: true, nullable: true, label: "DND5E.DescriptionChat"})
      }),
      identifier: new IdentifierField({ required: true, label: "DND5E.Identifier" }),
      source: new SourceField()
    };
  }

  /* -------------------------------------------- */
  /*  Data Migrations                             */
  /* -------------------------------------------- */

  /** @inheritDoc */
  static _migrateData(source) {
    super._migrateData(source);
    ItemDescriptionTemplate.#migrateSource(source);
  }

  /* -------------------------------------------- */

  /**
   * Convert source string into custom object.
   * @param {object} source  The candidate source data from which the model will be constructed.
   */
  static #migrateSource(source) {
    if ( ("source" in source) && (foundry.utils.getType(source.source) !== "Object") ) {
      source.source = { custom: source.source };
    }
  }

  /* -------------------------------------------- */
  /*  Data Preparation                            */
  /* -------------------------------------------- */

  /**
   * Prepare the source label.
   */
  prepareDescriptionData() {
    const uuid = this.parent.flags.dnd5e?.sourceId ?? this.parent._stats?.compendiumSource ?? this.parent.uuid;
    SourceField.prepareData.call(this.source, uuid);
  }

  /* -------------------------------------------- */
  /*  Getters                                     */
  /* -------------------------------------------- */

  /**
   * What properties can be used for this item?
   * @returns {Set<string>}
   */
  get validProperties() {
    return new Set(CONFIG.DND5E.validProperties[this.parent.type] ?? []);
  }

  /* -------------------------------------------- */
  /*  Helpers                                     */
  /* -------------------------------------------- */

  /**
   * Create the properties filter configuration for a type.
   * @param {string} type  Item type.
   * @returns {CompendiumBrowserFilterDefinitionEntry}
   */
  static compendiumBrowserPropertiesFilter(type) {
    return {
      label: "DND5E.Properties",
      type: "set",
      config: {
        choices: Object.entries(CONFIG.DND5E.itemProperties).reduce((obj, [k, v]) => {
          if ( CONFIG.DND5E.validProperties[type]?.has(k) ) obj[k] = v;
          return obj;
        }, {}),
        keyPath: "system.properties",
        multiple: true
      }
    };
  }
}

/**
 * Field that stores activities on an item.
 */
class ActivitiesField extends MappingField {
  constructor(options) {
    super(new ActivityField(), options);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  initialize(value, model, options) {
    const activities = Object.values(super.initialize(value, model, options));
    activities.sort((a, b) => a.sort - b.sort);
    return new ActivityCollection(model, activities);
  }
}

/* -------------------------------------------- */

/**
 * Field that stores activity data and swaps class based on activity type.
 */
class ActivityField extends foundry.data.fields.ObjectField {

  /** @override */
  static recursive = true;

  /* -------------------------------------------- */

  /**
   * Get the document type for this activity.
   * @param {object} value            Activity data being prepared.
   * @returns {typeof Activity|null}  Activity document type.
   */
  getModel(value) {
    return CONFIG.DND5E.activityTypes[value.type]?.documentClass ?? null;
  }

  /* -------------------------------------------- */

  /** @override */
  _cleanType(value, options) {
    if ( !(typeof value === "object") ) value = {};

    const cls = this.getModel(value);
    if ( cls ) return cls.cleanData(value, options);
    return value;
  }

  /* -------------------------------------------- */

  /** @override */
  initialize(value, model, options = {}) {
    const cls = this.getModel(value);
    if ( cls ) return new cls(value, { parent: model, ...options });
    return foundry.utils.deepClone(value);
  }

  /* -------------------------------------------- */

  /**
   * Migrate this field's candidate source data.
   * @param {object} sourceData  Candidate source data of the root model.
   * @param {any} fieldData      The value of this field within the source data.
   */
  migrateSource(sourceData, fieldData) {
    const cls = this.getModel(fieldData);
    if ( cls ) cls.migrateDataSafe(fieldData);
  }
}

/* -------------------------------------------- */

/**
 * Specialized collection type for stored activities.
 * @param {DataModel} model     The parent DataModel to which this ActivityCollection belongs.
 * @param {Activity[]} entries  The activities to store.
 */
class ActivityCollection extends Collection {
  constructor(model, entries) {
    super();
    this.#model = model;
    for ( const entry of entries ) {
      if ( !(entry instanceof BaseActivityData) ) continue;
      this.set(entry._id, entry);
    }
  }

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * The parent DataModel to which this ActivityCollection belongs.
   * @type {DataModel}
   */
  #model;

  /* -------------------------------------------- */

  /**
   * Pre-organized arrays of activities by type.
   * @type {Map<string, Set<string>>}
   */
  #types = new Map();

  /* -------------------------------------------- */
  /*  Methods                                     */
  /* -------------------------------------------- */

  /**
   * Fetch an array of activities of a certain type.
   * @param {string} type  Activity type.
   * @returns {Activity[]}
   */
  getByType(type) {
    return Array.from(this.#types.get(type) ?? []).map(key => this.get(key));
  }

  /* -------------------------------------------- */

  /**
   * Generator that yields activities for each of the provided types.
   * @param {string[]} types  Types to fetch.
   * @yields {Activity}
   */
  *getByTypes(...types) {
    for ( const type of types ) {
      for ( const activity of this.getByType(type) ) yield activity;
    }
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  set(key, value) {
    if ( !this.#types.has(value.type) ) this.#types.set(value.type, new Set());
    this.#types.get(value.type).add(key);
    return super.set(key, value);
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  delete(key) {
    this.#types.get(this.get(key)?.type)?.delete(key);
    return super.delete(key);
  }

  /* -------------------------------------------- */

  /**
   * Test the given predicate against every entry in the Collection.
   * @param {function(*, number, ActivityCollection): boolean} predicate  The predicate.
   * @returns {boolean}
   */
  every(predicate) {
    return this.reduce((pass, v, i) => pass && predicate(v, i, this), true);
  }

  /* -------------------------------------------- */

  /**
   * Convert the ActivityCollection to an array of simple objects.
   * @param {boolean} [source=true]  Draw data for contained Documents from the underlying data source?
   * @returns {object[]}             The extracted array of primitive objects.
   */
  toObject(source=true) {
    return this.map(doc => doc.toObject(source));
  }
}

/**
 * @typedef AdvantageModeData
 * @property {number|null} override               Whether the mode has been entirely overridden.
 * @property {AdvantageModeCounts} advantages     The advantage counts.
 * @property {AdvantageModeCounts} disadvantages  The disadvantage counts.
 */

/**
 * @typedef AdvantageModeCounts
 * @property {number} count          The number of applications of this mode.
 * @property {boolean} [suppressed]  Whether this mode is suppressed.
 */

/**
 * Subclass of NumberField that tracks the number of changes made to a roll mode.
 */
class AdvantageModeField extends foundry.data.fields.NumberField {
  /** @inheritDoc */
  static get _defaults() {
    return foundry.utils.mergeObject(super._defaults, {
      choices: AdvantageModeField.#values,
      initial: 0,
      label: "DND5E.AdvantageMode"
    });
  }

  /* -------------------------------------------- */

  /**
   * Allowed advantage mode values.
   * @type {number[]}
   */
  static #values = [-1, 0, 1];

  /* -------------------------------------------- */
  /*  Active Effect Integration                   */
  /* -------------------------------------------- */

  /** @override */
  _applyChangeAdd(value, delta, model, change) {
    // Add a source of advantage or disadvantage.
    if ( (delta !== -1) && (delta !== 1) ) return value;
    const counts = this.constructor.getCounts(model, change);
    if ( delta === 1 ) counts.advantages.count++;
    else counts.disadvantages.count++;
    return this.constructor.resolveMode(model, change, counts);
  }

  /* -