/**
 * Add GitLab login to your page.
 *
 * ### Setup
 *
 * #### Callback URL
 * ```
 * https://example.com/api/auth/callback/gitlab
 * ```
 *
 * #### Configuration
 *```js
 * import Auth from "@auth/core"
 * import GitLab from "@auth/core/providers/gitlab"
 *
 * const request = new Request(origin)
 * const response = await Auth(request, {
 *   providers: [GitLab({ clientId: GITLAB_CLIENT_ID, clientSecret: GITLAB_CLIENT_SECRET })],
 * })
 * ```
 *
 * ### Resources
 *
 *  - [GitLab OAuth documentation](https://docs.gitlab.com/ee/api/oauth2.html)
 *
 * ### Notes
 *
 * By default, Auth.js assumes that the GitLab provider is
 * based on the [OAuth 2](https://www.rfc-editor.org/rfc/rfc6749.html) specification.
 *
 * :::tip
 * Enable the `read_user` option in scope if you want to save the users email address on sign up.
 * :::
 *
 * :::tip
 *
 * The GitLab provider comes with a [default configuration](https://github.com/nextauthjs/next-auth/blob/main/packages/core/src/providers/gitlab.ts).
 * To override the defaults for your use case, check out [customizing a built-in OAuth provider](https://authjs.dev/guides/providers/custom-provider#override-default-options).
 *
 * :::
 *
 * :::info **Disclaimer**
 *
 * If you think you found a bug in the default configuration, you can [open an issue](https://authjs.dev/new/provider-issue).
 *
 * Auth.js strictly adheres to the specification and it cannot take responsibility for any deviation from
 * the spec by the provider. You can open an issue, but if the problem is non-compliance with the spec,
 * we might not pursue a resolution. You can ask for more help in [Discussions](https://authjs.dev/new/github-discussions).
 *
 * :::
 */
export default function GitLab(options) {
    return {
        id: "gitlab",
        name: "GitLab",
        type: "oauth",
        authorization: "https://gitlab.com/oauth/authorize?scope=read_user",
        token: "https://gitlab.com/oauth/token",
        userinfo: "https://gitlab.com/api/v4/user",
        profile(profile) {
            return {
                id: profile.id.toString(),
                name: profile.name ?? profile.username,
                email: profile.email,
                image: profile.avatar_url,
            };
        },
        style: {
            logo: "/gitlab.svg",
            logoDark: "/gitlab-dark.svg",
            bg: "#fff",
            text: "#FC6D26",
            bgDark: "#FC6D26",
            textDark: "#fff",
        },
        options,
    };
}
